/**
* @class ICTouchAPI.webApplicationManager
* @singleton
* @ignore
* @extends Object
*/
dojo.provide("ICTouchAPI.webApplicationManager");
dojo.declare("ICTouchAPI.webApplicationManager",
	null,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		currentlyLoadingApplications : 0,

		/**
		 * Time in seconds before we unload an application
		 * @property
		 * @type int
		 */
		TIME_FOR_UNLOADING: 60,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @private
		 * array of loaded web applications.
		 */
		_arrLoadedApplications : {},
		/**
		 * @private
		 */
		_arrLoadedWebapp : [],
		/**
		 * @private
		 */
		_boolSafeMode : false,
		/**
		 * @private
		 */
		_handleTick : null,
		/**
		 * @private
		 */
		_arrUnloadableApplication: [],

		/* ------------------------------------ Constructor --------------------------------------- */

                 /**
                 * @private
                 */
		constructor: function () {
			// If webapps are not loaded from the requireList, then make the
			// webApplicationManager receiptive to start_webapp events.
			if (!ICTouchAPI.devContext.loadWebappFromRequireList) {
				ICTouchAPI.eventServices.subscribeToEvent(this,"start_webapp",this._APILoadWebapp);
				this.currentlyLoadingApplications = 0;
			}
			ICTouchAPI.eventServices.subscribeToEvent(this,"end_of_init_IctDirectory",this.initDataStores);
			// If we are in remote display, call manually the initDataStores function to create the dataStore of contacts
			// (in remote display we will not receive the 'end_of_init_IctDirectory' event)'
			if (ICTouchAPI.remoteDisplay) {
				this.initDataStores();
			}
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

                /**
                 * @private
                 */
		getApplicationName : function(){
			return "webApplicationManager";
		},

		/**
		 * Get all loaded webApplications.
		 * @returns {Object} list of loaded webApplications indexed by their name.
                 * @private
		 */
		getLoadedApplications: function () {
			return this._arrLoadedApplications;
		},
		/**
		 * Get a webApplication's reference from its name.
		 * @param {String} strWebApplicationName Name of the webApplication
		 * @returns {Object} reference or false if undefined.
		 */
		getRefApplication: function (strWebApplicationName) {
			if (this._arrLoadedApplications[strWebApplicationName] == undefined) {
				return false;
			} else {
				return this._arrLoadedApplications[strWebApplicationName];
			}
		},
		/**
		 * Get the number of loaded webApplications.
		 * @returns {Int} number of loaded webApplications.
		 */
		getNumberOfApplications: function () {
			var i=0;
			for (var objApplication in this._arrLoadedApplications) {
				i++;
			}
			return i;
		},

		/**
		 * Get the current safe mode status
		 * @return {Bool} current safe mode status (on : true, off : false)
		 */
		getSafeModeStatus : function(){
			return this._boolSafeMode;
		},

		/**
		 * Set or remove the flag of an application that allows it to be unloaded when it has been closed
		 * @param {string} strWebApplicationName
		 * @param {string} boolUnloadable
		 */
		setApplicationUnloadable: function(strWebApplicationName, boolUnloadable) {
			if( boolUnloadable ) {
				this._arrUnloadableApplication[strWebApplicationName] = new Date().getTime();
			}
			else {
				delete this._arrUnloadableApplication[strWebApplicationName];
			}

			// Check if we still have unloadable application
			for(var i in this._arrUnloadableApplication) {
				this._startTickInterval();
				return;
			}

			// If we get here we don't have any app
			this._stopTickInterval();
			return;
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		initDataStores : function () {
			ICTouchAPI.dataStoreServices.createGlobalDataStore({

				name: "contacts",
				rowId: "contactId",

				// enable auto mapping
				mapping: true,

				// config object for data provider
				dataProvider: {
					onInit:   {
						func : ICTouchAPI.APIServices.IctDirectory.getLocalContacts,
						context : ICTouchAPI.APIServices.IctDirectory
					},
					onCreate: {
						event: ["DIR_CONTACTS_CREATED"],
						//func: ICTouchAPI.APIServices.IctDirectory.getContactById,
						//context : ICTouchAPI.APIServices.IctDirectory
						argumentName : "contacts" // Name of the event arguments (in case of multiple arguments)
					},
					onUpdate: {
						event: ["DIR_CONTACTS_UPDATED","DIR_CONTACTSINPREVIEW_CREATED","DIR_CONTACTSINPREVIEW_DELETED"],
						//func: ICTouchAPI.APIServices.IctDirectory.getContactById,
						//context : ICTouchAPI.APIServices.IctDirectory
						argumentName : "contacts" // Name of the event arguments (in case of multiple arguments)
					},
					onSynchro: {
						event: ["DIR_CONTENT_CHANGED"],
						//func: ICTouchAPI.APIServices.IctDirectory.getLocalContacts,
						//context : ICTouchAPI.APIServices.IctDirectory
					},
					onDelete: {
						event: ["DIR_CONTACTS_DELETED"],
						argumentName : "contactIds" // Name of the event arguments (in case of multiple arguments)
					}
				}

			});
		},

		/**
		 * First loads the list of all webapps, (webapp/requireList.js)
		 * Then loads every file in these webapps. (webapp/[oneWebapp]/requireList.js)
		 */
		loadInitFiles: function () {
			// Get the list of all webapps.
			// Load the requireList that define the webapp startup order
			dojo.require("webapp.requireList" + generalConfig.version, true);

			// Webapps are loaded from requireList
			if (ICTouchAPI.devContext.loadWebappFromRequireList) {
				webapp.startedWebappList = webapp.requireList;
			} // Else do nothing, webapp.startedWebappList is filled in by _APILoadWebapp

			this._arrLoadedWebapp = webapp.startedWebappList;

			//Unload the Settings webapp, if it is not required
			if (webapp.startedWebappList.indexOf("settings") === -1){
				this.unloadApplication("settings");
			}

			// For each webapp, get its js list of required files.
			dojo.forEach(webapp.requireList,
				function (oneWebapp) {
					if(webapp.startedWebappList.indexOf(oneWebapp) != -1 && (!webapp[oneWebapp] || (webapp[oneWebapp] && !webapp[oneWebapp]["data"]))) {
						// Load the webapp
						ICTouchAPI.webApplicationManager.loadWebapp(oneWebapp);

						// Once the webapp loaded, delete the webapp from the loaded webapp list
						var webappIndex = ICTouchAPI.webApplicationManager._arrLoadedWebapp.indexOf(oneWebapp);
						ICTouchAPI.webApplicationManager._arrLoadedWebapp.splice(webappIndex,1);
				}
					else {
						if (!webapp[oneWebapp]) {
							ICTouchAPI.debugServices.warning("ICTouchAPI.webApplicationManager - loadInitFiles / Hum, " + oneWebapp + " is not in the webapp require list.");
						}
						else {
							ICTouchAPI.debugServices.warning("ICTouchAPI.webApplicationManager - loadInitFiles / The webapp " + oneWebapp + " has been already loaded.");
						}
					}
				}
				);

			// For each webapp started but not present in the require list, load the webapp
			// following the start_webapp order list.
			// In safe mode, these webapps are not loaded
			if (!this.getSafeModeStatus()) {
				dojo.forEach(this._arrLoadedWebapp,
					function (oneWebapp) {
						if(!webapp[oneWebapp]) {
							ICTouchAPI.webApplicationManager.loadWebapp(oneWebapp);
						}
					}
					);
			}

			if (webapp.initialization && webapp.initialization.initWidget && webapp.initialization.initWidget.changeMessage) {
				webapp.initialization.initWidget.changeMessage("End of init");
			}

		},

		/**
		 * Load a web app from a namespace
		 * @param {string} strWebappName The name of the web app
		 * @param {string} strNamespace The web app namespace
		 */
		loadWebappFromNamespace : function(strWebappName, strNamespace) {
			// Load minified files
			if (!generalConfig.developer) {
				dojo.require(strNamespace + "." +strWebappName+".webapp_"+strWebappName+"_mini_"+generalConfig.version, true);
			}
			// Load normal files
			else  {
				ICTouchAPI.debugServices.debug("ICTouchAPI.webApplicationManager - loadWebappFromNamespace / require:" + strNamespace + "." + strWebappName+".requireList");
				dojo.require(strNamespace + "." + strWebappName+".requireList", true);

				dojo.forEach(awap[strWebappName]["requireList"],
					function (webappFile) {
						var webAppPath = strNamespace + "." + strWebappName;
						dojo.require(webAppPath+"."+webappFile+ generalConfig.version , true);
					}
				);
			}
		},

		/**
		 * Load a webapp with the webappname
		 * @param {String} strWebappName Name of the application to load
		 */
		loadWebapp : function(strWebappName){


			if (this._arrLoadedApplications["webapp." + strWebappName]) {
				return false;
			}
			if (webapp.initialization && webapp.initialization.initWidget && webapp.initialization.initWidget.changeMessage) {
				webapp.initialization.initWidget.changeMessage(_(["Loading webapp", strWebappName],"webapp.initialization"));
			}
			this.currentlyLoadingApplications++;
			// If not in developer mode, load minified file instead.
			if (!generalConfig.developer) {
				dojo.require("webapp."+strWebappName+".webapp_"+strWebappName+"_mini_"+generalConfig.version, true);

				// For each webapp properties
				for (var widget in webapp[strWebappName]) {
					// Check if it exists.
					if (webapp[strWebappName].hasOwnProperty(widget)) {
						// Check if it has a prototype
					/*if (webapp[strWebappName][widget] && webapp[strWebappName][widget].prototype) {
							// Check it has a cssFile property.
							var cssFile = webapp[strWebappName][widget].prototype.cssFile;
							if (cssFile) {
								// If it's an object
								if(typeof cssFile === "object") {
									// load every css file.
									dojo.forEach(cssFile,function(css){
										ICTouchAPI.skinServices.loadCssSkinFile("webapp."+strWebappName, css, false);
									});
								}
								else {
									ICTouchAPI.skinServices.loadCssSkinFile("webapp."+strWebappName, cssFile, false);
								}
							}
					}*/
					}
				}
			} else {
				dojo.require("webapp."+strWebappName+".requireList", true);
				dojo.forEach(webapp[strWebappName]["requireList"],
					function (webappFile) {
						var webAppPath = "webapp."+strWebappName;

						dojo.require(webAppPath+"."+webappFile+ generalConfig.version, true);
						//Load related css file(s)
					/*if(webapp[strWebappName][webappFile] && webapp[strWebappName][webappFile].prototype){
							var cssFile = webapp[strWebappName][webappFile].prototype.cssFile;
							if(cssFile){
								if(typeof cssFile === "object"){
									dojo.forEach(cssFile,function(css){
										ICTouchAPI.skinServices.loadCssSkinFile(webAppPath, css, false);
									});
								}
								else{
									ICTouchAPI.skinServices.loadCssSkinFile(webAppPath, cssFile, false);
								}
							}
					}*/
					}
					);
			}

			//Load webapp skins
			ICTouchAPI.skinServices.loadWebappStyle(strWebappName);

			if(webapp[strWebappName].loaded) {
				webapp[strWebappName].loaded();
			}
			if(typeof(webapp[strWebappName].load) == "function"){
				ICTouchAPI.addOnInit(webapp[strWebappName], function(){
					//try{
					if(webapp[strWebappName]){
						webapp[strWebappName].load();
					}
				});
			}

		},

		/**
		 * register a webApplication if it doesn't exist.
		 * @throws error existing application if it does already exist.
		 */
		registerApplication:function(objApplication){
			// Before registering a new webapp, check if it exists.
			if(this._arrLoadedApplications[objApplication.getApplicationName()]){
				throw {
					errorType : 'existing application',
					message : objApplication.getApplicationName() + ' application is already loaded'
				}
			}
			else{
				// If it doesn't exist.
				this._arrLoadedApplications[objApplication.getApplicationName()]=objApplication;
				// Notify it has been added.
				this._notifyAddedWebApp(objApplication.getApplicationName());
				// Decreases the loading applications counter.
				this.currentlyLoadingApplications --;
			}
		},

		unloadApplication: function (strApplicationName) {
			ICTouchAPI.debugServices.debug("ICTouchAPI.webApplicationManager - unloadApplication $ strApplicationName: " + strApplicationName);
			var split = strApplicationName.split(".");
			var appName = (split[0] == "webapp" ? split[1] : split[0]);
			if (typeof this._arrLoadedApplications["webapp."+appName] == "object") {

				this._arrLoadedApplications["webapp."+appName].unload()
				this.closeApplication("webapp."+appName);

			}

			delete this._arrLoadedApplications["webapp."+appName];
			delete webapp[appName];
		},

		unloadWebappFromNamespace : function(appName, namespace) {
			var listWidget = awap[appName].getLoadedWidgets();
			for( var strWidget in listWidget ) {
				var objWidget = listWidget[strWidget];
				ICTouchAPI.tools.destroyWidgets(objWidget.domNode);
				objWidget.webapp = null;
				delete listWidget[strWidget];
			}
			awap[appName].removeWidgetsRef();

			ICTouchAPI.debugServices.debug("ICTouchAPI.webApplicationManager - loadWebappFromNamespace / remove loadedModules for app: " + appName);

			var cpt	= 0;
			var regex = new RegExp('^awap\.'+appName, "gi");
			for(var i in dojo._loadedModules) {
				if(i.match(regex)) {
					var viewName = i.split(".");
					ICTouchAPI.AppBarServices.destroyAppBar(appName, viewName[2]);
					ICTouchAPI.AppBarServices.removeExitButton(appName, viewName[2]);
					delete dojo._loadedModules[i];
					cpt++;
				}
			}

			ICTouchAPI.debugServices.debug("ICTouchAPI.webApplicationManager - loadWebappFromNamespace / remove loadedUrls for app: " + appName);
			
			for (var i=dojo._loadedUrls.length-cpt;i < dojo._loadedUrls.length;i++) {
				var path = dojo._loadedUrls[i];
				dojo._loadedUrls[path] = false;
			}

			dojo._loadedUrls.splice(dojo._loadedUrls.length-cpt, cpt);
			ICTouchAPI.transitionServices._removeWebappScreens(namespace+"."+appName);
			delete this._arrLoadedApplications[namespace+"."+appName];
			delete awap[appName];
		},

		/**
		 * Destroy and remove any widget bounded to the webapp
		 * @param {string} strApplicationName name of the webapp
		 */
		closeApplication: function (strApplicationName) {
			ICTouchAPI.debugServices.debug("ICTouchAPI.webApplicationManager - closeApplication $ strApplicationName: " + strApplicationName);
			var objWebapp = dojo.getObject(strApplicationName);

			if( typeof objWebapp == "undefined" ) {
				ICTouchAPI.debugServices.warning("ICTouchAPI.webApplicationManager - closeApplication / Application " + strApplicationName + " does not exist");
				return;
			}

			var listWidget = objWebapp.getLoadedWidgets();
			for( var strWidget in listWidget ) {
				var objWidget = listWidget[strWidget];
				ICTouchAPI.tools.destroyWidgets(objWidget.domNode);

				delete listWidget[strWidget];
			}

			ICTouchAPI.transitionServices._removeWebappScreens(strApplicationName);
			objWebapp.removeWidgetsRef();

		},

		/**
		 * Notify the manager that we left an application
		 * This should only be called by transitionServices unless you know what you are doing
		 * @param {string} strApplicationName Name of the webapp
		 */
		notifyApplicationExit: function(strApplicationName) {


			if( typeof this._arrUnloadableApplication[strApplicationName] != "undefined" ) {
				// Unload this when the time has come
				this._arrUnloadableApplication[strApplicationName] = new Date().getTime();
			}
		},

		/**
		 * Notify the manager that we enterd an application
		 * This should only be called by transitionServices unless you know what you are doing
		 * @param {string} strApplicationName Name of the webapp
		 */
		notifyApplicationEnter: function(strApplicationName) {
			if( typeof this._arrUnloadableApplication[strApplicationName] != "undefined" ) {
				// Don't unload this at the moment
				this._arrUnloadableApplication[strApplicationName] = -1;
			}
		},

		/**
		 * Set/unset the safe mode
		 * @return {Bool} current safe mode status (on : true, off : false)
		 */
		toggleSafeMode : function(){
			this._boolSafeMode = !this._boolSafeMode;
			return this._boolSafeMode;
		},

		applicationsStillLoading: function () {
			return this.currentlyLoadingApplications > 0 ? true : false;
		},

		/* --------------------------------- Private Methods -------------------------------------- */


		/**
		 * @private
		 */
		_APILoadWebapp : function(objEvent){
			webapp.startedWebappList = webapp.startedWebappList || [];
			webapp.startedWebappList.push(objEvent.value);
		},

		/**
		 * @private
		 * Sends an notification when a webapp is launched/created.
		 * @param {String} webApplicationName Name of the webApplication
		 */
		_notifyAddedWebApp: function (webApplicationName) {
			dojo.publish("webApplicationManager.notify.added", [webApplicationName]);
		},
		/**
		 * @private
		 * Sends an notification when a webapp is closed/killed.
		 * @param {String} webApplicationName Name of the webApplication
		 */
		_notifyRemovedWebApp: function (webApplicationName) {
			dojo.publish("webApplicationManager.notify.removed", [webApplicationName]);
		},

		/**
		 * @private
		 * Stop the setInterval of the tick function do nothing if it isn't launched
		 */
		_stopTickInterval: function() {
			if( this._handleTick != null ) {
				// Stop tick function
				clearInterval(this._handleTick);
				this._handleTick = null;
			}
		},

		/**
		 * @private
		 * Start the setInterval of the tick function do nothing if it is already started
		 */
		_startTickInterval: function() {
			if( this._handleTick == null ) {
				// Create a tick function for unloading ( call it every 10seconds )
				var that = this;
				var func = function () {
					that._unloadTick();
				}
				this._handleTick = setInterval(func, 10000);
			}
		},

		/**
		 * @private
		 * Tick function called every 10 seconds it check if an app needs to be unloaded
		 */
		_unloadTick: function() {
			var now = new Date().getTime();
			var intervalInMS = this.TIME_FOR_UNLOADING * 1000;
			for(var strWebapp in this._arrUnloadableApplication) {
				var lastSeen = this._arrUnloadableApplication[strWebapp];
				if( lastSeen > 0 && ( lastSeen + intervalInMS ) < now ) {
					this._arrUnloadableApplication[strWebapp] = -1;
					this.closeApplication(strWebapp);
				}
			}
		}
	});

// Create Application Manager.
ICTouchAPI.webApplicationManager=new ICTouchAPI.webApplicationManager;
