/**
* @class ICTouchAPI.transitionServices
* @singleton
* @extends Object
* Manage the construction of new screens and the transition between screens
*/
dojo.provide("ICTouchAPI.transitionServices");

dojo.declare("ICTouchAPI.transitionServices",null,
{

	/* --------------------------------- Public attributes ------------------------------------ */




	/* --------------------------------- Private attributes ----------------------------------- */



	/* ------------------------------------ Constructor --------------------------------------- */

	/**
	 * reference of displayed widget
	 * @ignore
	 */
	_objCurrentScreen : null,


	/**
	 * reference of displayed actionBar
	 * @ignore
	 */
	_objCurrentActionBar: null,


	/**
	 * this attribut sets on or off the transition effect
	 * @ignore
	 */
	_transitionMode:false,

	/**
	 * @ignore
	 */
	_history: [],

	/**
	 * @ignore
	 */
	_stealthIScreens: {},

	/**
	 * @ignore
	 */
	_IScreens:{},

	/**
	 * @ignore
	 */
	_IScreensByWebapps: {},

	/**
	 * @ignore
	 */
	_IActionBars:{},

	/**
	 * @ignore
	 */
	_IActionBarsByWebapps: {},

	/**
	 * @ignore
	 */
	_isChangingScreen : false,

	/**
	 * @ignore
	 */
	_defaultScreen : {},

	/**
	 * @ignore
	 */
	_handlerSpinner : null,

	/**
	 * @ignore
	 */
	_objSpinner : null,

	/**
	 * @ignore
	 */
	_transitionHandler : null,

	/**
	 * @ignore
	 */
	_targetArgs : null,

	/**
	 * @ignore
	 */
	_transitionTimer : 0,

	/* ------------------------------------ Constructor --------------------------------------- */

	/**
	 * The Transition Services manages contruction of new screens, an manages graphical transition between screens
	 */
	constructor : function(){
		this._defaultScreen = {
			name: "webapp.homepage.getHomepage",
			params: {},
			position: "Middle"
		};

		this._transitionHandler = dojo.subscribe("unlockTransition",null,function(unlockState){
			if(unlockState){
				if(ICTouchAPI.transitionServices._transitionTimer){
					clearTimeout(ICTouchAPI.transitionServices._transitionTimer);
					ICTouchAPI.transitionServices._transitionTimer=null;
				}
				if(!ICTouchAPI.transitionServices._targetArgs.sync){
					ICTouchAPI.transitionServices._transitionTimer = setTimeout(function(){
						ICTouchAPI.transitionServices._moveToScreen(ICTouchAPI.transitionServices._targetArgs);
					}, 15);
				}else{
					ICTouchAPI.transitionServices._moveToScreen(ICTouchAPI.transitionServices._targetArgs);
				}
			}
			else {
				ICTouchAPI.transitionServices._isChangingScreen = false;
			}
		});

		ICTouchAPI.eventServices.subscribeToEvent(this, "presentationLoaded", this._presentationLoaded);
	},


	/* ----------------------------------- Getter / Setter------------------------------------- */

	/**
	 * Get the Application Service name
	 * @ignore
	 * @return {String} The Application Service Name
	 */
	getApplicationName:function(){
		return "transitionServices";
	},

	/**
	 * If we are switching to a screen, return its name, else return the current screen
	 * @ignore
	 * @return {String} The screen name
	 */
	getNextOrCurrentScreenName:function() {
		if (this._isChangingScreen) {
			return this._targetArgs.name;
		}
		return this.getCurrentScreenName();
	},
	
	/**
	 * Get the Current Screen Name
	 * @param {Boolean} visible if true only return a visible screen name, search through history latest non stealthy screen
	 * @return {String} The screen name
	 */
	getCurrentScreenName:function(visible){
		// return false if _objCurrentScreen is not defined (case of getCurrentScreenName called by getScreen for initialization webapp, no screen was previously existing)
		if (!this._objCurrentScreen) {
			return false;
		}
		var id = this._objCurrentScreen.id;
		if (this.isstealth(id) && visible) {
			// When no screen has been found, return default screen name
			id = this._defaultScreen.name;
			// Search the history for a non stealthy screen from newest to oldest
			var i = this._history.length;
			var name;
			for(var i = this._history.length-1; i >= 0; i--) {
				name = this._history[i];
				if (!this.isstealth(name)) {
					id = name;
					break;
				}
			}
		}
		return id;
	},

	/**
	 * Get the IFrame containing the current WebApp
	 * @ignore
	 * @return {Object} The IFrame
	 */
	getCurrentWebAppIFrameDocument: function () {
		return this._objCurrentScreen.getContentDocument();
	},

	/**
	 * Get the IFrame containing the current WebApp
	 * @ignore
	 * @return {Object} The IFrame
	 */
	getCurrentWebAppIFrame: function () {
		return this._objCurrentScreen._screenFrame;
	},

	/* ----------------------------------- Public methods ------------------------------------- */

	/**
	 * Get a screen, if screens exists, it is upped to the top and updated. Else, a new screen is created
	 * @param {Object} args JSON Object containing:<pre><code>
	 * - name: the name of the screen
	 * - params: the list of parameters
	 * - position: The position of the screen (ie: middle)
	 * - zindex: The z order index</code></pre>
	 * @return {Boolean} True if a transition has been done
	 */
	getScreen : function(args){
		// return false if the screen to get is the current screen displayed
		if(!dojo.getObject(args.name) || args.name == this.getCurrentScreenName(false)){
			return false;
		}
		if (this._isChangingScreen && !args.sync) {
			return false;
		}
		console.log("getScreen to :"+args.name);
		this._isChangingScreen = true;
		this._targetArgs = args;
		this._unlock(args.name);
	},

	/**
	 * Get the default screen
	 * @return {boolean} Was the transition successful
	 */
	getDefaultScreen : function() {
		return this.getScreen(this._defaultScreen);
	},

	/**
	 * Preload a screen when it doesn't exist
	 * @param {Object} args JSON Object containing:<pre><code>
	 * - name: the name of the screen
	 * - params: the list of parameters
	 * - position: The position of the screen (ie: middle)
	 * - zindex: The z order index</code></pre>
	 */
	preloadScreen : function(args) {
		// Some safeguards
		if( !args || typeof args.name !== "string" ) {
			console.warn("Please see the documentation on how to use preloadScreen");
			return;
		}
		if( !this._testScreenName(args.name) ) {
			console.warn("Screen name "+args.name+" does not exist");
			return;
		}
		// End of safeguards
		// Create the new screen if it doesn't exist
		if (!this._getIScreen(args.name)) {
			var screen = this._createIScreen(args.name, args.params, args.position || "Middle");

			screen._preFirstShow();

			// Hide the preloaded iframes just in case
			screen._hide();
		}
	},


	/**
	 * Display the Previous Screen
	 * @return {boolean} Was the transition successful
	 */
	back : function () {
		var lastScreen = this._history.pop();
		if (typeof lastScreen !== "undefined") {
			if (!this.isstealth(lastScreen)) {
				return this.getScreen({
					name : lastScreen,
					params: {}
				});
			} else {
				return this.back();
			}
		} else {
			return this.getDefaultScreen();
		}


	},

	/**
	 * Stealth a Widget
	 * A stealth widget is not added to the history stack
	 * @param {String} strWidgetName The name of the Widget
	 */
	stealth :function(strWidgetName) {
		this._stealthIScreens[strWidgetName] = true;
	},

	/**
	 * The widget will be added to the history stack again
	 * @param {String} strWidgetName The name of the Widget
	 */
	unstealth: function (strWidgetName) {
		this._stealthIScreens[strWidgetName] =  false;
	},

	/**
	 * Is a Widget Stealthy ?
	 * @param {String} strWidgetName The name of the Widget
	 * @return {Boolean} True is the Widget is Stealthy
	 */
	isstealth: function (strWidgetName) {
		return this._stealthIScreens[strWidgetName];
	},


	/**
	 * Set the first screen
	 * @param args params for the first widget constructor
	 * @ignore
	 */
	initialize : function(args){
		var screen;
		for (var i=0; i<args.length; i++) {
			screen = this._createIScreen(args[i].name,  args[i].params, args[i].position);
			// Show it because screens are created hidden
			screen._show();

		}
	},

	/**
	 * Add a stylesheet link to the header of the generic iframes
	 * @param {String} path : the path to the stylesheet file to link to the generic iframes
	 * @ignore
	 */
	addStyleToGenericIFrame : function(path) {
		ICTouchAPI.debugServices.debug("ICTouchAPI.transitionServices - addStyleToGenericIFrame / add to the generic iFrames the stylesheet with the path " + path);
		var i;
		// Ugly hack: get static iframes
		var screenObj = ICTouchAPI.transitionServices.screen.prototype._static_iframePool;

		// For each predefined bottom iframe
		for(i=0 ; i < screenObj.appbarFrame.length; i++) {
			ICTouchAPI.skinServices.addStyleToHeader(screenObj.appbarFrame[i]._getHead(), path);
		}

		// For each predefined screen iframe
		for(i=0; i < screenObj.screenFrame.length; i++) {
			ICTouchAPI.skinServices.addStyleToHeader(screenObj.screenFrame[i]._getHead(), path);
		}
	},

	/**
	 * Remove a stylesheet link from the header of the generic iframes
	 * @param {String} path : the path to the stylesheet file to remove from the generic iframes
	 * @ignore
	 */
	removeStyleToGenericIFrame : function(path) {
		ICTouchAPI.debugServices.debug("ICTouchAPI.transitionServices - removeStyleToGenericIFrame / remove to the generic iFrames the stylesheet with the path " + path);
		var i;
		// Ugly hack: get static iframes
		var screenObj = ICTouchAPI.transitionServices.screen.prototype._static_iframePool;

		// For each predefined bottom iframe
		for(i=0 ; i < screenObj.appbarFrame.length; i++) {
			ICTouchAPI.skinServices.removeStyleToHeader(screenObj.appbarFrame[i]._getHead(), path);
		}

		// For each predefined screen iframe
		for(i=0; i < screenObj.screenFrame.length; i++) {
			ICTouchAPI.skinServices.removeStyleToHeader(screenObj.screenFrame[i]._getHead(), path);
		}
	},

	/**
	 * Add CSS rule to the DOM Header of the generic iFrames
	 * @param {String} selector : the CSS Selector to add in the DOM Header of the generic iFrames
	 * @param {String} rule : the rule attached to the CSS Selector to add in the DOM Header of the generic iFrames
	 * @ignore
	 */
	addCssRuleToGenericIFrame : function(selector, rule) {
		ICTouchAPI.debugServices.debug("ICTouchAPI.transitionServices - addCssRuleToGenericIFrame / add to the generic iFrames the css selector " + selector + ", with the rule " + rule);
		var i;
		// Ugly hack: get static iframes
		var screenObj = ICTouchAPI.transitionServices.screen.prototype._static_iframePool;

		// For each predefined bottom iframe
		for(i=0 ; i < screenObj.appbarFrame.length; i++) {
			ICTouchAPI.skinServices.addCssRuleToFrame(screenObj.appbarFrame[i], selector, rule);
		}

		// For each predefined screen iframe
		for(i=0; i < screenObj.screenFrame.length; i++) {
			ICTouchAPI.skinServices.addCssRuleToFrame(screenObj.screenFrame[i], selector, rule);
		}
	},

	/**
	 * Remove a CSS rule from the DOM Header of the generic iFrames
	 * @param {String} selector: The CSS Selector to remove from the DOM Header of the generic iFrames
	 * @ignore
	 */
	removeCssRuleToGenericIFrame : function(selector) {
		ICTouchAPI.debugServices.debug("ICTouchAPI.transitionServices - removeCssRuleToGenericIFrame / remove to the generic iFrames the css selector " + selector);
		var i;
		// Ugly hack: get static iframes
		var screenObj = ICTouchAPI.transitionServices.screen.prototype._static_iframePool;

		// For each predefined bottom iframe
		for(i=0 ; i < screenObj.appbarFrame.length; i++) {
			ICTouchAPI.skinServices.removeCssRuleToFrame(screenObj.appbarFrame[i], selector);
		}

		// For each predefined screen iframe
		for(i=0; i < screenObj.screenFrame.length; i++) {
			ICTouchAPI.skinServices.removeCssRuleToFrame(screenObj.screenFrame[i], selector);
		}
	},


	/* ----------------------------------- Private methods ------------------------------------- */

        /**
         * @ignore
         */
	_presentationLoaded: function() {
		var lstWebapp = [];
		for(var i in webapp) {
			if(webapp[i]._strApplicationName!=null) {
				lstWebapp.push(webapp[i]._strApplicationName);
			}
		}
		ICTouchAPI.skinServices.linkWebappsStyles("webapp.mainframe",lstWebapp,false,false,true,false);
		ICTouchAPI.skinServices.linkWebappsStyles("webapp.appbarframe",lstWebapp,true,false,false,false);
		ICTouchAPI.eventServices.unsubscribeToEvent(this,"presentationLoaded");
	},

	/**
	 * Check if this screen exist and can be instantiated
	 * @ignore
	 * @param {String} screenName name of the screen to test
	 * @returns {boolean} if it exist
	 */
	_testScreenName: function(screenName) {
		var screen = ICTouchAPI.tools.getObjectReference(screenName);
		return (typeof screen === "function");
 	},

	/**
	 * @ignore
	 */
	_unlock : function(){
		dojo.publish("unlockTransition", [true]);
	},

	/**
	 * @ignore
	 */
	_basicUnlock : function(){
		dojo.publish("unlockTransition", [true]);
	},

	/**
	 * @ignore
	 */
	_getNextUnlock : function(name){
		var reg = /^webapp\.[^.]+/;
		var webappName = reg.exec(name);
		if(webappName == null) {
			reg = /^awap\.[^.]+/;
			webappName = reg.exec(name);
		}
		var refOnObject = dojo.getObject(webappName[0]);
		return dojo.hitch(
			(refOnObject ? refOnObject:{}),
			(refOnObject.unlock ? refOnObject.unlock : this._basicUnlock)
			);
	},

	/**
	 * @ignore
	 */
	_moveToScreen : function(args){
		// Some safeguards
		if( !args || typeof args.name !== "string" ) {
			console.warn("Please see the documentation on how to use getScreen");
			return false;
 		}
		if( !this._testScreenName(args.name) ) {
			console.warn("Screen name "+args.name+" does not exist");
			return false;
		}
		// End of safeguards

		args.params = args.params || {};
		/*if (this._isChangingScreen) {
 			return false;
 		}*/
		this._unlock = this._getNextUnlock(this._targetArgs.name);


		// Hide current screen if it's different than the required one
		if (this._objCurrentScreen && this._objCurrentScreen.id != args.name) {
			this._objCurrentScreen._hide();
			// Save history
			this._history.push(this._objCurrentScreen.id);
			this._checkApplicationExit(this._objCurrentScreen.id, args.name);
		}
		// Create the new screen if it doesn't exist
		if (!this._getIScreen(args.name)) {
			this._createSpinner();
			this._createIScreen(args.name, args.params, args.position || "Middle");
		}


		// If it's not the current one, show it.
		if (this._objCurrentScreen == null || this._objCurrentScreen.id != args.name) {
			this._getIScreen(args.name)._show();
		}

		if (args.zindex) {
			try{
				dojo.style(this._getIScreen(args.name), "z-index", args.zindex);
			}catch(err){
				console.warn("transitionServices.getScreen : trying to set a Zindex but argument is not a DOM object");
				console.warn(err);
			}
		}

		this._objCurrentScreen = this._getIScreen(args.name);

		this._isChangingScreen = false;
		this._removeSpinner();

		return true;
	},

	/**
	 * @ignore
	 */
	_getIScreen: function (strScreenName) {
		return this._IScreens[strScreenName];
	},

	/**
	 * Create an iframe and attach some utility method to it
	 * @ignore
	 */
	_createIFrame: function (strIFrameId, position, callback) {
		var documentFragment = document.createDocumentFragment();
		var iframe = document.createElement("iframe");
		documentFragment.appendChild(iframe);
		iframe.id = strIFrameId;
		iframe.src = "iframe.html";
		iframe.style.left="-1000px"; // Make the iframe hidden by default
		iframe.className = position || "Middle";

		if( typeof callback === "function" ) {
			var firstLoad = function() {
				// Remove listeners to avoid memory leaks
				iframe.removeEventListener('load', firstLoad, false);

				// iframe context in callback
				callback.call(iframe);
			}
			iframe.addEventListener('load', firstLoad, false);
 		}

		iframe._show = function () {
			this.style.left = "";
			var id = this.id;
			// publish an event when this iframe is visible
			setTimeout(function(){
				dojo.publish("iframe.show", [id]);
			}, 100);
		};
		iframe._hide = function () {
			// publish an event when this iframe is hidden
			dojo.publish("iframe.hide", [this.id]);
			this.style.left = "-1000px";
		};
		iframe._getHead = function () {
			return dojo.query("head",iframe.contentDocument)[0];
		}
		document.body.appendChild(documentFragment);
		return iframe;
	},

	/**
	 * @ignore
	 */
	_removeWebappScreens: function (strWebappName) {
		var iframes = this._IScreensByWebapps[strWebappName];
		if (iframes && iframes.length) {
			for (var i=0; i < iframes.length; i++) {
				var iframeId = iframes[i].id;
				this._removeScreen(iframeId, strWebappName);
			}
			delete this._IScreensByWebapps[strWebappName];
		}
	},

	/**
	 * Add a screen in the stack if transition enable else in the GlobalContainerMiddle
	 * Add the equivalent action bar to the _arrActionBars array
	 * @ignore
	 * @param {String} strScreenName name of the screen to create
	 * @param {Object} arrParams params used for the creation of the screen
	 * @param {Object} position location of the screen ( "Middle", "FullScreen", etc... )
	 * @return {Object} newly created screen
	 */
	_createIScreen:function(strScreenName, arrParams, position){
		var screen;
		var tmpObj = dojo.getObject(strScreenName);
		var screenContent = new tmpObj(arrParams);

		position = position || "Middle";

		// This is needed so ICTouchAPI.tools.getFrameId can work even when iframe isn't fully initialized
		screenContent.domNode.setAttribute("frameid", strScreenName);

		// If it's a webapp view use the new screen class, else create a simple iframe for it
		if( position === "Middle" ) {
			screen = new ICTouchAPI.transitionServices.screen(screenContent, strScreenName, screenContent.exclusiveIframe === true);
		}
		else {
			// Every screen that is not a "normal" screen is created the old way
			screen = this._createIFrame(strScreenName, position, function () {
				ICTouchAPI.skinServices.registerHeader(this._getHead(), screenContent.namespace, "webapp", this);
				screenContent.placeAt(this.contentDocument.body);
			});
		}

		this._IScreens[strScreenName] = screen;

		this._IScreensByWebapps[screenContent.namespace] = this._IScreensByWebapps[screenContent.namespace] || [];
		this._IScreensByWebapps[screenContent.namespace].push(screen);

		return screen;
	},

	/**
	 * Destroy a screen
	 * @ignore
	 * @param {String} screenName name of the screen to destroy / remove
	 */
	_removeScreen: function (strScreenName, strWebappName) {
		var objScreen = this._getIScreen(strScreenName);

		if( !objScreen ) {
			return;
		}
		if( typeof objScreen.destroy === "function" ) {
			// It's a "complex" screen, call the destroy method of it
			objScreen.destroy();
		}
		else {
			ICTouchAPI.skinServices.unregisterHeader(objScreen);
			// It's an iframe, do the cleaning stuff ourself
			dojo.empty(objScreen.contentDocument.body);
			// Remove iframe from dom
			dojo.destroy(objScreen);
		}
		delete this._IScreens[strScreenName];

		// Add it to stealth list
		this.stealth(strScreenName);
	},

	/**
	 * Replace the appbar from screen with an another one
	 * @ignore
	 * @param {String} screenName name of the screen where the appbar should be placed
	 * @param {Object} actionBar AppBarWidget object
	 */
	_setActionBar: function(screenName,refActionBar) {
		var screen = this._getIScreen(screenName);
		if( screen ) {
			screen.replaceAppbar(refActionBar);
 		}
 	},

	/*
	 * Returns an array containing the contentDocument of each loaded webapp's view
	 * @ignore
	 * Use with care it's not a toy
	 */
	getLoadedViewsContainer : function () {
		var _arr = [], i;
		for (i in this._IScreens) {
			if (this._IScreens.hasOwnProperty(i)) {
				_arr.push(this._IScreens[i].contentDocument);
			}
		}
		return _arr;
	},

	/*
	 * Check if we are leaving the current application and notify webApplicationManager if it's the case
	 * @ignore
	 * @param {String} Name of the view we are leaving
	 * @param {String} Name of the view we are entering
	 */
	_checkApplicationExit: function(strOldView, strNewView) {
		var splitOld = strOldView.split(".");
		var splitNew = strNewView.split(".");

		if( splitOld[1] != splitNew[1] ) {
			var appName = "webapp."+splitOld[1];
			ICTouchAPI.webApplicationManager.notifyApplicationExit(appName);
			appName = "webapp."+splitNew[1];
			ICTouchAPI.webApplicationManager.notifyApplicationEnter(appName);
		}
	},

	/**
	 * Create the spinner UIElement
	 * @ignore
	 */
	_createSpinner: function() {
		this._destroySpinner();
		this._objSpinner = new UIElements.Spinner.SpinnerControl({}, document.body);
	},

	/**
	 * Remove the spinner UIElement after 4 seconds
	 * @ignore
	 */
	_removeSpinner: function() {
		// Delay the end of the animation by 4seconds because we don't know when the images in CSS are fully loaded
		// Homepage seems to take that amount of time to load
		var context = this;
		this._handlerSpinner = setTimeout(function(){
			context._destroySpinner();
		}, 4000);
	},

	/**
	 * Destroy the spinner and don't fail if it isn't present
	 * @ignore
	 */
	_destroySpinner: function() {
		if( this._handlerSpinner ) {
			clearTimeout(this._handlerSpinner);
			this._handlerSpinner = null;
		}
		if( this._objSpinner ) {
			this._objSpinner.destroy();
			this._objSpinner = null;
		}
	}
});

// Create transition services
ICTouchAPI.transitionServices=new ICTouchAPI.transitionServices();

/**
* @class ICTouchAPI.transitionServices.screen
* @ignore
* @namespace ICTouchAPI.transitionServices
* @extends Object
* Ease the management of middle screens in createScreen.
* This API <b>must</b> be compatible with an iframe returned by createIframe ( _show, _hide methods )
*/
dojo.declare("ICTouchAPI.transitionServices.screen", null, {
	// Id of contained webapp
	id					: null,
	// Id of the appbar
	appbarId			: null,

	// Content widget
	_content			: null,
	// Appbar widget
	_appbar				: null,

	_screenFrame		: null,
	_appbarFrame		: null,

	// Must this screen have an exclusive iframe ?
	_exclusiveIframe	: false,

	// Is this the first time the screen has been shown
	_first				: true,

	// Defines
	SCREEN				: 0,
	APPBAR				: 1,

	// Static attribute were one iframe should always be ready
	_static_iframePool	: {screenIndex: 2, appbarIndex: 2, screenFrame: null, appbarFrame: null, init: false},

	constructor: function(content, id, exclusiveIframe) {
		this._content = content;
		// Set the id for this screen to the id of the content ( if available )
		if( typeof id === "string" ) {
			this.id = id;
			var appbarName = id.split(".");
			appbarName[0] = "appbar";
			this.appbarId = appbarName.join(".");
		}
		else {
			this.id = "mainframe_noname";
			this.appbarId = "appbarframe_noname";
		}
		this._exclusiveIframe = exclusiveIframe;
	},

	destroy: function() {
		this._hide();
		delete this._content;
		delete this._appbar;
	},

	getContentDocument: function() {
		return this._screenFrame ? this._screenFrame.contentDocument : null;
	},

	show: function() {
		this._show();
	},

	hide: function() {
		this._hide();
	},

	replaceAppbar: function(newAppbar) {
		if (newAppbar == this._appbar) {
			return;
		}

		this._appbar = newAppbar;
		if( this._appbarFrame !== null ) {
			this._cleanFrame(this._appbarFrame, false, this.APPBAR);
			this._attachToFrame(newAppbar, this._appbarFrame);
		}
	},

	_static_initPool: function() {
		if( this._static_initPool.init ) {
			return;
		}
		this._static_initPool.init = true;

		this._static_iframePool.screenFrame = [];
		this._static_iframePool.appbarFrame = [];

		this._static_iframePool.screenFrame[0] = this._static_createIframe("mainframe0", this.SCREEN);
		this._static_iframePool.appbarFrame[0] = this._static_createIframe("appbarframe0", this.APPBAR);

		this._static_iframePool.screenFrame[1] = this._static_createIframe("mainframe1", this.SCREEN);
		this._static_iframePool.appbarFrame[1] = this._static_createIframe("appbarframe1", this.APPBAR);

	},

	_static_createIframe: function(name, screen) {
		if( screen === this.SCREEN ) {
			return ICTouchAPI.transitionServices._createIFrame(name, "Middle", function() {
				ICTouchAPI.skinServices.registerHeader(this._getHead(), "mainframe", "webapp", this);
				name = null; // Helps the garbage collector
			});
		}
		else if( screen === this.APPBAR ) {
			return ICTouchAPI.transitionServices._createIFrame(name, "BottomBar", function() {
				ICTouchAPI.skinServices.registerHeader(this._getHead(), "appbarframe", "appbar", this);
				name = null; // Helps the garbage collector
			});
		}

		return null;
	},

	// Attach a widget to a frame
	_attachToFrame: function(widget, frame) {
		frame.contentDocument.adoptNode(widget.domNode);
		widget.placeAt(frame.contentDocument.body);
	},

	// Remove the widget from the frame and remove link that aren't part of the framework
	_cleanFrame: function(frame, removeHeaders, type) {
		// Remove content from this iframe
		dojo.empty(frame.contentDocument.body);
		if( removeHeaders ) {
			// transform to the type of skinServices
			type = type === this.SCREEN ? "webapp" : "appbar";
			ICTouchAPI.skinServices.changeTag(frame, "mainframe", type);
		}
	},

	/* Add specific link to header
	   frame is the frame to attach the content to
	   name of the webapp ( "webapp.communication" for example )
	   type screen or actionbar
	*/
	_addSpecificLinks: function(frame, name, type) {
		// transform to the type of skinServices
		type = type === this.SCREEN ? "webapp" : "appbar";

		ICTouchAPI.skinServices.changeTag(frame, name, type);
	},

	_show: function() {
		// Main screen
		// If this is an exclusive frame we only attach the head & body the first time
		if( !this._exclusiveIframe || this._first) {
			this._preFirstShow();
		}
		this._first = false;

		this._screenFrame._show();
		this._appbarFrame._show();

		this._notifyWebappShow( this._screenFrame.id );
	},


	_notifyWebappShow: function( id ) {
		//console.log( 'transition: screen._show', id );

		var splitId = id.split(".");

		if( undefined !== splitId[1] ) {
			dojo.publish( "ICTouchAPI.transitionServices.WEBAPP_SHOW" , [{'id': id}] );
			if (webapp[ splitId[1] ]) {
				webapp[ splitId[1] ].onShow( splitId[2] );
			}
		}
	},


	_preFirstShow : function() {
		// Get the name of the webapp
		var name = this._content.webapp.getApplicationName();
		// Only use the name of webapp and not the full namespace
		name = name.split(".")[1];

		this._screenFrame = this._getNextIframe(this.SCREEN);
		this._screenFrame.id = this.id;

		// Body
		this._attachToFrame(this._content, this._screenFrame);

		// Appbar frame
		this._appbarFrame = this._getNextIframe(this.APPBAR);
		this._appbarFrame.id = this.appbarId;
		var appbarName = this.appbarId.split(".");

		// Get new style of appbar
		this._appbar = ICTouchAPI.AppBarServices._getAppBarWidget(appbarName[1],appbarName[2]);
		if (this._appbar === null) {
			// If it doesn't work, get old style of appbar
			this._appbar = ICTouchAPI.AppBarServices.getAppBar(appbarName[1],appbarName[2]);
		}
		else if (this._appbar) {
			this._appbar._checkArrows();
		}

		// Body
		this._attachToFrame(this._appbar, this._appbarFrame);

		this._first = false;
	},

	_hide: function() {
		if( this._screenFrame === null ) {
			return;
		}
		this._notifyWebappHide( this._screenFrame.id );
		this._screenFrame._hide();
		if( this._appbarFrame === null ) {
			return;
		}
		this._appbarFrame._hide();

		// Only remove the webapp CSS when it's not an exclusive iframe because we will reuse it later
		if( !this._exclusiveIframe ) {
			this._screenFrame.id = "";
			this._cleanFrame(this._screenFrame, false, this.SCREEN);
			this._screenFrame = null;

			this._appbarFrame.id = "";
			this._cleanFrame(this._appbarFrame, false, this.APPBAR);
			this._appbarFrame = null;
		}
	},

	_notifyWebappHide: function( id ) {
		//console.log( 'transition: screen._hide', id );

		var splitId = id.split(".");

		if( undefined !== splitId[1] ) {
			dojo.publish( "ICTouchAPI.transitionServices.WEBAPP_HIDE" , [{'id': id}] );
			if (webapp[ splitId[1] ]) {
				webapp[ splitId[1] ].onHide();
			}
		}
	},

	_getNextIframe: function(screen) {
		// If an iframe is already defined this is a no-op
		if( this._screenFrame !== null && screen === this.SCREEN ) {
			return this._screenFrame;
		}
		if( this._appbarFrame !== null && screen === this.APPBAR ) {
			return this._appbarFrame;
		}

		var idx, list;
		if (screen === this.SCREEN) {
			idx = this._static_iframePool.screenIndex++;
			list = this._static_iframePool.screenFrame
		}
		else {
			idx = this._static_iframePool.appbarIndex++;
			list = this._static_iframePool.appbarFrame;
		}

		var iframe = list[idx%2];
		if( this._exclusiveIframe ) {
			var name = screen === this.SCREEN  ? "mainframe"+idx : "appbarframe"+idx;
			list[idx%2] = this._static_createIframe(name, screen);
		}

		return iframe;
	}
});

/* Initialize pool of iframes, because this is a STATIC method we must call it this way
  But this cannot be called in the first constructor call because the iframe won't be ready on time
*/
ICTouchAPI.transitionServices.screen.prototype._static_initPool();
