/**
* @class ICTouchAPI.toasterServices
* @singleton
* @extends Object
* @ignore
*/
dojo.provide("ICTouchAPI.toasterServices");
dojo.declare("ICTouchAPI.toasterServices",null,
{
	/* --------------------------------- Public attributes ------------------------------------ */

  /**
   * Callback when clicking outside the Toaster
   * @cfg {Function} onClickOutside Function to call
   */
	onOutsideClickClbk : null,

	/* --------------------------------- Private attributes ----------------------------------- */

	/**
	 * @private
	 */
	_arrIToasters : {},
	/**
	 * @private
	 */
	_currentDisplayed : "",

	/**
	 * List of "hidden" callbacks for toasters
	 * @private
	 */
	_arrOnHideCallback: [],

	/* ------------------------------------ Constructor --------------------------------------- */

	/**
	 * @private
	 */
	constructor : function() {
		this._arrIToasters = {};
		this._currentDisplayed = "";
		this._arrOnHideCallback = [];
		ICTouchAPI.eventServices.subscribeToEvent(this, "SENSKEY_HOME", this._eventPhysicalKeyHome);
	},

	/**
	 * @private
	 */
	getApplicationName: function(){
		return "toasterServices";
	},

	/* ----------------------------------- Getter / Setter------------------------------------- */

	/**
	 * Return the id of the view displayed in the toaster
	 * @return {string} id of the view
	 */
	getContentDisplayedId : function (){
		if (this._currentDisplayed && this._arrIToasters[this._currentDisplayed]){
			return this._arrIToasters[this._currentDisplayed].id;
		}
		return "";
	},
	
	/**
	 * Return the IToaster of the view displayed in the toaster
	 * @return {string} id of the view
	 */
	getContentDisplayed : function (){
		if (this._currentDisplayed && this._arrIToasters[this._currentDisplayed]){
			return this._arrIToasters[this._currentDisplayed];
		}
		return null;
	},
	
        /**
         * Set the callback when clicking outside the Toaster
         * @param {Function} func The callback
         */
        setOnClickOutsideCallback : function(func) {
            ICTouchAPI.toasterServices.onOutsideClickClbk = func;
        },

        /**
         * Get the callback when clcking outside the Toaster
         * @return {Function} The callback
         */
        getOnClickOutsideCallback : function() {
            return (ICTouchAPI.toasterServices.onOutsideClickClbk);
        },


	/* ----------------------------------- Public methods ------------------------------------- */

	/**
	 * Show a toaster with a DialogBox created for the user
	 * @param {Object} toasterContent The content to place inside the DialogBox. The content can be null
	 * @param {Object} params Additionnal parameters
	 * @return {MyICPhoneAPI.form.DialogBox} the DialogBox (that must be passed to hideContent to hide it).
	 */
	showDialogBox: function(toasterContent, params) {
		var arrToasterButtons = [];
		if( params.funcOk ) {
			arrToasterButtons.push({
				strButtonName: "TOASTER_OK",
				strButtonLabel: _("Ok","ICTouchAPI"),
				strButtonIcon: "generic-ok",
				callback: params.funcOk
			});
		}
		if( params.funcCancel ) {
			arrToasterButtons.push({
				strButtonName: "TOASTER_CANCEL",
				strButtonLabel: _("Cancel","ICTouchAPI"),
				strButtonIcon: "generic-cancel",
				callback: params.funcCancel
			});
		}
		var objContent = new UIElements.DialogBox.DialogBoxControl({
			strTitle: params.strTitle || "",
			arrButtons: arrToasterButtons
		});

		if( typeof toasterContent == "object" && toasterContent != null ) {
			objContent.placeContent(toasterContent);
		}
		this.showContent({
			"toasterContent" : objContent,
			"onHide": params.onHide
		});
		return objContent;
	},

	/**
	 * Show the toaster
	 * @param {Object} args Json Object<pre><code>
	 * {"toasterContent" : ...
	 * "onOutsideClick" : ...
	 * "onHide" : ... }</code></pre>
	 */
	showContent : function (args) {
		var toasterContent = args.toasterContent;
		var id = toasterContent.id;
		this.onOutsideClickClbk = args.onOutsideClick;
		if (this._currentDisplayed !== toasterContent.id) {
			if  (!this._arrIToasters[id]) {
				this._addToaster(toasterContent);
				this._arrOnHideCallback[id] = args.onHide;
			}
			if (this._currentDisplayed !== "") {
				this._arrIToasters[this._currentDisplayed]._hide();
				this._callOnHide(this._currentDisplayed);
			}
			this._currentDisplayed = toasterContent.id;
		}
		this._arrIToasters[this._currentDisplayed]._show();
	},

	/**
	 * Preload the toaster
	 * @param {Object} toasterContent The Toaster content
	 */
	preloadContent : function (toasterContent) {
		if  (!this._arrIToasters[toasterContent.id]) {
			var toaster = this._addToaster(toasterContent);
			toaster._hide();
		}
	},

	/**
	 * Reload the toaster
	 * @private
	 * @param {Object} toasterContent The Toaster content
	 */
	reloadContent : function(toasterContent) {
		if  (this._arrIToasters[toasterContent.id]) {
			delete this._arrIToasters[toasterContent.id];
			this._arrIToasters.splice(toasterContent.id, 1);
		}
		var toaster = this._addToaster(toasterContent);
		toaster._hide();
	},

	/**
	 * Hide all Toaster
	 */
	hideAll : function () {
		if (this._currentDisplayed !=="") {
			this._arrIToasters[this._currentDisplayed]._hide();
			this._callOnHide(this._currentDisplayed);
			this._currentDisplayed = "";
		}
	},

	/**
	 * Hide a specific Toaster
	 * @param {Object} objContent The toaster to hide
	 */
	hideContent : function (objContent) {
		if (!objContent) {
			ICTouchAPI.debugServices.warning("ICTouchAPI.toasterServices - hideContent / Hum, can't hide the toaster... Why not passing a defined argument next time?!");
			return;
		}
		var id = objContent.id;
		var context = this;
		setTimeout(function(){
			context._arrIToasters[id]._hide();
			if(context._currentDisplayed === id) {
				context._callOnHide(id);
				context._currentDisplayed = "";
			}
		}, 15);
	},

    destroyContent : function(objContent) {
		if(objContent){
			var id = objContent.id;
			var frame = ICTouchAPI.toasterServices._arrIToasters[id];

			frame._hide();
			if(ICTouchAPI.toasterServices._currentDisplayed === id) {
				ICTouchAPI.toasterServices._currentDisplayed = "";
				this._callOnHide(id);
			}
			ICTouchAPI.skinServices.unregisterHeader(frame);
			objContent.destroy();
			dojo.destroy(frame.contentDocument.body);
			dojo.destroy(frame);
			delete ICTouchAPI.toasterServices._arrIToasters[id];
			delete this._arrOnHideCallback[id];
		}
    },

	/**
	 * When clicking outside the toaster, launch the callback defined in onOutsideClickClbk
	 * @private
	 */
	onHideToaster : function () {
		if (this.onOutsideClickClbk && typeof this.onOutsideClickClbk.func === "function") {
			this.onOutsideClickClbk.func.apply(this.onOutsideClickClbk.context);
		} else {
			ICTouchAPI.toasterServices.destroyContent(ICTouchAPI.toasterServices.getContentDisplayed());
		}
	},

	/**
	 * Is a specified Toaster displayed
	 * @param {Object} content The content to check
	 * @return {Boolean} True if this Toaster is displayed
	 */
	isContentDisplayed : function(content){
		if(this._arrIToasters[this._currentDisplayed] != null) {
			return (this._arrIToasters[this._currentDisplayed].id === content.id);
		}
		else {
			return false;
		}
	},

	/* ----------------------------------- Private methods ------------------------------------- */

	/**
	 * @private
	 */
	_eventPhysicalKeyHome : function(objEvent) {
		if(objEvent && objEvent.value !== 0){
			ICTouchAPI.toasterServices.hideAll();
		}
	},

	/**
	 * @private
	 */
	_addToaster: function (toasterContent) {
		var documentFragment = document.createDocumentFragment();
		var iFrame = document.createElement("iframe");
		documentFragment.appendChild(iFrame);
		iFrame.id = toasterContent.id;
		iFrame.src = "iframe.html";
		iFrame.style.height="0px";
		iFrame.className = "FullScreen";
		iFrame.style.zIndex = "10000";
		toasterContent.attr("class", "Toaster");
		iFrame.addEventListener('load', function () {
			//ICTouchAPI.skinServices.registerWebappHeader(this.contentDocument.getElementsByTagName('head')[0], toasterContent.namespace, "Toaster");
			ICTouchAPI.skinServices.registerHeader(this.contentDocument.getElementsByTagName('head')[0], toasterContent.namespace,"webapp",this);
			toasterContent.placeAt(this.contentDocument.body);
			dojo.create("div", {className: "toasterMask"}, this.contentDocument.body);
			iFrame.contentDocument.addEventListener('mousedown', function (event) {
				if (event.target.nodeName == "DIV" && event.target.className == "toasterMask") {
					ICTouchAPI.toasterServices.onHideToaster();
				}
			},false);
		},false);
		iFrame.addEventListener('load', function () {
			iFrame.style.height="";
			document.body.appendChild(documentFragment);
			iFrame.removeEventListener('load', arguments.callee, false);
		},false);
		iFrame._show = function () {
			this.style.top = "";
			dojo.publish("toaster.show", [this.id]);
			/* DEBUG */  ICTouchAPI.debugServices.debug('[toasterServices] _show()');
		};
		iFrame._hide = function () {
			this.style.top = "1000px";
			dojo.publish("toaster.hide", [this.id]);
			/* DEBUG */  ICTouchAPI.debugServices.debug('[toasterServices] _hide()');
		};
		document.body.appendChild(iFrame);
		this._arrIToasters[toasterContent.id] = iFrame;
		return iFrame;
	},

	/**
	 * @private
	 */
	_callOnHide: function(id) {
		var onHide = this._arrOnHideCallback[id];
		if (typeof onHide == "function") {
			onHide();
		}
	}
});

ICTouchAPI.toasterServices=new ICTouchAPI.toasterServices();
