/**
* @class ICTouchAPI.supervisionServices
* @extends Object
* @singleton
 * supervisionServices provides an interface to web applications to manage the supervision.<br />
 * Note: only one supervision group is managed
 * <br />
 * The service publishes the following events (using dojo.publish):<br />
 *	- <b>supervision.groupAdded:</b> notify webapps that the user has been added to a supervision group.<br />
 *	Provide a string indicating the name of the group and an array of supervised users member of this group.<br />
 *	- <b>supervision.groupModified:</b> notify webapps that the group the user is in has been modified.
 *	Provide a string indicating the name of the group, an array of added users, an array of modified users and an array of users deleted from this group.
 *	The array of users deleted only contains the login of these users.<br />
 *	- <b>supervision.groupDeleted:</b> notify webapps that the user has been removed from the group.<br />
 *	- <b>supervision.incomingCallsNumberChanged:</b> notify that the number of incoming calls on supervised users changed. Provide this number.<br />
 *	- <b>supervision.hiddenStateChanged:</b> notify that the hidden state of supervision changed. Provide this state as a boolean.<br />
*/
dojo.provide("ICTouchAPI.supervisionServices");
dojo.declare("ICTouchAPI.supervisionServices", null, {
	
		/* --------------------------------- Public attributes ------------------------------------ */



		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		* @property
		* @type Object
		* @ignore
		*/
	   _arrMembers : [],

		/**
		* @property
		* @type string
		* @ignore
		*/
	   _strGroupName : "",

		/**
		* @property
		* @type number
		* @ignore
		*/
		_intIncomingCallsNb : 0,

		/**
		 * @property
		 * @type number
		 * @ignore
		 */
		_intNbMaxMembers : 15,

		/**
		 * @property
		 * @type boolean
		 * @ignore
		 */
		_boolSupervisionHidden : false,

		/* ------------------------------------ Constructor --------------------------------------- */

		/**
		* The supervision services manage all the supervision feature
		* @ignore
		*/
		constructor: function() {
			// Subscribe to Telephony events
			ICTouchAPI.eventServices.subscribeToEvent(this, "supGroupAdded", this._onSupGroupAdded);
			ICTouchAPI.eventServices.subscribeToEvent(this, "supGroupModified", this._onSupGroupModified);
			ICTouchAPI.eventServices.subscribeToEvent(this, "supGroupDeleted", this._onSupGroupDeleted);
			ICTouchAPI.eventServices.subscribeToEvent(this, "supGroupHidden", this._onSupGroupHidden);
			ICTouchAPI.eventServices.subscribeToEvent(this, "supUpdated", this._onSupUpdated);

			dojo.subscribe("HomepageLoaded", this, function() {
				if(ICTouchAPI.remoteDisplay){
					ICTouchAPI.APIServices.Telephony.getSupervisionCurrentMembers({context:this, callback:this._initGroup});
				}
			});
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * @ignore
		 * Because the eventServices wants an application name we provide one.
		 * @return {String} Return supervisionServices
		 */
		getApplicationName: function(){
			return "supervisionServices";
		},

		/**
		 * Return the maximum number of supervised users
		 * @return {Number} The maximum number of supervised users
		 */
		getMaxMembers : function() {
			return this._intNbMaxMembers;
		},

		/**
		 * Return the supervision hidden state.
		 * @return {Boolean} The supervision hidden state
		 */
		getSupervisionHidden : function() {
			return this._boolSupervisionHidden;
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		 * Return a user according to a login
		 * @param {String} strLogin is the unique login of the user you want to get
		 * @return {Object} The user corresponding to the given login (null if not found)
		 */
		getSupUserByLogin : function(strLogin) {
			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - getSupUserByLogin / Begin");
			for(var j = 0 ; j < this._arrMembers.length ; j++){
				if(this._arrMembers[j].identity.login == strLogin){
					return this._arrMembers[j];
				}
			}

			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - getSupUserByLogin / end");

			return null;
		},

		/**
		 * Return an array of users
		 * @param {number} intFrom the first index of the array
		 * @param {number} intTo the last index of the array
		 * @return {Object} The array of users in the given range
		 */
		getSupUsers : function(intFrom, intTo) {
			return this._arrMembers.slice(intFrom, intTo);
		},

		/**
		 * Run an action (call or pickup) on a user accoring to its ringing state
		 * @param {string} strLogin is the unique login of the user you want to pickup
		 */
		callPickupByLogin : function(strLogin) {
			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - callPickupByLogin / Begin");
			var objMember = null;

			// Check if the strLogin is valid
			for(var i = 0 ; i < this._arrMembers.length ; i++){
				if(this._arrMembers[i].identity.login == strLogin){
					objMember = this._arrMembers[i];
					break;
				}
			}

			// Pickup the call if there is an incoming call on the member, call him otherwise
			if(objMember){
				var supervisedDialableNumber = ICTouchAPI.tools.getContactPhone(objMember.identity, false);
				if(objMember.ringState && objMember.firstIncomingCall){
					ICTouchAPI.debugServices.debug("ICTouchAPI.supervisionServices - callPickupByLogin / Start pickup on login " + strLogin + " with callRef " + objMember.firstIncomingCall.callRef);
					ICTouchAPI.APIServices.Telephony.pickupCall({params:[objMember.firstIncomingCall.callRef, supervisedDialableNumber]});
				} else {
					ICTouchAPI.debugServices.debug("ICTouchAPI.supervisionServices - callPickupByLogin / Start call on login : " + strLogin);
					ICTouchAPI.APIServices.Telephony.startPhoneCall({params:[supervisedDialableNumber, false, false, false]});
				}
			} else {
				ICTouchAPI.debugServices.warning("ICTouchAPI.supervisionServices - callPickupByLogin / Login of the user not found : " + strLogin);
			}

			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - callPickupByLogin / end");
			return null;
		},

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 */
		_setSupervisionHidden : function(boolSupervisionHiddenState) {
			if(this._boolSupervisionHidden !== boolSupervisionHiddenState){
				this._boolSupervisionHidden = boolSupervisionHiddenState;
				dojo.publish("supervision.hiddenStateChanged", [boolSupervisionHiddenState]);
			}
		},
		
		/**
		 * @ignore
		 */
		_onSupGroupAdded : function() {
			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - _onSupGroupAdded / Begin");

			this._setSupervisionHidden(false);

			var objSupGroupAddedArguments = ICTouchAPI.tools.getEventArguments(arguments);
			var groupName = objSupGroupAddedArguments.groupName;
			var arrMembers = objSupGroupAddedArguments.listAddedMembers;

			this._initGroup(arrMembers, groupName);

			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - _onSupGroupAdded / End");
		},

		/**
		 * @ignore
		 */
		_initGroup : function(arrMembers, strGroupName) {
			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - _initGroup / Begin");


			for(var i =0 ; i < arrMembers.length ; i++){
				ICTouchAPI.debugServices.debug("ICTouchAPI.supervisionServices - _initGroup / addedMember "+ i + " : " + dojo.toJson(arrMembers[i]) );
				this._formatNewMember(arrMembers[i]);
			}

			ICTouchAPI.debugServices.debug("ICTouchAPI.supervisionServices - _onSupGroupAdded / groupName : " + strGroupName );

			// store the new members
			this._intIncomingCallsNb = 0;
			this._strGroupName = strGroupName || "";
			this._arrMembers = dojo.clone(arrMembers);

			// Notify webapps that a group is added
			dojo.publish("supervision.groupAdded", [this._strGroupName, arrMembers]);

			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - _initGroup / End");
		},
		
		/**
		 * @ignore
		 */
		_onSupGroupModified : function() {
			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - _onSupGroupModified / Begin");

			var objSupGroupModifiedArguments = ICTouchAPI.tools.getEventArguments(arguments);
			var groupName = objSupGroupModifiedArguments.groupName;
			var arrDeletedMemberLogin = objSupGroupModifiedArguments.listDeletedMembers;
			var arrAddedMember = objSupGroupModifiedArguments.listAddedMembers;
			var arrModifiedMember = objSupGroupModifiedArguments.listModifiedMembers;
			var i, j;

			ICTouchAPI.debugServices.debug("ICTouchAPI.supervisionServices - _onSupGroupModified / groupName : " + groupName );

			// Delete members
			if(arrDeletedMemberLogin){
				for(i = 0 ; i < arrDeletedMemberLogin.length ; i++){
					for(j = 0 ; j < this._arrMembers.length ; j++){
						if(this._arrMembers[j].identity.login == arrDeletedMemberLogin[i]){
							ICTouchAPI.debugServices.debug("ICTouchAPI.supervisionServices - _onSupGroupModified / deletedMember "+ i + " : " + arrDeletedMemberLogin[i] );
							this._arrMembers.splice(j, 1);
							break;
						}
					}
				}
			}

			// Add members
			if(arrAddedMember){
				for(i = 0 ; i < arrAddedMember.length && this._arrMembers.length < this.getMaxMembers(); i++){
					ICTouchAPI.debugServices.debug("ICTouchAPI.supervisionServices - _onSupGroupModified / addedMember "+ i + " : " + dojo.toJson(arrAddedMember[i]) );
					this._formatNewMember(arrAddedMember[i]);
					this._arrMembers.push(arrAddedMember[i]);
				}
			}

			// Modify members
			if(arrModifiedMember){
				for(i = 0 ; i < arrModifiedMember.length ; i++){
					this._formatMemberIdentity(arrModifiedMember[i]);
					for(j = 0 ; j < this._arrMembers.length ; j++){
						if(this._arrMembers[j].identity.login == arrModifiedMember[i].identity.login){
							ICTouchAPI.debugServices.debug("ICTouchAPI.supervisionServices - _onSupGroupModified / modifiedMember "+ i + " : " + dojo.toJson(arrModifiedMember[i]) );
							dojo.mixin(this._arrMembers[j], arrModifiedMember[i]);
							arrModifiedMember[i] = this._arrMembers[j];
							break;
						}
					}
				}
			}

			// Edit groupName if defined
			if(groupName){
				this._strGroupName = groupName;
			}

			// Notify webapps that a group is modified
			dojo.publish("supervision.groupModified", [groupName, arrAddedMember, arrModifiedMember, arrDeletedMemberLogin]);

			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - _onSupGroupModified / End");
		},
		
		/**
		 * @ignore
		 */
		_onSupGroupHidden : function() {
			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - _onSupGroupHidden / Begin");

			this._setSupervisionHidden(true);
			
			this._arrMembers = [];
			dojo.publish("supervision.groupDeleted", []);

			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - _onSupGroupHidden / End");
		},
		
		/**
		 * @ignore
		 */
		_onSupGroupDeleted : function() {
			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - _onSupGroupDeleted / Begin");
			
			this._setSupervisionHidden(false);
			
			this._arrMembers = [];
			dojo.publish("supervision.groupDeleted", []);

			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - _onSupGroupDeleted / End");
		},
		
		/**
		 * @ignore
		 */
		_onSupUpdated : function() {
			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - _onSupUpdated / Begin");

			var objSupUpdatedArguments = ICTouchAPI.tools.getEventArguments(arguments);
			var intNbCalls = objSupUpdatedArguments.nbIncomingCalls;
			var arrSup = objSupUpdatedArguments.listSupervisedUsers;
			ICTouchAPI.debugServices.debug("ICTouchAPI.supervisionServices - _onSupUpdated / intNbCalls "+intNbCalls);
			ICTouchAPI.debugServices.debug("ICTouchAPI.supervisionServices - _onSupUpdated / arrSup " + dojo.toJson(arrSup));
			var member;
			var arrModifiedMembers = [];

			for(var i = 0 ; i < arrSup.length ; i++){
				for(var j = 0 ; j < this._arrMembers.length ; j++){
					if(this._arrMembers[j].identity.login == arrSup[i].login){
						member = this._arrMembers[j];

						// Fill specific user new information
						member.presence = this._getPresenceIconFromTelState(arrSup[i].telState.value.value);
						member.ringState = this._getRingStateFromTelState(arrSup[i].telState.value.value);
						member.firstIncomingCall = arrSup[i].firstIncomingCall.value;
						member.firstIncomingCall.caller = member.firstIncomingCall.identity.value;
						delete member.firstIncomingCall.identity;
						member.firstIncomingCall.caller.officePhone = member.firstIncomingCall.caller.officePhone.value;

						arrModifiedMembers.push(member);
						break;
					}
				}
			}

			// webapps that a group is modified on the same way that in function _onSupGroupModified
			dojo.publish("supervision.groupModified", [null, [], arrModifiedMembers, []]);


			// Notify if the number of incoming calls on supervised users changed
			if(intNbCalls != this._intIncomingCallsNb){
				this._intIncomingCallsNb = intNbCalls;
				dojo.publish("supervision.incomingCallsNumberChanged", [intNbCalls])
			}

			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - _onSupUpdated / End");
		},
		
		/**
		 * @ignore
		 */
		_formatNewMember : function(objMember) {
			this._formatMemberIdentity(objMember);
			this._formatNewMemberPresence(objMember);
		},

		/**
		 * @ignore
		 */
		_formatNewMemberPresence : function(objMember) {
			objMember.presence = this._getPresenceIconFromTelState("UNKNOWN");
			objMember.ringState = this._getRingStateFromTelState("UNKNOWN");
		},

		/**
		 * @ignore
		 */
		_formatMemberIdentity : function(objMember) {
			objMember.identity = objMember.identity.value;
			objMember.identity.officePhone = objMember.identity.officePhone.value;
			objMember.identity.officePhone.type = objMember.identity.officePhone.type.value;
		},

		/**
		 * @ignore
		 */
		_getPresenceIconFromTelState : function(strTelState) {
			var telPresenceIcon;

			switch(strTelState){
				case "FREE":
				case "RINGING":
					telPresenceIcon = "userinfo-availablephone";
					break;
				case "BUSY":
				case "BUSYANDRINGING":
					telPresenceIcon = "userinfo-onthephone"
					break;
				default: // UNKNOWN and OUTOFSERVICE
					telPresenceIcon = "userinfo-appearoffline-offline"
					break;
			}

			return telPresenceIcon;
		},

		/**
		 * @ignore
		 */
		_getRingStateFromTelState : function(strTelState) {
			if(strTelState === "RINGING" || strTelState === "BUSYANDRINGING"){
				return true;
			} else {
				return false;
			}
		},

		/**
		 * @ignore
		 * Send logs of attributes of the service to the logger.
		 * @param {boolean} boolAdvancedLogs Boolean indicating if advanced or basic logs has to be written
		 */
		dump : function(boolAdvancedLogs){
			ICTouchAPI.debugServices.dump("dump function of supervisionServices with " + ((boolAdvancedLogs)?"advanced":"basic") + " logs option (bug advanced mode is not used yet");

			ICTouchAPI.debugServices.dump("Number of incoming call on supervised users : " + this._intIncomingCallsNb);

			ICTouchAPI.debugServices.dump("GroupName : " + this._strGroupName);
			for(var j in this._arrMembers){
				ICTouchAPI.debugServices.dump("Member " + j + " : " + dojo.toJson(this._arrMembers[j]));
			}
		}

	});
// Create supervision services
ICTouchAPI.supervisionServices=new ICTouchAPI.supervisionServices();
