/**
* @class ICTouchAPI.skinServices
* @singleton
* @ignore
* @extends Object
* Manage skins
*/

dojo.require("ICTouchAPI.cssHandler");

dojo.provide("ICTouchAPI.skinServices");

dojo.declare("ICTouchAPI.skinServices",null,
{

	/* --------------------------------- Public attributes ------------------------------------ */

	/**
	 * image used during the change of skin
	 * @property
	 * @type {Object}
	 */
	skinTransitionImage : null,


	/* --------------------------------- Private attributes ----------------------------------- */

	/**
	 * name of the current skin
	 * @property
	 * @private
	 * @type {String}
	 */
	_skinName : "Default",

	/**
	 * name of the current skin
	 * @property
	 * @private
	 * @type {String}
	 */
	_skinBackground : "",

	/**
	 * name of the associated full skin for partial skins
	 * @property
	 * @private
	 * @type {String}
	 */
	_associatedSkinName : "",

	/**
	 * content of the skinName/providedFiles.json for full or default skin and associatedFullSkin/providedFiles.json for partial skins
	 * @property
	 * @private
	 * @type {Object}
	 */
	_skinConfig: null,

	/**
	 * content of the skinName/providedFiles.json when usig a partial skin
	 * @property
	 * @private
	 * @type {Object}
	 */
	_associatedSkinConfig: null,

	/**
	 * array holding css handlers (independant agents)
	 * @property
	 * @private
	 * @type {Array}
	 */
	_arrCssHandlers : {},

	/**
	 * Max depth of skin heritage
	 * @property
	 * @private
	 * @type {int}
	 */
	_maxSkinLevel : 2,

	/**
	 * level associated to skin
	 * @property
	 * @private
	 * @type {Object}
	 */
	_skinLevel : {},




	/* ------------------------------------ Constructor --------------------------------------- */

	/**
	 * This method create the services and initialize it
	 */
	constructor: function (){
		//accessing the skin setting value
		this._accessSkinName();
		this._accessSkinProvidedFiles();
		this._loadSkinOnProvidedFiles();

		//check the skin on end_of_init event
		this._checkSkinOnEndOfInit();
	},


	/* ----------------------------------- Getter / Setter------------------------------------- */

	/**
	 * return the name of the application (used for event subscription ?)
         * @private
	 * @return {string} namespace.classname of the services
	 */
	getApplicationName : function(){
		return "ICTouchAPI.skinServices";
	},


	/**
	 * This method set the background defined by the user or the default background of the current skin
	 * @param {object} objSetting : the setting holding the user private background
	 *
	 */
	setBackground : function(objSetting){
		var bgdSrc;
		//we load it by changing the url of the DOM element with the .withBackground class
		//create the image and load it
		var img = new Image();
		var that=this;
		//if user has defined a private background url, use it
		if(objSetting && objSetting.value && objSetting.value != ""){
			bgdSrc = objSetting.value;
			img.onerror=function(){
				ICTouchAPI.settingServices.setSettingValue("UserBackgroundPicture", "");
			};
		}
		else { //if no background defined, use the default one of the current skin
			bgdSrc = "/library/ICTouchAPI/themes/" + this._skinName + "/images/default.jpg";
			img.onerror=function(){
				dojo.query(".withBackground").style("background-image", "");
			};
		}

		if(this._skinBackground != bgdSrc) {
			this._skinBackground = bgdSrc;
			img.src = bgdSrc;
			img.onload=function(){
				that._applyBackground(img);
			};
		}
	},


	/* ----------------------------------- Public methods ------------------------------------- */

	/**
	 * method used to set a frame :
	 *  - add the required css,
	 *  - add properties and functions to the frame,
	 *  - subscribe the frame to the related cssHandler
	 * @param {HTMLElement} header : header to be skinned
	 * @param {String} strWebappName : name of the webapp related to the header; used to get the tag
	 * @param {String} type : type of the frame
	 * @param {HTMLiFrame} frame : the targeted frame to be set up
	 */
	registerHeader : function(header,strWebappName,type,frame) {
		this._placeCssPosition(header);

		//getting the related headerTag
		var headerTag = (strWebappName.split("."))[1];
		headerTag = (headerTag ? headerTag : (strWebappName.split("."))[0]); //"ICTouchAPI" case only

		//setting the type
		type = (type == "appbar" ? "linkedIcons" : "linkedBasic");

		//creating handler if needed
		if(!this._arrCssHandlers[headerTag])
			this._arrCssHandlers[headerTag] = new ICTouchAPI.cssHandler(headerTag);
		if(this._associatedSkinName!="") {
			this._arrCssHandlers[headerTag].createSkinFragments(this._associatedSkinName);
		}
		this._arrCssHandlers[headerTag].createSkinFragments(this._skinName);

		//adding css
		if(this._associatedSkinName!="") {
			this._placeLink("ICTouchAPI", "linkedBasic", this._associatedSkinName, header);
			this._placeLink("ICTouchAPI", "linkedIcons", this._associatedSkinName, header);
			this._placeLink(headerTag, type, this._associatedSkinName, header);
		}
		//adding css
		this._placeLink("ICTouchAPI", "linkedBasic", this._skinName, header);
		this._placeLink("ICTouchAPI", "linkedIcons", this._skinName, header);
		this._placeLink(headerTag, type, this._skinName, header);

		//adding management tools to the frame
		if(frame){
			frame.cssTag  = headerTag;
			frame.cssType = type;

			this._setUpFrame(frame);
			frame.subToCss.push(dojo.subscribe("ICTouchAPI.linkedBasic",frame,frame.onCssHandlerEvent));
			frame.subToCss.push(dojo.subscribe("ICTouchAPI.linkedIcons",frame,frame.onCssHandlerEvent));
			frame.subToCss.push(dojo.subscribe(headerTag+"."+type,frame,frame.onCssHandlerEvent));
			frame.subToCss.push(dojo.subscribe("RemoveSkinCss",frame,frame.onRemoveSkinCssEvent));
		}
	},

	/**
	 * method used to add a stylesheet link to the header of a frame :
	 * @param {HTMLElement} header : the header in which the stylesheet file will be linked
	 * @param {String} url : the path to the stylesheet file to link to the header
	 * @private
	 */
	addStyleToHeader : function(header, url) {
	    // Check that the stylesheet is not already registered
		var links = header.getElementsByTagName("LINK");
		for (var i in links) {
			var node = links[i];
			if (node.href && node.href.indexOf(url) !== -1) {
				ICTouchAPI.debugServices.debug("ICTouchAPI.skinServices - addStyleToHeader / the stylesheet with url " + url + " is already registered to the header of the frame, exit");
				return;
			}
		}
		var link = dojo.create("link", {
							type  : "text/css",
							rel   : "stylesheet",
							media : "screen",
							href  : url
						});

		ICTouchAPI.debugServices.debug("ICTouchAPI.skinServices - addStyleToHeader / register the stylesheet with url " + url +" to the frame");
		dojo.place(link, header);
	},

	/**
	 * method used to remove a stylesheet link from the header of a frame :
	 * @param {HTMLElement} header : the header in which the stylesheet file will be removed
	 * @param {String} url : the path to the stylesheet file to remove from the header
	 * @private
	 */
	removeStyleToHeader : function(header, url) {
		var links = header.getElementsByTagName("LINK");
		for (var i in links) {
			var node = links[i];
			if (node.href && node.href.indexOf(url) !== -1) {
				ICTouchAPI.debugServices.debug("ICTouchAPI.skinServices - removeStyleToHeader / remove the stylesheet with url " + url +" from the frame");
				node.parentNode.removeChild(node);
				return;
			}
		}
	},

	/**
	 * method used to add a CSS rule to the DOM Header :
	 * @param {HTMLElement} frame : the frame in which the selector will be added
	 * @param {String} selector : the CSS Selector to add in the DOM Header
	 * @param {String} rule : the rule attached to the CSS Selector to add in the DOM Header
	 * @private
	 */
	addCssRuleToFrame : function(frame, selector, rule) {
	    var doc = frame.contentDocument,
	    style	= null,
		rules	= null;

	    // Get the styleSheet
	    style = doc.styleSheets[doc.styleSheets.length -1];

		// Get the rules
	    rules = style.cssRules;

	    // Check that the rule is not already registered
	    for (var i=0; i < rules.length; i++) {
           if (rules[i].selectorText == selector) {
				ICTouchAPI.debugServices.debug("ICTouchAPI.skinServices - addCssRuleToFrame / the css selector " + selector + " is already registered to the header of the frame, exit");
				return;
           }
        }

	    // add the selector & rule
		ICTouchAPI.debugServices.debug("ICTouchAPI.skinServices - addCssRuleToFrame / register the css selector " + selector + ", with the rule " + rule + " to the frame");
	    style.insertRule("." + selector + " { " + rule + "; }", style.cssRules.length);
	},

	/**
	 * method used to remove a CSS rule from the DOM Header
	 * @param {HTMLElement} frame : the frame from which the selector will be removed
	 * @param {String} selector : The CSS Selector to remove from the DOM Header
	 * @private
	 */
	removeCssRuleToFrame : function(frame, selector) {
		var doc = frame.contentDocument,
	    	style = null,
	    	rules = null,
	    	indexToRemove = -1;

	   	// Get the styleSheet
	    style = doc.styleSheets[doc.styleSheets.length -1];

	    // Get the rules
	    rules = style.cssRules;

	    // Get the index of the rule
	    for (var i=0, len=rules.length; i < len; i++) {
           if (rules[i].selectorText == selector) {
               indexToRemove = i;
               break;
           }
        }

        if(indexToRemove > -1) {
        	style.removeRule(indexToRemove);
        }

	},

	/**
	 * This method destroys a given frame listeners
	 * @param {HTMLiFrame} frame : the targeted iFrame
	 */
	unregisterHeader : function(frame){
		if(frame && frame.subToCss)
			dojo.forEach(frame.subToCss,dojo.unsubscribe);


	},

	/**
	 * method used to define a relation between a targeted webapp and an other webapp css
	 * @param {String} webappTarget : the target which requires to load the css of an other webapp
	 * @param {String} webappOwner : name of the webapp which own the required css. Array of string also can be used
	 */
	linkWebappsStyles : function(webappTarget, webappOwner, onlyIcon, onlyCss, duplicateIcon,duplicateCss){
		onlyIcon		= (onlyIcon ? onlyIcon : false);
		onlyCss			= (onlyCss ? onlyCss : false);
		duplicateIcon	= (duplicateIcon ? duplicateIcon : false);
		duplicateCss	= (duplicateCss ? duplicateCss : false);

		var target = webappTarget.split(".")[1];
		if(!this._arrCssHandlers[target]){
			this._arrCssHandlers[target] = new ICTouchAPI.cssHandler(target);
		}
		if(this._associatedSkinName!="") {
			this._arrCssHandlers[target].createSkinFragments(this._associatedSkinName);
		}
		this._arrCssHandlers[target].createSkinFragments(this._skinName);

		if(webappOwner!=null && typeof(webappOwner)!=="object") {
			webappOwner=[webappOwner];
		}
		var ownerCssHandler = [];
		for(var i in webappOwner) {
			var owner  = webappOwner[i].split(".")[1];
			if(!this._arrCssHandlers[owner]){
				this._arrCssHandlers[owner] = new ICTouchAPI.cssHandler(owner);
				if(this._associatedSkinName!="") {
					this._arrCssHandlers[owner].createSkinFragments(this._associatedSkinName);
				}
				this._arrCssHandlers[owner].createSkinFragments(this._skinName);
			}
			ownerCssHandler[i] = this._arrCssHandlers[owner];
		}

		this._arrCssHandlers[target].link(ownerCssHandler,onlyIcon, onlyCss, duplicateIcon,duplicateCss);
		webappTarget=null;
		webappOwner=null;
		onlyIcon=null;
		onlyCss=null;
		duplicateIcon=null;
		duplicateCss=null;
	},

	/**
	 * method called after end_of_init event has been received to change the skin
	 * @param {object} objSetting : returned skin setting
	 */
	skinSettingChanged : function(objSetting){
		var i;
		if(objSetting && objSetting.value != this._skinName){
			if (!ICTouchAPI.remoteDisplay && !webapp.settings.data.getResetInProgress()) {
				this._createSkinTransitionIFrame()._show();
			}

			var oldSkinName = this._skinName;
			var oldAssociatedSkinName = this._associatedSkinName;
			this._skinName = objSetting.value;
			this._associatedSkinName = "";
			this._skinConfig = null;
			this._associatedSkinConfig = null;

			for(i in this._arrCssHandlers){
				this._arrCssHandlers[i].setLock(true,arrSkins);
			}

			if(oldAssociatedSkinName!="") {
				delete this._skinLevel[oldAssociatedSkinName];
			}
			delete this._skinLevel[oldSkinName];
			this._accessSkinProvidedFiles();

			var removeAssociatedSkin = false;
			var removeSkin = false;
			var reloadAssociatedSkin = false;
			var reloadSkin = false;
			//Full skin to partial skin with different full skin
			if(oldAssociatedSkinName=="" && this._associatedSkinName!="" && oldSkinName!=this._associatedSkinName) {
				removeSkin = true;
				reloadAssociatedSkin = true;
				reloadSkin = true;
			}
			//Partial skin to full skin with different full skin
			else if(oldAssociatedSkinName!="" && this._associatedSkinName=="" && oldAssociatedSkinName!=this._skinName) {
				removeSkin = true;
				removeAssociatedSkin = true;
				reloadSkin = true;
			}
			//partial skin to partial skin with different skin
			else if(oldAssociatedSkinName!="" && this._associatedSkinName!="" && oldAssociatedSkinName!=this._associatedSkinName && oldSkinName!=this._associatedSkinName) {
				removeAssociatedSkin = true;
				removeSkin = true;
				reloadAssociatedSkin = true;
				reloadSkin = true;
			}
			//Full skin to full skin or partial skin to partial skin with same full skin
			else if(oldAssociatedSkinName==this._associatedSkinName && oldSkinName!=this._skinName) {
				removeSkin = true;
				reloadSkin = true;
			}
			//Full skin to partial skin with same full skin
			else if(oldAssociatedSkinName=="" && this._associatedSkinName!="" && oldSkinName==this._associatedSkinName) {
				reloadSkin = true;
			}
			//Partial skin to full skin with same full skin
			else if(oldAssociatedSkinName!="" && this._associatedSkinName=="" && oldAssociatedSkinName==this._skinName) {
				removeSkin = true;
			}


			//Remove unused CSS
			if(removeAssociatedSkin===true) {
				dojo.publish("RemoveSkinCss",[oldAssociatedSkinName]);
			}
			if(removeSkin===true) {
				dojo.publish("RemoveSkinCss",[oldSkinName]);
			}

			this._loadSkinOnProvidedFiles(reloadSkin,reloadAssociatedSkin);

			//Create header fragments
			for(var i in this._arrCssHandlers) {
				if(this._associatedSkinName!="") {
					this._arrCssHandlers[i].createSkinFragments(this._associatedSkinName);
				}
				this._arrCssHandlers[i].createSkinFragments(this._skinName);
			}

			//Reload webapp style
			for(i in webapp) {
				if(webapp[i]._strApplicationName!=null) {
					this.loadWebappStyle(i,reloadSkin,reloadAssociatedSkin);
				}
			}


			var arrSkins = [];
			if(reloadAssociatedSkin===true) {
				arrSkins.push(this._associatedSkinName);
			}
			if(reloadSkin===true) {
				arrSkins.push(this._skinName);
			}

			for(i in this._arrCssHandlers){
				this._arrCssHandlers[i].setLock(false,arrSkins);
			}
			this.loadBackground();

		}
	},

	/**
	 * This method is called during the initialization to load the background picture
	 * defined by the user. If user has not private background the skin background will be loaded
	 */
	loadBackground : function(){
            // Default settings for simulation mode
            if(!generalConfig.simulation) {
				ICTouchAPI.settingServices.getSetting("UserBackgroundPicture", this, this.setBackground);
            }
	},

	/**
	 * This method allows to change the type and the tag of an iFrame, update its css and listeners
	 * @param {HTMLiFrame} frame : the targeted iFrame
	 * @param {string} newTag : the new tag (ex: communication, email,...)
	 * @param {string} type : the new type (webapp / appbar)
	 */
	changeTag : function(frame,newTag,type){
		type = (type == "appbar" ? "linkedIcons" : "linkedBasic");

		if(type != frame.cssType){
			 if(type == "linkedIcons"){
				if(this._associatedSkinName!="") {
					this._placeLink("ICTouchAPI", type, this._associatedSkinName, frame.contentDocument.head);
				}
				this._placeLink("ICTouchAPI", type, this._skinName, frame.contentDocument.head);
				frame.subToCss.push(dojo.subscribe("ICTouchAPI."+type,frame,frame.onCssHandlerEvent));
				if(frame.cssTag == newTag){
					if(this._associatedSkinName!="") {
						this._placeLink(newTag, type, this._associatedSkinName, frame.contentDocument.head);
					}
					this._placeLink(newTag, type, this._skinName, frame.contentDocument.head);
					frame.subToCss.push(dojo.subscribe(newTag+"."+type,frame,frame.onCssHandlerEvent));
				}
			 }else{
				dojo.query("link[href$='icons.css']", frame.contentDocument.head).forEach(dojo.destroy);
				dojo.forEach(frame.subToCss,function(sub){
					if(sub[0] == "ICTouchAPI.linkedIcons" || sub[0] == frame.cssTag+".linkedIcons")
						dojo.unsubscribe(sub);
					}
				);
			 }
			 frame.cssType = type;
		}

		if(frame.cssTag != newTag){
			dojo.query("link[csstag='"+frame.cssTag+"']", frame.contentDocument.head).forEach(dojo.destroy);
			if(this._associatedSkinName!="") {
				this._placeLink(newTag, type, this._associatedSkinName, frame.contentDocument.head);
			}
			this._placeLink(newTag, type, this._skinName, frame.contentDocument.head);
			dojo.forEach(frame.subToCss,function(sub){
				if(sub[0] != "ICTouchAPI.linkedBasic" && sub[0] != "ICTouchAPI.linkedIcons")
					dojo.unsubscribe(sub);
			});
			frame.subToCss.push(dojo.subscribe(newTag+"."+type,frame,frame.onCssHandlerEvent));
			frame.cssTag = newTag;
			frame.cssType = type;
		}
	},


/* --------------------------------- Private Methods -------------------------------------- */

	/**
	 * This method check the existence of the providedFiles.json file related to the
	 * given skin name
	 * @param {string} strSkinName : the name of the skin to be retrieved
	 * @private
	 * @return {null|array} return null if the file does not exist else return an array holding the data related to the skin configuration
	 */
	getSkinConfig : function(strSkinName){
		//building the file path to be retrieved
		var skinConfigPath = dojo.moduleUrl("ICTouchAPI").path+"Config/themes/"+strSkinName+"/providedFiles.json";
		//accessing to the file content
		var skinConfigXHR = dojo.xhrGet({
			url: skinConfigPath,
			handleAs : "json",
			sync : true
		});
		//returning the content : null or an associative array
		if(skinConfigXHR.results[0]!=null) {
			skinConfigXHR.results[0].webapp = {};
			return skinConfigXHR.results[0];
		}
		else {
			return null;
		}
	},

	/**
	 * This method check the existence of a webapp providedFiles.json file related to the
	 * given skin name
	 * @param {string} strSkinName : the name of the skin to be retrieved
	 * @param {string} strWebapp : the name of the webapp to be retrieved
	 * @param {string} strNamespace : the strNamespace of the webapp to be retrieved
	 * @private
	 * @return {null|array} return null if the file does not exist else return an array holding the data related to the webapp skin configuration
	 */
	getWebappSkinConfig : function(strSkinName, strWebapp, strNamespace){
		//building the file path to be retrieved
		var skinConfigPath = dojo.moduleUrl(strNamespace ? strNamespace : "webapp").path+strWebapp+"/config/themes/"+ strSkinName +"/providedFiles.json";

		//accessing to the file content
		var skinConfigXHR = dojo.xhrGet({
			url: skinConfigPath,
			handleAs : "json",
			sync : true,
			failOk: true
		});
		//returning the content : null or an associative array
		return (skinConfigXHR.results[0] ? skinConfigXHR.results[0]:null);
	},

	/**
	 * This method return a dom fragment holding css matching the given tag and type
	 * @param {string} headerTag : the tag identifying the css
	 * @param {string} typeCss : the type - linkedBasic or linkedIcons
	 * @param {string} skinName : name of the skin
	 * @private
	 * @return {HTMLElement} return a dom fragment
	 */
	getCss : function(headerTag,typeCss,skinName){
		if(this._arrCssHandlers[headerTag])
			return this._arrCssHandlers[headerTag].getCss(typeCss,skinName);
		else
			return null;
	},


	/**
	 * This method load a webapp style
	 * @private
	 */
	loadWebappStyle : function(strWebappName,loadSkin,loadAssociatedSkin) {
		//Load providedFiles.json
		var associatedSkinConfig = null;
		var skinConfig = this.getWebappSkinConfig(this._skinName, strWebappName);
		if(this._associatedSkinName!="") {
			associatedSkinConfig = this.getWebappSkinConfig(this._associatedSkinName, strWebappName);
		}
		//If there is no providedFiles.json for this skin, load it from defaut skin
		if(skinConfig==null && associatedSkinConfig==null) {
			skinConfig = this.getWebappSkinConfig("Default", strWebappName);
		}

		//Create handler if not exist
		if(!this._arrCssHandlers[strWebappName]) {
			//Create header fragments
			this._arrCssHandlers[strWebappName] = new ICTouchAPI.cssHandler(strWebappName);
			if(this._associatedSkinName!="") {
				this._arrCssHandlers[strWebappName].createSkinFragments(this._associatedSkinName);
			}
			this._arrCssHandlers[strWebappName].createSkinFragments(this._skinName);
		}

		//Load CSS
		if(associatedSkinConfig!=null && loadAssociatedSkin!==false) {
			dojo.mixin(this._associatedSkinConfig["webapp"],associatedSkinConfig["webapp"]);
			this._loadWebappCssOnProvidedFiles(associatedSkinConfig, strWebappName);
		}
		if(skinConfig!=null && loadSkin!==false) {
			dojo.mixin(this._skinConfig["webapp"],skinConfig["webapp"]);
			this._loadWebappCssOnProvidedFiles(skinConfig, strWebappName);
		}
	},

	/**
	 * @private
	 * Send logs of attributes of the service to the logger.
	 * @param {boolean} boolAdvancedLogs Boolean indicating if advanced or basic logs has to be written
	 */
	dump : function(boolAdvancedLogs){
		ICTouchAPI.debugServices.dump("dump function of skinServices with " + ((boolAdvancedLogs)?"advanced":"basic") + " logs option");

		ICTouchAPI.debugServices.dump("Current skin name: " + this._skinName);
		if (boolAdvancedLogs) {
			ICTouchAPI.debugServices.dump("Associated skin name: " + this._associatedSkinName);
		}
	},

	/**
	 * Method called to preload skin transition iFrame
         * @private
	 */
	_preloadTransitionFrame : function(){
		//preloading the image for the transition iFrame
		this.skinTransitionImage = new Image();
		this.skinTransitionImage.src = "/library/ICTouchAPI/themes/skintransition.jpg";
		//create the iFrame (hidded)
		this._createSkinTransitionIFrame(this.skinTransitionImage.src);
	},

	/**
	 * Method called to create and return the iFrame holding the skin transition image
	 * @param {string} strImgPath : the skin transition image path
         * @private
	 * @return {iFrame} a reference on the expected iFrame (created or accessed and returned)
	 *
	 */
	 _createSkinTransitionIFrame : function(strImgPath){
		var myFrame = dojo.byId("skinTransition");
		if(!myFrame){
			var iFrame = dojo.create("iframe",{'id':'skinTransition','src':'iframe.html'},dojo.body());
			dojo.style(iFrame,{'height':'480px','width':'800px','top':'0px','left':'-1000px','zIndex':'19000'});

			var loaded = function () {
				var body = iFrame.contentDocument.body;
				var head = dojo.query("head", iFrame.contentDocument)[0];

				// Add CSS for Spinner
				dojo.create("link", {rel: "stylesheet", type: "text/css", href: "http://127.0.0.1/library/ICTouchAPI/UIElements/Spinner/themes/Default/css/base.css"}, head);
				dojo.create("img", {src: strImgPath, style: "pointer-events: none"}, body);
				this.spinner = new UIElements.Spinner.SpinnerControl({boolForeground: true}, body);

				iFrame.removeEventListener("load", loaded, false);
			};
			iFrame.addEventListener("load", loaded, false);

			iFrame._show = function(){
				clearTimeout(this.hideTimer);
				dojo.style(this,'left','0px');
				this.hideTimer = setTimeout(function(){
					iFrame._hide();
				}, 30000);
			};

			iFrame._hide = function(){
				clearTimeout(this.hideTimer);
				dojo.style(this,'left','-1000px');
			};
			//linking the iFrame to our document

			return iFrame;
		}else{
			//returning the existing iFrame
			return myFrame;
		}
	},

	/**
	 * This method get the skin setting value (SkinName)
         * @private
	 */
	_accessSkinName : function(){
            // Default settings for simulation mode
            if(!generalConfig.simulation) {
                //accessing the skin setting value
		ICTouchAPI.settingServices.getSetting("SkinsListChoice", this, function(objSetting){
			this._skinName = (objSetting ? objSetting.value : "Default");
		},null,true);
            }
	},

	/**
	 * This method get providedFiles content within dojo object
         * @private
	 */
	_accessSkinProvidedFiles : function(){
		this._skinConfig = this.getSkinConfig(this._skinName);
		if(!this._skinConfig){
			this._skinName = "Default";
			this._skinConfig = this.getSkinConfig(this._skinName);
		}
		if(this._skinConfig.type == "Partial"){
			//the skin which will be over-written by the partial skin
			this._associatedSkinName = this._skinConfig.associatedFullSkin;
			this._associatedSkinConfig = this.getSkinConfig(this._skinConfig.associatedFullSkin);
		}

		if(this._associatedSkinName=="") {
			this._skinLevel[this._skinName] = 0;
		}
		else {
			this._skinLevel[this._associatedSkinName] = 0;
			this._skinLevel[this._skinName] = 1;
		}
	},


	/**
	 * This method loads the skins from the content of the providedFiles (handle partial skin overload)
	 * @private
	 */
	_loadSkinOnProvidedFiles : function(loadSkin, loadAssociatedSkin){
		if(this._associatedSkinName && loadAssociatedSkin!==false){
			this._loadCssOnProvidedFiles(this._associatedSkinConfig);
		}
		if(loadSkin!==false) {
			this._loadCssOnProvidedFiles(this._skinConfig);
		}
	},

	/**
	 * According to the providedFiles data, this method :
	 * - create the cssHandlers
	 * - load cs files into handlers
	 * @private
	 */
	_loadCssOnProvidedFiles : function(skinConfig){
		var skinName = skinConfig.name;
		//framework
		if(!this._arrCssHandlers["ICTouchAPI"])
			this._arrCssHandlers["ICTouchAPI"] = new ICTouchAPI.cssHandler("ICTouchAPI");
		this._arrCssHandlers["ICTouchAPI"].createSkinFragments(skinName);
		if(skinConfig["ICTouchAPI"]) {
			this._arrCssHandlers["ICTouchAPI"].loadCssOnArrayOfFile("ICTouchAPI", "", skinName, skinConfig["ICTouchAPI"]);
		}

		if(skinConfig["UIElements"]){
			if(!generalConfig.cssDeveloper){
				if(skinConfig["UIElements"]["minifiedCss"])
					this._arrCssHandlers["ICTouchAPI"].loadCssOnArrayOfFile("UIElements", "minifiedCss", skinName, skinConfig["UIElements"]["minifiedCss"]);
			}else{
				for(var key in skinConfig["UIElements"]){
					if(key != "minifiedCss")
						this._arrCssHandlers["ICTouchAPI"].loadCssOnArrayOfFile("UIElements", key, skinName, skinConfig["UIElements"][key]);
				}
			}
		}

		//webapps
		for(var key in skinConfig["webapp"]){
			if(!this._arrCssHandlers[key])
				this._arrCssHandlers[key] = new ICTouchAPI.cssHandler(key);
			//this._arrCssHandlers[key].createSkinFragments(skinName);
			this._arrCssHandlers[key].loadCssOnArrayOfFile("webapp", key, skinName, skinConfig["webapp"][key]);
		}
	},

	/**
	 * According to the webapp providedFiles data, this method :
	 * - create the cssHandlers
	 * - load cs files into handlers
         * @private
	 */
	_loadWebappCssOnProvidedFiles : function(skinConfig, strWebappName){
		var skinName = skinConfig.name;
		this._arrCssHandlers[strWebappName].loadCssOnArrayOfFile("webapp", strWebappName, skinName, skinConfig["webapp"][strWebappName]);
	},



		/**
	 * Add properties and function to the given iFrame to allow it to manage its links
	 * @param {HTMLiFrame} frame the target iFrame to be set up
         * @private
	 */
	_setUpFrame : function(frame){
		if(!frame.subToCss)
			frame.subToCss = [];

		if(!frame.onCssHandlerEvent) {
			frame.onCssHandlerEvent = function(objEvent){
				try{
					var contentDocument = frame.contentDocument;
					if(frame.contentDocument==null) {
						contentDocument = frame;
					}
					var cssType = (objEvent.type==="linkedIcons"||objEvent.type==="icons") ? "icons" : "basic";

					dojo.query("link[csstag='"+objEvent.owner.getTag()+"'][skin='"+objEvent.skinName+"'][cssType='"+cssType+"']", contentDocument.head).forEach(dojo.destroy);
					ICTouchAPI.skinServices._placeLink(objEvent.owner.getTag(), objEvent.type, objEvent.skinName, contentDocument.head);
				}catch(e){
					console.warn("An event handler remeains on a destroyed frame : "+frame.id+" :: error : "+e);
				}
			}
		}
		if(!frame.onRemoveSkinCssEvent) {
			frame.onRemoveSkinCssEvent = function(skinName){
				try{
					var contentDocument = frame.contentDocument;
					if(frame.contentDocument==null) {
						contentDocument = frame;
					}
					dojo.query("link[skin='"+skinName+"']", contentDocument.head).forEach(dojo.destroy);
				}catch(e){
					console.warn("An event handler remeains on a destroyed frame : "+frame.id+" :: error : "+e);
				}
			}
		}
	},

	/**
	 * This method checks the skin on end_of_init event
         * @private
	 */
	_checkSkinOnEndOfInit : function(){
            //subscribing to the skin seeting once the initialization has been done
            var handler = dojo.subscribe('load step 6', this, function() {				
                // Preload transitionFrame on step 6 because we need the spinner in frame
                this._preloadTransitionFrame();

                // Default settings for simulation mode
                if(!generalConfig.simulation) {
                    //subscribing to the setting
                    ICTouchAPI.settingServices.subscribeToSetting(this, "SkinsListChoice", this.skinSettingChanged);

                    //checking the current skin value
                    ICTouchAPI.settingServices.getSetting("SkinsListChoice",this, function(objSetting){
                        //changing the skin without displaying the transition iFrame
                        if(objSetting && objSetting.value != this._skinName){
                                this.skinSettingChanged(objSetting);
                        }
                        //BACKGROUND
                        //listening the setting to be able to change the background afterwards
                        ICTouchAPI.settingServices.subscribeToSetting(this, "UserBackgroundPicture", this.setBackground);
                        //load the background : on end_of_init the background may not been checked if the skin has already been loaded
                        ICTouchAPI.settingServices.getSetting("UserBackgroundPicture", this, this.setBackground);
                    });
                }

                //unsubscribing to the event
                dojo.unsubscribe(handler);
            });
	},


	/**
	 * This method update the background with the given image object and computes the dimensions :
	 *  - if the image is too big, resize it
	 *  - if the image size is lower than the one of the display, center it
	 * @param {image} img the image object to use as background image
         * @private
	 */
	_applyBackground : function(img){
		//we load it by changing the url of the DOM element with the .withBackground class
		//create the image and load it
		if(img.width > 800 || img.height > 480){
			//compute the new dimensions matching the screen
			var ratedDimension = ICTouchAPI.tools.getRatedDimensions(img.width, img.height);
			//if the widht is the base of rating for the height
			if(ratedDimension.ratioW == 1){
				//set the width has 100% of the new screen, height will be resized proportionatly
				dojo.query(".withBackground").style("background-size", "100%");
			}else{
				//else the height is the base of the rate, height will have to fit 100% of the
				//screen height, then the width can be easily deducted
				//screen width : 800, picture width : 600, expected width : 400
				//600/800 = 3/4 : the current picture use 75% of the screen
				//resize from 600 down to 400 : 400/600 = 2/3 : 66,66%
				//global resizing rate become : 3/4*2/3 = 1/2 :=> 800->400
				dojo.query(".withBackground").style("background-size", (img.width/800)*ratedDimension.ratioW*100+"%");
			}
		}else{
			dojo.query(".withBackground").style("background-size", (img.width/800)*100+"%");
		}
		dojo.query(".withBackground").style("background-position", "center");
		dojo.query(".withBackground").style("background-image", "url('" + img.src + "')");
	},

	/**
	* place tag to help place skin link
        * @private
	*/
	_placeCssPosition : function(header) {
		//Place tag to help place skin link
		var helpTag;
		var i;
		for(i=0; i<ICTouchAPI.skinServices._maxSkinLevel; i++) {
			helpTag = dojo.create("cssposition",{
				"type":"ICTouchAPI",
				"level":i
			});
			dojo.place(helpTag,header);
		}
		for(i=0; i<ICTouchAPI.skinServices._maxSkinLevel; i++) {
			helpTag = dojo.create("cssposition",{
				"type":"webapp",
				"level":i
			});
			dojo.place(helpTag,header);
		}
	},

	/**
	 * place link to head
         * @private
	 */
	_placeLink : function(headerTag,type,skinName,header) {
		var position;
		var level = this._skinLevel[skinName];
		if(headerTag==="ICTouchAPI") {
			position = dojo.query("cssposition[type=ICTouchAPI][level="+level+"]",header)[0];
		}
		else {
			position = dojo.query("cssposition[type=webapp][level="+level+"]",header)[0];
		}
		var cssHeader = this._arrCssHandlers[headerTag].getCss(type,skinName);
		if(cssHeader!=null) {
			dojo.place(cssHeader,position,"before");
		}
	},

	/**
	 * Returns the path of css files and images based on the namespace
	 * @param {webappName} webappName application name
         * @private
	 */
	getCssPath : function(webappName) {
		var arrNamespace	= webappName.split('.');
		var strPath			= '/' + arrNamespace[0] + '/' + arrNamespace[1];

		if(arrNamespace[0] == "awap" && webapp.myICWall) {
			strPath	= webapp.myICWall.data.FORWARD_PROXY_URL + webapp.myICWall.data.LABS_URL + webapp.myICWall.data.LABS_REPOSITORY + "/" + arrNamespace[1]
		}

		return strPath;
	}

});
ICTouchAPI.skinServices = new ICTouchAPI.skinServices;
