/**
* @class ICTouchAPI.settingServices
* @singleton
* @extends Object
* Manage the subscription to the Phone Settings
* If a subscribed setting changes, the new setting will be broadcasted to the subscriber (webapp)
* <pre><code>
* Possible setting types:
* - "TYPE_ENABLER"
* - "TYPE_BOOLEAN"
* - "TYPE_CHOICE"
* - "TYPE_TEXT"
* - "TYPE_NUMERIC"
* - "TYPE_FLOAT"
* - "TYPE_LIST"
* - "TYPE_IPADDRESS"
* - "TYPE_IPDOMAIN"
* - "TYPE_IPPORT"
* - "TYPE_PASSWORD"
* - "TYPE_DATE"
* - "TYPE_DATETIME"
* - "TYPE_TIME"
* - "TYPE_PHONE"
* - "TYPE_WEBAPP"
* </code></pre>
*/
dojo.provide("ICTouchAPI.settingServices");
dojo.declare("ICTouchAPI.settingServices",null,
{
	/* --------------------------------- Public attributes ------------------------------------ */

	// Define setting types (pseudo-constants)
	TYPE_ENABLER    : "ENABLER[WEBAPP]",
	TYPE_BOOLEAN	: "BOOLEAN",
	TYPE_CHOICE		: "CHOICE",
	TYPE_TEXT		: "TEXT",
	TYPE_NUMERIC	: "NUMERIC",
	TYPE_FLOAT		: "FLOAT",
	TYPE_LIST		: "LIST",
	TYPE_IPADDRESS	: "IP_ADDRESS",
	TYPE_IPDOMAIN	: "IP_DOMAIN",
	TYPE_IPPORT		: "IP_PORT",
	TYPE_PASSWORD	: "PASSWORD",
	TYPE_DATE		: "DATE",
	TYPE_DATETIME	: "DATETIME",
	TYPE_TIME		: "TIME",
	TYPE_PHONE		: "PHONE_NUMBER",
	TYPE_WEBAPP		: "CUSTOM[WEBAPP]",

	MODE_USER			: "user",
	MODE_ADMIN			: "admin",
	MODE_DEFAULTADMIN	: "defaultAdmin",
	strMode				: "user",

	boolCurrentSettingAdmin : false,

	/* --------------------------------- Private attributes ----------------------------------- */

	/**
	 * @ignore
	 */
	_arrSettingSequence 	:[],

	/**
	 * @ignore
	 */
	_arrSequenceValue	 	:[],

	/**
	 * @ignore
	 */
	_arrReturnedSettings	:[],

	/**
	 * @ignore
	 */
	_arrReturnedSetting	:[],

	/**
	 * @ignore
	 * Actions management
	 */
	_arrActions	: {},

	/**
	 * @ignore
	 * Cache for settings
	 */
	_arrCachedSettings: {},

	/**
	 * @ignore
	 * Cache list of settings identified by mainCat, category, userMode
	 * This gets filled by getSettings
	 */
	_arrCachedCategories: [],

	/**
	 * @ignore
	 * Handles returned from calls to dojo.subscribe() by WebApps are stored here (used for unsubscriptions)
	 */
	_arrHandles:{},

	/* ------------------------------------ Constructor --------------------------------------- */

        /**
         * @ignore
         */
	constructor : function() {
		// The settingServices subscribes to the "ChangedSettingEvent" event through eventServices
		ICTouchAPI.eventServices.subscribeToEvent(this, "ChangedSettingEvent", this._onSettingChanged);
		ICTouchAPI.eventServices.subscribeToEvent(this, "presentationLoaded", this._clearCachedCategories);
	},

	/* ----------------------------------- Getter / Setter------------------------------------- */


	/**
         * Get the name of the application service
         * @return {String} the name of the Service
         * @ignore
         */
	getApplicationName : function() {
		return "settingServices";
	},


	/* ----------------------------------- Public methods ------------------------------------- */

	/**
	 * Method used by webApplications to subscribe to a setting
	 * @param {Object} objWebApp webApp which wants to subscribe to a setting
	 * @param {String} strSettingName The name (Id) of the setting to subscribe
	 * @param {Function} callback function called when the setting occurs
	 */
	subscribeToSetting : function(objWebApp, strSettingName, callback) {
		var webAppName = objWebApp.getApplicationName();
		if (this._arrHandles[webAppName] == undefined) {
			this._arrHandles[webAppName] = {};
		}
		this._arrHandles[webAppName][strSettingName] = dojo.subscribe("settingServices:" + strSettingName, objWebApp, callback);
		objWebApp=null;
		strSettingName=null;
		callback=null;
	},

	/**
	 * Method used by webApplications to unsubscribe to a setting
	 * @param {Object} objWebApp webApp which wants to unsubscribe to setting
	 * @param {String} strSettingName The name (Id) of the setting to unsubscribe
	 */
	unsubscribeToSetting : function(objWebApp, strSettingName) {
		var webAppName = objWebApp.getApplicationName();
		if (this._arrHandles[webAppName][strSettingName] != undefined) {
			dojo.unsubscribe(this._arrHandles[webAppName][strSettingName]);
			delete this._arrHandles[webAppName][strSettingName];
		}
	},

	/**
	 * Method used by webApplications to get all settings from a specific category of a main category
	 * @param {String} mainCat name of the main category
	 * @param {String} category name of the category
	 * @param {String} userMode user mode
	 * @param {Object} context context of the calling webApp
	 * @param {Function} callback function to be called as a callback in the webApp
	 * @param {Function} errorCallback function to be called as a error callback in the webApp
	 * @param {Object} callbackParams object containing parameters to be used by the callback
	 */
	getSettings : function(mainCat, category, userMode, context, callback, errorCallback, callbackParams) {
		ICTouchAPI.log("getSettings: " + mainCat + ", " + category + ", " + userMode, "SETTINGS");
		var categoryListId = mainCat+"_"+category+"_"+userMode;
		var params = {
				context: context,
				callback:callback,
				errorCallback:errorCallback,
				callbackParams:callbackParams,
				categoryListId: categoryListId,
				alreadyInCache: false
		};
		if (this._arrCachedCategories[categoryListId]) {
				// Settings are in cache, do not fetch them from SettingsManager
				params.alreadyInCache = true;
				var service = this;
				setTimeout(function(){
						// Simulate asynchronous call
						service._gotSettings(service._arrCachedCategories[categoryListId], params);
				}, 50);
		}
		else {
				// Get settings from business
				ICTouchAPI.APIServices.SettingsManager.getSettings({
						params:[mainCat, category, userMode],
						context:this,
						callback:this._gotSettings,
						errorCallback:this._notGotSettings,
						callbackParams: params
				});
		}
	},

	/**
	 * Method used by webApplications to get a setting of type sequence by its id
	 * @param {String} id id (name) of the setting
	 * @param {Object} context context of the calling webApp
	 * @param {Function} callback function to be called as a callback in the webApp
	 * @param {Object} callbackParams object containing parameters to be used by the callback
	 */
	getSettingSequence : function (id, context, callback, callbackParams) {
		ICTouchAPI.log("getSetting: " + id, "SETTINGS");
		if (this._arrCachedSettings[id]==undefined) {
			ICTouchAPI.APIServices.SettingsManager.getSetting({
				params:[id],
				context:this,
				callback:this._gotSettingSequence,
				callbackParams:{
					context: context,
					callback: callback,
					callbackParams: callbackParams
				}
			});
		}
		else {
			if (callback != undefined) {
				callback.apply(context, [this._arrCachedSettings[id], callbackParams]);
			}
		}
	},

	/**
	 * Add an action managment, one action will lauch one function
	 * @param {String} actionName action Name defined in the action field of the setting
	 * @param {Object} method method which will be called for this action
	 */
	addActionMethod : function(actionName, method){
		this._arrActions[actionName] = method;
	},

	/**
	 * Get the corresponding method for an action
	 * @param {String} actionName action Name used
	 * @return {Object} the corresponding function
	 */
	getMethodForAction : function(actionName){
		return this._arrActions[actionName];
	},

	/**
	 * Will call the right function corresponding to the action with the good parameters (in case of core method)
	 * @param {String} paramCallback the parameter given by the callback by UI
	 * @param {Object} paramAction parameters given manually
	 */
	callMethodForAction : function(paramCallback, paramAction){
		var func = this.getMethodForAction(paramAction.action)
		func({
			params:[paramCallback]
		});
	},


	/**
	 * Method used by webApplications to get a setting by its id
	 * @param {String} id (name) of the setting
	 * @param {Object} context of the calling webApp
	 * @param {Function} callback function to be called as a callback in the webApp
	 * @param {Object} callbackParams object containing parameters to be used by the callback
	 * @param {boolean} synchro used to define a synchronized xhr request
	 */
	getSetting : function (id, context, callback, callbackParams, synchro) {
		ICTouchAPI.log("getSetting: " + id, "SETTINGS");
		if (id.split(".")[0] == "awap" && webapp && webapp.myICWall) {
			webapp.myICWall.getSetting(id, context, callback, callbackParams, synchro);
		}
		else {
			var namespace = (context && context.declaredClass) ? context.declaredClass.split(".") : null;
			if (namespace && namespace[0] == "awap" && webapp && webapp.myICWall) {
				webapp.myICWall.getSetting(id, context, callback, callbackParams, synchro);
			}
			else {
		if (this._arrCachedSettings[id]==undefined || this._arrSettingSequence[id]) {
			ICTouchAPI.APIServices.SettingsManager.getSetting({
				params:[id],
				context:this,
				callback:this._gotSetting,
				callbackParams:{
					context: context,
					callback: callback,
					callbackParams: callbackParams
				},
				sync : (synchro ? synchro : false)
			});
		}
		else {
			if (callback != undefined) {
				callback.apply(context, [this._arrCachedSettings[id], callbackParams]);
			}
		}
			}
		}
		id=null;
		context=null;
		callback=null;
		callbackParams=null;
		synchro=null;
	},

	/**
	 * Method used by webApplications to get the value of a setting by its id
	 * @param {String} id (name) of the setting
	 * @param {Object} context of the calling webApp
	 * @param {Function} callback function to be called as a callback in the webApp
	 * @param {Object} callbackParams object containing parameters to be used by the callback
	 * @param {boolean} synchro used to define a synchronized xhr request
	 */
	getSettingValue : function (id, context, callback, callbackParams, synchro) {
		ICTouchAPI.log("getSettingValue: " + id, "SETTINGS");
		if (id.split(".")[0] == "awap" && webapp && webapp.myICWall) {
			webapp.myICWall.getSettingValue(id, context, callback, callbackParams, synchro);
		}
		else {
			var namespace = (context && context.declaredClass) ? context.declaredClass.split(".") : null;
			if (namespace && namespace[0] == "awap" && webapp && webapp.myICWall) {
				webapp.myICWall.getSettingValue(id, context, callback, callbackParams, synchro);
			}
			else {
		if (this._arrCachedSettings[id]==undefined) {
			ICTouchAPI.APIServices.SettingsManager.getSetting({
				params:[id],
				context:this,
				callback:this._gotSettingValue,
				callbackParams:{
					context: context,
					callback: callback,
					callbackParams: callbackParams
				},
				sync : (synchro ? synchro : false)
			});
		}
		else {
			if (callback != undefined) {
				callback.apply(context, [this._arrCachedSettings[id].jsValue, callbackParams]);
			}
		}
			}
		}
		id=null;
		context=null;
		callback=null;
		callbackParams=null;
		synchro=null;
	},

	/**
	 * Method used to get a list of settings from their ids
	 * The settings are obtained recursively
	 * @param {Array} ids list of settings ids
	 * @param {Object} context context of the calling webApp
	 * @param {Function} callback function to be called as a callback in the webApp
	 * @param {Object} callbackParams object containing parameters to be used by the callback
	 */
	getSettingsFromIds : function (ids, context, callback, callbackParams) {
		var objContext = {
			index: 0,
			ids: ids,
			results: [],
			context: context,
			callback: callback,
			callbackParams: callbackParams
		};
		this._getNextSettingFromId(objContext);
	},

	/**
	 * Method used by webApplications to set a new value to a setting
	 * @param {String} id (name) of the setting
	 * @param {String/Boolean/Number} jsValue new value which must be set in the setting
	 * @param {Object} context of the calling webApp
	 * @param {Function} callback function to be called as a callback in the webApp
	 * @param {Object} callbackParams object containing parameters to be used by the callback
	 */
	setSettingValue : function (id, jsValue, context, callback, callbackParams) {
		ICTouchAPI.log("setSettingValue: " + id, "SETTINGS");
		ICTouchAPI.logObject(id, "SETTINGS");
		if (id.split(".")[0] == "awap" && webapp && webapp.myICWall) {
			webapp.myICWall.setSettingValue(id, jsValue, context, callback, callbackParams);
		}
		else {
			var namespace = (context && context.declaredClass) ? context.declaredClass.split(".") : null;
			if (namespace && namespace[0] == "awap" && webapp && webapp.myICWall) {
				webapp.myICWall.setSettingValue(id, jsValue, context, callback, callbackParams);
			}
			else {
		if (this._arrCachedSettings[id] != undefined ) {
			if(this._arrCachedSettings[id].admin_rights == 2 && this._arrCachedSettings[id].user_rights != 2 && this.strMode != this.MODE_ADMIN){
				this._showAdminKeyboardForSetting({
					id : id,
					jsValue : jsValue,
					context : context,
					callback : callback,
					callbackParams : callbackParams
				});
			}
			if (this._isNewValueValidForSetting(this._arrCachedSettings[id], jsValue)) {
				// The setting must be cloned, so that the cache is NOT updated until we receive a ChangedSettingEvent.
				// In this way the setting in cache will not be modified if the validation fails on business side.
				var clonedSetting = dojo.clone(this._arrCachedSettings[id]);
				//SEQUENCE
				//if the setting is in the depencies array
				var id_sequence = this._arrSettingSequence[id];
				if(id_sequence){
					//we get the string value to send
					var newVal = this._arrSequenceValue[id_sequence];
					//we create a regexp to find the setting to be set up in the string which will be send to setting manager
					var forUpdate = new RegExp(id+"(=[^(\|;\|)]+){0,1}", "g");
					//we will replace the previous setting value by the new written one
					var strReplace = id+"="+jsValue;
					//we update the string
					newVal = ""+newVal; //without this line str.replace seems to fail for newVal
					newVal = newVal.replace(forUpdate,id+"="+jsValue);
					//The complete sequence value
					jsValue = newVal;
					//The id of the sequence
					id = id_sequence;
					//we save the modified sequence string value so that even the new written value will not be lost if we fail and be sent
					//on the next try
					this._arrSequenceValue[id_sequence] = newVal;
				}
				//END OF SEQUENCE
				clonedSetting.jsValue = jsValue;
				clonedSetting = this._formatSettingToICTGate(clonedSetting);
				ICTouchAPI.APIServices.SettingsManager.setSettingValue({
					params:[id, clonedSetting.value],
					context:this,
					callback:this._setSettingValueDone,
					callbackParams:{
						context: context,
						callback: callback,
						callbackParams: callbackParams
					}
				});
			}
			else {
				if (callback != undefined) {
					callback.apply(context, [false, callbackParams]);
				}
			}
			if (this.boolCurrentSettingAdmin){
				this.boolCurrentSettingAdmin = false;
				this.strMode = this.MODE_USER;
				ICTouchAPI.APIServices.ICTGate.logout();
			}
		}
		else {
			ICTouchAPI.APIServices.SettingsManager.getSetting({
				params:[id],
				context:this,
				callback:this._gotSettingSetSettingValue,
				callbackParams:{
					jsValue: jsValue,
					context: context,
					callback: callback,
					callbackParams: callbackParams
				}
			});
		}
			}
		}
		id=null;
		jsValue=null;
		context=null;
		callback=null;
		callbackParams=null;
	},

	/**
	 * Method called to know if a CURRENTLY displayed setting is from a SEQUENCE
	 * @param {String} id (name) of the setting
	 * @return {Boolean} if the setting belongs to a shown SEQUENCE
	 */
	isDisplayedSequence : function(id){
		if(this._arrSettingSequence[id]){
			return true;
		}
		return false;
	},

	/**
	 * Method called to know the id of SEQUENCE of a currently displayed setting
	 * @param {String} id (name) of the setting
	 * @return {Boolean} id of the SEQUENCE
	 */
	getDisplayedSequence : function(id){
		return this._arrSettingSequence[id];
	},

	/**
	 * Method used to build and return a typeData object depending on the setting type
	 * @param {Object} objSetting setting object
	 * @return {Object} objType
	 */
	getSettingTypeData : function(objSetting) {
		// CHOICE
		if (objSetting.type == this.TYPE_CHOICE)
			return {
				name: this.TYPE_CHOICE,
				params: null
			};
		// IPADDRESS
		if (objSetting.type == this.TYPE_IPADDRESS)
			return {
				name: this.TYPE_IPADDRESS,
				params: null
			};
		// IPDOMAIN
		if (objSetting.type == this.TYPE_IPDOMAIN)
			return {
				name: this.TYPE_IPDOMAIN,
				params: null
			};
		// DATE || DATETIME
		if (objSetting.type == this.TYPE_DATE || objSetting.type == this.TYPE_DATETIME)
			return {
				name: objSetting.type,
				params: null
			};
		// TIME
		if (objSetting.type == this.TYPE_TIME)
			return {
				name: this.TYPE_TIME,
				params: null
			};
		// PHONE
		if (objSetting.type == this.TYPE_PHONE)
			return {
				name: this.TYPE_PHONE,
				params: null
			};
		if (objSetting.type == this.TYPE_WEBAPP)
			return {
				name: this.TYPE_WEBAPP,
				params: null
			};
		var objType = {
			name: "",
			params: null
		};
		// BOOLEAN[name1,name2]
		var regexp = new RegExp("^"+this.TYPE_BOOLEAN+"(\\[(.+),(.+)\\])?", "i");
		var res = regexp.exec(objSetting.type);
		// "res" contains:
		//  0 => whole matching content if exists
		//  1 => whole type parameters if exists
		//  2 => first parameter
		//  3 => second parameter
		if (res && res[0]) {
			objType.name = this.TYPE_BOOLEAN;
			objType.params = {
				name1: "false",
				name2: "true"
			};
			if (res[1]) {
				objType.params.name1 = res[2];
				objType.params.name2 = res[3];
			}
			return objType;
		}

		// ENABLER[WEBAPP]
		if (objSetting.type == this.TYPE_ENABLER) {
			return {
				name: this.TYPE_ENABLER,
				params: {
					name1: "false",
					name2: "true"
				}
			};
		}

		// TIME[stepHours:stepMinutes:stepSeconds]
		regexp = new RegExp("^"+this.TYPE_TIME+"(\\[([^,]+):([^,]+):([^,]+)\\])?", "i");
		res = regexp.exec(objSetting.type);
		if (res && res[0]) {
			objType.name = this.TYPE_TIME;
			objType.params = {
				stepHours: null,
				stepMinutes: null,
				stepSeconds: null
			};
			if (res[1] ) {
				objType.params.stepHours = parseInt(res[2]);
				objType.params.stepMinutes = parseInt(res[3]);
				objType.params.stepSeconds = parseInt(res[4]);
			}
			return objType;
		}
		// NUMERIC[min,max,step]
		regexp = new RegExp("^"+this.TYPE_NUMERIC+"(\\[([^,]+),([^,]+)(,([^,]+))?\\])?", "i");
		res = regexp.exec(objSetting.type)
		if (res && res[0]) {
			objType.name = this.TYPE_NUMERIC;
			objType.params = {
				min: undefined,
				max: undefined,
				step: 1
			};
			if (res[1]) {
				if (res[2] != "*") objType.params.min = parseInt(res[2]);
				if (res[3] != "*") objType.params.max = parseInt(res[3]);
				if (res[4])
					objType.params.step = res[5];
			}
			return objType;
		}
		// IPPORT[min,max]
		regexp = new RegExp("^"+this.TYPE_IPPORT+"(\\[([^,]+),([^,]+)\\])?", "i");
		res = regexp.exec(objSetting.type)
		if (res && res[0]) {
			objType.name = this.TYPE_IPPORT;
			objType.params = {
				min: undefined,
				max: undefined
			};
			if (res[1]) {
				objType.params.min = parseInt(res[2]);
				objType.params.max = parseInt(res[3]);
			}
			return objType;
		}
		// FLOAT[min,max,step]
		regexp = new RegExp("^"+this.TYPE_FLOAT+"(\\[([^,]+),([^,]+)(,([^,]+))?\\])?", "i");
		res = regexp.exec(objSetting.type)
		if (res && res[0]) {
			objType.name = this.TYPE_FLOAT;
			objType.params = {
				min: undefined,
				max: undefined,
				step: 1
			};
			if (res[1]) {
				if (res[2] != "*") objType.params.min = res[2];
				if (res[3] != "*") objType.params.max = res[3];
				if (res[4])
					objType.params.step = res[5];
			}
			return objType;
		}
		// TEXT[min,max]
		regexp = new RegExp("^"+this.TYPE_TEXT+"(\\[([^,]+),([^,]+)\\])?", "i");
		res = regexp.exec(objSetting.type)
		if (res && res[0]) {
			objType.name = this.TYPE_TEXT;
			objType.params = {
				min: undefined,
				max: undefined
			};
			if (res[1]) {
				objType.params.min = res[2];
				objType.params.max = res[3];
			}
			return objType;
		}
		// PASSWORD[min,max]
		regexp = new RegExp("^"+this.TYPE_PASSWORD+"(\\[([^,]+),([^,]+)\\])?", "i");
		res = regexp.exec(objSetting.type)
		if (res && res[0]) {
			objType.name = this.TYPE_PASSWORD;
			objType.params = {
				min: undefined,
				max: undefined
			};
			if (res[1]) {
				objType.params.min = res[2];
				objType.params.max = res[3];
			}
			return objType;
		}
		// LIST[type]
		regexp = new RegExp("^"+this.TYPE_LIST+"(\\[(.+)\\])?", "i");
		res = regexp.exec(objSetting.type)
		if (res && res[0]) {
			objType.name = this.TYPE_LIST;
			objType.params = {
				listType: undefined
			};
			if (res[1])
				objType.params.listType = res[2];
			return objType;
		}
		return objType;
	},

	/* ----------------------------------- Private methods ------------------------------------- */

	/**
	 * @ignore
	 * Method called when a ChangedSettingEvent is triggered by eventServices
	 * @param {Object} params object returned with the event, params.value is the new setting in this case
	 */
	_onSettingChanged : function(params) {
		ICTouchAPI.log("Setting changed", "SETTINGS");
		ICTouchAPI.logObject(params, "SETTINGS");
		this._formatToJSAndCacheSetting(params.value);
		this._publishChangedSettingEvent(params.value.id, this._arrCachedSettings[params.value.id]);
	},

	/**
	 * @ignore
	 * Method called when a presentationLoaded event is triggered by eventServices
	 */
	_clearCachedCategories : function() {
		this._arrCachedCategories = [];
	},

	/**
	 * @ignore
	 * Method used as a callback to the getSettings() function
	 * @param {Array} arrSettings array containing returned settings
	 * @param {Object} callbackParams object containing webApp's context, callback and callbackParams
	 */
	_gotSettings : function(arrSettings, callbackParams) {
		//if we actually got settings
		if (arrSettings != undefined) {
			if (!callbackParams.alreadyInCache) {
				//we make them readable
				for (var i in arrSettings) {
					this._formatToJSAndCacheSetting(arrSettings[i]);
				}
				// Cache the result
				this._arrCachedCategories[callbackParams.categoryListId] = arrSettings;
			}
			//we initialize the setting array to be sent
			this._arrReturnedSettings = [];
			//we start an asychron chain of calls to the settingManager in order to build the right setting array
			//to be sent to the webapp setting .. else a solution using for loops would send an incomplet array due to
			//the asychron responses from the setting manager
			this._sequenceManagement(0,arrSettings.length,arrSettings,this._arrReturnedSettings,callbackParams);
		}
	},

	/**
	 * @ignore
	 * the function call itself in order to send the right settigns
	 * actually we got 2 chains :
	 * - one for each setting
	 * - an other one on the sequences
	 * the reason is that, as long as we don't get the true setting form the setting manager we cannot
	 * correctly build the _arrReturnedSettings array ... so we shouldn't send it to the setting webapp.
	 */
	_sequenceManagement : function(start,end,arrSettings,arrReturned,callbackParams){
		//once we have treated every setting
		if(start >= end){
			//we try to call the callback sent on the very first call from the setting's webapp
			if (callbackParams.callback != undefined) {
				// Call callback function with stored context and input parameters
				callbackParams.callback.apply(callbackParams.context, [arrReturned, callbackParams.callbackParams]);
			}
		}else{
			//else we have a setting to check ...
			//if we find a SEQUENCE
			var regExpType = new RegExp("SEQUENCE*");
			var isSequence = regExpType.test(arrSettings[start].type);
			if(isSequence){
				//get the sequence id
				var id_sequence = arrSettings[start].id;
				//we save the generic string value to send when setting a setting's value
				this._arrSequenceValue[id_sequence] = [arrSettings[start].value];
				//we get all the setting IDs and their value
				var ids_values = arrSettings[start].value.split("|;|");
				//we call the function which add the sequence with the depency array and call the true setting out from the setting manager
				this._gotSequence(	id_sequence,
					0,
					ids_values.length,
					ids_values,
					arrSettings[start].value,
					start,
					end,
					arrSettings,
					arrReturned,
					callbackParams
					);
			}else{
				//this is everything but a SEQUENCE, so we can simply add it to the array holding the settings
				arrReturned.push(arrSettings[start]);
				this._sequenceManagement(start + 1,end,arrSettings,arrReturned,callbackParams);
			}
		}
	},

   /**
	 * @ignore
	 * this function will :
	 * - add the dependency
	 * - call the setting manager
	 * - call itself through the callback function sent to the setting manager, this a way to synchronize the SettingManager's asynchron behavior
	 * so that the _arrSettingUIDependence array is actually fiilled in before to move further
	 */
	_gotSequence : function(id_sequence,start_ids,end_ids,ids_values,full_value,start,end,arrSettings,arrReturned,callbackParams){
		//once all the sequence has been treated
		if(start_ids >= end_ids){
			//we check the next setting
			this._sequenceManagement(start + 1,end,arrSettings,arrReturned,callbackParams);
		}else{
			//the SEQUENCE is not over yet
			var key_val = ids_values[start_ids].split("=");
			this._arrSettingSequence[key_val[0]] = id_sequence;
			var that = this;
			//add the real setting within the returned settings array
			var func = function(objSetting){
				objSetting.user_rights = that._arrCachedSettings[id_sequence].user_rights;
				objSetting.admin_rights = that._arrCachedSettings[id_sequence].admin_rights;
				//we add the correct true setting
				arrReturned.push(
					objSetting
					);
				//we call our caller on the next setting of the sequence, the caller will check if we can move
				//on the next setting or if some sequence's settings have not been treated yet
				this._gotSequence(	id_sequence,
					start_ids + 1,
					end_ids,
					ids_values,
					full_value,
					start,
					end,
					arrSettings,
					arrReturned,
					callbackParams
					);
			};
			this.getSetting(key_val[0],this,func);
		}
	},

	/**
	 * @ignore
	 * Method used as a callback to the getSetting() function
	 * @param {Object} objSetting returned setting
	 * @param {Object} callbackParams object containing webApp's context, callback and callbackParams
	 */
	_gotSettingSequence : function (objSetting, callbackParams) {
		if (objSetting.exist)
		{
			this._formatToJSAndCacheSetting(objSetting);
			//set objSetting in an array
			this._arrReturnedSetting	= [];
			var tmpArray = [objSetting];
			//call sequenceManagement
			this._sequenceManagement(0,tmpArray.length,tmpArray,this._arrReturnedSetting,callbackParams);
		}
		else if (callbackParams.callback != undefined) {
			// Call callback function with stored context and input parameters
			callbackParams.callback.apply(callbackParams.context, [null, callbackParams.callbackParams]);
		}
	},

	/**
	 * @ignore
	 * Method used as error callback to the getSettings() function
	 * @param {String} strErrorMsg error message from business  layer
	 * @param {Object} callbackParams object conaining webapps's context, callbacks
	 *
	 **/
	_notGotSettings : function(strErrorMsg, callbackParams) {
		// only call the error callback
		if (callbackParams.errorCallback != undefined) {
			// Call callback function with stored context and input parameters
			callbackParams.errorCallback.apply(callbackParams.context, [strErrorMsg, callbackParams.callbackParams]);
		}
	},

	/**
	 * @ignore
	 * Method used as a callback to the getSetting() function
	 * @param {Object} objSetting returned setting
	 * @param {Object} callbackParams object containing webApp's context, callback and callbackParams
	 */
	_gotSetting : function (objSetting, callbackParams) {
		if (objSetting.exist)
		{
			this._formatToJSAndCacheSetting(objSetting);
			if (callbackParams.callback != undefined) {
				// Call callback function with stored context and input parameters
				try{
					callbackParams.callback.apply(callbackParams.context, [this._arrCachedSettings[objSetting.id], callbackParams.callbackParams]);
				}catch(err){
					console.warn("settingServices._gotSetting: your callback function made an error listening setting = " + dojo.toJson(this._arrCachedSettings[objSetting.id]));
				}
			}
		}
		else if (callbackParams.callback != undefined) {
			// Call callback function with stored context and input parameters
			callbackParams.callback.apply(callbackParams.context, [null, callbackParams.callbackParams]);
		}
	},

	/**
	 * Method used as a callback to the getSettingValue() function
	 * @param {Object} objSetting returned setting
	 * @param {Object} callbackParams object containing webApp's context, callback and callbackParams
         * @ignore
	 */
	_gotSettingValue : function (objSetting, callbackParams) {
		if (objSetting.exist)
		{
			this._formatToJSAndCacheSetting(objSetting);

			if (callbackParams.callback != undefined) {
				// Call callback function with stored context and input parameters
				callbackParams.callback.apply(callbackParams.context, [this._arrCachedSettings[objSetting.id].jsValue, callbackParams.callbackParams]);
			}
		}
		else if (callbackParams.callback != undefined) {
			// Call callback function with stored context and input parameters
			callbackParams.callback.apply(callbackParams.context, [null, callbackParams.callbackParams]);
		}
	},

	/**
	 * @ignore
	 * Private recursive function; used to get one setting, consecutive to a call of getSettingsFromIds
	 */
	_getNextSettingFromId : function (objContext) {
		if (objContext.ids[objContext.index] === undefined) {
			// end of recursive call
			if (objContext.callback != undefined) {
				// Call callback function with stored context and input parameters
				objContext.callback.apply(objContext.context, [objContext.results, objContext.callbackParams]);
			}
			return;
		}
		this.getSetting(objContext.ids[objContext.index], this, this._gotNextSettingFromId, objContext);
	},

	/**
	 * @ignore
	 * Private function; callback that is called as a result of _getNextSettingFromId.
	 */
	_gotNextSettingFromId : function (objSetting, objContext) {
		// The setting is pushed into results array, index is incremented and we continue recursive call.
		objContext.results[objContext.ids[objContext.index]] = objSetting;
		objContext.index++;
		this._getNextSettingFromId(objContext);
	},

	/**
	 * @ignore
	 * Method used as a callback to the setSettingValue() function
	 * when the Setting needed to be retrieved from ICTGate because it was not in cache
	 * @param {Object} objSetting returned setting
	 * @param {Object} callbackParams object containing webApp's context, callback and callbackParams
	 */
	_gotSettingSetSettingValue : function(objSetting, callbackParams) {
		if (objSetting.exist) {
			this._formatToJSAndCacheSetting(objSetting);
			this.setSettingValue(objSetting.id, callbackParams.jsValue);
		}
		else {
			if (callbackParams.callback != undefined) {
				callbackParams.callback.apply(callbackParams.context, [false, callbackParams.callbackParams]);
			}
		}
	},

	/**
	 * @ignore
	 */
	_showAdminKeyboardForSetting : function(setting){
		var strUserMode = this.MODE_ADMIN;
		var funcCallback = this._logAdmin;
		var cxContext = this;
		ICTouchAPI.keyboardServices.deployKeyboard(
			ICTouchAPI.keyboardServices.TYPE_PASSWORD,
			null,
			{
				funcCallbackOk : function(password) {
					// Try login
					ICTouchAPI.APIServices.ICTGate.login({
						params:[strUserMode, password],
						context:cxContext,
						callback:funcCallback,
						callbackParams:setting
					});
				}
			}
			);
	},

	/**
	 * @ignore
	 */
	_logAdmin : function(strMsgLogin,setting){
		if (strMsgLogin == "Login Ok"){
			this.strMode = this.MODE_ADMIN;
			this.boolCurrentSettingAdmin = true;
			this.setSettingValue(setting.id, setting.jsValue, setting.context, setting.callback, setting.callbackParams);
		}
	},

	/**
	 * @ignore
	 * Method used as a callback to the setSettingValue() function
	 * @param {Object} result object representing the value of the UpdateReport returned by the SettingsManager
	 * @param {Object} callbackParams object containing webApp's context, callback and callbackParams
	 */
	_setSettingValueDone : function(result, callbackParams) {
		//callback to manage the return of needReboot
		var func = function(ret){
			//if needReboot send a boolean set to TRUE we set the rebootRequired variable to true
			if (typeof webapp.settings.data === "object") {
				webapp.settings.data.setReboot(ret);
			}
		}

		//we ask the platform whether we have to reboot or not
		this.needReboot({
			context : this,
			callback : func
		});

		//the result arg doesn't seem to be well passed
		if (callbackParams.callback != undefined) {
			callbackParams.callback.apply(callbackParams.context, [result, callbackParams.callbackParams]);
		}
	},

	/**
	 * @ignore
	 * Method used to determine if a new value is valid for a Setting object
	 * @param {Object} objSetting setting to check with new value
	 * @param {String/Boolean/Number} newValue jsValue to test for the Setting
	 * @return {Boolean} valid true or false
	 */
	_isNewValueValidForSetting : function(objSetting, newValue) {
		var valid = false;
		if (objSetting.typeData == undefined) {
			objSetting.typeData = this.getSettingTypeData(objSetting);
		}
		// Note: type "PHONE" is not validated from MMI.
		switch (objSetting.typeData.name) {
                        case this.TYPE_ENABLER:
			case this.TYPE_BOOLEAN:
				if (typeof newValue == "boolean") {
					valid = true;
				}
				break;
			case this.TYPE_NUMERIC:
				if (typeof newValue == "number") {
					if ((objSetting.typeData.params.min == undefined || newValue >= objSetting.typeData.params.min)
						&& (objSetting.typeData.params.max == undefined || newValue <= objSetting.typeData.params.max)) {
						valid = true;
					}
				}
				break;
			case this.TYPE_IPPORT:	// same as NUMERIC
				if (typeof newValue == "number") {
					if ((objSetting.typeData.params.min == undefined || newValue >= objSetting.typeData.params.min)
						&& (objSetting.typeData.params.max == undefined || newValue <= objSetting.typeData.params.max)) {
						valid = true;
					}
				}
				break;
			case this.TYPE_CHOICE:
				// The jsValue of a "CHOICE" setting must be the index of the value in the allowedValues array
				if (parseInt(newValue) >= 0 && parseInt(newValue) < objSetting.allowedValues.length) {
					valid = true;
				}
				break;
			case this.TYPE_IPADDRESS: {
				var ipAddressNumbers = newValue.split(".");
				if (ipAddressNumbers.length == 4) {
					valid = true;
					for (var i = 0; i < ipAddressNumbers.length; i++) {
						if (!(parseInt(ipAddressNumbers[i],10) >= 0 && parseInt(ipAddressNumbers[i],10) <= 255)) {
							valid = false;
						}
					}
				}
				break;
			}
			case this.TYPE_DATE: // Breakthrough
			case this.TYPE_DATETIME:
				valid = ( newValue instanceof Date );
				break;
			default : {
				valid = true;
				break;
			}
		}
		return valid;
	},

	/**
	 * @ignore
	 * Method used to format a Setting object to JS format (add jsValue and typeData attributes to object)
	 * After formatting, the Setting object is stored into the _arrCachedSettings array
	 * @param {Object} objSetting setting object
	 */
	_formatToJSAndCacheSetting : function(objSetting) {

		// Add "typeData" object to objSetting
		objSetting.typeData = this.getSettingTypeData(objSetting);
		if (objSetting.isUndefined === 0) {
			// Switch in case of jsValue must be different from value
			switch (objSetting.typeData.name) {
                                case this.TYPE_ENABLER:
				case this.TYPE_BOOLEAN:
					if (objSetting.value == "true") {
						objSetting.jsValue = true;
					}
					else if (objSetting.value == "false") {
						objSetting.jsValue = false;
					}
					break;
				case this.TYPE_NUMERIC:
					objSetting.jsValue = parseInt(objSetting.value);
					break;
				case this.TYPE_IPPORT:
					objSetting.jsValue = parseInt(objSetting.value);
					break;
				case this.TYPE_LIST:
					if(objSetting.value == "")
						objSetting.jsValue = [];
					else
						objSetting.jsValue = objSetting.value.split(";");
					break;
				case this.TYPE_CHOICE:
					objSetting.jsValue = -1;
					for (var i=0; i<objSetting.allowedValues.length; i++) {
						if (objSetting.allowedValues[i].id == objSetting.value) {
							objSetting.jsValue = i;
							break;
						}
					}
					break;
				case this.TYPE_DATE: // Breakthrough
				case this.TYPE_DATETIME:
					objSetting.jsValue = new Date();
					objSetting.jsValue.setTime(objSetting.value*1000);
					break;
				case this.TYPE_TEXT:
					objSetting.jsValue = objSetting.value;
					break;
				default:
					objSetting.jsValue = objSetting.value;
					break;
			}
		}
		else
		{
			// "UNDEFINED"
			objSetting.jsValue = objSetting.value;
		}
		// always cache it
		// objSetting is now cached in _arrCachedSettings
		this._arrCachedSettings[objSetting.id] = objSetting;

		// Also update category list with the new value, both user and admin
		this._updateSettingCategory(objSetting, objSetting.mainCategory+"_"+objSetting.category+"_"+this.MODE_USER);
		this._updateSettingCategory(objSetting, objSetting.mainCategory+"_"+objSetting.category+"_"+this.MODE_ADMIN);

		ICTouchAPI.log("Setting cached: " + objSetting.id, "SETTINGS");
	},

	/**
	 * @ignore
	 * Update the setting in this category
	 */
	_updateSettingCategory: function(objSetting, categoryListId) {
		if (this._arrCachedCategories[categoryListId]) {
				var objCategory = this._arrCachedCategories[categoryListId];
				var index;
				for(index in objCategory) {
						if (objCategory[index].id === objSetting.id) {
								objCategory[index] = objSetting;
						}
				}
		}
	},

	/**
	 * @ignore
	 * Method used to format a Setting object to ICTGate format (remove jsValue and typeData attributes to object)
	 * The input setting is cloned before being modified
	 * @param {Object} inputSetting Setting object to format for ICTGate
	 * @return {Object} objSetting Setting object formatted for ICTGate (cloned from input)
	 */
	_formatSettingToICTGate : function(inputSetting) {
		var objSetting = dojo.clone(inputSetting);

		switch (objSetting.typeData.name) {
                        case this.TYPE_ENABLER:
			case this.TYPE_BOOLEAN:
				if (objSetting.jsValue == true) {
					objSetting.value = "true";
				} else if (objSetting.jsValue == false) {
					objSetting.value = "false";
				}
				break;
			case this.TYPE_NUMERIC:
				objSetting.value = objSetting.jsValue+'';
				break;
			case this.TYPE_IPPORT:
				objSetting.value = objSetting.jsValue+'';
				break;
			case this.TYPE_CHOICE:
				objSetting.value = objSetting.allowedValues[objSetting.jsValue].id
				break;
			case this.TYPE_LIST:
				objSetting.value = objSetting.jsValue.join(";");
				break;
			case this.TYPE_DATE: // Breakthrough
			case this.TYPE_DATETIME:
				objSetting.value = parseInt(objSetting.jsValue.getTime()/1000, 10);
				break;
			case this.TYPE_TEXT:
				objSetting.value = objSetting.jsValue;
				break;
			default:
				objSetting.value = objSetting.jsValue;
				break;
		}
		delete objSetting.typeData;
		delete objSetting.jsValue;
		// Value may have changed, so also update it in input setting to keep the cache consistent
		inputSetting.value = objSetting.value;
		return objSetting;
	},

	/**
	 * @ignore
	 * Method used to broadcast to new setting provided by the ChangedSettingEvent to all subscribed webApps
	 * @param {String} strSettingName id (name) of the concerned setting = topic of dojo.subscribe/dojo.publish
	 * @param {Object} objSetting new setting
	 */
	_publishChangedSettingEvent : function(strSettingName, objSetting) {
		dojo.publish("settingServices:" + strSettingName, [objSetting]);
	},

	/**
     * Called to check weither we upgrade some settings before to enter in the webapp setting
     * the platform lock the webapp acces
     * @ignore
     */
	enterMMI : function(args) {
		ICTouchAPI.APIServices.SettingsManager.enterMMI(args);
	},

	/**
     * send an event expected by the settingManager to leave the webapp
     * @ignore
     */
	exitMMI : function() {
		ICTouchAPI.APIServices.SettingsManager.exitMMI();
	},

	/**
     * @ignore
     */
	needReboot : function(args){
		ICTouchAPI.APIServices.SettingsManager.needReboot(args);
	}
});

// Create Setting Service.
ICTouchAPI.settingServices=new ICTouchAPI.settingServices;
