/**
* @class ICTouchAPI.webWidget
* @extends Object
* @abstract
* Base Class for all UI Widget - Do not use it directly
*/
dojo.provide("ICTouchAPI.webWidget");
dojo.declare("ICTouchAPI.webWidget", dijit._Widget,
    {
	/**
         * @ignore
         */
        _status: '',

        /**
         * WidgetName could be it's id (better).
         * @ignore
         */
	strWidgetName: '',

        /**
         * Pointer to the associated webapp.
         * @ignore
         */
	webapp: null,

        /**
         * Node's id that will contain the widget which id is strWidgetName.
         * @ignore
         */
	containerId: '',

        /**
         * Animation if it was called at least one time
         * @ignore
         */
	animation: null,
	// Notify the transitionServices that this widget needs to get an exclusive iframe because it contains iframes
	exclusiveIframe: false,
	/**
	 * @ignore
	 */
	_subLanguageChanged : null,

    /**
     * Prior to anything else, builds a reference Node if it does not exist.
     * @ignore is this method !
     * @param {Array} params a list of parameters to pass to the constructor.
     * @param {Array} node name of the node where to place the widget. (containerId = )
     *               and facultative : the widget's id name (strWidgetName = )
     *               If not set, the name will be it's class name + a unique id.
     * @param {String/Number} position for dojo.place. Optional argument. Can be a number
     *               or one of the following strings: "before", "after", "replace", "only",
     *               "first", or "last". If omitted, "last" is assumed.
     *               The value of "only" replaces all children of the refNode.
     *               If the position argument is a number, it indicates
     *               that the node should be inserted as a child of
     *               refNode with this number (counting from 0).
     */
	postscript: function (params, node, position) {

		// If node.containerId is set, build a new node inside of it.
		if (node && node.containerId) {
            if (!dojo.byId(node.strWidgetName)) {
                // Get base node.
                var originNode = dojo.byId(node.containerId);
                // Create new node.
                var elt = dojo.doc.createElement("div");
                // place it inside it's parent.
                dojo.query(elt)
                    .place(originNode, position);
                // Give it an id if strWidgetName is given.
                if (node.strWidgetName) {
                    elt.id = node.strWidgetName;
                    // Set widget's future anchor point.
                    var node = dojo.byId(node.strWidgetName);
                } else {
                    // Set widget's future anchor point.
                    var node = elt;
                }
            }
		}
		// Kick off the widget's building process.
		this.create(params, node);
	},

        /**
	 * Destructor of the widget
	 */
	destroy: function() {		
		if( this.animation )
			this.animation.destroy();
		ICTouchAPI.i18nServices.unsubscribeAllI18nHandlers(this);

		dojo.unsubscribe(this._subLanguageChanged);

		for(var attr in this) {
			/*
			 * Search for scroll elements to be destroyed
			 * If firstElementChild or lastElementChild is null, uninit will crash
			*/
			// Search for scroll elements to be destroyed
			var elt = this[attr];
			if (elt && elt.IctScroll && elt.firstElementChild && elt.lastElementChild) {
				FXScroll.unInit(elt);
			}
		}

		this.inherited(arguments);
		this.webapp.removeWidget(this.strWidgetName);
	},

        /**
         * those methods will be bound to every widget that inherits webWidget
         * @ignore
         * @param {Array} args may contain the pointer to its parent webapp (pointer webapp)
         */
	constructor: function(args){
		ICTouchAPI.logObject(this, "WIDGETS");
		this._status = "QUEUED_STACK";
		this._subLanguageChanged = null;
		// Save the widget's parent webapp pointer if given.
		if (args.webapp) {
			this.webapp = args.webapp;
		} else {
			// Else, retrieve it from the declaredClass (could not work if missspelled).
			var strWebappName = this.declaredClass;
            var splitString = strWebappName.split(".");
			splitString.pop();
            this.namespace = splitString.join(".");
			// Require the webapp.
			this.webapp = ICTouchAPI.webApplicationManager.getRefApplication(this.namespace);
		}

        // Sets the widget's Id name and loads the skin file.
        var handle=dojo.connect(this, "postCreate",this, function(){
            // Whatever happened in the previous process, this.id is set and has
            // to be equal to strWidgetName.
            this.strWidgetName = this.id;
			// Register widget.
			this.webapp.registerWidget(this);
            // Parse the dom to find any widget to create
			dojo.parser.parse();
            dojo.disconnect(handle);
        });		
		

		this._subLanguageChanged = dojo.subscribe("languageChanged", this, function() {
			for (var i in this.attributeMap) {
				if(this[i] && this[i].isI18Ned) {
					this.attr(i, this[i]);
				}
			}
		});

		// This is some special code that enables the tracking of which element leaks in chrome's profiler
		// This works by creating a new object called like the widget we are currently constructing
		if (generalConfig.developer == true) {
			var splitString = this.declaredClass.split(".");
			var strLast = splitString.pop();
			eval("this.objTracking = new function myIC_WA_"+strLast+"(){}");
		}
	},

        /**
         * @ignore
         */
	postCreate: function () {
	},

	/**
         * Called when your widget is visible
         * @ignore
         */
	show:function () {
		this._status="VISIBLE";
	},

        /**
         * hide : function called when your widget is on the top of the stack but hidden
         * @ignore
         */
	hide:function () {
		this._status="TOP_STACK";
	},

        /**
         * function called when your widget goes to the top of the stack
         * @ignore
         */
	goOnTop:function () {
		this._status="TOP_STACK";
	},

        /**
         * Function called when the widget goes back to the stack.
         * @ignore
         */
	backToStack:function () {
		this._status="QUEUED_STACK";
	},

        /**
         * Get the name of the class that created the widget.
         * @ignore
         * @return {String} the class name.
         */
	getInstanceOf: function () {
		return this.declaredClass;
	},

       /**
	* Makes this widget stealth to transitionServices
	* @param {Boolean} boolStealth stealth or unstealth!
        */
	stealth: function (boolStealth) {
		if (boolStealth === false) {
			ICTouchAPI.transitionServices.unstealth(this.getInstanceOf());
		} else {
			ICTouchAPI.transitionServices.stealth(this.getInstanceOf());
		}

	},

        /**
         * Refresh the widget's view.
	 * Using this method, the widget might blink.
	 */
	updateWidget: function() {
        // Make a fake context to replace the current one.
        var fakeCtx = new dojox.dtl.Context();
        this.render(fakeCtx);
         // Then rebuild using the right one.
        this.render();
        // This is a trick to avoid duplicating nodes in a forloop.
        // There must be a better way to achieve that!
	},

	/**
	 * Update the widget, this method has to be overriden by each webwidget
	 */
	update : function(){

	},

	/**
	 * Update the widget content, this method may be reimplmented by webwidgets
	 * @return {Boolean} true if a full reload is required
	 */
	updateContent : function(params){
		params = null;
		return true;
	},

	/**
	 * Close the widget.
	 * Removes its dom, and deletes the object.
	 */
	close: function() {
		this.webapp.removeWidget(this.id);
	},

        /**
         * Set the widget moveable.
         * the whole content is clickable and draggable.
         */
	setMoveable: function () {
		new dojo.dnd.Moveable(
			dojo.byId(this.id));
	},

	/**
	 * Refresh this view by destroying every widget contained and redo them
	 * This should be used when other widget are contained within this widget
	 * @deprecated
	 */
	refresh: function() {
		// Clone the domNode of the widget ( without its children )
		var srcNodeRef = this.domNode.cloneNode(false);
		var nodeToClean = this.domNode;
		// Replace the fresh node with the old one
		dojo.place(srcNodeRef, this.domNode, "replace");

		ICTouchAPI.tools.destroyWidgets(nodeToClean);

		//This subscribe is removed on destroy function
		this._subLanguageChanged = dojo.subscribe("languageChanged", this, function() {
			for (var i in this.attributeMap) {
				if(this[i] && this[i].isI18Ned) {
					this.attr(i, this[i]);
				}
			}
		});

		if( params != undefined )
			this.create(params, srcNodeRef);
		else
			this.create({}, srcNodeRef);
	}
});
/**
* @class ICTouchAPI.eventServices
* @singleton
* @extends Object
* The Events class provide methods to subscribe and unsubscribe to backend events
*/
dojo.provide("ICTouchAPI.eventServices");
dojo.declare("ICTouchAPI.eventServices",null,{

/* --------------------------------- Public attributes ------------------------------------ */

/* --------------------------------- Private attributes ----------------------------------- */

	// List of subscribedEvents
	/**
	 * @ignore
	 */
	_arrSubscribedEvents: {},
	/**
	 * @ignore
	 */
	_xhr : null,
	/**
	 * @ignore
	 */
	_arrChannels: [],
	/**
	 * @ignore
	 */
	_nbChannels: 2,
	/**
	 * @ignore
	 */
	_lastChannel: null,
	/**
	 * @ignore
	 */
	_currChannel: null ,
	/**
	 * @ignore
	 */
	_nextChannel: null, // logCleanUp
	// Number of processed events
	/**
	 * @ignore
	 */
	_arrNbProcessedEvents: [],
	/**
	 * @ignore
	 */
	_arrNbStoredEvents: [],
	/**
	 * @ignore
	 */
	_areAllEventsTreated: false,
	/**
	 * @ignore
	 */
	_reconnectRequested: false,
	// {int} Temporization between retries to avoid freezing the ictouch in milliseconds.
	/**
	 * @ignore
	 */
	timeBetweenSeriesOfRetries: 2000,
	// {int} nb of retries between temporizations.
	/**
	 * @ignore
	 */
	nbOfConnexionRetries: 3,
	// {int} nb of current retries done before temporization.
	/**
	 * @ignore
	 */
	_tmpNbOfRetriesDone: 0,
	/**
	 * @ignore
	 */
	_gotCookie : false,
	/**
	 * @ignore
	 */
	_lastChannelAccess : null,

/* ------------------------------------ Constructor --------------------------------------- */

	// Initiliaze parameters
	/**
	 * @ignore
	 */
	constructor : function () {
		this._arrSubscribedEvents = {};
		this._lastChannel = this._nbChannels - 1;
		this._currChannel = this._lastChannel;
		this._nextChannel = (this._currChannel + 1) % this._nbChannels ; // logCleanUp
		
                // No channel for simulation mode
                if(!generalConfig.simulation) {
                    this._cookieWait();
                    this._arrChannels[this._nextChannel] = this._openChannel(); // logCleanUp
                }
		for(var i=0;i<this._nbChannels;i++) {
			// logCleanUp
			this._arrNbProcessedEvents[i] = 0 ;
			this._arrNbStoredEvents[i] = 0 ;
		}

                // No channel for simulation mode
                if(!generalConfig.simulation) {
                    this._initializeChannel();
                }
	},

/* ----------------------------------- Getter / Setter------------------------------------- */


/* ----------------------------------- Public methods ------------------------------------- */

    /**
	 * Method used by webapps to subscribe to a backend event
	 * @param {Object} ctx : callback execution context
	 * @param {String} strEventName : name of the subscribed event
	 * @param {Function} callback : function called when the event occurs
	 */
	subscribeToEvent : function(ctx, strEventName, callback){
		if(!this._arrSubscribedEvents[strEventName]){
			this._arrSubscribedEvents[strEventName]=new ICTouchAPI.eventServices.registeredEvent(strEventName);
		}
		this._arrSubscribedEvents[strEventName].addWebApp(ctx, callback);
		ctx=null;
		strEventName=null;
		callback=null;
	},

    /**
	 * Method used by webapps to unsubscribe to a backend event
	 * @param {Object} objWebApp : webapp execution context
	 * @param {String} strEventName : name of the subscribed event
	 */
	unsubscribeToEvent : function(objWebApp, strEventName){
		if(this._arrSubscribedEvents[strEventName]){
			// Removing webapp's subscription
			this._arrSubscribedEvents[strEventName].removeWebApp(objWebApp.getApplicationName());
			// If no webapp remaining
			if (!this._arrSubscribedEvents[strEventName].getNbWebapps()) {
				// Deleting the registered event
				delete this._arrSubscribedEvents[strEventName];
			}
		}
		objWebApp=null;
		strEventName=null;
	},

	/**
	 * Return list of Events Subscribed to the ICTGate
	 * @ignore
	 * @return {Array}
	 */
	listSubscribedEvents: function(){
		return this._arrSubscribedEvents;
	},

	/**
	 * Return the list of webapplications which subscribe to a particular event
	 * @ignore
	 * @param {String} strEventName
	 * @return {Array}
	 */
	listWebAppsForEvent: function(strEventName){
		return this._arrSubscribedEvents[strEventName];
	},

	/**
	 * Method used by the ICTGate for core event publishing
	 * @ignore
	 * @param {String} strEventName Name of the event which occurs
	 * @param {Object} params
	 */
	_corePublishEvent : function(strEventName, params){
		ICTouchAPI.log("Publishing " + strEventName, "EVENTS");
		dojo.publish("eventServices." + strEventName, params);
	},

/* --------------------------------- Private Methods -------------------------------------- */

	/**
	 * @ignore
	 * Send logs of attributes of the service to the logger.
	 * @param {boolean} boolAdvancedLogs Boolean indicating if advanced or basic logs has to be written
	 */
	dump : function(boolAdvancedLogs){
		ICTouchAPI.debugServices.dump("dump function of eventServices with " + ((boolAdvancedLogs)?"advanced":"basic") + " logs option");

		ICTouchAPI.debugServices.dump("Number of channels (_nbChannels): " + this._nbChannels);
		ICTouchAPI.debugServices.dump("Current channel used (_currChannel): " + this._currChannel);
		ICTouchAPI.debugServices.dump("Previous channel used (_lastChannel): " + this._lastChannel);
		ICTouchAPI.debugServices.dump("Next channel used (_nextChannel): " + this._nextChannel);
		ICTouchAPI.debugServices.dump("Last channel accessed (_lastChannelAccess): " + this._lastChannelAccess);
		ICTouchAPI.debugServices.dump("Number of events stored for the current channel (_arrNbStoredEvents[_currChannel]): " + this._arrNbStoredEvents[this._currChannel]);
		ICTouchAPI.debugServices.dump("Number of events processed for the current channel (_arrNbProcessedEvents[_currChannel]): " + this._arrNbProcessedEvents[this._currChannel]);
		ICTouchAPI.debugServices.dump("Are all events treated ? (_areAllEventsTreated): " + this._areAllEventsTreated);
		ICTouchAPI.debugServices.dump("Is reconnexion requested (_reconnectRequested): " + this._reconnectRequested);
		ICTouchAPI.debugServices.dump("Number of connexion retries done: " + this._tmpNbOfRetriesDone);

		if (boolAdvancedLogs) {
			// List of events currently subscribed
			ICTouchAPI.debugServices.dump("List of events currently subscribed:");
			for (var strEvent in this._arrSubscribedEvents) {
				ICTouchAPI.debugServices.dump("Event: " + strEvent + " is currently subscribed by " + this._arrSubscribedEvents[strEvent]._nbWebapps + " applications:");
				for (var application in this._arrSubscribedEvents[strEvent]._applications) {
					ICTouchAPI.debugServices.dump(" - " + application);
				}
			}
		}
	},

	/**
	 * @ignore
	 */
	_cookieWait:function(){
		var xhrtmp = new XMLHttpRequest();
		xhrtmp.open('GET',
			ICTouchAPI.eventChannelURI + "services/ICTGate/openSession",
			false
		);
		xhrtmp.send(null);
	},

	/**
	 * Open a fresh new pending channel
	 * @ignore
	 */
	_openChannel: function(){
		var newxhr = new XMLHttpRequest();
		newxhr.open('GET', ICTouchAPI.eventChannelURI + 'services/ICTGate/openEventChannel');
		// For use with cookies
		newxhr.withCredentials = true;
		newxhr.onreadystatechange = this.channelUpdated;
		newxhr.send(null);
		return newxhr;
	},

	/**
	 * switch to next channel
	 * @ignore
	 */
	_switchToNextChannel: function(){
		var tmp = this._currChannel ;
		this._currChannel++ ;
		this._currChannel = this._currChannel % this._nbChannels ;
		this._xhr = this._arrChannels[this._currChannel] ;
		if(this._arrChannels[tmp]){
			this._arrChannels[tmp].abort(); // logCleanUp
		}
		this._arrChannels[tmp] = this._openChannel();
	},

	/**
	 * Initialize the event channel with de ICTGate
	 * @ignore
	 */
	_initializeChannel: function(){
		ICTouchAPI.log("Initialize channel", "EVENTS");
		// Reset our flags
		this._arrNbProcessedEvents[this._currChannel] = 0;
		this._arrNbStoredEvents[this._currChannel] = 0;
		this._areAllEventsTreated= false;
		this._reconnectRequested=false;
		this._switchToNextChannel();
	},

	/**
	 * @ignore
	 */
	_myChannel: function(pxhr) {
		for(var i=0;i<ICTouchAPI.eventServices._nbChannels;i++) {
			if(ICTouchAPI.eventServices._arrChannels[i] == pxhr){
				ICTouchAPI.eventServices._lastChannelAccess=i ;
				return i;
			}
		}
	},

	/**
	 * @ignore
	 */
	channelUpdated : function(){
		ICTouchAPI.log("Channel updated", "EVENTS");

		ICTouchAPI.logObject(this, "EVENTS");

		/* events are waiting to be processed */
		if(this.readyState >= 3 ) {
			// Count the number of events currently stored in the eventServices._xhr object
			ICTouchAPI.eventServices._arrNbStoredEvents[ICTouchAPI.eventServices._myChannel(this)] = 0;
			var stop = 0;
			while (stop != -1) {
				stop = this.responseText.indexOf("((eventEnd))", stop+1);
				if (stop != -1) {
					ICTouchAPI.eventServices._arrNbStoredEvents[ICTouchAPI.eventServices._myChannel(this)]++;
				}
			}

			/*If channel got closed and all events have been processed, re open it.
			Else, set a flag to indicate that the channel has to be re open after all events have been processed.*/
			if (this.readyState == 4) {
				if (ICTouchAPI.eventServices._arrNbProcessedEvents[ICTouchAPI.eventServices._myChannel(this)] >= ICTouchAPI.eventServices._arrNbStoredEvents[ICTouchAPI.eventServices._myChannel(this)]) {
					ICTouchAPI.debugServices.debug("ICTouchAPI.eventServices - channelUpdated / (currentChannel: " + ICTouchAPI.eventServices._currChannel + "), readyState == 4 and all events has been processed ("+ICTouchAPI.eventServices._arrNbProcessedEvents[ICTouchAPI.eventServices._myChannel(this)]+">="+ICTouchAPI.eventServices._arrNbStoredEvents[ICTouchAPI.eventServices._myChannel(this)]+"). ICTGate channel is full, call _initializeChannel.");
					ICTouchAPI.eventServices._initializeChannel();
				} else {
					ICTouchAPI.debugServices.debug("ICTouchAPI.eventServices - channelUpdated / (currentChannel: " + ICTouchAPI.eventServices._currChannel + "), readyState == 4 and events are waiting to be processed ("+ICTouchAPI.eventServices._arrNbProcessedEvents[ICTouchAPI.eventServices._myChannel(this)]+"<"+ICTouchAPI.eventServices._arrNbStoredEvents[ICTouchAPI.eventServices._myChannel(this)]+"). ICTGate channel is full, request to reconnect.");
					ICTouchAPI.eventServices._reconnectRequested = true;
				}
			}

			// for each event stored
			for (var i=ICTouchAPI.eventServices._arrNbProcessedEvents[ICTouchAPI.eventServices._myChannel(this)]+1; i <= ICTouchAPI.eventServices._arrNbStoredEvents[ICTouchAPI.eventServices._myChannel(this)]; ++i) {

				// if this event is part of the unprocessed events
				if (i > ICTouchAPI.eventServices._arrNbProcessedEvents[ICTouchAPI.eventServices._myChannel(this)] ) {
					var firstIndex = 0;
					var secondIndex = 0;
					// we make our indexes go to the target event
					for (var j=0; j < i; j++) {
						firstIndex = this.responseText.indexOf("((eventStart))", secondIndex);
						secondIndex = this.responseText.indexOf("((eventEnd))", firstIndex);
					}
					// the target event is parsed and processed
					var temp = this.responseText.slice(firstIndex + 14, secondIndex);

					/*There's a trick there :
					The counter has to be increased before hands, otherwise an other
					callback may process it again even though it's already be sent.*/
					// Increase our processed events counter
					ICTouchAPI.eventServices._arrNbProcessedEvents[ICTouchAPI.eventServices._myChannel(this)]++;

					// Throws an error if bad formatted event (Webapp : crms00211314)
					try{
						// Convert the event to an event object.
						var currentEventObject=dojo.fromJson(temp);
					}catch(_error){
						ICTouchAPI.debugServices.error('ICTouchAPI.eventServices - channelUpdated / ' + _error + ' converting ' + temp,true);
						if(_error.stack) {
							ICTouchAPI.debugServices.error(_error.stack);
						}
						continue;
					}

					ICTouchAPI.debugServices.debug("ICTouchAPI.eventServices - channelUpdated / (currentChannel: " + ICTouchAPI.eventServices._currChannel + ", event number " + ICTouchAPI.eventServices._arrNbProcessedEvents[ICTouchAPI.eventServices._myChannel(this)] + "), process event: " + currentEventObject.eventName);
					try{
						// Publish the event.
						ICTouchAPI.eventServices._corePublishEvent(currentEventObject.eventName,currentEventObject.params);
					}catch(_error){
						ICTouchAPI.debugServices.error('ICTouchAPI.eventServices - channelUpdated / ' + _error + ' treating ' + temp,true);
						if(_error.stack)
							ICTouchAPI.debugServices.error(_error.stack);
						continue;
					}

					// If all events have been treated, set a flag.
					if (ICTouchAPI.eventServices._arrNbProcessedEvents[ICTouchAPI.eventServices._myChannel(this)] == ICTouchAPI.eventServices._arrNbStoredEvents[ICTouchAPI.eventServices._myChannel(this)]) {
						ICTouchAPI.eventServices._areAllEventsTreated = true;
						// If a reconnection request was previously set, then reconnect.
						if (ICTouchAPI.eventServices._reconnectRequested) {
							ICTouchAPI.eventServices._initializeChannel();
						}
					}else{
						// If there are still events to be treated
						ICTouchAPI.eventServices._areAllEventsTreated = false;
					}
				}
			}
		}
	},

	/**
	 * @ignore
	 */
	reconnect: function () {
		// If connexion got closed, re-initialize the event channel.
		// If the number of retries done is > to desired number of retries
		if (ICTouchAPI.eventServices._tmpNbOfRetriesDone >= ICTouchAPI.eventServices.nbOfConnexionRetries) {
			// Set a tempo before setting the number of retries done to 0 and then re-init the channel.
			setTimeout(function () {
				ICTouchAPI.eventServices._tmpNbOfRetriesDone = 0;
				ICTouchAPI.eventServices._initializeChannel();
			},
			ICTouchAPI.eventServices.timeBetweenSeriesOfRetries);

		// Else, if number of retries done is <, then retry.
		} else  if (ICTouchAPI.eventServices._tmpNbOfRetriesDone < ICTouchAPI.eventServices.nbOfConnexionRetries) {
			ICTouchAPI.eventServices._initializeChannel();
			ICTouchAPI.eventServices._tmpNbOfRetriesDone  +=1 ;
		}
	}

});

// Create Event Service.
ICTouchAPI.eventServices=new ICTouchAPI.eventServices();

dojo.provide("ICTouchAPI.eventServices.registeredEvent");
dojo.declare("ICTouchAPI.eventServices.registeredEvent",null,{


/* --------------------------------- Public attributes ------------------------------------ */

/* --------------------------------- Private attributes ----------------------------------- */

	// Name of the event
	/**
	 * @ignore
	 */
	_strEventName : "",
	// Applications that have subscribed to it
	/**
	 * @ignore
	 */
	_applications : {},
	// Number of applications using this event
	/**
	 * @ignore
	 */
	_nbWebapps : 0,

/* ------------------------------------ Constructor --------------------------------------- */

	/**
	 * @ignore
	 * Create a new event object
	 * @param {String} strEventName Name of the subscribed Event
	 * @return {Object} new event object
	 */
	constructor: function(strEventName){
		ICTouchAPI.log("Subscribing to " + strEventName, "EVENTS");
		// Saving the event name
		this._strEventName = strEventName;
		// Initializing attributes
		this._nbWebapps = 0;
		this._applications = {};
		// Subscribing to the event
                // No channel for simulation mode
                if(!generalConfig.simulation) {
                    this._coreEventSubscription();
                }
		return this;
	},

/* ----------------------------------- Getter / Setter------------------------------------- */

/* ----------------------------------- Public methods ------------------------------------- */

	/**
	 * Add a webApp to the current Event
	 * @ignore
	 * @param {Object} ctx
	 * @param {Object} callback
	 */
	addWebApp: function(ctx,callback){
		// Name of the application
		var name = ctx.getApplicationName();
		// If there's already a subscription
		if(this._applications[name]){
			// Unsubscribe the previous one
			dojo.unsubscribe(this._applications[name]);
		}
		// And replace it
		this._applications[name]=dojo.subscribe("eventServices." + this.getEventName(),ctx,callback);
		// Increase counter
		this._increaseNbWebapps();

		return this;
	},

	/**
	 * Remove a webApp to the current Event
	 * @ignore
	 * @param {String} webAppName
	 */
	removeWebApp: function(webAppName){
		// If there's such application
		if(this._applications[webAppName]){
			// Unsubscribe and delete its callback
			dojo.unsubscribe(this._applications[webAppName]);
			delete this._applications[webAppName];
			// If that was the last application using the event
			if (!this._decreaseNbWebapps()) {
				// Unsubscribe to this event so the eventServices won't receive it anymore
				this._coreEventUnsubscription();
			}
			return this;
        }
    },

/* --------------------------------- Private Methods -------------------------------------- */

	/**
	 * Subscribe to the event notification from the ICTGate
	 * @ignore
	 */
	_coreEventSubscription: function(){
		var xhrSubscribe = new XMLHttpRequest();
		xhrSubscribe.open('GET',
			ICTouchAPI.eventChannelURI + "services/ICTGate/subscribeToEvent?EventName={\"type\":\"QString\",\"value\":\""+this.getEventName()+"\"}&SocketId={\"type\":\"QString\",\"value\":\"0:0\"}",
			true
		);
		xhrSubscribe.send(null);
	},

	/**
	 * Unsubscribe to the event notification from the ICTGate
	 * @ignore
	 */
	_coreEventUnsubscription: function(){
		var xhrUnsubscribe = new XMLHttpRequest();
		xhrUnsubscribe.open('GET',
			ICTouchAPI.eventChannelURI + "services/ICTGate/unsubscribeToEvent?EventName={\"type\":\"QString\",\"value\":\""+this.getEventName()+"\"}&SocketId={\"type\":\"QString\",\"value\":\"0:0\"}",
			true
		);
		xhrUnsubscribe.send(null);
	},

	/**
	 * @ignore
	 */
	getEventName : function () {
		return this._strEventName;
	},

	/**
	 * @ignore
	 */
	_increaseNbWebapps : function () {
		return ++this._nbWebapps;
	},

	/**
	 * @ignore
	 */
	_decreaseNbWebapps : function () {
		return --this._nbWebapps;
	},

	/**
	 * @ignore
	 */
	getNbWebapps : function () {
		return this._nbWebapps;
	}

});
/**
* @class ICTouchAPI.settingServices
* @singleton
* @extends Object
* Manage the subscription to the Phone Settings
* If a subscribed setting changes, the new setting will be broadcasted to the subscriber (webapp)
* <pre><code>
* Possible setting types:
* - "TYPE_ENABLER"
* - "TYPE_BOOLEAN"
* - "TYPE_CHOICE"
* - "TYPE_TEXT"
* - "TYPE_NUMERIC"
* - "TYPE_FLOAT"
* - "TYPE_LIST"
* - "TYPE_IPADDRESS"
* - "TYPE_IPDOMAIN"
* - "TYPE_IPPORT"
* - "TYPE_PASSWORD"
* - "TYPE_DATE"
* - "TYPE_DATETIME"
* - "TYPE_TIME"
* - "TYPE_PHONE"
* - "TYPE_WEBAPP"
* </code></pre>
*/
dojo.provide("ICTouchAPI.settingServices");
dojo.declare("ICTouchAPI.settingServices",null,
{
	/* --------------------------------- Public attributes ------------------------------------ */

	// Define setting types (pseudo-constants)
	TYPE_ENABLER    : "ENABLER[WEBAPP]",
	TYPE_BOOLEAN	: "BOOLEAN",
	TYPE_CHOICE		: "CHOICE",
	TYPE_TEXT		: "TEXT",
	TYPE_NUMERIC	: "NUMERIC",
	TYPE_FLOAT		: "FLOAT",
	TYPE_LIST		: "LIST",
	TYPE_IPADDRESS	: "IP_ADDRESS",
	TYPE_IPDOMAIN	: "IP_DOMAIN",
	TYPE_IPPORT		: "IP_PORT",
	TYPE_PASSWORD	: "PASSWORD",
	TYPE_DATE		: "DATE",
	TYPE_DATETIME	: "DATETIME",
	TYPE_TIME		: "TIME",
	TYPE_PHONE		: "PHONE_NUMBER",
	TYPE_WEBAPP		: "CUSTOM[WEBAPP]",

	MODE_USER			: "user",
	MODE_ADMIN			: "admin",
	MODE_DEFAULTADMIN	: "defaultAdmin",
	strMode				: "user",

	boolCurrentSettingAdmin : false,

	/* --------------------------------- Private attributes ----------------------------------- */

	/**
	 * @ignore
	 */
	_arrSettingSequence 	:[],

	/**
	 * @ignore
	 */
	_arrSequenceValue	 	:[],

	/**
	 * @ignore
	 */
	_arrReturnedSettings	:[],

	/**
	 * @ignore
	 */
	_arrReturnedSetting	:[],

	/**
	 * @ignore
	 * Actions management
	 */
	_arrActions	: {},

	/**
	 * @ignore
	 * Cache for settings
	 */
	_arrCachedSettings: {},

	/**
	 * @ignore
	 * Cache list of settings identified by mainCat, category, userMode
	 * This gets filled by getSettings
	 */
	_arrCachedCategories: [],

	/**
	 * @ignore
	 * Handles returned from calls to dojo.subscribe() by WebApps are stored here (used for unsubscriptions)
	 */
	_arrHandles:{},

	/* ------------------------------------ Constructor --------------------------------------- */

        /**
         * @ignore
         */
	constructor : function() {
		// The settingServices subscribes to the "ChangedSettingEvent" event through eventServices
		ICTouchAPI.eventServices.subscribeToEvent(this, "ChangedSettingEvent", this._onSettingChanged);
		ICTouchAPI.eventServices.subscribeToEvent(this, "presentationLoaded", this._clearCachedCategories);
	},

	/* ----------------------------------- Getter / Setter------------------------------------- */


	/**
         * Get the name of the application service
         * @return {String} the name of the Service
         * @ignore
         */
	getApplicationName : function() {
		return "settingServices";
	},


	/* ----------------------------------- Public methods ------------------------------------- */

	/**
	 * Method used by webApplications to subscribe to a setting
	 * @param {Object} objWebApp webApp which wants to subscribe to a setting
	 * @param {String} strSettingName The name (Id) of the setting to subscribe
	 * @param {Function} callback function called when the setting occurs
	 */
	subscribeToSetting : function(objWebApp, strSettingName, callback) {
		var webAppName = objWebApp.getApplicationName();
		if (this._arrHandles[webAppName] == undefined) {
			this._arrHandles[webAppName] = {};
		}
		this._arrHandles[webAppName][strSettingName] = dojo.subscribe("settingServices:" + strSettingName, objWebApp, callback);
		objWebApp=null;
		strSettingName=null;
		callback=null;
	},

	/**
	 * Method used by webApplications to unsubscribe to a setting
	 * @param {Object} objWebApp webApp which wants to unsubscribe to setting
	 * @param {String} strSettingName The name (Id) of the setting to unsubscribe
	 */
	unsubscribeToSetting : function(objWebApp, strSettingName) {
		var webAppName = objWebApp.getApplicationName();
		if (this._arrHandles[webAppName][strSettingName] != undefined) {
			dojo.unsubscribe(this._arrHandles[webAppName][strSettingName]);
			delete this._arrHandles[webAppName][strSettingName];
		}
	},

	/**
	 * Method used by webApplications to get all settings from a specific category of a main category
	 * @param {String} mainCat name of the main category
	 * @param {String} category name of the category
	 * @param {String} userMode user mode
	 * @param {Object} context context of the calling webApp
	 * @param {Function} callback function to be called as a callback in the webApp
	 * @param {Function} errorCallback function to be called as a error callback in the webApp
	 * @param {Object} callbackParams object containing parameters to be used by the callback
	 */
	getSettings : function(mainCat, category, userMode, context, callback, errorCallback, callbackParams) {
		ICTouchAPI.log("getSettings: " + mainCat + ", " + category + ", " + userMode, "SETTINGS");
		var categoryListId = mainCat+"_"+category+"_"+userMode;
		var params = {
				context: context,
				callback:callback,
				errorCallback:errorCallback,
				callbackParams:callbackParams,
				categoryListId: categoryListId,
				alreadyInCache: false
		};
		if (this._arrCachedCategories[categoryListId]) {
				// Settings are in cache, do not fetch them from SettingsManager
				params.alreadyInCache = true;
				var service = this;
				setTimeout(function(){
						// Simulate asynchronous call
						service._gotSettings(service._arrCachedCategories[categoryListId], params);
				}, 50);
		}
		else {
				// Get settings from business
				ICTouchAPI.APIServices.SettingsManager.getSettings({
						params:[mainCat, category, userMode],
						context:this,
						callback:this._gotSettings,
						errorCallback:this._notGotSettings,
						callbackParams: params
				});
		}
	},

	/**
	 * Method used by webApplications to get a setting of type sequence by its id
	 * @param {String} id id (name) of the setting
	 * @param {Object} context context of the calling webApp
	 * @param {Function} callback function to be called as a callback in the webApp
	 * @param {Object} callbackParams object containing parameters to be used by the callback
	 */
	getSettingSequence : function (id, context, callback, callbackParams) {
		ICTouchAPI.log("getSetting: " + id, "SETTINGS");
		if (this._arrCachedSettings[id]==undefined) {
			ICTouchAPI.APIServices.SettingsManager.getSetting({
				params:[id],
				context:this,
				callback:this._gotSettingSequence,
				callbackParams:{
					context: context,
					callback: callback,
					callbackParams: callbackParams
				}
			});
		}
		else {
			if (callback != undefined) {
				callback.apply(context, [this._arrCachedSettings[id], callbackParams]);
			}
		}
	},

	/**
	 * Add an action managment, one action will lauch one function
	 * @param {String} actionName action Name defined in the action field of the setting
	 * @param {Object} method method which will be called for this action
	 */
	addActionMethod : function(actionName, method){
		this._arrActions[actionName] = method;
	},

	/**
	 * Get the corresponding method for an action
	 * @param {String} actionName action Name used
	 * @return {Object} the corresponding function
	 */
	getMethodForAction : function(actionName){
		return this._arrActions[actionName];
	},

	/**
	 * Will call the right function corresponding to the action with the good parameters (in case of core method)
	 * @param {String} paramCallback the parameter given by the callback by UI
	 * @param {Object} paramAction parameters given manually
	 */
	callMethodForAction : function(paramCallback, paramAction){
		var func = this.getMethodForAction(paramAction.action)
		func({
			params:[paramCallback]
		});
	},


	/**
	 * Method used by webApplications to get a setting by its id
	 * @param {String} id (name) of the setting
	 * @param {Object} context of the calling webApp
	 * @param {Function} callback function to be called as a callback in the webApp
	 * @param {Object} callbackParams object containing parameters to be used by the callback
	 * @param {boolean} synchro used to define a synchronized xhr request
	 */
	getSetting : function (id, context, callback, callbackParams, synchro) {
		ICTouchAPI.log("getSetting: " + id, "SETTINGS");
		if (id.split(".")[0] == "awap" && webapp && webapp.myICWall) {
			webapp.myICWall.getSetting(id, context, callback, callbackParams, synchro);
		}
		else {
			var namespace = (context && context.declaredClass) ? context.declaredClass.split(".") : null;
			if (namespace && namespace[0] == "awap" && webapp && webapp.myICWall) {
				webapp.myICWall.getSetting(id, context, callback, callbackParams, synchro);
			}
			else {
		if (this._arrCachedSettings[id]==undefined || this._arrSettingSequence[id]) {
			ICTouchAPI.APIServices.SettingsManager.getSetting({
				params:[id],
				context:this,
				callback:this._gotSetting,
				callbackParams:{
					context: context,
					callback: callback,
					callbackParams: callbackParams
				},
				sync : (synchro ? synchro : false)
			});
		}
		else {
			if (callback != undefined) {
				callback.apply(context, [this._arrCachedSettings[id], callbackParams]);
			}
		}
			}
		}
		id=null;
		context=null;
		callback=null;
		callbackParams=null;
		synchro=null;
	},

	/**
	 * Method used by webApplications to get the value of a setting by its id
	 * @param {String} id (name) of the setting
	 * @param {Object} context of the calling webApp
	 * @param {Function} callback function to be called as a callback in the webApp
	 * @param {Object} callbackParams object containing parameters to be used by the callback
	 * @param {boolean} synchro used to define a synchronized xhr request
	 */
	getSettingValue : function (id, context, callback, callbackParams, synchro) {
		ICTouchAPI.log("getSettingValue: " + id, "SETTINGS");
		if (id.split(".")[0] == "awap" && webapp && webapp.myICWall) {
			webapp.myICWall.getSettingValue(id, context, callback, callbackParams, synchro);
		}
		else {
			var namespace = (context && context.declaredClass) ? context.declaredClass.split(".") : null;
			if (namespace && namespace[0] == "awap" && webapp && webapp.myICWall) {
				webapp.myICWall.getSettingValue(id, context, callback, callbackParams, synchro);
			}
			else {
		if (this._arrCachedSettings[id]==undefined) {
			ICTouchAPI.APIServices.SettingsManager.getSetting({
				params:[id],
				context:this,
				callback:this._gotSettingValue,
				callbackParams:{
					context: context,
					callback: callback,
					callbackParams: callbackParams
				},
				sync : (synchro ? synchro : false)
			});
		}
		else {
			if (callback != undefined) {
				callback.apply(context, [this._arrCachedSettings[id].jsValue, callbackParams]);
			}
		}
			}
		}
		id=null;
		context=null;
		callback=null;
		callbackParams=null;
		synchro=null;
	},

	/**
	 * Method used to get a list of settings from their ids
	 * The settings are obtained recursively
	 * @param {Array} ids list of settings ids
	 * @param {Object} context context of the calling webApp
	 * @param {Function} callback function to be called as a callback in the webApp
	 * @param {Object} callbackParams object containing parameters to be used by the callback
	 */
	getSettingsFromIds : function (ids, context, callback, callbackParams) {
		var objContext = {
			index: 0,
			ids: ids,
			results: [],
			context: context,
			callback: callback,
			callbackParams: callbackParams
		};
		this._getNextSettingFromId(objContext);
	},

	/**
	 * Method used by webApplications to set a new value to a setting
	 * @param {String} id (name) of the setting
	 * @param {String/Boolean/Number} jsValue new value which must be set in the setting
	 * @param {Object} context of the calling webApp
	 * @param {Function} callback function to be called as a callback in the webApp
	 * @param {Object} callbackParams object containing parameters to be used by the callback
	 */
	setSettingValue : function (id, jsValue, context, callback, callbackParams) {
		ICTouchAPI.log("setSettingValue: " + id, "SETTINGS");
		ICTouchAPI.logObject(id, "SETTINGS");
		if (id.split(".")[0] == "awap" && webapp && webapp.myICWall) {
			webapp.myICWall.setSettingValue(id, jsValue, context, callback, callbackParams);
		}
		else {
			var namespace = (context && context.declaredClass) ? context.declaredClass.split(".") : null;
			if (namespace && namespace[0] == "awap" && webapp && webapp.myICWall) {
				webapp.myICWall.setSettingValue(id, jsValue, context, callback, callbackParams);
			}
			else {
		if (this._arrCachedSettings[id] != undefined ) {
			if(this._arrCachedSettings[id].admin_rights == 2 && this._arrCachedSettings[id].user_rights != 2 && this.strMode != this.MODE_ADMIN){
				this._showAdminKeyboardForSetting({
					id : id,
					jsValue : jsValue,
					context : context,
					callback : callback,
					callbackParams : callbackParams
				});
			}
			if (this._isNewValueValidForSetting(this._arrCachedSettings[id], jsValue)) {
				// The setting must be cloned, so that the cache is NOT updated until we receive a ChangedSettingEvent.
				// In this way the setting in cache will not be modified if the validation fails on business side.
				var clonedSetting = dojo.clone(this._arrCachedSettings[id]);
				//SEQUENCE
				//if the setting is in the depencies array
				var id_sequence = this._arrSettingSequence[id];
				if(id_sequence){
					//we get the string value to send
					var newVal = this._arrSequenceValue[id_sequence];
					//we create a regexp to find the setting to be set up in the string which will be send to setting manager
					var forUpdate = new RegExp(id+"(=[^(\|;\|)]+){0,1}", "g");
					//we will replace the previous setting value by the new written one
					var strReplace = id+"="+jsValue;
					//we update the string
					newVal = ""+newVal; //without this line str.replace seems to fail for newVal
					newVal = newVal.replace(forUpdate,id+"="+jsValue);
					//The complete sequence value
					jsValue = newVal;
					//The id of the sequence
					id = id_sequence;
					//we save the modified sequence string value so that even the new written value will not be lost if we fail and be sent
					//on the next try
					this._arrSequenceValue[id_sequence] = newVal;
				}
				//END OF SEQUENCE
				clonedSetting.jsValue = jsValue;
				clonedSetting = this._formatSettingToICTGate(clonedSetting);
				ICTouchAPI.APIServices.SettingsManager.setSettingValue({
					params:[id, clonedSetting.value],
					context:this,
					callback:this._setSettingValueDone,
					callbackParams:{
						context: context,
						callback: callback,
						callbackParams: callbackParams
					}
				});
			}
			else {
				if (callback != undefined) {
					callback.apply(context, [false, callbackParams]);
				}
			}
			if (this.boolCurrentSettingAdmin){
				this.boolCurrentSettingAdmin = false;
				this.strMode = this.MODE_USER;
				ICTouchAPI.APIServices.ICTGate.logout();
			}
		}
		else {
			ICTouchAPI.APIServices.SettingsManager.getSetting({
				params:[id],
				context:this,
				callback:this._gotSettingSetSettingValue,
				callbackParams:{
					jsValue: jsValue,
					context: context,
					callback: callback,
					callbackParams: callbackParams
				}
			});
		}
			}
		}
		id=null;
		jsValue=null;
		context=null;
		callback=null;
		callbackParams=null;
	},

	/**
	 * Method called to know if a CURRENTLY displayed setting is from a SEQUENCE
	 * @param {String} id (name) of the setting
	 * @return {Boolean} if the setting belongs to a shown SEQUENCE
	 */
	isDisplayedSequence : function(id){
		if(this._arrSettingSequence[id]){
			return true;
		}
		return false;
	},

	/**
	 * Method called to know the id of SEQUENCE of a currently displayed setting
	 * @param {String} id (name) of the setting
	 * @return {Boolean} id of the SEQUENCE
	 */
	getDisplayedSequence : function(id){
		return this._arrSettingSequence[id];
	},

	/**
	 * Method used to build and return a typeData object depending on the setting type
	 * @param {Object} objSetting setting object
	 * @return {Object} objType
	 */
	getSettingTypeData : function(objSetting) {
		// CHOICE
		if (objSetting.type == this.TYPE_CHOICE)
			return {
				name: this.TYPE_CHOICE,
				params: null
			};
		// IPADDRESS
		if (objSetting.type == this.TYPE_IPADDRESS)
			return {
				name: this.TYPE_IPADDRESS,
				params: null
			};
		// IPDOMAIN
		if (objSetting.type == this.TYPE_IPDOMAIN)
			return {
				name: this.TYPE_IPDOMAIN,
				params: null
			};
		// DATE || DATETIME
		if (objSetting.type == this.TYPE_DATE || objSetting.type == this.TYPE_DATETIME)
			return {
				name: objSetting.type,
				params: null
			};
		// TIME
		if (objSetting.type == this.TYPE_TIME)
			return {
				name: this.TYPE_TIME,
				params: null
			};
		// PHONE
		if (objSetting.type == this.TYPE_PHONE)
			return {
				name: this.TYPE_PHONE,
				params: null
			};
		if (objSetting.type == this.TYPE_WEBAPP)
			return {
				name: this.TYPE_WEBAPP,
				params: null
			};
		var objType = {
			name: "",
			params: null
		};
		// BOOLEAN[name1,name2]
		var regexp = new RegExp("^"+this.TYPE_BOOLEAN+"(\\[(.+),(.+)\\])?", "i");
		var res = regexp.exec(objSetting.type);
		// "res" contains:
		//  0 => whole matching content if exists
		//  1 => whole type parameters if exists
		//  2 => first parameter
		//  3 => second parameter
		if (res && res[0]) {
			objType.name = this.TYPE_BOOLEAN;
			objType.params = {
				name1: "false",
				name2: "true"
			};
			if (res[1]) {
				objType.params.name1 = res[2];
				objType.params.name2 = res[3];
			}
			return objType;
		}

		// ENABLER[WEBAPP]
		if (objSetting.type == this.TYPE_ENABLER) {
			return {
				name: this.TYPE_ENABLER,
				params: {
					name1: "false",
					name2: "true"
				}
			};
		}

		// TIME[stepHours:stepMinutes:stepSeconds]
		regexp = new RegExp("^"+this.TYPE_TIME+"(\\[([^,]+):([^,]+):([^,]+)\\])?", "i");
		res = regexp.exec(objSetting.type);
		if (res && res[0]) {
			objType.name = this.TYPE_TIME;
			objType.params = {
				stepHours: null,
				stepMinutes: null,
				stepSeconds: null
			};
			if (res[1] ) {
				objType.params.stepHours = parseInt(res[2]);
				objType.params.stepMinutes = parseInt(res[3]);
				objType.params.stepSeconds = parseInt(res[4]);
			}
			return objType;
		}
		// NUMERIC[min,max,step]
		regexp = new RegExp("^"+this.TYPE_NUMERIC+"(\\[([^,]+),([^,]+)(,([^,]+))?\\])?", "i");
		res = regexp.exec(objSetting.type)
		if (res && res[0]) {
			objType.name = this.TYPE_NUMERIC;
			objType.params = {
				min: undefined,
				max: undefined,
				step: 1
			};
			if (res[1]) {
				if (res[2] != "*") objType.params.min = parseInt(res[2]);
				if (res[3] != "*") objType.params.max = parseInt(res[3]);
				if (res[4])
					objType.params.step = res[5];
			}
			return objType;
		}
		// IPPORT[min,max]
		regexp = new RegExp("^"+this.TYPE_IPPORT+"(\\[([^,]+),([^,]+)\\])?", "i");
		res = regexp.exec(objSetting.type)
		if (res && res[0]) {
			objType.name = this.TYPE_IPPORT;
			objType.params = {
				min: undefined,
				max: undefined
			};
			if (res[1]) {
				objType.params.min = parseInt(res[2]);
				objType.params.max = parseInt(res[3]);
			}
			return objType;
		}
		// FLOAT[min,max,step]
		regexp = new RegExp("^"+this.TYPE_FLOAT+"(\\[([^,]+),([^,]+)(,([^,]+))?\\])?", "i");
		res = regexp.exec(objSetting.type)
		if (res && res[0]) {
			objType.name = this.TYPE_FLOAT;
			objType.params = {
				min: undefined,
				max: undefined,
				step: 1
			};
			if (res[1]) {
				if (res[2] != "*") objType.params.min = res[2];
				if (res[3] != "*") objType.params.max = res[3];
				if (res[4])
					objType.params.step = res[5];
			}
			return objType;
		}
		// TEXT[min,max]
		regexp = new RegExp("^"+this.TYPE_TEXT+"(\\[([^,]+),([^,]+)\\])?", "i");
		res = regexp.exec(objSetting.type)
		if (res && res[0]) {
			objType.name = this.TYPE_TEXT;
			objType.params = {
				min: undefined,
				max: undefined
			};
			if (res[1]) {
				objType.params.min = res[2];
				objType.params.max = res[3];
			}
			return objType;
		}
		// PASSWORD[min,max]
		regexp = new RegExp("^"+this.TYPE_PASSWORD+"(\\[([^,]+),([^,]+)\\])?", "i");
		res = regexp.exec(objSetting.type)
		if (res && res[0]) {
			objType.name = this.TYPE_PASSWORD;
			objType.params = {
				min: undefined,
				max: undefined
			};
			if (res[1]) {
				objType.params.min = res[2];
				objType.params.max = res[3];
			}
			return objType;
		}
		// LIST[type]
		regexp = new RegExp("^"+this.TYPE_LIST+"(\\[(.+)\\])?", "i");
		res = regexp.exec(objSetting.type)
		if (res && res[0]) {
			objType.name = this.TYPE_LIST;
			objType.params = {
				listType: undefined
			};
			if (res[1])
				objType.params.listType = res[2];
			return objType;
		}
		return objType;
	},

	/* ----------------------------------- Private methods ------------------------------------- */

	/**
	 * @ignore
	 * Method called when a ChangedSettingEvent is triggered by eventServices
	 * @param {Object} params object returned with the event, params.value is the new setting in this case
	 */
	_onSettingChanged : function(params) {
		ICTouchAPI.log("Setting changed", "SETTINGS");
		ICTouchAPI.logObject(params, "SETTINGS");
		this._formatToJSAndCacheSetting(params.value);
		this._publishChangedSettingEvent(params.value.id, this._arrCachedSettings[params.value.id]);
	},

	/**
	 * @ignore
	 * Method called when a presentationLoaded event is triggered by eventServices
	 */
	_clearCachedCategories : function() {
		this._arrCachedCategories = [];
	},

	/**
	 * @ignore
	 * Method used as a callback to the getSettings() function
	 * @param {Array} arrSettings array containing returned settings
	 * @param {Object} callbackParams object containing webApp's context, callback and callbackParams
	 */
	_gotSettings : function(arrSettings, callbackParams) {
		//if we actually got settings
		if (arrSettings != undefined) {
			if (!callbackParams.alreadyInCache) {
				//we make them readable
				for (var i in arrSettings) {
					this._formatToJSAndCacheSetting(arrSettings[i]);
				}
				// Cache the result
				this._arrCachedCategories[callbackParams.categoryListId] = arrSettings;
			}
			//we initialize the setting array to be sent
			this._arrReturnedSettings = [];
			//we start an asychron chain of calls to the settingManager in order to build the right setting array
			//to be sent to the webapp setting .. else a solution using for loops would send an incomplet array due to
			//the asychron responses from the setting manager
			this._sequenceManagement(0,arrSettings.length,arrSettings,this._arrReturnedSettings,callbackParams);
		}
	},

	/**
	 * @ignore
	 * the function call itself in order to send the right settigns
	 * actually we got 2 chains :
	 * - one for each setting
	 * - an other one on the sequences
	 * the reason is that, as long as we don't get the true setting form the setting manager we cannot
	 * correctly build the _arrReturnedSettings array ... so we shouldn't send it to the setting webapp.
	 */
	_sequenceManagement : function(start,end,arrSettings,arrReturned,callbackParams){
		//once we have treated every setting
		if(start >= end){
			//we try to call the callback sent on the very first call from the setting's webapp
			if (callbackParams.callback != undefined) {
				// Call callback function with stored context and input parameters
				callbackParams.callback.apply(callbackParams.context, [arrReturned, callbackParams.callbackParams]);
			}
		}else{
			//else we have a setting to check ...
			//if we find a SEQUENCE
			var regExpType = new RegExp("SEQUENCE*");
			var isSequence = regExpType.test(arrSettings[start].type);
			if(isSequence){
				//get the sequence id
				var id_sequence = arrSettings[start].id;
				//we save the generic string value to send when setting a setting's value
				this._arrSequenceValue[id_sequence] = [arrSettings[start].value];
				//we get all the setting IDs and their value
				var ids_values = arrSettings[start].value.split("|;|");
				//we call the function which add the sequence with the depency array and call the true setting out from the setting manager
				this._gotSequence(	id_sequence,
					0,
					ids_values.length,
					ids_values,
					arrSettings[start].value,
					start,
					end,
					arrSettings,
					arrReturned,
					callbackParams
					);
			}else{
				//this is everything but a SEQUENCE, so we can simply add it to the array holding the settings
				arrReturned.push(arrSettings[start]);
				this._sequenceManagement(start + 1,end,arrSettings,arrReturned,callbackParams);
			}
		}
	},

   /**
	 * @ignore
	 * this function will :
	 * - add the dependency
	 * - call the setting manager
	 * - call itself through the callback function sent to the setting manager, this a way to synchronize the SettingManager's asynchron behavior
	 * so that the _arrSettingUIDependence array is actually fiilled in before to move further
	 */
	_gotSequence : function(id_sequence,start_ids,end_ids,ids_values,full_value,start,end,arrSettings,arrReturned,callbackParams){
		//once all the sequence has been treated
		if(start_ids >= end_ids){
			//we check the next setting
			this._sequenceManagement(start + 1,end,arrSettings,arrReturned,callbackParams);
		}else{
			//the SEQUENCE is not over yet
			var key_val = ids_values[start_ids].split("=");
			this._arrSettingSequence[key_val[0]] = id_sequence;
			var that = this;
			//add the real setting within the returned settings array
			var func = function(objSetting){
				objSetting.user_rights = that._arrCachedSettings[id_sequence].user_rights;
				objSetting.admin_rights = that._arrCachedSettings[id_sequence].admin_rights;
				//we add the correct true setting
				arrReturned.push(
					objSetting
					);
				//we call our caller on the next setting of the sequence, the caller will check if we can move
				//on the next setting or if some sequence's settings have not been treated yet
				this._gotSequence(	id_sequence,
					start_ids + 1,
					end_ids,
					ids_values,
					full_value,
					start,
					end,
					arrSettings,
					arrReturned,
					callbackParams
					);
			};
			this.getSetting(key_val[0],this,func);
		}
	},

	/**
	 * @ignore
	 * Method used as a callback to the getSetting() function
	 * @param {Object} objSetting returned setting
	 * @param {Object} callbackParams object containing webApp's context, callback and callbackParams
	 */
	_gotSettingSequence : function (objSetting, callbackParams) {
		if (objSetting.exist)
		{
			this._formatToJSAndCacheSetting(objSetting);
			//set objSetting in an array
			this._arrReturnedSetting	= [];
			var tmpArray = [objSetting];
			//call sequenceManagement
			this._sequenceManagement(0,tmpArray.length,tmpArray,this._arrReturnedSetting,callbackParams);
		}
		else if (callbackParams.callback != undefined) {
			// Call callback function with stored context and input parameters
			callbackParams.callback.apply(callbackParams.context, [null, callbackParams.callbackParams]);
		}
	},

	/**
	 * @ignore
	 * Method used as error callback to the getSettings() function
	 * @param {String} strErrorMsg error message from business  layer
	 * @param {Object} callbackParams object conaining webapps's context, callbacks
	 *
	 **/
	_notGotSettings : function(strErrorMsg, callbackParams) {
		// only call the error callback
		if (callbackParams.errorCallback != undefined) {
			// Call callback function with stored context and input parameters
			callbackParams.errorCallback.apply(callbackParams.context, [strErrorMsg, callbackParams.callbackParams]);
		}
	},

	/**
	 * @ignore
	 * Method used as a callback to the getSetting() function
	 * @param {Object} objSetting returned setting
	 * @param {Object} callbackParams object containing webApp's context, callback and callbackParams
	 */
	_gotSetting : function (objSetting, callbackParams) {
		if (objSetting.exist)
		{
			this._formatToJSAndCacheSetting(objSetting);
			if (callbackParams.callback != undefined) {
				// Call callback function with stored context and input parameters
				try{
					callbackParams.callback.apply(callbackParams.context, [this._arrCachedSettings[objSetting.id], callbackParams.callbackParams]);
				}catch(err){
					console.warn("settingServices._gotSetting: your callback function made an error listening setting = " + dojo.toJson(this._arrCachedSettings[objSetting.id]));
				}
			}
		}
		else if (callbackParams.callback != undefined) {
			// Call callback function with stored context and input parameters
			callbackParams.callback.apply(callbackParams.context, [null, callbackParams.callbackParams]);
		}
	},

	/**
	 * Method used as a callback to the getSettingValue() function
	 * @param {Object} objSetting returned setting
	 * @param {Object} callbackParams object containing webApp's context, callback and callbackParams
         * @ignore
	 */
	_gotSettingValue : function (objSetting, callbackParams) {
		if (objSetting.exist)
		{
			this._formatToJSAndCacheSetting(objSetting);

			if (callbackParams.callback != undefined) {
				// Call callback function with stored context and input parameters
				callbackParams.callback.apply(callbackParams.context, [this._arrCachedSettings[objSetting.id].jsValue, callbackParams.callbackParams]);
			}
		}
		else if (callbackParams.callback != undefined) {
			// Call callback function with stored context and input parameters
			callbackParams.callback.apply(callbackParams.context, [null, callbackParams.callbackParams]);
		}
	},

	/**
	 * @ignore
	 * Private recursive function; used to get one setting, consecutive to a call of getSettingsFromIds
	 */
	_getNextSettingFromId : function (objContext) {
		if (objContext.ids[objContext.index] === undefined) {
			// end of recursive call
			if (objContext.callback != undefined) {
				// Call callback function with stored context and input parameters
				objContext.callback.apply(objContext.context, [objContext.results, objContext.callbackParams]);
			}
			return;
		}
		this.getSetting(objContext.ids[objContext.index], this, this._gotNextSettingFromId, objContext);
	},

	/**
	 * @ignore
	 * Private function; callback that is called as a result of _getNextSettingFromId.
	 */
	_gotNextSettingFromId : function (objSetting, objContext) {
		// The setting is pushed into results array, index is incremented and we continue recursive call.
		objContext.results[objContext.ids[objContext.index]] = objSetting;
		objContext.index++;
		this._getNextSettingFromId(objContext);
	},

	/**
	 * @ignore
	 * Method used as a callback to the setSettingValue() function
	 * when the Setting needed to be retrieved from ICTGate because it was not in cache
	 * @param {Object} objSetting returned setting
	 * @param {Object} callbackParams object containing webApp's context, callback and callbackParams
	 */
	_gotSettingSetSettingValue : function(objSetting, callbackParams) {
		if (objSetting.exist) {
			this._formatToJSAndCacheSetting(objSetting);
			this.setSettingValue(objSetting.id, callbackParams.jsValue);
		}
		else {
			if (callbackParams.callback != undefined) {
				callbackParams.callback.apply(callbackParams.context, [false, callbackParams.callbackParams]);
			}
		}
	},

	/**
	 * @ignore
	 */
	_showAdminKeyboardForSetting : function(setting){
		var strUserMode = this.MODE_ADMIN;
		var funcCallback = this._logAdmin;
		var cxContext = this;
		ICTouchAPI.keyboardServices.deployKeyboard(
			ICTouchAPI.keyboardServices.TYPE_PASSWORD,
			null,
			{
				funcCallbackOk : function(password) {
					// Try login
					ICTouchAPI.APIServices.ICTGate.login({
						params:[strUserMode, password],
						context:cxContext,
						callback:funcCallback,
						callbackParams:setting
					});
				}
			}
			);
	},

	/**
	 * @ignore
	 */
	_logAdmin : function(strMsgLogin,setting){
		if (strMsgLogin == "Login Ok"){
			this.strMode = this.MODE_ADMIN;
			this.boolCurrentSettingAdmin = true;
			this.setSettingValue(setting.id, setting.jsValue, setting.context, setting.callback, setting.callbackParams);
		}
	},

	/**
	 * @ignore
	 * Method used as a callback to the setSettingValue() function
	 * @param {Object} result object representing the value of the UpdateReport returned by the SettingsManager
	 * @param {Object} callbackParams object containing webApp's context, callback and callbackParams
	 */
	_setSettingValueDone : function(result, callbackParams) {
		//callback to manage the return of needReboot
		var func = function(ret){
			//if needReboot send a boolean set to TRUE we set the rebootRequired variable to true
			if (typeof webapp.settings.data === "object") {
				webapp.settings.data.setReboot(ret);
			}
		}

		//we ask the platform whether we have to reboot or not
		this.needReboot({
			context : this,
			callback : func
		});

		//the result arg doesn't seem to be well passed
		if (callbackParams.callback != undefined) {
			callbackParams.callback.apply(callbackParams.context, [result, callbackParams.callbackParams]);
		}
	},

	/**
	 * @ignore
	 * Method used to determine if a new value is valid for a Setting object
	 * @param {Object} objSetting setting to check with new value
	 * @param {String/Boolean/Number} newValue jsValue to test for the Setting
	 * @return {Boolean} valid true or false
	 */
	_isNewValueValidForSetting : function(objSetting, newValue) {
		var valid = false;
		if (objSetting.typeData == undefined) {
			objSetting.typeData = this.getSettingTypeData(objSetting);
		}
		// Note: type "PHONE" is not validated from MMI.
		switch (objSetting.typeData.name) {
                        case this.TYPE_ENABLER:
			case this.TYPE_BOOLEAN:
				if (typeof newValue == "boolean") {
					valid = true;
				}
				break;
			case this.TYPE_NUMERIC:
				if (typeof newValue == "number") {
					if ((objSetting.typeData.params.min == undefined || newValue >= objSetting.typeData.params.min)
						&& (objSetting.typeData.params.max == undefined || newValue <= objSetting.typeData.params.max)) {
						valid = true;
					}
				}
				break;
			case this.TYPE_IPPORT:	// same as NUMERIC
				if (typeof newValue == "number") {
					if ((objSetting.typeData.params.min == undefined || newValue >= objSetting.typeData.params.min)
						&& (objSetting.typeData.params.max == undefined || newValue <= objSetting.typeData.params.max)) {
						valid = true;
					}
				}
				break;
			case this.TYPE_CHOICE:
				// The jsValue of a "CHOICE" setting must be the index of the value in the allowedValues array
				if (parseInt(newValue) >= 0 && parseInt(newValue) < objSetting.allowedValues.length) {
					valid = true;
				}
				break;
			case this.TYPE_IPADDRESS: {
				var ipAddressNumbers = newValue.split(".");
				if (ipAddressNumbers.length == 4) {
					valid = true;
					for (var i = 0; i < ipAddressNumbers.length; i++) {
						if (!(parseInt(ipAddressNumbers[i],10) >= 0 && parseInt(ipAddressNumbers[i],10) <= 255)) {
							valid = false;
						}
					}
				}
				break;
			}
			case this.TYPE_DATE: // Breakthrough
			case this.TYPE_DATETIME:
				valid = ( newValue instanceof Date );
				break;
			default : {
				valid = true;
				break;
			}
		}
		return valid;
	},

	/**
	 * @ignore
	 * Method used to format a Setting object to JS format (add jsValue and typeData attributes to object)
	 * After formatting, the Setting object is stored into the _arrCachedSettings array
	 * @param {Object} objSetting setting object
	 */
	_formatToJSAndCacheSetting : function(objSetting) {

		// Add "typeData" object to objSetting
		objSetting.typeData = this.getSettingTypeData(objSetting);
		if (objSetting.isUndefined === 0) {
			// Switch in case of jsValue must be different from value
			switch (objSetting.typeData.name) {
                                case this.TYPE_ENABLER:
				case this.TYPE_BOOLEAN:
					if (objSetting.value == "true") {
						objSetting.jsValue = true;
					}
					else if (objSetting.value == "false") {
						objSetting.jsValue = false;
					}
					break;
				case this.TYPE_NUMERIC:
					objSetting.jsValue = parseInt(objSetting.value);
					break;
				case this.TYPE_IPPORT:
					objSetting.jsValue = parseInt(objSetting.value);
					break;
				case this.TYPE_LIST:
					if(objSetting.value == "")
						objSetting.jsValue = [];
					else
						objSetting.jsValue = objSetting.value.split(";");
					break;
				case this.TYPE_CHOICE:
					objSetting.jsValue = -1;
					for (var i=0; i<objSetting.allowedValues.length; i++) {
						if (objSetting.allowedValues[i].id == objSetting.value) {
							objSetting.jsValue = i;
							break;
						}
					}
					break;
				case this.TYPE_DATE: // Breakthrough
				case this.TYPE_DATETIME:
					objSetting.jsValue = new Date();
					objSetting.jsValue.setTime(objSetting.value*1000);
					break;
				case this.TYPE_TEXT:
					objSetting.jsValue = objSetting.value;
					break;
				default:
					objSetting.jsValue = objSetting.value;
					break;
			}
		}
		else
		{
			// "UNDEFINED"
			objSetting.jsValue = objSetting.value;
		}
		// always cache it
		// objSetting is now cached in _arrCachedSettings
		this._arrCachedSettings[objSetting.id] = objSetting;

		// Also update category list with the new value, both user and admin
		this._updateSettingCategory(objSetting, objSetting.mainCategory+"_"+objSetting.category+"_"+this.MODE_USER);
		this._updateSettingCategory(objSetting, objSetting.mainCategory+"_"+objSetting.category+"_"+this.MODE_ADMIN);

		ICTouchAPI.log("Setting cached: " + objSetting.id, "SETTINGS");
	},

	/**
	 * @ignore
	 * Update the setting in this category
	 */
	_updateSettingCategory: function(objSetting, categoryListId) {
		if (this._arrCachedCategories[categoryListId]) {
				var objCategory = this._arrCachedCategories[categoryListId];
				var index;
				for(index in objCategory) {
						if (objCategory[index].id === objSetting.id) {
								objCategory[index] = objSetting;
						}
				}
		}
	},

	/**
	 * @ignore
	 * Method used to format a Setting object to ICTGate format (remove jsValue and typeData attributes to object)
	 * The input setting is cloned before being modified
	 * @param {Object} inputSetting Setting object to format for ICTGate
	 * @return {Object} objSetting Setting object formatted for ICTGate (cloned from input)
	 */
	_formatSettingToICTGate : function(inputSetting) {
		var objSetting = dojo.clone(inputSetting);

		switch (objSetting.typeData.name) {
                        case this.TYPE_ENABLER:
			case this.TYPE_BOOLEAN:
				if (objSetting.jsValue == true) {
					objSetting.value = "true";
				} else if (objSetting.jsValue == false) {
					objSetting.value = "false";
				}
				break;
			case this.TYPE_NUMERIC:
				objSetting.value = objSetting.jsValue+'';
				break;
			case this.TYPE_IPPORT:
				objSetting.value = objSetting.jsValue+'';
				break;
			case this.TYPE_CHOICE:
				objSetting.value = objSetting.allowedValues[objSetting.jsValue].id
				break;
			case this.TYPE_LIST:
				objSetting.value = objSetting.jsValue.join(";");
				break;
			case this.TYPE_DATE: // Breakthrough
			case this.TYPE_DATETIME:
				objSetting.value = parseInt(objSetting.jsValue.getTime()/1000, 10);
				break;
			case this.TYPE_TEXT:
				objSetting.value = objSetting.jsValue;
				break;
			default:
				objSetting.value = objSetting.jsValue;
				break;
		}
		delete objSetting.typeData;
		delete objSetting.jsValue;
		// Value may have changed, so also update it in input setting to keep the cache consistent
		inputSetting.value = objSetting.value;
		return objSetting;
	},

	/**
	 * @ignore
	 * Method used to broadcast to new setting provided by the ChangedSettingEvent to all subscribed webApps
	 * @param {String} strSettingName id (name) of the concerned setting = topic of dojo.subscribe/dojo.publish
	 * @param {Object} objSetting new setting
	 */
	_publishChangedSettingEvent : function(strSettingName, objSetting) {
		dojo.publish("settingServices:" + strSettingName, [objSetting]);
	},

	/**
     * Called to check weither we upgrade some settings before to enter in the webapp setting
     * the platform lock the webapp acces
     * @ignore
     */
	enterMMI : function(args) {
		ICTouchAPI.APIServices.SettingsManager.enterMMI(args);
	},

	/**
     * send an event expected by the settingManager to leave the webapp
     * @ignore
     */
	exitMMI : function() {
		ICTouchAPI.APIServices.SettingsManager.exitMMI();
	},

	/**
     * @ignore
     */
	needReboot : function(args){
		ICTouchAPI.APIServices.SettingsManager.needReboot(args);
	}
});

// Create Setting Service.
ICTouchAPI.settingServices=new ICTouchAPI.settingServices;
/**
* @class ICTouchAPI.dataStoreServices
* @extends Object
* @singleton
* The DataStore class allows to get cache set of object records
*/
dojo.provide("ICTouchAPI.dataStoreServices");
dojo.declare("ICTouchAPI.dataStoreServices", null, {


	/**
	 * @ignore
	 */
	_publicStores : {},

	/**
	 * @ignore
	 */
	constructor : function () {

	},



	/**
	 * Get a reference on a Store object named 'name'
	 * @param {String} name The name of the data store
	 * @return {ICTouchAPI.Store} A reference on a Store object named 'name'
	 */
	getStore : function (name) {
		var store = this._publicStores[name];
		name=null;
		return store;
	},

	/**
	 * Create a Store object
	 * @ignore
	 * @param {Object} dsConfigObj<pre><code>
	 * name {String} name of the data store
	 * rowId {String} name of the data store record primary key
	 * mapping {Boolean} automatically create columns from object properties
	 * dataProvider {Object} data provider definition
	 * </code></pre>
	 * @return {Object} a reference on the freshly created Store object
	 */
	createGlobalDataStore : function (dsConfigObj) {
		if (!dsConfigObj.name) {
			throw new Error("a public dataStore must have a name");
		} else {
			if (dsConfigObj.name in this._publicStores) {
				return this.getStore(dsConfigObj.name);
			} else {
				return  this._publicStores[dsConfigObj.name] = this._createStore(dsConfigObj);
			}
		}
	},

	/**
		 * @class ICTouchAPI.Store
		 * @extends Object
		 * The Store class provides methods to handle a DataStore object and to fire events on DataStore object changes
		 */
	/**
	 * @event E_ITEM_ADDED
	 * fires when a new record is added
	 * @param {Object} record The new added record
	 */
	/**
	 * @event E_ITEM_REMOVED
	 * fires when a new record is removed
	 */
	/**
	 * @event E_ITEM_UPDATED
	 * fires when a new record is updated
	 * @param {Object} record The updated record
	 */
	/**
	 * @event E_DATA_LOADED
	 * fires when data ready to use or when huge amount of data changed
	 * @param {Object} records All the records contained into the datastore
	 */

	/**
	 * @ignore
	 */
	createDataStore : function (dsConfigObj) {
		return this._createStore(dsConfigObj);
	},

	/**
	 * @ignore
	 */
	_createStore : function (dsConfigObj) {
		return new this._storeFactory(dsConfigObj);
	},

	/**
	 * @ignore
	 */
	_storeFactory : function( dsConfigObj ) {

		var name = null;

		/* Internal objects */
		var Evt; // Event handler
		var Store; // Data
		var DataModel; // Data model
		var Record; // Record that define a row based on the column model
		var DataProvider; // Manager used to synchronize store with an external source


		/*
		 * EVENT
		 * Handle all event publishing and subscription
		 *
		 * The reason why dojo is not used is because we needed one "Hub" for each dataStore
		 * and not one Hub for all. That'd have required a unique id/name for the dataStores
		 * and this could'nt be guaranteed.
		 *
		 * Plus, using dojo's hub allows webapps to connect on dataStore's event to do nasty stuff.
		 *
		 * This implementation of the observer design pattern might be the lightest one
		 *
		 */

		Evt = new function Evt () {

			var topics = {};

			/* list of event fired */
			this.EVT_ITEM_ADDED = "itemAdded";
			this.EVT_ITEM_REMOVED = "itemRemoved";
			this.EVT_ITEM_UPDATED = "itemUpdated";
			this.EVT_ITEMS_LOADED ="itemsLoaded";

			/**
			 * Publish datas on the specified event to the hub
			 * @ignore
			 * @param {String} evtName
			 * @param {Object} dataObj
			 */
			this.publish =  function( evtName, dataObj ) {
				// If the topic exits
				if (topics[evtName]) {
					// The func that'll be called by the forEach loop on the topic list
					var func = function (handler) {
						// Call the callback in its native context
						handler[2].apply(handler[1], dataObj);
					}
					// Execute all callbacks
					topics[evtName].forEach(func);
				}
			};

			/**
			 * Provide subscrition to the specified event
			 * @param {String} evtName Event name
			 * @param {Object} c_context Callback context
			 * @param {Function} c_func Function called when event fired
			 * @return {Array} handler Handler to remove listener later
			 */
			this.addListener = function( evtName, c_context, c_func ) {
				// Check if topic exists, and if not, create an empty array.
				topics[evtName] = topics[evtName] ||[];
				// Create the handler that'll store all the information.
				var handler = [evtName, c_context, c_func];
				// Push the handler in the topic
				topics[evtName].push(handler);
				// Return the handler, it'll be usefull for removing the listener
				return handler;
			};

			/**
			 * Provide unsubscription to the specified event node
			 * @param {Array} handler Handler given by event listener adder
			 * @return {Boolean} action status
			 */
			this.removeListener = function( handler ) {
				// If topic exists
				if (topics[handler[0]]) {
					// Get the index of the handler
					var index = topics[handler[0]].indexOf(handler);
					// If the handler is in the array
					if (index >= 0) {
						// Remove it
						topics[handler[0]].splice(index, 1);
						// Warn it's been successful
						return true;
					} else {
						return false;
					}
				}else {
					return false;
				}
			};

			// Just for debugging purpose
			/**
			 * @ignore
			 */
			this.debug = function debug() {
			//				console.log(topics);
			}

		};



		/*
		 * STORE
		 * Provide methods for managing datas and fire related events
		 * Datas are managed with a simple structure :
		 *  - data: an array with natural index
		 *  - indexes: an object used to make link between columnIndex and data[] indexes
		 */

		Store = {

			events: true,
			mapFromData: false,

			/* raw datas */
			data: [],
			/* list of columnIndex */
			indexes: [],

			/**
			 * Add a record at the end of the store
			 * @param {Object} recordObj Record object
			 * @return {Boolean} action status
			 */
			add: function( recordObj ) {
				var added = false;

				if( Store.mapFromData === true )
				{
					if(!DataModel.boolModelDefined){
						var model = [];

						/* browse object properties that will be used as columns */
						for( var column in recordObj ) {
							model.push( {
								index: column
							} );
						}

						/* Create the data model */
						DataModel.create( model );
					}
				}

				/* Add a row at the end of data[] */
				if( undefined !== recordObj[ DataModel.indexColumn ] ) {
					Store._addTo( recordObj, recordObj[ DataModel.indexColumn ] );
					added = true;
				}

				/* EVENT - Operation succeed */
				if( added === true ) {
					/* Fire ADD event if enabled */
					if( Store.events === true ) {
						var data = {};
						data.value = recordObj[ DataModel.indexColumn ];
						Evt.publish( Evt.EVT_ITEM_ADDED, [recordObj] );
					}
				}

				return added;
			},


			/**
			 * Add a record to the store at the specified index
			 * @ignore
		 * @param {Number} index
			 * @param {Object} recordObj
			 */
			insert: function( index, recordObj ) {},


			/**
			 * Edit column(s) of record(s) that matches column(s) defined in the search object
			 * @ignore
			 * @param {Object} searchObj
			 * @param {Object} updateObj
			 */
			edit: function( searchObj, updateObj ) {},


			/**
			 * Edit column(s) for the record at the specified index
			 * @param {Object} recordIndex Record primary key value
			 * @param {Object} updateObj Record object
			 * @return {Boolean} action status
			 */
			editAt: function( recordIndex, updateObj ) {
				var updated = false;
				var objNew = {};

				if( Store._indexExists(recordIndex) === true )
				{
					/* get the current data model for validate properties to update (and check) */
					var columns = DataModel.get();

					/* Browse each requested property to update */
					for( var prop in updateObj ) {

						/* Check if property to update exists in data model */
						if( undefined !== columns[prop]) {

							/* Set the property of the requested record with its new value */
							if(Store._setProp( Store._getIndex(recordIndex), prop, updateObj[prop] )){
								updated = true;
								objNew[prop] = updateObj[prop];
							}
						}
					}
					/* EVENT - Operation succeed */
					if( updated === true ) {
						/* Fire UPDATE event if enabled */
						if( Store.events === true ) {
							var data = {};
							data.value = recordIndex ;
							var object = this.getAt(recordIndex);
							Evt.publish( Evt.EVT_ITEM_UPDATED, [object, objNew] );
						}
					}
				}
				return updated;
			},


			/**
			 * Remove record(s) that matches columns defined in the search object
			 * @ignore
			 * @param {Object} searchObj
			 */
			remove: function( searchObj ) {},


			/**
			 * Remove record at the specified index
			 * @param {Object} recordIndex Record primary key value
			 * @return {Boolean} action status
			 */
			removeAt: function( recordIndex ) {

				var removed = false;

				/* Check if a record exists with this index */
				if( Store._indexExists(recordIndex) === true )
				{
					/* Remove record from data... */
					Store.data.splice( Store._getIndex(recordIndex), 1 );
					/* ... and update the index */
					Store.indexes.splice( Store._getIndex(recordIndex), 1 );
					//delete Store.indexes[ recordIndex ];

					removed = true;

					/* EVENT - Fire REMOVE event if enabled */
					if( Store.events === true ) {
						Evt.publish( Evt.EVT_ITEM_REMOVED, [recordIndex] );
					}
				}

				return removed;

			},


			/**
			 * Return record at the specified index
		 * @param {Number} recordIndex Record primary key value
			 * @return {Object} record reference for the specified index
			 */
			getAt: function( recordIndex ) {

				var result = false;

				if( Store._indexExists( recordIndex ) === true ) {
					result = new Record( Store.data[ Store._getIndex(recordIndex) ] );
				}

				return result;
			},


			/**
			 * Returns a list of records
			 * @param {Object} configObj
			 * <pre><code>
                 * start {Number} start batch value
			 * max {Integer} batch size
			 * </code></pre>
			 * @return {Object} record references list, returns all if param is missing
			 */
			getAll: function( configObj ) {

				var list = [],
				listStart = 0,
				listMax = Store._length();

				/* Config is defined, using it for paginate */
				if( undefined !== configObj ) {
					if( undefined !== configObj.start && parseInt( configObj.start,10) ) {
						listStart = configObj.start;
					}
					if( undefined !== configObj.max && parseInt( configObj.max,10) && configObj.max <= listMax ) {
						listMax = configObj.max;
					}
				}

				/* Returns the portion of items in list */
				for( var i = listStart; i < listMax; i++ ) {
					list.push( Store.data[i] );
				}

				return list;
			},


			/**
			 * Search for records that matches
			 * @ignore
			 * @param {Array} arrFieldNames
			 * @param {String} search
			 */
			find: function( arrFieldNames, search ) {
				var func = function (item) {
					// If this item's attribute equals..
					for(var i = 0; i<arrFieldNames.length; i++){
						if (item[arrFieldNames[i]] === search) {
							return item;
						}
					}
				}
				return Store.data.filter(func);
			},

			/**
			 * @ignore UNTESTED!!
			 * Load data from another source and create column model from it
			 * @param {Array} arrDataList List of additional record objects to store
			 */
			loadData: function( arrDataList ) {

				Store._flush();

				/* Data model have to be defined from datas */
				if( Store.mapFromData === true )
				{
					var model = [];

					/* browse object properties that will be used as columns */
					for( var column in arrDataList[0] ) {
						model.push( {
							index: column
						} );
					}

					/* Create the data model */
					DataModel.create( model );
				}

				// Inserting data in store / No events will be fired until the end of addition
				Store.disableEvents();

				for( var i in arrDataList ) {
					Store.add( arrDataList[i] );
				}

				Store.enableEvents();

				/* All items have been added - Fire global event data loaded */
				Evt.publish( Evt.EVT_ITEMS_LOADED, [Store.data] );
			},



			/**
			 * @ignore
			 */
			enableEvents: function() {
				//				console.log( "[Datastore][Store] Enabling events" );
				Store.events = true;
			},

			/**
			 * @ignore
			 */
			disableEvents: function() {
				//				console.log( "[Datastore][Store] Disabling events" );
				Store.events = false;
			},


			/**
			 * Add record to store, add columnIndex to index
			 * @ignore
			 * @param {Object} recordObj
		 * @param {Number} columnIndex
			 */
			_addTo: function( recordObj, columnIndex ) {
				Store.data.push( recordObj );
				Store.indexes.push( columnIndex.toString() );
			},

			/**
			 * Flush store and index
			 * @ignore
			 */
			_flush: function() {
				Store.data = [];
				Store.indexes = [];
			},

			/**
			 * Returns the total number of items
			 * @ignore
			 */
			_length: function() {
				return Store.data.length;
			},

			/**
			 * Check if the recordIndex exists in store
			 * @ignore
			 * @param {String} recordIndex
			 */
			_indexExists: function( recordIndex ) {
				return (-1 !== Store.indexes.indexOf( recordIndex.toString() ) ) ? true : false;
			},

			/**
			 * Set property of a record at the specified index
			 * @ignore
		 * @param {Number} index
			 * @param {String} propName
			 * @param {String} propValue
		 * @return {Boolean} wether or not a change has been performed.
			 */
			_setProp: function( index, propName, propValue ) {
				if(Store.data[index][propName] != propValue){
					Store.data[index][propName] = propValue;
					return true;
				} else {
					return false;
				}
			},

			/**
			 * @ignore
			 */
			_getIndex: function( recordIndex ) {
				return Store.indexes.indexOf( recordIndex.toString() );
			}

		};



		/*
		 * COLUMN MODEL
		 * Used to define data organisation and to create records
		 */

		DataModel = {

			indexColumn: "",

			arrDatatypes: [ "integer", "string" ],

			/* defined columns for the store */
			arrColumns: {},

			boolModelDefined: false,

			/* optional functions used to validate data type */
			arrValidators: [],

			/**
			 * Create model for the store based on the config
			 * @ignore
			 * @param {Object} configObj
			 */
			create: function( configObj ) {
				//				console.log( "[Datastore][DataModel] Creating Model", configObj );

				for( var columnIndex in configObj ) {
					this.arrColumns[ configObj[ columnIndex].index ] = {};
				}
				if(configObj.length){
					this.boolModelDefined = true;
				}
			},

			/**
			 * Add a column to the model
			 * @param {Object} columnConfigObj
			 * @ignore
			 */
			addColumn: function( columnConfigObj ) {
				if(columnConfigObj && !DataModel.arrColumns[columnConfigObj.index]){
					DataModel.arrColumns[columnConfigObj.index] = {};
				}
			},

			/**
			 * Remove a column from the model
			 * @ignore
			 * @param {String} columnName
			 */
			removeColumn: function( columnName ) {},

			/**
			 * Add a data type validator for the specified column
			 * Will be used each time this column data is changed
			 * @ignore
			 * @param {String} columnName
			 * @param {Object} configObj
			 */
			addTypeValidator: function( columnName, configObj ) {},


			/**
			 * Returns the current model
			 * @ignore
			 */
			get: function() {
				return DataModel.arrColumns;
			}

		};



		/*
		 * RECORD
		 * Object based on data model and returned by Store queries
		 */

		/**
 * @ignore
		 */
		Record = function( data ) {

			/* Set all properties based on column model */
			for( var columnIndex in DataModel.arrColumns ) {
				this[ columnIndex ] = null;
			}

			this.getIndexName = function() {
				return DataModel.indexColumn;
			};

			// and set all values of this record
			for( var index in data ) {
				if( undefined !== this[index] ) {
					this[ index ] = data[ index ];
				}
			}

		};



		/*
		 * DATA PROVIDER
		 *
		 */

		DataProvider = {

			enabled: false,
			params: {},


			/**
			 * Set the config (events and callbacks)
			 * @ignore
			 * @param {Object} configObj
			 */
			configure: function( configObj ) {
				//				console.log( "[Datastore][DataProvider]", configObj );

				/* provider is enabled if config is valid */
				this.enabled = true;

				if( undefined !== configObj.onInit ) {
					this.params = configObj;
				}
			},

			/**
			 * @ignore
			 */
			getApplicationName: function () {
				return this.name;
			},

			/**
			 * @ignore
			 */
			init: function() {

				if( undefined !== this.params )	{
					if( undefined !== this.params.onInit ) {
						var params = {};
						params.callback = this.onInit;
						params.context = this;
						params.params = [-1,-1];

						this.params.onInit.func.call( this.params.onInit.context || window, params );
					}

					if( undefined !== this.params.onCreate ) {
						this.params.onCreate.event.forEach(function(evt) {
							ICTouchAPI.eventServices.subscribeToEvent(this, evt, this.dataCreate);
						},this);
					}
					if( undefined !== this.params.onUpdate ) {
						this.params.onUpdate.event.forEach(function(evt) {
							ICTouchAPI.eventServices.subscribeToEvent(this, evt, this.dataUpdate);
						},this);
					}
					if( undefined !== this.params.onDelete ) {
						this.params.onDelete.event.forEach(function(evt) {
							ICTouchAPI.eventServices.subscribeToEvent(this, evt, this.dataDelete);
						},this);
					}
					if( undefined !== this.params.onSynchro ) {
						this.params.onSynchro.event.forEach(function(evt) {
							ICTouchAPI.eventServices.subscribeToEvent(this, evt, this.dataSynchro);
						},this);
					}
				}
			},

			/**
			 * @ignore
			 */
			onInit: function(event) {
				/* If not empty, remove all items */
				Store.data = [];
				Store.indexes = [];

				/*  */
				Store.loadData( event );
			},

			/**
			 * @ignore
			 */
			dataSynchro: function(event) {
				var values;
				if(this.params.onCreate.argumentName){
					var objArguments = ICTouchAPI.tools.getEventArguments(arguments);
					values = objArguments[this.params.onCreate.argumentName];
				} else {
					values = event.value;
				}
				if(this.params.onInit.func){
					var params = {};
					params.callback = this.onInit;
					params.context = this;
					params.params = [-1,-1];

					this.params.onInit.func.call( this.params.onInit.context || window, params );
				} else {
					this.onInit.call(this, values);
				}
			},

			/**
			 * @ignore
			 */
			dataCreate: function(event) {
				var values;
				if(this.params.onCreate.argumentName){
					var objArguments = ICTouchAPI.tools.getEventArguments(arguments);
					values = objArguments[this.params.onCreate.argumentName];
				} else {
					values = event.value;
				}
				if(this.params.onCreate.func){
					var params = {};
					params.callback = this.onDataCreate;
					params.context = this;
					params.params = [values]; // contact_id, like "#CONTACT_1#"

					this.params.onCreate.func.call( this.params.onCreate.context || window, params );
				} else {
					this.onDataCreate.call(this, values);
				}
			},

			/**
			 * @ignore
			 */
			onDataCreate: function(event) {
				while(event.length > 0){
					Store.add( event.pop() );
				}
			},

			/**
			 * @ignore
			 */
			dataUpdate: function(event) {
				var values;
				if(this.params.onUpdate.argumentName){
					var objArguments = ICTouchAPI.tools.getEventArguments(arguments);
					values = objArguments[this.params.onUpdate.argumentName];
				} else {
					values = event.value;
				}
				if(this.params.onUpdate.func){
					var params = {};
					params.callback = this.onDataUpdate;
					params.context = this;
					params.params = [values]; // contact_id, like "#CONTACT_1#"

					this.params.onUpdate.func.call( this.params.onUpdate.context || window, params );
				} else {
					this.onDataUpdate.call(this, values);
				}
			},

			/**
			 * @ignore
			 */
			onDataUpdate: function(event) {
				var obj;
				while(event.length > 0){
					obj = event.pop();
					Store.editAt( obj.contactId, obj );
				}
			},

			/**
			 * @ignore
			 */
			dataDelete: function(event) {
				var values;
				if(this.params.onDelete.argumentName){
					var objArguments = ICTouchAPI.tools.getEventArguments(arguments);
					values = objArguments[this.params.onDelete.argumentName];
				} else {
					values = event.value;
				}
				while(values.length > 0){
					Store.removeAt(values.pop()); // contact_id, like "#CONTACT_1#"
				}
			}
		};

		/*
		 * Constructor
		 * @ignore
		 * @param {Object} configObj
		 */
		(function( configObj ) {
			name = configObj.name;

			/* Column definition if defined in this config object */
			if( undefined !== configObj.columns ) {
				DataModel.create( configObj.columns );
			}

			/* Activate auto mapping when Store load data */
			else if( undefined !== configObj.mapping && configObj.mapping === true ) {
				Store.mapFromData = true;
			}

			/* Set row index name */
			if( undefined !== configObj.rowId ) {
				DataModel.indexColumn = configObj.rowId;
			}

			/* Configure the data provider */
			if( undefined !== configObj.dataProvider ) {
				DataProvider.configure( configObj.dataProvider );
			}


			init();


		})( dsConfigObj );



		/**
		 * @ignore
		 */
		function init() {
			if( DataProvider.enabled ) {
				/* Call DataProvider init - this will start data loading and update event listenning */
				DataProvider.init();
			}
		}

		/*
		 *  Public interface ------------------------------
		 *  Provide public access on only necessary methods.
		 *  This interface will vary depending on access rights (read only, read/write)
		 */

		/* Event - access for all */

		this.addEventListener = Evt.addListener;
		this.removeEventListener = Evt.removeListener;

		/* Event - constants for listeners */
		this.E_DATA_LOADED = Evt.EVT_ITEMS_LOADED;
		this.E_ITEM_ADDED = Evt.EVT_ITEM_ADDED;
		this.E_ITEM_UPDATED = Evt.EVT_ITEM_UPDATED;
		this.E_ITEM_REMOVED = Evt.EVT_ITEM_REMOVED;


		/* Store - access for all */
		this.getList = Store.getAll;
		this.getAt = Store.getAt;
		this.find = Store.find;

		/* Store - write access required */
		this.add = Store.add;
		//this.insert = Store.insert;
		this.editAt = Store.editAt;
		//this.remove = Store.remove;
		this.removeAt = Store.removeAt;
		this.load = Store.loadData;

		/* DataModel - write access required */
		this.addColumn = function(obj){
			DataModel.addColumn(obj)
		};
	}
});
ICTouchAPI.dataStoreServices = new ICTouchAPI.dataStoreServices();

/**
 * @class ICTouchAPI.cssHandler
 * @extends Object
 * @ignore
 */
dojo.provide("ICTouchAPI.cssHandler");
dojo.declare("ICTouchAPI.cssHandler",null,
{

	/* --------------------------------- Public attributes ------------------------------------ */


	/* --------------------------------- Private attributes ----------------------------------- */
	_headerTag				: null,
	__basic					: null,
	_icons					: null,
	_linkedBasic			: null,
	_linkedIcons			: null,
	_arrListener			: null,
	_lockEventToDOM			: null,

	/* ------------------------------------ Constructor --------------------------------------- */

	/**
	 * This method create the cssHandler and initialize it
	 */
	constructor: function (headerTag){
		this._headerTag		= headerTag;

		this._basic			= {};
		this._icons			= {};
		this._linkedBasic	= {};
		this._linkedIcons	= {};

		this._arrListener	= [];
		this._lockEventToDOM = false;

		this._arrListener.push(dojo.subscribe("RemoveSkinCss",this,this.onRemoveSkinCssEvent));
	},


	/* ----------------------------------- Getter / Setter------------------------------------- */

	/**
	 * return a clone of the queried fragment
	 * @param {string} typeCss type of the expected css to be returned : basic, icons, linkedIcons, linkedBasic
	 * @param {string} skinName Name of the skin
	 * @return {HTMLElement} domFragment fragment holding the css related to the handlers
	 */
	getCss : function(typeCss,skinName){
		switch(typeCss){
			case "basic" :
				return dojo.clone(this._basic[skinName]);
			case "icons" :
				return dojo.clone(this._icons[skinName]);
			case "linkedIcons" :
				return dojo.clone(this._linkedIcons[skinName]);
			case "linkedBasic" :
				return dojo.clone(this._linkedBasic[skinName]);
		}
	},
	
	
	/**
	 * Return the managed tag
	 * @return {string} tag
	 */
	getTag : function(){
		return this._headerTag;
	},

	/**
	 * Set or unset a lock on the event publication to DOM elements.
	 * Do NOT lock event to other handlers.
	 * Publish linkedbasic and linkedicons event when the lock is set off.
	 * @param {boolean} value state of the lock inkedbasic and linkedicons event only for specified skins
	 * @param {Array} arrSkins Array o string, publish
	 */
	setLock : function(value,arrSkins){		
		var i;
		this._lockEventToDOM = value;
		if(value == false){
			if (arrSkins==null) {
				for (i in this._linkedIcons) {
					this.publishChangedEvent("linkedIcons",i);
				}
				for (i in this._linkedBasic) {
					this.publishChangedEvent("linkedBasic",i);
				}
			}
			else {
				for (i in arrSkins) {
					this.publishChangedEvent("linkedIcons",arrSkins[i]);
					this.publishChangedEvent("linkedBasic",arrSkins[i]);
				}
			}
		}

	},


	/* ----------------------------------- Public methods ------------------------------------- */

	/**
	 * Add links based on an array of files and publish an event to let know our targeted links we got new css
	 * @param {string} namespace_key0 first part ot the directory path to find the given css files (ICTouchAPI, webapp or UIElements)
	 * @param {string} namespace_key1 second part ot the directory path to find the given css files (Spinner, contacts, settings ...)
	 * @param {string} skinName name of the skin
	 * @param {array} arrayOfFiles array of the names of the files to be loaded and managed by this handler
	 */
	loadCssOnArrayOfFile : function(namespace_key0,namespace_key1,skinName,arrayOfFiles){		
		dojo.forEach(arrayOfFiles,function(fileName){
			if(fileName == "icons.css"){
				dojo.place(this._createLinkTag(namespace_key0,namespace_key1,skinName,fileName,"icons"),this._linkedIcons[skinName]);
				this._clean("linkedIcons",skinName);
				dojo.place(this._createLinkTag(namespace_key0,namespace_key1,skinName,fileName,"icons"),this._icons[skinName]);
				this._clean("icons",skinName);
			}else{
				dojo.place(this._createLinkTag(namespace_key0,namespace_key1,skinName,fileName,"basic"),this._linkedBasic[skinName]);
				this._clean("linkedBasic",skinName);
				dojo.place(this._createLinkTag(namespace_key0,namespace_key1,skinName,fileName,"basic"),this._basic[skinName]);
				this._clean("basic",skinName);
			}
		},this);
		//publishing for linked cssHandlers
		this.publishChangedEvent("basic",skinName);
		this.publishChangedEvent("icons",skinName);
		//publishing to allow related iframes to update their css links
		this.publishChangedEvent("linkedIcons",skinName);
		this.publishChangedEvent("linkedBasic",skinName);
	},

	/**
	 * methode used to manage our events
	 * @param {string} type type of the dom fragment which had been updated
	 * @param {string} skinName name of the skin
	 */
	publishChangedEvent : function(type,skinName){
		if(!this._lockEventToDOM || (type != "linkedIcons" && type != "linkedBasic")) {
			dojo.publish(this._headerTag+"."+type,[{'owner' : this,'type' : type, 'skinName' : skinName}]);
		}
	},

	/**
	 * allow an other cssHandler to link its css to ours and add a subsciption to any changes on the css of the owner
	 * @param {cssHandler} ownerHeader reference on the owner handler of the shared css
	 * @param {boolean} onlyIcon used to exclude copy of basic css
	 * @param {boolean} onlyCss used to exclude copy of icons css
	 * @param {boolean} duplicateIcon used to copy the icons css to the local linkedBasic fragment
	 * @param {boolean} duplicateCss used to copy the basic css to the local linkedIcons fragment
	 *
	 */
	link : function(ownerHeader,onlyIcon,onlyCss,duplicateIcon,duplicateCss){
		var i,j;
		for(i in ownerHeader) {			
			var ownerHeaderTag = ownerHeader[i].getTag();
			
			if(!onlyIcon){
				this._arrListener.push(dojo.subscribe(ownerHeaderTag+".basic",this,this.onLinkedBasicChange));
				for (j in this._linkedBasic) {
					this._placeLinkedBasic({'owner' : ownerHeader[i],'type' : "basic",skinName:j});
				}
			}
			if(!onlyCss){
				this._arrListener.push(dojo.subscribe(ownerHeaderTag+".icons",this,this.onLinkedIconsChange));
				for (j in this._linkedIcons) {
					this._placeLinkedIcons({'owner' : ownerHeader[i],'type' : "icons",skinName:j});
				}
			}
			if(duplicateIcon){
				this._arrListener.push(dojo.subscribe(ownerHeaderTag+".icons",this,this.onLinkedBasicChange));
				for (j in this._linkedBasic) {
					this._placeLinkedBasic({'owner' : ownerHeader[i],'type' : "icons",skinName:j});
				}
			}
			if(duplicateCss){
				for (j in this._linkedIcons) {
					this._arrListener.push(dojo.subscribe(ownerHeaderTag+".basic",this,this.onLinkedIconsChange));
				}
				this._placeLinkedIcons({'owner' : ownerHeader[i],'type' : "basic",skinName:j});
			}
		}
		
		for (i in this._linkedBasic) {
			this._clean("linkedBasic",i);			
			this.publishChangedEvent("linkedBasic",i);
		}		
		for (i in this._linkedIcons) {
			this._clean("linkedIcons",i);			
			this.publishChangedEvent("linkedIcons",i);
		}
		
	},

	/**
	 * when a css owner has been updated. For appbar.
	 * @param {object} objEvent object sent with the caught event
	 */
	onLinkedIconsChange : function(objEvent){
		this._placeLinkedIcons(objEvent);
		this._clean("linkedIcons",objEvent.skinName);
		this.publishChangedEvent("linkedIcons",objEvent.skinName);
	},
	

	/**
	 * when a css owner has been updated. for webapps.
	 * @param {object} objEvent object sent with the caught event
	 */
	onLinkedBasicChange : function(objEvent){
		this._placeLinkedBasic(objEvent);
		this._clean("linkedBasic",objEvent.skinName);
		this.publishChangedEvent("linkedBasic",objEvent.skinName);
	},

	/**
	 * when skin removed, update fragments
	 * @param {stringt} skinName skin to remove
	 */
	onRemoveSkinCssEvent : function(skinName){
		this.removeCss(skinName);
	},


	/**
	 * Destroy  all fragments in the handler of specific skin. (Destroy child links)
	 *
	 */
	removeCss : function(skinName){
		dojo.destroy(this._basic[skinName]);
		delete this._basic[skinName];
		dojo.destroy(this._icons[skinName]);
		delete this._icons[skinName];
		dojo.destroy(this._linkedBasic[skinName]);
		delete this._linkedBasic[skinName];
		dojo.destroy(this._linkedIcons[skinName]);
		delete this._linkedIcons[skinName];
	},

	/**
	 * Destroy all fragments in the handler. (Destroy child links)
	 * 
	 */
	removeAllCss : function(){
		var i;
		for (i in this._basic) {
			dojo.destroy(this._basic[i]);
			delete this._basic[i];
		}
		for (i in this._icons) {
			dojo.destroy(this._icons[i]);
			delete this._icons[i];

		}
		for (i in this._linkedBasic) {
			dojo.destroy(this._linkedBasic[i]);
			delete this._linkedBasic[i];
		}
		for (i in this._linkedIcons) {
			dojo.destroy(this._linkedIcons[i]);
			delete this._linkedIcons[i];
		}
	},

	/**
	 * Create dom fragments for a skin if doesn't exist
	 * @param {String} skinName Name of skin
	 */
	createSkinFragments : function(skinName) {
		if(this._basic[skinName] == null) {			
			this._basic[skinName] = document.createDocumentFragment();
		}
		if(this._icons[skinName] == null) {			
			this._icons[skinName] = document.createDocumentFragment();
		}
		if(this._linkedBasic[skinName] == null) {			
			this._linkedBasic[skinName] = document.createDocumentFragment();
		}
		if(this._linkedIcons[skinName] == null) {			
			this._linkedIcons[skinName] = document.createDocumentFragment();
		}
	},



	/* --------------------------------- Private Methods -------------------------------------- */

	/**
	 * Create the HTMLElement for the css
	 * @param {string} namespace_key0 first part ot the directory path to find the given css files (ICTouchAPI, webapp or UIElements)
	 * @param {string} namespace_key1 second part ot the directory path to find the given css files (Spinner, contacts, settings ...)
	 * @param {string} skinName name of the skin
	 * @param {string} fileName name of the file to be targeted by the link element
	 * @param {string} cssType type of CSS ("basic" or "icons")
	 */
	_createLinkTag: function (namespace_key0,namespace_key1,skinName,fileName,cssType) {
		if(namespace_key1 == "minifiedCss" && !generalConfig.cssDeveloper){
			return dojo.create("link",{
				'csstag':this._headerTag,
				'cssType':cssType,
				'rel':'stylesheet',
				'type':'text/css',
				'skin':skinName,
				'href':dojo.moduleUrl(namespace_key0+"."+namespace_key1,fileName).path
				}
			);
		}else{
			return dojo.create("link",{
				'csstag':this._headerTag,
				'cssType':cssType,
				'rel':'stylesheet',
				'type':'text/css',
				'skin':skinName,
				'href':dojo.moduleUrl(namespace_key0+(namespace_key1 ? "."+namespace_key1 : "" ),"themes/"+skinName+"/css/"+fileName).path
				}
			);
		}
		
	},

	/**
	 * Local getter to return a reference (not a copy)
	 * @param {string} typeCss type of the expected css to be returned : basic, icons, linkedIcons, linkedBasic
	 * @param {string} skinName name of the skin
	 * @return {HTMLElement} domFragment fragment holding the css related to the handlers
	 */
	_getCss : function(typeCss,skinName){
		switch(typeCss){
			case "basic" :
				return this._basic[skinName];
			case "icons" :
				return this._icons[skinName];
			case "linkedIcons" :
				return this._linkedIcons[skinName];
			case "linkedBasic" :
				return this._linkedBasic[skinName];
		}
	},
	
	/**
	 * Clean a fragment from any doublon of link
	 * @param {string} typeCss type type of the local fragment to be cleaned
	 * @param {string} skinName name of the skin
	 */
	_clean : function(typeCss,skinName){
		var frag = this._getCss(typeCss,skinName);
		var found = {};
		dojo.query("link", frag).forEach(function(linkTag){
			if(this.found[linkTag.href]){
				dojo.destroy(linkTag);
			}else{
				this.found[linkTag.href] = true;
				dojo.attr(linkTag,"csstag",this._headerTag);

			}
		},{'found' : found,'_headerTag' : this._headerTag});
	},

	_placeLinkedIcons : function(objEvent) {
		var skinName = objEvent.skinName;
		var ownerNode = objEvent.owner.getCss(objEvent.type,skinName);
		if(ownerNode!=null && this._linkedIcons[skinName]!=null) {
			dojo.place(ownerNode,this._linkedIcons[skinName]);
		}
	},

	_placeLinkedBasic : function(objEvent) {
		var skinName = objEvent.skinName;
		var ownerNode = objEvent.owner.getCss(objEvent.type,skinName);
		if(ownerNode!=null && this._linkedBasic[skinName]!=null) {
			dojo.place(ownerNode,this._linkedBasic[skinName]);
		}
	},

});

/**
* @class ICTouchAPI.skinServices
* @singleton
* @ignore
* @extends Object
* Manage skins
*/

dojo.require("ICTouchAPI.cssHandler");

dojo.provide("ICTouchAPI.skinServices");

dojo.declare("ICTouchAPI.skinServices",null,
{

	/* --------------------------------- Public attributes ------------------------------------ */

	/**
	 * image used during the change of skin
	 * @property
	 * @type {Object}
	 */
	skinTransitionImage : null,


	/* --------------------------------- Private attributes ----------------------------------- */

	/**
	 * name of the current skin
	 * @property
	 * @private
	 * @type {String}
	 */
	_skinName : "Default",

	/**
	 * name of the current skin
	 * @property
	 * @private
	 * @type {String}
	 */
	_skinBackground : "",

	/**
	 * name of the associated full skin for partial skins
	 * @property
	 * @private
	 * @type {String}
	 */
	_associatedSkinName : "",

	/**
	 * content of the skinName/providedFiles.json for full or default skin and associatedFullSkin/providedFiles.json for partial skins
	 * @property
	 * @private
	 * @type {Object}
	 */
	_skinConfig: null,

	/**
	 * content of the skinName/providedFiles.json when usig a partial skin
	 * @property
	 * @private
	 * @type {Object}
	 */
	_associatedSkinConfig: null,

	/**
	 * array holding css handlers (independant agents)
	 * @property
	 * @private
	 * @type {Array}
	 */
	_arrCssHandlers : {},

	/**
	 * Max depth of skin heritage
	 * @property
	 * @private
	 * @type {int}
	 */
	_maxSkinLevel : 2,

	/**
	 * level associated to skin
	 * @property
	 * @private
	 * @type {Object}
	 */
	_skinLevel : {},




	/* ------------------------------------ Constructor --------------------------------------- */

	/**
	 * This method create the services and initialize it
	 */
	constructor: function (){
		//accessing the skin setting value
		this._accessSkinName();
		this._accessSkinProvidedFiles();
		this._loadSkinOnProvidedFiles();

		//check the skin on end_of_init event
		this._checkSkinOnEndOfInit();
	},


	/* ----------------------------------- Getter / Setter------------------------------------- */

	/**
	 * return the name of the application (used for event subscription ?)
         * @private
	 * @return {string} namespace.classname of the services
	 */
	getApplicationName : function(){
		return "ICTouchAPI.skinServices";
	},


	/**
	 * This method set the background defined by the user or the default background of the current skin
	 * @param {object} objSetting : the setting holding the user private background
	 *
	 */
	setBackground : function(objSetting){
		var bgdSrc;
		//we load it by changing the url of the DOM element with the .withBackground class
		//create the image and load it
		var img = new Image();
		var that=this;
		//if user has defined a private background url, use it
		if(objSetting && objSetting.value && objSetting.value != ""){
			bgdSrc = objSetting.value;
			img.onerror=function(){
				ICTouchAPI.settingServices.setSettingValue("UserBackgroundPicture", "");
			};
		}
		else { //if no background defined, use the default one of the current skin
			bgdSrc = "/library/ICTouchAPI/themes/" + this._skinName + "/images/default.jpg";
			img.onerror=function(){
				dojo.query(".withBackground").style("background-image", "");
			};
		}

		if(this._skinBackground != bgdSrc) {
			this._skinBackground = bgdSrc;
			img.src = bgdSrc;
			img.onload=function(){
				that._applyBackground(img);
			};
		}
	},


	/* ----------------------------------- Public methods ------------------------------------- */

	/**
	 * method used to set a frame :
	 *  - add the required css,
	 *  - add properties and functions to the frame,
	 *  - subscribe the frame to the related cssHandler
	 * @param {HTMLElement} header : header to be skinned
	 * @param {String} strWebappName : name of the webapp related to the header; used to get the tag
	 * @param {String} type : type of the frame
	 * @param {HTMLiFrame} frame : the targeted frame to be set up
	 */
	registerHeader : function(header,strWebappName,type,frame) {
		this._placeCssPosition(header);

		//getting the related headerTag
		var headerTag = (strWebappName.split("."))[1];
		headerTag = (headerTag ? headerTag : (strWebappName.split("."))[0]); //"ICTouchAPI" case only

		//setting the type
		type = (type == "appbar" ? "linkedIcons" : "linkedBasic");

		//creating handler if needed
		if(!this._arrCssHandlers[headerTag])
			this._arrCssHandlers[headerTag] = new ICTouchAPI.cssHandler(headerTag);
		if(this._associatedSkinName!="") {
			this._arrCssHandlers[headerTag].createSkinFragments(this._associatedSkinName);
		}
		this._arrCssHandlers[headerTag].createSkinFragments(this._skinName);

		//adding css
		if(this._associatedSkinName!="") {
			this._placeLink("ICTouchAPI", "linkedBasic", this._associatedSkinName, header);
			this._placeLink("ICTouchAPI", "linkedIcons", this._associatedSkinName, header);
			this._placeLink(headerTag, type, this._associatedSkinName, header);
		}
		//adding css
		this._placeLink("ICTouchAPI", "linkedBasic", this._skinName, header);
		this._placeLink("ICTouchAPI", "linkedIcons", this._skinName, header);
		this._placeLink(headerTag, type, this._skinName, header);

		//adding management tools to the frame
		if(frame){
			frame.cssTag  = headerTag;
			frame.cssType = type;

			this._setUpFrame(frame);
			frame.subToCss.push(dojo.subscribe("ICTouchAPI.linkedBasic",frame,frame.onCssHandlerEvent));
			frame.subToCss.push(dojo.subscribe("ICTouchAPI.linkedIcons",frame,frame.onCssHandlerEvent));
			frame.subToCss.push(dojo.subscribe(headerTag+"."+type,frame,frame.onCssHandlerEvent));
			frame.subToCss.push(dojo.subscribe("RemoveSkinCss",frame,frame.onRemoveSkinCssEvent));
		}
	},

	/**
	 * method used to add a stylesheet link to the header of a frame :
	 * @param {HTMLElement} header : the header in which the stylesheet file will be linked
	 * @param {String} url : the path to the stylesheet file to link to the header
	 * @private
	 */
	addStyleToHeader : function(header, url) {
	    // Check that the stylesheet is not already registered
		var links = header.getElementsByTagName("LINK");
		for (var i in links) {
			var node = links[i];
			if (node.href && node.href.indexOf(url) !== -1) {
				ICTouchAPI.debugServices.debug("ICTouchAPI.skinServices - addStyleToHeader / the stylesheet with url " + url + " is already registered to the header of the frame, exit");
				return;
			}
		}
		var link = dojo.create("link", {
							type  : "text/css",
							rel   : "stylesheet",
							media : "screen",
							href  : url
						});

		ICTouchAPI.debugServices.debug("ICTouchAPI.skinServices - addStyleToHeader / register the stylesheet with url " + url +" to the frame");
		dojo.place(link, header);
	},

	/**
	 * method used to remove a stylesheet link from the header of a frame :
	 * @param {HTMLElement} header : the header in which the stylesheet file will be removed
	 * @param {String} url : the path to the stylesheet file to remove from the header
	 * @private
	 */
	removeStyleToHeader : function(header, url) {
		var links = header.getElementsByTagName("LINK");
		for (var i in links) {
			var node = links[i];
			if (node.href && node.href.indexOf(url) !== -1) {
				ICTouchAPI.debugServices.debug("ICTouchAPI.skinServices - removeStyleToHeader / remove the stylesheet with url " + url +" from the frame");
				node.parentNode.removeChild(node);
				return;
			}
		}
	},

	/**
	 * method used to add a CSS rule to the DOM Header :
	 * @param {HTMLElement} frame : the frame in which the selector will be added
	 * @param {String} selector : the CSS Selector to add in the DOM Header
	 * @param {String} rule : the rule attached to the CSS Selector to add in the DOM Header
	 * @private
	 */
	addCssRuleToFrame : function(frame, selector, rule) {
	    var doc = frame.contentDocument,
	    style	= null,
		rules	= null;

	    // Get the styleSheet
	    style = doc.styleSheets[doc.styleSheets.length -1];

		// Get the rules
	    rules = style.cssRules;

	    // Check that the rule is not already registered
	    for (var i=0; i < rules.length; i++) {
           if (rules[i].selectorText == selector) {
				ICTouchAPI.debugServices.debug("ICTouchAPI.skinServices - addCssRuleToFrame / the css selector " + selector + " is already registered to the header of the frame, exit");
				return;
           }
        }

	    // add the selector & rule
		ICTouchAPI.debugServices.debug("ICTouchAPI.skinServices - addCssRuleToFrame / register the css selector " + selector + ", with the rule " + rule + " to the frame");
	    style.insertRule("." + selector + " { " + rule + "; }", style.cssRules.length);
	},

	/**
	 * method used to remove a CSS rule from the DOM Header
	 * @param {HTMLElement} frame : the frame from which the selector will be removed
	 * @param {String} selector : The CSS Selector to remove from the DOM Header
	 * @private
	 */
	removeCssRuleToFrame : function(frame, selector) {
		var doc = frame.contentDocument,
	    	style = null,
	    	rules = null,
	    	indexToRemove = -1;

	   	// Get the styleSheet
	    style = doc.styleSheets[doc.styleSheets.length -1];

	    // Get the rules
	    rules = style.cssRules;

	    // Get the index of the rule
	    for (var i=0, len=rules.length; i < len; i++) {
           if (rules[i].selectorText == selector) {
               indexToRemove = i;
               break;
           }
        }

        if(indexToRemove > -1) {
        	style.removeRule(indexToRemove);
        }

	},

	/**
	 * This method destroys a given frame listeners
	 * @param {HTMLiFrame} frame : the targeted iFrame
	 */
	unregisterHeader : function(frame){
		if(frame && frame.subToCss)
			dojo.forEach(frame.subToCss,dojo.unsubscribe);


	},

	/**
	 * method used to define a relation between a targeted webapp and an other webapp css
	 * @param {String} webappTarget : the target which requires to load the css of an other webapp
	 * @param {String} webappOwner : name of the webapp which own the required css. Array of string also can be used
	 */
	linkWebappsStyles : function(webappTarget, webappOwner, onlyIcon, onlyCss, duplicateIcon,duplicateCss){
		onlyIcon		= (onlyIcon ? onlyIcon : false);
		onlyCss			= (onlyCss ? onlyCss : false);
		duplicateIcon	= (duplicateIcon ? duplicateIcon : false);
		duplicateCss	= (duplicateCss ? duplicateCss : false);

		var target = webappTarget.split(".")[1];
		if(!this._arrCssHandlers[target]){
			this._arrCssHandlers[target] = new ICTouchAPI.cssHandler(target);
		}
		if(this._associatedSkinName!="") {
			this._arrCssHandlers[target].createSkinFragments(this._associatedSkinName);
		}
		this._arrCssHandlers[target].createSkinFragments(this._skinName);

		if(webappOwner!=null && typeof(webappOwner)!=="object") {
			webappOwner=[webappOwner];
		}
		var ownerCssHandler = [];
		for(var i in webappOwner) {
			var owner  = webappOwner[i].split(".")[1];
			if(!this._arrCssHandlers[owner]){
				this._arrCssHandlers[owner] = new ICTouchAPI.cssHandler(owner);
				if(this._associatedSkinName!="") {
					this._arrCssHandlers[owner].createSkinFragments(this._associatedSkinName);
				}
				this._arrCssHandlers[owner].createSkinFragments(this._skinName);
			}
			ownerCssHandler[i] = this._arrCssHandlers[owner];
		}

		this._arrCssHandlers[target].link(ownerCssHandler,onlyIcon, onlyCss, duplicateIcon,duplicateCss);
		webappTarget=null;
		webappOwner=null;
		onlyIcon=null;
		onlyCss=null;
		duplicateIcon=null;
		duplicateCss=null;
	},

	/**
	 * method called after end_of_init event has been received to change the skin
	 * @param {object} objSetting : returned skin setting
	 */
	skinSettingChanged : function(objSetting){
		var i;
		if(objSetting && objSetting.value != this._skinName){
			if (!ICTouchAPI.remoteDisplay && !webapp.settings.data.getResetInProgress()) {
				this._createSkinTransitionIFrame()._show();
			}

			var oldSkinName = this._skinName;
			var oldAssociatedSkinName = this._associatedSkinName;
			this._skinName = objSetting.value;
			this._associatedSkinName = "";
			this._skinConfig = null;
			this._associatedSkinConfig = null;

			for(i in this._arrCssHandlers){
				this._arrCssHandlers[i].setLock(true,arrSkins);
			}

			if(oldAssociatedSkinName!="") {
				delete this._skinLevel[oldAssociatedSkinName];
			}
			delete this._skinLevel[oldSkinName];
			this._accessSkinProvidedFiles();

			var removeAssociatedSkin = false;
			var removeSkin = false;
			var reloadAssociatedSkin = false;
			var reloadSkin = false;
			//Full skin to partial skin with different full skin
			if(oldAssociatedSkinName=="" && this._associatedSkinName!="" && oldSkinName!=this._associatedSkinName) {
				removeSkin = true;
				reloadAssociatedSkin = true;
				reloadSkin = true;
			}
			//Partial skin to full skin with different full skin
			else if(oldAssociatedSkinName!="" && this._associatedSkinName=="" && oldAssociatedSkinName!=this._skinName) {
				removeSkin = true;
				removeAssociatedSkin = true;
				reloadSkin = true;
			}
			//partial skin to partial skin with different skin
			else if(oldAssociatedSkinName!="" && this._associatedSkinName!="" && oldAssociatedSkinName!=this._associatedSkinName && oldSkinName!=this._associatedSkinName) {
				removeAssociatedSkin = true;
				removeSkin = true;
				reloadAssociatedSkin = true;
				reloadSkin = true;
			}
			//Full skin to full skin or partial skin to partial skin with same full skin
			else if(oldAssociatedSkinName==this._associatedSkinName && oldSkinName!=this._skinName) {
				removeSkin = true;
				reloadSkin = true;
			}
			//Full skin to partial skin with same full skin
			else if(oldAssociatedSkinName=="" && this._associatedSkinName!="" && oldSkinName==this._associatedSkinName) {
				reloadSkin = true;
			}
			//Partial skin to full skin with same full skin
			else if(oldAssociatedSkinName!="" && this._associatedSkinName=="" && oldAssociatedSkinName==this._skinName) {
				removeSkin = true;
			}


			//Remove unused CSS
			if(removeAssociatedSkin===true) {
				dojo.publish("RemoveSkinCss",[oldAssociatedSkinName]);
			}
			if(removeSkin===true) {
				dojo.publish("RemoveSkinCss",[oldSkinName]);
			}

			this._loadSkinOnProvidedFiles(reloadSkin,reloadAssociatedSkin);

			//Create header fragments
			for(var i in this._arrCssHandlers) {
				if(this._associatedSkinName!="") {
					this._arrCssHandlers[i].createSkinFragments(this._associatedSkinName);
				}
				this._arrCssHandlers[i].createSkinFragments(this._skinName);
			}

			//Reload webapp style
			for(i in webapp) {
				if(webapp[i]._strApplicationName!=null) {
					this.loadWebappStyle(i,reloadSkin,reloadAssociatedSkin);
				}
			}


			var arrSkins = [];
			if(reloadAssociatedSkin===true) {
				arrSkins.push(this._associatedSkinName);
			}
			if(reloadSkin===true) {
				arrSkins.push(this._skinName);
			}

			for(i in this._arrCssHandlers){
				this._arrCssHandlers[i].setLock(false,arrSkins);
			}
			this.loadBackground();

		}
	},

	/**
	 * This method is called during the initialization to load the background picture
	 * defined by the user. If user has not private background the skin background will be loaded
	 */
	loadBackground : function(){
            // Default settings for simulation mode
            if(!generalConfig.simulation) {
				ICTouchAPI.settingServices.getSetting("UserBackgroundPicture", this, this.setBackground);
            }
	},

	/**
	 * This method allows to change the type and the tag of an iFrame, update its css and listeners
	 * @param {HTMLiFrame} frame : the targeted iFrame
	 * @param {string} newTag : the new tag (ex: communication, email,...)
	 * @param {string} type : the new type (webapp / appbar)
	 */
	changeTag : function(frame,newTag,type){
		type = (type == "appbar" ? "linkedIcons" : "linkedBasic");

		if(type != frame.cssType){
			 if(type == "linkedIcons"){
				if(this._associatedSkinName!="") {
					this._placeLink("ICTouchAPI", type, this._associatedSkinName, frame.contentDocument.head);
				}
				this._placeLink("ICTouchAPI", type, this._skinName, frame.contentDocument.head);
				frame.subToCss.push(dojo.subscribe("ICTouchAPI."+type,frame,frame.onCssHandlerEvent));
				if(frame.cssTag == newTag){
					if(this._associatedSkinName!="") {
						this._placeLink(newTag, type, this._associatedSkinName, frame.contentDocument.head);
					}
					this._placeLink(newTag, type, this._skinName, frame.contentDocument.head);
					frame.subToCss.push(dojo.subscribe(newTag+"."+type,frame,frame.onCssHandlerEvent));
				}
			 }else{
				dojo.query("link[href$='icons.css']", frame.contentDocument.head).forEach(dojo.destroy);
				dojo.forEach(frame.subToCss,function(sub){
					if(sub[0] == "ICTouchAPI.linkedIcons" || sub[0] == frame.cssTag+".linkedIcons")
						dojo.unsubscribe(sub);
					}
				);
			 }
			 frame.cssType = type;
		}

		if(frame.cssTag != newTag){
			dojo.query("link[csstag='"+frame.cssTag+"']", frame.contentDocument.head).forEach(dojo.destroy);
			if(this._associatedSkinName!="") {
				this._placeLink(newTag, type, this._associatedSkinName, frame.contentDocument.head);
			}
			this._placeLink(newTag, type, this._skinName, frame.contentDocument.head);
			dojo.forEach(frame.subToCss,function(sub){
				if(sub[0] != "ICTouchAPI.linkedBasic" && sub[0] != "ICTouchAPI.linkedIcons")
					dojo.unsubscribe(sub);
			});
			frame.subToCss.push(dojo.subscribe(newTag+"."+type,frame,frame.onCssHandlerEvent));
			frame.cssTag = newTag;
			frame.cssType = type;
		}
	},


/* --------------------------------- Private Methods -------------------------------------- */

	/**
	 * This method check the existence of the providedFiles.json file related to the
	 * given skin name
	 * @param {string} strSkinName : the name of the skin to be retrieved
	 * @private
	 * @return {null|array} return null if the file does not exist else return an array holding the data related to the skin configuration
	 */
	getSkinConfig : function(strSkinName){
		//building the file path to be retrieved
		var skinConfigPath = dojo.moduleUrl("ICTouchAPI").path+"Config/themes/"+strSkinName+"/providedFiles.json";
		//accessing to the file content
		var skinConfigXHR = dojo.xhrGet({
			url: skinConfigPath,
			handleAs : "json",
			sync : true
		});
		//returning the content : null or an associative array
		if(skinConfigXHR.results[0]!=null) {
			skinConfigXHR.results[0].webapp = {};
			return skinConfigXHR.results[0];
		}
		else {
			return null;
		}
	},

	/**
	 * This method check the existence of a webapp providedFiles.json file related to the
	 * given skin name
	 * @param {string} strSkinName : the name of the skin to be retrieved
	 * @param {string} strWebapp : the name of the webapp to be retrieved
	 * @param {string} strNamespace : the strNamespace of the webapp to be retrieved
	 * @private
	 * @return {null|array} return null if the file does not exist else return an array holding the data related to the webapp skin configuration
	 */
	getWebappSkinConfig : function(strSkinName, strWebapp, strNamespace){
		//building the file path to be retrieved
		var skinConfigPath = dojo.moduleUrl(strNamespace ? strNamespace : "webapp").path+strWebapp+"/config/themes/"+ strSkinName +"/providedFiles.json";

		//accessing to the file content
		var skinConfigXHR = dojo.xhrGet({
			url: skinConfigPath,
			handleAs : "json",
			sync : true,
			failOk: true
		});
		//returning the content : null or an associative array
		return (skinConfigXHR.results[0] ? skinConfigXHR.results[0]:null);
	},

	/**
	 * This method return a dom fragment holding css matching the given tag and type
	 * @param {string} headerTag : the tag identifying the css
	 * @param {string} typeCss : the type - linkedBasic or linkedIcons
	 * @param {string} skinName : name of the skin
	 * @private
	 * @return {HTMLElement} return a dom fragment
	 */
	getCss : function(headerTag,typeCss,skinName){
		if(this._arrCssHandlers[headerTag])
			return this._arrCssHandlers[headerTag].getCss(typeCss,skinName);
		else
			return null;
	},


	/**
	 * This method load a webapp style
	 * @private
	 */
	loadWebappStyle : function(strWebappName,loadSkin,loadAssociatedSkin) {
		//Load providedFiles.json
		var associatedSkinConfig = null;
		var skinConfig = this.getWebappSkinConfig(this._skinName, strWebappName);
		if(this._associatedSkinName!="") {
			associatedSkinConfig = this.getWebappSkinConfig(this._associatedSkinName, strWebappName);
		}
		//If there is no providedFiles.json for this skin, load it from defaut skin
		if(skinConfig==null && associatedSkinConfig==null) {
			skinConfig = this.getWebappSkinConfig("Default", strWebappName);
		}

		//Create handler if not exist
		if(!this._arrCssHandlers[strWebappName]) {
			//Create header fragments
			this._arrCssHandlers[strWebappName] = new ICTouchAPI.cssHandler(strWebappName);
			if(this._associatedSkinName!="") {
				this._arrCssHandlers[strWebappName].createSkinFragments(this._associatedSkinName);
			}
			this._arrCssHandlers[strWebappName].createSkinFragments(this._skinName);
		}

		//Load CSS
		if(associatedSkinConfig!=null && loadAssociatedSkin!==false) {
			dojo.mixin(this._associatedSkinConfig["webapp"],associatedSkinConfig["webapp"]);
			this._loadWebappCssOnProvidedFiles(associatedSkinConfig, strWebappName);
		}
		if(skinConfig!=null && loadSkin!==false) {
			dojo.mixin(this._skinConfig["webapp"],skinConfig["webapp"]);
			this._loadWebappCssOnProvidedFiles(skinConfig, strWebappName);
		}
	},

	/**
	 * @private
	 * Send logs of attributes of the service to the logger.
	 * @param {boolean} boolAdvancedLogs Boolean indicating if advanced or basic logs has to be written
	 */
	dump : function(boolAdvancedLogs){
		ICTouchAPI.debugServices.dump("dump function of skinServices with " + ((boolAdvancedLogs)?"advanced":"basic") + " logs option");

		ICTouchAPI.debugServices.dump("Current skin name: " + this._skinName);
		if (boolAdvancedLogs) {
			ICTouchAPI.debugServices.dump("Associated skin name: " + this._associatedSkinName);
		}
	},

	/**
	 * Method called to preload skin transition iFrame
         * @private
	 */
	_preloadTransitionFrame : function(){
		//preloading the image for the transition iFrame
		this.skinTransitionImage = new Image();
		this.skinTransitionImage.src = "/library/ICTouchAPI/themes/skintransition.jpg";
		//create the iFrame (hidded)
		this._createSkinTransitionIFrame(this.skinTransitionImage.src);
	},

	/**
	 * Method called to create and return the iFrame holding the skin transition image
	 * @param {string} strImgPath : the skin transition image path
         * @private
	 * @return {iFrame} a reference on the expected iFrame (created or accessed and returned)
	 *
	 */
	 _createSkinTransitionIFrame : function(strImgPath){
		var myFrame = dojo.byId("skinTransition");
		if(!myFrame){
			var iFrame = dojo.create("iframe",{'id':'skinTransition','src':'iframe.html'},dojo.body());
			dojo.style(iFrame,{'height':'480px','width':'800px','top':'0px','left':'-1000px','zIndex':'19000'});

			var loaded = function () {
				var body = iFrame.contentDocument.body;
				var head = dojo.query("head", iFrame.contentDocument)[0];

				// Add CSS for Spinner
				dojo.create("link", {rel: "stylesheet", type: "text/css", href: "http://127.0.0.1/library/ICTouchAPI/UIElements/Spinner/themes/Default/css/base.css"}, head);
				dojo.create("img", {src: strImgPath, style: "pointer-events: none"}, body);
				this.spinner = new UIElements.Spinner.SpinnerControl({boolForeground: true}, body);

				iFrame.removeEventListener("load", loaded, false);
			};
			iFrame.addEventListener("load", loaded, false);

			iFrame._show = function(){
				clearTimeout(this.hideTimer);
				dojo.style(this,'left','0px');
				this.hideTimer = setTimeout(function(){
					iFrame._hide();
				}, 30000);
			};

			iFrame._hide = function(){
				clearTimeout(this.hideTimer);
				dojo.style(this,'left','-1000px');
			};
			//linking the iFrame to our document

			return iFrame;
		}else{
			//returning the existing iFrame
			return myFrame;
		}
	},

	/**
	 * This method get the skin setting value (SkinName)
         * @private
	 */
	_accessSkinName : function(){
            // Default settings for simulation mode
            if(!generalConfig.simulation) {
                //accessing the skin setting value
		ICTouchAPI.settingServices.getSetting("SkinsListChoice", this, function(objSetting){
			this._skinName = (objSetting ? objSetting.value : "Default");
		},null,true);
            }
	},

	/**
	 * This method get providedFiles content within dojo object
         * @private
	 */
	_accessSkinProvidedFiles : function(){
		this._skinConfig = this.getSkinConfig(this._skinName);
		if(!this._skinConfig){
			this._skinName = "Default";
			this._skinConfig = this.getSkinConfig(this._skinName);
		}
		if(this._skinConfig.type == "Partial"){
			//the skin which will be over-written by the partial skin
			this._associatedSkinName = this._skinConfig.associatedFullSkin;
			this._associatedSkinConfig = this.getSkinConfig(this._skinConfig.associatedFullSkin);
		}

		if(this._associatedSkinName=="") {
			this._skinLevel[this._skinName] = 0;
		}
		else {
			this._skinLevel[this._associatedSkinName] = 0;
			this._skinLevel[this._skinName] = 1;
		}
	},


	/**
	 * This method loads the skins from the content of the providedFiles (handle partial skin overload)
	 * @private
	 */
	_loadSkinOnProvidedFiles : function(loadSkin, loadAssociatedSkin){
		if(this._associatedSkinName && loadAssociatedSkin!==false){
			this._loadCssOnProvidedFiles(this._associatedSkinConfig);
		}
		if(loadSkin!==false) {
			this._loadCssOnProvidedFiles(this._skinConfig);
		}
	},

	/**
	 * According to the providedFiles data, this method :
	 * - create the cssHandlers
	 * - load cs files into handlers
	 * @private
	 */
	_loadCssOnProvidedFiles : function(skinConfig){
		var skinName = skinConfig.name;
		//framework
		if(!this._arrCssHandlers["ICTouchAPI"])
			this._arrCssHandlers["ICTouchAPI"] = new ICTouchAPI.cssHandler("ICTouchAPI");
		this._arrCssHandlers["ICTouchAPI"].createSkinFragments(skinName);
		if(skinConfig["ICTouchAPI"]) {
			this._arrCssHandlers["ICTouchAPI"].loadCssOnArrayOfFile("ICTouchAPI", "", skinName, skinConfig["ICTouchAPI"]);
		}

		if(skinConfig["UIElements"]){
			if(!generalConfig.cssDeveloper){
				if(skinConfig["UIElements"]["minifiedCss"])
					this._arrCssHandlers["ICTouchAPI"].loadCssOnArrayOfFile("UIElements", "minifiedCss", skinName, skinConfig["UIElements"]["minifiedCss"]);
			}else{
				for(var key in skinConfig["UIElements"]){
					if(key != "minifiedCss")
						this._arrCssHandlers["ICTouchAPI"].loadCssOnArrayOfFile("UIElements", key, skinName, skinConfig["UIElements"][key]);
				}
			}
		}

		//webapps
		for(var key in skinConfig["webapp"]){
			if(!this._arrCssHandlers[key])
				this._arrCssHandlers[key] = new ICTouchAPI.cssHandler(key);
			//this._arrCssHandlers[key].createSkinFragments(skinName);
			this._arrCssHandlers[key].loadCssOnArrayOfFile("webapp", key, skinName, skinConfig["webapp"][key]);
		}
	},

	/**
	 * According to the webapp providedFiles data, this method :
	 * - create the cssHandlers
	 * - load cs files into handlers
         * @private
	 */
	_loadWebappCssOnProvidedFiles : function(skinConfig, strWebappName){
		var skinName = skinConfig.name;
		this._arrCssHandlers[strWebappName].loadCssOnArrayOfFile("webapp", strWebappName, skinName, skinConfig["webapp"][strWebappName]);
	},



		/**
	 * Add properties and function to the given iFrame to allow it to manage its links
	 * @param {HTMLiFrame} frame the target iFrame to be set up
         * @private
	 */
	_setUpFrame : function(frame){
		if(!frame.subToCss)
			frame.subToCss = [];

		if(!frame.onCssHandlerEvent) {
			frame.onCssHandlerEvent = function(objEvent){
				try{
					var contentDocument = frame.contentDocument;
					if(frame.contentDocument==null) {
						contentDocument = frame;
					}
					var cssType = (objEvent.type==="linkedIcons"||objEvent.type==="icons") ? "icons" : "basic";

					dojo.query("link[csstag='"+objEvent.owner.getTag()+"'][skin='"+objEvent.skinName+"'][cssType='"+cssType+"']", contentDocument.head).forEach(dojo.destroy);
					ICTouchAPI.skinServices._placeLink(objEvent.owner.getTag(), objEvent.type, objEvent.skinName, contentDocument.head);
				}catch(e){
					console.warn("An event handler remeains on a destroyed frame : "+frame.id+" :: error : "+e);
				}
			}
		}
		if(!frame.onRemoveSkinCssEvent) {
			frame.onRemoveSkinCssEvent = function(skinName){
				try{
					var contentDocument = frame.contentDocument;
					if(frame.contentDocument==null) {
						contentDocument = frame;
					}
					dojo.query("link[skin='"+skinName+"']", contentDocument.head).forEach(dojo.destroy);
				}catch(e){
					console.warn("An event handler remeains on a destroyed frame : "+frame.id+" :: error : "+e);
				}
			}
		}
	},

	/**
	 * This method checks the skin on end_of_init event
         * @private
	 */
	_checkSkinOnEndOfInit : function(){
            //subscribing to the skin seeting once the initialization has been done
            var handler = dojo.subscribe('load step 6', this, function() {				
                // Preload transitionFrame on step 6 because we need the spinner in frame
                this._preloadTransitionFrame();

                // Default settings for simulation mode
                if(!generalConfig.simulation) {
                    //subscribing to the setting
                    ICTouchAPI.settingServices.subscribeToSetting(this, "SkinsListChoice", this.skinSettingChanged);

                    //checking the current skin value
                    ICTouchAPI.settingServices.getSetting("SkinsListChoice",this, function(objSetting){
                        //changing the skin without displaying the transition iFrame
                        if(objSetting && objSetting.value != this._skinName){
                                this.skinSettingChanged(objSetting);
                        }
                        //BACKGROUND
                        //listening the setting to be able to change the background afterwards
                        ICTouchAPI.settingServices.subscribeToSetting(this, "UserBackgroundPicture", this.setBackground);
                        //load the background : on end_of_init the background may not been checked if the skin has already been loaded
                        ICTouchAPI.settingServices.getSetting("UserBackgroundPicture", this, this.setBackground);
                    });
                }

                //unsubscribing to the event
                dojo.unsubscribe(handler);
            });
	},


	/**
	 * This method update the background with the given image object and computes the dimensions :
	 *  - if the image is too big, resize it
	 *  - if the image size is lower than the one of the display, center it
	 * @param {image} img the image object to use as background image
         * @private
	 */
	_applyBackground : function(img){
		//we load it by changing the url of the DOM element with the .withBackground class
		//create the image and load it
		if(img.width > 800 || img.height > 480){
			//compute the new dimensions matching the screen
			var ratedDimension = ICTouchAPI.tools.getRatedDimensions(img.width, img.height);
			//if the widht is the base of rating for the height
			if(ratedDimension.ratioW == 1){
				//set the width has 100% of the new screen, height will be resized proportionatly
				dojo.query(".withBackground").style("background-size", "100%");
			}else{
				//else the height is the base of the rate, height will have to fit 100% of the
				//screen height, then the width can be easily deducted
				//screen width : 800, picture width : 600, expected width : 400
				//600/800 = 3/4 : the current picture use 75% of the screen
				//resize from 600 down to 400 : 400/600 = 2/3 : 66,66%
				//global resizing rate become : 3/4*2/3 = 1/2 :=> 800->400
				dojo.query(".withBackground").style("background-size", (img.width/800)*ratedDimension.ratioW*100+"%");
			}
		}else{
			dojo.query(".withBackground").style("background-size", (img.width/800)*100+"%");
		}
		dojo.query(".withBackground").style("background-position", "center");
		dojo.query(".withBackground").style("background-image", "url('" + img.src + "')");
	},

	/**
	* place tag to help place skin link
        * @private
	*/
	_placeCssPosition : function(header) {
		//Place tag to help place skin link
		var helpTag;
		var i;
		for(i=0; i<ICTouchAPI.skinServices._maxSkinLevel; i++) {
			helpTag = dojo.create("cssposition",{
				"type":"ICTouchAPI",
				"level":i
			});
			dojo.place(helpTag,header);
		}
		for(i=0; i<ICTouchAPI.skinServices._maxSkinLevel; i++) {
			helpTag = dojo.create("cssposition",{
				"type":"webapp",
				"level":i
			});
			dojo.place(helpTag,header);
		}
	},

	/**
	 * place link to head
         * @private
	 */
	_placeLink : function(headerTag,type,skinName,header) {
		var position;
		var level = this._skinLevel[skinName];
		if(headerTag==="ICTouchAPI") {
			position = dojo.query("cssposition[type=ICTouchAPI][level="+level+"]",header)[0];
		}
		else {
			position = dojo.query("cssposition[type=webapp][level="+level+"]",header)[0];
		}
		var cssHeader = this._arrCssHandlers[headerTag].getCss(type,skinName);
		if(cssHeader!=null) {
			dojo.place(cssHeader,position,"before");
		}
	},

	/**
	 * Returns the path of css files and images based on the namespace
	 * @param {webappName} webappName application name
         * @private
	 */
	getCssPath : function(webappName) {
		var arrNamespace	= webappName.split('.');
		var strPath			= '/' + arrNamespace[0] + '/' + arrNamespace[1];

		if(arrNamespace[0] == "awap" && webapp.myICWall) {
			strPath	= webapp.myICWall.data.FORWARD_PROXY_URL + webapp.myICWall.data.LABS_URL + webapp.myICWall.data.LABS_REPOSITORY + "/" + arrNamespace[1]
		}

		return strPath;
	}

});
ICTouchAPI.skinServices = new ICTouchAPI.skinServices;
/**
* @class ICTouchAPI.webApplicationManager
* @singleton
* @ignore
* @extends Object
*/
dojo.provide("ICTouchAPI.webApplicationManager");
dojo.declare("ICTouchAPI.webApplicationManager",
	null,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		currentlyLoadingApplications : 0,

		/**
		 * Time in seconds before we unload an application
		 * @property
		 * @type int
		 */
		TIME_FOR_UNLOADING: 60,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @private
		 * array of loaded web applications.
		 */
		_arrLoadedApplications : {},
		/**
		 * @private
		 */
		_arrLoadedWebapp : [],
		/**
		 * @private
		 */
		_boolSafeMode : false,
		/**
		 * @private
		 */
		_handleTick : null,
		/**
		 * @private
		 */
		_arrUnloadableApplication: [],

		/* ------------------------------------ Constructor --------------------------------------- */

                 /**
                 * @private
                 */
		constructor: function () {
			// If webapps are not loaded from the requireList, then make the
			// webApplicationManager receiptive to start_webapp events.
			if (!ICTouchAPI.devContext.loadWebappFromRequireList) {
				ICTouchAPI.eventServices.subscribeToEvent(this,"start_webapp",this._APILoadWebapp);
				this.currentlyLoadingApplications = 0;
			}
			ICTouchAPI.eventServices.subscribeToEvent(this,"end_of_init_IctDirectory",this.initDataStores);
			// If we are in remote display, call manually the initDataStores function to create the dataStore of contacts
			// (in remote display we will not receive the 'end_of_init_IctDirectory' event)'
			if (ICTouchAPI.remoteDisplay) {
				this.initDataStores();
			}
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

                /**
                 * @private
                 */
		getApplicationName : function(){
			return "webApplicationManager";
		},

		/**
		 * Get all loaded webApplications.
		 * @returns {Object} list of loaded webApplications indexed by their name.
                 * @private
		 */
		getLoadedApplications: function () {
			return this._arrLoadedApplications;
		},
		/**
		 * Get a webApplication's reference from its name.
		 * @param {String} strWebApplicationName Name of the webApplication
		 * @returns {Object} reference or false if undefined.
		 */
		getRefApplication: function (strWebApplicationName) {
			if (this._arrLoadedApplications[strWebApplicationName] == undefined) {
				return false;
			} else {
				return this._arrLoadedApplications[strWebApplicationName];
			}
		},
		/**
		 * Get the number of loaded webApplications.
		 * @returns {Int} number of loaded webApplications.
		 */
		getNumberOfApplications: function () {
			var i=0;
			for (var objApplication in this._arrLoadedApplications) {
				i++;
			}
			return i;
		},

		/**
		 * Get the current safe mode status
		 * @return {Bool} current safe mode status (on : true, off : false)
		 */
		getSafeModeStatus : function(){
			return this._boolSafeMode;
		},

		/**
		 * Set or remove the flag of an application that allows it to be unloaded when it has been closed
		 * @param {string} strWebApplicationName
		 * @param {string} boolUnloadable
		 */
		setApplicationUnloadable: function(strWebApplicationName, boolUnloadable) {
			if( boolUnloadable ) {
				this._arrUnloadableApplication[strWebApplicationName] = new Date().getTime();
			}
			else {
				delete this._arrUnloadableApplication[strWebApplicationName];
			}

			// Check if we still have unloadable application
			for(var i in this._arrUnloadableApplication) {
				this._startTickInterval();
				return;
			}

			// If we get here we don't have any app
			this._stopTickInterval();
			return;
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		initDataStores : function () {
			ICTouchAPI.dataStoreServices.createGlobalDataStore({

				name: "contacts",
				rowId: "contactId",

				// enable auto mapping
				mapping: true,

				// config object for data provider
				dataProvider: {
					onInit:   {
						func : ICTouchAPI.APIServices.IctDirectory.getLocalContacts,
						context : ICTouchAPI.APIServices.IctDirectory
					},
					onCreate: {
						event: ["DIR_CONTACTS_CREATED"],
						//func: ICTouchAPI.APIServices.IctDirectory.getContactById,
						//context : ICTouchAPI.APIServices.IctDirectory
						argumentName : "contacts" // Name of the event arguments (in case of multiple arguments)
					},
					onUpdate: {
						event: ["DIR_CONTACTS_UPDATED","DIR_CONTACTSINPREVIEW_CREATED","DIR_CONTACTSINPREVIEW_DELETED"],
						//func: ICTouchAPI.APIServices.IctDirectory.getContactById,
						//context : ICTouchAPI.APIServices.IctDirectory
						argumentName : "contacts" // Name of the event arguments (in case of multiple arguments)
					},
					onSynchro: {
						event: ["DIR_CONTENT_CHANGED"],
						//func: ICTouchAPI.APIServices.IctDirectory.getLocalContacts,
						//context : ICTouchAPI.APIServices.IctDirectory
					},
					onDelete: {
						event: ["DIR_CONTACTS_DELETED"],
						argumentName : "contactIds" // Name of the event arguments (in case of multiple arguments)
					}
				}

			});
		},

		/**
		 * First loads the list of all webapps, (webapp/requireList.js)
		 * Then loads every file in these webapps. (webapp/[oneWebapp]/requireList.js)
		 */
		loadInitFiles: function () {
			// Get the list of all webapps.
			// Load the requireList that define the webapp startup order
			dojo.require("webapp.requireList" + generalConfig.version, true);

			// Webapps are loaded from requireList
			if (ICTouchAPI.devContext.loadWebappFromRequireList) {
				webapp.startedWebappList = webapp.requireList;
			} // Else do nothing, webapp.startedWebappList is filled in by _APILoadWebapp

			this._arrLoadedWebapp = webapp.startedWebappList;

			//Unload the Settings webapp, if it is not required
			if (webapp.startedWebappList.indexOf("settings") === -1){
				this.unloadApplication("settings");
			}

			// For each webapp, get its js list of required files.
			dojo.forEach(webapp.requireList,
				function (oneWebapp) {
					if(webapp.startedWebappList.indexOf(oneWebapp) != -1 && (!webapp[oneWebapp] || (webapp[oneWebapp] && !webapp[oneWebapp]["data"]))) {
						// Load the webapp
						ICTouchAPI.webApplicationManager.loadWebapp(oneWebapp);

						// Once the webapp loaded, delete the webapp from the loaded webapp list
						var webappIndex = ICTouchAPI.webApplicationManager._arrLoadedWebapp.indexOf(oneWebapp);
						ICTouchAPI.webApplicationManager._arrLoadedWebapp.splice(webappIndex,1);
				}
					else {
						if (!webapp[oneWebapp]) {
							ICTouchAPI.debugServices.warning("ICTouchAPI.webApplicationManager - loadInitFiles / Hum, " + oneWebapp + " is not in the webapp require list.");
						}
						else {
							ICTouchAPI.debugServices.warning("ICTouchAPI.webApplicationManager - loadInitFiles / The webapp " + oneWebapp + " has been already loaded.");
						}
					}
				}
				);

			// For each webapp started but not present in the require list, load the webapp
			// following the start_webapp order list.
			// In safe mode, these webapps are not loaded
			if (!this.getSafeModeStatus()) {
				dojo.forEach(this._arrLoadedWebapp,
					function (oneWebapp) {
						if(!webapp[oneWebapp]) {
							ICTouchAPI.webApplicationManager.loadWebapp(oneWebapp);
						}
					}
					);
			}

			if (webapp.initialization && webapp.initialization.initWidget && webapp.initialization.initWidget.changeMessage) {
				webapp.initialization.initWidget.changeMessage("End of init");
			}

		},

		/**
		 * Load a web app from a namespace
		 * @param {string} strWebappName The name of the web app
		 * @param {string} strNamespace The web app namespace
		 */
		loadWebappFromNamespace : function(strWebappName, strNamespace) {
			// Load minified files
			if (!generalConfig.developer) {
				dojo.require(strNamespace + "." +strWebappName+".webapp_"+strWebappName+"_mini_"+generalConfig.version, true);
			}
			// Load normal files
			else  {
				ICTouchAPI.debugServices.debug("ICTouchAPI.webApplicationManager - loadWebappFromNamespace / require:" + strNamespace + "." + strWebappName+".requireList");
				dojo.require(strNamespace + "." + strWebappName+".requireList", true);

				dojo.forEach(awap[strWebappName]["requireList"],
					function (webappFile) {
						var webAppPath = strNamespace + "." + strWebappName;
						dojo.require(webAppPath+"."+webappFile+ generalConfig.version , true);
					}
				);
			}
		},

		/**
		 * Load a webapp with the webappname
		 * @param {String} strWebappName Name of the application to load
		 */
		loadWebapp : function(strWebappName){


			if (this._arrLoadedApplications["webapp." + strWebappName]) {
				return false;
			}
			if (webapp.initialization && webapp.initialization.initWidget && webapp.initialization.initWidget.changeMessage) {
				webapp.initialization.initWidget.changeMessage(_(["Loading webapp", strWebappName],"webapp.initialization"));
			}
			this.currentlyLoadingApplications++;
			// If not in developer mode, load minified file instead.
			if (!generalConfig.developer) {
				dojo.require("webapp."+strWebappName+".webapp_"+strWebappName+"_mini_"+generalConfig.version, true);

				// For each webapp properties
				for (var widget in webapp[strWebappName]) {
					// Check if it exists.
					if (webapp[strWebappName].hasOwnProperty(widget)) {
						// Check if it has a prototype
					/*if (webapp[strWebappName][widget] && webapp[strWebappName][widget].prototype) {
							// Check it has a cssFile property.
							var cssFile = webapp[strWebappName][widget].prototype.cssFile;
							if (cssFile) {
								// If it's an object
								if(typeof cssFile === "object") {
									// load every css file.
									dojo.forEach(cssFile,function(css){
										ICTouchAPI.skinServices.loadCssSkinFile("webapp."+strWebappName, css, false);
									});
								}
								else {
									ICTouchAPI.skinServices.loadCssSkinFile("webapp."+strWebappName, cssFile, false);
								}
							}
					}*/
					}
				}
			} else {
				dojo.require("webapp."+strWebappName+".requireList", true);
				dojo.forEach(webapp[strWebappName]["requireList"],
					function (webappFile) {
						var webAppPath = "webapp."+strWebappName;

						dojo.require(webAppPath+"."+webappFile+ generalConfig.version, true);
						//Load related css file(s)
					/*if(webapp[strWebappName][webappFile] && webapp[strWebappName][webappFile].prototype){
							var cssFile = webapp[strWebappName][webappFile].prototype.cssFile;
							if(cssFile){
								if(typeof cssFile === "object"){
									dojo.forEach(cssFile,function(css){
										ICTouchAPI.skinServices.loadCssSkinFile(webAppPath, css, false);
									});
								}
								else{
									ICTouchAPI.skinServices.loadCssSkinFile(webAppPath, cssFile, false);
								}
							}
					}*/
					}
					);
			}

			//Load webapp skins
			ICTouchAPI.skinServices.loadWebappStyle(strWebappName);

			if(webapp[strWebappName].loaded) {
				webapp[strWebappName].loaded();
			}
			if(typeof(webapp[strWebappName].load) == "function"){
				ICTouchAPI.addOnInit(webapp[strWebappName], function(){
					//try{
					if(webapp[strWebappName]){
						webapp[strWebappName].load();
					}
				});
			}

		},

		/**
		 * register a webApplication if it doesn't exist.
		 * @throws error existing application if it does already exist.
		 */
		registerApplication:function(objApplication){
			// Before registering a new webapp, check if it exists.
			if(this._arrLoadedApplications[objApplication.getApplicationName()]){
				throw {
					errorType : 'existing application',
					message : objApplication.getApplicationName() + ' application is already loaded'
				}
			}
			else{
				// If it doesn't exist.
				this._arrLoadedApplications[objApplication.getApplicationName()]=objApplication;
				// Notify it has been added.
				this._notifyAddedWebApp(objApplication.getApplicationName());
				// Decreases the loading applications counter.
				this.currentlyLoadingApplications --;
			}
		},

		unloadApplication: function (strApplicationName) {
			ICTouchAPI.debugServices.debug("ICTouchAPI.webApplicationManager - unloadApplication $ strApplicationName: " + strApplicationName);
			var split = strApplicationName.split(".");
			var appName = (split[0] == "webapp" ? split[1] : split[0]);
			if (typeof this._arrLoadedApplications["webapp."+appName] == "object") {

				this._arrLoadedApplications["webapp."+appName].unload()
				this.closeApplication("webapp."+appName);

			}

			delete this._arrLoadedApplications["webapp."+appName];
			delete webapp[appName];
		},

		unloadWebappFromNamespace : function(appName, namespace) {
			var listWidget = awap[appName].getLoadedWidgets();
			for( var strWidget in listWidget ) {
				var objWidget = listWidget[strWidget];
				ICTouchAPI.tools.destroyWidgets(objWidget.domNode);
				objWidget.webapp = null;
				delete listWidget[strWidget];
			}
			awap[appName].removeWidgetsRef();

			ICTouchAPI.debugServices.debug("ICTouchAPI.webApplicationManager - loadWebappFromNamespace / remove loadedModules for app: " + appName);

			var cpt	= 0;
			var regex = new RegExp('^awap\.'+appName, "gi");
			for(var i in dojo._loadedModules) {
				if(i.match(regex)) {
					var viewName = i.split(".");
					ICTouchAPI.AppBarServices.destroyAppBar(appName, viewName[2]);
					ICTouchAPI.AppBarServices.removeExitButton(appName, viewName[2]);
					delete dojo._loadedModules[i];
					cpt++;
				}
			}

			ICTouchAPI.debugServices.debug("ICTouchAPI.webApplicationManager - loadWebappFromNamespace / remove loadedUrls for app: " + appName);
			
			for (var i=dojo._loadedUrls.length-cpt;i < dojo._loadedUrls.length;i++) {
				var path = dojo._loadedUrls[i];
				dojo._loadedUrls[path] = false;
			}

			dojo._loadedUrls.splice(dojo._loadedUrls.length-cpt, cpt);
			ICTouchAPI.transitionServices._removeWebappScreens(namespace+"."+appName);
			delete this._arrLoadedApplications[namespace+"."+appName];
			delete awap[appName];
		},

		/**
		 * Destroy and remove any widget bounded to the webapp
		 * @param {string} strApplicationName name of the webapp
		 */
		closeApplication: function (strApplicationName) {
			ICTouchAPI.debugServices.debug("ICTouchAPI.webApplicationManager - closeApplication $ strApplicationName: " + strApplicationName);
			var objWebapp = dojo.getObject(strApplicationName);

			if( typeof objWebapp == "undefined" ) {
				ICTouchAPI.debugServices.warning("ICTouchAPI.webApplicationManager - closeApplication / Application " + strApplicationName + " does not exist");
				return;
			}

			var listWidget = objWebapp.getLoadedWidgets();
			for( var strWidget in listWidget ) {
				var objWidget = listWidget[strWidget];
				ICTouchAPI.tools.destroyWidgets(objWidget.domNode);

				delete listWidget[strWidget];
			}

			ICTouchAPI.transitionServices._removeWebappScreens(strApplicationName);
			objWebapp.removeWidgetsRef();

		},

		/**
		 * Notify the manager that we left an application
		 * This should only be called by transitionServices unless you know what you are doing
		 * @param {string} strApplicationName Name of the webapp
		 */
		notifyApplicationExit: function(strApplicationName) {


			if( typeof this._arrUnloadableApplication[strApplicationName] != "undefined" ) {
				// Unload this when the time has come
				this._arrUnloadableApplication[strApplicationName] = new Date().getTime();
			}
		},

		/**
		 * Notify the manager that we enterd an application
		 * This should only be called by transitionServices unless you know what you are doing
		 * @param {string} strApplicationName Name of the webapp
		 */
		notifyApplicationEnter: function(strApplicationName) {
			if( typeof this._arrUnloadableApplication[strApplicationName] != "undefined" ) {
				// Don't unload this at the moment
				this._arrUnloadableApplication[strApplicationName] = -1;
			}
		},

		/**
		 * Set/unset the safe mode
		 * @return {Bool} current safe mode status (on : true, off : false)
		 */
		toggleSafeMode : function(){
			this._boolSafeMode = !this._boolSafeMode;
			return this._boolSafeMode;
		},

		applicationsStillLoading: function () {
			return this.currentlyLoadingApplications > 0 ? true : false;
		},

		/* --------------------------------- Private Methods -------------------------------------- */


		/**
		 * @private
		 */
		_APILoadWebapp : function(objEvent){
			webapp.startedWebappList = webapp.startedWebappList || [];
			webapp.startedWebappList.push(objEvent.value);
		},

		/**
		 * @private
		 * Sends an notification when a webapp is launched/created.
		 * @param {String} webApplicationName Name of the webApplication
		 */
		_notifyAddedWebApp: function (webApplicationName) {
			dojo.publish("webApplicationManager.notify.added", [webApplicationName]);
		},
		/**
		 * @private
		 * Sends an notification when a webapp is closed/killed.
		 * @param {String} webApplicationName Name of the webApplication
		 */
		_notifyRemovedWebApp: function (webApplicationName) {
			dojo.publish("webApplicationManager.notify.removed", [webApplicationName]);
		},

		/**
		 * @private
		 * Stop the setInterval of the tick function do nothing if it isn't launched
		 */
		_stopTickInterval: function() {
			if( this._handleTick != null ) {
				// Stop tick function
				clearInterval(this._handleTick);
				this._handleTick = null;
			}
		},

		/**
		 * @private
		 * Start the setInterval of the tick function do nothing if it is already started
		 */
		_startTickInterval: function() {
			if( this._handleTick == null ) {
				// Create a tick function for unloading ( call it every 10seconds )
				var that = this;
				var func = function () {
					that._unloadTick();
				}
				this._handleTick = setInterval(func, 10000);
			}
		},

		/**
		 * @private
		 * Tick function called every 10 seconds it check if an app needs to be unloaded
		 */
		_unloadTick: function() {
			var now = new Date().getTime();
			var intervalInMS = this.TIME_FOR_UNLOADING * 1000;
			for(var strWebapp in this._arrUnloadableApplication) {
				var lastSeen = this._arrUnloadableApplication[strWebapp];
				if( lastSeen > 0 && ( lastSeen + intervalInMS ) < now ) {
					this._arrUnloadableApplication[strWebapp] = -1;
					this.closeApplication(strWebapp);
				}
			}
		}
	});

// Create Application Manager.
ICTouchAPI.webApplicationManager=new ICTouchAPI.webApplicationManager;
/**
* @class ICTouchAPI.webApplication
* @singleton
* @extends Object
* @ignore
*/
dojo.provide("ICTouchAPI.webApplication");
dojo.declare("ICTouchAPI.webApplication",
	null,
	{

		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @private
		 */
		_arrLoadedWidgets: null,
		/**
		 * @private
		 */
		_strApplicationName: '',
		/**
		 * @private
		 */
		_myIFrame : null,

		/* ------------------------------------ Constructor --------------------------------------- */

		/**
		 * @private
		 */
		constructor: function (){
			this._strApplicationName = this.declaredClass;
			this._arrLoadedWidgets = {};
			this._registerWebApp();
		},

		/**
		 * @private
		 */
		load: function () {

		},

		/**
		 * @private
		 */
		unload: function () {
			return true;
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * Get the webApplication's name.
		 * @return {String} name of the webApplication.
		 * @private
		 */
		getApplicationName: function () {
			return this._strApplicationName;
		},
		/**
		 * Get all loaded widgets for this webApplication.
		 * @return {List} List of widgets references.
		 * @private
		 */
		getLoadedWidgets: function () {
			return this._arrLoadedWidgets;
		},
		/**
		 * Get Number of loaded widgets.
		 * @return {int} number of loaded widgets.
		 * @private
		 */
		getNumberOfWidgets: function () {
			var i=0;
			for (var widget in this._arrLoadedWidgets) {
				i++;
			}
			return i;
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		* Create a widget.
		* @param {String} widgetName Name of the widget
		* @param {Object} node where to place the widget (dojo.byId(node))
		* @param {List}   args list of arguments to pass to the widget
		* @private
		*/
		createWidget: function (widgetName, args, node) {
			return new this[widgetName](args, node);
		},

		/**
		 * Called after widget creation to register it in its webapp.
		 * @private
		 */
		registerWidget: function (objWidget){
			// Before registering a new widget, check if it exists.
			if(this._arrLoadedWidgets[objWidget.strWidgetName]){
				throw {
					errorType : 'existing widget',
					message : 'This widget is already loaded'
				}
				return false;
			}
			else{
				// If it doesn't exist.
				this._arrLoadedWidgets[objWidget.strWidgetName]=objWidget;
				this._notifyAddWidget(objWidget.strWidgetName);
				return true;
			}
		},

		/**
		 * Remove the references to any widget from the control and data.
		 * If the automatic behavior fails you must implement it in the webapp control
		 * @private
		 */
		removeWidgetsRef: function() {
			// Remove every reference to a widget in the control and data
			this._removeWidgetRefFromObj(this);
			this._removeWidgetRefFromObj(this.data);
		},

		/**
		 * Remove a webapp's widget.
		 * @param {String} strWidgetName Name of the widget to remove.
		 * @private
		 */
		removeWidget: function (strWidgetName) {
			// Remove widget from array.
			delete this._arrLoadedWidgets[strWidgetName];
			this._notifyRemovedWidget(strWidgetName);
			return true;
		},

		/**
		 * unlock the transition to a another widget
		 * @param {string} name Application where you want to switch
		 * @private
		 */
		unlock : function(name){
			console.log("unlock webapp : "+this._strApplicationName);
			//this function is inherited by widgets which have to be over-write this one if they do
			//have to check their state when they are left
			//we implement this mother function in order to standarize the process and avoid errors
			dojo.publish("unlockTransition",[true]);
		},


		onHide: function() {
			//* DEBUG */ console.log( "webapp onHide", this._strApplicationName );
		},

		onShow: function( screenName ) {
			//* DEBUG */ console.log( "webapp onShow", this._strApplicationName, "- screen is", screenName );
		},

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @private
		 * Register webapp object in Application Manager
		 */
		_registerWebApp: function (){
			ICTouchAPI.webApplicationManager.registerApplication(this);
		},

		/**
		 * @private
		 * Remove every reference to a widget
		 * @param {Object} objToPurge to remove every reference to a widget
		 */
		_removeWidgetRefFromObj: function(objToPurge) {
			for( var ref in objToPurge ) {
				// Obj that will be tested
				var objToTest = objToPurge[ref];

				if( typeof objToTest == "object" && objToTest != null ) {
					if( objToTest instanceof dijit._Widget && objToTest.id != "" ) {
						delete objToPurge[ref];
					}
				}
			}
		},

		/**
		 * @private
		 * Publishes the information that a widget from this Webapp has been removed.
		 * @param {String} strWidgetName Name of the widget
		 */
		_notifyRemovedWidget: function (strWidgetName) {
			dojo.publish("webApplication.notify.removed", [[this._strApplicationName, strWidgetName]]);
		},
		/**
		 * @private
		 * Publishes the information that a widget from this Webapp has been added.
		 * @param {String} strWidgetName Name of the widget
		 */
		_notifyAddWidget: function (strWidgetName) {
			dojo.publish("webApplication.notify.added", [[this._strApplicationName, strWidgetName]]);
		},

		/**
		 * @private
		 * Send logs to the logger. To be redifined by the webapps according to their needs
		 * @param {boolean} boolAdvancedLogs Boolean indicating if advanced or basic logs has to be written
		 * (management of this boolean to be done by the webapp redefining this function)
		 */
		dump: function (boolAdvancedLogs) {
			ICTouchAPI.debugServices.dump("dump function of " + this.getApplicationName() + " with " + ((boolAdvancedLogs)?"advanced":"basic") + " logs option");
		}
	});

/**
* @class ICTouchAPI.transitionServices
* @singleton
* @extends Object
* Manage the construction of new screens and the transition between screens
*/
dojo.provide("ICTouchAPI.transitionServices");

dojo.declare("ICTouchAPI.transitionServices",null,
{

	/* --------------------------------- Public attributes ------------------------------------ */




	/* --------------------------------- Private attributes ----------------------------------- */



	/* ------------------------------------ Constructor --------------------------------------- */

	/**
	 * reference of displayed widget
	 * @ignore
	 */
	_objCurrentScreen : null,


	/**
	 * reference of displayed actionBar
	 * @ignore
	 */
	_objCurrentActionBar: null,


	/**
	 * this attribut sets on or off the transition effect
	 * @ignore
	 */
	_transitionMode:false,

	/**
	 * @ignore
	 */
	_history: [],

	/**
	 * @ignore
	 */
	_stealthIScreens: {},

	/**
	 * @ignore
	 */
	_IScreens:{},

	/**
	 * @ignore
	 */
	_IScreensByWebapps: {},

	/**
	 * @ignore
	 */
	_IActionBars:{},

	/**
	 * @ignore
	 */
	_IActionBarsByWebapps: {},

	/**
	 * @ignore
	 */
	_isChangingScreen : false,

	/**
	 * @ignore
	 */
	_defaultScreen : {},

	/**
	 * @ignore
	 */
	_handlerSpinner : null,

	/**
	 * @ignore
	 */
	_objSpinner : null,

	/**
	 * @ignore
	 */
	_transitionHandler : null,

	/**
	 * @ignore
	 */
	_targetArgs : null,

	/**
	 * @ignore
	 */
	_transitionTimer : 0,

	/* ------------------------------------ Constructor --------------------------------------- */

	/**
	 * The Transition Services manages contruction of new screens, an manages graphical transition between screens
	 */
	constructor : function(){
		this._defaultScreen = {
			name: "webapp.homepage.getHomepage",
			params: {},
			position: "Middle"
		};

		this._transitionHandler = dojo.subscribe("unlockTransition",null,function(unlockState){
			if(unlockState){
				if(ICTouchAPI.transitionServices._transitionTimer){
					clearTimeout(ICTouchAPI.transitionServices._transitionTimer);
					ICTouchAPI.transitionServices._transitionTimer=null;
				}
				if(!ICTouchAPI.transitionServices._targetArgs.sync){
					ICTouchAPI.transitionServices._transitionTimer = setTimeout(function(){
						ICTouchAPI.transitionServices._moveToScreen(ICTouchAPI.transitionServices._targetArgs);
					}, 15);
				}else{
					ICTouchAPI.transitionServices._moveToScreen(ICTouchAPI.transitionServices._targetArgs);
				}
			}
			else {
				ICTouchAPI.transitionServices._isChangingScreen = false;
			}
		});

		ICTouchAPI.eventServices.subscribeToEvent(this, "presentationLoaded", this._presentationLoaded);
	},


	/* ----------------------------------- Getter / Setter------------------------------------- */

	/**
	 * Get the Application Service name
	 * @ignore
	 * @return {String} The Application Service Name
	 */
	getApplicationName:function(){
		return "transitionServices";
	},

	/**
	 * If we are switching to a screen, return its name, else return the current screen
	 * @ignore
	 * @return {String} The screen name
	 */
	getNextOrCurrentScreenName:function() {
		if (this._isChangingScreen) {
			return this._targetArgs.name;
		}
		return this.getCurrentScreenName();
	},
	
	/**
	 * Get the Current Screen Name
	 * @param {Boolean} visible if true only return a visible screen name, search through history latest non stealthy screen
	 * @return {String} The screen name
	 */
	getCurrentScreenName:function(visible){
		// return false if _objCurrentScreen is not defined (case of getCurrentScreenName called by getScreen for initialization webapp, no screen was previously existing)
		if (!this._objCurrentScreen) {
			return false;
		}
		var id = this._objCurrentScreen.id;
		if (this.isstealth(id) && visible) {
			// When no screen has been found, return default screen name
			id = this._defaultScreen.name;
			// Search the history for a non stealthy screen from newest to oldest
			var i = this._history.length;
			var name;
			for(var i = this._history.length-1; i >= 0; i--) {
				name = this._history[i];
				if (!this.isstealth(name)) {
					id = name;
					break;
				}
			}
		}
		return id;
	},

	/**
	 * Get the IFrame containing the current WebApp
	 * @ignore
	 * @return {Object} The IFrame
	 */
	getCurrentWebAppIFrameDocument: function () {
		return this._objCurrentScreen.getContentDocument();
	},

	/**
	 * Get the IFrame containing the current WebApp
	 * @ignore
	 * @return {Object} The IFrame
	 */
	getCurrentWebAppIFrame: function () {
		return this._objCurrentScreen._screenFrame;
	},

	/* ----------------------------------- Public methods ------------------------------------- */

	/**
	 * Get a screen, if screens exists, it is upped to the top and updated. Else, a new screen is created
	 * @param {Object} args JSON Object containing:<pre><code>
	 * - name: the name of the screen
	 * - params: the list of parameters
	 * - position: The position of the screen (ie: middle)
	 * - zindex: The z order index</code></pre>
	 * @return {Boolean} True if a transition has been done
	 */
	getScreen : function(args){
		// return false if the screen to get is the current screen displayed
		if(!dojo.getObject(args.name) || args.name == this.getCurrentScreenName(false)){
			return false;
		}
		if (this._isChangingScreen && !args.sync) {
			return false;
		}
		console.log("getScreen to :"+args.name);
		this._isChangingScreen = true;
		this._targetArgs = args;
		this._unlock(args.name);
	},

	/**
	 * Get the default screen
	 * @return {boolean} Was the transition successful
	 */
	getDefaultScreen : function() {
		return this.getScreen(this._defaultScreen);
	},

	/**
	 * Preload a screen when it doesn't exist
	 * @param {Object} args JSON Object containing:<pre><code>
	 * - name: the name of the screen
	 * - params: the list of parameters
	 * - position: The position of the screen (ie: middle)
	 * - zindex: The z order index</code></pre>
	 */
	preloadScreen : function(args) {
		// Some safeguards
		if( !args || typeof args.name !== "string" ) {
			console.warn("Please see the documentation on how to use preloadScreen");
			return;
		}
		if( !this._testScreenName(args.name) ) {
			console.warn("Screen name "+args.name+" does not exist");
			return;
		}
		// End of safeguards
		// Create the new screen if it doesn't exist
		if (!this._getIScreen(args.name)) {
			var screen = this._createIScreen(args.name, args.params, args.position || "Middle");

			screen._preFirstShow();

			// Hide the preloaded iframes just in case
			screen._hide();
		}
	},


	/**
	 * Display the Previous Screen
	 * @return {boolean} Was the transition successful
	 */
	back : function () {
		var lastScreen = this._history.pop();
		if (typeof lastScreen !== "undefined") {
			if (!this.isstealth(lastScreen)) {
				return this.getScreen({
					name : lastScreen,
					params: {}
				});
			} else {
				return this.back();
			}
		} else {
			return this.getDefaultScreen();
		}


	},

	/**
	 * Stealth a Widget
	 * A stealth widget is not added to the history stack
	 * @param {String} strWidgetName The name of the Widget
	 */
	stealth :function(strWidgetName) {
		this._stealthIScreens[strWidgetName] = true;
	},

	/**
	 * The widget will be added to the history stack again
	 * @param {String} strWidgetName The name of the Widget
	 */
	unstealth: function (strWidgetName) {
		this._stealthIScreens[strWidgetName] =  false;
	},

	/**
	 * Is a Widget Stealthy ?
	 * @param {String} strWidgetName The name of the Widget
	 * @return {Boolean} True is the Widget is Stealthy
	 */
	isstealth: function (strWidgetName) {
		return this._stealthIScreens[strWidgetName];
	},


	/**
	 * Set the first screen
	 * @param args params for the first widget constructor
	 * @ignore
	 */
	initialize : function(args){
		var screen;
		for (var i=0; i<args.length; i++) {
			screen = this._createIScreen(args[i].name,  args[i].params, args[i].position);
			// Show it because screens are created hidden
			screen._show();

		}
	},

	/**
	 * Add a stylesheet link to the header of the generic iframes
	 * @param {String} path : the path to the stylesheet file to link to the generic iframes
	 * @ignore
	 */
	addStyleToGenericIFrame : function(path) {
		ICTouchAPI.debugServices.debug("ICTouchAPI.transitionServices - addStyleToGenericIFrame / add to the generic iFrames the stylesheet with the path " + path);
		var i;
		// Ugly hack: get static iframes
		var screenObj = ICTouchAPI.transitionServices.screen.prototype._static_iframePool;

		// For each predefined bottom iframe
		for(i=0 ; i < screenObj.appbarFrame.length; i++) {
			ICTouchAPI.skinServices.addStyleToHeader(screenObj.appbarFrame[i]._getHead(), path);
		}

		// For each predefined screen iframe
		for(i=0; i < screenObj.screenFrame.length; i++) {
			ICTouchAPI.skinServices.addStyleToHeader(screenObj.screenFrame[i]._getHead(), path);
		}
	},

	/**
	 * Remove a stylesheet link from the header of the generic iframes
	 * @param {String} path : the path to the stylesheet file to remove from the generic iframes
	 * @ignore
	 */
	removeStyleToGenericIFrame : function(path) {
		ICTouchAPI.debugServices.debug("ICTouchAPI.transitionServices - removeStyleToGenericIFrame / remove to the generic iFrames the stylesheet with the path " + path);
		var i;
		// Ugly hack: get static iframes
		var screenObj = ICTouchAPI.transitionServices.screen.prototype._static_iframePool;

		// For each predefined bottom iframe
		for(i=0 ; i < screenObj.appbarFrame.length; i++) {
			ICTouchAPI.skinServices.removeStyleToHeader(screenObj.appbarFrame[i]._getHead(), path);
		}

		// For each predefined screen iframe
		for(i=0; i < screenObj.screenFrame.length; i++) {
			ICTouchAPI.skinServices.removeStyleToHeader(screenObj.screenFrame[i]._getHead(), path);
		}
	},

	/**
	 * Add CSS rule to the DOM Header of the generic iFrames
	 * @param {String} selector : the CSS Selector to add in the DOM Header of the generic iFrames
	 * @param {String} rule : the rule attached to the CSS Selector to add in the DOM Header of the generic iFrames
	 * @ignore
	 */
	addCssRuleToGenericIFrame : function(selector, rule) {
		ICTouchAPI.debugServices.debug("ICTouchAPI.transitionServices - addCssRuleToGenericIFrame / add to the generic iFrames the css selector " + selector + ", with the rule " + rule);
		var i;
		// Ugly hack: get static iframes
		var screenObj = ICTouchAPI.transitionServices.screen.prototype._static_iframePool;

		// For each predefined bottom iframe
		for(i=0 ; i < screenObj.appbarFrame.length; i++) {
			ICTouchAPI.skinServices.addCssRuleToFrame(screenObj.appbarFrame[i], selector, rule);
		}

		// For each predefined screen iframe
		for(i=0; i < screenObj.screenFrame.length; i++) {
			ICTouchAPI.skinServices.addCssRuleToFrame(screenObj.screenFrame[i], selector, rule);
		}
	},

	/**
	 * Remove a CSS rule from the DOM Header of the generic iFrames
	 * @param {String} selector: The CSS Selector to remove from the DOM Header of the generic iFrames
	 * @ignore
	 */
	removeCssRuleToGenericIFrame : function(selector) {
		ICTouchAPI.debugServices.debug("ICTouchAPI.transitionServices - removeCssRuleToGenericIFrame / remove to the generic iFrames the css selector " + selector);
		var i;
		// Ugly hack: get static iframes
		var screenObj = ICTouchAPI.transitionServices.screen.prototype._static_iframePool;

		// For each predefined bottom iframe
		for(i=0 ; i < screenObj.appbarFrame.length; i++) {
			ICTouchAPI.skinServices.removeCssRuleToFrame(screenObj.appbarFrame[i], selector);
		}

		// For each predefined screen iframe
		for(i=0; i < screenObj.screenFrame.length; i++) {
			ICTouchAPI.skinServices.removeCssRuleToFrame(screenObj.screenFrame[i], selector);
		}
	},


	/* ----------------------------------- Private methods ------------------------------------- */

        /**
         * @ignore
         */
	_presentationLoaded: function() {
		var lstWebapp = [];
		for(var i in webapp) {
			if(webapp[i]._strApplicationName!=null) {
				lstWebapp.push(webapp[i]._strApplicationName);
			}
		}
		ICTouchAPI.skinServices.linkWebappsStyles("webapp.mainframe",lstWebapp,false,false,true,false);
		ICTouchAPI.skinServices.linkWebappsStyles("webapp.appbarframe",lstWebapp,true,false,false,false);
		ICTouchAPI.eventServices.unsubscribeToEvent(this,"presentationLoaded");
	},

	/**
	 * Check if this screen exist and can be instantiated
	 * @ignore
	 * @param {String} screenName name of the screen to test
	 * @returns {boolean} if it exist
	 */
	_testScreenName: function(screenName) {
		var screen = ICTouchAPI.tools.getObjectReference(screenName);
		return (typeof screen === "function");
 	},

	/**
	 * @ignore
	 */
	_unlock : function(){
		dojo.publish("unlockTransition", [true]);
	},

	/**
	 * @ignore
	 */
	_basicUnlock : function(){
		dojo.publish("unlockTransition", [true]);
	},

	/**
	 * @ignore
	 */
	_getNextUnlock : function(name){
		var reg = /^webapp\.[^.]+/;
		var webappName = reg.exec(name);
		if(webappName == null) {
			reg = /^awap\.[^.]+/;
			webappName = reg.exec(name);
		}
		var refOnObject = dojo.getObject(webappName[0]);
		return dojo.hitch(
			(refOnObject ? refOnObject:{}),
			(refOnObject.unlock ? refOnObject.unlock : this._basicUnlock)
			);
	},

	/**
	 * @ignore
	 */
	_moveToScreen : function(args){
		// Some safeguards
		if( !args || typeof args.name !== "string" ) {
			console.warn("Please see the documentation on how to use getScreen");
			return false;
 		}
		if( !this._testScreenName(args.name) ) {
			console.warn("Screen name "+args.name+" does not exist");
			return false;
		}
		// End of safeguards

		args.params = args.params || {};
		/*if (this._isChangingScreen) {
 			return false;
 		}*/
		this._unlock = this._getNextUnlock(this._targetArgs.name);


		// Hide current screen if it's different than the required one
		if (this._objCurrentScreen && this._objCurrentScreen.id != args.name) {
			this._objCurrentScreen._hide();
			// Save history
			this._history.push(this._objCurrentScreen.id);
			this._checkApplicationExit(this._objCurrentScreen.id, args.name);
		}
		// Create the new screen if it doesn't exist
		if (!this._getIScreen(args.name)) {
			this._createSpinner();
			this._createIScreen(args.name, args.params, args.position || "Middle");
		}


		// If it's not the current one, show it.
		if (this._objCurrentScreen == null || this._objCurrentScreen.id != args.name) {
			this._getIScreen(args.name)._show();
		}

		if (args.zindex) {
			try{
				dojo.style(this._getIScreen(args.name), "z-index", args.zindex);
			}catch(err){
				console.warn("transitionServices.getScreen : trying to set a Zindex but argument is not a DOM object");
				console.warn(err);
			}
		}

		this._objCurrentScreen = this._getIScreen(args.name);

		this._isChangingScreen = false;
		this._removeSpinner();

		return true;
	},

	/**
	 * @ignore
	 */
	_getIScreen: function (strScreenName) {
		return this._IScreens[strScreenName];
	},

	/**
	 * Create an iframe and attach some utility method to it
	 * @ignore
	 */
	_createIFrame: function (strIFrameId, position, callback) {
		var documentFragment = document.createDocumentFragment();
		var iframe = document.createElement("iframe");
		documentFragment.appendChild(iframe);
		iframe.id = strIFrameId;
		iframe.src = "iframe.html";
		iframe.style.left="-1000px"; // Make the iframe hidden by default
		iframe.className = position || "Middle";

		if( typeof callback === "function" ) {
			var firstLoad = function() {
				// Remove listeners to avoid memory leaks
				iframe.removeEventListener('load', firstLoad, false);

				// iframe context in callback
				callback.call(iframe);
			}
			iframe.addEventListener('load', firstLoad, false);
 		}

		iframe._show = function () {
			this.style.left = "";
			var id = this.id;
			// publish an event when this iframe is visible
			setTimeout(function(){
				dojo.publish("iframe.show", [id]);
			}, 100);
		};
		iframe._hide = function () {
			// publish an event when this iframe is hidden
			dojo.publish("iframe.hide", [this.id]);
			this.style.left = "-1000px";
		};
		iframe._getHead = function () {
			return dojo.query("head",iframe.contentDocument)[0];
		}
		document.body.appendChild(documentFragment);
		return iframe;
	},

	/**
	 * @ignore
	 */
	_removeWebappScreens: function (strWebappName) {
		var iframes = this._IScreensByWebapps[strWebappName];
		if (iframes && iframes.length) {
			for (var i=0; i < iframes.length; i++) {
				var iframeId = iframes[i].id;
				this._removeScreen(iframeId, strWebappName);
			}
			delete this._IScreensByWebapps[strWebappName];
		}
	},

	/**
	 * Add a screen in the stack if transition enable else in the GlobalContainerMiddle
	 * Add the equivalent action bar to the _arrActionBars array
	 * @ignore
	 * @param {String} strScreenName name of the screen to create
	 * @param {Object} arrParams params used for the creation of the screen
	 * @param {Object} position location of the screen ( "Middle", "FullScreen", etc... )
	 * @return {Object} newly created screen
	 */
	_createIScreen:function(strScreenName, arrParams, position){
		var screen;
		var tmpObj = dojo.getObject(strScreenName);
		var screenContent = new tmpObj(arrParams);

		position = position || "Middle";

		// This is needed so ICTouchAPI.tools.getFrameId can work even when iframe isn't fully initialized
		screenContent.domNode.setAttribute("frameid", strScreenName);

		// If it's a webapp view use the new screen class, else create a simple iframe for it
		if( position === "Middle" ) {
			screen = new ICTouchAPI.transitionServices.screen(screenContent, strScreenName, screenContent.exclusiveIframe === true);
		}
		else {
			// Every screen that is not a "normal" screen is created the old way
			screen = this._createIFrame(strScreenName, position, function () {
				ICTouchAPI.skinServices.registerHeader(this._getHead(), screenContent.namespace, "webapp", this);
				screenContent.placeAt(this.contentDocument.body);
			});
		}

		this._IScreens[strScreenName] = screen;

		this._IScreensByWebapps[screenContent.namespace] = this._IScreensByWebapps[screenContent.namespace] || [];
		this._IScreensByWebapps[screenContent.namespace].push(screen);

		return screen;
	},

	/**
	 * Destroy a screen
	 * @ignore
	 * @param {String} screenName name of the screen to destroy / remove
	 */
	_removeScreen: function (strScreenName, strWebappName) {
		var objScreen = this._getIScreen(strScreenName);

		if( !objScreen ) {
			return;
		}
		if( typeof objScreen.destroy === "function" ) {
			// It's a "complex" screen, call the destroy method of it
			objScreen.destroy();
		}
		else {
			ICTouchAPI.skinServices.unregisterHeader(objScreen);
			// It's an iframe, do the cleaning stuff ourself
			dojo.empty(objScreen.contentDocument.body);
			// Remove iframe from dom
			dojo.destroy(objScreen);
		}
		delete this._IScreens[strScreenName];

		// Add it to stealth list
		this.stealth(strScreenName);
	},

	/**
	 * Replace the appbar from screen with an another one
	 * @ignore
	 * @param {String} screenName name of the screen where the appbar should be placed
	 * @param {Object} actionBar AppBarWidget object
	 */
	_setActionBar: function(screenName,refActionBar) {
		var screen = this._getIScreen(screenName);
		if( screen ) {
			screen.replaceAppbar(refActionBar);
 		}
 	},

	/*
	 * Returns an array containing the contentDocument of each loaded webapp's view
	 * @ignore
	 * Use with care it's not a toy
	 */
	getLoadedViewsContainer : function () {
		var _arr = [], i;
		for (i in this._IScreens) {
			if (this._IScreens.hasOwnProperty(i)) {
				_arr.push(this._IScreens[i].contentDocument);
			}
		}
		return _arr;
	},

	/*
	 * Check if we are leaving the current application and notify webApplicationManager if it's the case
	 * @ignore
	 * @param {String} Name of the view we are leaving
	 * @param {String} Name of the view we are entering
	 */
	_checkApplicationExit: function(strOldView, strNewView) {
		var splitOld = strOldView.split(".");
		var splitNew = strNewView.split(".");

		if( splitOld[1] != splitNew[1] ) {
			var appName = "webapp."+splitOld[1];
			ICTouchAPI.webApplicationManager.notifyApplicationExit(appName);
			appName = "webapp."+splitNew[1];
			ICTouchAPI.webApplicationManager.notifyApplicationEnter(appName);
		}
	},

	/**
	 * Create the spinner UIElement
	 * @ignore
	 */
	_createSpinner: function() {
		this._destroySpinner();
		this._objSpinner = new UIElements.Spinner.SpinnerControl({}, document.body);
	},

	/**
	 * Remove the spinner UIElement after 4 seconds
	 * @ignore
	 */
	_removeSpinner: function() {
		// Delay the end of the animation by 4seconds because we don't know when the images in CSS are fully loaded
		// Homepage seems to take that amount of time to load
		var context = this;
		this._handlerSpinner = setTimeout(function(){
			context._destroySpinner();
		}, 4000);
	},

	/**
	 * Destroy the spinner and don't fail if it isn't present
	 * @ignore
	 */
	_destroySpinner: function() {
		if( this._handlerSpinner ) {
			clearTimeout(this._handlerSpinner);
			this._handlerSpinner = null;
		}
		if( this._objSpinner ) {
			this._objSpinner.destroy();
			this._objSpinner = null;
		}
	}
});

// Create transition services
ICTouchAPI.transitionServices=new ICTouchAPI.transitionServices();

/**
* @class ICTouchAPI.transitionServices.screen
* @ignore
* @namespace ICTouchAPI.transitionServices
* @extends Object
* Ease the management of middle screens in createScreen.
* This API <b>must</b> be compatible with an iframe returned by createIframe ( _show, _hide methods )
*/
dojo.declare("ICTouchAPI.transitionServices.screen", null, {
	// Id of contained webapp
	id					: null,
	// Id of the appbar
	appbarId			: null,

	// Content widget
	_content			: null,
	// Appbar widget
	_appbar				: null,

	_screenFrame		: null,
	_appbarFrame		: null,

	// Must this screen have an exclusive iframe ?
	_exclusiveIframe	: false,

	// Is this the first time the screen has been shown
	_first				: true,

	// Defines
	SCREEN				: 0,
	APPBAR				: 1,

	// Static attribute were one iframe should always be ready
	_static_iframePool	: {screenIndex: 2, appbarIndex: 2, screenFrame: null, appbarFrame: null, init: false},

	constructor: function(content, id, exclusiveIframe) {
		this._content = content;
		// Set the id for this screen to the id of the content ( if available )
		if( typeof id === "string" ) {
			this.id = id;
			var appbarName = id.split(".");
			appbarName[0] = "appbar";
			this.appbarId = appbarName.join(".");
		}
		else {
			this.id = "mainframe_noname";
			this.appbarId = "appbarframe_noname";
		}
		this._exclusiveIframe = exclusiveIframe;
	},

	destroy: function() {
		this._hide();
		delete this._content;
		delete this._appbar;
	},

	getContentDocument: function() {
		return this._screenFrame ? this._screenFrame.contentDocument : null;
	},

	show: function() {
		this._show();
	},

	hide: function() {
		this._hide();
	},

	replaceAppbar: function(newAppbar) {
		if (newAppbar == this._appbar) {
			return;
		}

		this._appbar = newAppbar;
		if( this._appbarFrame !== null ) {
			this._cleanFrame(this._appbarFrame, false, this.APPBAR);
			this._attachToFrame(newAppbar, this._appbarFrame);
		}
	},

	_static_initPool: function() {
		if( this._static_initPool.init ) {
			return;
		}
		this._static_initPool.init = true;

		this._static_iframePool.screenFrame = [];
		this._static_iframePool.appbarFrame = [];

		this._static_iframePool.screenFrame[0] = this._static_createIframe("mainframe0", this.SCREEN);
		this._static_iframePool.appbarFrame[0] = this._static_createIframe("appbarframe0", this.APPBAR);

		this._static_iframePool.screenFrame[1] = this._static_createIframe("mainframe1", this.SCREEN);
		this._static_iframePool.appbarFrame[1] = this._static_createIframe("appbarframe1", this.APPBAR);

	},

	_static_createIframe: function(name, screen) {
		if( screen === this.SCREEN ) {
			return ICTouchAPI.transitionServices._createIFrame(name, "Middle", function() {
				ICTouchAPI.skinServices.registerHeader(this._getHead(), "mainframe", "webapp", this);
				name = null; // Helps the garbage collector
			});
		}
		else if( screen === this.APPBAR ) {
			return ICTouchAPI.transitionServices._createIFrame(name, "BottomBar", function() {
				ICTouchAPI.skinServices.registerHeader(this._getHead(), "appbarframe", "appbar", this);
				name = null; // Helps the garbage collector
			});
		}

		return null;
	},

	// Attach a widget to a frame
	_attachToFrame: function(widget, frame) {
		frame.contentDocument.adoptNode(widget.domNode);
		widget.placeAt(frame.contentDocument.body);
	},

	// Remove the widget from the frame and remove link that aren't part of the framework
	_cleanFrame: function(frame, removeHeaders, type) {
		// Remove content from this iframe
		dojo.empty(frame.contentDocument.body);
		if( removeHeaders ) {
			// transform to the type of skinServices
			type = type === this.SCREEN ? "webapp" : "appbar";
			ICTouchAPI.skinServices.changeTag(frame, "mainframe", type);
		}
	},

	/* Add specific link to header
	   frame is the frame to attach the content to
	   name of the webapp ( "webapp.communication" for example )
	   type screen or actionbar
	*/
	_addSpecificLinks: function(frame, name, type) {
		// transform to the type of skinServices
		type = type === this.SCREEN ? "webapp" : "appbar";

		ICTouchAPI.skinServices.changeTag(frame, name, type);
	},

	_show: function() {
		// Main screen
		// If this is an exclusive frame we only attach the head & body the first time
		if( !this._exclusiveIframe || this._first) {
			this._preFirstShow();
		}
		this._first = false;

		this._screenFrame._show();
		this._appbarFrame._show();

		this._notifyWebappShow( this._screenFrame.id );
	},


	_notifyWebappShow: function( id ) {
		//console.log( 'transition: screen._show', id );

		var splitId = id.split(".");

		if( undefined !== splitId[1] ) {
			dojo.publish( "ICTouchAPI.transitionServices.WEBAPP_SHOW" , [{'id': id}] );
			if (webapp[ splitId[1] ]) {
				webapp[ splitId[1] ].onShow( splitId[2] );
			}
		}
	},


	_preFirstShow : function() {
		// Get the name of the webapp
		var name = this._content.webapp.getApplicationName();
		// Only use the name of webapp and not the full namespace
		name = name.split(".")[1];

		this._screenFrame = this._getNextIframe(this.SCREEN);
		this._screenFrame.id = this.id;

		// Body
		this._attachToFrame(this._content, this._screenFrame);

		// Appbar frame
		this._appbarFrame = this._getNextIframe(this.APPBAR);
		this._appbarFrame.id = this.appbarId;
		var appbarName = this.appbarId.split(".");

		// Get new style of appbar
		this._appbar = ICTouchAPI.AppBarServices._getAppBarWidget(appbarName[1],appbarName[2]);
		if (this._appbar === null) {
			// If it doesn't work, get old style of appbar
			this._appbar = ICTouchAPI.AppBarServices.getAppBar(appbarName[1],appbarName[2]);
		}
		else if (this._appbar) {
			this._appbar._checkArrows();
		}

		// Body
		this._attachToFrame(this._appbar, this._appbarFrame);

		this._first = false;
	},

	_hide: function() {
		if( this._screenFrame === null ) {
			return;
		}
		this._notifyWebappHide( this._screenFrame.id );
		this._screenFrame._hide();
		if( this._appbarFrame === null ) {
			return;
		}
		this._appbarFrame._hide();

		// Only remove the webapp CSS when it's not an exclusive iframe because we will reuse it later
		if( !this._exclusiveIframe ) {
			this._screenFrame.id = "";
			this._cleanFrame(this._screenFrame, false, this.SCREEN);
			this._screenFrame = null;

			this._appbarFrame.id = "";
			this._cleanFrame(this._appbarFrame, false, this.APPBAR);
			this._appbarFrame = null;
		}
	},

	_notifyWebappHide: function( id ) {
		//console.log( 'transition: screen._hide', id );

		var splitId = id.split(".");

		if( undefined !== splitId[1] ) {
			dojo.publish( "ICTouchAPI.transitionServices.WEBAPP_HIDE" , [{'id': id}] );
			if (webapp[ splitId[1] ]) {
				webapp[ splitId[1] ].onHide();
			}
		}
	},

	_getNextIframe: function(screen) {
		// If an iframe is already defined this is a no-op
		if( this._screenFrame !== null && screen === this.SCREEN ) {
			return this._screenFrame;
		}
		if( this._appbarFrame !== null && screen === this.APPBAR ) {
			return this._appbarFrame;
		}

		var idx, list;
		if (screen === this.SCREEN) {
			idx = this._static_iframePool.screenIndex++;
			list = this._static_iframePool.screenFrame
		}
		else {
			idx = this._static_iframePool.appbarIndex++;
			list = this._static_iframePool.appbarFrame;
		}

		var iframe = list[idx%2];
		if( this._exclusiveIframe ) {
			var name = screen === this.SCREEN  ? "mainframe"+idx : "appbarframe"+idx;
			list[idx%2] = this._static_createIframe(name, screen);
		}

		return iframe;
	}
});

/* Initialize pool of iframes, because this is a STATIC method we must call it this way
  But this cannot be called in the first constructor call because the iframe won't be ready on time
*/
ICTouchAPI.transitionServices.screen.prototype._static_initPool();
/**
* @class ICTouchAPI.I18nServices
* @singleton
* @extends Object
* I18n provides internationalization tools for the WebApps. This framework service can be used with or without i18n tag for Django templating.
* I18n returns a string translated in the current locale (defined in dojo.locale)
*/
dojo.provide("ICTouchAPI.i18nServices");
dojo.require("dojox.string.sprintf");
dojo.require("dojo.i18n");
dojo.declare("ICTouchAPI.i18nServices",null, {

	/**
         * @ignore
         */
	_languageAllowedValues : {},

	/**
         * @ignore
         * Know how many i18nized strings have currently subscribed to the languageChanged event.
         */
	_nbOfHandlers : 0,

	/**
         * @ignore
         */
	constructor : function(){
            // Default settings for simulation mode
            if(!generalConfig.simulation) {
		ICTouchAPI.settingServices.getSetting("Language", this, this.settingChanged);
		ICTouchAPI.settingServices.subscribeToSetting(this, "Language", this.settingChanged);
            }
	},

	/**
         * @ignore
         */
	getApplicationName : function(){
		return "ICTouchAPI.i18nServices";
	},

	/**
         * @ignore
         */
	settingChanged : function (objLanguage) {
		this._setLanguageAllowedValues(objLanguage.allowedValues);
		this.languageChanged(this.getLanguageName(objLanguage.jsValue));
	},

	/**
         * @ignore
         */
	languageChanged : function (ISOLang) {
		if (this.getLocale() !== ISOLang) {
			this.setLocale(ISOLang);
			dojo.publish("languageChanged");
		}
	},

	/**
         * @ignore
         */
	setLocale : function (locale) {
		dojo.locale = locale;
	},

	/**
         * @ignore
         */
	getLocale : function () {
		return dojo.locale;
	},


	/**
         * @ignore
         */
	changeLanguage : function (language) {
		language = language.toLowerCase();
		var langJSValue = this.getLanguageJSValue(language);
		if(langJSValue === false){
			langJSValue = this.getLanguageJSValue(language.split('-')[0]);
		}
		var langValue = typeof language === "string" && langJSValue !== false ? langJSValue : +(language);
		ICTouchAPI.settingServices.setSettingValue("Language", [langValue]);
	},

	/**
         * @ignore
         */
	getLanguageJSValue : function (language) {
		for (var i in this._languageAllowedValues) {
			if (this._languageAllowedValues.hasOwnProperty(i)) {
				if (this._languageAllowedValues[i].name === language) {
					return +(i);
				}
			}
		}
		return false;
	},

	/**
         * @ignore
         */
	getLanguageName : function (jsValue)  {
		return this._languageAllowedValues[jsValue].name;
	},

	/**
         * @ignore
         */
	_setLanguageAllowedValues : function (allowedValues) {
		this._languageAllowedValues = allowedValues;
	},

	/**
         * Translate a string, and replace variables by values
         * @param {String} toTranslate : String to translate
         * @param {String} i18nContext : context to use for translation
         * @param {Array} args : Translation variables values
         * @return {Object} : An object embedding everything that is needed to translate
         * @ignore
         */
	translate : function(toTranslate, i18nContext, args, bundle){

		if (toTranslate.isI18Ned){
			console.warn(toTranslate +" in " + i18nContext + " is already i18nized.");
		} else {

			// The i18nized object
			var objI18n =  {

				// Yes it is an i18nized object
				isI18Ned : true,

				// The callback that will be executed to assign a new string to a DOM element.
				subscribeCallback : null,

				// The i18n context
				i18nContext : i18nContext,

				// The text to translate
				toTranslate : toTranslate,

				stringsToReplace : args,
				
				getTranslation : function () {
					var translated = "";
					if (typeof objI18n.toTranslate === "string") {
						translated = objI18n._translate(objI18n.toTranslate, objI18n.i18nContext, objI18n.getStringsToReplace(), bundle);
					} else {
						for (var i in objI18n.toTranslate) {
							if (objI18n.toTranslate.hasOwnProperty(i)) {
								if (objI18n.toTranslate[i].isI18Ned){
									translated += objI18n.toTranslate[i].getTranslation();
								}
								else {
									translated += objI18n._translate(objI18n.toTranslate[i], objI18n.i18nContext, objI18n.getStringsToReplace(), bundle)+" ";
								}
							}
						}
					}
					return translated;
				},

				toString : function() {
					return this.getTranslation();
				},

				_translate : function (string, i18nContext, args, bundle) {
					bundle = bundle || "lang";
					dojo.requireLocalization(i18nContext,bundle);

					var translated=dojo.i18n.getLocalization(objI18n.i18nContext, bundle)[string];

					if(!translated) {
						translated = string;
					}

					if (args && args.length) {
						// Create the argument list for sprintf(translated, args[0], args[1], ...);
						args = dojo.isArray(args) ? args : [args];
						args.unshift(translated);
						translated = dojox.string.sprintf.apply(dojox.string, args);
						args.shift();
					}

					return translated;
				},

				fillInnerHTML : function (node) {
					this.subscribeCallback = function( ) {
						return node.innerHTML = this.getTranslation()
					};

					this.subscribeCallback();

					var unsub = dojo.subscribe("languageChanged", this, function (){
						if (node) {
							this.subscribeCallback();
						} else {
							console.warn("[i18nServices] it seems there was a leak with ");
							console.warn(node);
							// Unsubscribe
							dojo.unsubscribe(unsub);
						}

					});
					return unsub;
				},

				// Link this object with its environment
				i18nize : function (obj, name, subscribeCallback) {

					// If we try to i18nize an i18ned object.
					if (obj[name] &&  obj[name].isI18Ned) {

						// Get the text's translation
						var cb = obj[name].getTranslation;
						obj[name] = cb();
						// There will be a new subscribe handler.
						ICTouchAPI.i18nServices._nbOfHandlers++;

						// Save the callback so it can be used from other function of this i18ned object.
						this.subscribeCallback  = function () {
							// Returns true/false
							return subscribeCallback.apply(obj, [name, cb()+""]);
						}

						// subscribe to languageChanged event to assign the future translation to its domNode
						var unsub = dojo.subscribe("languageChanged", this, function () {

							// If the callback can't be executed there's a great chance it's because
							// it has disapeared
							if (!this.subscribeCallback()) {
								console.warn("[i18nServices] it seems there was a leak with ", obj, name);
								// Unsubscribe
								dojo.unsubscribe(unsub);
								unsub = null;
								ICTouchAPI.i18nServices._nbOfHandlers--;
							}
						});
						return unsub;

					} else {
						return false;
					}
				},

				// Change the text that is always translated
				changeText : function (toTranslate) {
					if (this.setToTranslate(toTranslate)) {
						// If it's a new one, execute the callback
						if (this.subscribeCallback) {
							return this.subscribeCallback();
						} else {
							return true;
						}
					}
				},

				/**
                                 * @ignore
                                 */
				setToTranslate : function (toTranslate) {
					if (toTranslate !== this.toTranslate) {
						this.toTranslate = toTranslate;
						return true;
					} else {
						return false;
					}
				},

				/**
                                 * @ignore
                                 */
				setStringsToReplace : function (args) {
					var currentStrings = this.stringsToReplace.join ? this.stringsToReplace.join() : this.stringsToReplace;
					var newStrings = args.join ? args.join() : args;

					if (currentStrings === newStrings) {
						return false;
					} else {
						this.stringsToReplace = args;
						if (this.subscribeCallback) {
							return this.subscribeCallback();
						} else {
							return true;
						}
					}
				},

				/**
                                 * @ignore
                                 */
				getStringsToReplace : function () {
					return dojo.isArray(this.stringsToReplace) ? this.stringsToReplace : [this.stringsToReplace];
				}

			}

			return objI18n;
		}


	},

	/**
         * @ignore
         */
	i18nize : function (list, i, callback) {

		var unsubscribeHandler;

		if (list[i] && typeof list[i].i18nize == "function"){
			// Prevents a bug when the same i18n object is used multiple times.
			list[i] = dojo.clone(list[i]);
			unsubscribeHandler = list[i].i18nize(list, i, callback);
		}

		return unsubscribeHandler;

	},


	i18nList : function (list, callback) {
		var unsubscribeHandlers = [], handler;

		for (var i in list) {
			if (list.hasOwnProperty(i)) {
				handler = this.i18nize(list, i, callback);
				if (handler) {
					unsubscribeHandlers.push(handler);
				}
			}
		}

		return unsubscribeHandlers;

	},

	unsubscribeAllI18nHandlers : function (widget) {

		if (widget._i18nizedHandlers) {

			var length = widget._i18nizedHandlers.length;
			for (var i=0; i<length; i++) {
				dojo.unsubscribe(widget._i18nizedHandlers[i]);
			}
			delete widget._i18nizedHandlers;

			this._nbOfHandlers -= i;
		}
	},

	addI18nHandlers : function (who, lstHandlers) {
		if (!who._i18nizedHandlers) {
			who._i18nizedHandlers = [];
		}
		who._i18nizedHandlers = who._i18nizedHandlers.concat(lstHandlers);
	},

	addI18nHandler : function (who, arrHandler) {
		if (!who._i18nizedHandlers) {
			who._i18nizedHandlers = [];
		}
		who._i18nizedHandlers.push(arrHandler);
	},

	areI18nStringsEquals : function (i18nA,i18nB) {
		if(i18nA.toTranslate.length !=i18nB.toTranslate.length){
			return 0;
		}
		for(var i=0;i<i18nA.toTranslate.length;i++){
			if(i18nA.toTranslate[i]!=i18nB.toTranslate[i]){
				return 0;
	}
		}
		return 1;
	},
	
	cloneI18n : function(i18nObject){
		if(i18nObject.isI18Ned){
			return _(i18nObject.toTranslate, i18nObject.i18nContext, i18nObject.stringsToReplace);
		} else {
			return i18nObject;
		}
	}


});
ICTouchAPI.i18nServices = new ICTouchAPI.i18nServices();

/**
 * Translate a string, and replace variables by values
 * @param {String} toTranslate : String to translate
 * @param {String} i18nContext : context to use for translation
 * @param {Array} args : Translation variables values
 * @return {Object} : An object embedding everything that is needed to translate
 */
_ = ICTouchAPI.i18nServices.translate;
/**
 * @class ICTouchAPI.WebActionButton
 * @singleton
 * @extends Object
 * @ignore
 */
dojo.provide("ICTouchAPI.webActionButton");
dojo.declare("ICTouchAPI.webActionButton",null,{
/* --------------------------------- Private attributes ----------------------------------- */
     /**
     * _actionButtonName
     * @property
     * @type String
     */
	_actionButtonName:"",			// name of button
    /**
     * _statusActionButton
     * @property
     * @type String
     */
	_statusActionButton:'',			// status of button
									// either 1 (visible),0 (hiden) or 2 (inactif)
    /**
     * _icon
     * @property
     * @type String
     */
	_icon:"",						// localisation of icon
    /**
     * _callback
     * @property
     * @type String
     */
	_callback:"",					// call function after a click on


/* ------------------------------------ Constructor --------------------------------------- */
	
/* ------------------------------------ Constructor --------------------------------------- */
	/**
	 * @private
	 */
        constructor: function(buttonName,buttonStatus,locationIcon,functCallback){
		this._actionButtonName = buttonName;
		this._statusActionButton = buttonStatus;
		this._icon = locationIcon;
		this._callback = functCallback;
	},

/* ----------------------------------- Getter / Setter------------------------------------- */
	/**
	 * @private
	 */
        getActionButtonName : function(){
		return this._actionButtonName;
	},

	/**
	 * @param {int} buttonStatus
         * @private
	 */
	setStatusButton : function(buttonStatus){
		this._statusActionButton = buttonStatus;
	}

})
/**
* @class ICTouchAPI.KeyboardTypes
* @singleton
* @extends Object
* Keyboard types
*
* <pre><code>
* Keyboard Types
* - "ALPHANUM"
* - "NUM"
* - "FILTER"
* - "DIALPAD"
* - "EMOTICON"
* - "PASSWORD"
* - "NUMSEARCH"
* - "ALPHASEARCH"
* - "DTMF"
* </code></pre>
*/
ICTouchAPI.KeyboardTypes = {

        /**
	 * @cfg {String} ALPHANUM Standard alpha numeric
         * @ignore
	 */
	ALPHANUM : "alphaNum",

	/**
	 * @cfg {String} NUM Standard numeric
         * @ignore
	 */
	NUM : "num",

        /**
	 * @cfg {String} FILTER Filtered
         * @ignore
	 */
	FILTER		: "filter",

	/**
	 * @cfg {String} DIALPAD Dialpad without additional button
         * @ignore
	 */
        DIALPAD		: "dialpad",

	/**
	 * @cfg {String} EMOTICON Emoticon buttons
         * @ignore
	 */
        EMOTICON	: "emoticons",

	/**
	 * @cfg {String} PASSWORD Standard alpha numeric with password
         * @ignore
	 */
	PASSWORD	: "password",

	/**
	 * @cfg {String} NUMSEARCH Dialpad with additional button to switch to alphaNum search
         * @ignore
	 */
	NUMSEARCH	: "numSearch",

	/**
	 * @cfg {String} ALPHASEARCH Standard alpha numeric with additional button to switch to dialpad
         * @ignore
	 */
	ALPHASEARCH	: "alphaSearch",

	/**
	 * @cfg {String} DTMF Dialpad with only a cancel button
         * @ignore
	 */
	DTMF		: "dtmf",

	/**
	 * @cfg {String} NUMMDRS Dialpad with additional button for Multi Device Routing Selection
         * @ignore
	 */
	NUMMDRS	: "numMultiDevice",

	/**
	 * @cfg {String} PASSWORD Standard numeric with password
         * @ignore
	 */
	NUMPASSWORD	: "numPassword"
}


/**
* @class ICTouchAPI.KeyboardAlphanumMode
* @singleton
* @extends Object
* Alphanumerique Keyboard Modes
*
* <pre><code>
* Alphanumerique Keyboard Modes
* - "NORMAL"
* - "CAPS"
* - "CAPS_LOCK"
* - "NUMERIC"
* </code></pre>
*/
ICTouchAPI.KeyboardAlphanumMode = {
	/**
	 * @cfg {String} NORMAL Normal keyboard mode
	 * @ignore
	 */
	NORMAL : "normal",
	/**
	 * @cfg {String} CAPS Non locked Caps keyboard mode
	 * @ignore
	 */
	CAPS : "caps",
	/**
	 * @cfg {String} CAPS_LOCK Locked Caps keyboard mode
	 */
	CAPS_LOCK : "capsLock",
	/**
	 * @cfg {String} NUMERIC Numeric keyboard mode
	 * @ignore
	 */
	NUMERIC : "numeric"
}


/**
* @class ICTouchAPI.keyboardServices
* @singleton
* @extends Object
* Keyboard management
*/
dojo.provide("ICTouchAPI.keyboardServices");
dojo.declare("ICTouchAPI.keyboardServices",null,
	{
		// Attributes.
		// Current keyboard.
		_objCurrentKeyboard: {},
		// Current characters sequence to build a candidate.
		_charSequence: "",
		// InputField that deployed the keyboard.
		_objCalledBy: {},
		// Initial caret position (absolute) when starting a charSequence.
		_initialCaretPos: 0,
		// Current caret position (absolute)
		_currentCaretPos: 0,
		// IF current language requires an IME.
		_IMERequired: false,
		// Loaded keyboards (1 for each keyboard type).
		_arrKeyboards: {},
		// A callback function if value is not to be returned in an inputField, like "make call"
		_funcCallbackOk: null,
		// A callback function called when the keyboard is closed (CLOSE_ACTION_EXIT or cancel if _funcCallbackManualExit is not defined).
		_funcCallbackExit: null,
		// A callback function called when the "Exit" key is pressed.
		_funcCallbackManualExit: null,
		// A callback function called when a key is pressed.
		_funcCallbackKeyPress: null,
		// A callback function called when a key is pressed.
		_funcCallbackKeyPressed: null,
		// A callback function called when a key is released.
		_funcCallbackKeyReleased: null,
		// Array of custom callbacks (they are created by webapps and accessed through UI Keyboard)
		_arrCustomCallbacks: [],
		// Set this to true to autoClose previous keyboard when opening a new one.
		autoClose: true,
		value: "",
		numPadLengthLimit : -1,
		// Popup for keyboard switch
		keyboardPopup : null,
		// List of the different keyboard
		sltList : null,
		//Calling webapp
		callingWebapp : null,

		// Constants used for function closeKeyboard
		CLOSE_ACTION_NONE	: -1,
		CLOSE_ACTION_EXIT	: 0,
		CLOSE_ACTION_OK		: 1,
		CLOSE_ACTION_MANUAL_EXIT : 2,

		/* Relation between a complex keyboard type "ICTouchAPI.KeyboardTypes" and an UI keyboard type with corresponding flags
		   There is also the function _setConditionalFlags that sets some flags
		*/
		_keyboardTypeMapping: {
			"alphaNum"		: {
				"simpleType"	: "alphaNum"
			},
			"num"			: {
				"simpleType"	: "num",
				'international'	: false,
				"dialSpace"		: false
			},
			"filter"		: {
				"simpleType"	: "alphaNum",
				"filter"		: true
			},
			"dialpad"		: {
				"simpleType"	: "num",
				'international'	: true,
				"dialSpace"		: true
				// Also got conditional flags
			},
			"emoticons"		: {
				"simpleType"	: "alphaNum",
				"emoticon"		: true
			},
			"password"		: {
				"simpleType"	: "alphaNum",
				"password"		: true
			},
			"numSearch"		: {
				"simpleType"	: "num",
				'international'	: true,
				"dialSpace"		: false
				// Also got conditional flags
			},
			"alphaSearch"	: {
				"simpleType"	: "alphaNum",
				"byDialpad"	: true
				// Also got conditional flags
			},
			"dtmf"			: {
				"simpleType"	: "num",
				"ok"				: false,
				"backspace"	: false,
				"dialSpace" : false
			},
			"numMultiDevice"			: {
				"simpleType"	: "num",
				"latest"	: true
			},
			"numPassword"		: {
				"simpleType"	: "num",
				"password"		: true
			}
		},
		/*
		define all the APEC keyboard, which is ready for IME indicator switch extension.
		 */
        imeLangSwitcher: {
            lastIME: "pinyin",
            switchIME: {
                "chinese": {
                    lang: ['apac_en', 'pinyin']
                },

                "japanese": {
                    lang: ['apac_en', 'Katakana', 'Hiragana']
                },

                "korean": {
                    lang: ['apac_en', 'Hangul']
                }
            }
        },
        /*
         * @ignore
         * this function is to remove the duplicated items in an array..
         */
		arrayUnique: function (array) {
            var tmpObj = {};
            for (var i = 0; i<array.length;++i) {
                tmpObj[array[i]] = array[i];
            }
            var results = [];
            for (index in tmpObj) {
                results.push(tmpObj[index]);
            }
            return results;
        },
        /*
         *@ignore
         * this function is to get all the APEC keyboard in the imeLangSwitcher.
         */
        _getAPACIMELangs: function () {
            var tempObject = this.imeLangSwitcher.switchIME;
            var tempArr = [];
            for (var item in tempObject) {
                var obj = tempObject[item];
                if (obj.hasOwnProperty("lang")) {
                    tempArr = tempArr.concat(obj.lang);
                }
            }
            return  this.arrayUnique(tempArr);
        },

        apacIMELangs: null,
        arrEmoIcon : null,
		arrEmoIconConvert: [],

		_hotelMode : false,
		_multiDeviceRoutingMode : false,
		_actionButtonInDialpad : 0,
		_arrDialpadKeys : [],
		_arrRegisteredKeys : [],
		_arrProgKeys : [],
		_arrProgKeysLocalisation : [],
		_customKeysUpdated : false,

        /*
         *@ignore
         * this function is to get last language when switch by IME indicator button.
         */
        getLastLang: function () {
            return this.imeLangSwitcher.lastIME;
        },

        /*
         *@ignore
         * this function is to get next language keyboard when switch keyboard by IME indicator button.
         */
        getNextLang: function (strLang) {
            if(strLang == "apac_en"){
                return this.imeLangSwitcher.lastIME;
            }else{
                this.imeLangSwitcher.lastIME = strLang;
                return "apac_en";
            }
           },

        /**
         * @ignore
         */
        constructor: function () {
            // Sets a flag if an IME is required.
            this._IMERequired = false;
            //Hi Soft
            this.apacIMELangs = this._getAPACIMELangs();

            var strPath="library/ICTouchAPI/UIElements/Keyboard/themes/Default/images/";
            var that=this;
            dojo.xhrGet( {
                url: "library/ICTouchAPI/UIElements/Keyboard/keyMaps/emoticon.json",
                handleAs: "json",
                sync: true,

                load: function(response) {
                        that.arrEmoIcon = response.arrEmoIcon;

                },

                error: function(response, ioArgs) {
                        console.error(ioArgs);
                }
            });
                    //console.log(that.arrEmoIcon);
            for(var i in this.arrEmoIcon){
                if(this.arrEmoIcon){
                    this.arrEmoIcon[i][0]=strPath+this.arrEmoIcon[i][0]+"_normal_24.png";
                    if(this.arrEmoIcon[i][1]){
                        var reg=RegExp("m-e");
                        this.arrEmoIcon[i][0]=this.arrEmoIcon[i][0].replace(reg,"m_e");
                        this.arrEmoIconConvert.push(this.arrEmoIcon[i]);
                    }
                }
            }
            dojo.subscribe("ModeHotel", this, this._hotelModeReceveid);

			dojo.subscribe("ModeMultiDeviceRouting", this, this._multiDeviceRoutingRecieved);

			dojo.subscribe("HomepageRegisterKey",this,this.registerDialpadKey);

			dojo.subscribe("HomepageUnregisterKey",this,this.unregisterDialpadKey);

			dojo.subscribe("languageChanged",this, this.applyDMTranslation);
       	},

		/**
		 * Get the keyboard's inputField value
		 * @return {string} inputField's value
		 */
		getValue: function () {
			return this._objCurrentKeyboard.getInputField().getInputFieldContent();
		},

		/**
		 * Get the name of the group the field belongs to, if any.
		 * @return {String} the group name, or false
         * @ignore
		 */
		getFieldGroupName: function () {
			if (this._objCalledBy) {
				return this._objCalledBy.linkedFields;
			} else {
				return false;
			}
		},

		/**
		 * Get current deployed keyboard
		 * @return {Object} current keyboard
		 */
		getCurrentKeyboard: function() {
			// WRK crms00199760
			return this._objCurrentKeyboard;
		},

		/**
		 * @ignore
		 */
		getApplicationName: function() {
			return "keyboardServices";
		},

		/**
		 * @ignore
		 */
		_getSimpleKeyboardType : function(type) {
			var keyboardType = this._keyboardTypeMapping[type];
			if( !keyboardType ) {
				ICTouchAPI.debugServices.debug("Unknown keyboard type: "+type, false);
				return null;
			}
			return this._keyboardTypeMapping[type].simpleType;
		},

		/**
		 * Regain focus of keyboard's input field.
		 * When a key is pressed, the focus is lost, then its content scrolls back to the beginning
		 * of the input field.
		 * @ignore
		 */
		focusInputField : function () {
			this._objCurrentKeyboard.getInputField().getInputFieldObject().focus();
		},

		/**
		 * Get InputField of current keyboard
		 * @return {Object} Current InputField
		 */
		getInputField : function () {
			return this._objCurrentKeyboard.getInputField();
		},

		/**
		 * One of the two inputs of this service. This input might trigger a candidate collection from the IME.
		 * @param {String} UTF8char the "keyed in" character (virtual or physical keyboard)
		 * @ignore
		 */
		keyPressed: function (UTF8char) {
			if (this.numPadLengthLimit != -1 && this._objCurrentKeyboard.getInputField().getInputFieldContent().length >= this.numPadLengthLimit) {
				return;
			}
			// WRK crms00199717
			if (UTF8char === "&amp;") {
				UTF8char = "&";
			}
			// WRK crms00199518
			else if (UTF8char === "&lt;") {
				UTF8char = "<";
			}
			else if (UTF8char === "&gt;") {
				UTF8char = ">";
			}

			var caretPos = this._objCurrentKeyboard.getInputField().getCaretPos();

			if ((this._currentCaretPos > caretPos) ||
				(caretPos > (this._initialCaretPos + this._charSequence.length))) {
				// If caretPos got out of previous charSquence, reset everything.
			}
			// Save current caret position.
			this._currentCaretPos = caretPos;

			if (this._funcCallbackKeyPress) {
					this._funcCallbackKeyPress(UTF8char);
				}

			// If IME is required,
			if (this._IMERequired) {
				// If it's a new char sequence
				if (ICTouchAPI.IMEServices.getCharSequenceLength()  === 0) {
					// Save current caret position as the start of the new char sequence.
					// This will be used to know where to start replacing the sequence by the selected candidate.
					this._initialCaretPos = this._currentCaretPos;
				}
				ICTouchAPI.IMEServices.keyPressed(UTF8char);
			} else {
				this.displayChar(UTF8char, false);
			}
		},

		/**
        * Display a string in IME input zone
        * */
        displayIMEString: function (UTF8String){
           this._objCurrentKeyboard.setIMEString(UTF8String);
        },

        /**
		 * @ignore
		 */
		displayChar : function(UTF8char, boolOnIMEInput) {
			if (boolOnIMEInput) {
				this._objCurrentKeyboard.addIMEChar(UTF8char)
				return;
			} else {
				this._objCurrentKeyboard.getInputField().addNewChar(UTF8char);
				if (this._funcCallbackKeyPressed) {
						this._funcCallbackKeyPressed(this._objCurrentKeyboard.getInputField().getInputFieldContent());
				}
			}
		},

		/**
		 * @ignore
		 */
		replaceChar : function(UTF8Char, boolOnIMEInput) {
            //Hi Soft
            this._objCurrentKeyboard.deleteIMEChar();
			this.displayChar(UTF8Char, boolOnIMEInput);
            //Hi Soft
        },
        //Hi Soft
        pushInputIntoField: function (clearInputZone) {
            this._objCurrentKeyboard.pushInputZoneIntoInputField(clearInputZone);
        },

        displayCharInInputField: function (ch) {
            this._objCurrentKeyboard.displayCharInInputField(ch, true);
        },

		/**
		 * @ignore
		 */
		displayCandidates : function(arrCandidates) {
			this._objCurrentKeyboard.recreateCandidateList(arrCandidates);
		},

		/**
		 * @ignore
		 */
		setIMERequirement : function(boolIMERequired) {
			this._IMERequired = boolIMERequired;
		},

		/**
		 * @ignore
		 */
		keyReleased: function(UTF8char) {
			if (this._funcCallbackKeyReleased) {
					this._funcCallbackKeyReleased(UTF8char);
				}
		},

		/**
		 * @ignore
		 */
        delChars: function (boolOnIMEInput) {
            //Hi Soft
            if (boolOnIMEInput && this._objCurrentKeyboard.deleteIMEChar()) {
                ICTouchAPI.IMEServices.delChar();
                return;
                //Hi Soft
            }
            this._objCurrentKeyboard.getInputField().delChars();
            if (this._funcCallbackKeyPressed) {
                this._funcCallbackKeyPressed(this._objCurrentKeyboard.getInputField().getInputFieldContent());
                //Hi Soft
                //}
            }
        },

		/**
		 * Close the keyboard.
		 * @param {Number} intAction action triggered: 0 = Exit, 1 = Ok, -1 = none
		 */
		closeKeyboard: function (intAction) {
			ICTouchAPI.toasterServices.hideContent(this._objCurrentKeyboard);
			if(!this._objCurrentKeyboard.closed){
				this._objCurrentKeyboard.closed = true;
				if (intAction == this.CLOSE_ACTION_OK) {
					if (this._objCalledBy) {
						this.sendInputFieldContent();
					} else if (this._funcCallbackOk) {
						this._funcCallbackOk(this.getValue());
					}
				}
				else if (intAction == this.CLOSE_ACTION_MANUAL_EXIT && this._funcCallbackManualExit) {
					this._funcCallbackManualExit();
				}
				else if ((intAction == this.CLOSE_ACTION_EXIT || intAction == this.CLOSE_ACTION_MANUAL_EXIT) && this._funcCallbackExit) {
					this._funcCallbackExit();
				}
			} else {
				ICTouchAPI.debugServices.warning("ICTouchAPI.keyboardServices - closeKeyboard / keyboard already closed... No action triggered.");
			}
			//this._objCurrentKeyboard.domNode.style.display = 'none';
		},

		/**
		 * @ignore
		 */
		sendInputFieldContent: function () {
			this._objCalledBy.setInputFieldContent(this._objCurrentKeyboard.getInputField().getInputFieldContent());
		},

		/**
		 * The second input of this service. Allows for direct input of candidates.
		 * Ends with a call to the virtual keyboard that will replace the char sequence by the candidate.
		 * @param {String} UTF8char is the selected candidate.
		 * @ignore
		 */
		candidateSelected: function (UTF8char) {
			ICTouchAPI.IMEServices.candidateSelected(UTF8char);
		},

		/**
		 * Deploys or not a keyboard according to target's version (MR/VHE)
		 * @param {String} type is the type of the keyboard (num, alphanum...)
		 * @param {Object} objCallingInputField is the inputField that fired up the keyboard.
		 * @param {Object} objCustomParams contains custom parameters used if no input field was defined.
		 *	It can contain
		 * <pre><code>
		 * - {String} strDefaultText a text to be displayed in the keyboard's inputField when open
		 * - {Boolean} bMultiLines specifies if caller is multiLines (so will be the keyboard's inputfield)
		 * - {String}  strDefaultMode initial mode for alphanumeric keyboard (ICTouchAPI.KeyboardAlphanumMode enumeration) (default: NORMAL)
		 * - {Boolean} showVoicemailButton hide or show the voicemail button
		 * - {Boolean} showAddcontactButton hide or show the AddContact button
		 * - {Boolean} hideStarsIfPasswordEmpty hide or not the stars displayed in the input field when the content is a password
		 * - {Function} funcCallbackOk the OK callback
		 * - {Function} funcCallbackExit the EXIT callback
		 * - {Function} funcCallbackKeyPress the callback when a key is pressed (called before the character sequence is modified)
		 * - {Function} funcCallbackKeyPressed the callback when a key is pressed (called after the character sequence is modified ; receives the character sequence in parameter)
		 * - {Function} funcCallbackKeyReleased the callback when a key is released
		 * - {Array} arrCustomCallbacks custom set of callbacks handled according to Keyboard type (see UIElement Keyboard)
		 * </code></pre>
		 */
		deployKeyboard: function (type, objCallingInputField, objCustomParams) {
			// If autoClose is set to true, then close the previous keyboard.
			if (this.autoClose === true && this._objCurrentKeyboard.domNode) {
				if (ICTouchAPI.toasterServices.isContentDisplayed(this._objCurrentKeyboard)) {
					this.closeKeyboard(ICTouchAPI.keyboardServices.CLOSE_ACTION_NONE);
				}
			}
			var _str, _numLockKeyboard;
			var multiline = false;
			this.numPadLengthLimit = -1;
			var keyboardType = this._getSimpleKeyboardType(type);
			if( !keyboardType ) {
				return;
			}

			objCustomParams = objCustomParams || {};
			//add the possibility to define its own strings and icon for OK and EXIT
			var okLabel = objCustomParams.strOkLabel || null;
			var okIcon = objCustomParams.strOkIcon || null;
			var exitLabel = objCustomParams.strExitLabel || null;
			var exitIcon = objCustomParams.strExitIcon || null;
			if(objCustomParams.callingWebapp){
				this.callingWebapp = objCustomParams.callingWebapp;
			}else{
				this.callingWebapp = null;
			}
			// If keyboard type is numeric we override the labels
			if (keyboardType === "num") {
				okLabel = _("", "ICTouchAPI");
				if (type === ICTouchAPI.KeyboardTypes.NUMSEARCH || type === ICTouchAPI.KeyboardTypes.DIALPAD) {
					exitLabel = _("Cancel", "ICTouchAPI");
					exitIcon = "generic-cancel";
				}
			}

			// Set the callback function and reset any previous save of the caller's inputfield
			this._funcCallbackOk = ICTouchAPI.tools.callbackToHitch(objCustomParams.funcCallbackOk);
			this._funcCallbackExit = ICTouchAPI.tools.callbackToHitch(objCustomParams.funcCallbackExit);
			this._funcCallbackManualExit = ICTouchAPI.tools.callbackToHitch(objCustomParams.funcCallbackManualExit);

			if (objCallingInputField) {
				this._funcCallbackKeyPress = null;
				this._funcCallbackKeyPressed = null;
				// WRK : crms00204576
				this._funcCallbackKeyReleased = null;
				this._arrCustomCallbacks = [];
				this._objCalledBy = objCallingInputField;
                _str = objCallingInputField.getInputFieldContent() || objCustomParams.strDefaultText || "";
				multiline = objCallingInputField.getMultiline();
				if (objCallingInputField.numLockKeyboard) {
					_numLockKeyboard = objCallingInputField.numLockKeyboard;
				}
				else {
					_numLockKeyboard = 0;
				}
			} else {
				// Set the callback function and reset any previous save of the caller's inputfield
				this._funcCallbackKeyPress = ICTouchAPI.tools.callbackToHitch(objCustomParams.funcCallbackKeyPress);
				this._funcCallbackKeyPressed = ICTouchAPI.tools.callbackToHitch(objCustomParams.funcCallbackKeyPressed);
				// WRK : crms00204576
				this._funcCallbackKeyReleased = ICTouchAPI.tools.callbackToHitch(objCustomParams.funcCallbackKeyReleased);
				if(objCustomParams.arrCustomCallbacks){
					this._arrCustomCallbacks = objCustomParams.arrCustomCallbacks;
				}				
				this.numPadLengthLimit = objCustomParams.numPadLengthLimit;
				this._objCalledBy = null;
				multiline = objCustomParams.multiline;
				_str = objCustomParams.strDefaultText || "";
				if(_numLockKeyboard==undefined)
					_numLockKeyboard = 0;
			}

			// Get the flags for this type of keyboard
			var keyboardFlags = this._keyboardTypeMapping[type];
			this._setConditionalFlags(keyboardFlags, type);
			// prevent the custom keys if it's a transfer
			if (objCustomParams && objCustomParams.strOkIcon && objCustomParams.strOkIcon.search("communication-deflect") != -1) {
				keyboardFlags.customKeys = false;
			}
			// Set multiline flag
			keyboardFlags['multiline'] = multiline;

			if (!this._arrKeyboards[keyboardType]) {
				this._arrKeyboards[keyboardType] = new UIElements.Keyboard.KeyboardControl({
					strType: keyboardType,
					strInputText : _str,
					strInputTitle : objCustomParams.strInputTitle ? objCustomParams.strInputTitle : null,
					numLockKeyboard: _numLockKeyboard,
					hideStarsIfPasswordEmpty : objCustomParams.hideStarsIfPasswordEmpty,
					strDefaultMode : objCustomParams.strDefaultMode,
					"keyboardFlags": keyboardFlags
				});
			} else {
				var title = objCustomParams.strInputTitle ? objCustomParams.strInputTitle : "";
				// Flags should be changed before anything else. Because we need the buttons to update their labels
				this._arrKeyboards[keyboardType].setTitle(title);
				this._arrKeyboards[keyboardType].setKeyboardFlags(keyboardFlags);
				this._arrKeyboards[keyboardType].setMode(objCustomParams.strDefaultMode);
				// Set keyboard's input field to caller's value.
				this._arrKeyboards[keyboardType].getInputField().setInputFieldContent(_str, false, objCustomParams.hideStarsIfPasswordEmpty);
				this._arrKeyboards[keyboardType].numLockKeyboard = _numLockKeyboard;
				this._arrKeyboards[keyboardType].refreshFlags();
				this._arrKeyboards[keyboardType]._updateButtonsIcons();
			}

			this._objCurrentKeyboard = this._arrKeyboards[keyboardType];
			//signal the other webapps that a keyboard is displayed on screen
			this._objCurrentKeyboard.closed = false;
			// Joel: need to swap key map after deploying keyboard, else it won't be updated according to its flags
			this._objCurrentKeyboard._swapKeyMap();
			this._objCurrentKeyboard.updateOkExitButtonLabels(okLabel, exitLabel);
			this._objCurrentKeyboard.updateOkExitButtonIcons(okIcon, exitIcon);
			dojo.publish("keyboardServices.deployKeyboard", [true]);

			//ICTouchAPI.toasterServices.showContent(this._objCurrentKeyboard);
            var isAPACIMELang = ICTouchAPI.keyboardServices.isAPACIMELang(this._objCurrentKeyboard.strLang);
            //update the IME flags, to active IME,or not.
            this.setIMERequirement(keyboardType == ICTouchAPI.KeyboardTypes.ALPHANUM && isAPACIMELang && !keyboardFlags.password);
            // update currentKeyboard Indicator
            if (keyboardType === ICTouchAPI.KeyboardTypes.ALPHANUM) {
                //update indicator when not password mode.
                this._objCurrentKeyboard._isPasswordKeyBoard = keyboardFlags.password;
                if (keyboardFlags.password && 'Hangul' == this._objCurrentKeyboard.strLang){
                    this.imeLangSwitcher.lastIME = "Hangul";
                    this.switchKeyboardFromIMEIndicator("apac_en");
                }
                this._objCurrentKeyboard.updateIMEIndicator();
                if(this._objCurrentKeyboard.strLang == "cyrillic_russian")
                    this._objCurrentKeyboard._replaceButton(1,true,this._objCurrentKeyboard.createRussianB);
                this._objCurrentKeyboard.clearIMEInput();
            }
			var actionOutsideClick = objCustomParams.actionOutsideClick ? objCustomParams.actionOutsideClick : ICTouchAPI.keyboardServices.CLOSE_ACTION_EXIT;
            ICTouchAPI.toasterServices.showContent({
                "toasterContent" : this._objCurrentKeyboard,
                "onOutsideClick" : {context : this, func: function () {this.closeKeyboard(actionOutsideClick)}}
            });

			// Put the focus on the InputField inside Keyboard
			this._objCurrentKeyboard.focus();
			type=null;
			objCallingInputField=null;
			objCustomParams=null;
		},

		/**
		 * Preload a keyboard so the first load is faster
		 * @param {String} type is the type of the keyboard (num, alphanum...)
		 */
		preloadKeyboard: function(type) {
			var keyboardType = this._getSimpleKeyboardType(type);
			if( !keyboardType ) {
				return;
			}
			if (!this._arrKeyboards[keyboardType]) {
				// Create minimal keyboard
				var keyboard = new UIElements.Keyboard.KeyboardControl({
					strType		: keyboardType,
					strInputText: ""
				});
				this._arrKeyboards[keyboardType] = keyboard;
				ICTouchAPI.toasterServices.preloadContent(keyboard);
			}
		},

		/**
		 * Is a keyboard displayed
		 * @return {Boolean} true if a keyboard is displayed, false otherwise
		 */
		isKeyboardDisplayed : function() {
			if(this._objCurrentKeyboard!=null && this._objCurrentKeyboard.id!=null) {
				return ICTouchAPI.toasterServices.isContentDisplayed(this._objCurrentKeyboard);
			}
			else {
				return false;
			}
		},

		/**
		 * @ignore
		 */
		doCustomCallback: function(strId, args) {
			if (this._arrCustomCallbacks && this._arrCustomCallbacks[strId]) {
				this._arrCustomCallbacks[strId](args);
			}
			switch(strId) {
				case "voicemail":
					if(webapp.communication) {
						webapp.communication.data._boolDialKeyEntered = false;
						webapp.communication.voiceMailBtnCallback();
					}
					this.closeKeyboard(this.CLOSE_ACTION_EXIT);
					break;
				case "goto-communication":
					if(webapp.history) {
						webapp.history.onHistoryEnter();
					}
					this.closeKeyboard(this.CLOSE_ACTION_EXIT);
					break;
				case "goto-progkeys":
					if(webapp.contacts) {
						webapp.contacts.onFavoritesEnter();
					}
					this.closeKeyboard(this.CLOSE_ACTION_EXIT);
					break;
				case "goto-contact":
					if(webapp.contacts) {
						webapp.contacts.onContactsEnter();
					}
					this.closeKeyboard(this.CLOSE_ACTION_EXIT);
					break;
				case "show-latest":
					if(webapp.userservices){
						webapp.userservices.displayCacheNumbersPopup();
					}
					break;
			}
		},

        /**
         * @ignore
         */
        switchKeyboard: function () {
            ICTouchAPI.settingServices.getSetting("Defaultkeyboard", this, this._switchKeyboard);
        },

        /**
		 * @ignore
		 */
        switchKeyboardFromIMEIndicator: function (strLang) {
            ICTouchAPI.settingServices.getSetting("Defaultkeyboard", this, function (objSetting) {
                var keyboardID = null;
                for (var i in objSetting.allowedValues) {
                    if (objSetting.allowedValues[i].name == strLang) {
                        keyboardID = objSetting.allowedValues[i].id;
                        break;
                    }
                }
                if (keyboardID != null) {
                    ICTouchAPI.debugServices.debug( "Defaultkeyboard:  " + keyboardID);
                    ICTouchAPI.settingServices.setSettingValue("Defaultkeyboard", keyboardID);
                } else {
                    console.warn("String language %s not found in settings.", strLang);
                }
            });
        },

        /**
         * @ignore
         */
        isAPACIMELang: function (strLang) {
            return this.apacIMELangs.some(function (elem) {
                return elem == strLang
            });
        },

         /**
		 * @ignore
		 */
          _switchKeyboard: function(objSetting) {
            var arrKeyboard = [];
            var context = this;
            for (var i in objSetting.allowedValues) {
                if (objSetting.allowedValues[i].name == 'apac_en') continue;
                var listItem = {};
                if(objSetting.allowedValues[i].id == objSetting.jsValue){
                    listItem.boolSelected = true;
                }else{
                    listItem.boolSelected = false;
                }
                listItem.strLabel = _(objSetting.allowedValues[i].name, "UIElements.Keyboard")
                listItem.intIndex=i;
                listItem.itemCallback = dojo.hitch(this,this.updateKeyboard);
                arrKeyboard.push(listItem);
            }

                    this.keyboardPopup = ICTouchAPI.popupServices.addNewPopup({
                        strTitle: _("Keyboard", "UIElements.Keyboard"),
                arrPopupButtons: [
                    {
                        strButtonName: "Keyboard_popup_ok",
                            strButtonLabel:  _("Ok", "ICTouchAPI"),
							strButtonIcon: "generic-ok",
                            callback: function() {
                                    context.updateKeyboard();
                            }
                        }]
                    }, "LOW");
                    this.sltList =  new UIElements.SelectList.SelectListControl({
                        arrItems : arrKeyboard,
                        boolMultiselect	: false
                    });
					var objPopup = ICTouchAPI.popupServices.getPopup(this.keyboardPopup.priority, this.keyboardPopup.position);
					objPopup.placeContent(this.sltList);

                    objPopup.show();
                },

        /*
		 * @ignore
		 */
	_setConditionalFlags : function(keyboardFlags, type) {
		if(this._hotelMode) {
			keyboardFlags.hotelMode = true;
		}
		else{
			keyboardFlags.hotelMode = false;
		}
		switch(type) {
			case ICTouchAPI.KeyboardTypes.NUMSEARCH:
				// set visibility of right button
				keyboardFlags.gotoCommunication = false;
				keyboardFlags.dialByName = false;
				keyboardFlags.gotoContact = false;
				keyboardFlags.gotoProgkeys = false;
				keyboardFlags.gotoDirectory = false;
				keyboardFlags.customKeys = false;
				keyboardFlags.customKeysUpdated = false;
				if(this._multiDeviceRoutingMode){
					if(this.callingWebapp == "userservices"){
						keyboardFlags.latest = false;
						if(typeof webapp.userservices !== "undefined") {
							keyboardFlags.latest = 1;
						}
					}else{
						keyboardFlags.latest = true;
					}
				}
				if(!this._hotelMode) {
					if(typeof webapp.search !== "undefined") {
							keyboardFlags.dialByName = 1;
					}
					if(typeof webapp.contacts !== "undefined") {
							keyboardFlags.gotoContact = 2;
						}
						if(typeof webapp.communication !== "undefined") {
							keyboardFlags.gotoCommunication = 3;
						}
					if(typeof webapp.contacts !== "undefined") {
						keyboardFlags.gotoProgkeys = 5;
					}
				}
				else {
					// if we are in hotel mode, custom keys must be displayed in dialpad
					if (this._customKeysUpdated){
						this._customKeysUpdated = false;
						keyboardFlags.customKeysUpdated = true;
					}
					keyboardFlags.customKeys = true;
					keyboardFlags.actionButtonInDialpad = this._actionButtonInDialpad;
					}
				break;

		}
	},


		/**
		 * @ignore
		 */
		updateKeyboard : function() {
			var keyboardID = this.sltList.getSelected()[0];

			ICTouchAPI.popupServices.removePopup(this.keyboardPopup);
			ICTouchAPI.settingServices.setSettingValue("Defaultkeyboard",keyboardID);
            ICTouchAPI.debugServices.debug('keyboardID=' + keyboardID);
			//this.closeKeyboard(this.CLOSE_ACTION_EXIT);
		},

		/**
		 * @ignore
		 * Guess the current webapp name
		 */
		_getCurrentWebappName: function() {
			var name = ICTouchAPI.transitionServices.getCurrentScreenName();
			// Special case here, search webapp is a stealth one but we still want to display it in keyboard
			if (name !== "webapp.search.getSearch") {
				var name = ICTouchAPI.transitionServices.getCurrentScreenName(true);
			}
			if (typeof name == "string") {
				var webapp = name.split('.')[1];
				name = webapp ? webapp : "homepage";
			}
			else {
				name = "homepage";
			}
			return name;
		},


		/**
		 * @ignore
		 * Callback of hotel mode subscribe
		 */
		_hotelModeReceveid : function() {
			this._hotelMode = true;
			// Default settings for simulation mode
            if(!generalConfig.simulation) {
				ICTouchAPI.settingServices.getSetting("ActionButtonInDialpad", this, this._actionButtonInDialpadReceived);
				ICTouchAPI.settingServices.subscribeToSetting(this,"ActionButtonInDialpad",this._actionButtonInDialpadReceived);
				ICTouchAPI.settingServices.getSetting("DialpadKeys", this, this._arrDialpadKeysReceivedFirst);
				ICTouchAPI.settingServices.subscribeToSetting(this,"DialpadKeys",this._arrDialpadKeysReceived);
            }
		},

		/**
		 * @ignore
		 * Callback of multi device rout
		 */
		_multiDeviceRoutingRecieved : function() {
			this._multiDeviceRoutingMode = true;
		},

		/**
		 * @ignore
		 * Callback of ActionuttonInDialpad setting
		 */
		_actionButtonInDialpadReceived : function(actionButtonInDialpad) {
			if(actionButtonInDialpad && actionButtonInDialpad.jsValue != undefined && actionButtonInDialpad.jsValue != null){
				this._actionButtonInDialpad = actionButtonInDialpad.jsValue;
				this._customKeysUpdated = true;
			}
		},

		/**
		 * @ignore
		 * Callback of DialpadKeys setting (first time)
		 */
		_arrDialpadKeysReceivedFirst : function(dialpadKeys){
			this._arrDialpadKeysReceived(dialpadKeys);
			if(dialpadKeys && dialpadKeys.jsValue != undefined && dialpadKeys.jsValue != null){
				ICTouchAPI.settingServices.getSettingValue("HomepageProgKeysLocalisation", this, this.loadedProgKeysLocalisationFirst);
				ICTouchAPI.settingServices.subscribeToSetting(this,"HomepageProgKeysLocalisation",this.loadedProgKeysLocalisation);
		}
		},

		/**
		 * @ignore
		 * Callback of DialpadKeys setting
		 */
		_arrDialpadKeysReceived : function(dialpadKeys){
			try {
				if(dialpadKeys && dialpadKeys.jsValue != undefined && dialpadKeys.jsValue != null){
					this._arrDialpadKeys = dialpadKeys.jsValue;
					this._customKeysUpdated = true;
				}
			} catch(err) {
				ICTouchAPI.debugServices.debug(err.description, false);
			}
		},

		getDialpadKeys : function(){
			return this._arrDialpadKeys;
		},

		/**
		 * @ignore
		 * Callback of HomepageRegisterKey event
		 */
		registerDialpadKey : function(strWebapp, strButtonName, strButtonIcon, strLabel, strStatusIcon, callBack, position, callbackLong){
			var button = {};
			button = {
				strButtonName: strButtonName,
				strButtonLabel: this.getKeyDMLabel(strLabel),
				callback : function(){
					callBack();
					ICTouchAPI.keyboardServices.closeKeyboard(ICTouchAPI.keyboardServices.CLOSE_ACTION_EXIT);
				},
				strButtonIcon: strButtonIcon
			};

			if(strLabel && !strLabel.isI18Ned){
				button.strLabelId = strLabel;
			}

			this._arrRegisteredKeys.push(button);
			this._customKeysUpdated = true;
		},

		unregisterDialpadKey : function(strButtonName, boolForceDeleteKey){
			for(var i = 0; i < this._arrRegisteredKeys.length; i++){
				if(this._arrRegisteredKeys[i].strButtonName === strButtonName){
					this._arrRegisteredKeys.splice(i, 1);
					break;
				}
			}
			this._customKeysUpdated = true;
		},

		/**
		 * @ignore
		 * Callback of HomepageProgKeysLocalisation setting (fist time)
		 */
		loadedProgKeysLocalisationFirst : function (value) {
			this.importDMTranslation(value);
			this.applyDMTranslation();
			ICTouchAPI.settingServices.getSetting("HomepageProgKeys", this, this.loadedProgKeysList);
			ICTouchAPI.settingServices.subscribeToSetting(this,"HomepageProgKeys",this.loadedProgKeysList);
		},

		/**
		 * @ignore
		 * Callback of HomepageProgKeysLocalisation setting (subscription)
		 */
		loadedProgKeysLocalisation : function (value) {
			value = (value && value.jsValue) || value;
			this.importDMTranslation(value);
			// apply the updated translations
			this.applyDMTranslation();
		},

		/**
		 * @private
		 * get the DM translation object from the Json value of the setting HomepageProgKeysLocalisation
		 */
		importDMTranslation : function (value) {
			try {
				if (value) {
					this._arrProgKeysLocalisation = dojo.fromJson(value);
				}
			} catch(err) {
				ICTouchAPI.debugServices.debug(err.description, false);
			}
		},

		/**
		 * @ignore
		 * Callback of HomepageProgKeys setting
		 */
		loadedProgKeysList : function(arrProgKeys){
			var _arrProgKeys = [];
			try {
				if(arrProgKeys && arrProgKeys.jsValue != undefined && arrProgKeys.jsValue != null){
					_arrProgKeys = dojo.fromJson(arrProgKeys.jsValue);
				}
			} catch(err) {
				ICTouchAPI.debugServices.debug(err.description, false);
			}
			this._arrProgKeys = [];
			for(var i=0;i<_arrProgKeys.length;i++)
			{
				if(_arrProgKeys[i]!==null && _arrProgKeys[i]!==undefined && _arrProgKeys[i]!=""){
					var strLabel = this.getKeyDMLabel(_arrProgKeys[i].label);
					var button = {
						strButtonName: _arrProgKeys[i].id,
						strButtonLabel: strLabel,
						strCustomClass: (_arrProgKeys[i].webAppName == "communication" && _arrProgKeys[i].action=="call"?'call':''),
						callback : function(){
							ICTouchAPI.keyboardServices.closeKeyboard(ICTouchAPI.keyboardServices.CLOSE_ACTION_EXIT);
						},
						strButtonIcon: _arrProgKeys[i].icon
					};
					// another closure to prevent the variable key to be overwritten in each iteration of the loop
					(function(key) {
						button.callback = function(){
							webapp[key.webAppName].onProgKeyClick(key);
						};
					})(_arrProgKeys[i]);
					// save strLabelId if it is not a i18n object to save the reference to the msgid in the DM translations
					if (!button.strButtonLabel.isI18Ned) {
						button.strLabelId = _arrProgKeys[i].label;
					}
					this._arrProgKeys.push(button);
				}
			}
			this._customKeysUpdated = true;
		},

		/**
		 * Get dialpad key from key id
		 */
		getCustomKey : function(strButtonName){
			if(strButtonName){
				for(var i = 0; i < this._arrRegisteredKeys.length; i++){
					if(this._arrRegisteredKeys[i].strButtonName === strButtonName){
						return this._arrRegisteredKeys[i];
					}
				}
				for(var j = 0; j < this._arrProgKeys.length; j++){
					if(this._arrProgKeys[j].strButtonName === strButtonName){
						return this._arrProgKeys[j];
					}
				}
			}
			return null;
		},

		/**
		 * Get Dialpad custom key label translation
		 */
		getKeyDMLabel : function (strLabelId) {
			// ICTouchAPI.debugServices.debug('ICTouchAPI.keyboardServices - getKeyDMLabel / search DM translation for strLabelId ' + strLabelId + " (current locale is: " + currentLocal + ")");

			var strLabel = "";
			// if the object is already i18ned, return it
			if (strLabelId.isI18Ned) {
			        ICTouchAPI.debugServices.debug('ICTouchAPI.keyboardServices - getKeyDMLabel / search DM translation for strLabelId ' + strLabelId.toTranslate + " (current locale is: " + currentLocal + ")");
				strLabel = strLabelId;
			}
			else if (this._arrProgKeysLocalisation) {
				var currentLocal = ICTouchAPI.i18nServices.getLocale();

				if (this._arrProgKeysLocalisation[currentLocal] && this._arrProgKeysLocalisation[currentLocal][strLabelId]){
					strLabel = this._arrProgKeysLocalisation[currentLocal][strLabelId];
					ICTouchAPI.debugServices.debug('ICTouchAPI.keyboardServices - getKeyDMLabel / DM translation found for strLabelId ' + strLabelId + " in locale " + currentLocal + ", translation is: " + strLabel);
				}
				// if the label is not defined in the current language, use the English translation if defined
				else if (this._arrProgKeysLocalisation["en"] && this._arrProgKeysLocalisation["en"][strLabelId]){
					strLabel = this._arrProgKeysLocalisation["en"][strLabelId];
					ICTouchAPI.debugServices.debug('ICTouchAPI.keyboardServices - getKeyDMLabel / no DM translation found for strLabelId ' + strLabelId + " in locale " + currentLocal + ", use english translation: " + strLabel);
				}
			}
			else {
			   ICTouchAPI.debugServices.debug('ICTouchAPI.keyboardServices - getKeyDMLabel / search DM translation for strLabelId ' + strLabelId + " (current locale is: " + currentLocal + ")");
			}
			// if the label hasn't been found (even in English), use the labelId
			if (!strLabel) {
				strLabel = strLabelId;
			}
			return strLabel;
		},

		// Change label with DM translation if exist
		applyDMTranslation : function() {
			ICTouchAPI.debugServices.info('ICTouchAPI.keyboardServices - applyDMTranslation');
			if (this._arrProgKeysLocalisation) {
				for (var i = 0; i < this._arrProgKeys.length; i++) {
					var button = this._arrProgKeys[i];
					if (button && button.strButtonLabel && !button.strButtonLabel.isI18Ned && button.strLabelId) {
						button.strButtonLabel = this.getKeyDMLabel(button.strLabelId);
						ICTouchAPI.debugServices.debug('ICTouchAPI.keyboardServices - applyDMTranslation / set computed label for button.strLabelId ' + button.strLabelId + " : " + button.strButtonLabel);
					}
				}
				for (var i = 0; i < this._arrRegisteredKeys.length; i++) {
					var button = this._arrRegisteredKeys[i];
					if (button && button.strButtonLabel && !button.strButtonLabel.isI18Ned && button.strLabelId) {
						button.strButtonLabel = this.getKeyDMLabel(button.strLabelId);
						ICTouchAPI.debugServices.debug('ICTouchAPI.keyboardServices - applyDMTranslation / set computed label for button.strLabelId ' + button.strLabelId + " : " + button.strButtonLabel);
					}
				}
			}
			this._customKeysUpdated = true;
		}


	});

ICTouchAPI.keyboardServices= new ICTouchAPI.keyboardServices();
/**
* @class ICTouchAPI.toasterServices
* @singleton
* @extends Object
* @ignore
*/
dojo.provide("ICTouchAPI.toasterServices");
dojo.declare("ICTouchAPI.toasterServices",null,
{
	/* --------------------------------- Public attributes ------------------------------------ */

  /**
   * Callback when clicking outside the Toaster
   * @cfg {Function} onClickOutside Function to call
   */
	onOutsideClickClbk : null,

	/* --------------------------------- Private attributes ----------------------------------- */

	/**
	 * @private
	 */
	_arrIToasters : {},
	/**
	 * @private
	 */
	_currentDisplayed : "",

	/**
	 * List of "hidden" callbacks for toasters
	 * @private
	 */
	_arrOnHideCallback: [],

	/* ------------------------------------ Constructor --------------------------------------- */

	/**
	 * @private
	 */
	constructor : function() {
		this._arrIToasters = {};
		this._currentDisplayed = "";
		this._arrOnHideCallback = [];
		ICTouchAPI.eventServices.subscribeToEvent(this, "SENSKEY_HOME", this._eventPhysicalKeyHome);
	},

	/**
	 * @private
	 */
	getApplicationName: function(){
		return "toasterServices";
	},

	/* ----------------------------------- Getter / Setter------------------------------------- */

	/**
	 * Return the id of the view displayed in the toaster
	 * @return {string} id of the view
	 */
	getContentDisplayedId : function (){
		if (this._currentDisplayed && this._arrIToasters[this._currentDisplayed]){
			return this._arrIToasters[this._currentDisplayed].id;
		}
		return "";
	},
	
	/**
	 * Return the IToaster of the view displayed in the toaster
	 * @return {string} id of the view
	 */
	getContentDisplayed : function (){
		if (this._currentDisplayed && this._arrIToasters[this._currentDisplayed]){
			return this._arrIToasters[this._currentDisplayed];
		}
		return null;
	},
	
        /**
         * Set the callback when clicking outside the Toaster
         * @param {Function} func The callback
         */
        setOnClickOutsideCallback : function(func) {
            ICTouchAPI.toasterServices.onOutsideClickClbk = func;
        },

        /**
         * Get the callback when clcking outside the Toaster
         * @return {Function} The callback
         */
        getOnClickOutsideCallback : function() {
            return (ICTouchAPI.toasterServices.onOutsideClickClbk);
        },


	/* ----------------------------------- Public methods ------------------------------------- */

	/**
	 * Show a toaster with a DialogBox created for the user
	 * @param {Object} toasterContent The content to place inside the DialogBox. The content can be null
	 * @param {Object} params Additionnal parameters
	 * @return {MyICPhoneAPI.form.DialogBox} the DialogBox (that must be passed to hideContent to hide it).
	 */
	showDialogBox: function(toasterContent, params) {
		var arrToasterButtons = [];
		if( params.funcOk ) {
			arrToasterButtons.push({
				strButtonName: "TOASTER_OK",
				strButtonLabel: _("Ok","ICTouchAPI"),
				strButtonIcon: "generic-ok",
				callback: params.funcOk
			});
		}
		if( params.funcCancel ) {
			arrToasterButtons.push({
				strButtonName: "TOASTER_CANCEL",
				strButtonLabel: _("Cancel","ICTouchAPI"),
				strButtonIcon: "generic-cancel",
				callback: params.funcCancel
			});
		}
		var objContent = new UIElements.DialogBox.DialogBoxControl({
			strTitle: params.strTitle || "",
			arrButtons: arrToasterButtons
		});

		if( typeof toasterContent == "object" && toasterContent != null ) {
			objContent.placeContent(toasterContent);
		}
		this.showContent({
			"toasterContent" : objContent,
			"onHide": params.onHide
		});
		return objContent;
	},

	/**
	 * Show the toaster
	 * @param {Object} args Json Object<pre><code>
	 * {"toasterContent" : ...
	 * "onOutsideClick" : ...
	 * "onHide" : ... }</code></pre>
	 */
	showContent : function (args) {
		var toasterContent = args.toasterContent;
		var id = toasterContent.id;
		this.onOutsideClickClbk = args.onOutsideClick;
		if (this._currentDisplayed !== toasterContent.id) {
			if  (!this._arrIToasters[id]) {
				this._addToaster(toasterContent);
				this._arrOnHideCallback[id] = args.onHide;
			}
			if (this._currentDisplayed !== "") {
				this._arrIToasters[this._currentDisplayed]._hide();
				this._callOnHide(this._currentDisplayed);
			}
			this._currentDisplayed = toasterContent.id;
		}
		this._arrIToasters[this._currentDisplayed]._show();
	},

	/**
	 * Preload the toaster
	 * @param {Object} toasterContent The Toaster content
	 */
	preloadContent : function (toasterContent) {
		if  (!this._arrIToasters[toasterContent.id]) {
			var toaster = this._addToaster(toasterContent);
			toaster._hide();
		}
	},

	/**
	 * Reload the toaster
	 * @private
	 * @param {Object} toasterContent The Toaster content
	 */
	reloadContent : function(toasterContent) {
		if  (this._arrIToasters[toasterContent.id]) {
			delete this._arrIToasters[toasterContent.id];
			this._arrIToasters.splice(toasterContent.id, 1);
		}
		var toaster = this._addToaster(toasterContent);
		toaster._hide();
	},

	/**
	 * Hide all Toaster
	 */
	hideAll : function () {
		if (this._currentDisplayed !=="") {
			this._arrIToasters[this._currentDisplayed]._hide();
			this._callOnHide(this._currentDisplayed);
			this._currentDisplayed = "";
		}
	},

	/**
	 * Hide a specific Toaster
	 * @param {Object} objContent The toaster to hide
	 */
	hideContent : function (objContent) {
		if (!objContent) {
			ICTouchAPI.debugServices.warning("ICTouchAPI.toasterServices - hideContent / Hum, can't hide the toaster... Why not passing a defined argument next time?!");
			return;
		}
		var id = objContent.id;
		var context = this;
		setTimeout(function(){
			context._arrIToasters[id]._hide();
			if(context._currentDisplayed === id) {
				context._callOnHide(id);
				context._currentDisplayed = "";
			}
		}, 15);
	},

    destroyContent : function(objContent) {
		if(objContent){
			var id = objContent.id;
			var frame = ICTouchAPI.toasterServices._arrIToasters[id];

			frame._hide();
			if(ICTouchAPI.toasterServices._currentDisplayed === id) {
				ICTouchAPI.toasterServices._currentDisplayed = "";
				this._callOnHide(id);
			}
			ICTouchAPI.skinServices.unregisterHeader(frame);
			objContent.destroy();
			dojo.destroy(frame.contentDocument.body);
			dojo.destroy(frame);
			delete ICTouchAPI.toasterServices._arrIToasters[id];
			delete this._arrOnHideCallback[id];
		}
    },

	/**
	 * When clicking outside the toaster, launch the callback defined in onOutsideClickClbk
	 * @private
	 */
	onHideToaster : function () {
		if (this.onOutsideClickClbk && typeof this.onOutsideClickClbk.func === "function") {
			this.onOutsideClickClbk.func.apply(this.onOutsideClickClbk.context);
		} else {
			ICTouchAPI.toasterServices.destroyContent(ICTouchAPI.toasterServices.getContentDisplayed());
		}
	},

	/**
	 * Is a specified Toaster displayed
	 * @param {Object} content The content to check
	 * @return {Boolean} True if this Toaster is displayed
	 */
	isContentDisplayed : function(content){
		if(this._arrIToasters[this._currentDisplayed] != null) {
			return (this._arrIToasters[this._currentDisplayed].id === content.id);
		}
		else {
			return false;
		}
	},

	/* ----------------------------------- Private methods ------------------------------------- */

	/**
	 * @private
	 */
	_eventPhysicalKeyHome : function(objEvent) {
		if(objEvent && objEvent.value !== 0){
			ICTouchAPI.toasterServices.hideAll();
		}
	},

	/**
	 * @private
	 */
	_addToaster: function (toasterContent) {
		var documentFragment = document.createDocumentFragment();
		var iFrame = document.createElement("iframe");
		documentFragment.appendChild(iFrame);
		iFrame.id = toasterContent.id;
		iFrame.src = "iframe.html";
		iFrame.style.height="0px";
		iFrame.className = "FullScreen";
		iFrame.style.zIndex = "10000";
		toasterContent.attr("class", "Toaster");
		iFrame.addEventListener('load', function () {
			//ICTouchAPI.skinServices.registerWebappHeader(this.contentDocument.getElementsByTagName('head')[0], toasterContent.namespace, "Toaster");
			ICTouchAPI.skinServices.registerHeader(this.contentDocument.getElementsByTagName('head')[0], toasterContent.namespace,"webapp",this);
			toasterContent.placeAt(this.contentDocument.body);
			dojo.create("div", {className: "toasterMask"}, this.contentDocument.body);
			iFrame.contentDocument.addEventListener('mousedown', function (event) {
				if (event.target.nodeName == "DIV" && event.target.className == "toasterMask") {
					ICTouchAPI.toasterServices.onHideToaster();
				}
			},false);
		},false);
		iFrame.addEventListener('load', function () {
			iFrame.style.height="";
			document.body.appendChild(documentFragment);
			iFrame.removeEventListener('load', arguments.callee, false);
		},false);
		iFrame._show = function () {
			this.style.top = "";
			dojo.publish("toaster.show", [this.id]);
			/* DEBUG */  ICTouchAPI.debugServices.debug('[toasterServices] _show()');
		};
		iFrame._hide = function () {
			this.style.top = "1000px";
			dojo.publish("toaster.hide", [this.id]);
			/* DEBUG */  ICTouchAPI.debugServices.debug('[toasterServices] _hide()');
		};
		document.body.appendChild(iFrame);
		this._arrIToasters[toasterContent.id] = iFrame;
		return iFrame;
	},

	/**
	 * @private
	 */
	_callOnHide: function(id) {
		var onHide = this._arrOnHideCallback[id];
		if (typeof onHide == "function") {
			onHide();
		}
	}
});

ICTouchAPI.toasterServices=new ICTouchAPI.toasterServices();
/**
 * @class ICTouchAPI.AppBarServices
 * @singleton
 * @extends Object
 * The AppBar save the action buttons which have to be displayed in the action bar for each main widget.<br/>
 * This list of action buttons is built from the static button list, the current widget button list, and the button list of the other webapps which require a link for the current widget. </br>
 * The webapps can add and remove some buttons and require an updating of the action bar.
 */
dojo.provide("ICTouchAPI.AppBarServices");
dojo.declare("ICTouchAPI.AppBarServices",null,{

	// attributs

	/**
         * @ignore
         */
	lstStaticButtons : {},			// list of objects which contains the action Static buttons for each widget

	/**
         * @ignore
         */
	lstExitButtons : {},			// list of objects which contains the exit buttons for each widget

	/**
         * @ignore
         */
	lstOtherWebAppButtons : {},     // list of objects which contains the action buttons coming from other webapp for each widget

	/**
         * @ignore
         */
	lstAppBarWidgets : {},       // list of objects wich contains the action bar for each widget

	/**
         * @ignore
         */
	lstGlobalButtons : [],			// array of buttons which are presents in all action bars

	//  methods of AppBarServices class

	lstAppBar : {},

	/**
         * @ignore
         */
	constructor : function(){
	},

	//  private methods

	createAppBar : function(strWebAppName, strWidgetName, _arrSkeleton, boolDisableScroll){
		if (!this.lstAppBar[strWebAppName]){
			this.lstAppBar[strWebAppName] = {};
		}
		if (!this.lstAppBar[strWebAppName][strWidgetName]){
			var appBarWidget = new UIElements.AppBar.AppBarWidget({
				webAppName:strWebAppName,
				widgetName:strWidgetName,
				boolDisableScroll:boolDisableScroll,
				arrSkeleton:_arrSkeleton
			});
			this.lstAppBar[strWebAppName][strWidgetName] = appBarWidget;
		}
		var appBar = this.lstAppBar[strWebAppName][strWidgetName];
		strWebAppName = null;
		strWidgetName = null;
		_arrSkeleton = null;
		return appBar;
	},

	/*
	 * This method doesn't create the appbar if it doesn't exist
	 * @param {String} strWebAppName
	 * @param {String} strWidgetName
	 * @return the appBar or null
	 */
	_getAppBar : function(strWebAppName, strWidgetName) {
		if( typeof this.lstAppBarWidgets[strWebAppName] == "undefined" ) {
			return null;
		}
		if( typeof this.lstAppBarWidgets[strWebAppName][strWidgetName] != "object" ) {
			return null;
		}

		return this.lstAppBarWidgets[strWebAppName][strWidgetName];
	},

	
	_getAppBarWidget : function(strWebAppName, strWidgetName) {
		var appBar = null;
		if( typeof this.lstAppBar[strWebAppName] !== "undefined" && typeof this.lstAppBar[strWebAppName][strWidgetName] == "object") {
			appBar = this.lstAppBar[strWebAppName][strWidgetName];
		}
		strWebAppName=null;
		strWidgetName=null;
		return appBar;
	},

	//  public methods


	/**
         * @ignore
         */
	updateAppBar : function () {
	},

	/**
	 * Add a button to the a webapp view
	 * @param {String} strWebAppName The name of the WebApp
	 * @param {String} strWidgetName The name of the View
	 * @param {UIElements.AppButton.AppButtonControl} objActionButton The AppBarButton to add
	 */
	addStaticButton : function(strWebAppName, strWidgetName, objActionButton){
		var buttonList;

		if (!this.lstStaticButtons[strWebAppName]) {
			this.lstStaticButtons[strWebAppName] = {}; // new object
		}
		if (!this.lstStaticButtons[strWebAppName][strWidgetName]) {
			this.lstStaticButtons[strWebAppName][strWidgetName] = []; // new array
		}
		buttonList = this.lstStaticButtons[strWebAppName][strWidgetName];

		var name = objActionButton.getButtonName();
		var appBar = this._getAppBar(strWebAppName, strWidgetName);
		if( appBar != null ) {
			appBar.addStaticButton(objActionButton);
		}else{
			appBar = this._getAppBarWidget(strWebAppName, strWidgetName);
			if( appBar != null ) {
				appBar.addStaticButton(objActionButton);
			}
		}

		if( typeof buttonList[name] != "undefined" ) {
			buttonList[name].destroy();
		}
		buttonList[name] = objActionButton; // index of array is the button name
	},

	/**
	 * Remove a button from a webapp view
	 * @param {String} strWebAppName The name of the WebApp
	 * @param {String} strWidgetName The name of the View
	 * @param {UIElements.AppButton.AppButtonControl} objActionButton The AppBarButton to add
	 */
	removeStaticButton : function(strWebAppName,strWidgetName,objActionButton){
		// get the name of button
		var name = objActionButton.getButtonName();
		// get the right button which is placed to strWidgetName of strWebAppName
		// because the button can be a clone of objActionButton
		var button = this.lstStaticButtons[strWebAppName][strWidgetName][name];
		// if button is a clone or original
		if (typeof button=="object" && button!=null && !button.original){
			//To destroy the cloned button
			objActionButton.destroyClone(strWebAppName,strWidgetName);
		}
		//remove button on appBar
		var appBar = this._getAppBar(strWebAppName, strWidgetName);
		if( appBar != null ) {
			appBar.removeStaticButton(objActionButton);
		}else{
			appBar = this._getAppBarWidget(strWebAppName, strWidgetName);
			if( appBar != null ) {
				appBar.removeStaticButton(objActionButton);
			}
		}
		// delete the reference of button, not the button, if no has the reference
		// of button, the garbage collector will destroy the button
		delete (this.lstStaticButtons[strWebAppName][strWidgetName][name]);
	},

	/**
         * Add a button to all views of a webapp or a set of webapps
	 * @param {Array} lstWebAppName An array containing the name of the webapps
	 * @param {UIElements.AppButton.AppButtonControl} objActionButton The button to add
	 */
	addStaticButtonMultipleAppBars : function(lstWebAppName, objActionButton){
		var strWebAppName,j;
		for (strWebAppName in lstWebAppName){
			var numberOfWidget = lstWebAppName[strWebAppName].length;
			for(j;j<numberOfWidget;j++){
				var strWidgetName = lstWebAppName[strWebAppName][j];
				var cloneButton = objActionButton.clone(strWebAppName,strWidgetName);
				this.addStaticButton(strWebAppName,strWidgetName,cloneButton);
			}
		}
	},

	/**
         * Add an extra button to all views of a webapp or a set of webapps
	 * @param {Array} lstWebAppName An array containing the name of the webapps
	 * @param {UIElements.AppButton.AppButtonControl} objActionButton The button to add
	 */
	addOtherWebAppButtonMultipleAppBars : function(lstWebAppName, objActionButton){
		var strWebAppName,j;
		for (strWebAppName in lstWebAppName){
			var numberOfWidget = lstWebAppName[strWebAppName].length;
			for(j;j<numberOfWidget;j++){
				var strWidgetName = lstWebAppName[strWebAppName][j];
				var cloneButton = objActionButton.clone(strWebAppName,strWidgetName);
				this.addOtherWebAppButton(strWebAppName,strWidgetName,cloneButton);
			}
		}
		lstWebAppName=null;
		objActionButton=null;
	},

	/**
	 * Add an extra button to a view of a WebApp
	 * @param {String} strWebAppName The name of the WebApp
	 * @param {String} strWidgetName The name of the View
	 * @param {UIElements.AppButton.AppButtonControl} objActionButton The AppBarButton to add
	 */
	addOtherWebAppButton : function(strWebAppName, strWidgetName, objActionButton){
		if (!this.lstOtherWebAppButtons[strWebAppName]) {
			this.lstOtherWebAppButtons[strWebAppName] = []; // new object
		}
		if (!this.lstOtherWebAppButtons[strWebAppName][strWidgetName]) {
			this.lstOtherWebAppButtons[strWebAppName][strWidgetName] = [];
		}
		var name = objActionButton.getButtonName();

		var appBar = this._getAppBar(strWebAppName, strWidgetName);
		if(appBar!=null){
			if (this.lstOtherWebAppButtons[strWebAppName][strWidgetName][name]) {
				appBar.removeOtherWebAppButton(appBar.getButton(name),false);
				this.lstOtherWebAppButtons[strWebAppName][strWidgetName][name].destroy();
				delete this.lstOtherWebAppButtons[strWebAppName][strWidgetName][name];
			}
			appBar.addOtherWebAppButton(objActionButton,false,false);
		}else{
			appBar = this._getAppBarWidget(strWebAppName, strWidgetName);
			if(appBar!=null){
				if (this.lstOtherWebAppButtons[strWebAppName][strWidgetName][name]) {
					appBar.removeOtherWebAppButton(appBar.getButton(name),true);
					this.lstOtherWebAppButtons[strWebAppName][strWidgetName][name].destroy();
					delete this.lstOtherWebAppButtons[strWebAppName][strWidgetName][name];
				}
				appBar.addOtherWebAppButton(objActionButton,false,true);
			}
		}
		this.lstOtherWebAppButtons[strWebAppName][strWidgetName][name] = objActionButton;
	},

	/**
	 * Remove extra button from a webapp view
	 * @param {String} strWebAppName The name of the WebApp
	 * @param {String} strWidgetName The name of the View
	 * @param {UIElements.AppButton.AppButtonControl} objActionButton The AppBarButton to add
	 */
	removeOtherWebAppButton : function(strWebAppName,strWidgetName,objActionButton){
		var name = objActionButton.getButtonName();
		var button = this.lstOtherWebAppButtons[strWebAppName][strWidgetName][name];
		// button is a clone or original ?
		if (!button.original){
			//To destroy the cloned button
			objActionButton.destroyClone(strWebAppName,strWidgetName);
		}


		var appBar = this._getAppBar(strWebAppName, strWidgetName);
		if( appBar != null ) {

			appBar.removeOtherWebAppButton(objActionButton,false);

		}else{
			appBar = this._getAppBarWidget(strWebAppName, strWidgetName);
			if( appBar != null ) {
				appBar.removeOtherWebAppButton(objActionButton,false);
			}
		}

		if (this.lstOtherWebAppButtons[strWebAppName][strWidgetName][name]) {
			this.lstOtherWebAppButtons[strWebAppName][strWidgetName][name].destroy();
			delete this.lstOtherWebAppButtons[strWebAppName][strWidgetName][name];
		}
	},

	/**
	 * Get the list of the buttons contained into a webapp view bar
	 * @param {String} strWebAppName The name of the WebApp
	 * @param {String} strWidgetName the name of the View
         * @return {Array} An array contained the list of the buttons or an empty array
	 */
	getStaticButtons : function(strWebAppName,strWidgetName){
		if(this.lstStaticButtons[strWebAppName]&&this.lstStaticButtons[strWebAppName][strWidgetName]){
			return this.lstStaticButtons[strWebAppName][strWidgetName];
		}
		else {
			return [];
		}
	},

	/**
	 * Get the list of the extra buttons contained into a webapp view bar
	 * @param {String} strWebAppName The name of the WebApp
	 * @param {String} strWidgetName the name of the View
         * @return {Array} An array contained the list of the buttons or an empty array
	 */
	getOtherWebAppButtons : function(strWebAppName,strWidgetName){
		if(this.lstOtherWebAppButtons[strWebAppName]&&this.lstOtherWebAppButtons[strWebAppName][strWidgetName]){
			return this.lstOtherWebAppButtons[strWebAppName][strWidgetName];
		}
		else{
			return []  ;
		}
	},

	/**
	 * Get the AppBar Widget for a webapp view
	 * @param {String} strWebAppName
	 * @param {String} strWidgetName
	 * @return {Object} The AppBar Widget (UI)
	 */
	getAppBar : function(strWebAppName,strWidgetName){
		var appBar;
		if(this.lstAppBar[strWebAppName] && this.lstAppBar[strWebAppName][strWidgetName]){
			appBar = this.lstAppBar[strWebAppName][strWidgetName];
		} else {
			if (!this.lstAppBarWidgets[strWebAppName]){
				this.lstAppBarWidgets[strWebAppName] = [];
			}
			if (!this.lstAppBarWidgets[strWebAppName][strWidgetName]){
				appBar = new UIElements.AppBar.AppBarWidget({
					webAppName:strWebAppName,
					widgetName:strWidgetName
				});
				this.lstAppBarWidgets[strWebAppName][strWidgetName] = appBar;
			} else {
				appBar = this.lstAppBarWidgets[strWebAppName][strWidgetName];
			}
		}
		//Add all global buttons if they are not already in
		for(var i in this.lstGlobalButtons){
			if (! appBar.getButton(this.lstGlobalButtons[i].strButtonName)){
				var uiBtn = new UIElements.AppButton.AppButtonControl(this.lstGlobalButtons[i]);
				this.addOtherWebAppButton(strWebAppName,strWidgetName,uiBtn,true);
			}
		}

		return appBar;
	},

	/**
	 * Remove the AppBar contained into a webapp view
	 * @param {String} strWebAppName The name of the WebApp
	 * @param {String} strWidgetName The name of the View
	 */
	destroyAppBar : function(strWebAppName,strWidgetName) {
		if (typeof this.lstAppBarWidgets[strWebAppName] == "undefined") {
			return;
		}

		if (this.lstAppBarWidgets[strWebAppName][strWidgetName]){
			if( typeof this.lstOtherWebAppButtons[strWebAppName] != "undefined" && typeof this.lstOtherWebAppButtons[strWebAppName][strWidgetName] != "undefined" ) {
				// Destroy Other webapp Button
				var list = this.lstOtherWebAppButtons[strWebAppName][strWidgetName];
				for(var i in list) {
					list[i].destroy();
					// list is a reference to the orginal array so it will be also deleted there
					delete list[i];
				}
			}

			if( typeof this.lstStaticButtons[strWebAppName] != "undefined" && typeof this.lstStaticButtons[strWebAppName][strWidgetName] != "undefined" ) {
				// Destroy static button
				list = this.lstStaticButtons[strWebAppName][strWidgetName];
				for(var i in list) {
					list[i].destroy();
					delete list[i];
				}
			}

			// Destroy AppBar
			this.lstAppBarWidgets[strWebAppName][strWidgetName].destroy();
			delete this.lstAppBarWidgets[strWebAppName][strWidgetName];
		}
	},

	/**
	 * @ignore
	 */
	reset : function(){
		this.lstAppBarWidgets = {};
	},

	/**
	 * Create an AppBar for a webapp view from another AppBar
	 * @param {String} strWebAppName The name of the WebApp
	 * @param {String} strWidgetName1 The name of the View to duplicate
	 * @param {String} strWidgetName2 The name of the view to create the AppBar
	 */
	setAppBar : function(strWebAppName,strWidgetName1,strWidgetName2){
		var appBar = this._getAppBar(strWebAppName,strWidgetName1);
		if(appBar != null){
			this.lstAppBarWidgets[strWebAppName][strWidgetName2] = appBar;

			// another service "transitionServices" has some references of appBar
			// in order to be more efficiency
			// it is necessary to set
			ICTouchAPI.transitionServices._setActionBar("webapp."+strWebAppName+"."+strWidgetName2,this.lstAppBarWidgets[strWebAppName][strWidgetName2]);
		}
		else{
			appBar = this._getAppBarWidget(strWebAppName,strWidgetName1);
			if(appBar!=null){
				this.lstAppBar[strWebAppName][strWidgetName2] = appBar;
				ICTouchAPI.transitionServices._setActionBar("webapp."+strWebAppName+"."+strWidgetName2,appBar);
			}
		}
		strWebAppName=null;
		strWidgetName1=null;
		strWidgetName2=null;
	},

	_setAppBar : function(strWebAppName,strWidgetName2,appBar){
		if(appBar!=null){
			ICTouchAPI.transitionServices._setActionBar("webapp."+strWebAppName+"."+strWidgetName2,appBar);
		}
	},

	/**
         * Add a button to all WebApps views
         * @param {UIElements.AppButton.AppButtonControl} btn The AppBarButton to add
         */
	addGlobalButton : function(btn){
		for (var k in this.lstGlobalButtons){
			if (this.lstGlobalButtons[k].strButtonName == btn.strButtonName){
				return;
			}
		}
		this.lstGlobalButtons.push(btn);
		var uiBtn,bar;
		for (var i in this.lstAppBarWidgets){
			if(this.lstAppBarWidgets[i]){
				for(var j in this.lstAppBarWidgets[i]){
					uiBtn = new UIElements.AppButton.AppButtonControl(btn);
					bar = this._getAppBar(i,j);
					if(uiBtn && bar){
						this.addOtherWebAppButton(bar.params.webAppName,bar.params.widgetName,uiBtn);
					}
				}
			}
		}
		for (var l in this.lstAppBar){
			if(this.lstAppBar[l]){
				for(var m in this.lstAppBar[l]){
					uiBtn = new UIElements.AppButton.AppButtonControl(btn);
					bar = this._getAppBarWidget(l,m);
					if(uiBtn && bar){
						this.addOtherWebAppButton(bar.params.webAppName,bar.params.widgetName,uiBtn);
					}
				}
			}
		}
	},

	/**
         * Remove a global button from all WebApps views
         * @param {string} btnName The name of the AppButton to remove
         */
	removeGlobalButton : function(btnName){
		var bar;
		if(this.lstAppBar){
			for (var k in this.lstAppBar){
				if(this.lstAppBar[k]){
					for(var l in this.lstAppBar[k]) {
						bar = this._getAppBarWidget(k,l);
						if(bar) {
							if(this.lstOtherWebAppButtons[bar.params.webAppName][bar.params.widgetName][btnName]){
								this.removeOtherWebAppButton(
									bar.params.webAppName,
									bar.params.widgetName,
									this.lstOtherWebAppButtons[bar.params.webAppName][bar.params.widgetName][btnName]
									);
							}
						}
					}
				}
			}
		}
		if(this.lstAppBarWidgets){
			for (var i in this.lstAppBarWidgets){
				if(this.lstAppBarWidgets[i]){
					for(var j in this.lstAppBarWidgets[i]) {
						bar = this._getAppBar(i,j);
						if(bar) {
							if(this.lstOtherWebAppButtons[bar.params.webAppName][bar.params.widgetName][btnName]){
								this.removeOtherWebAppButton(
									bar.params.webAppName,
									bar.params.widgetName,
									this.lstOtherWebAppButtons[bar.params.webAppName][bar.params.widgetName][btnName]
									);
							}
						}
					}
				}
			}
		}
		for (var m in this.lstGlobalButtons){
			if (this.lstGlobalButtons[m].strButtonName == btnName)
				this.lstGlobalButtons.splice(m,1);
		}
	},




	/**
	 * Add a Home or Back button to a WebApp view
	 * @param {String} strWebAppName The name of the WebApp
	 * @param {String} strWidgetName The name of the View
	 * @param {Boolean} isHome True if the button is a Home button. False for a Back button
	 * @param {Function} callback The callback function
	 */
	addExitButton : function(strWebAppName, strWidgetName, isHome, callback) {
		//Create label
		var strButtonLabel;
		var strButtonIcon;
		var boolNoCreate=false;
		if(isHome){
			strButtonLabel = _("Home", "ICTouchAPI");
			strButtonIcon = 'generic-homepage';
			if(this.lstExitButtons[strWebAppName] && this.lstExitButtons[strWebAppName][strWidgetName] && this.lstExitButtons[strWebAppName][strWidgetName].strButtonLabel == strButtonLabel.getTranslation()){
				boolNoCreate = true;
			}
		}
		else {
			strButtonLabel = _("Back", "ICTouchAPI");
			strButtonIcon = 'generic-back';
		}
		if(!boolNoCreate){
			if (!this.lstExitButtons[strWebAppName]) {
				this.lstExitButtons[strWebAppName] = {}; // new object
			}
			//Remove old exit button if exist
			if (this.lstExitButtons[strWebAppName][strWidgetName]!=null) {
				this.removeExitButton(strWebAppName, strWidgetName);
			}
			//Create button
			var buttonExit = {
				strButtonName: "BACK_BTN",
				strButtonLabel: strButtonLabel,
				strButtonIcon: strButtonIcon,
				callback: callback
			}
			var objButtonExit = new UIElements.AppButton.AppButtonControl(buttonExit);
			this.lstExitButtons[strWebAppName][strWidgetName] = objButtonExit; // index of array is the button name
			ICTouchAPI.AppBarServices.addStaticButton(strWebAppName, strWidgetName, objButtonExit);
		}
		strWebAppName=null;
		strWidgetName=null;
		isHome=null;
		callback=null;
	},

	/**
	 * Remove a Home or Back button from a WebApp view
	 * @param {String} strWebAppName The name of the WebApp
	 * @param {String} strWidgetName The name of the View
	 */
	removeExitButton : function(strWebAppName, strWidgetName){
		if(this.lstExitButtons[strWebAppName] && this.lstExitButtons[strWebAppName][strWidgetName]){
			var objButtonExit = this.lstExitButtons[strWebAppName][strWidgetName];
			if(objButtonExit!=null) {
				ICTouchAPI.AppBarServices.removeStaticButton(strWebAppName, strWidgetName, objButtonExit);
				objButtonExit.destroy();
				delete (this.lstExitButtons[strWebAppName][strWidgetName]);
			}
		}
	}


});
// End of the appBarServices class

// Instanciation of class
ICTouchAPI.AppBarServices = new ICTouchAPI.AppBarServices;
/**
* @class ICTouchAPI.popupServices
* @extends Object
* @singleton
* Manage and display popup in the Phone
*/
dojo.provide("ICTouchAPI.popupServices");
dojo.declare("ICTouchAPI.popupServices",null,

{
/* --------------------------------- Public attributes ------------------------------------ */
	/**
	 * Table of popups ordered by priority level and creation order
         * @ignore
	 */
	_arrPopup : [],
	/**
	 * List of business popup
         * @ignore
	 */
	_lstBusinessPopup : [],
	/**
	 * Object list of business popup id to delete on end of hPopup creation
         * @ignore
	 */
	_objListBusinessPopupToDelete : {},
	/**
	 * Reference to the DOM popup container
         * @ignore
	 */
	_popupContainer : null,
	/**
	 * Number of displayed popup : 0
         * @ignore
	 */
	_intDisplayedPopups : 0,
	/**
	 * Is the DOM container displayed
         * @ignore
	 */
	_isDisplayed : false,
	/**
	 * Reference to the current popup
         * @ignore
	 */
	_currentDisplayedPopup : null,
	/**
	 * @ignore
	 */
	_myIFrame : null,
	/**
	 * "Constant" definition for the priorities list
         * @ignore
	 */
	_lstPriorities : {
		"LOW" : 0,
		"MIDDLE" : 1,
		"HIGH" : 2,
		"CRITICAL" : 3
	},
	/**
	 * @ignore
	 */
	_isATimeout : false,

/* --------------------------------- Private attributes ----------------------------------- */

/* ------------------------------------ Constructor --------------------------------------- */

	/**
	 * @ignore
	 */
	constructor : function(){
		this._arrPopup = [];
		this._myIFrame = document.createElement('iframe');
		this._myIFrame.className = "FullScreen";
		this._myIFrame.src = "iframe.html";
		this._myIFrame.style.zIndex = "15000";
		this._myIFrame._show = function () {
            if(this.contentDocument) {
                var div = this.contentDocument.getElementById('opacityZone');
                div.style.backgroundColor = "rgba( 0, 0, 0, 0.8 )";
            }
			this.style.left = "";
		};
		this._myIFrame._hide = function () {
            if(this.contentDocument) {
                var div = this.contentDocument.getElementById('opacityZone');
                div.style.backgroundColor = "transparent";
            }
			this.style.left = "-1000px";
		};
		this._myIFrame.id="Popup";
		var that = this;
		this._myIFrame.onload = function () {
			var headerIFrame = dojo.query("head",that._myIFrame.contentDocument)[0];
			ICTouchAPI.skinServices.registerHeader(headerIFrame,"Popup","webapp",that._myIFrame);
			that._popupContainer = this.contentDocument.body;
			var opacityZone = document.createElement('div');
			opacityZone.id="opacityZone";
			that._popupContainer.appendChild(opacityZone);
		};
		this._myIFrame._hide();
		document.body.appendChild(this._myIFrame);
		ICTouchAPI.eventServices.subscribeToEvent(this, "SHOW_POPUP_EVENT", this._addBusinessPopup);
		ICTouchAPI.eventServices.subscribeToEvent(this, "HIDE_POPUP_EVENT", this._hideBusinessPopup);
	},

/* ----------------------------------- Getter / Setter------------------------------------- */

	/**
	 * Retrieve a popup
	 * @param {Number} priority Priority of the popup
	 * @param {Number} popup Position of the popup
	 *
	 */
	getPopup : function(priority, popup) {
		return this._arrPopup[priority][popup];
	},

	/**
	 * Retrieve popups of priority
	 * @param {Number} priority Priority of the popup
	 *
	 */
	getPopupsByPriority : function(priority){
		return this._arrPopup[priority];
	},

	/**
	 * Retrieve the numbers of openned popups
	 * @return {Number} Number of popups
	 */
	getNumberOfDisplayedPopups: function(){
		return this._intDisplayedPopups;
	},

/* ----------------------------------- Public methods ------------------------------------- */

	/**
	 * Create and publish a popup
	 * @param {Object} params Params corresponding to the UIElement Popup creation
	 * @param {String} priority Priority of the popup, one of : LOW | MIDDLE | HIGH | CRITICAL
         * @return {Object} Created popup reference, used for popup control
	 */
	addNewPopup : function(params, priority){
		params.selfDestroyCallback = dojo.hitch(this,this.onSelfDestroy)
		var priorityLocal = "";
		priorityLocal = this._lstPriorities[priority] || 0;
		for (var i in this._arrPopup[priorityLocal])
		{
			if (params.strContent.toTranslate && this._arrPopup[priorityLocal][i].strContent.toTranslate) {
				if (this._arrPopup[priorityLocal][i].strContent.toTranslate == params.strContent.toTranslate) {
					var popup = this._arrPopup[priorityLocal][i];
					return {position : popup.position, priority : popup.priority, id : popup.id};
				}
			}
		}
		var newPopupContainer = document.createElement("div");
		this._popupContainer.appendChild(newPopupContainer);
		var newPopup = new UIElements.DialogBox.DialogBoxControl(params, newPopupContainer);
		newPopup.attr("class", "Popup");
		newPopup.priority = priorityLocal;
		if(params.original && params.original.id) {
			newPopup.businessId = params.original.id;
		} else {
			newPopup.businessId = null;
		}
		if(params.id){
			newPopup.id=params.id;
		}

		this._registerPopup(newPopup);
		this._needDisplay(newPopup);
		params=null;
		priority=null;
		return {position:newPopup.position, priority:newPopup.priority ,id:newPopup.id};
	},

	/**
     *@ignore
     */
    onSelfDestroy : function(popup){
		if (this._arrPopup[popup.priority][popup.position]){
			this.removePopup(popup);
		}
		else{
			//Due to a bug in popup services (not easily reproductible)
			//Sometimes, the popup can not be close using the standard buttons
			//so Reset popup display state
			this._myIFrame._hide();
			this._isDisplayed = false;
			this._currentDisplayedPopup = null;
		}
		popup.destroy();

		console.log( "[popupServices] onSelfDestroy" );
	},

	/**
	 * Remove a popup
	 * @param {Object} popup Reference gave by the addNewPopup method
	 */
	removePopup : function(popup){
		var popupTmp = this._arrPopup[popup.priority][popup.position];
		if(popupTmp){
			this._unregisterPopup(popupTmp);
			this._noneedDisplay(popupTmp);
		}
		popup=null;
	},

	/**
	 * Remove a popup
	 * @param {Object} popup Reference gave by the addNewPopup method
	 */
	removePopupByID : function(popup){
		var popupTmp = null;
		for(var i in this._arrPopup[popup.priority])
		{
			if(popup.id==this._arrPopup[popup.priority][i].id){
				popupTmp = this._arrPopup[popup.priority][i];
				break;
			}
		}
		if(popupTmp){
			this._unregisterPopup(popupTmp);
			this._noneedDisplay(popupTmp);
		}
	},

	/**
	 * Register the popup in the popup list, according to his priority
	 * @ignore
	 * @param {Object} newPopup created popup
	 */
	_registerPopup : function(newPopup){
		if(!this._arrPopup[newPopup.priority]){
			this._arrPopup[newPopup.priority]=[];
		}
		this._arrPopup[newPopup.priority].push(newPopup);
		newPopup.position = this._arrPopup[newPopup.priority].length - 1;
	},

	/**
	 * Register the popup in the display stack / Check if the new popup has to be shown
	 * @ignore
	 * @param {Object} popup created popup
	 */
	_needDisplay : function(popup){
		this._intDisplayedPopups++;
		if(this._isDisplayed === false){
			this._myIFrame._show();
			this._isDisplayed = true;
			this._showPopup(popup);
		}else if(this._currentDisplayedPopup.priority <= popup.priority){
			this._showPopup(popup);
		}
	},

	/**
	 * Display a popup
	 * @ignore
	 * @param {Object} popup popup
	 */
	_showPopup : function(popup){
		if(this._currentDisplayedPopup){
			dojo.removeClass(this._currentDisplayedPopup.domNode, "currentPopup");
		}
		dojo.addClass(popup.domNode, "currentPopup");

		console.log( "[popupServices] _showPopup" );
		dojo.publish( "dialogbox.show", [] );

		popup.show();
		this._currentDisplayedPopup = popup;
	},

	/**
	 * Unregister the popup in the popup list
	 * @ignore
	 * @param {Object} popup popup
	 */
	_unregisterPopup : function(popup){
		this._arrPopup[popup.priority].splice([popup.position],1);
		if(this._arrPopup[popup.priority].length === 0 ){
			if(popup.priority === this._arrPopup.length-1)
				do{
					this._arrPopup.pop();
				}while(this._arrPopup.length != 0 && this._arrPopup[this._arrPopup.length-1] == undefined)
			else{
				this._arrPopup[popup.priority] = undefined;
			}
		} else {
			var i = popup.position;
			var popupTmp;
			while(this._arrPopup[popup.priority][i]!= null){
				if(this._arrPopup[popup.priority][i] && this._arrPopup[popup.priority][i].businessId) {
					var businessId = this._arrPopup[popup.priority][i].businessId;
					popupTmp = this._lstBusinessPopup[businessId];
					if(popupTmp) {
						popupTmp.hPopup.position = i;
						this._arrPopup[popup.priority][i].position = i;
					}
				}else{
					this._arrPopup[popup.priority][i].position = i;
				}
				i++;
			}
		}
	},

	/**
	 * Unregister a popup in the display stack / Check if the popup was the current popup
	 * @ignore
	 * @param {Object} popup popup
	 */
	_noneedDisplay : function(popup){
		this._intDisplayedPopups--;
		if(this._intDisplayedPopups === 0 ){
			this._myIFrame._hide();
			this._isDisplayed = false;
			this._currentDisplayedPopup = null;
		}else if(this._currentDisplayedPopup === popup){
			var highestPriority=this._arrPopup.length-1;
			var highestPriorityPopup = this._arrPopup[highestPriority][(this._arrPopup[highestPriority].length)-1];
			this._showPopup(highestPriorityPopup);
		}
		popup.destroy();

		console.log( "[popupServices] _noneedDisplay" );
		dojo.publish( "dialogbox.hide", [] );
	},

	/**
	 * Create and display a standard error popup
	 * @param {String} strTitle  Title of popup
	 * @param {String} strMessage  Error message
	 * @param {String} strOkBtn Label of the OK Button ("OK" by default)
	 * @param {Number} intId Id of the Popup
	 * @param {String} strIconOKBtn Icon of the OK Button ("generic-ok" by default)
	 */
	errorPopup : function(strTitle, strMessage, strOkBtn, intId, strIconOKBtn) {
		var arrButtons = [{
			strButtonName: "POPUP_OK_BTN",
			strButtonLabel: strOkBtn ? strOkBtn : _("OK","ICTouchAPI"),
			strButtonIcon : strIconOKBtn ? strIconOKBtn : "generic-ok",
			callback : function(){
				if (this.objParent)
					ICTouchAPI.popupServices.removePopup(this.objParent);
			}
		}];
		var popup = {
			strTitle: strTitle,
			strType: "error",
			strContent: strMessage,
			intID : intId,
			arrPopupButtons: arrButtons
		};
		ICTouchAPI.popupServices.addNewPopup(popup, "MEDIUM");
		strTitle=null;
		strMessage=null;
		strOkBtn=null;
		intId=null;
	},

	/**
	 * Create and display a standard popup with the given type
	 * @param {String} strType Type of popup ("info", "warning", "error", "fatal_error")
	 * @param {String} strTitle Title of popup
	 * @param {String} strMessage Error message
	 * @param {String} strOkBtn Label of the OK button ("OK" if no argument specified)
	 * @param {Number} intId Id of the popup
	 * @param {String} strIconOKBtn Icon of the OK Button ("generic-ok" if no argument specified)
	 */
	basicPopup : function(strType, strTitle, strMessage, strOkBtn, intId, strIconOKBtn) {
		var arrButtons = [{
			strButtonName: "POPUP_OK_BTN",
			strButtonLabel: strOkBtn ? strOkBtn : _("OK","ICTouchAPI"),
			strButtonIcon : strIconOKBtn ? strIconOKBtn : "generic-ok",
			callback : function(){
				if (this.objParent)
					ICTouchAPI.popupServices.removePopup(this.objParent);
			}
		}];
		var popup = {
			strTitle: strTitle,
			strType: strType,
			strContent: strMessage,
			intID : intId,
			arrPopupButtons: arrButtons
		};
		ICTouchAPI.popupServices.addNewPopup(popup, "MEDIUM");
	},

	/**
	 * @ignore
	 */
	_addBusinessPopup : function(){

		//Add a new business popup, thanks to event parameters
		//Expected parameters are :
		//- id : unique id of business popup
		//- timeout : timeout to automatically hide business popup
		//- events : list of events
		//- level : 0-1-2 (low), 3 (middle), 4 (high), 5 (critical)
		//- showDetails : boolean, with / without details
		//- text : content
		//- details : if details are needed

		var objBusinessPopup = ICTouchAPI.tools.getEventArguments(arguments);
		if(objBusinessPopup.id == undefined || this._lstBusinessPopup[objBusinessPopup.id] != undefined){
			return false;
		}
		if(objBusinessPopup.timeout <= 0 && (objBusinessPopup.events == undefined || objBusinessPopup.events.length == 0)){
			return false;
		}
		if(objBusinessPopup.showDetails != true ){
			objBusinessPopup.showDetails = false;
		}
		var objPopup = this._createValidBusinessPopup(objBusinessPopup);
		this._lstBusinessPopup[objBusinessPopup.id]= objPopup;
		objPopup.original = objBusinessPopup;
		objPopup.hPopup = this.addNewPopup(objPopup, objPopup.level);
		if(objBusinessPopup.timeout > 0){
			this._isATimeout = true;
			objPopup.hTimeout=window.setTimeout(function(){
				ICTouchAPI.popupServices._sendEvent(objBusinessPopup, "timeout");
				ICTouchAPI.popupServices._hideBusinessPopupWithId(objBusinessPopup.id);
			}, objBusinessPopup.timeout);
		}
		if(this._objListBusinessPopupToDelete[objBusinessPopup.id]) {
			this._hideBusinessPopupWithId(objBusinessPopup.id);
			delete(this._objListBusinessPopupToDelete[objBusinessPopup.id]);
		}
		return true;
	},

	/**
	 * Create the business popup
	 * @ignore
	 * @param {Array} objBusinessPopup key/value array of arguments, defining the popup
	 */
	_createValidBusinessPopup : function(objBusinessPopup){
		switch(objBusinessPopup.level){
			case 0:
			case 1:
			case 2:
				objBusinessPopup.level = "LOW";
				objBusinessPopup.strType = "info";
				break;
			case 3:
				objBusinessPopup.level = "MIDDLE";
				objBusinessPopup.strType = "warning";
				break;
			case 4:
				objBusinessPopup.level = "HIGH";
				objBusinessPopup.strType = "error";
				break;
			case 5:
				objBusinessPopup.level = "CRITICAL";
				objBusinessPopup.strType = "fatal_error";
				break;
			default :
				break;
		}

		if(objBusinessPopup.textParams) {
			for(var i=0; i<objBusinessPopup.textParams.length; i++) {
				if(!objBusinessPopup.textParams[i].isI18Ned) {
					objBusinessPopup.textParams[i] = _(objBusinessPopup.textParams[i], "UIElements.DialogBox");
				}
			}
		}
		objBusinessPopup.translatedText=_(objBusinessPopup.text,"UIElements.DialogBox",objBusinessPopup.textParams);
		if (objBusinessPopup.details) {
			objBusinessPopup.translatedDetails = _(objBusinessPopup.details,"UIElements.DialogBox",objBusinessPopup.detailsParams);
		}
		if(objBusinessPopup.showDetails){
			objBusinessPopup.translatedDetailsContent = objBusinessPopup.translatedText.getTranslation() + "<br /><br />" + objBusinessPopup.translatedDetails.getTranslation();
		}else{
			objBusinessPopup.translatedDetailsContent = objBusinessPopup.translatedText;
		}
		var objPopup = {
			strType			: objBusinessPopup.strType,
			strContent		: objBusinessPopup.translatedDetailsContent,
			strText			: objBusinessPopup.translatedText,
			strDetails		: objBusinessPopup.translatedDetails,
			showDetails 	: objBusinessPopup.showDetails,
			arrPopupButtons : [],
			level			: objBusinessPopup.level
		}
		var callback = function(strButtonName){
			ICTouchAPI.popupServices._userAction(objBusinessPopup.id,strButtonName);
		};
		var nbButtons = 0;
		if(objBusinessPopup.events != undefined) {
			if(objBusinessPopup.events["OK"]){
				objPopup.arrPopupButtons.push({
					strButtonLabel: _("Ok","ICTouchAPI"),
					callback : callback,
					strButtonIcon : 'generic-ok',
					strButtonName : "btn_OK"
				});
				nbButtons++;
			}
			if(objBusinessPopup.events["CANCEL"]){
				objPopup.arrPopupButtons.push({
					strButtonLabel: _("Cancel","ICTouchAPI"),
					callback : callback,
					strButtonIcon : 'generic-cancel',
					strButtonName : "btn_CANCEL"
				});
				nbButtons++;
			}
			if(objBusinessPopup.events["YES"]){
				objPopup.arrPopupButtons.push({
					strButtonLabel: _("Yes","ICTouchAPI"),
					callback : callback,
					strButtonIcon : 'generic-ok',
					strButtonName : "btn_YES"
				});
				nbButtons++;
			}
			if(objBusinessPopup.events["NO"]){
				objPopup.arrPopupButtons.push({
					strButtonLabel: _("No","ICTouchAPI"),
					callback : callback,
					strButtonIcon : 'generic-cancel',
					strButtonName : "btn_NO"
				});
				nbButtons++;
			}
		}
		if(nbButtons === 0){
			objPopup.arrPopupButtons.push({
				strButtonLabel: _("Dismiss","ICTouchAPI"),
				callback : callback,
				strButtonIcon : 'generic-cancel',
				strButtonName : "btn_DISMISS"
			});
		}
		if(objPopup.strDetails && objPopup.strDetails != ""){
			objPopup.arrPopupButtons.push({
					strButtonLabel: _("Details","ICTouchAPI"),
					callback : callback,
					strButtonIcon : 'generic-search',
					strButtonName : "btn_DETAILS"
				});
		}
		return objPopup;
	},

	/**
	 * Callback action of a business popup button
	 * @ignore
	 * @param {String} id id of the business popup
	 * @param {String} strButtonName name of the button clicked
	 */
	_userAction : function(id, strButtonName){
		var currPopup = this._lstBusinessPopup[id];
		if(strButtonName == "btn_DETAILS"){
			//switching from shown details mode to hidden details mode
			currPopup.original.showDetails = !currPopup.original.showDetails;
			//saving a reference on the original params
			var original = currPopup.original;
			//removing the currently displayed popup
			this._hideBusinessPopupWithId(id);
			//Building a popup showing the details within the content
			var objPopup = this._createValidBusinessPopup(original);
			//saving a reference on the original params
			objPopup.original = original;
			//adding the new popup
			objPopup.hPopup = this.addNewPopup(objPopup, objPopup.level);
			this._lstBusinessPopup[original.id]= objPopup;
			//allowing the timout params only if the details are not shown
			if(!original.showDetails){
				if(original.timeout > 0){
					this._isATimeout = true;
					objPopup.hTimeout=window.setTimeout(function(){
						ICTouchAPI.popupServices._sendEvent(original, "timeout");
						ICTouchAPI.popupServices._hideBusinessPopupWithId(original.id);
					}, original.timeout);
				}
			}

		}else{

			switch(strButtonName){
				case "btn_OK":
					this._sendEvent(currPopup.original, "OK");

					break;
				case "btn_CANCEL":
					this._sendEvent(currPopup.original, "CANCEL");

					break;
				case "btn_YES":
					this._sendEvent(currPopup.original, "YES");

					break;
				case "btn_NO":
					this._sendEvent(currPopup.original, "NO");

					break;
				case "btn_DISMISS":
					this._sendEvent(currPopup.original, "timeout");
					break;
				default :
					break;
			}

			this._hideBusinessPopupWithId(id);
			}

	},

	/**
	 * Send an event if specified in the popup object
	 * @ignore
	 * @param {Object} objPopup Popup sent by infra
	 * @param {String} strEventName name of the event to send
	 */
	_sendEvent	: function(objPopup, strEventName) {
		if( typeof objPopup.events != "undefined" && typeof objPopup.events[strEventName] == "string" ) {
			var infraEventName = objPopup.events[strEventName];
			ICTouchAPI.APIServices.ICTGate.notifyEventPresentation({params:[infraEventName]});
		}
	},

	/**
	 * Hide / destroy a business popup (asked through an event
	 * @ignore
	 * @param {Object} objEvent event object, with "value" defined as business popup id
	 */
	_hideBusinessPopup : function(objEvent){
		this._hideBusinessPopupWithId(objEvent.value);
	},

	/**
	 * Hide / destroy a business popup thanks to its ID
	 * @ignore
	 * @param {String} strPopupId id of the business popup
	 */
	_hideBusinessPopupWithId : function(strPopupId){
		var currPopup = this._lstBusinessPopup[strPopupId];
		if (currPopup) {
			if (currPopup.hPopup){
				if(currPopup.hTimeout && this._isATimeout){
					window.clearTimeout(currPopup.hTimeout);
					this._isATimeout = false;
				}
				this.removePopup(currPopup.hPopup);
				this._lstBusinessPopup[strPopupId] = null;
				delete(this._lstBusinessPopup[strPopupId]);
			} else {
				this._objListBusinessPopupToDelete[strPopupId] = true;
			}
		} else {
			console.warn("Popup not found.");
		}
	},

/* --------------------------------- Private Methods -------------------------------------- */
	/**
	 * Get Application Name
	 * @ignore
	 * @return {String} The name of the Application Service
	 */
	getApplicationName : function(){
		return "popupServices";
	}
});

ICTouchAPI.popupServices=new ICTouchAPI.popupServices();
/**
* @class ICTouchAPI.notificationServices
* @singleton
* @extends Object
* Manage and display notifications to the Phone Notification Bar
*/
dojo.provide("ICTouchAPI.notificationServices");
dojo.declare("ICTouchAPI.notificationServices",null,
{
	/* --------------------------------- Public attributes ------------------------------------ */
	/**
	 * first notification slot index
         * @ignore
	 */
	NOTIFICATION_FIRST_SLOT : 0,
	/**
	 * second notification slot index
         * @ignore
	 */
	NOTIFICATION_SECOND_SLOT : 1,
	/**
	 * third notification slot index
         * @ignore
	 */
	NOTIFICATION_THIRD_SLOT : 2,
	/**
	 * large notification slot index
         * @ignore
	 */
	NOTIFICATION_LARGE_SLOT : 3,
	/**
	 * hidden notification slot index
         * @ignore
	 */
	NOTIFICATION_HIDDEN_SLOT : 4,

	/* --------------------------------- Private attributes ----------------------------------- */
	/**
	 * @ignore
	 */
	_arrNotifications : [],

        /**
	 * @ignore
	 */

	_lstNotifications : {},

	enumType : {
		GREEN : "notificationBackgroundGreen",
		RED : "notificationBackgroundRed",
		BLUE : "notificationBackgroundBlue",
		ORANGE : "notificationBackgroundOrange",
		EMERGENCY : "emergency",
		WAITING : "waiting"
	},

        /**
	* @ignore
	*/
        constructor : function(){
		this._arrNotifications = [];
		this._lstNotifications = {};
	},

	/**
	* Create and add a new notification into the Notification area
	* @param {Object} objNotification : The notification to add
	* @return {Bool} True if successfull
	*/
	addNotification : function(objNotification){
		if(objNotification && objNotification.strNotificationName && this._lstNotifications[objNotification.strNotificationName] == undefined){
			if(!this._lstNotifications[objNotification.strNotificationName]){
				objNotification._intPosition = -1;
				this._lstNotifications[objNotification.strNotificationName]= objNotification;
			}
			if((objNotification.intNotificationValue && objNotification.intNotificationValue > 0) || objNotification.strNotificationTypeIcon) {
				this._arrNotifications.splice(0,0,objNotification);
				this._adjustPositions();
				this._publishModification("notificationAdded", [objNotification]);
			}
			objNotification=null;
			return true;
		}
		objNotification=null;
		return false;
	},

        /**
	* Remove a notification from the Notification area
	* @param {String} notificationName : name of the notification to remove
	*/
	removeNotification : function(notificationName){
		var objNotification = this.getNotificationByName(notificationName);
		if (objNotification) {
			this._delNotification(objNotification);
		}
		notificationName=null;
	},

	/**
	 * Adjust position of each notification
	 * @ignore
	 */
	_adjustPositions : function(){
		var notifLength = this._arrNotifications.length;
		for(var i=0; i<notifLength; i++){
			this._arrNotifications[i]._intPosition = i;
		}
	},

	/**
	 * Delete a notification from the table
	 * @ignore
	 * @param {String} objNotification : Name of the notification
	 */
	_delNotification : function(objNotification){
		delete this._lstNotifications[objNotification.strNotificationName];
		if(objNotification && objNotification._intPosition != -1){
			this._arrNotifications.splice(objNotification._intPosition,1);
			this._publishModification("notificationDeleted", [objNotification, objNotification._intPosition]);
			this._adjustPositions();
		}
	},

	/**
	 * update a notification (Value has changed)
	 * @ignore
	 * @param {Object} objNotification : Notification to update
	 */
	_updateNotification : function(objNotification){
		//INCREMENTATION OR DECREMENTATION ON AN EXISTING ELEMENT
		if(objNotification.intNotificationValue > 0 && objNotification._intPosition >= 0){
			//this._arrNotifications.splice(objNotification._intPosition,1);
			//objNotification._intPosition = -1;
			//this._arrNotifications.splice(0,0,objNotification);
		}
		//DECREMENTATION ON AN EXISTING ELEMENT THAT MUST DISAPPEAR
		else if(objNotification.intNotificationValue === 0  && objNotification._intPosition >= 0){
			this._arrNotifications.splice(objNotification._intPosition,1);
			objNotification._intPosition = -1;
		}
		//INCREMENTATION ON AN EXISTING ELEMENT THAT MUST APPEAR
		else if(objNotification.intNotificationValue > 0 && objNotification._intPosition === -1){
			this._arrNotifications.splice(0,0,objNotification);
		}
		this._adjustPositions();
	},

	/**
	 * refresh a notification with possible new params
	 * @param {String} notificationName : name of the notification to update
	 * @param {Object} objNotification : notification new params (optionnal)
	 */
	refreshNotification : function(notificationName, objNotification){
		var objOldNotification = this.getNotificationByName(notificationName);
		if(objOldNotification && objOldNotification.intNotificationValue !== undefined){
			var modified = false;
			for(var i in objNotification){
				if(!modified && (objOldNotification[i] != objNotification[i])){
					modified = true;
				}
				objOldNotification[i] = objNotification[i];
			}
			if(modified){
				var oldPosition = objNotification._intPosition;
				this._publishModification("notificationModified", [objOldNotification]);
			}
		}
		notificationName=null;
	},

        /**
	 * Increment the value of a notification
	 * @param {String} notificationName The name of the notification
	 */
	incNotification : function(notificationName){
		var objNotification = this.getNotificationByName(notificationName);
		if(objNotification && objNotification.intNotificationValue !== undefined){
			var oldPosition = objNotification._intPosition;
			objNotification.intNotificationValue++;
			this._updateNotification(objNotification);
			if(oldPosition == -1){
				this._publishModification("notificationAdded", [objNotification]);
			}
			else{
				this._publishModification("notificationUpdated", [objNotification, oldPosition]);
			}
		}
		notificationName=null;
	},

        /**
	 * Decrement the value of a notification
	 * @param {String} notificationName The name of the notification
	 */
	decNotification : function(notificationName){
		var objNotification = this.getNotificationByName(notificationName);
		if(objNotification && objNotification.intNotificationValue && objNotification.intNotificationValue > 0){
			var oldPosition = objNotification._intPosition;
			objNotification.intNotificationValue--;
			this._updateNotification(objNotification);
			if(objNotification.intNotificationValue === 0){
				this._publishModification("notificationDeleted", [objNotification, oldPosition]);
			}
			else{
				this._publishModification("notificationUpdated", [objNotification, oldPosition]);
			}
		}
		notificationName=null;
	},

        /**
	 * Set the value of a notification
	 * @param {String} notificationName The name of the notification
	 * @param {Number} value : The new value
	 */
	setNotificationValue : function(notificationName, value){
		var objNotification = this.getNotificationByName(notificationName);
		if(value >=0){
			objNotification.intNotificationValue = value;
			var oldPosition = objNotification._intPosition;
			this._updateNotification(objNotification);
			if(objNotification.intNotificationValue == 0){
				if(oldPosition != -1){
					this._publishModification("notificationDeleted", [objNotification,oldPosition]);
				}
			}
			else{
				if(oldPosition != -1){
					this._publishModification("notificationUpdated", [objNotification, oldPosition]);
				}
				else{
					this._publishModification("notificationAdded", [objNotification]);
				}
			}
		}
		notificationName=null;
		value=null;
	},

       /**
	 * Remove all notifications
	 */
	resetNotifications : function() {
		for(var i in this._lstNotifications){
			var objNotification = this._lstNotifications[i];
			if(objNotification.intNotificationValue && objNotification.intNotificationValue>0){
				this.setNotificationValue(objNotification.strNotificationName, 0);
			}
		}
	},

        /**
	 * Get all the registered notifications
	 * @return {Array} The notifications
	 */
	getNotifications : function(){
		return this._arrNotifications;
	},

        /**
	 * Get a notification by name
         * @param {String} strName The name of the notification
	 * @return {Object} The corresponding notification
	 */
	getNotificationByName : function(strName){
		return this._lstNotifications[strName];
	},

	/**
	 * Get a notification object by name
	 * @ignore
	 * @return {Object} objNotification : Object notification with the corresponding name
	 */
	_publishModification : function(type, object){
		dojo.publish("notification/"+type,object);
	}
});

ICTouchAPI.notificationServices = new ICTouchAPI.notificationServices();
/**
* @class ICTouchAPI.CapabilityServices
* @singleton
* @extends Object
* The Capabilities class provide an interface to webapp to easily listen to a capability and fetch its status.
* It also publish the event for an AppButton.
*/
dojo.provide("ICTouchAPI.CapabilityServices");
dojo.declare("ICTouchAPI.CapabilityServices", null , {

/* --------------------------------- Public attributes ------------------------------------ */

/* --------------------------------- Private attributes ----------------------------------- */

	// Defines
        /**
	 * @ignore
	 */
	AVAILABLE 				: 0,

        /**
	 * @ignore
	 */
	UNAVAILABLE 			: 1,

        /**
	 * @ignore
	 */
	TEMP_UNAVAILABLE		: 2,

        /**
	 * @ignore
	 */
	UNKNOWN					: -1,

        /**
	 * @ignore
	 */
	EVENT_ROOT				: "capabilities_changed_",

        /**
	 * @ignore
	 */
	arrGlobalCapabilities	: {}, // Current status of global capabilities ( for exemple [Telephony][makeCall] = AVAILABLE )

        /**
	 * @ignore
	 */
	arrInstanceCapabilities	: {}, // Current status of instance capabilities ( [Telephony][1001][hold] = UNAVAILABLE )

        /**
	 * @ignore
	 */
	arrEvents				: [ ], // List of events current listened ( strEvent => number of time listened )

        /**
	 * @ignore
	 */
	intSubscriptions		: 0, // Number of subscriptions currently given

	/**
	* @ignore
	*/
	intPublishNumber		: 0, // Number of capabilities' instances published
/* ------------------------------------ Constructor --------------------------------------- */

	/**
	 * @ignore
	 */
	constructor : function () {

	},

/* ----------------------------------- Getter / Setter------------------------------------- */


/* ----------------------------------- Public methods ------------------------------------- */

	/**
         * @ignore
	 * Because the eventServices wants an application name we provide one.
	 * @return {String} Return CapabilityServices
	 */
	getApplicationName: function(){
		return "CapabilityServices";
	},

	/**
	 * Add a new list of capacities to listen to.
	 * @param {String} strModule The name of the module capability.
	 * @param {String} Not used (backward compatibility only).
	 */
	subscribeToCapabilityList : function(strModule, strEvent){
		var that = this;
		ICTouchAPI.eventServices.subscribeToEvent(this, "capabilities_list_changed_" + strModule, function(){
			var args = ICTouchAPI.tools.getEventArguments(arguments);
			if(args.capabilities){
				for(var i in args.capabilities) {
					if(args.listId) {
						that._onInstanceChanged(strModule, args.listId, {value:args.capabilities[i]});
					} else {
						that._onCapabilityChanged(strModule, {value:args.capabilities[i]});
					}
				}
			}
		});

		//If capabilityList is removed, set to UNKNOWN associeted capabilities.
		ICTouchAPI.eventServices.subscribeToEvent(this, "removeCapabilityList_" + strModule, function(){
			var args = ICTouchAPI.tools.getEventArguments(arguments);
			if(args.listId ) {
				if (that.arrInstanceCapabilities[strModule] && that.arrInstanceCapabilities[strModule][args.listId]) {
					for( var i in that.arrInstanceCapabilities[strModule][args.listId] ) {
						if (that.arrInstanceCapabilities[strModule][args.listId][i] != that.UNKNOWN) {
							that.arrInstanceCapabilities[strModule][args.listId][i] = that.UNKNOWN;
							that._publishCapability(strModule, i, that.UNKNOWN, args.listId);
						}
					}
				}
			}
		});
	},

	/**
	 * Add a new instance's capacity to listen to, must be called previous to getCapability.
	 * @param {String} strModule The name of the module capability.
	 * @param {String} instanceId The name of the capability to listen to.
	 * @param {Number} strCapability The instance id to listen to.
	 * @param {Object} context The context of the callback.
	 * @param {Function} callback The function to be called when the capability changes.
	 * @return {Object} handler to unsubscribe
	 */
	subscribeToInstance : function(strModule, instanceId, strCapability, context, callback){
		var publishName = this.EVENT_ROOT + strModule + strCapability + instanceId;
		ICTouchAPI.debugServices.debug("ICTouchAPI.CapabilityServices - subscribeToInstance  / Subscribing to instance: " + publishName, " CAPABILITIES");

		var subscribeHandler = dojo.subscribe(publishName, context, callback);
		var eventName = this.EVENT_ROOT + strModule + "_" + instanceId; // Event name on dbus

		this._listenEvent(eventName, dojo.hitch(this, this._onInstanceChanged, strModule, instanceId));

		if(this.arrInstanceCapabilities[strModule] == undefined){
			this.arrInstanceCapabilities[strModule] = {} ;
		}
		if( this.arrInstanceCapabilities[strModule][instanceId] == undefined ){
			this.arrInstanceCapabilities[strModule][instanceId] = {};
		}

		if( this.arrInstanceCapabilities[strModule][instanceId][strCapability] == undefined )
		{
			this.arrInstanceCapabilities[strModule][instanceId][strCapability] = this.UNKNOWN;
			this._fetchCapability(strModule, strCapability, instanceId);
		}
		else
		{
			this._publishCapability(strModule, strCapability, this.arrInstanceCapabilities[strModule][instanceId][strCapability], instanceId);
		}

		this.intSubscriptions++;
		// Create handler for our disconnect function
		return [subscribeHandler, eventName, strModule, instanceId, strCapability];
	},

	/**
	 * Add a new capacity to listen to, must be called previous to getCapability.
	 * @param {String} strModule The name of the module capability.
	 * @param {String} strCapability The name of the capability to listen to.
	 * @param {Object} context The context of the callback.
	 * @param {Function} callback The function to be called when the capability changes.
	 * @return {Object} handler to unsubscribe
	 */
	subscribeToCapability : function(strModule, strCapability, context, callback){
		var publishName = this.EVENT_ROOT + strModule + strCapability;
		ICTouchAPI.debugServices.debug("ICTouchAPI.CapabilityServices - subscribeToCapability / Subscribing to capability: " + publishName, " CAPABILITIES");
		var subscribeHandler = dojo.subscribe(publishName, context, callback);
		var eventName = this.EVENT_ROOT + strModule; // Event name on dbus
		this._listenEvent(eventName, dojo.hitch(this, this._onCapabilityChanged, strModule));
		if(this.arrGlobalCapabilities[strModule] == undefined){
			this.arrGlobalCapabilities[strModule] = {};
		}
		if( this.arrGlobalCapabilities[strModule][strCapability] === undefined ){
			this.arrGlobalCapabilities[strModule][strCapability] = this.UNKNOWN;
			this._fetchCapability(strModule, strCapability);
		}
		else {
			this._publishCapability(strModule, strCapability, this.arrGlobalCapabilities[strModule][strCapability]);
		}
		this.intSubscriptions++;
		var returnedParams = [subscribeHandler, eventName, strModule, strCapability];
		strModule=null;
		strCapability=null;
		context=null;
		callback=null;
		return returnedParams;
	},

	/**
	 * Disconnect the dojo event connection, however keep the eventService subscription alive (the capability will still be handled in CapabilityService)
         * @param {Object} objHandler The handler of subscribeToInstance or subscribeToCapability
	 */
	disconnectHandler : function(objHandler) {
		// Sanity check on the handler passed
		if(objHandler){
			if( typeof objHandler.length != "number" || objHandler.length < 4 || objHandler.length > 5 ) {
				ICTouchAPI.debugServices.warning("ICTouchAPI.CapabilityServices - disconnectHandler / function treatment interrupted:  disconnectHandler not a handler, objHandler.length:" + objHandler.length);
				return;
			}
			this.intSubscriptions--;
			// Disconnect publish handler
			var subscribeHandler = objHandler.shift();
			dojo.unsubscribe(subscribeHandler);
			// Disconect from ICTGate if necessary
			var eventName = objHandler.shift();
			var cleanup = this._stopListening(eventName);
			if( cleanup ) {
				if( objHandler.length == 2 ) {
					this._cleanupCapability(objHandler[0], objHandler[1]);
				}
				else {
					this._cleanupInstance(objHandler[0], objHandler[1], objHandler[2]);
				}
			}
		}
		else {
			ICTouchAPI.debugServices.warning("ICTouchAPI.CapabilityServices - disconnectHandler / function treatment interrupted: objHandler is not defined");
		}

	},

	/**
	 * Retrieve the named capability of an instance
	 * @param {String} strModule The name of the module capability.
	 * @param {Number} instanceId The instance id of the capability.
	 * @param {String} strCapability The name of the capability to listen to.
	 * @return {Array} An array containing every capability of an instance
	 */
	getCapability: function(strModule, instanceId, strCapability){
		var isInstance = arguments.length > 2;
		strCapability = isInstance ? arguments[2] : arguments[1];
		var value = null;
		if ( !isInstance && this.arrGlobalCapabilities[strModule] !== undefined )
			value = this.arrGlobalCapabilities[strModule][strCapability];
		if( isInstance && this.arrInstanceCapabilities[strModule] !== undefined && this.arrInstanceCapabilities[strModule][instanceId] !== undefined )
			value = this.arrInstanceCapabilities[strModule][instanceId][strCapability];
		if (value == null || value == undefined)
			value = this.UNKNOWN;
		return value;
	},

	/**
	 * Retrieve the capabilities of a module and maybe for an instance (optionnal)
	 * @param {String} strModule The name of the module capability.
	 * @param {Number} instanceId The instance id of the capability.
	 * @return {Array} An array containing every capability of an instance or a module
	 */
	getCapabilities: function(strModule, instanceId){
		var isInstance = arguments.length > 1;
		var value = null;
		if ( !isInstance && this.arrGlobalCapabilities[strModule] !== undefined ){
			value = this.arrGlobalCapabilities[strModule];
		}
		if( isInstance && this.arrInstanceCapabilities[strModule] !== undefined && this.arrInstanceCapabilities[strModule][instanceId] !== undefined ){
			value = this.arrInstanceCapabilities[strModule][instanceId];
		}
		if (value == null || value == undefined){
			value = [];
		}
		return value;
	},

	/**
	 * Retrieve the whole capabilities of an instance
	 * @param {String} strModule The name of the module capability.
	 * @param {Number} instanceId The instance id of the capability.
	 * @return {Array} An array containing every capability of an instance
	 */
	getInstance: function(strModule, instanceId){
		if( this.arrInstanceCapabilities[strModule] !== undefined && this.arrInstanceCapabilities[strModule][instanceId] !== undefined )
			// Maybe clone this object to avoid any modification ?
			return this.arrInstanceCapabilities[strModule][instanceId];
		return null;
	},

	/**
	 * Force the publishing of an event about a capability
	 * @ignore
	 * @param {String} strModule The name of the module capability
	 * @param {String} strCapability The name of the capability
	 * @param {Number} status The capability value
	 * @param {String} instanceId
	 */
	_publishCapability: function(strModule, strCapability, status, instanceId){
		var publishName;
		if( instanceId === undefined ) {
			publishName = this.EVENT_ROOT + strModule + strCapability;
			ICTouchAPI.log("Publishing capability: " + publishName + " (value: " + status + ")", "CAPABILITIES");
			dojo.publish(publishName, [strCapability, status]);
		}
		else {
			publishName = this.EVENT_ROOT + strModule + strCapability + instanceId;
			ICTouchAPI.log("Publishing capability instance: " + publishName + " (value: " + status + ")", "CAPABILITIES");
			dojo.publish(publishName, [strCapability, status, instanceId, ++this.intPublishNumber]);
			// intPublishNumber is here to avoid bad order in capability management in buttons.
		}
	},

	/**
	 * Listen to a dbus event and increment it's counter
	 * @ignore
	 * @param {String} strEvent Event name
	 * @param {Function} callback Callback of the event
	 */
	_listenEvent: function(strEvent, callback){
		if( typeof this.arrEvents[strEvent] == "undefined" ) {
			// Start listening
			this.arrEvents[strEvent] = 0;
			ICTouchAPI.eventServices.subscribeToEvent(this, strEvent, callback);
		}
		this.arrEvents[strEvent]++;
	},

	/**
	 * Stop listening to an event, if the counter falls to 0 unsubsribe from the eventServices
	 * @ignore
	 * @param {String} strEvent Event name
	 * @return {Boolean} return true if it was the last one
	 */
	_stopListening: function(strEvent){
		if( typeof this.arrEvents[strEvent] == "undefined" ) {
			ICTouchAPI.debugServices.warning("ICTouchAPI.CapabilityServices - _stopListening $ return false: _stopListening on something we are not listening ? (this.arrEvents[strEvent] is undefined)");
			return false;
		}
		if( this.arrEvents[strEvent] > 1 ) {
			// Still something listening for this event
			this.arrEvents[strEvent]--;
			return false;
		}
		else {
			// Nothing is listening to this event anymore
			ICTouchAPI.eventServices.unsubscribeToEvent(this, strEvent);

			delete this.arrEvents[strEvent];
			return true;
		}
	},

	/**
	 * Remove the capability from the array and remove the module if it's empty
	 * @ignore
	 * @param {String} strModule Name of module
	 * @param {String} strCapability Name of capability
	 */
	_cleanupCapability: function(strModule, strCapability){
		var i, empty;
		delete this.arrGlobalCapabilities[strModule][strCapability];

		// Test if this associative array is empty ( array.length doesn't work here )
		empty = true;
		for( i in this.arrGlobalCapabilities[strModule] ) {
			empty = false;
		}
		if( empty ) {
			delete this.arrGlobalCapabilities[strModule];
		}
	},

	/**
	 * Remove the capability from the array and remove the instance and the module if they are empty
	 * @ignore
	 * @param {String} strModule Name of module
	 * @param {Number} instanceId Instance id of the capability
	 * @param {String} strCapability Name of capability
	 */
	_cleanupInstance: function(strModule, instanceId, strCapability){
		var i, empty;
		delete this.arrInstanceCapabilities[strModule][instanceId][strCapability];
		// Test if this associative array is empty ( array.length doesn't work here )
		empty = true;
		for( i in this.arrInstanceCapabilities[strModule][instanceId] ) {
			empty = false;
		}
		if( empty ) {
			delete this.arrInstanceCapabilities[strModule][instanceId];
			// Test again on module
			empty = true;
			for( i in this.arrInstanceCapabilities[strModule] ) {
				empty = false;
			}
			if( empty ) {
				delete this.arrInstanceCapabilities[strModule];
			}
		}
	},

/* --------------------------------- Private Methods -------------------------------------- */

	/**
	 * @ignore
	 * Send logs of attributes of the service to the logger.
	 * @param {Boolean} boolAdvancedLogs Boolean indicating if advanced or basic logs has to be written
	 */
	dump : function(boolAdvancedLogs){
		ICTouchAPI.debugServices.dump("dump function of CapabilityServices with " + ((boolAdvancedLogs)?"advanced":"basic") + " logs option");

		ICTouchAPI.debugServices.dump("Capability states available: 0 = AVAILABLE / 1 = UNAVAILABLE / TEMP_UNAVAILABLE = 2 / UNKNOWN = -1");
		ICTouchAPI.debugServices.dump("Number of subscriptions currently given (intSubscriptions): " + this.intSubscriptions);
		ICTouchAPI.debugServices.dump("Number of capabilities' instances published (intPublishNumber): " + this.intPublishNumber);

		if (boolAdvancedLogs) {
			// List of events current listened ( strEvent => number of time listened )
			var strEvent;
			for (strEvent in this.arrEvents) {
				ICTouchAPI.debugServices.dump(this.arrEvents[strEvent] + " capabilities are listened for event " + strEvent);
			}

			var strModule, instanceId, strCapability;
			ICTouchAPI.debugServices.dump("List of global capabilities: ");
			for (strModule in this.arrGlobalCapabilities) {
				ICTouchAPI.debugServices.dump("Global capabilities for strModule " + strModule + ":");
				for (strCapability in this.arrGlobalCapabilities[strModule]) {
					ICTouchAPI.debugServices.dump("Global capability state for strModule " + strModule + ", strCapability " + strCapability + ": " + this.arrGlobalCapabilities[strModule][strCapability]);
				}
			}
			ICTouchAPI.debugServices.dump("List of instance capabilities: ");
			for (strModule in this.arrInstanceCapabilities) {
				ICTouchAPI.debugServices.dump("Instance capabilities for strModule " + strModule + ":");
				for (instanceId in this.arrInstanceCapabilities[strModule]) {
					ICTouchAPI.debugServices.dump("Instance capabilities for strModule " + strModule + ", instanceId " + instanceId);
					for (strCapability in this.arrInstanceCapabilities[strModule][instanceId]) {
						ICTouchAPI.debugServices.dump("Capability state for strModule " + strModule + ", instanceId " + instanceId + ", strCapability " + strCapability + ": " + this.arrInstanceCapabilities[strModule][instanceId][strCapability]);
					}
				}
			}
		}
	},
	
	/**
	 * @ignore
	 */
	disconnectInstance : function(objHandler) {
		ICTouchAPI.debugServices.warning("ICTouchAPI.CapabilityServices - disconnectInstance / Use disconnectHandler instead of disconnectInstance");
		this.disconnectHandler(objHandler);
	},

	/**
	 * @ignore
	 */
	disconnectCapability : function(objHandler) {
		ICTouchAPI.debugServices.warning("ICTouchAPI.CapabilityServices - disconnectCapability / Use disconnectHandler instead of disconnectCapability");
		this.disconnectHandler(objHandler);
	},

	/**
	 * @ignore
	 */
	unsubscribeInstance : function() {
		ICTouchAPI.debugServices.warning("ICTouchAPI.CapabilityServices - unsubscribeInstance / Do not use unsubscribeInstance");
	},
        
	/**
	 * @ignore
	 */
	unsubscribeCapability : function() {
		ICTouchAPI.debugServices.warning("ICTouchAPI.CapabilityServices - unsubscribeCapability / Do not use unsubscribeCapability");
	},

	/**
	 * @ignore
	 */
	getOnceCapabilities: function(strModule, objEvent, idCapabilities, context, callBack){
		ICTouchAPI.APIServices[strModule].getCapabilities({params:[idCapabilities], context: context, callback: callBack, callbackParams: objEvent});
	},

	/**
	 * @ignore
	 */
	_onCapabilityChanged: function(strModule, objEvent){
		ICTouchAPI.log("Global capability changed", "CAPABILITIES");
		ICTouchAPI.logObject(objEvent, "CAPABILITIES");
		this._fetchedCapability(objEvent.value, {module : strModule});
	},

	/**
	 * @ignore
	 */
	_onInstanceChanged: function(strModule, instanceId, objEvent){
		ICTouchAPI.log("Instance capability changed", "CAPABILITIES");
		ICTouchAPI.logObject(objEvent, "CAPABILITIES");
		this._fetchedCapability(objEvent.value, {module : strModule, instanceId : instanceId});
	},

	/**
	 * @ignore
	 */
	_fetchCapability: function(strModule, strCapability, instanceId){
		if ( instanceId === undefined )
			ICTouchAPI.APIServices[strModule].getCapabilityGlobal({params: [strCapability], context: this, callback: this._fetchedCapability, callbackParams: {module : strModule}});
		else
			ICTouchAPI.APIServices[strModule].getCapabilityInstance({params: [instanceId, strCapability], context: this, callback: this._fetchedCapability, callbackParams: {module : strModule, instanceId : instanceId}});
	},

	/**
	 * @ignore
	 */
	_fetchedCapability: function(capability, params){
		if( params.instanceId === undefined ){
			this.arrGlobalCapabilities[params.module][capability.name] = capability.state;
			this._publishCapability(params.module, capability.name, capability.state);
		} else {
			if(this.arrInstanceCapabilities[params.module] == undefined){
				this.arrInstanceCapabilities[params.module] = {};
			}
			if(this.arrInstanceCapabilities[params.module][params.instanceId] == undefined){
				this.arrInstanceCapabilities[params.module][params.instanceId] = {};
			}
			this.arrInstanceCapabilities[params.module][params.instanceId][capability.name] = capability.state;
			this._publishCapability(params.module, capability.name, capability.state, params.instanceId);
		}
	}

});

ICTouchAPI.CapabilityServices = new ICTouchAPI.CapabilityServices();
/**
* @class ICTouchAPI.tools
* @singleton
* @extends Object
* Usefull class for managing Date/Time and other things...
*/
dojo.provide("ICTouchAPI.tools");
dojo.declare("ICTouchAPI.tools", null,
{
	/* --------------------------------- Public attributes ------------------------------------ */

	CALLDIRECTION_OUTGOING : "CALL_OUTGOING",
	CALLDIRECTION_INCOMING : "CALL_INCOMING",

	// List of pools
	POOL_SS			: "screensaver",
	POOL_BG			: "background",
	POOL_WU			: "wakeup",
	POOL_RM			: "ringingmelody",
	//POOL_AM			: "FileManagerAlarmMelodies",
	POOL_AM                 :"appointmentmelody",//the pool for notifications

	// Call log types enum
	enumComLogEntryType : {
		CallLog			: 0,
		CallBackRequest	: 1,
		VM				: 2,
		Conference		: 3,
		IM				: 4
	},
	
	// Contact types enum
	enumContactType : {
		// Local contact created by the user (manually or by copy of a search)
		LOCAL			: 0,
		// Remote contact
		UDA				: 1,
		// Remote contact
		BUDDY			: 2,
		// Local - Provided by setting "DirectoryDefaultContacts"
		DM				: 3,
		// Remote contact
		ICS				: 4,
		// Local - Automatically created when a Manager/assistant link is created
		MA				: 5,
		// Local - Automatically created when a comlog entry not matching a local contact is added. Those are not to be displayed in contacts/search.
		CACHED			: 6
	},

	// Date and time format settings
	strDateFormat		: "",
	boolTimeFormat		: false, // 12 \ 24

	intContactFirstNameOrder	: 1, // order "name + first name" by default, order should be defined by a setting in R250

	listDefaultAvatarTypes : {
			"user"			:"",
			"conference"	:"conf_",
			"im"			:"groupchat_",
			"video"			:"video_",
			"voicemail"		:"voicemail_"
	},
	listPresence : {
			"ACTIVITY_ONTHEPHONE" : "userinfo-onthephone",
			//"ACTIVITY_APPEAR_OFFLINE" : "userinfo-offline",
			"ACTIVITY_ONLINE": "userinfo-availablephone",
			"ACTIVITY_AWAY": "userinfo-away",
			"ACTIVITY_BUSY": "userinfo-busy"
	},
	listPresenceIM : {
			"ACTIVITY_ONTHEPHONE" : "userinfo-onthephone",
			"ACTIVITY_APPEAR_OFFLINE" : "userinfo-offline",
			"ACTIVITY_OFFLINE" : "userinfo-offline",
			"ACTIVITY_AWAY" : "userinfo-away",
			"ACTIVITY_LUNCH" : "userinfo-outtolunch",
			"ACTIVITY_OTHER" : "userinfo-offline",
			"ACTIVITY_UNKNOWN" : "userinfo-offline",
			"ACTIVITY_APPOINTMENT" : "userinfo-appointement",
			"ACTIVITY_ONLINE": "userinfo-available",
			"ACTIVITY_BUSY": "userinfo-busy",
			"ACTIVITY_BE_RIGHT_BACK" :"userinfo-berightback"
	},
	listDestinationsIcons: {
		"DST_ASSOCIATE" : "communication-routing-associate",
		"DST_ATTENDANT" : "communication-routing-assistant",
		"DST_MOBILE" : "communication-routing-mobile",
		"DST_VOICE_MAIL" : "communication-routing-voicemail",
		"Other" : "communication-routing-generic"
	},

	/* --------------------------------- Private attributes ----------------------------------- */



	/* ------------------------------------ Constructor --------------------------------------- */
	/**
         * @ignore
         */
        constructor : function() {
            // Default settings for simulation mode
            if(!generalConfig.simulation) {
		ICTouchAPI.settingServices.getSetting("TimeFormat", this, this.onTimeFormatChanged);
		ICTouchAPI.settingServices.getSetting("DateFormat", this, this.onDateFormatChanged);
		ICTouchAPI.settingServices.getSetting("ContactFirstNameOrder", this, this.onContactFirstNameOrderChanged);
		ICTouchAPI.settingServices.subscribeToSetting(this, "TimeFormat", this.onTimeFormatChanged);
		ICTouchAPI.settingServices.subscribeToSetting(this, "DateFormat", this.onDateFormatChanged);
		ICTouchAPI.settingServices.subscribeToSetting(this, "ContactFirstNameOrder", this.onContactFirstNameOrderChanged);
            }
	},


	/* ----------------------------------- Getter / Setter------------------------------------- */


	/* ----------------------------------- Public methods ------------------------------------- */

	/**
	 * Needed for settings and event services
         * @ignore
	 * @return {String} : the name of this : "tools"
	 */
	getApplicationName : function() {
		return "tools";
	},

    /**
	 * Transform the text of an emoIcon to the icon thaks to very useful regular expressions.
     *
	 * @return {String} : the path to the Icon
	 */
	textEmoIcon : function(text){
		var j;
		for( j=0;j<ICTouchAPI.keyboardServices.arrEmoIconConvert.length;j++){
			if(ICTouchAPI.keyboardServices.arrEmoIconConvert[j][1]){
				var reg = RegExp(ICTouchAPI.keyboardServices.arrEmoIconConvert[j][1], "g");
				text = text.replace(reg, "<img src="+ICTouchAPI.keyboardServices.arrEmoIconConvert[j][0]+">");
			}
		}
		return text;
    },

        /**
         * @ignore
         */
	onTimeFormatChanged : function(objSetting) {
		if (objSetting != null)
			this.boolTimeFormat = objSetting.jsValue;
	},

        /**
         * @ignore
         */
	getBoolTimeFormat : function() {
		return this.boolTimeFormat;
	},

        /**
         * @ignore
         */
	onDateFormatChanged : function(objSetting) {
		if (objSetting != null)
			this.strDateFormat = objSetting.allowedValues[objSetting.jsValue].name;
	},

        /**
         * @ignore
         */
	getDateFormat : function() {
		return this.strDateFormat;
	},

	/**
         * @ignore
         */
	onContactFirstNameOrderChanged : function(objSetting) {
		if (objSetting != null) {
			this.intContactFirstNameOrder = objSetting.jsValue;
		}
	},

	/**
	 * Return the name of a file without entire path
	 * @param {String} filePath The file path
         * @return {String} the name of the file
	 */
	basename : function(path) {
		return path.replace(/.*\//, "");
	},

	/**
	 * Return the html anchor text that follows a sharp #
	 * @param {String} path The path
         * @return {String} The anchor
	 */
	anchor : function(path) {
		var anchor = path.replace(/.*#/, "");
		if (anchor != path)
			return anchor;
		return "";  // no anchor
	},


	/**
         * Useful to translate some status to an another
         * @ignore
         */
	getPresenceLabel:function(strPresence){
		var strPres="";
		var listPresence = {
			"ACTIVITY_ONTHEPHONE" : "ACTIVITY_ONTHEPHONE",
			"ACTIVITY_APPEAR_OFFLINE" : "ACTIVITY_APPEAR_OFFLINE",
			"ACTIVITY_AWAY" : "ACTIVITY_AWAY",
			"ACTIVITY_LUNCH" : "ACTIVITY_LUNCH",
			"ACTIVITY_OTHER" : "ACTIVITY_APPEAR_OFFLINE",
			"ACTIVITY_UNKNOWN" : "ACTIVITY_APPEAR_OFFLINE",
			"ACTIVITY_APPOINTMENT" : "ACTIVITY_APPOINTMENT",
			"ACTIVITY_ONLINE"   : "ACTIVITY_ONLINE",
			"ACTIVITY_BUSY" : "ACTIVITY_BUSY",
			"ACTIVITY_BE_RIGHT_BACK":"ACTIVITY_BE_RIGHT_BACK"
		}
		strPres= listPresence[strPresence] || "ACTIVITY_APPEAR_OFFLINE";
		return strPres;

	},

	/**
	 * Get a default photo according to a type
	 * @ignore
         * @param {String} strType the type of the picture (user, conference, im, video, voicemail)
	 * @param {String} strSize The time minutes
	 * @return {String} The path of the image
	 */
	// Return the path of a defaultPhoto (user, conference, im, video, voicemail), with a defined size (50, 70, 100)
	getDefaultPhotoPath : function(strType, strSize) {
		var type = "";
		if(this.listDefaultAvatarTypes[strType]){
			type = this.listDefaultAvatarTypes[strType];
		}
		type += strSize;
		strType=null;
		strSize=null;
		return "http://localhost/library/ICTouchAPI/themes/Default/images/avatar_"+type+".png";
	},

        /**
         * @ignore
         */
	getTelephonyPresenceIconPath : function(strPresence, strSize) {
		var strPresenceIcon = this.listPresence[strPresence]||"userinfo-offline";
		if(strSize){
			strPresenceIcon = strPresenceIcon + '-' + strSize;
		}
		strPresence=null;
		strSize=null;
		return strPresenceIcon;
	},

	/**
	 * Get hours in 24 format from AM/PM format
	 * @param {Number} intHours in AM/PM format
	 * @param {Boolean} boolIsMorning, true if AM, false if PM
	 * @return {Number} The time in 24 format
	 */
	get24HourFrom12 : function(intHours, boolIsMorning){
		if(boolIsMorning){
			return parseInt(intHours)%12;
		}
		return parseInt(intHours) == 12 ? 12 : parseInt(intHours) + 12;
	},

	/**
	 * Get hours in AM/PM format from 24 format
	 * @param {Number} intHours in 24 format
	 * @return {Object} The time in AM/PM format
	 * intHours:(intHours + 11)%12 + 1,
	 * boolIsMorning:intHours - 12 >= 0 ? true : false,
	 */
	get12HourFrom24 : function(intHours){
		return{
			intHours:(parseInt(intHours) + 11)%12 + 1,
			boolIsMorning:parseInt(intHours) - 12 < 0 ? true : false
		}
	},

	/**
	 * Format a Time
	 * @param {Number} strHours: The time hours
	 * @param {Number} strMinutes: The time minutes
	 * @param {Boolean} bool24h True if the hours param format is 24h
	 * @return {String} The time in String format
	 */
	getFormatedTime : function (strHours, strMinutes, bool24h)
	{
		var intHours   = parseInt(strHours);
		var intMinutes = parseInt(strMinutes);

		var objDate = {};
		// create the time string
		var strTime = "";
		// 12 / 24 format
		if(bool24h)
		{

			objDate = new Date(0,0,0,intHours,intMinutes,0);
			strTime = objDate.format("HH:mm",dojo.locale);
		}
		else
		{

			if(intHours == 0) intHours = 24;
			objDate = new Date(0,0,0,intHours,intMinutes,0);
			strTime = objDate.format("hh:mm a",dojo.locale);
		}

		strHours=null;
		strMinutes=null;
		bool24h=null;
		return strTime;

	},

	/**
	 * Format a Time with the seconds
	 * @param {Number} strHours: The time hours
	 * @param {Number} strMinutes: The time minutes
	 * @param {Number} strSeconds: The time seconds
	 * @param {Boolean} bool24h True if the hours param format is 24h
	 * @return {String} The time in String format
	 */
	getFormatedTimeWithSeconds : function (strHours, strMinutes, strSeconds, bool24h)
	{


		var intHours   = parseInt(strHours);
		var intMinutes = parseInt(strMinutes);
		var intSeconds = parseInt(strSeconds);

		var objDate = {};
		// create the time string
		var strTime = "";
		// 12 / 24 format
		if(bool24h)
		{

			objDate = new Date(0,0,0,intHours,intMinutes,intSeconds);
			strTime = objDate.format("HH:mm:ss",dojo.locale);
		}
		else
		{

			if(intHours == 0) intHours = 24;
			objDate = new Date(0,0,0,intHours,intMinutes, intSeconds);
			strTime = objDate.format("hh:mm:ss a",dojo.locale);
		}

		return strTime;
	},

	/**
	 * Format a Date
	 * @param {Date} date The date
	 * @param {String} strFormat The date format to apply
	 * @return {String} The date in String format
         * @ignore
	 */
	getFormatedDate : function (date, strFormat)
	{
		// convert to dojoFormat
		//strFormat = strFormat.replace(/dd/, 'dd');
		strFormat = strFormat.replace(/mm/, 'MM');
		strFormat = strFormat.replace(/YYYY/, 'yyyy');

		var returnedDate = date.format(strFormat,dojo.locale);
		strFormat=null;
		date=null;
		return returnedDate;
	},

	/**
	 * Check the validity of an email address
	 * @param {String} strEmail The email address to check
	 * @return {Boolean} true if the email address is valid, false otherwise
	 */
	checkEmailAddress : function (strEmail)	{
		if(strEmail){
			var regex = /^[a-zA-Z0-9._%-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,4}$/;
			if (strEmail.match(regex)) {
				ICTouchAPI.debugServices.debug("ICTouchAPI.tools - checkEmailAddress / Email address is valid, return true");
				return true;
			}
			else {
				ICTouchAPI.debugServices.debug("ICTouchAPI.tools - checkEmailAddress / Email address is not valid, return false");
				return false;
			}
		}
		else {
			ICTouchAPI.debugServices.warning("ICTouchAPI.tools - checkEmailAddress / strEmail is not defined or empty, return false");
			return false;
		}
	},

	/**
	 * Convert the first character only of the String to upper case
	 * @param {String} str The text to convert
         * @return {String} The converted text
	 */
	ucfirst :function (str) {
		str += '';
		var f = str.charAt(0).toUpperCase();
		return f + str.substr(1);
	},


        /**
         * @ignore
         */
	onLoadedWebapp : function(strWebapp, context, callback) {
		if (ICTouchAPI.webApplicationManager.getRefApplication(strWebapp)) {
			var func = dojo.hitch(context, callback);
			func(strWebapp);
		}
		else {
			dojo.subscribe("webApplicationManager.notify.loaded." + strWebapp, this, function() {
				var func = dojo.hitch(context, callback);
				func(strWebapp);
			});
		}
	},

	/**
	 * Register a button on the HomePage (preview Homepage)
	 * @param {Object} args The button arguments contained:
         * <pre><code>
         * - {String} strWebapp The name of the Webapp
         * - {String} strButtonName The name of the button
         * - {String} strButtonIcon The name of the icon
         * - {String} strLabel The label of the button
         * - {String} strStatusIcon The status of the led (optional) can be 'notif-on, 'notif-off' to display the led or undefined to not have a led
         * </pre></code>
	 */
	registerHomepageButton : function(args) {

		/*var name = args[0];
		var reg = /^[a-zA-Z]+\.[a-zA-Z]+/;
		var webappName = reg.exec(name);
		ICTouchAPI.skinServices.linkWebappsStyles("webapp.homepage",webappName[0], false,false,true);*/
		if(!args[5]){
			args[5] = function(){
					ICTouchAPI.transitionServices.getScreen({
						name: args[0],
						params: {},
						position: args[6]
					});
			};
		}
		dojo.publish("HomepageRegisterButton", args);
		var handler = dojo.subscribe("HomepageLoaded", this, function() {
			dojo.publish("HomepageRegisterButton", args);
			dojo.unsubscribe(handler);
		});
	},

        /**
         * Unregister a button cotnained in the Homepage (preview Homepage)
         * @param {String} args The button name
         */
	unregisterHomepageButton : function(args) {
		dojo.publish("HomepageUnregisterButton", args);
	},

	/**
	 * Register a Container in the Home Page
	 * @param {Object} args A View containing at least a ContainerControl
	 */
	registerHomepageContainer : function(args) {
		dojo.publish("HomepageRegisterPreviewContainer", args);
		var handler = dojo.subscribe("HomepageLoaded", this, function() {
			dojo.publish("HomepageRegisterPreviewContainer", args);
			dojo.unsubscribe(handler);
		});
	},

	/**
	 * Unregister a Container in the Home Page
	 * @param {Object} args A View containing at least a ContainerControl
	 */
	unregisterHomepageContainer : function(args) {
		dojo.publish("HomepageUnregisterPreviewContainer", args);
	},

	/**
	 * Register a button on the HomePage (Button HomePage)
	 * @param {Object} args The button arguments contained:
         * <pre><code>
         * - {String} strWebapp The name of the Webapp
         * - {String} strButtonName The name of the button
         * - {String} strButtonIcon The name of the icon
         * - {String} strLabel The label of the button
         * - {String} strStatusIcon The status of the led (optional) can be 'notif-on, 'notif-off' to display the led or undefined to not have a led
         * </pre></code>
	 */
	registerHomepageKey : function(args) {
		/*var name = args[0];
		var reg = /^[a-zA-Z]+\.[a-zA-Z]+/;
		var webappName = reg.exec(name);
		ICTouchAPI.skinServices.linkWebappsStyles("webapp.homepage",webappName[0], false,false,true);*/
		if(!args[5]){
			args[5] = function(){
				ICTouchAPI.transitionServices.getScreen({
					name: args[0],
					params: {},
					position: args[6]
				});
			};
		}
		dojo.publish("HomepageRegisterKey", args);
		var handler = dojo.subscribe("HomepageLoaded", this, function() {
			dojo.publish("HomepageRegisterKey", args);
			dojo.unsubscribe(handler);
		});
	},

	/**
	 * Unregister a button contained in the Homepage (button Homepage)
	 * @param {String} args The button name
	 */
	unregisterHomepageKey : function(args) {
		dojo.publish("HomepageUnregisterKey", args);
	},

	/**
	 * Unregister a awap button contained in the Homepage (button Homepage)
	 * @param {String} args The button name
	 */
	unregisterHomepageAwapKey : function(args) {
		dojo.publish("HomepageUnregisterAwapKey", args);
	},

	/**
	 *  Take a width and a height checks their rate accordingly to the screen
	 *  size and return a set of proper dimensions
	 *  @param {Number} inWidth  width to be checked, in pixel
	 *  @param {Number} inHeight  height to be checked, in pixel
	 *  @return {Object} holds proper dimensions: top, left, width and height
         *  @ignore
	 */
	getRatedDimensions : function(inWidth,inHeight){
		var width, height, top, left;
		//ratio between the given width and the expected one
		var ratioW = inWidth/800;
		//ratio between the given height and the expected one
		var ratioH = inHeight/480;
		//if the picture is bigger than the screen
		if(inWidth>800 || inHeight>480) {
			//check the ratio between width and height
			var ratio = inWidth/inHeight;
			//if the width is the more stretched dimension
			if(ratioW >= ratioH) {
				width = 800; //set the new size
				ratioW = 1; //deifne the width as base of rated dimensions
				height = width/ratio; //compute the rated height
				ratioH = height/inHeight; //compute the new rate between coming height and new height
			}else { //else
				height = 480;
				ratioH = 1;
				width = height*ratio;
				ratioW = width/inWidth;
			}
		}
		else {
			//keep the given dimensions unchanged
			width = inWidth;
			height = inHeight;
		}
		//compute the new top and left positions
		top = (480-height)/2;
		left = (800-width)/2;

		return {
			'top' : top,
			'left' : left,
			'width' : width,
			'height' : height,
			'ratioW' : ratioW,
			'ratioH':ratioH
		};
	},

	/* ----------------------------------- Private methods ------------------------------------- */

	/**
	 * @ignore
	 */
	getChildren : function(type, id) {
		var node = dojo.query(type + "#" + id)[0];

		if (node != undefined) {
			var childrenIds = {};
			for (var i=0; i < node.childElementCount; i++) {
				childrenIds[i] = node.children[i].id;

			}

			return childrenIds;
		}
		else {
			return undefined;
		}
	},

	/**
	 * @ignore
	 */
	getClickableListTitle : function(iframeid, widgetname) {
		return dojo.query("li[class='ContainerTitleContent']", dojo.query("*[widgetname='" + widgetname + "']", dojo.query("iframe[id='" + iframeid + "']")[0].contentDocument)[0])[0];
	},

	/**
	 * @ignore
	 */
	getClickableListItems : function(iframeid, widgetname) {
		//return dojo.query("li[class~='Content']", dojo.query("*[widgetname='" + widgetname + "']", dojo.query("iframe[id='" + iframeid + "']")[0].contentDocument)[0]);
		return document.getElementById(iframeid).contentDocument.body.querySelectorAll("*[widgetname='"+ widgetname +"'] li[class~='Content']");

	},

	/**
	 * @ignore
	 */
	setRefreshable : function(objWidget) {
		ICTouchAPI.debugServices.warning("ICTouchAPI.tools - setRefreshable / Do not use setRefreshable anymore, it's broken");
		objWidget.refresh = function(params) {
			// Clone the domNode of the widget ( without it's children )
			var srcNodeRef = this.domNode.cloneNode(false);
			var nodeToClean = this.domNode;
			// Replace the fresh node with the old one
			dojo.place(srcNodeRef, this.domNode, "replace");

			// Restart the lifecycle of this UI
			this.destroy(true);
			if( params != undefined )
				this.create(params, srcNodeRef);
			else
				this.create({}, srcNodeRef);

			// Start background cleaning task
			setTimeout(function() {
				// Destroy every widget contained within this UI
				var list = dojo.query('[widgetId]', nodeToClean);
				ICTouchAPI.debugServices.debug("ICTouchAPI.tools - setRefreshable / Destroying "+list.length+" Widget while refreshing");
				list.forEach(function(domWidget){
					var objWidget = dijit.byNode(domWidget);
					// Don't destroy ourself
					if( objWidget != null && objWidget.id != nodeToClean.id) {
						objWidget.destroy();
					}
				});
				// Finish him ! ( destroy the old dom )
				dojo.destroy(nodeToClean);
			}, 0);
		};
	},

	/**
	 * @ignore
	 */
	getPresenceIconPath:function(strPresence, strSize) {
		var strPresenceIcon= this.listPresenceIM[strPresence]||"userinfo-invisible";
		if(strSize){
			strPresenceIcon=strPresenceIcon+"-"+strSize;
		}
		strPresence=null;
		strSize=null;
		return(strPresenceIcon);

	},



	/**
	 * Change the status icon of a button contained into the Homepage
	 * @param {Object} args An object containing:
	 * <pre><code>
	 * - {String} The name of the button
	 * - {String } The new status of the led ('notif-on' or 'notif-off'
	 * </code></pre>
	 */
	changeHomepageStatusIcon : function (args) {
		dojo.publish("HomepageChangeStatusIcon", args);
		var handler = dojo.subscribe("HomepageLoaded", this, function() {
			dojo.publish("HomepageChangeStatusIcon", args);
			dojo.unsubscribe(handler);
		});
	},

	/**
	 * @ignore
	 */
	getContactPhone : function(objContact, boolCanonical) {
		var phoneNumber = "";
		// Find contact number.
		if (objContact.officePhone) {
			phoneNumber = objContact.officePhone;
		}
		else if (objContact.personalHome) {
			phoneNumber = objContact.personalHome;
		}
		else if (objContact.personalMobile) {
			phoneNumber = objContact.personalMobile;
		}
		else if (objContact.personalFax) {
			phoneNumber = objContact.personalFax;
		}
		if(typeof phoneNumber == "object"){
			if(boolCanonical){
				phoneNumber = (phoneNumber.value) ? phoneNumber.value.canonicalNumber : phoneNumber.canonicalNumber;
			} else {
				phoneNumber = (phoneNumber.value) ? phoneNumber.value.number : phoneNumber.number;
			}
		}

		objContact=null;
		return phoneNumber;
	},

	/**
	 * @ignore
	 */
	getNumbersFromContact : function(objContact) {
		var _arrNumbers = [];
		// Find contact number.

		if (objContact.officePhone) {
			_arrNumbers.push(objContact.officePhone);
		}
		if (objContact.personalHome) {
			_arrNumbers.push(objContact.personalHome);
		}
		if (objContact.personalMobile) {
			_arrNumbers.push(objContact.personalMobile);
		}
		if (objContact.officePhoneCan) {
			_arrNumbers.push(objContact.officePhoneCan);
		}
		if (objContact.personalHomeCan) {
			_arrNumbers.push(objContact.personalHomeCan);
		}
		if (objContact.personalMobileCan) {
			_arrNumbers.push(objContact.personalMobileCan);
		}
		objContact=null;
		return _arrNumbers;
	},


	/**
	 * @ignore
	 */
	getContactPresentationList : function(objContact, context, callback, strI18nRef, callIcon, boolOnlyPhoneNumbers) {
		var list = [];
		var item;
		var arrLabels = this.getContactFieldLabels();
		var _arrNumbersFromActiveCall = webapp.communication ? webapp.communication.data.getNumbersFromActiveCall() : [];
		var itemIcon = callIcon ? callIcon : "communication-call";
		if(objContact){
			for (var i in arrLabels) {
				if ( ((boolOnlyPhoneNumbers && (arrLabels[i].type == "Phone" || arrLabels[i].type == "otherPhone")) || (!boolOnlyPhoneNumbers && arrLabels[i].type != "Header")) && (objContact[arrLabels[i].id] || (objContact.miscMap && objContact.miscMap[arrLabels[i].id]))) {
					item = {
						strId: arrLabels[i].id,
						strLabel: _(arrLabels[i].label, strI18nRef),
						strContent: (objContact[arrLabels[i].id]) ? (objContact[arrLabels[i].id]) : objContact.miscMap[arrLabels[i].id],
						strType: arrLabels[i].type
                        //strIcon:(arrLabels[i].type == "Phone") ? 'communication-call' : null
					};

					if(arrLabels[i].type == "Phone" || arrLabels[i].type == "otherPhone"){

						if(typeof objContact[arrLabels[i].id] == "object"){
							item.strContent = (objContact[arrLabels[i].id].value)?objContact[arrLabels[i].id].value.number : objContact[arrLabels[i].id].number;
						}
                                                if (item.strContent === "anonymous" || item.strContent === "***") {
                                                    item.strContent = _("anonymous","ICTouchAPI");
                                                    item.callback = null;
                                                    item.strIcon = null;
                                                }
                                                else {
						var boolFind = false;
						for(var j = 0 ; j < _arrNumbersFromActiveCall.length ; j++)
						{
							if(_arrNumbersFromActiveCall[j]==objContact[arrLabels[i].id] || _arrNumbersFromActiveCall[j]==objContact[arrLabels[i].id+"Can"]){
								boolFind = true;
								item.callback = null;
								item.strIcon = null;
								item.strHighlight = "highlight"
								break;
							}
						}
						if(!boolFind){
							item.callback = dojo.hitch(context, callback, item);
							item.strIcon = itemIcon;
						}
                                                }
					}
                                        else{
						item.callback = dojo.hitch(context, callback, item);
						item.strIcon = null;
					}

					list.push(item);
				}
			}
		}
		objContact=null;
		context=null;
		callback=null;
		strI18nRef=null;
		return list;
	},

	/**
	 * @ignore
	 */
	getContactFieldLabels : function() {
		var _arrLabels = [];
		_arrLabels['name']				= {id: 'name', label: "Last name", type: "Header", misc: false};
		_arrLabels['firstName']			= {id: 'firstName', label: "First name", type: "Header", misc: false};
		_arrLabels['personalFax']		= {id: 'personalFax', label: "Extension number", type: "otherPhone", misc: false};
		_arrLabels['officePhone']		= {id: 'officePhone', label: "Phone", type: "Phone", misc: false};
		_arrLabels['personalMobile']	= {id: 'personalMobile', label: "Mobile phone", type: "Phone", misc: false};
		_arrLabels['assistant']			= {id: 'assistant', label: "Other phone", type: "Phone", misc: false};
		_arrLabels['collaborationId']			= {id: 'collaborationId', label: "IM", type: "IM", misc: false};
		_arrLabels['email']				= {id: 'email', label: "E-mail", type: "Email", misc: false};
		_arrLabels['fax']				= {id: 'fax', label: "Fax", type: "Fax", misc: false};
		_arrLabels['personalHome']		= {id: 'personalHome', label: "Private phone", type: "Phone", misc: false};
		_arrLabels['personalEmail']		= {id: 'personalEmail', label: "Private e-mail", type: "Email", misc: false};
		_arrLabels['photo']				= {id: 'photo', label: "Picture", type: "Header", misc: false};
		_arrLabels['company']			= {id: 'company', label: "Company", type: "Header", misc: true};
		_arrLabels['job']				= {id: 'job', label: "Job title", type: "Header", misc: true};
		_arrLabels['address']			= {id: 'address', label: "Address", type: "Address", misc: true};
		_arrLabels['zip']				= {id: 'zip', label: "Zip", type: "Zip", misc: true};
		_arrLabels['city']				= {id: 'city', label: "City", type: "Header", misc: true};
		_arrLabels['country']			= {id: 'country', label: "Country", type: "Header", misc: true};
		return _arrLabels;
	},

	/**
	 * Return the display name of the contact according to the following rules:
	 * - if the objContact has a firstName and/or a name, compute the display name according to the order
	 *   defined by the boolean _boolLastNameFirstNameOrder (should be linked to a setting to add in R250):
	 *   - if _boolLastNameFirstNameOrder is true, return the display name in the order "lastName + firstName"
	 *     with a space between the lastName and firstName
	 *   - if _boolLastNameFirstNameOrder is false, return the display name in the order "firstName + lastName"
	 *     with a space between the firstName and lastName
	 * - else if the objContact has a phoneNumber, return the phoneNumber as display name
	 * - else, return empty string ""
	 *
	 * @param {Object}   the Contact object used to compute the display name
	 * @return {String} Returns the display name of the contact.
	 * @ignore
	 */
	getContactDisplayName : function(objContact) {
		var displayNameLabel = "";
		if (objContact){
			var name = (objContact.name)?objContact.name:"";
			var firstName = (objContact.firstName)?objContact.firstName:"";
			if (this.intContactFirstNameOrder == 0) {
				firstName = (firstName) ? firstName + " " : "";
				displayNameLabel = firstName + name;
			}
			else {
				name = (name) ? name + " " : "";
				displayNameLabel = name + firstName;
			}
			var phoneNumber = this.getContactPhone(objContact);
			if (!displayNameLabel && phoneNumber) {
				displayNameLabel = phoneNumber;
			}
		}
		return displayNameLabel;
	},

	/**
	 * @ignore
	 */
	isFirstNameFirstDisplayed : function() {
		return (this.intContactFirstNameOrder == 0);
	},

	/**
	 * @ignore
	 */
	getLogDataFromStatus : function(objLogData,strSize) {
		// default nothing
		var obj = {
			strIcon : "",
			strNotif : ""
		};

		// switch on datas entryType then other attribute
		switch (objLogData.entryType)
		{
			// incoming, outgoing, missed
			case this.enumComLogEntryType.CallLog:
				// we needs to switch on callReason

				switch(objLogData.callDirection.value.value)
				{
					case this.CALLDIRECTION_OUTGOING:
						if(objLogData.isAnswered == 0)
						{
							// out no answer (= red)
							obj.strIcon = "communication-log-outgoing-call-no-answer";
						}
						else
						{
							// out ok (= blue)
							obj.strIcon = "communication-log-outgoing-call";
						}
						break;
					case this.CALLDIRECTION_INCOMING:
						// is it a missed call ?
						if(objLogData.isAnswered == 0)
						{
							//missed  (incoming not answered = red)
							obj.strIcon = "communication-log-missed-call";
							obj.strNotif = "Missed";
						}
						else
						{
							// incoming answered (=blue)
							obj.strIcon = "communication-log-incoming-call";
						}
						break;
				}
				break;
			// callback request
			case this.enumComLogEntryType.CallBackRequest:
				obj.strIcon = "communication-log-call-me-back";
				obj.strNotif = "Callback";
				break;
			// VM
			case this.enumComLogEntryType.VM:
				if(objLogData.entryPriority == 2){
					obj.strIcon = "communication-log-log-voicemail-priority";
				} else {
					obj.strIcon = "communication-log-voicemail";
				}
				obj.strNotif = "Voicemail";
				break;
			// Conference
			case this.enumComLogEntryType.Conference:
				obj.strIcon = "communication-log-conference-finish";
				break;
			// IM
			case this.enumComLogEntryType.IM:
				obj.strIcon = "communication-log-instant-messaging";
				break;
		}
		if(strSize){
			strSize="-"+strSize;
			obj.strIcon+=strSize;

		}
		objLogData=null;
		return obj;
	},

	/**
	 * @ignore
	 */
	formatRoutingLabel : function(from, dest, number, boolOther, noI18nize) {
		var arrLabel = [];
		arrLabel.push(from);
		arrLabel.push("to");
		if(boolOther){
			// do not translate phonenumbers in case 'Other'
			arrLabel.push(number);
		}else{
			arrLabel.push(dest);
		}
		
		if(noI18nize){
			return arrLabel;
		} else {
			return _(arrLabel, "webapp.userservices");
		}
	},
	
	/**
	 * @ignore
	 */
	getRoutingIcon: function(dest) {
		var icon = this.listDestinationsIcons[dest];
		if (typeof icon !== "string") {
			icon = this.listDestinationsIcons["Other"];
		}
		return icon + "-32";
	},

	/**
	 * @ignore
	 */
	getObjectReference : function(objRefStr) {
		var arr = objRefStr.split('.');

		var myObj = window;
		var length = arr.length;
		for (var i=0; i<length; i++) {
			myObj = myObj[arr[i]];
			if(!myObj){
				//this is done so we will not ask for the attribute of an undefined object
				return false;
			}
		}

		return myObj;
	},

	/**
	 * @ignore
	 */
	getEventArguments : function(args) {
		var arr = {};
		if (args && args.length) {
			var len = args.length;
			for (var i = 0; i < len; i++) {
				arr[args[i].name] = args[i].value;
			}
		}
		args=null;
		return arr;
	},

	/**
	 * check recursively if all the fields of the given objects are equal
	 * @param {Object} obj1 The first object to compare
	 * @param {Object} obj2 The second object to compare
	 * @return {Boolean} true if the objects are identical, false otherwise
	 */
	compareObjects : function(obj1, obj2){
        var isIdentical = true;
		if (obj1 && obj2) {
		for (var i in obj1) {
			if (obj1[i] !== obj2[i]) {
				if ((typeof obj1[i] === "object") && (typeof obj2[i] === "object" )) {
					isIdentical = this.compareObjects(obj1[i], obj2[i]);
					if (!isIdentical) {
						break;
					}
				}
					else if (!obj1[i] && !obj2[i]){ // if one obj[i] is null and the other obj[i] is "" continue
						continue;
					}
				else {
					isIdentical = false;
					break;
					}
				}
			}
		}
		else if (obj1 != obj2) {
			isIdentical = false;
		}

		obj1 = null;
		obj2 = null;
		return isIdentical;
	},

	/**
	 * This function remove any text node that contains only whitespace that is a direct child of domList
	 * The purpose is to allow "display: inline-block" without any CSS hack while keeping a nice template
	 * @ignore
         * @param {HTMLElement} domList
	 */
	removeWhitespaceNodes : function(domList) {
		// Can't use dojo.query because it ignores text nodes
		var list = domList.childNodes;
		// Match when the string is only composed of whitespaces or empty
		var regex = /^\s*$/;
		for(var i=0; i<list.length; ++i) {
			var item = list[i];
			if( item.nodeType == domList.TEXT_NODE && item.nodeValue.match(regex) != null ) {
				domList.removeChild(item);
			}
		}
	},

	/**
	 * Destroy the widget by givin it's root node
         * @ignore
	 * @param {HTMLElement} domWidget of the widget to destroy
	 * @return {Number} number of widget destroyed
	 */
	destroyWidgetByNode : function(domWidget) {
		var objWidget = dijit.byNode(domWidget);
		// Destroy it if it's really a widget
		if( objWidget != null ) {

			objWidget.destroy();
			return 1;
		}
		return 0;
	},

	/**
	 * Destroy every widget contained within this domElement
	 * then destroy the domElement itself
         * @ignore
	 * @param {HTMLElement} domElement is the dom to wipe out
	 * @param {Boolean} boolAvoidRoot if there is a widget at the root of domElement, should it be kept
	 */
	destroyWidgets : function(domElement, boolAvoidRoot) {


		// Get the list of elements to clean
		var list = dojo.query("[widgetid]", domElement);

		var nbr = 0;
		// Start the tear down
		list.forEach(function(element){
			nbr += ICTouchAPI.tools.destroyWidgetByNode(element);
		});
		if( !boolAvoidRoot ) {
			nbr += ICTouchAPI.tools.destroyWidgetByNode(domElement);
		}

		ICTouchAPI.debugServices.debug("ICTouchAPI.tools - destroyWidgets / Destroying "+nbr+" Widget while refreshing");

		// Finish him ! ( destroy the old dom )
		dojo.destroy(domElement);

	},

	/**
	 * Retrieve frame's id from a node
         * @ignore
	 * @param {HTMLElement} domSearch search the frame's id attached to this node
	 * @return {String} return the id or null if the node isn't attached to any frame
	 */
	getFrameId : function(domSearch) {
		// Search every node for the attribute frameid until we reach the root
		while(domSearch && domSearch.getAttribute) {
			var strFrameId = domSearch.getAttribute("frameid");
			if( typeof strFrameId == "string" ) {
				return strFrameId;
			}
			domSearch = domSearch.parentNode;
		}
		return null;
	},

	/**
	 * Convert callback object to dojo.hitch. If callback is function, callback is directly returned.
	 * @ignore
         * @param {Object} callback Object containing :
	 *				- context : context of the webapp
	 *				- func : callback function
	 *				- param : optionnal parameter for callback. Can be String, Number or Boolean
	 * @return {Function} dojo.hitch function.
	 */
	callbackToHitch : function(callback) {
		if(callback!=null) {
			if (typeof(callback)==="function") {
				return callback;
			}
			else {
				if (callback.context && callback.func) {
					if (callback.param) {
						//if (typeof(callback.param)==="string" || typeof(callback.param)==="number" || typeof(callback.param)==="boolean") {
							return dojo.hitch(callback.context, callback.func, callback.param);
						//}
						//else {
						//	throw "incorrect param";
						//}
					}
					else {
						return dojo.hitch(callback.context, callback.func);
					}
				}
			}
		}
		return null;
	}
});

ICTouchAPI.tools = new ICTouchAPI.tools();
/**
 * @class ICTouchAPI.feedbackServices
 * @singleton
 * @extends Object
 * This service offers the possibility to attach a Feedback to each graphical element of the Phone
 */
dojo.provide("ICTouchAPI.feedbackServices");
dojo.declare("ICTouchAPI.feedbackServices",null,
{
/* --------------------------------- Public attributes ------------------------------------ */
    /**
    * SAFETY_DELAY
    * @property
    * @type Number
    * @ignore
    */
    SAFETY_DELAY: 80, // minimum delay between two haptic calls, in ms

    /**
    * senskeyServices
    * @property
    * @type Object
    * @ignore
    */
    senskeyServices : null,//senskey services

/* --------------------------------- Private attributes ------------------------------------ */
    /**
    * @property
    * @type Boolean
    * @ignore
    */
    _feedbackState: true,	// True : feedback activated

    /**
    * @property
    * @type Boolean
    * @ignore
    */
    _hapticState: true,			// True : Haptic activated

    /**
    * @property
    * @type Boolean
    * @ignore
    */
    _beepState: true,			// True : Beep activated

    /**
    * @property
    * @type Boolean
    * @ignore
    */
    _safetyLock: false, // True : ignore haptic effect, because previous haptic call was too recent

    /**
    * @property
    * @type Object
    * @ignore
    * List of available effects
    * Mapping beetween effects and the associated integer expected by IctWebClient
    */
    _feedbackEffectsValue: {
        "StrongClick"		: 0,
        "SharpClick"		: 1,
        "SoftBump"			: 2,
        "DoubleClick"		: 3,
        "TripleClick100"	: 4,
        "SoftBuzz100"		: 5,
        "StrongBuzz100"		: 6
    },

    /**
    * @property
    * @type String
    * @ignore
    */
	_defaultEffect : "SharpClick", // Default effect

    /**
    * @property
    * @type Object
    * @ignore
    * List of available feedbacks
    * Only haptic effects are implemented at the moment.
    */
    _feedbackTypes: {
        "endScroll" : {
            hapticMethod : "startEffect",
            hapticEffect : "StrongBuzz"
        },
        "keystroke" : {
            hapticMethod : "click",
            hapticEffect : "SoftBuzz"
        },
        "click"				: {
            hapticMethod : "click",
            hapticEffect : 	"SharpClick"
        }
    },

    /**
    * @property
    * @type String
    * @ignore
    */
    _defaultType: "click",// Default feedback type

/* ------------------------------------ Constructor --------------------------------------- */
    /**
     * @ignore
     */
    constructor: function () {
        //this._hapticState = (window.Haptic != undefined);

        // Default settings for simulation mode
        if(!generalConfig.simulation) {
            ICTouchAPI.settingServices.getSettingValue("HapticEnabled", this, this._hapticValueReceived);
            ICTouchAPI.settingServices.getSettingValue("BeepEnabled", this, this._beepValueReceived);

            ICTouchAPI.settingServices.subscribeToSetting(this, "HapticEnabled", this._hapticValueChanged);
            ICTouchAPI.settingServices.subscribeToSetting(this, "BeepEnabled", this._beepValueChanged);
        }
    },

/* ------------------------------------ getter/setter --------------------------------------- */
    /**
    * @ignore
    */
    getApplicationName: function() {
        return "feedbackServices";
    },

    /**
    * Get the state of the Feedback
    * @return {Boolean} True if the Feedback is enabled
    */
    getFeedbackState: function () {
        return this._feedbackState;
    },

    /**
    * Get the state of the haptic feedback
    * @return {Boolean} True if the haptic feedback is enabled
    */
    getHapticState: function () {
        return this._hapticState;
    },

    /**
    * Get the state of the beep feedback
    * @return {Boolean} True if the beep feedback is enabled
    */
    getBeepState: function () {
        return this._beepState;
    },

/* ------------------------------------ Public Methods --------------------------------------- */

    /**
     * Enable a feedback.
     */
    enable: function() {
        this._feedbackState = true;
    },

    /**
     * Disable the feedback.
     */
    disable: function () {
        this._feedbackState = false;
    },

    /**
     * Create a new Feedback effect to apply on a Graphical Element
     * @param {String} feedbackType The feedback's type
     * @return {Object} with the following methods :
     *		feedback(clickEvent)
     *		stop()
     */
    initFeedback: function (feedbackType) {
        var that = this;

        /**
        * @class ICTouchAPI.feedback
        * @namespace ICTouchAPI
        * @ignore
        * @extends Object
        * A Effect which can have one or several action (haptic, beep...)
        */
        return new function feedback() {

            /**
             * Start the feedback effect
             * @param {Event} clickEvent The associated Event (DOM Event)
             */
            this.feedback = function feedback(clickEvent) {
                if (that.getFeedbackState()) {
                    if ((that.getHapticState() || that.getBeepState()) && !ICTouchAPI.feedbackServices._safetyLock) {
                        ICTouchAPI.feedbackServices._safetyLock = true;
                        setTimeout(function() {
                            ICTouchAPI.feedbackServices._safetyLock = false;
                        }, ICTouchAPI.feedbackServices.SAFETY_DELAY);
                        var type = that._getFeedbackType(feedbackType);
                        if (that.getHapticState() && window.Haptic != undefined) {
							// Haptic.testClick() is used to test different Haptic effects through the whole software
							if (Haptic.testClick != undefined && webapp.haptic != undefined) {
								Haptic.testClick(webapp.haptic.data.intChoiceListSelected, clickEvent.clientX, clickEvent.clientY)
							}
							else {
								Haptic[type["hapticMethod"]](
									that._getFeedbackEffectValue(type["hapticEffect"]), clickEvent.clientX, clickEvent.clientY);
							}
                        }
                        if(that.getBeepState() && window.LowLevelBeep != undefined && LowLevelBeep.click != undefined) {
                            LowLevelBeep.click(0);
                        }
                    }
                }
            }

            /**
             * Stop the feedback effect
             */
            this.stop = function stop() {
                if (that.getFeedbackState()) {
                    if (that.getHapticState()) {
                        if (Haptic != undefined) {
                            Haptic.stopEffect();
                        }
                    }
//                    if (that.getBeepState()) {
//                        if (LowLevelBeep != undefined) {
//                            LowLevelBeep.stopEffect();
//                        }
//                    }
                }
            }
        }
    },

/* ------------------------------------ Private Methods --------------------------------------- */
    /**
     * @ignore
     * change the haptic state
     */
    _hapticValueReceived: function(hapticValue) {
        if (hapticValue != undefined) {
            this._hapticState = hapticValue;
        }
    },

    /**
     * @ignore
     * change the beep state
     */
     _beepValueReceived: function(beepValue) {
        if (beepValue != undefined) {
            this._beepState = beepValue;
        }
    },

    /**
     * @ignore
     * change the haptic state
     */
    _hapticValueChanged: function(hapticSetting) {
        this._hapticState = hapticSetting.jsValue;
    },

     /**
      * @ignore
      * change the beep state
      */
     _beepValueChanged: function(beepSetting) {
        this._beepState = beepSetting.jsValue;
    },

    /**
     * @param {String} feedback : the feedback's name to get the parameters.
     * @ignore
     */
    _getFeedbackType: function (feedback) {
        return (this._feedbackTypes[feedback] || this._feedbackTypes[this._defaultType]);
    },

    /**
     * @param {String} effect
	 * @return: the feedback's integer expected by IctWebClient associated to the given effect.
     * @ignore
     */
    _getFeedbackEffectValue: function (effect) {
        return (this._feedbackEffectsValue[effect] || this._feedbackEffectsValue[this._defaultEffect]);
    }

});

ICTouchAPI.feedbackServices = new ICTouchAPI.feedbackServices();
/**
* @class ICTouchAPI.webApplicationManager
* @singleton
* @ignore
* @extends Object
*/
dojo.provide("ICTouchAPI.navigationServices");
dojo.declare("ICTouchAPI.navigationServices",
	null,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		focusedView: null,

		/* --------------------------------- Private attributes ----------------------------------- */

		/* ------------------------------------ Constructor --------------------------------------- */

		constructor: function() {
			// If it's not 8052 don't start services
			if( generalConfig.version != "8052" )
				return;
			ICTouchAPI.eventServices.subscribeToEvent(this, "SENSEKEY_UP", this.eventUp);
			ICTouchAPI.eventServices.subscribeToEvent(this, "SENSEKEY_DOWN", this.eventDown);
			ICTouchAPI.eventServices.subscribeToEvent(this, "SENSEKEY_LEFT", this.eventLeft);
			ICTouchAPI.eventServices.subscribeToEvent(this, "SENSEKEY_RIGHT", this.eventRight);
			ICTouchAPI.eventServices.subscribeToEvent(this, "SENSEKEY_OK", this.eventOk);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		getApplicationName: function()
		{
			return "navigationServices";
		},

		getCurrentFocus: function()
		{
			return this.focusedView;
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		changeFocus: function(view)
		{
			this.focusedView = view;
		},

		navigate: function(eventName)
		{
			if( this.focusedView && this.focusedView.navigate )
				this.focusedView.navigate(eventName);
		},

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 */
		eventUp:	function() {
			this.navigate("UP");
		},
		/**
		 * @ignore
		 */
		eventDown:	function() {
			this.navigate("DOWN");
		},
		/**
		 * @ignore
		 */
		eventLeft:	function() {
			this.navigate("LEFT");
		},
		/**
		 * @ignore
		 */
		eventRight:	function() {
			this.navigate("RIGHT");
		},
		/**
		 * @ignore
		 */
		eventOk:	function() {
			this.navigate("OK");
		},
	});

// Create navigation services
ICTouchAPI.navigationServices=new ICTouchAPI.navigationServices();
/**
* @class ICTouchAPI.IMEServices
* @singleton
* @extends Object
* @ignore
* IME management
*/
dojo.provide("ICTouchAPI.IMEServices");
dojo.declare("ICTouchAPI.IMEServices",null,

	{
		/**
         * @private
         */
        _strLang: "", // "Hangul", "pinyin", "zhuyin"...

        /**
         * @private
         */
		_charSequence		: "",

        /**
         * @private
         */
        _maxCharSequenceLength: 6,
                /**
                 * @private
                 */
                _boolCandidateMode	: true, // true -> candidate mode, false -> wordAssociation

                /**
                 * @private
                 */
                lstIdSession		: {},

        /**
         * @private
         */
        imeSessionLang: {
//            Chinese IME
            'pinyin': 0,

//            Japanese IME
            'Katakana': 10,
            'Hiragana': 11,

//            Korea IME
            'Hangul': 20

        },

        /**
         * @private
         */
		constructor : function() {
		},

		/**
		 * Ime i18n translate function
		 * @private
		 */
		ime_ : function(toTranslate) {
			return _(toTranslate, "ICTouchAPI", [], "ime");
		},

        /**
		 * @private
		 */
		getStrLang : function() {
			return this._strLang;
		},

        /**
         * @private
         */
        UTF82Unicode: function Utf82Unicode(strUtf8) {
            var bstr = "";
            var nTotalChars = strUtf8.length; // total chars to be processed.
            var nOffset = 0; // processing point on strUtf8
            var nRemainingBytes = nTotalChars; // how many bytes left to be converted
            var nOutputPosition = 0;
            var iCode, iCode1, iCode2; // the value of the unicode.
            while (nOffset < nTotalChars) {
                iCode = strUtf8.charCodeAt(nOffset);
                if ((iCode & 0x80) == 0) // 1 byte.
                {
                    if (nRemainingBytes < 1) // not enough data
                        break;
                    bstr += String.fromCharCode(iCode & 0x7F);
                    nOffset++;
                    nRemainingBytes -= 1;
                }
                else if ((iCode & 0xE0) == 0xC0) // 2 bytes
                {
                    iCode1 = strUtf8.charCodeAt(nOffset + 1);
                    if (nRemainingBytes < 2 || // not enough data
                        (iCode1 & 0xC0) != 0x80) // invalid pattern
                    {
                        break;
                    }
                    bstr += String
                        .fromCharCode(((iCode & 0x3F) << 6) | (iCode1 & 0x3F));
                    nOffset += 2;
                    nRemainingBytes -= 2;
                } else if ((iCode & 0xF0) == 0xE0) // 3 bytes
                {
                    iCode1 = strUtf8.charCodeAt(nOffset + 1);
                    iCode2 = strUtf8.charCodeAt(nOffset + 2);
                    if (nRemainingBytes < 3 || // not enough data
                        (iCode1 & 0xC0) != 0x80 || // invalid pattern
                        (iCode2 & 0xC0) != 0x80) {
                        break;
                    }
                    bstr += String.fromCharCode(((iCode & 0x0F) << 12)
                        | ((iCode1 & 0x3F) << 6) | (iCode2 & 0x3F));
                    nOffset += 3;
                    nRemainingBytes -= 3;
                } else
                // 4 or more bytes -- unsupported
                    break;
            }
            if (nRemainingBytes != 0) { // bad UTF8 string.
                return "";
            }
            return bstr;
        },

		/**
		 * Change the language
		 * @param {String} language the new language to used. Can be "Hangul", "pinyin" or "zhuyin"
		 */
		changeLanguage : function(strLang) {
			this._strLang = strLang;
            var shouldStartIME = ICTouchAPI.keyboardServices.isAPACIMELang(strLang) && 'apac_en' != strLang;
            if (shouldStartIME) {
//				if(!this.lstIdSession[strLang])
                /* Int corresponding to language */
                ICTouchAPI.APIServices.IMEManager.startIMESession({params: [this.imeSessionLang[strLang]], context: this, callback: this.startedIMESession, callbackParams: {strLang: strLang}});
                this.resetCharSequence();
            }
            ICTouchAPI.keyboardServices.setIMERequirement(shouldStartIME);
        },
        /**
         * @private
         * */
        stoppedIMESession: function (result, params) {
            if (result && params.strLang) {
                this.lstIdSession[params.strLang] = null;
            } else {
                ICTouchAPI.popupServices.errorPopup(this.ime_("ERROR_IME"), this.ime_("ERROR_STOP_IME_SESSION"), this.ime_("Ok"));
            }
        },

		/**
         * @private
         */
		startedIMESession : function(idSession, params) {
			if(idSession != "NULL")
				this.lstIdSession[params.strLang] = idSession;
			else
				ICTouchAPI.popupServices.errorPopup(this.ime_("ERROR_IME"), this.ime_("ERROR_START_IME_SESSION"), this.ime_("Ok"));
		},

		/**
		 * Call core function getIMEWordAssociation
		 * @private
		 */
		getIMEWordAssociation : function(strWord) {
			this._boolCandidateMode = false;
			ICTouchAPI.APIServices.IMEManager.getIMEWordAssociation({params :[this.lstIdSession[this.getStrLang()], strWord], context :this, callback:this._gotIMEWordAssociation});
		},

		/**
         * Callback of getIMEWordAssociation
         * @private
         */
        _gotIMEWordAssociation: function (arrWordResults) {
            //Hi Soft
            if (!arrWordResults) return;
            var arrCandidates = arrWordResults.candidates;
            ICTouchAPI.keyboardServices.displayCandidates(arrCandidates);
        },

		/**
		 * Call core function getIMECandidates
		 * @private
		 */
		getIMECandidates : function() {
			this._boolCandidateMode = true;
			ICTouchAPI.APIServices.IMEManager.getIMECandidates({params :[this.lstIdSession[this.getStrLang()], this.getCharSequence()], context :this, callback:this._gotIMECandidates});
		},

		/**
         * Callback of getIMECandidates
         * @private
         */
		_gotIMECandidates : function(IMEResults) {
            if (!IMEResults) return;
			var arrCandidates = IMEResults.candidates;
            ICTouchAPI.debugServices.debug("candidates: " + arrCandidates);
            /**
             *  Japanese:
             *      Katakana
             *      Hiragana
             *
             *  Korean: Hangul
             *  Chinese: pinyin
             * */
            switch (this.getStrLang()) {
                case "pinyin" : // Pinyin : we just display it in the candidateList
                    ICTouchAPI.keyboardServices.displayCandidates(arrCandidates);
                    break;
                case "Hangul" : // Hangul : If several candidates, display the last in IME input
                    // Then validate the others and delete letters corresponding to them
                    if (arrCandidates.length > 1) {
                        // push the first one and replace candidate with the last character in arrCandidates
                        //1.push the characters in input zone into input field
                        ICTouchAPI.keyboardServices.pushInputIntoField();
                        //2. update _charSequence to the last character
                        var lastChar = this.getCharSequence()[this.getCharSequenceLength() - 1];
                        this.setCharSequence(lastChar, true);
                    } else { // If 1 candidate, just display it instead of the IME input
                        ICTouchAPI.keyboardServices.replaceChar(arrCandidates, true);
                    }
                    break;

                case "Katakana":
                case "Hiragana":
                    if (arrCandidates.length >= 1) {
                        ICTouchAPI.keyboardServices.displayCharInInputField(arrCandidates, true);
                        this.resetCharSequence();
                    } else if (this.getCharSequenceLength() > 0) {
                        /**
                         *
                         * */
//                        var lastChar = this.getCharSequence()[this.getCharSequenceLength() - 1];
//                        console.log("last Char=", lastChar);
//                        ICTouchAPI.keyboardServices.displayChar(lastChar, true);
                        ICTouchAPI.keyboardServices.displayIMEString(this.getCharSequence());
                    }
                    break;
                default :
                    break;
            }
        },

		/**
		 * Not for Hangul
		 * Called by keyboardService when we select a candidate
		 * @private
		 */
		candidateSelected : function(UTF8Char) {
			if(this._boolCandidateMode) { // Result of a getIMECandidate
				// We display the word and launch getIMEWordAssociation
				ICTouchAPI.keyboardServices.displayChar(UTF8Char, false);
				this.getIMEWordAssociation(UTF8Char);
			} else { // Result of a getIMEWordAssociation
				// We replace the last word by the associated word and relaunch getIMEWordAssociation
				ICTouchAPI.keyboardServices.replaceChar(UTF8Char, false);
				this.getIMEWordAssociation(UTF8Char);
			}
			this.resetCharSequence();
		},

		/**
         * Called by keyboard service when user press a key
         * @private
         */
        keyPressed: function (UTF8Char) {
            if (this.getCharSequenceLength() >= this._maxCharSequenceLength) return;
            var charToDisplay = UTF8Char;
            if (this.getStrLang() == 'Hangul' && this.getCharSequenceLength() == 0) {
                ICTouchAPI.keyboardServices.displayChar(UTF8Char, true);
                this.setCharSequence(this.getCharSequence() + charToDisplay, false);
                return;
            }
            if (this.getStrLang() == 'pinyin')ICTouchAPI.keyboardServices.displayChar(UTF8Char, true);
            if (charToDisplay != "") this.setCharSequence(this.getCharSequence() + charToDisplay, true);
        },

		/**
		 * Set the char sequence with parameter and launch a getIMECandidate if boolean parameter is true
		 * @private
		 */
		setCharSequence : function(charSequence, boolDoNotGetCandidate) {
			this._charSequence = charSequence;
			if(charSequence.length >0 && boolDoNotGetCandidate)
				this.getIMECandidates();
		},


		getCharSequence : function() {
			return this._charSequence;
		},

        getCharSequenceLength: function () {
            //Hi Soft
            return this.getCharSequence().length;
        },

		resetCharSequence : function() {
			this.setCharSequence("");
		},

		//Delete the last char
		delChar : function() {
            //Hi Soft
            // if current str lang is Japanese, the request should not be sent
            var sendRequest = ('Katakana' != this.getStrLang() && 'Hiragana' != this.getStrLang());
            this.setCharSequence(this.getCharSequence().slice(0, this.getCharSequenceLength() - 1), sendRequest);
        },


        // Used with Hangul
		// When we validate a word, we have to delete its letters in the char sequence
		deleteFirstLetters : function(intLetters) {
			if(this.getCharSequenceLength() < intLetters)
				this.resetCharSequence();
			else
				this.setCharSequence(this.getCharSequence().slice(intLetters, this.getCharSequenceLength()-intLetters), true);
		},

		zhuyin_ime_translate_input_char : function( strInput, lastChar)
		{
			var charToSend = 0;
			if (this.getCharSequenceLength() != 0 && strInput[0] == "\u02C9") {
				charToSend = strInput[0]; /* space is 1st tone if input zone isn't empty */
			}

			if (this.getCharSequenceLength() == 0 ) { /* this is the 1st input */
				charToSend = strInput[0];

			} else if ( this.getCharSequenceLength() == 1 ) { /* this is the 2nd input */
				if ( lastChar == strInput[0] ) { // loop machanism
					charToSend = zhuyin_ime_loop( lastChar, strInput );
				}

				if ( charToSend == 0 ) { // intelligent input
					charToSend = zhuyin_ime_intelligent_input (strInput);
				}

				if ( ( charToSend == 0 ) && ( strInput[0] == "\u3127" ) ) { // inter-vowel or tone?
					charToSend = zhuyin_ime_inter_vowl_input (lastChar);
				}

				if ( charToSend == 0 ) {
					charToSend = strInput[0];
				}

			} else { /* this is the 3rd or 4th input */
				switch ( strInput[0] ) /* tones must be in this position */
				{
					case "\u310D":
					case "\u3110":
					case "\u3117":
					case "\u3127":
						charToSend = strInput[2] ;
						break;

					default:
						charToSend = zhuyin_ime_inter_vowl_input (lastChar);

						if ( charToSend == 0 ) {
							charToSend = strInput[0];
						}
						break;
				}
			}
			if (charToSend != 0)
				return charToSend;
			else
				return "";
		},

		zhuyin_ime_intelligent_input : function(strInput) {
			switch (strInput[0])
			{
				case "\u310D":return "\u02C7"; /* [E] */
				case "\u3110":return "\u02CB"; /* [R] */
				case "\u3117":return "\u02CA"; /* [Y] */
				case "\u3126":return "\u311F"; /* [o] */
				case "\u310B":return "\u3125"; /* [S] */
				case "\u3111":return "\u311A"; /* [F] */
				case "\u3108":return "\u311E"; /* [Z] */
				case "\u3112":return "\u3122"; /* [V] */

				default:
					return 0;
			}
		},

		zhuyin_ime_inter_vowl_input : function( lastChar )
		{
			switch (lastChar)
			{
				case "\u3105": /* [b] */
				case "\u3109": /* [d] */
				case "\u3110": /* [j] */
				case "\u310c": /* [l] */
				case "\u3107": /* [m] */
				case "\u310b": /* [n] */
				case "\u3106": /* [p] */
				case "\u3111": /* [q] */
				case "\u310a": /* [t] */
				case "\u3112": /* [x] */
					return "\u3127";

				default:
					return "\u02d9";
			}
		},

		zhuyin_ime_loop : function ( lastChar, strInput )
		{
			var loop_char = 0;

			switch ( lastChar )
			{
				case "\u3105": /* [Q] */
				case "\u3109": /* [W] */
				case "\u3113": /* [T] */
				case "\u311a": /* [F] */
				case "\u3116": /* [G] */
				case "\u3119": /* [H] */
				case "\u311e": /* [Z] */
				case "\u3122": /* [V] */
					loop_char = strInput[2];
					break;
				case "\u3106": /* [q] */
				case "\u310a": /* [w] */
				case "\u3114": /* [t] */
				case "\u3111": /* [f] */
				case "\u3115": /* [g] */
				case "\u3118": /* [h] */
				case "\u3108": /* [z] */
				case "\u3112": /* [v] */
					loop_char = strInput[0];
					break;
				default:
					break;
			}

			if ( 0 != loop_char ) {
				this.delChars();
			}

			return loop_char;
		}
	});

ICTouchAPI.IMEServices = new ICTouchAPI.IMEServices();
/**
* @class ICTouchAPI.debugServices
* @singleton
* @extends Object
* The debugServices class provides utilities functions to log messages and JavaScript objects at different levels.
* <br/><br/>
* The phone manages the following log levels:<br/>
* <pre>
* - debug
* - info
* - notice
* - warning
* - err (error)
* - emerg (emergency)
* </pre>
* <br/>
* By default, only logs with levels greater than or equal to <b>err</b> will be saved into the log file <i>/var/log/WebApp.log</i>.
* <br/>
* <br/>
* To see others logs, you should:<br/>
* - Set the setting <b>developerMode</b> to true<br/>
* - Set the desired log level(ie: <b>debug</b> level allows to see all logs).<br/>
* <br/>
* You can change default log level by using the following command line: <i>level WebApp <b>new_level</b></i><br/>
* <br/>
* The following example changes the log level to <b>debug</b>:
* <pre>
* // Set the log level to <b>debug</b>
* level WebApp debug
* </pre>
* <br/>
* For example, if you change the log level to <b>notice</b>, all logs using levels <b>notice</b>, '<b>warning</b>, <b>err</b> and <b>emergency</b> will be saved into the log file.
* <br/>
* <br/>
* Logs with level greater than or equal to <b>err</b> will be saved into the defensive log file located in <i>/log/Defence.log</i>.
* <br/>
* <br/>
* The following example logs a simple text in the log file
* <br/>
* <pre>
* // To log a simple text into the log file
* ICTouchAPI.debugServices.info("My text");
* </pre>
* <br/>
* The following example logs a JSON object in the log file
* <br/>
* <pre>
* //To log a JSON object into the log file
* ICTouchAPI.debugServices.info(myObj);
* </pre>
*
*/
dojo.provide("ICTouchAPI.debugServices");
window.ICTouchAPI.debugServices = {

        // {Boolean} boolNotToJson is a private parameter than enables to not use dojo.toJson() on the parameter (avoir the creation of big structure)

        /**
	 * Write an info message to the log file.<br/>
         * Only enabled when the setting <b>developerMode</b> is set to <b>True</b>.
	 * @param {String/Object} info The information message/object to log
	 */
	info: function(info, boolNotToJson) {
		this._sendMessage(info, 1, boolNotToJson);
		info = null;
		boolNotToJson = null;
	},

	/**
	 * Write a debug message to the log file.<br/>
         * Only enabled when the setting <b>developerMode</b> is set to <b>True</b>.
	 * @param {String/Object} debug The debug message/Object to log
	 */
	debug: function(debug, boolNotToJson) {
		this._sendMessage(debug, 0, boolNotToJson);
		debug = null;
		boolNotToJson = null;
	},

	/**
	 * Write a notice message to the log file.<br/>
         * Only enabled when the setting <b>developerMode</b> is set to <b>True</b>.
	 * @param {String/Object} notice The notice message/Object to log
	 */
	notice: function(notice, boolNotToJson) {
		this._sendMessage(notice, 2, boolNotToJson);
		notice = null;
		boolNotToJson = null;
	},

	/**
	 * Write a warning message to the log file.<br/>
         * Only enabled when the setting <b>developerMode</b> is set to <b>True</b>.
	 * @param {String/Object} warning The warning message/Object to log
	 */
	warning: function(warning, boolNotToJson) {
		this._sendMessage(warning, 3, boolNotToJson);
		warning = null;
		boolNotToJson = null;
	},

	/**
	 * Write an error message to the log file.
	 * @param {String/Object} error The error message/Object to log
	 */
	error: function(error, boolNotToJson) {
		this._sendMessage(error, 4, boolNotToJson);
		error = null;
		boolNotToJson = null;
	},

	/**
	 * write an emergency message to the log file.<br/>
	 * @param {String/Object} emergency The emergency message/Object to log
	 */
	emergency: function(emergency, boolNotToJson) {
		this._sendMessage(emergency, 5, boolNotToJson);
		emergency = null;
		boolNotToJson = null;
	},
        
	/**
	 * Send an alert on the screen
	 * @param {String} toSend Object (boolToJson = true) or string to send
         * @ignore
	 */
	alert: function(toSend){
		alert(dojo.toJson(toSend));
		toSend = null;
	},

	/**
	 * Send an info in a log file directly by webkit. This method is used to dump information (called by webapp's dump method).
         * This method is not removed by the javascript code compressor.
	 * @param {String} toSend Object (boolNotToJson != true) or string to send
	 * @param {Boolean} boolNotToJson use dojo.toJson, in order to get a string from an object
         * @ignore
	 */
	dump: function(toSend, boolNotToJson) {
		this._sendMessage(toSend, 1, boolNotToJson);
		toSend = null;
		boolNotToJson = null;
	},

	/**
	 * @ignore
	 */
	_sendMessage : function(toSend, level, boolNotToJson){
		var strMessage = toSend;
		if(!boolNotToJson){
			strMessage = dojo.toJson(toSend);
		}
		if(ICTouchAPI.remoteDisplay){
			if (level == 0) {
				console.log("debug[|]"+strMessage);
			}
			if (level == 1) {
				console.info("info[|]"+strMessage);
			}
			if (level == 2) {
				console.info("notice[|]"+strMessage);
			}
			if (level == 3) {
				console.warn("warning[|]"+strMessage);
			}
			if (level == 4) {
				console.error("error[|]"+strMessage);
			}
			if (level == 5) {
				console.error("emergency[|]"+strMessage);
			}
		}
		else {
			console.log(level+"[|]"+strMessage);
		}
		toSend = null;
		level = null;
		boolNotToJson = null;
	}

};

// Global function allowing to call the dump function of the given webapp to log its main data (according to the dump function defined in the webapp)
dumpWebapp = function (strWebappName, boolAdvancedLogs) {
	if (webapp[strWebappName] && typeof webapp[strWebappName].dump === "function") {
		webapp[strWebappName].dump(boolAdvancedLogs);
	}
	else {
		ICTouchAPI.debugServices.warn('ICTouchAPI.dumpWebapp: webapp ' + strWebappName + " does not exist or does not have any dump function");
	}
};

// Global function allowing to call the dump function of the given service to log its main data (according to the dump function defined in the service)
dumpService =  function (strServiceName, boolAdvancedLogs) {
	if (ICTouchAPI[strServiceName] && typeof ICTouchAPI[strServiceName].dump === "function") {
		ICTouchAPI[strServiceName].dump(boolAdvancedLogs);
	}
	else {
		ICTouchAPI.debugServices.warn('ICTouchAPI.dumpService: service ' + strServiceName + " does not exist or does not have any dump function");
	}
};
/**
* @class ICTouchAPI.HttpServices
* @singleton
* @extends Object
* Manage an HTTP Connection to a remote server
*/
dojo.provide("ICTouchAPI.HttpServices");
window.ICTouchAPI.HttpServices = {

	/**
	 * Get an HTTP Ressource
	 * @param {Object} args : : JavaScript object of name/string value pairs :<pre><code>
	 * - method : Method to pass parameters to remote ressource. Method is "get" or "post" or "put" or "head"
 	 * - url : url of ressource
	 * - putData : data send for xhrPut request
	 * - postData : data send for xhrPost request
	 * - responseType : Type of server response. Type is "text" or "xml"
	 * - content : contains properties with strinf values. These properties will be serialized as nameX=valueX and passed to the request<br>
	 * - context : The given context
	 * - header : A JavaScript object of name/string value pairs. These values are sending to HTTP Headers(Example: Content-Type, X-Method-Override, or Content-Encoding)<br>
	 * - timeout : Milliseconds to wait for the response. If this time passes, the error callback is called<br>
	 * - callback : This function is called on response received
	 * - callbackParams : parameters given to the callback whic can be reused in the callback function
	 * - callbackError : This function is called on timeout</code></pre>
	 */
	httpRequest: function(args) {

		var url = args.url;
		//Add "http://" at begin if not present
		if (url.search("^http://")==-1 && url.search("^https://")==-1) {
			url = "http://"+url;
		}
		//Rewrite URL to use internal proxy
		if (url.search("^http://")!=-1) {
			url = url.replace(new RegExp("^http://"), "http://"+location.hostname+"/ext/");
		}
		else if (url.search("^https://")!=-1) {
			url = url.replace(new RegExp("^https://"), "http://"+location.hostname+"/exts/");
		}

		ICTouchAPI.debugServices.debug("ICTouchAPI.HttpServices - httpRequest / HTTPServices::httpRequest() to:" + url);

		var xhrArgs	= {
			url: url,
			timeout: args.timeout || 20000,
			handleAs: args.responseType || "text",
			content : args.content,
			callbackParams : args.callbackParams,
			// call the defined callback function on result received
			load: function () {
				return args.callback && args.callback.apply(args.context, arguments);
			},
			// call the defined callbackError function on timeout
			error: function () {
				return args.callbackError && args.callbackError.apply(args.context, arguments);
			}
		};

		if (args.headers) {
			xhrArgs.headers = args.headers;
		}

		if (args.method == "get") {
			return dojo.xhrGet(xhrArgs);
		} else if (args.method == "post") {
			if (args.postData) {
				xhrArgs.postData = args.postData;
			}
			return dojo.xhrPost(xhrArgs);
		} else if (args.method == "put") {
			if (args.putData) {
				xhrArgs.putData = args.putData;
			}
			return dojo.xhrPut(xhrArgs);
		} else if(args.method == "head"){
			return dojo.xhr("HEAD",xhrArgs);
		} else {
			return false;
		}
	}
};
/**
* @class ICTouchAPI.mode
* @namespace ICTouchAPI
* @extends Object
* @ignore
* mode holds the instruction to be runned when it is selected as the current mode of a led
*/
dojo.provide("ICTouchAPI.mode");

dojo.declare("ICTouchAPI.mode",null,
{
	/* --------------------------------- Public attributes ------------------------------------ */

	/* --------------------------------- Private attributes ----------------------------------- */

	/* ------------------------------------ Constructor --------------------------------------- */

	/**
	 * constructor
	 */
	constructor : function(name,ledId,priority,runFunction){
		this.strName = name;
		this.intLedId = ledId;
		this.intPriority = priority;
		this.run = dojo.hitch(this,runFunction);

		this.boolSate = false;
	},

	/**
	 * destroy
	 */
	destroy : function(){
		delete this.run;
	},

	/* ----------------------------------- Getter / Setter------------------------------------- */

	/**
	 * Return the state of the mode
	 * @return {boolean} the state
	 */
	getState : function(){
		return this.boolSate;
	},

	/**
	 * Return the priority of the mode
	 * @return {Numeric} the priority
	 */
	getPriority : function(){
		return this.intPriority;
	},

	/* ----------------------------------- Public methods ------------------------------------- */

	/**
	 * activate the mode
	 */
	activate : function(){
		this.boolSate = true;
	},

	/**
	 * desactivate the mode
	 */
	desactivate : function(){
		this.boolSate = false;
	}

/* --------------------------------- Private Methods -------------------------------------- */

});

	/**
	 *global dictionnary of leds colors names. Used to translate colors names to number
	 * @property
	 * @type {Object} 
	 */
ICTouchAPI.mode.prototype.arrColors = {
	'std':0,
	'blue':1,
	'red':2,
	'purple':3,
	'green':4,
	'cyan':5,
	'yellow':6,
	'white':7
};
	/**
	 * global mode value reference
	 * @cfg {Numeric} intOFF numerical value to be sent to the core system for OFF mode
	 */
ICTouchAPI.mode.prototype.intOFF = 0;
	/**
	 * global mode value reference
	 * @cfg {Numeric} intON numerical value to be sent to the core system for ON mode
	 */
ICTouchAPI.mode.prototype.intON = 1;
	/**
	 * global mode value reference
	 * @cfg {Numeric} intBLINK numerical value to be sent to the core system for BLINK mode
	 */
ICTouchAPI.mode.prototype.intBLINK = 100;/**
* @class ICTouchAPI.led
* @extends Object
* @ignore
* class led used to manage a led modes and priorities
*/
dojo.require("ICTouchAPI.mode");
dojo.provide("ICTouchAPI.led");

dojo.declare("ICTouchAPI.led",null,
{
	/* --------------------------------- Public attributes ------------------------------------ */

	/* --------------------------------- Private attributes ----------------------------------- */

	/* ------------------------------------ Constructor --------------------------------------- */

	constructor : function(name,computeModeFunction) {
		//name of the button : home; mute, ...
		this.intName = name;
		this.arrModes = {};

		/*this._createModeON();
		this._createModeOFF();
		this._createModeBLINK();*/
        this.createMode(ICTouchAPI.mode.prototype.intOFF,
        ICTouchAPI.mode.prototype.arrColors['std'], 'OFF', 0);
        this.createMode(ICTouchAPI.mode.prototype.intON,
        ICTouchAPI.mode.prototype.arrColors['std'], 'ON', 10);
        this.createMode(ICTouchAPI.mode.prototype.intBLINK,
        ICTouchAPI.mode.prototype.arrColors['std'], 'BLINK', 20);


		this.currentMode = null;

		if(computeModeFunction && typeof(computeModeFunction) == "function"){
			//this.computeMode = dojo.hitch(this,computeModeFunction);
		}

		this.activateMode('OFF');

	},

	/**
	 *Method called when the LED has to be destroyed
	 */
	destroy : function(){
		//for each registred mode
		for(var _mode in this.arrModes){
			this.arrModes[_mode].destroy();

		}

	},

	/* ----------------------------------- Getter / Setter------------------------------------- */

	/* ----------------------------------- Public methods ------------------------------------- */

	/**
	 * add a mode to the led instance
	 * @param {string} strModeName name of the mode to be loaded
	 * @param {mode} mode the mode object
	 */
	addMode : function(strModeName,mode){
		this.arrModes[strModeName] = mode;
	},

	/**
	 *	try to activate a mode on the led
	 *	@param {string} strModeName name of the mode to be activated
	 */
	activateMode : function(strModeName){
		this.arrModes[strModeName].activate();
		if(!this.currentMode || this.arrModes[strModeName].getPriority() > this.currentMode.getPriority()){
			this.currentMode = this.arrModes[strModeName];
			this.currentMode.run();
		}
	},

	/**
	 * try to desactivate a mode on the led
	 * @param {string} strModeName name of the mode to be desactivated
	 */
	desactivateMode : function(strModeName){
		if(strModeName != 'OFF'){
			this.arrModes[strModeName].desactivate();
			if(this.arrModes[strModeName] == this.currentMode){
				this.computeMode();
			}
		}
	},

	/* --------------------------------- Private Methods -------------------------------------- */

	/**
	 * @private
	 */
	computeMode : function(){
		//if the current mode has been desactivated
		if(!this.currentMode || !this.currentMode.getState()){
			//set the current to OFF by default
			this.currentMode = this.arrModes['OFF'];
		}

		//for each registred mode
		var mode = null;
		for(var _mode in this.arrModes){
			mode = this.arrModes[_mode];
			//which is activated
			if(mode.getState()){
				//if it has priority on the crurrent mode
				if(mode.getPriority() > this.currentMode.getPriority()){
					//set the current mode up to it
					this.currentMode = mode;
				}else{
			//keep the current mode
			}
			}
		}
          //run the new current mode and ..
		//if it returns false, the mode has desactivated it itslef
		if(this.currentMode && !this.currentMode.run()){
			//if the mode return false then the new current mode has to be computed once again
			this.computeMode();
		}

	},

	/**
	 * @private
	 */
    createMode : function(callInteger,colorInteger,modeName,modePriority){
                callInteger = (callInteger ? callInteger : 0);
                colorInteger = (colorInteger ? colorInteger : 0);
                modeName = (modeName ? modeName : 'customMode');
                modePriority = (modePriority ? modePriority : 0);

                var runFunc = function(){
                        //runned in mode

                    // Don't call platform in simulation mode
                    if(!generalConfig.simulation) {
                        ICTouchAPI.APIServices.Platform.setLedStatus({
                                params: [
                                        this.intLedId, //known within the running context : MODE
                                        callInteger,
                                        colorInteger,
                                        0, //unused but has to be sent
                                        0  //unused but has to be sent
                                ]
                        });
                    }
                    return true;
                }
                //led mode
                this.arrModes[modeName] = new ICTouchAPI.mode(modeName,this.intName,modePriority,runFunc);

    }

			});

	/**
	 * global dictionnary of leds names. Used to translate names to number.
	 * @property
	 * @type {Object}
	 */
ICTouchAPI.led.prototype.arrLedsNames =  {
	'mute' : 0,
	'volumeDown' : 1,
	'volumeUp' : 2,
	'handsFree' : 3,
	'userInfo' : 4,
	'communication' : 5,
	'dialer' : 6,
	'event' : 7,
	'home' : 8,
	'ringing' : 9
};/**
 * @class ICTouchAPI.ledServices
 * @singleton
 * @extends Object
 * The ledService provides an interface to web applications to manage the senskey LEDs.<br />
 * Currently, only the 'event' led is programmable.
 * <pre>
 * To turn on the 'event' led:
 * ICTouchAPI.ledServices.activate('event', 'ON');
 * <br />
 * To turn off the 'event' led:
 * ICTouchAPI.ledServices.desactivate('event', 'ON');
 * <br />
 * To blink the 'event' led:
 * ICTouchAPI.ledServices.activate('event', 'BLINK');
 * <br />
 * To stop blinking the 'event' led:
 * ICTouchAPI.ledServices.desactivate('event', 'BLINK');
 * </pre>
 */
dojo.require("ICTouchAPI.led");
dojo.provide("ICTouchAPI.ledServices");
dojo.declare("ICTouchAPI.ledServices",null,
{

	/* --------------------------------- Public attributes ------------------------------------ */

	/* --------------------------------- Private attributes ----------------------------------- */

        /**
         * @ignore
         */
        arrLEDs: null,

	/* ------------------------------------ Constructor --------------------------------------- */

	constructor : function() {
		//Array holding the LEDs
		this.arrLEDs = {};
		//creating LEDs
		this._fillInArrLEDs();
	},


	destroy : function(){
		dojo.forEach(this.arrLEDs,function(led){
			led.destroy();
		});
	},

	/* ----------------------------------- Getter / Setter------------------------------------- */

	/* ----------------------------------- Public methods ------------------------------------- */

	/**
	 *	Activate a mode on a led
	 *	@param {string} led The name of the led. Should be set to 'event'
	 *	@param {string} mode The name of the mode to be activated. Should be set to 'ON' or 'BLINK'
	 */
	activate : function(led, mode){
            this.arrLEDs[ICTouchAPI.led.prototype.arrLedsNames[led]].activateMode(mode);
            led=null;
            mode=null;
	},

	/**
	 * Desactivate a mode on a led
	 * @param {string} led The name of the led. Should be set to 'event'
	 * @param {string} mode The name of the mode to be desactivated. Should be set to 'ON' or 'BLINK'
	 */
	desactivate : function(led, mode){
            this.arrLEDs[ICTouchAPI.led.prototype.arrLedsNames[led]].desactivateMode(mode);
            led=null;
            mode=null;
	},


	/* --------------------------------- Private Methods -------------------------------------- */

   /**
	* Method called to create the LEDs
	* @ignore
	*/
	_fillInArrLEDs : function(){
		//creating leds
		for( var name in ICTouchAPI.led.prototype.arrLedsNames){
			this.arrLEDs[ICTouchAPI.led.prototype.arrLedsNames[name]] = new ICTouchAPI.led(ICTouchAPI.led.prototype.arrLedsNames[name]);
		}
        this.addCustomModeToLed('userInfo', ICTouchAPI.mode.prototype.intON,ICTouchAPI.mode.prototype.arrColors['green'], 'AVAILABLE', 11);
		this.addCustomModeToLed('userInfo', ICTouchAPI.mode.prototype.intON,ICTouchAPI.mode.prototype.arrColors['red'], 'UNAVAILABLE', 12);
		this.addCustomModeToLed('userInfo', ICTouchAPI.mode.prototype.intON,ICTouchAPI.mode.prototype.arrColors['blue'], 'OUTOFOFFICE', 13);
		this.addCustomModeToLed('userInfo', ICTouchAPI.mode.prototype.intON,ICTouchAPI.mode.prototype.arrColors['blue'], 'ROUTING', 14);
		this.addCustomModeToLed('userInfo', ICTouchAPI.mode.prototype.intON,ICTouchAPI.mode.prototype.arrColors['red'], 'DND-ON', 15);
		
		// Mode "FORCE-OFF" is here to force deactivation of the led. It is mainly used on hotel mode. It must keep highest priority on userInfo led.
		this.addCustomModeToLed('userInfo', ICTouchAPI.mode.prototype.intOFF,ICTouchAPI.mode.prototype.arrColors['std'], 'FORCE-OFF', 100);

		this.addCustomModeToLed('communication', ICTouchAPI.mode.prototype.intBLINK,ICTouchAPI.mode.prototype.arrColors['std'], 'N_CALLBACK', 11);
		this.addCustomModeToLed('communication', ICTouchAPI.mode.prototype.intBLINK,ICTouchAPI.mode.prototype.arrColors['std'], 'N_UNANS_CALL', 11);
		this.addCustomModeToLed('communication', ICTouchAPI.mode.prototype.intBLINK,ICTouchAPI.mode.prototype.arrColors['std'], 'N_VM', 11);
		this.addCustomModeToLed('communication', ICTouchAPI.mode.prototype.intBLINK,ICTouchAPI.mode.prototype.arrColors['std'], 'N_IM', 11);
		this.addCustomModeToLed('communication', ICTouchAPI.mode.prototype.intBLINK,ICTouchAPI.mode.prototype.arrColors['std'], 'N_HOTEL', 11);
		this.addCustomModeToLed('communication', ICTouchAPI.mode.prototype.intON,ICTouchAPI.mode.prototype.arrColors['std'], 'ONGOING_CALL', 12);
	},

	/**
	 * @ignore
	 */
    addCustomModeToLed : function(strLedName, callInteger, colorInteger, modeName,modePriority){
                var intLedId = ICTouchAPI.led.prototype.arrLedsNames[strLedName];
                var ledObj = this.arrLEDs[intLedId];

                ledObj.createMode(callInteger,colorInteger, modeName, modePriority);

        },

	/**
	 * @ignore
	 */
	_customCommunicationLed : function(){
		var intLedId = ICTouchAPI.led.prototype.arrLedsNames['communication'];
		var comLed = this.arrLEDs[intLedId];

		var runFunc = function(){
			//context : mode
			ICTouchAPI.APIServices.Platform.setLedStatus({
				params: [
				intLedId,
				ICTouchAPI.mode.prototype.intBLINK,
				ICTouchAPI.mode.prototype.arrColors['std'],
				0, //unused but has to be sent
				0  //unused but has to be sent
				]
			});
			return true;
		}
		//context : led
		comLed.addMode('N_CALLBACK',new ICTouchAPI.mode('N_CALLBACK',intLedId,11,runFunc));
		comLed.addMode('N_UNANS_CALL',new ICTouchAPI.mode('N_UNANS_CALL',intLedId,11,runFunc));
		comLed.addMode('N_VM',new ICTouchAPI.mode('N_VM',intLedId,11,runFunc));
		comLed.addMode('N_IM',new ICTouchAPI.mode('N_IM',intLedId,11,runFunc));
		comLed.addMode('N_HOTEL',new ICTouchAPI.mode('N_HOTEL',intLedId,11,runFunc));

		var runFunc = function(){
			//context : mode
			ICTouchAPI.APIServices.Platform.setLedStatus({
				params: [
				intLedId,
				ICTouchAPI.mode.prototype.intON,
				ICTouchAPI.mode.prototype.arrColors['std'],
				0, //unused but has to be sent
				0  //unused but has to be sent
				]
			});
			return true;
		}
		//context : led
		comLed.addMode('ONGOING_CALL',new ICTouchAPI.mode('ONGOING_CALL',intLedId,12,runFunc));

	}

});

ICTouchAPI.ledServices = new ICTouchAPI.ledServices();/**
* @class ICTouchAPI.presenceServices
* @singleton
* @ignore
* @extends Object
*/
dojo.provide("ICTouchAPI.presenceServices");
dojo.declare("ICTouchAPI.presenceServices",
	null,
	{
		/* --------------------------------- Public attributes ------------------------------------ */



		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		* @property
		* @type Boolean
		* @private
		*/
		imEnabled : false,

		/**
		* @property
		* @type Boolean
		* @private
		*/
		presenceAddedInDs : false,

		/**
		* @property
		* @type Boolean
		* @private
		*/
		boolIMAvailble : null,

		/**
		* @property
		* @type Boolean
		* @private
		*/
		boolTelephonyPresenceAvailable : null,
		/**
		* @property
		* @type Boolean
		* @private
		*/
		boolDisableIM : false,

		/* ------------------------------------ Constructor --------------------------------------- */

		/**
		* The presence services manage presence for contacts and update it in dataStore for webapps
		* @private
		*/
		constructor: function() {
			ICTouchAPI.eventServices.subscribeToEvent(this,"IM_ENABLED",this.initPresenceInDataStore);
			// Default settings for simulation mode
                        if(!generalConfig.simulation) {
			ICTouchAPI.settingServices.subscribeToSetting(this,"IMAvailable",this.onIMAvailable);
			ICTouchAPI.settingServices.getSetting("IMAvailable",this,this.onIMAvailable);
			ICTouchAPI.settingServices.subscribeToSetting(this,"TelephonyPresenceAvailable",this.onTelephonyPresence);
			ICTouchAPI.settingServices.getSetting("TelephonyPresenceAvailable",this,this.onTelephonyPresence);
                        }
			// If we are in remote display, call manually the initDataStores function to create the dataStore of contacts
			// (in remote display we will not receive the 'end_of_init_IctDirectory' event)'
			if (ICTouchAPI.remoteDisplay) {
				this.initPresenceInDataStore({
					value:true
				});
			}
			ICTouchAPI.eventServices.subscribeToEvent(this, "IM_BUDDIES_DELETED", this.onBuddyDeleted);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * Get Application Name
		 * @private
		 * @return {String} The name of the Application Service
		 */
		getApplicationName: function(){
			return "presenceServices";
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		* Get the state of the IM feature
		* @return {Boolean} True if the Im is enabled
		*/
		isImAvailable : function(){
			return this.imEnabled;
		},

		/**
		* Get the state telephony presence availability
		* @return {Boolean} True if the telephony presence is available
		*/
		isTelephonyPresenceAvailable : function() {
			return this.boolTelephonyPresenceAvailable;
		},

		/**
		* Get the state IM presence availability
		* @return {Boolean} True if the IM presence is available
		*/
		isIMPresenceAvailable : function() {
			return this.boolIMAvailble;
		},

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @private
		 */
		initPresenceInDataStore: function(event) {
			// Static value to disable presence feature because out of scope.
			if(!this.boolDisableIM){
				if(event && typeof event.value != "undefined"){
					var myDS = ICTouchAPI.dataStoreServices.getStore('contacts');
					var myDSContent;
					// IM may be enabled for managerAssistant, even if both settings of IM and telephonic presence are unavailable.
					if(event.value && (this.boolIMAvailble || this.boolTelephonyPresenceAvailable)){
						// Initiate presence in DataStore if not already done
						if(myDS && !this.presenceAddedInDs){
							// Add columns in dataModel
							myDS.addColumn({
								index: 'IMPresence'
							});
							myDS.addColumn({
								index: 'telephonyPresence'
							});
							// Subscribe to presenceChange event
							ICTouchAPI.eventServices.subscribeToEvent(this,"IM_PRESENCE_CHANGE",this.onPresenceChanged);
							this.presenceAddedInDs = true;
						}
						if(this.presenceAddedInDs){
							// If dataStore is initiated, get presence for every contact in it.
							myDSContent = myDS.getList();
							var arrContactId = [];
							for(var i in myDSContent){
								arrContactId.push(myDSContent.contactId);
							}
							ICTouchAPI.APIServices.InstantMessaging.getPresenceForContacts({
								params:[arrContactId],
								context:this,
								callback:this.setPresenceList
							});
						}
					}
					this.imEnabled = event.value;
					dojo.publish("IM_AVAILABILITY_CHANGED", [this.imEnabled]);
				}
			}
		},

		/**
		 * @private
		 */
		setPresenceList : function(presenceList) {
			if(presenceList && presenceList.length){
				var myDS = ICTouchAPI.dataStoreServices.getStore('contacts');
				for (var i = 0; i < presenceList.length; i++) {
					if(presenceList[i].contactId){
						var presence = {
							IMPresence : '',
							telephonyPresence : ''
						};
						if(this.boolIMAvailble){
							presence.IMPresence = presenceList[i].imPresenceValue.value.labelStatus;
						}
						if(this.boolTelephonyPresenceAvailable){
							presence.telephonyPresence = presenceList[i].telephonicPresenceValue.value.labelStatus;
						}
						if(presence.IMPresence != '' || presence.telephonyPresence != ''){
							myDS.editAt(presenceList[i].contactId, presence);
						}
					}
				}
			}
		},

		/**
		 * @private
		 */
		onPresenceChanged : function(presenceList) {
			if(presenceList && presenceList.value && presenceList.value.length){
				this.setPresenceList(presenceList.value);
			}
		},

		/**
		 ** @private
		 **/
		onIMAvailable : function(objSetting){
			if(objSetting){
				this.boolIMAvailble = objSetting.jsValue;
		    }

			if (ICTouchAPI.remoteDisplay) {
				this.initPresenceInDataStore({
					value:this.boolIMAvailble || this.boolTelephonyPresenceAvailable
				});
			}
		},

                /**
                 * @private
                 */
		onTelephonyPresence : function(objSetting){
			if(objSetting){
				this.boolTelephonyPresenceAvailable = objSetting.jsValue;
				dojo.publish("TELEPHONY_PRESENCE_AVAILABILITY_CHANGED", [this.boolTelephonyPresenceAvailable]);
			}

			if (ICTouchAPI.remoteDisplay) {
				this.initPresenceInDataStore({
					value:this.boolIMAvailble || this.boolTelephonyPresenceAvailable
				});
			}
		},

		/**
		 * @private
		 */
		onBuddyDeleted : function(arrContactId){
			var contactId;
			var arrContact;
			var presence = {
				IMPresence : '',
				telephonyPresence: ''
			};
			var myDS = ICTouchAPI.dataStoreServices.getStore('contacts');
			while(arrContactId.value.length > 0){
				contactId = arrContactId.value.pop();
				arrContact = myDS.find(["contactId"], contactId);

				if(this.isImAvailable() && arrContact[0]){
				myDS.editAt(arrContact[0].contactId, presence);
				}
		    }
		}

	});

// Create presence services
ICTouchAPI.presenceServices=new ICTouchAPI.presenceServices();



/**
* @class ICTouchAPI.hotelServices
* @extends Object
* @singleton
* Retrieves every user information from OXE in hotel mode, and provide a getter for these info.
 * <br />
* The service publishes the following events (using dojo.publish):<br />
*	- <b>GuestCheckIn:</b> published when the checkin state changes. Provide a boolean indicating if the state is checkin or not.<br />
*	- <b>GuestInfo:</b> published when new guest informations are received. Provide an object containing the fields guestName, guestNumber and setNumber.<br />
*/
dojo.provide("ICTouchAPI.hotelServices");
dojo.declare("ICTouchAPI.hotelServices", null , {

/* --------------------------------- Public attributes ------------------------------------ */

/* --------------------------------- Private attributes ----------------------------------- */

	// Defines
	/**
	 * @ignore
	 */
	boolCheckIn					: null,

	/**
	 * @ignore
	 */
	guestName					: "",

	/**
	 * Also called GPIN
	 * @ignore
	 */
	guestNumber					: "",

	/**
	 * @ignore
	 */
	setNumber					: "",

       
/* ------------------------------------ Constructor --------------------------------------- */

	/**
	 * @ignore
	 */
	constructor : function () {
		ICTouchAPI.eventServices.subscribeToEvent(this, "GuestInfo", this.notifyGuestInfo);
		ICTouchAPI.eventServices.subscribeToEvent(this, "GuestCheckIn", this.notifyGuestCheckIn);
	},

/* ----------------------------------- Getter / Setter------------------------------------- */


/* ----------------------------------- Public methods ------------------------------------- */

	/**
	 * Return all the information provided by OXE in Hotel mode.
	 * @return {Object} An object containing a the CheckIn state, a guest name, a guest number and a set number.
	 */
	getHotelInfo : function() {
		return {
			boolCheckIn : this.boolCheckIn,
			guestName : this.guestName,
			guestNumber : this.guestNumber, // GPIN
			setNumber : this.setNumber
		};
	},

/* --------------------------------- Private Methods -------------------------------------- */

	/**
	 * @ignore
	 * Because the eventServices wants an application name we provide one.
	 * @return {String} Return hotelServices
	 */
	getApplicationName: function(){
		return "hotelServices";
	},
	
	/**
	 * @ignore
	 */
	notifyGuestCheckIn : function(objEvent) {
		if(objEvent){
			var boolCheckInState = (objEvent.value) ? true : false;
			this.boolCheckIn = boolCheckInState;
			dojo.publish("GuestCheckIn", [boolCheckInState]);
		}
		objEvent = null;
	},

	/**
	 * @ignore
	 */
	notifyGuestInfo : function() {
		var objEventArguments = ICTouchAPI.tools.getEventArguments(arguments);

		// Update class attributes
		this.guestName = objEventArguments.guestName || "";
		this.guestNumber = objEventArguments.guestNumber || "";
		this.setNumber = objEventArguments.setNumber || "";

		var objInfo = {
			guestName : this.guestName,
			guestNumber : this.guestNumber,
			setNumber : this.setNumber
		}

		dojo.publish("GuestInfo", [objInfo]);
	},

	/**
	 * @ignore
	 */
	dump : function() {
		this.inherited(arguments);
		ICTouchAPI.debugServices.dump("HotelServices data:");
		ICTouchAPI.debugServices.dump(" - boolCheckIn: " + this.boolCheckIn);
		ICTouchAPI.debugServices.dump(" - guestName: " + this.guestName);
		ICTouchAPI.debugServices.dump(" - guestNumber: " + this.guestNumber);
		ICTouchAPI.debugServices.dump(" - setNumber: " + this.setNumber);
	}
});

ICTouchAPI.hotelServices = new ICTouchAPI.hotelServices();
/*
  Date.js, an Date class extension, adding localized format, parse and week numbering.
  Copyright (C) 2008 Henrik Lindqvist <henrik.lindqvist@llamalab.com>

  This library is free software: you can redistribute it and/or modify
  it under the terms of the GNU Lesser General Public License as published
  by the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/**
 * Extends the native <code>Date</code> class with additional functionality.
 * <p>Many of the date arithmetics are based on the information from
 * <a href="http://http://www.merlyn.demon.co.uk/frames-1.htm" target="_blank">http://www.merlyn.demon.co.uk/frames-1.htm</a>.</p>
 * @class Date
 * @version 0.1
 * @author Henrik Lindqvist &lt;<a href="mailto:henrik.lindqvist@llamalab.com">henrik.lindqvist@llamalab.com</a>&gt;
 * @ignore
 */

dojo.provide("ICTouchAPI.Date");


(function (d, dp) {

    /**
 * Get the date localization data for a specific language.
 * <p>Standard localizations included:</p>
 * <ul>
 *  <li><code>en-US</code></li>
 *  <li><code>iso</code> - Use for <a href="http://en.wikipedia.org/wiki/ISO_8601"
 *    target="_blank">ISO8601</a> and Schema <a href="http://www.w3.org/TR/xmlschema11-2/#dateTime"
 *    target="_blank"><code>dateTime</code></a></li>
 * </ul>
 * <p>A good source of localization data are
 * <a href="http://www.unicode.org/cldr/" target="_blank">http://www.unicode.org/cldr/</a>.</p>
 * @function {static object} i18n
 * @param {optional string} l - language, or user-agent language if omitted.
 * @returns localization data
 */
    d.i18n = function (l) {
        return (typeof l == 'string')
        ? (l in Date.i18n ? Date.i18n[l] : Date.i18n(l.substr(0, l.lastIndexOf('-'))))
        : (l || Date.i18n(navigator.language || navigator.browserLanguage || ''));
    };
    d.i18n.inherit = function (l, o) {
        l = Date.i18n(l);
        for (var k in l) if (typeof o[k] == 'undefined') o[k] = l[k];
        return o;
    };
    d.i18n[''] = // default
    d.i18n['en'] =
    d.i18n['en-US'] = {
        months: {
            abbr: [ 'Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec' ],
            full: [ 'January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December' ]
        },
        days: {
            init: [ 'S', 'M', 'T', 'W', 'T', 'F', 'S' ],
            abbr: [ 'Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat' ],
            full: [ 'Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday' ]
        },
        week: {   // Used by date pickers
            abbr: 'Wk',
            full: 'Week'
        },
        /*
  formats: {
    dateShort: 'M/d/yy',
    timeShort: 'h:mm a'
  },
    */

        ad: 'AD',
        am: 'AM',
        pm: 'PM',
        gmt: 'GMT',
        z: ':',   // Hour - minute separator
        Z: '',    // Hour - minute separator
        fdow: 0,  // First day of week
        mdifw: 1  // Minimum days in first week
    };
    d.i18n['iso'] = d.i18n.inherit('en', {
        Z: ':',
        fdow: 1,
        mdifw: 4
    });
    /**
 * Milliseconds in a week.
 * @property {static read number} WEEK
 */
    d.WEEK = 6048e5;
    /**
 * Milliseconds in a day.
 * @property {static read number} DAY
 */
    d.DAY = 864e5;
    /**
 * Milliseconds in an hour.
 * @property {static read number} HOUR
 */
    d.HOUR = 36e5;
    /**
 * Milliseconds in a minute.
 * @property {static read number} MINUTE
 */
    d.MINUTE = 6e4
    /**
 * Milliseconds in a second.
 * @property {static read number} SECOND
 */
    d.SECOND = 1000;
    /**
 * New <code>Date</code> instance for todays date, with time at midnight.
 * @function {static Date} today
 * @returns todays date at midnight.
 * @see datePart
 */
    d.today = function () {
        return new Date().datePart();
    };
    /**
 * Clone <code>this</code> date, creating a new instance.
 * @function {Date} clone
 * @returns clone of <code>this</code> date.
 */
    dp.clone = function() {
        return new Date(+this);
    };
    /**
 * Create a new instance with only the date part from <code>this</code> date.
 * <p>The time for the new date will be midnight.</p>
 * @function {Date} datePart
 * @returns the date at midnight.
 * @see timePart
 */
    dp.datePart = function () {
        with (this) return new Date(getFullYear(), getMonth(), getDate());
    };
    /**
 * Create a new instance with only the time part from <code>this</code> date.
 * <p>The date will be the JavaScript epoc (1970-01-01).</p>
 * @function {Date} timePart
 * @returns the time of <code>this</code> date.
 * @see datePart
 */
    dp.timePart = function () {
        with (this) return new Date(1970, 0, 1, getHours(), getMinutes(), getSeconds(), getMilliseconds());
    };
    /**
 * Get the first letter of localized days of week.
 * @function getDaysOfWeek
 * @param {string} l - localization.
 * @returns days of week (S,M,T,W,T,F,S).
 */
    d.getDaysOfWeek = function (l) {
        var i18n = Date.i18n(l);
        return i18n.days['init'];
    };
    /**
 * Set the "raw" unlocalized weekday.
 * @function setDay
 * @param {number} d - the weekday (0-6).
 * @see getDay (<a href="http://developer.mozilla.org/en/docs/Core_JavaScript_1.5_Reference:Global_Objects:Date:getDay" target="_blank">native</a>)
 */
    dp.setDay = function (d) {
        with (this) setDate((getDate() - getDay()) + d);
    };
    /**
 * Get the localized day of week.
 * @function {number} getDayOfWeek
 * @param {optional} o - first day of week number, language string or localization object.
 * @returns the weekday (1-7).
 * @see setDayOfWeek
 */
    dp.getDayOfWeek = function (o) {
        if (typeof o != 'number') o = Date.i18n(o).fdow;
        var d = this.getDay() - o;
        if (d < 0) d += 7;
        return d + 1;
    };
    /**
 * Set the localized day of week.
 * @function setDayOfWeek
 * @param {number} d - the weekday (1-7).
 * @param {optional} o - first day of week number, language string or localization object.
 * @see getDayOfWeek
 */
    dp.setDayOfWeek = function (d, o) {
        with (this) setDate((getDate() - getDayOfWeek(o)) + d);
    };
    /**
 * Get the maximum days in the month for <code>this</code> date.
 * @function {number} getDaysInMonth
 * @returns the number of days in this month, 1-31.
 */
    dp.getDaysInMonth = function () {
        with (this.clone()) {
            setDate(32);
            return 32 - getDate();
            }
    };
    /**
 * Get the maximum days in the year for <code>this</code> date.
 * @function {number} getDaysInYear
 * @returns the number of days in this year (1-366).
 */
    dp.getDaysInYear = function () {
        var y = this.getFullYear();
        return Math.floor((Date.UTC(y+1, 0, 1) - Date.UTC(y, 0, 1)) / Date.DAY);
    };
    /**
 * Get the day of the year for <code>this</code> date.
 * @function {number} getDayOfYear
 * @returns the day of this year (1-366).
 * @see setDayOfYear
 */
    dp.getDayOfYear = function () {
        return Math.floor((this - new Date(this.getFullYear(), 0, 1)) / Date.DAY) + 1;
    };
    /**
 * Set the day in the year for <code>this</code> date.
 * @function setDayOfYear
 * @param {number} d - the day of year (1-366).
 * @see getDayOfYear
 */
    dp.setDayOfYear = function (d) {
        this.setMonth(0, d);
    };
    /**
 * Get the week of month for <code>this</code> date.
 * @function {number} getWeekOfMonth
 * @param {optional} l - language string or localization object.
 * @returns the week number of this month (0-6).
 * @see setWeekOfMonth
 */
    dp.getWeekOfMonth = function (l) {
        l = Date.i18n(l);
        with (this.clone()) {
            setDate(1);
            var d = (7 - (getDay() - l.fdow)) % 7;
            d = (d < l.mdifw) ? -d : (7 - d);
            return Math.ceil((this.getDate() + d) / 7);
            }
    };
    /**
 * Set the week of month for <code>this</code> date.
 * @function setWeekOfMonth
 * @param {number} w - the week number of this month (1-6).
 * @param {optional} l - language string or localization object.
 * @see setWeekOfMonth
 */
    dp.setWeekOfMonth = function (w, l) {
        l = Date.i18n(l);
        with (this.clone()) {
            setDate(1);
            var d = (7 - (getDay() - l.fdow)) % 7;
            d = (d < l.mdifw) ? -d : (7 - d);
            setDate(d);
            }
    };
    /**
 * Get the week of year for <code>this</code> date.
 * @function {number} getWeekOfYear
 * @param {optional} l - language string or localization object.
 * @returns the week number of this year, 1-53.
 * @see setWeekOfMonth
 */
    dp.getWeekOfYear = function (l) {
        l = Date.i18n(l);
        with (this.clone()) {
            setMonth(0, 1);
            var d = (7 - (getDay() - l.fdow)) % 7;
            if (l.mdifw < d) d -= 7;
            setDate(d);
            var w = Math.ceil((+this - valueOf()) / Date.WEEK);
            return (w <= getWeeksInYear()) ? w : 1;
            }
    };
    /**
 * Set the week of year for <code>this</code> date.
 * @function setWeekOfYear
 * @param {number} w - the week number in this year, 1-53.
 * @param {optional} l - language string or localization object.
 * @see setWeekOfMonth
 */
    dp.setWeekOfYear = function (w, l) {
        l = Date.i18n(l);
        with (this) {
            setMonth(0, 1);
            var d = (7 - (getDay() - l.fdow)) % 7;
            if (l.mdifw < d) d -= 7;
            d += w * 7;
            setDate(d);
            }
    };
    /**
 * Get the maximum weeks in the year of <code>this</code> date.
 * @function {number} getWeeksInYear
 * @returns the number of weeks in this year, 1-53.
 */
    dp.getWeeksInYear = function () {
        var y = this.getFullYear();
        return 52 + (new Date(y, 0, 1).getDay() == 4 || new Date(y, 11, 31).getDay() == 4);
    };
    /**
 * Set the timezone offset for <code>this</code> date.
 * <p>This function only adjusts the date to the supplied offset,
 * it doesn&rsquo;t actually set the timezone.</p>
 * @function setTimezoneOffset
 * @param {number} o - offset in minutes.
 * @see getTimezoneOffset (<a href="http://developer.mozilla.org/en/docs/Core_JavaScript_1.5_Reference:Global_Objects:Date:getTimezoneOffset" traget="_blank">native</a>)
 */
    dp.setTimezoneOffset = function (o) {
        with (this) setTime(valueOf() + ((getTimezoneOffset() + -o) * Date.MINUTE));
    };

    /**
 * Format <code>this</code> date into a string.
 * <p>For pattern syntax, see <a href="http://java.sun.com/javase/6/docs/api/java/text/SimpleDateFormat.html" target="_blank">java.text.SimpleDateFormat</a>.</p>
 * @function format
 * @param {string} p - the pattern string of format
 * @param l - the language (string) or locationzation data used
 * @see i18n
 */
    dp.format = function (p, l) {
        var i18n = Date.i18n(l);
        var d = this;
        var pad = function (n, l) {
            for (n = String(n), l -= n.length; --l >= 0; n = '0'+n);
            return n;
        };
        var tz = function (n, s) {
            return ((n<0)?'+':'-')+pad(Math.abs(n/60),2)+s+pad(Math.abs(n%60),2);
        };
        return p.replace(/([aDdEFGHhKkMmSsWwyZz])\1*|'[^']*'|"[^"]*"/g, function (m) {
            l = m.length;
            switch (m.charAt(0)) {
                case 'a':
                    return (d.getHours() < 12) ? i18n.am : i18n.pm;
                case 'D':
                    return pad(d.getDayOfYear(), l);
                case 'd':
                    return pad(d.getDate(), l);
                case 'E':
                    return i18n.days[(l > 3)?'full':'abbr'][d.getDay()];
                case 'F':
                    return pad(d.getDayOfWeek(i18n), l);
                case 'G':
                    return i18n.ad;
                case 'H':
                    return pad(d.getHours(), l);
                case 'h':
                    return pad(d.getHours() % 12 || 12, l);
                case 'K':
                    return pad(d.getHours() % 12, l);
                case 'k':
                    return pad(d.getHours() || 24, l);
                case 'M':
                    return (l < 3)
                    ? pad(d.getMonth() + 1, l)
                    : i18n.months[(l > 3)?'full':'abbr'][d.getMonth()];
                case 'm':
                    return pad(d.getMinutes(), l);
                case 'S':
                    return pad(d.getMilliseconds(), l);
                case 's':
                    return pad(d.getSeconds(), l);
                case 'W':
                    return pad(d.getWeekOfMonth(i18n), l);
                case 'w':
                    return pad(d.getWeekOfYear(i18n), l);
                case 'y':
                    return (l == 2)
                    ? String(d.getFullYear()).substr(2)
                    : pad(d.getFullYear(), l);
                case 'Z':
                    return tz(d.getTimezoneOffset(), i18n.Z);
                case 'z':
                    return i18n.gmt+tz(d.getTimezoneOffset(), i18n.z);
                case "'":
                case '"':
                    return m.substr(1, l - 2);
                default:
                    throw new Error('Illegal pattern');
            }
        });
    }
    /**
 * Parse a date string.
 * This function replaces the built-in <code>parse</code> but
 * reverts back to the original if <code>p</code> is omitted.
 * <p>For pattern syntax, see <a href="http://java.sun.com/javase/6/docs/api/java/text/SimpleDateFormat.html" target="_blank">java.text.SimpleDateFormat</a>.</p>
 * @function {static Date} parse
 * @param {optional string} p - the pattern string of format.
 * @param {optional} l - the language (string) or locationzation data used.
 * @returns the parsed {@link Date}, or <code>NaN</code> on failure.
 * @see i18n
 */
    d.parse = function (s, p, l) {
        if (!p) return arguments.callee.original.call(this);
        var i18n = Date.i18n(l), d = new Date(1970,0,1,0,0,0,0);
        var pi = 0, si = 0, i, j, k, c;
        var num = function (x) {
            if (x) l = x;
            else if (!/[DdFHhKkMmSsWwy]/.test(p.charAt(pi))) l = Number.MAX_VALUE;
            for (i = si; --l >= 0 && /[0-9]/.test(s.charAt(si)); si++);
            if (i == si) throw 1;
            return parseInt(s.substring(i, si), 10);
        };
        var cmp = function (x) {
            if (s.substr(si, x.length).toLowerCase() != x.toLowerCase()) return false;
            si += x.length;
            return true;
        };
        var idx = function (x) {
            for (i = x.length; --i >= 0;) if (cmp(x[i])) return i+1;
            return 0;
        };
        try {
            while (pi < p.length) {
                c = p.charAt(l = pi);
                if (/[aDdEFGHhKkMmSsWwyZz]/.test(c)) {
                    while (p.charAt(++pi) == c);
                    l = pi - l;
                    switch (c) {
                        case 'a':
                            if (cmp(i18n.pm)) d.setHours(12 + d.getHours());
                            else if (!cmp(i18n.am)) throw 2;
                            break;
                        case 'D':
                            d.setDayOfYear(num());
                            break;
                        case 'd':
                            d.setDate(num());
                            break;
                        case 'E':
                            if (i = idx(i18n.days.full)) d.setDay(i - 1);
                            else if (i = idx(i18n.days.abbr)) d.setDay(i - 1);
                            else throw 3;
                            break;
                        case 'F':
                            d.setDayOfWeek(num(), i18n);
                            break;
                        case 'G':
                            if (!cmp(i18n.ad)) throw 4;
                            break;
                        case 'H':
                        case 'k':
                            d.setHours((i = num()) < 24 ? i : 0);
                            break;
                        case 'K':
                        case 'h':
                            d.setHours((i = num()) < 12 ? i : 0);
                            break;
                        case 'M':
                            if (l < 3) d.setMonth(num() - 1);
                            else if (i = idx(i18n.months.full)) d.setMonth(i - 1);
                            else if (i = idx(i18n.months.abbr)) d.setMonth(i - 1);
                            else throw 5;
                            break;
                        case 'm':
                            d.setMinutes(num());
                            break;
                        case 'S':
                            d.setMilliseconds(num());
                            break;
                        case 's':
                            d.setSeconds(num());
                            break;
                        case 'W':
                            d.setWeekOfMonth(num(), i18n);
                            break;
                        case 'w':
                            d.setWeekOfYear(num(), i18n);
                            break;
                        case 'y':
                            d.setFullYear((l == 2) ? 2000 + num() : num());
                            break;
                        case 'z':
                            if (!cmp(i18n.gmt)) throw 6;
                        case 'Z':
                            if (!/[+-]/.test(j = s.charAt(si++))) throw 6;
                            k = num(2) * 60;
                            if (!cmp(i18n[c])) throw 7;
                            k += num(2);
                            d.setTimezoneOffset((j == '+') ? -k : k);
                    }
                }
                else if (/["']/.test(c)) {
                    while (++pi < p.length && p.charAt(pi) != c);
                    if (!cmp(p.substring(l+1, pi++))) throw 8;
                }
                else {
                    while (pi < p.length && !/[aDdEFGHhKkMmSsWwyZz"']/.test(p.charAt(pi))) pi++;
                    if (!cmp(p.substring(l, pi))) throw 9;
                }
            }
            return d;
        }
        catch (e) {
            if (e > 0) return Number.NaN;
            throw e;
        }
    };
    d.parse.original = d.parse;

})(Date, Date.prototype);


var bDevMode = generalConfig.developer;
if (bDevMode) {
    dojo.require("ICTouchAPI.Date.requireList",true);
    dojo.forEach(ICTouchAPI.Date.requireList,
        function (newLang) {
            dojo.require(newLang);
        }
    );
} else {
    dojo.require("ICTouchAPI.Date.Date_mini", true);
}



/**
* @class ICTouchAPI.VideoServices
* @namespace ICTouchAPI
* @extends Object
* @ignore
*/
dojo.provide("ICTouchAPI.VideoServices");
dojo.declare("ICTouchAPI.VideoServices", null , {


	/* --------------------------------- Public attributes ------------------------------------ */

	/* --------------------------------- Private attributes ----------------------------------- */


	/**
	 * {Integer} Set to TRUE if video is currently shown.
	 * @private
	 */
	_videoDisplayed: 0,

	/**
	 * {Boolean} Set to TRUE if video com is ongoing (displayed or not).
	 * @private
	 */
	_videoComActive: false,


	_eventListenerHandleList: {},

	/**
	 * {Object} webapp show/hide event listener handle
	 * @private
	 */
	_webappEventListenerHandle: null,

	/**
	 * {Object} toaster show/hide event listener handle
	 * @private
	 */
	_toasterEventListenerHandle: null,

	/**
	 * {Object} popup show/hide event listener handle
	 * @private
	 */
	_popupEventListenerHandle: null,


	/**
	 * {Integer} webapp show/hide status code (100 = webapp is shown, 0 = webapp hidden)
	 * @private
	 */
	_webappShowStatus: 0,

	/**
	 * {Integer} toaster show/hide status code (10 = toaster is shown, 0 = toaster hidden)
	 * @private
	 */
	_toasterShowStatus: 0,

	/**
	 * {Integer} popup show/hide status code (1 = popup is shown, 0 = popup hidden)
	 * @private
	 */
	_popupShowStatus: 0,

	/**
	 * {Boolean} Is a Video mask set ? (false = no, true = yes)
	 * @private
	 */
	_maskSetStatus: false,

	/**
	 * {Object} Object used to store video details (position, size).
	 * Used to restore properly depending on webapp showing it
	 * @private
	 */
	_statesList: {},

	/**
	 * {String} audio toaster id
	 * @private
	 * @todo Is this id really static ?
	 */
	_audioToasterId: "webapp_audiosetting_getAudioToaster_0",

	_strReservedVideoUI: null,
	_bVideoDisplayedOnce: false,
	_VIDEO_OFF: 0, // Video is not displayed on the screen and video escalation is disabled
	_VIDEO_ON: 1, // Video is displayed on the screen and video escalation is enabled
	_VIDEO_HIDDEN: 2, // Video is not displayed on the screen but video escalation has been negociated

	/* EVENT LIST */
	_EVENT_CREATE_LOCAL: "ICTouchAPI.APIServices.Video.setLocalVideo",
	_EVENT_DELETE_LOCAL: "ICTouchAPI.APIServices.Video.deleteLocalVideo",
	_EVENT_WEBAPP_SHOW: "ICTouchAPI.transitionServices.WEBAPP_SHOW",
	_EVENT_WEBAPP_HIDE: "ICTouchAPI.transitionServices.WEBAPP_HIDE",
	_EVENT_TOASTER_SHOW: "toaster.show",
	_EVENT_TOASTER_HIDE: "toaster.hide",
	_EVENT_POPUP_SHOW: "dialogbox.show", // SHOW_POPUP_EVENT
	_EVENT_POPUP_HIDE: "dialogbox.hide", // HIDE_POPUP_EVENT


	/* ------------------------------------ Constructor --------------------------------------- */

	constructor : function () {

		this._videoDisplayed = this._VIDEO_OFF;

		this._webappShowStatus = 100;
		this._toasterShowStatus = 0;
		this._popupShowStatus = 0;

	},

	/* ----------------------------------- Getter / Setter------------------------------------- */

	/**
     * @public
	 * Because the eventServices wants an application name we provide one.
	 *
	 * @return {String} Return VideoServices
	 */
	getApplicationName: function() {
		return "VideoServices";
	},



	/* ----------------------------------- Public methods ------------------------------------- */


	/**
	 * Configure look and feel
	 *
	 * @param {Integer} background_color ARGB Widget background color
	 * @param {String} text main window displayed text when no video
	 * @param {Integer} text_color text ARGB color
	 * @param {String} pip_text PiP area displayed text when no video
	 */
	setParams: function( background_color, text, text_color, pip_text ) {
		ICTouchAPI.debugServices.info('ICTouchAPI.VideoServices - setParams');
		ICTouchAPI.APIServices.Video.setVideoMMIParams( background_color, text, text_color, pip_text );
	},


	/**
	 * @public
	 * Create and/or show local video if possible
	 *
	 * @param {Object} paramsObj contains x,y,w,h
	 */
	showLocal: function( paramsObj ) {
		ICTouchAPI.debugServices.info('ICTouchAPI.VideoServices - showLocal');
		return this.show( paramsObj, true );
	},


	/**
     * @public
	 * Create and/or show a video if possible (no video currently displayed)
	 *
	 * @param {Object} paramsObj contains x,y,w,h (callId, sipCallId, pip, ...)
	 * @param {Boolean} isLocal (optional) Set to TRUE if video is local. Default value is FALSE.
	 * Forced to FALSE if paramsObj.callId exists
	 * @param {Boolean} addOffset (optional) Set to FALSE if there is no need to calculate the absolute
	 * position of the iFrame that will be displayed behind the video. Default value is TRUE.
	 */
	show: function( paramsObj, isLocal, addOffset, _strOverrideCurrentScreen ) {
		ICTouchAPI.debugServices.info('ICTouchAPI.VideoServices - show');
		this._bVideoDisplayedOnce = true;
		var setResult = false;

		/* A video is already displayed */
		if(this._getVideoDisplayed() == this._VIDEO_ON) {
			/** @todo something to do in this case ? */
			/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - show / video is already displayed - callId: ' + paramsObj.callId);
		}

		else if( this._isVideoComActive() === true ) {
			if( this._isShowable() === true )
			{
				/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - show / video ready to be shown - callId: ' + paramsObj.callId);
						
				var current_screen_name;
				if(!_strOverrideCurrentScreen) {
					current_screen_name = this._getCurrentScreen();
				}
				else {
					current_screen_name = _strOverrideCurrentScreen;
				}
				/* Check if this webapp previously shown a video and restore it like it was */
				var state = this._getState(current_screen_name);

				if( state !== null )
				{
					/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - show / saved state found for '+ current_screen_name );

					if( state.isLocal === false ) {
						/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - show / showing video... (call showVideo)' );

						var setResult = ICTouchAPI.APIServices.Video.showVideo( state.params );

						if( setResult == true ) {
							this._postShow( current_screen_name, state.params, false );
						}
					}

				}

				else {
					/* DEBUG */  ICTouchAPI.debugServices.warning( 'ICTouchAPI.VideoServices - show / no saved state found for '+ current_screen_name );
				}

			}
			/** @todo something to do in this case ? */
		}

		/* No video is currently displayed - creating one */
		else
		{
			/* force isLocal to FALSE if a callId is defined */
			if( undefined !== paramsObj.callId ) {
				isLocal = false;
			} else if( undefined === isLocal ) {
				isLocal = true;
			}

			var current_screen_name;
			var displayedScreen;

			if(!_strOverrideCurrentScreen) {
				displayedScreen = this._getCurrentScreen();
			}
			else {
				displayedScreen = _strOverrideCurrentScreen;
			}
			
			if(paramsObj.callerScreen) {
				current_screen_name = paramsObj.callerScreen;
			}
			else {
				current_screen_name = displayedScreen;
			}			
			
			if(current_screen_name === displayedScreen) {
				//Get UI coordinate
				if(paramsObj.callerUI) {
					var UI = dijit.byId(paramsObj.callerUI);
					var params = UI.getStyle();
					if (params) {
						this.setParams( params.bgColor, params.mainMessage, params.textColor, params.pipMessage );
					}
					var objCoordinate = UI.getCoordinate();
					dojo.mixin(paramsObj, objCoordinate);
					UI = null;
					params = null;
				}
				
				/* Iframe offset if already defined - no need to calculate it */
				if( undefined !== addOffset && addOffset === false ) {					
				}
				/* Get offset of the iframe containing the UIElement */
				else {
					var frameOffset = this._getFrameOffset( current_screen_name );
					paramsObj.x += frameOffset[0];
					paramsObj.y += frameOffset[1];
				}

				/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - show / callId: ', paramsObj.callId );

				// LOCAL VIDEO
				if( isLocal === true ) {
					//Don't create video if it's reserved by an UI and the current UI isn't the right
					if(!this._strReservedVideoUI || this._strReservedVideoUI === paramsObj.callerUI ) {
						ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - show / call createLocalVideo');
						setResult = ICTouchAPI.APIServices.Video.createLocalVideo( paramsObj );
						this._strReservedVideoUI = null;
					}
					else {
						ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - show / do not call createLocalVideo');
						setResult = false;
					}
				}

				// VIDEO CALL
				else {
					if( undefined === paramsObj.pip ) {paramsObj.pip = 1;}
					if( undefined === paramsObj.mute ) {paramsObj.mute = 0;}

						//Don't create video if it's reserved by an UI and the current UI isn't the right
						if(!this._strReservedVideoUI || this._strReservedVideoUI === paramsObj.callerUI ) {
							ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - show / call createVideo');
							setResult = ICTouchAPI.APIServices.Video.createVideo( paramsObj );
							this._strReservedVideoUI = null;
						}
						else {
							ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - show / do not call call createVideo');
							setResult = false;
						}

					if( setResult === true ) {
						this._setVideoComActive(true);
					}
				}

				/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - show / setResult: ' + setResult);

				/* video has been created - set video states, register event listeners */
				if( setResult === true )  {
					this._postShow( current_screen_name, paramsObj, isLocal );
				}
			}

			//On first show, when target view isn't displayed
			else {
				this._bVideoDisplayedOnce = false;
				//Display the video on the next view open
				setResult = true;
				this._saveState( current_screen_name, {params: paramsObj, 'isLocal': isLocal, 'video': this._VIDEO_HIDDEN} );
				//Reserve the video
				this._strReservedVideoUI = paramsObj.callerUI;
				
				this._webappEventListenerHandle = this._registerEventListener(
					this._EVENT_WEBAPP_SHOW,
					function(evt) {
						
						/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - show / webapp first show event - _EVENT_WEBAPP_SHOW' );
						if( evt.id === current_screen_name ) {
							this._unregisterEventListener(this._webappEventListenerHandle);
							/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - show / webapp first show event - video ready to be shown' );
							//wait transitionServices update communication screen before call show function
							var context = this;
							setTimeout(function(){context.show(paramsObj, isLocal, addOffset)}, 0);							
						}
						else {
							/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - show / webapp first show event - default' );
						}						
					}	
				);
			}
		}

		if(paramsObj.callerUI) {
			if(setResult) {
				dijit.byId(paramsObj.callerUI).displayVideo();
			}
			else {
				dijit.byId(paramsObj.callerUI).displayError();
			}
		}
		return setResult;

	},


	/**
	 * @private
	 *
	 * @param {String} current_screen_name screen showing the video
	 * @param {Object} paramsObj contains x,y,w,h (callId, pip, ...)
	 * @param {Boolean} isLocal (optional) Set to TRUE if video is local. Default value is FALSE
	 * @param {String} closingToaster The name of the closing toaster, if any
	 *
	 */
	_postShow: function( current_screen_name, paramsObj, isLocal, closingToaster) {
		ICTouchAPI.debugServices.info('ICTouchAPI.VideoServices - _postShow');
		this._setVideoDisplayed(this._VIDEO_ON); // set video as displayed

		/* Considering that video is displayed, we can save video state and register listeners... */
		if( this._videoDisplayed )
		{
			this._saveState( current_screen_name, {params: paramsObj, 'isLocal': isLocal, 'video': this._VIDEO_ON} );

			/* Unregister previous existing listeners */
			this._unregisterShowHideListeners();

			/* Register webapp hide listener */
			this._webappEventListenerHandle = this._registerEventListener(
				this._EVENT_WEBAPP_HIDE, this._hideOnWebappHide
			);
			if (ICTouchAPI.toasterServices.getContentDisplayedId() && ICTouchAPI.toasterServices.getContentDisplayedId() != closingToaster) {
				/* Register toaster hide listener */
				this._hideOnToasterShow(ICTouchAPI.toasterServices.getContentDisplayedId());
			}
			else {
				/* Register toaster show listener */
				this._toasterEventListenerHandle = this._registerEventListener(
				this._EVENT_TOASTER_SHOW, this._hideOnToasterShow
			);
				}
			if (ICTouchAPI.popupServices.getNumberOfDisplayedPopups() > 0) {
				/* Register popup hide listener */
				this._hideOnPopupShow();
			}
			else {
				/* Register popup show listener */
				this._popupEventListenerHandle = this._registerEventListener(
					this._EVENT_POPUP_SHOW, this._hideOnPopupShow
				);
			}
			/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _postShow / video shown in ' + current_screen_name );
		}

	},

	/**
     * @public
	 * Hide current video and automatically re-show it if requested (autoRestoreVideo = TRUE).
	 *
	 * @param {Object} paramsObj
	 * @param {Boolean} autoRestoreVideo define if video can be restored automatically. Default value is FALSE.
	 * @param {String} strCurrentScreen The current screen name (optionnal)
	 */
	hide: function(paramsObj, autoRestoreVideo, strCurrentScreen) {
		ICTouchAPI.debugServices.info('ICTouchAPI.VideoServices - hide');
		if (typeof(strCurrentScreen) == 'undefined') {
			strCurrentScreen = this._getCurrentScreen();
		}

		if (paramsObj !== undefined) {
		    /* If paramsObj.videoStableState is set, then associate the videoStableState to the callId */
		    /* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - hide / callId: ' + paramsObj.callId);
		    var objState = (undefined !== paramsObj.callId) ? this._getStateByCallId(paramsObj.callId) : this._getState(strCurrentScreen);

		    if(objState !== null) {
				/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - hide / video: ' + paramsObj.videoStableState);
				objState.videoStableState = (undefined !== paramsObj.videoStableState) ? paramsObj.videoStableState : this._VIDEO_ON; // Save the stable video state
				this._saveState(strCurrentScreen,objState);
		    }
		}

		// if a mask has been set, disable it
		if (this._maskSetStatus === true) {
			ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - hide / call disableMask');
			this._maskSetStatus = !ICTouchAPI.APIServices.Video.disableMask( 1 );
		}
		if(this._getVideoDisplayed() == this._VIDEO_ON) {
			var hideResult = false,
			stateObj = this._getState( strCurrentScreen );

			/* stateObj will be used to define which type of video is displayed (local or call) */
			if( stateObj !== null ) {
				// LOCAL VIDEO
				if( stateObj.isLocal !== undefined && stateObj.isLocal === true ) {
				    	/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - hide / LOCAL VIDEO: call deleteLocalVideo');
					hideResult = ICTouchAPI.APIServices.Video.deleteLocalVideo();
				}

				// VIDEO CALL
				else {
					/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - hide / VIDEO CALL: call hideVideo with callId ' + stateObj.params.callId);
					hideResult = ICTouchAPI.APIServices.Video.hideVideo( stateObj.params.callId );
				}
			}

			/* video has been destroyed/hide */
			if( hideResult === true )
			{
				this._setVideoDisplayed(this._VIDEO_HIDDEN); // set video as not displayed

				/* Unregister previous existing listeners if auto restore not requested */
				if( undefined === autoRestoreVideo || autoRestoreVideo === false ) {
					this._unregisterShowHideListeners();
				}

				/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - hide / video hide success' );
			}

			else {
				/* DEBUG */ ICTouchAPI.debugServices.error( 'ICTouchAPI.VideoServices - hide / video hide error' );
			}
		} else {
			/* DEBUG */ ICTouchAPI.debugServices.warning( 'ICTouchAPI.VideoServices - hide / no video is currently displayed' );
			if( (undefined === autoRestoreVideo || autoRestoreVideo === false) && (objState && objState.videoStableState == this._VIDEO_HIDDEN )) {
                                this._unregisterShowHideListeners();
		}
		}

	},




	/**
     * @public
	 * Delete current displayed video
	 * @param {Number} callId The callId if any, undefined else
	 * @param {String} strScreen Optional, if undefined is proved in callId, the name of the screen associated to the video to destroy. If undefined, the currentScreen is used.
	 */
	destroy: function( callId, strScreen ) {
		ICTouchAPI.debugServices.info('ICTouchAPI.VideoServices - destroy');
		ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - destroy $ input function params | callId: ' + callId + ', strScreen:' + strScreen);
		if((this._getVideoDisplayed() == this._VIDEO_ON) || (this._getVideoDisplayed() == this._VIDEO_HIDDEN) || undefined !== callId || undefined != strScreen) {
			this._strReservedVideoUI = null;
			var destroyResult = false, stateObj = null, strStateId = null;
			if (undefined !== callId) {
				stateObj = this._getStateByCallId( callId );
				if (stateObj && stateObj.params) {
					strStateId = stateObj.params.callerScreen;
				}
			}
			else if (undefined !== strScreen) {
				stateObj = this._getState( strScreen );
				strStateId = strScreen;
			}
			else {
				stateObj = this._getState( this._getCurrentScreen() );
				strStateId = this._getCurrentScreen();
			}

			/* stateObj will be used to define which type of video is displayed (local or call) */
			if( stateObj !== null )
			{
				/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - destroy / state object found' );

				// VIDEO CALL
				if( stateObj.params.callId !== undefined ) {
					if(this._bVideoDisplayedOnce){
						/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - destroy / calling deleteVideo with callId '+ stateObj.params.callId );
						destroyResult = ICTouchAPI.APIServices.Video.deleteVideo( stateObj.params.callId );
					}
					if( destroyResult === true ) {
						this._setVideoComActive(false);
					}
				}

				// LOCAL VIDEO
				else {
					/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - destroy / calling deleteLocalVideo' );
					destroyResult = ICTouchAPI.APIServices.Video.deleteLocalVideo();
				}
			}

			/* video has been destroyed */
			if( destroyResult === true )
			{
				this._setVideoDisplayed(this._VIDEO_OFF); // set video as not displayed

				/* Unregister previous existing listeners */
				this._unregisterShowHideListeners();

				if( stateObj.params.callId !== undefined ) {
					// Destroy all the states with this callId
					this._deleteStateByCallId(stateObj.params.callId);
				}
				else {
					// Destroy the state by screen id
					this._deleteState(strStateId);
				}

				/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - destroy / video destroy success' );
			}

			else {
				/* DEBUG */  ICTouchAPI.debugServices.error( 'ICTouchAPI.VideoServices - destroy / video destroy error' );
			}
		}

		else {
			/* DEBUG */  ICTouchAPI.debugServices.warning( 'ICTouchAPI.VideoServices - destroy / no video is currently displayed' );
		}

	},




	/* ----------------------------------- Privates methods ------------------------------------- */

	/**
	 * @private
	 * For debug
	 */
	dump : function(boolAdvancedOption) {
		this.inherited(arguments);

		ICTouchAPI.debugServices.dump("Is video displayed (this._getVideoDisplayed()) (0: _VIDEO_OFF, 1: _VIDEO_ON, 2: _VIDEO_HIDDEN)? " + this._getVideoDisplayed());
		ICTouchAPI.debugServices.dump("Is video com active (displayed or not) (this._isVideoComActive())? " + this._isVideoComActive());
		ICTouchAPI.debugServices.dump("Shown status (this._getShowStatus()): " + this._getShowStatus());
		ICTouchAPI.debugServices.dump("Is video showable (i.e. if no content will be overlapped by the video area) (this._isShowable())? " + this._isShowable());

		if(boolAdvancedOption){
			ICTouchAPI.debugServices.dump("Events currently registered:");
			for (var evtName in this._eventListenerHandleList) {
				ICTouchAPI.debugServices.dump('Name of the event: '+ evtName);
				ICTouchAPI.debugServices.dump('Number of events registered for '+ evtName + ': ' + this._eventListenerHandleList[evtName].length);
				for(var i = 0; i < this._eventListenerHandleList[evtName].length; i++) {
					ICTouchAPI.debugServices.dump('Handler of events registered for '+ evtName + ', position ' + i + ': ' + this._eventListenerHandleList[evtName][i]);
				}
			}
			ICTouchAPI.debugServices.dump("Shown status:");
			ICTouchAPI.debugServices.dump(" - Webapp shown status (this._webappShowStatus): " + this._webappShowStatus);
			ICTouchAPI.debugServices.dump(" - Toaster shown status (this._toasterShowStatus): " + this._toasterShowStatus);
			ICTouchAPI.debugServices.dump(" - Popup shown status (this._popupShowStatus): " + this._popupShowStatus);

			ICTouchAPI.debugServices.dump("Registered events handlers:");
			ICTouchAPI.debugServices.dump(" - Webapp event listener (this._webappEventListenerHandle): " + this._webappEventListenerHandle);
			ICTouchAPI.debugServices.dump(" - Toaster event listener (this._toasterEventListenerHandle): " + this._toasterEventListenerHandle);
			ICTouchAPI.debugServices.dump(" - Popup event listener (this._popupEventListenerHandle): " + this._popupEventListenerHandle);

			ICTouchAPI.debugServices.dump("Is a Video mask set (this._maskSetStatus)? " + this._maskSetStatus);
			ICTouchAPI.debugServices.dump("_strReservedVideoUI: " + this._strReservedVideoUI);
			ICTouchAPI.debugServices.dump("_bVideoDisplayedOnce: " + this._bVideoDisplayedOnce);

			ICTouchAPI.debugServices.dump("_statesList (Object used to store video details (position, size)):");
			for (var state in this._statesList) {
				ICTouchAPI.debugServices.dump('Webapp id of the state: '+ state);
				ICTouchAPI.debugServices.dump('Params of the state: '+ dojo.toJson(this._statesList[state].params));
				ICTouchAPI.debugServices.dump('Is local video state? '+ this._statesList[state].isLocal);
				ICTouchAPI.debugServices.dump('Video state (0: _VIDEO_OFF, 1: _VIDEO_ON, 2: _VIDEO_HIDDEN): '+ this._statesList[state].video);
			}
		}
	},

	/**
     * @private
	 * Clear all variables of VideoServices (defense mechanism called by webapp communication when there is no ongoing call anymore)
	 */
	cleanVideoServices: function() {
		ICTouchAPI.debugServices.info('ICTouchAPI.VideoServices - cleanVideoServices');
		this._unregisterShowHideListeners();
		var stateObj, destroyResult;
		for (var screen_name in this._statesList) {
			stateObj = this._statesList[screen_name];
			/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - cleanVideoServices / state object for screen_name: ' + screen_name);
			if (stateObj)	{
				/* DEBUG */ ICTouchAPI.debugServices.warning('ICTouchAPI.VideoServices - cleanVideoServices / state object is defined for screen_name ' + screen_name + ': ' + dojo.toJson(stateObj) + ', delete it!');

				if (stateObj.params.callId !== undefined) {
					/* DEBUG */ ICTouchAPI.debugServices.warning('ICTouchAPI.VideoServices - cleanVideoServices / call deleteVideo for callId: ' + stateObj.params.callId + ' (this._getVideoDisplayed(): ' + this._getVideoDisplayed() + ')');
					destroyResult = ICTouchAPI.APIServices.Video.deleteVideo( stateObj.params.callId );
				}

				// LOCAL VIDEO
				else if (stateObj.isLocal) {
					/* DEBUG */ ICTouchAPI.debugServices.warning('ICTouchAPI.VideoServices - cleanVideoServices / state object is local, call deleteLocalVideo');
					destroyResult = ICTouchAPI.APIServices.Video.deleteLocalVideo();
				}
				/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - cleanVideoServices / destroyResult: ' + destroyResult);
			}
			this._deleteState(screen_name);
		}
		if (this._getVideoDisplayed() !== this._VIDEO_OFF) {
			this._setVideoDisplayed(this._VIDEO_OFF);
		}
		if (this._isVideoComActive()) {
			this._setVideoComActive(false);
		}
		this._setWebappShown(true);
		this._setPopupShown(false);
		this._setToasterShown(false);

		this._maskSetStatus = false;
		this._strReservedVideoUI = null;
		this._bVideoDisplayedOnce = false;
	},


	/**
	 * @private
	 * Return a boolean that define if video can be shown. In other words, verify if no content
	 * will be overlapped by the video area by checking the _getShowStatus() result.
	 *
	 * @return {Boolean} isShowable
	 */
	_isShowable: function() {

		var status = this._getShowStatus();
		/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _isShowable / status is: ' + status + ", return: " + (status == 100));

		return (status == 100) ? true : false;
	},

	/**
	 * @private
	 * Return a global status code for webapp, toaster & popup visibility.
	 * Video can be shown only if the returned value equals 100 (one hundred)
	 *
	 * This value is calculated with:
	 *  - _webappShowStatus (100 = webapp VISIBLE, 0 = webapp HIDDEN)
	 *  - _toasterShowStatus  (0 = toaster HIDDEN, 10 = toaster VISIBLE)
	 *  - _popupShowStatus (0 = popup HIDDEN, 1 = popup VISIBLE)
	 *
	 * For example, if webapp is showing a video and a toaster is showing up (keyboard),
	 * _toasterShowStatus will change from 0 to 10. So, global status will be 110 (100 + 10 + 0)
	 *
	 * @return {Integer} Can be equal from 0 to 111
	 */
	_getShowStatus: function() {
		return this._webappShowStatus + this._toasterShowStatus + this._popupShowStatus;
	},

	/**
	 * @private
	 * Set Webapp show status
	 * @param {Boolean} isShown set to TRUE if current webapp is visible.
	 */
	_setWebappShown: function( isShown ) {
		/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _setWebappShown / _webappShowStatus was: ' + this._webappShowStatus + ', it is now: ' + ((isShown) ? 100 : 0));
		this._webappShowStatus = (isShown) ? 100 : 0;
	},

	/**
	 * @private
	 * Set Toaster show status
	 * @param {Boolean} isShown set to TRUE if a toaster is visible.
	 */
	_setToasterShown: function( isShown ) {
		/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _setToasterShown / _toasterShowStatus was: ' + this._toasterShowStatus + ', it is now: ' + ((isShown) ? 10 : 0));
		this._toasterShowStatus = (isShown) ? 10 : 0;
	},

	/**
	 * @private
	 * Set Popup show status
	 * @param {Boolean} isShown set to TRUE if a popup is visible.
	 */
	_setPopupShown: function( isShown ) {
		/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _setPopupShown / _popupShowStatus was: ' + this._popupShowStatus + ', it is now: ' + ((isShown) ? 1 : 0));
		this._popupShowStatus = (isShown) ? 1 : 0;
	},



	/**
	 * @private
	 * Webapp hide event. Handle restore video visibility when webapp is shown again with the same context.
	 */
	_hideOnWebappHide: function() {

		/* DEBUG */ ICTouchAPI.debugServices.info( 'ICTouchAPI.VideoServices - _hideOnWebappHide');

		/* Hide video */
		this.hide( true, undefined );

		/* Set webapp visibility status to HIDDEN */
		this._setWebappShown( false );

		var current_screen_name = this._getCurrentScreen();

		/* Unregister previous listener */
		if( this._webappEventListenerHandle !== null ) {
			this._unregisterEventListener( this._webappEventListenerHandle );
			this._webappEventListenerHandle = null;
		}

		/* Register webapp show event - in order to reshow video previously hidden */
		this._webappEventListenerHandle = this._registerEventListener(
			this._EVENT_WEBAPP_SHOW,
			function(evt) {
				/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnWebappHide / webapp show event - _EVENT_WEBAPP_SHOW' );

				if( evt.id === current_screen_name )
				{
					/* Set webapp show status */
					this._setWebappShown( true );

					if( this._isShowable() === true )
					{
						/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnWebappHide / webapp show event - video ready to be shown' );

						/* Check if this webapp previously shown a video and restore it like it was */
						var state = this._getState(current_screen_name);

						if( state !== null )
						{
							/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnWebappHide / webapp show event - saved state found for '+ current_screen_name );

							// LOCAL VIDEO - need to create a new one
							if( state.isLocal === true ) {
								/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnWebappHide / webapp show event - showing LOCAL video...' );

								this.show( state.params, state.isLocal, true, evt.id );
							}

							// VIDEO CALL
							else {
								/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnWebappHide / webapp show event - showing video...' );

								var setResult = ICTouchAPI.APIServices.Video.showVideo( state.params );

								if( setResult == true ) {
									this._postShow( current_screen_name, state.params, false );
								}
							}

						}

						else {
							/* DEBUG */  ICTouchAPI.debugServices.warning( 'ICTouchAPI.VideoServices - _hideOnWebappHide / webapp show event - no saved state found for '+ current_screen_name );
						}

					}

					else {
						/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnWebappHide / webapp show event - screen name is different '+ evt.id + ' != ' + current_screen_name );
					}

				}

				else {
					/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnWebappHide / webapp show event - default' );
				}

			}

		);

	},


	/**
	 * @private
	 * Toaster show event. Handle restore video visibility (or mask) when toaster is hidden.
	 * @param {String} evt toaster id
	 */
	_hideOnToasterShow: function(evt) {
		/* DEBUG */ ICTouchAPI.debugServices.info('ICTouchAPI.VideoServices - _hideOnToasterShow');
		/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - _hideOnToasterShow $ input function params | evt ' + evt);


		var maskId = 1, maskWidth = 800, maskHeight = 166;


		/* Create a mask for the audio toaster */
		if( evt === this._audioToasterId ) {
			/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnToasterShow / create a mask for the audio toaster ');
			this._maskSetStatus = ICTouchAPI.APIServices.Video.enableMask( maskId, 0, 315, (maskWidth & 0xFFF8), (maskHeight & 0xFFFE) ); // mask w calculated overlap: 0, 210, 490, 62
		}
		/* or simply hide the video */
		else {
			/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnToasterShow / simply hide the video');
			this.hide(true,null, ICTouchAPI.transitionServices.getNextOrCurrentScreenName());
			/* Set toaster visibility status to VISIBLE */
			this._setToasterShown( true );
		}
		// on an even other than webapp.hide, assume that the screen is already loaded
		var current_screen_name = ICTouchAPI.transitionServices.getNextOrCurrentScreenName();
		

		/* Unregister previous listener */
		if( this._toasterEventListenerHandle !== null ) {
			this._unregisterEventListener( this._toasterEventListenerHandle );
			this._toasterEventListenerHandle = null;
		}

		this._toasterEventListenerHandle = this._registerEventListener(
			this._EVENT_TOASTER_HIDE,
			function( evt2 ) {
				/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnToasterShow / toaster hide event - id '+ evt2 );

				/* Set toaster show status */
				this._setToasterShown( false );

				if( this._isShowable() === true && this._getCurrentScreen() === current_screen_name ) {
					if( evt2 === this._audioToasterId ) {
						/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnToasterShow / toaster hide event - audio toaster');
						ICTouchAPI.APIServices.Video.disableMask( maskId );
						this._maskSetStatus = false;

						/* Register toaster show listener */
						this._toasterEventListenerHandle = this._registerEventListener(
							this._EVENT_TOASTER_SHOW, this._hideOnToasterShow
						);
					}

					else
					{
						/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - _hideOnToasterShow / toaster hide event - dialer or popup');
						/* Check if this webapp previously shown a video and restore it like it was */
						var state = this._getState(current_screen_name);

						if( state !== null )
						{
							// LOCAL VIDEO - need to create a new one
							if( state.isLocal === true ) {
							    	/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnToasterShow / toaster hide event - LOCAL VIDEO');
								this.show( state.params, state.isLocal );
							}

							// VIDEO CALL
							else {
								/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnToasterShow / toaster hide event - VIDEO CALL');
								/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnToasterShow / toaster hide event - state.videoStableState: ' + state.videoStableState);
								if (this._getVideoDisplayed() == this._VIDEO_HIDDEN && (undefined !== state.videoStableState && state.videoStableState != this._VIDEO_HIDDEN)) {
								    var setResult = ICTouchAPI.APIServices.Video.showVideo( state.params );

								    if( setResult == true ) {
								    	this._postShow( current_screen_name, state.params, false, evt2 );
								    }
								} else {
								    /* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnToasterShow / toaster hide event - video call is hidden');
								}
							}
						}
					}
				} else {
				    /* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnToasterShow / toaster hide event - hum... it\'s seems there is a problem when the toaster disappears');
				}
			}
		);

	},


	/**
	 * @private
	 * Popup show event. Handle restore video visibility when popup is hidden.
	 */
	_hideOnPopupShow: function() {

		/* DEBUG */ ICTouchAPI.debugServices.info( 'ICTouchAPI.VideoServices - _hideOnPopupShow');

		/* Hide video */
		this.hide( true, undefined, ICTouchAPI.transitionServices.getNextOrCurrentScreenName() );

		//Webapp can be hidden when a popup is displayed
		this._webappEventListenerHandle = this._registerEventListener(
			this._EVENT_WEBAPP_HIDE, this._hideOnWebappHide
		);

		/* Set popup visibility status to VISIBLE */
		this._setPopupShown( true );

		var current_screen_name = this._getCurrentScreen();

		/* Unregister previous listener */
		if( this._popupEventListenerHandle !== null ) {
			this._unregisterEventListener( this._popupEventListenerHandle );
			this._popupEventListenerHandle = null;
		}

		this._popupEventListenerHandle = this._registerEventListener(
			this._EVENT_POPUP_HIDE,
			function() {
				/* DEBUG */ ICTouchAPI.debugServices.info( "ICTouchAPI.VideoServices - _hideOnPopupShow / popup hide event" );

				/* Set popup show status */
				this._setPopupShown( false );

				/* Current screen is the one that contains the video */
				if( this._isShowable() === true && this._getCurrentScreen() === current_screen_name )
				{
					var state = this._getState(current_screen_name);

					if( state !== null ) {
						this._setPopupShown( false );

						// LOCAL VIDEO - need to create a new one
						if( state.isLocal === true ) {
							this.show( state.params, state.isLocal );
						}

						// VIDEO CALL
						else {
							var setResult = ICTouchAPI.APIServices.Video.showVideo( state.params );

							if( setResult == true ) {
								this._postShow( current_screen_name, state.params, false );
							}

						}

					}
				}
			}
		);

	},


	/**
	 * @private
	 * Used to save a state object that contains video details (local, coordinates, sizes) specific to a webapp (id)
	 * @param {String} id webapp id
	 * @param {Object} stateObj contains the state details.
	 */
	_saveState: function( id, stateObj ) {
		/* DEBUG */ ICTouchAPI.debugServices.info('ICTouchAPI.VideoServices - _saveState');
		/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - _saveState $ input function params | id: ' + id + ', stateObj: ' + dojo.toJson(stateObj));
		this._statesList[ id ] = stateObj;
	},

	/**
	 * @private
	 * Return a state object saved at 'id' OR false if nothing exists
	 * @param {String} id webapp id
	 * @return {Object} stateObj or FALSE
	 */
	_getState: function( id ) {
		/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - _getState $ input function params | id: ' + id);

		var stateObj = this._statesList[ id ];
		//console.log( stateObj );
		if( undefined === stateObj ) {
			/* DEBUG */ ICTouchAPI.debugServices.warning('ICTouchAPI.VideoServices - _getState / state no found!');
			stateObj = null;
		}

		/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - _getState $ return stateObj: ' + dojo.toJson(stateObj));
		return stateObj;
	},

	/**
	 * @private
	 * Delete a state object saved at 'id' OR false if nothing exists
	 * @param {String} id webapp id
	 * @return {Boolean} True if no error occurs 
	 */
	_deleteState: function( id ) {
		/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _deleteState $ input function params | id: ' + id );

		var stateObj = this._statesList[ id ];
		//console.log( stateObj );
		if( undefined === stateObj ) {
			/* DEBUG */ ICTouchAPI.debugServices.warning( 'ICTouchAPI.VideoServices - _deleteState / state no found!' );
			return false;
		}
		delete this._statesList[ id ];
		stateObj = null;
		return true;
	},

	/**
	 * @private
	 * Delete all states with a given callId OR false if nothing exists
	 * @param {String} callId
	 */
	_deleteStateByCallId : function(callId) {
		ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _deleteStateByCallId $ input function params | callId: '+ callId );
		var nbDeleted = 0;

		for(var screen_name in this._statesList){
			if( undefined !== this._statesList[ screen_name ].params.callId
				&& this._statesList[ screen_name ].params.callId === callId ) {

				ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _deleteStateByCallId $ delete state with callId: '+ callId + " and with screenName: "+screen_name);
				this._deleteState(screen_name);
			}

			nbDeleted++;
		}
		ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _deleteStateByCallId $ number of deleted states: '+ nbDeleted );
	},

	_getStateByCallId: function( callId ) {
		/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _getStateByCallId $ input function params | callId: '+ callId );

		var stateObj = null;

		for( var screen_name in this._statesList ) {

			if( undefined !== this._statesList[ screen_name ].params.callId
				&& this._statesList[ screen_name ].params.callId === callId ) {
				stateObj = this._statesList[ screen_name ];
				/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _getStateByCallId / state found for callId ' + callId + ' with screen name '+ screen_name );
				break;
			}

		}

		/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - _getStateByCallId $ return stateObj: ' + dojo.toJson(stateObj));
		return stateObj;
	},


	/**
	 * @private
	 * Used to define if a video is currently displayed.
	 * @param {Boolean} videoState set to TRUE if a video is displayed or FALSE if no video is visible.
	 */
	_setVideoDisplayed: function( videoState ) {

		if( undefined !== videoState ) {
			/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _setVideoDisplayed / _videoDisplayed was: ' + this._videoDisplayed + ', it is now: ' + videoState);
			this._videoDisplayed = videoState;
		}
	},

	/**
	 * @private
	 * Return a state object saved at 'id' OR false if nothing exists
	 * @param
	 * @return {Integer} _VIDEO_OFF; _VIDEO_ON; _VIDEO_HIDDEN
	 */

	_getVideoDisplayed: function() {
	    /* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _getVideoDisplayed $ return _videoDisplayed (0: _VIDEO_OFF, 1: _VIDEO_ON, 2: _VIDEO_HIDDEN): ' + this._videoDisplayed);
	    return this._videoDisplayed;
	},


	/**
	 * @private
	 * Used to define if a video communication is currently active.
	 * @param {Boolean} isActive set to TRUE if a video communication is active else FALSE.
	 */
	_setVideoComActive : function( isActive ) {

		if( undefined !== isActive ) {
			/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _setVideoComActive / _videoComActive was: ' + this._videoComActive + ', it is now: ' + isActive);
			this._videoComActive = new Boolean( isActive ).valueOf();
		}

	},

	/**
	 * @private
	 * Returns TRUE if a video communication is currently active (displayed or not).
	 * @return {Boolean} _videoComActive
	 */
	_isVideoComActive: function() {
	    /* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _isVideoComActive $ return _videoComActive: ' + this._videoComActive);
		return this._videoComActive;
	},


	//
	// HELPERS
	//

	/**
	 * @private
	 * @return {String} current webapp screen name
	 */
	_getCurrentScreen: function() {
	    /* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _getCurrentScreen $ return ' + ICTouchAPI.transitionServices.getCurrentScreenName());
		return ICTouchAPI.transitionServices.getCurrentScreenName();
	},


	/**
	 * Unregister all listeners used to catch show/hide events (webapp, toaster & popup)
	 * @private
	 */
	_unregisterShowHideListeners: function() {

		/* DEBUG */ ICTouchAPI.debugServices.info( 'ICTouchAPI.VideoServices - _unregisterShowHideListeners');

		/* unregister all previous listeners for each events */
		for( var evtName in this._eventListenerHandleList ) {
			/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _unregisterShowHideListeners / unregister all listeners for: '+ evtName );

			for( var i = 0, ln = this._eventListenerHandleList[evtName].length; i < ln; i++ ) {
				this._unregisterEventListener( this._eventListenerHandleList[evtName][i] );
			}

			delete( this._eventListenerHandleList[evtName] );
		}

		/* reset handle */
		this._webappEventListenerHandle = null;
		this._toasterEventListenerHandle = null;
		this._popupEventListenerHandle = null;

		/* reset visibility variables*/
/*
		this._videoDisplayed = false;
*/
		this._webappShowStatus = 100;
		this._toasterShowStatus = 0;
		this._popupShowStatus = 0;
	},

	//
	// EVENT LISTENERS
	//

	/**
	 * Event listener registering helper. Automatically set the context to 'this'.
	 * @param {String} evtName name of event to listen
	 * @param {Function} c_func function reference called when the event will fire
	 * @return {Array} Dojo subscribe handle used to unsubscribe.
	 */
	_registerEventListener: function( evtName, c_func ) {
		/* DEBUG */ ICTouchAPI.debugServices.info('ICTouchAPI.VideoServices - _registerEventListener');
		/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - _registerEventListener / registering ' + evtName);

		/* Register listener */
		var handle = dojo.subscribe( evtName, this, c_func );

		/* create handle list for this event if it does not already exists */
		if( undefined === this._eventListenerHandleList[ evtName ] ) {
			this._eventListenerHandleList[ evtName ] = [];
		}

		/* push handle in the list */
		if(this._eventListenerHandleList[evtName][0]){
			this._unregisterEventListener(this._eventListenerHandleList[evtName][0]);
			delete(this._eventListenerHandleList[evtName][0]);
		}
		this._eventListenerHandleList[evtName][0] = handle;

		/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - _registerEventListener $ return handle: ' + handle);
		return handle;
	},

	/**
	 * Event listener unregistering helper.
	 * @param {Array} handle dojo subscribe handle.
	 */
	_unregisterEventListener: function( handle ) {
		/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _unregisterEventListener / unregistering '+ handle.join(" ") );
		dojo.unsubscribe( handle );
	},



	//
	// OFFSET CALCULATION
	//

	/**
	 * @private
	 * Return the active frame offset
	 * @param {String} frameId iframe DOM id.
	 */
	_getFrameOffset: function( frameId ) {
		/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - _getFrameOffset $ input function params | frameId: ' + frameId);

		/* */
		var frameOffset = [ 0, 0 ],
			frameObj = null;
			frameObj = window.document.getElementById( frameId );

		if( null !== frameObj ) {
			frameOffset = this._getObjectOffset( frameObj );
		}

		/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - _getFrameOffset $ return frameOffset: ' + dojo.toJson(frameOffset));
		return frameOffset;
	},

	/**
	 * Calculate the offset for a DOM element.
	 * @param {Object} obj DOM object element
	 */
	_getObjectOffset: function( obj ) {
		var curtop, curleft;
		curleft = curtop = 0;

		/* get object relative offset */
	    if(obj.offsetParent) {

		    do {
		    	curleft += obj.offsetLeft;
		    	curtop += obj.offsetTop;

		    } while (obj = obj.offsetParent);

	    }

	    if( curleft < 0 ) {curleft = 0;}
	    if( curtop < 0 ) {curtop = 0;}

	    return [ curleft, curtop ];

	}


});

ICTouchAPI.VideoServices = new ICTouchAPI.VideoServices();
/**
* @class ICTouchAPI.supervisionServices
* @extends Object
* @singleton
 * supervisionServices provides an interface to web applications to manage the supervision.<br />
 * Note: only one supervision group is managed
 * <br />
 * The service publishes the following events (using dojo.publish):<br />
 *	- <b>supervision.groupAdded:</b> notify webapps that the user has been added to a supervision group.<br />
 *	Provide a string indicating the name of the group and an array of supervised users member of this group.<br />
 *	- <b>supervision.groupModified:</b> notify webapps that the group the user is in has been modified.
 *	Provide a string indicating the name of the group, an array of added users, an array of modified users and an array of users deleted from this group.
 *	The array of users deleted only contains the login of these users.<br />
 *	- <b>supervision.groupDeleted:</b> notify webapps that the user has been removed from the group.<br />
 *	- <b>supervision.incomingCallsNumberChanged:</b> notify that the number of incoming calls on supervised users changed. Provide this number.<br />
 *	- <b>supervision.hiddenStateChanged:</b> notify that the hidden state of supervision changed. Provide this state as a boolean.<br />
*/
dojo.provide("ICTouchAPI.supervisionServices");
dojo.declare("ICTouchAPI.supervisionServices", null, {
	
		/* --------------------------------- Public attributes ------------------------------------ */



		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		* @property
		* @type Object
		* @ignore
		*/
	   _arrMembers : [],

		/**
		* @property
		* @type string
		* @ignore
		*/
	   _strGroupName : "",

		/**
		* @property
		* @type number
		* @ignore
		*/
		_intIncomingCallsNb : 0,

		/**
		 * @property
		 * @type number
		 * @ignore
		 */
		_intNbMaxMembers : 15,

		/**
		 * @property
		 * @type boolean
		 * @ignore
		 */
		_boolSupervisionHidden : false,

		/* ------------------------------------ Constructor --------------------------------------- */

		/**
		* The supervision services manage all the supervision feature
		* @ignore
		*/
		constructor: function() {
			// Subscribe to Telephony events
			ICTouchAPI.eventServices.subscribeToEvent(this, "supGroupAdded", this._onSupGroupAdded);
			ICTouchAPI.eventServices.subscribeToEvent(this, "supGroupModified", this._onSupGroupModified);
			ICTouchAPI.eventServices.subscribeToEvent(this, "supGroupDeleted", this._onSupGroupDeleted);
			ICTouchAPI.eventServices.subscribeToEvent(this, "supGroupHidden", this._onSupGroupHidden);
			ICTouchAPI.eventServices.subscribeToEvent(this, "supUpdated", this._onSupUpdated);

			dojo.subscribe("HomepageLoaded", this, function() {
				if(ICTouchAPI.remoteDisplay){
					ICTouchAPI.APIServices.Telephony.getSupervisionCurrentMembers({context:this, callback:this._initGroup});
				}
			});
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * @ignore
		 * Because the eventServices wants an application name we provide one.
		 * @return {String} Return supervisionServices
		 */
		getApplicationName: function(){
			return "supervisionServices";
		},

		/**
		 * Return the maximum number of supervised users
		 * @return {Number} The maximum number of supervised users
		 */
		getMaxMembers : function() {
			return this._intNbMaxMembers;
		},

		/**
		 * Return the supervision hidden state.
		 * @return {Boolean} The supervision hidden state
		 */
		getSupervisionHidden : function() {
			return this._boolSupervisionHidden;
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		 * Return a user according to a login
		 * @param {String} strLogin is the unique login of the user you want to get
		 * @return {Object} The user corresponding to the given login (null if not found)
		 */
		getSupUserByLogin : function(strLogin) {
			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - getSupUserByLogin / Begin");
			for(var j = 0 ; j < this._arrMembers.length ; j++){
				if(this._arrMembers[j].identity.login == strLogin){
					return this._arrMembers[j];
				}
			}

			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - getSupUserByLogin / end");

			return null;
		},

		/**
		 * Return an array of users
		 * @param {number} intFrom the first index of the array
		 * @param {number} intTo the last index of the array
		 * @return {Object} The array of users in the given range
		 */
		getSupUsers : function(intFrom, intTo) {
			return this._arrMembers.slice(intFrom, intTo);
		},

		/**
		 * Run an action (call or pickup) on a user accoring to its ringing state
		 * @param {string} strLogin is the unique login of the user you want to pickup
		 */
		callPickupByLogin : function(strLogin) {
			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - callPickupByLogin / Begin");
			var objMember = null;

			// Check if the strLogin is valid
			for(var i = 0 ; i < this._arrMembers.length ; i++){
				if(this._arrMembers[i].identity.login == strLogin){
					objMember = this._arrMembers[i];
					break;
				}
			}

			// Pickup the call if there is an incoming call on the member, call him otherwise
			if(objMember){
				var supervisedDialableNumber = ICTouchAPI.tools.getContactPhone(objMember.identity, false);
				if(objMember.ringState && objMember.firstIncomingCall){
					ICTouchAPI.debugServices.debug("ICTouchAPI.supervisionServices - callPickupByLogin / Start pickup on login " + strLogin + " with callRef " + objMember.firstIncomingCall.callRef);
					ICTouchAPI.APIServices.Telephony.pickupCall({params:[objMember.firstIncomingCall.callRef, supervisedDialableNumber]});
				} else {
					ICTouchAPI.debugServices.debug("ICTouchAPI.supervisionServices - callPickupByLogin / Start call on login : " + strLogin);
					ICTouchAPI.APIServices.Telephony.startPhoneCall({params:[supervisedDialableNumber, false, false, false]});
				}
			} else {
				ICTouchAPI.debugServices.warning("ICTouchAPI.supervisionServices - callPickupByLogin / Login of the user not found : " + strLogin);
			}

			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - callPickupByLogin / end");
			return null;
		},

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 */
		_setSupervisionHidden : function(boolSupervisionHiddenState) {
			if(this._boolSupervisionHidden !== boolSupervisionHiddenState){
				this._boolSupervisionHidden = boolSupervisionHiddenState;
				dojo.publish("supervision.hiddenStateChanged", [boolSupervisionHiddenState]);
			}
		},
		
		/**
		 * @ignore
		 */
		_onSupGroupAdded : function() {
			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - _onSupGroupAdded / Begin");

			this._setSupervisionHidden(false);

			var objSupGroupAddedArguments = ICTouchAPI.tools.getEventArguments(arguments);
			var groupName = objSupGroupAddedArguments.groupName;
			var arrMembers = objSupGroupAddedArguments.listAddedMembers;

			this._initGroup(arrMembers, groupName);

			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - _onSupGroupAdded / End");
		},

		/**
		 * @ignore
		 */
		_initGroup : function(arrMembers, strGroupName) {
			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - _initGroup / Begin");


			for(var i =0 ; i < arrMembers.length ; i++){
				ICTouchAPI.debugServices.debug("ICTouchAPI.supervisionServices - _initGroup / addedMember "+ i + " : " + dojo.toJson(arrMembers[i]) );
				this._formatNewMember(arrMembers[i]);
			}

			ICTouchAPI.debugServices.debug("ICTouchAPI.supervisionServices - _onSupGroupAdded / groupName : " + strGroupName );

			// store the new members
			this._intIncomingCallsNb = 0;
			this._strGroupName = strGroupName || "";
			this._arrMembers = dojo.clone(arrMembers);

			// Notify webapps that a group is added
			dojo.publish("supervision.groupAdded", [this._strGroupName, arrMembers]);

			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - _initGroup / End");
		},
		
		/**
		 * @ignore
		 */
		_onSupGroupModified : function() {
			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - _onSupGroupModified / Begin");

			var objSupGroupModifiedArguments = ICTouchAPI.tools.getEventArguments(arguments);
			var groupName = objSupGroupModifiedArguments.groupName;
			var arrDeletedMemberLogin = objSupGroupModifiedArguments.listDeletedMembers;
			var arrAddedMember = objSupGroupModifiedArguments.listAddedMembers;
			var arrModifiedMember = objSupGroupModifiedArguments.listModifiedMembers;
			var i, j;

			ICTouchAPI.debugServices.debug("ICTouchAPI.supervisionServices - _onSupGroupModified / groupName : " + groupName );

			// Delete members
			if(arrDeletedMemberLogin){
				for(i = 0 ; i < arrDeletedMemberLogin.length ; i++){
					for(j = 0 ; j < this._arrMembers.length ; j++){
						if(this._arrMembers[j].identity.login == arrDeletedMemberLogin[i]){
							ICTouchAPI.debugServices.debug("ICTouchAPI.supervisionServices - _onSupGroupModified / deletedMember "+ i + " : " + arrDeletedMemberLogin[i] );
							this._arrMembers.splice(j, 1);
							break;
						}
					}
				}
			}

			// Add members
			if(arrAddedMember){
				for(i = 0 ; i < arrAddedMember.length && this._arrMembers.length < this.getMaxMembers(); i++){
					ICTouchAPI.debugServices.debug("ICTouchAPI.supervisionServices - _onSupGroupModified / addedMember "+ i + " : " + dojo.toJson(arrAddedMember[i]) );
					this._formatNewMember(arrAddedMember[i]);
					this._arrMembers.push(arrAddedMember[i]);
				}
			}

			// Modify members
			if(arrModifiedMember){
				for(i = 0 ; i < arrModifiedMember.length ; i++){
					this._formatMemberIdentity(arrModifiedMember[i]);
					for(j = 0 ; j < this._arrMembers.length ; j++){
						if(this._arrMembers[j].identity.login == arrModifiedMember[i].identity.login){
							ICTouchAPI.debugServices.debug("ICTouchAPI.supervisionServices - _onSupGroupModified / modifiedMember "+ i + " : " + dojo.toJson(arrModifiedMember[i]) );
							dojo.mixin(this._arrMembers[j], arrModifiedMember[i]);
							arrModifiedMember[i] = this._arrMembers[j];
							break;
						}
					}
				}
			}

			// Edit groupName if defined
			if(groupName){
				this._strGroupName = groupName;
			}

			// Notify webapps that a group is modified
			dojo.publish("supervision.groupModified", [groupName, arrAddedMember, arrModifiedMember, arrDeletedMemberLogin]);

			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - _onSupGroupModified / End");
		},
		
		/**
		 * @ignore
		 */
		_onSupGroupHidden : function() {
			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - _onSupGroupHidden / Begin");

			this._setSupervisionHidden(true);
			
			this._arrMembers = [];
			dojo.publish("supervision.groupDeleted", []);

			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - _onSupGroupHidden / End");
		},
		
		/**
		 * @ignore
		 */
		_onSupGroupDeleted : function() {
			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - _onSupGroupDeleted / Begin");
			
			this._setSupervisionHidden(false);
			
			this._arrMembers = [];
			dojo.publish("supervision.groupDeleted", []);

			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - _onSupGroupDeleted / End");
		},
		
		/**
		 * @ignore
		 */
		_onSupUpdated : function() {
			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - _onSupUpdated / Begin");

			var objSupUpdatedArguments = ICTouchAPI.tools.getEventArguments(arguments);
			var intNbCalls = objSupUpdatedArguments.nbIncomingCalls;
			var arrSup = objSupUpdatedArguments.listSupervisedUsers;
			ICTouchAPI.debugServices.debug("ICTouchAPI.supervisionServices - _onSupUpdated / intNbCalls "+intNbCalls);
			ICTouchAPI.debugServices.debug("ICTouchAPI.supervisionServices - _onSupUpdated / arrSup " + dojo.toJson(arrSup));
			var member;
			var arrModifiedMembers = [];

			for(var i = 0 ; i < arrSup.length ; i++){
				for(var j = 0 ; j < this._arrMembers.length ; j++){
					if(this._arrMembers[j].identity.login == arrSup[i].login){
						member = this._arrMembers[j];

						// Fill specific user new information
						member.presence = this._getPresenceIconFromTelState(arrSup[i].telState.value.value);
						member.ringState = this._getRingStateFromTelState(arrSup[i].telState.value.value);
						member.firstIncomingCall = arrSup[i].firstIncomingCall.value;
						member.firstIncomingCall.caller = member.firstIncomingCall.identity.value;
						delete member.firstIncomingCall.identity;
						member.firstIncomingCall.caller.officePhone = member.firstIncomingCall.caller.officePhone.value;

						arrModifiedMembers.push(member);
						break;
					}
				}
			}

			// webapps that a group is modified on the same way that in function _onSupGroupModified
			dojo.publish("supervision.groupModified", [null, [], arrModifiedMembers, []]);


			// Notify if the number of incoming calls on supervised users changed
			if(intNbCalls != this._intIncomingCallsNb){
				this._intIncomingCallsNb = intNbCalls;
				dojo.publish("supervision.incomingCallsNumberChanged", [intNbCalls])
			}

			ICTouchAPI.debugServices.info("ICTouchAPI.supervisionServices - _onSupUpdated / End");
		},
		
		/**
		 * @ignore
		 */
		_formatNewMember : function(objMember) {
			this._formatMemberIdentity(objMember);
			this._formatNewMemberPresence(objMember);
		},

		/**
		 * @ignore
		 */
		_formatNewMemberPresence : function(objMember) {
			objMember.presence = this._getPresenceIconFromTelState("UNKNOWN");
			objMember.ringState = this._getRingStateFromTelState("UNKNOWN");
		},

		/**
		 * @ignore
		 */
		_formatMemberIdentity : function(objMember) {
			objMember.identity = objMember.identity.value;
			objMember.identity.officePhone = objMember.identity.officePhone.value;
			objMember.identity.officePhone.type = objMember.identity.officePhone.type.value;
		},

		/**
		 * @ignore
		 */
		_getPresenceIconFromTelState : function(strTelState) {
			var telPresenceIcon;

			switch(strTelState){
				case "FREE":
				case "RINGING":
					telPresenceIcon = "userinfo-availablephone";
					break;
				case "BUSY":
				case "BUSYANDRINGING":
					telPresenceIcon = "userinfo-onthephone"
					break;
				default: // UNKNOWN and OUTOFSERVICE
					telPresenceIcon = "userinfo-appearoffline-offline"
					break;
			}

			return telPresenceIcon;
		},

		/**
		 * @ignore
		 */
		_getRingStateFromTelState : function(strTelState) {
			if(strTelState === "RINGING" || strTelState === "BUSYANDRINGING"){
				return true;
			} else {
				return false;
			}
		},

		/**
		 * @ignore
		 * Send logs of attributes of the service to the logger.
		 * @param {boolean} boolAdvancedLogs Boolean indicating if advanced or basic logs has to be written
		 */
		dump : function(boolAdvancedLogs){
			ICTouchAPI.debugServices.dump("dump function of supervisionServices with " + ((boolAdvancedLogs)?"advanced":"basic") + " logs option (bug advanced mode is not used yet");

			ICTouchAPI.debugServices.dump("Number of incoming call on supervised users : " + this._intIncomingCallsNb);

			ICTouchAPI.debugServices.dump("GroupName : " + this._strGroupName);
			for(var j in this._arrMembers){
				ICTouchAPI.debugServices.dump("Member " + j + " : " + dojo.toJson(this._arrMembers[j]));
			}
		}

	});
// Create supervision services
ICTouchAPI.supervisionServices=new ICTouchAPI.supervisionServices();
