/**
* @class ICTouchAPI.popupServices
* @extends Object
* @singleton
* Manage and display popup in the Phone
*/
dojo.provide("ICTouchAPI.popupServices");
dojo.declare("ICTouchAPI.popupServices",null,

{
/* --------------------------------- Public attributes ------------------------------------ */
	/**
	 * Table of popups ordered by priority level and creation order
         * @ignore
	 */
	_arrPopup : [],
	/**
	 * List of business popup
         * @ignore
	 */
	_lstBusinessPopup : [],
	/**
	 * Object list of business popup id to delete on end of hPopup creation
         * @ignore
	 */
	_objListBusinessPopupToDelete : {},
	/**
	 * Reference to the DOM popup container
         * @ignore
	 */
	_popupContainer : null,
	/**
	 * Number of displayed popup : 0
         * @ignore
	 */
	_intDisplayedPopups : 0,
	/**
	 * Is the DOM container displayed
         * @ignore
	 */
	_isDisplayed : false,
	/**
	 * Reference to the current popup
         * @ignore
	 */
	_currentDisplayedPopup : null,
	/**
	 * @ignore
	 */
	_myIFrame : null,
	/**
	 * "Constant" definition for the priorities list
         * @ignore
	 */
	_lstPriorities : {
		"LOW" : 0,
		"MIDDLE" : 1,
		"HIGH" : 2,
		"CRITICAL" : 3
	},
	/**
	 * @ignore
	 */
	_isATimeout : false,

/* --------------------------------- Private attributes ----------------------------------- */

/* ------------------------------------ Constructor --------------------------------------- */

	/**
	 * @ignore
	 */
	constructor : function(){
		this._arrPopup = [];
		this._myIFrame = document.createElement('iframe');
		this._myIFrame.className = "FullScreen";
		this._myIFrame.src = "iframe.html";
		this._myIFrame.style.zIndex = "15000";
		this._myIFrame._show = function () {
            if(this.contentDocument) {
                var div = this.contentDocument.getElementById('opacityZone');
                div.style.backgroundColor = "rgba( 0, 0, 0, 0.8 )";
            }
			this.style.left = "";
		};
		this._myIFrame._hide = function () {
            if(this.contentDocument) {
                var div = this.contentDocument.getElementById('opacityZone');
                div.style.backgroundColor = "transparent";
            }
			this.style.left = "-1000px";
		};
		this._myIFrame.id="Popup";
		var that = this;
		this._myIFrame.onload = function () {
			var headerIFrame = dojo.query("head",that._myIFrame.contentDocument)[0];
			ICTouchAPI.skinServices.registerHeader(headerIFrame,"Popup","webapp",that._myIFrame);
			that._popupContainer = this.contentDocument.body;
			var opacityZone = document.createElement('div');
			opacityZone.id="opacityZone";
			that._popupContainer.appendChild(opacityZone);
		};
		this._myIFrame._hide();
		document.body.appendChild(this._myIFrame);
		ICTouchAPI.eventServices.subscribeToEvent(this, "SHOW_POPUP_EVENT", this._addBusinessPopup);
		ICTouchAPI.eventServices.subscribeToEvent(this, "HIDE_POPUP_EVENT", this._hideBusinessPopup);
	},

/* ----------------------------------- Getter / Setter------------------------------------- */

	/**
	 * Retrieve a popup
	 * @param {Number} priority Priority of the popup
	 * @param {Number} popup Position of the popup
	 *
	 */
	getPopup : function(priority, popup) {
		return this._arrPopup[priority][popup];
	},

	/**
	 * Retrieve popups of priority
	 * @param {Number} priority Priority of the popup
	 *
	 */
	getPopupsByPriority : function(priority){
		return this._arrPopup[priority];
	},

	/**
	 * Retrieve the numbers of openned popups
	 * @return {Number} Number of popups
	 */
	getNumberOfDisplayedPopups: function(){
		return this._intDisplayedPopups;
	},

/* ----------------------------------- Public methods ------------------------------------- */

	/**
	 * Create and publish a popup
	 * @param {Object} params Params corresponding to the UIElement Popup creation
	 * @param {String} priority Priority of the popup, one of : LOW | MIDDLE | HIGH | CRITICAL
         * @return {Object} Created popup reference, used for popup control
	 */
	addNewPopup : function(params, priority){
		params.selfDestroyCallback = dojo.hitch(this,this.onSelfDestroy)
		var priorityLocal = "";
		priorityLocal = this._lstPriorities[priority] || 0;
		for (var i in this._arrPopup[priorityLocal])
		{
			if (params.strContent.toTranslate && this._arrPopup[priorityLocal][i].strContent.toTranslate) {
				if (this._arrPopup[priorityLocal][i].strContent.toTranslate == params.strContent.toTranslate) {
					var popup = this._arrPopup[priorityLocal][i];
					return {position : popup.position, priority : popup.priority, id : popup.id};
				}
			}
		}
		var newPopupContainer = document.createElement("div");
		this._popupContainer.appendChild(newPopupContainer);
		var newPopup = new UIElements.DialogBox.DialogBoxControl(params, newPopupContainer);
		newPopup.attr("class", "Popup");
		newPopup.priority = priorityLocal;
		if(params.original && params.original.id) {
			newPopup.businessId = params.original.id;
		} else {
			newPopup.businessId = null;
		}
		if(params.id){
			newPopup.id=params.id;
		}

		this._registerPopup(newPopup);
		this._needDisplay(newPopup);
		params=null;
		priority=null;
		return {position:newPopup.position, priority:newPopup.priority ,id:newPopup.id};
	},

	/**
     *@ignore
     */
    onSelfDestroy : function(popup){
		if (this._arrPopup[popup.priority][popup.position]){
			this.removePopup(popup);
		}
		else{
			//Due to a bug in popup services (not easily reproductible)
			//Sometimes, the popup can not be close using the standard buttons
			//so Reset popup display state
			this._myIFrame._hide();
			this._isDisplayed = false;
			this._currentDisplayedPopup = null;
		}
		popup.destroy();

		console.log( "[popupServices] onSelfDestroy" );
	},

	/**
	 * Remove a popup
	 * @param {Object} popup Reference gave by the addNewPopup method
	 */
	removePopup : function(popup){
		var popupTmp = this._arrPopup[popup.priority][popup.position];
		if(popupTmp){
			this._unregisterPopup(popupTmp);
			this._noneedDisplay(popupTmp);
		}
		popup=null;
	},

	/**
	 * Remove a popup
	 * @param {Object} popup Reference gave by the addNewPopup method
	 */
	removePopupByID : function(popup){
		var popupTmp = null;
		for(var i in this._arrPopup[popup.priority])
		{
			if(popup.id==this._arrPopup[popup.priority][i].id){
				popupTmp = this._arrPopup[popup.priority][i];
				break;
			}
		}
		if(popupTmp){
			this._unregisterPopup(popupTmp);
			this._noneedDisplay(popupTmp);
		}
	},

	/**
	 * Register the popup in the popup list, according to his priority
	 * @ignore
	 * @param {Object} newPopup created popup
	 */
	_registerPopup : function(newPopup){
		if(!this._arrPopup[newPopup.priority]){
			this._arrPopup[newPopup.priority]=[];
		}
		this._arrPopup[newPopup.priority].push(newPopup);
		newPopup.position = this._arrPopup[newPopup.priority].length - 1;
	},

	/**
	 * Register the popup in the display stack / Check if the new popup has to be shown
	 * @ignore
	 * @param {Object} popup created popup
	 */
	_needDisplay : function(popup){
		this._intDisplayedPopups++;
		if(this._isDisplayed === false){
			this._myIFrame._show();
			this._isDisplayed = true;
			this._showPopup(popup);
		}else if(this._currentDisplayedPopup.priority <= popup.priority){
			this._showPopup(popup);
		}
	},

	/**
	 * Display a popup
	 * @ignore
	 * @param {Object} popup popup
	 */
	_showPopup : function(popup){
		if(this._currentDisplayedPopup){
			dojo.removeClass(this._currentDisplayedPopup.domNode, "currentPopup");
		}
		dojo.addClass(popup.domNode, "currentPopup");

		console.log( "[popupServices] _showPopup" );
		dojo.publish( "dialogbox.show", [] );

		popup.show();
		this._currentDisplayedPopup = popup;
	},

	/**
	 * Unregister the popup in the popup list
	 * @ignore
	 * @param {Object} popup popup
	 */
	_unregisterPopup : function(popup){
		this._arrPopup[popup.priority].splice([popup.position],1);
		if(this._arrPopup[popup.priority].length === 0 ){
			if(popup.priority === this._arrPopup.length-1)
				do{
					this._arrPopup.pop();
				}while(this._arrPopup.length != 0 && this._arrPopup[this._arrPopup.length-1] == undefined)
			else{
				this._arrPopup[popup.priority] = undefined;
			}
		} else {
			var i = popup.position;
			var popupTmp;
			while(this._arrPopup[popup.priority][i]!= null){
				if(this._arrPopup[popup.priority][i] && this._arrPopup[popup.priority][i].businessId) {
					var businessId = this._arrPopup[popup.priority][i].businessId;
					popupTmp = this._lstBusinessPopup[businessId];
					if(popupTmp) {
						popupTmp.hPopup.position = i;
						this._arrPopup[popup.priority][i].position = i;
					}
				}else{
					this._arrPopup[popup.priority][i].position = i;
				}
				i++;
			}
		}
	},

	/**
	 * Unregister a popup in the display stack / Check if the popup was the current popup
	 * @ignore
	 * @param {Object} popup popup
	 */
	_noneedDisplay : function(popup){
		this._intDisplayedPopups--;
		if(this._intDisplayedPopups === 0 ){
			this._myIFrame._hide();
			this._isDisplayed = false;
			this._currentDisplayedPopup = null;
		}else if(this._currentDisplayedPopup === popup){
			var highestPriority=this._arrPopup.length-1;
			var highestPriorityPopup = this._arrPopup[highestPriority][(this._arrPopup[highestPriority].length)-1];
			this._showPopup(highestPriorityPopup);
		}
		popup.destroy();

		console.log( "[popupServices] _noneedDisplay" );
		dojo.publish( "dialogbox.hide", [] );
	},

	/**
	 * Create and display a standard error popup
	 * @param {String} strTitle  Title of popup
	 * @param {String} strMessage  Error message
	 * @param {String} strOkBtn Label of the OK Button ("OK" by default)
	 * @param {Number} intId Id of the Popup
	 * @param {String} strIconOKBtn Icon of the OK Button ("generic-ok" by default)
	 */
	errorPopup : function(strTitle, strMessage, strOkBtn, intId, strIconOKBtn) {
		var arrButtons = [{
			strButtonName: "POPUP_OK_BTN",
			strButtonLabel: strOkBtn ? strOkBtn : _("OK","ICTouchAPI"),
			strButtonIcon : strIconOKBtn ? strIconOKBtn : "generic-ok",
			callback : function(){
				if (this.objParent)
					ICTouchAPI.popupServices.removePopup(this.objParent);
			}
		}];
		var popup = {
			strTitle: strTitle,
			strType: "error",
			strContent: strMessage,
			intID : intId,
			arrPopupButtons: arrButtons
		};
		ICTouchAPI.popupServices.addNewPopup(popup, "MEDIUM");
		strTitle=null;
		strMessage=null;
		strOkBtn=null;
		intId=null;
	},

	/**
	 * Create and display a standard popup with the given type
	 * @param {String} strType Type of popup ("info", "warning", "error", "fatal_error")
	 * @param {String} strTitle Title of popup
	 * @param {String} strMessage Error message
	 * @param {String} strOkBtn Label of the OK button ("OK" if no argument specified)
	 * @param {Number} intId Id of the popup
	 * @param {String} strIconOKBtn Icon of the OK Button ("generic-ok" if no argument specified)
	 */
	basicPopup : function(strType, strTitle, strMessage, strOkBtn, intId, strIconOKBtn) {
		var arrButtons = [{
			strButtonName: "POPUP_OK_BTN",
			strButtonLabel: strOkBtn ? strOkBtn : _("OK","ICTouchAPI"),
			strButtonIcon : strIconOKBtn ? strIconOKBtn : "generic-ok",
			callback : function(){
				if (this.objParent)
					ICTouchAPI.popupServices.removePopup(this.objParent);
			}
		}];
		var popup = {
			strTitle: strTitle,
			strType: strType,
			strContent: strMessage,
			intID : intId,
			arrPopupButtons: arrButtons
		};
		ICTouchAPI.popupServices.addNewPopup(popup, "MEDIUM");
	},

	/**
	 * @ignore
	 */
	_addBusinessPopup : function(){

		//Add a new business popup, thanks to event parameters
		//Expected parameters are :
		//- id : unique id of business popup
		//- timeout : timeout to automatically hide business popup
		//- events : list of events
		//- level : 0-1-2 (low), 3 (middle), 4 (high), 5 (critical)
		//- showDetails : boolean, with / without details
		//- text : content
		//- details : if details are needed

		var objBusinessPopup = ICTouchAPI.tools.getEventArguments(arguments);
		if(objBusinessPopup.id == undefined || this._lstBusinessPopup[objBusinessPopup.id] != undefined){
			return false;
		}
		if(objBusinessPopup.timeout <= 0 && (objBusinessPopup.events == undefined || objBusinessPopup.events.length == 0)){
			return false;
		}
		if(objBusinessPopup.showDetails != true ){
			objBusinessPopup.showDetails = false;
		}
		var objPopup = this._createValidBusinessPopup(objBusinessPopup);
		this._lstBusinessPopup[objBusinessPopup.id]= objPopup;
		objPopup.original = objBusinessPopup;
		objPopup.hPopup = this.addNewPopup(objPopup, objPopup.level);
		if(objBusinessPopup.timeout > 0){
			this._isATimeout = true;
			objPopup.hTimeout=window.setTimeout(function(){
				ICTouchAPI.popupServices._sendEvent(objBusinessPopup, "timeout");
				ICTouchAPI.popupServices._hideBusinessPopupWithId(objBusinessPopup.id);
			}, objBusinessPopup.timeout);
		}
		if(this._objListBusinessPopupToDelete[objBusinessPopup.id]) {
			this._hideBusinessPopupWithId(objBusinessPopup.id);
			delete(this._objListBusinessPopupToDelete[objBusinessPopup.id]);
		}
		return true;
	},

	/**
	 * Create the business popup
	 * @ignore
	 * @param {Array} objBusinessPopup key/value array of arguments, defining the popup
	 */
	_createValidBusinessPopup : function(objBusinessPopup){
		switch(objBusinessPopup.level){
			case 0:
			case 1:
			case 2:
				objBusinessPopup.level = "LOW";
				objBusinessPopup.strType = "info";
				break;
			case 3:
				objBusinessPopup.level = "MIDDLE";
				objBusinessPopup.strType = "warning";
				break;
			case 4:
				objBusinessPopup.level = "HIGH";
				objBusinessPopup.strType = "error";
				break;
			case 5:
				objBusinessPopup.level = "CRITICAL";
				objBusinessPopup.strType = "fatal_error";
				break;
			default :
				break;
		}

		if(objBusinessPopup.textParams) {
			for(var i=0; i<objBusinessPopup.textParams.length; i++) {
				if(!objBusinessPopup.textParams[i].isI18Ned) {
					objBusinessPopup.textParams[i] = _(objBusinessPopup.textParams[i], "UIElements.DialogBox");
				}
			}
		}
		objBusinessPopup.translatedText=_(objBusinessPopup.text,"UIElements.DialogBox",objBusinessPopup.textParams);
		if (objBusinessPopup.details) {
			objBusinessPopup.translatedDetails = _(objBusinessPopup.details,"UIElements.DialogBox",objBusinessPopup.detailsParams);
		}
		if(objBusinessPopup.showDetails){
			objBusinessPopup.translatedDetailsContent = objBusinessPopup.translatedText.getTranslation() + "<br /><br />" + objBusinessPopup.translatedDetails.getTranslation();
		}else{
			objBusinessPopup.translatedDetailsContent = objBusinessPopup.translatedText;
		}
		var objPopup = {
			strType			: objBusinessPopup.strType,
			strContent		: objBusinessPopup.translatedDetailsContent,
			strText			: objBusinessPopup.translatedText,
			strDetails		: objBusinessPopup.translatedDetails,
			showDetails 	: objBusinessPopup.showDetails,
			arrPopupButtons : [],
			level			: objBusinessPopup.level
		}
		var callback = function(strButtonName){
			ICTouchAPI.popupServices._userAction(objBusinessPopup.id,strButtonName);
		};
		var nbButtons = 0;
		if(objBusinessPopup.events != undefined) {
			if(objBusinessPopup.events["OK"]){
				objPopup.arrPopupButtons.push({
					strButtonLabel: _("Ok","ICTouchAPI"),
					callback : callback,
					strButtonIcon : 'generic-ok',
					strButtonName : "btn_OK"
				});
				nbButtons++;
			}
			if(objBusinessPopup.events["CANCEL"]){
				objPopup.arrPopupButtons.push({
					strButtonLabel: _("Cancel","ICTouchAPI"),
					callback : callback,
					strButtonIcon : 'generic-cancel',
					strButtonName : "btn_CANCEL"
				});
				nbButtons++;
			}
			if(objBusinessPopup.events["YES"]){
				objPopup.arrPopupButtons.push({
					strButtonLabel: _("Yes","ICTouchAPI"),
					callback : callback,
					strButtonIcon : 'generic-ok',
					strButtonName : "btn_YES"
				});
				nbButtons++;
			}
			if(objBusinessPopup.events["NO"]){
				objPopup.arrPopupButtons.push({
					strButtonLabel: _("No","ICTouchAPI"),
					callback : callback,
					strButtonIcon : 'generic-cancel',
					strButtonName : "btn_NO"
				});
				nbButtons++;
			}
		}
		if(nbButtons === 0){
			objPopup.arrPopupButtons.push({
				strButtonLabel: _("Dismiss","ICTouchAPI"),
				callback : callback,
				strButtonIcon : 'generic-cancel',
				strButtonName : "btn_DISMISS"
			});
		}
		if(objPopup.strDetails && objPopup.strDetails != ""){
			objPopup.arrPopupButtons.push({
					strButtonLabel: _("Details","ICTouchAPI"),
					callback : callback,
					strButtonIcon : 'generic-search',
					strButtonName : "btn_DETAILS"
				});
		}
		return objPopup;
	},

	/**
	 * Callback action of a business popup button
	 * @ignore
	 * @param {String} id id of the business popup
	 * @param {String} strButtonName name of the button clicked
	 */
	_userAction : function(id, strButtonName){
		var currPopup = this._lstBusinessPopup[id];
		if(strButtonName == "btn_DETAILS"){
			//switching from shown details mode to hidden details mode
			currPopup.original.showDetails = !currPopup.original.showDetails;
			//saving a reference on the original params
			var original = currPopup.original;
			//removing the currently displayed popup
			this._hideBusinessPopupWithId(id);
			//Building a popup showing the details within the content
			var objPopup = this._createValidBusinessPopup(original);
			//saving a reference on the original params
			objPopup.original = original;
			//adding the new popup
			objPopup.hPopup = this.addNewPopup(objPopup, objPopup.level);
			this._lstBusinessPopup[original.id]= objPopup;
			//allowing the timout params only if the details are not shown
			if(!original.showDetails){
				if(original.timeout > 0){
					this._isATimeout = true;
					objPopup.hTimeout=window.setTimeout(function(){
						ICTouchAPI.popupServices._sendEvent(original, "timeout");
						ICTouchAPI.popupServices._hideBusinessPopupWithId(original.id);
					}, original.timeout);
				}
			}

		}else{

			switch(strButtonName){
				case "btn_OK":
					this._sendEvent(currPopup.original, "OK");

					break;
				case "btn_CANCEL":
					this._sendEvent(currPopup.original, "CANCEL");

					break;
				case "btn_YES":
					this._sendEvent(currPopup.original, "YES");

					break;
				case "btn_NO":
					this._sendEvent(currPopup.original, "NO");

					break;
				case "btn_DISMISS":
					this._sendEvent(currPopup.original, "timeout");
					break;
				default :
					break;
			}

			this._hideBusinessPopupWithId(id);
			}

	},

	/**
	 * Send an event if specified in the popup object
	 * @ignore
	 * @param {Object} objPopup Popup sent by infra
	 * @param {String} strEventName name of the event to send
	 */
	_sendEvent	: function(objPopup, strEventName) {
		if( typeof objPopup.events != "undefined" && typeof objPopup.events[strEventName] == "string" ) {
			var infraEventName = objPopup.events[strEventName];
			ICTouchAPI.APIServices.ICTGate.notifyEventPresentation({params:[infraEventName]});
		}
	},

	/**
	 * Hide / destroy a business popup (asked through an event
	 * @ignore
	 * @param {Object} objEvent event object, with "value" defined as business popup id
	 */
	_hideBusinessPopup : function(objEvent){
		this._hideBusinessPopupWithId(objEvent.value);
	},

	/**
	 * Hide / destroy a business popup thanks to its ID
	 * @ignore
	 * @param {String} strPopupId id of the business popup
	 */
	_hideBusinessPopupWithId : function(strPopupId){
		var currPopup = this._lstBusinessPopup[strPopupId];
		if (currPopup) {
			if (currPopup.hPopup){
				if(currPopup.hTimeout && this._isATimeout){
					window.clearTimeout(currPopup.hTimeout);
					this._isATimeout = false;
				}
				this.removePopup(currPopup.hPopup);
				this._lstBusinessPopup[strPopupId] = null;
				delete(this._lstBusinessPopup[strPopupId]);
			} else {
				this._objListBusinessPopupToDelete[strPopupId] = true;
			}
		} else {
			console.warn("Popup not found.");
		}
	},

/* --------------------------------- Private Methods -------------------------------------- */
	/**
	 * Get Application Name
	 * @ignore
	 * @return {String} The name of the Application Service
	 */
	getApplicationName : function(){
		return "popupServices";
	}
});

ICTouchAPI.popupServices=new ICTouchAPI.popupServices();
