/**
* @class ICTouchAPI.notificationServices
* @singleton
* @extends Object
* Manage and display notifications to the Phone Notification Bar
*/
dojo.provide("ICTouchAPI.notificationServices");
dojo.declare("ICTouchAPI.notificationServices",null,
{
	/* --------------------------------- Public attributes ------------------------------------ */
	/**
	 * first notification slot index
         * @ignore
	 */
	NOTIFICATION_FIRST_SLOT : 0,
	/**
	 * second notification slot index
         * @ignore
	 */
	NOTIFICATION_SECOND_SLOT : 1,
	/**
	 * third notification slot index
         * @ignore
	 */
	NOTIFICATION_THIRD_SLOT : 2,
	/**
	 * large notification slot index
         * @ignore
	 */
	NOTIFICATION_LARGE_SLOT : 3,
	/**
	 * hidden notification slot index
         * @ignore
	 */
	NOTIFICATION_HIDDEN_SLOT : 4,

	/* --------------------------------- Private attributes ----------------------------------- */
	/**
	 * @ignore
	 */
	_arrNotifications : [],

        /**
	 * @ignore
	 */

	_lstNotifications : {},

	enumType : {
		GREEN : "notificationBackgroundGreen",
		RED : "notificationBackgroundRed",
		BLUE : "notificationBackgroundBlue",
		ORANGE : "notificationBackgroundOrange",
		EMERGENCY : "emergency",
		WAITING : "waiting"
	},

        /**
	* @ignore
	*/
        constructor : function(){
		this._arrNotifications = [];
		this._lstNotifications = {};
	},

	/**
	* Create and add a new notification into the Notification area
	* @param {Object} objNotification : The notification to add
	* @return {Bool} True if successfull
	*/
	addNotification : function(objNotification){
		if(objNotification && objNotification.strNotificationName && this._lstNotifications[objNotification.strNotificationName] == undefined){
			if(!this._lstNotifications[objNotification.strNotificationName]){
				objNotification._intPosition = -1;
				this._lstNotifications[objNotification.strNotificationName]= objNotification;
			}
			if((objNotification.intNotificationValue && objNotification.intNotificationValue > 0) || objNotification.strNotificationTypeIcon) {
				this._arrNotifications.splice(0,0,objNotification);
				this._adjustPositions();
				this._publishModification("notificationAdded", [objNotification]);
			}
			objNotification=null;
			return true;
		}
		objNotification=null;
		return false;
	},

        /**
	* Remove a notification from the Notification area
	* @param {String} notificationName : name of the notification to remove
	*/
	removeNotification : function(notificationName){
		var objNotification = this.getNotificationByName(notificationName);
		if (objNotification) {
			this._delNotification(objNotification);
		}
		notificationName=null;
	},

	/**
	 * Adjust position of each notification
	 * @ignore
	 */
	_adjustPositions : function(){
		var notifLength = this._arrNotifications.length;
		for(var i=0; i<notifLength; i++){
			this._arrNotifications[i]._intPosition = i;
		}
	},

	/**
	 * Delete a notification from the table
	 * @ignore
	 * @param {String} objNotification : Name of the notification
	 */
	_delNotification : function(objNotification){
		delete this._lstNotifications[objNotification.strNotificationName];
		if(objNotification && objNotification._intPosition != -1){
			this._arrNotifications.splice(objNotification._intPosition,1);
			this._publishModification("notificationDeleted", [objNotification, objNotification._intPosition]);
			this._adjustPositions();
		}
	},

	/**
	 * update a notification (Value has changed)
	 * @ignore
	 * @param {Object} objNotification : Notification to update
	 */
	_updateNotification : function(objNotification){
		//INCREMENTATION OR DECREMENTATION ON AN EXISTING ELEMENT
		if(objNotification.intNotificationValue > 0 && objNotification._intPosition >= 0){
			//this._arrNotifications.splice(objNotification._intPosition,1);
			//objNotification._intPosition = -1;
			//this._arrNotifications.splice(0,0,objNotification);
		}
		//DECREMENTATION ON AN EXISTING ELEMENT THAT MUST DISAPPEAR
		else if(objNotification.intNotificationValue === 0  && objNotification._intPosition >= 0){
			this._arrNotifications.splice(objNotification._intPosition,1);
			objNotification._intPosition = -1;
		}
		//INCREMENTATION ON AN EXISTING ELEMENT THAT MUST APPEAR
		else if(objNotification.intNotificationValue > 0 && objNotification._intPosition === -1){
			this._arrNotifications.splice(0,0,objNotification);
		}
		this._adjustPositions();
	},

	/**
	 * refresh a notification with possible new params
	 * @param {String} notificationName : name of the notification to update
	 * @param {Object} objNotification : notification new params (optionnal)
	 */
	refreshNotification : function(notificationName, objNotification){
		var objOldNotification = this.getNotificationByName(notificationName);
		if(objOldNotification && objOldNotification.intNotificationValue !== undefined){
			var modified = false;
			for(var i in objNotification){
				if(!modified && (objOldNotification[i] != objNotification[i])){
					modified = true;
				}
				objOldNotification[i] = objNotification[i];
			}
			if(modified){
				var oldPosition = objNotification._intPosition;
				this._publishModification("notificationModified", [objOldNotification]);
			}
		}
		notificationName=null;
	},

        /**
	 * Increment the value of a notification
	 * @param {String} notificationName The name of the notification
	 */
	incNotification : function(notificationName){
		var objNotification = this.getNotificationByName(notificationName);
		if(objNotification && objNotification.intNotificationValue !== undefined){
			var oldPosition = objNotification._intPosition;
			objNotification.intNotificationValue++;
			this._updateNotification(objNotification);
			if(oldPosition == -1){
				this._publishModification("notificationAdded", [objNotification]);
			}
			else{
				this._publishModification("notificationUpdated", [objNotification, oldPosition]);
			}
		}
		notificationName=null;
	},

        /**
	 * Decrement the value of a notification
	 * @param {String} notificationName The name of the notification
	 */
	decNotification : function(notificationName){
		var objNotification = this.getNotificationByName(notificationName);
		if(objNotification && objNotification.intNotificationValue && objNotification.intNotificationValue > 0){
			var oldPosition = objNotification._intPosition;
			objNotification.intNotificationValue--;
			this._updateNotification(objNotification);
			if(objNotification.intNotificationValue === 0){
				this._publishModification("notificationDeleted", [objNotification, oldPosition]);
			}
			else{
				this._publishModification("notificationUpdated", [objNotification, oldPosition]);
			}
		}
		notificationName=null;
	},

        /**
	 * Set the value of a notification
	 * @param {String} notificationName The name of the notification
	 * @param {Number} value : The new value
	 */
	setNotificationValue : function(notificationName, value){
		var objNotification = this.getNotificationByName(notificationName);
		if(value >=0){
			objNotification.intNotificationValue = value;
			var oldPosition = objNotification._intPosition;
			this._updateNotification(objNotification);
			if(objNotification.intNotificationValue == 0){
				if(oldPosition != -1){
					this._publishModification("notificationDeleted", [objNotification,oldPosition]);
				}
			}
			else{
				if(oldPosition != -1){
					this._publishModification("notificationUpdated", [objNotification, oldPosition]);
				}
				else{
					this._publishModification("notificationAdded", [objNotification]);
				}
			}
		}
		notificationName=null;
		value=null;
	},

       /**
	 * Remove all notifications
	 */
	resetNotifications : function() {
		for(var i in this._lstNotifications){
			var objNotification = this._lstNotifications[i];
			if(objNotification.intNotificationValue && objNotification.intNotificationValue>0){
				this.setNotificationValue(objNotification.strNotificationName, 0);
			}
		}
	},

        /**
	 * Get all the registered notifications
	 * @return {Array} The notifications
	 */
	getNotifications : function(){
		return this._arrNotifications;
	},

        /**
	 * Get a notification by name
         * @param {String} strName The name of the notification
	 * @return {Object} The corresponding notification
	 */
	getNotificationByName : function(strName){
		return this._lstNotifications[strName];
	},

	/**
	 * Get a notification object by name
	 * @ignore
	 * @return {Object} objNotification : Object notification with the corresponding name
	 */
	_publishModification : function(type, object){
		dojo.publish("notification/"+type,object);
	}
});

ICTouchAPI.notificationServices = new ICTouchAPI.notificationServices();
