/**
 * @class ICTouchAPI.ledServices
 * @singleton
 * @extends Object
 * The ledService provides an interface to web applications to manage the senskey LEDs.<br />
 * Currently, only the 'event' led is programmable.
 * <pre>
 * To turn on the 'event' led:
 * ICTouchAPI.ledServices.activate('event', 'ON');
 * <br />
 * To turn off the 'event' led:
 * ICTouchAPI.ledServices.desactivate('event', 'ON');
 * <br />
 * To blink the 'event' led:
 * ICTouchAPI.ledServices.activate('event', 'BLINK');
 * <br />
 * To stop blinking the 'event' led:
 * ICTouchAPI.ledServices.desactivate('event', 'BLINK');
 * </pre>
 */
dojo.require("ICTouchAPI.led");
dojo.provide("ICTouchAPI.ledServices");
dojo.declare("ICTouchAPI.ledServices",null,
{

	/* --------------------------------- Public attributes ------------------------------------ */

	/* --------------------------------- Private attributes ----------------------------------- */

        /**
         * @ignore
         */
        arrLEDs: null,

	/* ------------------------------------ Constructor --------------------------------------- */

	constructor : function() {
		//Array holding the LEDs
		this.arrLEDs = {};
		//creating LEDs
		this._fillInArrLEDs();
	},


	destroy : function(){
		dojo.forEach(this.arrLEDs,function(led){
			led.destroy();
		});
	},

	/* ----------------------------------- Getter / Setter------------------------------------- */

	/* ----------------------------------- Public methods ------------------------------------- */

	/**
	 *	Activate a mode on a led
	 *	@param {string} led The name of the led. Should be set to 'event'
	 *	@param {string} mode The name of the mode to be activated. Should be set to 'ON' or 'BLINK'
	 */
	activate : function(led, mode){
            this.arrLEDs[ICTouchAPI.led.prototype.arrLedsNames[led]].activateMode(mode);
            led=null;
            mode=null;
	},

	/**
	 * Desactivate a mode on a led
	 * @param {string} led The name of the led. Should be set to 'event'
	 * @param {string} mode The name of the mode to be desactivated. Should be set to 'ON' or 'BLINK'
	 */
	desactivate : function(led, mode){
            this.arrLEDs[ICTouchAPI.led.prototype.arrLedsNames[led]].desactivateMode(mode);
            led=null;
            mode=null;
	},


	/* --------------------------------- Private Methods -------------------------------------- */

   /**
	* Method called to create the LEDs
	* @ignore
	*/
	_fillInArrLEDs : function(){
		//creating leds
		for( var name in ICTouchAPI.led.prototype.arrLedsNames){
			this.arrLEDs[ICTouchAPI.led.prototype.arrLedsNames[name]] = new ICTouchAPI.led(ICTouchAPI.led.prototype.arrLedsNames[name]);
		}
        this.addCustomModeToLed('userInfo', ICTouchAPI.mode.prototype.intON,ICTouchAPI.mode.prototype.arrColors['green'], 'AVAILABLE', 11);
		this.addCustomModeToLed('userInfo', ICTouchAPI.mode.prototype.intON,ICTouchAPI.mode.prototype.arrColors['red'], 'UNAVAILABLE', 12);
		this.addCustomModeToLed('userInfo', ICTouchAPI.mode.prototype.intON,ICTouchAPI.mode.prototype.arrColors['blue'], 'OUTOFOFFICE', 13);
		this.addCustomModeToLed('userInfo', ICTouchAPI.mode.prototype.intON,ICTouchAPI.mode.prototype.arrColors['blue'], 'ROUTING', 14);
		this.addCustomModeToLed('userInfo', ICTouchAPI.mode.prototype.intON,ICTouchAPI.mode.prototype.arrColors['red'], 'DND-ON', 15);
		
		// Mode "FORCE-OFF" is here to force deactivation of the led. It is mainly used on hotel mode. It must keep highest priority on userInfo led.
		this.addCustomModeToLed('userInfo', ICTouchAPI.mode.prototype.intOFF,ICTouchAPI.mode.prototype.arrColors['std'], 'FORCE-OFF', 100);

		this.addCustomModeToLed('communication', ICTouchAPI.mode.prototype.intBLINK,ICTouchAPI.mode.prototype.arrColors['std'], 'N_CALLBACK', 11);
		this.addCustomModeToLed('communication', ICTouchAPI.mode.prototype.intBLINK,ICTouchAPI.mode.prototype.arrColors['std'], 'N_UNANS_CALL', 11);
		this.addCustomModeToLed('communication', ICTouchAPI.mode.prototype.intBLINK,ICTouchAPI.mode.prototype.arrColors['std'], 'N_VM', 11);
		this.addCustomModeToLed('communication', ICTouchAPI.mode.prototype.intBLINK,ICTouchAPI.mode.prototype.arrColors['std'], 'N_IM', 11);
		this.addCustomModeToLed('communication', ICTouchAPI.mode.prototype.intBLINK,ICTouchAPI.mode.prototype.arrColors['std'], 'N_HOTEL', 11);
		this.addCustomModeToLed('communication', ICTouchAPI.mode.prototype.intON,ICTouchAPI.mode.prototype.arrColors['std'], 'ONGOING_CALL', 12);
	},

	/**
	 * @ignore
	 */
    addCustomModeToLed : function(strLedName, callInteger, colorInteger, modeName,modePriority){
                var intLedId = ICTouchAPI.led.prototype.arrLedsNames[strLedName];
                var ledObj = this.arrLEDs[intLedId];

                ledObj.createMode(callInteger,colorInteger, modeName, modePriority);

        },

	/**
	 * @ignore
	 */
	_customCommunicationLed : function(){
		var intLedId = ICTouchAPI.led.prototype.arrLedsNames['communication'];
		var comLed = this.arrLEDs[intLedId];

		var runFunc = function(){
			//context : mode
			ICTouchAPI.APIServices.Platform.setLedStatus({
				params: [
				intLedId,
				ICTouchAPI.mode.prototype.intBLINK,
				ICTouchAPI.mode.prototype.arrColors['std'],
				0, //unused but has to be sent
				0  //unused but has to be sent
				]
			});
			return true;
		}
		//context : led
		comLed.addMode('N_CALLBACK',new ICTouchAPI.mode('N_CALLBACK',intLedId,11,runFunc));
		comLed.addMode('N_UNANS_CALL',new ICTouchAPI.mode('N_UNANS_CALL',intLedId,11,runFunc));
		comLed.addMode('N_VM',new ICTouchAPI.mode('N_VM',intLedId,11,runFunc));
		comLed.addMode('N_IM',new ICTouchAPI.mode('N_IM',intLedId,11,runFunc));
		comLed.addMode('N_HOTEL',new ICTouchAPI.mode('N_HOTEL',intLedId,11,runFunc));

		var runFunc = function(){
			//context : mode
			ICTouchAPI.APIServices.Platform.setLedStatus({
				params: [
				intLedId,
				ICTouchAPI.mode.prototype.intON,
				ICTouchAPI.mode.prototype.arrColors['std'],
				0, //unused but has to be sent
				0  //unused but has to be sent
				]
			});
			return true;
		}
		//context : led
		comLed.addMode('ONGOING_CALL',new ICTouchAPI.mode('ONGOING_CALL',intLedId,12,runFunc));

	}

});

ICTouchAPI.ledServices = new ICTouchAPI.ledServices();