/**
* @class ICTouchAPI.I18nServices
* @singleton
* @extends Object
* I18n provides internationalization tools for the WebApps. This framework service can be used with or without i18n tag for Django templating.
* I18n returns a string translated in the current locale (defined in dojo.locale)
*/
dojo.provide("ICTouchAPI.i18nServices");
dojo.require("dojox.string.sprintf");
dojo.require("dojo.i18n");
dojo.declare("ICTouchAPI.i18nServices",null, {

	/**
         * @ignore
         */
	_languageAllowedValues : {},

	/**
         * @ignore
         * Know how many i18nized strings have currently subscribed to the languageChanged event.
         */
	_nbOfHandlers : 0,

	/**
         * @ignore
         */
	constructor : function(){
            // Default settings for simulation mode
            if(!generalConfig.simulation) {
		ICTouchAPI.settingServices.getSetting("Language", this, this.settingChanged);
		ICTouchAPI.settingServices.subscribeToSetting(this, "Language", this.settingChanged);
            }
	},

	/**
         * @ignore
         */
	getApplicationName : function(){
		return "ICTouchAPI.i18nServices";
	},

	/**
         * @ignore
         */
	settingChanged : function (objLanguage) {
		this._setLanguageAllowedValues(objLanguage.allowedValues);
		this.languageChanged(this.getLanguageName(objLanguage.jsValue));
	},

	/**
         * @ignore
         */
	languageChanged : function (ISOLang) {
		if (this.getLocale() !== ISOLang) {
			this.setLocale(ISOLang);
			dojo.publish("languageChanged");
		}
	},

	/**
         * @ignore
         */
	setLocale : function (locale) {
		dojo.locale = locale;
	},

	/**
         * @ignore
         */
	getLocale : function () {
		return dojo.locale;
	},


	/**
         * @ignore
         */
	changeLanguage : function (language) {
		language = language.toLowerCase();
		var langJSValue = this.getLanguageJSValue(language);
		if(langJSValue === false){
			langJSValue = this.getLanguageJSValue(language.split('-')[0]);
		}
		var langValue = typeof language === "string" && langJSValue !== false ? langJSValue : +(language);
		ICTouchAPI.settingServices.setSettingValue("Language", [langValue]);
	},

	/**
         * @ignore
         */
	getLanguageJSValue : function (language) {
		for (var i in this._languageAllowedValues) {
			if (this._languageAllowedValues.hasOwnProperty(i)) {
				if (this._languageAllowedValues[i].name === language) {
					return +(i);
				}
			}
		}
		return false;
	},

	/**
         * @ignore
         */
	getLanguageName : function (jsValue)  {
		return this._languageAllowedValues[jsValue].name;
	},

	/**
         * @ignore
         */
	_setLanguageAllowedValues : function (allowedValues) {
		this._languageAllowedValues = allowedValues;
	},

	/**
         * Translate a string, and replace variables by values
         * @param {String} toTranslate : String to translate
         * @param {String} i18nContext : context to use for translation
         * @param {Array} args : Translation variables values
         * @return {Object} : An object embedding everything that is needed to translate
         * @ignore
         */
	translate : function(toTranslate, i18nContext, args, bundle){

		if (toTranslate.isI18Ned){
			console.warn(toTranslate +" in " + i18nContext + " is already i18nized.");
		} else {

			// The i18nized object
			var objI18n =  {

				// Yes it is an i18nized object
				isI18Ned : true,

				// The callback that will be executed to assign a new string to a DOM element.
				subscribeCallback : null,

				// The i18n context
				i18nContext : i18nContext,

				// The text to translate
				toTranslate : toTranslate,

				stringsToReplace : args,
				
				getTranslation : function () {
					var translated = "";
					if (typeof objI18n.toTranslate === "string") {
						translated = objI18n._translate(objI18n.toTranslate, objI18n.i18nContext, objI18n.getStringsToReplace(), bundle);
					} else {
						for (var i in objI18n.toTranslate) {
							if (objI18n.toTranslate.hasOwnProperty(i)) {
								if (objI18n.toTranslate[i].isI18Ned){
									translated += objI18n.toTranslate[i].getTranslation();
								}
								else {
									translated += objI18n._translate(objI18n.toTranslate[i], objI18n.i18nContext, objI18n.getStringsToReplace(), bundle)+" ";
								}
							}
						}
					}
					return translated;
				},

				toString : function() {
					return this.getTranslation();
				},

				_translate : function (string, i18nContext, args, bundle) {
					bundle = bundle || "lang";
					dojo.requireLocalization(i18nContext,bundle);

					var translated=dojo.i18n.getLocalization(objI18n.i18nContext, bundle)[string];

					if(!translated) {
						translated = string;
					}

					if (args && args.length) {
						// Create the argument list for sprintf(translated, args[0], args[1], ...);
						args = dojo.isArray(args) ? args : [args];
						args.unshift(translated);
						translated = dojox.string.sprintf.apply(dojox.string, args);
						args.shift();
					}

					return translated;
				},

				fillInnerHTML : function (node) {
					this.subscribeCallback = function( ) {
						return node.innerHTML = this.getTranslation()
					};

					this.subscribeCallback();

					var unsub = dojo.subscribe("languageChanged", this, function (){
						if (node) {
							this.subscribeCallback();
						} else {
							console.warn("[i18nServices] it seems there was a leak with ");
							console.warn(node);
							// Unsubscribe
							dojo.unsubscribe(unsub);
						}

					});
					return unsub;
				},

				// Link this object with its environment
				i18nize : function (obj, name, subscribeCallback) {

					// If we try to i18nize an i18ned object.
					if (obj[name] &&  obj[name].isI18Ned) {

						// Get the text's translation
						var cb = obj[name].getTranslation;
						obj[name] = cb();
						// There will be a new subscribe handler.
						ICTouchAPI.i18nServices._nbOfHandlers++;

						// Save the callback so it can be used from other function of this i18ned object.
						this.subscribeCallback  = function () {
							// Returns true/false
							return subscribeCallback.apply(obj, [name, cb()+""]);
						}

						// subscribe to languageChanged event to assign the future translation to its domNode
						var unsub = dojo.subscribe("languageChanged", this, function () {

							// If the callback can't be executed there's a great chance it's because
							// it has disapeared
							if (!this.subscribeCallback()) {
								console.warn("[i18nServices] it seems there was a leak with ", obj, name);
								// Unsubscribe
								dojo.unsubscribe(unsub);
								unsub = null;
								ICTouchAPI.i18nServices._nbOfHandlers--;
							}
						});
						return unsub;

					} else {
						return false;
					}
				},

				// Change the text that is always translated
				changeText : function (toTranslate) {
					if (this.setToTranslate(toTranslate)) {
						// If it's a new one, execute the callback
						if (this.subscribeCallback) {
							return this.subscribeCallback();
						} else {
							return true;
						}
					}
				},

				/**
                                 * @ignore
                                 */
				setToTranslate : function (toTranslate) {
					if (toTranslate !== this.toTranslate) {
						this.toTranslate = toTranslate;
						return true;
					} else {
						return false;
					}
				},

				/**
                                 * @ignore
                                 */
				setStringsToReplace : function (args) {
					var currentStrings = this.stringsToReplace.join ? this.stringsToReplace.join() : this.stringsToReplace;
					var newStrings = args.join ? args.join() : args;

					if (currentStrings === newStrings) {
						return false;
					} else {
						this.stringsToReplace = args;
						if (this.subscribeCallback) {
							return this.subscribeCallback();
						} else {
							return true;
						}
					}
				},

				/**
                                 * @ignore
                                 */
				getStringsToReplace : function () {
					return dojo.isArray(this.stringsToReplace) ? this.stringsToReplace : [this.stringsToReplace];
				}

			}

			return objI18n;
		}


	},

	/**
         * @ignore
         */
	i18nize : function (list, i, callback) {

		var unsubscribeHandler;

		if (list[i] && typeof list[i].i18nize == "function"){
			// Prevents a bug when the same i18n object is used multiple times.
			list[i] = dojo.clone(list[i]);
			unsubscribeHandler = list[i].i18nize(list, i, callback);
		}

		return unsubscribeHandler;

	},


	i18nList : function (list, callback) {
		var unsubscribeHandlers = [], handler;

		for (var i in list) {
			if (list.hasOwnProperty(i)) {
				handler = this.i18nize(list, i, callback);
				if (handler) {
					unsubscribeHandlers.push(handler);
				}
			}
		}

		return unsubscribeHandlers;

	},

	unsubscribeAllI18nHandlers : function (widget) {

		if (widget._i18nizedHandlers) {

			var length = widget._i18nizedHandlers.length;
			for (var i=0; i<length; i++) {
				dojo.unsubscribe(widget._i18nizedHandlers[i]);
			}
			delete widget._i18nizedHandlers;

			this._nbOfHandlers -= i;
		}
	},

	addI18nHandlers : function (who, lstHandlers) {
		if (!who._i18nizedHandlers) {
			who._i18nizedHandlers = [];
		}
		who._i18nizedHandlers = who._i18nizedHandlers.concat(lstHandlers);
	},

	addI18nHandler : function (who, arrHandler) {
		if (!who._i18nizedHandlers) {
			who._i18nizedHandlers = [];
		}
		who._i18nizedHandlers.push(arrHandler);
	},

	areI18nStringsEquals : function (i18nA,i18nB) {
		if(i18nA.toTranslate.length !=i18nB.toTranslate.length){
			return 0;
		}
		for(var i=0;i<i18nA.toTranslate.length;i++){
			if(i18nA.toTranslate[i]!=i18nB.toTranslate[i]){
				return 0;
	}
		}
		return 1;
	},
	
	cloneI18n : function(i18nObject){
		if(i18nObject.isI18Ned){
			return _(i18nObject.toTranslate, i18nObject.i18nContext, i18nObject.stringsToReplace);
		} else {
			return i18nObject;
		}
	}


});
ICTouchAPI.i18nServices = new ICTouchAPI.i18nServices();

/**
 * Translate a string, and replace variables by values
 * @param {String} toTranslate : String to translate
 * @param {String} i18nContext : context to use for translation
 * @param {Array} args : Translation variables values
 * @return {Object} : An object embedding everything that is needed to translate
 */
_ = ICTouchAPI.i18nServices.translate;
