/**
 * @class ICTouchAPI.feedbackServices
 * @singleton
 * @extends Object
 * This service offers the possibility to attach a Feedback to each graphical element of the Phone
 */
dojo.provide("ICTouchAPI.feedbackServices");
dojo.declare("ICTouchAPI.feedbackServices",null,
{
/* --------------------------------- Public attributes ------------------------------------ */
    /**
    * SAFETY_DELAY
    * @property
    * @type Number
    * @ignore
    */
    SAFETY_DELAY: 80, // minimum delay between two haptic calls, in ms

    /**
    * senskeyServices
    * @property
    * @type Object
    * @ignore
    */
    senskeyServices : null,//senskey services

/* --------------------------------- Private attributes ------------------------------------ */
    /**
    * @property
    * @type Boolean
    * @ignore
    */
    _feedbackState: true,	// True : feedback activated

    /**
    * @property
    * @type Boolean
    * @ignore
    */
    _hapticState: true,			// True : Haptic activated

    /**
    * @property
    * @type Boolean
    * @ignore
    */
    _beepState: true,			// True : Beep activated

    /**
    * @property
    * @type Boolean
    * @ignore
    */
    _safetyLock: false, // True : ignore haptic effect, because previous haptic call was too recent

    /**
    * @property
    * @type Object
    * @ignore
    * List of available effects
    * Mapping beetween effects and the associated integer expected by IctWebClient
    */
    _feedbackEffectsValue: {
        "StrongClick"		: 0,
        "SharpClick"		: 1,
        "SoftBump"			: 2,
        "DoubleClick"		: 3,
        "TripleClick100"	: 4,
        "SoftBuzz100"		: 5,
        "StrongBuzz100"		: 6
    },

    /**
    * @property
    * @type String
    * @ignore
    */
	_defaultEffect : "SharpClick", // Default effect

    /**
    * @property
    * @type Object
    * @ignore
    * List of available feedbacks
    * Only haptic effects are implemented at the moment.
    */
    _feedbackTypes: {
        "endScroll" : {
            hapticMethod : "startEffect",
            hapticEffect : "StrongBuzz"
        },
        "keystroke" : {
            hapticMethod : "click",
            hapticEffect : "SoftBuzz"
        },
        "click"				: {
            hapticMethod : "click",
            hapticEffect : 	"SharpClick"
        }
    },

    /**
    * @property
    * @type String
    * @ignore
    */
    _defaultType: "click",// Default feedback type

/* ------------------------------------ Constructor --------------------------------------- */
    /**
     * @ignore
     */
    constructor: function () {
        //this._hapticState = (window.Haptic != undefined);

        // Default settings for simulation mode
        if(!generalConfig.simulation) {
            ICTouchAPI.settingServices.getSettingValue("HapticEnabled", this, this._hapticValueReceived);
            ICTouchAPI.settingServices.getSettingValue("BeepEnabled", this, this._beepValueReceived);

            ICTouchAPI.settingServices.subscribeToSetting(this, "HapticEnabled", this._hapticValueChanged);
            ICTouchAPI.settingServices.subscribeToSetting(this, "BeepEnabled", this._beepValueChanged);
        }
    },

/* ------------------------------------ getter/setter --------------------------------------- */
    /**
    * @ignore
    */
    getApplicationName: function() {
        return "feedbackServices";
    },

    /**
    * Get the state of the Feedback
    * @return {Boolean} True if the Feedback is enabled
    */
    getFeedbackState: function () {
        return this._feedbackState;
    },

    /**
    * Get the state of the haptic feedback
    * @return {Boolean} True if the haptic feedback is enabled
    */
    getHapticState: function () {
        return this._hapticState;
    },

    /**
    * Get the state of the beep feedback
    * @return {Boolean} True if the beep feedback is enabled
    */
    getBeepState: function () {
        return this._beepState;
    },

/* ------------------------------------ Public Methods --------------------------------------- */

    /**
     * Enable a feedback.
     */
    enable: function() {
        this._feedbackState = true;
    },

    /**
     * Disable the feedback.
     */
    disable: function () {
        this._feedbackState = false;
    },

    /**
     * Create a new Feedback effect to apply on a Graphical Element
     * @param {String} feedbackType The feedback's type
     * @return {Object} with the following methods :
     *		feedback(clickEvent)
     *		stop()
     */
    initFeedback: function (feedbackType) {
        var that = this;

        /**
        * @class ICTouchAPI.feedback
        * @namespace ICTouchAPI
        * @ignore
        * @extends Object
        * A Effect which can have one or several action (haptic, beep...)
        */
        return new function feedback() {

            /**
             * Start the feedback effect
             * @param {Event} clickEvent The associated Event (DOM Event)
             */
            this.feedback = function feedback(clickEvent) {
                if (that.getFeedbackState()) {
                    if ((that.getHapticState() || that.getBeepState()) && !ICTouchAPI.feedbackServices._safetyLock) {
                        ICTouchAPI.feedbackServices._safetyLock = true;
                        setTimeout(function() {
                            ICTouchAPI.feedbackServices._safetyLock = false;
                        }, ICTouchAPI.feedbackServices.SAFETY_DELAY);
                        var type = that._getFeedbackType(feedbackType);
                        if (that.getHapticState() && window.Haptic != undefined) {
							// Haptic.testClick() is used to test different Haptic effects through the whole software
							if (Haptic.testClick != undefined && webapp.haptic != undefined) {
								Haptic.testClick(webapp.haptic.data.intChoiceListSelected, clickEvent.clientX, clickEvent.clientY)
							}
							else {
								Haptic[type["hapticMethod"]](
									that._getFeedbackEffectValue(type["hapticEffect"]), clickEvent.clientX, clickEvent.clientY);
							}
                        }
                        if(that.getBeepState() && window.LowLevelBeep != undefined && LowLevelBeep.click != undefined) {
                            LowLevelBeep.click(0);
                        }
                    }
                }
            }

            /**
             * Stop the feedback effect
             */
            this.stop = function stop() {
                if (that.getFeedbackState()) {
                    if (that.getHapticState()) {
                        if (Haptic != undefined) {
                            Haptic.stopEffect();
                        }
                    }
//                    if (that.getBeepState()) {
//                        if (LowLevelBeep != undefined) {
//                            LowLevelBeep.stopEffect();
//                        }
//                    }
                }
            }
        }
    },

/* ------------------------------------ Private Methods --------------------------------------- */
    /**
     * @ignore
     * change the haptic state
     */
    _hapticValueReceived: function(hapticValue) {
        if (hapticValue != undefined) {
            this._hapticState = hapticValue;
        }
    },

    /**
     * @ignore
     * change the beep state
     */
     _beepValueReceived: function(beepValue) {
        if (beepValue != undefined) {
            this._beepState = beepValue;
        }
    },

    /**
     * @ignore
     * change the haptic state
     */
    _hapticValueChanged: function(hapticSetting) {
        this._hapticState = hapticSetting.jsValue;
    },

     /**
      * @ignore
      * change the beep state
      */
     _beepValueChanged: function(beepSetting) {
        this._beepState = beepSetting.jsValue;
    },

    /**
     * @param {String} feedback : the feedback's name to get the parameters.
     * @ignore
     */
    _getFeedbackType: function (feedback) {
        return (this._feedbackTypes[feedback] || this._feedbackTypes[this._defaultType]);
    },

    /**
     * @param {String} effect
	 * @return: the feedback's integer expected by IctWebClient associated to the given effect.
     * @ignore
     */
    _getFeedbackEffectValue: function (effect) {
        return (this._feedbackEffectsValue[effect] || this._feedbackEffectsValue[this._defaultEffect]);
    }

});

ICTouchAPI.feedbackServices = new ICTouchAPI.feedbackServices();
