/**
* @class ICTouchAPI.eventServices
* @singleton
* @extends Object
* The Events class provide methods to subscribe and unsubscribe to backend events
*/
dojo.provide("ICTouchAPI.eventServices");
dojo.declare("ICTouchAPI.eventServices",null,{

/* --------------------------------- Public attributes ------------------------------------ */

/* --------------------------------- Private attributes ----------------------------------- */

	// List of subscribedEvents
	/**
	 * @ignore
	 */
	_arrSubscribedEvents: {},
	/**
	 * @ignore
	 */
	_xhr : null,
	/**
	 * @ignore
	 */
	_arrChannels: [],
	/**
	 * @ignore
	 */
	_nbChannels: 2,
	/**
	 * @ignore
	 */
	_lastChannel: null,
	/**
	 * @ignore
	 */
	_currChannel: null ,
	/**
	 * @ignore
	 */
	_nextChannel: null, // logCleanUp
	// Number of processed events
	/**
	 * @ignore
	 */
	_arrNbProcessedEvents: [],
	/**
	 * @ignore
	 */
	_arrNbStoredEvents: [],
	/**
	 * @ignore
	 */
	_areAllEventsTreated: false,
	/**
	 * @ignore
	 */
	_reconnectRequested: false,
	// {int} Temporization between retries to avoid freezing the ictouch in milliseconds.
	/**
	 * @ignore
	 */
	timeBetweenSeriesOfRetries: 2000,
	// {int} nb of retries between temporizations.
	/**
	 * @ignore
	 */
	nbOfConnexionRetries: 3,
	// {int} nb of current retries done before temporization.
	/**
	 * @ignore
	 */
	_tmpNbOfRetriesDone: 0,
	/**
	 * @ignore
	 */
	_gotCookie : false,
	/**
	 * @ignore
	 */
	_lastChannelAccess : null,

/* ------------------------------------ Constructor --------------------------------------- */

	// Initiliaze parameters
	/**
	 * @ignore
	 */
	constructor : function () {
		this._arrSubscribedEvents = {};
		this._lastChannel = this._nbChannels - 1;
		this._currChannel = this._lastChannel;
		this._nextChannel = (this._currChannel + 1) % this._nbChannels ; // logCleanUp
		
                // No channel for simulation mode
                if(!generalConfig.simulation) {
                    this._cookieWait();
                    this._arrChannels[this._nextChannel] = this._openChannel(); // logCleanUp
                }
		for(var i=0;i<this._nbChannels;i++) {
			// logCleanUp
			this._arrNbProcessedEvents[i] = 0 ;
			this._arrNbStoredEvents[i] = 0 ;
		}

                // No channel for simulation mode
                if(!generalConfig.simulation) {
                    this._initializeChannel();
                }
	},

/* ----------------------------------- Getter / Setter------------------------------------- */


/* ----------------------------------- Public methods ------------------------------------- */

    /**
	 * Method used by webapps to subscribe to a backend event
	 * @param {Object} ctx : callback execution context
	 * @param {String} strEventName : name of the subscribed event
	 * @param {Function} callback : function called when the event occurs
	 */
	subscribeToEvent : function(ctx, strEventName, callback){
		if(!this._arrSubscribedEvents[strEventName]){
			this._arrSubscribedEvents[strEventName]=new ICTouchAPI.eventServices.registeredEvent(strEventName);
		}
		this._arrSubscribedEvents[strEventName].addWebApp(ctx, callback);
		ctx=null;
		strEventName=null;
		callback=null;
	},

    /**
	 * Method used by webapps to unsubscribe to a backend event
	 * @param {Object} objWebApp : webapp execution context
	 * @param {String} strEventName : name of the subscribed event
	 */
	unsubscribeToEvent : function(objWebApp, strEventName){
		if(this._arrSubscribedEvents[strEventName]){
			// Removing webapp's subscription
			this._arrSubscribedEvents[strEventName].removeWebApp(objWebApp.getApplicationName());
			// If no webapp remaining
			if (!this._arrSubscribedEvents[strEventName].getNbWebapps()) {
				// Deleting the registered event
				delete this._arrSubscribedEvents[strEventName];
			}
		}
		objWebApp=null;
		strEventName=null;
	},

	/**
	 * Return list of Events Subscribed to the ICTGate
	 * @ignore
	 * @return {Array}
	 */
	listSubscribedEvents: function(){
		return this._arrSubscribedEvents;
	},

	/**
	 * Return the list of webapplications which subscribe to a particular event
	 * @ignore
	 * @param {String} strEventName
	 * @return {Array}
	 */
	listWebAppsForEvent: function(strEventName){
		return this._arrSubscribedEvents[strEventName];
	},

	/**
	 * Method used by the ICTGate for core event publishing
	 * @ignore
	 * @param {String} strEventName Name of the event which occurs
	 * @param {Object} params
	 */
	_corePublishEvent : function(strEventName, params){
		ICTouchAPI.log("Publishing " + strEventName, "EVENTS");
		dojo.publish("eventServices." + strEventName, params);
	},

/* --------------------------------- Private Methods -------------------------------------- */

	/**
	 * @ignore
	 * Send logs of attributes of the service to the logger.
	 * @param {boolean} boolAdvancedLogs Boolean indicating if advanced or basic logs has to be written
	 */
	dump : function(boolAdvancedLogs){
		ICTouchAPI.debugServices.dump("dump function of eventServices with " + ((boolAdvancedLogs)?"advanced":"basic") + " logs option");

		ICTouchAPI.debugServices.dump("Number of channels (_nbChannels): " + this._nbChannels);
		ICTouchAPI.debugServices.dump("Current channel used (_currChannel): " + this._currChannel);
		ICTouchAPI.debugServices.dump("Previous channel used (_lastChannel): " + this._lastChannel);
		ICTouchAPI.debugServices.dump("Next channel used (_nextChannel): " + this._nextChannel);
		ICTouchAPI.debugServices.dump("Last channel accessed (_lastChannelAccess): " + this._lastChannelAccess);
		ICTouchAPI.debugServices.dump("Number of events stored for the current channel (_arrNbStoredEvents[_currChannel]): " + this._arrNbStoredEvents[this._currChannel]);
		ICTouchAPI.debugServices.dump("Number of events processed for the current channel (_arrNbProcessedEvents[_currChannel]): " + this._arrNbProcessedEvents[this._currChannel]);
		ICTouchAPI.debugServices.dump("Are all events treated ? (_areAllEventsTreated): " + this._areAllEventsTreated);
		ICTouchAPI.debugServices.dump("Is reconnexion requested (_reconnectRequested): " + this._reconnectRequested);
		ICTouchAPI.debugServices.dump("Number of connexion retries done: " + this._tmpNbOfRetriesDone);

		if (boolAdvancedLogs) {
			// List of events currently subscribed
			ICTouchAPI.debugServices.dump("List of events currently subscribed:");
			for (var strEvent in this._arrSubscribedEvents) {
				ICTouchAPI.debugServices.dump("Event: " + strEvent + " is currently subscribed by " + this._arrSubscribedEvents[strEvent]._nbWebapps + " applications:");
				for (var application in this._arrSubscribedEvents[strEvent]._applications) {
					ICTouchAPI.debugServices.dump(" - " + application);
				}
			}
		}
	},

	/**
	 * @ignore
	 */
	_cookieWait:function(){
		var xhrtmp = new XMLHttpRequest();
		xhrtmp.open('GET',
			ICTouchAPI.eventChannelURI + "services/ICTGate/openSession",
			false
		);
		xhrtmp.send(null);
	},

	/**
	 * Open a fresh new pending channel
	 * @ignore
	 */
	_openChannel: function(){
		var newxhr = new XMLHttpRequest();
		newxhr.open('GET', ICTouchAPI.eventChannelURI + 'services/ICTGate/openEventChannel');
		// For use with cookies
		newxhr.withCredentials = true;
		newxhr.onreadystatechange = this.channelUpdated;
		newxhr.send(null);
		return newxhr;
	},

	/**
	 * switch to next channel
	 * @ignore
	 */
	_switchToNextChannel: function(){
		var tmp = this._currChannel ;
		this._currChannel++ ;
		this._currChannel = this._currChannel % this._nbChannels ;
		this._xhr = this._arrChannels[this._currChannel] ;
		if(this._arrChannels[tmp]){
			this._arrChannels[tmp].abort(); // logCleanUp
		}
		this._arrChannels[tmp] = this._openChannel();
	},

	/**
	 * Initialize the event channel with de ICTGate
	 * @ignore
	 */
	_initializeChannel: function(){
		ICTouchAPI.log("Initialize channel", "EVENTS");
		// Reset our flags
		this._arrNbProcessedEvents[this._currChannel] = 0;
		this._arrNbStoredEvents[this._currChannel] = 0;
		this._areAllEventsTreated= false;
		this._reconnectRequested=false;
		this._switchToNextChannel();
	},

	/**
	 * @ignore
	 */
	_myChannel: function(pxhr) {
		for(var i=0;i<ICTouchAPI.eventServices._nbChannels;i++) {
			if(ICTouchAPI.eventServices._arrChannels[i] == pxhr){
				ICTouchAPI.eventServices._lastChannelAccess=i ;
				return i;
			}
		}
	},

	/**
	 * @ignore
	 */
	channelUpdated : function(){
		ICTouchAPI.log("Channel updated", "EVENTS");

		ICTouchAPI.logObject(this, "EVENTS");

		/* events are waiting to be processed */
		if(this.readyState >= 3 ) {
			// Count the number of events currently stored in the eventServices._xhr object
			ICTouchAPI.eventServices._arrNbStoredEvents[ICTouchAPI.eventServices._myChannel(this)] = 0;
			var stop = 0;
			while (stop != -1) {
				stop = this.responseText.indexOf("((eventEnd))", stop+1);
				if (stop != -1) {
					ICTouchAPI.eventServices._arrNbStoredEvents[ICTouchAPI.eventServices._myChannel(this)]++;
				}
			}

			/*If channel got closed and all events have been processed, re open it.
			Else, set a flag to indicate that the channel has to be re open after all events have been processed.*/
			if (this.readyState == 4) {
				if (ICTouchAPI.eventServices._arrNbProcessedEvents[ICTouchAPI.eventServices._myChannel(this)] >= ICTouchAPI.eventServices._arrNbStoredEvents[ICTouchAPI.eventServices._myChannel(this)]) {
					ICTouchAPI.debugServices.debug("ICTouchAPI.eventServices - channelUpdated / (currentChannel: " + ICTouchAPI.eventServices._currChannel + "), readyState == 4 and all events has been processed ("+ICTouchAPI.eventServices._arrNbProcessedEvents[ICTouchAPI.eventServices._myChannel(this)]+">="+ICTouchAPI.eventServices._arrNbStoredEvents[ICTouchAPI.eventServices._myChannel(this)]+"). ICTGate channel is full, call _initializeChannel.");
					ICTouchAPI.eventServices._initializeChannel();
				} else {
					ICTouchAPI.debugServices.debug("ICTouchAPI.eventServices - channelUpdated / (currentChannel: " + ICTouchAPI.eventServices._currChannel + "), readyState == 4 and events are waiting to be processed ("+ICTouchAPI.eventServices._arrNbProcessedEvents[ICTouchAPI.eventServices._myChannel(this)]+"<"+ICTouchAPI.eventServices._arrNbStoredEvents[ICTouchAPI.eventServices._myChannel(this)]+"). ICTGate channel is full, request to reconnect.");
					ICTouchAPI.eventServices._reconnectRequested = true;
				}
			}

			// for each event stored
			for (var i=ICTouchAPI.eventServices._arrNbProcessedEvents[ICTouchAPI.eventServices._myChannel(this)]+1; i <= ICTouchAPI.eventServices._arrNbStoredEvents[ICTouchAPI.eventServices._myChannel(this)]; ++i) {

				// if this event is part of the unprocessed events
				if (i > ICTouchAPI.eventServices._arrNbProcessedEvents[ICTouchAPI.eventServices._myChannel(this)] ) {
					var firstIndex = 0;
					var secondIndex = 0;
					// we make our indexes go to the target event
					for (var j=0; j < i; j++) {
						firstIndex = this.responseText.indexOf("((eventStart))", secondIndex);
						secondIndex = this.responseText.indexOf("((eventEnd))", firstIndex);
					}
					// the target event is parsed and processed
					var temp = this.responseText.slice(firstIndex + 14, secondIndex);

					/*There's a trick there :
					The counter has to be increased before hands, otherwise an other
					callback may process it again even though it's already be sent.*/
					// Increase our processed events counter
					ICTouchAPI.eventServices._arrNbProcessedEvents[ICTouchAPI.eventServices._myChannel(this)]++;

					// Throws an error if bad formatted event (Webapp : crms00211314)
					try{
						// Convert the event to an event object.
						var currentEventObject=dojo.fromJson(temp);
					}catch(_error){
						ICTouchAPI.debugServices.error('ICTouchAPI.eventServices - channelUpdated / ' + _error + ' converting ' + temp,true);
						if(_error.stack) {
							ICTouchAPI.debugServices.error(_error.stack);
						}
						continue;
					}

					ICTouchAPI.debugServices.debug("ICTouchAPI.eventServices - channelUpdated / (currentChannel: " + ICTouchAPI.eventServices._currChannel + ", event number " + ICTouchAPI.eventServices._arrNbProcessedEvents[ICTouchAPI.eventServices._myChannel(this)] + "), process event: " + currentEventObject.eventName);
					try{
						// Publish the event.
						ICTouchAPI.eventServices._corePublishEvent(currentEventObject.eventName,currentEventObject.params);
					}catch(_error){
						ICTouchAPI.debugServices.error('ICTouchAPI.eventServices - channelUpdated / ' + _error + ' treating ' + temp,true);
						if(_error.stack)
							ICTouchAPI.debugServices.error(_error.stack);
						continue;
					}

					// If all events have been treated, set a flag.
					if (ICTouchAPI.eventServices._arrNbProcessedEvents[ICTouchAPI.eventServices._myChannel(this)] == ICTouchAPI.eventServices._arrNbStoredEvents[ICTouchAPI.eventServices._myChannel(this)]) {
						ICTouchAPI.eventServices._areAllEventsTreated = true;
						// If a reconnection request was previously set, then reconnect.
						if (ICTouchAPI.eventServices._reconnectRequested) {
							ICTouchAPI.eventServices._initializeChannel();
						}
					}else{
						// If there are still events to be treated
						ICTouchAPI.eventServices._areAllEventsTreated = false;
					}
				}
			}
		}
	},

	/**
	 * @ignore
	 */
	reconnect: function () {
		// If connexion got closed, re-initialize the event channel.
		// If the number of retries done is > to desired number of retries
		if (ICTouchAPI.eventServices._tmpNbOfRetriesDone >= ICTouchAPI.eventServices.nbOfConnexionRetries) {
			// Set a tempo before setting the number of retries done to 0 and then re-init the channel.
			setTimeout(function () {
				ICTouchAPI.eventServices._tmpNbOfRetriesDone = 0;
				ICTouchAPI.eventServices._initializeChannel();
			},
			ICTouchAPI.eventServices.timeBetweenSeriesOfRetries);

		// Else, if number of retries done is <, then retry.
		} else  if (ICTouchAPI.eventServices._tmpNbOfRetriesDone < ICTouchAPI.eventServices.nbOfConnexionRetries) {
			ICTouchAPI.eventServices._initializeChannel();
			ICTouchAPI.eventServices._tmpNbOfRetriesDone  +=1 ;
		}
	}

});

// Create Event Service.
ICTouchAPI.eventServices=new ICTouchAPI.eventServices();

dojo.provide("ICTouchAPI.eventServices.registeredEvent");
dojo.declare("ICTouchAPI.eventServices.registeredEvent",null,{


/* --------------------------------- Public attributes ------------------------------------ */

/* --------------------------------- Private attributes ----------------------------------- */

	// Name of the event
	/**
	 * @ignore
	 */
	_strEventName : "",
	// Applications that have subscribed to it
	/**
	 * @ignore
	 */
	_applications : {},
	// Number of applications using this event
	/**
	 * @ignore
	 */
	_nbWebapps : 0,

/* ------------------------------------ Constructor --------------------------------------- */

	/**
	 * @ignore
	 * Create a new event object
	 * @param {String} strEventName Name of the subscribed Event
	 * @return {Object} new event object
	 */
	constructor: function(strEventName){
		ICTouchAPI.log("Subscribing to " + strEventName, "EVENTS");
		// Saving the event name
		this._strEventName = strEventName;
		// Initializing attributes
		this._nbWebapps = 0;
		this._applications = {};
		// Subscribing to the event
                // No channel for simulation mode
                if(!generalConfig.simulation) {
                    this._coreEventSubscription();
                }
		return this;
	},

/* ----------------------------------- Getter / Setter------------------------------------- */

/* ----------------------------------- Public methods ------------------------------------- */

	/**
	 * Add a webApp to the current Event
	 * @ignore
	 * @param {Object} ctx
	 * @param {Object} callback
	 */
	addWebApp: function(ctx,callback){
		// Name of the application
		var name = ctx.getApplicationName();
		// If there's already a subscription
		if(this._applications[name]){
			// Unsubscribe the previous one
			dojo.unsubscribe(this._applications[name]);
		}
		// And replace it
		this._applications[name]=dojo.subscribe("eventServices." + this.getEventName(),ctx,callback);
		// Increase counter
		this._increaseNbWebapps();

		return this;
	},

	/**
	 * Remove a webApp to the current Event
	 * @ignore
	 * @param {String} webAppName
	 */
	removeWebApp: function(webAppName){
		// If there's such application
		if(this._applications[webAppName]){
			// Unsubscribe and delete its callback
			dojo.unsubscribe(this._applications[webAppName]);
			delete this._applications[webAppName];
			// If that was the last application using the event
			if (!this._decreaseNbWebapps()) {
				// Unsubscribe to this event so the eventServices won't receive it anymore
				this._coreEventUnsubscription();
			}
			return this;
        }
    },

/* --------------------------------- Private Methods -------------------------------------- */

	/**
	 * Subscribe to the event notification from the ICTGate
	 * @ignore
	 */
	_coreEventSubscription: function(){
		var xhrSubscribe = new XMLHttpRequest();
		xhrSubscribe.open('GET',
			ICTouchAPI.eventChannelURI + "services/ICTGate/subscribeToEvent?EventName={\"type\":\"QString\",\"value\":\""+this.getEventName()+"\"}&SocketId={\"type\":\"QString\",\"value\":\"0:0\"}",
			true
		);
		xhrSubscribe.send(null);
	},

	/**
	 * Unsubscribe to the event notification from the ICTGate
	 * @ignore
	 */
	_coreEventUnsubscription: function(){
		var xhrUnsubscribe = new XMLHttpRequest();
		xhrUnsubscribe.open('GET',
			ICTouchAPI.eventChannelURI + "services/ICTGate/unsubscribeToEvent?EventName={\"type\":\"QString\",\"value\":\""+this.getEventName()+"\"}&SocketId={\"type\":\"QString\",\"value\":\"0:0\"}",
			true
		);
		xhrUnsubscribe.send(null);
	},

	/**
	 * @ignore
	 */
	getEventName : function () {
		return this._strEventName;
	},

	/**
	 * @ignore
	 */
	_increaseNbWebapps : function () {
		return ++this._nbWebapps;
	},

	/**
	 * @ignore
	 */
	_decreaseNbWebapps : function () {
		return --this._nbWebapps;
	},

	/**
	 * @ignore
	 */
	getNbWebapps : function () {
		return this._nbWebapps;
	}

});
