/**
* @class ICTouchAPI.debugServices
* @singleton
* @extends Object
* The debugServices class provides utilities functions to log messages and JavaScript objects at different levels.
* <br/><br/>
* The phone manages the following log levels:<br/>
* <pre>
* - debug
* - info
* - notice
* - warning
* - err (error)
* - emerg (emergency)
* </pre>
* <br/>
* By default, only logs with levels greater than or equal to <b>err</b> will be saved into the log file <i>/var/log/WebApp.log</i>.
* <br/>
* <br/>
* To see others logs, you should:<br/>
* - Set the setting <b>developerMode</b> to true<br/>
* - Set the desired log level(ie: <b>debug</b> level allows to see all logs).<br/>
* <br/>
* You can change default log level by using the following command line: <i>level WebApp <b>new_level</b></i><br/>
* <br/>
* The following example changes the log level to <b>debug</b>:
* <pre>
* // Set the log level to <b>debug</b>
* level WebApp debug
* </pre>
* <br/>
* For example, if you change the log level to <b>notice</b>, all logs using levels <b>notice</b>, '<b>warning</b>, <b>err</b> and <b>emergency</b> will be saved into the log file.
* <br/>
* <br/>
* Logs with level greater than or equal to <b>err</b> will be saved into the defensive log file located in <i>/log/Defence.log</i>.
* <br/>
* <br/>
* The following example logs a simple text in the log file
* <br/>
* <pre>
* // To log a simple text into the log file
* ICTouchAPI.debugServices.info("My text");
* </pre>
* <br/>
* The following example logs a JSON object in the log file
* <br/>
* <pre>
* //To log a JSON object into the log file
* ICTouchAPI.debugServices.info(myObj);
* </pre>
*
*/
dojo.provide("ICTouchAPI.debugServices");
window.ICTouchAPI.debugServices = {

        // {Boolean} boolNotToJson is a private parameter than enables to not use dojo.toJson() on the parameter (avoir the creation of big structure)

        /**
	 * Write an info message to the log file.<br/>
         * Only enabled when the setting <b>developerMode</b> is set to <b>True</b>.
	 * @param {String/Object} info The information message/object to log
	 */
	info: function(info, boolNotToJson) {
		this._sendMessage(info, 1, boolNotToJson);
		info = null;
		boolNotToJson = null;
	},

	/**
	 * Write a debug message to the log file.<br/>
         * Only enabled when the setting <b>developerMode</b> is set to <b>True</b>.
	 * @param {String/Object} debug The debug message/Object to log
	 */
	debug: function(debug, boolNotToJson) {
		this._sendMessage(debug, 0, boolNotToJson);
		debug = null;
		boolNotToJson = null;
	},

	/**
	 * Write a notice message to the log file.<br/>
         * Only enabled when the setting <b>developerMode</b> is set to <b>True</b>.
	 * @param {String/Object} notice The notice message/Object to log
	 */
	notice: function(notice, boolNotToJson) {
		this._sendMessage(notice, 2, boolNotToJson);
		notice = null;
		boolNotToJson = null;
	},

	/**
	 * Write a warning message to the log file.<br/>
         * Only enabled when the setting <b>developerMode</b> is set to <b>True</b>.
	 * @param {String/Object} warning The warning message/Object to log
	 */
	warning: function(warning, boolNotToJson) {
		this._sendMessage(warning, 3, boolNotToJson);
		warning = null;
		boolNotToJson = null;
	},

	/**
	 * Write an error message to the log file.
	 * @param {String/Object} error The error message/Object to log
	 */
	error: function(error, boolNotToJson) {
		this._sendMessage(error, 4, boolNotToJson);
		error = null;
		boolNotToJson = null;
	},

	/**
	 * write an emergency message to the log file.<br/>
	 * @param {String/Object} emergency The emergency message/Object to log
	 */
	emergency: function(emergency, boolNotToJson) {
		this._sendMessage(emergency, 5, boolNotToJson);
		emergency = null;
		boolNotToJson = null;
	},
        
	/**
	 * Send an alert on the screen
	 * @param {String} toSend Object (boolToJson = true) or string to send
         * @ignore
	 */
	alert: function(toSend){
		alert(dojo.toJson(toSend));
		toSend = null;
	},

	/**
	 * Send an info in a log file directly by webkit. This method is used to dump information (called by webapp's dump method).
         * This method is not removed by the javascript code compressor.
	 * @param {String} toSend Object (boolNotToJson != true) or string to send
	 * @param {Boolean} boolNotToJson use dojo.toJson, in order to get a string from an object
         * @ignore
	 */
	dump: function(toSend, boolNotToJson) {
		this._sendMessage(toSend, 1, boolNotToJson);
		toSend = null;
		boolNotToJson = null;
	},

	/**
	 * @ignore
	 */
	_sendMessage : function(toSend, level, boolNotToJson){
		var strMessage = toSend;
		if(!boolNotToJson){
			strMessage = dojo.toJson(toSend);
		}
		if(ICTouchAPI.remoteDisplay){
			if (level == 0) {
				console.log("debug[|]"+strMessage);
			}
			if (level == 1) {
				console.info("info[|]"+strMessage);
			}
			if (level == 2) {
				console.info("notice[|]"+strMessage);
			}
			if (level == 3) {
				console.warn("warning[|]"+strMessage);
			}
			if (level == 4) {
				console.error("error[|]"+strMessage);
			}
			if (level == 5) {
				console.error("emergency[|]"+strMessage);
			}
		}
		else {
			console.log(level+"[|]"+strMessage);
		}
		toSend = null;
		level = null;
		boolNotToJson = null;
	}

};

// Global function allowing to call the dump function of the given webapp to log its main data (according to the dump function defined in the webapp)
dumpWebapp = function (strWebappName, boolAdvancedLogs) {
	if (webapp[strWebappName] && typeof webapp[strWebappName].dump === "function") {
		webapp[strWebappName].dump(boolAdvancedLogs);
	}
	else {
		ICTouchAPI.debugServices.warn('ICTouchAPI.dumpWebapp: webapp ' + strWebappName + " does not exist or does not have any dump function");
	}
};

// Global function allowing to call the dump function of the given service to log its main data (according to the dump function defined in the service)
dumpService =  function (strServiceName, boolAdvancedLogs) {
	if (ICTouchAPI[strServiceName] && typeof ICTouchAPI[strServiceName].dump === "function") {
		ICTouchAPI[strServiceName].dump(boolAdvancedLogs);
	}
	else {
		ICTouchAPI.debugServices.warn('ICTouchAPI.dumpService: service ' + strServiceName + " does not exist or does not have any dump function");
	}
};
