/**
* @class ICTouchAPI.dataStoreServices
* @extends Object
* @singleton
* The DataStore class allows to get cache set of object records
*/
dojo.provide("ICTouchAPI.dataStoreServices");
dojo.declare("ICTouchAPI.dataStoreServices", null, {


	/**
	 * @ignore
	 */
	_publicStores : {},

	/**
	 * @ignore
	 */
	constructor : function () {

	},



	/**
	 * Get a reference on a Store object named 'name'
	 * @param {String} name The name of the data store
	 * @return {ICTouchAPI.Store} A reference on a Store object named 'name'
	 */
	getStore : function (name) {
		var store = this._publicStores[name];
		name=null;
		return store;
	},

	/**
	 * Create a Store object
	 * @ignore
	 * @param {Object} dsConfigObj<pre><code>
	 * name {String} name of the data store
	 * rowId {String} name of the data store record primary key
	 * mapping {Boolean} automatically create columns from object properties
	 * dataProvider {Object} data provider definition
	 * </code></pre>
	 * @return {Object} a reference on the freshly created Store object
	 */
	createGlobalDataStore : function (dsConfigObj) {
		if (!dsConfigObj.name) {
			throw new Error("a public dataStore must have a name");
		} else {
			if (dsConfigObj.name in this._publicStores) {
				return this.getStore(dsConfigObj.name);
			} else {
				return  this._publicStores[dsConfigObj.name] = this._createStore(dsConfigObj);
			}
		}
	},

	/**
		 * @class ICTouchAPI.Store
		 * @extends Object
		 * The Store class provides methods to handle a DataStore object and to fire events on DataStore object changes
		 */
	/**
	 * @event E_ITEM_ADDED
	 * fires when a new record is added
	 * @param {Object} record The new added record
	 */
	/**
	 * @event E_ITEM_REMOVED
	 * fires when a new record is removed
	 */
	/**
	 * @event E_ITEM_UPDATED
	 * fires when a new record is updated
	 * @param {Object} record The updated record
	 */
	/**
	 * @event E_DATA_LOADED
	 * fires when data ready to use or when huge amount of data changed
	 * @param {Object} records All the records contained into the datastore
	 */

	/**
	 * @ignore
	 */
	createDataStore : function (dsConfigObj) {
		return this._createStore(dsConfigObj);
	},

	/**
	 * @ignore
	 */
	_createStore : function (dsConfigObj) {
		return new this._storeFactory(dsConfigObj);
	},

	/**
	 * @ignore
	 */
	_storeFactory : function( dsConfigObj ) {

		var name = null;

		/* Internal objects */
		var Evt; // Event handler
		var Store; // Data
		var DataModel; // Data model
		var Record; // Record that define a row based on the column model
		var DataProvider; // Manager used to synchronize store with an external source


		/*
		 * EVENT
		 * Handle all event publishing and subscription
		 *
		 * The reason why dojo is not used is because we needed one "Hub" for each dataStore
		 * and not one Hub for all. That'd have required a unique id/name for the dataStores
		 * and this could'nt be guaranteed.
		 *
		 * Plus, using dojo's hub allows webapps to connect on dataStore's event to do nasty stuff.
		 *
		 * This implementation of the observer design pattern might be the lightest one
		 *
		 */

		Evt = new function Evt () {

			var topics = {};

			/* list of event fired */
			this.EVT_ITEM_ADDED = "itemAdded";
			this.EVT_ITEM_REMOVED = "itemRemoved";
			this.EVT_ITEM_UPDATED = "itemUpdated";
			this.EVT_ITEMS_LOADED ="itemsLoaded";

			/**
			 * Publish datas on the specified event to the hub
			 * @ignore
			 * @param {String} evtName
			 * @param {Object} dataObj
			 */
			this.publish =  function( evtName, dataObj ) {
				// If the topic exits
				if (topics[evtName]) {
					// The func that'll be called by the forEach loop on the topic list
					var func = function (handler) {
						// Call the callback in its native context
						handler[2].apply(handler[1], dataObj);
					}
					// Execute all callbacks
					topics[evtName].forEach(func);
				}
			};

			/**
			 * Provide subscrition to the specified event
			 * @param {String} evtName Event name
			 * @param {Object} c_context Callback context
			 * @param {Function} c_func Function called when event fired
			 * @return {Array} handler Handler to remove listener later
			 */
			this.addListener = function( evtName, c_context, c_func ) {
				// Check if topic exists, and if not, create an empty array.
				topics[evtName] = topics[evtName] ||[];
				// Create the handler that'll store all the information.
				var handler = [evtName, c_context, c_func];
				// Push the handler in the topic
				topics[evtName].push(handler);
				// Return the handler, it'll be usefull for removing the listener
				return handler;
			};

			/**
			 * Provide unsubscription to the specified event node
			 * @param {Array} handler Handler given by event listener adder
			 * @return {Boolean} action status
			 */
			this.removeListener = function( handler ) {
				// If topic exists
				if (topics[handler[0]]) {
					// Get the index of the handler
					var index = topics[handler[0]].indexOf(handler);
					// If the handler is in the array
					if (index >= 0) {
						// Remove it
						topics[handler[0]].splice(index, 1);
						// Warn it's been successful
						return true;
					} else {
						return false;
					}
				}else {
					return false;
				}
			};

			// Just for debugging purpose
			/**
			 * @ignore
			 */
			this.debug = function debug() {
			//				console.log(topics);
			}

		};



		/*
		 * STORE
		 * Provide methods for managing datas and fire related events
		 * Datas are managed with a simple structure :
		 *  - data: an array with natural index
		 *  - indexes: an object used to make link between columnIndex and data[] indexes
		 */

		Store = {

			events: true,
			mapFromData: false,

			/* raw datas */
			data: [],
			/* list of columnIndex */
			indexes: [],

			/**
			 * Add a record at the end of the store
			 * @param {Object} recordObj Record object
			 * @return {Boolean} action status
			 */
			add: function( recordObj ) {
				var added = false;

				if( Store.mapFromData === true )
				{
					if(!DataModel.boolModelDefined){
						var model = [];

						/* browse object properties that will be used as columns */
						for( var column in recordObj ) {
							model.push( {
								index: column
							} );
						}

						/* Create the data model */
						DataModel.create( model );
					}
				}

				/* Add a row at the end of data[] */
				if( undefined !== recordObj[ DataModel.indexColumn ] ) {
					Store._addTo( recordObj, recordObj[ DataModel.indexColumn ] );
					added = true;
				}

				/* EVENT - Operation succeed */
				if( added === true ) {
					/* Fire ADD event if enabled */
					if( Store.events === true ) {
						var data = {};
						data.value = recordObj[ DataModel.indexColumn ];
						Evt.publish( Evt.EVT_ITEM_ADDED, [recordObj] );
					}
				}

				return added;
			},


			/**
			 * Add a record to the store at the specified index
			 * @ignore
		 * @param {Number} index
			 * @param {Object} recordObj
			 */
			insert: function( index, recordObj ) {},


			/**
			 * Edit column(s) of record(s) that matches column(s) defined in the search object
			 * @ignore
			 * @param {Object} searchObj
			 * @param {Object} updateObj
			 */
			edit: function( searchObj, updateObj ) {},


			/**
			 * Edit column(s) for the record at the specified index
			 * @param {Object} recordIndex Record primary key value
			 * @param {Object} updateObj Record object
			 * @return {Boolean} action status
			 */
			editAt: function( recordIndex, updateObj ) {
				var updated = false;
				var objNew = {};

				if( Store._indexExists(recordIndex) === true )
				{
					/* get the current data model for validate properties to update (and check) */
					var columns = DataModel.get();

					/* Browse each requested property to update */
					for( var prop in updateObj ) {

						/* Check if property to update exists in data model */
						if( undefined !== columns[prop]) {

							/* Set the property of the requested record with its new value */
							if(Store._setProp( Store._getIndex(recordIndex), prop, updateObj[prop] )){
								updated = true;
								objNew[prop] = updateObj[prop];
							}
						}
					}
					/* EVENT - Operation succeed */
					if( updated === true ) {
						/* Fire UPDATE event if enabled */
						if( Store.events === true ) {
							var data = {};
							data.value = recordIndex ;
							var object = this.getAt(recordIndex);
							Evt.publish( Evt.EVT_ITEM_UPDATED, [object, objNew] );
						}
					}
				}
				return updated;
			},


			/**
			 * Remove record(s) that matches columns defined in the search object
			 * @ignore
			 * @param {Object} searchObj
			 */
			remove: function( searchObj ) {},


			/**
			 * Remove record at the specified index
			 * @param {Object} recordIndex Record primary key value
			 * @return {Boolean} action status
			 */
			removeAt: function( recordIndex ) {

				var removed = false;

				/* Check if a record exists with this index */
				if( Store._indexExists(recordIndex) === true )
				{
					/* Remove record from data... */
					Store.data.splice( Store._getIndex(recordIndex), 1 );
					/* ... and update the index */
					Store.indexes.splice( Store._getIndex(recordIndex), 1 );
					//delete Store.indexes[ recordIndex ];

					removed = true;

					/* EVENT - Fire REMOVE event if enabled */
					if( Store.events === true ) {
						Evt.publish( Evt.EVT_ITEM_REMOVED, [recordIndex] );
					}
				}

				return removed;

			},


			/**
			 * Return record at the specified index
		 * @param {Number} recordIndex Record primary key value
			 * @return {Object} record reference for the specified index
			 */
			getAt: function( recordIndex ) {

				var result = false;

				if( Store._indexExists( recordIndex ) === true ) {
					result = new Record( Store.data[ Store._getIndex(recordIndex) ] );
				}

				return result;
			},


			/**
			 * Returns a list of records
			 * @param {Object} configObj
			 * <pre><code>
                 * start {Number} start batch value
			 * max {Integer} batch size
			 * </code></pre>
			 * @return {Object} record references list, returns all if param is missing
			 */
			getAll: function( configObj ) {

				var list = [],
				listStart = 0,
				listMax = Store._length();

				/* Config is defined, using it for paginate */
				if( undefined !== configObj ) {
					if( undefined !== configObj.start && parseInt( configObj.start,10) ) {
						listStart = configObj.start;
					}
					if( undefined !== configObj.max && parseInt( configObj.max,10) && configObj.max <= listMax ) {
						listMax = configObj.max;
					}
				}

				/* Returns the portion of items in list */
				for( var i = listStart; i < listMax; i++ ) {
					list.push( Store.data[i] );
				}

				return list;
			},


			/**
			 * Search for records that matches
			 * @ignore
			 * @param {Array} arrFieldNames
			 * @param {String} search
			 */
			find: function( arrFieldNames, search ) {
				var func = function (item) {
					// If this item's attribute equals..
					for(var i = 0; i<arrFieldNames.length; i++){
						if (item[arrFieldNames[i]] === search) {
							return item;
						}
					}
				}
				return Store.data.filter(func);
			},

			/**
			 * @ignore UNTESTED!!
			 * Load data from another source and create column model from it
			 * @param {Array} arrDataList List of additional record objects to store
			 */
			loadData: function( arrDataList ) {

				Store._flush();

				/* Data model have to be defined from datas */
				if( Store.mapFromData === true )
				{
					var model = [];

					/* browse object properties that will be used as columns */
					for( var column in arrDataList[0] ) {
						model.push( {
							index: column
						} );
					}

					/* Create the data model */
					DataModel.create( model );
				}

				// Inserting data in store / No events will be fired until the end of addition
				Store.disableEvents();

				for( var i in arrDataList ) {
					Store.add( arrDataList[i] );
				}

				Store.enableEvents();

				/* All items have been added - Fire global event data loaded */
				Evt.publish( Evt.EVT_ITEMS_LOADED, [Store.data] );
			},



			/**
			 * @ignore
			 */
			enableEvents: function() {
				//				console.log( "[Datastore][Store] Enabling events" );
				Store.events = true;
			},

			/**
			 * @ignore
			 */
			disableEvents: function() {
				//				console.log( "[Datastore][Store] Disabling events" );
				Store.events = false;
			},


			/**
			 * Add record to store, add columnIndex to index
			 * @ignore
			 * @param {Object} recordObj
		 * @param {Number} columnIndex
			 */
			_addTo: function( recordObj, columnIndex ) {
				Store.data.push( recordObj );
				Store.indexes.push( columnIndex.toString() );
			},

			/**
			 * Flush store and index
			 * @ignore
			 */
			_flush: function() {
				Store.data = [];
				Store.indexes = [];
			},

			/**
			 * Returns the total number of items
			 * @ignore
			 */
			_length: function() {
				return Store.data.length;
			},

			/**
			 * Check if the recordIndex exists in store
			 * @ignore
			 * @param {String} recordIndex
			 */
			_indexExists: function( recordIndex ) {
				return (-1 !== Store.indexes.indexOf( recordIndex.toString() ) ) ? true : false;
			},

			/**
			 * Set property of a record at the specified index
			 * @ignore
		 * @param {Number} index
			 * @param {String} propName
			 * @param {String} propValue
		 * @return {Boolean} wether or not a change has been performed.
			 */
			_setProp: function( index, propName, propValue ) {
				if(Store.data[index][propName] != propValue){
					Store.data[index][propName] = propValue;
					return true;
				} else {
					return false;
				}
			},

			/**
			 * @ignore
			 */
			_getIndex: function( recordIndex ) {
				return Store.indexes.indexOf( recordIndex.toString() );
			}

		};



		/*
		 * COLUMN MODEL
		 * Used to define data organisation and to create records
		 */

		DataModel = {

			indexColumn: "",

			arrDatatypes: [ "integer", "string" ],

			/* defined columns for the store */
			arrColumns: {},

			boolModelDefined: false,

			/* optional functions used to validate data type */
			arrValidators: [],

			/**
			 * Create model for the store based on the config
			 * @ignore
			 * @param {Object} configObj
			 */
			create: function( configObj ) {
				//				console.log( "[Datastore][DataModel] Creating Model", configObj );

				for( var columnIndex in configObj ) {
					this.arrColumns[ configObj[ columnIndex].index ] = {};
				}
				if(configObj.length){
					this.boolModelDefined = true;
				}
			},

			/**
			 * Add a column to the model
			 * @param {Object} columnConfigObj
			 * @ignore
			 */
			addColumn: function( columnConfigObj ) {
				if(columnConfigObj && !DataModel.arrColumns[columnConfigObj.index]){
					DataModel.arrColumns[columnConfigObj.index] = {};
				}
			},

			/**
			 * Remove a column from the model
			 * @ignore
			 * @param {String} columnName
			 */
			removeColumn: function( columnName ) {},

			/**
			 * Add a data type validator for the specified column
			 * Will be used each time this column data is changed
			 * @ignore
			 * @param {String} columnName
			 * @param {Object} configObj
			 */
			addTypeValidator: function( columnName, configObj ) {},


			/**
			 * Returns the current model
			 * @ignore
			 */
			get: function() {
				return DataModel.arrColumns;
			}

		};



		/*
		 * RECORD
		 * Object based on data model and returned by Store queries
		 */

		/**
 * @ignore
		 */
		Record = function( data ) {

			/* Set all properties based on column model */
			for( var columnIndex in DataModel.arrColumns ) {
				this[ columnIndex ] = null;
			}

			this.getIndexName = function() {
				return DataModel.indexColumn;
			};

			// and set all values of this record
			for( var index in data ) {
				if( undefined !== this[index] ) {
					this[ index ] = data[ index ];
				}
			}

		};



		/*
		 * DATA PROVIDER
		 *
		 */

		DataProvider = {

			enabled: false,
			params: {},


			/**
			 * Set the config (events and callbacks)
			 * @ignore
			 * @param {Object} configObj
			 */
			configure: function( configObj ) {
				//				console.log( "[Datastore][DataProvider]", configObj );

				/* provider is enabled if config is valid */
				this.enabled = true;

				if( undefined !== configObj.onInit ) {
					this.params = configObj;
				}
			},

			/**
			 * @ignore
			 */
			getApplicationName: function () {
				return this.name;
			},

			/**
			 * @ignore
			 */
			init: function() {

				if( undefined !== this.params )	{
					if( undefined !== this.params.onInit ) {
						var params = {};
						params.callback = this.onInit;
						params.context = this;
						params.params = [-1,-1];

						this.params.onInit.func.call( this.params.onInit.context || window, params );
					}

					if( undefined !== this.params.onCreate ) {
						this.params.onCreate.event.forEach(function(evt) {
							ICTouchAPI.eventServices.subscribeToEvent(this, evt, this.dataCreate);
						},this);
					}
					if( undefined !== this.params.onUpdate ) {
						this.params.onUpdate.event.forEach(function(evt) {
							ICTouchAPI.eventServices.subscribeToEvent(this, evt, this.dataUpdate);
						},this);
					}
					if( undefined !== this.params.onDelete ) {
						this.params.onDelete.event.forEach(function(evt) {
							ICTouchAPI.eventServices.subscribeToEvent(this, evt, this.dataDelete);
						},this);
					}
					if( undefined !== this.params.onSynchro ) {
						this.params.onSynchro.event.forEach(function(evt) {
							ICTouchAPI.eventServices.subscribeToEvent(this, evt, this.dataSynchro);
						},this);
					}
				}
			},

			/**
			 * @ignore
			 */
			onInit: function(event) {
				/* If not empty, remove all items */
				Store.data = [];
				Store.indexes = [];

				/*  */
				Store.loadData( event );
			},

			/**
			 * @ignore
			 */
			dataSynchro: function(event) {
				var values;
				if(this.params.onCreate.argumentName){
					var objArguments = ICTouchAPI.tools.getEventArguments(arguments);
					values = objArguments[this.params.onCreate.argumentName];
				} else {
					values = event.value;
				}
				if(this.params.onInit.func){
					var params = {};
					params.callback = this.onInit;
					params.context = this;
					params.params = [-1,-1];

					this.params.onInit.func.call( this.params.onInit.context || window, params );
				} else {
					this.onInit.call(this, values);
				}
			},

			/**
			 * @ignore
			 */
			dataCreate: function(event) {
				var values;
				if(this.params.onCreate.argumentName){
					var objArguments = ICTouchAPI.tools.getEventArguments(arguments);
					values = objArguments[this.params.onCreate.argumentName];
				} else {
					values = event.value;
				}
				if(this.params.onCreate.func){
					var params = {};
					params.callback = this.onDataCreate;
					params.context = this;
					params.params = [values]; // contact_id, like "#CONTACT_1#"

					this.params.onCreate.func.call( this.params.onCreate.context || window, params );
				} else {
					this.onDataCreate.call(this, values);
				}
			},

			/**
			 * @ignore
			 */
			onDataCreate: function(event) {
				while(event.length > 0){
					Store.add( event.pop() );
				}
			},

			/**
			 * @ignore
			 */
			dataUpdate: function(event) {
				var values;
				if(this.params.onUpdate.argumentName){
					var objArguments = ICTouchAPI.tools.getEventArguments(arguments);
					values = objArguments[this.params.onUpdate.argumentName];
				} else {
					values = event.value;
				}
				if(this.params.onUpdate.func){
					var params = {};
					params.callback = this.onDataUpdate;
					params.context = this;
					params.params = [values]; // contact_id, like "#CONTACT_1#"

					this.params.onUpdate.func.call( this.params.onUpdate.context || window, params );
				} else {
					this.onDataUpdate.call(this, values);
				}
			},

			/**
			 * @ignore
			 */
			onDataUpdate: function(event) {
				var obj;
				while(event.length > 0){
					obj = event.pop();
					Store.editAt( obj.contactId, obj );
				}
			},

			/**
			 * @ignore
			 */
			dataDelete: function(event) {
				var values;
				if(this.params.onDelete.argumentName){
					var objArguments = ICTouchAPI.tools.getEventArguments(arguments);
					values = objArguments[this.params.onDelete.argumentName];
				} else {
					values = event.value;
				}
				while(values.length > 0){
					Store.removeAt(values.pop()); // contact_id, like "#CONTACT_1#"
				}
			}
		};

		/*
		 * Constructor
		 * @ignore
		 * @param {Object} configObj
		 */
		(function( configObj ) {
			name = configObj.name;

			/* Column definition if defined in this config object */
			if( undefined !== configObj.columns ) {
				DataModel.create( configObj.columns );
			}

			/* Activate auto mapping when Store load data */
			else if( undefined !== configObj.mapping && configObj.mapping === true ) {
				Store.mapFromData = true;
			}

			/* Set row index name */
			if( undefined !== configObj.rowId ) {
				DataModel.indexColumn = configObj.rowId;
			}

			/* Configure the data provider */
			if( undefined !== configObj.dataProvider ) {
				DataProvider.configure( configObj.dataProvider );
			}


			init();


		})( dsConfigObj );



		/**
		 * @ignore
		 */
		function init() {
			if( DataProvider.enabled ) {
				/* Call DataProvider init - this will start data loading and update event listenning */
				DataProvider.init();
			}
		}

		/*
		 *  Public interface ------------------------------
		 *  Provide public access on only necessary methods.
		 *  This interface will vary depending on access rights (read only, read/write)
		 */

		/* Event - access for all */

		this.addEventListener = Evt.addListener;
		this.removeEventListener = Evt.removeListener;

		/* Event - constants for listeners */
		this.E_DATA_LOADED = Evt.EVT_ITEMS_LOADED;
		this.E_ITEM_ADDED = Evt.EVT_ITEM_ADDED;
		this.E_ITEM_UPDATED = Evt.EVT_ITEM_UPDATED;
		this.E_ITEM_REMOVED = Evt.EVT_ITEM_REMOVED;


		/* Store - access for all */
		this.getList = Store.getAll;
		this.getAt = Store.getAt;
		this.find = Store.find;

		/* Store - write access required */
		this.add = Store.add;
		//this.insert = Store.insert;
		this.editAt = Store.editAt;
		//this.remove = Store.remove;
		this.removeAt = Store.removeAt;
		this.load = Store.loadData;

		/* DataModel - write access required */
		this.addColumn = function(obj){
			DataModel.addColumn(obj)
		};
	}
});
ICTouchAPI.dataStoreServices = new ICTouchAPI.dataStoreServices();

