/**
 * @class ICTouchAPI.cssHandler
 * @extends Object
 * @ignore
 */
dojo.provide("ICTouchAPI.cssHandler");
dojo.declare("ICTouchAPI.cssHandler",null,
{

	/* --------------------------------- Public attributes ------------------------------------ */


	/* --------------------------------- Private attributes ----------------------------------- */
	_headerTag				: null,
	__basic					: null,
	_icons					: null,
	_linkedBasic			: null,
	_linkedIcons			: null,
	_arrListener			: null,
	_lockEventToDOM			: null,

	/* ------------------------------------ Constructor --------------------------------------- */

	/**
	 * This method create the cssHandler and initialize it
	 */
	constructor: function (headerTag){
		this._headerTag		= headerTag;

		this._basic			= {};
		this._icons			= {};
		this._linkedBasic	= {};
		this._linkedIcons	= {};

		this._arrListener	= [];
		this._lockEventToDOM = false;

		this._arrListener.push(dojo.subscribe("RemoveSkinCss",this,this.onRemoveSkinCssEvent));
	},


	/* ----------------------------------- Getter / Setter------------------------------------- */

	/**
	 * return a clone of the queried fragment
	 * @param {string} typeCss type of the expected css to be returned : basic, icons, linkedIcons, linkedBasic
	 * @param {string} skinName Name of the skin
	 * @return {HTMLElement} domFragment fragment holding the css related to the handlers
	 */
	getCss : function(typeCss,skinName){
		switch(typeCss){
			case "basic" :
				return dojo.clone(this._basic[skinName]);
			case "icons" :
				return dojo.clone(this._icons[skinName]);
			case "linkedIcons" :
				return dojo.clone(this._linkedIcons[skinName]);
			case "linkedBasic" :
				return dojo.clone(this._linkedBasic[skinName]);
		}
	},
	
	
	/**
	 * Return the managed tag
	 * @return {string} tag
	 */
	getTag : function(){
		return this._headerTag;
	},

	/**
	 * Set or unset a lock on the event publication to DOM elements.
	 * Do NOT lock event to other handlers.
	 * Publish linkedbasic and linkedicons event when the lock is set off.
	 * @param {boolean} value state of the lock inkedbasic and linkedicons event only for specified skins
	 * @param {Array} arrSkins Array o string, publish
	 */
	setLock : function(value,arrSkins){		
		var i;
		this._lockEventToDOM = value;
		if(value == false){
			if (arrSkins==null) {
				for (i in this._linkedIcons) {
					this.publishChangedEvent("linkedIcons",i);
				}
				for (i in this._linkedBasic) {
					this.publishChangedEvent("linkedBasic",i);
				}
			}
			else {
				for (i in arrSkins) {
					this.publishChangedEvent("linkedIcons",arrSkins[i]);
					this.publishChangedEvent("linkedBasic",arrSkins[i]);
				}
			}
		}

	},


	/* ----------------------------------- Public methods ------------------------------------- */

	/**
	 * Add links based on an array of files and publish an event to let know our targeted links we got new css
	 * @param {string} namespace_key0 first part ot the directory path to find the given css files (ICTouchAPI, webapp or UIElements)
	 * @param {string} namespace_key1 second part ot the directory path to find the given css files (Spinner, contacts, settings ...)
	 * @param {string} skinName name of the skin
	 * @param {array} arrayOfFiles array of the names of the files to be loaded and managed by this handler
	 */
	loadCssOnArrayOfFile : function(namespace_key0,namespace_key1,skinName,arrayOfFiles){		
		dojo.forEach(arrayOfFiles,function(fileName){
			if(fileName == "icons.css"){
				dojo.place(this._createLinkTag(namespace_key0,namespace_key1,skinName,fileName,"icons"),this._linkedIcons[skinName]);
				this._clean("linkedIcons",skinName);
				dojo.place(this._createLinkTag(namespace_key0,namespace_key1,skinName,fileName,"icons"),this._icons[skinName]);
				this._clean("icons",skinName);
			}else{
				dojo.place(this._createLinkTag(namespace_key0,namespace_key1,skinName,fileName,"basic"),this._linkedBasic[skinName]);
				this._clean("linkedBasic",skinName);
				dojo.place(this._createLinkTag(namespace_key0,namespace_key1,skinName,fileName,"basic"),this._basic[skinName]);
				this._clean("basic",skinName);
			}
		},this);
		//publishing for linked cssHandlers
		this.publishChangedEvent("basic",skinName);
		this.publishChangedEvent("icons",skinName);
		//publishing to allow related iframes to update their css links
		this.publishChangedEvent("linkedIcons",skinName);
		this.publishChangedEvent("linkedBasic",skinName);
	},

	/**
	 * methode used to manage our events
	 * @param {string} type type of the dom fragment which had been updated
	 * @param {string} skinName name of the skin
	 */
	publishChangedEvent : function(type,skinName){
		if(!this._lockEventToDOM || (type != "linkedIcons" && type != "linkedBasic")) {
			dojo.publish(this._headerTag+"."+type,[{'owner' : this,'type' : type, 'skinName' : skinName}]);
		}
	},

	/**
	 * allow an other cssHandler to link its css to ours and add a subsciption to any changes on the css of the owner
	 * @param {cssHandler} ownerHeader reference on the owner handler of the shared css
	 * @param {boolean} onlyIcon used to exclude copy of basic css
	 * @param {boolean} onlyCss used to exclude copy of icons css
	 * @param {boolean} duplicateIcon used to copy the icons css to the local linkedBasic fragment
	 * @param {boolean} duplicateCss used to copy the basic css to the local linkedIcons fragment
	 *
	 */
	link : function(ownerHeader,onlyIcon,onlyCss,duplicateIcon,duplicateCss){
		var i,j;
		for(i in ownerHeader) {			
			var ownerHeaderTag = ownerHeader[i].getTag();
			
			if(!onlyIcon){
				this._arrListener.push(dojo.subscribe(ownerHeaderTag+".basic",this,this.onLinkedBasicChange));
				for (j in this._linkedBasic) {
					this._placeLinkedBasic({'owner' : ownerHeader[i],'type' : "basic",skinName:j});
				}
			}
			if(!onlyCss){
				this._arrListener.push(dojo.subscribe(ownerHeaderTag+".icons",this,this.onLinkedIconsChange));
				for (j in this._linkedIcons) {
					this._placeLinkedIcons({'owner' : ownerHeader[i],'type' : "icons",skinName:j});
				}
			}
			if(duplicateIcon){
				this._arrListener.push(dojo.subscribe(ownerHeaderTag+".icons",this,this.onLinkedBasicChange));
				for (j in this._linkedBasic) {
					this._placeLinkedBasic({'owner' : ownerHeader[i],'type' : "icons",skinName:j});
				}
			}
			if(duplicateCss){
				for (j in this._linkedIcons) {
					this._arrListener.push(dojo.subscribe(ownerHeaderTag+".basic",this,this.onLinkedIconsChange));
				}
				this._placeLinkedIcons({'owner' : ownerHeader[i],'type' : "basic",skinName:j});
			}
		}
		
		for (i in this._linkedBasic) {
			this._clean("linkedBasic",i);			
			this.publishChangedEvent("linkedBasic",i);
		}		
		for (i in this._linkedIcons) {
			this._clean("linkedIcons",i);			
			this.publishChangedEvent("linkedIcons",i);
		}
		
	},

	/**
	 * when a css owner has been updated. For appbar.
	 * @param {object} objEvent object sent with the caught event
	 */
	onLinkedIconsChange : function(objEvent){
		this._placeLinkedIcons(objEvent);
		this._clean("linkedIcons",objEvent.skinName);
		this.publishChangedEvent("linkedIcons",objEvent.skinName);
	},
	

	/**
	 * when a css owner has been updated. for webapps.
	 * @param {object} objEvent object sent with the caught event
	 */
	onLinkedBasicChange : function(objEvent){
		this._placeLinkedBasic(objEvent);
		this._clean("linkedBasic",objEvent.skinName);
		this.publishChangedEvent("linkedBasic",objEvent.skinName);
	},

	/**
	 * when skin removed, update fragments
	 * @param {stringt} skinName skin to remove
	 */
	onRemoveSkinCssEvent : function(skinName){
		this.removeCss(skinName);
	},


	/**
	 * Destroy  all fragments in the handler of specific skin. (Destroy child links)
	 *
	 */
	removeCss : function(skinName){
		dojo.destroy(this._basic[skinName]);
		delete this._basic[skinName];
		dojo.destroy(this._icons[skinName]);
		delete this._icons[skinName];
		dojo.destroy(this._linkedBasic[skinName]);
		delete this._linkedBasic[skinName];
		dojo.destroy(this._linkedIcons[skinName]);
		delete this._linkedIcons[skinName];
	},

	/**
	 * Destroy all fragments in the handler. (Destroy child links)
	 * 
	 */
	removeAllCss : function(){
		var i;
		for (i in this._basic) {
			dojo.destroy(this._basic[i]);
			delete this._basic[i];
		}
		for (i in this._icons) {
			dojo.destroy(this._icons[i]);
			delete this._icons[i];

		}
		for (i in this._linkedBasic) {
			dojo.destroy(this._linkedBasic[i]);
			delete this._linkedBasic[i];
		}
		for (i in this._linkedIcons) {
			dojo.destroy(this._linkedIcons[i]);
			delete this._linkedIcons[i];
		}
	},

	/**
	 * Create dom fragments for a skin if doesn't exist
	 * @param {String} skinName Name of skin
	 */
	createSkinFragments : function(skinName) {
		if(this._basic[skinName] == null) {			
			this._basic[skinName] = document.createDocumentFragment();
		}
		if(this._icons[skinName] == null) {			
			this._icons[skinName] = document.createDocumentFragment();
		}
		if(this._linkedBasic[skinName] == null) {			
			this._linkedBasic[skinName] = document.createDocumentFragment();
		}
		if(this._linkedIcons[skinName] == null) {			
			this._linkedIcons[skinName] = document.createDocumentFragment();
		}
	},



	/* --------------------------------- Private Methods -------------------------------------- */

	/**
	 * Create the HTMLElement for the css
	 * @param {string} namespace_key0 first part ot the directory path to find the given css files (ICTouchAPI, webapp or UIElements)
	 * @param {string} namespace_key1 second part ot the directory path to find the given css files (Spinner, contacts, settings ...)
	 * @param {string} skinName name of the skin
	 * @param {string} fileName name of the file to be targeted by the link element
	 * @param {string} cssType type of CSS ("basic" or "icons")
	 */
	_createLinkTag: function (namespace_key0,namespace_key1,skinName,fileName,cssType) {
		if(namespace_key1 == "minifiedCss" && !generalConfig.cssDeveloper){
			return dojo.create("link",{
				'csstag':this._headerTag,
				'cssType':cssType,
				'rel':'stylesheet',
				'type':'text/css',
				'skin':skinName,
				'href':dojo.moduleUrl(namespace_key0+"."+namespace_key1,fileName).path
				}
			);
		}else{
			return dojo.create("link",{
				'csstag':this._headerTag,
				'cssType':cssType,
				'rel':'stylesheet',
				'type':'text/css',
				'skin':skinName,
				'href':dojo.moduleUrl(namespace_key0+(namespace_key1 ? "."+namespace_key1 : "" ),"themes/"+skinName+"/css/"+fileName).path
				}
			);
		}
		
	},

	/**
	 * Local getter to return a reference (not a copy)
	 * @param {string} typeCss type of the expected css to be returned : basic, icons, linkedIcons, linkedBasic
	 * @param {string} skinName name of the skin
	 * @return {HTMLElement} domFragment fragment holding the css related to the handlers
	 */
	_getCss : function(typeCss,skinName){
		switch(typeCss){
			case "basic" :
				return this._basic[skinName];
			case "icons" :
				return this._icons[skinName];
			case "linkedIcons" :
				return this._linkedIcons[skinName];
			case "linkedBasic" :
				return this._linkedBasic[skinName];
		}
	},
	
	/**
	 * Clean a fragment from any doublon of link
	 * @param {string} typeCss type type of the local fragment to be cleaned
	 * @param {string} skinName name of the skin
	 */
	_clean : function(typeCss,skinName){
		var frag = this._getCss(typeCss,skinName);
		var found = {};
		dojo.query("link", frag).forEach(function(linkTag){
			if(this.found[linkTag.href]){
				dojo.destroy(linkTag);
			}else{
				this.found[linkTag.href] = true;
				dojo.attr(linkTag,"csstag",this._headerTag);

			}
		},{'found' : found,'_headerTag' : this._headerTag});
	},

	_placeLinkedIcons : function(objEvent) {
		var skinName = objEvent.skinName;
		var ownerNode = objEvent.owner.getCss(objEvent.type,skinName);
		if(ownerNode!=null && this._linkedIcons[skinName]!=null) {
			dojo.place(ownerNode,this._linkedIcons[skinName]);
		}
	},

	_placeLinkedBasic : function(objEvent) {
		var skinName = objEvent.skinName;
		var ownerNode = objEvent.owner.getCss(objEvent.type,skinName);
		if(ownerNode!=null && this._linkedBasic[skinName]!=null) {
			dojo.place(ownerNode,this._linkedBasic[skinName]);
		}
	},

});

