/**
* @class ICTouchAPI.VideoServices
* @namespace ICTouchAPI
* @extends Object
* @ignore
*/
dojo.provide("ICTouchAPI.VideoServices");
dojo.declare("ICTouchAPI.VideoServices", null , {


	/* --------------------------------- Public attributes ------------------------------------ */

	/* --------------------------------- Private attributes ----------------------------------- */


	/**
	 * {Integer} Set to TRUE if video is currently shown.
	 * @private
	 */
	_videoDisplayed: 0,

	/**
	 * {Boolean} Set to TRUE if video com is ongoing (displayed or not).
	 * @private
	 */
	_videoComActive: false,


	_eventListenerHandleList: {},

	/**
	 * {Object} webapp show/hide event listener handle
	 * @private
	 */
	_webappEventListenerHandle: null,

	/**
	 * {Object} toaster show/hide event listener handle
	 * @private
	 */
	_toasterEventListenerHandle: null,

	/**
	 * {Object} popup show/hide event listener handle
	 * @private
	 */
	_popupEventListenerHandle: null,


	/**
	 * {Integer} webapp show/hide status code (100 = webapp is shown, 0 = webapp hidden)
	 * @private
	 */
	_webappShowStatus: 0,

	/**
	 * {Integer} toaster show/hide status code (10 = toaster is shown, 0 = toaster hidden)
	 * @private
	 */
	_toasterShowStatus: 0,

	/**
	 * {Integer} popup show/hide status code (1 = popup is shown, 0 = popup hidden)
	 * @private
	 */
	_popupShowStatus: 0,

	/**
	 * {Boolean} Is a Video mask set ? (false = no, true = yes)
	 * @private
	 */
	_maskSetStatus: false,

	/**
	 * {Object} Object used to store video details (position, size).
	 * Used to restore properly depending on webapp showing it
	 * @private
	 */
	_statesList: {},

	/**
	 * {String} audio toaster id
	 * @private
	 * @todo Is this id really static ?
	 */
	_audioToasterId: "webapp_audiosetting_getAudioToaster_0",

	_strReservedVideoUI: null,
	_bVideoDisplayedOnce: false,
	_VIDEO_OFF: 0, // Video is not displayed on the screen and video escalation is disabled
	_VIDEO_ON: 1, // Video is displayed on the screen and video escalation is enabled
	_VIDEO_HIDDEN: 2, // Video is not displayed on the screen but video escalation has been negociated

	/* EVENT LIST */
	_EVENT_CREATE_LOCAL: "ICTouchAPI.APIServices.Video.setLocalVideo",
	_EVENT_DELETE_LOCAL: "ICTouchAPI.APIServices.Video.deleteLocalVideo",
	_EVENT_WEBAPP_SHOW: "ICTouchAPI.transitionServices.WEBAPP_SHOW",
	_EVENT_WEBAPP_HIDE: "ICTouchAPI.transitionServices.WEBAPP_HIDE",
	_EVENT_TOASTER_SHOW: "toaster.show",
	_EVENT_TOASTER_HIDE: "toaster.hide",
	_EVENT_POPUP_SHOW: "dialogbox.show", // SHOW_POPUP_EVENT
	_EVENT_POPUP_HIDE: "dialogbox.hide", // HIDE_POPUP_EVENT


	/* ------------------------------------ Constructor --------------------------------------- */

	constructor : function () {

		this._videoDisplayed = this._VIDEO_OFF;

		this._webappShowStatus = 100;
		this._toasterShowStatus = 0;
		this._popupShowStatus = 0;

	},

	/* ----------------------------------- Getter / Setter------------------------------------- */

	/**
     * @public
	 * Because the eventServices wants an application name we provide one.
	 *
	 * @return {String} Return VideoServices
	 */
	getApplicationName: function() {
		return "VideoServices";
	},



	/* ----------------------------------- Public methods ------------------------------------- */


	/**
	 * Configure look and feel
	 *
	 * @param {Integer} background_color ARGB Widget background color
	 * @param {String} text main window displayed text when no video
	 * @param {Integer} text_color text ARGB color
	 * @param {String} pip_text PiP area displayed text when no video
	 */
	setParams: function( background_color, text, text_color, pip_text ) {
		ICTouchAPI.debugServices.info('ICTouchAPI.VideoServices - setParams');
		ICTouchAPI.APIServices.Video.setVideoMMIParams( background_color, text, text_color, pip_text );
	},


	/**
	 * @public
	 * Create and/or show local video if possible
	 *
	 * @param {Object} paramsObj contains x,y,w,h
	 */
	showLocal: function( paramsObj ) {
		ICTouchAPI.debugServices.info('ICTouchAPI.VideoServices - showLocal');
		return this.show( paramsObj, true );
	},


	/**
     * @public
	 * Create and/or show a video if possible (no video currently displayed)
	 *
	 * @param {Object} paramsObj contains x,y,w,h (callId, sipCallId, pip, ...)
	 * @param {Boolean} isLocal (optional) Set to TRUE if video is local. Default value is FALSE.
	 * Forced to FALSE if paramsObj.callId exists
	 * @param {Boolean} addOffset (optional) Set to FALSE if there is no need to calculate the absolute
	 * position of the iFrame that will be displayed behind the video. Default value is TRUE.
	 */
	show: function( paramsObj, isLocal, addOffset, _strOverrideCurrentScreen ) {
		ICTouchAPI.debugServices.info('ICTouchAPI.VideoServices - show');
		this._bVideoDisplayedOnce = true;
		var setResult = false;

		/* A video is already displayed */
		if(this._getVideoDisplayed() == this._VIDEO_ON) {
			/** @todo something to do in this case ? */
			/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - show / video is already displayed - callId: ' + paramsObj.callId);
		}

		else if( this._isVideoComActive() === true ) {
			if( this._isShowable() === true )
			{
				/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - show / video ready to be shown - callId: ' + paramsObj.callId);
						
				var current_screen_name;
				if(!_strOverrideCurrentScreen) {
					current_screen_name = this._getCurrentScreen();
				}
				else {
					current_screen_name = _strOverrideCurrentScreen;
				}
				/* Check if this webapp previously shown a video and restore it like it was */
				var state = this._getState(current_screen_name);

				if( state !== null )
				{
					/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - show / saved state found for '+ current_screen_name );

					if( state.isLocal === false ) {
						/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - show / showing video... (call showVideo)' );

						var setResult = ICTouchAPI.APIServices.Video.showVideo( state.params );

						if( setResult == true ) {
							this._postShow( current_screen_name, state.params, false );
						}
					}

				}

				else {
					/* DEBUG */  ICTouchAPI.debugServices.warning( 'ICTouchAPI.VideoServices - show / no saved state found for '+ current_screen_name );
				}

			}
			/** @todo something to do in this case ? */
		}

		/* No video is currently displayed - creating one */
		else
		{
			/* force isLocal to FALSE if a callId is defined */
			if( undefined !== paramsObj.callId ) {
				isLocal = false;
			} else if( undefined === isLocal ) {
				isLocal = true;
			}

			var current_screen_name;
			var displayedScreen;

			if(!_strOverrideCurrentScreen) {
				displayedScreen = this._getCurrentScreen();
			}
			else {
				displayedScreen = _strOverrideCurrentScreen;
			}
			
			if(paramsObj.callerScreen) {
				current_screen_name = paramsObj.callerScreen;
			}
			else {
				current_screen_name = displayedScreen;
			}			
			
			if(current_screen_name === displayedScreen) {
				//Get UI coordinate
				if(paramsObj.callerUI) {
					var UI = dijit.byId(paramsObj.callerUI);
					var params = UI.getStyle();
					if (params) {
						this.setParams( params.bgColor, params.mainMessage, params.textColor, params.pipMessage );
					}
					var objCoordinate = UI.getCoordinate();
					dojo.mixin(paramsObj, objCoordinate);
					UI = null;
					params = null;
				}
				
				/* Iframe offset if already defined - no need to calculate it */
				if( undefined !== addOffset && addOffset === false ) {					
				}
				/* Get offset of the iframe containing the UIElement */
				else {
					var frameOffset = this._getFrameOffset( current_screen_name );
					paramsObj.x += frameOffset[0];
					paramsObj.y += frameOffset[1];
				}

				/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - show / callId: ', paramsObj.callId );

				// LOCAL VIDEO
				if( isLocal === true ) {
					//Don't create video if it's reserved by an UI and the current UI isn't the right
					if(!this._strReservedVideoUI || this._strReservedVideoUI === paramsObj.callerUI ) {
						ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - show / call createLocalVideo');
						setResult = ICTouchAPI.APIServices.Video.createLocalVideo( paramsObj );
						this._strReservedVideoUI = null;
					}
					else {
						ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - show / do not call createLocalVideo');
						setResult = false;
					}
				}

				// VIDEO CALL
				else {
					if( undefined === paramsObj.pip ) {paramsObj.pip = 1;}
					if( undefined === paramsObj.mute ) {paramsObj.mute = 0;}

						//Don't create video if it's reserved by an UI and the current UI isn't the right
						if(!this._strReservedVideoUI || this._strReservedVideoUI === paramsObj.callerUI ) {
							ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - show / call createVideo');
							setResult = ICTouchAPI.APIServices.Video.createVideo( paramsObj );
							this._strReservedVideoUI = null;
						}
						else {
							ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - show / do not call call createVideo');
							setResult = false;
						}

					if( setResult === true ) {
						this._setVideoComActive(true);
					}
				}

				/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - show / setResult: ' + setResult);

				/* video has been created - set video states, register event listeners */
				if( setResult === true )  {
					this._postShow( current_screen_name, paramsObj, isLocal );
				}
			}

			//On first show, when target view isn't displayed
			else {
				this._bVideoDisplayedOnce = false;
				//Display the video on the next view open
				setResult = true;
				this._saveState( current_screen_name, {params: paramsObj, 'isLocal': isLocal, 'video': this._VIDEO_HIDDEN} );
				//Reserve the video
				this._strReservedVideoUI = paramsObj.callerUI;
				
				this._webappEventListenerHandle = this._registerEventListener(
					this._EVENT_WEBAPP_SHOW,
					function(evt) {
						
						/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - show / webapp first show event - _EVENT_WEBAPP_SHOW' );
						if( evt.id === current_screen_name ) {
							this._unregisterEventListener(this._webappEventListenerHandle);
							/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - show / webapp first show event - video ready to be shown' );
							//wait transitionServices update communication screen before call show function
							var context = this;
							setTimeout(function(){context.show(paramsObj, isLocal, addOffset)}, 0);							
						}
						else {
							/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - show / webapp first show event - default' );
						}						
					}	
				);
			}
		}

		if(paramsObj.callerUI) {
			if(setResult) {
				dijit.byId(paramsObj.callerUI).displayVideo();
			}
			else {
				dijit.byId(paramsObj.callerUI).displayError();
			}
		}
		return setResult;

	},


	/**
	 * @private
	 *
	 * @param {String} current_screen_name screen showing the video
	 * @param {Object} paramsObj contains x,y,w,h (callId, pip, ...)
	 * @param {Boolean} isLocal (optional) Set to TRUE if video is local. Default value is FALSE
	 * @param {String} closingToaster The name of the closing toaster, if any
	 *
	 */
	_postShow: function( current_screen_name, paramsObj, isLocal, closingToaster) {
		ICTouchAPI.debugServices.info('ICTouchAPI.VideoServices - _postShow');
		this._setVideoDisplayed(this._VIDEO_ON); // set video as displayed

		/* Considering that video is displayed, we can save video state and register listeners... */
		if( this._videoDisplayed )
		{
			this._saveState( current_screen_name, {params: paramsObj, 'isLocal': isLocal, 'video': this._VIDEO_ON} );

			/* Unregister previous existing listeners */
			this._unregisterShowHideListeners();

			/* Register webapp hide listener */
			this._webappEventListenerHandle = this._registerEventListener(
				this._EVENT_WEBAPP_HIDE, this._hideOnWebappHide
			);
			if (ICTouchAPI.toasterServices.getContentDisplayedId() && ICTouchAPI.toasterServices.getContentDisplayedId() != closingToaster) {
				/* Register toaster hide listener */
				this._hideOnToasterShow(ICTouchAPI.toasterServices.getContentDisplayedId());
			}
			else {
				/* Register toaster show listener */
				this._toasterEventListenerHandle = this._registerEventListener(
				this._EVENT_TOASTER_SHOW, this._hideOnToasterShow
			);
				}
			if (ICTouchAPI.popupServices.getNumberOfDisplayedPopups() > 0) {
				/* Register popup hide listener */
				this._hideOnPopupShow();
			}
			else {
				/* Register popup show listener */
				this._popupEventListenerHandle = this._registerEventListener(
					this._EVENT_POPUP_SHOW, this._hideOnPopupShow
				);
			}
			/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _postShow / video shown in ' + current_screen_name );
		}

	},

	/**
     * @public
	 * Hide current video and automatically re-show it if requested (autoRestoreVideo = TRUE).
	 *
	 * @param {Object} paramsObj
	 * @param {Boolean} autoRestoreVideo define if video can be restored automatically. Default value is FALSE.
	 * @param {String} strCurrentScreen The current screen name (optionnal)
	 */
	hide: function(paramsObj, autoRestoreVideo, strCurrentScreen) {
		ICTouchAPI.debugServices.info('ICTouchAPI.VideoServices - hide');
		if (typeof(strCurrentScreen) == 'undefined') {
			strCurrentScreen = this._getCurrentScreen();
		}

		if (paramsObj !== undefined) {
		    /* If paramsObj.videoStableState is set, then associate the videoStableState to the callId */
		    /* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - hide / callId: ' + paramsObj.callId);
		    var objState = (undefined !== paramsObj.callId) ? this._getStateByCallId(paramsObj.callId) : this._getState(strCurrentScreen);

		    if(objState !== null) {
				/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - hide / video: ' + paramsObj.videoStableState);
				objState.videoStableState = (undefined !== paramsObj.videoStableState) ? paramsObj.videoStableState : this._VIDEO_ON; // Save the stable video state
				this._saveState(strCurrentScreen,objState);
		    }
		}

		// if a mask has been set, disable it
		if (this._maskSetStatus === true) {
			ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - hide / call disableMask');
			this._maskSetStatus = !ICTouchAPI.APIServices.Video.disableMask( 1 );
		}
		if(this._getVideoDisplayed() == this._VIDEO_ON) {
			var hideResult = false,
			stateObj = this._getState( strCurrentScreen );

			/* stateObj will be used to define which type of video is displayed (local or call) */
			if( stateObj !== null ) {
				// LOCAL VIDEO
				if( stateObj.isLocal !== undefined && stateObj.isLocal === true ) {
				    	/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - hide / LOCAL VIDEO: call deleteLocalVideo');
					hideResult = ICTouchAPI.APIServices.Video.deleteLocalVideo();
				}

				// VIDEO CALL
				else {
					/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - hide / VIDEO CALL: call hideVideo with callId ' + stateObj.params.callId);
					hideResult = ICTouchAPI.APIServices.Video.hideVideo( stateObj.params.callId );
				}
			}

			/* video has been destroyed/hide */
			if( hideResult === true )
			{
				this._setVideoDisplayed(this._VIDEO_HIDDEN); // set video as not displayed

				/* Unregister previous existing listeners if auto restore not requested */
				if( undefined === autoRestoreVideo || autoRestoreVideo === false ) {
					this._unregisterShowHideListeners();
				}

				/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - hide / video hide success' );
			}

			else {
				/* DEBUG */ ICTouchAPI.debugServices.error( 'ICTouchAPI.VideoServices - hide / video hide error' );
			}
		} else {
			/* DEBUG */ ICTouchAPI.debugServices.warning( 'ICTouchAPI.VideoServices - hide / no video is currently displayed' );
			if( (undefined === autoRestoreVideo || autoRestoreVideo === false) && (objState && objState.videoStableState == this._VIDEO_HIDDEN )) {
                                this._unregisterShowHideListeners();
		}
		}

	},




	/**
     * @public
	 * Delete current displayed video
	 * @param {Number} callId The callId if any, undefined else
	 * @param {String} strScreen Optional, if undefined is proved in callId, the name of the screen associated to the video to destroy. If undefined, the currentScreen is used.
	 */
	destroy: function( callId, strScreen ) {
		ICTouchAPI.debugServices.info('ICTouchAPI.VideoServices - destroy');
		ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - destroy $ input function params | callId: ' + callId + ', strScreen:' + strScreen);
		if((this._getVideoDisplayed() == this._VIDEO_ON) || (this._getVideoDisplayed() == this._VIDEO_HIDDEN) || undefined !== callId || undefined != strScreen) {
			this._strReservedVideoUI = null;
			var destroyResult = false, stateObj = null, strStateId = null;
			if (undefined !== callId) {
				stateObj = this._getStateByCallId( callId );
				if (stateObj && stateObj.params) {
					strStateId = stateObj.params.callerScreen;
				}
			}
			else if (undefined !== strScreen) {
				stateObj = this._getState( strScreen );
				strStateId = strScreen;
			}
			else {
				stateObj = this._getState( this._getCurrentScreen() );
				strStateId = this._getCurrentScreen();
			}

			/* stateObj will be used to define which type of video is displayed (local or call) */
			if( stateObj !== null )
			{
				/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - destroy / state object found' );

				// VIDEO CALL
				if( stateObj.params.callId !== undefined ) {
					if(this._bVideoDisplayedOnce){
						/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - destroy / calling deleteVideo with callId '+ stateObj.params.callId );
						destroyResult = ICTouchAPI.APIServices.Video.deleteVideo( stateObj.params.callId );
					}
					if( destroyResult === true ) {
						this._setVideoComActive(false);
					}
				}

				// LOCAL VIDEO
				else {
					/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - destroy / calling deleteLocalVideo' );
					destroyResult = ICTouchAPI.APIServices.Video.deleteLocalVideo();
				}
			}

			/* video has been destroyed */
			if( destroyResult === true )
			{
				this._setVideoDisplayed(this._VIDEO_OFF); // set video as not displayed

				/* Unregister previous existing listeners */
				this._unregisterShowHideListeners();

				if( stateObj.params.callId !== undefined ) {
					// Destroy all the states with this callId
					this._deleteStateByCallId(stateObj.params.callId);
				}
				else {
					// Destroy the state by screen id
					this._deleteState(strStateId);
				}

				/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - destroy / video destroy success' );
			}

			else {
				/* DEBUG */  ICTouchAPI.debugServices.error( 'ICTouchAPI.VideoServices - destroy / video destroy error' );
			}
		}

		else {
			/* DEBUG */  ICTouchAPI.debugServices.warning( 'ICTouchAPI.VideoServices - destroy / no video is currently displayed' );
		}

	},




	/* ----------------------------------- Privates methods ------------------------------------- */

	/**
	 * @private
	 * For debug
	 */
	dump : function(boolAdvancedOption) {
		this.inherited(arguments);

		ICTouchAPI.debugServices.dump("Is video displayed (this._getVideoDisplayed()) (0: _VIDEO_OFF, 1: _VIDEO_ON, 2: _VIDEO_HIDDEN)? " + this._getVideoDisplayed());
		ICTouchAPI.debugServices.dump("Is video com active (displayed or not) (this._isVideoComActive())? " + this._isVideoComActive());
		ICTouchAPI.debugServices.dump("Shown status (this._getShowStatus()): " + this._getShowStatus());
		ICTouchAPI.debugServices.dump("Is video showable (i.e. if no content will be overlapped by the video area) (this._isShowable())? " + this._isShowable());

		if(boolAdvancedOption){
			ICTouchAPI.debugServices.dump("Events currently registered:");
			for (var evtName in this._eventListenerHandleList) {
				ICTouchAPI.debugServices.dump('Name of the event: '+ evtName);
				ICTouchAPI.debugServices.dump('Number of events registered for '+ evtName + ': ' + this._eventListenerHandleList[evtName].length);
				for(var i = 0; i < this._eventListenerHandleList[evtName].length; i++) {
					ICTouchAPI.debugServices.dump('Handler of events registered for '+ evtName + ', position ' + i + ': ' + this._eventListenerHandleList[evtName][i]);
				}
			}
			ICTouchAPI.debugServices.dump("Shown status:");
			ICTouchAPI.debugServices.dump(" - Webapp shown status (this._webappShowStatus): " + this._webappShowStatus);
			ICTouchAPI.debugServices.dump(" - Toaster shown status (this._toasterShowStatus): " + this._toasterShowStatus);
			ICTouchAPI.debugServices.dump(" - Popup shown status (this._popupShowStatus): " + this._popupShowStatus);

			ICTouchAPI.debugServices.dump("Registered events handlers:");
			ICTouchAPI.debugServices.dump(" - Webapp event listener (this._webappEventListenerHandle): " + this._webappEventListenerHandle);
			ICTouchAPI.debugServices.dump(" - Toaster event listener (this._toasterEventListenerHandle): " + this._toasterEventListenerHandle);
			ICTouchAPI.debugServices.dump(" - Popup event listener (this._popupEventListenerHandle): " + this._popupEventListenerHandle);

			ICTouchAPI.debugServices.dump("Is a Video mask set (this._maskSetStatus)? " + this._maskSetStatus);
			ICTouchAPI.debugServices.dump("_strReservedVideoUI: " + this._strReservedVideoUI);
			ICTouchAPI.debugServices.dump("_bVideoDisplayedOnce: " + this._bVideoDisplayedOnce);

			ICTouchAPI.debugServices.dump("_statesList (Object used to store video details (position, size)):");
			for (var state in this._statesList) {
				ICTouchAPI.debugServices.dump('Webapp id of the state: '+ state);
				ICTouchAPI.debugServices.dump('Params of the state: '+ dojo.toJson(this._statesList[state].params));
				ICTouchAPI.debugServices.dump('Is local video state? '+ this._statesList[state].isLocal);
				ICTouchAPI.debugServices.dump('Video state (0: _VIDEO_OFF, 1: _VIDEO_ON, 2: _VIDEO_HIDDEN): '+ this._statesList[state].video);
			}
		}
	},

	/**
     * @private
	 * Clear all variables of VideoServices (defense mechanism called by webapp communication when there is no ongoing call anymore)
	 */
	cleanVideoServices: function() {
		ICTouchAPI.debugServices.info('ICTouchAPI.VideoServices - cleanVideoServices');
		this._unregisterShowHideListeners();
		var stateObj, destroyResult;
		for (var screen_name in this._statesList) {
			stateObj = this._statesList[screen_name];
			/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - cleanVideoServices / state object for screen_name: ' + screen_name);
			if (stateObj)	{
				/* DEBUG */ ICTouchAPI.debugServices.warning('ICTouchAPI.VideoServices - cleanVideoServices / state object is defined for screen_name ' + screen_name + ': ' + dojo.toJson(stateObj) + ', delete it!');

				if (stateObj.params.callId !== undefined) {
					/* DEBUG */ ICTouchAPI.debugServices.warning('ICTouchAPI.VideoServices - cleanVideoServices / call deleteVideo for callId: ' + stateObj.params.callId + ' (this._getVideoDisplayed(): ' + this._getVideoDisplayed() + ')');
					destroyResult = ICTouchAPI.APIServices.Video.deleteVideo( stateObj.params.callId );
				}

				// LOCAL VIDEO
				else if (stateObj.isLocal) {
					/* DEBUG */ ICTouchAPI.debugServices.warning('ICTouchAPI.VideoServices - cleanVideoServices / state object is local, call deleteLocalVideo');
					destroyResult = ICTouchAPI.APIServices.Video.deleteLocalVideo();
				}
				/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - cleanVideoServices / destroyResult: ' + destroyResult);
			}
			this._deleteState(screen_name);
		}
		if (this._getVideoDisplayed() !== this._VIDEO_OFF) {
			this._setVideoDisplayed(this._VIDEO_OFF);
		}
		if (this._isVideoComActive()) {
			this._setVideoComActive(false);
		}
		this._setWebappShown(true);
		this._setPopupShown(false);
		this._setToasterShown(false);

		this._maskSetStatus = false;
		this._strReservedVideoUI = null;
		this._bVideoDisplayedOnce = false;
	},


	/**
	 * @private
	 * Return a boolean that define if video can be shown. In other words, verify if no content
	 * will be overlapped by the video area by checking the _getShowStatus() result.
	 *
	 * @return {Boolean} isShowable
	 */
	_isShowable: function() {

		var status = this._getShowStatus();
		/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _isShowable / status is: ' + status + ", return: " + (status == 100));

		return (status == 100) ? true : false;
	},

	/**
	 * @private
	 * Return a global status code for webapp, toaster & popup visibility.
	 * Video can be shown only if the returned value equals 100 (one hundred)
	 *
	 * This value is calculated with:
	 *  - _webappShowStatus (100 = webapp VISIBLE, 0 = webapp HIDDEN)
	 *  - _toasterShowStatus  (0 = toaster HIDDEN, 10 = toaster VISIBLE)
	 *  - _popupShowStatus (0 = popup HIDDEN, 1 = popup VISIBLE)
	 *
	 * For example, if webapp is showing a video and a toaster is showing up (keyboard),
	 * _toasterShowStatus will change from 0 to 10. So, global status will be 110 (100 + 10 + 0)
	 *
	 * @return {Integer} Can be equal from 0 to 111
	 */
	_getShowStatus: function() {
		return this._webappShowStatus + this._toasterShowStatus + this._popupShowStatus;
	},

	/**
	 * @private
	 * Set Webapp show status
	 * @param {Boolean} isShown set to TRUE if current webapp is visible.
	 */
	_setWebappShown: function( isShown ) {
		/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _setWebappShown / _webappShowStatus was: ' + this._webappShowStatus + ', it is now: ' + ((isShown) ? 100 : 0));
		this._webappShowStatus = (isShown) ? 100 : 0;
	},

	/**
	 * @private
	 * Set Toaster show status
	 * @param {Boolean} isShown set to TRUE if a toaster is visible.
	 */
	_setToasterShown: function( isShown ) {
		/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _setToasterShown / _toasterShowStatus was: ' + this._toasterShowStatus + ', it is now: ' + ((isShown) ? 10 : 0));
		this._toasterShowStatus = (isShown) ? 10 : 0;
	},

	/**
	 * @private
	 * Set Popup show status
	 * @param {Boolean} isShown set to TRUE if a popup is visible.
	 */
	_setPopupShown: function( isShown ) {
		/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _setPopupShown / _popupShowStatus was: ' + this._popupShowStatus + ', it is now: ' + ((isShown) ? 1 : 0));
		this._popupShowStatus = (isShown) ? 1 : 0;
	},



	/**
	 * @private
	 * Webapp hide event. Handle restore video visibility when webapp is shown again with the same context.
	 */
	_hideOnWebappHide: function() {

		/* DEBUG */ ICTouchAPI.debugServices.info( 'ICTouchAPI.VideoServices - _hideOnWebappHide');

		/* Hide video */
		this.hide( true, undefined );

		/* Set webapp visibility status to HIDDEN */
		this._setWebappShown( false );

		var current_screen_name = this._getCurrentScreen();

		/* Unregister previous listener */
		if( this._webappEventListenerHandle !== null ) {
			this._unregisterEventListener( this._webappEventListenerHandle );
			this._webappEventListenerHandle = null;
		}

		/* Register webapp show event - in order to reshow video previously hidden */
		this._webappEventListenerHandle = this._registerEventListener(
			this._EVENT_WEBAPP_SHOW,
			function(evt) {
				/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnWebappHide / webapp show event - _EVENT_WEBAPP_SHOW' );

				if( evt.id === current_screen_name )
				{
					/* Set webapp show status */
					this._setWebappShown( true );

					if( this._isShowable() === true )
					{
						/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnWebappHide / webapp show event - video ready to be shown' );

						/* Check if this webapp previously shown a video and restore it like it was */
						var state = this._getState(current_screen_name);

						if( state !== null )
						{
							/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnWebappHide / webapp show event - saved state found for '+ current_screen_name );

							// LOCAL VIDEO - need to create a new one
							if( state.isLocal === true ) {
								/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnWebappHide / webapp show event - showing LOCAL video...' );

								this.show( state.params, state.isLocal, true, evt.id );
							}

							// VIDEO CALL
							else {
								/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnWebappHide / webapp show event - showing video...' );

								var setResult = ICTouchAPI.APIServices.Video.showVideo( state.params );

								if( setResult == true ) {
									this._postShow( current_screen_name, state.params, false );
								}
							}

						}

						else {
							/* DEBUG */  ICTouchAPI.debugServices.warning( 'ICTouchAPI.VideoServices - _hideOnWebappHide / webapp show event - no saved state found for '+ current_screen_name );
						}

					}

					else {
						/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnWebappHide / webapp show event - screen name is different '+ evt.id + ' != ' + current_screen_name );
					}

				}

				else {
					/* DEBUG */  ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnWebappHide / webapp show event - default' );
				}

			}

		);

	},


	/**
	 * @private
	 * Toaster show event. Handle restore video visibility (or mask) when toaster is hidden.
	 * @param {String} evt toaster id
	 */
	_hideOnToasterShow: function(evt) {
		/* DEBUG */ ICTouchAPI.debugServices.info('ICTouchAPI.VideoServices - _hideOnToasterShow');
		/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - _hideOnToasterShow $ input function params | evt ' + evt);


		var maskId = 1, maskWidth = 800, maskHeight = 166;


		/* Create a mask for the audio toaster */
		if( evt === this._audioToasterId ) {
			/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnToasterShow / create a mask for the audio toaster ');
			this._maskSetStatus = ICTouchAPI.APIServices.Video.enableMask( maskId, 0, 315, (maskWidth & 0xFFF8), (maskHeight & 0xFFFE) ); // mask w calculated overlap: 0, 210, 490, 62
		}
		/* or simply hide the video */
		else {
			/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnToasterShow / simply hide the video');
			this.hide(true,null, ICTouchAPI.transitionServices.getNextOrCurrentScreenName());
			/* Set toaster visibility status to VISIBLE */
			this._setToasterShown( true );
		}
		// on an even other than webapp.hide, assume that the screen is already loaded
		var current_screen_name = ICTouchAPI.transitionServices.getNextOrCurrentScreenName();
		

		/* Unregister previous listener */
		if( this._toasterEventListenerHandle !== null ) {
			this._unregisterEventListener( this._toasterEventListenerHandle );
			this._toasterEventListenerHandle = null;
		}

		this._toasterEventListenerHandle = this._registerEventListener(
			this._EVENT_TOASTER_HIDE,
			function( evt2 ) {
				/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnToasterShow / toaster hide event - id '+ evt2 );

				/* Set toaster show status */
				this._setToasterShown( false );

				if( this._isShowable() === true && this._getCurrentScreen() === current_screen_name ) {
					if( evt2 === this._audioToasterId ) {
						/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnToasterShow / toaster hide event - audio toaster');
						ICTouchAPI.APIServices.Video.disableMask( maskId );
						this._maskSetStatus = false;

						/* Register toaster show listener */
						this._toasterEventListenerHandle = this._registerEventListener(
							this._EVENT_TOASTER_SHOW, this._hideOnToasterShow
						);
					}

					else
					{
						/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - _hideOnToasterShow / toaster hide event - dialer or popup');
						/* Check if this webapp previously shown a video and restore it like it was */
						var state = this._getState(current_screen_name);

						if( state !== null )
						{
							// LOCAL VIDEO - need to create a new one
							if( state.isLocal === true ) {
							    	/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnToasterShow / toaster hide event - LOCAL VIDEO');
								this.show( state.params, state.isLocal );
							}

							// VIDEO CALL
							else {
								/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnToasterShow / toaster hide event - VIDEO CALL');
								/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnToasterShow / toaster hide event - state.videoStableState: ' + state.videoStableState);
								if (this._getVideoDisplayed() == this._VIDEO_HIDDEN && (undefined !== state.videoStableState && state.videoStableState != this._VIDEO_HIDDEN)) {
								    var setResult = ICTouchAPI.APIServices.Video.showVideo( state.params );

								    if( setResult == true ) {
								    	this._postShow( current_screen_name, state.params, false, evt2 );
								    }
								} else {
								    /* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnToasterShow / toaster hide event - video call is hidden');
								}
							}
						}
					}
				} else {
				    /* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _hideOnToasterShow / toaster hide event - hum... it\'s seems there is a problem when the toaster disappears');
				}
			}
		);

	},


	/**
	 * @private
	 * Popup show event. Handle restore video visibility when popup is hidden.
	 */
	_hideOnPopupShow: function() {

		/* DEBUG */ ICTouchAPI.debugServices.info( 'ICTouchAPI.VideoServices - _hideOnPopupShow');

		/* Hide video */
		this.hide( true, undefined, ICTouchAPI.transitionServices.getNextOrCurrentScreenName() );

		//Webapp can be hidden when a popup is displayed
		this._webappEventListenerHandle = this._registerEventListener(
			this._EVENT_WEBAPP_HIDE, this._hideOnWebappHide
		);

		/* Set popup visibility status to VISIBLE */
		this._setPopupShown( true );

		var current_screen_name = this._getCurrentScreen();

		/* Unregister previous listener */
		if( this._popupEventListenerHandle !== null ) {
			this._unregisterEventListener( this._popupEventListenerHandle );
			this._popupEventListenerHandle = null;
		}

		this._popupEventListenerHandle = this._registerEventListener(
			this._EVENT_POPUP_HIDE,
			function() {
				/* DEBUG */ ICTouchAPI.debugServices.info( "ICTouchAPI.VideoServices - _hideOnPopupShow / popup hide event" );

				/* Set popup show status */
				this._setPopupShown( false );

				/* Current screen is the one that contains the video */
				if( this._isShowable() === true && this._getCurrentScreen() === current_screen_name )
				{
					var state = this._getState(current_screen_name);

					if( state !== null ) {
						this._setPopupShown( false );

						// LOCAL VIDEO - need to create a new one
						if( state.isLocal === true ) {
							this.show( state.params, state.isLocal );
						}

						// VIDEO CALL
						else {
							var setResult = ICTouchAPI.APIServices.Video.showVideo( state.params );

							if( setResult == true ) {
								this._postShow( current_screen_name, state.params, false );
							}

						}

					}
				}
			}
		);

	},


	/**
	 * @private
	 * Used to save a state object that contains video details (local, coordinates, sizes) specific to a webapp (id)
	 * @param {String} id webapp id
	 * @param {Object} stateObj contains the state details.
	 */
	_saveState: function( id, stateObj ) {
		/* DEBUG */ ICTouchAPI.debugServices.info('ICTouchAPI.VideoServices - _saveState');
		/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - _saveState $ input function params | id: ' + id + ', stateObj: ' + dojo.toJson(stateObj));
		this._statesList[ id ] = stateObj;
	},

	/**
	 * @private
	 * Return a state object saved at 'id' OR false if nothing exists
	 * @param {String} id webapp id
	 * @return {Object} stateObj or FALSE
	 */
	_getState: function( id ) {
		/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - _getState $ input function params | id: ' + id);

		var stateObj = this._statesList[ id ];
		//console.log( stateObj );
		if( undefined === stateObj ) {
			/* DEBUG */ ICTouchAPI.debugServices.warning('ICTouchAPI.VideoServices - _getState / state no found!');
			stateObj = null;
		}

		/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - _getState $ return stateObj: ' + dojo.toJson(stateObj));
		return stateObj;
	},

	/**
	 * @private
	 * Delete a state object saved at 'id' OR false if nothing exists
	 * @param {String} id webapp id
	 * @return {Boolean} True if no error occurs 
	 */
	_deleteState: function( id ) {
		/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _deleteState $ input function params | id: ' + id );

		var stateObj = this._statesList[ id ];
		//console.log( stateObj );
		if( undefined === stateObj ) {
			/* DEBUG */ ICTouchAPI.debugServices.warning( 'ICTouchAPI.VideoServices - _deleteState / state no found!' );
			return false;
		}
		delete this._statesList[ id ];
		stateObj = null;
		return true;
	},

	/**
	 * @private
	 * Delete all states with a given callId OR false if nothing exists
	 * @param {String} callId
	 */
	_deleteStateByCallId : function(callId) {
		ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _deleteStateByCallId $ input function params | callId: '+ callId );
		var nbDeleted = 0;

		for(var screen_name in this._statesList){
			if( undefined !== this._statesList[ screen_name ].params.callId
				&& this._statesList[ screen_name ].params.callId === callId ) {

				ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _deleteStateByCallId $ delete state with callId: '+ callId + " and with screenName: "+screen_name);
				this._deleteState(screen_name);
			}

			nbDeleted++;
		}
		ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _deleteStateByCallId $ number of deleted states: '+ nbDeleted );
	},

	_getStateByCallId: function( callId ) {
		/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _getStateByCallId $ input function params | callId: '+ callId );

		var stateObj = null;

		for( var screen_name in this._statesList ) {

			if( undefined !== this._statesList[ screen_name ].params.callId
				&& this._statesList[ screen_name ].params.callId === callId ) {
				stateObj = this._statesList[ screen_name ];
				/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _getStateByCallId / state found for callId ' + callId + ' with screen name '+ screen_name );
				break;
			}

		}

		/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - _getStateByCallId $ return stateObj: ' + dojo.toJson(stateObj));
		return stateObj;
	},


	/**
	 * @private
	 * Used to define if a video is currently displayed.
	 * @param {Boolean} videoState set to TRUE if a video is displayed or FALSE if no video is visible.
	 */
	_setVideoDisplayed: function( videoState ) {

		if( undefined !== videoState ) {
			/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _setVideoDisplayed / _videoDisplayed was: ' + this._videoDisplayed + ', it is now: ' + videoState);
			this._videoDisplayed = videoState;
		}
	},

	/**
	 * @private
	 * Return a state object saved at 'id' OR false if nothing exists
	 * @param
	 * @return {Integer} _VIDEO_OFF; _VIDEO_ON; _VIDEO_HIDDEN
	 */

	_getVideoDisplayed: function() {
	    /* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _getVideoDisplayed $ return _videoDisplayed (0: _VIDEO_OFF, 1: _VIDEO_ON, 2: _VIDEO_HIDDEN): ' + this._videoDisplayed);
	    return this._videoDisplayed;
	},


	/**
	 * @private
	 * Used to define if a video communication is currently active.
	 * @param {Boolean} isActive set to TRUE if a video communication is active else FALSE.
	 */
	_setVideoComActive : function( isActive ) {

		if( undefined !== isActive ) {
			/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _setVideoComActive / _videoComActive was: ' + this._videoComActive + ', it is now: ' + isActive);
			this._videoComActive = new Boolean( isActive ).valueOf();
		}

	},

	/**
	 * @private
	 * Returns TRUE if a video communication is currently active (displayed or not).
	 * @return {Boolean} _videoComActive
	 */
	_isVideoComActive: function() {
	    /* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _isVideoComActive $ return _videoComActive: ' + this._videoComActive);
		return this._videoComActive;
	},


	//
	// HELPERS
	//

	/**
	 * @private
	 * @return {String} current webapp screen name
	 */
	_getCurrentScreen: function() {
	    /* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _getCurrentScreen $ return ' + ICTouchAPI.transitionServices.getCurrentScreenName());
		return ICTouchAPI.transitionServices.getCurrentScreenName();
	},


	/**
	 * Unregister all listeners used to catch show/hide events (webapp, toaster & popup)
	 * @private
	 */
	_unregisterShowHideListeners: function() {

		/* DEBUG */ ICTouchAPI.debugServices.info( 'ICTouchAPI.VideoServices - _unregisterShowHideListeners');

		/* unregister all previous listeners for each events */
		for( var evtName in this._eventListenerHandleList ) {
			/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _unregisterShowHideListeners / unregister all listeners for: '+ evtName );

			for( var i = 0, ln = this._eventListenerHandleList[evtName].length; i < ln; i++ ) {
				this._unregisterEventListener( this._eventListenerHandleList[evtName][i] );
			}

			delete( this._eventListenerHandleList[evtName] );
		}

		/* reset handle */
		this._webappEventListenerHandle = null;
		this._toasterEventListenerHandle = null;
		this._popupEventListenerHandle = null;

		/* reset visibility variables*/
/*
		this._videoDisplayed = false;
*/
		this._webappShowStatus = 100;
		this._toasterShowStatus = 0;
		this._popupShowStatus = 0;
	},

	//
	// EVENT LISTENERS
	//

	/**
	 * Event listener registering helper. Automatically set the context to 'this'.
	 * @param {String} evtName name of event to listen
	 * @param {Function} c_func function reference called when the event will fire
	 * @return {Array} Dojo subscribe handle used to unsubscribe.
	 */
	_registerEventListener: function( evtName, c_func ) {
		/* DEBUG */ ICTouchAPI.debugServices.info('ICTouchAPI.VideoServices - _registerEventListener');
		/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - _registerEventListener / registering ' + evtName);

		/* Register listener */
		var handle = dojo.subscribe( evtName, this, c_func );

		/* create handle list for this event if it does not already exists */
		if( undefined === this._eventListenerHandleList[ evtName ] ) {
			this._eventListenerHandleList[ evtName ] = [];
		}

		/* push handle in the list */
		if(this._eventListenerHandleList[evtName][0]){
			this._unregisterEventListener(this._eventListenerHandleList[evtName][0]);
			delete(this._eventListenerHandleList[evtName][0]);
		}
		this._eventListenerHandleList[evtName][0] = handle;

		/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - _registerEventListener $ return handle: ' + handle);
		return handle;
	},

	/**
	 * Event listener unregistering helper.
	 * @param {Array} handle dojo subscribe handle.
	 */
	_unregisterEventListener: function( handle ) {
		/* DEBUG */ ICTouchAPI.debugServices.debug( 'ICTouchAPI.VideoServices - _unregisterEventListener / unregistering '+ handle.join(" ") );
		dojo.unsubscribe( handle );
	},



	//
	// OFFSET CALCULATION
	//

	/**
	 * @private
	 * Return the active frame offset
	 * @param {String} frameId iframe DOM id.
	 */
	_getFrameOffset: function( frameId ) {
		/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - _getFrameOffset $ input function params | frameId: ' + frameId);

		/* */
		var frameOffset = [ 0, 0 ],
			frameObj = null;
			frameObj = window.document.getElementById( frameId );

		if( null !== frameObj ) {
			frameOffset = this._getObjectOffset( frameObj );
		}

		/* DEBUG */ ICTouchAPI.debugServices.debug('ICTouchAPI.VideoServices - _getFrameOffset $ return frameOffset: ' + dojo.toJson(frameOffset));
		return frameOffset;
	},

	/**
	 * Calculate the offset for a DOM element.
	 * @param {Object} obj DOM object element
	 */
	_getObjectOffset: function( obj ) {
		var curtop, curleft;
		curleft = curtop = 0;

		/* get object relative offset */
	    if(obj.offsetParent) {

		    do {
		    	curleft += obj.offsetLeft;
		    	curtop += obj.offsetTop;

		    } while (obj = obj.offsetParent);

	    }

	    if( curleft < 0 ) {curleft = 0;}
	    if( curtop < 0 ) {curtop = 0;}

	    return [ curleft, curtop ];

	}


});

ICTouchAPI.VideoServices = new ICTouchAPI.VideoServices();
