/**
* @class ICTouchAPI.CapabilityServices
* @singleton
* @extends Object
* The Capabilities class provide an interface to webapp to easily listen to a capability and fetch its status.
* It also publish the event for an AppButton.
*/
dojo.provide("ICTouchAPI.CapabilityServices");
dojo.declare("ICTouchAPI.CapabilityServices", null , {

/* --------------------------------- Public attributes ------------------------------------ */

/* --------------------------------- Private attributes ----------------------------------- */

	// Defines
        /**
	 * @ignore
	 */
	AVAILABLE 				: 0,

        /**
	 * @ignore
	 */
	UNAVAILABLE 			: 1,

        /**
	 * @ignore
	 */
	TEMP_UNAVAILABLE		: 2,

        /**
	 * @ignore
	 */
	UNKNOWN					: -1,

        /**
	 * @ignore
	 */
	EVENT_ROOT				: "capabilities_changed_",

        /**
	 * @ignore
	 */
	arrGlobalCapabilities	: {}, // Current status of global capabilities ( for exemple [Telephony][makeCall] = AVAILABLE )

        /**
	 * @ignore
	 */
	arrInstanceCapabilities	: {}, // Current status of instance capabilities ( [Telephony][1001][hold] = UNAVAILABLE )

        /**
	 * @ignore
	 */
	arrEvents				: [ ], // List of events current listened ( strEvent => number of time listened )

        /**
	 * @ignore
	 */
	intSubscriptions		: 0, // Number of subscriptions currently given

	/**
	* @ignore
	*/
	intPublishNumber		: 0, // Number of capabilities' instances published
/* ------------------------------------ Constructor --------------------------------------- */

	/**
	 * @ignore
	 */
	constructor : function () {

	},

/* ----------------------------------- Getter / Setter------------------------------------- */


/* ----------------------------------- Public methods ------------------------------------- */

	/**
         * @ignore
	 * Because the eventServices wants an application name we provide one.
	 * @return {String} Return CapabilityServices
	 */
	getApplicationName: function(){
		return "CapabilityServices";
	},

	/**
	 * Add a new list of capacities to listen to.
	 * @param {String} strModule The name of the module capability.
	 * @param {String} Not used (backward compatibility only).
	 */
	subscribeToCapabilityList : function(strModule, strEvent){
		var that = this;
		ICTouchAPI.eventServices.subscribeToEvent(this, "capabilities_list_changed_" + strModule, function(){
			var args = ICTouchAPI.tools.getEventArguments(arguments);
			if(args.capabilities){
				for(var i in args.capabilities) {
					if(args.listId) {
						that._onInstanceChanged(strModule, args.listId, {value:args.capabilities[i]});
					} else {
						that._onCapabilityChanged(strModule, {value:args.capabilities[i]});
					}
				}
			}
		});

		//If capabilityList is removed, set to UNKNOWN associeted capabilities.
		ICTouchAPI.eventServices.subscribeToEvent(this, "removeCapabilityList_" + strModule, function(){
			var args = ICTouchAPI.tools.getEventArguments(arguments);
			if(args.listId ) {
				if (that.arrInstanceCapabilities[strModule] && that.arrInstanceCapabilities[strModule][args.listId]) {
					for( var i in that.arrInstanceCapabilities[strModule][args.listId] ) {
						if (that.arrInstanceCapabilities[strModule][args.listId][i] != that.UNKNOWN) {
							that.arrInstanceCapabilities[strModule][args.listId][i] = that.UNKNOWN;
							that._publishCapability(strModule, i, that.UNKNOWN, args.listId);
						}
					}
				}
			}
		});
	},

	/**
	 * Add a new instance's capacity to listen to, must be called previous to getCapability.
	 * @param {String} strModule The name of the module capability.
	 * @param {String} instanceId The name of the capability to listen to.
	 * @param {Number} strCapability The instance id to listen to.
	 * @param {Object} context The context of the callback.
	 * @param {Function} callback The function to be called when the capability changes.
	 * @return {Object} handler to unsubscribe
	 */
	subscribeToInstance : function(strModule, instanceId, strCapability, context, callback){
		var publishName = this.EVENT_ROOT + strModule + strCapability + instanceId;
		ICTouchAPI.debugServices.debug("ICTouchAPI.CapabilityServices - subscribeToInstance  / Subscribing to instance: " + publishName, " CAPABILITIES");

		var subscribeHandler = dojo.subscribe(publishName, context, callback);
		var eventName = this.EVENT_ROOT + strModule + "_" + instanceId; // Event name on dbus

		this._listenEvent(eventName, dojo.hitch(this, this._onInstanceChanged, strModule, instanceId));

		if(this.arrInstanceCapabilities[strModule] == undefined){
			this.arrInstanceCapabilities[strModule] = {} ;
		}
		if( this.arrInstanceCapabilities[strModule][instanceId] == undefined ){
			this.arrInstanceCapabilities[strModule][instanceId] = {};
		}

		if( this.arrInstanceCapabilities[strModule][instanceId][strCapability] == undefined )
		{
			this.arrInstanceCapabilities[strModule][instanceId][strCapability] = this.UNKNOWN;
			this._fetchCapability(strModule, strCapability, instanceId);
		}
		else
		{
			this._publishCapability(strModule, strCapability, this.arrInstanceCapabilities[strModule][instanceId][strCapability], instanceId);
		}

		this.intSubscriptions++;
		// Create handler for our disconnect function
		return [subscribeHandler, eventName, strModule, instanceId, strCapability];
	},

	/**
	 * Add a new capacity to listen to, must be called previous to getCapability.
	 * @param {String} strModule The name of the module capability.
	 * @param {String} strCapability The name of the capability to listen to.
	 * @param {Object} context The context of the callback.
	 * @param {Function} callback The function to be called when the capability changes.
	 * @return {Object} handler to unsubscribe
	 */
	subscribeToCapability : function(strModule, strCapability, context, callback){
		var publishName = this.EVENT_ROOT + strModule + strCapability;
		ICTouchAPI.debugServices.debug("ICTouchAPI.CapabilityServices - subscribeToCapability / Subscribing to capability: " + publishName, " CAPABILITIES");
		var subscribeHandler = dojo.subscribe(publishName, context, callback);
		var eventName = this.EVENT_ROOT + strModule; // Event name on dbus
		this._listenEvent(eventName, dojo.hitch(this, this._onCapabilityChanged, strModule));
		if(this.arrGlobalCapabilities[strModule] == undefined){
			this.arrGlobalCapabilities[strModule] = {};
		}
		if( this.arrGlobalCapabilities[strModule][strCapability] === undefined ){
			this.arrGlobalCapabilities[strModule][strCapability] = this.UNKNOWN;
			this._fetchCapability(strModule, strCapability);
		}
		else {
			this._publishCapability(strModule, strCapability, this.arrGlobalCapabilities[strModule][strCapability]);
		}
		this.intSubscriptions++;
		var returnedParams = [subscribeHandler, eventName, strModule, strCapability];
		strModule=null;
		strCapability=null;
		context=null;
		callback=null;
		return returnedParams;
	},

	/**
	 * Disconnect the dojo event connection, however keep the eventService subscription alive (the capability will still be handled in CapabilityService)
         * @param {Object} objHandler The handler of subscribeToInstance or subscribeToCapability
	 */
	disconnectHandler : function(objHandler) {
		// Sanity check on the handler passed
		if(objHandler){
			if( typeof objHandler.length != "number" || objHandler.length < 4 || objHandler.length > 5 ) {
				ICTouchAPI.debugServices.warning("ICTouchAPI.CapabilityServices - disconnectHandler / function treatment interrupted:  disconnectHandler not a handler, objHandler.length:" + objHandler.length);
				return;
			}
			this.intSubscriptions--;
			// Disconnect publish handler
			var subscribeHandler = objHandler.shift();
			dojo.unsubscribe(subscribeHandler);
			// Disconect from ICTGate if necessary
			var eventName = objHandler.shift();
			var cleanup = this._stopListening(eventName);
			if( cleanup ) {
				if( objHandler.length == 2 ) {
					this._cleanupCapability(objHandler[0], objHandler[1]);
				}
				else {
					this._cleanupInstance(objHandler[0], objHandler[1], objHandler[2]);
				}
			}
		}
		else {
			ICTouchAPI.debugServices.warning("ICTouchAPI.CapabilityServices - disconnectHandler / function treatment interrupted: objHandler is not defined");
		}

	},

	/**
	 * Retrieve the named capability of an instance
	 * @param {String} strModule The name of the module capability.
	 * @param {Number} instanceId The instance id of the capability.
	 * @param {String} strCapability The name of the capability to listen to.
	 * @return {Array} An array containing every capability of an instance
	 */
	getCapability: function(strModule, instanceId, strCapability){
		var isInstance = arguments.length > 2;
		strCapability = isInstance ? arguments[2] : arguments[1];
		var value = null;
		if ( !isInstance && this.arrGlobalCapabilities[strModule] !== undefined )
			value = this.arrGlobalCapabilities[strModule][strCapability];
		if( isInstance && this.arrInstanceCapabilities[strModule] !== undefined && this.arrInstanceCapabilities[strModule][instanceId] !== undefined )
			value = this.arrInstanceCapabilities[strModule][instanceId][strCapability];
		if (value == null || value == undefined)
			value = this.UNKNOWN;
		return value;
	},

	/**
	 * Retrieve the capabilities of a module and maybe for an instance (optionnal)
	 * @param {String} strModule The name of the module capability.
	 * @param {Number} instanceId The instance id of the capability.
	 * @return {Array} An array containing every capability of an instance or a module
	 */
	getCapabilities: function(strModule, instanceId){
		var isInstance = arguments.length > 1;
		var value = null;
		if ( !isInstance && this.arrGlobalCapabilities[strModule] !== undefined ){
			value = this.arrGlobalCapabilities[strModule];
		}
		if( isInstance && this.arrInstanceCapabilities[strModule] !== undefined && this.arrInstanceCapabilities[strModule][instanceId] !== undefined ){
			value = this.arrInstanceCapabilities[strModule][instanceId];
		}
		if (value == null || value == undefined){
			value = [];
		}
		return value;
	},

	/**
	 * Retrieve the whole capabilities of an instance
	 * @param {String} strModule The name of the module capability.
	 * @param {Number} instanceId The instance id of the capability.
	 * @return {Array} An array containing every capability of an instance
	 */
	getInstance: function(strModule, instanceId){
		if( this.arrInstanceCapabilities[strModule] !== undefined && this.arrInstanceCapabilities[strModule][instanceId] !== undefined )
			// Maybe clone this object to avoid any modification ?
			return this.arrInstanceCapabilities[strModule][instanceId];
		return null;
	},

	/**
	 * Force the publishing of an event about a capability
	 * @ignore
	 * @param {String} strModule The name of the module capability
	 * @param {String} strCapability The name of the capability
	 * @param {Number} status The capability value
	 * @param {String} instanceId
	 */
	_publishCapability: function(strModule, strCapability, status, instanceId){
		var publishName;
		if( instanceId === undefined ) {
			publishName = this.EVENT_ROOT + strModule + strCapability;
			ICTouchAPI.log("Publishing capability: " + publishName + " (value: " + status + ")", "CAPABILITIES");
			dojo.publish(publishName, [strCapability, status]);
		}
		else {
			publishName = this.EVENT_ROOT + strModule + strCapability + instanceId;
			ICTouchAPI.log("Publishing capability instance: " + publishName + " (value: " + status + ")", "CAPABILITIES");
			dojo.publish(publishName, [strCapability, status, instanceId, ++this.intPublishNumber]);
			// intPublishNumber is here to avoid bad order in capability management in buttons.
		}
	},

	/**
	 * Listen to a dbus event and increment it's counter
	 * @ignore
	 * @param {String} strEvent Event name
	 * @param {Function} callback Callback of the event
	 */
	_listenEvent: function(strEvent, callback){
		if( typeof this.arrEvents[strEvent] == "undefined" ) {
			// Start listening
			this.arrEvents[strEvent] = 0;
			ICTouchAPI.eventServices.subscribeToEvent(this, strEvent, callback);
		}
		this.arrEvents[strEvent]++;
	},

	/**
	 * Stop listening to an event, if the counter falls to 0 unsubsribe from the eventServices
	 * @ignore
	 * @param {String} strEvent Event name
	 * @return {Boolean} return true if it was the last one
	 */
	_stopListening: function(strEvent){
		if( typeof this.arrEvents[strEvent] == "undefined" ) {
			ICTouchAPI.debugServices.warning("ICTouchAPI.CapabilityServices - _stopListening $ return false: _stopListening on something we are not listening ? (this.arrEvents[strEvent] is undefined)");
			return false;
		}
		if( this.arrEvents[strEvent] > 1 ) {
			// Still something listening for this event
			this.arrEvents[strEvent]--;
			return false;
		}
		else {
			// Nothing is listening to this event anymore
			ICTouchAPI.eventServices.unsubscribeToEvent(this, strEvent);

			delete this.arrEvents[strEvent];
			return true;
		}
	},

	/**
	 * Remove the capability from the array and remove the module if it's empty
	 * @ignore
	 * @param {String} strModule Name of module
	 * @param {String} strCapability Name of capability
	 */
	_cleanupCapability: function(strModule, strCapability){
		var i, empty;
		delete this.arrGlobalCapabilities[strModule][strCapability];

		// Test if this associative array is empty ( array.length doesn't work here )
		empty = true;
		for( i in this.arrGlobalCapabilities[strModule] ) {
			empty = false;
		}
		if( empty ) {
			delete this.arrGlobalCapabilities[strModule];
		}
	},

	/**
	 * Remove the capability from the array and remove the instance and the module if they are empty
	 * @ignore
	 * @param {String} strModule Name of module
	 * @param {Number} instanceId Instance id of the capability
	 * @param {String} strCapability Name of capability
	 */
	_cleanupInstance: function(strModule, instanceId, strCapability){
		var i, empty;
		delete this.arrInstanceCapabilities[strModule][instanceId][strCapability];
		// Test if this associative array is empty ( array.length doesn't work here )
		empty = true;
		for( i in this.arrInstanceCapabilities[strModule][instanceId] ) {
			empty = false;
		}
		if( empty ) {
			delete this.arrInstanceCapabilities[strModule][instanceId];
			// Test again on module
			empty = true;
			for( i in this.arrInstanceCapabilities[strModule] ) {
				empty = false;
			}
			if( empty ) {
				delete this.arrInstanceCapabilities[strModule];
			}
		}
	},

/* --------------------------------- Private Methods -------------------------------------- */

	/**
	 * @ignore
	 * Send logs of attributes of the service to the logger.
	 * @param {Boolean} boolAdvancedLogs Boolean indicating if advanced or basic logs has to be written
	 */
	dump : function(boolAdvancedLogs){
		ICTouchAPI.debugServices.dump("dump function of CapabilityServices with " + ((boolAdvancedLogs)?"advanced":"basic") + " logs option");

		ICTouchAPI.debugServices.dump("Capability states available: 0 = AVAILABLE / 1 = UNAVAILABLE / TEMP_UNAVAILABLE = 2 / UNKNOWN = -1");
		ICTouchAPI.debugServices.dump("Number of subscriptions currently given (intSubscriptions): " + this.intSubscriptions);
		ICTouchAPI.debugServices.dump("Number of capabilities' instances published (intPublishNumber): " + this.intPublishNumber);

		if (boolAdvancedLogs) {
			// List of events current listened ( strEvent => number of time listened )
			var strEvent;
			for (strEvent in this.arrEvents) {
				ICTouchAPI.debugServices.dump(this.arrEvents[strEvent] + " capabilities are listened for event " + strEvent);
			}

			var strModule, instanceId, strCapability;
			ICTouchAPI.debugServices.dump("List of global capabilities: ");
			for (strModule in this.arrGlobalCapabilities) {
				ICTouchAPI.debugServices.dump("Global capabilities for strModule " + strModule + ":");
				for (strCapability in this.arrGlobalCapabilities[strModule]) {
					ICTouchAPI.debugServices.dump("Global capability state for strModule " + strModule + ", strCapability " + strCapability + ": " + this.arrGlobalCapabilities[strModule][strCapability]);
				}
			}
			ICTouchAPI.debugServices.dump("List of instance capabilities: ");
			for (strModule in this.arrInstanceCapabilities) {
				ICTouchAPI.debugServices.dump("Instance capabilities for strModule " + strModule + ":");
				for (instanceId in this.arrInstanceCapabilities[strModule]) {
					ICTouchAPI.debugServices.dump("Instance capabilities for strModule " + strModule + ", instanceId " + instanceId);
					for (strCapability in this.arrInstanceCapabilities[strModule][instanceId]) {
						ICTouchAPI.debugServices.dump("Capability state for strModule " + strModule + ", instanceId " + instanceId + ", strCapability " + strCapability + ": " + this.arrInstanceCapabilities[strModule][instanceId][strCapability]);
					}
				}
			}
		}
	},
	
	/**
	 * @ignore
	 */
	disconnectInstance : function(objHandler) {
		ICTouchAPI.debugServices.warning("ICTouchAPI.CapabilityServices - disconnectInstance / Use disconnectHandler instead of disconnectInstance");
		this.disconnectHandler(objHandler);
	},

	/**
	 * @ignore
	 */
	disconnectCapability : function(objHandler) {
		ICTouchAPI.debugServices.warning("ICTouchAPI.CapabilityServices - disconnectCapability / Use disconnectHandler instead of disconnectCapability");
		this.disconnectHandler(objHandler);
	},

	/**
	 * @ignore
	 */
	unsubscribeInstance : function() {
		ICTouchAPI.debugServices.warning("ICTouchAPI.CapabilityServices - unsubscribeInstance / Do not use unsubscribeInstance");
	},
        
	/**
	 * @ignore
	 */
	unsubscribeCapability : function() {
		ICTouchAPI.debugServices.warning("ICTouchAPI.CapabilityServices - unsubscribeCapability / Do not use unsubscribeCapability");
	},

	/**
	 * @ignore
	 */
	getOnceCapabilities: function(strModule, objEvent, idCapabilities, context, callBack){
		ICTouchAPI.APIServices[strModule].getCapabilities({params:[idCapabilities], context: context, callback: callBack, callbackParams: objEvent});
	},

	/**
	 * @ignore
	 */
	_onCapabilityChanged: function(strModule, objEvent){
		ICTouchAPI.log("Global capability changed", "CAPABILITIES");
		ICTouchAPI.logObject(objEvent, "CAPABILITIES");
		this._fetchedCapability(objEvent.value, {module : strModule});
	},

	/**
	 * @ignore
	 */
	_onInstanceChanged: function(strModule, instanceId, objEvent){
		ICTouchAPI.log("Instance capability changed", "CAPABILITIES");
		ICTouchAPI.logObject(objEvent, "CAPABILITIES");
		this._fetchedCapability(objEvent.value, {module : strModule, instanceId : instanceId});
	},

	/**
	 * @ignore
	 */
	_fetchCapability: function(strModule, strCapability, instanceId){
		if ( instanceId === undefined )
			ICTouchAPI.APIServices[strModule].getCapabilityGlobal({params: [strCapability], context: this, callback: this._fetchedCapability, callbackParams: {module : strModule}});
		else
			ICTouchAPI.APIServices[strModule].getCapabilityInstance({params: [instanceId, strCapability], context: this, callback: this._fetchedCapability, callbackParams: {module : strModule, instanceId : instanceId}});
	},

	/**
	 * @ignore
	 */
	_fetchedCapability: function(capability, params){
		if( params.instanceId === undefined ){
			this.arrGlobalCapabilities[params.module][capability.name] = capability.state;
			this._publishCapability(params.module, capability.name, capability.state);
		} else {
			if(this.arrInstanceCapabilities[params.module] == undefined){
				this.arrInstanceCapabilities[params.module] = {};
			}
			if(this.arrInstanceCapabilities[params.module][params.instanceId] == undefined){
				this.arrInstanceCapabilities[params.module][params.instanceId] = {};
			}
			this.arrInstanceCapabilities[params.module][params.instanceId][capability.name] = capability.state;
			this._publishCapability(params.module, capability.name, capability.state, params.instanceId);
		}
	}

});

ICTouchAPI.CapabilityServices = new ICTouchAPI.CapabilityServices();
