/**
 * @class ICTouchAPI.AppBarServices
 * @singleton
 * @extends Object
 * The AppBar save the action buttons which have to be displayed in the action bar for each main widget.<br/>
 * This list of action buttons is built from the static button list, the current widget button list, and the button list of the other webapps which require a link for the current widget. </br>
 * The webapps can add and remove some buttons and require an updating of the action bar.
 */
dojo.provide("ICTouchAPI.AppBarServices");
dojo.declare("ICTouchAPI.AppBarServices",null,{

	// attributs

	/**
         * @ignore
         */
	lstStaticButtons : {},			// list of objects which contains the action Static buttons for each widget

	/**
         * @ignore
         */
	lstExitButtons : {},			// list of objects which contains the exit buttons for each widget

	/**
         * @ignore
         */
	lstOtherWebAppButtons : {},     // list of objects which contains the action buttons coming from other webapp for each widget

	/**
         * @ignore
         */
	lstAppBarWidgets : {},       // list of objects wich contains the action bar for each widget

	/**
         * @ignore
         */
	lstGlobalButtons : [],			// array of buttons which are presents in all action bars

	//  methods of AppBarServices class

	lstAppBar : {},

	/**
         * @ignore
         */
	constructor : function(){
	},

	//  private methods

	createAppBar : function(strWebAppName, strWidgetName, _arrSkeleton, boolDisableScroll){
		if (!this.lstAppBar[strWebAppName]){
			this.lstAppBar[strWebAppName] = {};
		}
		if (!this.lstAppBar[strWebAppName][strWidgetName]){
			var appBarWidget = new UIElements.AppBar.AppBarWidget({
				webAppName:strWebAppName,
				widgetName:strWidgetName,
				boolDisableScroll:boolDisableScroll,
				arrSkeleton:_arrSkeleton
			});
			this.lstAppBar[strWebAppName][strWidgetName] = appBarWidget;
		}
		var appBar = this.lstAppBar[strWebAppName][strWidgetName];
		strWebAppName = null;
		strWidgetName = null;
		_arrSkeleton = null;
		return appBar;
	},

	/*
	 * This method doesn't create the appbar if it doesn't exist
	 * @param {String} strWebAppName
	 * @param {String} strWidgetName
	 * @return the appBar or null
	 */
	_getAppBar : function(strWebAppName, strWidgetName) {
		if( typeof this.lstAppBarWidgets[strWebAppName] == "undefined" ) {
			return null;
		}
		if( typeof this.lstAppBarWidgets[strWebAppName][strWidgetName] != "object" ) {
			return null;
		}

		return this.lstAppBarWidgets[strWebAppName][strWidgetName];
	},

	
	_getAppBarWidget : function(strWebAppName, strWidgetName) {
		var appBar = null;
		if( typeof this.lstAppBar[strWebAppName] !== "undefined" && typeof this.lstAppBar[strWebAppName][strWidgetName] == "object") {
			appBar = this.lstAppBar[strWebAppName][strWidgetName];
		}
		strWebAppName=null;
		strWidgetName=null;
		return appBar;
	},

	//  public methods


	/**
         * @ignore
         */
	updateAppBar : function () {
	},

	/**
	 * Add a button to the a webapp view
	 * @param {String} strWebAppName The name of the WebApp
	 * @param {String} strWidgetName The name of the View
	 * @param {UIElements.AppButton.AppButtonControl} objActionButton The AppBarButton to add
	 */
	addStaticButton : function(strWebAppName, strWidgetName, objActionButton){
		var buttonList;

		if (!this.lstStaticButtons[strWebAppName]) {
			this.lstStaticButtons[strWebAppName] = {}; // new object
		}
		if (!this.lstStaticButtons[strWebAppName][strWidgetName]) {
			this.lstStaticButtons[strWebAppName][strWidgetName] = []; // new array
		}
		buttonList = this.lstStaticButtons[strWebAppName][strWidgetName];

		var name = objActionButton.getButtonName();
		var appBar = this._getAppBar(strWebAppName, strWidgetName);
		if( appBar != null ) {
			appBar.addStaticButton(objActionButton);
		}else{
			appBar = this._getAppBarWidget(strWebAppName, strWidgetName);
			if( appBar != null ) {
				appBar.addStaticButton(objActionButton);
			}
		}

		if( typeof buttonList[name] != "undefined" ) {
			buttonList[name].destroy();
		}
		buttonList[name] = objActionButton; // index of array is the button name
	},

	/**
	 * Remove a button from a webapp view
	 * @param {String} strWebAppName The name of the WebApp
	 * @param {String} strWidgetName The name of the View
	 * @param {UIElements.AppButton.AppButtonControl} objActionButton The AppBarButton to add
	 */
	removeStaticButton : function(strWebAppName,strWidgetName,objActionButton){
		// get the name of button
		var name = objActionButton.getButtonName();
		// get the right button which is placed to strWidgetName of strWebAppName
		// because the button can be a clone of objActionButton
		var button = this.lstStaticButtons[strWebAppName][strWidgetName][name];
		// if button is a clone or original
		if (typeof button=="object" && button!=null && !button.original){
			//To destroy the cloned button
			objActionButton.destroyClone(strWebAppName,strWidgetName);
		}
		//remove button on appBar
		var appBar = this._getAppBar(strWebAppName, strWidgetName);
		if( appBar != null ) {
			appBar.removeStaticButton(objActionButton);
		}else{
			appBar = this._getAppBarWidget(strWebAppName, strWidgetName);
			if( appBar != null ) {
				appBar.removeStaticButton(objActionButton);
			}
		}
		// delete the reference of button, not the button, if no has the reference
		// of button, the garbage collector will destroy the button
		delete (this.lstStaticButtons[strWebAppName][strWidgetName][name]);
	},

	/**
         * Add a button to all views of a webapp or a set of webapps
	 * @param {Array} lstWebAppName An array containing the name of the webapps
	 * @param {UIElements.AppButton.AppButtonControl} objActionButton The button to add
	 */
	addStaticButtonMultipleAppBars : function(lstWebAppName, objActionButton){
		var strWebAppName,j;
		for (strWebAppName in lstWebAppName){
			var numberOfWidget = lstWebAppName[strWebAppName].length;
			for(j;j<numberOfWidget;j++){
				var strWidgetName = lstWebAppName[strWebAppName][j];
				var cloneButton = objActionButton.clone(strWebAppName,strWidgetName);
				this.addStaticButton(strWebAppName,strWidgetName,cloneButton);
			}
		}
	},

	/**
         * Add an extra button to all views of a webapp or a set of webapps
	 * @param {Array} lstWebAppName An array containing the name of the webapps
	 * @param {UIElements.AppButton.AppButtonControl} objActionButton The button to add
	 */
	addOtherWebAppButtonMultipleAppBars : function(lstWebAppName, objActionButton){
		var strWebAppName,j;
		for (strWebAppName in lstWebAppName){
			var numberOfWidget = lstWebAppName[strWebAppName].length;
			for(j;j<numberOfWidget;j++){
				var strWidgetName = lstWebAppName[strWebAppName][j];
				var cloneButton = objActionButton.clone(strWebAppName,strWidgetName);
				this.addOtherWebAppButton(strWebAppName,strWidgetName,cloneButton);
			}
		}
		lstWebAppName=null;
		objActionButton=null;
	},

	/**
	 * Add an extra button to a view of a WebApp
	 * @param {String} strWebAppName The name of the WebApp
	 * @param {String} strWidgetName The name of the View
	 * @param {UIElements.AppButton.AppButtonControl} objActionButton The AppBarButton to add
	 */
	addOtherWebAppButton : function(strWebAppName, strWidgetName, objActionButton){
		if (!this.lstOtherWebAppButtons[strWebAppName]) {
			this.lstOtherWebAppButtons[strWebAppName] = []; // new object
		}
		if (!this.lstOtherWebAppButtons[strWebAppName][strWidgetName]) {
			this.lstOtherWebAppButtons[strWebAppName][strWidgetName] = [];
		}
		var name = objActionButton.getButtonName();

		var appBar = this._getAppBar(strWebAppName, strWidgetName);
		if(appBar!=null){
			if (this.lstOtherWebAppButtons[strWebAppName][strWidgetName][name]) {
				appBar.removeOtherWebAppButton(appBar.getButton(name),false);
				this.lstOtherWebAppButtons[strWebAppName][strWidgetName][name].destroy();
				delete this.lstOtherWebAppButtons[strWebAppName][strWidgetName][name];
			}
			appBar.addOtherWebAppButton(objActionButton,false,false);
		}else{
			appBar = this._getAppBarWidget(strWebAppName, strWidgetName);
			if(appBar!=null){
				if (this.lstOtherWebAppButtons[strWebAppName][strWidgetName][name]) {
					appBar.removeOtherWebAppButton(appBar.getButton(name),true);
					this.lstOtherWebAppButtons[strWebAppName][strWidgetName][name].destroy();
					delete this.lstOtherWebAppButtons[strWebAppName][strWidgetName][name];
				}
				appBar.addOtherWebAppButton(objActionButton,false,true);
			}
		}
		this.lstOtherWebAppButtons[strWebAppName][strWidgetName][name] = objActionButton;
	},

	/**
	 * Remove extra button from a webapp view
	 * @param {String} strWebAppName The name of the WebApp
	 * @param {String} strWidgetName The name of the View
	 * @param {UIElements.AppButton.AppButtonControl} objActionButton The AppBarButton to add
	 */
	removeOtherWebAppButton : function(strWebAppName,strWidgetName,objActionButton){
		var name = objActionButton.getButtonName();
		var button = this.lstOtherWebAppButtons[strWebAppName][strWidgetName][name];
		// button is a clone or original ?
		if (!button.original){
			//To destroy the cloned button
			objActionButton.destroyClone(strWebAppName,strWidgetName);
		}


		var appBar = this._getAppBar(strWebAppName, strWidgetName);
		if( appBar != null ) {

			appBar.removeOtherWebAppButton(objActionButton,false);

		}else{
			appBar = this._getAppBarWidget(strWebAppName, strWidgetName);
			if( appBar != null ) {
				appBar.removeOtherWebAppButton(objActionButton,false);
			}
		}

		if (this.lstOtherWebAppButtons[strWebAppName][strWidgetName][name]) {
			this.lstOtherWebAppButtons[strWebAppName][strWidgetName][name].destroy();
			delete this.lstOtherWebAppButtons[strWebAppName][strWidgetName][name];
		}
	},

	/**
	 * Get the list of the buttons contained into a webapp view bar
	 * @param {String} strWebAppName The name of the WebApp
	 * @param {String} strWidgetName the name of the View
         * @return {Array} An array contained the list of the buttons or an empty array
	 */
	getStaticButtons : function(strWebAppName,strWidgetName){
		if(this.lstStaticButtons[strWebAppName]&&this.lstStaticButtons[strWebAppName][strWidgetName]){
			return this.lstStaticButtons[strWebAppName][strWidgetName];
		}
		else {
			return [];
		}
	},

	/**
	 * Get the list of the extra buttons contained into a webapp view bar
	 * @param {String} strWebAppName The name of the WebApp
	 * @param {String} strWidgetName the name of the View
         * @return {Array} An array contained the list of the buttons or an empty array
	 */
	getOtherWebAppButtons : function(strWebAppName,strWidgetName){
		if(this.lstOtherWebAppButtons[strWebAppName]&&this.lstOtherWebAppButtons[strWebAppName][strWidgetName]){
			return this.lstOtherWebAppButtons[strWebAppName][strWidgetName];
		}
		else{
			return []  ;
		}
	},

	/**
	 * Get the AppBar Widget for a webapp view
	 * @param {String} strWebAppName
	 * @param {String} strWidgetName
	 * @return {Object} The AppBar Widget (UI)
	 */
	getAppBar : function(strWebAppName,strWidgetName){
		var appBar;
		if(this.lstAppBar[strWebAppName] && this.lstAppBar[strWebAppName][strWidgetName]){
			appBar = this.lstAppBar[strWebAppName][strWidgetName];
		} else {
			if (!this.lstAppBarWidgets[strWebAppName]){
				this.lstAppBarWidgets[strWebAppName] = [];
			}
			if (!this.lstAppBarWidgets[strWebAppName][strWidgetName]){
				appBar = new UIElements.AppBar.AppBarWidget({
					webAppName:strWebAppName,
					widgetName:strWidgetName
				});
				this.lstAppBarWidgets[strWebAppName][strWidgetName] = appBar;
			} else {
				appBar = this.lstAppBarWidgets[strWebAppName][strWidgetName];
			}
		}
		//Add all global buttons if they are not already in
		for(var i in this.lstGlobalButtons){
			if (! appBar.getButton(this.lstGlobalButtons[i].strButtonName)){
				var uiBtn = new UIElements.AppButton.AppButtonControl(this.lstGlobalButtons[i]);
				this.addOtherWebAppButton(strWebAppName,strWidgetName,uiBtn,true);
			}
		}

		return appBar;
	},

	/**
	 * Remove the AppBar contained into a webapp view
	 * @param {String} strWebAppName The name of the WebApp
	 * @param {String} strWidgetName The name of the View
	 */
	destroyAppBar : function(strWebAppName,strWidgetName) {
		if (typeof this.lstAppBarWidgets[strWebAppName] == "undefined") {
			return;
		}

		if (this.lstAppBarWidgets[strWebAppName][strWidgetName]){
			if( typeof this.lstOtherWebAppButtons[strWebAppName] != "undefined" && typeof this.lstOtherWebAppButtons[strWebAppName][strWidgetName] != "undefined" ) {
				// Destroy Other webapp Button
				var list = this.lstOtherWebAppButtons[strWebAppName][strWidgetName];
				for(var i in list) {
					list[i].destroy();
					// list is a reference to the orginal array so it will be also deleted there
					delete list[i];
				}
			}

			if( typeof this.lstStaticButtons[strWebAppName] != "undefined" && typeof this.lstStaticButtons[strWebAppName][strWidgetName] != "undefined" ) {
				// Destroy static button
				list = this.lstStaticButtons[strWebAppName][strWidgetName];
				for(var i in list) {
					list[i].destroy();
					delete list[i];
				}
			}

			// Destroy AppBar
			this.lstAppBarWidgets[strWebAppName][strWidgetName].destroy();
			delete this.lstAppBarWidgets[strWebAppName][strWidgetName];
		}
	},

	/**
	 * @ignore
	 */
	reset : function(){
		this.lstAppBarWidgets = {};
	},

	/**
	 * Create an AppBar for a webapp view from another AppBar
	 * @param {String} strWebAppName The name of the WebApp
	 * @param {String} strWidgetName1 The name of the View to duplicate
	 * @param {String} strWidgetName2 The name of the view to create the AppBar
	 */
	setAppBar : function(strWebAppName,strWidgetName1,strWidgetName2){
		var appBar = this._getAppBar(strWebAppName,strWidgetName1);
		if(appBar != null){
			this.lstAppBarWidgets[strWebAppName][strWidgetName2] = appBar;

			// another service "transitionServices" has some references of appBar
			// in order to be more efficiency
			// it is necessary to set
			ICTouchAPI.transitionServices._setActionBar("webapp."+strWebAppName+"."+strWidgetName2,this.lstAppBarWidgets[strWebAppName][strWidgetName2]);
		}
		else{
			appBar = this._getAppBarWidget(strWebAppName,strWidgetName1);
			if(appBar!=null){
				this.lstAppBar[strWebAppName][strWidgetName2] = appBar;
				ICTouchAPI.transitionServices._setActionBar("webapp."+strWebAppName+"."+strWidgetName2,appBar);
			}
		}
		strWebAppName=null;
		strWidgetName1=null;
		strWidgetName2=null;
	},

	_setAppBar : function(strWebAppName,strWidgetName2,appBar){
		if(appBar!=null){
			ICTouchAPI.transitionServices._setActionBar("webapp."+strWebAppName+"."+strWidgetName2,appBar);
		}
	},

	/**
         * Add a button to all WebApps views
         * @param {UIElements.AppButton.AppButtonControl} btn The AppBarButton to add
         */
	addGlobalButton : function(btn){
		for (var k in this.lstGlobalButtons){
			if (this.lstGlobalButtons[k].strButtonName == btn.strButtonName){
				return;
			}
		}
		this.lstGlobalButtons.push(btn);
		var uiBtn,bar;
		for (var i in this.lstAppBarWidgets){
			if(this.lstAppBarWidgets[i]){
				for(var j in this.lstAppBarWidgets[i]){
					uiBtn = new UIElements.AppButton.AppButtonControl(btn);
					bar = this._getAppBar(i,j);
					if(uiBtn && bar){
						this.addOtherWebAppButton(bar.params.webAppName,bar.params.widgetName,uiBtn);
					}
				}
			}
		}
		for (var l in this.lstAppBar){
			if(this.lstAppBar[l]){
				for(var m in this.lstAppBar[l]){
					uiBtn = new UIElements.AppButton.AppButtonControl(btn);
					bar = this._getAppBarWidget(l,m);
					if(uiBtn && bar){
						this.addOtherWebAppButton(bar.params.webAppName,bar.params.widgetName,uiBtn);
					}
				}
			}
		}
	},

	/**
         * Remove a global button from all WebApps views
         * @param {string} btnName The name of the AppButton to remove
         */
	removeGlobalButton : function(btnName){
		var bar;
		if(this.lstAppBar){
			for (var k in this.lstAppBar){
				if(this.lstAppBar[k]){
					for(var l in this.lstAppBar[k]) {
						bar = this._getAppBarWidget(k,l);
						if(bar) {
							if(this.lstOtherWebAppButtons[bar.params.webAppName][bar.params.widgetName][btnName]){
								this.removeOtherWebAppButton(
									bar.params.webAppName,
									bar.params.widgetName,
									this.lstOtherWebAppButtons[bar.params.webAppName][bar.params.widgetName][btnName]
									);
							}
						}
					}
				}
			}
		}
		if(this.lstAppBarWidgets){
			for (var i in this.lstAppBarWidgets){
				if(this.lstAppBarWidgets[i]){
					for(var j in this.lstAppBarWidgets[i]) {
						bar = this._getAppBar(i,j);
						if(bar) {
							if(this.lstOtherWebAppButtons[bar.params.webAppName][bar.params.widgetName][btnName]){
								this.removeOtherWebAppButton(
									bar.params.webAppName,
									bar.params.widgetName,
									this.lstOtherWebAppButtons[bar.params.webAppName][bar.params.widgetName][btnName]
									);
							}
						}
					}
				}
			}
		}
		for (var m in this.lstGlobalButtons){
			if (this.lstGlobalButtons[m].strButtonName == btnName)
				this.lstGlobalButtons.splice(m,1);
		}
	},




	/**
	 * Add a Home or Back button to a WebApp view
	 * @param {String} strWebAppName The name of the WebApp
	 * @param {String} strWidgetName The name of the View
	 * @param {Boolean} isHome True if the button is a Home button. False for a Back button
	 * @param {Function} callback The callback function
	 */
	addExitButton : function(strWebAppName, strWidgetName, isHome, callback) {
		//Create label
		var strButtonLabel;
		var strButtonIcon;
		var boolNoCreate=false;
		if(isHome){
			strButtonLabel = _("Home", "ICTouchAPI");
			strButtonIcon = 'generic-homepage';
			if(this.lstExitButtons[strWebAppName] && this.lstExitButtons[strWebAppName][strWidgetName] && this.lstExitButtons[strWebAppName][strWidgetName].strButtonLabel == strButtonLabel.getTranslation()){
				boolNoCreate = true;
			}
		}
		else {
			strButtonLabel = _("Back", "ICTouchAPI");
			strButtonIcon = 'generic-back';
		}
		if(!boolNoCreate){
			if (!this.lstExitButtons[strWebAppName]) {
				this.lstExitButtons[strWebAppName] = {}; // new object
			}
			//Remove old exit button if exist
			if (this.lstExitButtons[strWebAppName][strWidgetName]!=null) {
				this.removeExitButton(strWebAppName, strWidgetName);
			}
			//Create button
			var buttonExit = {
				strButtonName: "BACK_BTN",
				strButtonLabel: strButtonLabel,
				strButtonIcon: strButtonIcon,
				callback: callback
			}
			var objButtonExit = new UIElements.AppButton.AppButtonControl(buttonExit);
			this.lstExitButtons[strWebAppName][strWidgetName] = objButtonExit; // index of array is the button name
			ICTouchAPI.AppBarServices.addStaticButton(strWebAppName, strWidgetName, objButtonExit);
		}
		strWebAppName=null;
		strWidgetName=null;
		isHome=null;
		callback=null;
	},

	/**
	 * Remove a Home or Back button from a WebApp view
	 * @param {String} strWebAppName The name of the WebApp
	 * @param {String} strWidgetName The name of the View
	 */
	removeExitButton : function(strWebAppName, strWidgetName){
		if(this.lstExitButtons[strWebAppName] && this.lstExitButtons[strWebAppName][strWidgetName]){
			var objButtonExit = this.lstExitButtons[strWebAppName][strWidgetName];
			if(objButtonExit!=null) {
				ICTouchAPI.AppBarServices.removeStaticButton(strWebAppName, strWidgetName, objButtonExit);
				objButtonExit.destroy();
				delete (this.lstExitButtons[strWebAppName][strWidgetName]);
			}
		}
	}


});
// End of the appBarServices class

// Instanciation of class
ICTouchAPI.AppBarServices = new ICTouchAPI.AppBarServices;
