/**
 * @class ICTouchAPI.APIServices.Video
 * @extends Object
 * @ignore
 */
dojo.provide("ICTouchAPI.APIServices.Video");
dojo.declare("ICTouchAPI.APIServices.Video",[ICTouchAPI.APIServices.APIServicesBase],
{


    args: null, // last arguments used to restore in fake mode


    /**
     * @private
     */
    constructor : function () {

        /* Fake mode - just create an empty div positioned with absolute coordinates */
        this.fakeMode = false;

        /* Check if VideoAPIServices Javascript object exists, otherwise we need to
        to switch to fake mode (use methods that simulate some video content) */
        try {
            if( VideoAPIServices ) {}
        }
        catch(e) {
            /* It seems that VideoAPIServices does not exists... */
            this.fakeMode = true;
        }

    },


   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>capability of the module</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the name of the capability
     * <pre>
     * {String} the name of the capability
     * </pre>
     *
     * The callback will be called with the capability, following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;name {String}
     * &nbsp;&nbsp;state {Number}
     * }
     * </pre>
     */
    getCapabilityGlobal : function(args) {
            this.callCoreApplicationMethod({
                    domainName: "Video",
                    methodName: "getCapability",
                    context: args.context,
                    params: args.params,
                    paramType: ["QString"],
                    callback: args.callback,
                    callbackParams: args.callbackParams
            });
            args = null;
    },


    /**
     * Start a local video session
     *
     * @return {Boolean} created TRUE if video has been successfully created
     */
    createLocalVideo : function( args ) {
        //console.log( "ICTouchAPI.APIServices.Video", "setLocalVideo", args );

        var created = false; // false = an error occurred

        if( this.fakeMode === false ) {
            created = VideoAPIServices.createLocalVideo( args.x, args.y, args.w, args.h );
        } else {
            created = ICTouchAPI.APIServices.Video.setFakeVideo( args.x, args.y, args.w, args.h );
        }

        if(created == true ) {
            this.args = {params: args};
        }

        return created;
    },

    /**
     * Stop a local video session
     *
     * @return {Boolean} deleted TRUE if video has been successfully deleted
     */
    deleteLocalVideo : function() {
    	//console.log( "ICTouchAPI.APIServices.Video", "deleteLocalVideo" );

    	var deleted = false; // false = an error occurred

        if( this.fakeMode === false ) {
            deleted = VideoAPIServices.deleteLocalVideo();
        } else {
            deleted = ICTouchAPI.APIServices.Video.deleteFakeVideo();
        }

        return deleted;
    },


    /**
     * Display remote video given window parameters.
     *
     * @return {Boolean} shown TRUE if video has been successfully hidden
     */
    showVideo : function( args ) {

    	var shown = false; // false = an error occurred

        if( this.fakeMode === false ) {
            shown = VideoAPIServices.showVideo( args.callId, args.x, args.y, args.w, args.h );
        } else {
            shown = ICTouchAPI.APIServices.Video.setFakeVideo( args.x, args.y, args.w, args.h );
        }

        if(shown == true ) {
            this.args = {params: args};
        }

        return shown;
    },


    fullscreen: function( callId ) {},


    /**
     * Hide remote video from screen
     *
     * @return {Boolean} hidden TRUE if video has been successfully hidden
     */
    hideVideo : function( callId ) {

    	var hidden = false; // false = an error occurred

        if( this.fakeMode === false ) {
            hidden = VideoAPIServices.hideVideo( callId );
        } else {
            hidden = ICTouchAPI.APIServices.Video.deleteFakeVideo();
        }

        return hidden;
    },



    /**
     * Create a call video
     *
     * @return {Boolean} created TRUE if video has been successfully created
     */
    createVideo: function(args) {
        //console.log( "ICTouchAPI.APIServices.Video", "createVideo", args );

        var created = false; // false = an error occurred

        if( this.fakeMode === false ) {
            created = VideoAPIServices.createVideo( args.callId, args.sipCallId, args.x, args.y, args.w, args.h, args.pip, args.mute );
        } else {
            created = ICTouchAPI.APIServices.Video.setFakeVideo( args.x, args.y, args.w, args.h );
        }

        if(created == true ) {
            this.args = {params: args};
        }

        return created;
    },


    /**
     * Configure look and feel for a video session.
     * @param {Number} background_color ARGB Widget background color
     * @param {String} text main window displayed text when no video
     * @param {Number} text_color text ARGB color
     * @param {String} pip_text PiP area displayed text when no video
     */
    setVideoMMIParams: function( background_color, text, text_color, pip_text ) {

        var success = false; // false = an error occurred

        if( this.fakeMode === false ) {
            success = VideoAPIServices.setVideoMMIParams( background_color, text, text_color, pip_text );
        } else {
            success = true;
        }

        return success;
    },


    /**
     * Stop video session
     *
     * @param {Number} callId (call id)
     * @return {Boolean} deleted TRUE if video has been successfully deleted
     */
    deleteVideo : function( callId ) {
    	//console.log( "ICTouchAPI.APIServices.Video", "deleteVideo" );

    	var deleted = false; // false = an error occurred

        if( this.fakeMode === false ) {
            deleted = VideoAPIServices.deleteVideo( callId );
        } else {
            deleted = ICTouchAPI.APIServices.Video.deleteFakeVideo();
        }

        return deleted;
    },


    /**
     * Mask video according to parameters
     *
     * @param {Number} num mask number 0 (reserved to Pip) or 1.
     * @param {Number} x horizontal window absolute origin in pixels.
     * @param {Number} y vertical window absolute origin in pixels.
     * @param {Number} w video window width in pixels (MULTIPLE OF 8!!).
     * @param {Number} h video window height in pixels (MULTIPLE OF 2!!).
     * @return {Boolean} enabled TRUE if the mask has been successfully enabled
     */
    enableMask : function(num, x, y, w, h) {

    	var enabled = false; // false = an error occurred

        if( this.fakeMode === false ) {
            enabled = VideoAPIServices.enableMask( num, x, y, w, h );
        } else {
            enabled = ICTouchAPI.APIServices.Video.enableFakeMask( x, y, w, h );
        }

        return enabled;
    },

    /**
     * Remove mask
     *
     * @param {Number} num mask number
     * @return {Boolean} disabled TRUE if the mask has been successfully disabled
     */
    disableMask : function(num) {

    	/** @todo check which one of TRUE or FALSE means 'success' */
    	var disabled = false; // false = an error occurred

        if( this.fakeMode === false ) {
            disabled = VideoAPIServices.disableMask( num );
        } else {
            disabled = ICTouchAPI.APIServices.Video.disableFakeMask();
        }

        return disabled;
    },




    //
    // VIDEO FAKE METHODS
    //

    setFakeVideo: function( x, y, w, h ) {

    	var videoNodeName = 'fakevideo';

    	if( window.top.document.getElementById(videoNodeName) )
    	{
            var videoNode = window.top.document.getElementById(videoNodeName);
            var videoNodeContent = window.top.document.getElementById(videoNodeName+'content');
            videoNode.style.display = 'block';
            videoNode.style.left = x + "px";
            videoNode.style.top = y + "px";
            videoNode.style.width = (w) + "px";
            videoNode.style.height = (h) + "px";
            videoNodeContent.style.width = (w -2) + "px";
            videoNodeContent.style.height = (h -2) + "px";
    	}

    	else
    	{
            var videoNode = document.createElement("div");
            videoNode.id = videoNodeName;
            videoNode.style.left = x + "px";
            videoNode.style.top = y + "px";
            videoNode.style.width = (w) + "px";
            videoNode.style.height = (h) + "px";
            videoNode.style.position = 'absolute';
            videoNode.style.zIndex = '80000';
            videoNode.style.overflow = 'hidden';
            videoNode.onclick = function() {
                if( parseInt(videoNode.style.width) < 800 ) {
                    ICTouchAPI.APIServices.Video.setFakeVideo( 0, 0, 800, 480 );
                }
                else {
                    var params = ICTouchAPI.APIServices.Video.args.params;
                    ICTouchAPI.APIServices.Video.setFakeVideo( params.x, params.y, params.w, params.h );
                }
            };

            var videoNodeContent = document.createElement("div");
            videoNodeContent.id = videoNodeName + 'content';
            videoNodeContent.style.width = (w -2) + "px";
            videoNodeContent.style.height = (h -2) + "px";
            videoNodeContent.style.border = "1px solid #757575";
            videoNodeContent.style.background = "url(data:image/gif;base64,R0lGODlhDwAPALMMAP/FAP+wAP/CAP/AAP+vAP/EAP+sAP+2AP+rADMzMwAAAP///////wAAAAAAAAAAACH5BAEAAAwALAAAAAAPAA8AAARIkEml1lIpyT2tV0PGVZ6lFBTFkKVZUhfsxm76mWQqX+2SADrb65f76DKC4kwR0CSUvIuGkTgIeQrDlEoIsCiIrSTzFXE2wXMEADs=) 50% 50% no-repeat";
            videoNodeContent.style.backgroundColor = "#000";

            videoNode.appendChild( videoNodeContent );
            window.top.document.body.appendChild( videoNode );
    	}


    	return true;
    },

    deleteFakeVideo: function() {
    	if( window.top.document.getElementById('fakevideo') ) {
            window.top.document.getElementById('fakevideo').style.display = 'none';
    	}
    	return true;
    },

    /* Works only for full width mask (like audio toaster) */
    enableFakeMask: function(x,y,w,h) {

    	if( window.top.document.getElementById('fakevideo') ) {
            var videoNode = window.top.document.getElementById('fakevideo');

            /* Calculate the masked video sizes */
            var height = parseInt( videoNode.style.height, 10),
                    top = parseInt( videoNode.style.top, 10);
            // audiotoaster_height - (total_screen_height - (video_top_position + video_height) )
            var overlap = 165 - (480 - (top + height));
            var new_height = height - overlap;

            videoNode.style.height = new_height+ 'px';
    	}

    	return true;
    },

    disableFakeMask: function() {

    	if( window.top.document.getElementById('fakevideo') )
    	{
            var videoNode = window.top.document.getElementById('fakevideo');
            var videoNodeContent = window.top.document.getElementById('fakevideocontent');

            /* Calculate the full video height */
            var height = videoNodeContent.style.height;
            var new_height = parseInt( height, 10 ) + 2;

            videoNode.style.height = new_height + 'px';
    	}

    	return true;
    }

});

ICTouchAPI.APIServices.Video = new ICTouchAPI.APIServices.Video();