/**
 * @class ICTouchAPI.APIServices.Telephony
 * @singleton
 * @extends ICTouchAPI.APIServices.APIServicesBase
 * The telephony module is responsible for the general status
 * of the telephony on the phone. It provides all the services that are related to communication services.
 * <br><br>
 * Depending on the status of the phone, each feature is available or not.<br>
 * The information to know if a feature is available at a given moment is provided
 * by a capabilities system. So a webapp that is wanting to use a feature needs
 * first to check if the according capability is available.
 * <br>For example, if a webapp provides the "makeCall" feature, it should only make
 * it available when the "makeCall" capability is available.
 * <br><br>
 * The Telephony class is used as a singleton that carries the main phoning
 * state machine.<br>
 * The Telephony class is the owner of the PhoneCall class instances, it
 * creates and destroys them depending on the life of the according phone calls.<br>
 * The singleton reacts to the telephonic events from the SIPUA, to the service calls
 * from the presentation and provides events that describe the status of the telephony
 * module to all other modules that are connected on the DBUS. The general status of
 * the telephony is handled by a state machine that is owned by this class. The status
 * of each individual phone call is handled by a specific state machine owned by each
 * instance of the class PhoneCall.
 */
 
 
dojo.provide("ICTouchAPI.APIServices.Telephony");
dojo.declare("ICTouchAPI.APIServices.Telephony",[ICTouchAPI.APIServices.APIServicesBase],
{
	
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>list of the current phoneCalls</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Get the list of current phoneCalls.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with the list of the current phoneCalls,each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
     * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
     * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
     * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
     * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
     * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
     * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
     * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
     * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
     * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
     * &nbsp;&nbsp;startTime {Date} Start time
     * &nbsp;&nbsp;isConference {Boolean} Conferencing status
     * &nbsp;&nbsp;isConnected {Boolean}Connection status
     * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
     * &nbsp;&nbsp;isHeld {Boolean} Held Status
     * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
     * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
     * &nbsp;&nbsp;isActive {Boolean} Active Status
     * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
     * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
     * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
     * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
     * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
     * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
     * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
     * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
     * &nbsp;&nbsp;displayName {Boolean} Display Name
     * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
     * }
     * </pre>
     *
     * The parameter callId has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The parameter phoneNumber has the following structure:
     * <pre>
     * {String} sip phone number URI
     * </pre>  
     *
     * The parameter callDirection has the following structure:
     * <pre>
     * {String} call direction from the values,possilbe values are:
     * -"CALL_INCOMING" for incoming call
     * -"CALL_OUTGOING" for outgoing call
     * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
     * </pre>  
     *
     * The parameter startReason and endReason have the following structure:
     * <pre>
     * {String} call reason from the values,possilbe values are:
     * -"CALL_REASON_NORMAL" for normal call
     * -"CALL_REASON_TRANSFERRED" for transferred call
     * </pre>
     * @ignore
     */	
    getPhoneCalls : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "getPhoneCalls",
            context : args.context,
            callback : args.callback,
            errorCallback : args.errorCallback,
            callbackParams: args.callbackParams
        });
        args = null;
    },


    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>list of Ids of the current phoneCalls</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Get the list of current phoneCalls.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with the list of Ids of the current phoneCalls,each of them following this structure:
     *
     * <pre>
     * {String}  phone call identifier 
     * </pre>
     * @ignore
     */	
    getPhoneCallIds : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "getPhoneCallIds",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>data of the phoneCall</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Get phone call data from one phoneCall.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the identifier of the phoneCall,it has the following structure:
     * <pre>
     * {String}  phone call identifier 
     * </pre>
     
     * The callback will be called with the data of the phoneCall,following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
     * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
     * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
     * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
     * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
     * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
     * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
     * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
     * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
     * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
     * &nbsp;&nbsp;startTime {Date} Start time
     * &nbsp;&nbsp;isConference {Boolean} Conferencing status
     * &nbsp;&nbsp;isConnected {Boolean}Connection status
     * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
     * &nbsp;&nbsp;isHeld {Boolean} Held Status
     * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
     * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
     * &nbsp;&nbsp;isActive {Boolean} Active Status
     * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
     * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
     * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
     * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
     * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
     * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
     * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
     * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
     * &nbsp;&nbsp;displayName {Boolean} Display Name
     * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
     * }
     * </pre>
     *
     * The parameter callId has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The parameter phoneNumber has the following structure:
     * <pre>
     * {String} sip phone number URI
     * </pre>  
     *
     * The parameter callDirection has the following structure:
     * <pre>
     * {String} call direction from the values,possilbe values are:
     * -"CALL_INCOMING" for incoming call
     * -"CALL_OUTGOING" for outgoing call
     * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
     * </pre>  
     *
     * The parameter startReason and endReason have the following structure:
     * <pre>
     * {String} call reason from the values,possilbe values are:
     * -"CALL_REASON_NORMAL" for normal call
     * -"CALL_REASON_TRANSFERRED" for transferred call
     * </pre>
     * @ignore
     */	

    getPhoneCallFromId : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "getPhoneCallFromId",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>information of the started phoneCall</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Start an outgoing phone Call.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp; phoneNumber {Object} the phoneNumber identifier of the phoneCall.See the description below
     * &nbsp;&nbsp; hideIdentity {Boolean} indicates if the user explicitly asked to hide his identity : true to hide, false to use the default value deduced from the setting "IdentityRestriction"(false by default).
     * &nbsp;&nbsp; doDialableNumber {Boolean} if set to true, convert the given phone number(which may contain non-digit signs like ")", "+", ...) to a number that fits to the numbering plan.
     * &nbsp;&nbsp; bDTMF {Boolean} indicates if the DTMF keypad should be automatically opened on successful connection
     * }
     * </pre>
     *
     * The parameter phoneNumber has the following structure:
     * <pre>
     * {String} sip phone number URI
     * </pre>  
     *
     * The callback will be called with the data of the phoneCall,following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
     * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
     * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
     * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
     * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
     * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
     * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
     * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
     * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
     * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
     * &nbsp;&nbsp;startTime {Date} Start time
     * &nbsp;&nbsp;isConference {Boolean} Conferencing status
     * &nbsp;&nbsp;isConnected {Boolean}Connection status
     * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
     * &nbsp;&nbsp;isHeld {Boolean} Held Status
     * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
     * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
     * &nbsp;&nbsp;isActive {Boolean} Active Status
     * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
     * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
     * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
     * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
     * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
     * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
     * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
     * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
     * &nbsp;&nbsp;displayName {Boolean} Display Name
     * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
     * }
     * </pre>
     *
     * The parameter callId has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The parameter phoneNumber has the following structure:
     * <pre>
     * {String} sip phone number URI
     * </pre>  
     *
     * The parameter callDirection has the following structure:
     * <pre>
     * {String} call direction from the values,possilbe values are:
     * -"CALL_INCOMING" for incoming call
     * -"CALL_OUTGOING" for outgoing call
     * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
     * </pre>  
     *
     * The parameter startReason and endReason have the following structure:
     * <pre>
     * {String} call reason from the values,possilbe values are:
     * -"CALL_REASON_NORMAL" for normal call
     * -"CALL_REASON_TRANSFERRED" for transferred call
     * </pre>    
     */	
    startPhoneCall : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "startPhoneCall",
            context : args.context,
            params: args.params,
            paramType: ["PhoneNumber", "bool", "bool", "bool"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Decprecated
	 * Use startPhoneCall instead.
     * @ignore
     */
    startPhoneCallWithDialableNumber : function(args) {
		this._deprecated("startPhoneCallWithDialableNumber", "startPhoneCall", arguments);
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Answer to an incoming phone call.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;callId {Object} the identifier of a phoneCall answer.See the description below
     * &nbsp;&nbsp;hide {Boolean} indicates if the user explicitly asked to hide his identity : true to hide, false to use the default value deduced from the setting "IdentityRestriction".
     * }
     * </pre>
     *
     * The parameter callId has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */

    answerPhoneCall : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "answerPhoneCall",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId", "bool"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Ignore an incoming phone call. The phone is ringing but the user does not hook off after a few time the phone call is forwarded to the VM.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the identifier of the phoneCall to be ignored,it has the following structure:
     * <pre>
     * {String} the identifier of the phoneCall
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    ignoreCall : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "ignoreCall",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Reject an incoming phone call.This operation is provided to be able to stop an incoming call without hooking off the phone. While the phone is ringing, the user presses a button to stop the communication.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the identifier of a phoneCall rejected,it has the following structure:
     * <pre>
     * {String} the identifier of the phoneCall
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    rejectCall : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "rejectCall",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br> Stop a phoneCall.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the identifier of the phoneCall to be stopped,it has the following structure:
     * <pre>
     * {String} the identifier of the phoneCall
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    stopPhoneCall : function(args) {
        /*Call the method directly using IctWebClient module*/
        if (typeof TelephonyAPIServices != "undefined" && TelephonyAPIServices.stopPhoneCall) {
            TelephonyAPIServices.stopPhoneCall(args.params);
        }
        else {
            this.callCoreApplicationMethod({
                domainName : "Telephony",
                methodName : "stopPhoneCall",
                context : args.context,
                params: args.params,
                paramType: ["PhoneCallId"],
                callback : args.callback,
                callbackParams: args.callbackParams
            });
        }
        args = null;
    },


    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br> Start a conference between the first CallId c1 and the second CallId c2.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp; c1 {Object} the identifier of the first participant of the conference.See the description below.
     * &nbsp;&nbsp; c2 {Object} the identifier of the second participant of the conference.See the description below.
     * }
     * </pre>
     *  
     * params c1 and c2 have the following structure:
     * <pre>
     * {String} the identifier of the phoneCall
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    startConference : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "startConference",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId", "PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },


    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Add a participant to a conference.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp; callId  {Object} the identifier of the conference call.See the description below.
     * &nbsp;&nbsp; phoneNumber {Object} the phonenumber of the participant to add.See the description below.
     * &nbsp;&nbsp; makeDialableNumber {Boolean} if set to true, convert the given phone number(which may contain non-digit signs like ")", "+", ...) to a number that fits to the numbering plan.
     * }
     * </pre>
     *  
     * params callId has the following structure:
     * <pre>
     * {String} the identifier of the phoneCall
     * </pre>  
     *
     * The parameter phoneNumber has the following structure:
     * <pre>
     * {String} sip phone number URI
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    addParticipant : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "addParticipant",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId", "PhoneNumber", "bool"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

	transferConferenceToNumber : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "transferConferenceToNumber",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId", "QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },


    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Stop the current conference.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	stopConference : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "stopConference",
			context: args.context,
			params: {},
			callback: args.callback
		});
		args = null;
	},

	leaveConferenceN : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "leaveConferenceN",
			context: args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
			callback: args.callback
		});
		args = null;
	},
	
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br> End a conference.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the identifier of the conference call,it has the following structure:
     * <pre>
     * {String} the identifier of the conference call
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */

	stopConferenceN : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "stopConferenceN",
			context: args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
			callback: args.callback
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Drop a participant from a conference.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp; participantId {String} the id of the participants to drop (only one for the moment)
     * &nbsp;&nbsp; callId {Object} the identifier of the conference call.See the description below.
     * }
     * </pre>
     *
     * The parameter callId has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	dropParticipant : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "dropParticipant",
			context: args.context,
            params: args.params,
            paramType: ["QString", "PhoneCallId"],
			callback: args.callback
		});
		args = null;
	},


    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Deactivate the local (SIP) forward.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     */
    deactivateForward : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "deactivateForward",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Activate a forward.</b>
     * <br> The automatic "Forward" feature is set to the destination phone number.
     * <br> When an incoming call arrives, it is automatically forwarded depending on the type of forward set.
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the forward definition in a string form
     * <pre>
     * {String} the forward definition in a string form
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     */
    activateForward : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "activateForward",
            context : args.context,
            params: args.params,
            paramType: ["QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Activate a forward.</b>
     * <br> The automatic "Forward" feature is set to the destination phone number.
     * <br> When an incoming call arrives, it is automatically forwarded depending on the type of forward set.
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the forward type in a string form
     * <pre>
     * {String} the forward type in a string form
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    activateForwardDest : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "activateForwardDest",
            context : args.context,
            params: args.params,
            paramType: ["QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Implements a "Consult transfer".An existing phoneCall is transfered to an other existing phoneCall.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp; callId1 {Object} the identifier of the phoneCall to be transferred.See the description below.
     * &nbsp;&nbsp; callId2 {Object} the identifier of the phoneCall to be transferred.See the description below.
     * }
     * </pre>
     *  
     * params callId1 and callId2 have the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	transferPhoneCall : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "transferPhoneCall",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId", "PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },


    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Merge two calls into a conference.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp; callIdActive {Object} the identifier of the current active call.See the description below.
     * &nbsp;&nbsp; callIdHeld {Object} the identifier of the call to merge with the active call.See the description below.
     * }
     * </pre>
     *  
     * params callIdActive and callIdHeld have the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	mergePhoneCalls : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "mergePhoneCalls",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId", "PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Deflect an incoming phone call to a phone number.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp; callId  {Object} the identifier of the phoneCall to deflect.See the description below.
     * &nbsp;&nbsp; number {Object}  the destination number of the deflect.See the description below.
     * &nbsp;&nbsp; makeDialableNumber {Boolean} if set to true, convert the given phone number(which may contain non-digit signs like ")", "+", ...) to a number that fits to the numbering plan.
     * }
     * </pre>
     *  
     * params callId has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The parameter number has the following structure:
     * <pre>
     * {String} sip phone number URI
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	deflectToNumber : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "deflectToNumber",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId", "PhoneNumber", "bool"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },


    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Implements a "Blind transfer".An existing phoneCall is transfered to a phone number.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp; callId  {Object} the identifier of the existing phoneCall.See the description below.
     * &nbsp;&nbsp; n {Object}  the identifier of the phoneNumber to be transfered to.See the description below.
     * &nbsp;&nbsp; makeDialableNumber {Boolean} if set to true, convert the given phone number(which may contain non-digit signs like ")", "+", ...) to a number that fits to the numbering plan.
     * }
     * </pre>
     *  
     * params callId has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The parameter n has the following structure:
     * <pre>
     * {String} sip phone number URI
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	transferPhoneCallToNumber : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "transferPhoneCallToNumber",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId", "PhoneNumber", "bool"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },


    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Put the incoming phoneCall in hold.The caller is on hold by the callee.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params callId is a JSON, representing the identifier of a phoneCall put on hold,has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    putOnHold : function (args) {
		this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "putOnHold",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Retrieve an incoming phone call that is on hold.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params callId is a JSON, representing the identifier of the entry to be retrieved,has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	retrieveCall : function(args) {
		this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "retrieveCall",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Request the distant to call back.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params callId is a JSON, representing the identifier of the phonecall,has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	requestCallBack : function(args) {
		this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "requestCallBack",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Activate the "Do Not Disturb" feature.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the password 
     * <pre>
     * {String} password for the feature "Do Not Disturb"
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     */
    activateDnd : function(args) {
		args.params[0]	= Encryption.encryptRSA(args.params[0]);
		
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "activateDnd",
            context : args.context,
            params: args.params,
            paramType: ["QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Deactivate the "Do Not Disturb" feature.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     */
    deactivateDnd : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "deactivateDnd",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method sends a DTMF (dual-tone multi-frequency) coded digit into a call,for example when calling a vocal machine that requires answers based on DTMF.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp; callId  {Object} the identifier of the phoneCall destination of DTMF.See the description below.
     * &nbsp;&nbsp; digit {Number}  the DTMF digit (allowed value : 0 to 15)
     * }
     * </pre>
     *  
     * params callId has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     */
    sendDTMF : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "sendDtmf",
            params: args.params,
            paramType: ["PhoneCallId", "int"],
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method is to be called when the DTMF key is released.</b>
     * <br> It stops the DTMF tone if AudioDtmfDuration setting is set to "FollowKey"
     * <br>Note: in the mode "FollowKey", a timer ends the DTMF tone if this method is not called by presentation.
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     */
    sendDTMFRelease : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "sendDtmfKeyRelease",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>The presentation initiate a phoneCall,the dialer pad is ready to be used.This action was initiated from GUI, using the action "New Call".</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    initiateOutgoingCallGUI : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "initiateOutgoingCallGUI",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>The presentation initiate a phoneCall,the dialer pad is ready to be used.
     * <br>This action was initiated from Application level, after a physical hook off.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    initiateOutgoingCallBO : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "initiateOutgoingCallBO",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>The presentation shows the dialer pad,a key is pressed for the dialing.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    dialKeyEnter : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "dialKeyEnter",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Cancel an outgoing call.
     * <br>The presentation shows the dialer pad, the dial tone is activated then the phone call is stopped.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    cancelOutgoingCall : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "cancelOutgoingCall",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},


	getForwardTypeList : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "getForwardTypeList",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

	getForwardDestinationList : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "getForwardDestinationList",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>phoneCall </b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br> Create a phoneCall to the system to listen to the voice mail.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with the phoneCall,following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
     * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
     * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
     * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
     * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
     * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
     * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
     * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
     * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
     * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
     * &nbsp;&nbsp;startTime {Date} Start time
     * &nbsp;&nbsp;isConference {Boolean} Conferencing status
     * &nbsp;&nbsp;isConnected {Boolean}Connection status
     * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
     * &nbsp;&nbsp;isHeld {Boolean} Held Status
     * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
     * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
     * &nbsp;&nbsp;isActive {Boolean} Active Status
     * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
     * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
     * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
     * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
     * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
     * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
     * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
     * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
     * &nbsp;&nbsp;displayName {Boolean} Display Name
     * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
     * }
     * </pre>
     *
     * The parameter callId has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The parameter phoneNumber has the following structure:
     * <pre>
     * {String} sip phone number URI
     * </pre>  
     *
     * The parameter callDirection has the following structure:
     * <pre>
     * {String} call direction from the values,possilbe values are:
     * -"CALL_INCOMING" for incoming call
     * -"CALL_OUTGOING" for outgoing call
     * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
     * </pre>  
     *
     * The parameter startReason and endReason have the following structure:
     * <pre>
     * {String} call reason from the values,possilbe values are:
     * -"CALL_REASON_NORMAL" for normal call
     * -"CALL_REASON_TRANSFERRED" for transferred call
     * </pre>    
     */	
	listenVM : function(args) {
        this.callCoreApplicationMethod({
            domainName: "Telephony",
            methodName: "listenVM",
            context: args.context,
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },


    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Overflow an outgoing call to the callee voicemail if it exists.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the identifier of the phoneCall,it has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
        overflowToVM : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "overflowToVM",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Deflect an incoming phone call to the voice mail.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the identifier of the phoneCall,it has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
        deflectToVM : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "deflectToVM",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Deflect an incoming phone call to the DVA.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the identifier of the phoneCall deflected to the DVA,it has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
        deflectToDVA : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "deflectToDVA",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>capability of the module</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the name of the capability
     * <pre>
     * {String} the name of the capability
     * </pre>
     *
     * The callback will be called with the capability, following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;name {String} 
     * &nbsp;&nbsp;state {Number} 
     * }
     * </pre>
     */
	getCapabilityGlobal : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "getCapability",
			context: args.context,
			params: args.params,
			paramType: ["QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},


   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>capability of an object of the module</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;name {String}  Name of the capability
     * &nbsp;&nbsp;listId {Number} ID of the list of capabilities
     * }
     * </pre>
     *
     * The callback will be called with the capability, following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;name {String} 
     * &nbsp;&nbsp;state {Number} 
     * }
     * </pre>
     */
	getCapabilityInstance : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "getCapability",
			context: args.context,
			params: args.params,
			paramType: ["int", "QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *<b><br><br>Pickup an incoming call on another user.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp; otherCallRef {String}  Reference of the call to pickup  (on the remote user)
     * &nbsp; otherPhoneNumber {String}  Phone number on which the call is to pickup.
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	pickupCall : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "pickupCall",
			context: args.context,
			params: args.params,
			paramType: ["QString", "QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *<b><br><br>Intrude into a specified active call in a special mode (secrete or not) of another user.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp; otherCallRef {String} Reference of the call to intrude (on the remote user)
     * &nbsp; otherPhoneNumber {String} Phone number on which the call is to be intruded.
     * &nbsp; hidden {Boolean} True to hide the user for the participants of this call.
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	intrudeCall : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "intrudeCall",
			context: args.context,
			params: args.params,
			paramType: ["QString", "QString", "bool"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *<b><br><br>Get the notification context.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	getNotificationContext : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "getNotificationContext",
			context: args.context,
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

	/*getRemoteCall : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "getRemoteCall",
			context: args.context,
			callback: args.callback,
			callbackParams: args.callbackParams
		});
	},*/

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *<b><br><br>Call the sender of a visual voice mail message.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;  idMessage {String} the identifier of the message.
     * &nbsp;  number {String} the identifier of the message.
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	callMessageSender : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "callMessageSender",
            context : args.context,
            params: args.params,
            paramType: ["QString", "QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *<b><br><br>Forward a visual voice mail message by e-mail.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp; idMessage  {String} the identifier of the message to forward.
     * &nbsp; email {String} the e-mail address of the recipient.
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     */
	forwardMessageByEmail : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "forwardMessageByEmail",
            context : args.context,
            params: args.params,
            paramType: ["QString","QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *<b><br><br> Forward a visual voice mail message to another voice mail.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp; idMessage {Number} the identifier of the message to forward.
     * &nbsp; vmNumber {Object}  the phone number of the recipient.See the description below.
     * }
     * </pre>
     *
     * The parameter phoneNumber has the following structure:
     * <pre>
     * {String} sip phone number URI
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     */
	forwardMessageToVM : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "forwardMessageToVM",
            context : args.context,
            params: args.params,
            paramType: ["QString","PhoneNumber"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *<b><br><br>Start playing a visual voice mail message.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the identifier of the message to play
     * <pre>
     * {String} the identifier of the message 
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     */
	playMessage : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "playMessage",
            context : args.context,
            params: args.params,
            paramType: ["QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },
    
     /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *<b><br><br>Resume a paused visual voice mail message.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp; idMessage {String} the identifier of the message to resume.
     * &nbsp; offset {Number} the position from where the message has to be played again.
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     */

	resumeMessage : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "resumeMessage",
            context : args.context,
            params: args.params,
            paramType: ["QString","int"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *<b><br><br>Pause a playing visual voice mail message.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the identifier of the message to pause
     * <pre>
     * {String} the identifier of the message 
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     */
	pauseMessage : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "pauseMessage",
            context : args.context,
            params: args.params,
            paramType: ["QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *<b><br><br>Stop playing a visual voice mail message.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the identifier of the message to stop
     * <pre>
     * {String} the identifier of the message 
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     */
	stopMessage : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "stopMessage",
            context : args.context,
            params: args.params,
            paramType: ["QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Start recording call conversation in voice mail.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the callId of the call,it has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	startCallRecording : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "startCallRecording",
			context: args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
			callback: args.callback
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Pause recording the call conversation.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the callId of the call,it has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	pauseCallRecording : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "pauseCallRecording",
			context: args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
			callback: args.callback
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Resume recording the call conversation.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the callId of the call,it has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	resumeCallRecording : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "resumeCallRecording",
			context: args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
			callback: args.callback
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Stop recording the call conversation.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the callId of the call,it has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	stopCallRecording : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "stopCallRecording",
			context: args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
			callback: args.callback
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>list containing the other devices of the user</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Get the list of user devices.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with the list containing the other devices of the user( i.e all devices except local device), each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; _phoneNumber {String}
     * &nbsp;&nbsp; _label {String}
     * &nbsp;&nbsp; _state {Object} See the description below.
     * &nbsp;&nbsp; _idCapabilities {Number}
     * }
     * </pre>
     *
     * The parameter _state is a {Object},representing enumerated type values, used by applications.
     * Possible values are : 
     * <pre>
     * - "UNKNOWN" for 0 (Init value : device with unknown state)
     * - "IN_SERVICE " 1
     * - "OUT_OF_SERVICE " 2     
     * </pre>
     * @ignore
     */
     
    getOtherDevicesList : function(args) {
        this.callCoreApplicationMethod({
            domainName: "Telephony",
            methodName: "getOtherDevicesList",
            context: args.context,
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Switch de call from one user device to another user device</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp; callId {Object} callId of the call to be switched.See the description below
     * &nbsp;&nbsp; device {String} phone number of the device on which the call is switched
     * }
     * </pre>
     *
     * The parameter callId has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */

    switchCallOnDevice : function(args) {
        this.callCoreApplicationMethod({
            domainName: "Telephony",
            methodName: "switchCallOnDevice",
            context: args.context,
            params: args.params,
            paramType: ["PhoneCallId","QString"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Switch the call from a remote device to the local device</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the callId of the call to be switched,it has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    getCallFromDevice : function(args) {
        this.callCoreApplicationMethod({
            domainName: "Telephony",
            methodName: "getCallFromDevice",
            context: args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Terminate a call. Used to end a call on another device. same code as stopConferenceN.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the callId of the call to be switched,it has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    terminateCall : function(args) {
        this.callCoreApplicationMethod({
            domainName: "Telephony",
            methodName: "terminateCall",
            context: args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

	/**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Return the canonical number from a phone number. Used to get a phone number usable with the numbering library when making a call.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a string, corresponding to the phone number to be transformed by the telephony
     *
     * The callback will be called with the transformed phone number, in a string:
     *
     * <pre>
     * {String}  transformed phone number
     * </pre>
     * @public
     */
    makeCanonicalNumber : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "makeCanonicalNumber",
            context : args.context,
            params: args.params,
            paramType: ["QString"],
            callback : args.callback,
            callbackParams: args.callbackParams,
			sync : (args.sync ? args.sync : false)
        });
        args = null;
    },

	/**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Return the list of supervised users in the user's supervision group.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a {JSON} in parameter.
     *
     * <pre>
     * {JSON} list of supervised
     * </pre>
     * @public
     */
	getSupervisionCurrentMembers : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "getSupervisionCurrentMembers",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams,
			sync : (args.sync ? args.sync : false)
        });
        args = null;
    },

	/**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Login or logout from supervision group</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void.
     *
     * <pre>
     * void
     * </pre>
     * @public
     */
	loginLogoffSupervision : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "loginLogoffSupervision",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams,
			sync : (args.sync ? args.sync : false)
        });
        args = null;
    },
	
	/**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Activate/deactivate hide identity feature.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
	 * 
	 * params is a bool, corresponding to the hide identitiy state you want to set.
	 *
     * The callback will be called with a void.
     *
     * <pre>
     * void
     * </pre>
     * @ignore
     */
	modifyHideIdentity : function(args) {
		this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "modifyHideIdentity",
            context : args.context,
            params: args.params,
            paramType: ["bool"],
            callback : args.callback,
            callbackParams: args.callbackParams,
			sync : (args.sync ? args.sync : false)
        });
        args = null;
	},

	/**
	 * modify media
	 * 1st arg : callID
	 * 2nd arg : VideoActionType
	 *				VIDEO_START
	 *				VIDEO_STOP
	 *				VIDEO_NO_ACTION
     * @ignore
	 */
	modifyMedia : function(args) {
        this.callCoreApplicationMethod({
            domainName: "Telephony",
            methodName: "modifyMedia",
            context: args.context,
            params: args.params,
            paramType: ["PhoneCallId","VideoActionType"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},


    /**
     * Deprecated
	 * Use makeCanonicalNumber instead.
     * @ignore
     */
    deletePrefixFromNumber : function(args) {
		this._deprecated("deletePrefixFromNumber", "makeCanonicalNumber", arguments);
        args = null;
    },



    /**
     * @ignore
     */
	setRoutingOverflow : function(args){
		this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "setRoutingOverflow",
            context : args.context,
            params: args.params,
            paramType: ["OverflowDestination", "bool"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

    /**
     * @ignore
     */
	activateRoutingProfile : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "activateRoutingProfile",
            context : args.context,
            params: args.params,
            paramType: ["ProfileId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },


    /**
     * @ignore
     */
	cancelRoutingOverflow : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "cancelRoutingOverflow",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * @ignore
     */
	getRoutingProfiles : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "getRoutingProfiles",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * @ignore
     */
	getRoutingProfile : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "getRoutingProfile",
			params: args.params,
			paramType: ["ProfileId"],
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * @ignore
     */
	getRoutingState : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "getRoutingState",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * @ignore
     */
	setRoutingRoutes : function(args) {
		this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "setRoutingRoutes",
			params: args.params,
			paramType: ["QStringList", "bool"],
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

    /**
     * @ignore
     */
	setRoutingForward : function(args){
		this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "setRoutingForward",
            context : args.context,
            params: args.params,
            paramType: ["ForwardDestination"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

    /**
     * @ignore
     */
	openDoor : function(args) {
		this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "openDoor",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	}

/**
 * @event CallConnectedEvent 
 * This event is used to notify that a phone call has connected, which means that callers can speak to each other.
 * @param {Object} param contains the following structure:
 * <pre>
 * {    
 * &nbsp;&nbsp;callId {Object} telephony level call unique identifier.
 * &nbsp;&nbsp;phoneCall {Object} contains all the information related to this phone call instance.
 }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>
 *
 * The parameter phoneCall has the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
 * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
 * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
 * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
 * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
 * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
 * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
 * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
 * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
 * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
 * &nbsp;&nbsp;startTime {Date} Start time
 * &nbsp;&nbsp;isConference {Boolean} Conferencing status
 * &nbsp;&nbsp;isConnected {Boolean}Connection status
 * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
 * &nbsp;&nbsp;isHeld {Boolean} Held Status
 * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
 * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
 * &nbsp;&nbsp;isActive {Boolean} Active Status
 * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
 * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
 * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
 * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
 * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
 * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
 * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
 * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
 * &nbsp;&nbsp;displayName {Boolean} Display Name
 * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
 * }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>  
 *
 * The parameter phoneNumber has the following structure:
 * <pre>
 * {String} sip phone number URI
 * </pre>  
 *
 * The parameter callDirection has the following structure:
 * <pre>
 * {String} call direction from the values,possilbe values are:
 * -"CALL_INCOMING" for incoming call
 * -"CALL_OUTGOING" for outgoing call
 * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
 * </pre>  
 *
 * The parameter startReason and endReason have the following structure:
 * <pre>
 * {String} call reason from the values,possilbe values are:
 * -"CALL_REASON_NORMAL" for normal call
 * -"CALL_REASON_TRANSFERRED" for transferred call
 * </pre> 
 */

/**
 * @event CallEndedEvent 
 * This event is used to notify that a phone call (incoming or outgoing) is finished. This happens when a call is released by local or by distant.
 * In the case of distant release, this event happens when the hook-on invite is started.
 * @param {Object} param contains phoneCall which contains all the information related to this phone call instance,following the structure:
 * <pre>
 * {
 * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
 * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
 * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
 * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
 * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
 * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
 * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
 * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
 * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
 * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
 * &nbsp;&nbsp;startTime {Date} Start time
 * &nbsp;&nbsp;isConference {Boolean} Conferencing status
 * &nbsp;&nbsp;isConnected {Boolean}Connection status
 * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
 * &nbsp;&nbsp;isHeld {Boolean} Held Status
 * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
 * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
 * &nbsp;&nbsp;isActive {Boolean} Active Status
 * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
 * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
 * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
 * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
 * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
 * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
 * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
 * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
 * &nbsp;&nbsp;displayName {Boolean} Display Name
 * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
 * }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>  
 *
 * The parameter phoneNumber has the following structure:
 * <pre>
 * {String} sip phone number URI
 * </pre>  
 *
 * The parameter callDirection has the following structure:
 * <pre>
 * {String} call direction from the values,possilbe values are:
 * -"CALL_INCOMING" for incoming call
 * -"CALL_OUTGOING" for outgoing call
 * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
 * </pre>  
 *
 * The parameter startReason and endReason have the following structure:
 * <pre>
 * {String} call reason from the values,possilbe values are:
 * -"CALL_REASON_NORMAL" for normal call
 * -"CALL_REASON_TRANSFERRED" for transferred call
 * </pre> 
 */
 
 
/**
 * @event  CallIgnoredEvent
 * This event is used to notify that the call has been ignored
 * @param {Object} param contains the following structure:
 * <pre>
 * {    
 * &nbsp;&nbsp;callId {Object} telephony level call unique identifier.
 * &nbsp;&nbsp;phoneCall {Object} contains all the information related to this phone call instance.
 }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>
 *
 * The parameter phoneCall has the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
 * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
 * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
 * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
 * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
 * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
 * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
 * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
 * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
 * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
 * &nbsp;&nbsp;startTime {Date} Start time
 * &nbsp;&nbsp;isConference {Boolean} Conferencing status
 * &nbsp;&nbsp;isConnected {Boolean}Connection status
 * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
 * &nbsp;&nbsp;isHeld {Boolean} Held Status
 * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
 * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
 * &nbsp;&nbsp;isActive {Boolean} Active Status
 * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
 * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
 * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
 * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
 * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
 * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
 * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
 * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
 * &nbsp;&nbsp;displayName {Boolean} Display Name
 * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
 * }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>  
 *
 * The parameter phoneNumber has the following structure:
 * <pre>
 * {String} sip phone number URI
 * </pre>  
 *
 * The parameter callDirection has the following structure:
 * <pre>
 * {String} call direction from the values,possilbe values are:
 * -"CALL_INCOMING" for incoming call
 * -"CALL_OUTGOING" for outgoing call
 * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
 * </pre>  
 *
 * The parameter startReason and endReason have the following structure:
 * <pre>
 * {String} call reason from the values,possilbe values are:
 * -"CALL_REASON_NORMAL" for normal call
 * -"CALL_REASON_TRANSFERRED" for transferred call
 * </pre> 
 */
 
 /**
 * @event CallStartedEvent 
 * This event is used to notify that a phone call (incoming or outgoing) is starting to exist. This happens before the distant alerting or local ringing.
 * @param {Object} param contains phoneCall which contains all the information related to this phone call instance,following the structure:
 * <pre>
 * {
 * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
 * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
 * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
 * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
 * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
 * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
 * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
 * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
 * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
 * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
 * &nbsp;&nbsp;startTime {Date} Start time
 * &nbsp;&nbsp;isConference {Boolean} Conferencing status
 * &nbsp;&nbsp;isConnected {Boolean}Connection status
 * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
 * &nbsp;&nbsp;isHeld {Boolean} Held Status
 * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
 * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
 * &nbsp;&nbsp;isActive {Boolean} Active Status
 * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
 * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
 * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
 * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
 * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
 * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
 * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
 * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
 * &nbsp;&nbsp;displayName {Boolean} Display Name
 * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
 * }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>  
 *
 * The parameter phoneNumber has the following structure:
 * <pre>
 * {String} sip phone number URI
 * </pre>  
 *
 * The parameter callDirection has the following structure:
 * <pre>
 * {String} call direction from the values,possilbe values are:
 * -"CALL_INCOMING" for incoming call
 * -"CALL_OUTGOING" for outgoing call
 * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
 * </pre>  
 *
 * The parameter startReason and endReason have the following structure:
 * <pre>
 * {String} call reason from the values,possilbe values are:
 * -"CALL_REASON_NORMAL" for normal call
 * -"CALL_REASON_TRANSFERRED" for transferred call
 * </pre> 
 */


/**
 * @event  CallTransferedEvent  
 * This event is used to  notify that a the callee (1st phone call) has been transferred to another callee (2nd) in case of a consult transfer, or to a phone number in case of a blind transfer
 * @param {Object} param contains the following structure:
 * <pre>
 * {    
 * &nbsp;&nbsp;callId1 {Object} telephony level call unique identifier of the phone call to be transferred (1st).
 * &nbsp;&nbsp;phoneCall1 {Object} contains all the information related to 1st phone call.
 * &nbsp;&nbsp;callId2 {Object} telephony level call unique identifier of the phone call to be transferred to (2nd) if it is a consult transfer.
 * &nbsp;&nbsp;phoneCall2 {Object} contains all the information related to 2nd phone call.
 }
 * </pre>
 *
 * The parameter callId1 and callId2 have the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>
 *
 * The parameter phoneCall1 and phoneCall2 have the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
 * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
 * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
 * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
 * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
 * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
 * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
 * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
 * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
 * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
 * &nbsp;&nbsp;startTime {Date} Start time
 * &nbsp;&nbsp;isConference {Boolean} Conferencing status
 * &nbsp;&nbsp;isConnected {Boolean}Connection status
 * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
 * &nbsp;&nbsp;isHeld {Boolean} Held Status
 * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
 * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
 * &nbsp;&nbsp;isActive {Boolean} Active Status
 * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
 * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
 * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
 * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
 * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
 * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
 * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
 * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
 * &nbsp;&nbsp;displayName {Boolean} Display Name
 * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
 * }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>  
 *
 * The parameter phoneNumber has the following structure:
 * <pre>
 * {String} sip phone number URI
 * </pre>  
 *
 * The parameter callDirection has the following structure:
 * <pre>
 * {String} call direction from the values,possilbe values are:
 * -"CALL_INCOMING" for incoming call
 * -"CALL_OUTGOING" for outgoing call
 * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
 * </pre>  
 *
 * The parameter startReason and endReason have the following structure:
 * <pre>
 * {String} call reason from the values,possilbe values are:
 * -"CALL_REASON_NORMAL" for normal call
 * -"CALL_REASON_TRANSFERRED" for transferred call
 * </pre> 
 */     


/**
 * @event ConferenceStartedEvent  
 * This event is used to notify that a conference started
 * @param {Object} param contains the following structure:
 * <pre>
 * {     
 * &nbsp;&nbsp;callId1 {Object} telephony level call unique identifier of the conference 1st callee.
 * &nbsp;&nbsp;phoneCall1 {Object} contains all the information related to 1st phone call.
 * &nbsp;&nbsp;callId2 {Object} telephony level call unique identifier of the conference 2nd callee.
 * &nbsp;&nbsp;phoneCall2 {Object} contains all the information related to 2nd phone call.
 }
 * </pre>
 *
 * The parameter callId1 and callId2 have the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>
 *
 * The parameter phoneCall1 and phoneCall2 have the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
 * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
 * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
 * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
 * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
 * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
 * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
 * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
 * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
 * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
 * &nbsp;&nbsp;startTime {Date} Start time
 * &nbsp;&nbsp;isConference {Boolean} Conferencing status
 * &nbsp;&nbsp;isConnected {Boolean}Connection status
 * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
 * &nbsp;&nbsp;isHeld {Boolean} Held Status
 * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
 * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
 * &nbsp;&nbsp;isActive {Boolean} Active Status
 * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
 * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
 * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
 * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
 * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
 * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
 * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
 * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
 * &nbsp;&nbsp;displayName {Boolean} Display Name
 * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
 * }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>  
 *
 * The parameter phoneNumber has the following structure:
 * <pre>
 * {String} sip phone number URI
 * </pre>  
 *
 * The parameter callDirection has the following structure:
 * <pre>
 * {String} call direction from the values,possilbe values are:
 * -"CALL_INCOMING" for incoming call
 * -"CALL_OUTGOING" for outgoing call
 * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
 * </pre>  
 *
 * The parameter startReason and endReason have the following structure:
 * <pre>
 * {String} call reason from the values,possilbe values are:
 * -"CALL_REASON_NORMAL" for normal call
 * -"CALL_REASON_TRANSFERRED" for transferred call
 * </pre> 
 */     


/**
 * @event  ConferenceEndedEvent  
 * This event is used to notify that a conference ended
 * @param {Object} param contains the following structure:
 * <pre>
 * {    
 * &nbsp;&nbsp;callId1 {Object} telephony level call unique identifier of the phone call to be transferred (1st).
 * &nbsp;&nbsp;phoneCall1 {Object} contains all the information related to 1st phone call.
 * &nbsp;&nbsp;callId2 {Object} telephony level call unique identifier of the phone call to be transferred to (2nd) if it is a consult transfer.
 * &nbsp;&nbsp;phoneCall2 {Object} contains all the information related to 2nd phone call.
 }
 * </pre>
 *
 * The parameter callId1 and callId2 have the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>
 *
 * The parameter phoneCall1 and phoneCall2 have the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
 * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
 * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
 * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
 * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
 * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
 * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
 * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
 * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
 * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
 * &nbsp;&nbsp;startTime {Date} Start time
 * &nbsp;&nbsp;isConference {Boolean} Conferencing status
 * &nbsp;&nbsp;isConnected {Boolean}Connection status
 * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
 * &nbsp;&nbsp;isHeld {Boolean} Held Status
 * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
 * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
 * &nbsp;&nbsp;isActive {Boolean} Active Status
 * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
 * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
 * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
 * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
 * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
 * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
 * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
 * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
 * &nbsp;&nbsp;displayName {Boolean} Display Name
 * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
 * }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>  
 *
 * The parameter phoneNumber has the following structure:
 * <pre>
 * {String} sip phone number URI
 * </pre>  
 *
 * The parameter callDirection has the following structure:
 * <pre>
 * {String} call direction from the values,possilbe values are:
 * -"CALL_INCOMING" for incoming call
 * -"CALL_OUTGOING" for outgoing call
 * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
 * </pre>  
 *
 * The parameter startReason and endReason have the following structure:
 * <pre>
 * {String} call reason from the values,possilbe values are:
 * -"CALL_REASON_NORMAL" for normal call
 * -"CALL_REASON_TRANSFERRED" for transferred call
 * </pre> 
 */     

/**
 * @event  OutgoingCallAlertingEvent
 * This event is used to notify that an outgoing phone call has reached the callee. This happens at the beginning the distant alerting
 * @param {Object} param contains the following structure:
 * <pre>
 * {    
 * &nbsp;&nbsp;callId {Object} telephony level call unique identifier.
 * &nbsp;&nbsp;phoneCall {Object} contains all the information related to this phone call instance.
 }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>
 *
 * The parameter phoneCall has the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
 * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
 * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
 * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
 * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
 * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
 * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
 * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
 * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
 * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
 * &nbsp;&nbsp;startTime {Date} Start time
 * &nbsp;&nbsp;isConference {Boolean} Conferencing status
 * &nbsp;&nbsp;isConnected {Boolean}Connection status
 * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
 * &nbsp;&nbsp;isHeld {Boolean} Held Status
 * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
 * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
 * &nbsp;&nbsp;isActive {Boolean} Active Status
 * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
 * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
 * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
 * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
 * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
 * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
 * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
 * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
 * &nbsp;&nbsp;displayName {Boolean} Display Name
 * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
 * }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>  
 *
 * The parameter phoneNumber has the following structure:
 * <pre>
 * {String} sip phone number URI
 * </pre>  
 *
 * The parameter callDirection has the following structure:
 * <pre>
 * {String} call direction from the values,possilbe values are:
 * -"CALL_INCOMING" for incoming call
 * -"CALL_OUTGOING" for outgoing call
 * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
 * </pre>  
 *
 * The parameter startReason and endReason have the following structure:
 * <pre>
 * {String} call reason from the values,possilbe values are:
 * -"CALL_REASON_NORMAL" for normal call
 * -"CALL_REASON_TRANSFERRED" for transferred call
 * </pre> 
 */

/**
 * @event OutgoingCallFailedEvent
 * This event is used to notify that an outgoing phone call is failed. 
 * @param {Object} param contains the following structure:
 * <pre>
 * {    
 * &nbsp;&nbsp;callId {Object} telephony level call unique identifier.
 * &nbsp;&nbsp;phoneCall {Object} contains all the information related to this phone call instance.
 * &nbsp;&nbsp;callFailedReason {Number} 
 }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>
 *
 * The parameter phoneCall has the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
 * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
 * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
 * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
 * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
 * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
 * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
 * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
 * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
 * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
 * &nbsp;&nbsp;startTime {Date} Start time
 * &nbsp;&nbsp;isConference {Boolean} Conferencing status
 * &nbsp;&nbsp;isConnected {Boolean}Connection status
 * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
 * &nbsp;&nbsp;isHeld {Boolean} Held Status
 * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
 * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
 * &nbsp;&nbsp;isActive {Boolean} Active Status
 * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
 * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
 * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
 * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
 * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
 * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
 * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
 * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
 * &nbsp;&nbsp;displayName {Boolean} Display Name
 * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
 * }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>  
 *
 * The parameter phoneNumber has the following structure:
 * <pre>
 * {String} sip phone number URI
 * </pre>  
 *
 * The parameter callDirection has the following structure:
 * <pre>
 * {String} call direction from the values,possilbe values are:
 * -"CALL_INCOMING" for incoming call
 * -"CALL_OUTGOING" for outgoing call
 * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
 * </pre>  
 *
 * The parameter startReason and endReason have the following structure:
 * <pre>
 * {String} call reason from the values,possilbe values are:
 * -"CALL_REASON_NORMAL" for normal call
 * -"CALL_REASON_TRANSFERRED" for transferred call
 * </pre> 
 */ 
 


/**
 * @event OutgoingCallInitiatedEvent
 * This event is used to notify that an outgoing phone call is initiated. 
 */ 
 
 /**
 * @event OutgoingCallCancelledEvent
 * This event is used to notify that an outgoing phone call is cancelled. 
 */ 
 
 /**
 * @event GuestCheckIn 
 * This event is used to notify that the guest's status is changed.
 * @param {Boolean} param contains the status,checked in or checked out.
 */ 
 
 
 /**
 * @event GuestInfo 
 * This event is used to notify the guest's information.
 * @param {Object} param contains the following structure:
 * <pre>
 * {    
 * &nbsp;&nbsp;_guestName {String} 
 * &nbsp;&nbsp;_guestNumber {String} 
 * &nbsp;&nbsp;_setNumber {String} 
 * }
 * </pre> 
 */
 
 /**
 * @event GuestLanguage
 * This event is used to notify that the guest's language.
 * @param {String} param contains the language.
 */
 
 
 
});

ICTouchAPI.APIServices.Telephony = new ICTouchAPI.APIServices.Telephony();
