/**
 * @class ICTouchAPI.APIServices.APIServicesBase
 * @abstract
 * Abstract Class for all API Services - Do not use it directly
 *
 * Create API Interface Service.
 *
 * ICTGate services Path is HTTP://localhost:80/services/ : absolute HTTP://localhost:80/ : relative:services/
 * ICTouchAPI.APIServices=new ICTouchAPI.APIServices("HTTP://localhost:80/","services/");
 */
dojo.provide("ICTouchAPI.APIServices.APIServicesBase");
dojo.declare("ICTouchAPI.APIServices.APIServicesBase",null,
	{
	//defined in ICTouchAPI.main function
	_absolutePath:ICTouchAPI.applicationsCallURI,
	_relativePath:"services/",

	/**
	* @ignore
	*/
	constructor:function(){
		this._absolutePath = ICTouchAPI.applicationsCallURI;
	},

	/**
	 * Translates a javascript request into a http request for the ICTGate code application
	 * @param {String} domainName : called service/module (i.e. : telephony)
	 * @param {String} methodName : called method in service/module (i.e : makeCall)
	 * @param {Function} callback   : function to be called when response occurs
	 * @param {Array} params[]   : json required parameters for the methodName
	 * @returns response (must be json) is sent to callback function
	 */
	callCoreApplicationMethod : function(args){
		var url = this._absolutePath + this._relativePath + args.domainName + "/" + args.methodName;

		var json = {};
		if (args.params) {
			for (var i in args.params) {
				var subObj = {};
				if(typeof(args.paramType[i])=="object"){
					subObj['type'] = args.paramType[i].type;
					subObj['format'] = args.paramType[i].format;
					subObj['value'] = args.params[i];

				}else if(args.paramType[i] == 'QStringList'){
					subObj['type'] = "QString";
					subObj['format'] = 'list';
					subObj['value'] = args.params[i];
				} else {
					subObj['type'] = args.paramType[i];
					subObj['value'] = args.params[i];
				}

				json['json'+i] = dojo.toJson(subObj);
			}
		}

		dojo.xhrGet({
			url: url,
			handleAs: 'json',
			content: json,
			sync : (args.sync ? args.sync : false),
			load: function(data,ioArgs){
				var json = "";
				if (data && data.error == "false") {
					if (args.context && args.callback && typeof args.callback == "function") {
						args.context.__tempMethod__ = dojo.hitch(args.context, args.callback);
						json = data ? data.value : data;
						if (data && data.error == "true") {
							console.error("Error from business call: " + data.value);
							console.error("URL: " + ioArgs.url);
							if (args.errorCallback && typeof args.errorCallback == "function")
								args.context.__tempMethod__ = dojo.hitch(args.context, args.errorCallback);
							else
							{
								// clear the JSon
								json = {};
							}
						}

						// always call the callback
						args.context.__tempMethod__(json, args.callbackParams);
						delete args.context.__tempMethod__;
					}
				}
				else {
					json = data ? data.value : data;
					var func;
					if (args.context && args.errorCallback && typeof args.errorCallback == "function") {
						func = dojo.hitch(args.context, args.errorCallback);
						func(json, args.callbackParams);
					}
					else if (args.context && args.callback && typeof args.callback == "function") {
						func = args.context.__tempMethod__ = dojo.hitch(args.context, args.callback);
						func({}, args.callbackParams);
					}
				}
				args = null;
				data = null;
				ioArgs = null;
			},

			error: function(error, ioArgs){
				if (args.context && args.errorCallback && typeof args.errorCallback == "function") {
					var func = dojo.hitch(args.context, args.errorCallback);
					func({}, args.callbackParams);
				}
				// Better method for giving good errors display to developers

				if( error.stack )
					console.error(error.stack);
				else
					console.error(error);

				args = null;
				error = null;
				ioArgs = null;

			}
		});
	},

	/**
	 * @ignore
	 * Call this method with old function name and new one, it will execute it and console warn a message
	 * saying the old one is not to use anymore.
	 */
	_deprecated : function (oldName, newFunc, args) {
		console.warn(this.declaredClass + "." + oldName + " is deprecated, please use " + newFunc + " instead.");
		return this[newFunc].apply(this, args);
	}
});

//ICTouchAPI.APIServices=new ICTouchAPI.APIServices();
/**
 * @class ICTouchAPI.APIServices.Agenda
 * @extends Object
 * @ignore
 * In general there are 3 different operations that let the user to program, modify and remove a wake-up/appointment. By default the Agenda supports only one wake-up
 * and multiple appointments. The wake-up is reused every time, so it is not useful to remove it but just to modify it.
 * <br><br>A wake-up could be programmed locally, e.g. the Agenda manages it by itself. When the time for the wake-up comes, there will be a pop-up windows shown on the screen
 * with 2 or 3 buttons. These buttons let the user to manage the wake-up: STOP, REPEAT and SNOOZE. STOP stops the alarm immediately, REPEAT repeats the alarm after some time
 * interval defined by the user. SNOOZE lets the music used for a wake-up continue to play but removes the pop-up. The SNOOZE will appear only if the media source used for the
 * wake-up is a link, e.g. not a local file.
 * <br><br>A wake-up can be also a distant. In this case, Agenda is used just to contact the server delivering the telephony services. In our case the server should be an OXE. After the
 * request has been processed by OXE, Agenda do not controls anymore the wake-up. When the wake-up time comes, it is a special type of ring that is made. In this case Agenda
 * has just to update its state: turn off the wake-up led and remove the connectivity icon.
 * <br><br>Appointments are programmed always locally, e.g. managed by Agenda. Appointments can be programmed by the user himself or delivered by the Device Manager (programmed by an
 * administrator). For each appointment there is an associated notification. This notification occurs N minutes before the appointment, if there was enough time when the
 * appointment has been programmed. The appointment can be with or without alarm. When it is programmed with alarm, a pop-up will appear on the screen with 2 buttons: STOP and
 * REPEAT. STOP stops the appointments alarm and REPEAT, repeats it after some time interval defined by the user.
 */


dojo.provide("ICTouchAPI.APIServices.Agenda");
dojo.declare("ICTouchAPI.APIServices.Agenda",[ICTouchAPI.APIServices.APIServicesBase],
{
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>list of planned events</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Important notice : in case of repetitive appointments, only the first occurrence of the event is returned.</b>
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the type of planned events to retrieve.
     * Possible values are :
     * 
     * <pre>
     * - "WAKEUP" for WakeUp instances
     * - "APPOINTMENT" for Appointment instances
     * </pre>
     *
     * The callback will be called with the list of planned events, each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;id {Number} event identifier
     * &nbsp;&nbsp;type {String} "TYPE_WAKEUP" or "TYPE_APPOINTMENT"
     * &nbsp;&nbsp;label {String} Short description of the planned event
     * &nbsp;&nbsp;beginDate {Date} Begin date of the event. NB: For a WakeUp, time is only useful. Date is not taken into account
     * &nbsp;&nbsp;endDate {Date}  End date of the event. NB: For a WakeUp, time is only useful. Date is not taken into account
     * }
     * </pre>
     */
    
    findPlannedEventByType : function(args) {
        this.callCoreApplicationMethod({
            domainName: "Agenda",
            methodName: "findPlannedEventByType",
            context: args.context,
            params: args.params,
            paramType: ["QString"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>the Appointment instance</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b></b>
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the identifier of the appointment to retrieve.
     *
     * <pre>
     * {Number} the identifier of the appointment
     * </pre>
     * 
     * The callback will be called with the appointment,following this structure:
     *
     * <pre>
     * {    
     * &nbsp;&nbsp;description {String} free zone to describe the appointment.
     * &nbsp;&nbsp;place {String} The location of the appointment
     * &nbsp;&nbsp;alarmReminderDelay {Number} Delay in between the alarm and the beginning of the appointment, in minutes. Default value is 0, this value means that the alarm will start exactly at the programmed time.
     * &nbsp;&nbsp;repetitionEndDate {Date} End date for a repetitive event (inclusive). By default is not a valid date. In this case alarm will occur infinitely (daily, weekly, monthly).
     * &nbsp;&nbsp;puid {String} Identifier of the Appointment on PC side (for synchronization purpose) NB: the ICTOUCH itself doesn't initialize this field.
     * &nbsp;&nbsp;lastModificationDate {Date} Date of the last modification of the Appointment (for synchronization purpose)
     * &nbsp;&nbsp;readOnly {Boolean} boolean to check if the user is able to modify the Appointment(for synchronization purpose : this attribute is set to true when the Agenda synchronizes complex repetitive tasks.
     * &nbsp;&nbsp;                   This avoids difficult merges on PC side when the database is exported).
     * &nbsp;&nbsp;isDMAppointment {Boolean} boolean to check if the Appointment has been filled by the DM
     * &nbsp;&nbsp;alarm {JSON} associated alarm to the planned event.See the description below
     * &nbsp;&nbsp;notificationIdentifier {Number} identifier to the associated notification to the planned event.
     * &nbsp;&nbsp;repetitionPeriod {String} "OCCUR_ONE_TIME" or "OCCUR_DAILY" or "OCCUR_WEEKLY" or "OCCUR_MONTHLY"
     * &nbsp;&nbsp;id {Number} event identifier
     * &nbsp;&nbsp;type {String} "TYPE_WAKEUP" or "TYPE_APPOINTMENT"
     * &nbsp;&nbsp;label {String} Short description of the planned event
     * &nbsp;&nbsp;beginDate {Date} Begin date of the event. NB: For a WakeUp, time is only useful. Date is not taken into account
     * &nbsp;&nbsp;endDate {Date}  End date of the event. NB: For a WakeUp, time is only useful. Date is not taken into account
     * }
     * </pre>
     *
     * The parameter alarm has the following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;param id {Number}  Alarm identifier
     * &nbsp;&nbsp;idFile {Number} File identifier used to define the audio file used for the present alarm.
     * &nbsp;&nbsp;volume {Number}  volume level for the alarm's music
     * &nbsp;&nbsp;idCapabilities {Number} identifier of the associated capabilities of the present alarm.
     * }
     * </pre>
     */
    getAppointmentById : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Agenda",
            methodName : "getAppointmentById",
            context : args.context,
            params: args.params,
            paramType: ["int"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>WakeUp instance</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b></b>
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the identifier of the wake-up to retrieve.
     *
     * <pre>
     * {Number} the identifier of wakeup
     * </pre>
     *
     * The callback will be called with the wake-up, following this structure:
     *
     * <pre>
     * {    
     * &nbsp;&nbsp;alarm {JSON} associated alarm to the planned event.See the description below
     * &nbsp;&nbsp;notificationIdentifier {Number} identifier to the associated notification to the planned event.
     * &nbsp;&nbsp;repetitionPeriod {String} "OCCUR_ONE_TIME" or "OCCUR_DAILY" or "OCCUR_WEEKLY" or "OCCUR_MONTHLY"
     * &nbsp;&nbsp;id {Number} event identifier
     * &nbsp;&nbsp;type {String} "TYPE_WAKEUP" or "TYPE_APPOINTMENT"
     * &nbsp;&nbsp;label {String} Short description of the planned event
     * &nbsp;&nbsp;beginDate {Date} Begin date of the event. NB: For a WakeUp, time is only useful. Date is not taken into account
     * &nbsp;&nbsp;endDate {Date}  End date of the event. NB: For a WakeUp, time is only useful. Date is not taken into account
     * }
     * </pre>
     *
     * The parameter alarm has the following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;param id {Number}  Alarm identifier
     * &nbsp;&nbsp;idFile {Number} File identifier used to define the audio file used for the present alarm.
     * &nbsp;&nbsp;volume {Number}  volume level for the alarm's music
     * &nbsp;&nbsp;idCapabilities {Number} identifier of the associated capabilities of the present alarm.
     * }
     * </pre>
     */
	getWakeUpById : function(args) {
		this.callCoreApplicationMethod({
			domainName : "Agenda",
			methodName : "getWakeUpById",
			context : args.context,
			params: args.params,
			paramType: ["int"],
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>Appointment unique identifier</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method will save the appointment in the database.Then, a notification will be programmed via the scheduler,which will call notificationTriggered() on notification trigger.
     * <br>Finally, if the alarm is ON, a local alarm will be programmed via the Scheduler module, which will call localAlarmTriggered() on alarm trigger.</b>
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the event to schedule,it has the following structure:
     * <pre>
     * {    
     * &nbsp;&nbsp;description {String} free zone to describe the appointment.
     * &nbsp;&nbsp;place {String} The location of the appointment
     * &nbsp;&nbsp;alarmReminderDelay {Number} Delay in between the alarm and the beginning of the appointment, in minutes. Default value is 0, this value means that the alarm will start exactly at the programmed time.
     * &nbsp;&nbsp;repetitionEndDate {Date} End date for a repetitive event (inclusive). By default is not a valid date. In this case alarm will occur infinitely (daily, weekly, monthly).
     * &nbsp;&nbsp;puid {String} Identifier of the Appointment on PC side (for synchronization purpose) NB: the ICTOUCH itself doesn't initialize this field.
     * &nbsp;&nbsp;lastModificationDate {Date} Date of the last modification of the Appointment (for synchronization purpose)
     * &nbsp;&nbsp;readOnly {Boolean} boolean to check if the user is able to modify the Appointment(for synchronization purpose : this attribute is set to true when the Agenda synchronizes complex repetitive tasks. 
     * &nbsp;&nbsp;                   This avoids difficult merges on PC side when the database is exported).
     * &nbsp;&nbsp;isDMAppointment {Boolean} boolean to check if the Appointment has been filled by the DM
     * &nbsp;&nbsp;alarm {JSON} associated alarm to the planned event.See the description below
     * &nbsp;&nbsp;notificationIdentifier {Number} identifier to the associated notification to the planned event.
     * &nbsp;&nbsp;repetitionPeriod {String} "OCCUR_ONE_TIME" or "OCCUR_DAILY" or "OCCUR_WEEKLY" or "OCCUR_MONTHLY"
     * &nbsp;&nbsp;id {Number} event identifier
     * &nbsp;&nbsp;type {String} "TYPE_WAKEUP" or "TYPE_APPOINTMENT"
     * &nbsp;&nbsp;label {String} Short description of the planned event
     * &nbsp;&nbsp;beginDate {Date} Begin date of the event. NB: For a WakeUp, time is only useful. Date is not taken into account
     * &nbsp;&nbsp;endDate {Date}  End date of the event. NB: For a WakeUp, time is only useful. Date is not taken into account
     * }
     * </pre>
     *
     * The parameter alarm has the following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;param id {Number}  Alarm identifier
     * &nbsp;&nbsp;idFile {Number} File identifier used to define the audio file used for the present alarm.
     * &nbsp;&nbsp;volume {Number}  volume level for the alarm's music
     * &nbsp;&nbsp;idCapabilities {Number} identifier of the associated capabilities of the present alarm.
     * }
     * </pre>
     *
     * The callback will be called with the Appointment unique identifier
     * <pre>
     * {Number} the Appointment unique identifier
     * </pre>
     */
    addPlannedAppointment : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Agenda",
            methodName : "addPlannedEvent",
            context : args.context,
            params: args.params,
            paramType: ["Appointment"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>Wakeup unique identifier</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br>This method will save the wakeUp in the database.</b>
     * <br>Then, if the alarm is ON :
     * <br>-in a local context, the alarm will be programmed via the Scheduler module,which will call localAlarmTriggered() on alarm trigger;
     * <br>-in a distant context, the alarm will be scheduled via SIPUA. The alarm itself is managed like a automatic incoming call made by the PABX.
     *
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the event to schedule,it has the following structure:
     * <pre>
     * {    
     * &nbsp;&nbsp;alarm {JSON} associated alarm to the planned event.See the description below
     * &nbsp;&nbsp;notificationIdentifier {Number} identifier to the associated notification to the planned event.
     * &nbsp;&nbsp;repetitionPeriod {String} "OCCUR_ONE_TIME" or "OCCUR_DAILY" or "OCCUR_WEEKLY" or "OCCUR_MONTHLY"
     * &nbsp;&nbsp;id {Number} event identifier
     * &nbsp;&nbsp;type {String} "TYPE_WAKEUP" or "TYPE_APPOINTMENT"
     * &nbsp;&nbsp;label {String} Short description of the planned event
     * &nbsp;&nbsp;beginDate {Date} Begin date of the event. NB: For a WakeUp, time is only useful. Date is not taken into account
     * &nbsp;&nbsp;endDate {Date}  End date of the event. NB: For a WakeUp, time is only useful. Date is not taken into account
     * }
     * </pre>
     *
     * The parameter alarm has the following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;param id {Number}  Alarm identifier
     * &nbsp;&nbsp;idFile {Number} File identifier used to define the audio file used for the present alarm.
     * &nbsp;&nbsp;volume {Number}  volume level for the alarm's music
     * &nbsp;&nbsp;idCapabilities {Number} identifier of the associated capabilities of the present alarm.
     * }
     * </pre>  
     *
     * The callback will be called with the Wakeup unique identifier
     * <pre>
     * {Number} the Wakeup unique identifier
     * </pre>
     */
 
	addPlannedWakeUp : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Agenda",
            methodName : "addPlannedEvent",
            context :args.context,
            params: args.params,
            paramType: ["WakeUp"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Important notice: this method will update the appointment in the database.
     * If necessary, the notification will be reprogrammed via the scheduler, which will call notificationTriggered() on notification trigger.</b>
     * <br><br>If the alarm was deactivated (previously ON, now OFF),the associated task will be removed from the Scheduler.
     * <br><br>If the alarm was activated (previously OFF, now ON),the alarm will be programmed via the Scheduler module,which will call localAlarmTriggered() on alarm trigger;
     * <br><br>If the appointment time changed and the alarm stayed ON,the alarm will be reprogrammed via the Scheduler module,which will call localAlarmTriggered() on alarm trigger;
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the event to schedule,it has the following structure:
     * <pre>
     * {    
     * &nbsp;&nbsp;description {String} free zone to describe the appointment.
     * &nbsp;&nbsp;place {String} The location of the appointment
     * &nbsp;&nbsp;alarmReminderDelay {Number} Delay in between the alarm and the beginning of the appointment, in minutes. Default value is 0, this value means that the alarm will start exactly at the programmed time.
     * &nbsp;&nbsp;repetitionEndDate {Date} End date for a repetitive event (inclusive). By default is not a valid date. In this case alarm will occur infinitely (daily, weekly, monthly).
     * &nbsp;&nbsp;puid {String} Identifier of the Appointment on PC side (for synchronization purpose) NB: the ICTOUCH itself doesn't initialize this field.
     * &nbsp;&nbsp;lastModificationDate {Date} Date of the last modification of the Appointment (for synchronization purpose)
     * &nbsp;&nbsp;readOnly {Boolean} boolean to check if the user is able to modify the Appointment(for synchronization purpose : this attribute is set to true when the Agenda synchronizes complex repetitive tasks. 
     * &nbsp;&nbsp;                   This avoids difficult merges on PC side when the database is exported).
     * &nbsp;&nbsp;isDMAppointment {Boolean} boolean to check if the Appointment has been filled by the DM
     * &nbsp;&nbsp;alarm {JSON} associated alarm to the planned event.See the description below
     * &nbsp;&nbsp;notificationIdentifier {Number} identifier to the associated notification to the planned event.
     * &nbsp;&nbsp;repetitionPeriod {String} "OCCUR_ONE_TIME" or "OCCUR_DAILY" or "OCCUR_WEEKLY" or "OCCUR_MONTHLY"
     * &nbsp;&nbsp;id {Number} event identifier
     * &nbsp;&nbsp;type {String} "TYPE_WAKEUP" or "TYPE_APPOINTMENT"
     * &nbsp;&nbsp;label {String} Short description of the planned event
     * &nbsp;&nbsp;beginDate {Date} Begin date of the event. NB: For a WakeUp, time is only useful. Date is not taken into account
     * &nbsp;&nbsp;endDate {Date}  End date of the event. NB: For a WakeUp, time is only useful. Date is not taken into account
     * }
     * </pre>
     *
     * The parameter alarm has the following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;param id {Number}  Alarm identifier
     * &nbsp;&nbsp;idFile {Number} File identifier used to define the audio file used for the present alarm.
     * &nbsp;&nbsp;volume {Number}  volume level for the alarm's music
     * &nbsp;&nbsp;idCapabilities {Number} identifier of the associated capabilities of the present alarm.
     * }
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
	 */
    modifyPlannedAppointment : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Agenda",
            methodName : "modifyPlannedEvent",
            context : args.context,
            params: args.params,
            paramType: ["Appointment"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Important notice: this method will update the wakeup in the database.
     * <br>If necessary, the notification will be reprogrammed via the scheduler, which will call notificationTriggered() on notification trigger.</b>
     * <br><br>If the alarm was deactivated (previously ON, now OFF) :
     * <br>-in a local context, the alarm will be removed from the Scheduler module,
     * <br>-in a distant context, tje alarm will be cancelled by a call to SIPUA.
     * <br><br>If the alarm was activated (previously OFF, now ON), the alarm will be programmed via the Scheduler module or SIPUA, according to the local/distant context (see addPlannedEvent).
     *
     * <br><br>If the wakeUp time changed and the alarm stayed ON :
     * <br>-in a local context, the alarm will be reprogrammed via the Scheduler module,which will call localAlarmTriggered() on alarm trigger;
     * <br>-in a distant context, the alarm will be rescheduled via SIPUA.
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the event to schedule,it has the following structure:
     * <pre>
     * {    
     * &nbsp;&nbsp;alarm {JSON} associated alarm to the planned event.See the description below
     * &nbsp;&nbsp;notificationIdentifier {Number} identifier to the associated notification to the planned event.
     * &nbsp;&nbsp;repetitionPeriod {String} "OCCUR_ONE_TIME" or "OCCUR_DAILY" or "OCCUR_WEEKLY" or "OCCUR_MONTHLY"
     * &nbsp;&nbsp;id {Number} event identifier
     * &nbsp;&nbsp;type {String} "TYPE_WAKEUP" or "TYPE_APPOINTMENT"
     * &nbsp;&nbsp;label {String} Short description of the planned event
     * &nbsp;&nbsp;beginDate {Date} Begin date of the event. NB: For a WakeUp, time is only useful. Date is not taken into account
     * &nbsp;&nbsp;endDate {Date}  End date of the event. NB: For a WakeUp, time is only useful. Date is not taken into account
     * }
     * </pre>
     *
     * The parameter alarm has the following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;param id {Number}  Alarm identifier
     * &nbsp;&nbsp;idFile {Number} File identifier used to define the audio file used for the present alarm.
     * &nbsp;&nbsp;volume {Number}  volume level for the alarm's music
     * &nbsp;&nbsp;idCapabilities {Number} identifier of the associated capabilities of the present alarm.
     * }
     * </pre>  
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */
	modifyPlannedWakeUp : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Agenda",
            methodName : "modifyPlannedEvent",
            context : args.context,
            params: args.params,
            paramType: ["WakeUp"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Important notice : this method will remove the PlannedEvent from the database.</b>
     * <br><br>If the planned event is an instance of WakeUp,this method will remove the wakeUp from the database.
     * <br>If necessary, the alarm will be canceled :
     * <br>-in a local context, by removing the associated Scheduler task;
     * <br>-in a distant context, by calling SIPUA to remove it.
     * <br><br>If the planned event is an instance of Appointment,this method will remove the appointment from the database.
     * <br>The Scheduler task associated to the notification(s) will be deleted.If necessary, the alarm will also be canceled,by removing the associated task from the Scheduler.
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;plannedEventType {String}  the type of the PlannedEvent to delete
     * &nbsp;&nbsp;plannedEventId {Number} the identifier of the PlannedEvent to delete
     * }
     * </pre>
     *
     * params plannedEventType possible values are :     * 
     * <pre>
     * - "WAKEUP" for WakeUp instances
     * - "APPOINTMENT" for Appointment instances,
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */
    removePlannedEvent : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Agenda",
            methodName : "removePlannedEvent",
            context : args.context,
            params: args.params,
            paramType: ["QString","int"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Important notice : Stop the alarm definitively</b>
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the identifier of the alarm to stop.
     * 
     * <pre>
     *  {Number} for the identifier of the alarm
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */
    stopAlarm : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Agenda",
            methodName : "stopAlarm",
            context : args.context,
            params: args.params,
            paramType: ["int"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Important notice : stop the alarm and schedule the next ringing if necessary</b>
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the identifier of the alarm to repeat.
     * <pre>
     * {Number} for the identifier of the alarm
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */    
    repeatAlarm : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Agenda",
            methodName : "repeatAlarm",
            context : args.context,
            params: args.params,
            paramType: ["int"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },
    
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Important notice : Only available for a webradio : stop the alarm definitively but let the music play.</b>
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the identifier of the alarm to snooze.
     * 
     * <pre>
     * {Number} the identifier of the alarm
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */      
    snoozeAlarm : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Agenda",
            methodName : "snoozeAlarm",
            context : args.context,
            params: args.params,
            paramType: ["int"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>capability of the module</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the name of the capability
     * <pre>
     * {String} the name of the capability
     * </pre>
     *
     * The callback will be called with the capability, following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;name {String} 
     * &nbsp;&nbsp;state {Number} 
     * }
     * </pre>
     */

	getCapabilityGlobal : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Agenda",
			methodName: "getCapability",
			context: args.context,
			params: args.params,
			paramType: ["QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>capability of an object of the module</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;name {String}  Name of the capability
     * &nbsp;&nbsp;listId {Number} ID of the list of capabilities
     * }
     * </pre>
     *
     * The callback will be called with the capability, following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;name {String} 
     * &nbsp;&nbsp;state {Number} 
     * }
     * </pre>
     */

	getCapabilityInstance : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Agenda",
			methodName: "getCapability",
			context: args.context,
			params: args.params,
			paramType: ["int", "QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>the list of capability of an object</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the ID of the list of capabilities
     * <pre>
     * {Number} the ID of the list of capabilities
     * </pre>
     *
     * The callback will be called with a map of capabilities indexed by name, each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;name {String} 
     * &nbsp;&nbsp;state {Number} 
     * }
     * </pre>
     */

	getCapabilities : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Agenda",
			methodName: "getCapabilities",
			context: args.context,
			params: args.params,
			paramType: ["int"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	}
	
/**
 * @event AgendaContentChanged 
 * This event is used to notify that the Agenda's content has changed. Example: an appointment is added, modified or deleted.
 */	
	
/**
 * @event AlarmStarted 
 * This event is used to notify that a wake-up alarm is started.
 * @param {JSON} param contains a reference to the object wake-up which alarm has started. To check if the event is corresponding to the real alarm state of the wake-up, you can explore the value of the 'alarmState' variable of wakeUp, 
 * following the structure:
 * <pre>
 * {    
 * &nbsp;&nbsp;alarm {JSON} associated alarm to the planned event.See the description below
 * &nbsp;&nbsp;notificationIdentifier {Number} identifier to the associated notification to the planned event.
 * &nbsp;&nbsp;repetitionPeriod {String} "OCCUR_ONE_TIME" or "OCCUR_DAILY" or "OCCUR_WEEKLY" or "OCCUR_MONTHLY"
 * &nbsp;&nbsp;id {Number} event identifier
 * &nbsp;&nbsp;type {String} "TYPE_WAKEUP" or "TYPE_APPOINTMENT"
 * &nbsp;&nbsp;label {String} Short description of the planned event
 * &nbsp;&nbsp;beginDate {Date} Begin date of the event. NB: For a WakeUp, time is only useful. Date is not taken into account
 * &nbsp;&nbsp;endDate {Date}  End date of the event. NB: For a WakeUp, time is only useful. Date is not taken into account
 * }
 * </pre>
 *
 * The parameter alarm has the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp;param id {Number}  Alarm identifier
 * &nbsp;&nbsp;idFile {Number} File identifier used to define the audio file used for the present alarm.
 * &nbsp;&nbsp;volume {Number}  volume level for the alarm's music
 * &nbsp;&nbsp;idCapabilities {Number} identifier of the associated capabilities of the present alarm.
 * }
 * </pre>
 */
 
/**
 * @event AlarmStopped
 * This event is used to notify that a wake-up alarm is stopped.
 * @param {JSON} param contains a reference to the object wake-up which alarm has stopped. To check if the event is corresponding to the real alarm state of the wake-up, you can explore the value of the 'alarmState' variable of wakeUp,
 * following the structure:
 * <pre>
 * {    
 * &nbsp;&nbsp;alarm {JSON} associated alarm to the planned event.See the description below
 * &nbsp;&nbsp;notificationIdentifier {Number} identifier to the associated notification to the planned event.
 * &nbsp;&nbsp;repetitionPeriod {String} "OCCUR_ONE_TIME" or "OCCUR_DAILY" or "OCCUR_WEEKLY" or "OCCUR_MONTHLY"
 * &nbsp;&nbsp;id {Number} event identifier
 * &nbsp;&nbsp;type {String} "TYPE_WAKEUP" or "TYPE_APPOINTMENT"
 * &nbsp;&nbsp;label {String} Short description of the planned event
 * &nbsp;&nbsp;beginDate {Date} Begin date of the event. NB: For a WakeUp, time is only useful. Date is not taken into account
 * &nbsp;&nbsp;endDate {Date}  End date of the event. NB: For a WakeUp, time is only useful. Date is not taken into account
 * }
 * </pre>
 *
 * The parameter alarm has the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp;param id {Number}  Alarm identifier
 * &nbsp;&nbsp;idFile {Number} File identifier used to define the audio file used for the present alarm.
 * &nbsp;&nbsp;volume {Number}  volume level for the alarm's music
 * &nbsp;&nbsp;idCapabilities {Number} identifier of the associated capabilities of the present alarm.
 * }
 * </pre>
 */


/**
 * @event AlarmState
 * This event is used to notify that the wake-up alarm state has changed to ON or to OFF.
 * @param {JSON} param contains a reference to the object wake-up which alarm has changed its state.
 * To check what is the state of the alarm, you should access the internal variable of wakeUp - 'alarmState',
 * following the structure:
 * <pre>
 * {    
 * &nbsp;&nbsp;alarm {JSON} associated alarm to the planned event.See the description below
 * &nbsp;&nbsp;notificationIdentifier {Number} identifier to the associated notification to the planned event.
 * &nbsp;&nbsp;repetitionPeriod {String} "OCCUR_ONE_TIME" or "OCCUR_DAILY" or "OCCUR_WEEKLY" or "OCCUR_MONTHLY"
 * &nbsp;&nbsp;id {Number} event identifier
 * &nbsp;&nbsp;type {String} "TYPE_WAKEUP" or "TYPE_APPOINTMENT"
 * &nbsp;&nbsp;label {String} Short description of the planned event
 * &nbsp;&nbsp;beginDate {Date} Begin date of the event. NB: For a WakeUp, time is only useful. Date is not taken into account
 * &nbsp;&nbsp;endDate {Date}  End date of the event. NB: For a WakeUp, time is only useful. Date is not taken into account
 * }
 * </pre>
 *
 * The parameter alarm has the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp;param id {Number}  Alarm identifier
 * &nbsp;&nbsp;idFile {Number} File identifier used to define the audio file used for the present alarm.
 * &nbsp;&nbsp;volume {Number}  volume level for the alarm's music
 * &nbsp;&nbsp;idCapabilities {Number} identifier of the associated capabilities of the present alarm.
 * }
 * </pre>
 */
	
/**
 * @event  PlannedEventNotification
 * This event is used to send a notification to remind that an apppointment will start soon.
 * @param {JSON} param contains a reference to the object appointment that has been notified,
 * following the structure:
 * <pre>
 * {    
 * &nbsp;&nbsp;description {String} free zone to describe the appointment.
 * &nbsp;&nbsp;place {String} The location of the appointment
 * &nbsp;&nbsp;alarmReminderDelay {Number} Delay in between the alarm and the beginning of the appointment, in minutes. Default value is 0, this value means that the alarm will start exactly at the programmed time.
 * &nbsp;&nbsp;repetitionEndDate {Date} End date for a repetitive event (inclusive). By default is not a valid date. In this case alarm will occur infinitely (daily, weekly, monthly).
 * &nbsp;&nbsp;puid {String} Identifier of the Appointment on PC side (for synchronization purpose) NB: the ICTOUCH itself doesn't initialize this field.
 * &nbsp;&nbsp;lastModificationDate {Date} Date of the last modification of the Appointment (for synchronization purpose)
 * &nbsp;&nbsp;readOnly {Boolean} boolean to check if the user is able to modify the Appointment(for synchronization purpose : this attribute is set to true when the Agenda synchronizes complex repetitive tasks. 
 * &nbsp;&nbsp;                   This avoids difficult merges on PC side when the database is exported).
 * &nbsp;&nbsp;isDMAppointment {Boolean} boolean to check if the Appointment has been filled by the DM
 * &nbsp;&nbsp;alarm {JSON} associated alarm to the planned event.See the description below
 * &nbsp;&nbsp;notificationIdentifier {Number} identifier to the associated notification to the planned event.
 * &nbsp;&nbsp;repetitionPeriod {String} "OCCUR_ONE_TIME" or "OCCUR_DAILY" or "OCCUR_WEEKLY" or "OCCUR_MONTHLY"
 * &nbsp;&nbsp;id {Number} event identifier
 * &nbsp;&nbsp;type {String} "TYPE_WAKEUP" or "TYPE_APPOINTMENT"
 * &nbsp;&nbsp;label {String} Short description of the planned event
 * &nbsp;&nbsp;beginDate {Date} Begin date of the event. NB: For a WakeUp, time is only useful. Date is not taken into account
 * &nbsp;&nbsp;endDate {Date}  End date of the event. NB: For a WakeUp, time is only useful. Date is not taken into account
 * }
 * </pre>
 *
 * The parameter alarm has the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp;param id {Number}  Alarm identifier
 * &nbsp;&nbsp;idFile {Number} File identifier used to define the audio file used for the present alarm.
 * &nbsp;&nbsp;volume {Number}  volume level for the alarm's music
 * &nbsp;&nbsp;idCapabilities {Number} identifier of the associated capabilities of the present alarm.
 * }
 * </pre>
 */
	
	
});

ICTouchAPI.APIServices.Agenda = new ICTouchAPI.APIServices.Agenda();
/**
 * @class ICTouchAPI.APIServices.Comlog
 * @extends ICTouchAPI.APIServices.APIServicesBase
 * @singleton
 * The responsibilities of this component are :<br><br>
 * &nbsp;- Collect the information from all applications which need to be recorded. This is done
 *   by subscribing to the relevant events of these applications.<br>
 * &nbsp;- Ensure the persistence of the data by storing them in a database.<br>
 * &nbsp;- Give the list of the recordings to any other application. Several variants of the find
 *   method are available: with filter by log type, and/or by contact. The list can be sent
 *   totally or by pages.<br><br>
 * Note: the list is always sorted by date<br><br>
 * &nbsp;- Allow to mark events as read and to delete them.<br>
 * &nbsp;- Notify the ComLog presentation WebApp when the ComLog is changed.<br>
 * &nbsp;- Notify the top bar each time the count of unread logs of a given category is changed.<br>
 * &nbsp;- Delete oldest entries of the ComLog when the maximal size is reached.<br><br>
 * The ComLog application's role is to keep a track of the communication activities of the phone.
 * Each of these activities (phone call, incoming voicemail, Callback request, etc) is stored in
 * an internal database.
 */
dojo.provide("ICTouchAPI.APIServices.Comlog");
dojo.declare("ICTouchAPI.APIServices.Comlog",[ICTouchAPI.APIServices.APIServicesBase],
{
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>ComLogEntry entries</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Finds the entries of the ComLog matching the filter.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;{Number} the status
     * &nbsp;&nbsp;{Number} the type
     * }
     * </pre>
     *
     * The callback will be called with the list of planned events, each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; entryType {Object} Type of communication.
     * &nbsp;&nbsp; comLogEntryId {Number} Unique comlog entry identifier.
     * &nbsp;&nbsp; isRead {Boolean} Entry already (True) or not yet (False) read.
     * &nbsp;&nbsp; dateTime {Date} Date and time of the beginning of the logged event.
     * &nbsp;&nbsp; phoneNumber {String} Phone number of the correspondent. The string may be empty for the VM or if number is unknown.
     * &nbsp;&nbsp; comId {Number} Phone call identifier.
     * &nbsp;&nbsp; label {String} Name or phone number displayed in the ComLog summary.
     * &nbsp;&nbsp; contactId {String} Contact identifier of the correspondent if contact is defined, else correspondent's phone number.
     * &nbsp;&nbsp; endTime {Date} Date and time of the end of the logged event.
     * &nbsp;&nbsp; connectionTime {Date} Date and time of the beginning of the conversation.
     * &nbsp;&nbsp; isAnswered {Boolean} True if the call has been answered.
     * &nbsp;&nbsp; callDirection {Object} "Incoming" or "Outgoing".
     * &nbsp;&nbsp; startReason {Object} "normalCall" or "transferredCall".
     * &nbsp;&nbsp; endReason {Object} "normalCall" or "transferredCall".
     * &nbsp;&nbsp; countNew {Number} Number of new messages.
     * &nbsp;&nbsp; countOld {Number} Number of old messages.
     * &nbsp;&nbsp; com2Id {Number} Second phone call identifier.
     * &nbsp;&nbsp; wsRecordId {String} ICE comlog identifier.
     * &nbsp;&nbsp; idMail {Number} email identifier.
     * &nbsp;&nbsp; emailTitle {String} email subject.
     * &nbsp;&nbsp; conversationId {Number} Instant messaging identifier.
     * &nbsp;&nbsp; isWhiteList {Boolean} True if distant party belongs to correspondent's white list.
     * &nbsp;&nbsp; fromBossSecretary {Boolean} True if call routed to secretary phone in a manager/secretary context.
     * &nbsp;&nbsp; peerId {String} Identification of the peer in a manager/secretary context.
     * &nbsp;&nbsp; entryPriority {Object} comlog entry priority
     * }
     * </pre>
     *
     * Possible values of entryType are :
     *
     * <pre>
     * - "CallLog"          for Calllog
     * - "CallBackRequest"  for CR
     * - "VM"               for VoiceMail
     * - "Conference"       for Conference
     * - "IM"               for Instance Message
     * - "Email"            for Email
     * </pre>
     *
     * Possible value of callDirection are :
     *
     * <pre>
     * - "CALL_INCOMING"         for incoming call
     * - "CALL_OUTGOING"         for outgoing call
     * </pre>
     *
     * Possible value of startReason and endReason are :
     *
     * <pre>
     * - "CALL_REASON_NORMAL"       for a normal call
     * - "CALL_REASON_TRANSFERRED"  for a transferred call
     * </pre> 
     *   
     * Possible value of entryPriority are :
     *
     * <pre>
     * - "Low"      for low priority
     * - "Normal"   for normal priority
     * - "High"     for high priority
     * </pre>      
     */        
    findEntries : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Comlog",
            methodName : "findEntries",
            context : args.context,
            params: args.params,
            paramType: ["int", "int"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>ComLogEntry entries</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Finds the entries of the ComLog matching the filter for a given Contact.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;{String} the contact
     * &nbsp;&nbsp;{Number} the status
     * &nbsp;&nbsp;{Number} the type
     * }
     * </pre>
     *
     * The callback will be called with the list of planned events, each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; entryType {Object} Type of communication.
     * &nbsp;&nbsp; comLogEntryId {Number} Unique comlog entry identifier.
     * &nbsp;&nbsp; isRead {Boolean} Entry already (True) or not yet (False) read.
     * &nbsp;&nbsp; dateTime {Date} Date and time of the beginning of the logged event.
     * &nbsp;&nbsp; phoneNumber {String} Phone number of the correspondent. The string may be empty for the VM or if number is unknown.
     * &nbsp;&nbsp; comId {Number} Phone call identifier.
     * &nbsp;&nbsp; label {String} Name or phone number displayed in the ComLog summary.
     * &nbsp;&nbsp; contactId {String} Contact identifier of the correspondent if contact is defined, else correspondent's phone number.
     * &nbsp;&nbsp; endTime {Date} Date and time of the end of the logged event.
     * &nbsp;&nbsp; connectionTime {Date} Date and time of the beginning of the conversation.
     * &nbsp;&nbsp; isAnswered {Boolean} True if the call has been answered.
     * &nbsp;&nbsp; callDirection {Object} "Incoming" or "Outgoing".
     * &nbsp;&nbsp; startReason {Object} "normalCall" or "transferredCall".
     * &nbsp;&nbsp; endReason {Object} "normalCall" or "transferredCall".
     * &nbsp;&nbsp; countNew {Number} Number of new messages.
     * &nbsp;&nbsp; countOld {Number} Number of old messages.
     * &nbsp;&nbsp; com2Id {Number} Second phone call identifier.
     * &nbsp;&nbsp; wsRecordId {String} ICE comlog identifier.
     * &nbsp;&nbsp; idMail {Number} email identifier.
     * &nbsp;&nbsp; emailTitle {String} email subject.
     * &nbsp;&nbsp; conversationId {Number} Instant messaging identifier.
     * &nbsp;&nbsp; isWhiteList {Boolean} True if distant party belongs to correspondent's white list.
     * &nbsp;&nbsp; fromBossSecretary {Boolean} True if call routed to secretary phone in a manager/secretary context.
     * &nbsp;&nbsp; peerId {String} Identification of the peer in a manager/secretary context.
     * &nbsp;&nbsp; entryPriority {Object} comlog entry priority
     * }
     * </pre>
     *
     * Possible values of entryType are :
     *
     * <pre>
     * - "CallLog"          for Calllog
     * - "CallBackRequest"  for CR
     * - "VM"               for VoiceMail
     * - "Conference"       for Conference
     * - "IM"               for Instance Message
     * - "Email"            for Email
     * </pre>
     *
     * Possible value of callDirection are :
     *
     * <pre>
     * - "CALL_INCOMING"         for incoming call
     * - "CALL_OUTGOING"         for outgoing call
     * </pre>
     *
     * Possible value of startReason and endReason are :
     *
     * <pre>
     * - "CALL_REASON_NORMAL"       for a normal call
     * - "CALL_REASON_TRANSFERRED"  for a transferred call
     * </pre>
     *
     * Possible value of entryPriority are :
     *
     * <pre>
     * - "Low"      for low priority
     * - "Normal"   for normal priority
     * - "High"     for high priority
     * </pre>
     */
    findEntriesByContact : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Comlog",
            methodName : "findEntriesByContact",
            context : args.context,
            params: args.params,
            paramType: ["QString", "int", "int"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

	 /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>ComLogEntry entries</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Finds the entries of the ComLog matching the categories.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
	 * &nbsp;&nbsp;{QStringList} Categories Array of string, each entry is on this pattern: "categoriesId:filterReadId"
	 *
     * }
     * </pre>
     *
     * The callback will be called with the list of planned events, each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; entryType {Object} Type of communication.
     * &nbsp;&nbsp; comLogEntryId {Number} Unique comlog entry identifier.
     * &nbsp;&nbsp; isRead {Boolean} Entry already (True) or not yet (False) read.
     * &nbsp;&nbsp; dateTime {Date} Date and time of the beginning of the logged event.
     * &nbsp;&nbsp; phoneNumber {String} Phone number of the correspondent. The string may be empty for the VM or if number is unknown.
     * &nbsp;&nbsp; comId {Number} Phone call identifier.
     * &nbsp;&nbsp; label {String} Name or phone number displayed in the ComLog summary.
     * &nbsp;&nbsp; contactId {String} Contact identifier of the correspondent if contact is defined, else correspondent's phone number.
     * &nbsp;&nbsp; endTime {Date} Date and time of the end of the logged event.
     * &nbsp;&nbsp; connectionTime {Date} Date and time of the beginning of the conversation.
     * &nbsp;&nbsp; isAnswered {Boolean} True if the call has been answered.
     * &nbsp;&nbsp; callDirection {Object} "Incoming" or "Outgoing".
     * &nbsp;&nbsp; startReason {Object} "normalCall" or "transferredCall".
     * &nbsp;&nbsp; endReason {Object} "normalCall" or "transferredCall".
     * &nbsp;&nbsp; countNew {Number} Number of new messages.
     * &nbsp;&nbsp; countOld {Number} Number of old messages.
     * &nbsp;&nbsp; com2Id {Number} Second phone call identifier.
     * &nbsp;&nbsp; wsRecordId {String} ICE comlog identifier.
     * &nbsp;&nbsp; idMail {Number} email identifier.
     * &nbsp;&nbsp; emailTitle {String} email subject.
     * &nbsp;&nbsp; conversationId {Number} Instant messaging identifier.
     * &nbsp;&nbsp; isWhiteList {Boolean} True if distant party belongs to correspondent's white list.
     * &nbsp;&nbsp; fromBossSecretary {Boolean} True if call routed to secretary phone in a manager/secretary context.
     * &nbsp;&nbsp; peerId {String} Identification of the peer in a manager/secretary context.
     * &nbsp;&nbsp; entryPriority {Object} comlog entry priority
     * }
     * </pre>
     *
     * Possible values of entryType are :
     *
     * <pre>
     * - "CallLog"          for Calllog
     * - "CallBackRequest"  for CR
     * - "VM"               for VoiceMail
     * - "Conference"       for Conference
     * - "IM"               for Instance Message
     * - "Email"            for Email
     * </pre>
     *
     * Possible value of callDirection are :
     *
     * <pre>
     * - "CALL_INCOMING"         for incoming call
     * - "CALL_OUTGOING"         for outgoing call
     * </pre>
     *
     * Possible value of startReason and endReason are :
     *
     * <pre>
     * - "CALL_REASON_NORMAL"       for a normal call
     * - "CALL_REASON_TRANSFERRED"  for a transferred call
     * </pre>
     *
     * Possible value of entryPriority are :
     *
     * <pre>
     * - "Low"      for low priority
     * - "Normal"   for normal priority
     * - "High"     for high priority
     * </pre>
     */
     findEntriesByCategories: function(args) {
        this.callCoreApplicationMethod({
            domainName : "Comlog",
            methodName : "findEntriesByCategories",
            context : args.context,
            params: args.params,
            paramType : ["QStringList"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>ComLogEntry entries</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Finds the entries of the ComLog matching the filter for a given Contact.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;{String} the contact
     * &nbsp;&nbsp;{Number} the status
     * &nbsp;&nbsp;{Number} the type
     * &nbsp;&nbsp;{Number} the number
     * &nbsp;&nbsp;{Number} the start
     * }
     * </pre>
     *
     * The callback will be called with the list of planned events, each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; entryType {Object} Type of communication.
     * &nbsp;&nbsp; comLogEntryId {Number} Unique comlog entry identifier.
     * &nbsp;&nbsp; isRead {Boolean} Entry already (True) or not yet (False) read.
     * &nbsp;&nbsp; dateTime {Date} Date and time of the beginning of the logged event.
     * &nbsp;&nbsp; phoneNumber {String} Phone number of the correspondent. The string may be empty for the VM or if number is unknown.
     * &nbsp;&nbsp; comId {Number} Phone call identifier.
     * &nbsp;&nbsp; label {String} Name or phone number displayed in the ComLog summary.
     * &nbsp;&nbsp; contactId {String} Contact identifier of the correspondent if contact is defined, else correspondent's phone number.
     * &nbsp;&nbsp; endTime {Date} Date and time of the end of the logged event.
     * &nbsp;&nbsp; connectionTime {Date} Date and time of the beginning of the conversation.
     * &nbsp;&nbsp; isAnswered {Boolean} True if the call has been answered.
     * &nbsp;&nbsp; callDirection {Object} "Incoming" or "Outgoing".
     * &nbsp;&nbsp; startReason {Object} "normalCall" or "transferredCall".
     * &nbsp;&nbsp; endReason {Object} "normalCall" or "transferredCall".
     * &nbsp;&nbsp; countNew {Number} Number of new messages.
     * &nbsp;&nbsp; countOld {Number} Number of old messages.
     * &nbsp;&nbsp; com2Id {Number} Second phone call identifier.
     * &nbsp;&nbsp; wsRecordId {String} ICE comlog identifier.
     * &nbsp;&nbsp; idMail {Number} email identifier.
     * &nbsp;&nbsp; emailTitle {String} email subject.
     * &nbsp;&nbsp; conversationId {Number} Instant messaging identifier.
     * &nbsp;&nbsp; isWhiteList {Boolean} True if distant party belongs to correspondent's white list.
     * &nbsp;&nbsp; fromBossSecretary {Boolean} True if call routed to secretary phone in a manager/secretary context.
     * &nbsp;&nbsp; peerId {String} Identification of the peer in a manager/secretary context.
     * &nbsp;&nbsp; entryPriority {Object} comlog entry priority
     * }
     * </pre>
     *
     * Possible values of entryType are :
     *
     * <pre>
     * - "CallLog"          for Calllog
     * - "CallBackRequest"  for CR
     * - "VM"               for VoiceMail
     * - "Conference"       for Conference
     * - "IM"               for Instance Message
     * - "Email"            for Email
     * </pre>
     *
     * Possible value of callDirection are :
     *
     * <pre>
     * - "CALL_INCOMING"         for incoming call
     * - "CALL_OUTGOING"         for outgoing call
     * </pre>
     *
     * Possible value of startReason and endReason are :
     *
     * <pre>
     * - "CALL_REASON_NORMAL"       for a normal call
     * - "CALL_REASON_TRANSFERRED"  for a transferred call
     * </pre>
     *
     * Possible value of entryPriority are :
     *
     * <pre>
     * - "Low"      for low priority
     * - "Normal"   for normal priority
     * - "High"     for high priority
     * </pre>
     */
    findEntriesByContactNb : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Comlog",
            methodName : "findEntriesByContact",
            context : args.context,
            params: args.params,
            paramType: ["QString", "int", "int", "int", "int"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Deletes a ComLogEntry of the ComLog.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the identifier of ComLogEntry to delete.
     *
     * <pre>
     * {Number} for id of the ComLogEntry to delete
     * </pre>     
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */
    deleteEntry : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Comlog",
            methodName : "deleteEntry",
            context : args.context,
            params: args.params,
            paramType: ["qlonglong"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Deletes all ComLog entries.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *  
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */
    deleteAll : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Comlog",
            methodName : "deleteAll",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Deletes ComLog entries with selected types.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains a list of couples representing the filters for the deletion of comlog entries. Each of them has the following
     * <pre>
     * {String} logTypeFilter:readStatusFilter
     * </pre>
     *  
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     *
     * Possible values of logTypeFilter are :
     *
     * <pre>
     * - 0   for an incoming call
     * - 1   for an outgoing call
     * - 2   for an answered incoming call
     * - 3   for an answered outgoing call
     * - 4   for an unanswered incoming call
     * - 5   for an unanswered outgoing call
     * - 6   for a call back request
     * - 7   for voice mail 
     * - 8   for all kind of calls 
     * - 9   for an instant message
     * - 10  for an email
     * - 11  for a conference
     * </pre> 
     *
     * Possible valueis of readStatusFilter are :
     *
     * <pre>
     * - 0  for a not yet read comlog entry
     * - 1  for an already read comlog entry
     * - 2  for all comlog entries
     * </pre> 
     */
	deleteWithMoreFilters : function(args){
		this.callCoreApplicationMethod({
			domainName		: "Comlog",
			methodName		: "deleteWithMoreFilters",
			context			: args.context,
			params			: args.params,
			paramType		: ["QStringList"],
			callback		: args.callback,
			callbackParams	: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Marks an entry of the ComLog as read.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the identifier of ComLogEntry to be marked.
     *
     * <pre>
     * {Number} for id of the ComLogEntry to be marked
     * </pre>     
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */
    markEntryAsRead : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Comlog",
            methodName : "markEntryAsRead",
            context : args.context,
            params: args.params,
            paramType: ["qlonglong"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Marks all missed call entries of the ComLog as read.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */
    readAllMissedCall : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Comlog",
            methodName : "readAllMissedCall",
            context : args.context,
            params: args.params,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>number of events</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Counts the number of entries in the ComLog with a given read status and from a given category.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;{Number} the status
     * &nbsp;&nbsp;{Number} the type
     * }
     * </pre>   
     *
     * The callback will be called with a number of events:
     *
     * <pre>
     * int {Number}
     * </pre>
     */
    count : function(args){
		this.callCoreApplicationMethod({
			domainName		: "Comlog",
			methodName		: "count",
			context			: args.context,
			params			: args.params,
			paramType		: ["int","int"],
			callback		: args.callback,
			callbackParams	: args.callbackParams
		});
		args = null;
	}
	
/**
 * @event notifyCallBackRequestCountChanged
 * This event is used to sent when the number of unread callback request ComLogEntries is changed (increased or decreased).
 * @param {Number} param The number of unread callback request entries.
 */

/**
 * @event notifyComLogChanged
 * This event is used to sent when the structure of the Comlog is changed.
 */
 
 /**
 * @event notifyComLogEntryChanged
 * This event is used to sent when the number of unread voice mail ComLogEntries is changed (increased or decreased).
 * @param {Number} param contains the number of unread voice mail entries.
 */

/**
 * @event notifyMissedCallsCountChanged
 * This event is used to sent when the number of unread unanswered incoming call ComLogEntries is changed (increased or decreased).
 * @param {Number} param contains the number of unread unanswered incoming call entries.
 */
 
 /**
 * @event notifyVoiceMailCountChanged
 * This event is used to sent when the number of unread voice mail ComLogEntries is changed (increased or decreased).
 * @param {Number} param contains the number of unread voice mail entries.
 */

});

ICTouchAPI.APIServices.Comlog = new ICTouchAPI.APIServices.Comlog();
/**
 * @class ICTouchAPI.APIServices.IctDirectory
 * @singleton
 * @extends ICTouchAPI.APIServices.APIServicesBase
 * The IctDirectory module is the one used for all functions related to Directory Applications.<br>
 * It provides API to search users with different criterias and also functions to retrieve all information related to one.
 */
dojo.provide("ICTouchAPI.APIServices.IctDirectory");
dojo.declare("ICTouchAPI.APIServices.IctDirectory",[ICTouchAPI.APIServices.APIServicesBase],
{    
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>list of contacts</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method provides the sorted (by last name, first name) list of local contacts (including DM contacts)<br>
     * Particular rules :<br>
     * - to get all the contacts, specify startInd:-1 , length:-1 </b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;{Number} index of the first contact to return
     * &nbsp;&nbsp;{Number} number of contacts to return
     * }
     * </pre>
     *
     * The callback will be called with the list of contacts, each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; _contactId {String} The unique Id of a local contact                                     
     * &nbsp;&nbsp; _loginName {String} Login Name (when contact is copied from a search)                    
     * &nbsp;&nbsp; _salutation {String} Mr Mrs etc                                                             
     * &nbsp;&nbsp; _firstName {String} First Name                                                              
     * &nbsp;&nbsp; _name {String} Last Name                                                                    
     * &nbsp;&nbsp; _latinFirstName {String} not in use                                                         
     * &nbsp;&nbsp; _latinName {String} not in use yet                                                          
     * &nbsp;&nbsp; _language {String} not in use yet                                                           
     * &nbsp;&nbsp; _timezone {String} not in use yet                                                           
     * &nbsp;&nbsp; _officePhone {String} phone number (granted dialable in a further realease)                 
     * &nbsp;&nbsp; _attendant {String} name of the attendant (not yet in use)                               
     * &nbsp;&nbsp; _colleague {String} name of a colleague (not yet in use)                                 
     * &nbsp;&nbsp; _assistant {String} name of the assistant (not yet in use)                               
     * &nbsp;&nbsp; _email {String} email address                                                            
     * &nbsp;&nbsp; _fax {String} fax number (granted dialable in a further realease)                        
     * &nbsp;&nbsp; _voicemail {String} not in use yet                                                       
     * &nbsp;&nbsp; _personalHome {String} personnal phone number (granted dialable in a further realease)   
     * &nbsp;&nbsp; _personalMobile {String} personnal phone number (granted dialable in a further realease) 
     * &nbsp;&nbsp; _personalFax {String} personnal fax number (granted dialable in a further realease)      
     * &nbsp;&nbsp; _personalEmail {String} personal email address                                           
     * &nbsp;&nbsp; _collaborationId {String} Instant messaging address                                      
     * &nbsp;&nbsp; _misc {Object} A list of other attributes in a JSON structure (key-value)
     * &nbsp;&nbsp; _isPreviewed {Boolean} true if shown in user's Home page                                     
     * &nbsp;&nbsp; _isPrivate {Boolean} true if created by the user                                             
     * &nbsp;&nbsp; _type {Number} see enum IctDirectoryEnum                                                     
     * &nbsp;&nbsp; _photo {String} the url of contact photo                                                 
     * &nbsp;&nbsp; _melody {String} the url of ringing melody when this contact calls (not yet in use)      
     * &nbsp;&nbsp; _lastModificationDate {Date} as said                   
     * }            
     * </pre>
     */                
	getLocalContacts : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctDirectory",
			methodName : "getLocalContacts",
			params: args.params,
			paramType: ["int", "int"],
			context : args.context,
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>list of contacts</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method provides the sorted (by last name, first name) list of local contacts (including DM contacts) that are shown in the Home page<br>
     * Particular rules :<br>
     * - to get all the contacts, specify startInd:-1 , length:-1 </b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;{Number} index of the first contact to return
     * &nbsp;&nbsp;{Number} number of contacts to return
     * }
     * </pre>
     *
     * The callback will be called with the list of contacts, each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; _contactId {String} The unique Id of a local contact                                     
     * &nbsp;&nbsp; _loginName {String} Login Name (when contact is copied from a search)                    
     * &nbsp;&nbsp; _salutation {String} Mr Mrs etc                                                             
     * &nbsp;&nbsp; _firstName {String} First Name                                                              
     * &nbsp;&nbsp; _name {String} Last Name                                                                    
     * &nbsp;&nbsp; _latinFirstName {String} not in use                                                         
     * &nbsp;&nbsp; _latinName {String} not in use yet                                                          
     * &nbsp;&nbsp; _language {String} not in use yet                                                           
     * &nbsp;&nbsp; _timezone {String} not in use yet                                                           
     * &nbsp;&nbsp; _officePhone {String} phone number (granted dialable in a further realease)                 
     * &nbsp;&nbsp; _attendant {String} name of the attendant (not yet in use)                               
     * &nbsp;&nbsp; _colleague {String} name of a colleague (not yet in use)                                 
     * &nbsp;&nbsp; _assistant {String} name of the assistant (not yet in use)                               
     * &nbsp;&nbsp; _email {String} email address                                                            
     * &nbsp;&nbsp; _fax {String} fax number (granted dialable in a further realease)                        
     * &nbsp;&nbsp; _voicemail {String} not in use yet                                                       
     * &nbsp;&nbsp; _personalHome {String} personnal phone number (granted dialable in a further realease)   
     * &nbsp;&nbsp; _personalMobile {String} personnal phone number (granted dialable in a further realease) 
     * &nbsp;&nbsp; _personalFax {String} personnal fax number (granted dialable in a further realease)      
     * &nbsp;&nbsp; _personalEmail {String} personal email address                                           
     * &nbsp;&nbsp; _collaborationId {String} Instant messaging address                                      
     * &nbsp;&nbsp; _misc {Object} A list of other attributes defined by keyword, value
     * &nbsp;&nbsp; _isPreviewed {Boolean} true if shown in user's Home page                                     
     * &nbsp;&nbsp; _isPrivate {Boolean} true if created by the user                                             
     * &nbsp;&nbsp; _type {Number} see enum IctDirectoryEnum                                                     
     * &nbsp;&nbsp; _photo {String} the url of contact photo                                                 
     * &nbsp;&nbsp; _melody {String} the url of ringing melody when this contact calls (not yet in use)      
     * &nbsp;&nbsp; _lastModificationDate {Date} as said                   
     * }            
     * </pre>
     */  
	getAllContactsInPreview : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctDirectory",
			methodName : "getAllContactsInPreview",
			params: args.params,
			paramType: ["int", "int"],
			context : args.context,
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>list of contacts</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method provides a Contact list of 1 local contact, or empty, of the passed contactId<br>
     * - user to provide contact Id<br>
     *
     * Particular rules :<br>
     * - this method is currently not able to return more than 1 contact (but this should not happen to have many contacts with the same id)<br></b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the contact Id.
     * <pre>
     * {String} the contact Id
     * </pre>
     *
     * The callback will be called with the list of contacts, each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; _contactId {String} The unique Id of a local contact                                     
     * &nbsp;&nbsp; _loginName {String} Login Name (when contact is copied from a search)                    
     * &nbsp;&nbsp; _salutation {String} Mr Mrs etc                                                             
     * &nbsp;&nbsp; _firstName {String} First Name                                                              
     * &nbsp;&nbsp; _name {String} Last Name                                                                    
     * &nbsp;&nbsp; _latinFirstName {String} not in use                                                         
     * &nbsp;&nbsp; _latinName {String} not in use yet                                                          
     * &nbsp;&nbsp; _language {String} not in use yet                                                           
     * &nbsp;&nbsp; _timezone {String} not in use yet                                                           
     * &nbsp;&nbsp; _officePhone {String} phone number (granted dialable in a further realease)                 
     * &nbsp;&nbsp; _attendant {String} name of the attendant (not yet in use)                               
     * &nbsp;&nbsp; _colleague {String} name of a colleague (not yet in use)                                 
     * &nbsp;&nbsp; _assistant {String} name of the assistant (not yet in use)                               
     * &nbsp;&nbsp; _email {String} email address                                                            
     * &nbsp;&nbsp; _fax {String} fax number (granted dialable in a further realease)                        
     * &nbsp;&nbsp; _voicemail {String} not in use yet                                                       
     * &nbsp;&nbsp; _personalHome {String} personnal phone number (granted dialable in a further realease)   
     * &nbsp;&nbsp; _personalMobile {String} personnal phone number (granted dialable in a further realease) 
     * &nbsp;&nbsp; _personalFax {String} personnal fax number (granted dialable in a further realease)      
     * &nbsp;&nbsp; _personalEmail {String} personal email address                                           
     * &nbsp;&nbsp; _collaborationId {String} Instant messaging address                                      
     * &nbsp;&nbsp; _misc {Object} A list of other attributes defined by keyword, value
     * &nbsp;&nbsp; _isPreviewed {Boolean} true if shown in user's Home page                                     
     * &nbsp;&nbsp; _isPrivate {Boolean} true if created by the user                                             
     * &nbsp;&nbsp; _type {Number} see enum IctDirectoryEnum                                                     
     * &nbsp;&nbsp; _photo {String} the url of contact photo                                                 
     * &nbsp;&nbsp; _melody {String} the url of ringing melody when this contact calls (not yet in use)      
     * &nbsp;&nbsp; _lastModificationDate {Date} as said                   
     * }            
     * </pre>
     */
	getContactById : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctDirectory",
			methodName : "getContactById",
			context : args.context,
			params: args.params,
			paramType: ["QString"],
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>result status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method starts a search of contacts (local at least, remote on option) matching a N-worded string<br>
     * Particular rules :<br>
     * - searchId is mandatory and provides a way to retrieve the found contacts later, and to purge the search results (for memory deallocation)<br>
     * - "startSearch" is designed to be asynchroneous; the application will, later, emit 0 to N "DIR_SEARCH_RESULTS_ (searchId)" events,
     * and one DIR_SEARCH_COMPLETED_(searchId) event when the search is finished.<br>
     * - localOnly, if specified true, will not try to retrieve contacts from UDA directories<br>
     * - name can be multi-worded :<br>
     * - simplest case, one word: it will search the contacts whose last name or first name starts with name parameter<br>
     * - elaborated cases, several words (separated by blank) : it will search the contacts who have all the words in the beginning of the last name and/or first name<br>
     * - the number of possible words is limited by the setting "DirMaxWord", but the above rule makes useful only 1 or 2 words<br>
     * - in case of remote search, all the available UDA directories are scanned<br>
     *
     * Emitted events after startSearchContactByName(searchId) :<br>
     *
     * - DIR_SEARCH_RESULTS_(searchId) , with param int nbResults (1 to N)<br>
     * - DIR_SEARCH_COMPLETED_(searchId), with param int nbResults (0 to N)<br>
     * There can be zero to several successive DIR_SEARCH_RESULTS_(searchId), and always one final DIR_SEARCH_COMPLETED_(searchId).
     * Each DIR_SEARCH_RESULT provided quickly the information on the search progress, with the total increasing nbResults reached at this stage.
     * When a search step provides 0 results, no DIR_SEARCH_RESULTS is sent, thus avoids disturbing the client for nothing new.
     * When the client gets a DIR_SEARCH_RESULT_(searchId) corresponding to his search, he then can fetch part or all results using method "GET_SEARCH_RESULTS".
     * This allows a fast display while search is still going on.
     * The client may also want to wait "DIR_SEARCH_COMPLETED_(searchId)" (nbResults) and the retrieve part or all of the contacts.<br>
     *
     * Caution: the getSearchResults that the client would then use to get part of the contacts before the end of the search,
     * would always get the set of [start, number] contacts AFTER a sort of the current search results, not knowing if further results would or not insert names in this list.
     * So, a client should not consider that pages requested before the DIR_SEARCH_COMPLETED are correctly sorted</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;{String}   mandatory: allows to link search request on one side, resulting events , requests and results on the other side          
     * &nbsp;&nbsp;{String}   see method description
     * &nbsp;&nbsp;{Boolean}  if specified and true, will not try to retrieve contacts from UDA directories; and will return only zero to one DIR_SEARCH_RESULTS_(searchId)  
     * }
     * </pre>
     *
     * The callback will be called with a result status, which is always 0 (success).  
     */
	startSearchContactByName : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctDirectory",
			methodName : "startSearchContactByName",
			context : args.context,
			params: args.params,
			paramType: ["QString", "QString", "int", "bool"],
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

	startSearchContactByAllFields : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctDirectory",
			methodName : "startSearchContactByAllFields",
			context : args.context,
			params: args.params,
			paramType: ["QString", "QString", "bool"],
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>result status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method is IDENTICAL to startSearchContactByName,
     * excepted the parameter which is a phone number.
     * The same mechanism allows an asynchrone notification of successive sets of results.
     * Please refer to method startSearchContactByName</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;{String}   mandatory: allows to link search request on one side, resulting events , requests and results on the other side
     * &nbsp;&nbsp;{String}   see method description
     * &nbsp;&nbsp;{Boolean}  if specified and true, will not try to retrieve contacts from UDA directories; and will return only zero to one DIR_SEARCH_RESULTS_(searchId)
     * }
     * </pre>
     *
     * The callback will be called with a result status, which is always 0 (success).  
     */
	startSearchContactByPhoneNumber : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctDirectory",
			methodName : "startSearchContactByPhoneNumber",
			context : args.context,
			params: args.params,
			paramType: ["QString", "QString", "bool"],
			callback : args.callback,
			callbackParams: args.callbackParams,
			errorCallback : args.errorCallback
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>result status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method creates a contact in the local database<br>
     * - user to provide all information<br>
     *
     *  Particular rules :<br>
     *  - if request Id is "DM" the contact type is "DM" else it is local<br>
     *  - event DIR_CONTACT_CREATED is sent if ok,  excepted for a DM contact<br>
     *  - contact last modification date is set to current date time<br>
     *  - contact Id is calculated depending on origin: DM or not<br></b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;{String} a request Id
     * &nbsp;&nbsp;{Object}   the Contact to create
     * }
     * </pre>
     
     * The second parameter has the following structs: 
     * <pre>
     * {
     * _contactId {String} The unique Id of a local contact                                     
     * _loginName {String} Login Name (when contact is copied from a search)                    
     * _salutation {String} Mr Mrs etc                                                             
     * _firstName {String} First Name                                                              
     * _name {String} Last Name                                                                    
     * _latinFirstName {String} not in use                                                         
     * _latinName {String} not in use yet                                                          
     * _language {String} not in use yet                                                           
     * _timezone {String} not in use yet                                                           
     * _officePhone {String} phone number (granted dialable in a further realease)                 
     * _attendant {String} name of the attendant (not yet in use)                               
     * _colleague {String} name of a colleague (not yet in use)                                 
     * _assistant {String} name of the assistant (not yet in use)                               
     * _email {String} email address                                                            
     * _fax {String} fax number (granted dialable in a further realease)                        
     * _voicemail {String} not in use yet                                                       
     * _personalHome {String} personnal phone number (granted dialable in a further realease)   
     * _personalMobile {String} personnal phone number (granted dialable in a further realease) 
     * _personalFax {String} personnal fax number (granted dialable in a further realease)      
     * _personalEmail {String} personal email address                                           
     * _collaborationId {String} Instant messaging address                                      
     * _misc {Object} A list of other attributes defined by keyword, value
     * _isPreviewed {Boolean} true if shown in user's Home page                                     
     * _isPrivate {Boolean} true if created by the user                                             
     * _type {Number} see enum IctDirectoryEnum                                                     
     * _photo {String} the url of contact photo                                                 
     * _melody {String} the url of ringing melody when this contact calls (not yet in use)      
     * _lastModificationDate {Date} as said                   
     * }            
     * </pre>
     *
     * The callback will be called with a result status.  
     * Possible value are:
     * <pre>
     * {
     * &nbsp;&nbsp; 0 if successful    
     * &nbsp;&nbsp; 6 during synchronization (contact it then read-only)
     * &nbsp;&nbsp; 3 if the max number was reached ;
     * &nbsp;&nbsp; -1: database error
     * }
     * </pre>     
     */
	createLocalContact : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctDirectory",
			methodName : "createLocalContact",
			context : args.context,
			params: args.params,
			paramType: ["QString", "Contact"],
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>result status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method updates a contact in the local database, using contactId key<br>
     * - user to provide all information<br>
     *
     * Particular rules :<br>
     * - only the pure local contacts can be updated (DM cannot, and other types, remote, are only in memory and are not updated)<br>
     * - event DIR_CONTACT_UPDATED is sent if ok<br>
     * - it removes the old photo in the filesystem if different<br>
     * - contact last modification date is set to current date time<br></b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;{String} a request Id
     * &nbsp;&nbsp;{Object}   the Contact to create
     * }
     * </pre>
     
     * The second parameter has the following structs: 
     * <pre>
     * {
     * _contactId {String} The unique Id of a local contact                                     
     * _loginName {String} Login Name (when contact is copied from a search)                    
     * _salutation {String} Mr Mrs etc                                                             
     * _firstName {String} First Name                                                              
     * _name {String} Last Name                                                                    
     * _latinFirstName {String} not in use                                                         
     * _latinName {String} not in use yet                                                          
     * _language {String} not in use yet                                                           
     * _timezone {String} not in use yet                                                           
     * _officePhone {String} phone number (granted dialable in a further realease)                 
     * _attendant {String} name of the attendant (not yet in use)                               
     * _colleague {String} name of a colleague (not yet in use)                                 
     * _assistant {String} name of the assistant (not yet in use)                               
     * _email {String} email address                                                            
     * _fax {String} fax number (granted dialable in a further realease)                        
     * _voicemail {String} not in use yet                                                       
     * _personalHome {String} personnal phone number (granted dialable in a further realease)   
     * _personalMobile {String} personnal phone number (granted dialable in a further realease) 
     * _personalFax {String} personnal fax number (granted dialable in a further realease)      
     * _personalEmail {String} personal email address                                           
     * _collaborationId {String} Instant messaging address                                      
     * _misc {Object} A list of other attributes in a JSON structure (key-value)
     * _isPreviewed {Boolean} true if shown in user's Home page                                     
     * _isPrivate {Boolean} true if created by the user                                             
     * _type {Number} see enum IctDirectoryEnum                                                     
     * _photo {String} the url of contact photo                                                 
     * _melody {String} the url of ringing melody when this contact calls (not yet in use)      
     * _lastModificationDate {Date} as said                   
     * }            
     * </pre>
     *
     * The callback will be called with a result status.  
     * Possible value are:
     * <pre>
     * {
     * &nbsp;&nbsp; 0 if successful    
     * &nbsp;&nbsp; 1 if the found contact is not a pure local
     * &nbsp;&nbsp; 2 if the contact is not found though its contactId
     * &nbsp;&nbsp; 6 during synchronization
     * &nbsp;&nbsp; -1: database error
     * }
     * </pre>     
     */
	modifyLocalContact : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctDirectory",
			methodName : "modifyLocalContact",
			context : args.context,
			params: args.params,
			paramType: ["QString", "Contact"],
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>result status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method definitely deletes a contact in the local database, using contactId key<br>
     * - user to provide key<br>
     * Particular rules :<br>
     * - event DIR_CONTACT_DELETED is sent if ok, but not for DM delete<br>
     * - it removes the old photo in the file system if different<br></b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;{String} a request Id
     * &nbsp;&nbsp;{Object}   the Contact to create
     * }
     * </pre>
     
     * The second parameter has the following structs: 
     * <pre>
     * {
     * _contactId {String} The unique Id of a local contact                                     
     * _loginName {String} Login Name (when contact is copied from a search)                    
     * _salutation {String} Mr Mrs etc                                                             
     * _firstName {String} First Name                                                              
     * _name {String} Last Name                                                                    
     * _latinFirstName {String} not in use                                                         
     * _latinName {String} not in use yet                                                          
     * _language {String} not in use yet                                                           
     * _timezone {String} not in use yet                                                           
     * _officePhone {String} phone number (granted dialable in a further realease)                 
     * _attendant {String} name of the attendant (not yet in use)                               
     * _colleague {String} name of a colleague (not yet in use)                                 
     * _assistant {String} name of the assistant (not yet in use)                               
     * _email {String} email address                                                            
     * _fax {String} fax number (granted dialable in a further realease)                        
     * _voicemail {String} not in use yet                                                       
     * _personalHome {String} personnal phone number (granted dialable in a further realease)   
     * _personalMobile {String} personnal phone number (granted dialable in a further realease) 
     * _personalFax {String} personnal fax number (granted dialable in a further realease)      
     * _personalEmail {String} personal email address                                           
     * _collaborationId {String} Instant messaging address                                      
     * _misc {Object} A list of other attributes in a JSON structure (key-value)
     * _isPreviewed {Boolean} true if shown in user's Home page                                     
     * _isPrivate {Boolean} true if created by the user                                             
     * _type {Number} see enum IctDirectoryEnum                                                     
     * _photo {String} the url of contact photo                                                 
     * _melody {String} the url of ringing melody when this contact calls (not yet in use)      
     * _lastModificationDate {Date} as said                   
     * }            
     * </pre>
     *
     * The callback will be called with a result status.  
     * Possible value are:
     * <pre>
     * {
     * &nbsp;&nbsp; 0 if successful    
     * &nbsp;&nbsp; 1 if the found contact is of type UDA or ICE (remote contacts in memory)
     * &nbsp;&nbsp; 2 if the contact is not found though its contactId
     * &nbsp;&nbsp; 6 during synchronization
     * &nbsp;&nbsp; -1: database error
     * }
     * </pre>     
     */
	deleteLocalContact : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctDirectory",
			methodName : "deleteLocalContact",
			context : args.context,
			params: args.params,
			paramType: ["QString", "QString"],
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>String</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method copys the contact to local</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * Params contains the contact Id
     * <pre>
     * {String} the contact Id
     * </pre>
     *
     * The callback will be called with a String.
     */
	copyContactToLocal : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctDirectory",
			methodName : "copyContactToLocal",
			context : args.context,
			params: args.params,
			paramType: ["QString"],
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>sorted list of ContactSummary</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This Method retrieves a list of found ContactSummary in a given search. Only basic information of the contacts are contained in the ContactSummary (contactId, name, firstName, officePhone, email, type).
	 * The whole contact object can be obtained using the function getContactFromSearch. Please refer to "getContactFromSearch" for more details about this function.
     * Please refer to "startSearchContactByName" for more details about the asynchroneous mechanism of searchs.<br>
     *
     * Particular rules :<br>
     * - startInd and length are mandatory, there are no default values to get all the list<br>
     * - startInd : specify 0 to get the first contact<br>
     * - to get all the found contacts, specify startind=0, length = the number of found contacts (nbResults returned by the event DIR_SEARCH_RESULTS or by the final DIR_SEARCH_COMPLETED)<br>
     * Caution: the  getSearchResults that the client would then use to get part of the contacts before the end of the search,
     * would always get the set of [start, number] contacts AFTER a sort of the current search results, not knowing if further results would or not insert names in this list.
     * So, a client should not consider that pages requested before the DIR_SEARCH_COMPLETED are correctly sorted</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;{String} the reference of the search
     * &nbsp;&nbsp;{Number} the index of the first contact
     * &nbsp;&nbsp;{Number} the number of wanted contacts starting from startInd
     * }
     * </pre>
     *
     * The callback will be called with the list of contacts, each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; _contactId {String} The unique Id of a local contact                                                 
     * &nbsp;&nbsp; _firstName {String} First Name                                                              
     * &nbsp;&nbsp; _name {String} Last Name                                                      
     * &nbsp;&nbsp; _officePhone {String} phone number (granted dialable in a further realease)                           
     * &nbsp;&nbsp; _email {String} email address                
     * }            
     * </pre>
     */     
	getSearchResults : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctDirectory",
			methodName : "getSearchResults",
			context : args.context,
			params: args.params,
			paramType: ["QString", "int", "int"],
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>contact</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This Method retrieves a found contact in a given search. Unlike the ContactSummary obtained using getSearchResults, this contact contains the detailed informations.
	 * 
     * Particular rules :<br>
     * - searchId and contactId are mandatory<br>
     * - searchId : specify the id of the search from which the contact has to be get<br>
     * - contactId : specify the id of the contact to be get
     * Caution: once a new search has been launched with a new search id, the previous search is no longer available in IctDirectory.
	 * Therefore, a client has to get the whole contact needed BEFORE starting a new search with a new search id</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;{String} the reference of the search
     * &nbsp;&nbsp;{Number} the reference of the contact to get
     * }
     * </pre>
     *
     * The callback will be called with the contact corresponding to the given contactId. The contact returned has the following structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; _contactId {String} The unique Id of a local contact
     * &nbsp;&nbsp; _loginName {String} Login Name (when contact is copied from a search)
     * &nbsp;&nbsp; _salutation {String} Mr Mrs etc
     * &nbsp;&nbsp; _firstName {String} First Name
     * &nbsp;&nbsp; _name {String} Last Name
     * &nbsp;&nbsp; _latinFirstName {String} not in use
     * &nbsp;&nbsp; _latinName {String} not in use yet
     * &nbsp;&nbsp; _language {String} not in use yet
     * &nbsp;&nbsp; _timezone {String} not in use yet
     * &nbsp;&nbsp; _officePhone {String} phone number (granted dialable in a further realease)
     * &nbsp;&nbsp; _attendant {String} name of the attendant (not yet in use)
     * &nbsp;&nbsp; _colleague {String} name of a colleague (not yet in use)
     * &nbsp;&nbsp; _assistant {String} name of the assistant (not yet in use)
     * &nbsp;&nbsp; _email {String} email address
     * &nbsp;&nbsp; _fax {String} fax number (granted dialable in a further realease)
     * &nbsp;&nbsp; _voicemail {String} not in use yet
     * &nbsp;&nbsp; _personalHome {String} personnal phone number (granted dialable in a further realease)
     * &nbsp;&nbsp; _personalMobile {String} personnal phone number (granted dialable in a further realease)
     * &nbsp;&nbsp; _personalFax {String} personnal fax number (granted dialable in a further realease)
     * &nbsp;&nbsp; _personalEmail {String} personal email address
     * &nbsp;&nbsp; _collaborationId {String} Instant messaging address
     * &nbsp;&nbsp; _misc {Object} A list of other attributes in a JSON structure (key-value)
     * &nbsp;&nbsp; _isPreviewed {Boolean} true if shown in user's Home page
     * &nbsp;&nbsp; _isPrivate {Boolean} true if created by the user
     * &nbsp;&nbsp; _type {Number} see enum IctDirectoryEnum
     * &nbsp;&nbsp; _photo {String} the url of contact photo
     * &nbsp;&nbsp; _melody {String} the url of ringing melody when this contact calls (not yet in use)
     * &nbsp;&nbsp; _lastModificationDate {Date} as said
     * </pre>
     */
	getContactFromSearch : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctDirectory",
			methodName : "getContactFromSearch",
			context : args.context,
			params: args.params,
			paramType: ["QString", "QString"],
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>result status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method is IDENTICAL to startSearchContactByName,
     * excepted the parameter which is a phone number.
     * The same mechanism allows an asynchrone notification of successive sets of results.
     * Please refer to method startSearchContactByName</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;{String}   mandatory: allows to link search request on one side, resulting events , requests and results on the other side
     * &nbsp;&nbsp;{String}   see method description
     * &nbsp;&nbsp;{Boolean}  if specified and true, will not try to retrieve contacts from UDA directories; and will return only zero to one DIR_SEARCH_RESULTS_(searchId)
     * }
     * </pre>
     *
     * The callback will be called with a result status, which is always 0 (success).  
     */
	purgeSearchResults : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctDirectory",
			methodName : "purgeSearchResults",
			context : args.context,
			params: args.params,
			paramType: ["QString"],
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>result status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method changes the "isPreview" flag to TRUE so that the contact is shown on the Home page<br>
     * - user to provide contact Id<br>
     * Particular rules :<br>
     * - event DIR_CONTACTINPREVIEW_CREATED is sent if ok, but not for DM contact</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;{String}   a request Id
     * &nbsp;&nbsp;{String}   the contact Id
     * }
     * </pre>
     *
     * The callback will be called with a result status, which is always 0 (success).  
     * <pre>
     * {
     * &nbsp;&nbsp; 0 if successful    
     * &nbsp;&nbsp; 4 if the found contact is already flagged
     * &nbsp;&nbsp; 6 during synchronization
     * &nbsp;&nbsp; -1: database error
     * }
     * </pre>     
     */
	addContactToPreview : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctDirectory",
			methodName : "addContactToPreview",
			context : args.context,
			params: args.params,
			paramType: ["QString", "QString"],
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>result status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method changes the "isPreview" flag to FALSE so that the contact is not shown on the Home page<br>
     * - user to provide contact Id<br>
     * Particular rules :<br>
     * - event DIR_CONTACTINPREVIEW_DELETED is sent if ok, but not for DM contact</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;{String}   a request Id
     * &nbsp;&nbsp;{String}   the contact Id
     * }
     * </pre>
     *
     * The callback will be called with a result status, which is always 0 (success).  
     * <pre>
     * {
     * &nbsp;&nbsp; 0 if successful    
     * &nbsp;&nbsp; 4 if the found contact is already flagged
     * &nbsp;&nbsp; 6 during synchronization
     * &nbsp;&nbsp; -1: database error
     * }
     * </pre>     
     */
	removeContactFromPreview : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctDirectory",
			methodName : "removeContactFromPreview",
			context : args.context,
			params: args.params,
			paramType: ["QString", "QString"],
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	}
});

ICTouchAPI.APIServices.IctDirectory = new ICTouchAPI.APIServices.IctDirectory();
/**
 * @class ICTouchAPI.APIServices.FileManager
 * @extends ICTouchAPI.APIServices.APIServicesBase
 * @singleton
 * The file manager module is in charge of all the file manager features.<br>
 * The file manager module provides services in order to manage actions on files.<br>
 * These files can be stored on the phone itself,on an USB Key
 * or can be a link to a streaming audio or video.
 */
dojo.provide("ICTouchAPI.APIServices.FileManager");
dojo.declare("ICTouchAPI.APIServices.FileManager",[ICTouchAPI.APIServices.APIServicesBase],
{
//    getMediaTypeList : function(context, callback, callbackParams) {
//        this.callCoreApplicationMethod({
//            domainName : "FileManager",
//            methodName : "getMediaTypeList",
//            context : context,
//            callback : callback,
//            callbackParams: callbackParams
//        });
//    },

//    getMediaList : function(params, context, callback, callbackParams) {
//        this.callCoreApplicationMethod({
//            domainName : "FileManager",
//            methodName : "getMediaList",
//            context : context,
//            params: params,
//            paramType: ["int"],
//            callback : callback,
//            callbackParams: callbackParams
//        });
//    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>result status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Deletes a file</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the identifier of the file.
     * Possible values are :
     *
     * <pre>
     * {Number} the identifier of the file.
     * </pre>
     *
     * The callback will be called with a status. Possible value are:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; 0   removal successful
     * &nbsp;&nbsp; -1  technical problem
     * &nbsp;&nbsp; -3  unknown file identifier
     * &nbsp;&nbsp; -6  permission denied
     * &nbsp;&nbsp; -7  the file does not exist
     * }
     * </pre>
     */
    deleteFile : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "deleteFile",
            context : args.context,
            params: args.params,
            paramType: ["int"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>result status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Renames a file</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     *
     * <pre>
     * {
     * &nbsp;&nbsp;{String} the new name of the file
     * &nbsp;&nbsp;{String} the new path of the file
     * }
     * </pre>
     *
     * The callback will be called with a status. Possible value are:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; 0   rename successful
     * &nbsp;&nbsp; -1  technical problem
     * &nbsp;&nbsp; -3  unknown file identifier
     * &nbsp;&nbsp; -4  already exists
     * &nbsp;&nbsp; -5  name already exists
     * &nbsp;&nbsp; -6  permission denied
     * }
     * </pre>
     */
    renameFile : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "renameFile",
            context : args.context,
            params: args.params,
            paramType: ["int", "QString", "QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

	moveFileToDevice : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "moveFileToDevice",
            context : args.context,
            params: args.params,
            paramType: ["int", "enum"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>result status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method adds a category of files.<br>
     * By default, 4 categories will be created on module installation:<br>
     * Other : all the files which are not supported by the phone<br>
     * Audio : it gathers files with extensions like jpg,bmp,gif and png<br>
     * Picture : it gathers files with extensions like ogg, mp3 and wav<br>
     * Web Radio : it gathers url and all informations related to the link:name of the radio,...<br></b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     *
     * <pre>
     * {
     * &nbsp;&nbsp;{String} the name of the category
     * &nbsp;&nbsp;{Number} the rank of the category
     * &nbsp;&nbsp;{Boolean} false means the user can't suppress the category
     * }
     * </pre>
     *
     * The callback will be called with a status. Possible value are:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; 0   insertion successful
     * &nbsp;&nbsp; -14  order
     * &nbsp;&nbsp; -15 the category already exists
     * }
     * </pre>
     */
	addFileCategory : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "addFileCategory",
            context : args.context,
            params: args.params,
            paramType: ["FileCategory"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>information of a category</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method gets all the information of a category  which is passed as an argument</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the name of the category
     *
     * <pre>
     * {String} the name of the category
     * </pre>
     *
     * The callback will be called with a JSON with the folloing structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;idFileCategory {Number} associated identifier for the current file category
     * &nbsp;&nbsp;caregoryName {String} name of the current category
     * &nbsp;&nbsp;categoryOrder {Number} order of the current category
     * &nbsp;&nbsp;deletable {Boolean} defines if a category can be deleted or not. Some categories are system defined and can not be modified by the user.
     * }
     * </pre>
     */
	getFileCategory : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "getFileCategory",
            context : args.context,
            params: args.params,
            paramType: ["QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>result status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method modifies a category</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON which contains the following structure :
     *
     * <pre>
     * {
     * &nbsp;&nbsp;idFileCategory {Number} associated identifier for the current file category
     * &nbsp;&nbsp;caregoryName {String} name of the current category
     * &nbsp;&nbsp;categoryOrder {Number} order of the current category
     * &nbsp;&nbsp;deletable {Boolean} defines if a category can be deleted or not. Some categories are system defined and can not be modified by the user.
     * }
     * </pre>
     *
     * The callback will be called with a status. Possible value are:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; 0   modification successful
     * &nbsp;&nbsp; -1  technical problem
     * &nbsp;&nbsp; -2 the category does not exist
     * &nbsp;&nbsp; -14 the order of the category must be >= 0
     * &nbsp;&nbsp; -15 the category already exists
     * }
     * </pre>
     */
	modifyFileCategory : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "modifyFileCategory",
            context : args.context,
            params: args.params,
            paramType: ["FileCategory"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>result status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method deletes a FileCategory according to its identifier. All the FileType linked with
     * this category will be removed. All the File will be attached to the 'Other' category with
	 * a FileType 'unknown'</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the identifier of the category to delete.
     *
     * <pre>
     * {Number} the identifier of the category to delete
     * </pre>
     *
     * The callback will be called with a status. Possible value are:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; 0   deletion successful
     * &nbsp;&nbsp; -1  technical problem
     * &nbsp;&nbsp; -2 the category does not exist
     * &nbsp;&nbsp; -6 the permission is denied : the category can not be deleted
     * }
     * </pre>
     */
	deleteFileCategory : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "deleteFileCategory",
            context : args.context,
            params: args.params,
            paramType: ["int"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>list of File</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method gets all the files corresponding to the category in argument.
     * If the index is incorrect, an empty list will be return. The startInd must be positive.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     *
     * <pre>
     * {
     * {String} the name of the category
     * {Number} ndex of the first category to return (0 for the first one)
     * {Number} the number of categories to return
     * }
     * </pre>
     *
     * The callback will be called with a list of File, each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;url {String} file URL.
     * &nbsp;&nbsp;rights {String} file access rights. Possible users: USER,GROUP,OTHER. Possible access rights for each user: 'r', 'w', 'x'.
     * &nbsp;&nbsp;size {Number} total size of the file in bytes.
     * &nbsp;&nbsp;device {String} device in which the file is located: 'USERSPACE', 'System' and 'USB'.
     * &nbsp;&nbsp;category {Object} the file category in which the current file is classified.
     * &nbsp;&nbsp;typeFile {Object} type of the file.
     * &nbsp;&nbsp;isLink {Number} defines if a file is a link or not. Possible values: -1 value for an empty file object, 0 the file is not a link, 1 the file is a link.
     * &nbsp;&nbsp;name {String} short name of the file, e.g. without it's full path.
     * &nbsp;&nbsp;streaming {Boolean} if TRUE the file is a streaming source, FALSE otherwise.
     * }
     * </pre>
     *
     * The typeFile parameter has the following structure :
     *
     * <pre>
     * {
     * &nbsp;&nbsp;idFileType {Number} identifier of the current file type
     * &nbsp;&nbsp;extension {String} extension associated to the current file type (ex: mp3)
     * &nbsp;&nbsp;header {String} associate header (magic number) for the current file type (ex: audio/mpeg)
     * &nbsp;&nbsp;category {Object} associated file category to the current file type
     * }
     * </pre>
     *
     * The category parameter has the following structure :
     *
     * <pre>
     * {
     * &nbsp;&nbsp;idFileCategory {Number} associated identifier for the current file category
     * &nbsp;&nbsp;caregoryName {String} name of the current category
     * &nbsp;&nbsp;categoryOrder {Number} order of the current category
     * &nbsp;&nbsp;deletable {Boolean} defines if a category can be deleted or not. Some categories are system defined and can not be modified by the user.
     * }
     * </pre>
     */
	getFileDescriptions : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "getFileDescriptions",
            context : args.context,
            params: args.params,
            paramType: ["QString","int","int"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>list of File</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method gets all the files according to different criteria</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * Params is an array which contains the following structure :
     *
     * <pre>
     * {
     * {String} the name of the category. If the name does not match with any of the stored categories, an empty list is returned.
     * {Number} index of the first File to return (0 for the first one), has to be positive.
     * {Number} the number of File to return, has to be positive.
     * {String} a list of type (actually, the magic number) to select, separeted by the special character ";"(for instance : "audio/mpeg;audio/x-wav").
     * {String} the list of the device identifier separated by the special character ";" (for instance : "0;1")
     * }
     * </pre>
     *
     * Possible value of the device identifier are:
     * <pre>
     * {
     * &nbsp;&nbsp;0 for local disk
     * &nbsp;&nbsp;1 for the USB key
     * }
     * <pre>
     *
     * The callback will be called with a list of File, each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;url {String} file URL.
     * &nbsp;&nbsp;rights {String} file access rights. Possible users: USER,GROUP,OTHER. Possible access rights for each user: 'r', 'w', 'x'.
     * &nbsp;&nbsp;size {Number} total size of the file in bytes.
     * &nbsp;&nbsp;device {String} device in which the file is located: 'USERSPACE', 'System' and 'USB'.
     * &nbsp;&nbsp;category {Object} the file category in which the current file is classified.
     * &nbsp;&nbsp;typeFile {Object} type of the file.
     * &nbsp;&nbsp;isLink {Number} defines if a file is a link or not. Possible values: -1 value for an empty file object, 0 the file is not a link, 1 the file is a link.
     * &nbsp;&nbsp;name {String} short name of the file, e.g. without it's full path.
     * &nbsp;&nbsp;streaming {Boolean} if TRUE the file is a streaming source, FALSE otherwise.
     * }
     * </pre>
     *
     * The typeFile parameter has the following structure :
     *
     * <pre>
     * {
     * &nbsp;&nbsp;idFileType {Number} identifier of the current file type
     * &nbsp;&nbsp;extension {String} extension associated to the current file type (ex: mp3)
     * &nbsp;&nbsp;header {String} associate header (magic number) for the current file type (ex: audio/mpeg)
     * &nbsp;&nbsp;category {Object} associated file category to the current file type
     * }
     * </pre>
     *
     * The category parameter has the following structure :
     *
     * <pre>
     * {
     * &nbsp;&nbsp;idFileCategory {Number} associated identifier for the current file category
     * &nbsp;&nbsp;caregoryName {String} name of the current category
     * &nbsp;&nbsp;categoryOrder {Number} order of the current category
     * &nbsp;&nbsp;deletable {Boolean} defines if a category can be deleted or not. Some categories are system defined and can not be modified by the user.
     * }
     * </pre>
     */
	getFilesWithFilter : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "getFilesWithFilter",
            context : args.context,
            params: args.params,
            paramType: ["QString","int","int", "QString", "QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>result status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method returns the total number of files present in category 'nameCategory'</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the name of the category.
     *
     * <pre>
     * {String} name of the category
     * </pre>
     *
     * The callback will be called with a status. Possible value are:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; 0   total number of files
     * &nbsp;&nbsp; -2 the category does not exist
     * }
     * </pre>
     */
	getFilesInCategorySize : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "getFilesInCategorySize",
            context : args.context,
            params: args.params,
            paramType: ["QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method checks if a file exists according to its path</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the full path of the file.
     *
     * <pre>
     * {String} the full path of the file
     * </pre>
     *
     * The callback will be called with a status. Possible value are:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; true   the file already exists
     * &nbsp;&nbsp; false  it does not exist
     * }
     * </pre>
     */
	fileExists : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "fileExists",
            context : args.context,
            params: args.params,
            paramType: ["QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This methods copy a file to a specified device. If successful, the event {@link #FILE_LIST_CHANGED} is fired.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * Params is an array which contains the following structure :
     *
     * <pre>
     * {
     * &nbsp;&nbsp;{Number} the identifier of the file
     * &nbsp;&nbsp;{Number} the device where the file will be copied
     * &nbsp;&nbsp;{Boolean} if the destination file already exists, the FileManager will overwrite it if overwrite is true.
     * </pre>
     *
     * Possible value of the device identifier are:
     * <pre>
     * {
     * &nbsp;&nbsp;0 for local disk
     * &nbsp;&nbsp;1 for the USB key
     * }
     * <pre>
     *
     * The callback will be called with a status. Possible value are:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; 0   copy successful
     * &nbsp;&nbsp; -1  technical problem
     * &nbsp;&nbsp; -3  unknown file identifier
     * &nbsp;&nbsp; -6  the permission is denied
     * &nbsp;&nbsp; -8  the file is a link
     * &nbsp;&nbsp; -9  the file is already on device
     * &nbsp;&nbsp; -10 the USB key is not connected
     * &nbsp;&nbsp; -11 unknown device identifier
     * &nbsp;&nbsp; -12 the device is full
     * &nbsp;&nbsp; -13 the file exists but overwrite forbidden
     * }
     * </pre>
     */
	copyFileToDevice : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "copyFileToDevice",
            context : args.context,
            params: args.params,
            paramType: ["int","int","bool"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method adds a new file type which will be managed by the category.<br>
     * All the files of these type will be displayed in the selected category.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * Params is an array which contains the following structure :
     *
     * <pre>
     * {
     * &nbsp;&nbsp;{String} the name of the category
     * &nbsp;&nbsp;{String}  the filname extension (ex : wav, gif ,...)
     * &nbsp;&nbsp;{String} the magic number of this type of file (ex: audio/x-wav, image/gif, ...)
     * }
     * </pre>
     *
     * The callback will be called with a status. Possible value are:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; 0   insertion successful
     * &nbsp;&nbsp; -2  the category does not exist
     * &nbsp;&nbsp; -18  the file type already exists
     * }
     * </pre>
     */
	addFileTypeToCategory : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "addFileTypeToCategory",
            context : args.context,
            params: args.params,
            paramType: ["QString","QString","QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method adds a new pool. This new pool can be used by webapp application.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * Params contains the name of the pool.
     *
     * <pre>
     * {String} the name of the pool, <b>the length of the name of the pool must be less or equal to 45 characters</b>
     * </pre>
     *
     * The callback will be called with a status. Possible value are:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; 0   insertion successful
     * &nbsp;&nbsp; -16  the pool already exists
     * }
     * </pre>
     */
	addPool : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "addPool",
            context : args.context,
            params: args.params,
            paramType: ["QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method checks if a file is stored into a pool</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * Params is an array which contains the following structure :
     *
     * <pre>
     * {
     * &nbsp;&nbsp;{String} the name of the pool
     * &nbsp;&nbsp;{Number} :the File identifier
     * }
     * </pre>
     *
     * The callback will be called with a status. Possible value are:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; 1   the file is in pool
     * &nbsp;&nbsp; 0   the file does not exist in pool
     * &nbsp;&nbsp; -1  technical problem
     * &nbsp;&nbsp; -7  the file does not exist
     * &nbsp;&nbsp; -17 the pool does not exist
     * }
     * </pre>
     */
	fileExistsInPool : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "fileExistsInPool",
            context : args.context,
            params: args.params,
            paramType: ["QString", "int"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method save a link into the FileManager
     * If the operation is successful , the event {@link #FILE_LIST_CHANGED} is sent.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * Params is an array which contains the following structure :
     *
     * <pre>
     * {
     * &nbsp;&nbsp;{String} the url of the link
     * &nbsp;&nbsp;{String} :the category associated with the link
     * &nbsp;&nbsp;{String} the name of the link
     * }
     * </pre>
     *
     * The callback will be called with a status. Possible value are:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; 0   insertion successful
     * &nbsp;&nbsp; -1  technical problem
     * &nbsp;&nbsp; -2  the category does not exist
     * &nbsp;&nbsp; -4  the url already exists
     * &nbsp;&nbsp; -5  the name already existst
     * }
     * </pre>
     */
	putLinkUnderFMControl : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "putLinkUnderFMControl",
            context : args.context,
            params: args.params,
            paramType: ["QString", "QString", "QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method copies the File on the local disk of the device.
     * If the operation is successful , the event {@link #FILE_LIST_CHANGED} is sent.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * Params contains the full path of the File.
     *
     * <pre>
     * {String} the full path of the File
     * </pre>
     *
     * The callback will be called with a status. Possible value are:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; 0   identifier of the File
     * &nbsp;&nbsp; -1  technical problem
     * &nbsp;&nbsp; -5  the name of the file already exists
     * &nbsp;&nbsp; -7  the file does not exist
     * &nbsp;&nbsp; -9  the file is already stored on the device
     * &nbsp;&nbsp; -12  the device is full
     * }
     * </pre>
     */
	putUnderFMControl : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "putUnderFMControl",
            context : args.context,
            params: args.params,
            paramType: ["QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },
	///////// TODO
	modifyLink : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "modifyLink",
            context : args.context,
            params: args.params,
            paramType: ["int", "QString", "QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method adds a file in a pool.<br>
     * The file will be both displayed in his category and in the pool.
	 * One file can be inserted in several pools.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * Params is an array which contains the following structure :
     *
     * <pre>
     * {
     * &nbsp;&nbsp;{String} the name of the pool
     * &nbsp;&nbsp;{String} :the path of the file
     * }
     * </pre>
     *
     * The callback will be called with a status. Possible value are:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; 0   insertion successful
     * &nbsp;&nbsp; -6  permission denied
     * &nbsp;&nbsp; -7  the file does not exist
     * &nbsp;&nbsp; -17 the pool does not exist
     * }
     * </pre>
     */
	addFileInPool : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "addFileInPool",
            context : args.context,
            params: args.params,
            paramType: ["QString", "QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method removes a File from a dedicated Pool.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * Params is an array which contains the following structure :
     *
     * <pre>
     * {
     * &nbsp;&nbsp;{String} the name of the pool
     * &nbsp;&nbsp;{String} the full path of the File to removes
     * }
     * </pre>
     *
     * The callback will be called with a status. Possible value are:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; 2   technical problem
     * &nbsp;&nbsp; 0   deletion successful
     * &nbsp;&nbsp; -6  permission denied: can not remove the file
     * &nbsp;&nbsp; -7  the file does not exist
     * &nbsp;&nbsp; -17 the pool does not exist
     * }
     * </pre>
     */
	removeFileFromPool : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "removeFileFromPool",
            context : args.context,
            params: args.params,
            paramType: ["QString", "QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>list of File</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method gets all the File stored into the pool passed as an argument.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * Params is an array which contains the following structure :
     *
     * <pre>
     * {
     * &nbsp;&nbsp;{String} the name of the pool
     * &nbsp;&nbsp;{Number} index of the first File to return (0 for the first one), has to be positive.
     * &nbsp;&nbsp;{Number} the number of File to return, has to be positive.
     * }
     * </pre>
     *
     * The callback will be called with a list of File, each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;url {String} file URL.
     * &nbsp;&nbsp;rights {String} file access rights. Possible users: USER,GROUP,OTHER. Possible access rights for each user: 'r', 'w', 'x'.
     * &nbsp;&nbsp;size {Number} total size of the file in bytes.
     * &nbsp;&nbsp;device {String} device in which the file is located: 'USERSPACE', 'System' and 'USB'.
     * &nbsp;&nbsp;category {Object} the file category in which the current file is classified.
     * &nbsp;&nbsp;typeFile {Object} type of the file.
     * &nbsp;&nbsp;isLink {Number} defines if a file is a link or not. Possible values: -1 value for an empty file object, 0 the file is not a link, 1 the file is a link.
     * &nbsp;&nbsp;name {String} short name of the file, e.g. without it's full path.
     * &nbsp;&nbsp;streaming {Boolean} if TRUE the file is a streaming source, FALSE otherwise.
     * }
     * </pre>
     *
     * The typeFile parameter has the following structure :
     *
     * <pre>
     * {
     * &nbsp;&nbsp;idFileType {Number} identifier of the current file type
     * &nbsp;&nbsp;extension {String} extension associated to the current file type (ex: mp3)
     * &nbsp;&nbsp;header {String} associate header (magic number) for the current file type (ex: audio/mpeg)
     * &nbsp;&nbsp;category {Object} associated file category to the current file type
     * }
     * </pre>
     *
     * The category parameter has the following structure :
     *
     * <pre>
     * {
     * &nbsp;&nbsp;idFileCategory {Number} associated identifier for the current file category
     * &nbsp;&nbsp;caregoryName {String} name of the current category
     * &nbsp;&nbsp;categoryOrder {Number} order of the current category
     * &nbsp;&nbsp;deletable {Boolean} defines if a category can be deleted or not. Some categories are system defined and can not be modified by the user.
     * }
     * </pre>
     */
    getFilesInPool : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "getFilesInPool",
            context : args.context,
            callback : args.callback,
			params	: args.params,
			paramType : ["QString","int","int"],
            callbackParams: args.callbackParams
        });
        args = null;
    },

    getPoolSize : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "getPoolSize",
            context : args.context,
            callback : args.callback,
			params	: args.params,
			paramType : ["QString"],
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>FileCategory</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method gets the FileCategory for a file thanks to the full path of the file.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * Params params contains the full path of the File.
     *
     * <pre>
     * {String} the full path of the File
     * </pre>
     *
     * The callback will be called with a FileCategory which contains the following structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;idFileCategory {Number} associated identifier for the current file category
     * &nbsp;&nbsp;caregoryName {String} name of the current category
     * &nbsp;&nbsp;categoryOrder {Number} order of the current category
     * &nbsp;&nbsp;deletable {Boolean} defines if a category can be deleted or not. Some categories are system defined and can not be modified by the user.
     * }
     * </pre>
     */
    getCategoryForFile : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "getCategoryForFile",
            context : args.context,
            callback : args.callback,
			params	: args.params,
			paramType : ["QString"],
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>a list of FileCategory</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method gets all categories of file that are available between startInd and length
     * NB: if the index is incorrect, an empty list will be return. The startInd must be positive.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * Params is an array which contains the following structure :
     *
     * <pre>
     * {
     * &nbsp;&nbsp;{Number} index of the first category to return (0 for the first one)
     * &nbsp;&nbsp;{Number} the number of categories to return
     * }
     * </pre>
     *
     * The callback will be called with a list of FileCategory, each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;idFileCategory {Number} associated identifier for the current file category
     * &nbsp;&nbsp;caregoryName {String} name of the current category
     * &nbsp;&nbsp;categoryOrder {Number} order of the current category
     * &nbsp;&nbsp;deletable {Boolean} defines if a category can be deleted or not. Some categories are system defined and can not be modified by the user.
     * }
     * </pre>
     */
    getAllFileCategories : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "getAllFileCategories",
            context : args.context,
            callback : args.callback,
			params	: args.params,
			paramType : ["int","int"],
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>String</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method gets all the magic numbers which are stored under the same category.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * Params params contains the name of the category.
     *
     * <pre>
     * {String} the name of the category
     * </pre>
     *
     * The callback will be called with a String.
     *
     * <pre>
     * {String} a list of magic number seperated by the special character ";", for instance : "application/ogg;audio/mpeg;audio/x-wav").
     * </pre>
     */
    getExtensions : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "getExtensions",
            context : args.context,
            callback : args.callback,
			params	: args.params,
			paramType : ["QString"],
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>the file</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method returns the file corresponding to the identifier passed as an argument.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * Params contains the identifier of the file to search:
     *
     * <pre>
     * {Number} the identifier of the file to search
     * </pre>
     *
     * The callback will be called with a list of File, each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;url {String} file URL.
     * &nbsp;&nbsp;rights {String} file access rights. Possible users: USER,GROUP,OTHER. Possible access rights for each user: 'r', 'w', 'x'.
     * &nbsp;&nbsp;size {Number} total size of the file in bytes.
     * &nbsp;&nbsp;device {String} device in which the file is located: 'USERSPACE', 'System' and 'USB'.
     * &nbsp;&nbsp;category {Object} the file category in which the current file is classified.
     * &nbsp;&nbsp;typeFile {Object} type of the file.
     * &nbsp;&nbsp;isLink {Number} defines if a file is a link or not. Possible values: -1 value for an empty file object, 0 the file is not a link, 1 the file is a link.
     * &nbsp;&nbsp;name {String} short name of the file, e.g. without it's full path.
     * &nbsp;&nbsp;streaming {Boolean} if TRUE the file is a streaming source, FALSE otherwise.
     * }
     * </pre>
     *
     * The typeFile parameter has the following structure :
     *
     * <pre>
     * {
     * &nbsp;&nbsp;idFileType {Number} identifier of the current file type
     * &nbsp;&nbsp;extension {String} extension associated to the current file type (ex: mp3)
     * &nbsp;&nbsp;header {String} associate header (magic number) for the current file type (ex: audio/mpeg)
     * &nbsp;&nbsp;category {Object} associated file category to the current file type
     * }
     * </pre>
     *
     * The category parameter has the following structure :
     *
     * <pre>
     * {
     * &nbsp;&nbsp;idFileCategory {Number} associated identifier for the current file category
     * &nbsp;&nbsp;caregoryName {String} name of the current category
     * &nbsp;&nbsp;categoryOrder {Number} order of the current category
     * &nbsp;&nbsp;deletable {Boolean} defines if a category can be deleted or not. Some categories are system defined and can not be modified by the user.
     * }
     * </pre>
     */
    getFile : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "getFile",
            context : args.context,
            callback : args.callback,
			params	: args.params,
			paramType : ["int"],
            callbackParams: args.callbackParams
        });
        args = null;
    },

	/*
	 * @ignore
	 */
	getDefaultAlarmMelody : function(args) {
		this.callCoreApplicationMethod({
            domainName : "IctFileManager",
            methodName : "getDefaultAlarmMelody",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	}

/**
 * @event FILE_LIST_CHANGED
 * This event is used to notify that several files have been changed.
 * @param {Object} param contains the list of the file categories, each of them following this structure:
 * <pre>
 * {
 * &nbsp;&nbsp;idFileCategory {Number} associated identifier for the current file category
 * &nbsp;&nbsp;caregoryName {String} name of the current category
 * &nbsp;&nbsp;categoryOrder {Number} order of the current category
 * &nbsp;&nbsp;deletable {Boolean} defines if a category can be deleted or not. Some categories are system defined and can not be modified by the user.
 * }
 * </pre>
 */

/**
 * @event FILE_DELETED
 * This event is used to notify that a File has been deleted.
 * @param {Number} param contains the identifier of the file that has been deleted.
 */

});

ICTouchAPI.APIServices.FileManager = new ICTouchAPI.APIServices.FileManager();
/**
 * @class ICTouchAPI.APIServices.GreenManager
 * @singleton
 * @extends ICTouchAPI.APIServices.APIServicesBase
 *  This API manages the screensaver.
 */

dojo.provide("ICTouchAPI.APIServices.GreenManager");
dojo.declare("ICTouchAPI.APIServices.GreenManager",[ICTouchAPI.APIServices.APIServicesBase],
{
    setScreensaverState : function(args) {
        this.callCoreApplicationMethod({
            domainName : "GreenManager",
            methodName : "setScreensaverState",
            context : args.context,
            params: args.params,
            paramType: ["bool"],
            callback : args.callback
        });
        args = null;
    }

/**
 * @event Screensaver_Off
 * This event occurs when the screensaver is displayed and when the user touchs the screen.
 */
    
/**
 * @event Screensaver_On
 * This event is sent when the screensaver delay has expired (no screen interaction during this delay)
 */
 });



ICTouchAPI.APIServices.GreenManager = new ICTouchAPI.APIServices.GreenManager();
/**
 * @ignore
 */
dojo.provide("ICTouchAPI.APIServices.MediaPlayer");
dojo.declare("ICTouchAPI.APIServices.MediaPlayer",[ICTouchAPI.APIServices.APIServicesBase],
{
    moduleLaunch : function(args) {
        this.callCoreApplicationMethod({
            domainName : "MediaPlayer",
            methodName : "moduleLaunch",
            context : args.context,
			params: args.params,
			paramType: ["QString","int","int","int","int"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    moduleVolume : function(args) {
        this.callCoreApplicationMethod({
            domainName : "MediaPlayer",
            methodName : "moduleVolume",
            context : args.context,
            params: args.params,
            paramType: ["double"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    moduleRewind : function(args) {
        this.callCoreApplicationMethod({
            domainName : "MediaPlayer",
            methodName : "moduleRewind",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    modulePlayPause : function(args) {
        this.callCoreApplicationMethod({
            domainName : "MediaPlayer",
            methodName : "modulePlayPause",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    moduleStop : function(args) {
        this.callCoreApplicationMethod({
            domainName : "MediaPlayer",
            methodName : "moduleStop",
            context : args.context,
            params: args.params,
            paramType: ["int"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    moduleMute : function(args) {
        this.callCoreApplicationMethod({
            domainName : "MediaPlayer",
            methodName : "moduleMute",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    moduleUnmute : function(args) {
        this.callCoreApplicationMethod({
            domainName : "MediaPlayer",
            methodName : "moduleUnmute",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    playAlarm : function(args) {
        this.callCoreApplicationMethod({
            domainName : "MediaPlayer",
            methodName : "playAlarm",
            context : args.context,
			params: args.params,
			paramType: ["int","QString","double","bool"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    launchMP : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctMPInterface",
            methodName : "moduleLaunch",
            context : args.context,
			params: args.params,
			paramType: ["int","QString","int","int","int","int","double","bool","qlonglong"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    playMedia : function(args) {
	    this.callCoreApplicationMethod({
		    domainName : "IctMPInterface",
		    methodName : "playMedia",
		    context : args.context,
		    params: args.params,
		    paramType: ["int","QString"],
		    callback : args.callback,
		    callbackParams: args.callbackParams
	    });
	    args = null;
    },

    playMediaWithSeek : function(args) {
	    this.callCoreApplicationMethod({
		    domainName : "IctMPInterface",
		    methodName : "playMediaWithSeek",
		    context : args.context,
		    params: args.params,
		    paramType: ["int","QString","qlonglong"],
		    callback : args.callback,
		    callbackParams: args.callbackParams
	    });
	    args = null;
    },
    getMediaSessionInfos : function(args) {
	this.callCoreApplicationMethod({
		domainName : "IctMPInterface",
		methodName : "getMediaSessionInfos",
		context: args.context,
		params: args.params,
		paramType: [],
		callback: args.callback,
		callbackParams: args.callbackParams
	});
	args = null;
    },

    playPauseMP : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctMPInterface",
            methodName : "moduleLaunch",
            context : args.context,
			params: args.params,
			paramType: ["int"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    rewindMP : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctMPInterface",
            methodName : "moduleLaunch",
            context : args.context,
			params: args.params,
			paramType: ["int"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    stopMP : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctMPInterface",
            methodName : "moduleLaunch",
            context : args.context,
			params: args.params,
			paramType: ["int"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    setVolumeMP : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctMPInterface",
            methodName : "moduleLaunch",
            context : args.context,
			params: args.params,
			paramType: ["double"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    muteMP : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctMPInterface",
            methodName : "moduleLaunch",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    unMuteMP : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctMPInterface",
            methodName : "moduleLaunch",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    }

});

ICTouchAPI.APIServices.MediaPlayer = new ICTouchAPI.APIServices.MediaPlayer();
/**
 * @ignore
 */
dojo.provide("ICTouchAPI.APIServices.SettingsManager");
dojo.declare("ICTouchAPI.APIServices.SettingsManager",[ICTouchAPI.APIServices.APIServicesBase],
{
    getMainCategories : function(args) {
        this.callCoreApplicationMethod({
            domainName : "SettingsManager",
            methodName : "getMainCategories",
            context : args.context,
            params: args.params,
            paramType: ["QString"],
            callback : args.callback,
            errorCallback : args.errorCallback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    getCategories : function(args) {
        this.callCoreApplicationMethod({
            domainName : "SettingsManager",
            methodName : "getCategories",
            context : args.context,
            params: args.params,
            paramType: ["QString", "QString"],
            callback : args.callback,
            errorCallback : args.errorCallback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    getSettings : function(args) {
        this.callCoreApplicationMethod({
            domainName : "SettingsManager",
            methodName : "getSettings",
            context : args.context,
            params: args.params,
            paramType: ["QString", "QString", "QString"],
            callback : args.callback,
            errorCallback : args.errorCallback,
            callbackParams: args.callbackParams
        });
        args = null;
    },


	getMainCategoriesWithCheck : function(args) {
        this.callCoreApplicationMethod({
            domainName : "SettingsManager",
            methodName : "getMainCategoriesWithCheck",
            context : args.context,
            params: args.params,
            paramType: ["QString"],
            callback : args.callback,
            errorCallback : args.errorCallback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    getCategoriesWithCheck : function(args) {
        this.callCoreApplicationMethod({
            domainName : "SettingsManager",
            methodName : "getCategoriesWithCheck",
            context : args.context,
            params: args.params,
            paramType: ["QString", "QString"],
            callback : args.callback,
            errorCallback : args.errorCallback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    getSetting : function(args) {
        this.callCoreApplicationMethod({
            domainName : "SettingsManager",
            methodName : "getSetting",
            context : args.context,
            params: args.params,
            paramType: ["QString"],
            callback : args.callback,
            callbackParams: args.callbackParams,
			sync : (args.sync ? args.sync : false)
        });
        args = null;
    },

    getSettingValue : function(args) {
        this.callCoreApplicationMethod({
            domainName : "SettingsManager",
            methodName : "getSettingValue",
            context : args.context,
            params: args.params,
            paramType: ["QString"],
            callback : args.callback,
            callbackParams: args.callbackParams,
			sync : (args.sync ? args.sync : false)
        });
        args = null;
    },

    setSetting : function(args) {
        this.callCoreApplicationMethod({
            domainName : "SettingsManager",
            methodName : "setSetting",
            context : args.context,
            params: args.params,
            paramType: ["Setting"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    setSettingValue : function(args) {
        this.callCoreApplicationMethod({
            domainName : "SettingsManager",
            methodName : "setSettingValue",
            context : args.context,
            params: args.params,
            paramType: ["QString", "QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    //called to check weither we upgrade some settings before to enter in the webapp setting
    //the platform lock the webapp acces
    enterMMI : function(args) {
        this.callCoreApplicationMethod({
            domainName : "SettingsManager",
            methodName : "enterMMI",
            context : args.context,
            callback : args.callback
        });
        args = null;
    },

    //send an event expected by the settingManager to leave the webapp
    exitMMI : function() {
        ICTouchAPI.APIServices.ICTGate.notifyEventPresentation({
            params:["exitMMI"]
        });
    },

	needReboot : function(args) {

        this.callCoreApplicationMethod({
            domainName : "SettingsManager",
            methodName : "needReboot",
            context : args.context,
            callback : args.callback
        });
        args = null;
    }
});

ICTouchAPI.APIServices.SettingsManager = new ICTouchAPI.APIServices.SettingsManager();
/**
 * @class ICTouchAPI.APIServices.Telephony
 * @singleton
 * @extends ICTouchAPI.APIServices.APIServicesBase
 * The telephony module is responsible for the general status
 * of the telephony on the phone. It provides all the services that are related to communication services.
 * <br><br>
 * Depending on the status of the phone, each feature is available or not.<br>
 * The information to know if a feature is available at a given moment is provided
 * by a capabilities system. So a webapp that is wanting to use a feature needs
 * first to check if the according capability is available.
 * <br>For example, if a webapp provides the "makeCall" feature, it should only make
 * it available when the "makeCall" capability is available.
 * <br><br>
 * The Telephony class is used as a singleton that carries the main phoning
 * state machine.<br>
 * The Telephony class is the owner of the PhoneCall class instances, it
 * creates and destroys them depending on the life of the according phone calls.<br>
 * The singleton reacts to the telephonic events from the SIPUA, to the service calls
 * from the presentation and provides events that describe the status of the telephony
 * module to all other modules that are connected on the DBUS. The general status of
 * the telephony is handled by a state machine that is owned by this class. The status
 * of each individual phone call is handled by a specific state machine owned by each
 * instance of the class PhoneCall.
 */
 
 
dojo.provide("ICTouchAPI.APIServices.Telephony");
dojo.declare("ICTouchAPI.APIServices.Telephony",[ICTouchAPI.APIServices.APIServicesBase],
{
	
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>list of the current phoneCalls</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Get the list of current phoneCalls.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with the list of the current phoneCalls,each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
     * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
     * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
     * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
     * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
     * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
     * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
     * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
     * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
     * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
     * &nbsp;&nbsp;startTime {Date} Start time
     * &nbsp;&nbsp;isConference {Boolean} Conferencing status
     * &nbsp;&nbsp;isConnected {Boolean}Connection status
     * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
     * &nbsp;&nbsp;isHeld {Boolean} Held Status
     * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
     * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
     * &nbsp;&nbsp;isActive {Boolean} Active Status
     * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
     * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
     * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
     * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
     * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
     * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
     * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
     * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
     * &nbsp;&nbsp;displayName {Boolean} Display Name
     * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
     * }
     * </pre>
     *
     * The parameter callId has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The parameter phoneNumber has the following structure:
     * <pre>
     * {String} sip phone number URI
     * </pre>  
     *
     * The parameter callDirection has the following structure:
     * <pre>
     * {String} call direction from the values,possilbe values are:
     * -"CALL_INCOMING" for incoming call
     * -"CALL_OUTGOING" for outgoing call
     * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
     * </pre>  
     *
     * The parameter startReason and endReason have the following structure:
     * <pre>
     * {String} call reason from the values,possilbe values are:
     * -"CALL_REASON_NORMAL" for normal call
     * -"CALL_REASON_TRANSFERRED" for transferred call
     * </pre>
     * @ignore
     */	
    getPhoneCalls : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "getPhoneCalls",
            context : args.context,
            callback : args.callback,
            errorCallback : args.errorCallback,
            callbackParams: args.callbackParams
        });
        args = null;
    },


    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>list of Ids of the current phoneCalls</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Get the list of current phoneCalls.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with the list of Ids of the current phoneCalls,each of them following this structure:
     *
     * <pre>
     * {String}  phone call identifier 
     * </pre>
     * @ignore
     */	
    getPhoneCallIds : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "getPhoneCallIds",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>data of the phoneCall</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Get phone call data from one phoneCall.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the identifier of the phoneCall,it has the following structure:
     * <pre>
     * {String}  phone call identifier 
     * </pre>
     
     * The callback will be called with the data of the phoneCall,following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
     * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
     * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
     * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
     * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
     * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
     * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
     * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
     * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
     * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
     * &nbsp;&nbsp;startTime {Date} Start time
     * &nbsp;&nbsp;isConference {Boolean} Conferencing status
     * &nbsp;&nbsp;isConnected {Boolean}Connection status
     * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
     * &nbsp;&nbsp;isHeld {Boolean} Held Status
     * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
     * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
     * &nbsp;&nbsp;isActive {Boolean} Active Status
     * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
     * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
     * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
     * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
     * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
     * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
     * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
     * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
     * &nbsp;&nbsp;displayName {Boolean} Display Name
     * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
     * }
     * </pre>
     *
     * The parameter callId has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The parameter phoneNumber has the following structure:
     * <pre>
     * {String} sip phone number URI
     * </pre>  
     *
     * The parameter callDirection has the following structure:
     * <pre>
     * {String} call direction from the values,possilbe values are:
     * -"CALL_INCOMING" for incoming call
     * -"CALL_OUTGOING" for outgoing call
     * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
     * </pre>  
     *
     * The parameter startReason and endReason have the following structure:
     * <pre>
     * {String} call reason from the values,possilbe values are:
     * -"CALL_REASON_NORMAL" for normal call
     * -"CALL_REASON_TRANSFERRED" for transferred call
     * </pre>
     * @ignore
     */	

    getPhoneCallFromId : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "getPhoneCallFromId",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>information of the started phoneCall</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Start an outgoing phone Call.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp; phoneNumber {Object} the phoneNumber identifier of the phoneCall.See the description below
     * &nbsp;&nbsp; hideIdentity {Boolean} indicates if the user explicitly asked to hide his identity : true to hide, false to use the default value deduced from the setting "IdentityRestriction"(false by default).
     * &nbsp;&nbsp; doDialableNumber {Boolean} if set to true, convert the given phone number(which may contain non-digit signs like ")", "+", ...) to a number that fits to the numbering plan.
     * &nbsp;&nbsp; bDTMF {Boolean} indicates if the DTMF keypad should be automatically opened on successful connection
     * }
     * </pre>
     *
     * The parameter phoneNumber has the following structure:
     * <pre>
     * {String} sip phone number URI
     * </pre>  
     *
     * The callback will be called with the data of the phoneCall,following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
     * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
     * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
     * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
     * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
     * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
     * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
     * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
     * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
     * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
     * &nbsp;&nbsp;startTime {Date} Start time
     * &nbsp;&nbsp;isConference {Boolean} Conferencing status
     * &nbsp;&nbsp;isConnected {Boolean}Connection status
     * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
     * &nbsp;&nbsp;isHeld {Boolean} Held Status
     * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
     * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
     * &nbsp;&nbsp;isActive {Boolean} Active Status
     * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
     * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
     * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
     * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
     * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
     * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
     * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
     * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
     * &nbsp;&nbsp;displayName {Boolean} Display Name
     * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
     * }
     * </pre>
     *
     * The parameter callId has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The parameter phoneNumber has the following structure:
     * <pre>
     * {String} sip phone number URI
     * </pre>  
     *
     * The parameter callDirection has the following structure:
     * <pre>
     * {String} call direction from the values,possilbe values are:
     * -"CALL_INCOMING" for incoming call
     * -"CALL_OUTGOING" for outgoing call
     * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
     * </pre>  
     *
     * The parameter startReason and endReason have the following structure:
     * <pre>
     * {String} call reason from the values,possilbe values are:
     * -"CALL_REASON_NORMAL" for normal call
     * -"CALL_REASON_TRANSFERRED" for transferred call
     * </pre>    
     */	
    startPhoneCall : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "startPhoneCall",
            context : args.context,
            params: args.params,
            paramType: ["PhoneNumber", "bool", "bool", "bool"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Decprecated
	 * Use startPhoneCall instead.
     * @ignore
     */
    startPhoneCallWithDialableNumber : function(args) {
		this._deprecated("startPhoneCallWithDialableNumber", "startPhoneCall", arguments);
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Answer to an incoming phone call.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;callId {Object} the identifier of a phoneCall answer.See the description below
     * &nbsp;&nbsp;hide {Boolean} indicates if the user explicitly asked to hide his identity : true to hide, false to use the default value deduced from the setting "IdentityRestriction".
     * }
     * </pre>
     *
     * The parameter callId has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */

    answerPhoneCall : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "answerPhoneCall",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId", "bool"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Ignore an incoming phone call. The phone is ringing but the user does not hook off after a few time the phone call is forwarded to the VM.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the identifier of the phoneCall to be ignored,it has the following structure:
     * <pre>
     * {String} the identifier of the phoneCall
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    ignoreCall : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "ignoreCall",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Reject an incoming phone call.This operation is provided to be able to stop an incoming call without hooking off the phone. While the phone is ringing, the user presses a button to stop the communication.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the identifier of a phoneCall rejected,it has the following structure:
     * <pre>
     * {String} the identifier of the phoneCall
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    rejectCall : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "rejectCall",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br> Stop a phoneCall.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the identifier of the phoneCall to be stopped,it has the following structure:
     * <pre>
     * {String} the identifier of the phoneCall
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    stopPhoneCall : function(args) {
        /*Call the method directly using IctWebClient module*/
        if (typeof TelephonyAPIServices != "undefined" && TelephonyAPIServices.stopPhoneCall) {
            TelephonyAPIServices.stopPhoneCall(args.params);
        }
        else {
            this.callCoreApplicationMethod({
                domainName : "Telephony",
                methodName : "stopPhoneCall",
                context : args.context,
                params: args.params,
                paramType: ["PhoneCallId"],
                callback : args.callback,
                callbackParams: args.callbackParams
            });
        }
        args = null;
    },


    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br> Start a conference between the first CallId c1 and the second CallId c2.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp; c1 {Object} the identifier of the first participant of the conference.See the description below.
     * &nbsp;&nbsp; c2 {Object} the identifier of the second participant of the conference.See the description below.
     * }
     * </pre>
     *  
     * params c1 and c2 have the following structure:
     * <pre>
     * {String} the identifier of the phoneCall
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    startConference : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "startConference",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId", "PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },


    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Add a participant to a conference.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp; callId  {Object} the identifier of the conference call.See the description below.
     * &nbsp;&nbsp; phoneNumber {Object} the phonenumber of the participant to add.See the description below.
     * &nbsp;&nbsp; makeDialableNumber {Boolean} if set to true, convert the given phone number(which may contain non-digit signs like ")", "+", ...) to a number that fits to the numbering plan.
     * }
     * </pre>
     *  
     * params callId has the following structure:
     * <pre>
     * {String} the identifier of the phoneCall
     * </pre>  
     *
     * The parameter phoneNumber has the following structure:
     * <pre>
     * {String} sip phone number URI
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    addParticipant : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "addParticipant",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId", "PhoneNumber", "bool"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

	transferConferenceToNumber : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "transferConferenceToNumber",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId", "QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },


    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Stop the current conference.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	stopConference : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "stopConference",
			context: args.context,
			params: {},
			callback: args.callback
		});
		args = null;
	},

	leaveConferenceN : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "leaveConferenceN",
			context: args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
			callback: args.callback
		});
		args = null;
	},
	
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br> End a conference.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the identifier of the conference call,it has the following structure:
     * <pre>
     * {String} the identifier of the conference call
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */

	stopConferenceN : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "stopConferenceN",
			context: args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
			callback: args.callback
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Drop a participant from a conference.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp; participantId {String} the id of the participants to drop (only one for the moment)
     * &nbsp;&nbsp; callId {Object} the identifier of the conference call.See the description below.
     * }
     * </pre>
     *
     * The parameter callId has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	dropParticipant : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "dropParticipant",
			context: args.context,
            params: args.params,
            paramType: ["QString", "PhoneCallId"],
			callback: args.callback
		});
		args = null;
	},


    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Deactivate the local (SIP) forward.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     */
    deactivateForward : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "deactivateForward",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Activate a forward.</b>
     * <br> The automatic "Forward" feature is set to the destination phone number.
     * <br> When an incoming call arrives, it is automatically forwarded depending on the type of forward set.
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the forward definition in a string form
     * <pre>
     * {String} the forward definition in a string form
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     */
    activateForward : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "activateForward",
            context : args.context,
            params: args.params,
            paramType: ["QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Activate a forward.</b>
     * <br> The automatic "Forward" feature is set to the destination phone number.
     * <br> When an incoming call arrives, it is automatically forwarded depending on the type of forward set.
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the forward type in a string form
     * <pre>
     * {String} the forward type in a string form
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    activateForwardDest : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "activateForwardDest",
            context : args.context,
            params: args.params,
            paramType: ["QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Implements a "Consult transfer".An existing phoneCall is transfered to an other existing phoneCall.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp; callId1 {Object} the identifier of the phoneCall to be transferred.See the description below.
     * &nbsp;&nbsp; callId2 {Object} the identifier of the phoneCall to be transferred.See the description below.
     * }
     * </pre>
     *  
     * params callId1 and callId2 have the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	transferPhoneCall : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "transferPhoneCall",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId", "PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },


    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Merge two calls into a conference.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp; callIdActive {Object} the identifier of the current active call.See the description below.
     * &nbsp;&nbsp; callIdHeld {Object} the identifier of the call to merge with the active call.See the description below.
     * }
     * </pre>
     *  
     * params callIdActive and callIdHeld have the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	mergePhoneCalls : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "mergePhoneCalls",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId", "PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Deflect an incoming phone call to a phone number.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp; callId  {Object} the identifier of the phoneCall to deflect.See the description below.
     * &nbsp;&nbsp; number {Object}  the destination number of the deflect.See the description below.
     * &nbsp;&nbsp; makeDialableNumber {Boolean} if set to true, convert the given phone number(which may contain non-digit signs like ")", "+", ...) to a number that fits to the numbering plan.
     * }
     * </pre>
     *  
     * params callId has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The parameter number has the following structure:
     * <pre>
     * {String} sip phone number URI
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	deflectToNumber : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "deflectToNumber",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId", "PhoneNumber", "bool"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },


    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Implements a "Blind transfer".An existing phoneCall is transfered to a phone number.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp; callId  {Object} the identifier of the existing phoneCall.See the description below.
     * &nbsp;&nbsp; n {Object}  the identifier of the phoneNumber to be transfered to.See the description below.
     * &nbsp;&nbsp; makeDialableNumber {Boolean} if set to true, convert the given phone number(which may contain non-digit signs like ")", "+", ...) to a number that fits to the numbering plan.
     * }
     * </pre>
     *  
     * params callId has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The parameter n has the following structure:
     * <pre>
     * {String} sip phone number URI
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	transferPhoneCallToNumber : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "transferPhoneCallToNumber",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId", "PhoneNumber", "bool"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },


    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Put the incoming phoneCall in hold.The caller is on hold by the callee.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params callId is a JSON, representing the identifier of a phoneCall put on hold,has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    putOnHold : function (args) {
		this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "putOnHold",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Retrieve an incoming phone call that is on hold.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params callId is a JSON, representing the identifier of the entry to be retrieved,has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	retrieveCall : function(args) {
		this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "retrieveCall",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Request the distant to call back.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params callId is a JSON, representing the identifier of the phonecall,has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	requestCallBack : function(args) {
		this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "requestCallBack",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Activate the "Do Not Disturb" feature.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the password 
     * <pre>
     * {String} password for the feature "Do Not Disturb"
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     */
    activateDnd : function(args) {
		args.params[0]	= Encryption.encryptRSA(args.params[0]);
		
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "activateDnd",
            context : args.context,
            params: args.params,
            paramType: ["QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Deactivate the "Do Not Disturb" feature.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     */
    deactivateDnd : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "deactivateDnd",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method sends a DTMF (dual-tone multi-frequency) coded digit into a call,for example when calling a vocal machine that requires answers based on DTMF.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp; callId  {Object} the identifier of the phoneCall destination of DTMF.See the description below.
     * &nbsp;&nbsp; digit {Number}  the DTMF digit (allowed value : 0 to 15)
     * }
     * </pre>
     *  
     * params callId has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     */
    sendDTMF : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "sendDtmf",
            params: args.params,
            paramType: ["PhoneCallId", "int"],
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>This method is to be called when the DTMF key is released.</b>
     * <br> It stops the DTMF tone if AudioDtmfDuration setting is set to "FollowKey"
     * <br>Note: in the mode "FollowKey", a timer ends the DTMF tone if this method is not called by presentation.
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     */
    sendDTMFRelease : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "sendDtmfKeyRelease",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>The presentation initiate a phoneCall,the dialer pad is ready to be used.This action was initiated from GUI, using the action "New Call".</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    initiateOutgoingCallGUI : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "initiateOutgoingCallGUI",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>The presentation initiate a phoneCall,the dialer pad is ready to be used.
     * <br>This action was initiated from Application level, after a physical hook off.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    initiateOutgoingCallBO : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "initiateOutgoingCallBO",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>The presentation shows the dialer pad,a key is pressed for the dialing.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    dialKeyEnter : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "dialKeyEnter",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Cancel an outgoing call.
     * <br>The presentation shows the dialer pad, the dial tone is activated then the phone call is stopped.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    cancelOutgoingCall : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "cancelOutgoingCall",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},


	getForwardTypeList : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "getForwardTypeList",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

	getForwardDestinationList : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "getForwardDestinationList",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>phoneCall </b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br> Create a phoneCall to the system to listen to the voice mail.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with the phoneCall,following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
     * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
     * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
     * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
     * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
     * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
     * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
     * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
     * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
     * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
     * &nbsp;&nbsp;startTime {Date} Start time
     * &nbsp;&nbsp;isConference {Boolean} Conferencing status
     * &nbsp;&nbsp;isConnected {Boolean}Connection status
     * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
     * &nbsp;&nbsp;isHeld {Boolean} Held Status
     * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
     * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
     * &nbsp;&nbsp;isActive {Boolean} Active Status
     * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
     * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
     * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
     * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
     * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
     * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
     * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
     * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
     * &nbsp;&nbsp;displayName {Boolean} Display Name
     * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
     * }
     * </pre>
     *
     * The parameter callId has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The parameter phoneNumber has the following structure:
     * <pre>
     * {String} sip phone number URI
     * </pre>  
     *
     * The parameter callDirection has the following structure:
     * <pre>
     * {String} call direction from the values,possilbe values are:
     * -"CALL_INCOMING" for incoming call
     * -"CALL_OUTGOING" for outgoing call
     * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
     * </pre>  
     *
     * The parameter startReason and endReason have the following structure:
     * <pre>
     * {String} call reason from the values,possilbe values are:
     * -"CALL_REASON_NORMAL" for normal call
     * -"CALL_REASON_TRANSFERRED" for transferred call
     * </pre>    
     */	
	listenVM : function(args) {
        this.callCoreApplicationMethod({
            domainName: "Telephony",
            methodName: "listenVM",
            context: args.context,
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },


    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Overflow an outgoing call to the callee voicemail if it exists.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the identifier of the phoneCall,it has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
        overflowToVM : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "overflowToVM",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Deflect an incoming phone call to the voice mail.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the identifier of the phoneCall,it has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
        deflectToVM : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "deflectToVM",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Deflect an incoming phone call to the DVA.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the identifier of the phoneCall deflected to the DVA,it has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
        deflectToDVA : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "deflectToDVA",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>capability of the module</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the name of the capability
     * <pre>
     * {String} the name of the capability
     * </pre>
     *
     * The callback will be called with the capability, following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;name {String} 
     * &nbsp;&nbsp;state {Number} 
     * }
     * </pre>
     */
	getCapabilityGlobal : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "getCapability",
			context: args.context,
			params: args.params,
			paramType: ["QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},


   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>capability of an object of the module</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;name {String}  Name of the capability
     * &nbsp;&nbsp;listId {Number} ID of the list of capabilities
     * }
     * </pre>
     *
     * The callback will be called with the capability, following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;name {String} 
     * &nbsp;&nbsp;state {Number} 
     * }
     * </pre>
     */
	getCapabilityInstance : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "getCapability",
			context: args.context,
			params: args.params,
			paramType: ["int", "QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *<b><br><br>Pickup an incoming call on another user.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp; otherCallRef {String}  Reference of the call to pickup  (on the remote user)
     * &nbsp; otherPhoneNumber {String}  Phone number on which the call is to pickup.
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	pickupCall : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "pickupCall",
			context: args.context,
			params: args.params,
			paramType: ["QString", "QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *<b><br><br>Intrude into a specified active call in a special mode (secrete or not) of another user.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp; otherCallRef {String} Reference of the call to intrude (on the remote user)
     * &nbsp; otherPhoneNumber {String} Phone number on which the call is to be intruded.
     * &nbsp; hidden {Boolean} True to hide the user for the participants of this call.
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	intrudeCall : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "intrudeCall",
			context: args.context,
			params: args.params,
			paramType: ["QString", "QString", "bool"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *<b><br><br>Get the notification context.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	getNotificationContext : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "getNotificationContext",
			context: args.context,
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

	/*getRemoteCall : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "getRemoteCall",
			context: args.context,
			callback: args.callback,
			callbackParams: args.callbackParams
		});
	},*/

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *<b><br><br>Call the sender of a visual voice mail message.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;  idMessage {String} the identifier of the message.
     * &nbsp;  number {String} the identifier of the message.
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	callMessageSender : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "callMessageSender",
            context : args.context,
            params: args.params,
            paramType: ["QString", "QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *<b><br><br>Forward a visual voice mail message by e-mail.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp; idMessage  {String} the identifier of the message to forward.
     * &nbsp; email {String} the e-mail address of the recipient.
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     */
	forwardMessageByEmail : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "forwardMessageByEmail",
            context : args.context,
            params: args.params,
            paramType: ["QString","QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *<b><br><br> Forward a visual voice mail message to another voice mail.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp; idMessage {Number} the identifier of the message to forward.
     * &nbsp; vmNumber {Object}  the phone number of the recipient.See the description below.
     * }
     * </pre>
     *
     * The parameter phoneNumber has the following structure:
     * <pre>
     * {String} sip phone number URI
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     */
	forwardMessageToVM : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "forwardMessageToVM",
            context : args.context,
            params: args.params,
            paramType: ["QString","PhoneNumber"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *<b><br><br>Start playing a visual voice mail message.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the identifier of the message to play
     * <pre>
     * {String} the identifier of the message 
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     */
	playMessage : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "playMessage",
            context : args.context,
            params: args.params,
            paramType: ["QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },
    
     /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *<b><br><br>Resume a paused visual voice mail message.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp; idMessage {String} the identifier of the message to resume.
     * &nbsp; offset {Number} the position from where the message has to be played again.
     * }
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     */

	resumeMessage : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "resumeMessage",
            context : args.context,
            params: args.params,
            paramType: ["QString","int"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *<b><br><br>Pause a playing visual voice mail message.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the identifier of the message to pause
     * <pre>
     * {String} the identifier of the message 
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     */
	pauseMessage : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "pauseMessage",
            context : args.context,
            params: args.params,
            paramType: ["QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *<b><br><br>Stop playing a visual voice mail message.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the identifier of the message to stop
     * <pre>
     * {String} the identifier of the message 
     * </pre>
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     */
	stopMessage : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "stopMessage",
            context : args.context,
            params: args.params,
            paramType: ["QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Start recording call conversation in voice mail.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the callId of the call,it has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	startCallRecording : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "startCallRecording",
			context: args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
			callback: args.callback
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Pause recording the call conversation.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the callId of the call,it has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	pauseCallRecording : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "pauseCallRecording",
			context: args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
			callback: args.callback
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Resume recording the call conversation.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the callId of the call,it has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	resumeCallRecording : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "resumeCallRecording",
			context: args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
			callback: args.callback
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Stop recording the call conversation.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the callId of the call,it has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
	stopCallRecording : function(args) {
		this.callCoreApplicationMethod({
			domainName: "Telephony",
			methodName: "stopCallRecording",
			context: args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
			callback: args.callback
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>list containing the other devices of the user</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Get the list of user devices.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with the list containing the other devices of the user( i.e all devices except local device), each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; _phoneNumber {String}
     * &nbsp;&nbsp; _label {String}
     * &nbsp;&nbsp; _state {Object} See the description below.
     * &nbsp;&nbsp; _idCapabilities {Number}
     * }
     * </pre>
     *
     * The parameter _state is a {Object},representing enumerated type values, used by applications.
     * Possible values are : 
     * <pre>
     * - "UNKNOWN" for 0 (Init value : device with unknown state)
     * - "IN_SERVICE " 1
     * - "OUT_OF_SERVICE " 2     
     * </pre>
     * @ignore
     */
     
    getOtherDevicesList : function(args) {
        this.callCoreApplicationMethod({
            domainName: "Telephony",
            methodName: "getOtherDevicesList",
            context: args.context,
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Switch de call from one user device to another user device</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp; callId {Object} callId of the call to be switched.See the description below
     * &nbsp;&nbsp; device {String} phone number of the device on which the call is switched
     * }
     * </pre>
     *
     * The parameter callId has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */

    switchCallOnDevice : function(args) {
        this.callCoreApplicationMethod({
            domainName: "Telephony",
            methodName: "switchCallOnDevice",
            context: args.context,
            params: args.params,
            paramType: ["PhoneCallId","QString"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Switch the call from a remote device to the local device</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the callId of the call to be switched,it has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    getCallFromDevice : function(args) {
        this.callCoreApplicationMethod({
            domainName: "Telephony",
            methodName: "getCallFromDevice",
            context: args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Terminate a call. Used to end a call on another device. same code as stopConferenceN.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the callId of the call to be switched,it has the following structure:
     * <pre>
     * {String} phone call identifier
     * </pre>  
     *
     * The callback will be called with a void
     * <pre>
     * void {void}
     * </pre>
     * @ignore
     */
    terminateCall : function(args) {
        this.callCoreApplicationMethod({
            domainName: "Telephony",
            methodName: "terminateCall",
            context: args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

	/**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Return the canonical number from a phone number. Used to get a phone number usable with the numbering library when making a call.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a string, corresponding to the phone number to be transformed by the telephony
     *
     * The callback will be called with the transformed phone number, in a string:
     *
     * <pre>
     * {String}  transformed phone number
     * </pre>
     * @public
     */
    makeCanonicalNumber : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "makeCanonicalNumber",
            context : args.context,
            params: args.params,
            paramType: ["QString"],
            callback : args.callback,
            callbackParams: args.callbackParams,
			sync : (args.sync ? args.sync : false)
        });
        args = null;
    },

	/**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Return the list of supervised users in the user's supervision group.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a {JSON} in parameter.
     *
     * <pre>
     * {JSON} list of supervised
     * </pre>
     * @public
     */
	getSupervisionCurrentMembers : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "getSupervisionCurrentMembers",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams,
			sync : (args.sync ? args.sync : false)
        });
        args = null;
    },

	/**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Login or logout from supervision group</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void.
     *
     * <pre>
     * void
     * </pre>
     * @public
     */
	loginLogoffSupervision : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "loginLogoffSupervision",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams,
			sync : (args.sync ? args.sync : false)
        });
        args = null;
    },
	
	/**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Activate/deactivate hide identity feature.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
	 * 
	 * params is a bool, corresponding to the hide identitiy state you want to set.
	 *
     * The callback will be called with a void.
     *
     * <pre>
     * void
     * </pre>
     * @ignore
     */
	modifyHideIdentity : function(args) {
		this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "modifyHideIdentity",
            context : args.context,
            params: args.params,
            paramType: ["bool"],
            callback : args.callback,
            callbackParams: args.callbackParams,
			sync : (args.sync ? args.sync : false)
        });
        args = null;
	},

	/**
	 * modify media
	 * 1st arg : callID
	 * 2nd arg : VideoActionType
	 *				VIDEO_START
	 *				VIDEO_STOP
	 *				VIDEO_NO_ACTION
     * @ignore
	 */
	modifyMedia : function(args) {
        this.callCoreApplicationMethod({
            domainName: "Telephony",
            methodName: "modifyMedia",
            context: args.context,
            params: args.params,
            paramType: ["PhoneCallId","VideoActionType"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},


    /**
     * Deprecated
	 * Use makeCanonicalNumber instead.
     * @ignore
     */
    deletePrefixFromNumber : function(args) {
		this._deprecated("deletePrefixFromNumber", "makeCanonicalNumber", arguments);
        args = null;
    },



    /**
     * @ignore
     */
	setRoutingOverflow : function(args){
		this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "setRoutingOverflow",
            context : args.context,
            params: args.params,
            paramType: ["OverflowDestination", "bool"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

    /**
     * @ignore
     */
	activateRoutingProfile : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "activateRoutingProfile",
            context : args.context,
            params: args.params,
            paramType: ["ProfileId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },


    /**
     * @ignore
     */
	cancelRoutingOverflow : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "cancelRoutingOverflow",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * @ignore
     */
	getRoutingProfiles : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "getRoutingProfiles",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * @ignore
     */
	getRoutingProfile : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "getRoutingProfile",
			params: args.params,
			paramType: ["ProfileId"],
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * @ignore
     */
	getRoutingState : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "getRoutingState",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * @ignore
     */
	setRoutingRoutes : function(args) {
		this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "setRoutingRoutes",
			params: args.params,
			paramType: ["QStringList", "bool"],
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

    /**
     * @ignore
     */
	setRoutingForward : function(args){
		this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "setRoutingForward",
            context : args.context,
            params: args.params,
            paramType: ["ForwardDestination"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

    /**
     * @ignore
     */
	openDoor : function(args) {
		this.callCoreApplicationMethod({
            domainName : "Telephony",
            methodName : "openDoor",
            context : args.context,
            params: args.params,
            paramType: ["PhoneCallId"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	}

/**
 * @event CallConnectedEvent 
 * This event is used to notify that a phone call has connected, which means that callers can speak to each other.
 * @param {Object} param contains the following structure:
 * <pre>
 * {    
 * &nbsp;&nbsp;callId {Object} telephony level call unique identifier.
 * &nbsp;&nbsp;phoneCall {Object} contains all the information related to this phone call instance.
 }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>
 *
 * The parameter phoneCall has the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
 * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
 * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
 * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
 * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
 * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
 * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
 * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
 * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
 * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
 * &nbsp;&nbsp;startTime {Date} Start time
 * &nbsp;&nbsp;isConference {Boolean} Conferencing status
 * &nbsp;&nbsp;isConnected {Boolean}Connection status
 * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
 * &nbsp;&nbsp;isHeld {Boolean} Held Status
 * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
 * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
 * &nbsp;&nbsp;isActive {Boolean} Active Status
 * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
 * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
 * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
 * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
 * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
 * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
 * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
 * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
 * &nbsp;&nbsp;displayName {Boolean} Display Name
 * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
 * }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>  
 *
 * The parameter phoneNumber has the following structure:
 * <pre>
 * {String} sip phone number URI
 * </pre>  
 *
 * The parameter callDirection has the following structure:
 * <pre>
 * {String} call direction from the values,possilbe values are:
 * -"CALL_INCOMING" for incoming call
 * -"CALL_OUTGOING" for outgoing call
 * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
 * </pre>  
 *
 * The parameter startReason and endReason have the following structure:
 * <pre>
 * {String} call reason from the values,possilbe values are:
 * -"CALL_REASON_NORMAL" for normal call
 * -"CALL_REASON_TRANSFERRED" for transferred call
 * </pre> 
 */

/**
 * @event CallEndedEvent 
 * This event is used to notify that a phone call (incoming or outgoing) is finished. This happens when a call is released by local or by distant.
 * In the case of distant release, this event happens when the hook-on invite is started.
 * @param {Object} param contains phoneCall which contains all the information related to this phone call instance,following the structure:
 * <pre>
 * {
 * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
 * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
 * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
 * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
 * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
 * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
 * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
 * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
 * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
 * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
 * &nbsp;&nbsp;startTime {Date} Start time
 * &nbsp;&nbsp;isConference {Boolean} Conferencing status
 * &nbsp;&nbsp;isConnected {Boolean}Connection status
 * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
 * &nbsp;&nbsp;isHeld {Boolean} Held Status
 * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
 * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
 * &nbsp;&nbsp;isActive {Boolean} Active Status
 * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
 * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
 * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
 * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
 * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
 * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
 * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
 * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
 * &nbsp;&nbsp;displayName {Boolean} Display Name
 * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
 * }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>  
 *
 * The parameter phoneNumber has the following structure:
 * <pre>
 * {String} sip phone number URI
 * </pre>  
 *
 * The parameter callDirection has the following structure:
 * <pre>
 * {String} call direction from the values,possilbe values are:
 * -"CALL_INCOMING" for incoming call
 * -"CALL_OUTGOING" for outgoing call
 * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
 * </pre>  
 *
 * The parameter startReason and endReason have the following structure:
 * <pre>
 * {String} call reason from the values,possilbe values are:
 * -"CALL_REASON_NORMAL" for normal call
 * -"CALL_REASON_TRANSFERRED" for transferred call
 * </pre> 
 */
 
 
/**
 * @event  CallIgnoredEvent
 * This event is used to notify that the call has been ignored
 * @param {Object} param contains the following structure:
 * <pre>
 * {    
 * &nbsp;&nbsp;callId {Object} telephony level call unique identifier.
 * &nbsp;&nbsp;phoneCall {Object} contains all the information related to this phone call instance.
 }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>
 *
 * The parameter phoneCall has the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
 * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
 * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
 * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
 * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
 * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
 * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
 * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
 * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
 * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
 * &nbsp;&nbsp;startTime {Date} Start time
 * &nbsp;&nbsp;isConference {Boolean} Conferencing status
 * &nbsp;&nbsp;isConnected {Boolean}Connection status
 * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
 * &nbsp;&nbsp;isHeld {Boolean} Held Status
 * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
 * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
 * &nbsp;&nbsp;isActive {Boolean} Active Status
 * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
 * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
 * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
 * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
 * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
 * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
 * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
 * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
 * &nbsp;&nbsp;displayName {Boolean} Display Name
 * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
 * }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>  
 *
 * The parameter phoneNumber has the following structure:
 * <pre>
 * {String} sip phone number URI
 * </pre>  
 *
 * The parameter callDirection has the following structure:
 * <pre>
 * {String} call direction from the values,possilbe values are:
 * -"CALL_INCOMING" for incoming call
 * -"CALL_OUTGOING" for outgoing call
 * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
 * </pre>  
 *
 * The parameter startReason and endReason have the following structure:
 * <pre>
 * {String} call reason from the values,possilbe values are:
 * -"CALL_REASON_NORMAL" for normal call
 * -"CALL_REASON_TRANSFERRED" for transferred call
 * </pre> 
 */
 
 /**
 * @event CallStartedEvent 
 * This event is used to notify that a phone call (incoming or outgoing) is starting to exist. This happens before the distant alerting or local ringing.
 * @param {Object} param contains phoneCall which contains all the information related to this phone call instance,following the structure:
 * <pre>
 * {
 * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
 * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
 * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
 * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
 * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
 * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
 * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
 * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
 * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
 * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
 * &nbsp;&nbsp;startTime {Date} Start time
 * &nbsp;&nbsp;isConference {Boolean} Conferencing status
 * &nbsp;&nbsp;isConnected {Boolean}Connection status
 * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
 * &nbsp;&nbsp;isHeld {Boolean} Held Status
 * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
 * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
 * &nbsp;&nbsp;isActive {Boolean} Active Status
 * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
 * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
 * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
 * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
 * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
 * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
 * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
 * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
 * &nbsp;&nbsp;displayName {Boolean} Display Name
 * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
 * }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>  
 *
 * The parameter phoneNumber has the following structure:
 * <pre>
 * {String} sip phone number URI
 * </pre>  
 *
 * The parameter callDirection has the following structure:
 * <pre>
 * {String} call direction from the values,possilbe values are:
 * -"CALL_INCOMING" for incoming call
 * -"CALL_OUTGOING" for outgoing call
 * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
 * </pre>  
 *
 * The parameter startReason and endReason have the following structure:
 * <pre>
 * {String} call reason from the values,possilbe values are:
 * -"CALL_REASON_NORMAL" for normal call
 * -"CALL_REASON_TRANSFERRED" for transferred call
 * </pre> 
 */


/**
 * @event  CallTransferedEvent  
 * This event is used to  notify that a the callee (1st phone call) has been transferred to another callee (2nd) in case of a consult transfer, or to a phone number in case of a blind transfer
 * @param {Object} param contains the following structure:
 * <pre>
 * {    
 * &nbsp;&nbsp;callId1 {Object} telephony level call unique identifier of the phone call to be transferred (1st).
 * &nbsp;&nbsp;phoneCall1 {Object} contains all the information related to 1st phone call.
 * &nbsp;&nbsp;callId2 {Object} telephony level call unique identifier of the phone call to be transferred to (2nd) if it is a consult transfer.
 * &nbsp;&nbsp;phoneCall2 {Object} contains all the information related to 2nd phone call.
 }
 * </pre>
 *
 * The parameter callId1 and callId2 have the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>
 *
 * The parameter phoneCall1 and phoneCall2 have the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
 * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
 * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
 * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
 * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
 * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
 * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
 * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
 * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
 * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
 * &nbsp;&nbsp;startTime {Date} Start time
 * &nbsp;&nbsp;isConference {Boolean} Conferencing status
 * &nbsp;&nbsp;isConnected {Boolean}Connection status
 * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
 * &nbsp;&nbsp;isHeld {Boolean} Held Status
 * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
 * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
 * &nbsp;&nbsp;isActive {Boolean} Active Status
 * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
 * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
 * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
 * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
 * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
 * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
 * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
 * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
 * &nbsp;&nbsp;displayName {Boolean} Display Name
 * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
 * }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>  
 *
 * The parameter phoneNumber has the following structure:
 * <pre>
 * {String} sip phone number URI
 * </pre>  
 *
 * The parameter callDirection has the following structure:
 * <pre>
 * {String} call direction from the values,possilbe values are:
 * -"CALL_INCOMING" for incoming call
 * -"CALL_OUTGOING" for outgoing call
 * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
 * </pre>  
 *
 * The parameter startReason and endReason have the following structure:
 * <pre>
 * {String} call reason from the values,possilbe values are:
 * -"CALL_REASON_NORMAL" for normal call
 * -"CALL_REASON_TRANSFERRED" for transferred call
 * </pre> 
 */     


/**
 * @event ConferenceStartedEvent  
 * This event is used to notify that a conference started
 * @param {Object} param contains the following structure:
 * <pre>
 * {     
 * &nbsp;&nbsp;callId1 {Object} telephony level call unique identifier of the conference 1st callee.
 * &nbsp;&nbsp;phoneCall1 {Object} contains all the information related to 1st phone call.
 * &nbsp;&nbsp;callId2 {Object} telephony level call unique identifier of the conference 2nd callee.
 * &nbsp;&nbsp;phoneCall2 {Object} contains all the information related to 2nd phone call.
 }
 * </pre>
 *
 * The parameter callId1 and callId2 have the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>
 *
 * The parameter phoneCall1 and phoneCall2 have the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
 * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
 * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
 * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
 * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
 * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
 * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
 * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
 * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
 * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
 * &nbsp;&nbsp;startTime {Date} Start time
 * &nbsp;&nbsp;isConference {Boolean} Conferencing status
 * &nbsp;&nbsp;isConnected {Boolean}Connection status
 * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
 * &nbsp;&nbsp;isHeld {Boolean} Held Status
 * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
 * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
 * &nbsp;&nbsp;isActive {Boolean} Active Status
 * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
 * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
 * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
 * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
 * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
 * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
 * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
 * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
 * &nbsp;&nbsp;displayName {Boolean} Display Name
 * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
 * }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>  
 *
 * The parameter phoneNumber has the following structure:
 * <pre>
 * {String} sip phone number URI
 * </pre>  
 *
 * The parameter callDirection has the following structure:
 * <pre>
 * {String} call direction from the values,possilbe values are:
 * -"CALL_INCOMING" for incoming call
 * -"CALL_OUTGOING" for outgoing call
 * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
 * </pre>  
 *
 * The parameter startReason and endReason have the following structure:
 * <pre>
 * {String} call reason from the values,possilbe values are:
 * -"CALL_REASON_NORMAL" for normal call
 * -"CALL_REASON_TRANSFERRED" for transferred call
 * </pre> 
 */     


/**
 * @event  ConferenceEndedEvent  
 * This event is used to notify that a conference ended
 * @param {Object} param contains the following structure:
 * <pre>
 * {    
 * &nbsp;&nbsp;callId1 {Object} telephony level call unique identifier of the phone call to be transferred (1st).
 * &nbsp;&nbsp;phoneCall1 {Object} contains all the information related to 1st phone call.
 * &nbsp;&nbsp;callId2 {Object} telephony level call unique identifier of the phone call to be transferred to (2nd) if it is a consult transfer.
 * &nbsp;&nbsp;phoneCall2 {Object} contains all the information related to 2nd phone call.
 }
 * </pre>
 *
 * The parameter callId1 and callId2 have the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>
 *
 * The parameter phoneCall1 and phoneCall2 have the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
 * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
 * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
 * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
 * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
 * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
 * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
 * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
 * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
 * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
 * &nbsp;&nbsp;startTime {Date} Start time
 * &nbsp;&nbsp;isConference {Boolean} Conferencing status
 * &nbsp;&nbsp;isConnected {Boolean}Connection status
 * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
 * &nbsp;&nbsp;isHeld {Boolean} Held Status
 * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
 * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
 * &nbsp;&nbsp;isActive {Boolean} Active Status
 * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
 * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
 * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
 * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
 * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
 * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
 * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
 * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
 * &nbsp;&nbsp;displayName {Boolean} Display Name
 * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
 * }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>  
 *
 * The parameter phoneNumber has the following structure:
 * <pre>
 * {String} sip phone number URI
 * </pre>  
 *
 * The parameter callDirection has the following structure:
 * <pre>
 * {String} call direction from the values,possilbe values are:
 * -"CALL_INCOMING" for incoming call
 * -"CALL_OUTGOING" for outgoing call
 * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
 * </pre>  
 *
 * The parameter startReason and endReason have the following structure:
 * <pre>
 * {String} call reason from the values,possilbe values are:
 * -"CALL_REASON_NORMAL" for normal call
 * -"CALL_REASON_TRANSFERRED" for transferred call
 * </pre> 
 */     

/**
 * @event  OutgoingCallAlertingEvent
 * This event is used to notify that an outgoing phone call has reached the callee. This happens at the beginning the distant alerting
 * @param {Object} param contains the following structure:
 * <pre>
 * {    
 * &nbsp;&nbsp;callId {Object} telephony level call unique identifier.
 * &nbsp;&nbsp;phoneCall {Object} contains all the information related to this phone call instance.
 }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>
 *
 * The parameter phoneCall has the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
 * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
 * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
 * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
 * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
 * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
 * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
 * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
 * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
 * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
 * &nbsp;&nbsp;startTime {Date} Start time
 * &nbsp;&nbsp;isConference {Boolean} Conferencing status
 * &nbsp;&nbsp;isConnected {Boolean}Connection status
 * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
 * &nbsp;&nbsp;isHeld {Boolean} Held Status
 * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
 * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
 * &nbsp;&nbsp;isActive {Boolean} Active Status
 * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
 * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
 * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
 * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
 * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
 * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
 * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
 * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
 * &nbsp;&nbsp;displayName {Boolean} Display Name
 * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
 * }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>  
 *
 * The parameter phoneNumber has the following structure:
 * <pre>
 * {String} sip phone number URI
 * </pre>  
 *
 * The parameter callDirection has the following structure:
 * <pre>
 * {String} call direction from the values,possilbe values are:
 * -"CALL_INCOMING" for incoming call
 * -"CALL_OUTGOING" for outgoing call
 * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
 * </pre>  
 *
 * The parameter startReason and endReason have the following structure:
 * <pre>
 * {String} call reason from the values,possilbe values are:
 * -"CALL_REASON_NORMAL" for normal call
 * -"CALL_REASON_TRANSFERRED" for transferred call
 * </pre> 
 */

/**
 * @event OutgoingCallFailedEvent
 * This event is used to notify that an outgoing phone call is failed. 
 * @param {Object} param contains the following structure:
 * <pre>
 * {    
 * &nbsp;&nbsp;callId {Object} telephony level call unique identifier.
 * &nbsp;&nbsp;phoneCall {Object} contains all the information related to this phone call instance.
 * &nbsp;&nbsp;callFailedReason {Number} 
 }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>
 *
 * The parameter phoneCall has the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp;callId {Object} Unique phoneCall identifier, local to the telephony module. This is usually different from the SIP call id (sipCallId).See the description below
 * &nbsp;&nbsp;sipCallId {Number} Unique phoneCall identifier as provided by the SIPUA module
 * &nbsp;&nbsp;phoneNumber {Object} Phone number string: "SIP" "URI".See the description below
 * &nbsp;&nbsp;phoneNumberCan {Object} Canonical Phone number string: Canonical form of the phone number
 * &nbsp;&nbsp;activeDevicePhoneNumber {String} The phone number of the device on which the call is ongoing
 * &nbsp;&nbsp;callDirection {Object} The phone number of the device on which the call is ongoing.See the description below
 * &nbsp;&nbsp;startReason {Object}  Reason for the start of the call.See the description below
 * &nbsp;&nbsp;endReason {Object}  Reason for the end of the call.See the description below
 * &nbsp;&nbsp;callType {Number} Call type(value 0 for internal call or value 1 for external call or value 2 for appointment call or value 3 for special call)
 * &nbsp;&nbsp;idCapabilities {Number} Identifier of the capability list
 * &nbsp;&nbsp;startTime {Date} Start time
 * &nbsp;&nbsp;isConference {Boolean} Conferencing status
 * &nbsp;&nbsp;isConnected {Boolean}Connection status
 * &nbsp;&nbsp;isOnHold {Boolean} OnHold Status
 * &nbsp;&nbsp;isHeld {Boolean} Held Status
 * &nbsp;&nbsp;isRinging {Boolean} Ringing Status
 * &nbsp;&nbsp;isAlerting {Boolean} Alerting Status
 * &nbsp;&nbsp;isActive {Boolean} Active Status
 * &nbsp;&nbsp;isIgnoring {Boolean} Ignoring Status
 * &nbsp;&nbsp;isBusy {Boolean} Busy Status (true for an outgoing call if the distant is busy)
 * &nbsp;&nbsp;isLocal {Boolean}Is phone call local to the device
 * &nbsp;&nbsp;answerAfter {Boolean} Auto answer information
 * &nbsp;&nbsp;isMutedByDistant {Boolean} Call session muted by distnat during conference
 * &nbsp;&nbsp;isMediaSecured {Boolean} Current RTP media is secured
 * &nbsp;&nbsp;isVVM {Boolean} Indicates if the phone call is a visual voice mail session
 * &nbsp;&nbsp;isFiltered {Boolean} Indicate if the phone call is a call filtered by manager and redirected to user = assistant
 * &nbsp;&nbsp;displayName {Boolean} Display Name
 * &nbsp;&nbsp;manager {String} The manager's login if the call is filtered    
 * }
 * </pre>
 *
 * The parameter callId has the following structure:
 * <pre>
 * {String} phone call identifier
 * </pre>  
 *
 * The parameter phoneNumber has the following structure:
 * <pre>
 * {String} sip phone number URI
 * </pre>  
 *
 * The parameter callDirection has the following structure:
 * <pre>
 * {String} call direction from the values,possilbe values are:
 * -"CALL_INCOMING" for incoming call
 * -"CALL_OUTGOING" for outgoing call
 * -"CALL_DIR_UNKNOWN" for call direction not yet defined      
 * </pre>  
 *
 * The parameter startReason and endReason have the following structure:
 * <pre>
 * {String} call reason from the values,possilbe values are:
 * -"CALL_REASON_NORMAL" for normal call
 * -"CALL_REASON_TRANSFERRED" for transferred call
 * </pre> 
 */ 
 


/**
 * @event OutgoingCallInitiatedEvent
 * This event is used to notify that an outgoing phone call is initiated. 
 */ 
 
 /**
 * @event OutgoingCallCancelledEvent
 * This event is used to notify that an outgoing phone call is cancelled. 
 */ 
 
 /**
 * @event GuestCheckIn 
 * This event is used to notify that the guest's status is changed.
 * @param {Boolean} param contains the status,checked in or checked out.
 */ 
 
 
 /**
 * @event GuestInfo 
 * This event is used to notify the guest's information.
 * @param {Object} param contains the following structure:
 * <pre>
 * {    
 * &nbsp;&nbsp;_guestName {String} 
 * &nbsp;&nbsp;_guestNumber {String} 
 * &nbsp;&nbsp;_setNumber {String} 
 * }
 * </pre> 
 */
 
 /**
 * @event GuestLanguage
 * This event is used to notify that the guest's language.
 * @param {String} param contains the language.
 */
 
 
 
});

ICTouchAPI.APIServices.Telephony = new ICTouchAPI.APIServices.Telephony();
/**
 * @ignore
 */
dojo.provide("ICTouchAPI.APIServices.Platform");
dojo.declare("ICTouchAPI.APIServices.Platform",[ICTouchAPI.APIServices.APIServicesBase],
{
    initLock : function(args){
        this.callCoreApplicationMethod({
            domainName : "Platform",
            methodName : "initLock",
            context : args.context,
            params: args.params,
            paramType: ["bool"], //types of params
            callback : args.callback,
            callbackParams: args.callbackParams  // optional: to send to callback some parameters of context
                                            // the callback can be a method of another webapp which
                                            // could use the callbackParams
        });
        args = null;
    },

    setDateTime : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Platform",
            methodName : "setDateTime",
            context : args.context,
            params: args.params,
            paramType: ["Timestamp"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    setTimeZone : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Platform",
            methodName : "setTimeZone",
            context : args.context,
            params: args.params,
            paramType: ["float"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    login : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Platform",
            methodName : "login",
            context : args.context,
            params: args.params,
            paramType: ["QString", "QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

setLedStatus : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Platform",
            methodName : "setLedStatus",
            context : args.context,
            params: args.params,
            paramType: ["int", "int", "int", "int" ,"int"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

	setMute : function(args) {
        this.callCoreApplicationMethod({
            domainName: "Platform",
            methodName: "setMute",
            context: args.context,
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

	DMLogin : function(args) {
		
        this.callCoreApplicationMethod({
            domainName: "Platform",
            methodName: "DMLogin",
            context: args.context,
            params: args.params,
            paramType: ["QString", "QString"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    // Ludovic Cadart
    // pour simuler
    initLockSimulate : function(args){


        var method = dojo.hitch(args.context, args.callback);
        method(args.params);
		args = null;
    },

	resetToFactory : function(args) {
        this.callCoreApplicationMethod({
            domainName : "Platform",
            methodName : "resetToFactory",
            context : args.context,
            params: args.params,
            paramType: ["QString", "QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    }
 });

ICTouchAPI.APIServices.Platform = new ICTouchAPI.APIServices.Platform();
/**
 * @ignore
 */
dojo.provide("ICTouchAPI.APIServices.ApplicationManager");
dojo.declare("ICTouchAPI.APIServices.ApplicationManager",[ICTouchAPI.APIServices.APIServicesBase],
{
    presentationReady : function(args){
        this.callCoreApplicationMethod({
            domainName : "ApplicationManager",
            methodName : "presentationReady",
            context : args.context,
            callback : args.callback
        });
        args = null;
    }
 });

ICTouchAPI.APIServices.ApplicationManager = new ICTouchAPI.APIServices.ApplicationManager();
/**
 * @ignore
 */
dojo.provide("ICTouchAPI.APIServices.ICTGate");
dojo.declare("ICTouchAPI.APIServices.ICTGate",[ICTouchAPI.APIServices.APIServicesBase],
{
    notifyEventPresentation : function(args){
        this.callCoreApplicationMethod({
            domainName : "ICTGate",
            methodName : "notifyEventPresentation",
            params: args.params,
            paramType: ["QString"] //types of params

        });
        args = null;
    },

    syslog : function(args) {
        this.callCoreApplicationMethod({
            domainName: "ICTGate",
            methodName: "syslog",
            context: args.context,
            params: args.params,
            paramType: ["int", "QString"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },


    login : function(args) {
		
        this.callCoreApplicationMethod({
            domainName : "ICTGate",
            methodName : "login",
            context : args.context,
            params: args.params,
            paramType: ["QString", "QString"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    logout : function(args) {
        this.callCoreApplicationMethod({
            domainName : "ICTGate",
            methodName : "logout",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    }
});

ICTouchAPI.APIServices.ICTGate = new ICTouchAPI.APIServices.ICTGate();
/**
 * @class ICTouchAPI.APIServices.Email
 * @extends Object
 * @ignore
 * The Email module is in charge of all the emails features.<br>
 * It provides functions allowing users to manage their mails and receive events related to them.
 */
dojo.provide("ICTouchAPI.APIServices.Email");
dojo.declare("ICTouchAPI.APIServices.Email",[ICTouchAPI.APIServices.APIServicesBase],
{
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>list of all mails in database</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>the list of all mails in database corresponding to the criteria.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the search criteria.
     * 
     * <pre>
     * {String} for the search criteria.
     * </pre>
     *
     *
     * The callback will be called with the list of all mails in database, each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence
     * &nbsp;&nbsp;_fromAdress {String} Address of the sender, as received in mail argument
     * &nbsp;&nbsp;_title {String} Title of the mail, as received in mail argument                                                    
     * &nbsp;&nbsp;_status {String} Status of the mail : DRAFT, SENT, READ or UNREAD.                                                 
     * &nbsp;&nbsp;_saved {Date} Date of the last modification.                                                                   
     * &nbsp;&nbsp;_sentReceived {Date} Date of the sent or receive depending of the mail status.                                 
     * &nbsp;&nbsp;_textBody {Date} Body of the mail.                                                                             
     * &nbsp;&nbsp;_formattedBody {String} Body of the mail , formatted in different format (for exemple, html part of the mail).     
     * &nbsp;&nbsp;_formattedType {String} Type of the formattedBody (Text or HTML).                                                  
     * &nbsp;&nbsp;_charset {String} Original charset of the message. The message is converted in UTF-8 before being sent on DBUS.    
     * &nbsp;&nbsp;_mailTo {Object} List of recipients.
     * &nbsp;&nbsp;_mailAtt (JSON) : List of attachments.
     * }
     * </pre>
     *
     * The parameter _mailTo has the following structs:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idMailTo {Number} Unique mail attachment identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_toAdress {String} Recipient Address.
     * }
     * </pre>
     *
     * The parameter _mailAtt has the following structs:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idMailAtt {Number} Unique mail attachment identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_fileId {Number} ID of the file representing the attachment.
     * &nbsp;&nbsp;_mimeType {String} Mime type of the file.
     * &nbsp;&nbsp;_fileName {String} Name of the attachment.                                
     * &nbsp;&nbsp;_ignoredAttachments {Boolean} Set to true if some attachments have been ignored.                                       
     * }
     * </pre>
     */            
	getAllMails : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctMail",
			methodName : "getAllMails",
			context : args.context,
			params: args.params,
            paramType: ["QString"],
			callback : args.callback,
			errorCallback : args.errorCallback
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>list of unread mails in database</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>the list of unread mails in database corresponding to the criteria.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the search criteria.
     * 
     * <pre>
     * {String} for the search criteria.
     * </pre>
     *
     *
     * The callback will be called with the list of unread mails in database, each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence
     * &nbsp;&nbsp;_fromAdress {String} Address of the sender, as received in mail argument
     * &nbsp;&nbsp;_title {String} Title of the mail, as received in mail argument                                                    
     * &nbsp;&nbsp;_status {String} Status of the mail : DRAFT, SENT, READ or UNREAD.                                                 
     * &nbsp;&nbsp;_saved {Date} Date of the last modification.                                                                   
     * &nbsp;&nbsp;_sentReceived {Date} Date of the sent or receive depending of the mail status.                                 
     * &nbsp;&nbsp;_textBody {Date} Body of the mail.                                                                             
     * &nbsp;&nbsp;_formattedBody {String} Body of the mail , formatted in different format (for exemple, html part of the mail).     
     * &nbsp;&nbsp;_formattedType {String} Type of the formattedBody (Text or HTML).                                                  
     * &nbsp;&nbsp;_charset {String} Original charset of the message. The message is converted in UTF-8 before being sent on DBUS.    
     * &nbsp;&nbsp;_mailTo {Object} List of recipients.
     * &nbsp;&nbsp;_mailAtt (JSON) : List of attachments.
     * }
     * </pre>
     *
     * The parameter _mailTo has the following structs:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idMailTo {Number} Unique mail attachment identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_toAdress {String} Recipient Address.
     * }
     * </pre>
     *
     * The parameter _mailAtt has the following structs:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idMailAtt {Number} Unique mail attachment identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_fileId {Number} ID of the file representing the attachment.
     * &nbsp;&nbsp;_mimeType {String} Mime type of the file.
     * &nbsp;&nbsp;_fileName {String} Name of the attachment.                                
     * &nbsp;&nbsp;_ignoredAttachments {Boolean} Set to true if some attachments have been ignored.                                       
     * }
     * </pre>
     */     
	getUnreadMails : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctMail",
			methodName : "getUnreadMails",
			context : args.context,
			params: args.params,
            paramType: ["QString"],
			callback : args.callback,
			errorCallback: args.errorCallback
		});
		args = null;
	},
	
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Set the mail to the "READ" status.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the id of the mail to be set as read.
     * 
     * <pre>
     * {Number} for the id of the mail
     * </pre>
     *
     *
     * The callback will be called with the status, Possible value are:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; -1 EMAIL_NOT_FOUND
     * &nbsp;&nbsp; 21 mail already Read
     * &nbsp;&nbsp; 22 mail in another status
     * &nbsp;&nbsp; 0 SUCCESS
     * }
     * </pre>
     */    	
        readMail : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctMail",
            methodName : "readMail",
            context : args.context,
            params: args.params,
            paramType: ["int"]
        });
        args = null;
    },
    
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Update mail attributes and send the event {@link #MAIL_UPDATE_NTF} containing the new MailId.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON which contains the following structure : 
     * 
     * <pre>
     * {
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence
     * &nbsp;&nbsp;_fromAdress {String} Address of the sender, as received in mail argument
     * &nbsp;&nbsp;_title {String} Title of the mail, as received in mail argument                                                    
     * &nbsp;&nbsp;_status {String} Status of the mail : DRAFT, SENT, READ or UNREAD.                                                 
     * &nbsp;&nbsp;_saved {Date} Date of the last modification.                                                                   
     * &nbsp;&nbsp;_sentReceived {Date} Date of the sent or receive depending of the mail status.                                 
     * &nbsp;&nbsp;_textBody {Date} Body of the mail.                                                                             
     * &nbsp;&nbsp;_formattedBody {String} Body of the mail , formatted in different format (for exemple, html part of the mail).     
     * &nbsp;&nbsp;_formattedType {String} Type of the formattedBody (Text or HTML).                                                  
     * &nbsp;&nbsp;_charset {String} Original charset of the message. The message is converted in UTF-8 before being sent on DBUS.    
     * &nbsp;&nbsp;_mailTo {Object} List of recipients.
     * &nbsp;&nbsp;_mailAtt (JSON) : List of attachments.
     * }
     * </pre>
     *
     * The parameter _mailTo has the following structs:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idMailTo {Number} Unique mail attachment identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_toAdress {String} Recipient Address.
     * }
     * </pre>
     *
     * The parameter _mailAtt has the following structs:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idMailAtt {Number} Unique mail attachment identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_fileId {Number} ID of the file representing the attachment.
     * &nbsp;&nbsp;_mimeType {String} Mime type of the file.
     * &nbsp;&nbsp;_fileName {String} Name of the attachment.                                
     * &nbsp;&nbsp;_ignoredAttachments {Boolean} Set to true if some attachments have been ignored.                                       
     * }
     * </pre>
     *
     * The callback will be called with the status, Possible value are:
     *
     * <pre>
     * {                                            
     * &nbsp;&nbsp; -999 ERR_EMAIL_DISABLED 
     * &nbsp;&nbsp; 20 EMAIL_NOT_FOUND      
     * &nbsp;&nbsp; 10 MAILBOX_FULL         
     * &nbsp;&nbsp; 5 Linked file not found 
     * &nbsp;&nbsp; 2 Database error        
     * &nbsp;&nbsp; 0 SUCCESS                                           
     * }            
     * </pre>
     */        
        updateMail : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctMail",
            methodName : "updateMail",
            context : args.context,
            params: args.params,
            paramType: ["Mail"],
			errorCallback : args.errorCallback
        });
        args = null;
        },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>list of mails</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Search a mail and return the list of mails corresponding to the criteria.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;{String} The search criteria
     * &nbsp;&nbsp;{Number} The page Number
     * &nbsp;&nbsp;{Number} The length of a page
     * }
     * </pre>
     *
     * The callback will be called with the list of mails in database, each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence
     * &nbsp;&nbsp;_fromAdress {String} Address of the sender, as received in mail argument
     * &nbsp;&nbsp;_title {String} Title of the mail, as received in mail argument                                                    
     * &nbsp;&nbsp;_status {String} Status of the mail : DRAFT, SENT, READ or UNREAD.                                                 
     * &nbsp;&nbsp;_saved {Date} Date of the last modification.                                                                   
     * &nbsp;&nbsp;_sentReceived {Date} Date of the sent or receive depending of the mail status.                                 
     * &nbsp;&nbsp;_textBody {Date} Body of the mail.                                                                             
     * &nbsp;&nbsp;_formattedBody {String} Body of the mail , formatted in different format (for exemple, html part of the mail).     
     * &nbsp;&nbsp;_formattedType {String} Type of the formattedBody (Text or HTML).                                                  
     * &nbsp;&nbsp;_charset {String} Original charset of the message. The message is converted in UTF-8 before being sent on DBUS.    
     * &nbsp;&nbsp;_mailTo {Object} List of recipients.
     * &nbsp;&nbsp;_mailAtt (JSON) : List of attachments.
     * }
     * </pre>
     *
     * The parameter _mailTo has the following structs:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idMailTo {Number} Unique mail attachment identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_toAdress {String} Recipient Address.
     * }
     * </pre>
     *
     * The parameter _mailAtt has the following structs:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idMailAtt {Number} Unique mail attachment identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_fileId {Number} ID of the file representing the attachment.
     * &nbsp;&nbsp;_mimeType {String} Mime type of the file.
     * &nbsp;&nbsp;_fileName {String} Name of the attachment.                                
     * &nbsp;&nbsp;_ignoredAttachments {Boolean} Set to true if some attachments have been ignored.                                       
     * }
     * </pre>
     */         
		findMail : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctMail",
            methodName : "findMail",
            context : args.context,
            params: args.params,
            paramType: ["QString", "int", "int"],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
        },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>list of mails</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Search a mail and return the list of mails corresponding to the criteria. In case of error the event {@link #MAIL_FIND_ERROR_NTF} is send</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;{String} The search criteria
     * &nbsp;&nbsp;{Number} The page Number
     * &nbsp;&nbsp;{Number} The length of a page
     * }
     * </pre>
     *
     * The callback will be called with the list of mails in database, each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence
     * &nbsp;&nbsp;_fromAdress {String} Address of the sender, as received in mail argument
     * &nbsp;&nbsp;_title {String} Title of the mail, as received in mail argument                                                    
     * &nbsp;&nbsp;_status {String} Status of the mail : DRAFT, SENT, READ or UNREAD.                                                 
     * &nbsp;&nbsp;_saved {Date} Date of the last modification.                                                                   
     * &nbsp;&nbsp;_sentReceived {Date} Date of the sent or receive depending of the mail status.                                 
     * &nbsp;&nbsp;_textBody {Date} Body of the mail.                                                                             
     * &nbsp;&nbsp;_formattedBody {String} Body of the mail , formatted in different format (for exemple, html part of the mail).     
     * &nbsp;&nbsp;_formattedType {String} Type of the formattedBody (Text or HTML).                                                  
     * &nbsp;&nbsp;_charset {String} Original charset of the message. The message is converted in UTF-8 before being sent on DBUS.    
     * &nbsp;&nbsp;_mailTo {Object} List of recipient.
     * &nbsp;&nbsp;_mailAtt (JSON) : List of attachments.
     * }
     * </pre>
     *
     * The parameter _mailTo has the following structs:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idMailTo {Number} Unique mail attachment identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_toAdress {String} Recipient Address.
     * }
     * </pre>
     *
     * The parameter _mailAtt has the following structs:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idMailAtt {Number} Unique mail attachment identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_fileId {Number} ID of the file representing the attachment.
     * &nbsp;&nbsp;_mimeType {String} Mime type of the file.
     * &nbsp;&nbsp;_fileName {String} Name of the attachment.                                
     * &nbsp;&nbsp;_ignoredAttachments {Boolean} Set to true if some attachments have been ignored.                                       
     * }
     * </pre>
     */       
		findDraftMail : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctMail",
            methodName : "findDraftMail",
            context : args.context,
            params: args.params,
            paramType: ["QString", "int", "int"],
            callback : args.callback,
			errorCallback : args.errorCallback
        });
        args = null;
        },
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>the eMail in database</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>The eMail in database corresponding to the id.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the identifier of mail.
     * 
     * <pre>
     * {Number} for the identifier of mail
     * </pre>
     *
     *
     * The callback will be called with the mails in database, each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence
     * &nbsp;&nbsp;_fromAdress {String} Address of the sender, as received in mail argument
     * &nbsp;&nbsp;_title {String} Title of the mail, as received in mail argument                                                    
     * &nbsp;&nbsp;_status {String} Status of the mail : DRAFT, SENT, READ or UNREAD.                                                 
     * &nbsp;&nbsp;_saved {Date} Date of the last modification.                                                                   
     * &nbsp;&nbsp;_sentReceived {Date} Date of the sent or receive depending of the mail status.                                 
     * &nbsp;&nbsp;_textBody {Date} Body of the mail.                                                                             
     * &nbsp;&nbsp;_formattedBody {String} Body of the mail , formatted in different format (for exemple, html part of the mail).     
     * &nbsp;&nbsp;_formattedType {String} Type of the formattedBody (Text or HTML).                                                  
     * &nbsp;&nbsp;_charset {String} Original charset of the message. The message is converted in UTF-8 before being sent on DBUS.    
     * &nbsp;&nbsp;_mailTo {Object} List of recipients.
     * &nbsp;&nbsp;_mailAtt (JSON) : List of attachments.
     * }
     * </pre>
     *
     * The parameter _mailTo has the following structs:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idMailTo {Number} Unique mail attachment identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_toAdress {String} Recipient Address.
     * }
     * </pre>
     *
     * The parameter _mailAtt has the following structs:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idMailAtt {Number} Unique mail attachment identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_fileId {Number} ID of the file representing the attachment.
     * &nbsp;&nbsp;_mimeType {String} Mime type of the file.
     * &nbsp;&nbsp;_fileName {String} Name of the attachment.                                
     * &nbsp;&nbsp;_ignoredAttachments {Boolean} Set to true if some attachments have been ignored.                                       
     * }
     * </pre>
     */       
        getMail : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctMail",
            methodName : "getMail",
            context : args.context,
            params: args.params,
            paramType: ["int"],
			errorCallback : args.errorCallback
        });
        args = null;
        },
        
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>status of the action</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Send a mail and set the mail's status to SENT, and send the event {@link #MAIL_SEND_NTF}.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the id of the mail to be sent, using configured mail server, user and password.
     * 
     * <pre>
     * {Number} the id of the mail to be sent, using configured mail server, user and password.
     * </pre>
     *
     *
     * The callback will be called with the status, Possible value are:
     *
     * <pre>
     * {                                              
     * &nbsp;&nbsp; -999 ERR_EMAIL_DISABLED 
     * &nbsp;&nbsp; 31 SERVER_ERROR             
     * &nbsp;&nbsp; 20 EMAIL_NOT_FOUND          
     * &nbsp;&nbsp; 23 EMAIL_ALREADY_SENT       
     * &nbsp;&nbsp; 10 MAILBOX_FULL             
     * &nbsp;&nbsp; 5 Linked file not found     
     * &nbsp;&nbsp; 2 Database error  
     * &nbsp;&nbsp; 0 SUCCESS                             
     * }            
     * </pre>       
     */                
        sendMail : function(args ) {
        this.callCoreApplicationMethod({
            domainName : "IctMail",
            methodName : "sendMail",
            context : args.context,
            params: args.params,
            paramType: ["int"],
			errorCallback :args.errorCallback
        });
        args = null;
        },
        
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>status of mail saved</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Save a mail in database.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON which contains the following structure : 
     * 
     * <pre>
     * {
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence
     * &nbsp;&nbsp;_fromAdress {String} Address of the sender, as received in mail argument
     * &nbsp;&nbsp;_title {String} Title of the mail, as received in mail argument                                                    
     * &nbsp;&nbsp;_status {String} Status of the mail : DRAFT, SENT, READ or UNREAD.                                                 
     * &nbsp;&nbsp;_saved {Date} Date of the last modification.                                                                   
     * &nbsp;&nbsp;_sentReceived {Date} Date of the sent or receive depending of the mail status.                                 
     * &nbsp;&nbsp;_textBody {Date} Body of the mail.                                                                             
     * &nbsp;&nbsp;_formattedBody {String} Body of the mail , formatted in different format (for exemple, html part of the mail).     
     * &nbsp;&nbsp;_formattedType {String} Type of the formattedBody (Text or HTML).                                                  
     * &nbsp;&nbsp;_charset {String} Original charset of the message. The message is converted in UTF-8 before being sent on DBUS.    
     * &nbsp;&nbsp;_mailTo {Object} List of recipients.
     * &nbsp;&nbsp;_mailAtt (JSON) : List of attachments.
     * }
     * </pre>
     *
     * The parameter _mailTo has the following structs:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idMailTo {Number} Unique mail attachment identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_toAdress {String} Recipient Address.
     * }
     * </pre>
     *
     * The parameter _mailAtt has the following structs:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idMailAtt {Number} Unique mail attachment identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_fileId {Number} ID of the file representing the attachment.
     * &nbsp;&nbsp;_mimeType {String} Mime type of the file.
     * &nbsp;&nbsp;_fileName {String} Name of the attachment.                                
     * &nbsp;&nbsp;_ignoredAttachments {Boolean} Set to true if some attachments have been ignored.                                       
     * }
     * </pre>
     *
     * The callback will be called with the status, Possible value are:
     *
     * <pre>
     * {                                             
     * &nbsp;&nbsp; -999  ERR_EMAIL_DISABLED    
     * &nbsp;&nbsp; 31 SERVER_ERROR             
     * &nbsp;&nbsp; 20 EMAIL_NOT_FOUND          
     * &nbsp;&nbsp; 23 EMAIL_ALREADY_SENT       
     * &nbsp;&nbsp; 10 MAILBOX_FULL             
     * &nbsp;&nbsp; 5 Linked file not found     
     * &nbsp;&nbsp; 2 Database error  
     * &nbsp;&nbsp; 0 SUCCESS                             
     * }            
     * </pre>       
     */         
		saveMail : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctMail",
            methodName : "saveMail",
            context : args.context,
            params: args.params,
            paramType: ["Mail"],
			errorCallback : args.errorCallback
        });
        args = null;
        },

        

        updateSettingValue : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctMail",
            methodName : "updateSettingValue",
            context : args.context,
            params: args.params,
            paramType: [],
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
        },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>status of deletion</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Delete a mail and send the event {@link #MAIL_DELETE_NTF}.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the id of mail to delete.
     * 
     * <pre>
     * {Number} for the id of mail to delete.
     * </pre>
     *
     *
     * The callback will be called with the status, Possible value are:
     *
     * <pre>
     * {                                     
     * &nbsp;&nbsp; -1 EMAIL_NOT_FOUND        
     * &nbsp;&nbsp; 2 Database error          
     * &nbsp;&nbsp; 0 SUCCESS                 
     * }            
     * </pre>       
     */               
        deleteMail : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctMail",
            methodName : "deleteMail",
            context : args.context,
            params: args.params,
            paramType: ["int"],
			errorCallback : args.errorCallback
        });
        args = null;
        },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>status of receive</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Retrieve mails from the server and store it from the database.
     * Invoked by the Presentation on RECEIVE_MAIL_EVENT event.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON which contains the following structure : 
     * 
     * <pre>
     * {
     * &nbsp;&nbsp;_evtName {String} name of the event.
     * &nbsp;&nbsp;_argList {Object} a list of other attributes in a JSON structure
     * &nbsp;&nbsp;_argTypes {JSONO} A list of other attributes in a JSON structure
     * }
     * </pre>
     *
     *
     * The callback will be called with the status, Possible value are:
     *
     * <pre>
     * {                                             
     * &nbsp;&nbsp; -999  ERR_EMAIL_DISABLED 
     * &nbsp;&nbsp; 31 SERVER_ERROR          
     * &nbsp;&nbsp; 30 SERVER_UNREACHABLE        
     * &nbsp;&nbsp; 10 MAILBOX_FULL          
     * &nbsp;&nbsp; 2 Database error         
     * &nbsp;&nbsp; 0 SUCCESS                 
     * }       
     * </pre>       
     */                
		receiveMails : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctMail",
            methodName : "receiveMails",
            context : args.context
        });
        args = null;
        },
        
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>status of sent</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Update and Send a mail.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON which contains the following structure : 
     * 
     * <pre>
     * {
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence
     * &nbsp;&nbsp;_fromAdress {String} Address of the sender, as received in mail argument
     * &nbsp;&nbsp;_title {String} Title of the mail, as received in mail argument                                                    
     * &nbsp;&nbsp;_status {String} Status of the mail : DRAFT, SENT, READ or UNREAD.                                                 
     * &nbsp;&nbsp;_saved {Date} Date of the last modification.                                                                   
     * &nbsp;&nbsp;_sentReceived {Date} Date of the sent or receive depending of the mail status.                                 
     * &nbsp;&nbsp;_textBody {Date} Body of the mail.                                                                             
     * &nbsp;&nbsp;_formattedBody {String} Body of the mail , formatted in different format (for exemple, html part of the mail).     
     * &nbsp;&nbsp;_formattedType {String} Type of the formattedBody (Text or HTML).                                                  
     * &nbsp;&nbsp;_charset {String} Original charset of the message. The message is converted in UTF-8 before being sent on DBUS.    
     * &nbsp;&nbsp;_mailTo {Object} List of recipients.
     * &nbsp;&nbsp;_mailAtt (JSON) : List of attachments.
     * }
     * </pre>
     *
     * The parameter _mailTo has the following structs:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idMailTo {Number} Unique mail attachment identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_toAdress {String} Recipient Address.
     * }
     * </pre>
     *
     * The parameter _mailAtt has the following structs:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idMailAtt {Number} Unique mail attachment identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_fileId {Number} ID of the file representing the attachment.
     * &nbsp;&nbsp;_mimeType {String} Mime type of the file.
     * &nbsp;&nbsp;_fileName {String} Name of the attachment.                                
     * &nbsp;&nbsp;_ignoredAttachments {Boolean} Set to true if some attachments have been ignored.                                       
     * }
     * </pre>
     *
     *
     * The callback will be called with the status of sent, Possible value are:
     *
     * <pre>
     * {                                             
     * &nbsp;&nbsp; -999  ERR_EMAIL_DISABLED 
     * &nbsp;&nbsp; 31 SERVER_ERROR          
     * &nbsp;&nbsp; 20 EMAIL_NOT_FOUND       
     * &nbsp;&nbsp; 23 EMAIL_ALREADY_SENT    
     * &nbsp;&nbsp; 10 MAILBOX_FULL          
     * &nbsp;&nbsp; 5 Linked file not found  
     * &nbsp;&nbsp; 2 Database error         
     * &nbsp;&nbsp; 0 SUCCESS                 
     * }
     * </pre>                  
     */                                           
		saveAndSendMail : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctMail",
            methodName : "saveAndSendMail",
            context : args.context,
            params: args.params,
            paramType: ["Mail"],
			errorCallback : args.errorCallback
        });
        args = null;
        },
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>status of sent</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Update and Send a mail. Set the mail's status to SENT, and send the event {@link #MAIL_SEND_NTF}.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON which contains the following structure : 
     * 
     * <pre>
     * {
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence
     * &nbsp;&nbsp;_fromAdress {String} Address of the sender, as received in mail argument
     * &nbsp;&nbsp;_title {String} Title of the mail, as received in mail argument                                                    
     * &nbsp;&nbsp;_status {String} Status of the mail : DRAFT, SENT, READ or UNREAD.                                                 
     * &nbsp;&nbsp;_saved {Date} Date of the last modification.                                                                   
     * &nbsp;&nbsp;_sentReceived {Date} Date of the sent or receive depending of the mail status.                                 
     * &nbsp;&nbsp;_textBody {Date} Body of the mail.                                                                             
     * &nbsp;&nbsp;_formattedBody {String} Body of the mail , formatted in different format (for exemple, html part of the mail).     
     * &nbsp;&nbsp;_formattedType {String} Type of the formattedBody (Text or HTML).                                                  
     * &nbsp;&nbsp;_charset {String} Original charset of the message. The message is converted in UTF-8 before being sent on DBUS.    
     * &nbsp;&nbsp;_mailTo {Object} List of recipients.
     * &nbsp;&nbsp;_mailAtt (JSON) : List of attachments.
     * }
     * </pre>
     *
     * The parameter _mailTo has the following structs:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idMailTo {Number} Unique mail attachment identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_toAdress {String} Recipient Address.
     * }
     * </pre>
     *
     * The parameter _mailAtt has the following structs:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idMailAtt {Number} Unique mail attachment identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_idMail {Number} Unique mail identifier, calculated by a database sequence.
     * &nbsp;&nbsp;_fileId {Number} ID of the file representing the attachment.
     * &nbsp;&nbsp;_mimeType {String} Mime type of the file.
     * &nbsp;&nbsp;_fileName {String} Name of the attachment.                                
     * &nbsp;&nbsp;_ignoredAttachments {Boolean} Set to true if some attachments have been ignored.                                       
     * }
     * </pre>
     *
     *
     * The callback will be called with the status of sent, Possible value are:
     *
     * <pre>
     * {                                             
     * &nbsp;&nbsp; -999  ERR_EMAIL_DISABLED 
     * &nbsp;&nbsp; 31 SERVER_ERROR          
     * &nbsp;&nbsp; 20 EMAIL_NOT_FOUND       
     * &nbsp;&nbsp; 23 EMAIL_ALREADY_SENT    
     * &nbsp;&nbsp; 10 MAILBOX_FULL          
     * &nbsp;&nbsp; 5 Linked file not found  
     * &nbsp;&nbsp; 2 Database error         
     * &nbsp;&nbsp; 0 SUCCESS                 
     * }
     * </pre>                  
     */          
		updateAndSendMail : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IctMail",
            methodName : "updateAndSendMail",
            context : args.context,
            params: args.params,
            paramType: ["Mail"]
        });
        args = null;
        }

/**
 * @event MAILBOX_FULL_NTF
 * This event is used to sent after various actions that fail due to reachet number of locally stored emails limit.
 */

/**
 * @event MAIL_DELETE_NTF
 * This event is used to notify that a mail was deleted.
 * @param {Object} param contains the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp;status {Number} The status of deletion.
 * &nbsp;&nbsp;idMail {Number} ID of the deleted mail.
 * }
 * </pre>
 */

/**
 * @event MAIL_GET_ERROR_NTF
 * This event is used to notify that the method {@link #getAllMails} or {@link #getUnreadMails} failed.
 * @param {Number} param contains the status of action.
 */

/**
 * @event MAIL_RECEIVE_NTF
 * This event is used to notify that a mail was received.
 * @param {Object} param contains the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp;status {Number} The status of deletion.
 * &nbsp;&nbsp;nbnew {Number} Number of new mail.
 * &nbsp;&nbsp;nbunread {Number} Number of unread mail.
 * &nbsp;&nbsp;nbtot {Number} Total number of mail.
 * }
 * </pre>
 */

/**
 * @event MAIL_SAVE_NTF
 * This event is used to notify that a mail was saved.
 * @param {Object} record contains the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp;status {Number} The status of deletion.
 * &nbsp;&nbsp;idMail {Number} ID of the deleted mail.
 * }
 * </pre>
 */

/**
 * @event MAIL_SEND_NTF
 * This event is used to notify that a mail was send.
 * @param {Object} param contains the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp;status {Number} The status of send.
 * &nbsp;&nbsp;idMail {Number} ID of the deleted mail.
 * }
 * </pre>
 */

/**
 * @event MAIL_UPDATE_NTF
 * This event is used to notify that a mail was updated.
 * @param {Object} param contains the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp;status {Number} The status of update.
 * &nbsp;&nbsp;idMail {Number} ID of the deleted mail.
 * }
 * </pre>
 */

});

ICTouchAPI.APIServices.Email = new ICTouchAPI.APIServices.Email();
/**
 * @class ICTouchAPI.APIServices.InstantMessaging
 * @ignore
 * @extends Object
 * The IctIM module provides all Instant Messaging (IM) and presence features :<br>
 * <br>-Change the IM presence of the user (the telephonic presence is managed automatically),
 * <br>-Manage a buddy list : add/remove buddies and associate a nickname ("display name") to each of them,
 * <br> -Be notified about the presence changes of all buddies,
 * <br>-Send instant messages to a buddy,
 * <br> -Receive instant messages from any contact, and answer to it.
 * <br><br>This module sends notifications about :<br>
 * <br>-Buddy list modifications,
 * <br>-Presence modifications relative to the ictouch user and its buddies,
 * <br>-Instant messages sent and received.
 */
 
 
dojo.provide("ICTouchAPI.APIServices.InstantMessaging");
dojo.declare("ICTouchAPI.APIServices.InstantMessaging",[ICTouchAPI.APIServices.APIServicesBase],
{
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Send a new instant message to answer to the conversation corresponding to the given identifier.If the given conversation is already closed, a new one will be opened.</b>
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;idConversation {Number}  the identifier of the conversation
     * &nbsp;&nbsp;message {String} the content of the instant message
     * }
     * </pre>
     *
     * The callback will be called with a value,but it is deprecated, do not use
     *
     * <pre>
     * {Number} deprecated, do not use
     * </pre>
     */
	
	answerIM : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctIM",
			methodName : "answerIM",
			context : args.context,
			params: args.params,
			paramType: ["int", "QString"],
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b> the identifier of the conversation</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Send a instant message to the contact corresponding to the given identifier. A new conversation will be created if necessary.</b>
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;contactId {String} the unique identifier of the contact
     * &nbsp;&nbsp;message {String} the content of the instant message
     * }
     * </pre>
     *
     * The callback will be called with the identifier of the conversation created
     *
     * <pre>
     * {Number} the identifier of the conversation created
     * </pre>
     */

	sendIM : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctIM",
			methodName : "sendIM",
			context : args.context,
			params: args.params,
			paramType: ["QString", "QString"],
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>the list of Conversation objects</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Retrieve the conversation corresponding to the given identifier.</b>
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the unique identifier of the conversation
     * <pre>
     * {Number} the unique identifier of the conversation
     * </pre>
     *
     * The callback will be called with the list of Conversation objects,each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; _idConversation {String} The unique identifier of the instant conversation.
     * &nbsp;&nbsp; _phoneNumber {String}  The phone number of the distant associated to the conversation.
     * &nbsp;&nbsp; _contactId {String}  The contactId, unique identifier of the distant (empty if the distant is not a buddy).
     * &nbsp;&nbsp; _messageList (JSON)  List of the 15 last instant messages of the conversation.See the description below.
     * &nbsp;&nbsp; _conversationDateTime {Date}  The date&time of the Conversation (set when closing the conversation).
     * &nbsp;&nbsp; _isRead {Boolean} Indicates if the conversation has been read or not by the ictouch user.
     * &nbsp;&nbsp; _isClosed {Boolean} Indicates if the conversation has been closed or not by the ictouch user. A conversation must be closed before being deleted.
     * &nbsp;&nbsp; _sipuaImId {String}  The unique identifier used by SIPUA to identify the conversation.
     * }
     * </pre>
     *
     *The parameter _messageList is  the List of the 15 last instant messages of the conversation,each of them has the following structure:
     * <pre>
     * {
     * &nbsp;&nbsp; _idInstantMessage {Number} The unique identifier of the instant message.
     * &nbsp;&nbsp; _direction {Number}The direction of the instant message (0 for an incoming message, 1 for an outgoing message).
     * &nbsp;&nbsp; _contactId {String} The unique identifier the distant as a contact.
     * &nbsp;&nbsp; _message {String} The text content of the instant message.
     * &nbsp;&nbsp; _sendTime {Date} The date&time when the instant message was sent.
     * &nbsp;&nbsp; _receivedTime {Date} The date&time when the instant message was received.
     * &nbsp;&nbsp; _idConversation {Number} The identifier of the Conversation containing this instant message.   
     * }
     * </pre>       
     */

	getConversationById : function(args) {
	this.callCoreApplicationMethod({
		domainName : "IctIM",
		methodName : "getConversationById",
		context : args.context,
        params: args.params,
        paramType: ["int"],
		callback : args.callback,
		callbackParams: args.callbackParams
	});
	args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Set as read the conversation corresponding to the given identifier.</b>
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the unique identifier of the conversation
     * <pre>
 	   * {Number} the unique identifier of the conversation
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */

	setConversationAsRead : function(args) {
	this.callCoreApplicationMethod({
		domainName : "IctIM",
		methodName : "setConversationAsRead",
		context : args.context,
        params: args.params,
        paramType: ["int"],
		callback : args.callback,
		callbackParams: args.callbackParams
	});
	args = null;
	},

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>capability of the module</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the name of the capability
     * <pre>
     * {String} the name of the capability
     * </pre>
     *
     * The callback will be called with the capability, following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;name {String} 
     * &nbsp;&nbsp;state {Number} 
     * }
     * </pre>
     */

	getCapability : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctIM",
            methodName: "getCapability",
            context: args.context,
            params: args.params,
            paramType: ["QString"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Update the IM presence of the current ictouch user.</b>
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;idIMStatus {Number} the identifier of the new status (-1 to ignore)
     * &nbsp;&nbsp; message {String} a personnalized message
     * }
     * </pre>
     *
     * The callback will be called with void
     *
     * <pre>
     * {void} void
     * </pre>
     */

	setStatus : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctIM",
            methodName: "setStatus",
            context: args.context,
            params: args.params,
            paramType: ["int" , "QString"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>the value of the Status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Getter method for attribute Status.</b>
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     *
     * The callback will be called with status value
     *
     * <pre>
     * {Number} status value
     * </pre>
     */

	getStatus : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctIM",
            methodName: "getStatus",
            context: args.context,
            params: args.params,
            paramType: [],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>the presence data</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Retrieve the presence data associated to the current ictouch user.</b>
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     *
     * The callback will be called with he list of Presence objects,each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; _imPresenceValue {JSON} The IM presence status of the user.See the description below.
     * &nbsp;&nbsp; _telephonicPresenceValue {JSON} The telephonic presence status of the user.See the description below.
     * &nbsp;&nbsp; _personnalizedMessage {String} A personalized message that can be filled by the user.
     * &nbsp;&nbsp;                                to give more details about its presence status.
     * &nbsp;&nbsp; _imAddress {String} The IM address of the user.
     * &nbsp;&nbsp; _phoneNumber {String} The phone number of the user.
     * &nbsp;&nbsp; _displayName {String} The display name (nickname) used to identify a buddy.
     * &nbsp;&nbsp;                       This name will be displayed instead of the phone number when an instant message is received from this user.
     * &nbsp;&nbsp; _groupName {String} The name of the buddy list containing this user.
     * &nbsp;&nbsp; _contactId {String} The unique identifier of the user as a contact.
     * }
     * </pre>
     *
     * The parameter _imPresenceValue and _telephonicPresenceValue are the presence statuses ,each of them has the following structure:
     * <pre>
     * {
     * &nbsp;&nbsp; _idStatus {Number} The unique identifier of the status. See _labelStatus for possible values.
     * &nbsp;&nbsp; _labelStatus {String} The label of the status.See the description below.
     * &nbsp;&nbsp; _myTeamWorkStatus {Number} Status value for MyTeamwork. N.B : deprecated.
     * }
     * </pre>   
     *   
     * The _labelStatus Possible values are :
     * <pre>
     *-"ACTIVITY_APPEAR_OFFLINE" (_idStatus=1) : can be set manually,
     *-"ACTIVITY_ONLINE" (_idStatus=2) : can be set manually,
     *-"ACTIVITY_AWAY" (_idStatus=3) : automatic status,
     *-"ACTIVITY_LUNCH" (_idStatus=4) : can be set manually,
     *-"ACTIVITY_ONTHEPHONE" (_idStatus=5) : automatic status,
     *-"ACTIVITY_OTHER" (_idStatus=6) : read-only status, not used yet,
     *-"ACTIVITY_UNKNOWN" (_idStatus=7) : default status,
     *-"ACTIVITY_BUSY" (_idStatus=8) : can be set manually,
     *-"ACTIVITY_BE_RIGHT_BACK" (_idStatus=9) : can be set manually.
     * </pre>     
     */


	getPresence : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctIM",
            methodName: "getPresence",
            context: args.context,
            params: args.params,
            paramType: [],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>all the possible values for PresenceStatus</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Retrieve all the possible values for PresenceStatus.</b>
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     *
     * The callback will be called with the list of PresenceStatus objects,each of them following this structure:
     * <pre>
     * {
     * &nbsp;&nbsp; _idStatus {Number} The unique identifier of the status. See _labelStatus for possible values.
     * &nbsp;&nbsp; _labelStatus {String} The label of the status.See the description below.
     * &nbsp;&nbsp; _myTeamWorkStatus {Number} Status value for MyTeamwork. N.B : deprecated.
     * }
     * </pre>   
     *   
     * The _labelStatus Possible values are :
     * <pre>
     *-"ACTIVITY_APPEAR_OFFLINE" (_idStatus=1) : can be set manually,
     *-"ACTIVITY_ONLINE" (_idStatus=2) : can be set manually,
     *-"ACTIVITY_AWAY" (_idStatus=3) : automatic status,
     *-"ACTIVITY_LUNCH" (_idStatus=4) : can be set manually,
     *-"ACTIVITY_ONTHEPHONE" (_idStatus=5) : automatic status,
     *-"ACTIVITY_OTHER" (_idStatus=6) : read-only status, not used yet,
     *-"ACTIVITY_UNKNOWN" (_idStatus=7) : default status,
     *-"ACTIVITY_BUSY" (_idStatus=8) : can be set manually,
     *-"ACTIVITY_BE_RIGHT_BACK" (_idStatus=9) : can be set manually.
     * </pre>     
     */

	getAvailableStatuses : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctIM",
            methodName: "getAvailableStatuses",
            context: args.context,
            params: args.params,
            paramType: [],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },


    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>the presence data</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Retrieve the presence data associated to the current ictouch user.</b>
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     *
     * The callback will be called with he list of Presence objects,each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; _imPresenceValue {JSON} The IM presence status of the user.See the description below.
     * &nbsp;&nbsp; _telephonicPresenceValue {JSON} The telephonic presence status of the user.See the description below.
     * &nbsp;&nbsp; _personnalizedMessage {String} A personalized message that can be filled by the user.
     * &nbsp;&nbsp;                                to give more details about its presence status.
     * &nbsp;&nbsp; _imAddress {String} The IM address of the user.
     * &nbsp;&nbsp; _phoneNumber {String} The phone number of the user.
     * &nbsp;&nbsp; _displayName {String} The display name (nickname) used to identify a buddy.
     * &nbsp;&nbsp;                       This name will be displayed instead of the phone number when an instant message is received from this user.
     * &nbsp;&nbsp; _groupName {String} The name of the buddy list containing this user.
     * &nbsp;&nbsp; _contactId {String} The unique identifier of the user as a contact.
     * }
     * </pre>
     *
     *The parameter _imPresenceValue and _telephonicPresenceValue are the presence statuses ,each of them has the following structure:
     * <pre>
     * {
     * &nbsp;&nbsp; _idStatus {Number} The unique identifier of the status. See _labelStatus for possible values.
     * &nbsp;&nbsp; _labelStatus {String} The label of the status.See the description below.
     * &nbsp;&nbsp; _myTeamWorkStatus {Number} Status value for MyTeamwork. N.B : deprecated.
     * }
     * </pre>   
     *   
     * The _labelStatus Possible values are :
     * <pre>
     *-"ACTIVITY_APPEAR_OFFLINE" (_idStatus=1) : can be set manually,
     *-"ACTIVITY_ONLINE" (_idStatus=2) : can be set manually,
     *-"ACTIVITY_AWAY" (_idStatus=3) : automatic status,
     *-"ACTIVITY_LUNCH" (_idStatus=4) : can be set manually,
     *-"ACTIVITY_ONTHEPHONE" (_idStatus=5) : automatic status,
     *-"ACTIVITY_OTHER" (_idStatus=6) : read-only status, not used yet,
     *-"ACTIVITY_UNKNOWN" (_idStatus=7) : default status,
     *-"ACTIVITY_BUSY" (_idStatus=8) : can be set manually,
     *-"ACTIVITY_BE_RIGHT_BACK" (_idStatus=9) : can be set manually.
     * </pre>     
     */

	getPresence : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctIM",
            methodName: "getPresence",
            context: args.context,
            params: args.params,
            paramType: [],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b> the presence data associated to the contact </b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Retrieve the presence data associated to the contact corresponding to the given identifier.</b>
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the unique identifier of the contact
     * <pre>
     * {String} the unique identifier of the contact
     * </pre>
     *
     * The callback will be called with he list of Presence objects,each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; _imPresenceValue {JSON} The IM presence status of the user.See the description below.
     * &nbsp;&nbsp; _telephonicPresenceValue {JSON} The telephonic presence status of the user.See the description below.
     * &nbsp;&nbsp; _personnalizedMessage {String} A personalized message that can be filled by the user.
     * &nbsp;&nbsp;                                to give more details about its presence status.
     * &nbsp;&nbsp; _imAddress {String} The IM address of the user.
     * &nbsp;&nbsp; _phoneNumber {String} The phone number of the user.
     * &nbsp;&nbsp; _displayName {String} The display name (nickname) used to identify a buddy.This name will be displayed
     * &nbsp;&nbsp;                       instead of the phone number when an instant message is received from this user.
     * &nbsp;&nbsp; _groupName {String} The name of the buddy list containing this user.
     * &nbsp;&nbsp; _contactId {String} The unique identifier of the user as a contact.
     * }
     * </pre>
     *
     *The parameter _imPresenceValue and _telephonicPresenceValue are the presence statuses ,each of them has the following structure:
     * <pre>
     * {
     * &nbsp;&nbsp; _idStatus {Number} The unique identifier of the status. See _labelStatus for possible values.
     * &nbsp;&nbsp; _labelStatus {String} The label of the status.See the description below.
     * &nbsp;&nbsp; _myTeamWorkStatus {Number} Status value for MyTeamwork. N.B : deprecated.
     * }
     * </pre>   
     *   
     * The _labelStatus Possible values are :
     * <pre>
     *-"ACTIVITY_APPEAR_OFFLINE" (_idStatus=1) : can be set manually,
     *-"ACTIVITY_ONLINE" (_idStatus=2) : can be set manually,
     *-"ACTIVITY_AWAY" (_idStatus=3) : automatic status,
     *-"ACTIVITY_LUNCH" (_idStatus=4) : can be set manually,
     *-"ACTIVITY_ONTHEPHONE" (_idStatus=5) : automatic status,
     *-"ACTIVITY_OTHER" (_idStatus=6) : read-only status, not used yet,
     *-"ACTIVITY_UNKNOWN" (_idStatus=7) : default status,
     *-"ACTIVITY_BUSY" (_idStatus=8) : can be set manually,
     *-"ACTIVITY_BE_RIGHT_BACK" (_idStatus=9) : can be set manually.
     * </pre>     
     */

	getPresenceForContact : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctIM",
            methodName: "getPresenceForContact",
            context: args.context,
            params: args.params,
            paramType: ["QString"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b> the presence data associated to the contacts </b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Retrieve the presence data associated to the contacts corresponding to the given identifiers.</b>
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains a list of contactId,each of them has the following 
     * <pre>
     * {String} the unique identifier of the contact
     * </pre>
     *
     * The callback will be called with he list of Presence objects,each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; _imPresenceValue {JSON} The IM presence status of the user.See the description below.
     * &nbsp;&nbsp; _telephonicPresenceValue {JSON} The telephonic presence status of the user.See the description below.
     * &nbsp;&nbsp; _personnalizedMessage {String} A personalized message that can be filled by the user.
     * &nbsp;&nbsp;                                to give more details about its presence status.
     * &nbsp;&nbsp; _imAddress {String} The IM address of the user.
     * &nbsp;&nbsp; _phoneNumber {String} The phone number of the user.
     * &nbsp;&nbsp; _displayName {String} The display name (nickname) used to identify a buddy.
     * &nbsp;&nbsp;                       This name will be displayed instead of the phone number when an instant message is received from this user.
     * &nbsp;&nbsp; _groupName {String} The name of the buddy list containing this user.
     * &nbsp;&nbsp; _contactId {String} The unique identifier of the user as a contact.
     * }
     * </pre>
     *
     *The parameter _imPresenceValue and _telephonicPresenceValue are the presence statuses ,each of them has the following structure:
     * <pre>
     * {
     * &nbsp;&nbsp; _idStatus {Number} The unique identifier of the status. See _labelStatus for possible values.
     * &nbsp;&nbsp; _labelStatus {String} The label of the status.See the description below.
     * &nbsp;&nbsp; _myTeamWorkStatus {Number} Status value for MyTeamwork. N.B : deprecated.
     * }
     * </pre>   
     *   
     * The _labelStatus Possible values are :
     * <pre>
     *-"ACTIVITY_APPEAR_OFFLINE" (_idStatus=1) : can be set manually,
     *-"ACTIVITY_ONLINE" (_idStatus=2) : can be set manually,
     *-"ACTIVITY_AWAY" (_idStatus=3) : automatic status,
     *-"ACTIVITY_LUNCH" (_idStatus=4) : can be set manually,
     *-"ACTIVITY_ONTHEPHONE" (_idStatus=5) : automatic status,
     *-"ACTIVITY_OTHER" (_idStatus=6) : read-only status, not used yet,
     *-"ACTIVITY_UNKNOWN" (_idStatus=7) : default status,
     *-"ACTIVITY_BUSY" (_idStatus=8) : can be set manually,
     *-"ACTIVITY_BE_RIGHT_BACK" (_idStatus=9) : can be set manually.
     * </pre>     
     */

	getPresenceForContacts : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctIM",
            methodName: "getPresenceForContacts",
            context: args.context,
            params: args.params,
            paramType: ["QStringList"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Modify the display name of a contact belonging to the buddy list.</b>
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp; contactId {String} the unique identifier of the contact
     * &nbsp;&nbsp; displayName {String} the new display name of the buddy
     * }
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */

	modifyContactfromBuddyList : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctIM",
            methodName: "modifyContactFromBuddyList",
            context: args.context,
            params: args.params,
            paramType: ["QString", "QString"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Add a contact to the buddy list.</b>
     * <br>contactId is used as the main key (if the imAddress corresponding to the contactId is not the same as the imAddress parameter, we use the imAddress corresponding to the contactId)
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp; imAddress {String} the IM address of the contact
     * &nbsp;&nbsp; contactId {String} the unique identifier of the contact
     * &nbsp;&nbsp; displayName {String} the display name of the new buddy
     * }
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */

	addContactInBuddyList : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctIM",
            methodName: "addContactInBuddyList",
            context: args.context,
            params: args.params,
            paramType: ["QString", "QString", "QString"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Remove a contact from the buddy list.</b>
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the unique identifier of the contact
     * <pre>
     * {String} the unique identifier of the contact
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */

	removeContactFromBuddyList : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctIM",
            methodName: "removeContactFromBuddyList",
            context: args.context,
            params: args.params,
            paramType: ["QString"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

  /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Close the conversation corresponding to the given identifier.</b>
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the unique identifier of the conversation
     * <pre>
     * {Number}  the unique identifier of the conversation
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */

	closeConversation : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctIM",
            methodName: "closeConversation",
            context: args.context,
            params: args.params,
            paramType: ["int"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

  /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Search the collaborationId associated with the given login in ContactProfile.
     * The response will be sent asynchronously in {@link #IM_EVENT_GET_COLLABID_RESULT}.</b>
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the login used as search key
     * <pre>
     * {String} The login used as search key
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */

	getCollaborationId : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctIM",
            methodName: "getCollaborationId",
            context: args.context,
            params: args.params,
            paramType: ["QString"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },
    
/**
 * @event IM_BUDDY_ADDED
 * This event is used to notify that a new buddy has been added to the buddy list of the ictouch user.
 * @param {String} param contains the IM address of the added buddy.
 */	 
 
/**
 * @event IM_BUDDY_DELETED
 * This event is used to notify that a buddy has been removed from the buddy list of the ictouch user.
 * @param {String} param contains the IM address of the removed buddy.
 */	 
 
 /**
 * @event IM_BUDDY_LIST_MODIFIED
 * This event is used to notify that the buddy list content has changed.
 * @param {String} param contains the name of the updated buddy list.
 */	 
 
 /**
 * @event IM_CONVERSATION_READ 
 * This event is used to notify that a conversation has been read.
 * @param {Number} param contains the unique identifier of the conversation.
 */

 /**
 * @event IM_CONVERSATION_DELETED
 * This event is used to notify that a conversation has been deleted.
 * All instant messages belonging to this conversation were deleted too.
 * @param {Number} param contains the unique identifier of the conversation.
 */

 /**
 * @event IM_CONVERSATION_CLOSED 
 * This event is used to notify that a conversation has been closed,and can't be used anymore to store instant messages. If the user wants to send an instant
 * message to the same user, a new conversation will be created.
 * @param {Number} param contains the unique identifier of the conversation.
 */

 /**
 * @event IM_SENT 
 * This event is used to notify that IM and presence features are now ready to be used (presence subscribed and buddy list retrieved).
 * @param {JSON} param contains the following structure:
 * <pre>
 * {    
 * &nbsp;&nbsp;message {JSON}  the data of the message.See the description below.
 * &nbsp;&nbsp;displayName {String} the display name (nickname) of the sender.
 * &nbsp;&nbsp;phoneNumber {String} the phone number of the sender.
 * }
 * </pre>
 *
 * The parameter message has the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp; _idInstantMessage {Number} The unique identifier of the instant message.
 * &nbsp;&nbsp; _direction {Number}The direction of the instant message (0 for an incoming message, 1 for an outgoing message).
 * &nbsp;&nbsp; _contactId {String} The unique identifier the distant as a contact.
 * &nbsp;&nbsp; _message {String} The text content of the instant message.
 * &nbsp;&nbsp; _sendTime {Date} The date&time when the instant message was sent.
 * &nbsp;&nbsp; _receivedTime {Date} The date&time when the instant message was received.
 * &nbsp;&nbsp; _idConversation {Number} The identifier of the Conversation containing this instant message.   
 * }
 * </pre> 
 */
 
 /**
 * @event IM_RECEIVED 
 * This event is used to notify that a new instant message has been received.
 * @param {JSON} param contains the following structure:
 * <pre>
 * {    
 * &nbsp;&nbsp;message {JSON}  the data of the message.See the description below.
 * &nbsp;&nbsp;displayName {String} the display name (nickname) of the distant.
 * &nbsp;&nbsp;phoneNumber {String} the phone number of the distant.
 * }
 * </pre>
 *
 * The parameter message has the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp; _idInstantMessage {Number} The unique identifier of the instant message.
 * &nbsp;&nbsp; _direction {Number}The direction of the instant message (0 for an incoming message, 1 for an outgoing message).
 * &nbsp;&nbsp; _contactId {String} The unique identifier the distant as a contact.
 * &nbsp;&nbsp; _message {String} The text content of the instant message.
 * &nbsp;&nbsp; _sendTime {Date} The date&time when the instant message was sent.
 * &nbsp;&nbsp; _receivedTime {Date} The date&time when the instant message was received.
 * &nbsp;&nbsp; _idConversation {Number} The identifier of the Conversation containing this instant message.   
 * }
 * </pre> 
 */
 
 /**
 * @event IM_EVENT_GET_COLLABID_RESULT 
 * This event is used to return the result of a call to IctIM::getCollaborationId.
 * @param {JSON} param contains the following structure:
 * <pre>
 * {    
 * &nbsp;&nbsp;login {String} the login used as the search key. 
 * &nbsp;&nbsp;collaborationId {String} the collaborationId that was found (empty if nothing found).
 * }
 * </pre> 
 */
 
 /**
 * @event IM_PRESENCE_CHANGE 
 * This event is used to notify that the presence of one buddy at least has been changed.
 * @param {JSON} param contains the list of updated data of each presence,each of them following the structure:
 * <pre>
 * {
 * &nbsp;&nbsp; _imPresenceValue {JSON} The IM presence status of the user.See the description below.
 * &nbsp;&nbsp; _telephonicPresenceValue {JSON} The telephonic presence status of the user.See the description below.
 * &nbsp;&nbsp; _personnalizedMessage {String} A personalized message that can be filled by the user.
 * &nbsp;&nbsp;                                to give more details about its presence status.
 * &nbsp;&nbsp; _imAddress {String} The IM address of the user.
 * &nbsp;&nbsp; _phoneNumber {String} The phone number of the user.
 * &nbsp;&nbsp; _displayName {String} The display name (nickname) used to identify a buddy.
 * &nbsp;&nbsp;                       This name will be displayed instead of the phone number when an instant message is received from this user.
 * &nbsp;&nbsp; _groupName {String} The name of the buddy list containing this user.
 * &nbsp;&nbsp; _contactId {String} The unique identifier of the user as a contact.
 * }
 * </pre>
 *
 * The parameter _imPresenceValue and _telephonicPresenceValue are the presence statuses ,each of them has the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp; _idStatus {Number} The unique identifier of the status. See _labelStatus for possible values.
 * &nbsp;&nbsp; _labelStatus {String} The label of the status.See the description below.
 * &nbsp;&nbsp; _myTeamWorkStatus {Number} Status value for MyTeamwork. N.B : deprecated.
 * }
 * </pre>   
 *   
 * The _labelStatus Possible values are :
 * <pre>
 *-"ACTIVITY_APPEAR_OFFLINE" (_idStatus=1) : can be set manually,
 *-"ACTIVITY_ONLINE" (_idStatus=2) : can be set manually,
 *-"ACTIVITY_AWAY" (_idStatus=3) : automatic status,
 *-"ACTIVITY_LUNCH" (_idStatus=4) : can be set manually,
 *-"ACTIVITY_ONTHEPHONE" (_idStatus=5) : automatic status,
 *-"ACTIVITY_OTHER" (_idStatus=6) : read-only status, not used yet,
 *-"ACTIVITY_UNKNOWN" (_idStatus=7) : default status,
 *-"ACTIVITY_BUSY" (_idStatus=8) : can be set manually,
 *-"ACTIVITY_BE_RIGHT_BACK" (_idStatus=9) : can be set manually.
 * </pre>     
 */
 

 /**
 * @event IM_SET_PRESENCE_NTF 
 * This event is used to notify that the presence of the ictouch user has been changed.
 * @param {JSON} param contains the updated data of the presence,following the structure:
 * <pre>
 * {
 * &nbsp;&nbsp; _imPresenceValue {JSON} The IM presence status of the user.See the description below.
 * &nbsp;&nbsp; _telephonicPresenceValue {JSON} The telephonic presence status of the user.See the description below.
 * &nbsp;&nbsp; _personnalizedMessage {String} A personalized message that can be filled by the user.
 * &nbsp;&nbsp;                                to give more details about its presence status.
 * &nbsp;&nbsp; _imAddress {String} The IM address of the user.
 * &nbsp;&nbsp; _phoneNumber {String} The phone number of the user.
 * &nbsp;&nbsp; _displayName {String} The display name (nickname) used to identify a buddy.
 * &nbsp;&nbsp;                       This name will be displayed instead of the phone number when an instant message is received from this user.
 * &nbsp;&nbsp; _groupName {String} The name of the buddy list containing this user.
 * &nbsp;&nbsp; _contactId {String} The unique identifier of the user as a contact.
 * }
 * </pre>
 *
 * The parameter _imPresenceValue and _telephonicPresenceValue are the presence statuses ,each of them has the following structure:
 * <pre>
 * {
 * &nbsp;&nbsp; _idStatus {Number} The unique identifier of the status. See _labelStatus for possible values.
 * &nbsp;&nbsp; _labelStatus {String} The label of the status.See the description below.
 * &nbsp;&nbsp; _myTeamWorkStatus {Number} Status value for MyTeamwork. N.B : deprecated.
 * }
 * </pre>   
 *   
 * The _labelStatus Possible values are :
 * <pre>
 *-"ACTIVITY_APPEAR_OFFLINE" (_idStatus=1) : can be set manually,
 *-"ACTIVITY_ONLINE" (_idStatus=2) : can be set manually,
 *-"ACTIVITY_AWAY" (_idStatus=3) : automatic status,
 *-"ACTIVITY_LUNCH" (_idStatus=4) : can be set manually,
 *-"ACTIVITY_ONTHEPHONE" (_idStatus=5) : automatic status,
 *-"ACTIVITY_OTHER" (_idStatus=6) : read-only status, not used yet,
 *-"ACTIVITY_UNKNOWN" (_idStatus=7) : default status,
 *-"ACTIVITY_BUSY" (_idStatus=8) : can be set manually,
 *-"ACTIVITY_BE_RIGHT_BACK" (_idStatus=9) : can be set manually.
 * </pre>     
 */

});

ICTouchAPI.APIServices.InstantMessaging = new ICTouchAPI.APIServices.InstantMessaging();
/**
 * @class ICTouchAPI.APIServices.ProgrammableKeys
 * @singleton
 * @extends ICTouchAPI.APIServices.APIServicesBase
 * IctProgrammableKeys module manages the keys of the Programmable Keys application.<br>
 * A key comes along with an action to do. This module also manages actions.<br>
 * The keys and actions are saved in a persistent database.
 */
 
 
dojo.provide("ICTouchAPI.APIServices.ProgrammableKeys");
dojo.declare("ICTouchAPI.APIServices.ProgrammableKeys",[ICTouchAPI.APIServices.APIServicesBase],
{
	
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>all keys</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * 
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with the list of keys,each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idKey {Number} The key object identifier
     * &nbsp;&nbsp;_idDM {String} Ignored
     * &nbsp;&nbsp;_idAction {Number} The action object identifier
     * &nbsp;&nbsp;_container {String} An arbitrary string chosen by the application
     * &nbsp;&nbsp;_position {Number} The button order displayed on the screen
     * &nbsp;&nbsp;_icon {String} The file relative name of the icon
     * &nbsp;&nbsp;_type {Number} possible value is 0 or 1 or -1 
     * &nbsp;&nbsp;_aom_type {Number} type
     * &nbsp;&nbsp;_aom_addr {Number} address
     * }
     * </pre>
     */	
    getAllProgrammableKeys : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctProgrammableKeys",
            methodName: "getAllProgrammableKeys",
            context: args.context,
            params: args.params,
            paramType: [],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },
    
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>all keys</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Retrieve all keys that fit a given container</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the container
     * <pre>
     * {String} the container
     * </pre>
     *
     * The callback will be called with the list of keys,each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idKey {Number} The key object identifier
     * &nbsp;&nbsp;_idDM {String} Ignored
     * &nbsp;&nbsp;_idAction {Number} The action object identifier
     * &nbsp;&nbsp;_container {String} An arbitrary string chosen by the application
     * &nbsp;&nbsp;_position {Number} The button order displayed on the screen
     * &nbsp;&nbsp;_icon {String} The file relative name of the icon
     * &nbsp;&nbsp;_type {Number} possible value is 0 or 1 or -1
     * &nbsp;&nbsp;_aom_type {Number} type
     * &nbsp;&nbsp;_aom_addr {Number} address
     * }
     * </pre>
     */	
	getProgrammableKeysByContainer : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctProgrammableKeys",
            methodName: "getProgrammableKeysByContainer",
            context: args.context,
            params: args.params,
            paramType: ["QString"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},
	
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>key identifier</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Create a key</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the keys of the Programmable Key application,it has the following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;_idKey {Number} The key object identifier
     * &nbsp;&nbsp;_idDM {String} Ignored
     * &nbsp;&nbsp;_idAction {Number} The action object identifier
     * &nbsp;&nbsp;_container {String} An arbitrary string chosen by the application
     * &nbsp;&nbsp;_position {Number} The button order displayed on the screen
     * &nbsp;&nbsp;_icon {String} The file relative name of the icon
     * &nbsp;&nbsp;_type {Number} possible value is 0 or 1 or -1
     * &nbsp;&nbsp;_aom_type {Number} type
     * &nbsp;&nbsp;_aom_addr {Number} address
     * }
     * </pre>
     *
     * The callback will be called with the key identifier,possible values are :
     * <pre>
     * - -2 : Error
     * - -1 : The data base is full
     * -  n  : The key identifier
     * </pre>
     */	
	
	createProgrammableKey : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctProgrammableKeys",
            methodName: "createProgrammableKey",
            context: args.context,
            params: args.params,
            paramType: ["Key"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},
	
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Delete a key</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the key id
     * <pre>
     * {Number} the key id
     * </pre>
     *
     * The callback will be called with status,possible values are :
     * <pre>
     *  - -2 : Error
     *  -  0 : Success
     * </pre>
     */	
	deleteProgrammableKey : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctProgrammableKeys",
            methodName: "deleteProgrammableKey",
            context: args.context,
            params: args.params,
            paramType: ["qlonglong"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},
	
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>key</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Retrieve a key</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the key id
     * <pre>
     * {Number} the key id
     * </pre>
     *
     * The callback will be called with the list of keys
     * -A list of one key in case of success
     * -A list of null length if the key has not been found
     * each of keys following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;_idKey {Number} The key object identifier
     * &nbsp;&nbsp;_idDM {String} Ignored
     * &nbsp;&nbsp;_idAction {Number} The action object identifier
     * &nbsp;&nbsp;_container {String} An arbitrary string chosen by the application
     * &nbsp;&nbsp;_position {Number} The button order displayed on the screen
     * &nbsp;&nbsp;_icon {String} The file relative name of the icon
     * &nbsp;&nbsp;_type {Number} possible value is 0 or 1 or -1
     * &nbsp;&nbsp;_aom_type {Number} type
     * &nbsp;&nbsp;_aom_addr {Number} address
     * }
     * </pre>
     */		
	
	getProgrammableKey : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctProgrammableKeys",
            methodName: "getProgrammableKey",
            context: args.context,
            params: args.params,
            paramType: ["qlonglong"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},
	
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Change a key</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the keys of the Programmable Key application,it has the following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;_idKey {Number} The key object identifier
     * &nbsp;&nbsp;_idDM {String} Ignored
     * &nbsp;&nbsp;_idAction {Number} The action object identifier
     * &nbsp;&nbsp;_container {String} An arbitrary string chosen by the application
     * &nbsp;&nbsp;_position {Number} The button order displayed on the screen
     * &nbsp;&nbsp;_icon {String} The file relative name of the icon
     * &nbsp;&nbsp;_type {Number} possible value is 0 or 1 or -1
     * &nbsp;&nbsp;_aom_type {Number} type
     * &nbsp;&nbsp;_aom_addr {Number} address
     * }
     * </pre>
     *
     * The callback will be called with status,possible values are :
     * <pre>
     *  - -2 : Error
     *  -  0 : Success
     * </pre>
     */	
	modifyProgrammableKey : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctProgrammableKeys",
            methodName: "modifyProgrammableKey",
            context: args.context,
            params: args.params,
            paramType: ["Key"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},
	
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>all the actions</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Retrieve all the actions</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with the list of actions,each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp; _idAction {Number} The action object identifier
     * &nbsp;&nbsp; _idDM {String} Ignored
     * &nbsp;&nbsp; _type {Number} the action type,possible values are:
     * &nbsp;&nbsp;               - 0 for no type,
     * &nbsp;&nbsp;               - 1 for direct call,
     * &nbsp;&nbsp;               - 2 for shortcut access key,
     * &nbsp;&nbsp;               - 3 for shortcut action key,
     * &nbsp;&nbsp;               - 4 for flip flop
     * &nbsp;&nbsp; _action {String} The javascript function name
     * &nbsp;&nbsp; _webAppName {String} The webapp name which contains the function
     * &nbsp;&nbsp; _arguments {String} The arguments required by the function
     * &nbsp;&nbsp; _label {String} The string to display
     * &nbsp;&nbsp; _shortLabel {String} The string to display     
     * }
     * </pre>
     */	
	
	getAllActions : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctProgrammableKeys",
            methodName: "getAllActions",
            context: args.context,
            params: args.params,
            paramType: [],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},
	
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>action identifier</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Create an action</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the action of the Programmable Key application ,it has the following structure:
     * <pre>
     * {
     * &nbsp;&nbsp; _idAction {Number} The action object identifier
     * &nbsp;&nbsp; _idDM {String} Ignored
     * &nbsp;&nbsp; _type {Number} the action type,possible values are:
     * &nbsp;&nbsp;                - 0 for no type,
     * &nbsp;&nbsp;                - 1 for direct call,
     * &nbsp;&nbsp;                - 2 for shortcut access key,
     * &nbsp;&nbsp;                - 3 for shortcut action key,
     * &nbsp;&nbsp;                - 4 for flip flop
     * &nbsp;&nbsp; _action {String} The javascript function name
     * &nbsp;&nbsp; _webAppName {String} The webapp name which contains the function
     * &nbsp;&nbsp; _arguments {String} The arguments required by the function
     * &nbsp;&nbsp; _label {String} The string to display
     * &nbsp;&nbsp; _shortLabel {String} The string to display     
     * }
     * </pre>
     *
     * The callback will be called with the action identifier
     * <pre>
     * {Number} the action identifier
     * </pre>
     */
	addAction : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctProgrammableKeys",
            methodName: "addAction",
            context: args.context,
            params: args.params,
            paramType: ["Action"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},
	   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Change an action</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is a JSON,representing the action of the Programmable Key application ,it has the following structure:
     * <pre>
     * {
     * &nbsp;&nbsp; _idAction {Number} The action object identifier
     * &nbsp;&nbsp; _idDM {String} Ignored
     * &nbsp;&nbsp; _type {Number} the action type,possible values are:
     * &nbsp;&nbsp;                - 0 for no type,
     * &nbsp;&nbsp;                - 1 for direct call,
     * &nbsp;&nbsp;                - 2 for shortcut access key,
     * &nbsp;&nbsp;                - 3 for shortcut action key,
     * &nbsp;&nbsp;                - 4 for flip flop
     * &nbsp;&nbsp; _action {String} The javascript function name
     * &nbsp;&nbsp; _webAppName {String} The webapp name which contains the function
     * &nbsp;&nbsp; _arguments {String} The arguments required by the function
     * &nbsp;&nbsp; _label {String} The string to display
     * &nbsp;&nbsp; _shortLabel {String} The string to display     
     * }
     * </pre>
     *
     * The callback will be called with status,possible values are :
     * <pre>
     *  - -2 : Error
     *  -  0 : Success
     * </pre>
     */	
	modifyAction : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctProgrammableKeys",
            methodName: "modifyAction",
            context: args.context,
            params: args.params,
            paramType: ["Action"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},
	
	   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Delete an action</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the action id
     * <pre>
     * {Number} the action id
     * </pre>
     *
     * The callback will be called with status,possible values are :
     * <pre>
     *  - -2 : Error
     *  -  0 : Success
     * </pre>
     */	
	deleteAction : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctProgrammableKeys",
            methodName: "deleteAction",
            context: args.context,
            params: args.params,
            paramType: ["qlonglong"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},
	
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>action list</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Retrieve an action</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the action id
     * <pre>
     * {Number} the action id
     * </pre>
     *
     * The callback will be called with the list of actions
     * -A list of one action in case of success
     * -A list of null length if the action has not been found
     * each of actions following this structure:
     * <pre>
     * {
     * &nbsp;&nbsp; _idAction {Number} The action object identifier
     * &nbsp;&nbsp; _idDM {String} Ignored
     * &nbsp;&nbsp; _type {Number} the action type,possible values are:
     * &nbsp;&nbsp;                - 0 for no type,
     * &nbsp;&nbsp;                - 1 for direct call,
     * &nbsp;&nbsp;                - 2 for shortcut access key,
     * &nbsp;&nbsp;                - 3 for shortcut action key,
     * &nbsp;&nbsp;                - 4 for flip flop
     * &nbsp;&nbsp; _action {String} The javascript function name
     * &nbsp;&nbsp; _webAppName {String} The webapp name which contains the function
     * &nbsp;&nbsp; _arguments {String} The arguments required by the function
     * &nbsp;&nbsp; _label {String} The string to display
     * &nbsp;&nbsp; _shortLabel {String} The string to display     
     * }
     * </pre>
     */		
	getAction : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctProgrammableKeys",
            methodName: "getAction",
            context: args.context,
            params: args.params,
            paramType: ["qlonglong"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},
	
	   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>status</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Change the position of several keys</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the list of other attributes in a JSON structure (key-value)
     *
     * The callback will be called with status
     * <pre>
     * {Boolean} true when all changes success
     * </pre>
     */	
	changeProgrammableKeysPosition : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctProgrammableKeys",
            methodName: "changeProgrammableKeysPosition",
            context: args.context,
            params: args.params,
            paramType: ["QMap<int,int>"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>capability of the module</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the name of the capability
     * <pre>
     * {String} the name of the capability
     * </pre>
     *
     * The callback will be called with the capability, following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;name {String} 
     * &nbsp;&nbsp;state {Number} 
     * }
     * </pre>
     */

	getCapability : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctProgrammableKeys",
            methodName: "getCapability",
            context: args.context,
            params: args.params,
            paramType: ["QString"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	 }
	 
/**
 * @event KEY_CREATE_NTF
 * This event is used to notify that a key is created.
 * @param {Object} param contains the created key,following the structure:
 * <pre>
 * {
 * &nbsp;&nbsp;_idKey {Number} The key object identifier
 * &nbsp;&nbsp;_idDM {String} Ignored
 * &nbsp;&nbsp;_idAction {Number} The action object identifier
 * &nbsp;&nbsp;_container {String} An arbitrary string chosen by the application
 * &nbsp;&nbsp;_position {Number} The button order displayed on the screen
 * &nbsp;&nbsp;_icon {String} The file relative name of the icon
 * &nbsp;&nbsp;_type {Number} possible value is 0 or 1 or -1 
 * &nbsp;&nbsp;_aom_type {Number} type
 * &nbsp;&nbsp;_aom_addr {Number} address
 * }
 * </pre>
 */

/**
 * @event KEY_MODIFY_NTF
 * This event is used to notify that the key is changed.
 * @param {Object} param contains the changed key,following the structure:
 * <pre>
 * {
 * &nbsp;&nbsp;_idKey {Number} The key object identifier
 * &nbsp;&nbsp;_idDM {String} Ignored
 * &nbsp;&nbsp;_idAction {Number} The action object identifier
 * &nbsp;&nbsp;_container {String} An arbitrary string chosen by the application
 * &nbsp;&nbsp;_position {Number} The button order displayed on the screen
 * &nbsp;&nbsp;_icon {String} The file relative name of the icon
 * &nbsp;&nbsp;_type {Number} possible value is 0 or 1 or -1 
 * &nbsp;&nbsp;_aom_type {Number} type
 * &nbsp;&nbsp;_aom_addr {Number} address
 * }
 * </pre>
 */	 

/**
 * @event KEY_DELETE_NTF
 * This event is used to notify that the key is deleted.
 * @param {Number} param contains the deleted key identifier
 */	 
	 
/**
 * @event ACTION_CREATE_NTF
 * This event is used to notify that an action is created.
 * @param {Object} param contains the created action,following the structure:
 * <pre>
 * {
 * &nbsp;&nbsp; _idAction {Number} The action object identifier
 * &nbsp;&nbsp; _idDM {String} Ignored
 * &nbsp;&nbsp; _type {Number} the action type,possible values are:
 * &nbsp;&nbsp;                - 0 for no type,
 * &nbsp;&nbsp;                - 1 for direct call,
 * &nbsp;&nbsp;                - 2 for shortcut access key,
 * &nbsp;&nbsp;                - 3 for shortcut action key,
 * &nbsp;&nbsp;                - 4 for flip flop
 * &nbsp;&nbsp; _action {String} The javascript function name
 * &nbsp;&nbsp; _webAppName {String} The webapp name which contains the function
 * &nbsp;&nbsp; _arguments {String} The arguments required by the function
 * &nbsp;&nbsp; _label {String} The string to display
 * &nbsp;&nbsp; _shortLabel {String} The string to display     
 * }
 * </pre>
 */	 
 
 /**
 * @event ACTION_MODIFY_NTF
 * This event is used to notify that an action is changed.
 * @param {Object} param contains the changed action,following the structure:
 * <pre>
 * {
 * &nbsp;&nbsp; _idAction {Number} The action object identifier
 * &nbsp;&nbsp; _idDM {String} Ignored
 * &nbsp;&nbsp; _type {Number} the action type,possible values are:
 * &nbsp;&nbsp;                - 0 for no type,
 * &nbsp;&nbsp;                - 1 for direct call,
 * &nbsp;&nbsp;                - 2 for shortcut access key,
 * &nbsp;&nbsp;                - 3 for shortcut action key,
 * &nbsp;&nbsp;                - 4 for flip flop
 * &nbsp;&nbsp; _action {String} The javascript function name
 * &nbsp;&nbsp; _webAppName {String} The webapp name which contains the function
 * &nbsp;&nbsp; _arguments {String} The arguments required by the function
 * &nbsp;&nbsp; _label {String} The string to display
 * &nbsp;&nbsp; _shortLabel {String} The string to display     
 * }
 * </pre>
 */	 
 
 /**
 * @event ACTION_DELETE_NTF
 * This event is used to notify that an action is deleted.
 * @param {Number} param contains the deleted action identifier
 */
 
});

ICTouchAPI.APIServices.ProgrammableKeys = new ICTouchAPI.APIServices.ProgrammableKeys();
/**
 * @ignore
 */
dojo.provide("ICTouchAPI.APIServices.IctUser");
dojo.declare("ICTouchAPI.APIServices.IctUser",[ICTouchAPI.APIServices.APIServicesBase],
{
	lockRequest : function(args) {
        this.callCoreApplicationMethod({
            domainName: "IctUser",
            methodName: "lockRequest",
            context: args.context,
            paramType: [],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

	unLockRequest : function(args) {
		args.params[0]	= Encryption.encryptRSA(args.params[0]);
		
        this.callCoreApplicationMethod({
            domainName: "IctUser",
            methodName: "unLockRequest",
            context: args.context,
            params: args.params,
            paramType: ["QString"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

	login : function() {
		this.callCoreApplicationMethod({
			domainName: "IctUser",
			methodName: "login"
		});
	},

	getManagerList : function(args) {
		this.callCoreApplicationMethod({
			domainName: "IctUser",
			methodName: "getManagerList",
			context: args.context,
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

	getAssistantList : function(args) {
		this.callCoreApplicationMethod({
			domainName: "IctUser",
			methodName: "getAssistantList",
			context: args.context,
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

	consultWhiteList : function(args) {
		this.callCoreApplicationMethod({
			domainName: "IctUser",
			methodName: "consultWhiteList",
			context: args.context,
			params: args.params,
            paramType: [],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

	addContactsInWhiteList : function(args) {
		this.callCoreApplicationMethod({
			domainName: "IctUser",
			methodName: "addContactsInWhiteList",
			context: args.context,
			params: args.params,
			paramType: ["QStringList"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

	removeContactsInWhiteList : function(args) {
		this.callCoreApplicationMethod({
			domainName: "IctUser",
			methodName: "removeContactsInWhiteList",
			context: args.context,
			params: args.params,
			paramType: ["QStringList"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

	deleteWhiteList : function(args) {
		this.callCoreApplicationMethod({
			domainName: "IctUser",
			methodName: "deleteWhiteList",
			context: args.context,
			params: args.params,
			paramType: [],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

	getFilteringLevel : function(args) {
		this.callCoreApplicationMethod({
			domainName: "IctUser",
			methodName: "getFilteringLevel",
			context: args.context,
			params: args.params,
			paramType: ["QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

	activateFiltering : function(args) {
		this.callCoreApplicationMethod({
			domainName: "IctUser",
			methodName: "activateFiltering",
			context: args.context,
			params: args.params,
			paramType: ["QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

	deActivateFiltering : function(args) {
		this.callCoreApplicationMethod({
			domainName: "IctUser",
			methodName: "deActivateFiltering",
			context: args.context,
			params: args.params,
			paramType: ["QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

	activateSupervision : function(args) {
		this.callCoreApplicationMethod({
			domainName: "IctUser",
			methodName: "activateSupervision",
			context: args.context,
			params: args.params,
			paramType: ["QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

	deActivateSupervision : function(args) {
		this.callCoreApplicationMethod({
			domainName: "IctUser",
			methodName: "deActivateSupervision",
			context: args.context,
			params: args.params,
			paramType: ["QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

	setFilteringLevel : function(args) {
		this.callCoreApplicationMethod({
			domainName: "IctUser",
			methodName: "setFilteringLevel",
			context: args.context,
			params: args.params,
			paramType: ["QString", "QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

	isInWhiteList : function(args) {
		this.callCoreApplicationMethod({
			domainName: "IctUser",
			methodName: "isInWhiteList",
			context: args.context,
			params: args.params,
			paramType: ["QStringList"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

	getCapabilityGlobal : function(args) {
		this.callCoreApplicationMethod({
			domainName: "IctUser",
			methodName: "getCapability",
			context: args.context,
			params: args.params,
			paramType: ["QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

	getCapabilityInstance : function(args) {
		this.callCoreApplicationMethod({
			domainName: "IctUser",
			methodName: "getCapability",
			context: args.context,
			params: args.params,
			paramType: ["int", "QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

	eraseUserData : function(args) {
		args.params[0]	= Encryption.encryptRSA(args.params[0]);
		
		this.callCoreApplicationMethod({
			domainName: "IctUser",
			methodName: "eraseUserData",
			context: args.context,
			params: args.params,
			paramType: ["QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

	unLockVVMAccessRequest : function(args) {
		this.callCoreApplicationMethod({
			domainName: "IctUser",
			methodName: "unLockVVMAccessRequest",
			context: args.context,
			params: args.params,
			paramType: ["QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

	lockVVMAccessRequest : function(args) {
		this.callCoreApplicationMethod({
			domainName: "IctUser",
			methodName: "lockVVMAccessRequest",
			context: args.context,
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	}
});

ICTouchAPI.APIServices.IctUser = new ICTouchAPI.APIServices.IctUser();
/**
 * @ignore
 */
dojo.provide("ICTouchAPI.APIServices.Session");
dojo.declare("ICTouchAPI.APIServices.Session",[ICTouchAPI.APIServices.APIServicesBase],
{
	getRemoteDeviceList : function(args) {
        this.callCoreApplicationMethod({
            domainName: "Session",
            methodName: "getRemoteDeviceList",
            context: args.context,
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

	getCapability : function(args) {
        this.callCoreApplicationMethod({
            domainName: "Session",
            methodName: "getCapability",
            context: args.context,
            params: args.params,
            paramType: ["QString"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },
});

ICTouchAPI.APIServices.Session = new ICTouchAPI.APIServices.Session();
/**
 * @ignore
 */
dojo.provide("ICTouchAPI.APIServices.Bluetooth");
dojo.declare("ICTouchAPI.APIServices.Bluetooth",[ICTouchAPI.APIServices.APIServicesBase],
{

	BTRemovePairing : function(args) {
        this.callCoreApplicationMethod({
            domainName: "ictbtmgr",
            methodName: "BTRemovePairing",
            context: args.context,
            params: args.params,
            paramType: ["QString", "int"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

	BTShowBondedDevice : function(args) {
        this.callCoreApplicationMethod({
            domainName: "ictbtmgr",
            methodName: "BTShowBondedDevice",
            context: args.context,
            params: args.params,
            paramType: ["QString"],
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

	BTStartInquiry : function(args) {
        this.callCoreApplicationMethod({
            domainName: "ictbtmgr",
            methodName: "BTStartInquiry",
            context: args.context,
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

	BTStopInquiry : function(args) {
        this.callCoreApplicationMethod({
            domainName: "ictbtmgr",
            methodName: "BTStopInquiry",
            context:args. context,
            callback: args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
	},

	BTInitiatePairing : function(args) {
		this.callCoreApplicationMethod({
			domainName: "ictbtmgr",
			methodName: "BTInitiatePairing",
			context: args.context,
			params: args.params,
			paramType: ["QString", "QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
    },
});

ICTouchAPI.APIServices.Bluetooth = new ICTouchAPI.APIServices.Bluetooth();
/**
 * @class ICTouchAPI.APIServices.IctMPInterface
 * @singleton
 * @extends ICTouchAPI.APIServices.APIServicesBase
 * IctMPInterface is a module that communicates directly, using DBus, with the MediaPlayer. IctMPinterface can demand to MediaPlayer to play, pause... music files. <br>
 * This is the class that external web applications must use. Such applications can not have the right to access directly to the infrastructure applications, MediaPlayer taking part of this set.
 */


dojo.provide("ICTouchAPI.APIServices.IctMPInterface");
dojo.declare("ICTouchAPI.APIServices.IctMPInterface",[ICTouchAPI.APIServices.APIServicesBase],
{

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Playes a media file.
     * deprecated instead if using this method, you must use void playMedia()</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;idFile {Number} the File identifier
     * &nbsp;&nbsp; filePath {String} the full url of the File
     * &nbsp;&nbsp; x {Number} x position  -  !!NOT USED!!
     * &nbsp;&nbsp; y {Number} y position  -  !!NOT USED!!
     * &nbsp;&nbsp; w {Number} w position  -  !!NOT USED!!
     * &nbsp;&nbsp; h {Number} h position  -  !!NOT USED!!
     * &nbsp;&nbsp; volume {Number} the volume to set  -  !!NOT USED!!
     * &nbsp;&nbsp; loop {Boolean} true if the music will be played in loop  -  !!NOT USED!!
     * &nbsp;&nbsp; index {Number} index
     * }
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */
	launchMP : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctMPInterface",
			methodName : "launchMP",
			context : args.context,
			params: args.params,
			paramType: ["int","QString","int","int","int","int","double","bool","qlonglong"],
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},
    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Plays a media file.
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;idFile {Number} identifier of the File that we want to play.
     * &nbsp;&nbsp; filePath {String} full File path.
     * }
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */
	playMedia : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctMPInterface",
			methodName : "playMedia",
			context : args.context,
			params: args.params,
			paramType: ["int","QString"],
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Plays a media file.
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;idFile {Number} identifier of the File that we want to play.
     * &nbsp;&nbsp; filePath {String} full File path.
     * &nbsp;&nbsp;  seek {qlonglong} : seek (0 to start at beginning).
     * }
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */
	playMediaWithSeek : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctMPInterface",
			methodName : "playMediaWithSeek",
			context : args.context,
			params: args.params,
			paramType: ["int","QString","qlonglong"],
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Send the media sessin data to the presentation.
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */
	getMediaSessionInfos : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctMPInterface",
			methodName : "getMediaSessionInfos",
			context: args.context,
			params: args.params,
			paramType: [],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},


    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Puts the currently playing melody in pause or replayes a paused melody.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains a File identifier
     * <pre>
 	   * {Number} a File identifier.
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */

	playPauseMP : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctMPInterface",
			methodName : "playPauseMP",
			context : args.context,
			params: args.params,
			paramType: ["int"],
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Puts the reading pointer of the currently loaded file in 0 position.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains a File identifier
     * <pre>
 	   * {Number} a File identifier.
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */

	rewindMP : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctMPInterface",
			methodName : "rewindMP",
			context : args.context,
			params: args.params,
			paramType: ["int"],
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Stops the currently playing file.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains a File identifier
     * <pre>
 	   * {Number} a File identifier.
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */


	stopMP : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctMPInterface",
			methodName : "stopMP",
			context : args.context,
			params: args.params,
			paramType: ["int"],
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>call method setVolumeMP</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the volume value
     * <pre>
 	   * {Number} a volume value
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */

	setVolumeMP : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctMPInterface",
			methodName : "setVolumeMP",
			context : args.context,
			params: args.params,
			paramType: ["double"],
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>call method muteMP</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */

	muteMP : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctMPInterface",
			methodName : "muteMP",
			context : args.context,
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>call method unMuteMP</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */

	unMuteMP : function(args) {
		this.callCoreApplicationMethod({
			domainName : "IctMPInterface",
			methodName : "unMuteMP",
			context : args.context,
			callback : args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>capability of the module</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the name of the capability
     * <pre>
     * {String} the name of the capability
     * </pre>
     *
     * The callback will be called with the capability, following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;name {String}
     * &nbsp;&nbsp;state {Number}
     * }
     * </pre>
     */

	getCapabilityGlobal : function(args) {
		this.callCoreApplicationMethod({
			domainName: "IctMPInterface",
			methodName: "getCapability",
			context: args.context,
			params: args.params,
			paramType: ["QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>capability of an object of the module</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params is an array which contains the following structure :
     * <pre>
     * {
     * &nbsp;&nbsp;name {String}  Name of the capability
     * &nbsp;&nbsp;listId {Number} ID of the list of capabilities
     * }
     * </pre>
     *
     * The callback will be called with the capability, following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;name {String}
     * &nbsp;&nbsp;state {Number}
     * }
     * </pre>
     */

	getCapabilityInstance : function(args) {
		this.callCoreApplicationMethod({
			domainName: "IctMPInterface",
			methodName: "getCapability",
			context: args.context,
			params: args.params,
			paramType: ["int", "QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>the list of capability of an object</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the ID of the list of capabilities
     * <pre>
     * {Number} the ID of the list of capabilities
     * </pre>
     *
     * The callback will be called with a map of capabilities indexed by name, each of them following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;name {String}
     * &nbsp;&nbsp;state {Number}
     * }
     * </pre>
     */

	getCapabilities : function(args) {
		this.callCoreApplicationMethod({
			domainName: "IctMPInterface",
			methodName: "getCapabilities",
			context: args.context,
			params: args.params,
			paramType: ["int"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Asks MediaPlayer to sends its current state. The state as an event.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */

	mediaPlayerSendState : function(args) {
		this.callCoreApplicationMethod({
			domainName: "IctMPInterface",
			methodName: "mediaPlayerSendState",
			context: args.context,
			params: args.params,
			paramType: [],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Preview of a ring tone (play it). Use it only in settings application to preview a ringtone before choose its in settings application.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the file path
     * <pre>
 	   * {String} the full url of the File.
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */

	preview : function(args) {
		this.callCoreApplicationMethod({
			domainName: "IctMPInterface",
			methodName: "preview",
			context: args.context,
			params: args.params,
			paramType: ["QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

	  /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Preview of a ring tone (play it). Use it only in settings application to preview a ringtone before choose its in settings application.</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the file path
     * <pre>
 	   * {String} the full url of the File.
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */

	volumePreview : function(args) {
		this.callCoreApplicationMethod({
			domainName: "IctMPInterface",
			methodName: "volumePreview",
			context: args.context,
			params: args.params,
			paramType: ["QString","double"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

    /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b></b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     * <b><br><br>Stops a preview</b>
     *
     * @param {Object} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * The callback will be called with a void:
     *
     * <pre>
     * void {void}
     * </pre>
     */

	stopPreview : function(args) {
		this.callCoreApplicationMethod({
			domainName: "IctMPInterface",
			methodName: "stopPreview",
			context: args.context,
			params: args.params,
			paramType: [],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	}
})

ICTouchAPI.APIServices.IctMPInterface = new ICTouchAPI.APIServices.IctMPInterface();/**
 * @ignore
 */
dojo.provide("ICTouchAPI.APIServices.IctAudio");
dojo.declare("ICTouchAPI.APIServices.IctAudio",[ICTouchAPI.APIServices.APIServicesBase],
{
    getMute : function(args) {
        this.callCoreApplicationMethod({
            domainName : "ictaudio",
            methodName : "getMute",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    setMute : function(args) {
        this.callCoreApplicationMethod({
            domainName : "ictaudio",
            methodName : "setMute",
            context : args.context,
            params: args.params,
            paramType: ["bool"],
            callback : args.callback
        });
        args = null;
    },

    getAvailableVoicemode : function(args) {
        this.callCoreApplicationMethod({
            domainName : "ictaudio",
            methodName : "getAvailableVoicemode",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

    getCurrentVoicemode : function(args) {
        this.callCoreApplicationMethod({
            domainName : "ictaudio",
            methodName : "getCurrentVoicemode",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
        });
        args = null;
    },

	setVoicemode : function(args) {
        this.callCoreApplicationMethod({
            domainName : "ictaudio",
            methodName : "setVoicemode",
            context : args.context,
			params: args.params,
            paramType: ["int"],
			callback : args.callback
        });
        args = null;
    },

	getVoicemodeVol : function(args) {
        this.callCoreApplicationMethod({
            domainName : "ictaudio",
            methodName : "getVoicemodeVol",
            context : args.context,
			params: args.params,
            paramType: ["int"],
			callback : args.callback
		});
		args = null;
    },

	getVoicemodeMaxVol : function(args) {
        this.callCoreApplicationMethod({
            domainName : "ictaudio",
            methodName : "getVoicemodeMaxVol",
            context : args.context,
			params: args.params,
            paramType: ["int"],
			callback : args.callback,
			callbackParams: args.callbackParams
        });
        args = null;
    },

	setVoicemodeVol : function(args) {
        this.callCoreApplicationMethod({
            domainName : "ictaudio",
            methodName : "setVoicemodeVol",
            context : args.context,
			params: args.params,
            paramType: ["int","int"],
			callback : args.callback
        });
        args = null;
    },

	getVoicemodeStatus : function(args) {
		this.callCoreApplicationMethod({
            domainName : "ictaudio",
            methodName : "getVoicemodeStatus",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
		});
		args = null;
	},

	getVoiceSignalLimitationDMconfig : function(args) {
		this.callCoreApplicationMethod({
            domainName : "ictaudio",
            methodName : "GetVoiceSignalLimitationDMConfig",
            context : args.context,
            callback : args.callback,
            callbackParams: args.callbackParams
		});
		args = null;
	},

	enableAutoVolume : function(args) {
		this.callCoreApplicationMethod({
            domainName : "ictaudio",
            methodName : "EnableAutoVolume",
            context : args.context,
			params: args.params,
            paramType: ["uint"],
			callback : args.callback,
			callbackParams: args.callbackParams
        });
        args = null;
	},

	keepStateAudioEnhancement : function(args) {
		this.callCoreApplicationMethod({
            domainName : "ictaudio",
            methodName : "KeepStateAudioEnhancement",
            context : args.context,
			params: args.params,
            paramType: ["uint"],
			callback : args.callback,
			callbackParams: args.callbackParams
        });
        args = null;
	}

});

ICTouchAPI.APIServices.IctAudio = new ICTouchAPI.APIServices.IctAudio();
/**
* @class ICTouchAPI.WSManager
* @singleton
* @extends Object
* @ignore
*/
dojo.provide("ICTouchAPI.APIServices.WSManager");
dojo.declare("ICTouchAPI.APIServices.WSManager",[ICTouchAPI.APIServices.APIServicesBase],
{
    /**
     * submitUserPassword
     * @param [array] args containing the following parameters:
     *  	string password corresponding to the password
     *		boolean canceled indicating if the user pressed on Ok (false) of on exit (true)
     * @ignore
     */
    submitUserPassword : function(args) {
		args.params[0]	= Encryption.encryptRSA(args.params[0]);
		
        this.callCoreApplicationMethod({
            domainName : "WSManager",
            methodName : "submitUserPassword",
            context : args.context,
            params: args.params,
            paramType: ["QString", "bool"],
            callback : args.callback,
			callbackParams : args.callbackParams
        });
        args = null;
    },

    /**
     * changePassword
     * @param args containing the following parameter:
     *  	string oldPassword corresponding to the new password
     *  	string newPassword corresponding to the new password
     *  @ignore
     */
    changePassword : function(args) {
		args.params[0]	= Encryption.encryptRSA(args.params[0]);
		args.params[1]	= Encryption.encryptRSA(args.params[1]);
		
        this.callCoreApplicationMethod({
            domainName : "WSManager",
            methodName : "changePassword",
            context : args.context,
            params: args.params,
            paramType: ["QString","QString"],
            callback : args.callback,
			callbackParams : args.callbackParams
        });
        args = null;
    },

    /**
     * reconnectAfterPwdChanged
     * @param args containing the following parameter:
     *  	string password corresponding to the new password
     * @ignore
     */
    reconnectAfterPwdChanged : function(args) {
		args.params[0]	= Encryption.encryptRSA(args.params[0]);

        this.callCoreApplicationMethod({
            domainName : "WSManager",
            methodName : "reconnectAfterPwdChanged",
            context : args.context,
            params: args.params,
            paramType: ["QString"],
            callback : args.callback,
			callbackParams : args.callbackParams
        });
        args = null;
    },

	getCapabilityGlobal : function(args) {
		this.callCoreApplicationMethod({
			domainName: "WSManager",
			methodName: "getCapability",
			context: args.context,
			params: args.params,
			paramType: ["QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

	getCapabilityInstance : function(args) {
		this.callCoreApplicationMethod({
			domainName: "WSManager",
			methodName: "getCapability",
			context: args.context,
			params: args.params,
			paramType: ["int", "QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	}
 });

ICTouchAPI.APIServices.WSManager = new ICTouchAPI.APIServices.WSManager();/**
 * @ignore
 */
dojo.provide("ICTouchAPI.APIServices.ICTSaveRestore");
dojo.declare("ICTouchAPI.APIServices.ICTSaveRestore",[ICTouchAPI.APIServices.APIServicesBase],
{
    save : function(){
		this.callCoreApplicationMethod({
														domainName : "ICTSaveRestore",
														methodName : "startSaveOnRepository"
														});
	},

	restore : function(){
		this.callCoreApplicationMethod({
														domainName : "ICTSaveRestore",
														methodName : "startRestoreFromRepository"
														});
	},

	getCapabilityGlobal : function(args) {
		this.callCoreApplicationMethod({
			domainName: "ICTSaveRestore",
			methodName: "getCapability",
			context: args.context,
			params: args.params,
			paramType: ["QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

	getCapabilityInstance : function(args) {
		this.callCoreApplicationMethod({
			domainName: "ICTSaveRestore",
			methodName: "getCapability",
			context: args.context,
			params: args.params,
			paramType: ["int", "QString"],
			callback: args.callback,
			callbackParams: args.callbackParams
		});
		args = null;
	},

	 resetToDefault : function(args) {
        this.callCoreApplicationMethod({
            domainName : "ICTSaveRestore",
            methodName : "resetToDefault",
            context : args.context,
            params: args.params,
			paramType: [],
            callback : args.callback,
			callbackParams: args.callbackParams
        });
        args = null;
    }

});

ICTouchAPI.APIServices.ICTSaveRestore = new ICTouchAPI.APIServices.ICTSaveRestore();


/* ----------------------------------------------------------------------------
 * This file was automatically generated by XSLT javascriptGenerator
 * Version 0.1
 * 
 * This file is not intended to be easily readable and contains a number of 
 * coding conventions designed to improve portability and efficiency. Do not make
 * changes to this file unless you know what you are doing.Modify the xml interface 
 * file instead. 
 * ----------------------------------------------------------------------------- */

/**
 * @class ICTouchAPI.APIServices.Sync
 * @extends Object
 * @ignore
 * Functions provided to communicate with system regarding Sync Module.<br><br>
 * Methods have 3 different visibility levels:<br>
 * - Public    : means a method can be called from Webapps<br>  
 * - Protected : means a method can be called only if the user has entered a password <br>  
 * - Partially (protected) : means the method may need authentication<br><br>  
 * Events have also 3 different visibility levels:<br>
 * - Public    : means the event can be called and subscribe <br>  
 * - Subscribe : means the event can only be subscribed <br>  
 * - Notify    : means the event can only be notified <br>  
 */
dojo.provide("ICTouchAPI.APIServices.Sync");
dojo.declare("ICTouchAPI.APIServices.Sync",[ICTouchAPI.APIServices.APIServicesBase],
{
	
        /**
         * method syncStart This method has the visibility level: .
	 */
  		syncStart: function(args) {
        this.callCoreApplicationMethod({
            domainName: "Sync",
            methodName: "syncStart",
            context: args.context,
            simulate: false,
            
            callback: args.callback,
            errorCallback : args.errorCallback,
            callbackParams: args.callbackParams
});
args = null;
}
,
        /**
         * method syncStop This method has the visibility level: .
	 */
  		syncStop: function(args) {
        this.callCoreApplicationMethod({
            domainName: "Sync",
            methodName: "syncStop",
            context: args.context,
            simulate: false,
            
            callback: args.callback,
            errorCallback : args.errorCallback,
            callbackParams: args.callbackParams
});
args = null;
}

        /** 
         * @event syncFinishedEvent This event has the visibility level: .
         */


});
ICTouchAPI.APIServices.Sync= new ICTouchAPI.APIServices.Sync();
/**
 * @class ICTouchAPI.APIServices.Video
 * @extends Object
 * @ignore
 */
dojo.provide("ICTouchAPI.APIServices.Video");
dojo.declare("ICTouchAPI.APIServices.Video",[ICTouchAPI.APIServices.APIServicesBase],
{


    args: null, // last arguments used to restore in fake mode


    /**
     * @private
     */
    constructor : function () {

        /* Fake mode - just create an empty div positioned with absolute coordinates */
        this.fakeMode = false;

        /* Check if VideoAPIServices Javascript object exists, otherwise we need to
        to switch to fake mode (use methods that simulate some video content) */
        try {
            if( VideoAPIServices ) {}
        }
        catch(e) {
            /* It seems that VideoAPIServices does not exists... */
            this.fakeMode = true;
        }

    },


   /**
     * Executes the given callback <i>(callback)</i> within the given context <i>(context)</i> with the <b>capability of the module</b> corresponding to the given parameters <i>(params)</i> along with additional callback parameters <i>(callbackParams)</i>.
     *
     * @param {JSON} args The following structure:
     * <pre>
     * {
     * &nbsp;&nbsp;params {Object} The function parameters. See below for the details
     * &nbsp;&nbsp;context {Object} the webapp context
     * &nbsp;&nbsp;callback {Function} The webapp callback
     * &nbsp;&nbsp;callbackParams: The additional callback parameters or null
     * }
     * </pre>
     *
     * params contains the name of the capability
     * <pre>
     * {String} the name of the capability
     * </pre>
     *
     * The callback will be called with the capability, following this structure:
     *
     * <pre>
     * {
     * &nbsp;&nbsp;name {String}
     * &nbsp;&nbsp;state {Number}
     * }
     * </pre>
     */
    getCapabilityGlobal : function(args) {
            this.callCoreApplicationMethod({
                    domainName: "Video",
                    methodName: "getCapability",
                    context: args.context,
                    params: args.params,
                    paramType: ["QString"],
                    callback: args.callback,
                    callbackParams: args.callbackParams
            });
            args = null;
    },


    /**
     * Start a local video session
     *
     * @return {Boolean} created TRUE if video has been successfully created
     */
    createLocalVideo : function( args ) {
        //console.log( "ICTouchAPI.APIServices.Video", "setLocalVideo", args );

        var created = false; // false = an error occurred

        if( this.fakeMode === false ) {
            created = VideoAPIServices.createLocalVideo( args.x, args.y, args.w, args.h );
        } else {
            created = ICTouchAPI.APIServices.Video.setFakeVideo( args.x, args.y, args.w, args.h );
        }

        if(created == true ) {
            this.args = {params: args};
        }

        return created;
    },

    /**
     * Stop a local video session
     *
     * @return {Boolean} deleted TRUE if video has been successfully deleted
     */
    deleteLocalVideo : function() {
    	//console.log( "ICTouchAPI.APIServices.Video", "deleteLocalVideo" );

    	var deleted = false; // false = an error occurred

        if( this.fakeMode === false ) {
            deleted = VideoAPIServices.deleteLocalVideo();
        } else {
            deleted = ICTouchAPI.APIServices.Video.deleteFakeVideo();
        }

        return deleted;
    },


    /**
     * Display remote video given window parameters.
     *
     * @return {Boolean} shown TRUE if video has been successfully hidden
     */
    showVideo : function( args ) {

    	var shown = false; // false = an error occurred

        if( this.fakeMode === false ) {
            shown = VideoAPIServices.showVideo( args.callId, args.x, args.y, args.w, args.h );
        } else {
            shown = ICTouchAPI.APIServices.Video.setFakeVideo( args.x, args.y, args.w, args.h );
        }

        if(shown == true ) {
            this.args = {params: args};
        }

        return shown;
    },


    fullscreen: function( callId ) {},


    /**
     * Hide remote video from screen
     *
     * @return {Boolean} hidden TRUE if video has been successfully hidden
     */
    hideVideo : function( callId ) {

    	var hidden = false; // false = an error occurred

        if( this.fakeMode === false ) {
            hidden = VideoAPIServices.hideVideo( callId );
        } else {
            hidden = ICTouchAPI.APIServices.Video.deleteFakeVideo();
        }

        return hidden;
    },



    /**
     * Create a call video
     *
     * @return {Boolean} created TRUE if video has been successfully created
     */
    createVideo: function(args) {
        //console.log( "ICTouchAPI.APIServices.Video", "createVideo", args );

        var created = false; // false = an error occurred

        if( this.fakeMode === false ) {
            created = VideoAPIServices.createVideo( args.callId, args.sipCallId, args.x, args.y, args.w, args.h, args.pip, args.mute );
        } else {
            created = ICTouchAPI.APIServices.Video.setFakeVideo( args.x, args.y, args.w, args.h );
        }

        if(created == true ) {
            this.args = {params: args};
        }

        return created;
    },


    /**
     * Configure look and feel for a video session.
     * @param {Number} background_color ARGB Widget background color
     * @param {String} text main window displayed text when no video
     * @param {Number} text_color text ARGB color
     * @param {String} pip_text PiP area displayed text when no video
     */
    setVideoMMIParams: function( background_color, text, text_color, pip_text ) {

        var success = false; // false = an error occurred

        if( this.fakeMode === false ) {
            success = VideoAPIServices.setVideoMMIParams( background_color, text, text_color, pip_text );
        } else {
            success = true;
        }

        return success;
    },


    /**
     * Stop video session
     *
     * @param {Number} callId (call id)
     * @return {Boolean} deleted TRUE if video has been successfully deleted
     */
    deleteVideo : function( callId ) {
    	//console.log( "ICTouchAPI.APIServices.Video", "deleteVideo" );

    	var deleted = false; // false = an error occurred

        if( this.fakeMode === false ) {
            deleted = VideoAPIServices.deleteVideo( callId );
        } else {
            deleted = ICTouchAPI.APIServices.Video.deleteFakeVideo();
        }

        return deleted;
    },


    /**
     * Mask video according to parameters
     *
     * @param {Number} num mask number 0 (reserved to Pip) or 1.
     * @param {Number} x horizontal window absolute origin in pixels.
     * @param {Number} y vertical window absolute origin in pixels.
     * @param {Number} w video window width in pixels (MULTIPLE OF 8!!).
     * @param {Number} h video window height in pixels (MULTIPLE OF 2!!).
     * @return {Boolean} enabled TRUE if the mask has been successfully enabled
     */
    enableMask : function(num, x, y, w, h) {

    	var enabled = false; // false = an error occurred

        if( this.fakeMode === false ) {
            enabled = VideoAPIServices.enableMask( num, x, y, w, h );
        } else {
            enabled = ICTouchAPI.APIServices.Video.enableFakeMask( x, y, w, h );
        }

        return enabled;
    },

    /**
     * Remove mask
     *
     * @param {Number} num mask number
     * @return {Boolean} disabled TRUE if the mask has been successfully disabled
     */
    disableMask : function(num) {

    	/** @todo check which one of TRUE or FALSE means 'success' */
    	var disabled = false; // false = an error occurred

        if( this.fakeMode === false ) {
            disabled = VideoAPIServices.disableMask( num );
        } else {
            disabled = ICTouchAPI.APIServices.Video.disableFakeMask();
        }

        return disabled;
    },




    //
    // VIDEO FAKE METHODS
    //

    setFakeVideo: function( x, y, w, h ) {

    	var videoNodeName = 'fakevideo';

    	if( window.top.document.getElementById(videoNodeName) )
    	{
            var videoNode = window.top.document.getElementById(videoNodeName);
            var videoNodeContent = window.top.document.getElementById(videoNodeName+'content');
            videoNode.style.display = 'block';
            videoNode.style.left = x + "px";
            videoNode.style.top = y + "px";
            videoNode.style.width = (w) + "px";
            videoNode.style.height = (h) + "px";
            videoNodeContent.style.width = (w -2) + "px";
            videoNodeContent.style.height = (h -2) + "px";
    	}

    	else
    	{
            var videoNode = document.createElement("div");
            videoNode.id = videoNodeName;
            videoNode.style.left = x + "px";
            videoNode.style.top = y + "px";
            videoNode.style.width = (w) + "px";
            videoNode.style.height = (h) + "px";
            videoNode.style.position = 'absolute';
            videoNode.style.zIndex = '80000';
            videoNode.style.overflow = 'hidden';
            videoNode.onclick = function() {
                if( parseInt(videoNode.style.width) < 800 ) {
                    ICTouchAPI.APIServices.Video.setFakeVideo( 0, 0, 800, 480 );
                }
                else {
                    var params = ICTouchAPI.APIServices.Video.args.params;
                    ICTouchAPI.APIServices.Video.setFakeVideo( params.x, params.y, params.w, params.h );
                }
            };

            var videoNodeContent = document.createElement("div");
            videoNodeContent.id = videoNodeName + 'content';
            videoNodeContent.style.width = (w -2) + "px";
            videoNodeContent.style.height = (h -2) + "px";
            videoNodeContent.style.border = "1px solid #757575";
            videoNodeContent.style.background = "url(data:image/gif;base64,R0lGODlhDwAPALMMAP/FAP+wAP/CAP/AAP+vAP/EAP+sAP+2AP+rADMzMwAAAP///////wAAAAAAAAAAACH5BAEAAAwALAAAAAAPAA8AAARIkEml1lIpyT2tV0PGVZ6lFBTFkKVZUhfsxm76mWQqX+2SADrb65f76DKC4kwR0CSUvIuGkTgIeQrDlEoIsCiIrSTzFXE2wXMEADs=) 50% 50% no-repeat";
            videoNodeContent.style.backgroundColor = "#000";

            videoNode.appendChild( videoNodeContent );
            window.top.document.body.appendChild( videoNode );
    	}


    	return true;
    },

    deleteFakeVideo: function() {
    	if( window.top.document.getElementById('fakevideo') ) {
            window.top.document.getElementById('fakevideo').style.display = 'none';
    	}
    	return true;
    },

    /* Works only for full width mask (like audio toaster) */
    enableFakeMask: function(x,y,w,h) {

    	if( window.top.document.getElementById('fakevideo') ) {
            var videoNode = window.top.document.getElementById('fakevideo');

            /* Calculate the masked video sizes */
            var height = parseInt( videoNode.style.height, 10),
                    top = parseInt( videoNode.style.top, 10);
            // audiotoaster_height - (total_screen_height - (video_top_position + video_height) )
            var overlap = 165 - (480 - (top + height));
            var new_height = height - overlap;

            videoNode.style.height = new_height+ 'px';
    	}

    	return true;
    },

    disableFakeMask: function() {

    	if( window.top.document.getElementById('fakevideo') )
    	{
            var videoNode = window.top.document.getElementById('fakevideo');
            var videoNodeContent = window.top.document.getElementById('fakevideocontent');

            /* Calculate the full video height */
            var height = videoNodeContent.style.height;
            var new_height = parseInt( height, 10 ) + 2;

            videoNode.style.height = new_height + 'px';
    	}

    	return true;
    }

});

ICTouchAPI.APIServices.Video = new ICTouchAPI.APIServices.Video();/**
 * @ignore
 */
dojo.provide("ICTouchAPI.APIServices.IMEManager");
dojo.declare("ICTouchAPI.APIServices.IMEManager",[ICTouchAPI.APIServices.APIServicesBase],
{
	//QString startIMESession(IME selection);
    startIMESession : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IMEManager",
            methodName : "startIMESession",
            context : args.context,
            params: args.params,
            paramType: ["int"],
            callback : args.callback,
			callbackParams : args.callbackParams
        });
        args = null;
    },

	//bool changeIMESelection(QString sessionID, IME selection);
	changeIMESelection : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IMEManager",
            methodName : "changeIMESelection",
            context : args.context,
            params: args.params,
            paramType: ["QString","int"],
            callback : args.callback,
			callbackParams : args.callbackParams
        });
        args = null;
    },

	//bool stopIMESession(QString sessionID);
	stopIMESession : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IMEManager",
            methodName : "stopIMESession",
            context : args.context,
            params: args.params,
            paramType: ["QString"],
            callback : args.callback,
			callbackParams : args.callbackParams
        });
        args = null;
    },

	//MEResults getIMECandidates(QString sessionID, QString input);
	getIMECandidates : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IMEManager",
            methodName : "getIMECandidates",
            context : args.context,
            params: args.params,
            paramType: ["QString","QString"],
            callback : args.callback,
			callbackParams : args.callbackParams
        });
        args = null;
    },

	//IMEResults getIMEWordAssociation(QString sessionID, QChar selection);
	getIMEWordAssociation : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IMEManager",
            methodName : "getIMEWordAssociation",
            context : args.context,
            params: args.params,
            paramType: ["QString","QChar"],
            callback : args.callback,
			callbackParams : args.callbackParams
        });
        args = null;
    },

	//bool isValid(QString sessionID, QChar character);
	isValid : function(args) {
        this.callCoreApplicationMethod({
            domainName : "IMEManager",
            methodName : "isValid",
            context : args.context,
            params: args.params,
            paramType: ["QString","QChar"],
            callback : args.callback,
			callbackParams : args.callbackParams
        });
        args = null;
    }

 });

ICTouchAPI.APIServices.IMEManager = new ICTouchAPI.APIServices.IMEManager();