dojo.provide("webapp.search.controlBase");
dojo.declare("webapp.search.controlBase", [ICTouchAPI.webApplication],
{
	mainWidget              : null,
	objSearchData           : null,
	clbkBackButton          : null,
	strCurrentText          : "",
	intResultsCounter		: 0,
	intCurrentSearchId		: -1,
	boolOngoingSearch		: false,
	boolErgoSearchInitiated	: false, // used show or not the empty states, if the search has been initiated or not
	boolPurgeResults		: true, // used show or not the empty states, if the search has been initiated or not

	DELAY_BEFORE_SEARCH		: 1500,		// in ms
	_waitKeyJustPressed		: 0,

	load : function(){
		// Get default call method
		ICTouchAPI.settingServices.getSetting("NewCallDefaultMethod", this, this.getDefaultCallMethod);
		ICTouchAPI.settingServices.getSetting("MinSearchCharacter", this, this.setMinCharacterValue);
		ICTouchAPI.settingServices.subscribeToSetting(this, "NewCallDefaultMethod", this.getDefaultCallMethod);
		ICTouchAPI.settingServices.subscribeToSetting(this, "MinSearchCharacter", this.setMinCharacterValue);

		// Subscribe to the event containing the search results
		ICTouchAPI.eventServices.subscribeToEvent(this, "DIR_SEARCH_RESULTS", this.resultsReceived);
		ICTouchAPI.eventServices.subscribeToEvent(this, "DIR_SEARCH_COMPLETED_BY_NAME", this.resultsLastReceived);
	},

	dump : function(boolAdvancedLogs) {
		this.inherited(arguments);
		ICTouchAPI.debugServices.dump("Search settings :");
		ICTouchAPI.debugServices.dump(" - NewCallDefaultMethod: " + ((this.data._boolDialpadByDefault)?"dialpad":"byname"));
		ICTouchAPI.debugServices.dump(" - MinSearchCharacter: " + this.data._intMinCharacter);
		ICTouchAPI.debugServices.dump("Is ongoing search (boolOngoingSearch) ? : " + this.boolOngoingSearch);
		ICTouchAPI.debugServices.dump("Is new search initiated (boolErgoSearchInitiated) ? : " + this.boolErgoSearchInitiated);
		ICTouchAPI.debugServices.dump("Current SearchId (intCurrentSearchId): " + this.intCurrentSearchId);
		ICTouchAPI.debugServices.dump("Current text input for the search (strCurrentText): " + this.strCurrentText);
		ICTouchAPI.debugServices.dump("Number of results for the current search (intResultsCounter): " + this.intResultsCounter);
	},

	buttonCallback : function(buttonId) {
		var data = webapp.search.data;
		switch (buttonId)
		{
			case data.BACK:
			{
				if (!this.data.isPreviousHome()) {
					this.clbkBackButton();
					this.purgeResults();
				}
				else {   // Default "back" action
					dojo.publish("OpenHomepage");
					this.purgeResults();
				}
				break;
			}
			case data.REFINE_SEARCH:
				dojo.publish("search/NewSearch", [true]);
				this.objSearchData.boolForceByname = true;
				this.showKeyboard();
				break;
			default:
				console.warn("buttonCallbak : no callback is defined for this button id:"+buttonId);
				break;
		}
	},

	resetSearch : function() {
		if(this.boolPurgeResults){
			this.stopOngoingSearch();
			this.purgeResults();
			this.strCurrentText = "";
		}else{
			this.boolPurgeResults = true;
		}
		this.showKeyboard();
	},

	/**
	 * doSearch method is the main entry point to the search webapp.
	 * It raises the keyboard and listens pressed keys to match directory contacts.
	 *
	 * @param {Object} args is an object that can contain the following values:
	 * - contactDataTypes, which is an array of recognized contact data types defined as character strings. These strings would be matched against contact's "AddressMap" keys.
	 *   Example: ["phone number", "fax number"]
	 * - secondaryInformation, information of the ContactSummary to be displayed at the right of the contact's display name in the search results. It can be:
	 *		"contactId", "name", "firstName", "phoneNumber", "officePhone", "email", "type"
	 * - lineClickCallback, which is called when the user selected a contact. The lineClickCallback object can also define the following optionnal fields:
	 *		- neededData: the information about the contact that the lineClickCallback needs. The neededData can be:
	 *			- phoneNumber: directly return to the callback the phoneNumber extracted from the ContactSummary obtained in the search result (more performant),
	 *			- ContactSummary: directly return to the callback the ContactSummary obtained in the search result (more performant, but less information available),
	 *			- Contact: ask the whole contact to IctDirectory. This whole contact object is returned to the callback (costly for performances),
	 *			- other field: a field of the ContactSummary object (contactId, name, firstName, officePhone, email, type).
	 *				If the field asked is not contained into the ContactSummary, the whole contact is returned to the callback.
	 *		- leftIconImage: the icon to use for the left icon
	 *		- leftIconImageConditionnalDisplay: a condition on the contact result indicating if the left icon has to be displayed or not
	 *		(ex: "phoneNumber" -> the left icon will be displayed only if the contact corresponding to this result has a phoneNumber)
	 *   This callback will be called with parameters objContact, objClickedData. objClickedData can be null.
	 * - rightIconClickCallback, which is called when the user clicked on the right icon of the line. The rightIconClickCallback object can also define the following optionnal fields:
	 *			- neededData: the information about the contact that the rightIconClickCallback needs. The neededData can be:
	 *			- phoneNumber: directly return to the callback the phoneNumber extracted from the ContactSummary obtained in the search result (more performant),
	 *			- ContactSummary: directly return to the callback the ContactSummary obtained in the search result (more performant, but less information available),
	 *			- Contact: ask the whole contact to IctDirectory. This whole contact object is returned to the callback (costly for performances),
	 *			- other field: a field of the ContactSummary object (contactId, name, firstName, officePhone, email, type).
	 *				If the field asked is not contained into the ContactSummary, the whole contact is returned to the callback.
	 *		- rightIconImage: the icon to use for the right icon
	 *		- rightIconImageConditionnalDisplay: a condition on the contact result indicating if the right icon has to be displayed or not
	 *		(ex: "phoneNumber" -> the right icon will be displayed only if the contact corresponding to this result has a phoneNumber)
	 * - boolLocalContactsOnly, boolean; if specified and true, will only search local contacts (will not try to retrieve contacts from UDA directories)
	 * - useDefaultCallMethod, boolean; allows to switch between dialpad and search keyboard
	 * - onOpenDialpad (optional), function called when "useDefaultCallMethod" is true and the dial-pad is being opened
	 * - dialpadObject (optional), object used when "useDefaultCallMethod" is true to have a particular dial-pad object
	 * - stealth (optional), boolean; if true, add this view to stealth list in transitionServices.
	 * @param {Function} clbkBackButton can be used to override the default BACK_BTN behavior.
	 */
	doSearch : function(args, clbkBackButton) {
		webapp.search.objSearchData = args;
		webapp.search.clbkBackButton = clbkBackButton;
		webapp.search.resetSearch();
		if(this.mainWidget) {
			this.mainWidget._changeBackButton();
		}
		args = null;
		clbkBackButton = null;
	},

	clbkKeyPressed : function(str) {
		ICTouchAPI.debugServices.info("webapp.search.controlBase - clbkKeyPressed");
		// If predictive search was ongoing, clear previous results (IctDirectory + graphically)
		if (this.boolOngoingSearch) {
			this.stopOngoingSearch();
			this.purgeResults();
		}
		this.strCurrentText = str;
		if (str.length >= webapp.search.data.getIntMinCharacter()) {
			this._waitKeyJustPressed++;
			setTimeout(function() {
				webapp.search._waitKeyJustPressed--;
				if (webapp.search._waitKeyJustPressed == 0) {
					ICTouchAPI.debugServices.debug("webapp.search.controlBase - clbkKeyPressed / launch predictive search for this.strCurrentText = " + webapp.search.strCurrentText);
					webapp.search._restartSearch();
				}
			}, this.DELAY_BEFORE_SEARCH);
		}
	},

	/**
	 * Clear the results displayed.
	 */
	purgeResults : function() {
		ICTouchAPI.debugServices.info("webapp.search.controlBase - purgeResults");
		// If needed, clear the results displayed
		if (this.boolErgoSearchInitiated) {
			ICTouchAPI.debugServices.debug("webapp.search.controlBase - purgeResults / this.intResultsCounter = " + this.intResultsCounter + ", purge previous results");
			// no search initiated yet, don't display empty state at next reload of getContentSearch
			this.boolErgoSearchInitiated = false;
			dojo.publish("webapp.search.emptyItems");
			// We set the container title with the appropriate label (Search)
			this.mainWidget.setContainerTitle(this.data.NO_SEARCH);
			this.intResultsCounter = 0;
		}
	},

	/**
	 * If a search is ongoing, ask the directory to interrupt it.
	 */
	stopOngoingSearch : function() {
		ICTouchAPI.debugServices.info("webapp.search.controlBase - stopOngoingSearch");
		if (this.boolOngoingSearch) {
			// Search is interrupted, so boolOngoingSearch -> false
			this.boolOngoingSearch = false;
			// Notify the directory that the results of the previous search be purged
			ICTouchAPI.debugServices.debug("webapp.search.controlBase - stopOngoingSearch / call IctDirectory.purgeSearchResults for currentSearchId: " + this.getCurrentSearchId());
			ICTouchAPI.APIServices.IctDirectory.purgeSearchResults({
				params: [this.getCurrentSearchId()]
			});
		}
	},

	/**
	 * Restart (or start) the search, is text was entered by the user.
	 */
	_restartSearch : function() {
		ICTouchAPI.debugServices.info("webapp.search.controlBase - _restartSearch");
		this.stopOngoingSearch();
		if (this.strCurrentText) {
			// We restart the search so we purge the previous search results (if any)
			this.purgeResults();

			// We set the container title with the appropriate label (Search in progress...)
			this.mainWidget.setContainerTitle(this.data.ONGOING_SEARCH);

			// When (re)starting a search, we must increase the search id so that we can ignore results provided from previous search ids.
			var searchString;

			this.intCurrentSearchId++;
			this.intCurrentSearchId %= 10000; // prevent overflow; assume that 10000 previous search was finished!!
			searchString = this.getCurrentSearchId();
			// Do the search
			this.boolOngoingSearch = true;
			// Search is initiated, empty state will be displayed at next reload of getContentSearch
			this.boolErgoSearchInitiated = true;
			ICTouchAPI.debugServices.debug("webapp.search.controlBase - _restartSearch / call startSearchContactByName for searchString: " + searchString + " with strCurrentText: " + webapp.search.strCurrentText);
			ICTouchAPI.APIServices.IctDirectory.startSearchContactByName({
				params:[searchString, webapp.search.strCurrentText, 250, this.objSearchData.boolLocalContactsOnly ? true : false],
				context:webapp.search
			});
		}
		ICTouchAPI.debugServices.info("webapp.search.controlBase - _restartSearch done");
	},

	/**
	 * Callback of DIR_SEARCH_RESULTS.
	 * Search is completed. If the results received well correspond to the searchId ongoing, display them.
	 */
	resultsReceived : function() {
		ICTouchAPI.debugServices.info("webapp.search.controlBase - resultsReceived");
		var objArguments = ICTouchAPI.tools.getEventArguments(arguments);
		var resultSearchId = objArguments.searchId;
		ICTouchAPI.debugServices.debug("webapp.search.controlBase - resultsReceived / Results received for searchId: " + resultSearchId + " | current searchId of webapp search is: " + this.getCurrentSearchId());

		// Check if the results received well comes from the current search and that the ongoing search was not interrupted
		if (resultSearchId == this.getCurrentSearchId() && this.boolOngoingSearch) {
			var arrResults = objArguments.listContactSummary;
			var nbResults = objArguments.nbResults;
			this._resultProcessing(resultSearchId, arrResults, nbResults);
		}
	},

	/**
	 * Callback of DIR_SEARCH_COMPLETED_BY_NAME.
	 * Search is completed. If the results received well correspond to the searchId ongoing, display them.
	 */
	resultsLastReceived : function() {
		ICTouchAPI.debugServices.info("webapp.search.controlBase - resultsLastReceived");
		var objArguments = ICTouchAPI.tools.getEventArguments(arguments);
		var resultSearchId = objArguments.searchId;
		ICTouchAPI.debugServices.debug("webapp.search.controlBase - resultsLastReceived / Results received for searchId: " + resultSearchId + " | current searchId of webapp search is: " + this.getCurrentSearchId());

		// Check if the results received well comes from the current search and that the ongoing search was not interrupted
		if (resultSearchId == this.getCurrentSearchId() && this.boolOngoingSearch) {
			var arrResults = objArguments.listContactSummary;
			var nbResults = objArguments.nbResults;
			this._resultProcessing(resultSearchId, arrResults, nbResults);
			// results are received, so the search is ended -> boolOngoingSearch = false
			this.boolOngoingSearch = false;
			// results are received, so the modified button has to be shown
			var appBar = ICTouchAPI.AppBarServices.getAppBar("search", "getSearch");
			var button = appBar.getButton(webapp.search.data.REFINE_SEARCH);
			if(button.strButtonStatus == "unavailable"){
				button.show();
			}

			if(this.intResultsCounter === 0) {
				// No result => display the popup, display the empty state and change the label of the container title
				ICTouchAPI.popupServices.basicPopup("warning", "", _("Search completed with no result.","webapp.search"), _("OK","webapp.search"));
				dojo.publish("webapp.search.showEmptyState");
			}
			// We set the container title with the appropriate label (Search completed)
			this.mainWidget.setContainerTitle(this.data.SEARCH_DONE);
		}
		ICTouchAPI.debugServices.info("webapp.search.controlBase - resultsReceived done");
	},

	_resultProcessing : function(resultSearchId, arrResults, nbResults) {
		this.intResultsCounter = nbResults;
		var items = [];
		// build the array of items to be displayed
		if (arrResults && arrResults.length > 0) {
			ICTouchAPI.debugServices.debug("webapp.search.controlBase - resultsReceived / SearchId: " + resultSearchId + " | nb of results is: " + arrResults.length + " on " + this.intResultsCounter);
			var objItem ;
			for (var i = 0 ; i < arrResults.length ; i++) {
				var contactSummary = arrResults[i];
				if(contactSummary.type !== ICTouchAPI.tools.enumContactType.CACHED){
					var contactPhone = ICTouchAPI.tools.getContactPhone(contactSummary);
					objItem = {
						intIndex: contactSummary.contactId,
						strPrimaryContent: ICTouchAPI.tools.getContactDisplayName(contactSummary),
						strSecondaryContent: "",
						objContact: contactSummary
					};

					// get the secondary content to be displayed (if needed)
					objItem.arrAdditionalContents = this.getArrAdditionalContent(contactSummary, contactPhone);
					// get the icons to be displayed (if needed)
					objItem.arrIcons = this.getArrIcons(contactSummary, contactPhone);

					items.push(objItem);
				}
			}

			dojo.publish("webapp.search.newItems", [items]);
		}
	},

	/**
	 * Called by resultsReceived to format the display.
	 * Search is completed. If the results received well correspond to the searchId ongoing, display them.
	 * get the secondary content to be displayed (if needed)
	 */
	getArrAdditionalContent : function(contactSummary, contactPhone) {
		// check if we have to display a secondary content
		var secondaryInformation = "";
		if (this.objSearchData.secondaryInformation) {
			if (this.objSearchData.secondaryInformation == "phoneNumber") {
				secondaryInformation = contactPhone;
			}
			else if (contactSummary[this.objSearchData.secondaryInformation]) {
				secondaryInformation = contactSummary[this.objSearchData.secondaryInformation];
			}
		}
		return [secondaryInformation];
	},

	getArrIcons : function(contactSummary, contactPhone) {
		// check if we have to display the icons or not
		var leftIcon = "";
		var rightIcon = "";
		// Check if the contact is local (in the directory -> LOCAL = 0, DM = 3, MA = 5),
		// or if it is remote, (UDA = 1, BUDDY = 2, ICS = 4)
		var boolIsRemoteContact = (contactSummary.type != ICTouchAPI.tools.enumContactType.LOCAL && contactSummary.type != ICTouchAPI.tools.enumContactType.DM && contactSummary.type != ICTouchAPI.tools.enumContactType.MA) ? true : false;
		if (this.objSearchData.lineClickCallback && (this.objSearchData.lineClickCallback.leftIconImage || this.objSearchData.lineClickCallback.leftIconImageRemoteContact)) {
				// Check if the left icon has to be displayed
				if (!this.objSearchData.lineClickCallback.leftIconImageConditionnalDisplay // if there is no condition to display the image
						|| (this.objSearchData.lineClickCallback.leftIconImageConditionnalDisplay == "phoneNumber" && contactPhone) // the condition is the phone number and the contact has a phone number
						|| contactSummary[this.objSearchData.lineClickCallback.leftIconImageConditionnalDisplay]) { // the contact has the conditionnal field
					leftIcon = (boolIsRemoteContact && this.objSearchData.lineClickCallback.leftIconImageRemoteContact) ? this.objSearchData.lineClickCallback.leftIconImageRemoteContact : this.objSearchData.lineClickCallback.leftIconImage;
				}
			}
		if (this.objSearchData.rightIconClickCallback && (this.objSearchData.rightIconClickCallback.rightIconImage || this.objSearchData.rightIconClickCallback.rightIconImageRemoteContact)) {
				// Check if the right icon has to be displayed
				if (!this.objSearchData.rightIconClickCallback.rightIconImageConditionnalDisplay // if there is no condition to display the image
						|| (this.objSearchData.rightIconClickCallback.rightIconImageConditionnalDisplay == "phoneNumber" && contactPhone) // the condition is the phone number and the contact has a phone number
						|| contactSummary[this.objSearchData.rightIconClickCallback.rightIconImageConditionnalDisplay]) { // the contact has the conditionnal field
					rightIcon = (boolIsRemoteContact && this.objSearchData.rightIconClickCallback.rightIconImageRemoteContact) ? this.objSearchData.rightIconClickCallback.rightIconImageRemoteContact : this.objSearchData.rightIconClickCallback.rightIconImage;
				}
		}
		return ["", "","",leftIcon,rightIcon];
	},

	searchLineClick : function(item) {
		if (this.objSearchData.lineClickCallback) {
			var contactSummary = item.objContact;
			this.getNeededSearchResultData(contactSummary, this.objSearchData.lineClickCallback);
		}
	},

	searchRightIconClick : function(item) {
		if (this.objSearchData.rightIconClickCallback) {
			var contactSummary = item.objContact;
			this.getNeededSearchResultData(contactSummary, this.objSearchData.rightIconClickCallback);
		}
	},

	getNeededSearchResultData : function(contactSummary, callback) {
		// if the neededData is the phoneNumber, directly call the callback with the
		// officePhone contained in the contactSumary obtained during the search
		if (callback.neededData == "phoneNumber") {
			callback.func.call(callback.context, contactSummary.officePhone);
		}
		// if the neededData is the contact summary, directly call the callback with the
		// ContactSumary obtained during the search
		else if (callback.neededData == "ContactSummary" || !callback.neededData) {
			callback.func.call(callback.context, contactSummary);
		}
		// if the neededData is the whole contact, we have to retreive the whole contact from
		// IctDirectory search results
		else if (callback.neededData == "Contact") {
			ICTouchAPI.APIServices.IctDirectory.getContactFromSearch({
				params:[this.getCurrentSearchId(), contactSummary.contactId],
				context:callback.context,
				callback: callback.func
			});
		}
		// if the field required is directly available in the simplified contact, use it
		else if (contactSummary[callback.neededData]) {
			callback.func.call(callback.context, contactSummary[callback.neededData]);
		}
		// else, we have to retreive the whole contact from IctDirectory search results
		else {
			ICTouchAPI.APIServices.IctDirectory.getContactFromSearch({
				params:[this.getCurrentSearchId(), contactSummary.contactId],
				context:callback.context,
				callback: callback.func
			});
		}
	},

	showKeyboard : function() {
		// Force the search by name instead of using the setting _boolDialpadByDefault
		if (this.objSearchData.boolForceByname) {
			this.objSearchData.boolForceByname = false;
			this.showSwitchableKeyboard(false);
		}
		// Force the dialpad instead of using the setting _boolDialpadByDefault
		else if (this.objSearchData.boolForceDialpad) {
			this.objSearchData.boolForceDialpad = false;
			this.showSwitchableKeyboard(true);
		}
		else if (this.objSearchData.useDefaultCallMethod) {
			this.showSwitchableKeyboard(this.data._boolDialpadByDefault);
		} else {
			ICTouchAPI.transitionServices.getScreen({
				name: "webapp.search.getSearch",
				params: {}
			});

			var context = this;
			var funcCallbackOk = function(){
				context._funcCallbackOk(arguments);
			};
			var funcCallbackExit = function(){
				context._funcCallbackExit(arguments);
			};
			ICTouchAPI.keyboardServices.deployKeyboard(ICTouchAPI.KeyboardTypes.ALPHANUM, null, {
				strDefaultText: this.strCurrentText,
				funcCallbackKeyPressed: dojo.hitch(this, this.clbkKeyPressed),
				funcCallbackExit : funcCallbackExit,
				funcCallbackOk	: funcCallbackOk
			});
		}
	},

	_funcCallbackOk : function(){
		var funcCallbackOk = function(){
			return;
		};
		if(this.objSearchData.dialpadObject){
			if(typeof this.objSearchData.dialpadObject.funcCallbackOk === "function"){
				funcCallbackOk = this.objSearchData.dialpadObject.funcCallbackOk;
			}
		}
		// If a byNameObject is given, its callbacks overwrite dialpadObject's callbacks
		if (this.objSearchData.byNameObject) {
			if(this.objSearchData.byNameObject.funcCallbackOk && typeof this.objSearchData.byNameObject.funcCallbackOk === "function"){
				funcCallbackOk = this.objSearchData.byNameObject.funcCallbackOk;
			}
		}
		// If less than IntMinCharacter were entered in the keyboard when pressing OK button, launch the search
		if (this.strCurrentText && this.strCurrentText.length < this.data.getIntMinCharacter()) {
			this._restartSearch();
		}
		funcCallbackOk();
	},

	_funcCallbackExit : function(){
		var funcCallbackExit = function(){
			return;
		};
		if(this.objSearchData.dialpadObject){
			if(typeof this.objSearchData.dialpadObject.funcCallbackExit === "function"){
				funcCallbackExit = this.objSearchData.dialpadObject.funcCallbackExit;
			}
		}
		// If a byNameObject is given, its callbacks overwrite dialpadObject's callbacks
		if (this.objSearchData.byNameObject) {
			if(this.objSearchData.byNameObject.funcCallbackExit && typeof this.objSearchData.byNameObject.funcCallbackExit === "function"){
				funcCallbackExit = this.objSearchData.byNameObject.funcCallbackExit;
			}
		}
		funcCallbackExit();
	},

	showSwitchableKeyboard : function(boolDialpad) {
		//this.strCurrentText = "";
		if (!boolDialpad) {
			dojo.publish("communication/switchKeyboard",[true]);
			ICTouchAPI.transitionServices.getScreen({
				name: "webapp.search.getSearch",
				params: {}
			});
			if (typeof this.objSearchData.onOpenByName == "function") {
				this.objSearchData.onOpenByName();
			}
			var context = this;
			var funcCallbackOk = function(){
				context._funcCallbackOk(arguments);
			};
			var funcCallbackExit = function(){
				context._funcCallbackExit(arguments);
			};
			ICTouchAPI.keyboardServices.deployKeyboard(ICTouchAPI.KeyboardTypes.ALPHASEARCH, null, {
				strDefaultText: this.strCurrentText,
				funcCallbackKeyPressed: dojo.hitch(this, this.clbkKeyPressed),
				funcCallbackExit : funcCallbackExit,
				funcCallbackOk	: funcCallbackOk,
				arrCustomCallbacks: {
					"dial-pad": dojo.hitch(this, this.showSwitchableKeyboard, true)
				},
				showVoicemailButton : webapp.search.objSearchData.showVoicemailButton,
				showAddcontactButton : webapp.search.objSearchData.showAddcontactButton,
				callingWebapp : webapp.search.objSearchData.callingWebapp
			});
		}
		else {
			dojo.publish("communication/switchKeyboard",[false]);
			// show numeric
			if (typeof this.objSearchData.onOpenDialpad == "function") {
				this.objSearchData.onOpenDialpad();
			}
			var objDialpad = this.objSearchData.dialpadObject;
			if (objDialpad == undefined) {
				objDialpad = {
					funcCallbackOk : function() {
						webapp.search.objSearchData.lineClickCallback.func.call(webapp.search.objSearchData.lineClickCallback.context, null, {
							strContent: this.getValue()
						});
					},
					funcCallbackExit : function() {
						if(webapp.search.objSearchData.funcCallbackExit) {
							webapp.search.objSearchData.funcCallbackExit();
						}
					},
					showVoicemailButton : webapp.search.objSearchData.showVoicemailButton,
					showAddcontactButton : webapp.search.objSearchData.showAddcontactButton,
					callingWebapp : webapp.search.objSearchData.callingWebapp
				};
			}
			objDialpad.arrCustomCallbacks = {
				"keyboard": dojo.hitch(this, this.showSwitchableKeyboard, false)
			};
			ICTouchAPI.keyboardServices.deployKeyboard(ICTouchAPI.KeyboardTypes.NUMSEARCH, null, objDialpad);
		}
	},

	// Called when the user push "go to contact" button in dialpad
	showContacts: function() {
		ICTouchAPI.keyboardServices.closeKeyboard(ICTouchAPI.keyboardServices.CLOSE_ACTION_EXIT);
		ICTouchAPI.transitionServices.getScreen({
			name: "webapp.contacts.getContact",
			params: {}
		});
	},

	getCurrentSearchId : function() {
		return "webapp_search_" + this.intCurrentSearchId;
	},

	getDefaultCallMethod : function(objSetting) {
		webapp.search.data._boolDialpadByDefault = !(objSetting != undefined && objSetting.jsValue == 1);
	},

	isRightIcon : function() {
		return (webapp.search.objSearchData.contactDataTypes && webapp.search.objSearchData.contactDataTypes.length > 0);
	},

	setMinCharacterValue : function(objSetting) {
		if(objSetting) {
			webapp.search.data.setIntMinCharacter(objSetting.jsValue);
		}
	},

	unlock: function(name) {
		// Close keyboard when we leave search
		if (name && name.split('.')[1] !== "search") {
			ICTouchAPI.keyboardServices.closeKeyboard(ICTouchAPI.keyboardServices.CLOSE_ACTION_EXIT);
		}

		this.inherited(arguments);
	}
});
dojo.require("webapp.search.controlBase");
dojo.provide("webapp.search");
dojo.declare("webapp.search", webapp.search.controlBase,
{

});

webapp.search = new webapp.search();
dojo.provide("webapp.search.dataBase");
dojo.declare("webapp.search.dataBase",null,{
	// Member data used for object construction
	_arrAppButtons          : [],
	_arrSysButtons          : [],

	// i18n labels for the container title
	NO_SEARCH				: _("Search", "ICTouchAPI"),
	ONGOING_SEARCH			: _("Search in progress...", "webapp.search"),
	SEARCH_DONE				: _("Search completed", "webapp.search"),

	// Constant values for the button's id
	BACK                    : "BACK_BTN",
	NEW_SEARCH              : "NEW_SEARCH_BTN",
	REFINE_SEARCH           : "REFINE_SEARCH_BTN",

	// Other
	_boolDialpadByDefault		: true,
	_intMinCharacter			: 0,

	constructor : function(){
		var func = dojo.hitch( webapp.search, webapp.search.buttonCallback);
		//this._arrAppButtons[this.NEW_SEARCH] = {strButtonName: this.NEW_SEARCH, strButtonLabel: _('New', 'ICTouchAPI'), strButtonIcon: 'generic-search-by-name', callback: func};
		this._arrAppButtons[this.REFINE_SEARCH] = {strButtonName: this.REFINE_SEARCH, strButtonLabel: _('Search', 'ICTouchAPI'), strButtonIcon: 'generic-search-by-name', callback: func};

		// Register action bars in AppBarServices for child widgets
		var appBar = ICTouchAPI.AppBarServices.getAppBar("search", "getSearch");
		for (var k in this._arrAppButtons) {
			var objButton = new UIElements.AppButton.AppButtonControl(this._arrAppButtons[k]);
			this._arrAppButtons[k].objAppButton = objButton;
			appBar.addActionButton(objButton);
		}
	},

	getAppBarSysButtons : function() {
		return this._arrSysButtons;
	},

	getAppBarAppButtons : function() {
		return this._arrAppButtons;
	},

	getTitle : function() {
		//Returns default title for no ongoing search
		return this.NO_SEARCH;
	},

	isPreviousHome : function() {
		if (webapp.search.clbkBackButton && typeof webapp.search.clbkBackButton === "function")
			return false;
		else //Default "back" action,
			return true;
	},

	getIntMinCharacter : function() {
		return this._intMinCharacter;
	},

	setIntMinCharacter : function(value) {
		this._intMinCharacter = value;
	}

});
dojo.require("webapp.search.dataBase");
dojo.provide("webapp.search.data");
dojo.declare("webapp.search.data",webapp.search.dataBase,{
});

webapp.search.data = new webapp.search.data;
dojo.provide("webapp.search.getSearchBase");
dojo.declare("webapp.search.getSearchBase",
	[ICTouchAPI.webWidget, dojox.dtl._Templated],
	{
		container   : null,

		constructor: function() {

			/* For transistionServices; go back to previous application */
			this.stealth();
		},

		postCreate:function() {
			var data = webapp.search.data;
			var am = new UIElements.ApplicationMode.FullControl({
				content : {
					name    : "webapp.search.getSearchContent",
					params  : {}
				},
				objTitle : {
					strLabel: data.getTitle()
				}
			}, this.mainPart);
			this.container = am.getContent();
			webapp.search.mainWidget = this;

			// System buttons
			var _arrSys = data.getAppBarSysButtons();
			for (var i in _arrSys) {
				ICTouchAPI.AppBarServices.addStaticButton("search", "getSearch", _arrSys[i]);
			}

			this._changeBackButton();

			ICTouchAPI.AppBarServices.setAppBar("search", "getSearch", "getSearch");
		},

		_changeBackButton: function() {
			var isPrevious = this.webapp.data.isPreviousHome();
			var backFunc = dojo.hitch(webapp.search, webapp.search.buttonCallback);
			ICTouchAPI.AppBarServices.addExitButton("search", "getSearch", isPrevious, backFunc);
						},

		reloadContainer : function() {
			var data = webapp.search.data;
			this.container.reload({
				objTitle: {
					strLabel: data.getTitle()
				},
				objContent: {
					name: "webapp.search.getSearchContent",
					params: {}
				}
			});
		},

		setContainerTitle : function(strTitle) {
			this.container.getTitle().changeTitle(strTitle);
		},

		destroy : function(){
			this.container.destroy();
			this.inherited(arguments);
		}
	}
	);
dojo.require("webapp.search.getSearchBase");
dojo.provide("webapp.search.getSearch");
dojo.declare("webapp.search.getSearch",
	webapp.search.getSearchBase,
	{
		templatePath	: dojo.moduleUrl("webapp.search","templates/getSearch8082.html"),
    }
);
dojo.provide("webapp.search.getSearchContentBase");
dojo.declare("webapp.search.getSearchContentBase",
	[ICTouchAPI.webWidget, dojox.dtl._Templated],
	{
		_hNewItemsSubscribe : null, // handler of webapp.search.newItems event subscription
		_hEmptyListSubscribe : null, // handler of webapp.search.emptyItems event subscription
		_hManageEmptyStateVisibility : null, // handler of webapp.search.showEmptyState event subscription

		UIDataList : null,
		UIEmptyState : null,
		UILoading : null,

		postCreate:function() {
			var funcLineClick = dojo.hitch(webapp.search, webapp.search.searchLineClick);
			var funcRightIconClick = dojo.hitch(webapp.search, webapp.search.searchRightIconClick);

			this.UIDataList = new UIElements.DataList.DataListControl({
				arrItems: [],
				boolShowAdditionalContents: true,
				lineClickedCallback: funcLineClick,
				iconClickedCallback: funcRightIconClick,
			}, this.domContainer);

			this._hNewItemsSubscribe = dojo.subscribe("webapp.search.newItems",  this, function (items) {
				this.UIDataList.pushItems(items, true);
				items = null;
			});
			this._hEmptyListSubscribe = dojo.subscribe("webapp.search.emptyItems", this, function () {
				this.UIDataList.emptyItems();
				this.UIEmptyState.setVisibility(false);
			});

			// Don't use the EmptyState UI provided by the DataList, use a one defined in search in order to choose when it has to be shown
			// (it should only be displayed when there is no results, not when we don't have started a search yet)
			this.UIEmptyState = new UIElements.EmptyState.EmptyStateControl({
				strEmptyMessage: _("No search results", "webapp.search"),
				boolVisible: false
			}, this.domEmptyState);

			this._hManageEmptyStateVisibility = dojo.subscribe("webapp.search.showEmptyState",  this, function () {
				this.UIEmptyState.setVisibility(true);
			});

		},

		destroy : function(){
			if(this.UIDataList){
				dojo.unsubscribe(this._hNewItemsSubscribe);
				dojo.unsubscribe(this._hEmptyListSubscribe);
				dojo.unsubscribe(this._hManageEmptyStateVisibility);
				this.UIDataList.destroy();
			}
			this.inherited(arguments);
		}
	}
);
dojo.require("webapp.search.getSearchContentBase");
dojo.provide("webapp.search.getSearchContent");
dojo.declare("webapp.search.getSearchContent",
	webapp.search.getSearchContentBase,
	{
        // The path to the template is defined here
        templatePath	: dojo.moduleUrl("webapp.search","templates/getSearchContent8082.html"),
	}
);
dojo.provide("webapp.search.getFiltersBase");
dojo.declare("webapp.search.getFiltersBase",
	[ICTouchAPI.webWidget, dojox.dtl._Templated],
	{
		UIselectList : null,

		postCreate:function(){
			var func = function(intIndex) {
                webapp.search.data.intNewFilter = intIndex;
            };
            var _arr = webapp.search.data.getFilters();
            for (var i in _arr)
                _arr[i].boolSelected = (_arr[i].intIndex == webapp.search.data.intCurrentFilter);
			this.UIselectList = new UIElements.SelectList.SelectListControl({
				arrItems: _arr,
				intContainerSize: 4,
				boolMultiselect: false,
				itemCallback: func
			}, this.domSelectList);
		},

		destroy : function(){
			this.UIselectList.destroy();
			this.inherited(arguments);
		}
	}
);
dojo.require("webapp.search.getFiltersBase");
dojo.provide("webapp.search.getFilters");
dojo.declare("webapp.search.getFilters",
	webapp.search.getFiltersBase,
	{
		templatePath: dojo.moduleUrl("webapp.search","templates/getFilters8082.html")
	}
);
