#!/bin/sh
#
# upgrade_scripts.sh
# script that contains a collection of functions to
# manage upgrade process start up.
#
# Changelog
#
#	- Wed Aug 17 2011 - Guillaume Catto <guillaume.catto@alcatel-lucent.fr>
#	* file's creation
#
#       - Tue Jun 26 2012 - Jerry ZHOU <jerry.zhou@alcatel-lucent.com>
#       * Fix crms00382768: remove "upgrade.pid" when "ADMCFG_UPDATE_ENABLE" set false
#
#       - The Aug 16 2012 - Jerry ZHOU <jerry.zhou@alcatel-lucent.com>
#       * Fix crms00390845: call 'backup_restore' to backup restore rpm to '/data/restore_rpm_bk'
#
#	- Wed Oct 17 2012 - Ziming Xu <Ziming.b.Xu@alcatel-lucent.com>
#	* Fix crms00402118: upgrade from R250 to R260 failed by "dwl seturl" command
#
#       - Wed Jan 23 2012 - Ziming Xu <Ziming.b.Xu@alcatel-lucent.com>
#       * PR341502: Reduce Reboot Frequency
#
#       - Fri Feb 09 2013 - Ziming Xu <Ziming.b.Xu@alcatel-lucent.com>
#       * Fix crms00417047: confusion about the style of dmurl update
#
# Source global script to access some common functions
. /usr/lib/upgrade/rpmdbbackup-addon.sh
# PR341502 zimingxu+
. /etc/functions
# PR341502 zimingxu-

##################################
# Global variables               #
##################################
RUN_ACTION=""
SIGNAL_TRAPPED=0
CHECK_FLAG=0
#crms00386821 jerryzh+
WAIT_PRESENTATION_FLAG=0
#crms00386821 jerryzh-
SET_RESTORE=0
UPGD_PART=0
update=$(which update)
dblock=/.dblcheck.lock
# +++ crms00xxxxxx
async_locked=/.async.lock
# --- crms00xxxxxx
#crms00386821 jerryzh+
presentation_ready_flag=/tmp/.presentation_ready_flag
#crms00386821 jerryzh-
# crms00402118 zimingxu-
STATUS=/usr/sbin/upgd_status
# crms00402118 zimingxu+
# PR341502 zimingxu+
mutex_param=mtx_param                                
mutex_context=ctx_upgdparam                          
mutex_check=0
# PR341502 zimingxu-

##################################
# Verify all input directories   #
##################################
set_upgrade_directories() {
    local FUNCTION=set_upgrade_directories
    ENTER_FUNCTION

    : "${attr}Create missing directories${default}"
    local elem=""
#crms00390845 jerryzh+
    local dirlist="$workdir $tmpdir $restoredir $otherrpmdir $ext_data_path $ext_system_data_path"
#crms00390845 jerryzh-
    for elem in $dirlist; do
	[ ! -d $elem ] && mkdir -m 0755 -p $elem
    done

    RETURN
}

##################################
# Verify flags used when flash   #
# partitions are processed       #
##################################
get_flash_status() {
    local FUNCTION=get_flash_status
    ENTER_FUNCTION
    local ret=
    local flag="/boot/.flash_$1"

    if [ -f $flag ]; then
	# Verify if RPM is available in database, if yes remove it
	# so that it can be reinstalled later
	rpm -q $1 > /dev/null 2>&1
	[ $? -eq 0 ] &&	rpm -e --nodeps $1
	printf "(ko) "
	ret=1
    else
	printf "(ok) "
	ret=0
    fi

    RETURN $ret
}	

check_flash_flags() {
    local FUNCTION=check_flash_flags
    ENTER_FUNCTION

    : "${attr}Verify flags used when flashing partition${default}"
    local elem="", ret=0
    local flagslist="upgrade_image u-boot ibl"

    printf "Get flash status for element: "
    for elem in $flagslist; do
	printf "$elem"
	get_flash_status $elem
	ret=$(expr $? + $ret)
    done

    printf "\n"

    RETURN $ret
}

##################################
# Management of "UPGRADE" status #
##################################
manage_post_step2() {
    local FUNCTION=manage_post_step2
    ENTER_FUNCTION
    # +++ crms00310043 +++
    # extract data from kernel's command line
    : "${attr}Parse kernel command line${default}"
    kern_cmdline=$(cat /proc/cmdline)
	
    # verify that flashed component IBL/U-boot have the right version number
    verify_component "ibl" "ibl"
    verify_component "sbl" "u-boot"
    # --- crms00310043 ---

    # In case something goes wrong or a restore has been done
    # retrieve parameters from the DM
    if [[ "$UPGD_STATUS" == "pre2" ]]; then
	    [[ ! -z "$dmconfig" ]] && $dmconfig
    fi

    RETURN
}

##################################
# Verify upgrade status          #
##################################
get_upgrade_status() {
    local FUNCTION=get_upgrade_status
    ENTER_FUNCTION

    : "${attr}Get upgrade status and analyze it${default}"
    # Get status
    if [ -f $statusfile ]; then
	source $statusfile
	UPGD_STATUS=$STEP
    else
	UPGD_STATUS="pre0"
    fi

    # Analyse it
    case $UPGD_STATUS in
	pre0)
	    RUN_ACTION="build"
	    # Truncate file size to only contains the current upgrade logs
	    [[ $CHECK_FLAG == 0 ]] && :> ${reportfile}
	    ;;
	pre2|2)
	    RUN_ACTION="build"
	    manage_post_step2
	    ;;
	0|pre1)
	    RUN_ACTION="execute"
	    ;;
	*)
	    unset RUN_ACTION
	    logger -s -t upgrade -p local0.NOTICE "No need to run this step ($UPGD_STATUS)"
	    ;;
    esac
    
    RETURN
}

##################################
# Global checks                  #
##################################
load_current_settings() {
    local FUNCTION=load_current_settings
    ENTER_FUNCTION

    : "${attr}Load CURRENT settings${default}"
    local elem=""
# crms00417047 zimingxu+
    local settings_list="ENETCFG_DM ENETCFG_DM_BACKUP ENETCFG_UPGRADE_FILE ENETCFG_DM_PROTOCOL ENETCFG_DM_PORT WEBAPP_DEPLOYMENT_MODE ENETCFG_DHCP_MODE"
# crms00417047 zimingxu-

# PR341502 zimingxu+
    mutex_create_context $mutex_context                  
    mutex_check=1                                        
    mutex_lock $mutex_param $mutex_context
# PR341502 zimingxu-
    for elem in $settings_list; do
	: "${attr}Load $elem current setting value${default}"
	get_setting_value "$elem"
    done
# PR341502 zimingxu+
    mutex_unlock $mutex_param $mutex_context
# PR341502 zimingxu-

    RETURN
}

##################################
# validate loaded settings       #
##################################
validate_settings() {
    local FUNCTION=validate_settings
    ENTER_FUNCTION

    : "${attr}Verify imported CURRENT settings${default}"
    local ret=

    # First check UPGRADE and DM settings
    validate_parameters
    ret=$?

    # Then define servers' PORTs and PORTOCOLs
    if [[ $ret == 0 ]]; then
	get_servers_parameters
	ret=$?
    fi

    RETURN $ret
}

##################################
# Build upgrade scenario         #
##################################
build_scenario_start() {
    local FUNCTION=build_scenario_start
    ENTER_FUNCTION

    : "${attr}Start building upgrade scenario${default}"
    local cmdline=

    if [[ $CHECK_FLAG == 1 ]]; then
	cmdline="-c"
    else
        : "${attr}==== Set \"STEP=pre0\" ====${default}"
	echo "STEP=pre0" > $statusfile
    fi
	
    # Build cmdline
    if [ -z "$BACKUP_PARAMETERS" ]; then
	cmdline="$cmdline -d $WEBAPP_DEPLOYMENT_MODE $MAIN_PARAMETERS $ENETCFG_UPGRADE_FILE"
    else
	cmdline="$cmdline -d $WEBAPP_DEPLOYMENT_MODE -b $BACKUP_PARAMETERS $MAIN_PARAMETERS $ENETCFG_UPGRADE_FILE"
    fi
	
    $scenario $cmdline
    RETURN $?
}

##################################
# Execution of upgrade scenario  #
##################################
execute_scenario_start() {
    local FUNCTION=execute_scenario_start
    ENTER_FUNCTION

    : "${attr}Start executing scenario from the beginning${default}"
    start_update 1 $WEBAPP_DEPLOYMENT_MODE
    ret=$?
	
    : "${attr}Verify if a restore is needed${default}"
    if [[ $ret == 2 && -e $scenariofile ]]; then
	: "${attr}Restart an upgrade step${default}"
	start_update 1
	ret=$?
    fi

    RETURN $ret
}

##################################
# Signals management             #
##################################
clean_signal() {
    local pid=$(pidof -s build-scenario update)

    logger -s -t upgrade -p local0.ERR "Signal trapped forcing end of script!"
    if [ -z "$pid" ]; then
	SIGNAL_TRAPPED=1
    else
        if [[ $CHECK_FLAG == 0 ]]; then
	    cleanup 1

            : "${attr}==== Set \"STEP=pre2\" ====${default}"
	    echo "STEP=pre2" > $statusfile
        fi
        exit 4 
    fi
}

# PR341502 zimingxu+
exit_handle() {                                        
    [[ $mutex_check -eq 1 ]] && { mutex_unlock $mutex_param $mutex_context; mutex_delete_context $mutex_context; }
}
# PR341502 zimingxu+

##################################
# Main                           #
##################################

# Save process PID
PROGNAME=$(basename $0)
save_pid

# Trap signal: HUP => do nothing
trap "" HUP
# Trap signals: INT QUIT KILL => call function to do some cleanup
trap clean_signal INT QUIT KILL
# PR341502 zimingxu+
trap exit_handle EXIT
# PR341502 zimingxu-

#crms00386821 jerryzh+
# PR341502 zimingxu+
if [[ $# -ge 1 ]]; then
    case "$1" in
        check)
            CHECK_FLAG=1
        ;;
        wait_presentation)
            WAIT_PRESENTATION_FLAG=1
        ;;
        cmdline_args)
            shift
            upgrade_params="$@"
        ;;
# PR341502 zimingxu-
    esac
fi
#crms00386821 jerryzh-

set_upgrade_directories

# In case of a check, do not verify if upgrade is enabled or not
# neither verify if component have been flashed successfully
if [[ $CHECK_FLAG == 0 ]]; then
	get_upgrade_status

	# Verify if an upgrade can be started but only if status is pre0
	if [[ "$RUN_ACTION" == "build" ]]; then
		get_setting_value "ADMCFG_UPDATE_ENABLE"
#crms00390845 jerryzh+
# crms00402118 zimingxu-
        if [[ "$ADMCFG_UPDATE_ENABLE" != "true" ]] && $STATUS; then
            unset RUN_ACTION
        fi
# crms00402118 zimingxu+
#crms00390845 jerryzh-
	fi
            
	# Do not continue if neither the scenario's build or its execution are requested
#crms00382768 jerryzh+
	if [[ -z "$RUN_ACTION" ]]; then
	    remove_pid              
	    exit 0
    fi
#crms00382768 jerryzh-

	check_flash_flags
	if [[ $? == 1 ]]; then
		logger -s -t upgrade -p local0.ERR "Flashing component failed, restarting upgrade process"
		cleanup
		RUN_ACTION="build"
	fi
else
	RUN_ACTION="build"
fi

# Retrieve useful settings
load_current_settings
# PR341502 zimingxu+
eval $upgrade_params
# PR341502 zimingxu-

# Validate loaded settings
validate_settings
ret=$?
echo "ENETCFG_UPGRADE_FILE= $ENETCFG_UPGRADE_FILE"

#crms00386821 jerryzh+
if [[ $WAIT_PRESENTATION_FLAG -eq 1 ]]; then
    wait_time_out=300
    while [[ ! -e $presentation_ready_flag ]] && [[ $wait_time_out -gt 0 ]]; do
        sleep 10
        wait_time_out=`expr $wait_time_out - 10`     
    done
fi
#crms00386821 jerryzh-

# Execute the right function now
if [[ $ret == 0 ]]; then
#crms00390845 jerryzh+
    if [[ "$RUN_ACTION" == "execute" ]]; then
        # backup restore rpms
        backup_restore      
    fi
#crms00390845 jerryzh-
	eval ${RUN_ACTION}_scenario_start
	ret=$?

	# If build of scenario succeeded, execute it right now in the same environment
	if [[ -e $scenariofile && $CHECK_FLAG == 0 && "$RUN_ACTION" == "build" && $ret == 0 && $SIGNAL_TRAPPED == 0 ]]; then
#crms00390845 jerryzh+
        # backup restore rpms
        backup_restore
#crms00390845 jerryzh-
        execute_scenario_start
        ret=$?
    fi
fi

# Treat the returned code
analyze_returned_code $ret

[[ $SIGNAL_TRAPPED == 1 ]] && ret=4

# In all cases, remove stored pid
remove_pid

exit $ret

