#!/bin/sh
#
# This script defines some functions used to
# backup and restore the rpmdb. They will be
# used in upgrade startup services' scripts.
#
# Changelog:
#
#	- Tue May 04 2010 - Guillaume Catto <guillaume.catto@alcatel-lucent.com>
#	* Add a function to rotate file
#
#	- Thu May 06 2010 - Guillaume Catto <guillaume.catto@alcatel-lucent.com>
#	* Add function to write logs to report file
#
#	- Wed May 26 2010 - Guillaume Catto <guillaume.catto@alcatel-lucent.com>
#	* Unify the way returned code is managed into all scripts
#
#	- Wed Jun 02 2010 - Guillaume Catto <guillaume.catto@alcatel-lucent.com>
# 	* Remount the SYSTEM filsystem without the "sync" option
#
#	- Mon Jun 07 2010 - Guillaume Catto <guillaume.catto@alcatel-lucent.com>
#	* Modify cleanup function to also list and remove hidden files
#
#	- Tue Jun 15 2010 - Guillaume Catto <guillaume.catto@alcatel-lucent.com>
#   	* Fix crms00238623: take into account the reboot flag before cleaning up upgrade files
#
#	- Wed Jun 16 2010 - Guillaume Catto <guillaume.catto@alcatel-lucent.com>
#	* Fix crms00238705: do not delete .serversip.bak configuration file while doing the cleanup
#
#	- Wed Jun 23 2010 Guillaume Catto <guillaume.catto@alcatel-lucent.com>
#	* Fix crms00239986: Remove old gpg signature files due to the change of their destination directory
#
#	- Thu Jul 01 2010 - Guillaume Catto <guillaume.catto@alcate-lucent.com>
#	* Fix crms00242620: Call a script to check if there is no multiple packages with the same names 
#	  available in the database
#	* Fix crms00242598: Change the reset type (soft => hard)
#
#	- Tue Jul 27 2010 - Guillaume Catto <guillaume.catto@alcatel-lucent.com>
#	* Fix crms00246998: in static dhcp mode, send alse send the protocol used with the server's IP/Name
#
#	- Mon Aug 02 2010 - Guillaume Catto <guillaume.catto@alcatel-lucent.com>
#	* Fix crms00249742: while retriving backuped upgrade parameters, the one not set are retrieved from defaults
#
#	- Wed Aug 06 2010 - Guillaume Catto <guillaume.catto@alcatel-lucent.com>
#	* Mount the SYSTEM partition in async mode to install RPMs
#	* Remove the check of the database from this script
#	* add a function to remove temporary RPM files at the end of an upgrade
#	* Modify the way the URL are detected
#	* Add at the end of the upgrade step 5 a check of the rpm database to fix issues
#
#	- Wed Aug 18 2010 - Guillaume Catto <guillaume.catto@alcatel-lucent.com>
#	* add a lock on the check of RPM database
#
#	- Thu Aug 26 2010 - Guillaume Catto <guillaume.catto@alcatel-lucent.com>
#	* Fix crms00xxxxxx: add a lock on async mode
#
#	- Wed Sep 01 2010 - Guillaume Catto <guillaume.catto@alcatel-lucent.com>
#	* Fix crms00256623: truncate upgrade report file if they are too big
#
#	- Wed Sep 08 2010 - Guillaume Catto <guillaume.catto@alcatel-lucent.com>
#	* Fix crms00258187: do a cleanup if an update has been started but input parameters are wrong
#
#	- Wed Sep 15 2010 - Guillaume Catto <guillaume.catto@alcatel-lucent.com>
#	* Fix crms00259952: improve the check of input parameters
#
#	- Wed Oct 08 2010 - Guillaume Catto <guillaume.catto@alcatel-lucent.com>
#	* Fix crms00265443: add the support of DynamicAlcatel DHCP mode
#
#	- Mon Oct 25 2010 - Guillaume Catto <guillaume.catto@alcatel-lucent.com>
#	* Fix crms00259524: manage webapp deployment mode setting
#
#	- Mon Nov 15 2010 - Guillaume Catto <guillaume.catto@alcatel-lucent.com>
#	* Fix crms00273199: treat "request upgrade" flag to force a reboot
#
#	- Fri Jan 28 2011 - Guillaume Catto <guillaume.catto@alcatel-lucent.com>
#	* Fix crms00289400: in case of error in upgrade process, keep the "in upgrade" status
#
#	- Fri Feb 11 2011 - Guillaume Catto <guillaume.catto@alcatel-lucent.com>
#	* Fix crms00290807: update all reboot reasons to include information on versions
#
#	- Fri Mar 25 2011 - Guillaume Catto <guillaume.catto@alcatel-lucent.com>
#	* Fix crms00303412: limit the use of hard reset in upgrade process
#
#	- Tue Mar 29 2011 - Guillaume Catto <guillaume.catto@alcatel-lucent.com>
#	* Fix crms00303424: support the use of tcp ports in STATIC mode
#
#	- Thu Apr 28 2011 - Guillaume Catto <guillaume.catto@alcatel-lucent.com>
#	* Fix crms00310043: add function to check ibl/u-boot version's from kernel command line
#
#	- Mon Jul 18 2011 - Guillaume Catto <guillaume.catto@alcatel-lucent.com>
#	* Fix crms00326993: remove code that manage upgrade_report files as their management is done using logrotate
#
#	- Tue Feb 29 2012 - Thanh lam NGUYEN <thanh-lam.nguyen@alcatel-lucent.com>
#	* Fix crms00362595: add binary paths, do_restore and is_unlock function
#
#	- Mon July 30 2012 - Ziming B Xu <Ziming.b.Xu@alcatel-lucent.com>
#	* Fix crms00386875 Change the cause of parsing from removing all dm relative 
# 	  configuration files to delete only the checksum files.
#
#	- Wed Aug 15 2012 - Ziming Xu <Ziming.b.Xu@alcatel-lucent.com>
#	* Feature ID: 411654 - Non standard https port support in DHCP mode
#
#	- Thu Aug 16 2012 - Jerry ZHOU <jerry.zhou@alcatel-lucent.com>
#	* Fix crms00390845: add function 'backup_restore' and fix error of function 'analyze_return_code'
#
#	- Thu Sep 6 2012 - Jerry ZHOU <jerry.zhou@alcatel-lucent.com>
#	* Fix crms00394219: remount DATA/CONFIG to /system on UPGRADE partition
#
#	- Thu Sep 13 2012 - Jerry ZHOU <jerry.zhou@alcatel-lucent.com>
#	* Fix crms00395560: remove code for fix crms00394219, change remount DATA/CONFIG partition in 'C' code
#
#	- Thu Jan 24 2013 - Jerry ZHOU <jerry.zhou@alcatel-lucent.com>
#	* PR261951: Reduce time of unavailability during boot time
#
#	- Fri Feb 09 2013 - Ziming Xu <Ziming.b.Xu@alcatel-lucent.com>
#	* Fix crms00417047: confusion about the style of dmurl update
#
#	- Mon Jun 03 2013 - Ziming Xu <Ziming.b.Xu@alcatel-lucent.com>
#	* Fix crms00432759: upgrade failed from R300.01.004.1 to R200.01.035.2
#
##################################
# Source external scripts        #
##################################

source /etc/init.d/rc.config

##################################
# Global variables               #
##################################
# system data
system_prefix=${system_prefix:-}

#rpm data
rpmdb_path=/var/lib/rpm
ubootfile=/config/u-boot/u-boot.cfg
workdir=/config/upgrade

# +++++ crms00362595 +++++
restoredir=/config/restore/rpm
# ----- crms00362595 -----
#crms00390845 jerryzh+
backup_restoredir=/data/restore_rpm_bk
otherrpmdir=/data/rpm
#crms00390845 jerryzh-
restoreflag=$workdir/.restore
retryflag=$workdir/.retry
statusfile=$workdir/status
rebootconfig=$workdir/reboot
scenariofile=$workdir/scenario.xml
# +++++ crms00362595 +++++
profilefile=$workdir/profile.xml
profilesum=$workdir/profile.xml.sum
# ----- crms00362595 -----
backup_server_file=$workdir/.serversip.bak
reportfile=$workdir/upgrade_report
backuplog_file=$workdir/upgrade.log
tmpdir=/tmp/upgrade
rundir=/var/run

procmount=/proc/mounts

# Fix crms00239986: remove old certificates files
old_xmlkey_file=/etc/trust_store/ICTouch_keys.xml
old_cert_file=/etc/trust_store/ICTouch_ca.pem
old_log_file=/upgd/upgrade.log

send_msg=$(which send_infra)
rebootset=$(which setnextboot)
reboot=$(which reboot)
update=$(which update)
scenario=$(which build-scenario)
dblcheck=$(which dblcheck)
dmconfig=$(which dmconfig)
# +++++ crms00362595 +++++
dwl=/usr/sbin/debug/dwl
unlock=$system_prefix/usr/sbin/.unlock
# ----- crms00362595 -----

# +++ crms00273199
request_upgrade=/config/upgrade/.request_upgd
# --- crms00273199

# +++ crms00294436
ext_data_path=/data/resources
ext_system_data_path=${ext_data_path}/system
# --- crms00294436

PREFIX=
# +++ crms00290807 +++
unset REBOOT_PART
unset TXT_POPUP
# --- crms00290807 ---

# +++ HACK +++ R250 +++
restore_flag=$workdir/.restore
# --- HACK --- R250 ---

process_management=/usr/lib/upgrade/upgrade_scripts.sh
rundir=/var/run
childpid=$rundir/upgrade.pid
dbg_flag=/tmp/dbgupgd_script

# crms00417047 zimingxu+
STATUS=/usr/sbin/upgd_status
# crms00417047 zimingxu-

# define_aliases
#---------------
# ARGS:
#   <none>  define macros that are used in the different functions
define_aliases() {
    local -
    set +x

    esc=$'\x1b'
    attr="$esc[7m"
    errorattr="$esc[31;7m"
    default="$esc[0m"
    ROOT=${ROOT:-}
    HOME_ADMIN_SYS="/home/admin"
    HOME_ADMIN_UPG="/system/home/admin"

    alias DEFAULT_DEBUG_OFF="local ldebug=1"
    alias defPS4="PS4=\${PS4_TPLT:-\"\$esc[\$((\$\$%6+31))m+(\$\$)\$FUNCTION:\$esc[0m \"}"
    alias IN_DEBUG="([[ -f $HOME_ADMIN_SYS/.debug.upgrade ]] || [[ -f $HOME_ADMIN_UPG/.debug.upgrade ]])"
    alias IN_NODEBUG="([[ -f $HOME_ADMIN_SYS/.nodebug.upgrade ]] || [[ -f $HOME_ADMIN_UPG/.nodebug.upgrade ]])"
#PR261951 jerryzh+
    if [[ -f $HOME_ADMIN_SYS/.debug.upgrade ]] || [[ -f $HOME_ADMIN_UPG/.debug.upgrade ]]; then
    	alias ENTER_FUNCTION="
    	[[ \${ldebug:-0} -eq 1 ]] && { local -; set +x; }
    	: \"---- ENTER_FUNCTION\"
    	: \"\$esc[4mfrom \${EF_function:-unknown()}\${default}\"
    	local OIFS=\$IFS IFS=\",\"
    	local EF_function=\"\$FUNCTION(\"\"\$*\"\")\"
    	IFS=\$OIFS
    	local _in_xflag=\${-//[^x]}
    	local PS4
    	defPS4
    	if IN_DEBUG; then
    	    local -; set -x
    	fi
    	if IN_NODEBUG; then
    	    local -; set +x
    	fi
    	(IFS=\",\"; : \"\${attr}START \$EF_function\${default}\")
    	: \"---- ENTER_FUNCTION END\"
    	"
# CRMS00312748 END
#   NOTE:
#       - RETURN macro in a short 'if' has to be like
#           [[ -f <file> ]] && { RETURN <ret_code>; }
#         The '{' and '; }' are VERY VERY important
#       - The ret_code has to be an immediate or a variable other than "$?".
#         "$?" is destroyed by the RETURN macro.
    	alias RETURN="
    	: \"\${attr}LEAVING \$EF_function\${default}\"
    	if [[ \"\$_in_xflag\" == \"\${-//[^x]}\" ]]; then
    	    [[ -n \"\$_in_xflag\" ]] && set -x || set +x
    	fi
    	return"
    else
        alias ENTER_FUNCTION=""
        alias RETURN="return" 
    fi
#PR261951 jerryzh-
}
define_aliases
##################################
# Change SYSTEM mount options    #
##################################
remount_system_part() {
    local FUNCTION=remount_system_part
    ENTER_FUNCTION 
    local mntpoint="/"

    # In case we are in UPGRADE partition, change the SYSTEM mount point
    [[ ! -z "$system_prefix" ]] && mntpoint=$system_prefix

    # before forcing the installation in async mode, check if it is authorized
    if [[ -f $async_locked ]]; then
        : "${attr}Installation in async mode is locked${default}"
        if [[ ! -z "$system_prefix" ]]; then
            echo "Installation in async mode is locked" >> $backuplog_file
        else
            logger -s -t upgrade -p local0.ERROR "Installation in async mode is locked"
        fi
        RETURN 1
    fi

    # remount the SYSTEM's partition in async
    cat $procmount | grep "SYSTEM" | grep "sync" > /dev/null
    if [[ $? == 0 ]]; then
        logger -s -t upgrade -p local0.WARNING "Remount SYSTEM partition in \"async\" mode"
        mount $mntpoint -o remount,rw,async
        if [[ $? != 0 ]]; then
	    if [[ ! -z "$system_prefix" ]]; then
                echo "Failed to remount SYSTEM in async mode" >> $backuplog_file
            else
                logger -s -t upgrade -p local0.ERROR "Failed to remount SYSTEM in async mode"
            fi
        fi
    fi

    RETURN 0
}

##################################
# Verify settings IP/protocol    #
##################################
set_ipprotocol_parameter() {
    local FUNCTION=set_ipprotocol_parameter
    ENTER_FUNCTION
    local ip=$1
    local protocol=$2
    # +++ crms00303424 +++
    local port=$3

    : "${attr}compute ip/protocol/port value from ${ip}/${protocol}/${port}${default}"
	# --- crms00303424 ---
    # If server's Name/IP is not available, this is an error
    [ -z "${ip}" ] && { RETURN 1; }

    # +++ fix crms00249742 +++
    # Check that $ip is not an url, if so it is an error
    is_url "$ip"
    [ $? -eq 1 ] && { RETURN 2; }
    # --- fix crms00249742 ---

    # Check if associated protocol is defined
    if [ -z "${protocol}" ]; then
        # +++ crms00345690 - force dummy protocol +++
        RESULT="noprot:${ip}"
        # --- crms00345690 ---        
    else
        RESULT="${protocol}:${ip}"
    fi

    # +++ crms00303424 +++
    # Verify that a port has been specified
    [ ! -z "${port}" ] && RESULT="${RESULT}:${port}"
    # --- crms00303424 ---

    RETURN 0
}

# +++ crms00265443
# Differentiate the 3 available dhcp mode: static vs dynamic and dynamic alcatel
# --- crms00265443
#    in static mode: both ip/protocol are needed
#    in dynamic mode: only the ip parameter is requested
select_needed_parameters() {
    local FUNCTION=select_needed_parameters
    ENTER_FUNCTION
    local dhcpmode=$1
    local retval=0

    : "${attr}==== dhcpmode: ${dhcpmode} ====${default}"
    case ${dhcpmode} in
        "Static")
	    # +++ crms00303424 +++
            set_ipprotocol_parameter "$2" "$3" "$4"
			# --- crms00303424 ---

            # +++ crms00259952
            [ $? -ne 0 ] && retval=1
            # --- crms00259952
            ;;
# +++ crms00265443
        "Dynamic"|"DynamicAlcatel")
# --- crms00265443
# FID411654
# IP/Protocol/Port are also required to support non standard https port in Dynamic mode
            set_ipprotocol_parameter "$2" "$3" "$4"
# FID411654 END
            [ $? -ne 0 ] && unset RESULT
            ;;
        *)
            echo "Unrecognized dhcp mode (${dhcpmode})"
            retval=2
            ;;
    esac

    RETURN ${retval}
}

# Retrieve the values of server's Name/IP and protocol used
# depending on the server
get_servers_parameters() {
    local FUNCTION=get_servers_parameters
    ENTER_FUNCTION
    local retval=1

    unset RESULT
    unset MAIN_PARAMETERS
    unset BACKUP_PARAMETERS

    # Associate them to get the final value
    : "${attr}Computing main value ip/protocol${default}"
	# +++ crms00303424 +++
# crms00417047 zimingxu+
    select_needed_parameters "${ENETCFG_DHCP_MODE}" "${ENETCFG_DM}" "${ENETCFG_DM_PROTOCOL}" "${ENETCFG_DM_PORT}"
# crms00417047 zimingxu-
	# --- crms00303424 ---
    retval=$?
    # +++ crms00259952
    [ ${retval} -ne 0 ] && logger -s -t upgrade -p local0.ERROR "main ip/protocol is invalid" && return ${retval}
    MAIN_PARAMETERS=${RESULT}
    unset RESULT

    : "${attr}Computing backup value ip/protocol${default}"
# crms00417047 zimingxu+
    [ -z "${ENETCFG_DM_BACKUP}" ] && logger -s -t upgrade -p local0.WARNING "No backup server found" && return 0
	# +++ crms00303424 +++
    select_needed_parameters "${ENETCFG_DHCP_MODE}" "${ENETCFG_DM_BACKUP}" "${ENETCFG_DM_BACKUP_PROTOCOL}" "${ENETCFG_DM_BACKUP_PORT}"
	# --- crms00303424 ---
# crms00417047 zimingxu-
    retval=$?
    [ ${retval} -eq 0 ] && BACKUP_PARAMETERS=${RESULT} && unset RESULT || logger -t upgrade -p local0.WARNING "backup ip/protocol is invalid"
    # --- crls00259952

    RETURN 0
}

# Check if given path is an url or not
is_url() {
    local FUNCTION=is_url
    ENTER_FUNCTION
    local url=$1

    # Parse the given parameter to retrieve the protocol used
    url=${url%%://*}
    if [ "${url}" == "http" ] || [ "${url}" == "https" ]; then
        RETURN 1
    fi

    RETURN 0
}

# Parse the backup file to fill the variables with the old upgrade servers' IP
retrieve_backup_ip() {
    local FUNCTION=retrieve_backup_ip
    ENTER_FUNCTION

    if [ ! -f $backup_server_file ]; then
        echo "No backup of upgrade servers information has been found"
        RETURN 1
    fi

    # +++ fix crms00249742 +++
    # File is available, reset all parameters before reading it
    # This is to prevent the use of wrong default parameters in case of missing ones
# crms00432759 zimingxu+
# crms00417047 zimingxu+
    unset ENETCFG_DM ENETCFG_DM_PROTOCOL ENETCFG_DM_BACKUP ENETCFG_DM_BACKUP_PROTOCOL ENETCFG_UPGRADE_FILE \
          ENETCFG_DM_PORT ENETCFG_DM_BACKUP_PORT ENETCFG_DM_URL ENETCFG_DM_BACKUP_URL 
# crms00417047 zimingxu-
# crms00432759 zimingxu-

    while read line; do
    case ${line%=*} in
# crms00417047 zimingxu+
        "ENETCFG_DM")
            ENETCFG_DM=${line#*=}
# crms00417047 zimingxu-
            ;;
# crms00432759 zimingxu+
        "ENETCFG_DM_URL")
            ENETCFG_DM_URL=${line#*=}
            ;;
# crms00432759 zimingxu-
        "ENETCFG_DM_PROTOCOL")
            ENETCFG_DM_PROTOCOL=${line#*=}
            ;;
# crms00417047 zimingxu+
        "ENETCFG_DM_BACKUP")
            ENETCFG_DM_BACKUP=${line#*=}
# crms00417047 zimingxu-
            ;;
# crms00432759 zimingxu+
        "ENETCFG_DM_BACKUP_URL")
            ENETCFG_DM_BACKUP_URL=${line#*=}
            ;;
# crms00432759 zimingxu-
        "ENETCFG_DM_BACKUP_PROTOCOL")
            ENETCFG_DM_BACKUP_PROTOCOL=${line#*=}
            ;;
        "ENETCFG_UPGRADE_FILE")
            ENETCFG_UPGRADE_FILE=${line#*=}
            ;;
        # +++ crms00303424 +++
        "ENETCFG_DM_PORT")
            ENETCFG_DM_PORT=${line#*=}
        ;;
        "ENETCFG_DM_BACKUP_PORT")
            ENETCFG_DM_BACKUP_PORT=${line#*=}
        ;;
        # --- crms00303424 ---
    esac
    done < $backup_server_file

# crms00432759 zimingxu+
    if [[ -z "$ENETCFG_DM" ]] && [[ ! -z "$ENETCFG_DM_URL" ]]; then
        ENETCFG_DM=$ENETCFG_DM_URL
    fi

    if [[ -z "$ENETCFG_DM_BACKUP" ]] && [[ ! -z "$ENETCFG_DM_BACKUP_URL" ]]; then
        ENETCFG_DM_BACKUP=$ENETCFG_DM_BACKUP_URL
    fi
# crms00432759 zimingxu-

    RETURN 0
}

# Split the URL into the PROTOCOL/IP/PATH components
split_old_upgrade_url () {
    local FUNCTION=split_old_upgrade_url
    ENTER_FUNCTION

# FID411654
    local oldurl=$1 dm_server
# FID411654 END

    : "Splitting old url ${oldurl}"

    # +++ fix crms00249742 +++
    if [ -z "${oldurl}" ]; then
        echo "No old upgrade URL found, exiting"
        RETURN 1
    fi
    # --- fix crms00249742 ---

    ENETCFG_DM_PROTOCOL=${oldurl%://*}
    oldurl=${oldurl##*://}
# FID411654
    dm_server=${oldurl%%/*}
    if test "X${dm_server%:*}" != "X"
    then
# crms00417047 zimingxu+
        ENETCFG_DM=${dm_server%:*}
# crms00417047 zimingxu-
        echo ${dm_server#*:} | grep -E ^[[:digit:]]\{1,5\}$ && ENETCFG_DM_PORT=${dm_server#*:} || ENETCFG_DM_PORT=
    fi
    ENETCFG_UPGRADE_FILE=${oldurl#${dm_server}*}                       
# FID411654 END    

    RETURN 0
}

# Check at least 2 entry parameters. If they are not valid, try to retrieve the old ones from a backup file
validate_parameters () {
    local FUNCTION=validate_parameters
    ENTER_FUNCTION

# crms00417047 zimingxu+
    # +++ crms00259952
    $STATUS
    if [ $? -ne 0 ]; then
        logger -s -t upgrade -p local0.ERROR "Upgrade is in progress, using stored ones"
# crms00417047 zimingxu-
        retrieve_backup_ip
        if [ $? -ne 0 ]; then
            # Last chance: test the old way
            logger -t upgrade -p local0.ERROR "Stored settings are invalid, trying older ENETCFG_UPGRADE setting"
            split_old_upgrade_url "$ENETCFG_UPGRADE"
            # +++ crms00258187
            if [ $? -ne 0 ]; then
                logger -s -t upgrade -p local0.ERROR "Failed to retrieve valid parameters"
                [ -z "$1" ] && cleanup 1
		# +++ crms00289400

                : "${attr}==== Set \"STEP=pre0\" ====${default}"
		echo "STEP=pre0" > $statusfile
		# --- crms00289400
                RETURN 1
            fi
            # --- crms00258187
        fi
    fi

    logger -s -t upgrade -p local0.NOTICE "Upgrade parameters are valid"
    # -- crms00259952
    RETURN 0
}

# +++ crms00290807 +++
# Parse the reboot flag to extract requested reboot partition and displayed text
get_reboot_data() {
    local FUNCTION=get_reboot_data
    ENTER_FUNCTION
    local tmpstr=

    # +++ crms00303412 +++
    if [ -z "$rebootconfig" ] || [ ! -f $rebootconfig ]; then
	RETURN 1
    fi
    tmpstr=$(cat $rebootconfig)
    # --- crms00303412 ---

    # split string
    REBOOT_PART=${tmpstr%%\#*}
    TXT_POPUP=${tmpstr##*\#}

    # make sure the string is not empty, if so add a default error
    [ -z "$TXT_POPUP" ] && TXT_POPUP="requested by Upgrade"

    RETURN 0
}

# Define a global reboot function that check if reboot is authorized
# log in the upgrade report file the current date
launch_reboot() {
    local FUNCTION=launch_reboot
    ENTER_FUNCTION

    if [ $# -eq 1 ]; then
	REBOOT_OWNER=$1
    else
	REBOOT_PART=$1
	TXT_POPUP=$2
	REBOOT_OWNER=$3
    fi

    : "${attr}Set next reboot to $REBOOT_PART${default}"
    $rebootset $REBOOT_PART
    RETVAL=$?
	
    if [ $RETVAL -eq 0 ]; then
        rm -f $rebootconfig
	write_to_report "\n\nRebooting on $REBOOT_PART"
	write_report_footer
	sync

	# Fix crms00242598
	# +++ crms00303412 +++
        : "${attr}=== Reboot --reason \"$TXT_POPUP\" --owner \"$REBOOT_OWNER\" ===${default}"
	$reboot --reason "$TXT_POPUP" --owner "$REBOOT_OWNER" $REBOOT_TYPE
	# --- crms00303412 ---
    fi

    RETURN $RETVAL
}
# --- crms00290807 ---

rmflag() {
    local FUNCTION=rmflag
    ENTER_FUNCTION

    [[ -e $1 ]] && rm -f $1

    RETURN
}

# Fix crms00239571: do not remove a log file created under upgrade working dir
cleanup () {
    local FUNCTION=cleanup
    ENTER_FUNCTION
    local full_cleanup=$1
    local list="upgrade_report*|serversip.bak|upgrade.log|.restore"

    # Add the profile.xml file and its checksum to the list of non removed files
    # +++ crms00273199
    # Add ${request_upgrade} flag to the list of non removed files
    if [ -z "${full_cleanup}" ] || [ ${full_cleanup} -ne 1 ];then
        list="${list}|profile.xml*|${request_upgrade}"
    fi
	# --- crms00273199

    # Find all files that do not match the request
    files=$(find $workdir/ ! -type d | grep -Ev "${list}")

    echo "Cleaning all upgrade flags"
    rm -f $files

    # Fix crms00239986
    for elem in $old_xmlkey_file $old_cert_file $async_locked; do
        rmfile ${elem}
    done

    # Remove old log file
    rm -f $(find $old_log_file* 2> /dev/null)

    RETURN 0
}

check_retry() {
    local FUNCTION=check_retry
    ENTER_FUNCTION

    if [ -e $retryflag ]; then
	echo "Step has been restarted but fails again, upgrade is rebooting"
	# +++ crms00258187: do a full cleanup
	cleanup 1
	# --- crms00258187
	# +++ crms00289400
        : "${attr}==== Set \"STEP=pre2\" ====${default}"
	echo "STEP=pre2" > $statusfile
	# --- crms00289400

	# Remove "retry" flag once checked
	rm -f ${retryflag}
    else
        touch $retryflag
    fi

    launch_reboot "upgrade"

    RETURN
}

update_backupdb_path() {
    local FUNCTION=update_backupdb_path
    ENTER_FUNCTION
    local ROOT=

    ROOT=`cat /proc/cmdline | sed 's/^.* root=.*:\(\w*\) .*/\1/'`
    # Check that the u-boot configuration file is available
    if [ "$ROOT" == "" ] ; then
        [[ ! -e $ubootfile ]] && echo "Nou-boot.cfg file found" && { RETURN 1; }

        while read line; do
	    case "$line" in root=*)
	        ROOT=${line#*=}
	    ;;
    	    esac
	done < $ubootfile
    fi

    if [ "$ROOT" != "" ] ; then
	$rebootset $ROOT
    fi

    [[ "$ROOT" == "UPGRADE" ]] && PREFIX=/system

    RETURN
}

start_update() {
    local FUNCTION=start_update
    ENTER_FUNCTION
    local STEP=$1 RET=0
    # +++ crms00259524
    local cmdline="" deployment_mode=""

    [[ ! -e $update ]] && echo "update tool has not been found !" && { RETURN 1; }

    if [[ $STEP -eq 1 ]] && [[ $# -eq 2 ]]; then
        deployment_mode=$2
    fi
    # --- crms00259524

    # Check whether SYSTEM partition is mounted with "sync" option
    remount_system_part

    # +++ crms00303412 +++
    # Remove reboot flag in case it exists
    : "${attr}Remove existing reboot flag(s)${default}"
    rbtfiles=$(find $workdir -name "reboot*" | grep -v "hard")
    [ ! -z "$rbtfiles" ] && rm -f $rbtfiles
    # --- crms00303412 ---
    # In caseno scenario has been found, it is an error
    if [[ ! -r $scenariofile ]]; then
	logger -s -t upgrade -p local0.ERR "No upgrade scenario found, this should not happen !"
	cleanup 
	RETURN 1
    fi

    # +++ Fix crms00246998
    if [[ $SET_RESTORE != 1 ]]; then
	if [[ $UPGD_PART == 0 ]]; then
	    get_servers_parameters
	    RET=$?
	fi

	if [[ $RET == 0 ]]; then
    	    if [ ! -z "${BACKUP_PARAMETERS}" ]; then
        	cmdline="-b ${BACKUP_PARAMETERS}"
	    fi

	    # +++ crms00259524
	    if [ ! -z "${deployment_mode}" ]; then
		cmdline="${cmdline} -d ${deployment_mode}"
	    fi
    	    # --- crms00259524

	    cmdline="${cmdline} -m ${MAIN_PARAMETERS} -u ${ENETCFG_UPGRADE_FILE}"
	else
            logger -s -t upgrade -p local0.ERR "Failed to retrieve servers' Name/IP, exiting"
	fi
    fi
	
    if [[ $RET == 0 ]]; then
	$update --step "$STEP" ${cmdline}
	RET=$?
    fi

    RETURN $RET
}

# +++ crms00289400
# Remove temporary rpm files
remove_rpm_tmp_files() {
    local FUNCTION=remove_rpm_tmp_files
    ENTER_FUNCTION
    local elem=
    local pattern=$1
    local ret=1

    # check if pattern exists
    [ -z "$pattern" ] && { RETURN 1; }

	# Remove files that match pattern
    # Find some temporary rpm due to copy errors
    for elem in $(find / -name "$pattern"); do
        echo "Removing rpm temporary file ${elem}"
        if [ -d ${elem} ]; then
            rm -rf ${elem}
            ret=$?
        else
            rm -f ${elem}
            ret=$?
        fi

        [ $ret -ne 0 ] && break
    done

    RETURN $ret
}

# In case of power failure, rpm temporary files can still waste some disk space
# this function find and remove them
find_rpm_temporary_files() {
    local FUNCTION=find_rpm_temporary_files
    ENTER_FUNCTION

    # Find some temporary rpm due to copy errors
    remove_rpm_tmp_files "*;*"
    [ $? -ne 0 ] && { RETURN 1; }

    # Find also all files marked as .rpmsave
    remove_rpm_tmp_files "*.rpmsave"
    [ $? -ne 0 ] && { RETURN 1; }
    
    # Find also all files marked as .rpmnew
    remove_rpm_tmp_files "*.rpmnew"
    [ $? -ne 0 ] && { RETURN 1; }
    
    RETURN 0
}
# --- crms00289400

##################################
# Upgrade post installation step #
##################################
upgrade_post_installation() {
    local FUNCTION=upgrade_post_installation
    ENTER_FUNCTION

    # Fix crms00219093: cleanup configuration files before rebooting
    if [[ ! -z "$dmconfig" ]]; then
# CRMS00386875
        $dmconfig "reparse"
# CRMS00386875 END
    else # in UPGRADE partition, no dmconfig
        rmfile "/config/dm/.cksum*"
    fi

    rmfile $backup_server_file

    # Check for rpm temporary files
    find_rpm_temporary_files

    RETURN
}

##################################
# Treat installation errors      #
##################################
analyze_returned_code() {
    local FUNCTION=analyze_returned_code
    ENTER_FUNCTION
    local returned_value=$1

    # Do a syncrhonization of the data
    sync

    # Treat the returned data
    case $returned_value in
        0|1)
            echo "$0 exited successfully"
        ;;
        2)
            echo "$0 exited successfully"
    #crms00390845 jerryzh+
            # upgrade successfully, remove backup restore pool
            rm -rf $backup_restoredir
    #crms00390845 jerryzh-
            # Run post installation treatment
            upgrade_post_installation
        ;;
    #crms00390845 jerryzh+
        13)
    #crms00390845 jerryzh-
            # Error while processing external datas... startup must end
            echo "$0 exited with an error in external components"

            # Clean some flags and some files in working direcotry
            rmfile $scenariofile
            upgrade_post_installation
        ;;
        *)
            # All others returned values are errors
            echo "$0 failed (error code $returned_value)"
            check_retry
        ;;
    esac
    # --- crms00290807 ---

    RETURN
}

write_report_header() {
    local FUNCTION=write_report_header
    ENTER_FUNCTION
    local step=$1
    local banner=$(printf "%0.120d" "0" | tr "0" "-")
    local part="SYSTEM"

    echo "$banner\nAction started at: $(date)\n\n" >> $reportfile
    [ $step -eq 2 ] && part="UPGRADE"

    echo "Running action in $part partition\n" >> $reportfile

    RETURN
}

write_to_report() {
    local FUNCTION=write_to_report
    ENTER_FUNCTION
    local msg=$1

    echo "$msg" >> $reportfile

    RETURN
}

write_report_footer() {
    local FUNCTION=write_report_footer
    ENTER_FUNCTION
    local banner=$(printf "%0.120d" "0" | tr "0" "-")

    echo "$banner\n\nAction ended at: $(date)\n$banner\n\n" >> $reportfile

    RETURN
}

write_end_report() {
    local FUNCTION=write_end_report
    ENTER_FUNCTION

    echo "$1"
    write_to_report "$1"
    write_report_footer

    RETURN
}

# +++ crms00310043 +++
compare_versions() {
    local FUNCTION=compare_versions
    ENTER_FUNCTION
    ret=0
	
    # Sanity check, if function is not called with all parameters
    # force component installation
    if [ $# -ne 2 ]; then
	ret=1
	RETURN
    fi

    # Then compare parameters
    [ $1 -eq $2 ] && { RETURN; }
    [ $1 -gt $2 ] && ret=-1 || ret=1

    RETURN
}

# Parse kernel command line to extract component version and
# compare it with its value in RPM database. In case they are different,
# remove component from it
compare_component_version() {
    local FUNCTION=compare_component_version
    ENTER_FUNCTION
    local kern_ver rpm_db tmp

    # keep "=x.x ....."
    tmp=${kern_cmdline##*${cmdline_name}=}
    # then remove " ....."
    kern_ver=${tmp%% *}

    # get data from rpm database
    : "${attr}Get data from RPM database${default}"
    rpm -q ${rpm_name} > /dev/null
    if [ $? -ne 0 ]; then
	logger -s -t upgrade -p local0.WARNING "rpm command failed"
	ret=0
	RETURN
    fi

    rpm_db=$(rpm -q ${rpm_name} --qf '%{VERSION}')

    # compare major version number
    : "${attr}Compare versions (${kern_ver} & ${rpm_db})${default}"
    compare_versions ${kern_ver%%.*} ${rpm_db%%.*}
    if [ $ret -eq 0 ]; then
	# compare minor version number
	compare_versions ${kern_ver##*.} ${rpm_db##*.}
    fi

    RETURN
}

# usage: verify_component cmdline_name rpm_name
# returns:
#	- 1 if rpm database component is greater
#	- -1 if kernel command line component is greater
#	- 0 if both match
verify_component() {
    local FUNCTION=verify_component
    ENTER_FUNCTION
    cmdline_name=$1
    rpm_name=$2

    # First check that requested option can be found on kernel command line
    : "${attr}Verify option $cmdline_name in kernel command line${default}"
    echo "$kern_cmdline" | grep $cmdline_name > /dev/null
    if [ $? -ne 0 ]; then
	logger -s -t upgrade -p local0.WARNING "$cmdline_name option not found in kernel command line"
	RETURN
    fi
	
    : "${attr}Option has been found, process the parsing${default}"
    compare_component_version
    if [ $ret -gt 0 ]; then
	logger -s -t upgrade -p local0.ERR "Component $rpm_name has not been flashed successfully, erasing it from RPM database"
	rpm -e --nodeps --allmatches $rpm_name
	[ $? -ne 0 ] && logger -s -t upgrade -p local0.ERR "Removing of component $rpm_name from RPM database failed"
    fi

    RETURN
}
# --- crms00310043 ---

get_application_manager_status() {
    local FUNCTION=get_application_manager_status
    ENTER_FUNCTION
    # Check for sanity flag: if present, skip this test
    [ -f ${sanityflag} ] && { RETURN 1; }

    # Check for application manager
    [ -f ${applicationManagerPid} ] && [ -d /proc/$(cat ${applicationManagerPid}) ] && { RETURN 1; }

    # No sanity flag and no application manager, clean all upgrade flags
    logger -s -t upgrade -p local0.ERROR "No application manager found !"

    RETURN 0
}

##################################
# Get upgrade process PID        #
##################################
child_pid_status() {
    local FUNCTION=child_pid_status
    ENTER_FUNCTION
    RETVAL=0

    if [[ -e $childpid ]]; then 
	CHILDPID=$(cat $childpid)
	RETVAL=1
	logger -t upgrade -p local0.WARNING "Process already running ($CHILDPID)"
    fi

    RETURN $RETVAL
}

##################################
# Manage pid of running process  #
##################################
save_pid() {
    local FUNCTOIN=save_pid
    ENTER_FUNCTION

    [[ ! -d $rundir ]] && mkdir -p $rundir
    echo "$$" > $childpid

    RETURN
}

remove_pid() {
    rmfile $childpid
}

##################################
# Get setting's value            #
##################################
get_setting_value() {
    local FUNCTION=get_setting_value
    ENTER_FUNCTION

    : "${attr}Load CURRENT settings' value for $setting${default}"
    local setting=$1
    local name=

    [ -z "$setting" ] && return 1

    # Introduce an indirection to use variable's content as a new one
    # which will contain its setting's value
    name=$(echo $setting)
    eval "${name}=$(CLISettings get CURRENT $setting 2>/dev/null)"
	
    RETURN 0
}

# +++ crms00326993 +++
rmfile() {
    local FUNCTION=rmfile
    ENTER_FUNCTION

    if [ ! -f $1 ]; then
	RETVAL=1
    else
	logger -s -t upgrade -p local0.NOTICE "rmfile: Removing file $1"
	rm -f $1
	RETVAL=$?
    fi

    [ $RETVAL -ne 0 ] && logger -s -t upgrade -p local0.WARNING "Failed to remove file $1"

    RETURN
}
# -- crms 00326993 ---

current_version() {
    local FUNCTION=current_version
    ENTER_FUNCTION
    local ret=1 version=${system_prefix:-}/$VERSIONFILE

    if [[ -e "$version" ]]; then
	source $version
	echo "current_version=$VersionSoft"
	ret=0
    fi

    RETURN $ret
}

next_version() {
    local FUNCTION=next_version
    ENTER_FUNCTION
    local ret=1

    if [[ -e "$scenariofile" ]]; then
	echo "next_version=$(awk 'BEGIN{RS=">[^[:graph:]]*<"}/version[[:blank:]]*>/{v=gensub(/.*>[^[:graph:]]*/,"","",$0);v=gensub(/[^[:graph:]]*<.*/,"","",v); print v; nextfile}' $scenariofile)"
	ret=0
    fi

    RETURN $ret
}
# +++++ crms00362595 +++++
#######
# is_unlocked function
# check if locked version of upgrade
# Return code:
#        0      lock version
#        1      unlock version
#######
is_unlocked() {
    local FUNCTION=is_unlocked
    ENTER_FUNCTION

    [[ -x $unlock ]] && [[ "$($unlock upgrade)" == "true" ]]

    RETURN $?
}

do_restore() {
    local list1 list2 ret go
    local FUNCTION=do_restore
    ENTER_FUNCTION
	
    # verify retore pool before 
    list1="$(sort $restoredir/.base_rpmlist)"
    list2="$(cd $restoredir && { ls -A *.rpm | sort; })"
    : "${attr}======== list1 ========${default}"
    : "${attr}$list1${default}"
    : "${attr}======== list2 ========${default}"
    : "${attr}$list2${default}"
    : "${attr}=======================${default}"
    if [[ "$list1" != "$list2" ]]; then
	#restore pool is KO
	logger -t upgrade -p local0.EMERG "Restore pool is not complete ! Restore refused."
	go=0
    else
	go=1
    fi
    if [[ $go -ne 0 ]]; then
	# do restore
	touch $restoreflag
	launch_reboot "UPGRADE" "restore start" "upgrade"
	ret=0
    else
	ret=1
    fi

    RETURN $ret
}

# ----- crms00362595 -----

#crms00390845 jerryzh+
backup_restore() {
    local FUNCTION=backup_restore restore_rpm_size other_rpm_size remain_space ret=1
    ENTER_FUNCTION

    [[ -d $backup_restoredir ]] && rm -rf $backup_restoredir

    if [[ -d $restoredir ]]; then
        restore_rpm_size=`du $restoredir | awk '{print $1}'`
        remain_space=`df | grep "ubi0:DATA" | awk '{print $4}'`

        # 1024KB is redundancy
        restore_rpm_size=`expr $restore_rpm_size + 1024`
        if [[ $remain_space -ge $restore_rpm_size ]]; then
            logger -s -t upgrade -p local0.NOTICE "remain space is enough, backup restore rpms"
            cp -rf $restoredir $backup_restoredir
            ret=$?
        else 
            other_rpm_size=`du $otherrpmdir | awk '{print $1}'`
            remain_space=`expr $remain_space + $other_rpm_size`
            if [[ $remain_space -ge $restore_rpm_size ]]; then
                logger -s -t upgrade -p local0.NOTICE "remain space is not enough, remove data on /data/rpm"
                rm -rf $otherrpmdir
                cp -rf $restoredir $backup_restoredir
	        ret=$?
            else
                logger -s -t upgrade -p local0.WARNING "remain space is not enough, please manually release data on /data"
            fi
        fi
        if [[ $ret -eq 0 ]]; then
            logger -s -t upgrade -p local0.NOTICE "success to backup restore pool, clear end of upgrade flag"
        else
            logger -s -t upgrade -p local0.ERR "failed to backup restore rpms"
        fi  
    fi

    RETURN $ret
}
#crms00390845 jerryzh-
