#!/bin/sh

#######################################################################
# CHANGELOG
#===========
#
# 2013/02/27    PR#442574    Jerry ZHOU <jerry.zhou@alcatel-lucent.com>
#               file's creation
# 2013/03/07    PR#442574    Terry Zhao
#				curl tftp_blkszie config, add progress showing.
#######################################################################

##################################
# Global variables               #
##################################
LANPBX_DOWNLOAD_FAILED=1
LANPBX_PARSE_FAILED=2
KEYXML_DOWNLOAD_FAILED=3
GPGKEY_IMPORT_FAILED=4
NOEIMAGE_DOWNLOAD_FAILED=5
NOEIMAGE_VERIFY_FAILED=6
NOERPM_SIG_VERIFY_FAILED=7
NOE_VERSION_CHECK_FAILED=8
OTHER_ERR=10

KEYXML_FILE="key.xml"
LANPBX_FILE="lanpbx.cfg"

NOE_RPM_PATH="/data/noe_upgrade"
NOE_RPM="noe_image.rpm"
NOE_BIN="noe8082v"
NOE_VERSION="noe_version"

TFTP1_ADDR=""
TFTP2_ADDR=""
PORT=""

MAIN_SERVER=""
BACK_SERVER=""
DL_PORT=""
NOE_RESTRICT_VERSION=45060   #4.50.60  4*10000+50*100+60

source /etc/init.d/rc.config


usage() {
    local script_name=`basename $0`

    echo "================================================"
    echo "USAGE:                                          "
    echo "  $script_name tftp1_addr <tftp2_addr> port     "
    echo "================================================"
}


show_progress()
{
    while true
    do
        echo '-'"\b\c"; 
        sleep 1
        echo '\\'"\b\c"
        sleep 1
        echo "|\b\c"; 
        sleep 1
        echo "/\b\c";$sl 
        sleep 1
    done  
}


dowait() {
    local ret=0 nostderr=0

    while [[ "${1##--}" != "$1" ]]; do
        case "$1" in
            --nostderr) nostderr=1
            ;;
        esac
        shift
    done
    dowaitcmd="$1" shift

    if [[ $nostderr -eq 1 ]]; then
        $dowaitcmd 2>/dev/null "$@" &
    else
        $dowaitcmd "$@" &
    fi

    dowaitpid=$!
    echo $dowaitpid > /var/run/dowait.pid

    show_progress &
    progress_pid=$!
    wait $dowaitpid
    ret=$?
    [[ -d /proc/$dowaitpid ]] && kill $dowaitpid
    dowaitpid=-1

    [[ -d /proc/$progress_pid ]] && kill $progress_pid
    progress_pid=-1

    rm /var/run/dowait.pid -f

    return $ret
}


config_server(){
    local ret=1
# reset config
    MAIN_SERVER=""
    BACK_SERVER=""
    DL_PORT=""

    while [[ $# -gt 0 ]]; do
    case $1 in
        --m=*) 
            MAIN_SERVER=${1##*=}
            ret=0
            ;;
        --b=*) BACK_SERVER=${1##*=};;
        --p=*) DL_PORT=${1##*=};;
    esac
    shift
    done

    return $ret
}


swap_server(){
    local ret

    if [[ ! -z $MAIN_SERVER ]] && [[ ! -z $BACK_SERVER ]]; then
        tmp=$MAIN_SERVER
        MAIN_SERVER=$BACK_SERVER
        BACK_SERVER=$tmp
        ret=0
    else
        ret=1
    fi

    return $ret
}


tftp_download() {
    local protol="tftp" \
    file=$1 \
    url="" \
    ret=1

    if [[ ! -z "$MAIN_SERVER" ]]; then
        url="${protol:+$protol://}${MAIN_SERVER:+$MAIN_SERVER}${DL_PORT:+:$DL_PORT}"
    fi

    if [[ ! -z $url ]]; then
        dowait --nostderr curl --tftp-blksize 1428 --connect-timeout 5 --retry 0 -o /tmp/$file --silent --output /dev/null $url/$file
        ret=$?
    fi

    return $ret
}


download() {
    local ret=1 swap dlretry \
    file=$1

    dlretry=3
    swap=0
    while [[ $dlretry -ge 0 ]] && [[ $ret -ne 0 ]]; do
        tftp_download $file
        ret=$?

        if [[ $ret -ne 0 ]]; then
            echo "download from $MAIN_SERVER failed !!!"
            echo "try to download from $BACK_SERVER ..."
            if swap_server; then
                if [[ $swap -eq 0 ]]; then
                    swap=1
                else
                    swap=0
                fi
            else
                swap=0
            fi
        fi
        
        if [[ $swap -eq 0 ]]; then
            dlretry=$((dlretry-1))
            dowait sleep $((RANDOM%5))
        fi
    done

    return $ret
}


parse_lanpbx() {
    local ret=0 lanpbx=$1

#crms00442067 tingh
    eval `sed -n '/^TYPE=/p' $1 | sed -n '1p'`
    [[ $? -ne 0 ]] && echo "Error: lanpbx.cfg is bad format" && ret=1        

    return $ret
}


import_gpgkey() {
    local ret cmd keyxml_file=$1

    cmd="sigupdate -pubkey $keyxml_file"
    eval "$cmd"
    ret=$?

    return $ret
}


check_secure_status() {
    local lock_status ret=1\
    UNLOCK="/usr/sbin/.unlock"

    if [[ -x $UNLOCK ]]; then
        lock_status=`$UNLOCK upgrade`
        if [[ "$lock_status" == "true" ]]; then
            ret=0
        fi
    fi

    return $ret
}


decap_noe_binary() {
    local ret cmd in_file=$1 out_file=$2

    if [ ! -f $in_file ]; then
        echo "decap_noe_binary: $in_file missing!!!"
        return -1
    fi

    cmd="dd if=$in_file of=$out_file bs=256 skip=1"
    dowait eval "$cmd"
    ret=$?

    return $ret
}


verify_noe_rpm() {
    local ret=1 sig_status verify_status
    rpm_file=$1

    if [ ! -f $rpm_file ]; then
        echo "verify_noe_rpm: $rpm_file missing!!!"
        return -1
    fi

    sig_status=`rpm -qpi $rpm_file | grep Signature | awk '{print $3}'`

    if [[ "$sig_status" != "(none)" ]]; then
        verify_status=`rpm -K $rpm_file | awk '{print $5}'`

        [[ "$verify_status" == "OK" ]] && ret=0
    fi

    return $ret
}

check_noe_version() {
    local ret in_file=$1 out_file=$2 ver1 ver2 ver3 ver HW_TP
   
    HW_TP=`/usr/bin/CLISettings get CURRENT FAB_HW_TOUCHPANEL 2>/dev/null`
    if [ `echo $HW_TP | grep '^touchscreen_CY8CTMG120'` ]; then
        echo "UNICORN touchpanel"
        return 0
    else
        echo "TRULY touchpanel"
    fi

    if [ ! -f $in_file ]; then
        echo "check_noe_version: $in_file missing!!!"
        return 1
    fi

    cmd="dd if=$in_file of=$out_file bs=1 count=7 skip=27"
    dowait eval "$cmd"
    ret=$?

    if [[ $ret -ne 0 ]]; then
        echo "check_noe_version failed"
        return 1
    fi

    echo "NOE version `cat $out_file`"
    
    ver1=`cut -d . -f 1 $out_file`
    ver2=`cut -d . -f 2 $out_file`
    ver3=`cut -d . -f 3 $out_file`
    ver=`expr 10000 \* $ver1 + 100 \* $ver2 + $ver3`

    if [ $ver -lt $NOE_RESTRICT_VERSION ]; then
        echo "version not supported"
        return 1
    fi

    return 0
}

#########################################################
## Main management function
#########################################################
progress_management(){
    local ret swap dlretry msg insecure

## prepare job before process start
##---------------------------------
    [[ -d /data/rpm ]] && rm -fr /data/rpm/*
    [[ ! -d $NOE_RPM_PATH ]] && mkdir -p $NOE_RPM_PATH
    [[ -f $NOE_RPM_PATH/$NOE_RPM ]] && rm -f $NOE_RPM_PATH/$NOE_RPM
    [[ -f $NOE_RPM_PATH/$NOE_BIN ]] && rm -f $NOE_RPM_PATH/$NOE_BIN
    [[ -f /tmp/$NOE_RPM ]] && rm -f /tmp/$NOE_RPM
    [[ -f /tmp/$NOE_BIN ]] && rm -f /tmp/$NOE_BIN

##show step 3
    /usr/sbin/jpg2fb $step3Splash

## config lanpbx.cfg download server
##----------------------------------
    cmd="config_server ${TFTP1_ADDR:+--m=$TFTP1_ADDR} ${TFTP2_ADDR:+--b=$TFTP2_ADDR} ${PORT:+--p=$PORT}"
    eval "$cmd"
    ret=$?

    if [[ $ret -ne 0 ]]; then
        msg="config lanpbx.cfg download server failed"
        echo "Error: $msg"
        return $OTHER_ERR
    fi

## start download lanpbx.cfg
##-------------------------
    msg="start download lanpbx.cfg ..."
    echo $msg
    displaymsg $msg
    download $LANPBX_FILE
    ret=$?

## check lanpbx.cfg download result
##---------------------------------
    if [[ $ret -ne 0 ]]; then
        msg="lanpbx.cfg download failed"
        echo "Error: $msg"
        displaymsg $msg
        return $LANPBX_DOWNLOAD_FAILED
    else
        msg="lanpbx.cfg download OK"
        echo $msg
        displaymsg $msg
    fi

## start parse lanpbx
##-------------------
    displaymsg "parse the lanpbx.cfg ..."
    parse_lanpbx /tmp/$LANPBX_FILE
    ret=$?

## remove the temporary lanpbx file
##----------------------------------
    rm /tmp/$LANPBX_FILE -f        

    if [[ $ret -ne 0 ]]; then
        msg="unrecognize lanpbx.cfg"
        echo "Error: $msg"
        displaymsg $msg
        return $LANPBX_PARSE_FAILED
    else
        msg="lanpbx.cfg parse OK"
        echo $msg
        displaymsg $msg
    fi

##show step 4
##--------------------------------
    /usr/sbin/jpg2fb $step4Splash

## config key.xml and noe binary download server
##----------------------------------------------
    if [[ ! -z $BIN_DOWNLOAD ]]; then
        if [[ a"$BIN_DOWNLOAD" == a"0.0.0.0" || a"$BIN_DOWNLOAD" == a"255.255.255.255" ]]; then
            msg="ILLegal BIN_DOWNLOAD Addr $BIN_DOWNLOAD !!!"
            echo $msg
            return $OTHER_ERR
        fi

        cmd="config_server ${BIN_DOWNLOAD:+--m=$BIN_DOWNLOAD} ${PORT_DOWNLOAD:+--p=$PORT_DOWNLOAD}"
        eval "$cmd"
        else
        if [[ a"$IP_DOWNLOAD" == a"0.0.0.0" || a"$IP_DOWNLOAD" == a"255.255.255.255" || a"$IP_DOWNLOAD" == a"" ]]; then
            msg="ILLegal IP_DOWNLOAD Addr $IP_DOWNLOAD !!!"
            echo $msg
            return $OTHER_ERR
        fi

        cmd="config_server ${IP_DOWNLOAD:+--m=$IP_DOWNLOAD} ${IP_DOWNLOAD_RD:+--b=$IP_DOWNLOAD_RD} ${PORT_DOWNLOAD:+--p=$PORT_DOWNLOAD}"
        eval "$cmd"
    fi
    ret=$?

    if [[ $ret -ne 0 ]]; then
        msg="config NOE binary download server failed"
        echo "Error: $msg"
        return $NOEIMAGE_DOWNLOAD_FAILED
    fi

## check secure status
##--------------------
    check_secure_status
    insecure=$?

#############################
## check secure status. 
## lock version, need download key.xml and verify signature
##---------------------------------------------------------
    if [[ $insecure -ne 0 ]]; then
## start download key.xml
##-----------------------
        msg="start download key.xml ..."
        echo $msg
        download $KEYXML_FILE
        ret=$?

## import GPG key to rpm database
##-------------------------------
        if [[ $ret -ne 0 ]]; then
            msg="key.xml download failed"
            echo "Error: $msg"
            displaymsg $msg
            return $KEYXML_DOWNLOAD_FAILED
        else
            msg="key.xml download OK"
            echo $msg
        fi

        import_gpgkey /tmp/$KEYXML_FILE
        ret=$?

## remove the temporary key.xml
##-----------------------------
        rm /tmp/$KEYXML_FILE -f        

        if [[ $ret -ne 0 ]]; then
            msg="GPG key import failed"
            echo "Error: $msg"
            displaymsg $msg
            return $GPGKEY_IMPORT_FAILED
        fi

    else
        msg="insecure mode, no key.xml download"
        echo "Info: $msg"
    fi

## start download noe binary
##-------------------------
    msg="start download noe image ..."
    echo $msg
    displaymsg $msg
    download $NOE_BIN
    ret=$?

    if [[ $ret -ne 0 ]]; then
        msg="noe binary download failed"
        echo $msg
        displaymsg $msg
        return $NOEIMAGE_DOWNLOAD_FAILED        
        else
        msg="noe binary download OK"
        echo $msg
        displaymsg $msg        
    fi

## check NOE version: Truly touchpanel does not support lower versions
## --------------------------------------
    echo "checking NOE version..."
    check_noe_version /tmp/$NOE_BIN /tmp/$NOE_VERSION
    ret=$?
    rm -f /tmp/$NOE_VERSION
    
   if [[ $ret -ne 0 ]]; then
        msg="NOE version not supported"
        echo $msg
        return $NOE_VERSION_CHECK_FAILED
    else
        msg="check NOE version OK"
        echo $msg
    fi
## remove the DHS3 header from noe binary
##---------------------------------------
    echo "moving $NOE_BIN to /data partition ..."
    dowait mv /tmp/$NOE_BIN $NOE_RPM_PATH
    decap_noe_binary $NOE_RPM_PATH/$NOE_BIN /tmp/$NOE_RPM
    ret=$?

## remove the noe binary to save space
##------------------------------------
    rm $NOE_RPM_PATH/$NOE_BIN -f

    if [[ $ret -ne 0 ]]; then
        msg="noe binary decapsulate failed"
        echo $msg
        return $NOEIMAGE_VERIFY_FAILED
    else
        msg="noe binary decapsulate OK"
        echo $msg
    fi

    if [[ $insecure -ne 0 ]]; then
## verify noe RPM signature
##-------------------------        
        verify_noe_rpm /tmp/$NOE_RPM
        ret=$?

        if [[ $ret -ne 0 ]]; then
            msg="noe rpm signature verify failed"
            echo $msg
            displaymsg $msg
            return $NOERPM_SIG_VERIFY_FAILED
        else
            msg="noe rpm signature verify OK"
            echo $msg
            displaymsg $msg
        fi
    fi

## move noe rpm to destination
##----------------------------    
    echo "backup $NOE_RPM to /data partition ..."

    if [ ! -f /tmp/$NOE_RPM ]; then
        echo "progress_management: /tmp/$NOE_RPM missing!!!"
        return -1
    fi

    dowait mv /tmp/$NOE_RPM $NOE_RPM_PATH/$NOE_RPM

    return $ret
}




#################
## MAIN
#################
case $# in
    2)
        TFTP1_ADDR=$1
        PORT=$2
    ;;
    3)
        TFTP1_ADDR=$1
        TFTP2_ADDR=$2
        PORT=$3
    ;;
    *)
        usage
        exit $OTHER_ERR
    ;;
esac

if [[ a"$TFTP1_ADDR" == a"0.0.0.0"  ||  a"$TFTP1_ADDR" == a"255.255.255.255"  ||  a"$TFTP1_ADDR" == a""  ||  a"$TFTP2_ADDR" == a"0.0.0.0"  ||  a"$TFTP2_ADDR" == a"255.255.255.255" ]]; then
	msg="Illgeal TFTP Addr !!"
	echo $msg
	displaymsg $msg
	exit $OTHER_ERR
fi

progress_management
ret=$?

exit $ret
