/**
 * @class webapp.audiosetting.controlBase
 * @namespace webapp.audiosetting
 */
dojo.provide("webapp.audiosetting.controlBase");
dojo.declare("webapp.audiosetting.controlBase",[ICTouchAPI.webApplication],
{
	constructor : function() {
		ICTouchAPI.settingServices.subscribeToSetting(this, "AudioRingLock", this.onAudioRingLockChanged);
	},

	/**
	 * Callback on the AudioRingLock setting
	 * @param {Object} audioRingLockSetting The AudioRingLock setting
	 */
	onAudioRingLockChanged : function(audioRingLockSetting){
		//Save the new value
		this.data.isRingVolumeLocked = audioRingLockSetting.jsValue;
		audioRingLockSetting=null;
	}
});
/**
 * @class webapp.audiosetting.control
 * @namespace webapp.audiosetting
 * @extends webapp.audiosetting.controlBase
 */
dojo.require("webapp.audiosetting.controlBase");
dojo.provide("webapp.audiosetting");
dojo.declare("webapp.audiosetting",webapp.audiosetting.controlBase,{

	/**
	 * Toaster view
	 * @property
	 * @type View
	 */
	objToasterUI		: null,

	/**
	 * Id of the previous view in the toaster
	 * @property
	 * @type String
	 */
	oldToasterContentId : "",

	constructor : function(){
		ICTouchAPI.skinServices.linkWebappsStyles("webapp.audiosetting", "webapp.audiosetting", true, true, true);
		ICTouchAPI.eventServices.subscribeToEvent(this, "audio_key_event", this.onAudioKeyEvent);
		ICTouchAPI.eventServices.subscribeToEvent(this, "voicemode_trans", this.onVoicemodeTrans);

		ICTouchAPI.eventServices.subscribeToEvent(this, "presentationLoaded", this._presentationLoaded);
        ICTouchAPI.eventServices.subscribeToEvent(this, "CallStartedEvent", this.closeAudioToaster);

		ICTouchAPI.eventServices.subscribeToEvent(this, "send_sense_key_event", this.handsFreeCallback);

		ICTouchAPI.settingServices.subscribeToSetting(this, "AudioRingVolume", this.onVolumeValueChanged);
		ICTouchAPI.settingServices.subscribeToSetting(this, "AudioHandsetVolume", this.onVolumeValueChanged);
		ICTouchAPI.settingServices.subscribeToSetting(this, "AudioHeadsetVolume", this.onVolumeValueChanged);
		ICTouchAPI.settingServices.subscribeToSetting(this, "AudioSpeakerPhoneVolume", this.onVolumeValueChanged);
		ICTouchAPI.settingServices.subscribeToSetting(this, "MPSpeakerPhoneVolume", this.onMediaVolumeChanged);
		ICTouchAPI.settingServices.subscribeToSetting(this, "MPHandsetVolume", this.onMediaVolumeChanged);
		ICTouchAPI.settingServices.subscribeToSetting(this, "MPHeadsetVolume", this.onMediaVolumeChanged);
		ICTouchAPI.settingServices.subscribeToSetting(this, "MPPreviewVolume", this.onMediaVolumeChanged);
		//ICTouchAPI.settingServices.subscribeToSetting(this, "MPAlarmVolume", this.onMediaVolumeChanged);

		ICTouchAPI.settingServices.subscribeToSetting(this, "AudioRingLock", this.onAudioRingLockChanged);

		ICTouchAPI.eventServices.subscribeToEvent(this, "SENSKEY_HOME", this.eventPhysicalKeyHome);

	},

	/**
	 * Event when the presentation layer is loaded to build the toaster
	 */
	_presentationLoaded:function(){
		ICTouchAPI.eventServices.unsubscribeToEvent(this,"presentationLoaded");
		this.buildAll();

		// AVAD
		ICTouchAPI.settingServices.subscribeToSetting(this, "AudioVoiceSignalLimitation", this.onAudioVoiceSignalLimitation);
		ICTouchAPI.settingServices.getSetting("AudioVoiceSignalLimitation", this, this.onAudioVoiceSignalLimitation);
		ICTouchAPI.settingServices.subscribeToSetting(this, "AudioVoiceSignalLimitationState", this.onAudioVoiceSignalLimitationState);
		ICTouchAPI.settingServices.getSetting("AudioVoiceSignalLimitationState", this, this.onAudioVoiceSignalLimitationState);
	},


	/**
	 * Build the Audiotoaster and preload it
	 */
	buildAll : function(){
		this.objToasterUI = new webapp.audiosetting.getAudioToaster({},null);
		this.objToasterUI.setVoicemodeText(this.data.getCurrentVoicemodeText());
		this.objToasterUI.setMaxSlider(this.data.getCurrentVoicemodeMaxVolume());
		ICTouchAPI.toasterServices.preloadContent(this.objToasterUI);
	},

	/**
	 * Callback of setting AudioVoiceSignalLimitation. Determines if the button audioQuality must be displayed.
	 * @params {object} objAVSLSetting setting AudioVoiceSignalLimitation
	 */
	onAudioVoiceSignalLimitation : function(objAVSLSetting) {
		if(objAVSLSetting){
			this.data.setBoolQualityAudioDisplay(objAVSLSetting.jsValue);
		}
	},

	/**
	 * Callback of setting AudioVoiceSignalLimitationState. Determines if the audioQuality is activated.
	 * @params {object} objAVSLSetting setting AudioVoiceSignalLimitationState
	 */
	onAudioVoiceSignalLimitationState : function(objAVSLSSetting) {
		if(objAVSLSSetting){
			this.data.setBoolQualityAudio(objAVSLSSetting.jsValue);
		}
	},

	/**
	 * Callback for buttons
	 */
	handleButtons : function (buttonId)	{
		this.resetTimer(true);
		if(buttonId == this.data.MUTE_BTN){
			//On click on the mute button, switch mute activation
			ICTouchAPI.APIServices.IctAudio.setMute({
				params:[!this.data.muteStatus],
				context:this
			});
		}
		else if(buttonId == this.data.HIDE_BTN){
			this.closeAudioToaster();
		}
		else if(buttonId == this.data.QUALITY_AUDIO_BTN){
			var bool_audioQuality = this.data.getBoolQualityAudio();
			//Audio quality is currently enabled
			if(bool_audioQuality){
				ICTouchAPI.APIServices.IctAudio.enableAutoVolume({
					params:[0],
					context:this
				});
				this.data.setBoolQualityAudio(false);
				this.objToasterUI.qualityAudioButtonUi.setStatusIcon('notif-off');
			}else{//Audio quality is currently disabled
				ICTouchAPI.APIServices.IctAudio.enableAutoVolume({
					params:[1],
					context:this
				});

				this.data.setBoolQualityAudio(true);
				this.objToasterUI.qualityAudioButtonUi.setStatusIcon('notif-on');
			}
		}
		else{
			var voicemode = this.data.getcorrespondingVoicemodeToButton(buttonId);
			if (voicemode){
				ICTouchAPI.APIServices.IctAudio.setVoicemode({
					params:[voicemode],
					context:this
				});
			}
		}
		buttonId = null;
	},

	eventPhysicalKeyHome : function(objEvent) {
		if(objEvent && objEvent.value !== 0){
			this.oldToasterContentId = "";
		}
	},

	handsFreeCallback : function(){
		var objArguments = ICTouchAPI.tools.getEventArguments(arguments);
		if (objArguments.senseKey ==1){
			this.openAudioToaster();
		}
	},

	updateAutoVolumeButton : function() {
		if(this.objToasterUI){
			// Show or hide the button
			if(this.data.getBoolQualityAudioDisplay()){
				this.objToasterUI.qualityAudioButtonUi.show();
			} else {
				this.objToasterUI.qualityAudioButtonUi.hide();
			}
			
			// Manage the status icon
			if(this.data.getBoolQualityAudio()){
				this.objToasterUI.qualityAudioButtonUi.setStatusIcon(this.data.STATUS_ON_ICON);
			} else {
				this.objToasterUI.qualityAudioButtonUi.setStatusIcon(this.data.STATUS_OFF_ICON);
			}
		}
	},

	/**
	 * Open the audio toaster and start the timer to hide it after 10 seconds
	 */
	openAudioToaster: function() {
		if(this.objToasterUI){
			this.objToasterUI.volumeSliderUI.refreshDisplay();
		    webapp.audiosetting.data.boolShownToaster = true;
			if (this.oldToasterContentId == "" && ICTouchAPI.toasterServices.getContentDisplayedId() != this.objToasterUI.id){
				this.oldToasterContentId = ICTouchAPI.toasterServices.getContentDisplayedId();
			}
			ICTouchAPI.toasterServices.showContent({
				"toasterContent" : this.objToasterUI
			});
			this.startTimer(true);
		}
	},

	/**
	 * Close the audio toaster and stop the timer
	 */
	closeAudioToaster : function(){
		ICTouchAPI.APIServices.ICTGate.notifyEventPresentation({
			params:["presentationHideAudiotoaster"]
		});
		ICTouchAPI.toasterServices.hideContent(webapp.audiosetting.objToasterUI);
		webapp.audiosetting.resetTimer(false);
		if (webapp.audiosetting.oldToasterContentId != "") {
			ICTouchAPI.toasterServices.showContent({
				toasterContent:{
					id:webapp.audiosetting.oldToasterContentId
				}
			})
		}
		webapp.audiosetting.oldToasterContentId = "";
		if(webapp.audiosetting.data.boolShownToaster) {
		    webapp.audiosetting.data.boolShownToaster = false;
		}
	},

	/**
	 * Function to stop the timer
	 * @param {Boolean} restart boolean to know if we have to restart again the timer
	 */
	resetTimer : function(restart){
		if( this.timerHandler ) {
			clearTimeout(this.timerHandler);
			this.timerHandler = null;
		}
		if(restart){
			this.startTimer();
		}
		restart= null;
	},

	/**
	 * Start the timer
	 */
	startTimer : function() {
		if (this.timerHandler){
			this.resetTimer(false);
		}
		this.timerHandler = setTimeout(this.closeAudioToaster, 10000);
	},


	/**
	 * Callback of audio_key_event event
	 */
	onAudioKeyEvent: function() {
		var objArguments = ICTouchAPI.tools.getEventArguments(arguments);
		var showToaster = false;
		//If the event is about mute
		if (objArguments.audioevt === this.data.EVT_MUTE_ON && this.data.getMuteStatus() !== true){
			this.muteValueChanged(true);
			showToaster = true;
		}
		//If the mute value has changed to false
		else if(objArguments.audioevt === this.data.EVT_MUTE_OFF && this.data.getMuteStatus() !== false){
			this.muteValueChanged(false);
			showToaster = true;
		}
        else if (objArguments.audioevt === this.data.EVT_VOL_MINUS || objArguments.audioevt === this.data.EVT_VOL_PLUS ){
			showToaster = true;
		}
        else if (objArguments.keyevent === this.data.EVT_KEY_MINUS || objArguments.keyevent === this.data.EVT_KEY_PLUS ){
			showToaster = true;
		}
		//open the toaster if needed
		if (showToaster){
			this.openAudioToaster();
		}

	},

	/**
	 * callback when we receive the voicemode_trans event
	 */
	onVoicemodeTrans : function(){
		//No events are currently correctly sent by ictaudio module for the moment
		//Force to get the new values by calling ictaudio methods
		//-------------------TEMPORARY FIX
		this.data.getCurrentVoicemode();
		this.data.getVoicemodesStatus();
	//-------------------TO BE REMOVED

	/*var objArguments = ICTouchAPI.tools.getEventArguments(arguments);
		//Voicemode has changed
		this.voicemodeChanged(objArguments.old_state,objArguments.new_state);
		*/
	},

	/**
	 * Callback for the volume settings
	 * @param {Object} volumeSetting The volumeSetting object
	 */
	onVolumeValueChanged : function(volumeSetting){
		//Save the new value
		this.data.arrVolumeValues[volumeSetting.id] = volumeSetting.jsValue;
		//If if it the current volume displayed
		if (this.objToasterUI && this.data.isCurrentVoicemodeLinkedToSetting(volumeSetting.id) && this.objToasterUI.getVolumeValue() != volumeSetting.jsValue){
			//Change the sldier value
			this.objToasterUI.setSliderValue(volumeSetting.jsValue);
		}
		volumeSetting=null;
	},

	/**
	 * Callback for the media volume settings
	 * @param {Object} volumeSetting The mediaVolumeSetting object
	 */
	onMediaVolumeChanged : function(volumeSetting){
		volumeSetting.jsValue = parseFloat(volumeSetting.jsValue*5);
		this.onVolumeValueChanged(volumeSetting);
		volumeSetting = null;
	},

	/**
	 * Callback on the AudioRingLock setting
	 * @param {Object} lockSetting The AudioRingLock setting
	 */
	onAudioRingLockChanged : function(lockSetting){
		this.data.gotRingLockValue(lockSetting.jsValue);
		lockSetting = null;
	},
	/**
	 * Whan the mute value has changed
	 * @param {Boolean} value new mute value
	 */
	muteValueChanged : function(value){
		//Save the new value
		this.data.muteStatus = value;

		//Update the button with the status icon
		//And turn on/off the led
		if (value){
			this.data.objMuteButton.strStatusIcon = this.data.STATUS_ON_ICON;
			ICTouchAPI.ledServices.activate('mute', 'ON');
		}
		else{
			this.data.objMuteButton.strStatusIcon = this.data.STATUS_OFF_ICON;
			ICTouchAPI.ledServices.desactivate('mute', 'ON');
		}

		//If the toaster was already built
		if (this.objToasterUI){
			//Update the appbutton ui element
			this.objToasterUI.changeMuteStatusIcon(this.data.objMuteButton.strStatusIcon);
		}
		value=null;
	},

	/**
	 * When the current voicemode has changed
	 * @param {Int} oldVoicemode Old Voicemode
	 * @param {Int} newVoicemode New Voicemode
	 */
	voicemodeChanged: function (oldVoicemode,newVoicemode){
		//If voicemode has not really changed
		//This can happen with an ictaudio bug
		if (oldVoicemode == newVoicemode){
			return;
		}
		//reset the timer to avoid closing toaster if not needed
		//It will be restarted if needed at the end of this function
		this.resetTimer(true);

		//Desactivate the old voicemode
		if (oldVoicemode && this.data.arrObjVoicemodeButtons[oldVoicemode]){
			this.data.arrObjVoicemodeButtons[oldVoicemode].strStatusIcon = this.data.STATUS_OFF_ICON;
			if (this.objToasterUI){
				this.objToasterUI.changeVoicemodeStatusIcon(oldVoicemode,this.data.STATUS_OFF_ICON);
			}
		}

		//Turn off the handsfree led if the old voicemode was this type
		if(oldVoicemode == this.data.VOICEMODE_INTERNAL_HANDFREE || oldVoicemode == this.data.VOICEMODE_INTERNAL_GROUP_LISTENING ||
			oldVoicemode == this.data.VOICEMODE_EXTERNAL_HANDFREE || oldVoicemode == this.data.VOICEMODE_EXTERNAL_GROUP_LISTENING) {
			ICTouchAPI.ledServices.desactivate('handsFree', 'ON');
		}


		//Set the new Voicemode
		this.data.currentVoicemode = newVoicemode;

		//Activate the new voicemode
		if (this.data.arrObjVoicemodeButtons[newVoicemode]){
			this.data.arrObjVoicemodeButtons[newVoicemode].strStatusIcon = this.data.STATUS_ON_ICON;
			//Get the max value if we have'nt already it
			if (!this.data.arrMaxVolumes[newVoicemode]){
				this.askForMaxVolume(newVoicemode);
			}
			if (this.objToasterUI){
				//Get the max value to update the slider
				if (this.data.arrMaxVolumes[newVoicemode]){
					this.objToasterUI.setMaxSlider(this.data.arrMaxVolumes[newVoicemode]);
				}
				this.objToasterUI.setSliderSeekable(this.data.isVoicemodeVolumeChangeable(newVoicemode));
				this.objToasterUI.changeVoicemodeStatusIcon(newVoicemode,this.data.STATUS_ON_ICON);
				this.objToasterUI.setSliderValue(this.data.getCurrentVoicemodeVolume());
				this.objToasterUI.setVoicemodeText(this.data.getCurrentVoicemodeText());

			}
		}

		//Turn on the handsfree led if the new voicemode is this type
		if(newVoicemode == this.data.VOICEMODE_INTERNAL_HANDFREE || newVoicemode == this.data.VOICEMODE_INTERNAL_GROUP_LISTENING ||
			newVoicemode == this.data.VOICEMODE_EXTERNAL_HANDFREE || newVoicemode == this.data.VOICEMODE_EXTERNAL_GROUP_LISTENING) {
			ICTouchAPI.ledServices.activate('handsFree', 'ON');
		}

		//Some exception case to display the toaster
		if (this.objToasterUI && oldVoicemode &&
			!(oldVoicemode == this.data.VOICEMODE_IDLE && newVoicemode == this.data.VOICEMODE_RING )
			&& oldVoicemode != this.data.VOICEMODE_RING
			&& oldVoicemode != this.data.VOICEMODE_IDLE
			&& oldVoicemode != this.data.VOICEMODE_STREAMING
			&& newVoicemode != this.data.VOICEMODE_IDLE){
			this.openAudioToaster();
		}
		else{
			//If the toaster is always displayed when it must be hidden, close it
			// (example case, intiate a call, change the headset/handset volume, when the call is automatically cancelled, it switch to idle mode)
			if(this.objToasterUI && ICTouchAPI.toasterServices.getContentDisplayedId() == this.objToasterUI.id){
				this.closeAudioToaster();
			}
		}
		oldVoicemode=null;
		newVoicemode=null;

	},

	/**
	 * When the status of a voicemode changed
	 * @param {Int} voicemode voicemode
	 * @param {Int} status new status
	 */
	voicemodeStatusChanged : function(voicemode,status){
		//Save the new status
		this.data.arrObjVoicemodeButtons[voicemode].voicemodeStatus = status;
		//If the toaster was already built, update the appButton Ui according to the new status
		if (this.objToasterUI){
			if (status == this.data.STATUS_UNAVAILABLE){
				this.objToasterUI.hideButton(voicemode);
			}
			else if (status == this.data.STATUS_DISABLE){
				this.objToasterUI.greyButton(voicemode);
			}
			else if (status == this.data.STATUS_ENABLE){
				this.objToasterUI.showButton(voicemode);
			}
		}
		voicemode=null;
		status=null;
	},

	/*
	 *Call this function when the seekable value has changed
	 *@param {Boolean} newValue new value
	 */
	seekableChanged : function(newValue){
		if (this.objToasterUI){
			this.objToasterUI.setSliderSeekable(newValue);
		}
		newValue=null;
	},

	/**
	 * Get the max volume value
	 * @param {Int} voicemode Voicemode for which we want the max volume
	 */
	askForMaxVolume : function(voicemode){
		if (voicemode == this.data.VOICEMODE_IDLE){
			//Simulate the voicemode ring because the IDLE one has no volume
			voicemode = this.data.VOICEMODE_RING;
		}
		this.data.getMaxVolume(voicemode);
		voicemode=null;
	},

	/**
	 * When the max volume has changed
	 * @param {Int} mode voicemode
	 * @param {Int} maxVol new max volume
	 */
	maxVolumeChanged : function(mode, maxVol){
		if (mode == this.data.currentVoicemode && this.objToasterUI){
			//Change the slider max value
			this.objToasterUI.setMaxSlider(maxVol);
			//update again the value after having chnage the max value
			this.objToasterUI.setSliderValue(this.data.getCurrentVoicemodeVolume());
		}
		mode=null;
		maxVol=null;
	},

	/**
	 * Callback when we have change the slider value
	 */
	handleVolumeSliderEndChange :function(){
		//Get the current slider value
		var value = this.objToasterUI.getSliderValue();
		//get the current setting
		var setting = this.data.getCurrentVoicemodeSetting();
		//Change the value if it is the streaming one (other module setting)
		if (setting == this.data.getVoicemodeSetting(this.data.VOICEMODE_STREAMING)){
			value = value/5;
		}
		ICTouchAPI.settingServices.setSettingValue(setting, value);
		this.resetTimer(true);
	},

	/**
	 * Callback when we begin the change the slider value
	 */
	handleVolumeSliderBeginChange :function(){
		this.resetTimer(false);
	}


});

webapp.audiosetting=new webapp.audiosetting();
/**
 * @class webapp.audiosetting.dataBase
 * @namespace webapp.audiosetting
 */
dojo.provide("webapp.audiosetting.dataBase");
dojo.declare("webapp.audiosetting.dataBase", null, {

	/**
	 * Is the ring volume is locked
	 * @property
	 * @type Boolean
	 */
	isRingVolumeLocked  : false,
	/**
	 * Current volume for each voicemode
	 * @property
	 * @type IntArray
	 */
	arrVolumeValues		: [],

	constructor : function() {
		ICTouchAPI.settingServices.getSettingValue("AudioRingLock", this, this.gotRingLockValue);

	},

	/**
	 * Get the mute value from audio module
	 */
	getMuteValue : function (){
		ICTouchAPI.APIServices.IctAudio.getMute({
			context:this,
			callback: this.gotMuteValue
			});
	},

	/**
	 * Get the current voicemode value from audio module
	 */
	getCurrentVoicemode : function(){
		ICTouchAPI.APIServices.IctAudio.getCurrentVoicemode({
			context:this,
			callback: this.gotCurrentVoicemode
			});
	},

	/**
	 * Get the voicemode status from audio module
	 */
	getVoicemodesStatus : function(){
		ICTouchAPI.APIServices.IctAudio.getVoicemodeStatus({
			context:this,
			callback: this.gotVoicemodesStatus
			});
	},



	getMaxVolume : function(voicemode){
		ICTouchAPI.APIServices.IctAudio.getVoicemodeMaxVol({
			params :[voicemode],
			context:this,
			callback:this.gotVoicemodeMaxVolume,
			callbackParams:voicemode
		});
		voicemode = null;
	}

});
/**
 * @class webapp.audiosetting.data
 * @namespace webapp.audiosetting
 * @extends webapp.audiosetting.dataBase
 */
dojo.require("webapp.audiosetting.dataBase");
dojo.provide("webapp.audiosetting.data");
dojo.declare("webapp.audiosetting.data", webapp.audiosetting.dataBase, {
	/**
	 * Controller
	 * @property
	 * @type Controller
	 */
	control  : webapp.audiosetting,

	//VOICEMODE STATES
	VOICEMODE_IDLE						: 0,
	VOICEMODE_HANDSET					: 1,
	VOICEMODE_HEADSET					: 2,
	VOICEMODE_INTERNAL_HANDFREE			: 3,
	VOICEMODE_INTERNAL_GROUP_LISTENING	: 4,
	VOICEMODE_EXTERNAL_HANDFREE			: 5,
	VOICEMODE_EXTERNAL_GROUP_LISTENING	: 6,
	VOICEMODE_RING						: 7,
	VOICEMODE_STREAMING					: 8,

	//AUDIOEVENT TYPES
	EVT_NOTHING			: 0,
	EVT_OFFHOOK			: 1,
	EVT_ONHOOK			: 2,
	EVT_VOL_PLUS		: 3,
	EVT_VOL_MINUS		: 4,
	EVT_MUTE_ON			: 5,
	EVT_MUTE_OFF		: 6,
	EVT_ROUTING_CHANGED	: 7,
	EVT_KEY_PLUS	: 17,
	EVT_KEY_MINUS	: 18,

	//VOICEMODE STATUS
	STATUS_NEVER			:-1,
	STATUS_DISABLE			:0,
	STATUS_UNAVAILABLE		:2,
	STATUS_ENABLE			:1,

	//Buttons ID
	QUALITY_AUDIO_BTN		: "QUALITY_AUDIO_BTN",
	MUTE_BTN				: "MUTE_BTN",
	HIDE_BTN				: "HIDE_BTN",
	HANDSET_BTN				: "HANDSET_BTN",
	HEADSET_BTN				: "HEADSET_BTN",
	EXTERNAL_HANDSFREE_BTN  : "EXTERNAL_HANDSFREE_BTN",
	INTERNAL_HANDSFREE_BTN  : "INTERNAL_HANDSFREE_BTN",
	INTERNAL_GROUP_BTN		: "INTERNAL_GROUP_BTN",
	EXTERNAL_GROUP_BTN		: "EXTERNAL_GROUP_BTN",
	IDLE_BTN				: "IDLE_BTN",
	STREAMING_BTN			: "STREAMING_BTN",
	RING_BTN				: "RING_BTN",

	// Status Icons for appButtons
	STATUS_ON_ICON			: 'notif-on',
	STATUS_OFF_ICON			: 'notif-off',

	//Arrays to save datas
	/**
	 * Max Volume for each voicemode
	 * @property
	 * @type IntArray
	 */
	arrMaxVolumes					: [], //Maximum volume for each Voicemode
	/**
	 * Label for each voicemode
	 * @property
	 * @type StringArray
	 */
	arrVoicemodeLabels				: [], //Labels for each Voicemode
	/**
	 * ObjButton for each voicemode
	 * @property
	 * @type ObjectArray
	 */
	arrObjVoicemodeButtons			: [], //Button model for each voicemode
	/**
	 * Setting name for each voicemode
	 * @property
	 * @type StringArray
	 */
	arrVoicemodeSettings			: [], //Volume Settings used for each voicemode
	/**
	 * Secondary setting name for some voicemode
	 * @property
	 * @type StringArray
	 */
	arrSecondaryVoicemodeSettings	: [], //array used if a voicemode is linked to multiple settings
	/**
	 * Current volume for each voicemode
	 * @property
	 * @type IntArray
	 */
	arrVolumeValues					: [], //Volume Values for each Setting

	/**
	 * Curretn volume for the streaming mode
	 * @property
	 * @type Float
	 */
	streamingVolume			: 0,

	/**
	 * Button Object for the mute button
	 * @property
	 * @type Object
	 */
	objMuteButton			: {}, //Button model for the mute


	/**
	 * Button Object for the quality audio button
	 * @property
	 * @type Object
	 */
	objQualityAudioButton : null,

	/**
	 * Current voicemode
	 * @property
	 * @type Int
	 */
	currentVoicemode		: null, //Current voicemode

	/**
	 * Curretn Mute status
	 * @property
	 * @type Boolean
	 */
	muteStatus				: false, //Current mute status

	/**
	 * Notify we want to stop current media play 
	 * @property
	 * @type Boolean
	 */
	boolShownToaster		: false,

	/**
	 * Is audio quality (AVAD/AGC) activated ?
	 * @property
	 * @type Boolean
	 */
	boolQualityAudio		: false,

	/**
	 * Do we display audio quality (AVAD/AGC) button ?
	 * @property
	 * @type Boolean
	 */
	boolQualityAudioDisplay	: false,

	_arrAppBarSkeleton		: [],
	//STATUS
	APPBAR_CLOSE			: 0,
	APPBAR_AUDIOOPTIONS		: 1,
	APPBAR_AUDIOVOICEMODE	: 2,

	constructor : function(){
		var func = {
			context:this.control,
			func:this.control.handleButtons
		};

		this._arrAppBarSkeleton[this.APPBAR_CLOSE] = [this.HIDE_BTN];
		this._arrAppBarSkeleton[this.APPBAR_AUDIOOPTIONS] = [this.MUTE_BTN,this.QUALITY_AUDIO_BTN];
		this._arrAppBarSkeleton[this.APPBAR_AUDIOVOICEMODE] = [this.IDLE_BTN,this.HANDSET_BTN,this.HEADSET_BTN,this.INTERNAL_HANDSFREE_BTN,this.INTERNAL_GROUP_BTN,this.EXTERNAL_HANDSFREE_BTN,this.EXTERNAL_GROUP_BTN,this.RING_BTN,this.STREAMING_BTN];

		//Define the MuteButton
		this.objMuteButton = {
			strButtonName: this.MUTE_BTN,
			strButtonLabel: _("Mute", "webapp.audiosetting"),
			strButtonIcon: 'audiosetting-mute',
			strStatusIcon: this.STATUS_OFF_ICON,
			callback: func
		};

		//Define the Quality Audio button
		this.objQualityAudioButton = {
			strButtonName: this.QUALITY_AUDIO_BTN,
			strButtonLabel: "Auto",
			strButtonIcon: 'audiosetting-quality-audio',
			strStatusIcon: this.STATUS_OFF_ICON,
			callback: func
		};

		this.objHideButton = {
			strButtonName: this.HIDE_BTN,
			strButtonLabel: _("Hide", "ICTouchAPI"),
			strButtonIcon: 'generic-down',
			callback: func
		};

		//Define the mode buttons
		this.arrObjVoicemodeButtons[this.VOICEMODE_IDLE] = {
			strButtonName: this.IDLE_BTN,
			strButtonLabel: _("Idle", "webapp.audiosetting"),
			intVoicemode : this.VOICEMODE_IDLE,
			strButtonIcon: 'audiosetting-handset',
			strStatusIcon: this.STATUS_OFF_ICON,
			voicemodeStatus:this.STATUS_NEVER,
			isVolumeChangeable:this.isRingVolumeLocked,
			callback: func
		};
		this.arrObjVoicemodeButtons[this.VOICEMODE_HANDSET] = {
			strButtonName: this.HANDSET_BTN,
			strButtonLabel: _("Handset", "webapp.audiosetting"),
			intVoicemode : this.VOICEMODE_HANDSET,
			strButtonIcon: 'audiosetting-handset',
			strStatusIcon: this.STATUS_OFF_ICON,
			voicemodeStatus:this.STATUS_AVAILABLE,
			isVolumeChangeable:true,
			callback: func
		};
		this.arrObjVoicemodeButtons[this.VOICEMODE_HEADSET] = {
			strButtonName: this.HEADSET_BTN,
			strButtonLabel: _("Headset", "webapp.audiosetting"),
			intVoicemode : this.VOICEMODE_HEADSET,
			strButtonIcon: 'audiosetting-headset',
			strStatusIcon: this.STATUS_OFF_ICON,
			voicemodeStatus:this.STATUS_AVAILABLE,
			isVolumeChangeable:true,
			callback: func
		};
		this.arrObjVoicemodeButtons[this.VOICEMODE_INTERNAL_HANDFREE] = {
			strButtonName: this.INTERNAL_HANDSFREE_BTN,
			strButtonLabel: _("Internal Handsfree", "webapp.audiosetting"),
			intVoicemode : this.VOICEMODE_INTERNAL_HANDFREE,
			strButtonIcon: 'audiosetting-handsfree',
			strStatusIcon: this.STATUS_OFF_ICON,
			voicemodeStatus:this.STATUS_AVAILABLE,
			isVolumeChangeable:true,
			callback: func
		};
		this.arrObjVoicemodeButtons[this.VOICEMODE_INTERNAL_GROUP_LISTENING] = {
			strButtonName: this.INTERNAL_GROUP_BTN,
			strButtonLabel: _("Internal Group Listening", "webapp.audiosetting"),
			intVoicemode : this.VOICEMODE_INTERNAL_GROUP_LISTENING,
			strButtonIcon: 'audiosetting-group-listening',
			strStatusIcon: this.STATUS_OFF_ICON,
			voicemodeStatus:this.STATUS_AVAILABLE,
			isVolumeChangeable:true,
			callback: func
		};
		this.arrObjVoicemodeButtons[this.VOICEMODE_EXTERNAL_HANDFREE] = {
			strButtonName: this.EXTERNAL_HANDSFREE_BTN,
			strButtonLabel: _("External Handsfree", "webapp.audiosetting"),
			intVoicemode : this.VOICEMODE_EXTERNAL_HANDFREE,
			strButtonIcon: 'audiosetting-bluetooth-handsfree',
			strStatusIcon: this.STATUS_OFF_ICON,
			voicemodeStatus:this.STATUS_AVAILABLE,
			isVolumeChangeable:true,
			callback: func
		};
		this.arrObjVoicemodeButtons[this.VOICEMODE_EXTERNAL_GROUP_LISTENING] = {
			strButtonName: this.EXTERNAL_GROUP_BTN,
			strButtonLabel: _("External Group Listening", "webapp.audiosetting"),
			intVoicemode : this.VOICEMODE_EXTERNAL_GROUP_LISTENING,
			strButtonIcon: 'audiosetting-externalaudio',
			strStatusIcon: this.STATUS_OFF_ICON,
			voicemodeStatus:this.STATUS_AVAILABLE,
			isVolumeChangeable:true,
			callback: func
		};
		this.arrObjVoicemodeButtons[this.VOICEMODE_RING] = {
			strButtonName: this.RING_BTN,
			strButtonLabel: _("Ring", "webapp.audiosetting"),
			intVoicemode : this.VOICEMODE_RING,
			strButtonIcon: 'audiosetting-externalaudio',
			strStatusIcon: this.STATUS_OFF_ICON,
			voicemodeStatus:this.STATUS_NEVER,
			isVolumeChangeable:this.isRingVolumeLocked,
			callback: func
		};
		this.arrObjVoicemodeButtons[this.VOICEMODE_STREAMING] = {
			strButtonName: this.STREAMING_BTN,
			strButtonLabel: _("Streaming", "webapp.audiosetting"),
			intVoicemode : this.VOICEMODE_STREAMING,
			strButtonIcon: 'audiosetting-externalaudio',
			strStatusIcon: this.STATUS_OFF_ICON,
			voicemodeStatus:this.STATUS_NEVER,
			isVolumeChangeable:true,
			callback: func
		};

		//Define the labels mode
		this.arrVoicemodeLabels[this.VOICEMODE_IDLE] = "Ring Volume";
		this.arrVoicemodeLabels[this.VOICEMODE_HANDSET] = "Handset Volume";
		this.arrVoicemodeLabels[this.VOICEMODE_HEADSET] = "Headset Volume";
		this.arrVoicemodeLabels[this.VOICEMODE_INTERNAL_HANDFREE] = "Speaker Volume";
		this.arrVoicemodeLabels[this.VOICEMODE_INTERNAL_GROUP_LISTENING] = "Speaker Volume";
		this.arrVoicemodeLabels[this.VOICEMODE_EXTERNAL_HANDFREE] = "Speaker Volume";
		this.arrVoicemodeLabels[this.VOICEMODE_EXTERNAL_GROUP_LISTENING] = "Speaker Volume";
		this.arrVoicemodeLabels[this.VOICEMODE_RING] = "Ring Volume";
		this.arrVoicemodeLabels[this.VOICEMODE_STREAMING] = "Streaming Volume";

		//Define the settings used for each mode
		this.arrVoicemodeSettings[this.VOICEMODE_IDLE] = "AudioRingVolume";
		this.arrVoicemodeSettings[this.VOICEMODE_HANDSET] = "AudioHandsetVolume";
		this.arrVoicemodeSettings[this.VOICEMODE_HEADSET] = "AudioHeadsetVolume";
		this.arrVoicemodeSettings[this.VOICEMODE_INTERNAL_HANDFREE] = "AudioSpeakerPhoneVolume";
		this.arrVoicemodeSettings[this.VOICEMODE_INTERNAL_GROUP_LISTENING] = "AudioSpeakerPhoneVolume";
		this.arrVoicemodeSettings[this.VOICEMODE_EXTERNAL_HANDFREE] = "AudioSpeakerPhoneVolume";
		this.arrVoicemodeSettings[this.VOICEMODE_EXTERNAL_GROUP_LISTENING] = "AudioSpeakerPhoneVolume";
		this.arrVoicemodeSettings[this.VOICEMODE_RING] = "AudioRingVolume";
		this.arrVoicemodeSettings[this.VOICEMODE_STREAMING] = "MPSpeakerPhoneVolume";

		//Define the other settings used for the Streaming voicemode
		this.arrSecondaryVoicemodeSettings[this.VOICEMODE_STREAMING] = [];
		this.arrSecondaryVoicemodeSettings[this.VOICEMODE_STREAMING].push("MPHandsetVolume");
		this.arrSecondaryVoicemodeSettings[this.VOICEMODE_STREAMING].push("MPHeadsetVolume");
		this.arrSecondaryVoicemodeSettings[this.VOICEMODE_STREAMING].push("MPAlarmVolume");
		this.arrSecondaryVoicemodeSettings[this.VOICEMODE_STREAMING].push("MPPreviewVolume");


		//The streaming mode is not managed by audio module, so we have to set the value by hand
		this.arrMaxVolumes[this.VOICEMODE_STREAMING] = 7;


		//Get the curretn mute Value
		this.getMuteValue();

		//Get the current voicemode status
		this.getVoicemodesStatus();

		//Get the Current Voicemode
		this.getCurrentVoicemode();

		//Get the current volume values
		this.getVolumesValues();
	},

	/**
	 * Callback while we get the mute value
	 * @param {Boolean} value new mute value
	 */
	gotMuteValue : function (value){
		//Inform controller
		this.control.muteValueChanged(value);
		value=null;
	},

	/**
	 * Callback when we get the current voicemode
	 * @param {Int} intVoicemode the new current voicemode
	 */
	gotCurrentVoicemode : function (intVoicemode){
		//Inform controller
		this.control.voicemodeChanged(this.currentVoicemode,intVoicemode);
		intVoicemode=null;
	},

	/**
	 * Callback when we get the voicemode status
	 * @param {IntArray} arrStatus status of all voicemodes
	 */
	gotVoicemodesStatus : function(arrStatus){

		//Simulate the samùe behaviour as 01.018.0
		//Will be deleted when the ictaudio module will be right


		for (var i=0;i<arrStatus.length; i++){
			if (this.arrObjVoicemodeButtons[i].voicemodeStatus != this.STATUS_NEVER && this.arrObjVoicemodeButtons[i].voicemodeStatus != arrStatus[i]){
				//Inform controller for each voicemode
				this.control.voicemodeStatusChanged(i, arrStatus[i]);
			}
		}
		arrStatus=null;
	},

	/**
	 * Get the current volume values for each setting
	 */
	getVolumesValues : function(){
		//For each defined settings
		for(var i=0;i<this.arrVoicemodeSettings.length;i++){
			//If we haven't already try to get the value'
			if (typeof (this.arrVolumeValues[this.arrVoicemodeSettings[i]]) == "undefined"){
				//fill the array with a false value to inform we have begin to ask the setting value
				this.arrVolumeValues[this.arrVoicemodeSettings[i]] = -1;
				//Get the setting Value
				ICTouchAPI.settingServices.getSetting(this.arrVoicemodeSettings[i], this, this.gotVolumeValue);
			}
		}
	},

	/**
	 * Callback when we get the volume setting value
	 * @param {Object} objSetting the volume setting
	 */
	gotVolumeValue : function(objSetting){
		//Save the value
		this.arrVolumeValues[objSetting.id] = objSetting.jsValue;
		objSetting=null;
	},

	/**
	 * Callback of AudioRingLock setting
	 * @param {Boolean} new AudioRingLock value
	 */
	gotRingLockValue :function(value) {
		this.isRingVolumeLocked = value;
		this.updateVoicemodeChangeable(!value);
		value=null;
	},

	/**
	 * Update the volumeChangeable attribute of the concerned voicemodes
	 * @param {Boolean} value new changeable value
	 */
	updateVoicemodeChangeable : function (value){
		for(var i=0; i<this.arrVoicemodeSettings.length;i++){
			//Find the concerned voicemode using the AudioRingVolume setting
			if (this.arrVoicemodeSettings[i] == "AudioRingVolume"){
				this.arrObjVoicemodeButtons[i].isVolumeChangeable = value;
				if (i == this.currentVoicemode){
					this.control.seekableChanged(value);
				}
			}
		}
		value=null;
	},

	/**
	 * Callback when we get the max volume value of a voicemode
	 * @param {Int} maxVol the max volume
	 * @param {Int} mode The corresponding voicemode
	 */
	gotVoicemodeMaxVolume : function (maxVol, mode){
		this.arrMaxVolumes[mode] = maxVol;
		this.control.maxVolumeChanged(mode, maxVol);
		//Same max value for RING and IDLE
		if (mode == this.VOICEMODE_RING){
			this.gotVoicemodeMaxVolume(maxVol,this.VOICEMODE_IDLE);
		}
		maxVol=null;
		mode=null;
	},

	/**
	 * get the models for the voicemode buttons
	 * @return {ObjectArray} The voicemode Buttons
	 */
	getArrVoicemodeButtons : function (){
		return this.arrObjVoicemodeButtons;
	},

	/**
	 * Get the model for the mute button
	 * @return {Object} The mute button
	 */
	getMuteObjButton : function(){
		return this.objMuteButton;
	},


	getHideObjButton : function(){
		return this.objHideButton;
	},


	/**
	 * Verify if the given setting is related to the current voicemode
	 * @param {String} settingName the setting to test
	 * @return {Boolean} isCurrentVoicemodeLinkedToGivenSetting
	 */
	isCurrentVoicemodeLinkedToSetting : function(settingName){
		//If the setting is the primary one of the voicemode
		if (this.arrVoicemodeSettings[this.currentVoicemode] == settingName){
			settingName=null;
			return true;
		}
		//find if the setting if one of the secondary of the voicemode
		if (this.arrSecondaryVoicemodeSettings[this.currentVoicemode]){
			for (var i=0; i < this.arrSecondaryVoicemodeSettings[this.currentVoicemode].length ; i++){
				if (this.arrSecondaryVoicemodeSettings[this.currentVoicemode][i] == settingName){
					settingName=null;
					return true;
				}
			}
		}
		settingName=null;
		return false;
	},

	/**
	 * Get the volume of the curent voicemode
	 * @return {Int} Volume of the current voicemode
	 */
	getCurrentVoicemodeVolume : function(){
		return this.arrVolumeValues[this.arrVoicemodeSettings[this.currentVoicemode]];
	},

	/**
	 * Get the Max volume of the curent voicemode
	 * @return {Int} Max Volume of the current voicemode
	 */
	getCurrentVoicemodeMaxVolume : function(){
		return this.arrMaxVolumes[this.currentVoicemode];
	},

	/**
	 * Get the label for the current voicemode
	 * @return {String} the Label of the voicemode
	 */
	getCurrentVoicemodeText : function(){
		return this.arrVoicemodeLabels[this.currentVoicemode];
	},

	/**
	 * By giving the strButtonName, it return the corresponding voicemode
	 * @param {String} buttonId id of the button
	 * @return {Int} Voicemode of the button
	 */
	getcorrespondingVoicemodeToButton : function(buttonId){
		for(var i=0;i<this.arrObjVoicemodeButtons.length;i++){
			if (this.arrObjVoicemodeButtons[i].strButtonName == buttonId){
				buttonId=null;
				return this.arrObjVoicemodeButtons[i].intVoicemode;
			}
		}
		buttonId=null;
		return null;
	},

	/**
	 * Return the setting used by the current voicemode
	 * @return {String} The setting of the current voicemode
	 */
	getCurrentVoicemodeSetting : function(){
		return this.arrVoicemodeSettings[this.currentVoicemode];
	},

	/**
	 * Get the setting used for the following voicemode
	 * @param {Int} voicemode The voicemode
	 * @return {String} The setting of the given voicemode
	 */
	getVoicemodeSetting : function(voicemode){
		return this.arrVoicemodeSettings[voicemode];
		voicemode=null;
	},

	/**
	 * Verify if the volume is changeable for the following voicemode
	 * @param {Int} voicemode The voicemode
	 * @return {Boolean} Is the volume changeable
	 */
	isVoicemodeVolumeChangeable : function(voicemode){
		return this.arrObjVoicemodeButtons[voicemode].isVolumeChangeable;
		voicemode=null;
	},

	/**
	 * Get the actual mute status
	 * @return {Boolean} Mute status
	 */
	getMuteStatus : function(){
		return this.muteStatus;
	},

	/**
	 * Function to know if the current volume is the max one
	 * @return {Boolean} isCurrentVoicemodeVolumeMax
	 */
	isCurrentVoicemodeVolumeMax : function(){
		return this.arrVolumeValues[this.arrVoicemodeSettings[this.currentVoicemode]] === this.arrMaxVolumes[this.currentVoicemode];
	},

	/**
	 * Function to know if the current volume is the min one
	 * @return {Boolean} isCurrentVoicemodeVolumeMin
	 */
	isCurrentVoicemodeVolumeMin : function(){
		return this.arrVolumeValues[this.arrVoicemodeSettings[this.currentVoicemode]] === 0;
	},


	getObjQualityAudioButton : function(){
		this.objQualityAudioButton.strStatusIcon = (this.getBoolQualityAudio()) ? this.STATUS_ON_ICON : this.STATUS_OFF_ICON;
		return this.objQualityAudioButton;
	},

	getBoolQualityAudio : function(){
		return this.boolQualityAudio;
	},

	setBoolQualityAudio : function(_bool){
		if(this.boolQualityAudio != _bool){
			this.boolQualityAudio = _bool;
			this.control.updateAutoVolumeButton();
		}
	},

	getBoolQualityAudioDisplay : function(){
		return this.boolQualityAudioDisplay;
	},

	setBoolQualityAudioDisplay : function(_bool){
		if(this.boolQualityAudioDisplay != _bool){
			this.boolQualityAudioDisplay = _bool;
			this.control.updateAutoVolumeButton();
		}
	}

});

webapp.audiosetting.data = new webapp.audiosetting.data();/**
 * @class webapp.audiosetting.getAudioToasterBase
 * @namespace webapp.audiosetting
 */
dojo.provide("webapp.audiosetting.getAudioToasterBase");
dojo.declare("webapp.audiosetting.getAudioToasterBase",
	[ICTouchAPI.webWidget, dojox.dtl._Templated],
	{
	}
	);
/**
 * @class webapp.audiosetting.getAudioToaster
 * @namespace webapp.audiosetting
 * @extends webapp.audiosetting.getAudioToasterBase
 */
dojo.require("webapp.audiosetting.getAudioToasterBase");
dojo.provide("webapp.audiosetting.getAudioToaster");
dojo.declare("webapp.audiosetting.getAudioToaster",
	webapp.audiosetting.getAudioToasterBase,
	{
		templatePath: dojo.moduleUrl("webapp.audiosetting","templates/getAudioToaster8082.html"),

		/**
		 * DomElement of the Bar
		 * @property
		 * @type domElement
		 */
		domAppBarClose				: null,
		domAppBarAudioOptions		: null,
		domAppBarAudioVoicemode		: null,
		/**
		 * DomElement of the Label
		 * @property
		 * @type domElement
		 */
		domVolumeLabel			: null,
		/**
		 * DomElement of the Slider
		 * @property
		 * @type domElement
		 */
		domVolumeSlider			: null,

		/**
		 * Current Displayed label
		 * @property
		 * @type String
		 */
		strVolumeLabel			: "",

		/**
		 * Slider UIElement
		 * @property
		 * @type GraduateSlider
		 */
		volumeSliderUI			: null,

		/**
		 * Appbar UIElement
		 * @property
		 * @type AppBar
		 */
		appBarUiClose			: null,
		appBarUiAudioOptions	: null,
		appBarUiAudioVoicemode	: null,
		/**
		 * Mute AppButton
		 * @property
		 * @type AppButton
		 */
		muteButtonUi			: null,
		/**
		 * Voicemode AppButtons
		 * @property
		 * @type AppButtonArray
		 */
		arrVoicemodeButtonUi	: [],

		/**
		 * Current Volume value
		 * @property
		 * @type Float
		 */
		volumeValue				: 0,

		appBars					: [],

		attributeMap: {
			strVolumeLabel: {
				node: "domVolumeLabel",
				type: "innerHTML"
			}
		},

		postCreate : function(){
			this.strVolumeLabel = _("Ring Volume", "webapp.audiosetting");
			this.buildAppBar();
			this.buildSlider();
		},

		/**
		 * Build the appbar
		 */
		buildAppBar: function() {
			//Create the appbar
			this.appBarUiClose = ICTouchAPI.AppBarServices.createAppBar("audiosetting","getAudioToaster0",this.webapp.data._arrAppBarSkeleton[0]);
			this.appBarUiAudioOptions = ICTouchAPI.AppBarServices.createAppBar("audiosetting","getAudioToaster1",this.webapp.data._arrAppBarSkeleton[1]);
			this.appBarUiAudioVoicemode = ICTouchAPI.AppBarServices.createAppBar("audiosetting","getAudioToaster2",this.webapp.data._arrAppBarSkeleton[2], true);

			//Build AppBarUiClose
			this.hideButtonUi = new UIElements.AppButton.AppButtonControl(this.webapp.data.getHideObjButton());
			this.appBarUiClose.addAppButton(this.hideButtonUi,false);
			this.appBarUiClose.buildDom();

			//Build AppBarUiAudioOptions
			this.muteButtonUi = new UIElements.AppButton.AppButtonControl(this.webapp.data.getMuteObjButton());
			this.appBarUiAudioOptions.addAppButton(this.muteButtonUi,false);
			this.qualityAudioButtonUi = new UIElements.AppButton.AppButtonControl(this.webapp.data.getObjQualityAudioButton());
			this.appBarUiAudioOptions.addAppButton(this.qualityAudioButtonUi,false);
			this.appBarUiAudioOptions.buildDom();
			if(!this.webapp.data.getBoolQualityAudioDisplay()){
				this.qualityAudioButtonUi.hide();
			}

			var voicemodesButtons = this.webapp.data.getArrVoicemodeButtons();
			for(var i in voicemodesButtons){
				var uiButton = new UIElements.AppButton.AppButtonControl(voicemodesButtons[i]);
				this.arrVoicemodeButtonUi[voicemodesButtons[i].intVoicemode] = uiButton;
				if (voicemodesButtons[i].voicemodeStatus != this.webapp.data.STATUS_NEVER){
					//Add it to the appbar
					this.appBarUiAudioVoicemode.addAppButton(uiButton,false);
					if (voicemodesButtons[i].voicemodeStatus == this.webapp.data.STATUS_UNAVAILABLE){
						uiButton.hide();
					}else if(voicemodesButtons[i].voicemodeStatus == this.webapp.data.STATUS_DISABLE){
						uiButton.greyTint();
					}else{
						ICTouchAPI.debugServices.error("webapp.audiosettings -- getAudioToaster -- buildAppBar -- Unknow voicemode status");
					}
				}
			}
			this.appBarUiAudioVoicemode.buildDom();

			//Place appBars in the dom
			this.appBarUiClose.placeAt(this.domAppBarClose);
			this.appBarUiAudioOptions.placeAt(this.domAppBarAudioOptions);
			this.appBarUiAudioVoicemode.placeAt(this.domAppBarAudioVoicemode);
		},

		/**
		 * Build the slider
		 */
		buildSlider : function(){
			//Create the ui
			this.volumeSliderUI = new UIElements.Slider.Graduate({
				intValue: this.webapp.data.getCurrentVoicemodeVolume(),
				intMinimum: 0,
				intMaximum: this.webapp.data.getCurrentVoicemodeMaxVolume(),
				boolShowTime: false,
				boolShowButtonLabel : false,
				boolSeekable: true
			}, this.domVolumeSlider);
			//Add it a callback
			this.volumeSliderUI.seekingstart = dojo.hitch(webapp.audiosetting, webapp.audiosetting.handleVolumeSliderBeginChange);
			this.volumeSliderUI.seekingend = dojo.hitch(webapp.audiosetting, webapp.audiosetting.handleVolumeSliderEndChange);
		},

		/**
		 * Change the status icon of the mute button
		 * @param {Boolean} statusValue new status
		 */
		changeMuteStatusIcon : function(statusValue){
			this.muteButtonUi.setStatusIcon(statusValue);
			statusValue = null;
		},

		/**
		 * Change the status icon of any voicemode button
		 * @param {Int} intVoicemode the concernede voicemode
		 * @param {Boolean} statusValue new status
		 */
		changeVoicemodeStatusIcon : function(intVoicemode,statusValue){
			if (this.arrVoicemodeButtonUi[intVoicemode]){
				this.arrVoicemodeButtonUi[intVoicemode].setStatusIcon(statusValue);
			}
			intVoicemode=null;
			statusValue=null;
		},

		/**
		 * Hide a voicemode button
		 * @param {Int} intVoicemode The voicemode to hide
		 */
		hideButton : function(intVoicemode){
			if (this.arrVoicemodeButtonUi[intVoicemode]){
				this.arrVoicemodeButtonUi[intVoicemode].hide();
			}
			intVoicemode=null;
		},

		/**
		 * Grey a voicemode button
		 * @param {Int} intVoicemode The voicemode to Grey
		 */
		greyButton : function(intVoicemode){
			if (this.arrVoicemodeButtonUi[intVoicemode]){
				this.arrVoicemodeButtonUi[intVoicemode].greyTint();
			}
			intVoicemode=null;
		},

		/**
		 * Show a voicemode button
		 * @param {Int} intVoicemode The voicemode to Show
		 */
		showButton : function(intVoicemode){
			if (this.arrVoicemodeButtonUi[intVoicemode]){
				this.arrVoicemodeButtonUi[intVoicemode].show();
			}
			intVoicemode=null;
		},

		/**
		 * Change the slider value
		 * @param {Int} value New slider value
		 */
		setSliderValue: function(value){
			this.volumeValue = value;
			var intValue = Math.round(value);
			this.volumeSliderUI.setValue(intValue);
			value=null;
		},

		/**
		 * Change the voicemode label
		 * @param {String} newText The new Text to display
		 */
		setVoicemodeText : function(newText){
			this.strVolumeLabel.changeText(newText);
			ICTouchAPI.i18nServices.unsubscribeAllI18nHandlers(this.strVolumeLabel);
			var i18nHandler = this.strVolumeLabel.fillInnerHTML(this.domVolumeLabel);
			ICTouchAPI.i18nServices.addI18nHandler(this.strVolumeLabel, i18nHandler);
			newText=null;
		},

		/**
		 * Change the max value of the volume slider
		 * @param {Int} intMax New max slider value
		 */
		setMaxSlider : function(intMax){
			this.volumeSliderUI.intMaximum = intMax;
			intMax=null;
		},

		/**
		 * Get the current slider value
		 * @return {Int} the current displayed Value
		 */
		getSliderValue : function(){
			return this.volumeSliderUI.getValue();
		},

		/**
		 * Get the saved value, not the slider one
		 * @return {Float} the current Saved Value
		 */
		getVolumeValue : function(){
			return this.volumeValue;
		},

		/**
		 * Change the seekable behaviour of the volume slider
		 * @param {Boolean} boolValue The new seekable value
		 */
		setSliderSeekable : function(boolValue){
			this.volumeSliderUI.setSeekable(boolValue);
			boolValue=null;
		}


	}
	);