/**
 * @class webapp.audiosetting.data
 * @namespace webapp.audiosetting
 * @extends webapp.audiosetting.dataBase
 */
dojo.require("webapp.audiosetting.dataBase");
dojo.provide("webapp.audiosetting.data");
dojo.declare("webapp.audiosetting.data", webapp.audiosetting.dataBase, {
	/**
	 * Controller
	 * @property
	 * @type Controller
	 */
	control  : webapp.audiosetting,

	//VOICEMODE STATES
	VOICEMODE_IDLE						: 0,
	VOICEMODE_HANDSET					: 1,
	VOICEMODE_HEADSET					: 2,
	VOICEMODE_INTERNAL_HANDFREE			: 3,
	VOICEMODE_INTERNAL_GROUP_LISTENING	: 4,
	VOICEMODE_EXTERNAL_HANDFREE			: 5,
	VOICEMODE_EXTERNAL_GROUP_LISTENING	: 6,
	VOICEMODE_RING						: 7,
	VOICEMODE_STREAMING					: 8,

	//AUDIOEVENT TYPES
	EVT_NOTHING			: 0,
	EVT_OFFHOOK			: 1,
	EVT_ONHOOK			: 2,
	EVT_VOL_PLUS		: 3,
	EVT_VOL_MINUS		: 4,
	EVT_MUTE_ON			: 5,
	EVT_MUTE_OFF		: 6,
	EVT_ROUTING_CHANGED	: 7,
	EVT_KEY_PLUS	: 17,
	EVT_KEY_MINUS	: 18,

	//VOICEMODE STATUS
	STATUS_NEVER			:-1,
	STATUS_DISABLE			:0,
	STATUS_UNAVAILABLE		:2,
	STATUS_ENABLE			:1,

	//Buttons ID
	QUALITY_AUDIO_BTN		: "QUALITY_AUDIO_BTN",
	MUTE_BTN				: "MUTE_BTN",
	HIDE_BTN				: "HIDE_BTN",
	HANDSET_BTN				: "HANDSET_BTN",
	HEADSET_BTN				: "HEADSET_BTN",
	EXTERNAL_HANDSFREE_BTN  : "EXTERNAL_HANDSFREE_BTN",
	INTERNAL_HANDSFREE_BTN  : "INTERNAL_HANDSFREE_BTN",
	INTERNAL_GROUP_BTN		: "INTERNAL_GROUP_BTN",
	EXTERNAL_GROUP_BTN		: "EXTERNAL_GROUP_BTN",
	IDLE_BTN				: "IDLE_BTN",
	STREAMING_BTN			: "STREAMING_BTN",
	RING_BTN				: "RING_BTN",

	// Status Icons for appButtons
	STATUS_ON_ICON			: 'notif-on',
	STATUS_OFF_ICON			: 'notif-off',

	//Arrays to save datas
	/**
	 * Max Volume for each voicemode
	 * @property
	 * @type IntArray
	 */
	arrMaxVolumes					: [], //Maximum volume for each Voicemode
	/**
	 * Label for each voicemode
	 * @property
	 * @type StringArray
	 */
	arrVoicemodeLabels				: [], //Labels for each Voicemode
	/**
	 * ObjButton for each voicemode
	 * @property
	 * @type ObjectArray
	 */
	arrObjVoicemodeButtons			: [], //Button model for each voicemode
	/**
	 * Setting name for each voicemode
	 * @property
	 * @type StringArray
	 */
	arrVoicemodeSettings			: [], //Volume Settings used for each voicemode
	/**
	 * Secondary setting name for some voicemode
	 * @property
	 * @type StringArray
	 */
	arrSecondaryVoicemodeSettings	: [], //array used if a voicemode is linked to multiple settings
	/**
	 * Current volume for each voicemode
	 * @property
	 * @type IntArray
	 */
	arrVolumeValues					: [], //Volume Values for each Setting

	/**
	 * Curretn volume for the streaming mode
	 * @property
	 * @type Float
	 */
	streamingVolume			: 0,

	/**
	 * Button Object for the mute button
	 * @property
	 * @type Object
	 */
	objMuteButton			: {}, //Button model for the mute


	/**
	 * Button Object for the quality audio button
	 * @property
	 * @type Object
	 */
	objQualityAudioButton : null,

	/**
	 * Current voicemode
	 * @property
	 * @type Int
	 */
	currentVoicemode		: null, //Current voicemode

	/**
	 * Curretn Mute status
	 * @property
	 * @type Boolean
	 */
	muteStatus				: false, //Current mute status

	/**
	 * Notify we want to stop current media play 
	 * @property
	 * @type Boolean
	 */
	boolShownToaster		: false,

	/**
	 * Is audio quality (AVAD/AGC) activated ?
	 * @property
	 * @type Boolean
	 */
	boolQualityAudio		: false,

	/**
	 * Do we display audio quality (AVAD/AGC) button ?
	 * @property
	 * @type Boolean
	 */
	boolQualityAudioDisplay	: false,

	_arrAppBarSkeleton		: [],
	//STATUS
	APPBAR_CLOSE			: 0,
	APPBAR_AUDIOOPTIONS		: 1,
	APPBAR_AUDIOVOICEMODE	: 2,

	constructor : function(){
		var func = {
			context:this.control,
			func:this.control.handleButtons
		};

		this._arrAppBarSkeleton[this.APPBAR_CLOSE] = [this.HIDE_BTN];
		this._arrAppBarSkeleton[this.APPBAR_AUDIOOPTIONS] = [this.MUTE_BTN,this.QUALITY_AUDIO_BTN];
		this._arrAppBarSkeleton[this.APPBAR_AUDIOVOICEMODE] = [this.IDLE_BTN,this.HANDSET_BTN,this.HEADSET_BTN,this.INTERNAL_HANDSFREE_BTN,this.INTERNAL_GROUP_BTN,this.EXTERNAL_HANDSFREE_BTN,this.EXTERNAL_GROUP_BTN,this.RING_BTN,this.STREAMING_BTN];

		//Define the MuteButton
		this.objMuteButton = {
			strButtonName: this.MUTE_BTN,
			strButtonLabel: _("Mute", "webapp.audiosetting"),
			strButtonIcon: 'audiosetting-mute',
			strStatusIcon: this.STATUS_OFF_ICON,
			callback: func
		};

		//Define the Quality Audio button
		this.objQualityAudioButton = {
			strButtonName: this.QUALITY_AUDIO_BTN,
			strButtonLabel: "Auto",
			strButtonIcon: 'audiosetting-quality-audio',
			strStatusIcon: this.STATUS_OFF_ICON,
			callback: func
		};

		this.objHideButton = {
			strButtonName: this.HIDE_BTN,
			strButtonLabel: _("Hide", "ICTouchAPI"),
			strButtonIcon: 'generic-down',
			callback: func
		};

		//Define the mode buttons
		this.arrObjVoicemodeButtons[this.VOICEMODE_IDLE] = {
			strButtonName: this.IDLE_BTN,
			strButtonLabel: _("Idle", "webapp.audiosetting"),
			intVoicemode : this.VOICEMODE_IDLE,
			strButtonIcon: 'audiosetting-handset',
			strStatusIcon: this.STATUS_OFF_ICON,
			voicemodeStatus:this.STATUS_NEVER,
			isVolumeChangeable:this.isRingVolumeLocked,
			callback: func
		};
		this.arrObjVoicemodeButtons[this.VOICEMODE_HANDSET] = {
			strButtonName: this.HANDSET_BTN,
			strButtonLabel: _("Handset", "webapp.audiosetting"),
			intVoicemode : this.VOICEMODE_HANDSET,
			strButtonIcon: 'audiosetting-handset',
			strStatusIcon: this.STATUS_OFF_ICON,
			voicemodeStatus:this.STATUS_AVAILABLE,
			isVolumeChangeable:true,
			callback: func
		};
		this.arrObjVoicemodeButtons[this.VOICEMODE_HEADSET] = {
			strButtonName: this.HEADSET_BTN,
			strButtonLabel: _("Headset", "webapp.audiosetting"),
			intVoicemode : this.VOICEMODE_HEADSET,
			strButtonIcon: 'audiosetting-headset',
			strStatusIcon: this.STATUS_OFF_ICON,
			voicemodeStatus:this.STATUS_AVAILABLE,
			isVolumeChangeable:true,
			callback: func
		};
		this.arrObjVoicemodeButtons[this.VOICEMODE_INTERNAL_HANDFREE] = {
			strButtonName: this.INTERNAL_HANDSFREE_BTN,
			strButtonLabel: _("Internal Handsfree", "webapp.audiosetting"),
			intVoicemode : this.VOICEMODE_INTERNAL_HANDFREE,
			strButtonIcon: 'audiosetting-handsfree',
			strStatusIcon: this.STATUS_OFF_ICON,
			voicemodeStatus:this.STATUS_AVAILABLE,
			isVolumeChangeable:true,
			callback: func
		};
		this.arrObjVoicemodeButtons[this.VOICEMODE_INTERNAL_GROUP_LISTENING] = {
			strButtonName: this.INTERNAL_GROUP_BTN,
			strButtonLabel: _("Internal Group Listening", "webapp.audiosetting"),
			intVoicemode : this.VOICEMODE_INTERNAL_GROUP_LISTENING,
			strButtonIcon: 'audiosetting-group-listening',
			strStatusIcon: this.STATUS_OFF_ICON,
			voicemodeStatus:this.STATUS_AVAILABLE,
			isVolumeChangeable:true,
			callback: func
		};
		this.arrObjVoicemodeButtons[this.VOICEMODE_EXTERNAL_HANDFREE] = {
			strButtonName: this.EXTERNAL_HANDSFREE_BTN,
			strButtonLabel: _("External Handsfree", "webapp.audiosetting"),
			intVoicemode : this.VOICEMODE_EXTERNAL_HANDFREE,
			strButtonIcon: 'audiosetting-bluetooth-handsfree',
			strStatusIcon: this.STATUS_OFF_ICON,
			voicemodeStatus:this.STATUS_AVAILABLE,
			isVolumeChangeable:true,
			callback: func
		};
		this.arrObjVoicemodeButtons[this.VOICEMODE_EXTERNAL_GROUP_LISTENING] = {
			strButtonName: this.EXTERNAL_GROUP_BTN,
			strButtonLabel: _("External Group Listening", "webapp.audiosetting"),
			intVoicemode : this.VOICEMODE_EXTERNAL_GROUP_LISTENING,
			strButtonIcon: 'audiosetting-externalaudio',
			strStatusIcon: this.STATUS_OFF_ICON,
			voicemodeStatus:this.STATUS_AVAILABLE,
			isVolumeChangeable:true,
			callback: func
		};
		this.arrObjVoicemodeButtons[this.VOICEMODE_RING] = {
			strButtonName: this.RING_BTN,
			strButtonLabel: _("Ring", "webapp.audiosetting"),
			intVoicemode : this.VOICEMODE_RING,
			strButtonIcon: 'audiosetting-externalaudio',
			strStatusIcon: this.STATUS_OFF_ICON,
			voicemodeStatus:this.STATUS_NEVER,
			isVolumeChangeable:this.isRingVolumeLocked,
			callback: func
		};
		this.arrObjVoicemodeButtons[this.VOICEMODE_STREAMING] = {
			strButtonName: this.STREAMING_BTN,
			strButtonLabel: _("Streaming", "webapp.audiosetting"),
			intVoicemode : this.VOICEMODE_STREAMING,
			strButtonIcon: 'audiosetting-externalaudio',
			strStatusIcon: this.STATUS_OFF_ICON,
			voicemodeStatus:this.STATUS_NEVER,
			isVolumeChangeable:true,
			callback: func
		};

		//Define the labels mode
		this.arrVoicemodeLabels[this.VOICEMODE_IDLE] = "Ring Volume";
		this.arrVoicemodeLabels[this.VOICEMODE_HANDSET] = "Handset Volume";
		this.arrVoicemodeLabels[this.VOICEMODE_HEADSET] = "Headset Volume";
		this.arrVoicemodeLabels[this.VOICEMODE_INTERNAL_HANDFREE] = "Speaker Volume";
		this.arrVoicemodeLabels[this.VOICEMODE_INTERNAL_GROUP_LISTENING] = "Speaker Volume";
		this.arrVoicemodeLabels[this.VOICEMODE_EXTERNAL_HANDFREE] = "Speaker Volume";
		this.arrVoicemodeLabels[this.VOICEMODE_EXTERNAL_GROUP_LISTENING] = "Speaker Volume";
		this.arrVoicemodeLabels[this.VOICEMODE_RING] = "Ring Volume";
		this.arrVoicemodeLabels[this.VOICEMODE_STREAMING] = "Streaming Volume";

		//Define the settings used for each mode
		this.arrVoicemodeSettings[this.VOICEMODE_IDLE] = "AudioRingVolume";
		this.arrVoicemodeSettings[this.VOICEMODE_HANDSET] = "AudioHandsetVolume";
		this.arrVoicemodeSettings[this.VOICEMODE_HEADSET] = "AudioHeadsetVolume";
		this.arrVoicemodeSettings[this.VOICEMODE_INTERNAL_HANDFREE] = "AudioSpeakerPhoneVolume";
		this.arrVoicemodeSettings[this.VOICEMODE_INTERNAL_GROUP_LISTENING] = "AudioSpeakerPhoneVolume";
		this.arrVoicemodeSettings[this.VOICEMODE_EXTERNAL_HANDFREE] = "AudioSpeakerPhoneVolume";
		this.arrVoicemodeSettings[this.VOICEMODE_EXTERNAL_GROUP_LISTENING] = "AudioSpeakerPhoneVolume";
		this.arrVoicemodeSettings[this.VOICEMODE_RING] = "AudioRingVolume";
		this.arrVoicemodeSettings[this.VOICEMODE_STREAMING] = "MPSpeakerPhoneVolume";

		//Define the other settings used for the Streaming voicemode
		this.arrSecondaryVoicemodeSettings[this.VOICEMODE_STREAMING] = [];
		this.arrSecondaryVoicemodeSettings[this.VOICEMODE_STREAMING].push("MPHandsetVolume");
		this.arrSecondaryVoicemodeSettings[this.VOICEMODE_STREAMING].push("MPHeadsetVolume");
		this.arrSecondaryVoicemodeSettings[this.VOICEMODE_STREAMING].push("MPAlarmVolume");
		this.arrSecondaryVoicemodeSettings[this.VOICEMODE_STREAMING].push("MPPreviewVolume");


		//The streaming mode is not managed by audio module, so we have to set the value by hand
		this.arrMaxVolumes[this.VOICEMODE_STREAMING] = 7;


		//Get the curretn mute Value
		this.getMuteValue();

		//Get the current voicemode status
		this.getVoicemodesStatus();

		//Get the Current Voicemode
		this.getCurrentVoicemode();

		//Get the current volume values
		this.getVolumesValues();
	},

	/**
	 * Callback while we get the mute value
	 * @param {Boolean} value new mute value
	 */
	gotMuteValue : function (value){
		//Inform controller
		this.control.muteValueChanged(value);
		value=null;
	},

	/**
	 * Callback when we get the current voicemode
	 * @param {Int} intVoicemode the new current voicemode
	 */
	gotCurrentVoicemode : function (intVoicemode){
		//Inform controller
		this.control.voicemodeChanged(this.currentVoicemode,intVoicemode);
		intVoicemode=null;
	},

	/**
	 * Callback when we get the voicemode status
	 * @param {IntArray} arrStatus status of all voicemodes
	 */
	gotVoicemodesStatus : function(arrStatus){

		//Simulate the samùe behaviour as 01.018.0
		//Will be deleted when the ictaudio module will be right


		for (var i=0;i<arrStatus.length; i++){
			if (this.arrObjVoicemodeButtons[i].voicemodeStatus != this.STATUS_NEVER && this.arrObjVoicemodeButtons[i].voicemodeStatus != arrStatus[i]){
				//Inform controller for each voicemode
				this.control.voicemodeStatusChanged(i, arrStatus[i]);
			}
		}
		arrStatus=null;
	},

	/**
	 * Get the current volume values for each setting
	 */
	getVolumesValues : function(){
		//For each defined settings
		for(var i=0;i<this.arrVoicemodeSettings.length;i++){
			//If we haven't already try to get the value'
			if (typeof (this.arrVolumeValues[this.arrVoicemodeSettings[i]]) == "undefined"){
				//fill the array with a false value to inform we have begin to ask the setting value
				this.arrVolumeValues[this.arrVoicemodeSettings[i]] = -1;
				//Get the setting Value
				ICTouchAPI.settingServices.getSetting(this.arrVoicemodeSettings[i], this, this.gotVolumeValue);
			}
		}
	},

	/**
	 * Callback when we get the volume setting value
	 * @param {Object} objSetting the volume setting
	 */
	gotVolumeValue : function(objSetting){
		//Save the value
		this.arrVolumeValues[objSetting.id] = objSetting.jsValue;
		objSetting=null;
	},

	/**
	 * Callback of AudioRingLock setting
	 * @param {Boolean} new AudioRingLock value
	 */
	gotRingLockValue :function(value) {
		this.isRingVolumeLocked = value;
		this.updateVoicemodeChangeable(!value);
		value=null;
	},

	/**
	 * Update the volumeChangeable attribute of the concerned voicemodes
	 * @param {Boolean} value new changeable value
	 */
	updateVoicemodeChangeable : function (value){
		for(var i=0; i<this.arrVoicemodeSettings.length;i++){
			//Find the concerned voicemode using the AudioRingVolume setting
			if (this.arrVoicemodeSettings[i] == "AudioRingVolume"){
				this.arrObjVoicemodeButtons[i].isVolumeChangeable = value;
				if (i == this.currentVoicemode){
					this.control.seekableChanged(value);
				}
			}
		}
		value=null;
	},

	/**
	 * Callback when we get the max volume value of a voicemode
	 * @param {Int} maxVol the max volume
	 * @param {Int} mode The corresponding voicemode
	 */
	gotVoicemodeMaxVolume : function (maxVol, mode){
		this.arrMaxVolumes[mode] = maxVol;
		this.control.maxVolumeChanged(mode, maxVol);
		//Same max value for RING and IDLE
		if (mode == this.VOICEMODE_RING){
			this.gotVoicemodeMaxVolume(maxVol,this.VOICEMODE_IDLE);
		}
		maxVol=null;
		mode=null;
	},

	/**
	 * get the models for the voicemode buttons
	 * @return {ObjectArray} The voicemode Buttons
	 */
	getArrVoicemodeButtons : function (){
		return this.arrObjVoicemodeButtons;
	},

	/**
	 * Get the model for the mute button
	 * @return {Object} The mute button
	 */
	getMuteObjButton : function(){
		return this.objMuteButton;
	},


	getHideObjButton : function(){
		return this.objHideButton;
	},


	/**
	 * Verify if the given setting is related to the current voicemode
	 * @param {String} settingName the setting to test
	 * @return {Boolean} isCurrentVoicemodeLinkedToGivenSetting
	 */
	isCurrentVoicemodeLinkedToSetting : function(settingName){
		//If the setting is the primary one of the voicemode
		if (this.arrVoicemodeSettings[this.currentVoicemode] == settingName){
			settingName=null;
			return true;
		}
		//find if the setting if one of the secondary of the voicemode
		if (this.arrSecondaryVoicemodeSettings[this.currentVoicemode]){
			for (var i=0; i < this.arrSecondaryVoicemodeSettings[this.currentVoicemode].length ; i++){
				if (this.arrSecondaryVoicemodeSettings[this.currentVoicemode][i] == settingName){
					settingName=null;
					return true;
				}
			}
		}
		settingName=null;
		return false;
	},

	/**
	 * Get the volume of the curent voicemode
	 * @return {Int} Volume of the current voicemode
	 */
	getCurrentVoicemodeVolume : function(){
		return this.arrVolumeValues[this.arrVoicemodeSettings[this.currentVoicemode]];
	},

	/**
	 * Get the Max volume of the curent voicemode
	 * @return {Int} Max Volume of the current voicemode
	 */
	getCurrentVoicemodeMaxVolume : function(){
		return this.arrMaxVolumes[this.currentVoicemode];
	},

	/**
	 * Get the label for the current voicemode
	 * @return {String} the Label of the voicemode
	 */
	getCurrentVoicemodeText : function(){
		return this.arrVoicemodeLabels[this.currentVoicemode];
	},

	/**
	 * By giving the strButtonName, it return the corresponding voicemode
	 * @param {String} buttonId id of the button
	 * @return {Int} Voicemode of the button
	 */
	getcorrespondingVoicemodeToButton : function(buttonId){
		for(var i=0;i<this.arrObjVoicemodeButtons.length;i++){
			if (this.arrObjVoicemodeButtons[i].strButtonName == buttonId){
				buttonId=null;
				return this.arrObjVoicemodeButtons[i].intVoicemode;
			}
		}
		buttonId=null;
		return null;
	},

	/**
	 * Return the setting used by the current voicemode
	 * @return {String} The setting of the current voicemode
	 */
	getCurrentVoicemodeSetting : function(){
		return this.arrVoicemodeSettings[this.currentVoicemode];
	},

	/**
	 * Get the setting used for the following voicemode
	 * @param {Int} voicemode The voicemode
	 * @return {String} The setting of the given voicemode
	 */
	getVoicemodeSetting : function(voicemode){
		return this.arrVoicemodeSettings[voicemode];
		voicemode=null;
	},

	/**
	 * Verify if the volume is changeable for the following voicemode
	 * @param {Int} voicemode The voicemode
	 * @return {Boolean} Is the volume changeable
	 */
	isVoicemodeVolumeChangeable : function(voicemode){
		return this.arrObjVoicemodeButtons[voicemode].isVolumeChangeable;
		voicemode=null;
	},

	/**
	 * Get the actual mute status
	 * @return {Boolean} Mute status
	 */
	getMuteStatus : function(){
		return this.muteStatus;
	},

	/**
	 * Function to know if the current volume is the max one
	 * @return {Boolean} isCurrentVoicemodeVolumeMax
	 */
	isCurrentVoicemodeVolumeMax : function(){
		return this.arrVolumeValues[this.arrVoicemodeSettings[this.currentVoicemode]] === this.arrMaxVolumes[this.currentVoicemode];
	},

	/**
	 * Function to know if the current volume is the min one
	 * @return {Boolean} isCurrentVoicemodeVolumeMin
	 */
	isCurrentVoicemodeVolumeMin : function(){
		return this.arrVolumeValues[this.arrVoicemodeSettings[this.currentVoicemode]] === 0;
	},


	getObjQualityAudioButton : function(){
		this.objQualityAudioButton.strStatusIcon = (this.getBoolQualityAudio()) ? this.STATUS_ON_ICON : this.STATUS_OFF_ICON;
		return this.objQualityAudioButton;
	},

	getBoolQualityAudio : function(){
		return this.boolQualityAudio;
	},

	setBoolQualityAudio : function(_bool){
		if(this.boolQualityAudio != _bool){
			this.boolQualityAudio = _bool;
			this.control.updateAutoVolumeButton();
		}
	},

	getBoolQualityAudioDisplay : function(){
		return this.boolQualityAudioDisplay;
	},

	setBoolQualityAudioDisplay : function(_bool){
		if(this.boolQualityAudioDisplay != _bool){
			this.boolQualityAudioDisplay = _bool;
			this.control.updateAutoVolumeButton();
		}
	}

});

webapp.audiosetting.data = new webapp.audiosetting.data();