/**
 * @class webapp.audiosetting.control
 * @namespace webapp.audiosetting
 * @extends webapp.audiosetting.controlBase
 */
dojo.require("webapp.audiosetting.controlBase");
dojo.provide("webapp.audiosetting");
dojo.declare("webapp.audiosetting",webapp.audiosetting.controlBase,{

	/**
	 * Toaster view
	 * @property
	 * @type View
	 */
	objToasterUI		: null,

	/**
	 * Id of the previous view in the toaster
	 * @property
	 * @type String
	 */
	oldToasterContentId : "",

	constructor : function(){
		ICTouchAPI.skinServices.linkWebappsStyles("webapp.audiosetting", "webapp.audiosetting", true, true, true);
		ICTouchAPI.eventServices.subscribeToEvent(this, "audio_key_event", this.onAudioKeyEvent);
		ICTouchAPI.eventServices.subscribeToEvent(this, "voicemode_trans", this.onVoicemodeTrans);

		ICTouchAPI.eventServices.subscribeToEvent(this, "presentationLoaded", this._presentationLoaded);
        ICTouchAPI.eventServices.subscribeToEvent(this, "CallStartedEvent", this.closeAudioToaster);

		ICTouchAPI.eventServices.subscribeToEvent(this, "send_sense_key_event", this.handsFreeCallback);

		ICTouchAPI.settingServices.subscribeToSetting(this, "AudioRingVolume", this.onVolumeValueChanged);
		ICTouchAPI.settingServices.subscribeToSetting(this, "AudioHandsetVolume", this.onVolumeValueChanged);
		ICTouchAPI.settingServices.subscribeToSetting(this, "AudioHeadsetVolume", this.onVolumeValueChanged);
		ICTouchAPI.settingServices.subscribeToSetting(this, "AudioSpeakerPhoneVolume", this.onVolumeValueChanged);
		ICTouchAPI.settingServices.subscribeToSetting(this, "MPSpeakerPhoneVolume", this.onMediaVolumeChanged);
		ICTouchAPI.settingServices.subscribeToSetting(this, "MPHandsetVolume", this.onMediaVolumeChanged);
		ICTouchAPI.settingServices.subscribeToSetting(this, "MPHeadsetVolume", this.onMediaVolumeChanged);
		ICTouchAPI.settingServices.subscribeToSetting(this, "MPPreviewVolume", this.onMediaVolumeChanged);
		//ICTouchAPI.settingServices.subscribeToSetting(this, "MPAlarmVolume", this.onMediaVolumeChanged);

		ICTouchAPI.settingServices.subscribeToSetting(this, "AudioRingLock", this.onAudioRingLockChanged);

		ICTouchAPI.eventServices.subscribeToEvent(this, "SENSKEY_HOME", this.eventPhysicalKeyHome);

	},

	/**
	 * Event when the presentation layer is loaded to build the toaster
	 */
	_presentationLoaded:function(){
		ICTouchAPI.eventServices.unsubscribeToEvent(this,"presentationLoaded");
		this.buildAll();

		// AVAD
		ICTouchAPI.settingServices.subscribeToSetting(this, "AudioVoiceSignalLimitation", this.onAudioVoiceSignalLimitation);
		ICTouchAPI.settingServices.getSetting("AudioVoiceSignalLimitation", this, this.onAudioVoiceSignalLimitation);
		ICTouchAPI.settingServices.subscribeToSetting(this, "AudioVoiceSignalLimitationState", this.onAudioVoiceSignalLimitationState);
		ICTouchAPI.settingServices.getSetting("AudioVoiceSignalLimitationState", this, this.onAudioVoiceSignalLimitationState);
	},


	/**
	 * Build the Audiotoaster and preload it
	 */
	buildAll : function(){
		this.objToasterUI = new webapp.audiosetting.getAudioToaster({},null);
		this.objToasterUI.setVoicemodeText(this.data.getCurrentVoicemodeText());
		this.objToasterUI.setMaxSlider(this.data.getCurrentVoicemodeMaxVolume());
		ICTouchAPI.toasterServices.preloadContent(this.objToasterUI);
	},

	/**
	 * Callback of setting AudioVoiceSignalLimitation. Determines if the button audioQuality must be displayed.
	 * @params {object} objAVSLSetting setting AudioVoiceSignalLimitation
	 */
	onAudioVoiceSignalLimitation : function(objAVSLSetting) {
		if(objAVSLSetting){
			this.data.setBoolQualityAudioDisplay(objAVSLSetting.jsValue);
		}
	},

	/**
	 * Callback of setting AudioVoiceSignalLimitationState. Determines if the audioQuality is activated.
	 * @params {object} objAVSLSetting setting AudioVoiceSignalLimitationState
	 */
	onAudioVoiceSignalLimitationState : function(objAVSLSSetting) {
		if(objAVSLSSetting){
			this.data.setBoolQualityAudio(objAVSLSSetting.jsValue);
		}
	},

	/**
	 * Callback for buttons
	 */
	handleButtons : function (buttonId)	{
		this.resetTimer(true);
		if(buttonId == this.data.MUTE_BTN){
			//On click on the mute button, switch mute activation
			ICTouchAPI.APIServices.IctAudio.setMute({
				params:[!this.data.muteStatus],
				context:this
			});
		}
		else if(buttonId == this.data.HIDE_BTN){
			this.closeAudioToaster();
		}
		else if(buttonId == this.data.QUALITY_AUDIO_BTN){
			var bool_audioQuality = this.data.getBoolQualityAudio();
			//Audio quality is currently enabled
			if(bool_audioQuality){
				ICTouchAPI.APIServices.IctAudio.enableAutoVolume({
					params:[0],
					context:this
				});
				this.data.setBoolQualityAudio(false);
				this.objToasterUI.qualityAudioButtonUi.setStatusIcon('notif-off');
			}else{//Audio quality is currently disabled
				ICTouchAPI.APIServices.IctAudio.enableAutoVolume({
					params:[1],
					context:this
				});

				this.data.setBoolQualityAudio(true);
				this.objToasterUI.qualityAudioButtonUi.setStatusIcon('notif-on');
			}
		}
		else{
			var voicemode = this.data.getcorrespondingVoicemodeToButton(buttonId);
			if (voicemode){
				ICTouchAPI.APIServices.IctAudio.setVoicemode({
					params:[voicemode],
					context:this
				});
			}
		}
		buttonId = null;
	},

	eventPhysicalKeyHome : function(objEvent) {
		if(objEvent && objEvent.value !== 0){
			this.oldToasterContentId = "";
		}
	},

	handsFreeCallback : function(){
		var objArguments = ICTouchAPI.tools.getEventArguments(arguments);
		if (objArguments.senseKey ==1){
			this.openAudioToaster();
		}
	},

	updateAutoVolumeButton : function() {
		if(this.objToasterUI){
			// Show or hide the button
			if(this.data.getBoolQualityAudioDisplay()){
				this.objToasterUI.qualityAudioButtonUi.show();
			} else {
				this.objToasterUI.qualityAudioButtonUi.hide();
			}
			
			// Manage the status icon
			if(this.data.getBoolQualityAudio()){
				this.objToasterUI.qualityAudioButtonUi.setStatusIcon(this.data.STATUS_ON_ICON);
			} else {
				this.objToasterUI.qualityAudioButtonUi.setStatusIcon(this.data.STATUS_OFF_ICON);
			}
		}
	},

	/**
	 * Open the audio toaster and start the timer to hide it after 10 seconds
	 */
	openAudioToaster: function() {
		if(this.objToasterUI){
			this.objToasterUI.volumeSliderUI.refreshDisplay();
		    webapp.audiosetting.data.boolShownToaster = true;
			if (this.oldToasterContentId == "" && ICTouchAPI.toasterServices.getContentDisplayedId() != this.objToasterUI.id){
				this.oldToasterContentId = ICTouchAPI.toasterServices.getContentDisplayedId();
			}
			ICTouchAPI.toasterServices.showContent({
				"toasterContent" : this.objToasterUI
			});
			this.startTimer(true);
		}
	},

	/**
	 * Close the audio toaster and stop the timer
	 */
	closeAudioToaster : function(){
		ICTouchAPI.APIServices.ICTGate.notifyEventPresentation({
			params:["presentationHideAudiotoaster"]
		});
		ICTouchAPI.toasterServices.hideContent(webapp.audiosetting.objToasterUI);
		webapp.audiosetting.resetTimer(false);
		if (webapp.audiosetting.oldToasterContentId != "") {
			ICTouchAPI.toasterServices.showContent({
				toasterContent:{
					id:webapp.audiosetting.oldToasterContentId
				}
			})
		}
		webapp.audiosetting.oldToasterContentId = "";
		if(webapp.audiosetting.data.boolShownToaster) {
		    webapp.audiosetting.data.boolShownToaster = false;
		}
	},

	/**
	 * Function to stop the timer
	 * @param {Boolean} restart boolean to know if we have to restart again the timer
	 */
	resetTimer : function(restart){
		if( this.timerHandler ) {
			clearTimeout(this.timerHandler);
			this.timerHandler = null;
		}
		if(restart){
			this.startTimer();
		}
		restart= null;
	},

	/**
	 * Start the timer
	 */
	startTimer : function() {
		if (this.timerHandler){
			this.resetTimer(false);
		}
		this.timerHandler = setTimeout(this.closeAudioToaster, 10000);
	},


	/**
	 * Callback of audio_key_event event
	 */
	onAudioKeyEvent: function() {
		var objArguments = ICTouchAPI.tools.getEventArguments(arguments);
		var showToaster = false;
		//If the event is about mute
		if (objArguments.audioevt === this.data.EVT_MUTE_ON && this.data.getMuteStatus() !== true){
			this.muteValueChanged(true);
			showToaster = true;
		}
		//If the mute value has changed to false
		else if(objArguments.audioevt === this.data.EVT_MUTE_OFF && this.data.getMuteStatus() !== false){
			this.muteValueChanged(false);
			showToaster = true;
		}
        else if (objArguments.audioevt === this.data.EVT_VOL_MINUS || objArguments.audioevt === this.data.EVT_VOL_PLUS ){
			showToaster = true;
		}
        else if (objArguments.keyevent === this.data.EVT_KEY_MINUS || objArguments.keyevent === this.data.EVT_KEY_PLUS ){
			showToaster = true;
		}
		//open the toaster if needed
		if (showToaster){
			this.openAudioToaster();
		}

	},

	/**
	 * callback when we receive the voicemode_trans event
	 */
	onVoicemodeTrans : function(){
		//No events are currently correctly sent by ictaudio module for the moment
		//Force to get the new values by calling ictaudio methods
		//-------------------TEMPORARY FIX
		this.data.getCurrentVoicemode();
		this.data.getVoicemodesStatus();
	//-------------------TO BE REMOVED

	/*var objArguments = ICTouchAPI.tools.getEventArguments(arguments);
		//Voicemode has changed
		this.voicemodeChanged(objArguments.old_state,objArguments.new_state);
		*/
	},

	/**
	 * Callback for the volume settings
	 * @param {Object} volumeSetting The volumeSetting object
	 */
	onVolumeValueChanged : function(volumeSetting){
		//Save the new value
		this.data.arrVolumeValues[volumeSetting.id] = volumeSetting.jsValue;
		//If if it the current volume displayed
		if (this.objToasterUI && this.data.isCurrentVoicemodeLinkedToSetting(volumeSetting.id) && this.objToasterUI.getVolumeValue() != volumeSetting.jsValue){
			//Change the sldier value
			this.objToasterUI.setSliderValue(volumeSetting.jsValue);
		}
		volumeSetting=null;
	},

	/**
	 * Callback for the media volume settings
	 * @param {Object} volumeSetting The mediaVolumeSetting object
	 */
	onMediaVolumeChanged : function(volumeSetting){
		volumeSetting.jsValue = parseFloat(volumeSetting.jsValue*5);
		this.onVolumeValueChanged(volumeSetting);
		volumeSetting = null;
	},

	/**
	 * Callback on the AudioRingLock setting
	 * @param {Object} lockSetting The AudioRingLock setting
	 */
	onAudioRingLockChanged : function(lockSetting){
		this.data.gotRingLockValue(lockSetting.jsValue);
		lockSetting = null;
	},
	/**
	 * Whan the mute value has changed
	 * @param {Boolean} value new mute value
	 */
	muteValueChanged : function(value){
		//Save the new value
		this.data.muteStatus = value;

		//Update the button with the status icon
		//And turn on/off the led
		if (value){
			this.data.objMuteButton.strStatusIcon = this.data.STATUS_ON_ICON;
			ICTouchAPI.ledServices.activate('mute', 'ON');
		}
		else{
			this.data.objMuteButton.strStatusIcon = this.data.STATUS_OFF_ICON;
			ICTouchAPI.ledServices.desactivate('mute', 'ON');
		}

		//If the toaster was already built
		if (this.objToasterUI){
			//Update the appbutton ui element
			this.objToasterUI.changeMuteStatusIcon(this.data.objMuteButton.strStatusIcon);
		}
		value=null;
	},

	/**
	 * When the current voicemode has changed
	 * @param {Int} oldVoicemode Old Voicemode
	 * @param {Int} newVoicemode New Voicemode
	 */
	voicemodeChanged: function (oldVoicemode,newVoicemode){
		//If voicemode has not really changed
		//This can happen with an ictaudio bug
		if (oldVoicemode == newVoicemode){
			return;
		}
		//reset the timer to avoid closing toaster if not needed
		//It will be restarted if needed at the end of this function
		this.resetTimer(true);

		//Desactivate the old voicemode
		if (oldVoicemode && this.data.arrObjVoicemodeButtons[oldVoicemode]){
			this.data.arrObjVoicemodeButtons[oldVoicemode].strStatusIcon = this.data.STATUS_OFF_ICON;
			if (this.objToasterUI){
				this.objToasterUI.changeVoicemodeStatusIcon(oldVoicemode,this.data.STATUS_OFF_ICON);
			}
		}

		//Turn off the handsfree led if the old voicemode was this type
		if(oldVoicemode == this.data.VOICEMODE_INTERNAL_HANDFREE || oldVoicemode == this.data.VOICEMODE_INTERNAL_GROUP_LISTENING ||
			oldVoicemode == this.data.VOICEMODE_EXTERNAL_HANDFREE || oldVoicemode == this.data.VOICEMODE_EXTERNAL_GROUP_LISTENING) {
			ICTouchAPI.ledServices.desactivate('handsFree', 'ON');
		}


		//Set the new Voicemode
		this.data.currentVoicemode = newVoicemode;

		//Activate the new voicemode
		if (this.data.arrObjVoicemodeButtons[newVoicemode]){
			this.data.arrObjVoicemodeButtons[newVoicemode].strStatusIcon = this.data.STATUS_ON_ICON;
			//Get the max value if we have'nt already it
			if (!this.data.arrMaxVolumes[newVoicemode]){
				this.askForMaxVolume(newVoicemode);
			}
			if (this.objToasterUI){
				//Get the max value to update the slider
				if (this.data.arrMaxVolumes[newVoicemode]){
					this.objToasterUI.setMaxSlider(this.data.arrMaxVolumes[newVoicemode]);
				}
				this.objToasterUI.setSliderSeekable(this.data.isVoicemodeVolumeChangeable(newVoicemode));
				this.objToasterUI.changeVoicemodeStatusIcon(newVoicemode,this.data.STATUS_ON_ICON);
				this.objToasterUI.setSliderValue(this.data.getCurrentVoicemodeVolume());
				this.objToasterUI.setVoicemodeText(this.data.getCurrentVoicemodeText());

			}
		}

		//Turn on the handsfree led if the new voicemode is this type
		if(newVoicemode == this.data.VOICEMODE_INTERNAL_HANDFREE || newVoicemode == this.data.VOICEMODE_INTERNAL_GROUP_LISTENING ||
			newVoicemode == this.data.VOICEMODE_EXTERNAL_HANDFREE || newVoicemode == this.data.VOICEMODE_EXTERNAL_GROUP_LISTENING) {
			ICTouchAPI.ledServices.activate('handsFree', 'ON');
		}

		//Some exception case to display the toaster
		if (this.objToasterUI && oldVoicemode &&
			!(oldVoicemode == this.data.VOICEMODE_IDLE && newVoicemode == this.data.VOICEMODE_RING )
			&& oldVoicemode != this.data.VOICEMODE_RING
			&& oldVoicemode != this.data.VOICEMODE_IDLE
			&& oldVoicemode != this.data.VOICEMODE_STREAMING
			&& newVoicemode != this.data.VOICEMODE_IDLE){
			this.openAudioToaster();
		}
		else{
			//If the toaster is always displayed when it must be hidden, close it
			// (example case, intiate a call, change the headset/handset volume, when the call is automatically cancelled, it switch to idle mode)
			if(this.objToasterUI && ICTouchAPI.toasterServices.getContentDisplayedId() == this.objToasterUI.id){
				this.closeAudioToaster();
			}
		}
		oldVoicemode=null;
		newVoicemode=null;

	},

	/**
	 * When the status of a voicemode changed
	 * @param {Int} voicemode voicemode
	 * @param {Int} status new status
	 */
	voicemodeStatusChanged : function(voicemode,status){
		//Save the new status
		this.data.arrObjVoicemodeButtons[voicemode].voicemodeStatus = status;
		//If the toaster was already built, update the appButton Ui according to the new status
		if (this.objToasterUI){
			if (status == this.data.STATUS_UNAVAILABLE){
				this.objToasterUI.hideButton(voicemode);
			}
			else if (status == this.data.STATUS_DISABLE){
				this.objToasterUI.greyButton(voicemode);
			}
			else if (status == this.data.STATUS_ENABLE){
				this.objToasterUI.showButton(voicemode);
			}
		}
		voicemode=null;
		status=null;
	},

	/*
	 *Call this function when the seekable value has changed
	 *@param {Boolean} newValue new value
	 */
	seekableChanged : function(newValue){
		if (this.objToasterUI){
			this.objToasterUI.setSliderSeekable(newValue);
		}
		newValue=null;
	},

	/**
	 * Get the max volume value
	 * @param {Int} voicemode Voicemode for which we want the max volume
	 */
	askForMaxVolume : function(voicemode){
		if (voicemode == this.data.VOICEMODE_IDLE){
			//Simulate the voicemode ring because the IDLE one has no volume
			voicemode = this.data.VOICEMODE_RING;
		}
		this.data.getMaxVolume(voicemode);
		voicemode=null;
	},

	/**
	 * When the max volume has changed
	 * @param {Int} mode voicemode
	 * @param {Int} maxVol new max volume
	 */
	maxVolumeChanged : function(mode, maxVol){
		if (mode == this.data.currentVoicemode && this.objToasterUI){
			//Change the slider max value
			this.objToasterUI.setMaxSlider(maxVol);
			//update again the value after having chnage the max value
			this.objToasterUI.setSliderValue(this.data.getCurrentVoicemodeVolume());
		}
		mode=null;
		maxVol=null;
	},

	/**
	 * Callback when we have change the slider value
	 */
	handleVolumeSliderEndChange :function(){
		//Get the current slider value
		var value = this.objToasterUI.getSliderValue();
		//get the current setting
		var setting = this.data.getCurrentVoicemodeSetting();
		//Change the value if it is the streaming one (other module setting)
		if (setting == this.data.getVoicemodeSetting(this.data.VOICEMODE_STREAMING)){
			value = value/5;
		}
		ICTouchAPI.settingServices.setSettingValue(setting, value);
		this.resetTimer(true);
	},

	/**
	 * Callback when we begin the change the slider value
	 */
	handleVolumeSliderBeginChange :function(){
		this.resetTimer(false);
	}


});

webapp.audiosetting=new webapp.audiosetting();
