/**
 * @class UIElements._base
 * @extends dijit._Widget
 * @abstract
 * Abstract class for all UI Elements
 */
dojo.provide("UIElements._base");
dojo.declare("UIElements._base",
	dijit._Widget,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		_connectedEvents : [],

                /**
		 * @ignore
		 */
		_subLanguageChanged : null,

		_highlightClass: "selected",
		_handlerStopHighlight : null,
		_domCurrentHighlight : null,

		/**
		 * setTimeout handler for the long press callback feature
		 * @ignore
		 * @type {Boolean}
		 */
		_waitingHighlight : null,
		/**
		 * true if we must wait for highlighting, false otherwise
		 * @ignore
		 * @type {Boolean}
		 */
		_enableWaitingHighlight : false,
		/**
		 * setTimeout handler for the long press callback feature
		 * @ignore
		 */
		_handlerLongPress : null,


		/* ------------------------------------ Constructor --------------------------------------- */

		constructor: function(args){
			var splitString = this.declaredClass.split(".");
			var strLast = splitString.pop();
			this.namespace = splitString.join(".");
			this._subLanguageChanged = null;

                        if(!generalConfig.simulation) {
			this._subLanguageChanged = dojo.subscribe("languageChanged", this, function() {
				for (var i in this.attributeMap) {
					if(this[i] && this[i].isI18Ned) {
						this.attr(i, this[i]);
					}
				}
			});
                        }

			this._connectedEvents = [];

			/*
			 * This is some special code that enables the tracking of which element leaks in chrome's profiler
			 * This works by creating a new object called like the widget we are currently constructing
			 */
			if (generalConfig.developer == true) {
				eval("this.objTracking = new function myIC_UI_"+strLast+"(){}");
			}
		},

		destroy : function(){
			dojo.unsubscribe(this._subLanguageChanged);
			this._unsubscribeHighlight(this._handlerStopHighlight);
			this._handlerStopHighlight = null;
			if (this._waitingHighlight) {
				window.clearTimeout(this._waitingHighlight);
				this._waitingHighlight = null;
			}
			for(var attr in this) {
				/*
				 * Search for scroll elements to be destroyed
				 * If firstElementChild or lastElementChild is null, uninit will crash
				*/
				var elt = this[attr];
				if (elt && elt.IctScroll && elt.firstElementChild && elt.lastElementChild) {
					FXScroll.unInit(elt);
				}
			}

			this.inherited(arguments);
			// Before destroying the UIElement, remove all its i18n handlers to avoid memory leaks.
			ICTouchAPI.i18nServices.unsubscribeAllI18nHandlers(this);
			// Also remove all registered events.
			this._removeAllEvents();
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * Get the name of the class that created the widget.
                 * @ignore
		 * @return {String} the class name.
		 */
		getInstanceOf: function (){
			return this.declaredClass;
		},

		/**
		 * Method for changing the UIElement's value ( if applicable )
                 * @ignore
		 */
		setValue: function () {
			ICTouchAPI.debugServices.info("setValue not defined for "+this.declaredClass, false);
		},

		/* ----------------------------------- Public methods ------------------------------------- */


		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * Clear all timers
		 * @ignore
		 */
		_clearTimers: function() {
			if (this._handlerLongPress) {
				clearTimeout(this._handlerLongPress);
			}
			this._handlerLongPress = null;
		},

		/**
		 * Launch this.funcCallback on mouse down event
		 *  @param {Event} clickEvent mouse down event
		 */
		startHandlerLongPress: function(liNode) {
			// Start the long press timer if a longpress callback is present
			if (typeof this.eventLongPress == "function") {
				this._handlerLongPress = setTimeout(dojo.hitch(this, this.eventLongPress,liNode), 1000);
			}
		},

		/**
		 * Launch this.funcCallbackRelease on mouse up event
		 */
		stopHandlerLongPress: function() {
			this._clearTimers();
		},

		/**
		 * Wait a moment then apply the highlighting effect
                 * @ignore
		 * @return {Boolean} highlight effect will start

		 */
		_startHighlight: function(e, args) {
			if (e && e.currentTarget) {
				// So it works with both dojoAttachPoint and attachEvent method
				var context = (args && args.PARENT_WIDGET) ? args.PARENT_WIDGET : this;
				context._stopHighlight();
				context._handlerStopHighlight = context._subscribeHighlight(e.currentTarget);
				context._domCurrentHighlight = e.currentTarget;
				var _currentTarget = e.currentTarget;

				// Trigger the long press handler
				context.startHandlerLongPress(_currentTarget);

				if(context._enableWaitingHighlight) {
				    var func = function (){
					    dojo.addClass(_currentTarget, context._highlightClass);
				    }
				    context._waitingHighlight = window.setTimeout(func, 100);
				} else {
				    dojo.addClass(_currentTarget, context._highlightClass);
				}
				return true;
			}
			return false;
		},

		/**
		 * Stop the highlighting effect
                 * @ignore
		* @return {Object} the HtmlElement if the user did a click , null in other case
		 */
		_stopHighlight: function(e) {
			if (this._domCurrentHighlight) {
				if (this._waitingHighlight) {
					window.clearTimeout(this._waitingHighlight);
					this._waitingHighlight = null;
				}
				this._unsubscribeHighlight(this._handlerStopHighlight);
				this._handlerStopHighlight = null;
				dojo.removeClass(this._domCurrentHighlight, this._highlightClass);
				var clickedElement = this._domCurrentHighlight;
				this._domCurrentHighlight = null;

				// If the focus is released, stop the long press handling
				this.stopHandlerLongPress();
				return (e && e.type == "mouseup") ? clickedElement : null;
			}
			return null;
		},

		/**
		 * Subscribe to all events that will stop the highlight
		 * @ignore
		 * @return {Object} opaque subscribe handler
		 */
		_subscribeHighlight: function(element) {
			var handler = [];
			handler.push(FXScroll.observeState("start", this, this._stopHighlight));
			handler.push(dojo.connect(element, "mouseup", this, this._stopHighlight));
			handler.push(dojo.connect(element, "mouseleave", this, this._stopHighlight));
			return handler;
		},

		/**
		 * Unsubscribe the handler
		 * @ignore
		 * @param {Object} Object returned by _subscribeHighlight
		 */
		_unsubscribeHighlight: function(handler) {
			if (!handler) {
				return;
			}
			var item;
			// First item is the FXScroll handler
			item = handler.shift();
			FXScroll.unobserveState(item);
			// Rest is dojo connect
			while (handler.length) {
				item = handler.pop();
				dojo.disconnect(item);
			}
		},

		/**
		 * @param {String} type element type
		 * @return {HTMLElement} a dom Element with associated methods for :
		 * @ignore
		 * - mixing in params
		 * - attaching event (so they're referenced and destroyed in the end)
		 * - removing the event
		 * - storing extra information
		 * - appending other nodes
		 */
		newElement : function (type) {
			var elem = type != "documentFragment" ?  document.createElement(type) : document.createDocumentFragment();
			elem.mixin = this._mixin();
			elem.attachEvent = this._attachEvent(elem);
			elem._removeEvent = this._removeEvent(elem);
			elem._infos = {};
			elem._children = [];
			elem._removeMyEvents = this._removeMyEvents(elem);
			elem._removeChildren = function () {
					var l=elem._children.length;
					for (var i =0; i<l; i++) {
							elem._children[i].remove(true);
					}
			};
			elem.append = this.append;
			elem.condAppend = this._condAppend;
			elem.storeInfo = function (infoName, infoValue) {
				this._infos[infoName] = infoValue;
				return infoValue;
			}
			elem.getInfo = function (infoName) {
				return this._infos[infoName] || false;
			}
			elem.remove = function (leaveDom) {
				ICTouchAPI.i18nServices.unsubscribeAllI18nHandlers(this);
				this._removeChildren();
				this._removeMyEvents();
				if (this.parentNode && !leaveDom) {
					this.parentNode.removeChild(this);
				}
				return true;
			}
			return elem;
		},

		/**
		 * Update the widget content, this method may be redefined by each uiElement
		 * @return {boolean} true if a full reload is required
		 * @ignore
		 */
		updateContent : function(params){
			params = null;
			return true;
		},

		/**
		 * @ignore
		 * Call this method with old function name and new one, it will execute it and console warn a message
		 * saying the old one is not to use anymore.
		 */
		_deprecated : function (oldName, newFunc, args) {
			console.warn(this.declaredClass + "." + oldName + " is deprecated, please use " + newFunc + " instead.");
			return this[newFunc].apply(this, args);
		},

		/**
		 * @ignore
		 * Attaches an event and storing it.
		 * Event that'll be removed at UIElement destruction
		 */
		_attachEvent : function () {
			var that = this;

			return function (args, callbackArgs) {

				if (!args) {
					return false;
				} else {
					// Enclose callback
					var callback = function(event) {
						if (typeof that[args.func] === "function") {
							that[args.func].apply(this, [event, callbackArgs]);
						}
					}

					// Store its info
					that._storeEvent(this, {
						"event" : args.event,
						"func" : callback,
						"capture" : args.capture
					});


					callbackArgs = callbackArgs || {};
					// callbackArgs will be given as custom arguments to the
					// encapsulated callback.
					callbackArgs.PARENT_WIDGET = that;

					this.addEventListener(args.event, callback, args.capture);

					return this;
				}

			}
		},

		/**
		 * @ignore
		 * Remove the event
		 */
		_removeEvent : function (to) {
			return function (args) {
				to.removeEventListener(args.event, args.func, args.capture);
				return to;
			}
		},

		/**
	 * @ignore
	 * Mixes in extra params, also auto i18nize strings.
	 */
		_mixin : function () {
			return function (from) {
				for (var i in from) {
					if (from.hasOwnProperty(i)) {
						if (i === "innerHTML" && typeof from[i] === "object" && from[i].isI18Ned) {
							ICTouchAPI.i18nServices.addI18nHandler(this, from[i].fillInnerHTML(this));
						} else if (i === "style") {
							this.setAttribute(i, from[i]);
						} else {
							this[i] = from[i];
						}
					}
				}
				return this;
			}
		},

		/**
		 * @ignore
		 * Appends child to given node if condition is satisfied
		 */
		_condAppend : function  (ifWhat, appendWhat) {
			if (!!(ifWhat)) {
				this.append(appendWhat);
			}
			return this;
		},

		append : function (appendWhat) {
				this.appendChild(appendWhat);
				this._children.push(appendWhat);
		},

		/**
		 * @ignore
		 * Save event in database
		 */
		_storeEvent : function (node, args) {
			this._connectedEvents.push({
				"node" : node,
				"args" : args
			});
		},

		/**
		 * @ignore
		 * Removes all widget's events
		 */
		_removeAllEvents : function () {
			var event;
			while (event = this._connectedEvents.pop()) {
				event.node._removeEvent(event.args);
			}
		},

		/**
		 * @ignore
		 * Remove specifics elem's events
		 */
		_removeMyEvents : function (elem) {
			var that = this;
			return function () {
				for (var i=0; i<that._connectedEvents.length; i++) {
					if (that._connectedEvents.hasOwnProperty(i)) {
						if (that._connectedEvents[i]["node"] === elem) {
							elem._removeEvent(that._connectedEvents[i]["args"]);
							that._connectedEvents.splice(i, 1);
							i--;
						}
					}
				}
			}
		}

	});
/**
 * @class UIElements.AppBar.AppBarWidgetBase
 * @extends UIElements._base
 * Abstract Class. Do not use it directly.
 */
dojo.provide("UIElements.AppBar.AppBarWidgetBase");
dojo.declare("UIElements.AppBar.AppBarWidgetBase",
	[UIElements._base,
	dijit._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Action buttons of the current widget
                 * @ignore
		 * @property
		 * @type Array
		 */
		arrActionButtons : null,
		/**
		 * Action buttons of system
                 * @ignore
		 * @property
		 * @type Array
		 */
		arrStaticButtons : null,
		/**
		 * Action buttons of other webApp
                 * @ignore
		 * @property
		 * @type Array
		 */
		arrOtherWebAppButtons : null,
		/**
		 * webapp name
                 * @ignore
		 * @property
		 * @type String
		 */
		webAppName :"",
		/**
		 * widget name
                 * @ignore
		 * @property
		 * @type String
		 */
		widgetName :"",

                /**
                 * @ignore
                 */
		arrSkeleton : [],

		/**
		 * Disable scroll
         * @ignore
		 * @property
		 * @type String
		 */
		boolDisableScroll : false,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		domLastActionButton : null,
		/**
		 * @ignore
		 */
		domButtons : null,

		/* ------------------------------------ Constructor --------------------------------------- */

		constructor : function(){
			// initialise the attributs of instances
			this.arrActionButtons = [];
			this.arrOtherWebAppButtons = [];
			this.arrStaticButtons = [];
		},

		postMixInProperties : function(){
			this.initArrayButtons();
		},

		postCreate : function(){
			if( this.domButtons ) {
				ICTouchAPI.tools.removeWhitespaceNodes(this.domButtons);
			}
		},

		/**
		 * Destroy
		 */
		destroy: function() {
			this.removeAllActionButtons();
			this.inherited(arguments);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * Returns a button of the AppBar according to his name
		 * @param {String} name the name of the button to be retrieved
		 */
		getButton: function(name) {
			for(var i in this.arrActionButtons)
			{
				if(name == this.arrActionButtons[i].strButtonName){
					return this.arrActionButtons[i];
				}
			}

			for(var i in this.arrOtherWebAppButtons)
			{
				if(name == this.arrOtherWebAppButtons[i].strButtonName){
					return this.arrOtherWebAppButtons[i];
				}
			}

			for(var i in this.arrStaticButtons)
			{
				if(name == this.arrStaticButtons[i].strButtonName){
					return this.arrStaticButtons[i];
				}
			}

			return null;
		},

		/**
		 * Returns all action buttons of the AppBar widget
                 * @return {UIElements.AppButton.AppButtonControl[]} The list of buttons
		 */
		getAllButtons: function() {
			return this.arrActionButtons;
		},

		/**
		 * Returns all others buttons of the AppBar widget
                 * @return {UIElements.AppButton.AppButtonControl[]} The list of buttons
		 */
		getAllOtherWebappButtons: function() {
			return this.arrOtherWebAppButtons;
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		 * initialise the arrStaticButtons and arrOtherWebAppButtons arrays in requesting
		 * the buttons to appBarServices
                 * @ignore
		 */
		initArrayButtons : function(){
			// Reinitialize to zero element
			this.arrStaticButtons = [];
			this.arrOtherWebAppButtons = [];
			// Get the Static buttons
			var arrayOfButtons = ICTouchAPI.AppBarServices.getStaticButtons(this.webAppName,this.widgetName);
			for (var button in  arrayOfButtons){
				this.arrStaticButtons.push(arrayOfButtons[button]);
			}
			// Get the otherWebApp buttons
			arrayOfButtons =  ICTouchAPI.AppBarServices.getOtherWebAppButtons(this.webAppName,this.widgetName);
			for ( button in  arrayOfButtons){
				this.arrOtherWebAppButtons.push(arrayOfButtons[button]);
			}
		},

		/**
		 * Only remaining for backward compatibility
                 * @ignore
		 */
		updateAppBar: function() {

		},

		/**
		 * Adds an action ( normal ) button into this AppBar widget
		 * @param {UIElements.AppButton.AppButtonControl} objButton UI Button to add
		 */
		addActionButton : function(objButton, boolAvoidPush) {
			// Adopt it so chrome doesn't throw an exception
			this.domNode.ownerDocument.adoptNode(objButton.domNode)
			// Is this the first button added ?
			if( this.domLastActionButton == null ) {
				objButton.placeAt(this.domButtons, "first");
			}
			else {
				objButton.placeAt(this.domLastActionButton, "after");
			}
			this.domLastActionButton = objButton.domNode;

			if( boolAvoidPush == false || typeof boolAvoidPush == "undefined" ) {
				this.arrActionButtons.push(objButton);
			}

			objButton.setAppBar(this);
			this._checkArrows();
		},


                /**
                 * @ignore
                 */
		addAppButton : function(objButton) {
			if(typeof objButton != "string"){
				if(this.arrSkeleton.indexOf(objButton.getButtonName())!=-1){
					// Adopt it so chrome doesn't throw an exception
					this.domNode.ownerDocument.adoptNode(objButton.domNode)
					this.arrActionButtons[objButton.getButtonName()] = objButton;
					objButton.setAppBar(this);
				}
			}else{
				if(this.arrSkeleton.indexOf(objButton)!=-1){
					this.arrActionButtons[objButton] = objButton;
				}
			}
		},

                /**
                 * @ignore
                 */
		rebuildDom : function(){
			//this.destroyDom();
			this.buildDom();
		},

                /**
                 * @ignore
                 */
		destroyDom : function() {
			dojo.empty(this.domButtons);
			dojo.empty(this.domLastActionButton);
		},

                /**
                 * @ignore
                 */
		buildDom : function() {
			var boolFirst = true;
			for(var i in this.arrSkeleton){
				var _currentBtn = this.getButton(this.arrSkeleton[i]);
				if(_currentBtn){
					if(boolFirst){
						_currentBtn.placeAt(this.domButtons, "first");
						boolFirst = false;
					}else {
						_currentBtn.placeAt(this.domLastActionButton, "after");
					}
					this.domLastActionButton = _currentBtn.domNode;
				}
			}
			this._checkArrows();

			/*for(var i in this.arrActionButtons){
				var _currentBtn = this.arrActionButtons[i];
				if(typeof _currentBtn != "string"){
					console.error(_currentBtn.getButtonName());
					if(boolFirst){
						_currentBtn.placeAt(this.domButtons, "first");
						boolFirst = false;
					}else {
						_currentBtn.placeAt(this.domLastActionButton, "after");
					}
					this.domLastActionButton = _currentBtn.domNode;
				}
			}
			this._checkArrows();*/
		},

                /**
                 * @ignore
                 */
		hideButton : function(btnName){
			this.arrActionButtons[btnName].hide();
		},

                /**
                 * @ignore
                 */
		showButton : function(btnName){
			this.arrActionButtons[btnName].show();
		},

                /**
                 * @ignore
                 */
		getSkeleton : function(){
			return this.arrSkeleton;
		},

		/**
		 * Removes all actions buttons contained in the AppBar widget
		 */
		removeAllActionButtons : function() {
			var obj;
			for(var i=0; i<this.arrActionButtons.length; ++i) {
				obj = this.arrActionButtons[i];
				obj.destroy();
			}
			this.arrActionButtons = [];
			this.domLastActionButton = null;
		},

		/**
		 * Removes an action button contained in the AppBar widget
		 * the actionButton array.
		 * @param {UIElements.AppButton.AppButtonControl} objButton
		 */
		removeActionButton : function(objButton){
			var lengthArrActionButtons = this.arrActionButtons.length;
			for(var i = 0; i<lengthArrActionButtons;i++){
				if(this.arrActionButtons[i]===objButton){
					this.arrActionButtons.splice(i,1)
					objButton.destroy();
					this._checkArrows();
					return;
				}
			}
		},

		/**
		 * Adds a list of action button to the AppBar widget
		 * @param {UIElements.AppButton.AppButtonControl[]} arrayButtons
		*/
		addActionButtonArray: function(arrayButtons){
			dojo.forEach(arrayButtons, function(button){
				this.addActionButton(button);
			}, this);
		},

		/**
		 * Used by the contained buttons to notify the appBar that something has changed
		 * Should only be used by appButtons
                 * @ignore
		 */
		buttonChanged: function() {
			this._checkArrows();
		},

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 * Empty function defined in 8082
		 */
		_checkArrows: function() {

		}
	});
/**
 * @class UIElements.AppBar.AppBarWidget
 * @extends UIElements.AppBar.AppBarWidgetBase
 * This class represents the graphical user interface of an AppBar widget.
 *
 * Each view contains an AppBar widget that can be used to propose action buttons for the user so than he can interact with your web application.
 *
 * The web application requests the {ICTouchAPI.AppBarServices} to provide the action bar corresponding to its widget. In other words,no webapp has to instanciate an AppBar.
 * As soon as the webapp gets the AppBar, the webapp can add/remove some action buttons.
 *
 * When the AppBar of a view is building or updating, the AppBar requests automatically the static buttons and the others buttons attached to this view.
 *
 * Here is an example of adding a new action button in the AppBar widget:
 *
 *
 *     // Add an action button to an AppBar widget
 *     postCreate : function() {
 *
 *          // Initialize the callback function
 *          var func = dojo.hitch(this, this.onButtonPress);
 *
 *          // Create the Button
 *          var saveBtn = new UIElements.AppButton.AppButtonControl({
 *              strButtonName:  "saveButton",       // Button identifier
 *              strButtonLabel: "Save",             // Label displayed
 *              strButtonIcon:  "saveImgCss",       // CSS class of the image button
 *              callback:       func                // Callback function called
 *          });
 *
 *          // Get the AppBar Widget
 *          var appBar = ICTouchAPI.AppBarServices.getAppBar("webappName", "viewName");
 *
 *          // Add the button
 *          appBar.addActionButton(saveBtn);
 *     },
 *
 *     // Define the callback function when the user presses on the save button
 *     onButtonPress : function() {
 *
 *          // Do something
 *          ...
 *
 *     },
 *
 */
dojo.require("UIElements.AppBar.AppBarWidgetBase");
dojo.provide("UIElements.AppBar.AppBarWidget");
dojo.declare("UIElements.AppBar.AppBarWidget",
	UIElements.AppBar.AppBarWidgetBase,
	{

		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		domStaticButtons: null,
		/**
		 * @ignore
		 */
		domAppBarLeft	: null,
		/**
		 * @ignore
		 */
		domArrowLeft	: null,
		/**
		 * @ignore
		 */
		domArrowRight	: null,
		/**
		 * @ignore
		 * 0: nowhere, 1: utmost left, 2: somewhere in the middle, 3: utmost right
		 */
		_intLastArrow	: 0,
		/**
		 * @ignore
		 */
		_boolArrow		: false,
		/**
		 * @ignore
		 * The path to the template is defined here
		 */
		templatePath: dojo.moduleUrl("UIElements.AppBar","templates/AppBar8082.html"),
		/**
		 * @ignore
		 */
		cssFile    : ["base.css","8082.css"],

		/* ------------------------------------ Constructor --------------------------------------- */

		postCreate : function(){
			var i;

			// place the buttons into this bar
			for( i=0; i<this.arrStaticButtons.length; i++){
				this.addStaticButton(this.arrStaticButtons[i], true);
			}

			for( i=0; i<this.arrActionButtons.length; i++) {
				this.addActionButton(this.arrActionButtons[i], true);
			}

			for( i=0; i<this.arrOtherWebAppButtons.length; i++) {
				this.addOtherWebAppButton(this.arrOtherWebAppButtons[i], true,true);
			}

			if( this.domButtons != null && !this.boolDisableScroll) {
				this._checkArrows();
				FXScroll.init({
					"handler"	: this.domButtons,
					"easing"	: true,
					"scrollType" : "horizontal",
					// Not used for pagination but more as an "onscroll" event
					"pagination": {
						"callback"	: this._updateArrows,
						"context"	: this
					}
				});
			}

			this.inherited(arguments);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		 * Add a static button into this bar. Only the 'homepage' button should be added as static.
		 * @param {UIElements.AppButton.AppButtonControl} objButton UI Button to add
                 */
		addStaticButton : function(objButton, boolAvoidPush) {
			// Adopt it so chrome doesn't throw an exception
			this.domNode.ownerDocument.adoptNode(objButton.domNode)

			objButton.placeAt(this.domStaticButtons, "last");
			dojo.addClass(this.domNode, "hasStatic");

			if( boolAvoidPush == false || typeof boolAvoidPush == "undefined" ) {
				this.arrStaticButtons.push(objButton);
			}

			objButton.setAppBar(this);
			this._checkArrows();
		},

		/**
		 * Remove a static button from this bar. Only the 'homepage' button should be removed
		 * @param {UIElements.AppButton.AppButtonControl} objButton UI Button to remove
		 */
		removeStaticButton : function(objButton) {
			for(var i in this.arrStaticButtons) {
				if(this.arrStaticButtons[i]==objButton) {
					this.arrStaticButtons.splice(i,1);
				}
			}

			this._checkArrows();
		},

		/**
		 * Add an action ( from an other webapp ) button into this bar
		 * @param {UIElements.AppButton.AppButtonControl} objButton UI Button to add
                 */
		addOtherWebAppButton : function(objButton, boolAvoidPush, boolRebuild) {
			// Adopt it so chrome doesn't throw an exception
			this.domNode.ownerDocument.adoptNode(objButton.domNode)

			objButton.placeAt(this.domButtons, "last");

			if( boolAvoidPush == false || typeof boolAvoidPush == "undefined" ) {
				this.arrOtherWebAppButtons.push(objButton);
			}
			if(boolRebuild){
				this.rebuildDom();
			}
			objButton.setAppBar(this);
			this._checkArrows();
		},

		/**
		 * Remove an action ( from an other webapp ) button from this bar
		 * @param {UIElements.AppButton.AppButtonControl} objButton UI Button to remove
		 */
		removeOtherWebAppButton : function(objButton) {
			for(var i in this.arrOtherWebAppButtons) {
				if(this.arrOtherWebAppButtons[i]==objButton) {
					this.arrOtherWebAppButtons.splice(i,1);
				}
			}
			this._checkArrows();
		},

		/**
		 * change the scroll capture
                 * @ignore
		 * @param {Object} capture
		 */
		changeScrollCapture : function(capture) {
			this.scroll.changeCapture(capture);
		},

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 */
		_updateArrows: function() {
			if( !this._boolArrow ) {
				return 0;
			}
			var status = 2;

			var s = this.domButtons;
			// If the buttons are scrolled to the left most.
			//   This condition should be first so when the appBar is offscreen both condition can be true
			//   but the scroll is on the left
			if (s.scrollLeft < 5) {
				status = 1;
			// If the buttons are scrolled to the right most
			} else if (s.scrollLeft + 5 >= s.scrollWidth - s.clientWidth) {
				status = 3
			}

			if( this._intLastArrow !== status ) {
				dojo.toggleClass(this.domArrowLeft, "show", status > 1);
				dojo.toggleClass(this.domArrowRight, "show", status < 3);
				this._intLastArrow = status;
			}
			return 0;
		},

		/**
		 * @ignore
		 */
		_checkArrows: function() {
			//var i;
			// Number buttons before we need to scroll ( with and without static buttons )
			var max = this.arrStaticButtons.length > 0 ? 5 : 8;
			// Number of visible buttons
			var nbr = 0;
			for (var i in this.arrActionButtons){
				if(typeof this.arrActionButtons[i] != "string"){
					if(this.arrActionButtons[i].getVisibility()){
						nbr+=1;
					}
				}
			}
			for (var j in this.arrOtherWebAppButtons){
				if(typeof this.arrOtherWebAppButtons[j] != "string"){
					if(this.arrOtherWebAppButtons[j].getVisibility()){
						nbr+=1;
					}
				}
			}
			if( nbr > max ) {
				if(!this._boolArrow){
					this._boolArrow = true;
				}
				// 2 means somewhere in the middle. Otherwise, we do not have to test
				if(this._intLastArrow !== 1){
					// Call this to show the arrows
					this._updateArrows();
				}
			}
			else {
				this._intLastArrow = 0;
				this._boolArrow = false;
				dojo.removeClass(this.domArrowLeft, "show");
				dojo.removeClass(this.domArrowRight, "show");
			}
		}

	});
/**
 * @class UIElements.AppButton.ButtonControlBase
 * @extends UIElements._base
 * Abstract Class - Do not use it directly.
 */
dojo.provide("UIElements.AppButton.ButtonControlBase");
dojo.declare("UIElements.AppButton.ButtonControlBase",
	[UIElements._base, dijit._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Status of button (either onDisplay,offDisplay or unavailable)
		 * @cfg {String} strButtonStatus
		 */
		strButtonStatus : null,
		/**
		 * Name of the button
		 * @cfg {String} strButtonName
		 */
		strButtonName   : "",

		/**
		 * Status text
		 * @property
		 * @type String
		 * @ignore
		 */
		strStatusText   : "",

		/**
		 * Status icon (either 'notif-on', 'notif-off', or '')
		 * @cfg {String} strStatusIcon
		 */
		strStatusIcon   : "",

		/**
		 * Button label
		 * @cfg {String} strButtonLabel
		 */
		strButtonLabel	: "",

		/**
		 * Button icon. CSS style that embeds your image
		 * @cfg {String} strButtonIcon
		 */
		strButtonIcon   : "",

		/**
		 * Is the button embedded inside a popup
		 * @cfg {String} boolPopup
		 */
		boolPopup		: false,

		/**
		 * Call back function
		 * @cfg {Function} callback
		 */
		callback		: null,

		/**
		 * call back function for long press
		 * @property
		 * @type Function
		 */
		callbackLong	: null,
		
		/**
		 * feedback
		 * @ignore
		 * @property
		 * @type Function
		 */
		myFeedback		: null,
		/**
		 * scroll object
		 * @ignore
		 * @property
		 * @type UIEffects.Scroll
		 */
		objScroll		: null,
		/**
		 * Array of showing events
		 * @ignore
		 * @property
		 * @type Array
		 */
		eventsShowing	: [],
		/**
		 * Array of hiding events
		 * @ignore
		 * @property
		 * @type Array
		 */
		eventsHiding	: [],
		/**
		 * Array of Tinting events
		 * @ignore
		 * @property
		 * @type Array
		 */
		eventsTinting	: [],
		/**
		 * Array of capabilities which impact this button
		 * @ignore
		 * @property
		 * @type Array
		 */
		arrCapabilities : [],
		/**
		 * Array of capabilities handler related to this button
		 * @ignore
		 * @property
		 * @type Array
		 */
		arrCapabilitiesHandler : [],
		/**
		 *  Array of instances handler related to this button
		 *  @ignore
		 * @property
		 * @type Array
		 */
		arrCapabilitiesInstancesHandler : [],
		/**
		 * mode icon
		 * @ignore
		 * @property
		 * @type String
		 */
		strModeIcon   : null,

		/* --------------------------------- Private attributes ----------------------------------- */

		/*
		 * Hide / show / grey status depending of the capabilities the button subscribed to.
		 * A webapp can not show a button that is hidden by capability.
		 * @ignore
		 */
		_capabilityStatus : null,

		/*
		 * Hide / show / grey status depending of custom management of the button (by webapps).
		 * A webapp can hide / grey a button that is shown by the capabalities.
		 * @ignore
		 */
		_customStatus : null,

		/*
		 * List of available status
		 * @ignore
		 */
		_lstStatus : {
			ON : "onDisplay",
			OFF : "offDisplay",
			GREY : "unavailable"
		},

		/**
		 * @ignore
		 */
		domStatus		: null,
		/**
		 * @ignore
		 */
		domButton		: null,
		/**
		 * @ignore
		 * true if there is already an onmousedown on element
		 * useful in order to interpret an onclick after an onmouseup
		 */
		_eventOnMouseDownDone : false,
		/**
		 * @ignore
		 */
		_callbackFunc	: null,
		/**
		 * @ignore
		 */
		_handlerLongPress : null,
		/**
		 * @ignore
		 */
		_arrLastCapabilityCounter : null,
		/**
		 * @ignore
		 */
		attributeMap: {
			strButtonLabel: {
				node: "domLabel",
				type: "innerHTML"
			},
			strStatusText: {
				node: "domStatus",
				type: "innerHTML"
			}
		},

		_highlightClass: "active",
		/**
		 * @ignore
		 */
		_boolMoveable :true,//boolean in order to know it the progkey webapp can move the button or not

		/* ------------------------------------ Constructor --------------------------------------- */

		constructor : function() {
			//this.myFeedback = ICTouchAPI.feedbackServices.initFeedback("click");
			this.eventsHiding = [];
			this.eventsShowing = [];
			this.eventsTinting = [];

			// Initialize capabilities counters.
			this._arrLastCapabilityCounter = [];
			for(var i = 0 ; i > this.arrCapabilities.length ; i++) {
				this._arrLastCapabilityCounter[this.arrCapabilities[i].capability] = 0;
			}

			// Initialize capability and custom status.
			this._capabilityStatus = this._lstStatus.ON;
			this._customStatus = this._lstStatus.ON;
			this.strButtonStatus = this._lstStatus.ON;
		},


		postCreate : function() {
			if(this.strButtonStatus && this.domNode)
				dojo.addClass(this.domNode, this.strButtonStatus);
			if(this.strButtonIcon && this.domIcon)
				dojo.addClass(this.domIcon, this.strButtonIcon);
			if (this.strButtonName || this.strButtonName == 0){
				this.domNode.setAttribute("buttonname",this.strButtonName);
			}
			else {
				console.warn("Button '" + this.strButtonLabel + "' doesn't have a strButtonName!");
			}
			if (this.strStatusIcon){
				dojo.addClass(this.domStatus, "withStatusIcon");
				dojo.addClass(this.domStatus, this.strStatusIcon);
			}

			this.subscribeCapabilities();
		},

		destroy : function() {
			this.removeAllEvents();

			this.unSubscribeCapabilities();

			//delete this.myFeedback;
			delete this.callback;

			this.inherited(arguments);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * Returns the name of the button
		 * @return {String} name of button
		 */
		getButtonName : function(){
			return this.strButtonName;
		},

		/**
		 * Get visibility status of this button
		 * @return {Boolean} the visibility of this button
		 */
		getVisibility: function() {
			return this.strButtonStatus == "onDisplay" || this.strButtonStatus == "unavailable";
		},

		/**
		* Updates the label
        * @param {String} strButtonLabel new button label
        */
		setButtonLabel : function (strButtonLabel) {
			this.attr("strButtonLabel", strButtonLabel);
		},

		/**
		 * Updates the text status by the given argument
		 * @param {String} strStatusText status of button in text (on or off)
		 * @ignore
		 */
		setStatusText: function(strStatusText){
			dojo.removeClass(this.domStatus, this.strStatusText);
			this.strStatusText = strStatusText;
			dojo.addClass(this.domStatus, this.strStatusText);
		},

		/**
		 * Updates the icon status by the given argument
		 * @param {String} strStatusIcon status of button led (either notif-on or notif-off)
		 */
		setStatusIcon    : function(strStatusIcon){
			this.attr("strStatusIcon", strStatusIcon);
		},

		/**
		 * setModeIcon replaces the icon mode by the given argument
		 * @ignore
		 * @param {String} strModeIcon mode of button in icon
		 */
		setModeIcon : function(strModeIcon){
			this.attr("strModeIcon", strModeIcon);
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		 * removes mode icon
		 * @ignore
		 */
		removeModeIcon : function(){
			this.attr("strModeIcon", "");
		},

		/**
		 * clone method clones this in order to make a button with the same
		 * behavior. The clone is displayed in action bar of widgetName of
		 * webAppName.
		 * This method should be only called by the framework, especially appBarServicesis.
		 * It is useful when the webapp requires to display the same button in several action bars.
		 * @ignore
		 * @param {String} strWebAppName name of webapp
		 * @param {String} strWidgetName name of widget
		 */
		subscribeCapabilities:function() {
			var length = this.arrCapabilities.length;

			var subscribeHandler = null;
			this.arrCapabilitiesHandler = [];
			this.arrCapabilitiesInstancesHandler = [];
			for (var i = 0; i < length; i++) {

				var objCap = this.arrCapabilities[i];
				if (objCap.instance == undefined) {
					subscribeHandler = ICTouchAPI.CapabilityServices.subscribeToCapability(objCap.module, objCap.capability, this, this._capabilityChanged);
					this.arrCapabilitiesHandler[objCap.module + objCap.capability] = subscribeHandler;
				}
				else if (objCap.instance >= 0) {
					subscribeHandler = ICTouchAPI.CapabilityServices.subscribeToInstance(objCap.module, objCap.instance, objCap.capability, this, this._capabilityChanged);
					this.arrCapabilitiesInstancesHandler[objCap.module + objCap.capability + objCap.instance] = subscribeHandler;
				}
			}
		},

		/**
		 *  Disconnect an capability handler and reconnect an instance handler
		 *  @ignore
		 *  @param {String} strModule the module whch has to reconnect its capability
		 *  @param {Number} intInstance the instance to be econnected
		 */
		reconnectCapabilitiesToInstance:function(strModule, intInstance) {
			ICTouchAPI.debugServices.info("UIElements.AppButton.ButtonControlBase - reconnectCapabilitiesToInstance $ input function params | (button " + this.strButtonName + ") | strModule: " + strModule + " | intInstance: " + intInstance);
			var length = this.arrCapabilities.length;
			var subscribeHandler = null;
			var noResubscribeToInstance = false; // bool used to avoid unecessary treatment (do not unsubscribe/resubscribe the instance if it is already subscribed)
			for (var i = 0; i < length; i++) {
				var objCap = this.arrCapabilities[i];
				if (objCap.module == strModule && objCap.instance != undefined) {
					// remove all subsciption to the given capability whatever the instance is
					for (var j in this.arrCapabilitiesInstancesHandler) {
						if (j.indexOf(objCap.module + objCap.capability) !== -1) {
							if (j == objCap.module + objCap.capability + intInstance) {
								ICTouchAPI.debugServices.debug("UIElements.AppButton.ButtonControlBase - reconnectCapabilitiesToInstance / j: " + j + ", button " + this.strButtonName + " do not need to unsubscribe to intInstance " + intInstance + " because it is already susbribed");
								// instance to reconnect is already subscribed, do not unsubscribe/resubscribe it
								noResubscribeToInstance = true;
							}
							else {
								ICTouchAPI.debugServices.debug("UIElements.AppButton.ButtonControlBase - reconnectCapabilitiesToInstance / call disconnectHandler for " + this.strButtonName + " | objCap.instance: " + objCap.instance + " (disconnect j: " + j + ")");
								subscribeHandler = this.arrCapabilitiesInstancesHandler[j];
								ICTouchAPI.CapabilityServices.disconnectHandler(subscribeHandler);
								delete (this.arrCapabilitiesInstancesHandler[j]);
							}
						}
					}
					objCap.instance = intInstance;
					if (objCap.instance >= 0 && !noResubscribeToInstance) {
						ICTouchAPI.debugServices.debug("UIElements.AppButton.ButtonControlBase - reconnectCapabilitiesToInstance / call subscribeToInstance for " + this.strButtonName + " | objCap.instance: " + objCap.instance);
						subscribeHandler = ICTouchAPI.CapabilityServices.subscribeToInstance(objCap.module, objCap.instance, objCap.capability, this, this._capabilityChanged);
						this.arrCapabilitiesInstancesHandler[objCap.module + objCap.capability + objCap.instance] = subscribeHandler;
					}
					else {
						ICTouchAPI.debugServices.debug("UIElements.AppButton.ButtonControlBase - reconnectCapabilitiesToInstance / noResubscribeToInstance: " + noResubscribeToInstance + ", button " + this.strButtonName + " do not need to resubscribe to objCap.instance " + objCap.instance + " because it is already susbribed");
					}
				}
			}
		},

		/**
		 * unsubscribe all capabilities
		 * @ignore
		 */
		unSubscribeCapabilities: function() {
			var subscribeHandler;

			for(var i in this.arrCapabilitiesHandler ) {
				subscribeHandler = this.arrCapabilitiesHandler[i];
				ICTouchAPI.CapabilityServices.disconnectHandler(subscribeHandler);
				delete this.arrCapabilitiesHandler[i];
			}

			for(var j in this.arrCapabilitiesInstancesHandler ) {
				subscribeHandler = this.arrCapabilitiesInstancesHandler[j];
				ICTouchAPI.CapabilityServices.disconnectHandler(subscribeHandler);
				delete this.arrCapabilitiesInstancesHandler[j];
			}
		},

		/**
		 * Set capabilityStatus to "OFF"
		 */
		hide: function(){
			this._customStatus = this._lstStatus.OFF;
			this._applyStatus();
		},

		/**
		 * Set capabilityStatus to "ON"
		 */
		show: function(){
			this._customStatus = this._lstStatus.ON;
			this._applyStatus();
		},

		/**
		 * Set capabilityStatus to "GREY"
		 */
		greyTint: function(){
			this._customStatus = this._lstStatus.GREY;
			this._applyStatus();
		},

		/**
		 * addEventsShowing method adds the events given in arguments
		 * which shows the button
		 * @ignore
		 * @param {String[]}    list of event names
		 */
		addEventsShowing: function(){
			for(var i=0;i<arguments.length;i++){
				var handle = dojo.subscribe(arguments[i], this, this.show);
				// note: with this.show(), show() is running
				// when subscribe occurs write only
				// the name of method (without ())

				// save the handle in the eventsShowing array
				this.eventsShowing[arguments[i]]=handle;
			}
		},

		/**
		 * removeEventsShowing method removes the events given in arguments
		 * which shows the button
		 * @ignore
		 * @param {String[]} list of event names
		 */
		removeEventsShowing: function(){
			for(var i=0;i<arguments.length;i++){
				// get the handle from the eventsShowing array
				// before unsubscribing
				dojo.unsubscribe(this.eventsShowing[arguments[i]]);
				// delete the reference of handle
				delete (this.eventsShowing[arguments[i]]);
			}
		},

		/**
		 * addEventsHiding method adds the events given in arguments
		 * which hides the button
		 * @ignore
		 * @param {String[]} list of event names
		 */
		addEventsHiding: function(){
			for(var i=0;i<arguments.length;i++){
				var handle = dojo.subscribe(arguments[i], this, this.hide);
				// save the handle in the eventsHiding array
				this.eventsHiding[arguments[i]]=handle;
			}
		},

		/**
		 * removeEventsHiding method removes the events given in arguments
		 * which hides the button
		 * @ignore
		 * @param {String[]} list of event names
		 */
		removeEventsHiding: function(){
			for(var i=0;i<arguments.length;i++){
				dojo.unsubscribe(this.eventsHiding[arguments[i]]);
				delete (this.eventsHiding[arguments[i]]);
			}
		},

		/**
		 * addEventsGreyTinting method adds the events given in arguments
		 * which greytints the button
		 * @ignore
		 * @param {String[]} list of event names
		 */
		addEventsGreyTinting: function(){
			for(var i=0;i<arguments.length;i++){
				var handle = dojo.subscribe(arguments[i], this, this.greyTint);
				// save the handle in the eventsTinting array
				this.eventsTinting[arguments[i]]=handle;
			}
		},

		/**
		 * removeEventsGreyTinting method removes the events given in arguments
		 * which greytints the button
		 * @ignore
		 * @param {String[]}    list of event names
		 */
		removeEventsGreyTinting: function(){
			for(var i=0;i<arguments.length;i++){
				dojo.unsubscribe(this.eventsTinting[arguments[i]]);
				delete (this.eventsTinting[arguments[i]]);
			}
		},

		/**
		 * removeAllEvents method removes all events currently subscribed
		 * @ignore
		 */
		removeAllEvents: function(){
			var i;
			for (i in this.eventsTinting) {
				dojo.unsubscribe(this.eventsTinting[i]);
				delete (this.eventsTinting[i]);
			}
			for (i in this.eventsHiding) {
				dojo.unsubscribe(this.eventsHiding[i]);
				delete (this.eventsHiding[i]);
			}
			for (i in this.eventsShowing) {
				dojo.unsubscribe(this.eventsShowing[i]);
				delete (this.eventsShowing[i]);
			}
		},



		/* --------------------------------- Private Methods -------------------------------------- */

		_startHighlight: function() {
			this.boolLongPress = false;
			// Launch default highlight and get its status
			var activated = this.inherited(arguments);
			// Add the active class on the icon if necessary
			dojo.toggleClass(this.domIcon, "active", activated);
			if (typeof this.callbackLong == "function") {
				this._handlerLongPress = setTimeout(dojo.hitch(this, this._actionLongpress), 500);
			}
		},
		
		_actionLongpress : function() {
			// Longpress has been triggered don't callback on mouseup
			this.boolLongPress = true;
			this._handlerLongPress = null;
			this.callbackLong(this.strButtonName);
		},
		
		_stopHighlight: function() {
			// Launch default stopHighlight and check if a click was issued
			var clicked = this.inherited(arguments);
			dojo.removeClass(this.domIcon, "active");
			if (clicked && this.strButtonStatus === "onDisplay" && !this.boolLongPress) {
				this._callbackFunc(this.strButtonName);
			}
			
			if (this._handlerLongPress) {
				clearTimeout(this._handlerLongPress);
				this._handlerLongPress = null;
			}
		},
		
		/**
		 * @ignore
		 */
		_setStrStatusIconAttr : function(value) {
			if(value && value.length){
				value = value + "-" + this.intStatusIconSize;
				if(this.strStatusIcon != value){
					dojo.toggleClass(this.domStatus, "withStatusIcon", value != "");
					dojo.removeClass(this.domStatus, this.strStatusIcon);
					dojo.addClass(this.domStatus, value);
					this.strStatusIcon = value;
				}
			}
		},

		/**
		 * @ignore
		 */
		_setStrModeIconAttr : function(value) {
			if(this.strModeIcon && this.strModeIcon != ""){
				dojo.removeClass(this.domMode, this.strModeIcon+"-32");
				dojo.removeClass(this.domMode, "modeOn");
			}
			if(value && value.length){
				dojo.addClass(this.domMode, value + "-32");
				dojo.addClass(this.domMode, "modeOn");
				this.strModeIcon = value;
			}
		},

		/**
		 * @ignore
		 */
		_setCallbackAttr : function(callback) {
			this._callbackFunc = ICTouchAPI.tools.callbackToHitch(callback);
		},


		/**
		 * @ignore
		 * Function is called when a capability related to this button has changed
		 * The button is set visible and active only when all capabilities associated with it are AVAILABLE
		 * If only one capability is UNAVAILABLE, button is hidden
		 * If at least one is TEMP_UNAVAILABLE or UNKNOWN and all other are AVAILABLE, button is grey tinted
		 */
		_capabilityChanged : function(capability, status, instance, intCapabilityCounter) {
			ICTouchAPI.debugServices.info("UIElements.AppButton.ButtonControlBase - _capabilityChanged $ input function params | capability: " + capability + " | status: " + status + " | instance: " + instance + " | intCapabilityCounter: " + intCapabilityCounter);
			if(intCapabilityCounter){
				if(!this._arrLastCapabilityCounter[capability] || intCapabilityCounter > this._arrLastCapabilityCounter[capability]){
					this._arrLastCapabilityCounter[capability] = intCapabilityCounter;
				} else {
					ICTouchAPI.debugServices.warning("UIElements.AppButton.ButtonControlBase - _capabilityChanged / Do not treat capability " + capability + " with instance " + instance + " because a more recent capability value was already treated. Old number : " + this._arrLastCapabilityCounter[capability] + " , new number : " +intCapabilityCounter);
					return;
				}
			}
			var newState = 1;	// show
			for (var i in this.arrCapabilities) {
				var objCap = this.arrCapabilities[i];
				var capstate;
				if (instance != undefined)
					capstate = ICTouchAPI.CapabilityServices.getCapability(objCap.module, instance, objCap.capability);
				else
					capstate = ICTouchAPI.CapabilityServices.getCapability(objCap.module, objCap.capability);
				if (capstate == ICTouchAPI.CapabilityServices.UNAVAILABLE) {
					newState = 0;	// hide
					break;	// If unavailable we can break the loop
				}
				else if (capstate != ICTouchAPI.CapabilityServices.AVAILABLE)
					newState = 2;	// grey tinted
			}
			switch (newState) {
				case 1:
					this._capabilityShow();
					break;
				case 0:
					this._capabilityHide();
					break;
				default:
					this._capabilityHide();
					break;
			}
		},

		/**
		 * Shows/hides/tints in grey the button according to capabilityStatus and customStatus
		 * @ignore
		 */
		_applyStatus: function(){
			if(this._capabilityStatus == this._lstStatus.OFF || this._customStatus == this._lstStatus.OFF){
				this._hide();
			} else if(this._capabilityStatus == this._lstStatus.GREY || this._customStatus == this._lstStatus.GREY) {
				this._greyTint();
			} else {
				this._show();
			}
		},

		/**
		 * Set capabilityStatus to "OFF"
		 *  @ignore
		 */
		_capabilityHide: function(){
			this._capabilityStatus = this._lstStatus.OFF;
			this._applyStatus();
		},

		/**
		 * Set capabilityStatus to "ON"
		 *  @ignore
		 */
		_capabilityShow: function(){
			this._capabilityStatus = this._lstStatus.ON;
			this._applyStatus();
		},

		/**
		 * Set capabilityStatus to "GREY"
		 * @ignore
		 */
		_capabilityGreyTint: function(){
			this._capabilityStatus = this._lstStatus.GREY;
			this._applyStatus();
		},

		/**
		 * Hides the button
		 *  @ignore
		 */
		_hide: function(){
			if(this.strButtonStatus != this._lstStatus.OFF){
				dojo.removeClass(this.domNode, this.strButtonStatus);
				this.strButtonStatus = this._lstStatus.OFF;
				dojo.addClass(this.domNode, this.strButtonStatus);
			}
		},

		/**
		 * Unhides / shows the button
		 *  @ignore
		 */
		_show: function(){
			if(this.strButtonStatus != this._lstStatus.ON){
				dojo.removeClass(this.domNode, this.strButtonStatus);
				this.strButtonStatus = this._lstStatus.ON;
				dojo.addClass(this.domNode, this.strButtonStatus);
			}
		},

		/**
		 * Deactivates and tints the button in grey.
		 * @ignore
		 */
		_greyTint: function(){
			if(this.strButtonStatus != this._lstStatus.GREY){
				dojo.removeClass(this.domNode, this.strButtonStatus);
				this.strButtonStatus = this._lstStatus.GREY;
				dojo.addClass(this.domNode, this.strButtonStatus);
			}
		},

	}
	);
/**
 * @class UIElements.AppButton.ButtonControl
 * @extends UIElements.AppButton.ButtonControlBase
 *
 * This class is the graphical representation of a button.
 *
 * A button can be used in a container with a 2/3 or a full layout to propose interaction with the user.
 *
 * A button contains a label, an icon an optionally a led that can be switched on or not.
 *
 * You can detect if the user has pressed on a button by given a callback function that will be called each time the user presses the button.
 *
 * Here is an example of creating a button:
 *
 * The first step is to create the DOM attach point in the HTML template file
 *     // Creating the HTML template associated to the button
 *     // Use a CSS style to place your button at the right place and to define the correct size
 *     <div dojoAttachPoint="myButton" class="buttonCls" />
 *
 * Next, you can create the button in your JavaScript view
 *     postCreate : function() {
 *          // Initialize the callback function
 *          var func = dojo.hitch(this, this.onButtonPress);
 *
 *          // Create the Button
 *          var btn = new UIElements.AppButton.ButtonControl({
 *              strButtonName:  "aButton",          // Button identifier
 *              strButtonLabel: "A Button",         // Label displayed
 *              strButtonIcon:  "buttonImgCss",     // CSS class of the image button
 *              strStatusIcon:  "notif-on",        // Led is switched off
 *              callback:       func                // Callback function called
 *          });
 *      },
 *
 *      onButtonPress : function() {
 *          // Do something
 *          // ...
 *      }
 *      
 * Finally, define the buttonCls CSS style
 *     .buttonCls {
 *         // Mandatory, define button size
 *         width:120px,
 *         height:124px,
 *         // Optional, define button position
 *         // ...
 *     }
 *
 */
dojo.require("UIElements.AppButton.ButtonControlBase");
dojo.provide("UIElements.AppButton.ButtonControl");
dojo.declare("UIElements.AppButton.ButtonControl",
	UIElements.AppButton.ButtonControlBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Status text
		 * @property
		 * @type String
                 * @ignore
		 */
		strStatusText		: "",

                /**
		 * Button icon. Use a CSS style that embeds your image
		 * @cfg {String} strButtonIcon
		 */

		strButtonIcon		: "",

                /**
		 * status icon size
                 * @ignore
		 * @property
		 * @type Number
		 */
		intStatusIconSize	: 16,

                /**
		 * icon size
                 * @ignore
		 * @property
		 * @type Number
		 */
		intIconSize			: 64,

                /**
		 * popup icon size
                 * @ignore
		 * @property
		 * @type Number
		 */
		intPopupIconSize	: 48,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 * The path to the template is defined here
		 */
		templatePath	: dojo.moduleUrl("UIElements.AppButton", "templates/Button8082.html"),
		/**
		 * @ignore
		 */
		cssFile         : ["base.css", "8082.css"],

		/* ------------------------------------ Constructor --------------------------------------- */

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 */
		_setStrButtonIconAttr : function(value) {
			if( this.boolPopup ) {
				value = value + "-" + this.intPopupIconSize;
			}
			else {
				value = value + "-" + this.intIconSize;
			}

			// Toggle status icon
			dojo.removeClass(this.domIcon, this.strButtonIcon);
			dojo.addClass(this.domIcon, value);
			this.strButtonIcon = value;
		}
	}
	);
/**
 * @class UIElements.AppButton.AppButtonControlBase
 * @extends UIElements.AppButton.ButtonControl
 * Abstract Class - Do not use it directly.
 */
dojo.provide("UIElements.AppButton.AppButtonControlBase");
dojo.declare("UIElements.AppButton.AppButtonControlBase",
	UIElements.AppButton.ButtonControl,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		_objAppBar		: null,

		/* ------------------------------------ Constructor --------------------------------------- */

		destroy : function(){
			// Break cyclic reference
			delete this._objAppBar;
			this.inherited(arguments);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */


		/**
		 * Set the parent of this AppButton so it will be notified when the visibility has changed
		 * @param {UIElements.AppBar.AppBarWidget} appBar The appBar that contains this button
		 */
		setAppBar: function(appBar) {
			if( typeof appBar.buttonChanged == "function" || appBar == null ) {
				this._objAppBar = appBar;
			}
		},

		/* ----------------------------------- Public methods ------------------------------------- */



	/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * Hides the action button.
		 * Warn the appBar if the visibility status change (for arrows)
		 * @ignore
		 */
		_hide: function(){
			var boolVisible = this.getVisibility();
			this.inherited(arguments);
			// Notify the appBar that the visibility has changed
			if( this._objAppBar && boolVisible) {
				this._objAppBar.buttonChanged();
			}
		},

		/**
		 * Unhides/shows the action button.
		 * Warn the appBar if the visibility status change (for arrows)
		 * @ignore
		 */
		_show: function(){
			var boolVisible = this.getVisibility();
			this.inherited(arguments);
			// Notify the appBar that the visibility has changed
			if( this._objAppBar && !boolVisible) {
				this._objAppBar.buttonChanged();
			}
		},

		/**
		 * Deactivates and tints the button in grey.
		 * Warn the appBar if the visibility status change (for arrows)
		 *  @ignore
		 */
		_greyTint: function(){
			var boolVisible = this.getVisibility();
			this.inherited(arguments);
			// Notify the appBar that the visibility has changed
			if( this._objAppBar && !boolVisible) {
				this._objAppBar.buttonChanged();
			}
		}

	}
	);
/**
 * @class UIElements.AppButton.AppButtonControl
 * @extends UIElements.AppButton.AppButtonControlBase
 * This class is the graphical representation of an action button.
 *
 * An action button is a button dedicated to be used in an AppBar widget.
 *
 * An action button contains a label, an icon an optionally a led that can be switched on or not.
 *
 * You can detect if the user has pressed on an action button by given a callback function that will be called each time the user presses the button.
 *
 * Here is an example of creating an action button with a led
 *
 *     // Define the action button
 *     ledBtn : null,
 *
 *     // Create and add the action button to the AppBar widget of my view
 *     postCreate : function() {
 *
 *          // Initialize the callback function
 *          var func = dojo.hitch(this, this.onLedButtonPress);
 *
 *          // Creation of the action button
 *          this.ledBtn = new UIElements.AppButton.AppButtonControl({
 *              strButtonName:  "ledButton",       // Button identifier
 *              strButtonLabel: "My Button",       // Label displayed
 *              strButtonIcon:  "buttonImgCss",    // CSS class of the image button
 *              strStatusIcon:  "notif-off",       // Led is switched off
 *              callback:       func               // Callback function called
 *          });
 *
 *          // Add the button to the AppBar widget
 *          ICTouchAPI.AppBarServices.getAppBar("webappName", "viewName").addActionButton(ledBtn);
 *     },
 *
 *     // A function that switches on the led on my button
 *     switchOnTheLed : function() {
 *
 *          // Switch on the led
 *          this.ledBtn.setStatusIcon("notif-on");
 *     },
 *     
 *     // A function that switches off the led on my button
 *     switchOffTheLed : function() {
 *
 *          // Switch off the led
 *          this.ledBtn.setStatusIcon("notif-off");
 *     },
 *
 *     // Define the callback function when the user presses on the save button
 *     onLedButtonPress : function() {
 *
 *          // Do something
 *          ...
 *     },
 *
 */
dojo.require("UIElements.AppButton.AppButtonControlBase");
dojo.provide("UIElements.AppButton.AppButtonControl");
dojo.declare("UIElements.AppButton.AppButtonControl",
	UIElements.AppButton.AppButtonControlBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * original
                 * @ignore
		 * @property
		 * @type Boolean
		 */
		original			: true,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 * The path to the template is defined here
		 */
		templatePath	: dojo.moduleUrl("UIElements.AppButton", "templates/AppButton8082.html"),
		/**
		 * @ignore
		 */
		cssFile         : ["base.css", "8082.css"]

	/* ------------------------------------ Constructor --------------------------------------- */

	/* ----------------------------------- Getter / Setter------------------------------------- */

	/* ----------------------------------- Public methods ------------------------------------- */

	/* --------------------------------- Private Methods -------------------------------------- */

	}
	);
/**
* @class UIElements.ApplicationMode.FullControlBase
* @extends UIElements.ApplicationMode.ControlBase
* Abstract Class - Do not use it directly.
*/
dojo.require("UIElements.ApplicationMode.ControlBase");
dojo.provide("UIElements.ApplicationMode.FullControlBase");
dojo.declare("UIElements.ApplicationMode.FullControlBase",
	UIElements.ApplicationMode.ControlBase, {

		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Reference on the layout content
		 * @property
		 * @type HTMLElement
                 * @ignore
		 */
		objContent       	: null,
		/**
		 * Reference on the layout title
		 * @property
		 * @type HTMLElement
                 * @ignore
		 */
		objTitle        	: null,

		/**
		 * JSON object that defines the view displayed inside this layout
		 * This field must contain the name of the view and the parameters to send to this view
		 * @cfg {Object} content (required)
		 * @cfg {String} content.name
		 * @cfg {Object} content.params
		 */
		content: null,

		 /**
		 * JSON object that defines the title of the container (To act as a ContainerTitle)
		 * This field must contain the label displayed
		 * @cfg {Object} title (required)
		 * @cfg {String} title.strLabel
		 */
		title: null,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		domContent			: null,
		/**
		 * @ignore
		 */
		_uiContent			: null,

		/* ------------------------------------ Constructor --------------------------------------- */

		postMixInProperties : function() {
			if (this.content) {
				this.objContent = this.content;
			}
			if (this.title) {
				this.objTitle = this.title;
			}
		},

		postCreate: function() {
			if(!generalConfig.simulation) {
				if (this.objTitle != undefined) {
					this._uiContent = new UIElements.Container.ContainerControl({
						"objTitle": this.objTitle,
						"objContent": {
							name: this.objContent.name,
							params: this.objContent.params
						}
					}, this.domContent);
				}
				else {
					eval("this._uiContent = new "+this.objContent.name+"(this.objContent.params,this.domContent)");
				}
			}
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * Get the container associated to view contained into this full layout
		 * @return {UIElements.Container.ContainerControl} the object representing the full layout
		 */
		getContent: function() {
			return this._uiContent;
		}

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */

	}
	);
/**
* @class UIElements.ApplicationMode.FullControl
* @extends UIElements.ApplicationMode.FullControlBase
* This class is the graphical representation of a webapp screen with a full layout.
*
* The full layout displays a single container that takes all the available space in the screen while keeping the top bar and an action bar. A title is displayed above this container.
*
* If you want to display a webapp view using a 1/3 - 2/3 layout, see the {@link UIElements.ApplicationMode.ThirdControl} class.
*
* Here is an example of creating a full layout in the main view:
*     postCreate : function() {
*           // Create the full layout
*           var full = new UIElements.ApplicationMode.FullControl({
*               content : {
*                   name: "webapp.myWebapp.getFullContent",
*                   params: {}
*               }
*           }, this.fullDomNode);
*     }
*
* In the template associated to this view, don't forget to add the fullDomNode attach point
*     <div dojoAttachPoint="fullDomNode" />
*
* In the view getFullContent, add the following code to have a container
*     postCreate : function() {
*          // Create the container
*           var container = new UIElements.Container.ContainerControl({
*               title: {
*                   strLabel: "Full Layout title"
*               }
*               content: {
*                   name:"UIElements.DataList.DataListControl",
*                   params:{arrItems:[]}
*               }
*           }, this.containerDomNode);
*     }
*
* Instead of adding a DataList, other UIElement like a {@link UIElements.PresentationList.PresentationListControl} or an other view can be added using the objContent parameter
*      objContent: {
*          name:"webapp.myWebapp.getOtherView",
*          params:{}
*      }
*
* Don't forget to add the titleDomNode attach point in the template getFullContent.html
*     <div dojoAttachPoint="containerDomNode" />
*
*/
dojo.require("UIElements.ApplicationMode.FullControlBase");
dojo.provide("UIElements.ApplicationMode.FullControl");

dojo.declare("UIElements.ApplicationMode.FullControl",
	UIElements.ApplicationMode.FullControlBase,
	{

		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 * The path to the template is defined here
		 */
		templatePath	: dojo.moduleUrl("UIElements.ApplicationMode","templates/Full8082.html"),
		/**
		 * @ignore
		 */
		cssFile         : ["base.css","8082.css"]

		/* ------------------------------------ Constructor --------------------------------------- */

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */
	}
	);
/**
* @class UIElements.ApplicationMode.PreviewControlBase
* @extends UIElements.ApplicationMode.ControlBase
* @private
* Abstract Class<br>
* Preview mode : Manages the way you organize the Webapps containers across the screen
*/
dojo.require("UIElements.ApplicationMode.ControlBase");
dojo.provide("UIElements.ApplicationMode.PreviewControlBase");
dojo.declare("UIElements.ApplicationMode.PreviewControlBase",
	UIElements.ApplicationMode.ControlBase,
	{

		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		domPreviewList: null,

		/* ------------------------------------ Constructor --------------------------------------- */

		constructor: function() {

		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		 * Create preview iframe right under li tag and populate it on iframe load event
		 * @param {HTMLElement} div
		 * @param {Number} pos position to add the preview in preview list
		 * @param {Function} firstload callback launched at the end of the preview creation
		 */
		addPreview: function(div,pos,firstload) {
			var li = dojo.create("li", {
				"class": "ZonePreview",
				"position": pos
			}, this.domPreviewList, pos);
			this._createIFrame(li, pos, firstload)
		},

		/**
		 * Replace a preview with an other
		 * @param {Number} pos position of the previex to replace
		 * @param {Function} firstload callback launched at the end of the new preview creation
		 */
		replacePreview: function(pos,firstload) {
			var domPreviews = dojo.query("li.ZonePreview", this.domPreviewList);
			var iframe = dojo.query("iframe", domPreviews[pos])[0];
			dojo.destroy(iframe);
			this._createIFrame(domPreviews[pos], pos, firstload);
		},

		/**
		 * Moves the preview from a source position to a destinatin position
		 * @param {Number} intSrc Source position
		 * @param {Number} intDest Destination position
		 */
		movePreview: function(intSrc, intDest) {
			var destPos;
			if (intDest>intSrc) {
				destPos = intDest+2;
			}
			else {
				destPos = intDest+1;
			}
			var oldDomPreviews = dojo.query("li.ZonePreview", this.domPreviewList);
			dojo.place(oldDomPreviews[intSrc], this.domPreviewList, destPos);
			var newDomPreviews = dojo.query("li.ZonePreview", this.domPreviewList);
			var i, iMax=newDomPreviews.length;
			for(i=0; i<iMax; i++) {
				newDomPreviews[i].setAttribute("id", "Preview_"+i);
				newDomPreviews[i].setAttribute("position", i);
			}
		},

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 */
		_createIFrame: function(elem, pos,firstload) {
			var myfr = dojo.create("iframe", {
				"src": "iframe.html" ,
				"class": "Preview" ,
				"id": "Preview_"+pos
			});
			myfr.addEventListener('load', firstload, false);
			dojo.place(myfr, elem, pos);
		},

	}
	);
/**
* @class UIElements.ApplicationMode.PreviewControl
* @extends UIElements.ApplicationMode.PreviewControlBase
* @private
* Preview mode : Manages the way you organize the Webapps containers across the screen
*/
dojo.require("UIElements.ApplicationMode.PreviewControlBase");
dojo.provide("UIElements.ApplicationMode.PreviewControl");
dojo.declare("UIElements.ApplicationMode.PreviewControl",
	UIElements.ApplicationMode.PreviewControlBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 * The path to the template is defined here
		 */
		templatePath	: dojo.moduleUrl("UIElements.ApplicationMode","templates/Preview8082.html"),
		/**
		 * @ignore
		 */
		cssFile         : ["base.css","8082.css"],

		/* ------------------------------------ Constructor --------------------------------------- */

		postCreate : function() {
			// Add the scrolling functionality
                        // crms00334612_begin : don't scroll the homepage preview mode
			/*
			this.scroll = FXScroll.init({
				"handler" : this.domScroll,
				"easing" : true,
				"scrollType" : "horizontal"
			});
                        */
                       // crms00334612_end

		}

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */

	}
	);
/**
* @class UIElements.ApplicationMode.ThirdControlBase
* @extends UIElements.ApplicationMode.ControlBase
* Abstract Class - Do not use it directly.
*/
dojo.require("UIElements.ApplicationMode.ControlBase");
dojo.provide("UIElements.ApplicationMode.ThirdControlBase");
dojo.declare("UIElements.ApplicationMode.ThirdControlBase",
	UIElements.ApplicationMode.ControlBase,
	{
	/* --------------------------------- Public attributes ------------------------------------ */

	/* --------------------------------- Private attributes ----------------------------------- */

	/* ------------------------------------ Constructor --------------------------------------- */

	/* ----------------------------------- Getter / Setter------------------------------------- */

	/* ----------------------------------- Public methods ------------------------------------- */

	/* --------------------------------- Private Methods -------------------------------------- */
	}
	);
/**
* @class UIElements.ApplicationMode.ThirdControl
* @extends UIElements.ApplicationMode.ThirdControlBase
* This class is the graphical representation of a webapp screen with a 1/3 - 2/3 layout.
*
* The 1/3 - 2/3 layout displays two containers that take all the available space in the screen while keeping the top bar and an action bar.
*
* - The 1/3 container that is used to display a {@link UIElements.MenuList.MenuListControl}
*
* - The 2/3 container that is used to display a list like a {@link UIElements.DataList.DataListControl} or a {@link UIElements.PresentationList.PresentationListControl} or any other graphical elements.
*
* If you want to display a webapp view using a full layout, see the {@link UIElements.ApplicationMode.FullControl} class.
*
* Here is an example of creating a 1/3 - 2/3 layout in the main view:
*     // Declare the layout
*     layout:null,
*
*     postCreate : function() {
*           // Define the callback function that is called when an item is selected in the 1/3 container list.
*           var itemMenuPressedCallback = dojo.hitch(this, this.onMenuItemPressed);
*
*           // Create the 1/3 - 2/3 layout
*           this.layout = new UIElements.ApplicationMode.ThirdControl({
*               menu : {
*                   name: "webapp.myWebapp.getMenuContent",
*                   params: {callBackFunc:itemMenuPressedCallback}
*               },
*               container : {
*                   name: "webapp.myWebapp.getContainerContent",
*                   params: {}
*               }
*           }, this.thirdDomNode);
*     },
*
*     // Callback when an item in the view getMenuContent is selected.
*     onMenuItemPressed : function() {
*           // Do something, ie: reload an other container in the 2/3 part
*           this.layout.setContainer("webapp.myWebapp.getAnOtherContainer", {});
*     }
*
* In the template associated to this view, don't forget to add the thirdDomNode attach point
*     <div dojoAttachPoint="thirdDomNode" />
*
* In the view getMenuContent (1/3 container), define the MenuList
*
*     // Items in the list see the {@link UIElements.MenuList.MenuListControl} class to see an example of how to populate a list.
*     arrItems: []
*
*     // The callBack function (value is given by the main view using the 'params' property
*     callBackFunc: null,
*
*     postCreate : function() {
*            var container = new UIElements.Container.ContainerControl({
*               title: {
*                   strLabel: "1/3 Layout title"
*               }
*               content: {
*                   name:"UIElements.MenuList.MenuListControl",
*                   params:{
*                       arrItems:[],
*                       callback: this.callBackFunc
*                   }
*               }
*           }, this.menuDomNode);
*     }
*
* In the template associated to the getMenuContent, don't forget to add the associated DOM attach point
*     <div dojoAttachPoint="menuDomNode" />
*
* In the view getContainerContent (2/3 container), the minimum that must be defined is a container
*     postCreate : function() {
*           // Create the container
*           var container = new UIElements.Container.ContainerControl({
*               title: {
*                   strLabel: "2/3 Layout title"
*               }
*               content: {
*                   name:"UIElements.DataList.DataListControl",
*                   params:{arrItems:[]}
*               }
*           }, this.containerDomNode);
*      }
*
* Instead of adding a DataList, other UIElement like a {@link UIElements.PresentationList.PresentationListControl} or an other view can be added using the objContent parameter
*      objContent: {
*          name:"webapp.myWebapp.getOtherView",
*          params:{}
*      }
*
* In the template associated to the getContainerContent, don't forget to add the DOM attach points of all UIElements added
*     <div dojoAttachPoint="containerDomNode" />
*
*/
dojo.require("UIElements.ApplicationMode.ThirdControlBase");
dojo.provide("UIElements.ApplicationMode.ThirdControl");
dojo.declare("UIElements.ApplicationMode.ThirdControl",
	UIElements.ApplicationMode.ThirdControlBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Reference on menu object
		 * @property
		 * @type HTMLElement
		 * @ignore
		 */
		objMenu				: null,
		/**
		 * Reference on the container object
		 * @property
		 * @type HTMLElement
		 * @ignore
		 */
		objContainer		: null,
		/**
		 * Reference on the menu title object
		 * @property
		 * @type HTMLElement
		 * @ignore
		 */
		objMenuTitle		: null,
		/**
		 * Reference on the contaienr title object
		 * @property
		 * @type HTMLElement
		 * @ignore
		 */
		objContainerTitle	: null,

		/**
		 * JSON object that defines the view displayed in the 1/3 part of the layout
		 * This field must contain the name of the view and the parameters to send to this view
		 * @cfg {Object} menu (required)
		 * @cfg {String} menu.name
		 * @cfg {Object} menu.params
		 */
		menu: null,

		/**
		 * JSON object that defines the view displayed in the 2/3 part of the layout
		 * This field must contain the name of the view and the parameters to send to this view
		 * @cfg {Object} container (required)
		 * @cfg {String} container.name
		 * @cfg {Object} container.params
		 */
		container: null,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		domMenu				: null,
		/**
		 * @ignore
		 */
		domContainer		: null,
		/**
		 * @ignore
		 * The path to the template is defined here
		 */
		templatePath	: dojo.moduleUrl("UIElements.ApplicationMode","templates/Third8082.html"),
		/**
		 * @ignore
		 */
		cssFile         : ["base.css","8082.css"],
		/**
		 * @ignore
		 */
		_uiMenu				: null,
		/**
		 * @ignore
		 */
		_uiContainer		: null,

		/* ------------------------------------ Constructor --------------------------------------- */

		postMixInProperties : function() {
			if (this.menu)
				this.objMenu = this.menu;
			if (this.container)
				this.objContainer = this.container;
		},

		postCreate: function() {
			if(!generalConfig.simulation) {
				// If strMenuTitle was defined, load an intermediate "ContainerControl" instead of directly loading the widget.
				if (this.objMenuTitle != null) {
					this._uiMenu = new UIElements.Container.ContainerControl({
						objTitle: this.objMenuTitle,
						objContent: {
							name: this.objMenu.name,
							params: this.objMenu.params
						}
					}, this.domMenu);
				}
				else {
					eval("this._uiMenu = new "+this.objMenu.name+"(this.objMenu.params,this.domMenu)");
				}
				// If strContainerTitle was defined, load an intermediate "ContainerControl" instead of directly loading the widget.
				var div = dojo.create("div",{},this.domContainer);
				if (this.objContainerTitle != null) {
					this._uiContainer = new UIElements.Container.ContainerControl({
						objTitle: this.objContainerTitle,
						objContent: {
							name: this.objContainer.name,
							params: this.objContainer.params
						}
					}, div);
				}
				else{
					eval("this._uiContainer = new "+this.objContainer.name+"(this.objContainer.params,div)");
				}
			}
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * Get the container associated to view contained into 1/3 part of the layout
		 * @return {UIElements.Container.ContainerControl} the object representing the 1/3 part of the layout
         */
		getMenu: function() {
			return this._uiMenu;
		},

		/**
		 * Get the container associated to view contained into 2/3 part of the layout
		 * @return {UIElements.Container.ContainerControl} the object representing the 2/3 part of the layout
         */
		getContainer: function() {
			return this._uiContainer;
		},

		/**
		 * Set or update the view contained into the 2/3 par of the layout
		 * @param {String} strContainerName the name of the view
		 * @param {Object} objContainerParams the params (JSON Object with keys / values)
         */
		setContainer: function(strContainerName,objContainerParams) {
			this.objContainer.name   = strContainerName;
			this.objContainer.params = objContainerParams;
			if( this._uiContainer != null )
				this._uiContainer.destroy();
			this.domContainer.id = ""; // resets

			// new objContainer
			var div = dojo.create("div",{},this.domContainer);
			var tempObj = ICTouchAPI.tools.getObjectReference(this.container.name);
			this._uiContainer = new tempObj(this.container.params, div);
		}

	/* ----------------------------------- Public methods ------------------------------------- */

	/* --------------------------------- Private Methods -------------------------------------- */

	}
	);
/**
 * @class UIElements.Calendar.CalendarControlBase
 * @extends UIElements._base
 * @private
 * Abstract Class - Do not use it directly.
 */

dojo.provide("UIElements.Calendar.CalendarControlBase");

dojo.declare("UIElements.Calendar.CalendarControlBase",
	[UIElements._base,dojox.dtl._Templated],
	{

		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Default size of Calendar
		 * @property
		 * @type String
		 */
		strCalendarType				: "medium", // Default size of Calendar
		/**
		 * today
		 * @property
		 * @type Date
		 */
		today						: null,
		/**
		 * date
		 * @property
		 * @type Date
		 */
		date						: null,
		/**
		 * month
		 * @property
		 * @type Number
		 */
		month						: null,
		/**
		 * year
		 * @property
		 * @type Number
		 */
		year						: null,
		/**
		 * First day of month
		 * @property
		 * @type String
		 */
		firstDayOfMonth				: null,
		/**
		 * selected dom node
		 * @property
		 * @type HTMLElement
		 */
		selectedNode				: null,
		/**
		 * selected date
		 * @property
		 * @type Date
		 */
		selectedDate				: null,


		/**
		 * Array of plannaed events
		 * @property
		 * @type Array
		 */
		arrPlannedEvents			: [],

		/**
		 * Callback function runned when a value is selected
		 * @property
		 * @type Function
		 */
		onValueSelectedCallBack		: null,
		/**
		 * DO we have to show the header ?
		 * @property
		 * @type Boolean
		 */
		boolShowHeader				: false,


		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		_onValueSelectedCallBackFunc	: null,

		/* ------------------------------------ Constructor --------------------------------------- */


		constructor : function() {
			this.date = new Date();
			this.selectedDate = this.date;
		},


		postCreate : function() {
			this.firstDayOfMonth = dojo.cldr.supplemental.getFirstDayOfWeek(this.lang);
			this.today = new Date();
			var dayNames = Date.getDaysOfWeek(dojo.locale);
			var headNode = this.UIElementsCalendarDaysHead;
			this.trNode = dojo.create("tr", null, headNode);
			var d = this.firstDayOfMonth;

			//display days label
			for (var i=this.firstDayOfMonth; i < 7+this.firstDayOfMonth; i++) {
				if (d >= 7) d = d-7;
				var thNode = dojo.create("th", null, this.trNode, i);
				dojo.addClass(thNode, "dijitReset dijitCalendarDayLabelTemplate");
				var spanNode = dojo.create("span", {
					innerHTML: dayNames[d]
				}, thNode, d);
				dojo.addClass(spanNode, "dijitCalendarDayLabel");
				d++;
			}
			this._populateDays();
			this._selectDateInCurrentMonth(this.today);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * return the selected date
		 * @return {Date} selected Date
		 */
		getValue : function () {
			return this.selectedDate;
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		 * decrease month
		 * @return {Date} this.date
		 */
		decMonth: function() {
			this.date = dojo.date.add(new Date(this.year, this.month, 1), "month", -1);
			this._populateDays();
			var sDate = this.date;
			if (this.today.getMonth()==this.month && this.today.getFullYear()==this.year) {
				sDate = new Date(this.date.getFullYear(), this.date.getMonth(), this.today.getDate());
				this._selectDateInCurrentMonth(sDate);
			}
			return this.date;
		},


		/**
		 * increase month
		 * @return {Date} this.date
		 */
		incMonth: function() {
			this.date = dojo.date.add(new Date(this.year, this.month, 1), "month", 1);
			this._populateDays();
			var sDate = this.date;
			if (this.today.getMonth()==this.month && this.today.getFullYear()==this.year) {
				sDate = new Date(this.date.getFullYear(), this.date.getMonth(), this.today.getDate());
				this._selectDateInCurrentMonth(sDate);
			}
			return this.date;
		},


		/**
		 * select a Date(change month if needed)
		 * @param {Date} sDate date
		 */
		selectDate : function(sDate) {
			if(sDate.getFullYear()!=this.year || sDate.getMonth()!=this.month) {
				this.date = new Date(sDate.getFullYear(), sDate.getMonth(), 1);
				this._populateDays();
			}
			this._selectDateInCurrentMonth(sDate);
		},

                /**
                 * Get the date
                 * @ignore
                 */
		getDate : function(){
			return this.date;
		},

                /**
                 * @ignore
                 */
		refreshDays : function(){
			var dayNames = Date.getDaysOfWeek(dojo.locale);
			var headNode = this.UIElementsCalendarDaysHead;
			dojo.destroy(this.trNode);
			this.trNode = dojo.create("tr", null, headNode);
			var d = this.firstDayOfMonth;

			//display days label
			for (var i=this.firstDayOfMonth; i < 7+this.firstDayOfMonth; i++) {
				if (d >= 7) d = d-7;
				var thNode = dojo.create("th", null, this.trNode, i);
				dojo.addClass(thNode, "dijitReset dijitCalendarDayLabelTemplate");
				var spanNode = dojo.create("span", {
					innerHTML: dayNames[d]
				}, thNode, d);
				dojo.addClass(spanNode, "dijitCalendarDayLabel");
				d++;
			}
		},
		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * on Value Selected
		 * @param {Date} date selected (may happen, on same day)
		 *
		 */
		onValueSelected : function(date) {
			if(this._onValueSelectedCallBackFunc) {
				this._onValueSelectedCallBackFunc(date);
			}
		},

		/**
		 * @ignore
		 * Display days of current month
		 */
		_populateDays: function() {
			this.selectedNode = null;
			var bodyNode = this.UIElementsCalendarDaysBody;
			bodyNode.innerHTML="";	/* Clear days tbody*/
			this.year = this.date.getFullYear();
			this.month = this.date.getMonth();
			var firstDay = new Date(this.year, this.month, 1);
			var isPreviousMonth = 1;
			var offset = firstDay.getDay()*(-1) + this.firstDayOfMonth;

			if(this.boolShowHeader) {
				if(dojo.locale == "ko"){
					// Korean date needs a specific display with a special character after the year and after the month. It is done by js for month but has to be added manually for year.
					this.UIElementsCalendarMonth.innerHTML = firstDay.format('yyyy년 MMMM',dojo.locale);
				}
				else {
					this.UIElementsCalendarMonth.innerHTML = firstDay.format('MMMM yyyy',dojo.locale);
				}
			}

			//iterate weeks of current month
			for (var i=0; i<6; i++) {
				var trNode = dojo.create("tr", null, bodyNode);
				dojo.addClass(trNode, "dijitReset dijitCalendarWeekTemplate");
				//iterate days of current week
				for (var j=0; j<7; j++) {
					var tdNode = dojo.create("td", null, trNode);
					var iterateDay = dojo.date.add(firstDay, "day", offset);

					//Different class between days of month and days of previous or next month
					if(iterateDay.getMonth() == this.month) {
						dojo.addClass(tdNode, "dijitCalendarCurrentMonth dijitCalendarDateTemplate");
						if(isPreviousMonth == 1) isPreviousMonth = 0;
					}
					else {
						if(isPreviousMonth == 1)
							dojo.addClass(tdNode, "dijitCalendarPreviousMonth dijitCalendarDateTemplate");
						else
							dojo.addClass(tdNode, "dijitCalendarNextMonth dijitCalendarDateTemplate");
					}
					//add eventThisDay class where an event is planned
					var strKeyDate = iterateDay.format('yyyy-MM-dd',dojo.locale);
					if(this.arrPlannedEvents.hasOwnProperty(strKeyDate)) {
						dojo.addClass(tdNode, "eventThisDay");
					}

					//Display week Number
					if(this.firstDayOfMonth == iterateDay.getDay()) {
						var weekNumber = iterateDay.format('w',dojo.locale);
                                                var emNode = dojo.create("em", {
							innerHTML: weekNumber
						}, tdNode);

						dojo.addClass(emNode, "dijitCalendarWeekLabel");
					}
					//Display day
					var spanNode = dojo.create("span", {
						innerHTML: iterateDay.getDate()
					}, tdNode);
					dojo.connect(spanNode, "onclick", this, this._onDayClick);
					dojo.addClass(spanNode, "dijitCalendarDateLabel");
					offset++;
				}
			}
		},




		/**
		 * @ignore
		 * Event where a day is selected
		 */
		_onDayClick: function(evt){
			dojo.stopEvent(evt);
			var node = evt.currentTarget;
			if (dojo.hasClass(node, "dijitCalendarDateLabel")) {
				//Deselect old date and select clicked date
				if (this.selectedNode!=null)
					dojo.removeClass(this.selectedNode.parentNode, "dijitCalendarSelectedDate");
				this.selectedNode = node;
				dojo.addClass(node.parentNode, "dijitCalendarSelectedDate");
				//Detect and format clicked date
				var sDate = new Date(this.year, this.month, node.innerHTML);
				if (dojo.hasClass(node.parentNode, "dijitCalendarPreviousMonth"))
					sDate = dojo.date.add(sDate, "month", -1);
				else if (dojo.hasClass(node.parentNode, "dijitCalendarNextMonth"))
					sDate = dojo.date.add(sDate, "month", 1);
				this.selectedDate = sDate;
				this.onValueSelected(this.selectedDate);
			}
		},





		/**
		 * @ignore
		 * Select a Date in current month
		 */
		_selectDateInCurrentMonth : function(sDate) {
			var bodyNode = this.UIElementsCalendarDaysBody;
			var nodes = dojo.query(".dijitCalendarDateLabel", bodyNode);
			var iMax = nodes.length, i = 0;
			if (this.selectedNode!=null)
				dojo.removeClass(this.selectedNode.parentNode, "dijitCalendarSelectedDate");
			while(i<iMax) {
				if(parseInt(nodes[i].innerHTML)==sDate.getDate() && dojo.hasClass(nodes[i].parentNode, "dijitCalendarCurrentMonth")) {
					dojo.addClass(nodes[i].parentNode, "dijitCalendarSelectedDate");
					this.selectedNode = nodes[i];
				}
				i++;
			}
		},

		/**
		 * @ignore
		 */
		_setOnValueSelectedCallBackAttr : function(callback) {
			this._onValueSelectedCallBackFunc = ICTouchAPI.tools.callbackToHitch(callback);
		},

		/**
		 * @ignore
		 * Compatibility with original Dojo Calendar
		 */
		_getDisplayMonthAttr : function () {
			this.date.format('MMMM yyyy',dojo.locale);
			return this.date.format('MMMM yyyy',dojo.locale);
		}
	}
	);
/**
 * @class UIElements.Calendar.CalendarControl
 * @extends UIElements.Calendar.CalendarControlBase
 * @private
 * This UIElement is used as a container to prevent content to be included without a containing structure. This UIElement can be titled or not.
 */
dojo.require("UIElements.Calendar.CalendarControlBase");
dojo.provide("UIElements.Calendar.CalendarControl");
dojo.declare("UIElements.Calendar.CalendarControl",
	UIElements.Calendar.CalendarControlBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Default size of Calendar
		 * @property
		 * @type String
		 */
		strCalendarType				: "small",

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		templatePath				: dojo.moduleUrl("UIElements.Calendar","templates/Calendar8082.html"),
		/**
		 * @ignore
		 */
		templateString				: null,
		/**
		 * @ignore
		 */
		cssFile                     : ["base.css", "8082.css"],

	/* ------------------------------------ Constructor --------------------------------------- */

	/* ----------------------------------- Getter / Setter------------------------------------- */

	/* ----------------------------------- Public methods ------------------------------------- */

	/* --------------------------------- Private Methods -------------------------------------- */

	}
	);

/**
 * @class UIElements.List.ListControlBase
 * @namespace UIElements.List
 * @extends UIElements._base
 * Abstract Class - Do not use it directly
 */
dojo.provide("UIElements.List.ListControlBase");
dojo.declare("UIElements.List.ListControlBase",
	[UIElements._base, dojox.dtl._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Items to display in the list.
		 * 
	 * This property is public for backward compatibility reasons.
		 * <b>It is recommended not to modify it but instead to use the available functions</b>
		 * (i.e. pushItems, popItems, updateItem....).
	 * @property
	 * @type {Array}
	 */
		arrItems : [],

		/**
		 * Nb of elements Max to be displayed in the DOM for pagination.
		 *
		 * This property can be set when instantiating the UIElement.
		 * @cfg {Number}
	 */
		nbElementsMax : 0,

		/**
		 * Describes the differences between item types (normal, title...).
		 *
		 * One of these types can be set for each item. Default type of the item is "normal".
		 * "Title" corresponds to title of collapsable items.
		 * @property {Object}
	 */
		types : {},

		/**
	 * Do categories have to automatically collapse?
		 *
		 * This property can be set when instantiating the UIElement.
		 * Then it can be changed using the appropriated getter/setter.
		 * @cfg {Boolean}
	 */
		autoCollapse : true,

		/**
		 * Do principal entries have to automatically fold when another one is unfolding?
		 *
		 * This property can be set when instantiating the UIElement.
		 * Then it can be changed using the appropriated getter/setter.
		 * @cfg {Boolean}
		 */
		autoFold : true,

		/**
		 * Do principal entries have to automatically unfold when clicking on a principal item?
		 *
		 * This property can be set when instantiating the UIElement.
		 * Then it can be changed using the appropriated getter/setter.
		 * @cfg {Boolean}
		 */
		autoUnfoldOnClick : true,

		/**
		 * Do principal entries have to be automatically folded when they are added to the list?
		 *
		 * This property can be set when instantiating the UIElement.
		 * Then it can be changed using the appropriated getter/setter.
		 * @cfg {Boolean}
		 */
		insertAsFolded : true,

		/**
	 * Does the list have to autoSelect the first item?
		 *
		 * This property can be set when instantiating the UIElement.
		 * @cfg {Boolean}
	 */
		autoSelectFirst : false,

		/**
		 * Does the list have to scroll ?
		 *
		 * This property can be set when instantiating the UIElement.
		 * @cfg {Boolean}
		 */
		boolScroll : true,

		/**
		 * Callback called when an element is selected.
		 *
		 * This property is set when instantiating the UIElement.
		 * @cfg {Function}
	 */
		callback : {},

		/**
		 * Callback called when a category is collapsed.
		 *
		 * This property is set when instantiating the UIElement.
		 * @cfg {Function}
	 */
		funcCollapsed : null,

		/**
		 * Display an empty state when this list is empty.
		 *
		 * This property can be set when instantiating the UIElement.
		 * @cfg {Boolean}
		 */
		showEmptyState: false,

		/**
		 * Empty state text, null show the default text from EmptyState.
		 *
		 * This property can be set when instantiating the UIElement.
		 * @cfg {String}
		 */
		emptyText: "",

		/**
		 * Do we select graphically an item we selected (by click or by js) ?
		 * 
		 * This property can be set when instantiating the UIElement.
		 * @cfg {Boolean}
		 */
		boolNotSelectable : false,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
	 * the list's root dom node
	 * @ignore
	 */
		_list: null,
		/**
	 * The current selected item
	 * @ignore
	 */
		_currentSelected : null,
		/**
	 * The current uncollapsed category
	 * @ignore
	 */
		_currentUnCollapsed : null,
		/**
	 * An array listing every LIs in the dom
	 * @ignore
	 */
		_LisInDom : [],

		/**
		 * EmptyState object
		 * @ignore
		 */
		_emptyState : null,

		/**
		 * Static value for _subType
		 * PRINCIPAL_ENTRY is the _subType for principal items
		 * @static
		 * @type {String}
		 * @ignore
		 */
		PRINCIPAL_ENTRY : "principalEntry",

		/**
		 * Static value for _subType
		 * SUB_ENTRY is the _subType for secondary items
		 * @static
		 * @type {String}
		 * @ignore
		 */
		SUB_ENTRY : "subEntry",

		/* ------------------------------------ Constructor --------------------------------------- */


		/**
	 * @ignore
	 */
		constructor : function () {
			// Describes every type used in these lists.
			// Can't see more than two, this is linked with the style sheet
			this.types = {
				title : {
					className : "Title",
					callback : {
						event : "click",
						func : "toggleCollapse",
						capture : true
					}
				},
				normal : {
					className : "Content",
					callback : {
						event : "click",
						func : "selectedLi",
						capture : false
					}
				}
			};

			// Take care to correctly init every attribute
			// so they are not shared between UIElements of same type.
			this.arrItems = [];


                        //crms00331797: Due to scroll enhancement and back color change, sometimes it seems that the end of the list is reached
			this.nbElementsMax = 10;
			this.autoCollapse = true;
			this.autoSelectFirst = true;

			this._list = null;
			this._currentSelected = null;
			this._currentUnCollapsed = null;
			this._LisInDom = [];

			this.callback = {};
			this.funcCollapsed = null;
		},

		/**
	 * @ignore
	 */
		postMixInProperties : function() {
			if(!generalConfig.simulation) {
				// Categories can only auto collapse if asked and if there are categories!
				this.setAutoCollapse(this._isListWithCategories() && !!this.getAutoCollapse());

				// Clean types in arrItems.
				this._cleanTypes(this.arrItems);

				// Add bCollapse to all items
				// The List will be (un)collapsed at start according to autoCollapse.
				this._setItemsCollapse(this.arrItems, this.getAutoCollapse());

				// Initializes the UL dom node
				this._initList()

				// And adds elements to it.
				.push(this.nbElementsMax);

				// Auto select first item
				this.autoSelectFirst &&  this.selectFirst();
		    }
		},

		/**
	 * @ignore
	 */
		postCreate : function () {                    
			if(!generalConfig.simulation) {

				// Appends the list to the visible dom
				this.domNode.appendChild(this._list);
				if(this.boolScroll){
					// Adds the scroll functionality
					FXScroll.init({
						handler : this.domNode,
						// Scroll of vertical type.
						scrollType : "vertical",
						// The pagination object
						// When scrolling up/down, the callback is called and does
						// what is necessary to add and remove dom nodes.
						pagination : {
							"callback" : this.pagination,
							"context" : this
						}
					});
				}
	
				if (this.showEmptyState) {
					var params = {
						boolBackground: true,
						boolIcon: false
					};
					if (this.emptyText) {
						params.strEmptyMessage = this.emptyText;
					}
					this._emptyState = new UIElements.EmptyState.EmptyStateControl(params);
					this._emptyState.placeAt(this.domNode, "first");
				}
			}
		},

		/**
	 * @ignore
	 */
		destroy : function () {
			// Go over each li and "remove" it to unsubscribe all i18n object
			var len = this._LisInDom.length;
			for (var i=0; i<len; ++i) {
				this._LisInDom[i].remove(false);
			}

			if (this._emptyState) {
				this._emptyState.destroy();
				delete this._emptyState;
			}
			
			this.inherited(arguments);
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
	 * Returns the selected item
	 * @return {Object} the current selected item
	 */
		getCurrentSelectedItem : function () {
			var currentSelected = this.getCurrentSelected();
			return (currentSelected && currentSelected._itemSource);
		},

		/**
		 * Returns the number of items in the list (ignore the number of secondary items)
		 * @return {Number} the number of items (without taking into account potential secondary items)
		 */
		getNbItems : function () {
			return this._getArrItemsWithoutSecondaryItems().length;
		},

		/**
	 * Scrolls the list to an item
	 * @param {Number} itemNb index of the item to scroll to
	 * @return {Object} Returns the li of the pointer that we've scrolled to.
	 */
		scrollToItemByPosition : function (itemNb) {
			// Before uncollapsing a new category,
			// collapse the previous one, if autoCollapse is enabled.
			this.getAutoCollapse() && this.collapseAll();
			// Get previous item's category
			var previousCat = this._getPreviousCatItem(itemNb);
			// Unollapse it.
			this._setCategoryCollapse(previousCat, false);
			// Empty the list of domNodes
			this._emptyList();
			// Reset the scroll
			this._resetDomScroll();
			// Insert the new nodes
			var firstDomNodeDisplayed = this.insert(0, itemNb, this.nbElementsMax, false);
			// Fill the list from the top until max nb of elements to display is reached.
			this._fillList();
			// return the first dom node
			return firstDomNodeDisplayed;
		},

		/**
	 * Select an item according to its position.
		 * 
		 * Note that:
		 * <ul>
		 * <li> it will scroll to the item's position (except if "doNotScroll" is set to true) and select the item </li>
		 * <li> it will not execute the clickCallback function of the list, except if "click" is set to true </li>
		 * <li> if the item to select is a principal item and the list's flag "autoUnfoldOnClick" is true (that's the case by default),
		 * the item will be automatically unfolded (except if "doNotUnfold" is set to true) </li>
		 * <li> if the item to select is a secondary item and the principal item is currently folded, the principal item will be
		 * automatically unfolded and the secondary item will be selected (except if "doNotUnfold" is set to true) </li>
		 * </ul>
		 * @param {Object} itemPos position of the item to select
		 * @param {Boolean} click handle click ? If true, the clickCallback function of the list will be executed
		 * @param {Boolean} doNotScroll if true, select the item without scrolling to its position
		 * @param {Boolean} doNotUnfold if true, do not unfold the primary entry when selecting it
		 */
		selectItemByPosition : function (itemPos, click, doNotScroll, doNotUnfold) {
		    var item = this.arrItems[itemPos];
			if (item) {
				this.selectItemByIndex(item.intIndex, click, doNotScroll, doNotUnfold);
			}
			else {
				ICTouchAPI.debugServices.warning("UIElements.List.ListControlBase - selectItemByPosition / Couldn't select item as item at position " + itemPos + " was not defined!");
			}
		},

		/**
		 * Select an item according to its intIndex property.
		 * 
		 * Note that:
		 * <ul>
		 * <li> it will scroll to the item's position (except if "doNotScroll" is set to true) and select the item </li>
		 * <li> it will not execute the clickCallback function of the list, except if "click" is set to true </li>
		 * <li> if the item to select is a principal item and the list's flag "autoUnfoldOnClick" is true (that's the case by default),
		 * the item will be automatically unfolded (except if "doNotUnfold" is set to true) </li>
		 * <li> if the item to select is a secondary item and the principal item is currently folded, the principal item will be
		 * automatically unfolded and the secondary item will be selected (except if "doNotUnfold" is set to true) </li>
		 * </ul>
		 * @param {String} index intIndex of the itemm to select
		 * @param {Boolean} click handle click ? If true, the clickCallback function of the list will be executed
		 * @param {Boolean} doNotScroll if true, select the item without scrolling to its position
		 * @param {Boolean} doNotUnfold if true, do not unfold the primary entry when selecting it
		 */
		selectItemByIndex : function (index, click, doNotScroll, doNotUnfold) {
		    var item = this.findItem("intIndex", index);
			if (item) {
				// if the selected entry is a secondary item and the principal item is folded, unfold it
				if (!doNotUnfold && this.isSecondaryItem(item) && this._isItemFolded(this.getPrincipalItem(item))) {
					this.unfoldPrincipalItem(this.getPrincipalItem(item));
				}
				if(doNotScroll) {
					// Select the item
					this._selectLi(this._getItemInDom(item), !click, doNotUnfold);
				} else {
					// Select the item we're scrolling to
					this._selectLi(this.scrollToItemByPosition(this._getItemPosition(item)), !click, doNotUnfold);
				}
			}
			else {
				ICTouchAPI.debugServices.warning("UIElements.List.ListControlBase - selectItemByIndex / Couldn't select item, item with intIndex " + index + " was not found!");
		    }
		},


		/**
	 * Select first list item
	 * Only works if there's no categories.
	 */
		selectFirst : function () {
			// If not a list with categories and there are items
			if (!this._isListWithCategories()) {
				// Select first item
				this._selectLi(this._LisInDom[0]);
			}
		},

		/**
	 * Uncollapse a category
	 * By default, the first item will be selected.
	 * @param {Object} item elem to uncollapse
	 * @param {Boolean} dontSelectFirstItem don't select the first item
	 * @return {Object} this
	 */
		uncollapseCategory : function (item, dontSelectFirstItem) {
			// Before uncollapsing a new one,
			// collapse the previous one, if autoCollapse is enabled.
			this.getAutoCollapse() && this.collapseAll();
			// Set the whole category uncollapsed
			this._setCategoryCollapse(item, false);
			// Insert its domNodes
			// The first inserted domNode will be returned.
			this._emptyList();
			this._resetDomScroll();
			var firstNode = this.insert(0, this._getItemPosition(item),  this.nbElementsMax, false);
			// fill the list at the top if needed.
			this._fillList();
			// Select first item if required and it exists.
			if (!dontSelectFirstItem) {
				// Select the domNode
				this._selectLi(this._getNextLiInDom(firstNode));
			}
			return this;

		},

		/**
	 * Collapse a category
	 * @param {Object} item elem to collapse
	 * @return {Object} this
	 */
		collapseCategory : function (item) {
			// Set all the category collapsed
			this._setCategoryCollapse(item, true);
			// Remove its domNodes if any
			var elem = this._getItemInDom(item);

			if  (elem) {
				this.slice(this.getLiPosition(elem)+1, this._getNbLisDisplayedInCat(elem), true);
			}
			// fill the list at the top if needed.
			this._fillList();
			// Call the associated callback.
			this._executeCallback("funcCollapsed");

			return this;

		},

		/**
	 * Collapse all categories
	 * @return {Object} this
	 */
		collapseAll : function () {
			// Collapse current if any
			var currentUnCollapsed = this._getCurrentUnCollapsed();
			if (currentUnCollapsed) {
				this.collapseCategory(currentUnCollapsed);
			}
			return this;
		},

		/**
	 * Returns current selected dom node
	 * @return {Object} currently selected object
	 */
		getCurrentSelected : function () {
			return this._currentSelected;
		},

		/**
	 * Deselect the current item.
	 */
		deselect : function () {
			this._deSelectCurrent();
		},

		/**
	 * Get autoCollapse value
	 * @return {Boolean} autoCollapse value
	 */
		getAutoCollapse : function () {
			return this.autoCollapse;
		},

		/**
	 * Set autoCollapse value;
	 * @param {Boolean} bool new value for autoCollapse
	 * @return {Boolean} results of the affectation
	 */
		setAutoCollapse : function (bool) {
			return this.autoCollapse = bool;
		},

		/**
		 * Get autoFold value
		 * @return {Boolean} autoFold value
		 */
		getAutoFold : function () {
			return this.autoFold;
		},

		/**
		 * Set autoFold value;
		 * @param {Boolean} bool new value for autoFold
		 * @return {Boolean} results of the affectation
		 */
		setAutoFold : function (bool) {
			return this.autoFold = bool;
		},

		/**
		 * Get autoUnfoldOnClick value
		 * @return {Boolean} autoUnfoldOnClick value
		 */
		getAutoUnfoldOnClick : function () {
			return this.autoUnfoldOnClick;
		},

		/**
		 * Set autoUnfoldOnClick value;
		 * @param {Boolean} bool new value for autoUnfoldOnClick
		 * @return {Boolean} results of the affectation
		 */
		setAutoUnfoldOnClick : function (bool) {
			return this.autoUnfoldOnClick = bool;
		},

		/**
		 * Get insertAsFolded value
		 * @return {Boolean} insertAsFolded value
		 */
		getInsertAsFolded : function () {
			return this.insertAsFolded;
		},

		/**
		 * Set insertAsFolded value;
		 * @param {Boolean} bool new value for insertAsFolded
		 * @return {Boolean} results of the affectation
		 */
		setInsertAsFolded : function (bool) {
			return this.insertAsFolded = bool;
		},

		/**
	 * Just give an item the select's style
	 * @param {Number} intIndex index of the element
	 * @param {Boolean} boolSelect
	 * @return {Object}
	 */
		toggleEmphasis : function (intIndex, boolSelect) {
			return this._toggleSelectEntryByItem(this.findItem("intIndex", intIndex), boolSelect);
		},

		/**
	 * Hides an item
		 * @param {Number} intPosition position of item to hide
	 */
		hideItem : function (intPosition) {
			this._setItemInvisibility(this.arrItems[intPosition], true);
			this._removeItemInDom(this.arrItems[intPosition]);
		},

		/**
	 * Hides an item
	 * @param {Number} intIndex index of the item to hide
	 */
		hideItemByIntIndex : function (intIndex) {
			this.hideItem(this._getItemPositionByIntIndex(intIndex));
		},

		/**
	 * Shows an item
		 * @param {Object} intPosition position of item to show
	 */
		showItem : function (intPosition) {
			this._setItemInvisibility(this.arrItems[intPosition], false);
			// if the item to show is in first position of arrItems, unshift it to the DOM
			if (intPosition === 0) {
				this.unshift(1);
			}
			// if the item to show is in last position of arrItems, push it to the DOM
			else if (intPosition === this.arrItems.length-1) {
				this.push(1);
			}
			// else insert it after the previous displayable item of arrItems
			else {
				// get the previous item which is displayable (not callapsed and not invisible)
				var previousDisplayableItem = this._getPreviousDisplayableItem(intPosition);
				if (previousDisplayableItem) {
					var insertPointDom = this.getLiPosition(this._getItemInDom(previousDisplayableItem));
					// if we found an item displayable, get its position in DOM (if it is not in DOM, it is not visible because of the pagination,
					// so do not insert the shownItem in the DOM - it will be inserted when its position will be in the visible area)
					if (insertPointDom !== undefined) {
						this.insert(insertPointDom, intPosition, 1, true);
					}
				}
				// if there is no item displayable before the one to show, unshift if to the DOM
				else {
					this.unshift(1);
				}
			}
		},

		/**
		 * Shows an item
		 * @param {String/Number} intIndex index of the item to show
	 */
		showItemByIntIndex : function (intIndex) {
			this.showItem(this._getItemPositionByIntIndex(intIndex));
		},

		/**
		 * Find item according to one of their attributes value.
	 * @param {Object} what the property to test
	 * @param {Object} equals the value of the property that is looked for
		 * @return {Object} The first element that match the search
	 */
		findItem : function (what, equals) {
			return this.findItems(what, equals)[0];
		},

		/**
		 * Find items according to one of their attributes value.
		 * @param {Object} what the property to test
		 * @param {Object} equals the value of the property that is looked for
		 * @return {Array} The elements that match the search
		 */
		findItems : function (what, equals) {
			var func = function (item) {
				// If this item's attribute equals..
				return (item && item[what] === equals);
			}
			return this.arrItems.filter(func);
		},

		/**
		 * Get an item from its intIndex.
		 * @param {String/Number} intIndex the item's index
		 * @return {Object} the first item matching the given intIndex
		 */
		getItemByIntIndex : function(intIndex) {
			return this.findItem("intIndex", intIndex);
		},

		/**
		 * Get an item from its position (in arrItems).
		 * @param {Number} intPosition the item's position
		 * @return {Object} the item at the given position (in arrItems)
		 */
		getItemByPosition : function (intPosition) {
			return this.arrItems[intPosition];
		},


		/**
		 * Check if an item is a principal item.
		 * @param {Object} item the item to check
		 * @return {Boolean} true if the item is a principal item, false otherwise
		 */
		isPrincipalItem: function (item) {
			// Check if an item is a principal item (i.e. it has the _subType equal to this.PRINCIPAL_ENTRY)
			return (item && this._getItemSubType(item) === this.PRINCIPAL_ENTRY);
		},
		

		/**
		 * Check if an item is a secondary item.
		 * @param {Object} item the item to check
		 * @return {Boolean} true if the item is a secondary item, false otherwise
		 */
		isSecondaryItem: function (item) {
			// Check if the item is a secondary item (i.e. it has the _subType equal to this.SUB_ENTRY and its field _principalEntryIndex is defined)
			return (item && (this._getItemSubType(item) === this.SUB_ENTRY && item._principalEntryIndex !== undefined));
		},


		/**
		 * Get the secondary items associated to a principal item.
		 * @param {String/Number} principalEntryIndex the principal item's intIndex
		 * @return {Array} an array containing the secondary items of this principal item (empty array if there is no secondary items)
		 */
		getSecondaryItems : function (principalEntryIndex) {
			return this.findItems("_principalEntryIndex", principalEntryIndex);
		},


		/**
		 * Get the number of secondary items associated to a principal item.
		 * @param {String/Number} principalEntryIndex the principal item's intIndex
		 * @return {Number} the number of secondary items for this principal item (0 if there is no secondary items for this principal item)
		 */
		getNbSecondaryItems : function (principalEntryIndex) {
			var secondaryItems = this.getSecondaryItems(principalEntryIndex);
			return (secondaryItems && secondaryItems.length) || 0;
		},


		/**
		 * Get the principal item's intIndex from a secondary item.
		 * @param {Object} secondaryItem the secondary item
		 * @return {String/Number} the principal item's intIndex
		 */
		getPrincipalItemIndex : function (secondaryItem) {
			return (secondaryItem && secondaryItem._principalEntryIndex);
		},


		/**
		 * Get the principal item's position (in arrItems) from a secondary item.
		 * @param {Object} secondaryItem the secondary item
		 * @return {Number} the principal item's position (in arrItems)
		 */
		getPrincipalItemPosition : function (secondaryItem) {
			return this._getItemPositionByIntIndex(this.getPrincipalItemIndex(secondaryItem));
		},


		/**
		 * Get the principal item from a secondary item.
		 * @param {Object} secondaryItem the secondary item
		 * @return {Object} the principal item
		 */
		getPrincipalItem : function (secondaryItem) {
			return this.getItemByIntIndex(this.getPrincipalItemIndex(secondaryItem));
		},

		/**
	 * Push a new item at the end.
	 * @param {Object} item item to be pushed
	 * @param {Boolean} addNow if the element is to be displayed right away
	 * @return {Boolean} result of the push operation
	 */
		pushItem : function (item,  addNow) {
			// Split collapsable items if arrSecondaryItems is filled
			var items = this._splitSecondaryItems(item);
			var nbItems = items.length;
			this.arrItems = this.arrItems.concat(items);
			this._checkForEmpty();
			return addNow && this.push(nbItems);
		},

		/**
	 * Push an array of new items at the end.
	 * @param {Array} arrItems array of the items to be pushed
	 * @param {Boolean} addNow if the element is to be displayed right away
	 * @return {Boolean} result of the push operation
	 */
		pushItems : function (arrItems, addNow) {
			var func = function (item) {
				this.pushItem(item, addNow);
			}
			var returnValue = arrItems.forEach(func, this);
			this._checkForEmpty();
			return returnValue;
		},

		/**
		 * Push secondary item(s) to a principal item.
		 * Note: If the item was not a principal item (i.e. it had 0 secondary items), it will be displayed as a principal item after the push operation.
		 * ex: [..., item_x, sub_item_x1, sub_item_x2, item_x+1,...] -> [..., item_x, sub_item_x1, sub_item_x2, <b>sub_item_x3</b>, item_x+1,...]
		 * @param {String/Number} principalEntryIndex the index of the principal entry to which the secondary items will be added
		 * @param {Array} secondaryItems the secondary items to be added
		 * @return {Array} the result of the insert
		 */
		pushSecondaryItems : function (principalEntryIndex, secondaryItems) {
			return this.insertSecondaryItems(principalEntryIndex, this.getNbSecondaryItems(principalEntryIndex)+1, secondaryItems);
		},

		/**
		 * Add a new item at the begining.
	 * @param {Object} item item to be unshifted
	 * @param {Boolean} addNow to be added now ?
	 * @return {Boolean} result of the operation
	 */
		unshiftItem : function (item, addNow) {
			// Split collapsable items if arrSecondaryItems is filled
			var items = this._splitSecondaryItems(item);
			var nbItems = items.length;
			this.arrItems = items.concat(this.arrItems);
			this._checkForEmpty();
			return addNow && this.unshift(nbItems);
		},

		/**
		 * Add multiple items from the begining.
	 * @param {Array} arrItems array of the items to be unshifted
	 * @param {Boolean} addNow to be added now ?
	 * @return {Boolean} result of the operation
	 */
		unshiftItems : function (arrItems, addNow) {
			var func = function (item) {
				this.unshiftItem(item, addNow);
			}
			var returnValue = arrItems.forEach(func, this);
			this._checkForEmpty();
			return returnValue;
		},

		/**
		 * Unshift secondary item(s) to a principal item.
		 * Note: If the item was not a principal item (i.e. it had 0 secondary items), it will be displayed as a principal item after the unshift operation.
		 * ex: [..., item_x, sub_item_x1, sub_item_x2, item_x+1,...] -> [..., item_x, <b>sub_item_x0</b>, sub_item_x1, sub_item_x2, item_x+1,...]
		 * @param {String/Number} principalEntryIndex the index of the principal entry to which the secondary items will be added
		 * @param {Array} secondaryItems the secondary items to be added
		 * @return {Array} the result of the insert
		 */
		unshiftSecondaryItems : function (principalEntryIndex, secondaryItems) {
			return this.insertSecondaryItems(principalEntryIndex, 0, secondaryItems);
		},

		/**
		 * Update an item and refresh its corresponding li element.
		 * 
		 * If the item provided is a secondary item, it will be updated.
		 *
		 * If the item provided is a principal item, it will be updated and its secondary items will be updated too:
		 * <ul>
		 * <li> modification/addition/suppression of secondary items is managed,
		 * <li> if the item was not a principal item but the item provided in updateItem is, its secondary items will be automatically added
		 *		and the item will appear as a principal item (according to the graphical implementation of the UIElements inheriting List),
		 * <li> if the item was a principal item but the item provided in updateItem doesn't have secondary items, its secondary items will
		 *		be removed and it will not appear as a principal item anymore (according to the graphical implementation of the UIElements inheriting List).
		 * </ul>
		 *
		 * For performance reason, <b>updateItem</b> checks if the item needs to be updated (i.e. fully destroyed and rebuilt, really CPU coslty) using the
		 * abstract function "\_isUpdateNeeded" which has to be implemented by UIElements inheriting MenuList. If the function "\_isUpdateNeeded"
		 * is not overwritten, by default the item will be updated.
		 *
		 * It is possible to bypass the test done in "\_isUpdateNeeded" by setting the argument bForceUpdate to true <u>(not recommended)</u>.
		 * @param {Object} item item to be updtated (can be standard item, principal item or secondary item)
		 * @param {Boolean} bForceUpdate force the update (don't use the function "\_isUpdateNeeded" to check if an update is needed) <u>(not recommended)</u>
		 */
		updateItem : function (item, bForceUpdate) {
			// Get the old item.
			var oldItem = this.findItem("intIndex", item.intIndex);
			if (oldItem) {
				var selectedItem;
				// if item has secondary items provided, check if the item was already a principal item (look for secondary items to be added/removed/updated)
				// or not (add the secondary items and the item will become a principal item)
				if (item.arrSecondaryItems) {
					// if the old item was already a principal item, check if some of its secondary items have to be added/removed/updated
					if (this.isPrincipalItem(oldItem)) {
						// if the selected item was a secondary item of this item, save its index to re-select it after, as it may have been moved
						selectedItem = (this.isSecondaryItem(this.getCurrentSelectedItem()) && this.getPrincipalItem(this.getCurrentSelectedItem()) == oldItem) ? this.getCurrentSelectedItem() : undefined;
						var i;
						// browse through the new item's secondary items and look for secondary items to update/add in oldItem
						for (i = 0 ; i < item.arrSecondaryItems.length ; i++) {
							// if the secondary item at position 'i' was already at position 'i' in the oldItem, update this secondary item (if needed)
							if (oldItem.arrSecondaryItems[i] && item.arrSecondaryItems[i].intIndex === oldItem.arrSecondaryItems[i].intIndex) {
								this.updateItem(item.arrSecondaryItems[i]);
							}
							// else the secondary item at position 'i' was not at the same position in the oldItem, insert this secondary item at position 'i'
							else {
								this.insertSecondaryItems(item.intIndex, i, item.arrSecondaryItems[i]);
							}
						}
						// browse through oldItem's secondary items and look for secondary items to remove
						for (i = 0 ; i < oldItem.arrSecondaryItems.length ; i++) {
							// if the secondary item at position 'i' in the oldItem is not at the same position in the new item, remove this secondary item and
							// all the secondary items following this one; then exit the loop
							if (!item.arrSecondaryItems[i]) {
								this.sliceSecondaryItems(item.intIndex, i, oldItem.arrSecondaryItems.length-i);
								break;
							}
						}

					}
					// else the old item was not a principal item, so insert all the secondary items (the item will automatically become a principal item)
					else {
						this.insertSecondaryItems(item.intIndex, 0, item.arrSecondaryItems);
					}
				}
				// item doesn't have secondary items provided, check if the item was already a principal item. If it was, remove all its secondary items
				// (the item will automatically become a standard item)
				else if (this.isPrincipalItem(oldItem)) {
					// if the selected item was a secondary item of this item, save its index to select the principal item, as the secondaries will be removed
					selectedItem = (this.isSecondaryItem(this.getCurrentSelectedItem()) && this.getPrincipalItem(this.getCurrentSelectedItem()) == oldItem) ? this.getCurrentSelectedItem() : undefined;
					this.emptySecondaryItems(oldItem.intIndex);
				}
				// determine if the item need to be updated graphycally:
				// if bForceUpdate is not set to true, check if the item really need to be updated, using function _isUpdateNeeded (to be
				// overwritten by UIElements inheriting List, otherwise the update will always be done even if it is not necessary)
				var boolUpdate = bForceUpdate || this._isUpdateNeeded(oldItem, item );
			// Get its index
			var idx = this._getItemPosition(oldItem);
			// Mix the new item in the old one.
			this.arrItems[idx] = dojo.mixin(oldItem, item);
				// update graphycally or not according to the computed boolUpdate
				if (boolUpdate) {
			// Get its position in the dom.
			var lipos = this.getLiPosition(this._getItemInDom(oldItem));
			// Refresh the li
			this._updateLi(lipos, idx);
				}
				// if selectedItem is defined, the selected item before the update was a secondary item of the principal item updating.
				if (selectedItem) {
					// if the item which was selected is still in the list, select it again (it may have been removed/re-added during the update)
					if (this.getItemByIntIndex(selectedItem.intIndex)) {
						this.selectItemByIndex(selectedItem.intIndex, false, true, true);
					}
					// if the item which was selected is no longer in the list, select the item updating (i.e. the parent of seletedItem, which has been deleted during the update)
					else {
						this.selectItemByIndex(this.getPrincipalItemIndex(selectedItem), true);
					}
				}
			}
			else {
				ICTouchAPI.debugServices.warning("UIElements.List.ListControlBase - updateItem / Couldn't update item as the item provided don't exists in the list.");
			}
		},

		/**
		 * Pop an item from the list.
		 * Note: If the item to pop is a secondary item, the whole corresponding principal item is popped (including its secondary items)
		 * ex: [..., item_x, sub_item_x1, sub_item_x2, item_x+1,...] -> [..., item_x, sub_item_x1, item_x+1,...]
		 * @param {Array} completeList if the list should'nt be completed
		 * @return {Object} popped Item(s)
	 */
		popItem : function (completeList) {
			// Check if the item to pop is a secondary item. If that's the case, pop the whole last item (including its secondary items)
			if (this._getItemSubType(this.arrItems[this.arrItems.length-1]) === this.SUB_ENTRY) {
				return this.sliceItems(this.getPrincipalItemPosition(this.arrItems[this.arrItems.length-1]), this.arrItems.length, !completeList);
			}
			else {
			var item = this.arrItems.pop();
			this._checkForEmpty();
			return this._removeItemInDom(item, !completeList);
			}
		},

		/**
		 * Pop a secondary item from a principal item.
		 * Note: If the principal item don't have secondary items anymore after the pop operation, it will not be displayed as a principal item anymore.
		 * @param {String/Number} principalEntryIndex the index of the principal item from which the secondary item will be popped
		 * @param {Array} completeList if the list should'nt be completed
		 * @return {Object} popped item
		 */
		popSecondaryItem : function (principalEntryIndex, completeList) {
			return this.sliceItem(this._getLastSecondaryItemPosition(principalEntryIndex), !completeList);
		},

		/**
	 * Pop multiple items from the list
	 * @param {Number} nb number of element to pop out
		 * @return {Array} popped elements
	 */
		popItems : function (nb) {
			for (var i=0; i<nb; i++) {
				this.popItem();
			}
			this._checkForEmpty();
			return this.arrItems;
		},

		/**
		 * Remove one item from the top of the list.
		 * Note: If the item to shift is a principal item, the whole principal item is shifted, including its secondary items
	 * @param {Array} completeList List from which to shift an item
	 * @return {Object} li or this
	 */
		shiftItem : function (completeList) {
			// Check if the item to shift is a principal item. If that's the case, shift the whole first item (including its secondary items)
			if (this.arrItems[0]._subType === this.PRINCIPAL_ENTRY) {
				return this.sliceItems(0, this._getLastSecondaryItemPosition(this.arrItems[0].intIndex)+1, !completeList);
			}
			else {
			var item = this.arrItems.shift();
			this._checkForEmpty();
			return this._removeItemInDom(item, !completeList);
			}
		},

		/**
		 * Shift a secondary item from a principal item.
		 * Note: If the principal item don't have secondary items anymore after the shift operation, it will not be displayed as a principal item anymore.
		 * ex: [..., item_x, sub_item_x1, sub_item_x2, item_x+1,...] -> [..., item_x, sub_item_x2, item_x+1,...]
		 * @param {String/Number} principalEntryIndex the index of the principal item from which the secondary item will be shifted
		 * @param {Array} completeList if the list should'nt be completed
		 * @return {Object} shifted item
		 */
		shiftSecondaryItem : function (principalEntryIndex, completeList) {
			return this.sliceItem(this._getFirstSecondaryItemPosition(principalEntryIndex), !completeList);
		},

		/**
	 * Remove multiple items from the top of the list.
	 * @param {Number} nb number of elements to shifted
	 * @return {Array} local array of items
	 */
		shiftItems : function (nb) {
			for (var i=0; i<nb; i++) {
				this.shiftItem();
			}
			this._checkForEmpty();
			return this.arrItems;
		},

		/**
		 * Remove an item and its corresponding li.
		 * Note: If the item to slice is a principal item, the whole principal item is sliced, including its secondary items.
		 * @param {Number} itemPos position of element to slice
		 * @param {Boolean} dontCompleteList don't complete the list
		 * @return {Object} sliced li or false
		 */
		sliceItem : function (itemPos, dontCompleteList) {
			if (this.arrItems[itemPos]) {
				if (this.arrItems[itemPos]._subType === this.PRINCIPAL_ENTRY) {
					// Splice arrItems
					return this.sliceItems(itemPos, this._getLastSecondaryItemPosition(this.arrItems[itemPos].intIndex) + 1 - itemPos, dontCompleteList);
				} else {
					var isSecondaryItem = this.isSecondaryItem(this.arrItems[itemPos]);
					var secondaryItemPosition = isSecondaryItem && this._getSecondaryItemPosition(this.arrItems[itemPos]);
					var wasLastSecondaryItem = isSecondaryItem && this._isLastSecondaryItem(this.arrItems[itemPos]);
					var item = this.arrItems.splice(itemPos, 1)[0];
					// manage the case a secondary item is spliced (refresh the principal item and the last secondary item if needed)
					if (isSecondaryItem) {
						var principalItem = this.getPrincipalItem(item);
						// update the array of secondary items of the principal item
						principalItem.arrSecondaryItems.splice(secondaryItemPosition, 1);
						// Update graphically the principal item if needed
						this._checkUpdatePrincipalItem(principalItem.intIndex);
						// if we removed the last secondary item and the principal item is still a principal item after the slice,
						// add the class LastOne to its last secondary item
						if (wasLastSecondaryItem && this.isPrincipalItem(principalItem)) {
							this._updateSecondaryItemLastOneClass(this._getLastSecondaryItem(principalItem.intIndex), true);
						}
					}
				this._checkForEmpty();
				return this._removeItemInDom(item, dontCompleteList);
				}
			} else {
				return false;
			}
		},

		/**
		 * Remove secondary item(s) from a principal item (according to its position in arrSecondaryItems) and its corresponding li.
		 * If the secondaryItemStart is lower than 0, secondaryItemStart is set to 0 (the slice will start from the first secondary item).
		 * If secondaryItemStart + range is greater than the number of secondary items currently present, the slice will stop at the last secondary item
		 * Note: If the principal item don't have secondary items anymore after the slice operation, it will not be displayed as a principal item anymore
		 * ex: [..., item_x, sub_item_x1, sub_item_x2, sub_item_x3, sub_item_x4, item_x+1,...] -> [..., item_x, sub_item_x1, sub_item_x4, item_x+1,...]
		 * @param {String/Number} principalEntryIndex index of the principal item
		 * @param {Number} secondaryItemStart position of first secondary item to slice
		 * @param {Number} range range of secondary items to be sliced
		 * @param {Boolean} dontCompleteList don't complete the list
		 * @return {Object} sliced li or false
		 */
		sliceSecondaryItems : function (principalEntryIndex, secondaryItemStart, range, dontCompleteList) {
			// check that the principal item well exists... if not, just return empty array
			if (!this.getItemByIntIndex(principalEntryIndex)) {
				ICTouchAPI.debugServices.warning("UIElements.List.ListControlBase - sliceSecondaryItems / Couldn't remove secondary items as there is no item having the given principalEntryIndex: " + principalEntryIndex);
				return [];
			}
			else {
				// check that secondaryItemPos is well in the range of secondaryItems, else if out of range slice the last secondary item
				var nbSecondaryItems = this.getNbSecondaryItems(principalEntryIndex);
				secondaryItemStart = (0 < secondaryItemStart) ? secondaryItemStart : 0;
				range = (secondaryItemStart + range <= nbSecondaryItems) ? range : nbSecondaryItems;
				// compute the position of secondaryItemStart in arrItems
				var startSlice = this._getItemPosition(this.getSecondaryItems(principalEntryIndex)[secondaryItemStart]);
				// remove the secondary items
				var slicedItems = this.sliceItems(startSlice, range, dontCompleteList);
				// update the array of secondary items of the principal item
				this.getItemByIntIndex(principalEntryIndex).arrSecondaryItems.splice(secondaryItemStart, range);
				// if there is no longuer secondary items for the principal item, update it
				this._checkUpdatePrincipalItem(principalEntryIndex);
				// if we removed the last secondary item and the principal item is still a principal item after the slice, add the class LastOne to its last secondary item
				if (secondaryItemStart+range === nbSecondaryItems && this.isPrincipalItem(this.getItemByIntIndex(principalEntryIndex))) {
					this._updateSecondaryItemLastOneClass(this._getLastSecondaryItem(principalEntryIndex), true);
				}
				return slicedItems;
			}
		},
		
		/**
		 * Remove an item and its corresponding li given its intIndex.
		 * The item to remove can also be a secondary item.
		 * If the item to remove is a principal item, the whole principal item is removed, including its secondary items.
	 * If the deleted item was selected, it selects the new one and executes the callback.
	 * dontAutoSelectNext true if you don't want to have the next item to be selected
		 * @param {String/Number} intIndex the intIndex property of the item to delete
	 * @param {Boolean} dontAutoSelectNext to prevent the next item to be selected
		 * @return {Object} the removed item (an array is a principal item is removed with its secondary items)
	 */
		removeItemByIntIndex : function (intIndex, dontAutoSelectNext) {
			var sliceItem = null;
			// Get the item in the list according to the intIndex
			var item = this.getItemByIntIndex(intIndex);
			if(item) {
				// If the item is in the dom
				var isInDom = this._getItemInDom(item);
				// Get the position of the item in the dom
				var itemPos = this._LisInDom.indexOf(isInDom);
				// Is the current select item.
				var currentSelected = !!(isInDom === this.getCurrentSelected() || (this.getCurrentSelectedItem() && this.isSecondaryItem(this.getCurrentSelectedItem()) && this.getPrincipalItem(this.getCurrentSelectedItem()) == item));
				// Remove the item from the list and from the dom
				sliceItem =  this.sliceItem(this._getItemPosition(item));
				// And select the new one which took the place.
				if (currentSelected) {
					// the currentSelected item (or the principal item of the currentSelected item) has been removed, so set CurrentSelected to null
					this._setCurrentSelected(null);
					if (!dontAutoSelectNext) {
					// Select the new one which took the place (the "next one")
					// Or select the previous one if we were at the end of the list.
					this._selectLi(this._LisInDom[itemPos]) || this._selectLi(this._LisInDom[itemPos-1]);
				}
				}

			}
			return sliceItem;
		},

		/**
		 * Remove item(s) and their corresponding li
		 * Do not use if the list contains secondary items (do not managed by sliceItems)
		 * @ignore
		 * @param {Number} start starting position
	 * @param {Number} range range od items to be sliced
	 * @param {Boolean} dontCompleteList don't complete the list
	 * @return {Array} sliced Items
	 */
		sliceItems : function (start, range, dontCompleteList) {
			
			// Splice arrItems
			var items = this.arrItems.splice(start, range);
			// function remove lis
			var func = function (item) {
				this._removeItemInDom(item, dontCompleteList);
			}
			// Execution function for each item
			items.forEach(func, this);
			this._checkForEmpty();
			// Return an array of removed elements
			return items;
		},

		/**
		 * Insert item(s) at a given insert point.
		 * Note that the provided insert point don't take into account the secondary items that some items can have.
		 * Ex: if insertItems is called with insertPoint = 2, the items will be inserted before the 2nd item
		 * (whatever the item at position 0 and 1 have secondary items or not).
		 * @param {Number} insertPoint insert point
		 * @param {Array} items elements to be added
	 * @return {Array} the result of the insert
	 */
		insertItems : function (insertPoint, items) {
			// Check if items is a list, else fake one
			items = dojo.isArray(items) ? items : [items];
			// Split collapsable items if arrSecondaryItems is filled
			items = this._splitSecondaryItemsFromArray(items);
			// Insert the items in the items list (take into account the potential secondary items that some items can have by using _computeInsertPoint)
			this.arrItems = this._insertInArray(this.arrItems, this._computeInsertPoint(insertPoint), items);
			this._checkForEmpty();
			// Get the next index of the first inserted item.
			var newIndex = this._getItemPosition(items[0]);
			// Get the previous item in the dom, if any
			var  previousItemIndex = this.getLiPosition(this._getItemInDom(this.arrItems[newIndex-1]));
			// Insert after the previous item or at the end.
			var elt = this.insert(previousItemIndex, newIndex, items.length, true);
			items = null;
			return elt;
		},

		/**
		 * insert secondary item(s) to a principal item at a given position
		 * If the insertPoint is lower than 0, the secondary items are inserted at the position 0 (in first position of the secondary items).
		 * If the insertPoint is greater than the number of secondary items currently present, the secondary items are inserted at the end of the secondary items array.
		 * Note: If the item was not a principal item (i.e. it had 0 secondary items), it will be displayed as a principal item
		 * ex: [..., item_x, sub_item_x1, sub_item_x2, item_x+1,...] -> [..., item_x, sub_item_x1, <b>sub_item_x3</b>, sub_item_x2, item_x+1,...]
		 * @param {String/Number} principalEntryIndex the index of the principal entry to which the secondary items will be added
		 * @param {Number} insertPoint insert point
		 * @param {Array} secondaryItems the secondary elements to be added
		 * @return {Array} the result of the insert
		 */
		insertSecondaryItems : function (principalEntryIndex, insertPoint, secondaryItems) {
			// check that the principal item well exists... if not, just return empty array
			if (!this.getItemByIntIndex(principalEntryIndex)) {
				ICTouchAPI.debugServices.warning("UIElements.List.ListControlBase - insertSecondaryItems / Couldn't insert secondary items as there is no item having the given argument principalEntryIndex (" + principalEntryIndex + ')!');
				return [];
			}
			if (!secondaryItems) {
				ICTouchAPI.debugServices.warning("UIElements.List.ListControlBase - insertSecondaryItems / Couldn't insert secondary items as argument secondaryItems is not defined!");
				return [];
			}
			else {
				// Check if items is a list, else fake one
				secondaryItems = dojo.isArray(secondaryItems) ? secondaryItems : [secondaryItems];
				// format secondary items
				var principalEntry = this.getItemByIntIndex(principalEntryIndex);
				this._formatSecondaryItems(secondaryItems, principalEntry);
				// compute the position of insertPoint in arrItems
				var insertPosition;
				// check that the principal item we wants to insert items is already a principal item ; if that's not the case, we insert the secondary items directly after the principal item
				if (this.isPrincipalItem(principalEntry)) {
					// check that insertPoint is well in the range of secondaryItems, else if out of range insert them after the last secondary item
					var nbSecondaryItems = this.getNbSecondaryItems(principalEntryIndex);
					insertPoint = (0 < insertPoint) ? insertPoint : 0;
					insertPoint = (insertPoint <= nbSecondaryItems) ? insertPoint : nbSecondaryItems;
					// If we insert the secondaryItem in last position, remove the class "LastOne" from the current last item 
					// (the class will be automatically added to the new one at its creation by _getLiTemplate)
					if (insertPoint === nbSecondaryItems) {
						this._updateSecondaryItemLastOneClass(this._getLastSecondaryItem(principalEntryIndex), false);
					}
					insertPosition = this._getFirstSecondaryItemPosition(principalEntryIndex) + insertPoint;
					// update the array of secondary items of the principal item
					principalEntry.arrSecondaryItems = this._insertInArray(principalEntry.arrSecondaryItems, insertPoint, secondaryItems);
				}
				else {
					// insert the secondary items directly after the principal item
					insertPosition = this._getItemPosition(principalEntry) + 1;
					// create the array of secondary items of the principal item
					principalEntry.arrSecondaryItems = secondaryItems;
				}
				// Insert the items in the items list
				this.arrItems = this._insertInArray(this.arrItems, insertPosition, secondaryItems);
				// Update the principal item if needed
				this._checkUpdatePrincipalItem(principalEntryIndex);
				this._checkForEmpty();
				// if the principal item is unfolded, add the secondary items in the DOM
				if (!this._isItemFolded(principalEntry)) {
					// Get the next index of the first inserted item.
					var newIndex = this._getItemPosition(secondaryItems[0]);
					// Get the previous item in the dom, if any
					var  previousItemIndex = this.getLiPosition(this._getItemInDom(this._getPreviousDisplayableItem(newIndex)));
					// Insert after the previous item or at the end.
					var elt = this.insert(previousItemIndex, newIndex, secondaryItems.length, true);
				}
				secondaryItems = null;
				return elt;
			}
		},

		/**
	 * Remove all items from the list
	 */
		emptyItems : function() {
			return this.sliceItems(0, this.arrItems.length);
		},


		/**
		 * Remove all secondary items of a principal item and their corresponding li
		 * Note: If the principal item will not be displayed as a principal item anymore
		 * ex: [..., item_x, sub_item_x1, sub_item_x2, sub_item_x3, sub_item_x4, item_x+1,...] -> [..., item_x, item_x+1,...]
		 * @param {String/Number} principalEntryIndex index of the principal item
		 * @return {Object} sliced li or false
		 */
		emptySecondaryItems : function(principalEntryIndex) {
			return this.sliceSecondaryItems(principalEntryIndex, 0, this.getNbSecondaryItems(principalEntryIndex));
		},

		/**
		 * Unfold a principal item
		 * @param {Object} principalEntry principal item to unfold
		 */
		unfoldPrincipalItem : function (principalEntry) {
			// unfold the item only if it is curretnly folded
			if (this._isItemFolded(principalEntry)) {
				// Before unfolding a new one,
				// fold the other ones, if autoFold is enabled.
				this.getAutoFold() && this.foldAllPrincipalItems();
				// set principal item's folded flag as unfolded
				this._setItemFolded(principalEntry, false);
				// refresh principal item to appear unfolded
				this._updatePrincipalItem(principalEntry);
				// show all the secondary items associated to this principal item
				var func = function (item) {
					this.showItemByIntIndex(item.intIndex);
				}
				this.getSecondaryItems(principalEntry.intIndex).forEach(func, this);
			}
		},

		/**
		 * Unfold a principal item from its intIndex
		 * @param {String/Number} principalEntryIndex principal item's intIndex to unfold
		 */
		unfoldPrincipalItemByIndex : function (principalEntryIndex) {
			this.unfoldPrincipalItem(this.getItemByIntIndex(principalEntryIndex));
		},

		/**
		 * Fold a principal item
		 * @param {Object} principalEntry principal item to fold
		 */
		foldPrincipalItem : function (principalEntry) {
			// fold the item only if it is currently unfolded
			if (!this._isItemFolded(principalEntry)) {
				// set principal item's folded flag as folded
				this._setItemFolded(principalEntry, true);
				// refresh principal item to appear folded
				this._updatePrincipalItem(principalEntry);
				// hide all the secondary items associated to this principal item
				// and look if one of the secondary items was selected before the folding
				var currentSelectedItem = this.getCurrentSelectedItem();
				var selectedSecondaryItem = false;
				var func = function (item) {
					if (item === currentSelectedItem) {
						selectedSecondaryItem = true;
					}
					this.hideItemByIntIndex(item.intIndex);
				}
				this.getSecondaryItems(principalEntry.intIndex).forEach(func, this);
				// if a secondary item was selected when folding, select the principal item after all the secondary items has been hidden
				if (selectedSecondaryItem) {
					this.selectItemByIndex(principalEntry.intIndex, true, true, true);
				}
			}
		},

		/**
		 * Fold a principal item from its intIndex
		 * @param {String/Number} principalEntryIndex principal item's intIndex to fold
		 */
		foldPrincipalItemByIndex : function (principalEntryIndex) {
			this.foldPrincipalItem(this.getItemByIntIndex(principalEntryIndex));
		},

		/**
		 * Fold all the unfolded principal items
		 */
		foldAllPrincipalItems : function () {
			var arrUnfoldedItems = this.findItems("_isFolded", false);

			var func = function (item) {
				this.foldPrincipalItem(item);
			}
			arrUnfoldedItems.forEach(func, this);
		},

		/**
		 * Fold a primary item if it is unfolded, unfold it if it is folded
		 * @param {Object} principalEntry principal item to fold/unfold
		 */
		foldUnfoldPrincipalItem : function (principalEntry) {
			if (this._isItemFolded(principalEntry)) {
				this.unfoldPrincipalItem(principalEntry);
			}
			else {
				this.foldPrincipalItem(principalEntry);
			}
		},

		/**
		 * Fold a primary item if it is unfolded, unfold it if it is folded
		 * @param {String/Number} principalEntryIndex principal item's intIndex to fold/unfold
		 */
		foldUnfoldPrincipalItemByIndex : function (principalEntryIndex) {
			this.foldUnfoldPrincipalItem(this.getItemByIntIndex(principalEntryIndex));
		},

		/**
	 *   If long press callback is defined, execute callback.
	 *   The selected entry item index is given in parameter
	 *   @param {Object} liNode the selected item entry
	 */
		eventLongPress: function(liNode) {
			if (this.callbackLongPress && liNode && liNode._itemSource) {
				this._executeCallback("callbackLongPress", liNode._itemSource.intIndex);
			}
		},

		/* ----------------------------------- Private methods ------------------------------------- */


		/**
	 * @ignore
	 * launch pagination if the list has been scrolled
	 * @return {number} the amount of pixel scrolled
	 */
		pagination : function () {
                        var scrollTop = this.domNode.scrollTop;

                        // If the list is scrolled more than 50 pixels
			if (scrollTop >= 50) {
				// Push an item at the end of the list and remove the first one.
				// Return the amount of pixels to "unscroll"
				return -50 * this.push(1, true);
			// If the list is scrolled to the top most
			} else if (!scrollTop) {
				// Add an item at the top of the list and remove the last one
				return 50 *this.unshift(1, true);
			} else {
				// If nothing changed, return 0.
				return 0;
			}

		},

		/**
	 * A method to add (nb) elements at the end of the dom node.
	 * If the number of elements gets to big, removes the top nodes
	 * @ignore
	 * @param {Number} nb  number of elements to be added
	 * @param {Boolean} remove  true if we have to remove when nb > max
	 * @return {Number} the final array length
	 */
		push : function (nb, remove) {

			// We want to add stuff after the last displayed item :
			var start = this._getLastDisplayedItem()+1;
			// Can't add more than nbElementsMax item
			var range = remove && nb > this.nbElementsMax ? this.ElementsMax : nb;
			// Get a list of Lis to display, from a list of items
			var newElements = this._buildList(this._getNodesToDisplay(start, range));
			// Insert the domNodes in the main list and store the references as well.
			this._insertNodes(newElements[0], newElements[1], 0);
			// If asked, remove any exceeding node.
			remove && this.shift(this._getNbOfExceedingNodes());
			// Return the number of elements really pushed
			return newElements[1].length;
		},

		/**
	 * A method to remove (nb) elements at the end of the dom node.
	 * @ignore
	 * @param {Number} nb number of elements to pop out
	 * @return {Array} poped elements
	 */
		pop : function (nb) {
			// Calling the slice method with the starting point and the range
			// calculated from the current list length
			return this.slice(this._LisInDom.length-nb, nb);
		},

		/**
	 * Removes (nb) elements at the begining of the dom node
	 * @ignore
	 * @param {Number} nb number of element to pop out
	 * @return {Number} number of sliced arrays
	 */
		shift : function (nb) {
			// Calling the slice method with the starting point and the range
			return this.slice(0, nb);
		},

		/**
	 * Add (nb) elements at the begining of the dom node
	 * @ignore
	 * @param {Number} nb number of elements to be added
	 * @param {Boolean} remove true if we have to remove when nb > max
	 * @return {Number} the final array length
	 */
		unshift : function (nb, remove) {
			// As we want to insert nodes before the first displayed
			var start = this._getFirstDisplayedItem();
			// Be sure not to add more elements than possible
			var range = remove && nb > this.nbElementsMax ? this.ElementsMax : nb;
			// Get a list of Lis to display, from a list of items
			var newElements = this._buildList(this._getNodesToDisplay(start, range, true));
			// Insert the domNodes in the main list and store the references as well.
			this._insertNodes(newElements[0], newElements[1], this._LisInDom[0]);
			// If asked, remove any exceeding node.
			remove && this.pop(this._getNbOfExceedingNodes());
			// Return the number of elements really unshifted
			return newElements[1].length;
		},

		/**
	 * Removes a number of dom nodes from a specific point
	 * @ignore
	 * @param {Number} start starting point
	 * @param {Number} range range of item to be sliced
	 * @param {Boolean} add notice the query for addign element at the end
	 * @return {Array} number of sliced elements
	 */
		slice : function (start, range, add) {
			// Remove n domNodes
			for (var i=0, nb=0; i<range; i++, nb++) {
				this._removeOneLi(start);
			}
			// If asked, add n elements at the end.
			add && this.push(-this._getNbOfExceedingNodes());
			//add && this.unshift(-this._getNbOfExceedingNodes());
			// Returns the number of domNodes sliced.
			return nb;
		},

		/**
	 * Insert items from an Li
	 * @ignore
	 * @param {Object} fromLi  Elements to be added
	 * @param {Object} fromItem  starting Element
	 * @param {Number} nb  starting Element
	 * @param {Boolean} remove do we have to remove some elements ?
	 * @return {Array} poped elements
	 */
		insert : function (fromLi, fromItem, nb, remove) {

			// In this case, range is of course limited by the available nodes to display.
			var nextLi = this._LisInDom[fromLi+1];
			// Can't be more than the nbElementsMax
			// (check that the number of displayable elements between fromItem and the item after the insert point (fromLi) is not greater than nbElementsMax)
			var maxRange = nextLi ? (this._getNbDisplayableItems(fromItem, this._getItemPosition(nextLi._itemSource))) : this.nbElementsMax;
			// As long as nb is defined and less than maxRange
			var range = (nb > maxRange) || !nb ? maxRange : nb;
			// Get the elements
			var newElements = this._buildList(this._getNodesToDisplay(fromItem, range));
			// Insert them
			this._insertNodes(newElements[0], newElements[1], nextLi);

			// If asked, remove any exceeding node starting from the end
			remove && (this._getNbOfExceedingNodes() > 0) && this.pop(this._getNbOfExceedingNodes());
			// Then from the start
			remove && (this._getNbOfExceedingNodes() > 0) && this.shift(this._getNbOfExceedingNodes());
			// Or if not enough domNodes were added
			remove && (this._getNbOfExceedingNodes() < 0) && this.unshift(-this._getNbOfExceedingNodes());
			// Actually its more like a verify nb of nodes than a remove boolean
			remove && (this._getNbOfExceedingNodes() < 0) && this.push(-this._getNbOfExceedingNodes());

			// Returns a reference to the first domNode displayed.
			return newElements[1][0];
		},

		/**
		 * Find the previous displayable item in arrItems from a given pisition.
		 * @ignore
		 * @param {Number} intPosition position from which search the previous displayable item
		 * @return {Object} The previous displayable item (undefined if not found)
		 */
		_getPreviousDisplayableItem : function (intPosition) {
			var previousDisplayableItem;
			for (var i = intPosition-1; i >= 0 ; i--) {
				if (this._itemDisplayable(this.arrItems[i])) {
					previousDisplayableItem = this.arrItems[i];
					break;
				}
			}
			return previousDisplayableItem;
		},

		/**
		 * Get the number of displayable items in arrItems from a item to another one.
		 * @ignore
		 * @param {Number} fromItem position from which get the number of displayable items
		 * @param {Number} toItem position to which get the number of displayable items
		 * @return {Number} The number of displayable items
		 */
		_getNbDisplayableItems : function (fromItem, toItem) {
			var nbDisplayable = 0;
			for (var i = fromItem ; i < toItem ; i++) {
				if (this._itemDisplayable(this.arrItems[i])) {
					nbDisplayable++;
				}
			}
			return nbDisplayable;
		},


		/**
	 * @ignore
	 * Returns the index of a given li in the list.
	 * @param {Object} li  targeted element
	 * @return {Number} the position
	 */
		getLiPosition : function (li) {
			return +(this._LisInDom.indexOf(li));
		},

		/**
	 * un/collapse a category
	 * This method is called when a click occurs on a category.
	 * @ignore
	 * @param {Event} event the click event, unused
	 * @param {Array} args objects to collapse or not
	 */
		toggleCollapse : function (event, args) {
			// Is it collapsed?
			if (this._itemSource.bCollapse) {
				// Uncollapse the new one
				args.PARENT_WIDGET.uncollapseCategory(this._itemSource);
			} else {
				// Else, collapse the current.
				args.PARENT_WIDGET.collapseCategory(this._itemSource);
			}

		},

		/**
	 * Set a collapse value + style to a dom node
	 * @ignore
	 * @param {Object} item element to be setted up
	 * @param {Boolean} bCollapse collapse state
	 * @return {Boolean} new collapse state
	 */
		setCollapse : function (item, bCollapse) {
			if (item) {
				var elem = this._getItemInDom(item);
				if (elem) {
					dojo.toggleClass(elem, "collapsed", bCollapse);
				}
				this._setItemCollapse(item, bCollapse);
				return bCollapse;
			}
		},

		/**
	 * Called when a click occured on an item
	 * @ignore
	 * @param {Event} event the click event
	 * @param {Object} args args to be browsed
	 */
		selectedLi : function (event, args) {
			args.PARENT_WIDGET._selectLi(this, false, false, true);
		},

		/**
	 * Abstract Class - To be overloaded by child class
	 * Returns an li dom node for a given item.
	 * @ignore
		 * @abstract
	 * @param {Object} item the item from which the template is built
	 * @return {HTMLElement} an li dom node
	 */
		_getLiTemplate: function (item) {
			
		},

		/**
		 * Abstract function called when a request to update an item is done.
		 * By default, if this function is not overwritten, the function returns true so that the whole item will always be destroyed and rebuilt graphically.
		 * As the graphical operations are costly for CPU, overwrite this function in UIElements inheriting "List" would allow to gain performances by not
		 * refreshing the whole item when it's not needed.
		 * Ex:
		 * <ul>
		 * <li> if nothing has changed in the fields of the object concerning graphical element's part (label/icons/style...), the UIElement can return false as
		 *		an update is not needed (will avoid a useless costly refresh). </li>
		 * <li> if only a label changed in the newItem compared to what it is in the item displayed, the UIElement can change it direcly using _updateItemText
		 *		(will change only the text direcly in the dom) and return false (will be better for performances than rebuilding all the entry). </li>
		 *	</ul>
		 * @ignore
		 * @abstract
		 * @param {Object} oldItem the current state of the item
		 * @param {Object} newItem the new item
		 * @return {Boolean} true if an update is needed, false otherwise
		 */
		_isUpdateNeeded: function (oldItem, newItem) {			
			return true;
		},


		/**
		 * Abstract function called when the state folded/unfolded of a principal item changes.
		 * Function to be overwitten by UIElements inheriting "List" to manage graphically this changement.
		 * @ignore
		 * @abstract
		 * @param {Object} item the principal item
		 */
		_updateArrawPrincipalItem: function (item) {

		},

		/**
		 * Abstract function called when a secondary item [become/is not anymore] the last secondary item of its principal item.
		 * Function to be overwitten by UIElements inheriting "List" to manage graphically this changement.
		 * @ignore
		 * @abstract
		 * @param {Object} item the secondary item
		 * @param {Boolean} bLastOne true if the item is the last secondary item of its principal item, false otherwise
		 */
		_updateSecondaryItemLastOneClass: function (item, bLastOne) {

		},

		/**
	 *    Update an li given its index and the item's index
	 *    @ignore
	 *    @param {Number} liIndex the index of the li to update
	 *    @param {Number} itemIndex the index of the corresponding item
	 */
		_updateLi : function (liIndex, itemIndex) {
			// If the li element exists.
			if (this._LisInDom[liIndex]) {
				// Remove the old one.
				this._removeOneLi(liIndex);
				// Build the new one
				var newElement = this._buildList(this._getNodesToDisplay(itemIndex, 1));
				// Insert it.
				this._insertNodes(newElement[0], newElement[1], this._LisInDom[liIndex]);
			}
		},

		/**
	 * Reset the scroll for this UIElement
	 * @ignore
	 */
		_resetDomScroll : function () {
			if(this.boolScroll){
				FXScroll.reset(this.domNode);
			}
		},

		/**
	 * A generic function for executing the UI element callbacks
	 * @ignore
	 * @param {String} name the name of the callback to execute
	 * @param {Object} params the callback's parameters
		 * @param {Boolean} boolByClick true if the callback is executed following a user click (not a selection called by the webapp)
	 * @return {Boolean} the result of the callback execution or false
	 */
		_executeCallback : function(name, params, boolByClick) {
			// If the callback is a function
			var cb = this[name];
			if (typeof cb === "function") {
				// Execute it
				return cb(params, boolByClick || false);
			// If it's an object
			} else if (cb && typeof cb.func === "function") {
				// Execute it in the given context, or in the current one.
				cb.func.call(cb.context || window, params || null, boolByClick || false);
			} else {
				return false;
			}
		},

		/**
	 *   Get nb of domNodes currently displayed
	 *   @ignore
	 *   @return {Number} the number of displayed domNodes
	 */
		_getNbOfDomNodes : function () {
			return this._LisInDom.length;
		},

		/**
	 * Get the difference between the number of existing domNodes and the nbElementsMax
	 * @ignore
	 * @return {Number} the difference
	 */
		_getNbOfExceedingNodes : function () {
			var maxRange = this.arrItems.length > this.nbElementsMax ? this.nbElementsMax : this.arrItems.length;
			return (this._getNbOfDomNodes() - maxRange);
		},

		/**
	 * Empty the list of domNodes
	 * @ignore
	 * @return {Number} the result of the pop function
	 */
		_emptyList : function () {
			return this.pop(this._getNbOfDomNodes());
		},

		/**
	 * Removes one dom node
	 * @ignore
	 * @param {Number} liNb the index of the dom node
	 * @return the result of the remove operation or this
	 */
		_removeOneLi : function (liNb) {
			var li = this._removeLiNb(liNb);
			return li ? li.remove() : this;
		},

		/**
	 * Collapse one li
	 * @ignore
	 * @param {Number} liNb the index of the li
	 */
		_collapseOneLi : function (liNb) {
			this._setItemCollapse(this._LisInDom[liNb]._itemSource, true);
		},

		/**
	 * Uncollapse one li
	 * @ignore
	 * @param {Number} liNb the index of the li
	 */
		_uncollapseOneLi : function (liNb) {
			this._setItemCollapse(this._LisInDom[liNb]._itemSource, false);
		},

		/**
	 * Removes the index from the list of references.
	 * @ignore
	 * @param {Number} liNb the index of the li
	 */
		_removeLiNb : function (liNb) {
			return this._LisInDom.splice(liNb, 1)[0] || false;
		},


		/**
	 * Add nodes at a specific point
	 * @ignore
	 * @param {HTMLElement} domNodes a dom fragment with the li domNodes
	 * @param {HTMLElement} nodesRef the references of the domNodes to insert
	 * @param {HTMLElement} position a domNode from which to insert
	 */
		_insertNodes : function (domNodes, nodesRef, position) {
			// Insert nodes before an other one, or append at the end.
			this._list.insertBefore(domNodes, position);
			// Also insert the references in the _LisInDom array
			this._LisInDom = this._insertInArray(this._LisInDom, this.getLiPosition(position), nodesRef);

		},

		/**
	 * A generic function for adding items to an array from an insertion point
	 * @ignore
	 * @param {Array} list the list to which to add items
	 * @param {Number} insertPoint the index from which to add the items
	 * @param {Array} items the items to add
	 */
		_insertInArray : function (list, insertPoint, items) {
			items = dojo.isArray(items) ? items : [items];
			var head = insertPoint >= 0 ? list.splice(insertPoint) : [];
			return list.concat(items).concat(head);
		},

		/**
	 * Returns the last displayed element in the list.
	 * @ignore
	 * @return {Number} the position or -1 if none
	 */
		_getLastDisplayedItem : function () {
			var length = this._LisInDom.length;
			if (length > 0) {
				return this._getItemPosition(this._LisInDom[this._LisInDom.length-1]._itemSource);
			} else {
				return -1;
			}
		},

		/**
	 * Returns the first displayed element in the list
	 * @ignore
	 * @return {Number} the position
	 */
		_getFirstDisplayedItem : function () {
			if (typeof this._LisInDom[0] === "object") {
				return this._getItemPosition(this._LisInDom[0]._itemSource);
			}
		},

		/**
	 *  Return the type of a domNode
	 *  @ignore
	 *  @param {HTMLElement} li the domNode
	 *  @return {String} title || normal
	 */
		_getLiType : function (li) {
			if (typeof li === "object" && li.getAttribute) {
				if (li.className.split(" ").indexOf(this.types["title"].className) >= 0) {
					return "title";
				} else {
					return "normal";
				}
			} else {
				return false;
			}
		},

		/**
	 * Checks if the given list contains categories
	 * @ignore
	 * @return {Boolean} true if it's a list with categories
	 */
		_isListWithCategories : function () {
			var func = function (item) {
				if (item.strType === "title") {
					return true;
				}
			};
			return this.arrItems.some(func);
		},

		/**
	 * Clearly identify title type from normal type
	 * @ignore
	 * @param {Array} array the list of items from which to clean types
	 */
		_cleanTypes : function (array) {
			// Do the cleaning on arrItems.
			if (array && array.forEach) {
				array.forEach(this._cleanItemType, this);
			}
		},

		/**
	 * Set the strType property to "normal" if it's not defined
	 * @ignore
	 * @param {Object} item the item to clean
	 * @return {Boolean} if the item was cleaned
	 */
		_cleanItemType : function (item) {
			// If no type is defined
			if (!item.strType) {
				// set it to normal.
				item.strType = "normal";
				return true;
			} else {
				return false;
			}
		},


		/**
		 * @ignore
		 * Return the list of items without the potential secondary items
		 * @return {Array} the array of list's items without the potential secondary items
		 */
		_getArrItemsWithoutSecondaryItems : function () {
			var func = function (item) {
				// return this item if it is not a secondary item
				return (item && !this.isSecondaryItem(item));
			}
			return this.arrItems.filter(func,this);
		},


		/**
		 * @ignore
		 * Convert an insert point given which don't take into account the secondary items that some items can have to the real position in the whole arrItems array
		 * (which contains the secondary items).
		 * ex: [item_a, item_b, sub_item_b1, sub_item_b2, item_c,...] -> we want insert before item_c, so to position 2, but real position is arrItems is in fact 4
		 * (because of the secondary items of item_b)
		 * @param {Number} insertPoint insert point wanted without taking into account the secondary items that some items can have
		 * @return {Number} the insert point in arrItems
		 */
		_computeInsertPoint : function (insertPoint) {
			var arrayWithoutSecondariesItems = this._getArrItemsWithoutSecondaryItems();
			return this._getItemPosition(arrayWithoutSecondariesItems[insertPoint]);
		},


		/**
		 * @ignore
		 * Get the position (relative to arrSecondaryItems of a principal item, not relative to arrItems) of a secondary item.
		 * @param {Object} secondaryItem the secondary item to get the position
		 * @return {Number} the position (relative to arrSecondaryItems) of the secondary item
		 */
		_getSecondaryItemPosition : function (secondaryItem) {
			return this.getSecondaryItems(this.getPrincipalItem(secondaryItem).intIndex).indexOf(secondaryItem);
		},


		/**
		 * @ignore
		 * Get the first secondary item associated to a given principal item (from the principal item's intIndex).
		 * @param {String/Number} principalEntryIndex the intIndex of the principal item
		 * @return {Object} the first secondary item associated to this principal item
		 */
		_getFirstSecondaryItem : function (principalEntryIndex) {
			var secondaryItems = this.getSecondaryItems(principalEntryIndex);
			return (secondaryItems && secondaryItems[0]);
		},

		/**
		 * @ignore
		 * Get the intIndex of the first secondary item associated to a given principal item (from the principal item's intIndex).
		 * @param {String/Number} principalEntryIndex the intIndex of the principal item
		 * @return {Number} the intIndex of the first secondary item associated to this principal item
		 */
		_getFirstSecondaryItemIndex : function (principalEntryIndex) {
			var firstSecondaryItem = this._getFirstSecondaryItem(principalEntryIndex);
			return (firstSecondaryItem && firstSecondaryItem.intIndex);
		},

		/**
		 * @ignore
		 * Get the position (in arrItems) of the first secondary item associated to a given principal item (from the principal item's intIndex).
		 * ex: [item_a, item_b, sub_item_b1, sub_item_b2, item_c,...] -> position of sub_item_b1 is 2
		 * @param {String/Number} principalEntryIndex the intIndex of the principal item
		 * @return {Number} the position (in arrItems) of the first secondary item associated to this principal item
		 */
		_getFirstSecondaryItemPosition : function (principalEntryIndex) {
			return this._getItemPosition(this._getFirstSecondaryItem(principalEntryIndex));
		},


		/**
		 * @ignore
		 * Get the last secondary item associated to a given principal item (from the principal item's intIndex).
		 * @param {String/Number} principalEntryIndex the intIndex of the principal item
		 * @return {Object} the last secondary item associated to this principal item
		 */
		_getLastSecondaryItem : function (principalEntryIndex) {
			var secondaryItems = this.getSecondaryItems(principalEntryIndex);
			return (secondaryItems && secondaryItems[secondaryItems.length-1]);
		},

		/**
		 * @ignore
		 * Get the intIndex of the last secondary item associated to a given principal item (from the principal item's intIndex).
		 * @param {String/Number} principalEntryIndex the intIndex of the principal item
		 * @return {Number} the intIndex of the last secondary item associated to this principal item
		 */
		_getLastSecondaryItemIndex : function (principalEntryIndex) {
			var lastSecondaryItem = this._getLastSecondaryItem(principalEntryIndex);
			return (lastSecondaryItem && lastSecondaryItem.intIndex);
		},

		/**
		 * @ignore
		 * Get the position (in arrItems) of the last secondary item associated to a given principal item (from the principal item's intIndex).
		 * ex: [item_a, item_b, sub_item_b1, sub_item_b2, item_c,...] -> position of sub_item_b2 is 3
		 * @param {String/Number} principalEntryIndex the intIndex of the principal item
		 * @return {Number} the position (in arrItems) of the last secondary item associated to this principal item
		 */
		_getLastSecondaryItemPosition : function (principalEntryIndex) {
			return this._getItemPosition(this._getLastSecondaryItem(principalEntryIndex));
		},

		/**
		 * @ignore
		 * Check if an item is the last secondary item of its principal item.
		 * @param {Object} secondaryItem the secondary item to check
		 * @return {Boolean} true if the item is the secondary item, false otherwise
		 */
		_isLastSecondaryItem: function (secondaryItem) {
			var principalItem = this.getPrincipalItem(secondaryItem);
			return (principalItem && this._getLastSecondaryItem(principalItem.intIndex) === secondaryItem);
		},


		/**
		 * @ignore
		 * Format an array of items provided with secondary items (item[i].arrSecondaryItems) to an array of item with secondary items split
		 * after their respective principal item. Specific fields of principal items and secondary items are filled.
		 * The array returned is directly insertable into arrItems.
		 * ex: [item_a.arrSecondaryItems=[sub_item_a1, sub_item_a2], item_b.arrSecondaryItems=[sub_item_b1, sub_item_b2]]
		 * -> [item_a, sub_item_a1, sub_item_a2, item_b, sub_item_b1, sub_item_b2]
		 * @param {Array} array the initial array of items to format
		 * @return {Array} the formatted array with secondary items split and entries's fields filled
		 */
		_splitSecondaryItemsFromArray: function (array) {
			var itemsList = [];
			if (array && dojo.isArray(array)) {
				for (var i = 0 ; i < array.length ; i++) {
					itemsList = itemsList.concat(this._splitSecondaryItems(array[i]));
				}
			}
			return itemsList;
		},

		/**
		 * @ignore
		 * Format an item provided with secondary items (item[i].arrSecondaryItems) to an array of item with secondary items split
		 * after their principal item. Specific fields of principal item and secondary items are filled.
		 * ex: item_a.arrSecondaryItems=[sub_item_a1, sub_item_a2]
		 * -> [item_a, sub_item_a1, sub_item_a2]
		 * Note:
		 *  - if the item provided in input don't have secondary items in its field arrSecondaryItems, the function returns an array with just this item: [item]
		 *  - if the item provided in input is not defined, the function returns an empty array: []
		 * @param {Object} item the initial item format
		 * @return {Array} the array with secondary items split and entries's fields filled
		 */
		_splitSecondaryItems: function (item) {
			if (item) {
				// Add missing normal type to item
				this._cleanItemType(item);
				if (item.arrSecondaryItems && item.arrSecondaryItems.length > 0) {
					// set the principal item 's flag _subType to PRINCIPAL_ENTRY
					this._setItemSubType(item, this.PRINCIPAL_ENTRY);
					// set the principal item's flag _isFolded according to the list's flag insertAsFolded
					this._setItemFolded(item, this.getInsertAsFolded());
					// add the principal item to the first position of the array
					var itemsList = [item];
					// format the secondary items
					this._formatSecondaryItems(item.arrSecondaryItems, item);
					// push the formatted secondary items after the principal item
					itemsList = itemsList.concat(item.arrSecondaryItems);
					return itemsList;
				} else {
					return [item];
				}
			} else {
				return [];
			}
		},

		/**
		 * @ignore
		 * Format an array of secondary items (the array stored in principalItem.arrSecondaryItems) by filling the specific fields of each secondary items.
		 * Specific fields are:
		 *  - _subType (set to this.SUB_ENTRY)
		 *  - _principalEntryIndex (set to the intIndex of the principal item provided in input)
		 * @param {Array} arrSecondaryItems the array of secondary items to format
		 * @param {Object} principalEntry the principal item associated to the secondary items
		 */
		_formatSecondaryItems: function (arrSecondaryItems, principalEntry) {
			for (var i = 0 ; i < arrSecondaryItems.length ; i++) {
				this._formatSecondaryItem(arrSecondaryItems[i], principalEntry);
			}
		},

		/**
		 * @ignore
		 * Format a secondary item by filling its specific fields.
		 * The secondary item is automatically formatted as invisible if the principal item is folded,
		 * visible is the principal item is unfolded.
		 * Specific fields are:
		 *  - _principalEntryIndex (set to the intIndex of the principal item provided in input)
		 *  - _subType (set to this.SUB_ENTRY)
		 * @param {Object} secondaryItem the secondary item to format
		 * @param {Object} principalEntry the principal item associated to this secondary item
		 * @return {Boolean} return true if the item is successfully formatted, false if the item was not defined or not an object
		 */
		_formatSecondaryItem: function (secondaryItem, principalEntry) {
			if (secondaryItem && typeof secondaryItem === "object") {
				// Add missing normal type to item
				this._cleanItemType(secondaryItem);
				// Add specific fields for secondary entry item
				secondaryItem._principalEntryIndex = principalEntry.intIndex;
				this._setItemSubType(secondaryItem, this.SUB_ENTRY);
				// set the invisibility property of the secondary item according to the state fold/unfold of the principal item
				// if the state fold/unfold of the principal item is undefined, it means that we are adding a secondary item to an item
				// which was not principal before, so set invisibility to the value of autoFold
				var boolVisibility = (this._isItemFolded(principalEntry) === undefined) ? this.getAutoFold() : this._isItemFolded(principalEntry);
				this._setItemInvisibility(secondaryItem, boolVisibility);
				return true;
			} else {
				return false;
			}
		},

		/**
		 * @ignore
		 * Update, if needed, a principal item:
		 *  - set the sepcific field (_subType = this.PRINCIPAL_ENTRY) and update graphically the item if it had no secondary items but it has now,
		 *  - delete the specific field (_subType) and update graphically the item if it had secondary items but it has no one anymore.
		 * Usefull to check if the principal entry has to be updated after an addition or deletion of secondary entries.
		 * @param {String/Number} principalEntryIndex the intIndex of the principal item to check for update
		 * @return {Boolean} return true if the item is a principal item (it has secondary item(s)), false if the item is not a principal item (it has no secondary items)
		 */
		_checkUpdatePrincipalItem: function (principalEntryIndex) {
			var principalItem = this.getItemByIntIndex(principalEntryIndex);
			if (this.getNbSecondaryItems(principalEntryIndex) > 0) {
				// If the principal item was not a principal item before, it is now, so set its internal variable and update it graphically
				if (!this.isPrincipalItem(principalItem)) {
					// set the principal item subtype to PRINCIPAL_ENTRY
					this._setItemSubType(principalItem, this.PRINCIPAL_ENTRY);
					// if the _isFolded property is not defined, set it to the value of autoFold
					if (this._isItemFolded(principalItem) === undefined) {
						this._setItemFolded(principalItem, this.getAutoFold())
					}
					// Update graphically the principal item so that it appears as a principal item
					this.updateItem(principalItem, true);
				}
				return true;
			} else {
				// If the principal item was a principal item before, it isn't anymore, so remove its internal variable and update it graphically
				if (this.isPrincipalItem(principalItem)) {
					// Remove subType PRINCIPAL_ENTRY
					delete principalItem._subType;
					// Remove isFolded flag
					delete principalItem._isFolded;
					// Remove the array of secondary items
					delete principalItem.arrSecondaryItems;
					// Update graphically the principal item so that it doesn't appear as a principal item anymore
					this.updateItem(principalItem, true);
				}
				return false;
			}
		},

		/**
		 * @ignore
		 * Refresh principal item to appear folded or unfolded, according to its state
		 *  - if the principal item has a unfoldedContent object defined (different data to display when the item is unfolded), refresh the whole item
		 *  - otherwise, just refresh the arraw (better performances).
		 * @param {Object} principalItem the principal item to update
		 */
		_updatePrincipalItem: function (principalItem) {
			if (principalItem.unfoldedContent !== undefined) {
				this.updateItem(principalItem, true);
			}
			else {
				this._updateArrawPrincipalItem(principalItem);
			}
		},


		/**
	 * Creates the root of the list
	 * @ignore
	 * @return this
	 */
		_initList : function () {
			this._list = this.newElement("ul")
			.mixin( {
				"className" : this.namespace.split(".")[1]
			})
			return this;
		},

		/**
	 * Converts a list of items to a list of LIs
	 * @ignore
	 * @param {Array} list a list of items to convert to LIs
	 * @return {Array} the list of dom elements and their references
	 */
		_buildList : function (list) {
			var nb ,  i, newLi, lisReference = [], domFragment = document.createDocumentFragment();
			nb = list.length;
			// As long as there'll be item to transform to domNodes
			for (i=0; i<nb; i++) {
				// Get the li's template
				newLi = this._getLiTemplate(list[i]);
				// And save its former item
				newLi._itemSource =  list[i];
				// Add its reference to the list
				lisReference.push(newLi);

				// Add specific information
				if (list[i].strType === "title")  {
					dojo.toggleClass(newLi, "collapsed", list[i].bCollapse);
				} else {
					!!list[i].bSelect && this._selectCurrent(newLi);
				}
				// Append the domNode to the list
				domFragment.appendChild(newLi);
			}
			// Returns both.
			return [domFragment, lisReference];
		},


		/**
	 * Returns the list of items in a category
	 * @ignore
	 * @param {Number} start the index of the item from which to search items
	 * @return {Array} the list of items
	 */
		_getItemsInCategory : function (start) {
			return this._getItemsList(start+1, 0, "normal", true);
		},


		/**
	 * From start, find the next category domNode
	 * @ignore
	 * @param {Number} start the index from which to search for the next category
	 * @return {HTMLElement} the LI dom node
	 */
		_getNextCatLi : function (start) {
			// Searching from next Li, looking for only 1 result of type "title"
			// Returns the first element in the list.
			return this._getLisList(start+1, 1, "title")[0];
		},

		/**
	 * From start, find the previous category domNode
	 * @ignore
	 * @param {Number} start the index from which to search for the previous category
	 * @return {HTMLElement} the LI dom node
	 */
		_getPreviousCatLi : function (start) {
			// Searching from this li, looking for only 1 li of type title,
			// don't stop until found, and reverse the search array
			return this._getLisList(start, 1, "title", false, true)[0];
		},


		/**
	 * From itemNb, find the previous category item
	 * @ignore
	 * @param {Number} itemNb the index from which to search for the previous category item
	 * @return {Object} the item object
	 */
		_getPreviousCatItem : function (itemNb) {
			// Searching from this item, looking for only 1 item of type title,
			// don't stop until found, and reverse the search array
			return this._getItemsList(itemNb, 1, "title", false, true)[0];
		},

		/**
	 * Return a list of items, searching with the parameters :
	 * @ignore
	 * @param {Number} start From start
	 * @param {Number} range A maximum of range elements
	 * @param {String} type With specific type or all
	 * @param {Boolean} stop Must stop right after the next one if it doesn't match required type.
	 * @param {Boolean} reverse Reverses the list
	 * @return {Array} the list of items found
	 */
		_getItemsList : function (start, range, type, stop, reverse) {
			// Of type... or all
			type = type || (this.getAutoCollapse() ? "title" : "*");
			var results = [];
			range = range || undefined;

			var arrCopy;
			if (reverse) {
				// If reverse, use the list from 0 to start point and reverse it.
				arrCopy = this.arrItems.slice(0,start).reverse();
			} else {
				// Else, use from start to end.
				arrCopy = this.arrItems.slice(start);
			}
			// If some items correspond to criterium, push them into results[]
			var func = function (item) {
				if (item.strType === type || type === "*") {
					results.push(item);
					return false;
				} else {
					// Stop if needed, else continue
					return stop || false;
				}
			};
			arrCopy.some(func);
			// Keep range
			results = results.slice(0, range);
			// Returns the results
			if (reverse) {
				return results.reverse();
			} else {
				return results;
			}

		},

		/**
	 * Return a list of Lis, searching with the parameters :
	 * @ignore
	 * @param {Number} start From start
	 * @param {Number} range A maximum of range elements
	 * @param {String} type With specific type or all
	 * @param {Boolean} stop Must stop right after the next one if it doesn't match required type.
	 * @param {Boolean} reverse Reverses the list
	 * @return {Array} the list of items found
	 */
		_getLisList : function (start, range, type, stop, reverse) {
			var results = [];
			var arrCopy;
			if (reverse) {
				arrCopy = this._LisInDom.slice(0,start).reverse();
			} else {
				arrCopy = this._LisInDom.slice(start);
			}
			var func = function (item) {
				if (this._getLiType(item) === type) {
					results.push(item);
					return false;
				} else {
					return stop || false;
				}
			};
			arrCopy.some(func, this);
			results = results.slice(0, range);
			if (reverse) {
				return results.reverse();
			} else {
				return results;
			}
		},

		/**
	 * Get an item's corresponding item in dom
	 * @ignore
	 * @param {Object} item the item to be searched for in the dom
	 * @return {HTMLElement} the item in the DOM
	 *
	 */
		_getItemInDom : function (item) {
			var itemInDom;
			var func = function (li) {
				if (li._itemSource == item)  {
					itemInDom = li;
					return li;
				}
			}
			this._LisInDom.some(func, this);
			return itemInDom;
		},

		/**
	 * From the items list, return the nodes that can be displayed
	 * @ignore
	 * @param {Number} start get the nodes from this item's index
	 * @param {Number} range the number of items to get
	 * @param {Boolean} reverse reverse the array?
	 * @return {Array} the list of items.
	 */
		_getNodesToDisplay : function (start, range, reverse) {
			range = range || undefined;
			var arrCopy;
			if (reverse) {
				arrCopy = this.arrItems.slice(0,start).reverse();
			} else {
				arrCopy = this.arrItems.slice(start);
			}
			var func = function (item) {
				// These items are all of type "title" and all the others that are not collapsed
				if (this._itemDisplayable(item)) {
					return item;
				}
			};
			arrCopy = arrCopy.filter(func, this).slice(0, range);
			if (reverse) {
				return arrCopy.reverse();
			} else {
				return arrCopy;
			}
		},

		/**
	 * Tests if an item is displayable or not
	 * @ignore
	 * @param {Object} item the item to test
	 * @return {Boolean} if the item is displayable
	 */
		_itemDisplayable : function (item) {
			if (item.strType === "title" ||
				!item.bCollapse) {
				if (!this._getItemInvisibility(item)) {
					return true;
				} else {
					return false;
				}
			} else {
				return false;
			}
		},

		/**
	 * Get the index of an item in the items list
	 * @ignore
	 * @param {Object} item the item to get the index from
	 * @return {Number} the index
	 */
		_getItemPosition : function (item) {
			return this.arrItems.indexOf(item);
		},

		/**
	 * Get the index of an item in the items list
	 * @ignore
	 * @param {String} intIndex the item's index
	 * @return {Number} the index of the item in the list
	 */
		_getItemPositionByIntIndex : function(intIndex) {
			return this._getItemPosition(this.findItem("intIndex", intIndex));
		},

		/**
	 * Get the next dom li
	 * @ignore
	 * @param {HTMLElement} li the dom node to search from
	 *
	 */
		_getNextLiInDom : function (li) {
			// Get the position of the given li
			var liPos = this._LisInDom.indexOf(li);
			// If it's in the array and is not the last one
			if (liPos >= 0 && liPos < this._getNbOfDomNodes()) {
				// Return the next element
				return this._LisInDom[liPos+1];
			} else {
				// Return false if there's not next element
				return false;
			}
		},

		/**
	 * Fills in the list from its top without moving the domNodes
	 * @ignore
	 */
		_fillList : function () {
			// Number of pixels to scroll after the dom nodes are added
			var nbOfPixelsToScroll = 0;
			// As long as there are nodes to be added
			while ((this._getNbOfExceedingNodes() < 0) && (this.unshift(1, false)) ) {
				// Increase the nb pixels to scroll by the height of an element
				// TODO : what if 50 changes?
				nbOfPixelsToScroll +=50;
			}
			// Scroll it
			if (nbOfPixelsToScroll > 0)
				this.domNode.scrollTop += nbOfPixelsToScroll;
		},

		/**
	 * Un/collapse a category
	 * @ignore
	 * @param {Object} item the category item
	 * @param {Boolean} bCollapse true if the category is to collapse
	 * @return {Array} the list of items that've been collapsed
	 */
		_setCategoryCollapse : function (item, bCollapse) {
			if (item) {
				// Its domNode
				this.setCollapse(item, bCollapse);
				!bCollapse && this._setCurrentUnCollapsed(item);
				// What's inside
				var itemsList = this._getItemsInCategory(this._getItemPosition(item));
				this._setItemsCollapse(itemsList, bCollapse);
				// Returns the list of items collapsed
				return itemsList;
			} else {
				return false;
			}
		},

		/**
	 * Returns the number of lis displayed in a category
	 * @ignore
	 * @param {Object} elem the li of the category
	 * @return {Number} the number of lis displayed
	 *
	 */
		_getNbLisDisplayedInCat : function (elem) {
			// Get the position of this li
			var start = this.getLiPosition(elem);
			// find the next category
			var nextCat = this.getLiPosition(this._getNextCatLi(start));
			nextCat = nextCat >= 0 ? nextCat : this._LisInDom.length;
			return nextCat-start-1;
		},
		
				/**
	 * Select an li
	 * @ignore
	 * @param {HTMLElement} liNode the li domNode to select
	 * @param {Boolean} dontExecCallback if the callback must be executed or not
		 * @param {Boolean} dontUnfold if the entry must be unfold or not
		 * @param {Boolean} boolByClick if _selectLi was called following to a user click (not a click called by the webapp)
		 */
		_selectLi : function (liNode, dontExecCallback, dontUnfold, boolByClick) {
			if (liNode && this._selectCurrent(liNode,liNode._itemSource.boolNotSelectable)) {
			    if (!dontExecCallback && this.callback && liNode._itemSource) {
					this._executeCallback("callback", liNode._itemSource.intIndex, boolByClick);
				}
				// if auto unfold on click is enabled and the selected entry is a principal item, unfold it
				if (!dontUnfold && this.getAutoUnfoldOnClick() && this.isPrincipalItem(liNode._itemSource)) {
					this.unfoldPrincipalItem(liNode._itemSource);
				}
				// if the selected entry is a secondary item and the principal item is folded, unfold it
				if (!dontUnfold && this.isSecondaryItem(liNode._itemSource) && this._isItemFolded(this.getPrincipalItem(liNode._itemSource))) {
					this.unfoldPrincipalItem(this.getPrincipalItem(liNode._itemSource));
				}
				// if auto fold is enabled and the selected entry is not a principal or a secondary item, fold all the principal items
				if (!dontUnfold && this.getAutoFold() && !this.isPrincipalItem(liNode._itemSource) && !this.isSecondaryItem(liNode._itemSource)) {
					this.foldAllPrincipalItems();
			    }
			    return liNode;		
			} else {
				return false;
			}
		},

		/**
	 * Set current selected dom node info
	 * @ignore
	 * @param {HTMLElement} elem the element that is selected
	 */
		_setCurrentSelected : function (elem) {
			this._currentSelected = elem;
		},

		/**
	 * Select a dom node and unselect the old one.
		 * Note that if the dom node to select corresponds to a secondary item, the associated principal item's dom node will also appear selected.
	 * @ignore
	 * @param {HTMLElement} elem the dom node to select
	 * @param {boolean} boolNotSelectable specify if the element appears or not graphically selected
	 * @return {HTMLElement} the selected element or false if it can't be selected
	 */
		_selectCurrent : function (elem, boolNotSelectable) {
			if (this._getLiType(elem) === "normal") {
				this._deSelectCurrent(elem);
				if (!this.boolNotSelectable && !boolNotSelectable) {
				    this._toggleSelectEntryByLi(elem, true);
					// if the item to select is a secondary item, select its principal item
					if (elem._itemSource && this.isSecondaryItem(elem._itemSource)) {
						var liPrincipalItem = this._getItemInDom(this.getPrincipalItem(elem._itemSource));
						// the principal item can not beeing in the dom (pagination), so check if it exists before trying to unselect it
						// if the principal item is not displayed currently, set it flag bSelect so that it will be displayed as unselected if we scroll to its position
						if (liPrincipalItem) {
							this._toggleAppearSelectedEntryByLi(liPrincipalItem, true);
						}
						else {
							this._toggleAppearSelectedItem(this.getPrincipalItem(elem._itemSource), true);
						}
					}
				}
				this._setCurrentSelected(elem);
				return elem;
			} else {
				return false;
			}
		},

		/**
	 * Toggle select an entry given its li
	 * @ignore
	 * @param {HTMLElement} li the li domNode to de/select
	 * @param {Boolean} bSelect select or deselect
	 * @return {HTMLElement} the selected li
	 */
		_toggleSelectEntryByLi : function (li, bSelect) {
			this._toggleSelectLi(li, bSelect);
			this._toggleSelectItem(li._itemSource, bSelect);
			return li;
		},


		/**
	 * Toggle select an entry given its item
	 * @ignore
	 * @param {Object} item the item to de/select
	 * @param {Boolean} bSelect select or deselect
	 * @return {Object} the selected item
	 */
		_toggleSelectEntryByItem : function (item, bSelect) {
			this._toggleSelectItem(item, bSelect);
			var func = function (li) {
				if (li._itemSource == item) {
					return this._toggleSelectLi(li, bSelect);
				}
			};
			return this._LisInDom.filter(func, this);
		},

		/**
	 * Add/Remove the selected class of a domNode
	 * @ignore
	 * @param {HTMLElement} li the li node to modify
	 * @param {Boolean} boolSelect true : add, false : remove
	 * @return dojo.toggleClass return
	 */
		_toggleSelectLi : function (li, boolSelect) {
			return dojo.toggleClass(li, "selected", boolSelect);
		},

		/**
	 * Set the selected property of an item
	 * @ignore
	 * @param {Object} item the item to modify
	 * @param {Boolean} boolSelect true : selected, false : not selected
	 * @return {Object} the item
	 */
		_toggleSelectItem : function (item, boolSelect) {
			return item.bSelect = boolSelect;
		},


		/**
		 * Toggle appear selected or not appear selected an entry given its li
		 * @ignore
		 * @param {HTMLElement} li the li domNode to toggle appear selected or not appear selected
		 * @param {Boolean} boolAppearSelected appear selected or not appear selected
		 * @return {HTMLElement} the appear selected li
		 */
		_toggleAppearSelectedEntryByLi : function (li, boolAppearSelected) {
			this._toggleAppearSelectedLi(li, boolAppearSelected);
			this._toggleAppearSelectedItem(li._itemSource, boolAppearSelected);
			return li;
		},

		/**
		 * Toggle appear selected or not appear selected an entry given its item
		 * @ignore
		 * @param {Object} item the item to toggle appear selected or not appear selected
		 * @param {Boolean} boolAppearSelected appear selected or not appear selected
		 * @return {Object} the appear selected item
		 */
		_toggleAppearSelectedEntryByItem : function (item, boolAppearSelected) {
			this._toggleAppearSelectedItem(item, boolAppearSelected);
			var func = function (li) {
				if (li._itemSource == item) {
					return this._toggleAppearSelectedLi(li, boolAppearSelected);
				}
			};
			return this._LisInDom.filter(func, this);
		},

		/**
		 * Add/Remove the appearSelected class of a domNode
		 * @ignore
		 * @param {HTMLElement} li the li node to modify
		 * @param {Boolean} boolAppearSelected true : add, false : remove
		 * @return dojo.toggleClass return
		 */
		_toggleAppearSelectedLi : function (li, boolAppearSelected) {
			return dojo.toggleClass(li, "appearSelected", boolAppearSelected);
		},

		/**
		 * Set the appear selected property of an item
		 * @ignore
		 * @param {Object} item the item to modify
		 * @param {Boolean} boolAppearSelected true : appear selected, false : not appear selected
		 * @return {Object} the item
		 */
		_toggleAppearSelectedItem : function (item, boolAppearSelected) {
			return item.bAppearSelected = !!boolAppearSelected;
		},

		/**
		 * Get the appear selected property of an item
		 * @ignore
		 * @param {Object} item the item to test
		 * @return {Boolean} true if the item has the property appear selected enabled, false otherwise
		 */
		_isAppearSelected : function (item) {
			return !!item.bAppearSelected;
		},

		/**
	 * Deselect current item/domNode
	 * @ignore
		 * @param {HTMLElement} elem (optional parameter) the dom node currently beeing selected
		 */
		_deSelectCurrent : function (elem) {
			if (this.getCurrentSelected() != null) {
				this._toggleSelectEntryByLi(this.getCurrentSelected(), false);
				// if the current item to deselect is a secondary item, also deselect its principal item
				// (only if the item currently beeing selected is not a secondary item of the same principal item than the one to deselect)
				if (this.isSecondaryItem(this.getCurrentSelectedItem()) && this.getPrincipalItem(this.getCurrentSelectedItem()) && (!elem || this.getPrincipalItemIndex(elem) !== this.getPrincipalItemIndex(this.getCurrentSelectedItem()))) {
					var liPrincipalItem = this._getItemInDom(this.getPrincipalItem(this.getCurrentSelectedItem()));
					// the principal item can not beeing in the dom (pagination), so check if it exists before trying to unselect it
					// if the principal item is not displayed currently, set it flag bSelect so that it will be displayed as unselected if we scroll to its position
					if (liPrincipalItem) {
						this._toggleAppearSelectedEntryByLi(liPrincipalItem, false);
					}
					else {
						this._toggleAppearSelectedItem(this.getPrincipalItem(this.getCurrentSelectedItem()), false);
					}
				}
			}
		},

		/**
	 * Set a list of items to un/collapse
	 * @ignore
	 * @param {Array} list the list of items to set to un/collapse
	 * @param {Boolean} bCollapse the value of collapse
	 * @return {Array} the list of modified items
	 */
		_setItemsCollapse : function (list, bCollapse) {
			var func = function (item) {
				item.bCollapse = bCollapse;
			}
			list.map(func);

			return list;
		},

		/**
	 * Set an item to un/collapse
	 * @ignore
	 * @param {Object} item the item to set to un/collapse
	 * @param {Boolean} bCollapse the value of collapse
	 * @return {Boolean} the value of bCollapse
	 */
		_setItemCollapse : function (item, bCollapse) {
			if (item && (typeof item == "object")) {
				item.bCollapse = bCollapse;
				return bCollapse;
			}
		},

		/**
	 * Set the current uncollapsed property
	 * @ignore
	 * @param {Object} item the category's item
	 * @return {Object} the category's item
	 */
		_setCurrentUnCollapsed : function (item) {
			return this._currentUnCollapsed = item;
		},

		/**
	 * Get the current uncollapsed category
	 * @ignore
	 * @return {Object} the category's item
	 */
		_getCurrentUnCollapsed : function () {
			return this._currentUnCollapsed;
		},

		/**
	 * Update an item's text (and its Li)
	 * @ignore
	 * @param {Number} itemNb the index of the item
	 * @param {String} which the name of the property to modify
	 * @param {String} newText the new text to assign
	 */
		_updateItemText : function (itemNb,  which, newText) {
			// If its attribute has a changeText function
			if (this.arrItems[itemNb][which] && (typeof this.arrItems[itemNb][which].changeText == "function")) {
				// Then call it to change the text.
				this.arrItems[itemNb][which].changeText(newText);
				return true;
			} else {
				// Else find the domNode to refresh in the list of lis.
				var func = function (item) {
					// Is it this one?
					if (item._itemSource === this.arrItems[itemNb]) {
						// If the new text is a fresh i18nObject
						if (typeof newText === "object" && newText.isI18Ned) {
							// Do i18n stuff
							ICTouchAPI.i18nServices.addI18nHandler(this, newText.fillInnerHTML(item));
						} else {
							// Or just innerHTML the new text.
							item.getElementsByClassName(which)[0].innerHTML = newText;
						}
					}
				};
				return this._LisInDom.forEach(func, this);
			}
		},

		/**
	 * Get an item invisibility property
	 * @ignore
	 * @param {Object} item the item from which to get the property
	 * @return {Boolean} the value
	 */
		_getItemInvisibility : function (item) {
			return item._invisible;
		},

		/**
	 * Set an item invisibility property
	 * @ignore
	 * @param {Object} item the item to set the property to
	 * @param {Boolean} invisibility the value to set
	 * @return {Boolean} the value
	 */
		_setItemInvisibility : function (item, invisibility) {
			return item._invisible = !!invisibility;
		},
		
		/**
		 * Get the item's subType property
		 * @ignore
		 * @param {Object} item the item from which to get the property
		 * @return {String} the subType value
		 */
		_getItemSubType : function (item) {
			return item._subType;
		},

		/**
		 * Set the item's subType property
		 * @ignore
		 * @param {Object} item the item to set the property to
		 * @param {Boolean} subType the value to set
		 * @return {String} the subType value
		 */
		_setItemSubType : function (item, subType) {
			return item._subType = subType;
		},

		/**
		 * Get the item's folded property
		 * @ignore
		 * @param {Object} item the item from which to get the property
		 * @return {Boolean} the value
		 */
		_isItemFolded : function (item) {
			return item._isFolded;
		},

		/**
		 * Set the item's folded property
		 * @ignore
		 * @param {Object} item the item to set the property to
		 * @param {Boolean} folded the value to set
		 * @return {Boolean} the value
		 */
		_setItemFolded : function (item, folded) {
			return item._isFolded = !!folded;
		},

		/**
	 * METHODE 1.18
	 * NE PAS UTILISER EN DEHORS DE CONTACTS
	 * DOIT DISPARAITRE AU PROFIT DE FILTRES/TRIS DES DATASTORE
	 * @ignore
	 */
		_DONT_USE_locateItem : function (what, superiorTo) {
			var func = function (item) {
				var superior = false;
				for(var i in what) {
					if (item.objContact[what[i]].toLowerCase() > superiorTo[i].toLowerCase()) {
						superior = true;
						break;
					}
					else if (item.objContact[what[i]].toLowerCase() < superiorTo[i].toLowerCase()) {
						break;
					}
				}
				if(superior === true) {
					return item;
				}
				else {
					return null;
				}
			}
			return this._getItemPosition(this.arrItems.filter(func)[0]);
		},

		/**
	 * Remove a dom element given its item.
	 * By default, add new elements to the visible list
	 * @ignore
	 * @param {Object} item the item to remove
	 * @param {Boolean} dontAdd to set to true if the list must not autocomplete itself after removing
	 * @return {HTMLElement} the li that is removed
	 */
		_removeItemInDom : function (item, dontAdd) {
			var removed = this._removeOneLi(this._LisInDom.indexOf(this._getItemInDom(item)));
			!dontAdd && (this._getNbOfExceedingNodes() < 0) && this.push(-this._getNbOfExceedingNodes(), true);
			!dontAdd && (this._getNbOfExceedingNodes() < 0) && this.unshift(-this._getNbOfExceedingNodes(), true);
			return removed;
		},

		/**
		 * Check if the list is empty and display the empty state if needed
		 * @ignore
		 */
		_checkForEmpty : function() {
			if (this._emptyState) {
				var isEmpty = this.arrItems.length <= 0;
				this._emptyState.setVisibility(isEmpty);
				if (typeof this.domNode.deactivate === "function" && isEmpty) {
					this.domNode.deactivate();
				}
				else if (typeof this.domNode.activate === "function" && !isEmpty) {
					this.domNode.activate();
				}
			}
		}

	});
dojo.require("UIElements.List.ListControlBase");
dojo.provide("UIElements.List.ListControl");
dojo.declare("UIElements.List.ListControl",
	UIElements.List.ListControlBase,
	{
		
	});
/**
 * @class UIElements.MenuList.MenuListControlBase
 * @extends UIElements.List.ListControlBase
 * Abstract Class - Do not use it directly
 * - {Object} objIconLeft The object defining the icon displayed between the avatar and the primary and secondary content
 * - {Object} arrSecondaryIcon The array containing icons displayed left of the second text content
 */
dojo.provide("UIElements.MenuList.MenuListControlBase");
dojo.declare("UIElements.MenuList.MenuListControlBase",
	UIElements.List.ListControl,
	{
		/**
		 * CSS highlight class
		 * @ignore
		 */
		_highlightClass : "highlighted",

		/**
		 * Height (in pixel) of an element of the list (li) (Default height is 50px)
		 * @cfg {Number}
		 */
		intLiHeight : 50,

		/**
		 * Set boolPreview to true when the MenuList UIElement is instantiated in a container in a preview of Homepage preview.
		 *
		 * This property can be set when instantiating the UIElement.
		 * @cfg {Boolean}
		 */
		boolPreview : false,

		/**
		 * @ignore
		 */
		constructor : function() {
			this.types.normal.callbackLeftIcon = {
				event : "click",
				func : "_clickedIconLeft",
				capture : true
			}
		},

		/**
		 * Update an item its primary content
		 * @param {Number} itemNb number of element
		 * @param {String} text new value
		 * @return {Boolean} result of the operation
		 */
		updatePrimaryContent : function (itemNb, text) {
			return this._updateItemText(itemNb, "strPrimaryContent", text);
		},

		/**
		 * Update an item its secondary content
		 * @param {Number} itemNb number of element
		 * @param {String} text new value
		 * @return {Boolean} result of the operation
		 */
		updateSecondaryContent : function (itemNb, text) {
			return this._updateItemText(itemNb, "strSecondaryContent", text);
		},

		/**
		 * Backward compatibility
		 * @deprecated
		 * @return {Boolean} the value returned by the right function call
		 */
		updateTimeForItem : function () {
			return this._deprecated("updateTimeForItem", "updateSecondaryContent", arguments);
		},

		/**
		 * Returns an li dom node for a given item.
		 * @ignore
		 * @param {Object} item the item from which the template is built
		 * @return {HTMLElement} an li dom node
		 */
		_getLiTemplate: function (item) {

			// Compute the content of the item:
			// - if the item is a principal item, unfolded, and it has an alternate unfoldedContent object defined, the alternate content
			//   provided into this unfoldedContent object will overwrite the content provided basically to the item
			//   (if the item is folded, the content provided at the root of the item will be applied)
			// - otherwise, the content provided at the root of item will be applied
			var objContent = this._computeContent(item);
			
			// li is a new Element
			var li = this.newElement("li")
			// In which I mix in a new className
			.mixin({
				"className" :  this.types[item.strType || "normal"]["className"]
					+ (this.isPrincipalItem(item) ?	" " + "Principal" : "")
					+ (this.isSecondaryItem(item) ?	" " + "Secondary" : "")
					+ (this._isAppearSelected(item) ?	" " + "appearSelected" : "")
					+ ( objContent.additionalStyle ?	" " + objContent.additionalStyle : "" ),
				"style" : "height:" + (this.intLiHeight + 1) + "px"
			})
			// I can also attach an event to it
			.attachEvent(this.types[item.strType || "normal"]["callback"], {
				// the Element can even receive custom params, this is an example :
				"myArgs" : "example"
			})
			.attachEvent({event: "mousedown", func: "_startHighlight", capture: false});

			// compute the width of the item label according the elements which are displayed or not in the item
			// (the base width is not the same if the menulist is in preview, as containers in preview are narrower
			// than containers in thirdControl)
			var menuListLabelWidth = this.boolPreview ? 238 : 268;

			// If the subtype is set to "PRINCIPAL_ENTRY", add the possibility on the left to collapse/uncollapse.
			if(this.isPrincipalItem(item)){
				// A div element
				var divIconPrincipalLeft = this.newElement("div")
				// With this className
				.mixin({
					"className" : "MenuListIconLeft " + "MenuListLeftPrincipal " + (this._isItemFolded(item) ? "folded " : "unfolded "),
					"style" : "height:" + this.intLiHeight + "px"
				})
				.attachEvent({event: "click", func: "_clickedIconLeft", capture:false}, {item: item});
				li.append(divIconPrincipalLeft);

				var leftSeparator = this._createSeparator();
				li.append(leftSeparator);
				menuListLabelWidth -= 45 ; //(size of separator : 1 + size of the clickable div : 44)

			}

			// If the item is a secondary item, add the left 'L' symbol
			if(this.isSecondaryItem(item)){
				// A div element
				var divSecondaryLeft = this.newElement("div")
					// With this className
					.mixin({
						"className" : "MenuListIconLeft " + "MenuListLeftSecondary " + (this._isLastSecondaryItem(item) ? "LastOne" : "")
					});
				li.append(divSecondaryLeft);
				menuListLabelWidth -= 45 ; //(size of the crochet div : 45)
			}
			
			// A div element
			var divEntry = this.newElement("div")
			// With this className
			.mixin({
				"className" : "MenuListItem",
				"style" : "height:" + this.intLiHeight + "px"
			})

			if (typeof objContent.strPicture != "undefined" && objContent.strPicture != "" && objContent.strPicture != null) {
				// Include img in a div set to the size of the MenuList item
				var divPicture = this.newElement("div")
					.mixin({
						"style" : "height:" + this.intLiHeight + "px" + ";" +
								"width:" + this.intLiHeight + "px"
					});
				divPicture.append(this.newElement("img")
					.mixin({
						// Max height/witdh of the picture is the size of the MenuList item
						"className" : "MenuListPicture",
						"src" : objContent.strPicture,
						"style" : "max-height:" + this.intLiHeight + "px" + ";" +
							"max-width:" + this.intLiHeight + "px"
					}));
				divEntry.append(divPicture);
				menuListLabelWidth -= this.intLiHeight;
			}

			// Default margin-left between items
			var marginLeft = 12;
			var divIconsLeft;
			if (objContent.objIconLeft) {
			    // Calculate the general margin-left space between items
			    // By default margin-left is set to 12px; If a width has been defined for the icon left; then the margin-left (spacing)
			    // between items is calculated in a proportion of (8/3)
			    marginLeft = objContent.objIconLeft.width ? Math.floor(objContent.objIconLeft.width/(8/3)) : 12;

				// A div element
				divIconsLeft = this.newElement("div")
					// With this className
					.mixin({
						"className" : "MenuListIconLeft " + (objContent.objIconLeft.icon ? objContent.objIconLeft.icon : ""),
						"style" :   "width:" + (objContent.objIconLeft.width ? objContent.objIconLeft.width : 32) + "px" + ";" +
							    "margin-left:" + marginLeft + "px"
					    });
				divEntry.append(divIconsLeft);
			    menuListLabelWidth -= marginLeft + (objContent.objIconLeft.width ? objContent.objIconLeft.width : 32);
			}

			// Then add a div for the texts
			var MenuListLabel = this.newElement("div")
			.mixin({
				"className" : "MenuListLabel" + (objContent.alternateStyle ? " alternate" :""),
				"style" : "margin-left:" + marginLeft + "px"
			});
			menuListLabelWidth -= marginLeft;

			divEntry.append(MenuListLabel);

			// Set the primary icon if any
			if (item.strType == "title" || (typeof objContent.strPrimaryIcon !== "undefined" && objContent.strPrimaryIcon !== "")) {

				var divPrimaryIcon = this.newElement("span")
				.mixin({
					"className" : (objContent.strPrimaryIcon || "") + " MenuListIconRight",
					"style" : "width: " + this.intLiHeight + "px;" +
						  "height: " + this.intLiHeight + "px;"
				});

				// If a callback is defined for right icon, add a separator and attack the callback to the dom.
				if(this.callbackPrimaryIcon){
					var rightSeparator = this._createSeparator();
					divEntry.append(rightSeparator);
					menuListLabelWidth -= 1 ; //(size of separator)

					divPrimaryIcon.attachEvent({event: "click", func: "_clickedPrimaryIcon", capture:false}, {item: item});

					divPrimaryIcon.className += " MenuListIconRightClickable";
				}
				
				divEntry.append(divPrimaryIcon);
				menuListLabelWidth -= this.intLiHeight;
			}

			if (typeof objContent.strPrimaryContent != "undefined") {
				MenuListLabel.append(
					this.newElement("span")
					// Every text content is automatically l10n if i18ned
					.mixin({
						"className" : "strPrimaryContent",
						"innerHTML" :  objContent.strPrimaryContent,
						"style"		: "width : "+menuListLabelWidth+"px;"
					})
				);
			}

			if (typeof objContent.strPrimaryContent != "undefined") {
				MenuListLabel.append(
					this.newElement("br")
				);
			}

			// Add icons in the secondary line content
			if (typeof objContent.arrSecondaryIcon != "undefined") {
			    for(var counterIcon = 0; counterIcon < objContent.arrSecondaryIcon.length; counterIcon++) {
				MenuListLabel.append (
				    this.newElement("span").mixin({
					"className" : objContent.arrSecondaryIcon[counterIcon].icon + " MenuListSecondaryIcon",
					"style" : "width:" + objContent.arrSecondaryIcon[counterIcon].width + "px;"
				    })
				);
				if (typeof objContent.arrSecondaryIcon[counterIcon].width != "undefined") {
				    menuListLabelWidth -= objContent.arrSecondaryIcon[counterIcon].width;
				}
			    }
			}

			// Add label in the secondary line content
			if (objContent.strSecondaryContent) {
			    if ((objContent.strSecondaryContent != null) && (objContent.strSecondaryContent != "")) {
				MenuListLabel.append(
					this.newElement("span")
					.mixin({
						"className" : "strSecondaryContent",
						"innerHTML" : objContent.strSecondaryContent,
						"style" : "width: "+menuListLabelWidth+"px;" +
							  "vertical-align: top;" +
							  (objContent.strSecondaryContentStyle ? "font-style: " + objContent.strSecondaryContentStyle + ";" : "") +
							  (objContent.strSecondaryContentWeight ? "font-weight: " + objContent.strSecondaryContentWeight + ";" : "")
					})
				);
			    }
			}
			
			li.append(divEntry);

			// Returns the li.
			return li;
		},

		/**
		 * Create an HTMLElement separator of 1 pixel
		 * @return {HTMLElement} separator
		 * @ignore
		 */
		_createSeparator : function() {
			var objSeparator = this.newElement("div").mixin({
				className: "MenuListSeparatorExtDiv",
				style: "height: " + this.intLiHeight + "px;"
			});
			objSeparator.append(
				this.newElement("div").mixin({
					className: "MenuListSeparator",
					// 3/5 of the hight for the separator, and 1/5 free on the top and on the bottom
					style: "margin-top: " + Math.ceil((this.intLiHeight)*0.2) + "px; height: "+ Math.floor((this.intLiHeight)*0.6) + "px;"
				})
			);

			return objSeparator;
		},


		/**
		 * Compare oldItem and newItem and determine if the item needs to be updated
		 * (i.e. fields of the newItem have been modified and have an impact graphically)
		 * @param {Object} oldItem the current state of the item
		 * @param {Object} newItem the new item
		 * @return {Boolean} true if an update is needed, false otherwise
		 * @ignore
		 */
		_isUpdateNeeded: function (oldItem, newItem) {
			if (oldItem.strPrimaryIcon === newItem.strPrimaryIcon &&
				oldItem.strPrimaryContent === newItem.strPrimaryContent &&
				ICTouchAPI.tools.compareObjects(oldItem.arrSecondaryIcon, newItem.arrSecondaryIcon) &&
				oldItem.strSecondaryContent === newItem.strSecondaryContent &&
				oldItem.strSecondaryContentStyle === newItem.strSecondaryContentStyle &&
				oldItem.strSecondaryContentWeight === newItem.strSecondaryContentWeight &&
				oldItem.strPicture === newItem.strPicture &&
				ICTouchAPI.tools.compareObjects(oldItem.objIconLeft, newItem.objIconLeft) &&
				oldItem.alternateStyle === newItem.alternateStyle &&
				oldItem.additionalStyle === newItem.additionalStyle &&
				ICTouchAPI.tools.compareObjects(oldItem.unfoldedContent, newItem.unfoldedContent)) {
				
				return false;
			}
			return true;
		},


		/**
		 * Update the class "LastOne" of a given secondary item.
		 * @ignore
		 * @param {Object} item the secondary item
		 * @param {Boolean} bLastOne if true, add the "LastOne" class, remove it otherwise
		 */
		_updateSecondaryItemLastOneClass : function (item, bLastOne) {
			var domItem = this._getItemInDom(item);
			var elem = domItem && domItem.getElementsByClassName("MenuListLeftSecondary")[0];
			if (elem) { // If elem is not defined, that's because item is not currently displayed in dom (because of pagination or folded)
				if (bLastOne) {
					dojo.addClass(elem,"LastOne");
				}
				else {
					dojo.removeClass(elem,"LastOne");
				}
			}
		},

		/**
		 * Update the arraw displayed in front of principal items, according to the state folded/unfolded of the item.
		 * @ignore
		 * @abstract
		 * @param {Object} item the principal item
		 */
		_updateArrawPrincipalItem : function (item) {
			var domItem = this._getItemInDom(item);
			var elem = domItem && domItem.getElementsByClassName("MenuListLeftPrincipal")[0];
			if (elem) { // If elem is not defined, that's because item is not currently displayed in dom (because of pagination)
				if (this._isItemFolded(item)) {
					dojo.removeClass(elem,"unfolded");
					dojo.addClass(elem,"folded");
				}
				else {
					dojo.removeClass(elem,"folded");
					dojo.addClass(elem,"unfolded");
				}
			}
		},

		/**
		 * Compute the content of the item:
		 * - if the item is a principal item, unfolded, and it has an alternate unfoldedContent object defined, the alternate content
		 *   provided into this unfoldedContent object will overwrite the content provided basically to the item
		 *   (if the item is folded, the content provided at the root of the item will be applied)
		 *  - otherwise, the content provided at the root of item will be applied
		 * @return {Object} an object containing all the computed styles
		 * @ignore
		 */
		_computeContent : function(item) {
			// Compute the content of the item:
			// - if the item is a principal item, unfolded, and it has an alternate unfoldedContent object defined, the alternate content
			//   provided into this unfoldedContent object will overwrite the content provided basically to the item
			//   (if the item is folded, the content provided at the root of the item will be applied)
			// - otherwise, the content provided at the root of item will be applied
			var objContent;
			if(this.isPrincipalItem(item) && item.unfoldedContent && !this._isItemFolded(item)){
				objContent = {
					strPrimaryIcon : item.unfoldedContent.strPrimaryIcon !== undefined ? item.unfoldedContent.strPrimaryIcon : item.strPrimaryIcon,
					strPrimaryContent : item.unfoldedContent.strPrimaryContent !== undefined ? item.unfoldedContent.strPrimaryContent : item.strPrimaryContent,
					arrSecondaryIcon : item.unfoldedContent.arrSecondaryIcon !== undefined ? item.unfoldedContent.arrSecondaryIcon : item.arrSecondaryIcon,
					strSecondaryContent : item.unfoldedContent.strSecondaryContent !== undefined ? item.unfoldedContent.strSecondaryContent : item.strSecondaryContent,
					strSecondaryContentStyle : item.unfoldedContent.strSecondaryContentStyle !== undefined ? item.unfoldedContent.strSecondaryContentStyle : item.strSecondaryContentStyle,
					strSecondaryContentWeight : item.unfoldedContent.strSecondaryContentWeight !== undefined ? item.unfoldedContent.strSecondaryContentWeight : item.strSecondaryContentWeight,
					strPicture : item.unfoldedContent.strPicture !== undefined ? item.unfoldedContent.strPicture : item.strPicture,
					objIconLeft : item.unfoldedContent.objIconLeft !== undefined ? item.unfoldedContent.objIconLeft : item.objIconLeft,
					alternateStyle : item.unfoldedContent.alternateStyle !== undefined ? item.unfoldedContent.alternateStyle : item.alternateStyle,
					additionalStyle : item.unfoldedContent.additionalStyle !== undefined ? item.unfoldedContent.additionalStyle : item.additionalStyle
				}
			}
			else {
				objContent = {
					strPrimaryIcon : item.strPrimaryIcon,
					strPrimaryContent : item.strPrimaryContent,
					arrSecondaryIcon : item.arrSecondaryIcon,
					strSecondaryContent : item.strSecondaryContent,
					strSecondaryContentStyle : item.strSecondaryContentStyle,
					strSecondaryContentWeight : item.strSecondaryContentWeight,
					strPicture : item.strPicture,
					objIconLeft : item.objIconLeft,
					alternateStyle : item.alternateStyle,
					additionalStyle : item.additionalStyle
				}
			}
			return objContent;
		},

		/**
		 * @param {HTMLElement} element UI element to be paginated
		 * @ignore
		 */
		_clickedIconLeft : function(event, args) {
			if (args.PARENT_WIDGET.foldUnfoldPrincipalItem) {
				event.cancelBubble = true;
				args.PARENT_WIDGET.foldUnfoldPrincipalItem.call(args.PARENT_WIDGET, args.item);
			}
		},

		/**
		 * @param {HTMLElement} element UI element to be paginated
		 * @ignore
		 */
		_clickedPrimaryIcon : function(event, args) {

			if (args.PARENT_WIDGET.callbackPrimaryIcon) {
				event.cancelBubble = true;
				args.PARENT_WIDGET._executeCallback("callbackPrimaryIcon", args.item);
			}
		}

	});
/**
* @class UIElements.MenuList.MenuListControl
* @extends UIElements.MenuList.MenuListControlBase
* The MenuList displays items that contain at least the following fields:
* <pre><code>
* - {String} strPrimaryContent The primary text content displayed
* - {String} strSecondaryContent The second text content displayed
* - {String} strPicture The path to the picture displayed
* - {String} strPrimaryIcon The CSS class for the Primary icon
* - {String} strsecondaryIcon The CSS class for the Secondary icon
* - {Number} intIndex The index for retrieving the item
* </code></pre>
* Depending to your need, the item can contains any additionnal field
*/
dojo.require("UIElements.MenuList.MenuListControlBase");
dojo.provide("UIElements.MenuList.MenuListControl");
dojo.declare("UIElements.MenuList.MenuListControl",
	UIElements.MenuList.MenuListControlBase,
	{
		// The path to the template is defined here
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * path to the template
		 * @property
		 * @type {String}
		 */
		templatePath	: dojo.moduleUrl("UIElements.MenuList","templates/MenuList8082.html"),

		/**
		 * Array of css files
		 * @property
		 * @type {Array}
		 */
		cssFile         : ["base.css", "8082.css"],

		/* --------------------------------- Private attributes ----------------------------------- */

		/* ------------------------------------ Constructor --------------------------------------- */

		constructor : function(){

		}

	/* ----------------------------------- Getter / Setter------------------------------------- */

	/* ----------------------------------- Public methods ------------------------------------- */

	/* --------------------------------- Private Methods -------------------------------------- */

	}
	);
/**
* @class UIElements.CustomList.CustomListControlBase
* @extends UIElements._base
* Abstract Class - Do not use it
*/
dojo.provide("UIElements.CustomList.CustomListControlBase");
dojo.declare("UIElements.CustomList.CustomListControlBase",
	[UIElements._base, dojox.dtl._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Store the CustomList items
		 * @cfg {Object[]} arrItems
		 */
		arrItems                    : [ ],
                
		/**
		 * Enable or not the multiple selection of items
		 * @cfg {Boolean} boolMultiselect
		 */
		boolMultiselect				: false,

		/**
		 * Maximal number of items that can be selected
		 * @cfg {Boolean} intSelectLimt
		 */
		intSelectLimit				: -1,

		/**
		 * callback function
		 * @cfg {Function} callback
		 */
		callback					: null,



		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		domArrLi                    : [ ],
		/**
		 * @ignore
		 */
		_callbackFunc				: null,
		/**
		 * @ignore
		 */
		_itemList					: [],
		/**
		 * true if we must wait for highlighting, false otherwise
		 * @ignore
		 * @type {Boolean}
		 */
		_enableWaitingHighlight : true,

		/* ------------------------------------ Constructor --------------------------------------- */

		constructor		: function(){
			this.arrItems  = [ ];
			this.domArrLi  = [ ];
			this._itemList = [ ];
		},

		postCreate : function () {
			for(var i=0; i<this.arrItems.length; ++i)
			{
				var item = new UIElements.CustomList.CustomListItemControl(this.arrItems[i], this.domArrLi[i]);
				this._itemList.push(item);
				this.connect(item.domNode, "click", this._clicked);
			}
			FXScroll.init({
				"handler" : this.domNode,
				"easing" : true,
				"scrollType" : "vertical"
			});
		},

		destroy: function() {
			this.inherited(arguments);
			if (this.domNode)
			{
				FXScroll.unInit(this.domNode);
			}
			if (this._itemList)
			{
				dojo.forEach(this._itemList, function(item) {
					item.destroy();
				});
				delete this._itemList;
			}
			delete this.callback;
			delete this._callbackFunc;
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		* Get the list if selected item index
                * @return {Number[]} Array of selected items index
		*/
		getSelectedItemsIndex : function (){
			var arrIndex = new Array();
			var selectedItems = dojo.query(".activated", this.domNode);
			for(var i=0;i < selectedItems.length; i++){
				arrIndex.push(selectedItems[i].intIndex);
			}
			return arrIndex;
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		* Is one or more items selected ?
                * @return {Boolean} return true if at least one item is selected
		*/
		hasSelectedItems : function (){
			return (dojo.query(".activated", this.domNode).length)?true:false;
		},


		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 */
		_clicked : function(event) {
			if (!this.boolMultiselect)
			{
				if (dojo.hasClass(event.currentTarget, "activated"))
				{
					dojo.removeClass(event.currentTarget, "activated");
				}
				else
				{
					this._unselectAll();
					dojo.toggleClass(event.currentTarget, "activated");
				}
				if(this._callbackFunc) {
					this._callbackFunc(event.currentTarget.intIndex, dojo.hasClass(event.currentTarget, "activated"));
				}
			}
			else
			{
				var selectedItems = dojo.query(".activated", this.domNode);
				if (this.intSelectLimit < 0 || selectedItems.length < this.intSelectLimit || dojo.hasClass(event.currentTarget, "activated")) {
					dojo.toggleClass(event.currentTarget, "activated");
					if(this._callbackFunc) {
						this._callbackFunc(event.currentTarget.intIndex, dojo.hasClass(event.currentTarget, "activated"));
					}
				}
			}
		},

		/**
		 * @ignore
		 */
		_unselectAll:function(){
			var selectedItems = dojo.query(".activated", this.domNode);
			for(var i=0;i < selectedItems.length; i++) {
				dojo.removeClass(selectedItems[i], "activated");
			}
		},

		/**
		 * @ignore
		 */
		_setCallbackAttr : function(callback) {
			this._callbackFunc = ICTouchAPI.tools.callbackToHitch(callback);
		},

		/**
		 * @ignore
		 */
		updateContent : function(arrItems) {
			var modified = false;
			for(var i = 0; i < arrItems.length; i++){
				if(this._itemList[i].updateContent(arrItems[i])){
					modified = true;
					break;
				}
			}
			return modified;
		}

	}
	);
/**
* @class UIElements.CustomList.CustomListControl
* @extends UIElements.CustomList.CustomListControlBase
*
* This class is the graphical representation of a generic list.
*
* The CustomList allows you to display any view that you have previously defined as an item of the list.
*
* This list can mix different types of view: Each item can have its proper view.
*
* Here is an example of creating a CustomList that render a simple view
*
* The first task is to define a view that will be used to render the item. Here is a minimal view:
*     dojo.require("webapp.myWebapp.getItemViewBase");
*     dojo.provide("webapp.myWebapp.getItemView");
*     dojo.declare("webapp.myWebapp.getItemView", webapp.myWebapp.getItemViewBase, {
*           templatePath: dojo.moduleUrl("webapp.myWebapp", "templates/getItemView8082.html"),
*
*           postCreate : function() {
*           }
*     }
*
* The next step is to create the HTML template associated to this view (getItemView8082.html).
* Here is a simple representation of the view
*     <div id="webapp.myWebapp.getItemView" widgetname="webapp.myWebapp.getItemView">
*           <ul>
*               <li>First line</li>
*               <li>Second line</li>
*           <ul>
*     </div>
*
* Now that we have the view, we can create the CustomList in the 2/3 part of the main view for example
*
*     // Create the view that will be used to display my item
*     var itemUI = new webapp.myWebapp.getItemView({});
*
*     // Define the array used by the CustomList to handle the list data
*     var items = [];
*
*     // Add an item to the list
*     items.push({
*           intIndex: 0,
*           boolSelected: false,
*           objWidget: itemUI
*     };
*
*     postCreate : function() {
*         var func = dojo.hitch(this, this.onItemPressed);
*
*         var customList = new UIElements.Container.ContainerControl({
*           title: {
*               strLabel: "My Custom List"
*           }
*           content: {
*               name: "UIElements.CustomList.CustomListItem",
*               params: {
*                   arrItems: items,
*                   boolMultiSelect: false,
*                   itemCallback: func
*               }
*           }
*         }, domCustomList);
*     },
*
*     onItemPressed : function(idx) {
*         // Do something
*         // ...
*     }
*
* Don't forget to add the DOM attach point to the template associated to the 2/3 part of the view
*     <div dojoAttachPoint="domCustomList" />
*
*/
dojo.require("UIElements.CustomList.CustomListControlBase");
dojo.provide("UIElements.CustomList.CustomListControl");
dojo.declare("UIElements.CustomList.CustomListControl",
	UIElements.CustomList.CustomListControlBase,
	{

		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		templatePath			: dojo.moduleUrl("UIElements.CustomList","templates/CustomList8082.html"),
		/**
		 * @ignore
		 */
		cssFile                 : ["base.css", "8082.css"]

	/* ------------------------------------ Constructor --------------------------------------- */

	/* ----------------------------------- Getter / Setter------------------------------------- */

	/* ----------------------------------- Public methods ------------------------------------- */

	/* --------------------------------- Private Methods -------------------------------------- */
	}
	);
/**
* @class UIElements.CustomList.CustomListItemControlBase
* @ignore
* @extends UIElements._base
* Abstract Class
* Item of the custom list
*/
dojo.provide("UIElements.CustomList.CustomListItemControlBase");
dojo.declare("UIElements.CustomList.CustomListItemControlBase",
	[UIElements._base, dojox.dtl._Templated],
	{

		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Widget
		 * @property
		 * @type {Widget}
		 */
		objWidget					: null,
		/**
		 * index
		 * @property
		 * @type Number
		 */
		intIndex					: -1,
		/**
		 * selected state
		 * @property
		 * @type Boolean
		 */
		boolSelected				: false,


		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @private
		 */
		domContent					: null,
		/**
		 * true if we must wait for highlighting, false otherwise
		 * @private
		 * @type {Boolean}
		 */
		_enableWaitingHighlight : true,

		/* ------------------------------------ Constructor --------------------------------------- */

		postMixInProperties : function() {
		},

		postCreate : function () {
                        if(!generalConfig.simulation) {
                            if (this.boolSelected) {
                                    dojo.addClass(this.domNode, "selected");
                            }
                            if (dojo.isFunction(this.objWidget.placeAt)){
                                    this.objWidget.placeAt(this.domContent, "replace");
                            }
                            else {
                                    dojo.place(this.objWidget, this.domContent, "replace");
                            }
                            this.domNode.intIndex = this.intIndex;
                        }
		},

		destroy: function() {
			if (this.objWidget && typeof this.objWidget.destroy == "function") {
				this.objWidget.destroy();
			}
			delete this.objWidget;
			this.inherited(arguments);
		},

	/* ----------------------------------- Getter / Setter------------------------------------- */

	/* ----------------------------------- Public methods ------------------------------------- */

	/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @private
		 */
		updateContent : function(widgetParams) {
			var modified = true;
			if(this.objWidget && typeof this.objWidget.updateContent == "function") {
				modified = this.objWidget.updateContent(widgetParams);
			}
			return modified;
		}
	}
	);/**
* @class UIElements.CustomList.CustomListItemControl
* @extends UIElements.CustomList.CustomListItemControlBase
* @ignore
*/
dojo.require("UIElements.CustomList.CustomListItemControlBase");
dojo.provide("UIElements.CustomList.CustomListItemControl");
dojo.declare("UIElements.CustomList.CustomListItemControl",
	UIElements.CustomList.CustomListItemControlBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * @private
		 */
		templatePath			: dojo.moduleUrl("UIElements.CustomList","templates/CustomListItem8082.html")

	/* --------------------------------- Private attributes ----------------------------------- */

	/* ------------------------------------ Constructor --------------------------------------- */

	/* ----------------------------------- Getter / Setter------------------------------------- */

	/* ----------------------------------- Public methods ------------------------------------- */

	/* --------------------------------- Private Methods -------------------------------------- */
	}
	);/**
* @class UIElements.Container.ContainerControlBase
* @extends UIElements._base
* Abstract Class - Do not use it directly.
*/
dojo.provide("UIElements.Container.ContainerControlBase");
dojo.declare("UIElements.Container.ContainerControlBase",
	[UIElements._base, dojox.dtl._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */
		/**
        * Content of the Container
        * @property
        * @type HtmlElement
                 * @ignore
        */
		objContent	: null,

		/**
        * Title of the container
        * @property
        * @type String
                 * @ignore
        */
		objTitle	: null,

		/* --------------------------------- Private attributes ----------------------------------- */
		/**
		* @ignore
        */
		_uiTitle	: null,
		/**
		* @ignore
        */
		_uiContent	: null,
		/**
		* @ignore
        */
		_lastObjContent : null,
		/**
		* @ignore
        */
		_boolTitle	: false,
		/**
		 * @ignore
        */
		_boolTab	: false,
		/**
		 * @ignore
		 * Animation if it was called at least one time
        */
		_objAnimation	: null,

		/**
		 * JSON object that defines the view or the UIElement displayed inside this container
		 * This field must contain the name of the view or the class (including the namespace) of the UIElement and the parameters to send to this view or UIElement
		 * @cfg {Object} content (required)
		 * @cfg {String} content.name
		 * @cfg {Object} content.params
		 */
		content : null,

		/**
		 * JSON object that defines the title of this container
		 * This field must contain the name of the view and the parameters to send to this view
		 * @cfg {Object} title (required)
		 * @cfg {String} title.strLabel
		 */
		 title: null,


		/* ------------------------------------ Constructor --------------------------------------- */

		postMixInProperties : function() {
			this._boolTitle = this.objTitle != null;
			if(this.objTitle ){
				if(!this.objTitle.arrItems){
					this.objTitle = {
						arrItems : [this.objTitle]
					}
				} else {
					if(this.objTitle.arrItems.length > 1){
						this._boolTab = true;
					}
				}
			}
		},

		postCreate : function(){
			if(!generalConfig.simulation) {
				this._createContent(true);
			}
		},

		destroy: function() {
			// Destroying title
			if(this._uiTitle) {
				this._uiTitle.destroy();
				delete this._uiTitle;
			}

			// Destroying nested widget
			if (this.domContent != undefined) {
			var domWidget = dojo.query("> *", this.domContent);
			if( domWidget.length > 0 )
			{
				ICTouchAPI.tools.destroyWidgets(domWidget[0]);
			}
			}
			delete this._uiContent;

			if( this._objAnimation )
				this._objAnimation.destroy();
			this.inherited(arguments);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
        * Returns the content object created in the container
                 * @return {HTMLElement} Content of the container
        */
		getContent : function(){
			return this._uiContent;
		},

		/**
        * Returns the title object created in the container
                 * @return {UIElements.ContainerTitle.ContainerTitleControl} Title of the container
        */
		getTitle : function(){
			return this._uiTitle;
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		 * This will only refresh the content and takes no arguments
		 */
		refresh: function() {
			this._createContent(false);
		},

		/**
         * reloads the UIElement in order to change its content. same params sent to the constructor (objContent, objTitle)
         * @param {Object} objParams
		   @param {String} objParams.title  Title of the container
		   @param {HTMLElement} objParams.content  Content of the container (MenuList for example)
         */
		reload : function(objParams) {
			var noTitle = false;
			var noContent = false;
			this._lastObjContent = this.objContent;
			if(objParams.objTitle) {
				if(objParams.objTitle.length){
				this.objTitle = objParams.objTitle;
				} else {
					this.objTitle = [objParams.objTitle];
				}
			} else {
				noTitle = true;
			}
			if(objParams.objContent) {
				this.objContent = objParams.objContent;
			}
			else {
				noContent = true;
			}
			this._createContent(false, noTitle, noContent);
			objParams = null;
		},

		/**
         * Only used by a 2/3 container
         * Hide it under the 1/3 container at its left
         * @param {Function} funcCallback is called when the animation ends
         * @ignore
         */
		moveAway: function(funcCallback) {
			this.nextCallback = funcCallback;
			if( this._objAnimation )
				this._objAnimation.destroy();

			var func = dojo.hitch(this, this._callbackDispatcher);
			this._objAnimation = new UIEffects.Transition.Slide(this.domNode, {
				strDirection: "left",
				funcTransitionEnd: func
			});
			this._objAnimation.launch();
			funcCallback = null;
		},

		/**
         * Only used by a 2/3 container
         * Show it again to the user
         * @param {Function} funcCallback is called when the animation ends
         * @ignore
         */
		moveBack: function(funcCallback) {
			this.nextCallback = funcCallback;
			if( this._objAnimation )
				this._objAnimation.revert();
			else if (funcCallback)
				funcCallback();
			funcCallback = null;
		},

		/**
        * Call method "selectTitleById" of containerTitle (if there is one) to select it.
        * @param {number} intIndex is the index of the title to select.
        * @param {boolean} bNoCallback indicated if we want to execute the callback or just select graphically.
         */
		selectTitleById : function(intIndex, bNoCallback) {
			if(this._uiTitle){
				this._uiTitle.selectTitleById(intIndex, bNoCallback);
			}
		},
		
		/**
         * Change the left Icon of the container title
		 * @param {string} strIcon the icon to put
		 * @param {string} intIndex the index of the title in case of tabs
         */
		setTitleLeftIcon : function(strIcon, intIndex) {
			this.objTitle.strLeftIcon = strIcon;
			this._uiTitle._setStrLeftIconAttr(strIcon, intIndex || 0);
		},

		/**
         * Change the right Icon of the container title
		 * @param {string} strIcon the icon to put
		 * @param {string} intIndex the index of the title in case of tabs
         */
		setTitleRightIcon : function(strIcon, intIndex) {
			this.objTitle.strRightIcon = strIcon;
			this._uiTitle._setStrRightIconAttr(strIcon, intIndex || 0);
		},

		/* --------------------------------- Private Methods -------------------------------------- */
		/**
		 * call the next callback if it exists
		 * @ignore
		 */
		_callbackDispatcher: function() {
			if( this.nextCallback )
			{
				this.nextCallback();
				this.nextCallback = null;
			}
		},

		/**
		 * Create the containg item and remove the old one if necessary
		 * Call updateContent of the content widget, to only update the content if possible
		 * @ignore
		 */
		_createContent: function(first, noTitle, noContent) {
			try
			{
				if (this._boolTitle && noTitle!==true)
				{
					if (first)
					{
						this._uiTitle = new UIElements.ContainerTitle.ContainerTitleControl(this.objTitle, this.domTitle);
					}
					else
					{
						if(this._uiTitle){
							this._uiTitle.updateContent(this.objTitle);
					}
					}
				}
				if ((this.objContent.name || this.objContent.obj) && noContent!==true)
				{
                    var boolReloadRequired = true;
                    if (!first)
					{
                        if(this._lastObjContent.name == this.objContent.name){
                            // update the content with the widget's method
                            boolReloadRequired = this._uiContent.updateContent(this.objContent.params);
                        } else {
                            // We changed the container, store it as last for next createContent
                            this._lastObjContent.name = this.objContent.name;
                        }

                        if(boolReloadRequired){
                            var domWidget = dojo.query("> *", this.domContent);
                            if(domWidget.length > 0){
                                ICTouchAPI.tools.destroyWidgets(domWidget[0]);
                            }
                        }
                    }
                    if(boolReloadRequired){
					// Create the div element for the MenuList
					var div = dojo.create("div", { }, this.domContent);
					div.setAttribute("style","height:inherit;");
					if(this.objContent.name) {
						var newObj = dojo.getObject(this.objContent.name);
						this._uiContent = new newObj(this.objContent.params, div);
					}
					else {
						// Place the Widget in the div created
						this.objContent.obj.placeAt(div);
						this._uiContent = this.objContent.obj;
					}
				}
                }
			}
			catch(e)
			{
				console.error("Error while creating or refreshing a Container "+this.objContent.name);
				if( e.stack ){
					console.error(e.stack);
				}else{
					console.error(e);
				}
			}
		}
	}
	);
/**
* @class UIElements.Container.ContainerControl
* @extends UIElements.Container.ContainerControlBase
* This class is the graphical representation of a container.
*
* A container is the frame of any UIElements or view: by default, a container is used to render a list but it can render a view that displays "own made" UIElements based on HTML and CSS.
*
* Concerning the 1/3 layout, the only restriction is that the container rendered into this 1/3 layout can only embed a {@link UIElements.MenuList.MenuListControl}.
*
* There is no restriction on what can be used in the container rendered into a Full or a 2/3 layout.
*
* Here is an exemple of how to create a container with a menuList
*
*     // Array to popupate MenuList data
*     data: [],
*
*     postCreate : function() {
*           var itemPressedCallback = dojo.hitch(this, this.onItemPressed);
*
*           var container = new UIElements.Container.ContainerControl({
*               title: {
*                   strLabel: "My container title"
*               }
*               content: {
*                   name:"UIElements.MenuList.MenuListControl",
*                   params:{
*                       arrItems:this.data,
*                       callback: itemPressedCallback
*                   }
*               }
*           }, this.menuDomNode);
*     },
*
*     // On item pressed
*     onItemPressed : function(item) {
*       // Do something
*       // ...
*     }
*
*/
dojo.require("UIElements.Container.ContainerControlBase");
dojo.provide("UIElements.Container.ContainerControl");
dojo.declare("UIElements.Container.ContainerControl",
	UIElements.Container.ContainerControlBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * defines the height of the Container: Half/Full
		 * @property
		 * @type String
                 * @ignore
		 */
		strHeight	: "Full",

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		templatePath	: dojo.moduleUrl("UIElements.Container","templates/Container8082.html"),
		/**
		 * @ignore
		 */
		cssFile         : ["base.css", "8082.css"]

	/* ------------------------------------ Constructor --------------------------------------- */
	/* ----------------------------------- Getter / Setter------------------------------------- */
	/* ----------------------------------- Public methods ------------------------------------- */
	/* --------------------------------- Private Methods -------------------------------------- */
	}
	);/**
* @class UIElements.ContainerTitle.ContainerTitleControlBase
* @extends UIElements._base
* Abstract Class - Do not use it directly.
*/

dojo.provide("UIElements.ContainerTitle.ContainerTitleControlBase");
dojo.declare("UIElements.ContainerTitle.ContainerTitleControlBase",
	[UIElements._base, dojox.dtl._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Label displayed
		 * @cfg {String} strLabel (required)
		 */
		strLabel 		: "",


		/**
        * Size of the icon
        * @property
        * @type Number
		 * @ignore
        */
		intIconSize		: 32,

		/**
        * Array with the complete items.
        * @property
        * @type Array
		* One element :
		* (String) strLeftIcon, Left icon - Name of the CSS class, Icon size should be 32x32
		* (String) strRightIcon, Right icon - Name of the CSS class, Icon size should be 32x32
		* (String) strLabel, a label (i18ned or not) for the title
		* (number) intIndex, an id for the title (optionnal, will correspond to the row number of the array by default)
        */
		arrItems		: [],

		/**
        * Array with all i18n labels.
        * @property
        * @type Array
        */
		arrStrLabelI18n : [],
		/* --------------------------------- Private attributes ------------------------------------ */

		/**
        * @ignore
        */
	    arrStrLabelI18nHandler : [],
		/**
        * @ignore
        */
		domTitleContainer	: null,
		/**
        * @ignore
        */
		domLabel		: [],
		/**
        * @ignore
        */
		domLabelBottom: [],
		/**
        * @ignore
        */
		domSecondaryLabel: [],
		/**
        * @ignore
        */
		domIcon: [],
		/**
        * @ignore
        */
		domLeftIcon		: [],
		/**
        * @ignore
        */
		domRightIcon	: [],
		/**
        * @ignore
        */
		domUlTitle		: [],
		/**
        * @ignore
        */
		bMultiTitle		: false,
		/**
        * @ignore
        */
		_intSelected	: null,
		/**
        * @ignore
        */
		templatePath: dojo.moduleUrl("UIElements.ContainerTitle", "templates/ContainerTitle.html"),




		/* ------------------------------------ Constructor --------------------------------------- */

		constructor : function() {
			this.domLabel = [];
			this.domLabelBottom = [];
			this.domSecondaryLabel = [];
			this.domIcon = [];
			this.domLeftIcon = [];
			this.domRightIcon = [];
			this.domUlTitle = [];
			this.arrStrLabelI18n = [];
			this.arrItems = [];
			this.arrStrLabelI18nHandler = [];
		},
		
		postMixInProperties : function() {
			if (this.arrItems.length === 0) {
				this.arrItems.push({
					strLeftIcon			: this.strLeftIcon,
					strRightIcon		: this.strRightIcon,
					strLabelBottom		: this.strLabelBottom,
					strSecondaryLabel	: this.strSecondaryLabel,
					strLabel			: this.strLabel,
					intIndex			: 0
				});
				delete this.strLeftIcon;
				delete this.strRightIcon;
				delete this.strLabelBottom;
				delete this.strSecondaryLabel;
				delete this.strLabel;
			}
		},

		postCreate : function() {			
                        if(!generalConfig.simulation) {
			for(var j = 0 ; j < this.arrItems.length ; j++){
				if(typeof this.arrItems[j].intIndex == "undefined"){
					this.arrItems[j].intIndex = j;
				}
				this.domUlTitle[j].intIndex = this.arrItems[j].intIndex;
				var strTitleSuffix = "Left";
				if(j !== 0){
					if(j !== this.arrItems.length-1){
						strTitleSuffix = "Middle"
					} else {
						strTitleSuffix = "Right"
					}
				}
				dojo.addClass(this.domUlTitle[j], "ContainerTitle"+strTitleSuffix);
				this.changeTitle(this.arrItems[j].strLabel, j);
				if(!generalConfig.simulation) {
				this._createLeftIcon(this.arrItems[j].strLeftIcon, j);
				this._createRightIcon(this.arrItems[j].strRightIcon, j);
				}

				if (this.arrItems[j].strIcon && this.domIcon[j]) {
				    dojo.addClass(this.domIcon[j], this.arrItems[j].strIcon);
				}
			}
			
			if(this.arrItems.length > 1){
				this.bMultiTitle = true;
				dojo.addClass(this.domTitleContainer, "ContainerTabTitle");
			}
                        }
		},

		/* --------------------------------- Public methods ----------------------------------- */

		/**
        * Stop an event propagation
        * @param {Event} event the event whom propagation has to be stopped
                 * @ignore
        */
		clickLeftIcon:function(event){
			event.stopPropagation();
			event = null;
		},

		/**
        * Stop an event propagation
        * @param {Event} event the event whom propagation has to be stopped
                 * @ignore
        */
		clickRightIcon:function(event){
			event.stopPropagation();
			event = null;
		},

		/**
        * Stop an event propagation
        * @param {Event} event the event whom propagation has to be stopped
                 * @ignore
        */
		clickTitle:function(event){
			this.selectTitleById(event.currentTarget.intIndex);
			event.stopPropagation();
			event = null;
		},

		/**
        * Select a title by its intIndex
        * @param {number} intIndex is the index of the title to select.
        * @param {boolean} bNoCallback indicated if we want to execute the callback or just select graphically.
        */
		selectTitleById : function(intIndex, bNoCallback) {
			if(this.bMultiTitle){
				this.setSelected(intIndex);
			}
			if(this.clickTitleCallback && !bNoCallback){
				if(typeof this.clickTitleCallback == "function"){
					this.clickTitleCallback(intIndex);
				} else {
					this.clickTitleCallback.func.call(this.clickTitleCallback.context, intIndex);
				}
			}
		},


		/* ----------------------------------- Getter / Setter------------------------------------- */
		/**
        * set the label
        * @param {String} strLabel new label
        * @param {number} intIndex index of the title to change
		 * @ignore
        */
		setStrLabel : function (strLabel, intIndex) {
			if(typeof intIndex == "undefined"){
				intIndex = 0;
			}
			if (this.arrStrLabelI18n[intIndex]) {
				if(strLabel != this.arrStrLabelI18n[intIndex].toTranslate){
					this.arrStrLabelI18n[intIndex].changeText(strLabel);
				}
			} else {
				if(strLabel != this.arrItems[intIndex].strLabel){
					this.arrItems[intIndex].strLabel = strLabel;
					this.domLabel[intIndex].innerHTML = strLabel;
				}
			}
		},

		/**
	* set the label of the counter below the label
        * @param {String} strLabelBottom new label for the counter
        * @param {number} intIndex index of the title to change
        */
		setStrLabelBottom: function(strLabelBottom, intIndex) {
			if (this.arrItems[intIndex].strLabelBottom) {
			    if (strLabelBottom != this.arrItems[intIndex].strLabelBottom) {
				this.arrItems[intIndex].strLabelBottom = strLabelBottom;
				this.domLabelBottom[intIndex].innerHTML = strLabelBottom;
			    }
			} else {
				this.arrItems[intIndex].strLabelBottom = strLabelBottom;
				this.domLabelBottom[intIndex].innerHTML = strLabelBottom;

			}
		},

		/**
		* set the secondary label of the title
        * @param {String} strSecondaryLabel new label
        * @param {number} intIndex index of the title to change
        */
		setSecondaryLabel: function(strSecondaryLabel, intIndex) {
			if (this.arrItems[intIndex].strSecondaryLabel) {
			    if (strSecondaryLabel != this.arrItems[intIndex].strSecondaryLabel) {
					this.arrItems[intIndex].strSecondaryLabel = strSecondaryLabel;
					this.domSecondaryLabel[intIndex].innerHTML = strSecondaryLabel;
			    }
			} else {
				this.arrItems[intIndex].strSecondaryLabel = strSecondaryLabel;
				this.domSecondaryLabel[intIndex].innerHTML = strSecondaryLabel;

			}
		},

		/**
		* Change the title
		* @param {String} title New title to display
        * @param {number} intIndex index of the title to change
        */
		changeTitle : function (title, intIndex) {
			if(typeof intIndex == "undefined"){
				intIndex = 0;
			}
			if (typeof title === "string" || title.length !== undefined) {
				this.setStrLabel(title, intIndex);
			}
			// retro compatibility
			else if (title.isI18Ned) {
				this._setI18nedLabel(title, intIndex);
			}
			title = null;
		},

		updateContent : function (arrObjTitle) {
			for(var i = 0 ; i < arrObjTitle.length ; i++){
				this._setStrLeftIconAttr(arrObjTitle[i].strLeftIcon, i);
				this.changeTitle(arrObjTitle[i].strLabel, i);
				if (arrObjTitle[i].strLabelBottom) {
				    this.setStrLabelBottom(arrObjTitle[i].strLabelBottom, i);
				}
				if (arrObjTitle[i].strSecondaryLabel) {
				    this.setSecondaryLabel(arrObjTitle[i].strSecondaryLabel, i);
				}
				this._setStrRightIconAttr(arrObjTitle[i].strRightIcon, i);
			}
			arrObjTitle = null;
		},

		/* --------------------------------- Private methods ----------------------------------- */

		
		/**
        * @ignore
        */
		setSelected : function(intIndex) {
			if(this._intSelected != intIndex){
				var intPosition = this._getItemPositionById(intIndex);
				if(this._intSelected != null){
					var intLastSelectedPosition = this._getItemPositionById(this._intSelected);
					dojo.toggleClass(this.domUlTitle[intLastSelectedPosition], "selected", false);
					if(this.domUlTitle[intLastSelectedPosition-1]){
						dojo.toggleClass(this.domUlTitle[intLastSelectedPosition-1], "shadow_right_5", false);
					}
					if(this.domUlTitle[intLastSelectedPosition+1]){
						dojo.toggleClass(this.domUlTitle[intLastSelectedPosition+1], "shadow_left_5", false);
					}
				}
				dojo.toggleClass(this.domUlTitle[intPosition], "selected", true);
				if(this.domUlTitle[intPosition-1]){
					dojo.toggleClass(this.domUlTitle[intPosition-1], "shadow_right_5", true);
				}
				if(this.domUlTitle[intPosition+1]){
					dojo.toggleClass(this.domUlTitle[intPosition+1], "shadow_left_5", true);
				}
				this._intSelected = intIndex;
			}
		},

		_setI18nedLabel : function(i18nLabel, intIndex) {
			// store the new i18n object
			this.arrStrLabelI18n[intIndex] = i18nLabel;
			// if a i18n object was previously used, unsubscribe its handler
			if (this.arrStrLabelI18nHandler[intIndex]) {
				dojo.unsubscribe(this.arrStrLabelI18nHandler[intIndex]);
			}
			// update the dom with the new label and save the handler for the new i18n object
			var handler = this.arrStrLabelI18n[intIndex].fillInnerHTML(this.domLabel[intIndex]);
			this.arrStrLabelI18nHandler[intIndex] = handler;
			i18nLabel = null;
		},

		/**
        * refresh right icon if changed
        * @ignore
        */
		_setStrRightIconAttr : function(value, intIndex) {
			if(value != this.arrItems[intIndex].strRightIcon){
				this._createRightIcon(value, intIndex);
			}
			value = null;
		},

		/**
        * refresh left icon if changed
        * @ignore
        */
		_setStrLeftIconAttr : function(value, intIndex) {
			if(value != this.arrItems[intIndex].strLeftIcon){
				this._createLeftIcon(value, intIndex);
			}
			value = null;
		},

		_getItemPositionById : function(intIndex) {
			for(var i = 0 ; i < this.arrItems.length ; i++){
				if(this.arrItems[i].intIndex == intIndex){
					return i;
				}
			}
			return 0;
		},

		/**
        * create the left Icon
        * @ignore
        */
		_createLeftIcon : function(value, intIndex){
			dojo.removeClass(this.domLeftIcon[intIndex], this.arrItems[intIndex].strLeftIcon + "-" + this.intIconSize);
			if( value )	{
				dojo.style(this.domLeftIcon[intIndex].parentNode, "display", "");
				dojo.addClass(this.domLeftIcon[intIndex], value + "-" + this.intIconSize)
			} else {
				dojo.style(this.domLeftIcon[intIndex].parentNode, "display", "none");
			}
			this.arrItems[intIndex].strLeftIcon = value;
			this._checkForCenterText(intIndex);
			value = null;
		},

		/**
        * create the right Icon
        * @ignore
        */
		_createRightIcon : function(value, intIndex){
			dojo.removeClass(this.domRightIcon[intIndex], this.arrItems[intIndex].strRightIcon + "-" + this.intIconSize);
			if( value )	{
				dojo.style(this.domRightIcon[intIndex].parentNode, "display", "");
				dojo.addClass(this.domRightIcon[intIndex], value + "-" + this.intIconSize)
			} else {
				dojo.style(this.domRightIcon[intIndex].parentNode, "display", "none");
				}
			this.arrItems[intIndex].strRightIcon = value;
			this._checkForCenterText(intIndex);
			value = null;
		},

		/**
        * Center (or not) the text
        * @ignore
        */
		_checkForCenterText : function(intIndex) {
			var boolCentered = this.arrItems[intIndex].strLeftIcon && this.arrItems[intIndex].strRightIcon;

			// Set it to boolean "true" or "false" for the toggle class
				if(boolCentered){
					boolCentered = true;
				} else {
					boolCentered = false;
				}

			dojo.toggleClass(this.domLabel[intIndex], "centered", boolCentered);
		},
		
		destroy: function() {
			for (var i in this.arrStrLabelI18nHandler) {
				if (this.arrStrLabelI18nHandler[i]) {
					dojo.unsubscribe(this.arrStrLabelI18nHandler[i]);
		}
			}

			for (var i in this.domLabel) {
				dojo.destroy(this.domLabel[i]);
				this.domLabel[i] = null;
			}

			this.inherited(arguments);
		}

	});
/**
* @class UIElements.ContainerTitle.ContainerTitleControl
* @extends UIElements.ContainerTitle.ContainerTitleControlBase
* This class is the graphical representation of the title of the container.
*
* The container title contains a label and may contains two icons (in the right corner and in the left corner).
*
* Only the title can be changed at runtime
*
* Here is an exemple of how to change the title of the container
*
*     // Reference to the container
*     container: null,
*
*     postCreate : function() {
*           var itemPressedCallback = dojo.hitch(this, this.onItemPressed);
*
*           this.container = new UIElements.Container.ContainerControl({
*               title: {
*                   strLabel: "My container title"
*               }
*               content: {
*                   name:"UIElements.MenuList.MenuListControl",
*                   params:{
*                       arrItems:[],
*                       callback: itemPressedCallback
*                   }
*               }
*           }, this.menuDomNode);
*     },
*
*     onItemPressed : function() {
*           // Get the ContainerTitle UIElement
*           var titleUI = this.container.getTitle();
*
*           // Update the title
*           titleUI.changeTitle("My new container title");
*     }
*
*/
dojo.require("UIElements.ContainerTitle.ContainerTitleControlBase");
dojo.provide("UIElements.ContainerTitle.ContainerTitleControl");
dojo.declare("UIElements.ContainerTitle.ContainerTitleControl",
	UIElements.ContainerTitle.ContainerTitleControlBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */


		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		templatePath		: dojo.moduleUrl("UIElements.ContainerTitle", "templates/ContainerTitle8082.html"),
		/**
		 * @ignore
		 */
		cssFile				: ["base.css","8082.css"],
		/**
		 * @ignore
		 */
		_objFeedback		: null,

		/* ------------------------------------ Constructor --------------------------------------- */

		constructor	: function() {
		//this._objFeedback = ICTouchAPI.feedbackServices.initFeedback("click");
		},

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 */
		_feedbackLeft : function(clickEvent) {
			if (this.strLeftIcon) {
		//this._objFeedback.feedback(clickEvent);
		}
		},

		/**
		 * @ignore
		 */
		_feedbackRight : function(clickEvent) {
			if (this.strRightIcon) {
		//this._objFeedback.feedback(clickEvent);
		}
		}
	}
	);
/**
* @class UIElements.DataList.DataListControlBase
* @extends UIElements.List.ListControlBase
* Abstract Class - Do not use it directly
* The DataList is an expandable list used to display data : the list elements can contain extra fields.
*/
dojo.provide("UIElements.DataList.DataListControlBase");

dojo.declare("UIElements.DataList.DataListControlBase",
	UIElements.List.ListControl,
	{

		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Callback on line pressed
		 * @cfg {Function} lineClickedCallback
		 */
		lineClickedCallback         : null,

		/**
		 * Callback on icon pressed
		 * @cfg {Function} iconClickedCallback
		 */
		iconClickedCallback         : null,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * CSS highlight class
		 * @ignore
		 */
		_highlightClass : "highlighted",
		
		/**
		 * true if we must wait for highlighting, false otherwise
		 * @ignore
		 * @type {Boolean}
		 */
		_enableWaitingHighlight : true,


		/* --------------------------------- Private Methods -------------------------------------- */


		/**
		 * @ignore
		 */
		_getLiTemplate : function _getLiTemplate(item) {
			var icons = item.arrIcons;
			var li = this.newElement("li").mixin({
				className:"selectableLine"
			})
			.attachEvent({event: "click", func: "_clickedLine", capture:false})

			.attachEvent({event: "mousedown", func: "_startHighlight", capture: false});

			if (icons[0]) {
				var icon0 = this.newElement("div").mixin({
					className: "DataListIconStatus " + icons[0]
				});
				li.appendChild(icon0);
			}


			if (icons[1] || icons[2]) {
				var notifs = this.newElement("div").mixin({
					className: "DataListIconNotifications"
				});

				if (icons[1]) {
					var icon1 = this.newElement("span").mixin({
						className: icons[1]
					});
					notifs.appendChild(icon1);
				}

				if (icons[2]) {
					var icon2 = this.newElement("span").mixin({
						className: icons[2]
					});
					notifs.appendChild(icon2);
				}
				li.appendChild(notifs);
			}

			var main = this.newElement("div").mixin({
				className: "DataListMain"
			});

			if (item.strPrimaryContent) {
				main.innerHTML = item.strPrimaryContent;
			}


			if (item.strSecondaryContent) {
				main.innerHTML = "<br />" + item.strSecondaryContent;
			}

			li.appendChild(main);

			if (item.arrAdditionalContents && item.arrAdditionalContents.length) {
				var secondary = this.newElement("div").mixin({
					className: "DataListSecondary"
				});
				secondary.innerHTML = item.arrAdditionalContents.join(" ");
				li.appendChild(secondary);
			}


			if (icons[3]) {
					var icon3 = this.newElement("div").mixin({
						className: "DataListIcon DataListIconOther " + icons[3]
					});
					li.appendChild(icon3);
			}

			if (icons[3] && icons[4]) {
					var separator = this.newElement("div").mixin({
						className: "DataListSeparator"
					});
					li.appendChild(separator);
			}

			if (icons[4]) {
					var icon4 = this.newElement("div").mixin({
						className: "DataListAction"
					}).attachEvent({event: "click", func: "_clickedIcon", capture:false}, {item: item});
					
					var span = this.newElement("span").mixin({
						className: "DataListIcon " + icons[4]
					});
					icon4.appendChild(span);

					li.append(icon4);
			}
			return li;
		},

		/**
		 * @param {HTMLElement} element UI element to be paginated
		 * @return {Function} function
		 * @ignore
		 */
		_clickedIcon : function (event, args) {
			if (args.PARENT_WIDGET.iconClickedCallback) {
			event.cancelBubble = true;
				args.PARENT_WIDGET.iconClickedCallback(args.item);
			}
		},

		/**
		 * @ignore
		 */
		_clickedLine : function (event, args) {
			if (args.PARENT_WIDGET.lineClickedCallback)
				args.PARENT_WIDGET.lineClickedCallback(this._itemSource);
		},

		/**
		 * @ignore
		 */
		_feedback : function (event) {
		//this._objFeedback.feedback(event);
		},


		/* ----------------------------------- Public methods ------------------------------------- */

		/**
	 * Remove one item from the top of the list
	 * @param {Array} completeList List from which to shift an item
	 * @return {Object} li or this
	 */
		shiftItem : function (item) {
			ICTouchAPI.debugServices.error("DATALIST shiftItem");
			this._checkForEmpty();
			return this._removeItemInDom(item, true);
		},

		/**
	 * Remove multiple items from the top of the list.
	 * @param {Number} nb number of elements to shifted
	 * @return {Array} local array of items
	 */
		shiftItems : function () {
			var arrItemsCopy = dojo.clone(this.arrItems);
			var itemToNotDisplay = arrItemsCopy.splice(this.nbElementsMax-1,arrItemsCopy.length-1);

			for (var i in itemToNotDisplay) {
				this.shiftItem(itemToNotDisplay[i]);
			}
			this._checkForEmpty();
			return this.arrItems;
		},


			/**
	 * Push a new item at the end.
	 * @param {Object} item item to be pushed
	 * @param {Boolean} addNow if the element is to be displayed right away
	 * @return {Boolean} result of the push operation
	 */
		pushItem : function (item,  addNow) {
			this._cleanItemType(item);
			this.arrItems.push(item);
			this._checkForEmpty();
			//return addNow && this.push(1);
		},

		/**
	 * Push an array of new items at the end.
	 * @param {Array} arrItems array of the items to be pushed
	 * @return {Boolean} result of the push operation
	 */
		pushItems : function (arrItems, addNow) {
			var func = function (item) {
				this._cleanItemType(item);
				this.pushItem(item, addNow);
			}
			arrItems.forEach(func, this);
			for(var i= 0 ; i < this.nbElementsMax ; i++ ){
				this.push(1);
			}
			//this.shiftItems();
			this._checkForEmpty();
		}

	});
/**
* @class UIElements.DataList.DataListControl
* @extends UIElements.DataList.DataListControlBase
*
* This class is the graphical representation of a data list.
*
* A DataList is a list dedicated to be used in a full layout (don't use it in a 1/3 or a 2/3 layout, because the display is not adapted).
*
* This list can display a primary content (strPrimaryContent), an additionnal content and a maximum of 2 icons at the right.
*
* The last icon displayed can be attached to a specific callback thanks to the property *iconClickedCallback*.
*
* By default the DataList doesn't highlight the selected item automatically. If you want to highlight it, you should use the function {@link UIElements.List.ListControlBase#toggleEmphasis}
*
* Here is an example of creating a DataList in a full layout. See the class {@link UIElements.ApplicationMode.FullControl} to understand how to create a full layout.
*
*     items: [];
*
*     postCreate : function() {
*
*           var iconCallback = dojo.hitch(this, this.onIconPressed),
*               lineCallback = dojo.hitch(this, this.onLinePressed);

*          // Push a item to the list
*          this.items.push({
*               intIndex: 0,
*               strPrimaryContent: "Contact A",
*               arrAdditionnalContents: ["80874", "80675"],                     // Additionnal contents
*               boolEmphasis: false,                                            // Highlight this item or not
*               arrIcons: ["", "", "", "phoneIconClass", "contactCardClass"]    // Icons to be displayed
*          });
*
*          // Create the container
*          var container = new UIElements.Container.ContainerControl({
*               title: {
*                   strLabel: "Full DataList"
*               }
*               content: {
*                   name:"UIElements.DataList.DataListControl",
*                   params:{
*                       arrItems: this.items,
*                       boolShowAdditionalContents: true,
*                       lineClickedCallback: lineCallback,
*                       iconClickedCallback: iconCallback,
*                   }
*               }
*          }, this.containerDomNode);
*     },
*
*     onIconPressed : function(idx) {
*           // Do something
*           // ...
*     },
*
*     onLinePressed : function (idx) {
*           // Do something
*           // ...
*     }
*
* An item in the list is a JSON Object that has the following structure
*     var item = {
*          intIndex: 0,
*          strPrimaryContent: "Contact A",
*          arrAdditionnalContents: ["80874", "80675"],                     // Additionnal contents
*          boolEmphasis: false,                                            // Highlight this item or not
*          arrIcons: ["", "", "", "phoneIconClass", "contactCardClass"]    // Icons to be displayed
*     };
*
* The field *arrAdditionnalContents* is an array of Strings that can be displayed. You're are not limited concerning the number of Strings displayed. Be careful, to not override the screen.
*
* The field *arrIcons* is an array of Strings that represents the icons (CSS classes) displayed on the list. Be careful to define the same kind of array (with 5 elements) and use only the two last to display your icons.
*/
dojo.require("UIElements.DataList.DataListControlBase");
dojo.provide("UIElements.DataList.DataListControl");

dojo.declare("UIElements.DataList.DataListControl",
	UIElements.DataList.DataListControlBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Array of items
		 * @property
		 * @type {Array}
                 * @ignore
		 */
		arrSecondary				: [ ],

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		templatePath: dojo.moduleUrl("UIElements.DataList","templates/DataList8082.html"),
		/**
		 * @ignore
		 */
		cssFile         : ["base.css","8082.css"],

		/* ------------------------------------ Constructor --------------------------------------- */

		constructor : function(){
			this.arrSecondary 		= [ ];
		},

		/* ------------------------------------- Destructor --------------------------------------- */

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 */
		_showSecondary: function(i){
			if (!this.arrItems[i].strSecondaryContent || this.arrItems[i].strSecondaryContent == ""){
				dojo.addClass(this.arrSecondary[i], "OneLineContent");
			}
		}

	});
dojo.provide("UIElements.GlobalContainer.GlobalContainerControlBase");
dojo.declare("UIElements.GlobalContainer.GlobalContainerControlBase",
	[UIElements._base, dijit._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/*
		 * top object
		 * @property
		 * @type Object
		 */
		objTop      : null,

		domTop      : null,
		domBottom   : null,

		/* --------------------------------- Private attributes ----------------------------------- */

		/* ------------------------------------ Constructor --------------------------------------- */

		postCreate:function() {
			//test if the argument top.name exists
			// note: test necessary due to the initialization webapp which
			// does not need the top bar
			if(this.objTop && this.objTop.name){
				var tempObj = ICTouchAPI.tools.getObjectReference(this.objTop.name);
				new tempObj(this.objTop.params, this.domTop);
			}
		}
		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */
	}
	);
dojo.require("UIElements.GlobalContainer.GlobalContainerControlBase");
dojo.provide("UIElements.GlobalContainer.GlobalContainerControl");
dojo.declare("UIElements.GlobalContainer.GlobalContainerControl",
	UIElements.GlobalContainer.GlobalContainerControlBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */
		// The paths to the template and the css file are defined here
		templatePath    : dojo.moduleUrl("UIElements.GlobalContainer","templates/GlobalContainer8082.html"),
		cssFile         : ["base.css", "8082.css"],


		domMiddle       : null


		/* --------------------------------- Private attributes ----------------------------------- */

		/* ------------------------------------ Constructor --------------------------------------- */

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */
	}
	);
/**
* @class UIElements.InputField.InputFieldControlBase
* @extends UIElements._base
* Abstract Class - Do not use it directly
*/
dojo.provide("UIElements.InputField.InputFieldControlBase");
dojo.declare("UIElements.InputField.InputFieldControlBase",
	[UIElements._base, dojox.dtl._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Text displayed into the InputField element
		 * @cfg {String} strInputText
		 */
		strInputText	: "",

		/**
		 * Text displayed into the InputField (used when boolPassword = true)
		 * @cfg {String} strInputContent
		 */
		strInputContent		: "",

		/**
		 * Keyboard to display. See {@link ICTouchAPI.KeyboardTypes} for more information about the available keyboards
		 * @cfg {String} keyboardType
		 */
		keyboardType	: "",

		/**
		 * Initial mode for alphanumeric keyboard (ICTouchAPI.KeyboardAlphanumMode enumeration) (default: NORMAL)
		 * @property
		 * @type String
		 */
		strDefaultMode	: null,

		/**
		 * Default text to be displayed in the keyboard when activated
		 * @cfg {String} strKeyboardDefaultText
		 */
		strKeyboardDefaultText	: null,

		/**
		 * Callback fired when the OK button is pressed
		 * @cfg {Function} funcCallback
		 */
		funcCallback	: null,

		/**
		 * Callback fired when the OK button of the keyboard is pressed
		 * @cfg {Function} funcCallbackOk
                 * @ignore
		 */
		funcCallbackOk	: null,

		/**
		 * Callback fired when the EXIT button of the keyboard is pressed
		 * @cfg {Function} funcCallbackExit
		 */
		funcCallbackExit: null,

		/**
		 * Name of the group of fields this input belongs to.
		 * @property
		 * @type Function
                 * @ignore
		 */
		linkedFields	: "",

		/**
		 * Displayed an InputField with several lines. Not compatible with a PresentationList (Should be set to false when used with a PresentationList)
		 * @cfg {Boolean} isMultiline
		 */
		isMultiline	: false,

		/**
		 * Maximum number of lines allowed. Not compatible with a PresentationList (Should not be used with a PresentationList)
		 * @cfg {Number} intMaxLine
		 */
		intMaxLine		: 3,

/**
		 * Should the inputfield directly use the max line number, even if the content is empty ? (note, isMultiline should be set to true)
		 * @cfg {Boolean} boolForceMaxLineSize
		 */
		boolForceMaxLineSize : false,

		/**
		 * if true, the value of the input filed has chanegd
		 * @property
		 * @type Boolean
                 * @ignore
		 */
		valueChanged	: false,

		/**
		 * Label of the keyboard OK button
		 * @cfg {String} strOK
		 */
		strOk			: null,

		/**
		 * Label of the Exit button
		 * @cfg {String} strExit
		 */
		strExit			: null,

		/**
		 * Callback called when clicking outside the keyboard
		 * @property
		 * @type Function
                 * @ignore
		 */
		actionOutsideClick : null,

		/**
		 * Display InputField as a password field. Show stars instead of actual characters.
		 * @cfg {Boolean} boolPassword
		 */
		boolPassword : false,

		/**
		 * Enable of disable the inputField. If the InputField is disabled, the keyboard will not be displayed.
		 * @cfg {Boolean} boolEnabled
		 */
		boolEnabled : true,

		/**
		 * Hides or not stars if password is empty
		 * @cfg {Boolean} hideStarsIfPasswordEmpty
		 */
		hideStarsIfPasswordEmpty : false,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		numLockKeyboard	: 0,
		/**
		 * @ignore
		 */
		strEventId		: null,
		/**
		 * @ignore
		 */
		domInput		: null,
		/**
		 * @ignore
		 */
		domTextarea		: null,
		/**
		 * @ignore
		 */
		oldScrollHeight : 0,
		/**
		 * @ignore
		 */
		domExtraLabel	: null,
		/**
		 * @ignore
		 */
		domExtra		: null,
		/**
		 * @ignore
		 */
		domExtraIcon	: null,
		/**
		 * @ignore
		 */
		_extraCallback	: null,
		/**
		 * @ignore
		 */
		_extraAutorepeat: false,
		/**
		 * @ignore
		 */
		_autorepeatInterval: null,
		/*
		 * @ignore
		 */
		_autorepeatTimeout: null,
		/**
		 * @ignore
		 */
		_extraLabelSubscribe : null,
		/**
		 * @ignore
		 */
		myFeedback				: null,

		/* ------------------------------------ Constructor --------------------------------------- */

		constructor: function () {
			this._arrInputThreshold = [];
			this.myFeedback = ICTouchAPI.feedbackServices.initFeedback("click");
		},

		postMixInProperties: function() {
			// Backward compatibility
			if (typeof this.isMultiLines === "boolean")
			{
				this.isMultiline = this.isMultiLines;
				delete this.isMultiLines;
			}
		},

		postCreate : function () {
			if (this.strInputIcon){
				dojo.addClass(this.domWithIcon, "opIcon");
				dojo.addClass(this.domIcon, this.strInputIcon);
			}
			if(this.strInputText && !this.strInputContent) {
				this.strInputContent = this.strInputText;
			}
			if(!this.strInputText && this.strInputContent) {
				this.strInputText = this.strInputContent;
			}
			if (!this.boolEnabled){
				this.getInputFieldObject().setAttribute("disabled","disabled");
			}
			this.setMultiline(this.isMultiline);
		},

		destroy: function() {
			if (this._extraLabelSubscribe) {
				dojo.unsubscribe(this._extraLabelSubscribe);
				this._extraLabelSubscribe = null;
			}
			this._clearExtraTimers();
			this.inherited(arguments);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * Get the text of this element
		 * @return {String} text of element
		 */
		getInputFieldContent: function () {
			return dojox.dtl.filter.htmlstrings.striptags(this.strInputContent);
		},

		/**
		 * Set the text of this element
		 * @param {String} strContent Text to display in this element
		 * @param {Boolean} boolCallback Execute callback or not
		 * @param {Boolean} hideStarsIfPasswordEmpty In password mode, the empty string will be replace by "*****", if parameter is set to true, it will not replace the content in case of an empty string
		 * @return {String} text of element
                 * @ignore
		 */
		setInputFieldContent: function (strContent, boolCallback, hideStarsIfPasswordEmpty) {
			var field = this.getInputFieldObject();
			this.strInputText = strContent;
			this.strInputContent = strContent;
			if(this.boolPassword && (this.strInputText || (!hideStarsIfPasswordEmpty && !this.hideStarsIfPasswordEmpty))){
				this._convertInputToPassword(this.strInputText.length);
			}
			if (boolCallback == undefined || boolCallback == null || boolCallback == true){
				if(this.funcCallback){
					this.funcCallback(strContent);
				}
			}
			field.value = this.strInputText;
			field.selectionStart = this.strInputText.length;
			field.selectionEnd = this.strInputText.length;
			this.domTextarea.rows = 1;
			this._updateInputFieldRows();

			return strContent;
		},

		/**
		 * Set a new text int the InputField
		 * @param {String} newValue New text to display
		 */
		setValue: function (newValue) {
			this.setInputFieldContent(newValue, false);
		},

		/**
		 * Should this field be a password field. This will erase the content
		 * @param {Boolean} password is this text field a password field
		 */
		setPassword: function (password) {
			this.boolPassword = password;
			this.setInputFieldContent("", false);
		},

		/**
		 * Retrieve maximum number of lines allowed
		 * @return {Number}  maximum number of lines
		 */
		getMaxLine: function() {
			return this.intMaxLine;
		},

		/**
		 * Set maximum number of lines allowed
		 * @param {Number} nbLine new maximum number of lines
		 */
		setMaxLine: function(nbLine) {
			this.intMaxLine = nbLine;
		},

		/**
		 * Retrieve html dom element of the input field
		 * @return {HTMLElement}  Html dom element
                 * @ignore
		 */
		getInputFieldObject: function () {
			return this.isMultiline ? this.domTextarea : this.domInput;
		},

		/**
		 * Retrieve html dom element's length
		 * @return {Number}  Html dom element's length
                 * @ignore
		 */
		getInputFieldLength: function () {
			return this.getInputFieldObject().length;
		},

		/**
		 * Retrieve input field selection start index and selection end index
		 * @return {Array}  Input field selection start and end
		 */
		getSelection: function () {
			var field = this.getInputFieldObject();
			return [
			field.selectionStart,
			field.selectionEnd
			];
		},

		/**
		 * Retrieve cursor position in the input field
		 * @return {Number}  cursor position
		 */
		getCaretPos: function () {
			return this.getInputFieldObject().selectionStart;
		},

		/**
		 * Retrieve Multiline status
		 * @return {Boolean}  true if multiline is allowed
		 */
		getMultiline: function () {
			return this.isMultiline;
		},

		/**
                 * Switch to a multi lines or single line InputField
                 * @param {Boolean} boolMultiline True to switch to multi lines InputField
                 */
                setMultiline: function(boolMultiline) {
			var multiline = false;
			if(boolMultiline===true) {
				multiline = true;
			}
			this.isMultiline = multiline;
			dojo.toggleClass(this.domInput, "hidden", multiline);
			dojo.toggleClass(this.domTextarea, "hidden", !multiline);
			this.setInputFieldContent(this.strInputContent, false, this.hideStarsIfPasswordEmpty);
		},

		/**
		 * Retrieve ForceMaxLineSize status
		 * @return {Boolean}  true if ForceMaxLineSize is set
		 */
		getForceMaxLineSize: function () {
			return this.boolForceMaxLineSize;
		},

		/**
		* Set ForceMaxLineSize
		* @param {Boolean} boolForceMaxLineSize True to activate boolForceMaxLineSize
		*/
		setForceMaxLineSize: function(boolForceMaxLineSize) {
			this.boolForceMaxLineSize = boolForceMaxLineSize;
			// update the textarea
			this._updateInputFieldRows();
		},

		/**
		 * Add or remove the extra button
		 * @param {Function} callback function to call when the user click or null to remove the button
		 * @param {Boolean} should the button autorepeat when the button stay pressed
		 * @param {String} label to show or null to remove the label
		 * @param {String} icon for the button
                 * @ignore
		 */
		setExtraButton: function(callback, autorepeat, label, icon) {
			// Because this label can't be handled by this widget lifecycle
			if (this._extraLabelSubscribe) {
				dojo.unsubscribe(this._extraLabelSubscribe);
				this._extraLabelSubscribe = null;
			}

			this.domExtraLabel.innerHTML = "";
			if (typeof callback == "function") {
				this._extraCallback = callback;
				if (label && label.fillInnerHTML) {
					this._extraLabelSubscribe = label.fillInnerHTML(this.domExtraLabel);
				}
				if (icon && typeof icon == "string") {
					this.domExtraIcon.className = "icon "+icon+"-32";
				}
				else {
					this.domExtraIcon.className = "";
				}
				dojo.addClass(this.domNode, "withExtra");
				this._extraAutorepeat = autorepeat;
			}
			else {
				this._extraCallback = null;
				dojo.removeClass(this.domNode, "withExtra");
			}
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		 * Del characters from selection's start to end.
		 * If selection's start and end are equals, dels the previous char.
                 * @ignore
		 */
		delChars: function() {
			// Dels chars from selectionStart to selectionEnd.
			// If selectionStart equals selectionEnd, then dels the previous char.
			var field = this.getInputFieldObject();
			var intSelectionStart = field.selectionStart;
			var intSelectionEnd = field.selectionEnd;

			// If no selection is made
			if (intSelectionStart === intSelectionEnd) {
				// If caret is at position 0, no need to do anything
				if (intSelectionStart > 0) {
					// Removes the character before caret position
					if (intSelectionStart === this.strInputText.length) {
						this.strInputText = this.strInputText.substring(0, this.strInputText.length - 1);
						this.strInputContent = this.strInputContent.substring(0, this.strInputContent.length - 1);
					}
					else {
						this.strInputText = this.strInputText.substring(0, intSelectionStart - 1) + this.strInputText.substring(intSelectionStart);
						this.strInputContent = this.strInputContent.substring(0, intSelectionStart - 1) + this.strInputContent.substring(intSelectionStart);
					}
					field.value = this.strInputText;
					// Sets back the caret position.
					field.selectionStart = intSelectionStart-1;
					field.selectionEnd = intSelectionStart-1;
				}
			}
			// If several characters are selected
			else {
				// Removes selected chars
				this.strInputText = this.strInputText.substring(0, intSelectionStart) + this.strInputText.substring(intSelectionEnd);
				this.strInputContent = this.strInputContent.substring(0, intSelectionStart) + this.strInputContent.substring(intSelectionEnd);
				field.value = this.strInputText;
				// Sets back the caret position.
				field.selectionStart = intSelectionStart;
				field.selectionEnd = intSelectionStart;
			}

			this._updateInputFieldRows();
		},

		/**
		 * This method is called when the virtual keyboard is used
		 * this.strInputContent is manually updated with given UTF8char
		 * @param {String} UTF8char
                 * @ignore
		 */
		addNewChar : function(UTF8char) {
			var field = this.getInputFieldObject();
			var intSelectionStart = field.selectionStart;
			var intSelectionEnd = field.selectionEnd;

			if (UTF8char != '') {
				this.strInputContent = this.strInputContent.substring(0, intSelectionStart) + UTF8char + this.strInputContent.substring(intSelectionEnd);
				this.strInputText = this.strInputText.substring(0, intSelectionStart) + UTF8char + this.strInputText.substring(intSelectionEnd);
				field.value = this.strInputText;
				if (this.boolPassword) {
					this._convertInputButLastCharToPassword();
				}
				field.selectionEnd = field.selectionStart = intSelectionStart + 1;
			}

			this.domInput.scrollLeft = this.domInput.scrollWidth;
			this._updateInputFieldRows();
		},

		/**
		 * Launch the Keyboard
		 * @param {Event} event
                 * @ignore
		 */
		deployKeyboard: function (event) {
			if (this.boolEnabled){
				this.valueChanged = true;
				if(this.strEventId!=null)
					dojo.publish(this.strEventId, [this]);
				if( this.keyboardType !== "" )
				{
					ICTouchAPI.keyboardServices.deployKeyboard(this.keyboardType, this, {
						strOkLabel : this.strOk,
						strExitLabel : this.strExit,
						funcCallbackOk : this.funcCallbackOk,
						funcCallbackExit : this.funcCallbackExit,
						actionOutsideClick : this.actionOutsideClick,
						strDefaultText : this.strKeyboardDefaultText,
						strInputTitle : this.strInputTitle,
						hideStarsIfPasswordEmpty : this.hideStarsIfPasswordEmpty,
						strDefaultMode: this.strDefaultMode
					});
					if(event)
						dojo.stopEvent(event);
				}
			}
		},

		/**
		 * Enable or disable the inputField element
		 * @param {Boolean} disable True to disable the InputField
		 */
		toggleDisabledMode : function(disable){
			this.boolEnabled = !disable;
			if (disable){
				this.getInputFieldObject().setAttribute("disabled","disabled");
			}
			else{
				this.getInputFieldObject().removeAttribute("disabled");
			}
		},

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 */
		_extraButtonDown: function(clickEvent) {
			this._feedBack(clickEvent);
			if (typeof this._extraCallback == "function") {
				this._extraCallback();
				if (this._extraAutorepeat) {
					this._autorepeatTimeout = setTimeout(dojo.hitch(this, this._extraRepeat), 500);
				}
			}
		},

                /**
                 * @ignore
                 */
		_clearExtraTimers: function() {
			if (this._autorepeatTimeout) {
				clearTimeout(this._autorepeatTimeout);
				this._autorepeatTimeout = null;
			}
			if (this._autorepeatInterval) {
				clearInterval(this._autorepeatInterval);
				this._autorepeatInterval = null;
			}
		},


                /**
                 * @ignore
                 */
		_extraRepeat: function() {
			this._extraCallback();
			this._autorepeatTimeout = null;
			this._autorepeatInterval = setInterval(this._extraCallback, 100);
		},

		/**
		 * @ignore
		 */
		_updateInputFieldRows: function () {
			if (this.isMultiline) {
				// If we force the line size
				if (this.boolForceMaxLineSize) {
					this.domTextarea.rows = this.intMaxLine;
				}else {
					var rows = Math.floor(this.domTextarea.scrollHeight / 30);
					// Clamp value between 1 and intMaxLine
					rows = Math.max(Math.min(rows, this.intMaxLine), 1);
					this.domTextarea.rows = rows;
					if(this.oldScrollHeight !== this.domTextarea.scrollHeight) {
						var diffScrollHeight = this.domTextarea.scrollHeight - this.oldScrollHeight;
						this.oldScrollHeight = this.domTextarea.scrollHeight;
						this.domTextarea.scrollTop += diffScrollHeight;

					}
				}
			}
		},

		/**
		 * @ignore
		 * This method is triggered when a physical keyboard is used
		 * The value in inputField is not refreshed at this time
		 * Both this.strInputText and this.strInputContent are manually updated with new event.keychar
		 */
		_onKeyPress : function(event) {
			var field = this.domInput;
			var intSelectionStart = field.selectionStart;
			var intSelectionEnd = field.selectionEnd;

			// A new letter / number was typed
			if (event.keyChar !== '') {
				this.addNewChar(event.keyChar);
				dojo.stopEvent(event);
			}
			// Backspace key was hit
			else if (event.keyCode === 8) {
				if (intSelectionStart === intSelectionEnd) {
					if (intSelectionStart > 0) {
						if (intSelectionStart === this.strInputText.length) {
							this.strInputText = this.strInputText.substring(0, this.strInputText.length - 1);
							this.strInputContent = this.strInputContent.substring(0, this.strInputContent.length - 1);
						}
						else {
							this.strInputText = this.strInputText.substring(0, intSelectionStart - 1) + this.strInputText.substring(intSelectionStart);
							this.strInputContent = this.strInputContent.substring(0, intSelectionStart - 1) + this.strInputContent.substring(intSelectionStart);
						}
						// Replace the carrot at the right place ( else it will show up a the end )
						field.selectionStart = intSelectionStart-1;
					}
				}
				else {
					this.strInputText = this.strInputText.substring(0, intSelectionStart) + this.strInputText.substring(intSelectionEnd);
					this.strInputContent = this.strInputContent.substring(0, intSelectionStart) + this.strInputContent.substring(intSelectionEnd);
				}
			}
			// Delete key was hit
			else if (event.keyCode === 46) {
				if (intSelectionStart === intSelectionEnd) {
					if (intSelectionStart < this.strInputText.length) {
						if (intSelectionStart === 0) {
							this.strInputText = this.strInputText.substring(1);
							this.strInputContent = this.strInputContent.substring(1);
						}
						else {
							this.strInputText = this.strInputText.substring(0, intSelectionStart) + this.strInputText.substring(intSelectionStart+1);
							this.strInputContent = this.strInputContent.substring(0, intSelectionStart) + this.strInputContent.substring(intSelectionStart+1);
						}
					}
				}
				else {
					this.strInputText = this.strInputText.substring(0, intSelectionStart) + this.strInputText.substring(intSelectionEnd);
					this.strInputContent = this.strInputContent.substring(0, intSelectionStart) + this.strInputContent.substring(intSelectionEnd);
				}
			}
			this.domInput.value = this.strInputText;
			dojo.stopEvent(event);
		},

		/**
		 * @ignore
		 * This method converts field.value but last char in stars
		 */
		_convertInputButLastCharToPassword : function() {
			// The whole string but last char is replaced by ? chars
			var field = this.getInputFieldObject();
			if(this.strInputContent.length == 1){
				this.strInputText = this.strInputContent;
			}
			var strLength = this.strInputText.length;
			var str = ""
			for (var i=0; i<strLength; i++) {
				if (i == strLength - 1) {
					str += this.strInputText.charAt(i);
				}
				else {
					str += '*';
				}
			}
			this.strInputText = str;
			field.value = this.strInputText;
			// 1 second later, the whole string including last char needs to be replaced by * chars
			var func = dojo.hitch(this, this._convertInputToPassword);
			setTimeout (function(){
				func(strLength);
			}, 1000);
		},

		/**
		 * @ignore
		 * This method converts field.value in stars
		 */
		_convertInputToPassword : function (currentStrLength) {
			// This test is made so no replacement is made if another char has been typed during the 3 seconds delay
			if(!currentStrLength || currentStrLength <= 0){
				var field = this.getInputFieldObject();
				this.strInputText = "*****";
				field.value = this.strInputText;
			}else{
				if (this.strInputText.length == currentStrLength) {
					var field = this.getInputFieldObject();
					var str = "";
					for (var i=0; i<currentStrLength; i++) {
						str += '*';
					}
					this.strInputText = str;
					var intSelectionStart = field.selectionStart;
					var intSelectionEnd = field.selectionEnd;
					// After setting the value of a field we must restore selectionStart and selectionEnd
					field.value = this.strInputText;
					field.selectionStart = intSelectionStart;
					field.selectionEnd = intSelectionEnd;
				}
			}
		},


                /**
                 * @ignore
                 */
		_feedBack:function(clickEvent){
				this.myFeedback.feedback(clickEvent);
		}
	}
	);
/**
* @class UIElements.InputField.InputFieldControl
* @extends UIElements.InputField.InputFieldControlBase
*
* This class is the graphical representation of an input field.
*
* An input field is a text area used to capture information from the user.
*
* The input field could be used directly on a input DOM element or in a {@link UIElements.PresentationList.PresentationListControl}.
*
* In both case, when the user presses on the input field, the keyboard is displayed to catch the user's input.
*
* Here are two exemples of how to display an InputField.
*
* 1/ InputField attached to a DOM node
*
* To display the InputField, an Input node should be created into the HTML template
*
*     <input type="text" readonly="true" class="InputField" dojoAttachPoint="domInputFieldContent" />
*
* The next task is to create the view that will display the InputField element. This view can be the 2/3 of a layout.
*
*     postCreate : function() {
*
*       // Callbacks
*       var funcOk = dojo.hitch(this, this.OnInputOk),
*           funcExit = dojo.hitch(this, this.OnInputExit);
*
*       var input = new UIElements.InputField.InputFieldControl({
*           keyboardType: ICTouchAPI.KeyboardTypes.ALPHANUM,    // keyboard type
*           strInputField: "Paris",                             // Default text displayed
*           isMultiLines: false,                                // Simple line or multiline
*           strOk: "OK",                                        // Text displayed for the Keyboard OK button
*           strExit: "Exit",                                    // Text displayed for the Exit Keyboard button
*           funcCallback: funcOk,                               // Callback on OK button pressed
*           funcCallbackExit: funcExit                          // Callback on Exit button pressed
*           }, this.domInputFieldContent);
*     },
*
*     OnInputOk : function() {
*       // Do something
*       // ...
*     },
*
*     OnInputExit : function() {
*       // Do something
*       // ...
*     }
*
* Using this method, you should define a CSS style to place correctly your InputField in your view.
*
* See {@link ICTouchAPI.KeyboardTypes} class for the list of available keyboards.
*
* 2/ InputField included in a PresentationList
*
* This second solution describes how to add an InputField into a PresentationList element
*
* First, create the DOM node associated to the PresentationList
*
*     <div dojoAttachPoint="domPresentationList" />
*
* Now, in the associated view, the InputField element should be an item of the PresentationList
*
*     postCreate : function() {
*
*       // callbacks
*       var funcOk = dojo.hitch(this, this.OnInputOk),
*           funcExit = dojo.hitch(this, this.OnInputExit);
*
*       // PresentationList items
*       var data = [];
*
*       data.push({
            strLabel: "City",       
*           objContent: new UIElements.InputFIeld.InputFieldControl({
*               keyboardType: ICTouchAPI.KeyboardTypes.ALPHANUM,
*               strInputField: "Paris",
*               isMultiLines: false,
*               strOk: "OK",
*               strExit: "Exit",
*               funcCallback: funcOk,
*               funcCallbackExit: funcExit
*           })
*       });
*
*       this.presentationList = new UIElements.PresentationList.PresentationListControl({
*           arrItems: data,
*           boolShowLabel: true
*       }, this.domPresentationList);
*     },
*
*     OnInputOk : function(inputText) {
*       // Do something
*       // ...
*     },
*
*     OnInputExit : function(inputText) {
*       // Do something
*       // ...
*     }
*
* New text entered using the keyboard is sent as a parameter to the callback.
*
*/
dojo.require("UIElements.InputField.InputFieldControlBase");
dojo.provide("UIElements.InputField.InputFieldControl");
dojo.declare("UIElements.InputField.InputFieldControl",
	UIElements.InputField.InputFieldControlBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		templatePath		: dojo.moduleUrl("UIElements.InputField","templates/InputField8082.html"),
		/**
		 * @ignore
		 */
		cssFile        		: "base.css",
		/**
		 * @ignore
		 */
		_objFeedback		: null,

		/* ------------------------------------ Constructor --------------------------------------- */

		constructor : function() {
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 */
		_feedback : function (event) {
		}
	});
/**
* @class UIElements.InputField.TimeBase
* @extends UIElements._base
* @ignore
* Abstract Class - Do not use it directly
*/
dojo.provide("UIElements.InputField.TimeBase");
dojo.declare("UIElements.InputField.TimeBase",
	[UIElements._base, dijit._Templated], {

		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Hour that will be displayed in this element ( always in 24 hour format )
		 * @property
		 * @type Number
		 */
		intHour			: 12,

		/**
		 * Hour that will be displayed in this element
		 * @property
		 * @type Number
		 */
		intMinute		: 00,

		/**
		 * Display the time in 24 hour format
		 * @property
		 * @type Boolean
		 */
		is24HourFormat	: true,

		/**
		 * Should we show a popup to enable the user to change the time
		 * @property
		 * @type Boolean
		 */
		boolDialog		: true,

		/**
		 * Callback when element is activated
		 * @property
		 * @type Function
		 */
		funcClicked		: null,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @private
		 */
		_objToaster		: null,

		/* ------------------------------------ Constructor --------------------------------------- */

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @private
		 */
		_getDialogValues: function() {
			var objTime = this._objToaster.getContent().getTime();
			this.intHour = objTime[0];
			this.intMinute = objTime[1];
			this._updateInput();
			this._closeDialog();
		},

		/**
		 * @private
		 */
		_closeDialog: function() {
			ICTouchAPI.toasterServices.hideContent(this._objToaster);
			this._objToaster.destroy();
		},

		/**
		 * @private
		 */
		_setIntHourAttr: function(hour) {
			this.intHour = hour;
			this._updateInput();
		},

		/**
		 * @private
		 */
		_setIntMinuteAttr: function(minute) {
			this.intMinute = minute;
			this._updateInput();
		},

		/**
		 * @private
		 */
		_setIs24HourFormatAttr: function(is24) {
			this.is24HourFormat = is24;
			this._updateInput();
		},

		/**
		 * @private
		 */
		_updateInput: function() {
			var date = new Date(1970, 0, 1, this.intHour, this.intMinute);
			var pattern = this.is24HourFormat ? "HH:mm" : "h:mm a";
			this.domInput.value = date.format(pattern,dojo.locale);
		},

		/**
		 * @private
		 */
		_clicked: function() {
			if( this.boolDialog ) {
				this._deployTimeSelect();
			}
			else if( this.funcClicked != null ) {
				this.funcClicked();
			}
		},

		/**
		 * @private
		 */
		_deployTimeSelect: function() {
			var objTimeSelect = new UIElements.ValueSelect.TimeSelect({
				intHour: this.intHour,
				intMinute: this.intMinute
			});
			var objDialogParam = {
				funcOk: dojo.hitch(this, this._getDialogValues),
				funcCancel: dojo.hitch(this, this._closeDialog)
			};
			this._objToaster = ICTouchAPI.toasterServices.showDialBox(objTimeSelect);
		}
	});
/**
* @class UIElements.InputField.Time
* @extends UIElements.InputField.TimeBase
* @ignore
*/
dojo.require("UIElements.InputField.TimeBase");
dojo.provide("UIElements.InputField.Time");
dojo.declare("UIElements.InputField.Time",
	UIElements.InputField.TimeBase, {
		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @private
		 */
		templatePath		: dojo.moduleUrl("UIElements.InputField","templates/Time8082.html"),
		/**
		 * @private
		 */
		cssFile        		: "base.css"

		/* ------------------------------------ Constructor --------------------------------------- */

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */

	});/**
* @class UIElements.InputField.SimpleBase
* @extends UIElements._base
* Abstract Class - Do not use it directly
* 
*/
dojo.provide("UIElements.InputField.SimpleBase");
dojo.declare("UIElements.InputField.SimpleBase",
	[UIElements._base, dijit._Templated], {

		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Text displayed inside the simple Input field
		 * @cfg {String} strValue
		 */
		strValue		: "",

		/**
		 * Callback fired when the user presses on the simple input field
		 * @cfg {Function} funcClicked
		 */
		funcClicked		: null,

		/**
		 * Html dom element
		 * @property
		 * @type Function
                 * @ignore
		 */
		domInput		: null,
	
		/* --------------------------------- Private attributes ----------------------------------- */

		/* ------------------------------------ Constructor --------------------------------------- */

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 */
		_clicked: function() {
			if( this.funcClicked != null ) {
				this.funcClicked(this.strValue);
			}
		},
	
		/**
		 * @ignore
		 */
		_setStrValueAttr: function(value) {
			this.strValue = value;
			this.domInput.value = value;
		}
	});
/**
* @class UIElements.InputField.Simple
* @extends UIElements.InputField.SimpleBase
*
* This class is the graphical representation of a simple input field.
*
* The simple input field only fires a callback when the user presses on it and is generally used to display a list of choices rendered in a view (in full or 2/3 mode).
*
* The simple input field can be used in a {@link UIElements.PresentationList.PresentationListControl} or directly in a HTML DOM node.
*
* Here is an example of creating a simple input field
*
*     simpleInput : null,
*
*     postCreate : function() {
*
*       // Define the callback
*       var func = dojo.hitch(this, this.OnSimpleInputPress);
*
*       // Create the simple input field
*       this.simpleInput = new UIElements.InputField.Simple({
*           funcClicked: func,
*           strValue: "Choose a city..."
*       }, this.domInputSimple);
*
*     },
*
*     OnSimpleInputPress : function() {
*       // Do something
*       // ...
*     },
*
*/
dojo.require("UIElements.InputField.SimpleBase");
dojo.provide("UIElements.InputField.Simple");
dojo.declare("UIElements.InputField.Simple",
	UIElements.InputField.SimpleBase, {
		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		templatePath		: dojo.moduleUrl("UIElements.InputField","templates/Simple8082.html"),
		/**
		 * @ignore
		 */
		cssFile        		: "base.css"

		/* ------------------------------------ Constructor --------------------------------------- */

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */
	});/**
 * @class UIElements.Keyboard.KeyboardControlBase
 * @extends UIElements._base
 * @ignore
 * Abstract Class<br>
 * Deploys or not a keyboard according to target's version
 */
dojo.provide("UIElements.Keyboard.KeyboardControlBase");
dojo.declare("UIElements.Keyboard.KeyboardControlBase",
	[UIElements._base, dojox.dtl._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Contains the keys of the keyboard
		 * @property
		 * @type Array
		 */
		arrKeys         : [ ],


        arrEmoIcon:null,

		/**
		 * Contains the dom element keys of the keyboard
		 * @property
		 * @type Array
		 */
		domArrKeys      : [ ],
		/**
		 * rows of buttons
		 * @property
		 * @type Array
		 */

		domIMEOtherButtons : [ ],
		/**
		 * i18n path for translation
		 * @property
		 * @type Array
		 */
		i18n : "UIElements.Keyboard",
		/**
		 * InputField of the keyboard
		 * @property
		 * @type UIElements.InputField.InputField
		 */
		objInputField	: null,
		/**
		 * OK button
		 * @property
		 * @type UIElements.Keyboard.Key
		 */
		objOkButton		: null,
		/**
		 * Exit button
		 * @property
		 * @type UIElements.Keyboard.Key
		 */
		objExitButton   : null,

		/**
		 * defines the type of the keyboard to display
		 * values : num, alphaNum, filter, filterNDial, emoticons, password
		 * @property
		 * @type String
		 */
		strType			: "",

		/**
		 * defines the title  of the input field to display
		 * @property
		 * @type String
		 */
		strInputTitle			: null,

		/**
		 * specifies if caller is multiLines (so will be the keyboard's inputfield)
		 * @property
		 * @type Boolean
		 */
		bMultiLines		: false,
		/**
		 * Keyboard 'language' (azerty / qwerty / qwertz)
		 * @property
		 * @type String
		 */
		strLang			: "",


		/**
		 * If true, in password mode, hide stars if password is empty
		 * @property
		 * @type Boolean
		 */
		hideStarsIfPasswordEmpty : false,

		/**
		 * Allow the keyboard services to display or remove some buttons. The goal is that we only have to handle 2 types:
		 * - numeric keyboard
		 * - alphaNumeric keyboard
		 * @property
		 * @type Object
		 * Example :
		   this.keyboardFlags = {
		 	"dialByName": false,
		 	"gotoContact": false,
		 	"cancel": true,
		 	"ok": true,
		 	"backspace": true,
		 	"password": false,
		 	"emoticon": false,
		 	"filter": false,
		 	"byDialpad": false,
		 	"multiline": false,
		 	"gotoCommunication": false,
		 	"gotoProgkeys": false,
			"gotoDirectory": false,
			"dialSpace": false
			"imeIndicator": false
		  };
		 */
		keyboardFlags	: null,
		/**
		 * keymap for each Keyboard 'language' (azerty / qwerty / qwertz)
		 * @property
		 * @type Array
		 */
		languages : {},
		/**
		 * Button OK label
		 * @property
		 * @type String
		 */
		strOkLabel		: null,
		/**
		 * Button Exit label
		 * @property
		 * @type String
		 */
		strExitLabel	: null,
		/**
		 * Button OK icon
		 * @property
		 * @type String
		 */
		strOkIcon		: null,
		/**
		 * Button Exit icon
		 * @property
		 * @type String
		 */
		strExitIcon		: null,
		/*
		 * initial mode for alphanumeric keyboard (ICTouchAPI.KeyboardAlphanumMode enumeration)
		 * @property
		 * @type String
		 */
		strDefaultMode : null,

		/* --------------------------------- Private attributes ----------------------------------- */

        /**
         * @Private
         */
		objIMEInputField: "",
        /**
         * @Private
         * to contorl div IME load or not, can not be defined as 0.
         */
        boolIMEMode     :1,
        /**
         * @Private
         */
		arrLines        : [ ],
        /**
         * @Private
         */
		arrCandidates	: [ ],
        /**
         * @Private
         * the candidates start page number
         */
        candidatesPage: 0,
        /**
         * @Private
         * the candidates in one page
         */
        candidatesPageSize: 10,
        /**
         * @Private
         */
        imeIndicatorLabel:  "",
		/**
		 * @private
		 */
		attributeMap: {
			strInputTitle: {
				node: "domInputTitle",
				type: "innerHTML"
			}
		},

		/**
		 * @private
		 */
		numLockKeyboard : 0,
		/**
		 * @private
		 * This are the default value when no flags are given, user flags will override them
		 */
		_defaultKeyboardFlags	: {
			"dialByName": false,
			"gotoContact": false,
			"cancel": true,
			"ok": true,
			"backspace": true,
			"password": false,
			"emoticon": false,
			"filter": false,
			"byDialpad": false,
			"multiline": false,
			"gotoCommunication": false,
			"gotoProgkeys": false,
			"international": false,
			"gotoDirectory": false,
			"latest":false,
			"dialSpace": false,
			"hotelMode": false,
			"customKeysUpdated" : false,
            "actionButtonInDialpad": 0,
            "imeIndicator": true
		},

		/**
		 * @private
		 * Associative array defining internal flags :
		 * (bool) _flags["star"] if the star key is on and active
         * (bool) _flags["emoticon"] if the emoticon is on and active
         * (bool) _flags["emoticonLock"] if the emoticon is on and locked
		 * (bool) _flags["caps"] if the caps key is on and active
		 * (bool) _flags["capsLock"] if the caps key is on, active and locked.
		 * (bool) _flags["starKey"] is the key defining which extended keyboard is displayed.
		 * (bool) _flags["numLock"] if the alphaNum's numpad is active and locked.
		 * (bool) _flags["ime"] if the ime is required
		 */
		_flags			: {},
		/**
		 * @private
		 */
		_strClassToAdd  : null,
		/**
		 * @private
		 * International button
		 */
		_objInternational : null,

		/**
		 * @private
		 * Space button
		 */
		_objSpace : null,
        /**
         * @private
         */
        _isPasswordKeyBoard: false,
		/* ------------------------------------ Constructor --------------------------------------- */

		constructor : function(){
			// Init all attributes so they won't be shared between instances.
			this.arrKeys		= [ ];
			this.domArrKeys     = [ ];
			this.domIMEOtherButtons = [ ];
			this.arrLines		= [ ];
            //add candidate buttons.
            for (var i = 0; i < this.candidatesPageSize; ++i) {
                this.arrCandidates.push("&nbsp;");
            }
			this.strLang= "azerty_french";
			this._setDefaultModeFlags();
			ICTouchAPI.settingServices.subscribeToSetting(this, "Defaultkeyboard", this._gotDefaultKeyboard);
			this.keyboardFlags = {};
		},

		postMixInProperties : function() {
			// Initialise Ok and Exit button label and icon
			this._setOkExitLabels(this.strOkLabel, this.strExitLabel);
			this._setOkExitIcons(this.strOkIcon, this.strExitIcon)
			this.refreshFlags();
			// Generate keyboard
			this._loadKeyMaps();
			if( this._defaultKeyboardFlags.emoticon !== this.keyboardFlags.emoticon ) {
					this._loadEmoMap("emoticon");
			}
			this.arrKeys = this._getKeyboardSkeleton(this.strType);
			var lineLength = this.strType == "num" ? 5 : 13;
			var remainingWidth = lineLength; // How many space is still left on the current row ( arrLine )
			var arrLine = []; // This contain the current line of widget for the template
			var arrWidthToSkip = []; // This contain how many cell should be ignored due to rowspan
			var i, j;

			/*
			  In the template we need rows of buttons ( arrLines ) but in javascript we only have a big key array ( arrKeys ).
			  So for each row we must compute how many buttons fits in taking into account colspan and rowspan.
			*/
			for (i=0; i<this.arrKeys.length; i++) {
				var currentKey = this.arrKeys[i];

				/*
				  If the last button did fill the row we need to create the next row
				  Using "while" because we could have a line filled only with rowspan
				  for exemple when every button got a 2 height size
				*/
				while (remainingWidth <= 0) {
					arrWidthToSkip.shift();
					remainingWidth = lineLength;
					// Substracting the space taken by rowspan from the last row
					if( typeof arrWidthToSkip[0] === "number" ) {
						remainingWidth -= arrWidthToSkip[0];
					}
					this.arrLines.push(arrLine);
					arrLine = [];
				}

				// Size of the button
				currentKey.intInputWidth = (currentKey.intInputWidth || 1);
				currentKey.intInputHeight = (currentKey.intInputHeight || 1);

				// Substract remaining space
				remainingWidth -= currentKey.intInputWidth;
				arrLine.push(currentKey);

				// When the button got height, increment the array for the next rows
				for (j=1; j<currentKey.intInputHeight; ++j) {
					arrWidthToSkip[j] = arrWidthToSkip[j] || 0;
					arrWidthToSkip[j] += currentKey.intInputWidth;
				}

				// Change the attributes of the currentKey
				currentKey.objElement.attr("intWidth", currentKey.intInputWidth);
				currentKey.objElement.attr("intHeight", currentKey.intInputHeight);
			}
			this.arrLines.push(arrLine);
		},

		postCreate : function () {
			var i;
			dojo.addClass(this.domNode, this._strClassToAdd);
			for (i=0; i<this.arrKeys.length; ++i){
				this.arrKeys[i].objElement.placeAt(this.domArrKeys[i]);
			}
			if(this.domIMEOther) {
				this.scroll = FXScroll.init({
					"handler" : this.domIMEOther,
					"easing" : true
				});
			}

			// getSetting moved from constructor to postCreate because of a bug when preloading qwerty keyboard linked
			// to asynchronous call of the getSetting callback
			ICTouchAPI.settingServices.getSetting("Defaultkeyboard", this, this._gotDefaultKeyboard);
			this.setKeyboardFlags(this.keyboardFlags, true);
			this.setMode(this.strDefaultMode)
			this.placeTitle();
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * Get application name
		 */
		getApplicationName : function() {
			return "KeyboardControl";
		},

		setTitle : function(strTitle){
			if (this.strInputTitle != strTitle) {
                            this.strInputTitle = strTitle;
                            this.placeTitle();
                        }
		},

		placeTitle : function(){
			if (this.strInputTitle && this.strInputTitle.isI18Ned){
				this.domInputTitle.innerHTML = this.strInputTitle.getTranslation();
			}
			else{
				this.domInputTitle.innerHTML = this.strInputTitle;
			}
		},

		/**
		 * Set the keyboard flags.
		 * This will add or remove button according to the flags. With some flags, put the order position instead of true
		 * If the flag isn't set it will revert back to its default value
		 * @param {Object} flags Associative array with the new flags
		 * <pre><code>
		   setKeyboardFlags({
		 		"dialByName": false,		//button order, num keyboard only
		 		"gotoContact": false,		//button order, num keyboard only
		 		"cancel": true,
		 		"ok": true,
		 		"backspace": true,
		 		"password": false,
		 		"emoticon": false,
		 		"filter": false,
		 		"byDialpad": false,
		 		"multiline": false,
				"gotoCommunication": false,	//button order, num keyboard only
				"gotoProgkeys": false,
				"international": false,
				"dialSpace": false,
				"hotelMode": false,
				"customKeysUpdated" : false,
				"actionButtonInDialpad" : 0
		  },true);
		  </pre></code>
		 * @param {Boolean} _first
		 */
		setKeyboardFlags: function(flags, _first) {
			// _first is only set from inside KeyboardControl and should only called from postCreate or any of its subroutines
			// Clone the defaults flags so we don't change it
			var newFlags = dojo.clone(this._defaultKeyboardFlags);
			dojo.mixin(newFlags, flags);
			if( this.strType == "num" ) {
				// If any flag has changed, we must reorder everything
				if (_first ||
					newFlags.gotoCommunication !== this.keyboardFlags.gotoCommunication ||
					newFlags.gotoProgkeys !== this.keyboardFlags.gotoProgkeys ||
 					newFlags.gotoContact !== this.keyboardFlags.gotoContact ||
					newFlags.dialByName !== this.keyboardFlags.dialByName ||
					newFlags.gotoDirectory !== this.keyboardFlags.gotoDirectory ||
					newFlags.latest !== this.keyboardFlags.latest ||
					newFlags.hotelMode
					) {
					if(newFlags.hotelMode){// if we are in hotel mode, create custom keys
						var customKeys = ICTouchAPI.keyboardServices.getDialpadKeys();
						var intIndex = 0;
						var intKeyboardIndex = 0;
						if(newFlags.customKeysUpdated || (newFlags.customKeys && !this.keyboardFlags.customKeys)){
							for(intIndex = 0; intIndex < 8; intIndex++){
								var customKey = ICTouchAPI.keyboardServices.getCustomKey(customKeys[intIndex]);
								intKeyboardIndex=intIndex<4?5*intIndex+1:(intIndex-3)*5;
								if (customKey && customKeys[intIndex] != null) {
									this._replaceButton(intKeyboardIndex, true, null, customKey);
								}
								else {
									// if the button has been removed, replace it by an empty button
									this._replaceButton(intKeyboardIndex, false, null, null);
								}
							}
							switch(newFlags.actionButtonInDialpad){
								case 0:
									this._replaceButton(23, false, null, null);
									break;
								case 1:
									this._replaceButton(23, true, this.createDialByName);
									break;
								case 2:
									this._replaceButton(23, true, this.createGotoDirectory);
									break;
								case 3:
									if(customKeys[8]){
										this._replaceButton(23, true, null, ICTouchAPI.keyboardServices.getCustomKey(customKeys[8]));
									}
									else{
										this._replaceButton(23, false, null, null);
									}
									break;
							}
						}
						else if(!newFlags.customKeys && this.keyboardFlags.customKeys){
							for(intIndex = 0; intIndex < 8; intIndex++){
								intKeyboardIndex=intIndex<4?5*intIndex+1:(intIndex-3)*5;
								this._replaceButton(intKeyboardIndex, false, null, null);
							}
						}
						
					}
					else{
					var itemToTest = [];
					if(newFlags.gotoCommunication) itemToTest[newFlags.gotoCommunication] = true;
					if(newFlags.gotoProgkeys) itemToTest[newFlags.gotoProgkeys] = true;
 					if(newFlags.gotoContact) itemToTest[newFlags.gotoContact] = true;
					if(newFlags.dialByName) itemToTest[newFlags.dialByName] = true;
					if(newFlags.gotoDirectory) itemToTest[newFlags.gotoDirectory] = true;
					if(newFlags.latest) itemToTest[newFlags.latest] = true;
					var firstPosition = 23;
					var position = firstPosition;
					for(var i = itemToTest.length-1; i >= 0; i--) {
						if(itemToTest[i]!=null) {
							if(newFlags.latest===i) {
								this._replaceButton(position, true, this.createShowLatestNumber);
								position = this._getPreviousPosition(newFlags.latest, position);
							}
							if(newFlags.gotoCommunication===i) {
								this._replaceButton(position, true, this.createGotoCommunication);
								position = this._getPreviousPosition(newFlags.gotoCommunication, position);
							}
							if(newFlags.gotoDirectory===i) {
								this._replaceButton(position, true, this.createGotoDirectory);
								position = this._getPreviousPosition(newFlags.gotoDirectory, position);
							}
							if(newFlags.dialByName===i) {
								this._replaceButton(position, true, this.createDialByName);
								position = this._getPreviousPosition(newFlags.dialByName, position);
							}
							if(newFlags.gotoContact===i) {
								this._replaceButton(position, true, this.createGotoContact);
								position = this._getPreviousPosition(newFlags.gotoContact, position);
							}
							if(newFlags.gotoProgkeys===i) {
								this._replaceButton(position, true, this.createGotoProgkeys);
								position = this._getPreviousPosition(newFlags.gotoProgkeys, position);
							}
						}
					}
					}


					// Fill the rest with empty buttons
					if(!newFlags.hotelMode){
					while (position >= 5) {
						this._replaceButton(position, false, null);
						position = this._getPreviousPosition(true, position);
					}
				}
				}

				// Backspace button is integrated into InputField
				if( newFlags.backspace !== this.keyboardFlags.backspace || _first ) {
					var callback = newFlags.backspace ? dojo.hitch(this, this._backspaceCallback) : null;
					this.getInputField().setExtraButton(callback, true, _("delete", this.i18n), "delete");
				}
				// Enable the '+' key when international mode is enabled
				if( newFlags.international !== this.keyboardFlags.international || _first ) {
					// this._internationalKeyPressed is a special case, it don't need context
					var callback = newFlags.international ? this._internationalKeyPressed : null;
					this._objInternational.objElement.setLongCallback(callback);
				}
				// Enable the ' ' key on dialpad 1
				if( newFlags.dialSpace !== this.keyboardFlags.dialSpace || _first ) {
					var callback = newFlags.dialSpace ? this._spaceKeyPressed : null;
					this._objSpace.objElement.setLongCallback(callback);
				}
			} else if( this.strType == "alphaNum" ) {
				// Flags only meaningful in alphanum keyboard
				// crms00334755: dial-pad button not displayed anymore in keyboards / let in comment just in case of new ergo requirements...
				/*else if( newFlags.byDialpad !== this.keyboardFlags.byDialpad || _first ) {
					this._replaceButton(4, newFlags.byDialpad, this.createByDialpad);
				}*/
				// Multiline support
				if( newFlags.multiline !== this.keyboardFlags.multiline || _first ) {
					this._replaceButton(30, newFlags.multiline, this.createCarriageReturn);
					// Set multiline InputField
					this.getInputField().setMultiline(newFlags.multiline);
				}
				if( newFlags.backspace !== this.keyboardFlags.backspace || _first ) {
					this._replaceButton(17, newFlags.backspace, this.createBackspace);
				}
			}
			if( newFlags.cancel !== this.keyboardFlags.cancel || _first ) {
				var pos = this.strType === "num" ? 21 : 31;
				this._replaceButton(pos, newFlags.cancel, this.createCancel);
			}
			if( newFlags.ok !== this.keyboardFlags.ok || _first ) {
				var pos = this.strType === "num" ? 22 : 41;
				this._replaceButton(pos, newFlags.ok, this.createOk);
			}
			if( newFlags.password !== this.keyboardFlags.password ) {
				this.getInputField().setPassword(newFlags.password);
			}
			this.keyboardFlags = newFlags;
		},

		/**
		 * Set the alphaNumeric keyboard mode.
		 * This will change alphaNumeric keyboard into lowercase, uppercase and numeric mode.		 *
		 * @param {String} strMode Name of the mode (ICTouchAPI.KeyboardAlphanumMode enumeration) (default: NORMAL)
		 */
		setMode : function(strMode) {
			if(this.strType !== "alphaNum") {
				return;
			}
			else if(strMode === ICTouchAPI.KeyboardAlphanumMode.NORMAL) {
				this._setDefaultModeFlags();
			}
			else if(strMode === ICTouchAPI.KeyboardAlphanumMode.CAPS) {
				this._setDefaultModeFlags();
				this._flags["caps"] = 1;
			}
			else if(strMode === ICTouchAPI.KeyboardAlphanumMode.CAPS_LOCK) {
				this._setDefaultModeFlags();
				this._flags["caps"] = 1;
				this._flags["capsLock"] = 1;
			}
			else if(strMode === ICTouchAPI.KeyboardAlphanumMode.NUMERIC) {
				this._setDefaultModeFlags();
				this._flags["numLock"] = 1;
			}
			else {
				this._setDefaultModeFlags();
			}
			this._swapKeyMap();
			this._updateButtonsIcons();
		},

		/**
		 * Get the actual InputField Element
		 * This should only be called by the keyboardServices
		 * @return {UIElements.InputField.InputFieldControl} the InputField of this keyboard
		 */
		getInputField: function() {
			return this.objInputField.objElement;
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		 * Refresh flag boolIMEMode
		 */
		refreshFlags : function() {
			this._flags["ime"] = this.boolIMEMode;
		},

		/**
		 * this is called when the keyboard has been inserted into toaster
		 * So we steal focus and place it on our InputField. A problem arise when preloading keyboards
		 * so we scroll back the body.
		 */
		placeAt : function () {
			this.inherited(arguments);
			this.focus();
			document.body.scrollTop = 0;
		},

		/**
		 * Replace the current icon of OK button and EXIT button.
		 * @param {String} strOkIcon button OK icon
		 * @param {String} strExitIcon button Exit icon
		 */
		updateOkExitButtonIcons : function(strOkIcon, strExitIcon) {
			this._setOkExitIcons(strOkIcon, strExitIcon);
			if(this.objOkButton && this.objOkButton.objElement && this.keyboardFlags.ok ){
				this.objOkButton.objElement.attr("strButtonIcon", this.strOkIcon);
			}
			if(this.objExitButton && this.objExitButton.objElement && this.keyboardFlags.cancel){
				this.objExitButton.objElement.attr("strButtonIcon", this.strExitIcon);
			}
		},

		/**
		 * When this keyboard takes focus that means that the user can type with physical keyboard into InputField
		 */
		focus: function() {
			this.getInputField().getInputFieldObject().focus();
		},

		/**
		 * Replace the current label of OK button and EXIT button.
		 * @param {String} strOkLabel button OK label
		 * @param {String} strExitLabel button Exit label
		 */
		updateOkExitButtonLabels : function(strOkLabel, strExitLabel) {
			this._setOkExitLabels(strOkLabel, strExitLabel);
			if(this.objOkButton && this.objOkButton.objElement && this.strOkLabel && this.keyboardFlags.ok ){
				this.objOkButton.objElement.replaceText(this.strOkLabel);
			}
			if(this.objExitButton && this.objExitButton.objElement && this.strExitLabel && this.keyboardFlags.cancel ){
				this.objExitButton.objElement.replaceText(this.strExitLabel);
			}
		},

		/**
		 * Create keyboard button "international"
		 */
		createInternational: function() {
			var context = this;
			var key = {
				intInputWidth : 1,
				intInputHeight: 1,
				objElement :   new UIElements.Keyboard.Key({
					strButtonType : 'letter',
					strButtonText : '0',
					strSubText    : '+',
					strButtonName : "keyboard_International",
					intInputWidth : 3,
					intInputHeight: 1,
					funcCallback: function (value) {
						context.keyboardLetterCallback(value);
					},
					funcCallbackLong: null, // Will be defined by the keyboardFlags
					funcCallbackRelease : function(value) {
						ICTouchAPI.keyboardServices.keyReleased(value);
					}
				})
			}
			return key;
		},

		_internationalKeyPressed: function() {
						ICTouchAPI.keyboardServices.keyPressed('+');
					},

		/**
		 * Create keypad button "space"
		 */
		createSpace: function() {
			var context = this;
			var key = {
				intInputWidth : 1,
				intInputHeight: 1,
				objElement :   new UIElements.Keyboard.Key({
					strButtonType : 'letter',
					strButtonText : '1',
					strSubText    : '&#9251;',
					strButtonName : "dialPad1",
					intInputWidth : 3,
					intInputHeight: 1,
					funcCallback: function (value) {
						context.keyboardLetterCallback(value);
					},
					funcCallbackLong: null, // Will be defined by the keyboardFlags
					funcCallbackRelease : function(value) {
						ICTouchAPI.keyboardServices.keyReleased(value);
					}
				})
			}
			return key;
		},
		_spaceKeyPressed: function() {
						ICTouchAPI.keyboardServices.keyPressed(' ');
					},

		/**
		 * Create keyboard button "Dial By Name"
		 */
		createDialByName: function() {
			return this.createButton({
				intInputWidth : 1,
				intInputHeight: 1,
				strButtonText : _("Search", "ICTouchAPI"),
				strButtonIcon : "generic-search-by-name",
				strButtonName : "keyboard_search",
				boolAttachParent : true,
				callbackClick: function () {
					ICTouchAPI.keyboardServices.doCustomCallback("keyboard");
				}
			});
		},

		/**
		 * Create keyboard button "Go to contact"
		 */
		createGotoContact: function() {
			return this.createButton({
				intInputWidth : 1,
				intInputHeight: 1,
				strButtonText : _("contacts", this.i18n),
				strButtonIcon : "contacts-application",
				strButtonName : "keyboard_contact",
				boolAttachParent : true,
				callbackClick: function () {
					ICTouchAPI.keyboardServices.doCustomCallback("goto-contact");
				}
			});
		},

		/**
		 * Create keyboard button "Go to communication"
		 */
		createGotoCommunication: function() {
			return this.createButton({
				intInputWidth : 1,
				intInputHeight: 1,
				strButtonText : _("communication", this.i18n),
				strButtonIcon : "communication-application",
				strButtonName : "keyboard_communication",
				boolAttachParent : true,
				callbackClick: function () {
					ICTouchAPI.keyboardServices.doCustomCallback("goto-communication");
				}
			});
		},

		/**
		 * Create keyboard button "Go to communication"
		 */
		createGotoProgkeys: function() {
			return this.createButton({
				intInputWidth : 1,
				intInputHeight: 1,
				strButtonText : _("favorites", this.i18n),
				strButtonIcon : "contacts-favorites-application",
				strButtonName : "keyboard_progkeys",
				boolAttachParent : true,
				callbackClick: function () {
					ICTouchAPI.keyboardServices.doCustomCallback("goto-progkeys");
				}
			});
		},

		/**
		 * Create keyboard button "Go to Directory"
		 */
		createGotoDirectory: function() {
			return this.createButton({
				intInputWidth : 1,
				intInputHeight: 1,
				strButtonText : _("directory", this.i18n),
				strButtonIcon : "sdk-hotel-directory",
				strButtonName : "keyboard_directory",
				boolAttachParent : true,
				callbackClick: function () {
					ICTouchAPI.keyboardServices.doCustomCallback("goto-contact");
				}
			});
		},


		/**
		 * Create keyboard button "Show latest numbers"
		 */
		createShowLatestNumber: function() {
			return this.createButton({
				intInputWidth : 1,
				intInputHeight: 1,
				strButtonText : _("Latest", this.i18n),
				strButtonIcon : "userservices-latest-numbers",
				strButtonName : "keyboard_latest",
				boolAttachParent : true,
				callbackClick: function () {
					ICTouchAPI.keyboardServices.doCustomCallback("show-latest");
				}
			});
		},

		// Create keyboard button

		/**
		 * Create keyboard button "Filter"
		 */
		createFilter: function() {
			return this.createButton({
				strButtonText: _("filter", this.i18n),
				strButtonIcon: "generic-filter",
				strButtonName: "keyboard_FILTER",
				callback: function () {
					ICTouchAPI.keyboardServices.doCustomCallback("filter");
				}
			});
		},

		/**
		 * Create keyboard button "Dialpad"
		 * crms00334755: dial-pad button not displayed anymore in keyboards / let in comment just in case of new ergo requirements...
		 */
		/*createByDialpad: function() {
			return this.createButton({
				strButtonIcon: "dialpad",
				strButtonName: "keyboard_DIAL",
				callback: function () {
					ICTouchAPI.keyboardServices.doCustomCallback("dial-pad");
				}
			});
		},*/


		/**
		 * Create an inputField UIElement with specified args.
		 * There must be only one inputField, its reference will be stored in this.objInputField
		 * @param {Array} args array with desired arguments to pass to the UIElement.
		 *	intInputWidth {Int} the width of the key
		 *	strInputText {String} the default text to display in the inputField
		 *	strInputIcon {String} name of the icon to place before the inputField
		 * @return {UIElement.InputField} this.objInputField
		 * <pre><code>
		 		this.createInputField({
		 			intInputWidth	: 9,
		 			strInputText	: this.strInputText,
		 			strInputIcon	: 'notif-16px-denied'
		 		}),
		  </pre></code>
		 */
		createInputField: function  (args) {
			this.objInputField =  {
				intInputWidth		: args.intInputWidth  || null,
				intInputHeight		: args.intInputHeight || null,
				objElement			: new UIElements.InputField.InputFieldControl( {
					strInputText	: args.strInputText,
					strInputContent : args.strInputText,
					strInputIcon	: args.strInputIcon,
					boolPassword	: args.boolPassword,
					hideStarsIfPasswordEmpty : args.hideStarsIfPasswordEmpty
				})
			};
			return this.objInputField;
		},

		/**
		 * Create a backspace key from SimpleButton UIElement.
		 * @param {Array} args array with desired arguments to pass to the UIElement.
		 *	intInputWidth {Int} the width of the key
		 *	strButtonType {String} By default, a backspace key is of TextButton type
		 *	strButtonText {String} Name of the backspace key
		 * @return the object
		 * <pre><code>
		 		this.createBackspace({})
		  </pre></code>
		 */
		createBackspace : function (args) {
			var context = this;
			return 		{
				intInputWidth :  args.intInputWidth || 1,
				intInputHeight :  args.intInputHeight || 1,
				objElement :   new UIElements.Keyboard.Key({
					strButtonIcon : "delete",
					strButtonName : "keyboard_DEL",
					strButtonText : null,
					boolAutorepeat: true,
					// Both needed because the size of the icon ( 32 or 64 ) is computed on creation
					intWidth     : args.intInputWidth || 1,
					intHeight    : args.intInputHeight || 1,

					funcCallback: dojo.hitch(context, context._backspaceCallback),
					funcCallbackRelease : function() {
						ICTouchAPI.keyboardServices.keyReleased('backspace');
					}
				})
			}
		},

        /*
         * Create IME indicator key
         * @param {Array} args array with desired arguments to pass to the UIElement.
         *    intInputWidth {Int} the width of the key
         *    strButtonType {String} By default, a backspace key is of TextButton type
         *    strButtonText {String} Name of the backspace key
         * @return the object
         * */
        createIMEIndicator: function (args) {
            var context = this;
            this.imeIndicatorLabel = this.imeIndicatorLabel || "EN";
            this.objIMEIndicatorButton = {
                objElement: new UIElements.Keyboard.Key({
//                    strButtonIcon: "delete",
                    strButtonName: "keyboard_IME_Indicator",
                    strButtonText: this.imeIndicatorLabel,
                    boolAutorepeat: false,
                    // Both needed because the size of the icon ( 32 or 64 ) is computed on creation
                    intWidth: args.intInputWidth || 1,
                    intHeight: args.intInputHeight || 1,

                    funcCallback: dojo.hitch(context, context._imeIndicatorCallback),
                    funcCallbackRelease: function () {
                        ICTouchAPI.keyboardServices.keyReleased('imeIndicator');
                    }
                })
            };
            return this.objIMEIndicatorButton;
        },
        /**
         * create IME ok button
         * @returns {boolean}
         * @constructor
         */
        IMEPressOK: function () {
            // if OK pressed in IME mode
            var isIMELang = ICTouchAPI.keyboardServices.isAPACIMELang(this.strLang);
            if (!isIMELang || 'apac_en' == this.strLang || !this.domIMEInput.value)
                return false;
            for (var index in this.domIMEInput.value) {
                this.getInputField().addNewChar(this.domIMEInput.value[index]);
            }
            this.clearIMEInput();
            ICTouchAPI.IMEServices.resetCharSequence();
            this.recreateCandidateList("");
            return true;
        },
		/**
		 * Create keyboard button "OK"
		 * <pre><code>
		  createOK({
		 		intInputHeight: "50",
		 		intInputWidth : "100",
		 		strButtonIcon: "generic-ok",
		 		strButtonText: "OK",
		 		strCustomClass: "ok",
		 		strButtonName: 'keyboard_OK',
		 		callback: function () {
		 			ICTouchAPI.keyboardServices.closeKeyboard(ICTouchAPI.keyboardServices.CLOSE_ACTION_OK);
		 		}
		 	}
		  );
		  </pre></code>
		 */
		createOk : function(args) {
            var that=this;
			this.objOkButton = this.createButton({
				intInputHeight: args.intInputHeight || 1,
				intInputWidth : args.intInputWidth || 1,
				strButtonIcon: "generic-ok",
				strButtonText: this.strOkLabel,
				strCustomClass: "ok",
				boolAttachParent: true,
				strButtonName: 'keyboard_OK',
				callbackClick: function () {
                    //to push the content from ime input zone to ime input field.
                    if (that.IMEPressOK())
                        return;
                    if(that._flags["emoticon"]||that._flags["emoticonLock"]){
                        that._flags["emoticon"]=false;
                        that._flags["emoticonLock"]=false;
                    }
					ICTouchAPI.keyboardServices.closeKeyboard(ICTouchAPI.keyboardServices.CLOSE_ACTION_OK);
				}
			});
			return this.objOkButton;
		},

		/**
		 * Create keyboard button "Cancel"
		 * <pre><code>
		  createCancel({
		 		intInputHeight: "50",
		 		intInputWidth : "100",
		 		strButtonIcon: "generic-cancel",
		 		strButtonText: "Cancel",
		 		strCustomClass: "Cancel",
		 		strButtonName: 'keyboard_Cancel',
		 		callback: function () {
		 			ICTouchAPI.keyboardServices.closeKeyboard(ICTouchAPI.keyboardServices.CLOSE_ACTION_CANCEL);
		 		}
		 	}
		  );
		  </pre></code>
		 */
		createCancel : function(args) {
            var that=this;
			this.objExitButton = this.createButton({
				intInputHeight: args.intInputHeight || 1,
				intInputWidth : args.intInputWidth || 1,
				strButtonIcon: "generic-cancel",
				strButtonText : this.strExitLabel,
				strCustomClass : "cancel",
				strButtonName : "keyboard_EXIT",
				// Don't make whole Cancel button sensitive in numeric mode
				boolAttachParent: true,
				callbackClick: function () {
                     if(that._flags["emoticon"]||that._flags["emoticonLock"]){
                        that._flags["emoticon"]=false;
                        that._flags["emoticonLock"]=false;
                    }
					ICTouchAPI.keyboardServices.closeKeyboard(ICTouchAPI.keyboardServices.CLOSE_ACTION_MANUAL_EXIT);
				}
			});
			return this.objExitButton;
		},

		/**
		 * Create a Key for the keyboard UIElement.
		 * @param {Array} args array with desired arguments to pass to the UIElement.
		 *	intInputWidth {Int} the width of the key
		 *	strButtonText {String} the button's name
		 *	callback {Func} the callback function
		 * @return the object
		 * <pre><code>
		  	this.createTextButton({
		 		strButtonText: 'OK',
		 		callback: function () {
		 			ICTouchAPI.keyboardServices.closeKeyboard(true);
		 		}
		 	}),
		 	</pre></code>
		 */
		createButton : function (args) {
			var width  = args.intInputWidth  || 1;
			var height = args.intInputHeight || 1;
			if (typeof args.boolAttachParent === "undefined") {
				args.boolAttachParent = true;
			}
			var objButton = {
				intInputWidth  : width,
				intInputHeight : height,
				objElement :   new UIElements.Keyboard.Key({
					strButtonName	: args.strButtonName,
					strButtonText	: args.strButtonText	|| null,
					strButtonIcon	: args.strButtonIcon	|| null,
					strCustomClass	: args.strCustomClass	|| null,
					strNotification	: args.strNotification	|| null,
					boolAttachParent: args.boolAttachParent,
					intWidth		: width,
					intHeight		: height,
					funcCallback	: args.callback || null,
					funcCallbackClick : args.callbackClick || null
				})
			};
			args = null;
			return objButton;
		},

		/**
		 * Create a SpaceBar key from SimpleButton UIElement.
		 * @param {Array} args array with desired arguments to pass to the UIElement.
		 *	intInputWidth {Int} the width of the key
		 *	strButtonType {String} By default, a SpaceBar key is of KeyboardLetter type
		 *	strButtonText {String} Name of the SpaceBar key
		 * @return the object
		 * <pre><code>
		 		this.createSpaceBar({}),
		  </pre></code>
		 */
		createSpaceBar: function (args) {
			return {
				intInputWidth: args.intInputWidth || 3,
				objElement : new UIElements.Keyboard.Key({
					strButtonType: 'letter',
					strButtonText: args.strButtonText || _('SPACE', this.i18n),
					strButtonName: args.strButtonName,
					funcCallback: function () {
						ICTouchAPI.keyboardServices.keyPressed(" ");
					},
					funcCallbackRelease : function() {
						ICTouchAPI.keyboardServices.keyReleased(" ");
					}
				})
			}
		},

		/**
		 * Create a keyboard letter from SimpleButton UIElement.
		 * @param {String} buttonName the name of the key
		 * @param {String} strLetter the char to display
		 * @param {String} strSubLetter the small text next to the main label
		 * @param {Number} intWidth the width of the key
		 * @param {Number} intHeight the width of the key
		 * @return the object
		 * <pre><code>
		 		// the #2 key, with "abc" as subletter, with a width of 3 and the name of "dialpad_3"
		 		this.createKeyboardLetter("2", "abc", 3, "dialpad_3")
		 		// the "z" letter, width a default width of 1.
		 		this.createKeyboardLetter("z")
		  </pre></code>
		 */
       createKeyboardLetter : function (buttonName, strLetter, strSubLetter, intWidth, intHeight) {
			// Save scope for callback.
			var that = this;
			intWidth  = intWidth || 1;
			intHeight = intHeight || 1;
			var key = {
				intInputWidth : intWidth,
				intInputHeight: intHeight,
				emoticon : false,
				strValue : strLetter,
				objElement :   new UIElements.Keyboard.Key({
					strButtonType : 'letter',
					strButtonText : strLetter,
					strSubText    : strSubLetter,
					strButtonName : buttonName,
					intInputWidth : intWidth,
					intInputHeight: intHeight,
					funcCallback: function (value) {
						if(key.emoticon){
							that.keyboardEmoCallback(key.strValue);
						} else {
							that.keyboardLetterCallback(value);
						}
					},
					funcCallbackRelease : function(value) {
						ICTouchAPI.keyboardServices.keyReleased(value);
					}
				}),
				swapContent : function (strText, strEmo) {
					var boolEmo = strEmo!=null && strEmo!='';
					if(this.emoticon == false){
						if(boolEmo){
							this.objElement.replaceText('');
							this.objElement.setIcon(strEmo)
						}
						else {
							this.objElement.replaceText(strText);
						}
					} else {
						if(boolEmo){
							this.objElement.replaceText('');
							this.objElement.replaceIcon(strEmo);
						} else {
							this.objElement.removeIcon();
							this.objElement.replaceText(strText);
						}
					}
					this.strValue = strText;
					this.emoticon = boolEmo;
				}
			};
			return key;
		},
		/**
		 * get the text corresponding to the Emo
		 */
        keyboardEmoCallback : function(strValue){
			this.keyboardLetterCallback(strValue);
            var field = this.objInputField.objElement.getInputFieldObject();
			var intSelectionStart = field.selectionStart;
            intSelectionStart++;
            field.selectionStart = intSelectionStart;
            if(this._flags["emoticon"]&&!this._flags["emoticonLock"]){
                    this._flags["emoticon"] = 0;
					this._swapKeyMap();
            }
        },

		/**
		 * Could it be a lack of understanding of how javascript builds its objects,
		 * but that function had to be defined outside of the key's scope but inside this
		 * createPrevNextButton so it is accessible from within the objects 'attributes' and methods
		 * during creation.
		 * It needs no parameter and returns true or false wether the icon + callback have to be a carriage return.
		 * @return {bool} if true, display the carriage Return, else does not display it.
		 */
		createCarriageReturn: function () {
			return {
				intInputWidth: 	1,
				intInputHeight: 1,
				objElement: new UIElements.Keyboard.Key({
					strButtonIcon: "return",
					funcCallback: function () {
						ICTouchAPI.keyboardServices.keyPressed("\n");
					}
				})
			};
		},

		/**
		 * Create an empty button with specified width.
		 * @param {Number} intWidth width of the empty button.
		 * @param {Number} intHeight height of the empty button.
		 * @return the object
		 * <pre><code>
		 	// An empty button of width 2
		 	this.createEmptyButton(2);
		  </pre></code>
		 */
		createEmptyButton: function (intWidth, intHeight) {
			return {
				intInputWidth : intWidth  || 1,
				intInputHeight: intHeight || 1,
				objElement : new UIElements.Keyboard.Key({
					strButtonType: "empty"
				})
			}
		},

		/* --------------------------------- Private Methods -------------------------------------- */
        /**
         * @Private
         * only used for Pinyin keyboard. to display the all candidates of chinese words.
         */
        createPagedCandidateList: function () {
            var pageCandidates;
            var startIndex = this.candidatesPage * this.candidatesPageSize;
            var endIndex = (this.candidatesPage + 1) * this.candidatesPageSize;
            if(endIndex > this.arrCandidates.length){
                pageCandidates = this.arrCandidates.slice(startIndex);
            }else{
                pageCandidates = this.arrCandidates.slice(startIndex, endIndex);
            }
            var candidateButtons = dojo.query("li", this.domIMEOtherList);
            //add every candidate for this page
            for (var x = 0; x < candidateButtons.length; x++) {
                if (x >= pageCandidates.length) {
                    candidateButtons[x].style.display = "none";
                    continue;
                }
                candidateButtons[x].innerHTML = pageCandidates[x];
                candidateButtons[x].style.display = "";
            }
            //clear page button
            this.domIMEPageUp.style.color = this.candidatesPage > 0 ? "white" : "grey";
            this.domIMEPageDown.style.color = endIndex < this.arrCandidates.length ? "white" : "grey";
            pageCandidates = null;
        },

		/**
		 * @private
		 */
		recreateCandidateList : function(arrNewCandidates) {
            if (null != arrNewCandidates)
				this.arrCandidates = arrNewCandidates;
            // Display the top IME bar according to if the candidates list is empty
            if (arrNewCandidates || this.domIMEInput.value)
                this.setIMEVisible(true);
            this.candidatesPage = 0;
            // get this.candidatesPage page candidate list
            this.createPagedCandidateList();
		},

		/**
		 * @private
		 */
		clearIMEInput : function() {
            this.domIMEInput.value = "";
            this.setIMEVisible(false);
            ICTouchAPI.IMEServices.resetCharSequence();
		},

		/**
		 * @private
		 */
		deleteIMEChar : function() {
            if (!this.domIMEInput.value)
                return false;
            this.domIMEInput.value = this.domIMEInput.value.substring(0, this.domIMEInput.value.length - 1);
            if (!this.domIMEInput.value)
                this.setIMEVisible(false);
            return true;
        },

        /**
         * set dom IME Area visible
         * @private
         */
        setIMEVisible: function (visible) {
            var styleStr = visible ? "block" : "none";
            dojo.style(this.domIME, "display", styleStr);
        },

        /**
         * @private
         */
        setIMEString: function(str){
            this.domIMEInput.value = str;
            this.setIMEVisible(str);
        },
		/**
		 * @private
		 */
		addIMEChar : function(strChar) {
			var field = this.domIMEInput;
			field.value += strChar;
            this.setIMEVisible(field.value);
        },

        /**
         * @private
         */
        pushInputZoneIntoInputField: function (clearInputZone) {
            var currentValue = this.domIMEInput.value;
            var inputField = this.getInputField();
            for (var index in currentValue) {
                inputField.addNewChar(currentValue[index]);
            }

            if (clearInputZone)
                this.clearIMEInput();
        },

        /**
         * @private
         */
        displayCharInInputField: function (ch, clearInputZone) {
            for (var i = 0; i < ch.length; ++i) {
                this.getInputField().addNewChar(ch[i]);
            }
            if (clearInputZone)
                this.clearIMEInput();
		},

		/**
		 * @private
		 */
		_loadEmoMap: function (keyMap) {
			var that = this;
			// Load key maps from file.
			keyMap = keyMap || this.strLang;
			dojo.xhrGet( {
					url: "library/ICTouchAPI/UIElements/Keyboard/keyMaps/"+keyMap+".json",
					handleAs: "json",
					sync: true,

					load: function(response) {
						that.arrEmoIcon = response.arrEmoIcon;

					},

					error: function(response, ioArgs) {
						console.error(ioArgs);
					}
				});

		},

		/**
		 * @private
		 */
        _populateIndicatorLabelText: function (strLang) {
            var selectedLangLabel = _("indicator_" + strLang, this.i18n).getTranslation();
            var toSwitchLang = 'apac_en' == strLang ? ICTouchAPI.keyboardServices.getLastLang() : 'apac_en';
            var toSwitchLangLabel = _("indicator_" + toSwitchLang, this.i18n).getTranslation();
            return (selectedLangLabel + "/").bold() + toSwitchLangLabel.sub().fontcolor("grey");
        },

        /**
         * @private
         */
        updateIMEIndicator: function () {
//            create ime indicator button
            var isAPACIMELang = ICTouchAPI.keyboardServices.isAPACIMELang(this.strLang);
            if(isAPACIMELang){
                this.imeIndicatorLabel = this._populateIndicatorLabelText(this.strLang);
            }else{
                this.imeIndicatorLabel = "";
            }
            var displayIndicator = isAPACIMELang && !this._isPasswordKeyBoard && this.strType == ICTouchAPI.KeyboardTypes.ALPHANUM;
            this._replaceButton(1, displayIndicator, this.createIMEIndicator);

            },

        /**
         * @private
         */
        setIMEStuff: function (strLang) {
            var isAPACIMELang = ICTouchAPI.keyboardServices.isAPACIMELang(this.strLang);
            this.updateIMEIndicator();
            this.clearIMEInput();
            this.recreateCandidateList("");
            if (this.strType == ICTouchAPI.KeyboardTypes.ALPHANUM && !this._isPasswordKeyBoard) {
                ICTouchAPI.IMEServices.changeLanguage(strLang);
            }else{
                ICTouchAPI.keyboardServices.setIMERequirement(false);
            }
        },

		/**
		 * @private
		 */
		_gotDefaultKeyboard : function(defaultKeyboard) {
            if (defaultKeyboard) {
				this.strLang = defaultKeyboard.allowedValues[defaultKeyboard.jsValue].name;
                this.setIMEStuff(this.strLang);
                }
			else
				console.warn("KeyboardControl: setting Defaultkeyboard not provided.");
			this._loadKeyMaps();
            if(this.strLang == "cyrillic_russian" && this.strType == ICTouchAPI.KeyboardTypes.ALPHANUM)
                this._replaceButton(1,true,this.createRussianB);
            //replace button back to letter button when cyrillic_russian
			// Nothing to replace if nothing was there before
			if (this.arrKeys.length > 0) {
				this._swapKeyMap();
			}
		},

		/**
         * @private
         */
        createRussianB: function () {
            // Save scope for callback.
            var that = this;
            var BTNcontent = this.arrDefault[this.strLang][1] ? this.arrDefault[this.strLang][1][0] : "";
            this.objButton = {
                intInputWidth: 1,
                intInputHeight: 1,
                objElement: new UIElements.Keyboard.Key({
                    strButtonType: 'letter',
                    strButtonText: BTNcontent,
                    strButtonName: "alphaNum32",
                    boolAutorepeat: false,
                    // Both needed because the size of the icon ( 32 or 64 ) is computed on creation
                    intWidth: 1,
                    intHeight: 1,
                    funcCallback: function (value) {
                        that.keyboardLetterCallback(value);
                    },
                    funcCallbackRelease: function (value) {
                        ICTouchAPI.keyboardServices.keyReleased(value);
                    }
                }),
                swapContent: function (strText, strEmo) {
                        var boolEmo = strEmo != null && strEmo != '';
                        if (this.emoticon == false) {
                            if (boolEmo) {
                                this.objElement.replaceText('');
                                this.objElement.setIcon(strEmo)
                            }
                            else {
                                this.objElement.replaceText(strText);
                            }
                        } else {
                            if (boolEmo) {
                                this.objElement.replaceText('');
                                this.objElement.replaceIcon(strEmo);
                            } else {
                                this.objElement.removeIcon();
                                this.objElement.replaceText(strText);
                            }
                        }
                        this.strValue = strText;
                        this.emoticon = boolEmo;
                    }
                };

            return this.objButton;
        },

		/**
		 * Replace a button at position
		 * if custom is true, create a button using the function provided or else create an empty button
		 * @private
		 */
		_replaceButton: function(position, custom, functionRef, objButton) {
			var oldkey = this.arrKeys[position];
			var width = oldkey.intInputWidth || 1;
			var height = oldkey.intInputHeight || 1;
			this.arrKeys[position].objElement.destroy();
			if( custom ) {
				if(functionRef){
				this.arrKeys[position] = functionRef.call(this, {
					intInputHeight: height,
					intInputWidth: width
				});
			}
				else if(objButton){
					this.arrKeys[position] = this.createButton({
						intInputWidth : 1,
						intInputHeight: 1,
						strButtonText : objButton.strButtonLabel,
						strButtonIcon : objButton.strButtonIcon,
						strButtonName : objButton.strButtonName,
						strCustomClass : objButton.strCustomClass,
						boolAttachParent : true,
						callbackClick: objButton.callback
					});
				}
			}
			else {
				this.arrKeys[position] = this.createEmptyButton(width, height);
			}
			this.arrKeys[position].objElement.placeAt(this.domArrKeys[position], "only");
		},

		/**
		 * Initialize ok and exit labels
		 * @private
		 */
		_setOkExitLabels : function(strOkLabel, strExitLabel) {
			this.strOkLabel = strOkLabel || _("ok", this.i18n);
			this.strExitLabel = strExitLabel || _("Cancel", this.i18n);
		},

		/**
		 * @private
		 */
		_setOkExitIcons : function(strOkIcon, strExitIcon) {
			this.strOkIcon = strOkIcon || "generic-ok";
			this.strExitIcon = strExitIcon || "generic-cancel";
		},

		/**
		 * @private
		 */
		_loadKeyMaps: function (keyMap) {
			var that = this;
			// Load key maps from file.
			keyMap = keyMap || this.strLang;
			if (!this.languages[keyMap]) {
				dojo.xhrGet( {
					url: "library/ICTouchAPI/UIElements/Keyboard/keyMaps/"+keyMap+".json",
					handleAs: "json",
					sync: true,

					load: function(response) {
						that.arrStar = response.arrStar;
						that.arrDefault = response.arrDefault;
						that.languages[keyMap] = response;
					},

					error: function(response, ioArgs) {
						console.error(ioArgs);
					}
				});
			} else {
				this.arrStar = this.languages[keyMap].arrStar;
				this.arrDefault = this.languages[keyMap].arrDefault;
			}
		},

		/**
		 * @private
		 */
		_candidateSelected : function(args) {
			ICTouchAPI.keyboardServices.candidateSelected(args.toElement.innerText);
            this.clearIMEInput();

		},

        /**
         * @private
         */
        _candidatePageUp: function (args) {
            if(this.candidatesPage <= 0)
                return;
            --this.candidatesPage;
            this.createPagedCandidateList();
        },

        /**
         * @private
         */
        _candidatePageDown: function (args) {
            var endIndex = (this.candidatesPage + 1 ) * this.candidatesPageSize;
            if(endIndex >= this.arrCandidates.length)
                return;
            ++this.candidatesPage;
            this.createPagedCandidateList();
		},

        /**
         * @private
         */
        _callbackEmoticon : function(context){
                if(!this._flags["emoticon"]&&!this._flags["emoticonLock"]){
                        this._flags["emoticon"]=true;
                }
                else if(this._flags["emoticon"]&&!this._flags["emoticonLock"]){
                        this._flags["emoticonLock"]=true;
                }
                else if(this._flags["emoticon"]&&this._flags["emoticonLock"]){
                        this._flags["emoticon"]=false;
                        this._flags["emoticonLock"]=false;
                }
                this._swapKeyMap();
                this._updateButtonsIcons();
        },

		/**
		 * Toggle flags
		 * @private
		 * @param {bool} flag the flag to turn on/off
		 * @return the _flags object
		 */
		_toggleFlag: function (flag) {
			this._flags[flag] = this._flags[flag] == 0 ? 1: 0;
			return this._flags;
		},

		/**
		 * @private
		 * Turn all flags to off but specified flags.
		 * @param {List} butFlag a list of flags that should not be modified.
		 * @return the _flags array.
		 */
		_allFlagsOff: function (butFlag) {
			butFlag = butFlag || [];
			butFlag.join(",");
			for (var flag in this._flags) {
				if (butFlag.indexOf(flag, 0) == -1) {
					this._flags[flag] = 0;
				}
			}
			return this._flags;
		},

		/**
		 * @private
		 */
		_swapKeyMap: function (strLetter) {
			// No swap possible when type is num-pad.
			if (this.strType === "num")
				return;
			var i, _tmpKeys;
			strLetter = strLetter || this._flags["starKey"];
			// If STAR + Key
			if (this._flags["star"] && strLetter) {
				// If there's a replacement keyMap
				if (this.arrStar[""+strLetter.toLowerCase()]) {
					// Pick up the new keyMap to display.
					_tmpKeys  = this.arrStar[""+strLetter.toLowerCase()];
				}
				else {
					// No swap to do as no keyMap is defined for this key: set the flags back to default and return
					this._flags["star"] = 0;
					this._flags["starKey"] = "";
					return;
				}
			// If no STAR + Key corresponding, load the default key map.
			} else {
				if (this._flags["numLock"]) {
					_tmpKeys = this.arrDefault["arrNum"];
					if(this._flags["emoticon"]) {
                        _tmpKeys=this.arrEmoIcon;
                }
				}
                else {
					_tmpKeys = this.arrDefault[this.strLang];
				}
			}

			for (i=0; i<this.arrKeys.length; ++i) {
				if (this.arrKeys[i].swapContent) {
					if (_tmpKeys[i] === undefined) {
						this.arrKeys[i].swapContent("", "");
					} else {
				this.arrKeys[i].swapContent(_tmpKeys[i][this._flags["caps"]], _tmpKeys[i][2+this._flags["caps"]]);
			}
				}
			}
		},

		/**
		 * @private
		 * Keyboard's skeleton that must be completed with getKeyMap
		 * @param {String} type whether "alphaNum" or "dialpad"
		 * @return {list} the list of keys that compose the skeleton.
		 * @private
		 */
		_getKeyboardSkeleton: function(type) {
			var that = this;
			if (type==="alphaNum") {
				// Class to add on the domNode after templating
				this._strClassToAdd = "alphaNum";
				return  [
				this.createButton({
					strButtonText: '123',
					strNotification:'notif-off',
					strButtonName:'keyboard_NUM',
					callback: function () {
						if (that._flags["numLock"] === 0) {
							that._flags["numLock"] = 1;
							that._flags["star"] = 0;
						} else if (that._flags["numLock"] === 1) {
							that._flags["numLock"] = 0;
							that._flags["emoticon"] = 0;
							that._flags["emoticonLock"] = 0;
						}
						that._swapKeyMap();
						that._updateButtonsIcons();
					}
				}),
				// the keyboard letter at position #1, is not always there in the layout json file, so protect ourselves from "undefined"
				this.createKeyboardLetter('alphaNum32', this.arrDefault[this.strLang][1] ? this.arrDefault[this.strLang][1][0] : ""),
				this.createInputField({
					intInputWidth	: 9,
					strInputText	  : this.strInputText,
					boolPassword	  : this.keyboardFlags.password,
					hideStarsIfPasswordEmpty : this.hideStarsIfPasswordEmpty
				}),
//                    this.createEmptyButton(1, 1),
				// the keyboard letter at position #3, is not always there in the layout json file, so protect ourselves from "undefined"
				this.createKeyboardLetter('alphaNum33', this.arrDefault[this.strLang][3] ? this.arrDefault[this.strLang][3][0] : ""),
				this.createButton({
					strButtonIcon:'language',
					strButtonName: 'keyboard_FR',
					callback: function () {
						ICTouchAPI.keyboardServices.switchKeyboard();
					}
				}),
				this.createButton({
					strNotification:'notif-off',
					strButtonText: _('Alt', this.i18n),
					strButtonName:'keyboard_STAR',
					callback: function () {
						if (that._flags["numLock"] === 1) {
							that._callbackEmoticon();
						}
						else {
						that._toggleFlag("star");
						dojo.publish("UIElement.keyboard.star", [that._flags["star"]]);
						if( !that._flags["star"] ) {
							if (that._flags["starKey"] != ""){
								that._flags["starKey"] = ""
							}
							that._swapKeyMap();
						}
						}
						that._updateButtonsIcons();
					}
				}),
				this.createKeyboardLetter('alphaNum1', this.arrDefault[this.strLang][6][0]),
				this.createKeyboardLetter('alphaNum2', this.arrDefault[this.strLang][7][0]),
				this.createKeyboardLetter('alphaNum3', this.arrDefault[this.strLang][8][0]),
				this.createKeyboardLetter('alphaNum4', this.arrDefault[this.strLang][9][0]),
				this.createKeyboardLetter('alphaNum5', this.arrDefault[this.strLang][10][0]),
				this.createKeyboardLetter('alphaNum6', this.arrDefault[this.strLang][11][0]),
				this.createKeyboardLetter('alphaNum7', this.arrDefault[this.strLang][12][0]),
				this.createKeyboardLetter('alphaNum8', this.arrDefault[this.strLang][13][0]),
				this.createKeyboardLetter('alphaNum9', this.arrDefault[this.strLang][14][0]),
				this.createKeyboardLetter('alphaNum10', this.arrDefault[this.strLang][15][0]),
				this.createKeyboardLetter('alphaNum11', this.arrDefault[this.strLang][16][0]),
				this.createEmptyButton(1, 1), // Delete button
				this.createButton({
					strButtonIcon:'caps',
					strNotification: 'notif-off',
					strButtonName: 'keyboard_CAPS',
					callback: function () {
						if (that._flags["caps"] === 0 &&
							that._flags["capsLock"] === 0) {
							that._flags["caps"] = 1;
						} else if (that._flags["caps"] === 1 &&
							that._flags["capsLock"] === 0) {
							that._flags["capsLock"] = 1;
						} else if (that._flags["caps"] === 1 &&
							that._flags["capsLock"] === 1) {
							that._flags["caps"] = 0;
							that._flags["capsLock"] = 0;
						}
						that._swapKeyMap();
						that._updateButtonsIcons();
					}
				}),
				this.createKeyboardLetter('alphaNum12', this.arrDefault[this.strLang][19][0]),
				this.createKeyboardLetter('alphaNum13', this.arrDefault[this.strLang][20][0]),
				this.createKeyboardLetter('alphaNum14', this.arrDefault[this.strLang][21][0]),
				this.createKeyboardLetter('alphaNum15', this.arrDefault[this.strLang][22][0]),
				this.createKeyboardLetter('alphaNum16', this.arrDefault[this.strLang][23][0]),
				this.createKeyboardLetter('alphaNum17', this.arrDefault[this.strLang][24][0]),
				this.createKeyboardLetter('alphaNum18', this.arrDefault[this.strLang][25][0]),
				this.createKeyboardLetter('alphaNum19', this.arrDefault[this.strLang][26][0]),
				this.createKeyboardLetter('alphaNum20', this.arrDefault[this.strLang][27][0]),
				this.createKeyboardLetter('alphaNum21', this.arrDefault[this.strLang][28][0]),
				this.createKeyboardLetter('alphaNum22', this.arrDefault[this.strLang][29][0]),
				this.createEmptyButton(1, 1), // Carriage return
				this.createEmptyButton(1, 1), // Exit button
				this.createKeyboardLetter('alphaNum23', this.arrDefault[this.strLang][32][0]),
				this.createKeyboardLetter('alphaNum24', this.arrDefault[this.strLang][33][0]),
				this.createKeyboardLetter('alphaNum25', this.arrDefault[this.strLang][34][0]),
				this.createKeyboardLetter('alphaNum26', this.arrDefault[this.strLang][35][0]),
				this.createSpaceBar({
					strButtonName:'alphaNum27'
				}),
				this.createKeyboardLetter('alphaNum28', this.arrDefault[this.strLang][37][0]),
				this.createKeyboardLetter('alphaNum29', this.arrDefault[this.strLang][38][0]),
				this.createKeyboardLetter('alphaNum30', this.arrDefault[this.strLang][39][0]),
				this.createKeyboardLetter('alphaNum31', this.arrDefault[this.strLang][40][0]),
				this.createEmptyButton(1, 1) // Ok button
				];
			}

			else if (type === "num") {
				// Class to add on the domNode after templating
				this._strClassToAdd = "dialpad";

				return [
				this.createInputField({
					intInputWidth : 5,
					intInputHeight: 1,
					strInputText  : this.strInputText
				}),

				this.createEmptyButton(1, 1),
				(this._objSpace = this.createSpace()), //DialPad1
				this.createKeyboardLetter('dialPad2', "2", "abc"),
				this.createKeyboardLetter('dialPad3', "3", "def"),
				this.createEmptyButton(1, 1),

				this.createEmptyButton(1, 1),
				this.createKeyboardLetter('dialPad4', "4", "ghi"),
				this.createKeyboardLetter('dialPad5', "5", "jkl"),
				this.createKeyboardLetter('dialPad6', "6", "mno"),
				this.createEmptyButton(1, 1),

				this.createEmptyButton(1, 1),
				this.createKeyboardLetter('dialPad7', "7", "pqrs"),
				this.createKeyboardLetter('dialPad8', "8", "tuv"),
				this.createKeyboardLetter('dialPad9', "9", "wxyz"),
				this.createEmptyButton(1, 1),

				this.createEmptyButton(1, 1),
				this.createKeyboardLetter('dialPad10', "*"),
				(this._objInternational = this.createInternational()),
				this.createKeyboardLetter('dialPad12', "#"),
				this.createEmptyButton(1, 1),

				this.createEmptyButton(1, 1), // Exit button
				this.createEmptyButton(3, 1), // Ok button
				this.createEmptyButton(1, 1)  // Delete button
				];
			}
			else {
				console.warn("No keyboard loaded", type);
			}
			return [];
		},

		/**
		 * @private
		 */
		_updateButtonsIcons: function()
		{
			var len = this.arrKeys.length;
			for(var i=0; i<len; ++i)
			{
				var button = this.arrKeys[i].objElement;
				switch( button.strButtonName )
				{
					case 'keyboard_STAR':
						if (this._flags["numLock"]) {
							button.replaceText(_("emo", this.i18n));
							button.replaceIcon("im-emote-bigsmile");
						} else  {
							button.replaceText(_("Alt", this.i18n));
							button.removeIcon() ;
						}
						if( this._flags["star"] || this._flags["emoticonLock"] )
							button.attr("strNotification", "notif-on");
						else
							button.attr("strNotification", "notif-off");
						break;
					case 'keyboard_NUM':
						if( this._flags["numLock"] )
							button.attr("strNotification", "notif-on");
						else
							button.attr("strNotification", "notif-off");
						break;
					case 'keyboard_CAPS':
						if( this._flags["capsLock"] )
							button.attr("strNotification", "notif-on");
						else
							button.attr("strNotification", "notif-off");
						break;
                }
            }
		},

		/**
		 * Block the event propagation so the InputField doesn't lose focus
		 * But don't block the click on actual field
		 * @private
		*/
		_block : function(e) {
			var tagname = e.target.tagName.toLowerCase();
			if( tagname !== "input" && tagname !== "textarea" ) {
				dojo.stopEvent(e);
			}
		},

		/**
		 * @private
		 */
		keyboardLetterCallback : function(buttonText) {
			// If an extended keyboard is required.
			if (this._flags["star"]) {
				// Is it already displayed?
				if (!this._flags["starKey"]) {
					// If not, set a flag with the desired keyboard (defined by a letter).
					this._flags["starKey"] = buttonText;
					// Swap the keyboard.
					this._swapKeyMap(buttonText);
				} else {
					// If the extended keyboard is already displayed
					ICTouchAPI.keyboardServices.keyPressed(buttonText);
					// Set the flags back to default so it won't be displayed after keypress.
					this._flags["star"] = 0;
					this._flags["starKey"] = "";
					if (this._flags["caps"] === 1 && this._flags["capsLock"] === 0) {
						this._flags["caps"] = 0;
					}
					// Swap the keyboard back to default.
					this._swapKeyMap();
				}
				this._updateButtonsIcons();
			} else {
				// If no extended keyboard required or displayed.
				ICTouchAPI.keyboardServices.keyPressed(buttonText);
				// Swap back if no caps lock
				if (this._flags["caps"] === 1 && this._flags["capsLock"] === 0) {
					this._flags["caps"] = 0;
					// Swap the keyboard back to default.
					this._swapKeyMap();
					this._updateButtonsIcons();
				}
				// Swap back if no num lock
				/*if (this._flags["num"] === 1 && this._flags["numLock"] === 0) {
					this._flags["num"] = 0;
					// Swap the keyboard back to default.
					this._swapKeyMap();
					this._updateButtonsIcons();
				}*/
			}
		},

        /**
         * @private
         */
		_setDefaultModeFlags : function() {
			this._flags["caps"] = 0;
			this._flags["star"] = 0;
            this._flags["emoticon"] = 0;
            this._flags["emoticonLock"] = 0;
			this._flags["capsLock"] = 0;
			this._flags["starKey"] = "";
			//this._flags["num"] = 0;
			this._flags["numLock"] = 0;
			this._flags["multiLines"] = false;
			this._flags["ime"] = 0;
		},

	/**
	 * @private
	 */
        _imeIndicatorCallback: function () {
            /**
             * this.imeIndicatorState 0: EN else localeIME
             * */
            var nextLang = ICTouchAPI.keyboardServices.getNextLang(this.strLang);
            ICTouchAPI.keyboardServices.switchKeyboardFromIMEIndicator(nextLang);
        },
	/**
	 * @private
	 */
	_backspaceCallback: function() {
            var imeRequired = ICTouchAPI.keyboardServices._IMERequired;
            ICTouchAPI.keyboardServices.delChars(imeRequired);
	},

	/**
	 * Get next position for the buttons in dialpad mode
	 * @private
	 */
	_getNextPosition: function(flag, oldposition) {
		if (!flag) {
			return oldposition;
		}

		if (oldposition >= 23) {
			return -1;
		}
		else if (oldposition >= 20) {
			return 23;
		}
		else {
			return oldposition + 5;
		}
	},

	/**
	 * Get next position for the buttons in dialpad mode
	 * @private
	 */
	_getPreviousPosition: function(flag, oldposition) {
		if (!flag) {
			return oldposition;
		}

		if (oldposition > 20) {
			return 20;
		}
		else if(oldposition <= 5){
			return -1;
		}
		else {
			return oldposition - 5;
		}
	}


});
/**
 * @class UIElements.Keyboard.KeyboardControl
 * @extends UIElements.Keyboard.KeyboardControlBase
 * @ignore
 */
dojo.require("UIElements.Keyboard.KeyboardControlBase");
dojo.provide("UIElements.Keyboard.KeyboardControl");
dojo.declare("UIElements.Keyboard.KeyboardControl",
	UIElements.Keyboard.KeyboardControlBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @private
		 */
		cssFile         : ["base.css","icon.css"],
		/**
		 * @private
		 */
		templatePath	: dojo.moduleUrl("UIElements.Keyboard","templates/Keyboard8082.html")

		/* ------------------------------------ Constructor --------------------------------------- */

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */

	}
	);
/**
 * @class UIElements.Keyboard.KeyBase
 * @ignore
 * @extends UIElements._base
 * Abstract Class<br>
 * Represents a key of the keyboard
 */
dojo.provide("UIElements.Keyboard.KeyBase");
dojo.declare("UIElements.Keyboard.KeyBase",
	[UIElements._base, dijit._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * name of button
		 * @property
		 * @type String
		 */
		strButtonName			: null,
		/**
		 * button type
		 * @property
		 * @type String
		 */
		strButtonType			: "button",
		/**
		 * Css class for keys with notification
		 * @property
		 * @type String
		 */
		strNotification			: null,
		/**
		 * CssClass of the key
		 * @property
		 * @type String
		 */
		strCustomClass			: null,
		/**
		 * Text of the key
		 * @property
		 * @type String
		 */
		strButtonText			: null,
		/**
		 * icon of the key
		 * @property
		 * @type String
		 */
		strButtonIcon			: null,
		/**
		 * sub text of the key
		 * @property
		 * @type String
		 */
		strSubText				: null,

		/**
		 * Width of the key
		 * @property
		 * @type Number
		 */
		intWidth				: 1,
		/**
		 * Height of the key
		 * @property
		 * @type Number
		 */
		intHeight				: 1,
		/**
		 * Function called on mouse down event
		 * @property
		 * @type Function
		 */
		funcCallback			: null,
		/**
		 * Function called on long press on button. Mutually exclusive with autorepeat
		 * @property
		 * @type Function
		 */
		funcCallbackLong		: null,
		/**
		 * Function called on mouse up event
		 * @property
		 * @type Function
		 */
		funcCallbackRelease		: null,
		/**
		 * Function called on click event
		 * @property
		 * @type Function
		 */
		funcCallbackClick		: null,
		/**
		 * Should the callback function be repeated when the user keeps pressing. Mutually exclusive with long press
		 * @property
		 * @type Boolean
		 */
		boolAutorepeat			: false,
		/**
		 * Attach on parent node, attach mousedown and mouseup on parent instead of domNode.
		 * Useful when the whole td should be active instead of the button
		 * @property
		 * @type Boolean
		 */
		boolAttachParent		: true,


		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @private
		 */
		myFeedback				: null,
		/**
		 * @private
		 */
		domButtonIcon			: null,
		/**
		 * @private
		 */
		domButtonText			: null,
		/**
		 * @private
		 */
		domSubText				: null,
		/**
		 * @private
		 * Base class of this UI
		 */
		_strBaseClass			: "Key keyMainDiv",
		/**
		 * setTimeout handler for the long press callback feature
		 * @private
		 */
		_handlerLongPress		: null,
		/**
		 * @private
		 */
		_autorepeatTimeout		: null,
		/**
		 * @private
		 */
		_autorepeatInterval		: null,

		/**
		 * @private
		 * Are mouse events connected
		 */
		_eventsAttached			: false,

		/**
		 * @private
		 */
		attributeMap: {
			strButtonText: {
				node: "domButtonText",
				type: "innerHTML"
			},
			strSubText: {
				node: "domSubText",
				type: "innerHTML"
			}
		},

		/**
		 * @private
		 * True untill the postCreate is finished
		 */
		_initializing : true,

		/* ------------------------------------ Constructor --------------------------------------- */

		constructor : function(args) {
			this.myFeedback = ICTouchAPI.feedbackServices.initFeedback("click");
		},

		postMixInProperties : function() {
			this.strButtonText = this.strButtonText || "";
			this.strSubText = this.strSubText || "";

			//
		},

		postCreate : function () {
			if (this.strButtonName) {
				this.domNode.setAttribute("buttonname", this.strButtonName);
			}
			this._updateClasses();
			this._attachEventParent();

			this._initializing = false;
		},

		placeAt: function() {
			/*
			 Keyboard does create this UIElements a domNode so in postCreate this.domNode.parentNode is null
			 But after placeAt has been called there should be a parentNode
			*/
			this.inherited(arguments);
			this._attachEventParent();
		},

		destroy: function() {
			this._clearTimers();
			this.inherited(arguments);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * Replaces key text
		 * @param {String} strText new text for the key
		 */
		replaceText: function (strText) {
			if(strText==null || strText=="") {
				strText = " ";
			}
			if (strText.isI18Ned) {
				strText = strText.getTranslation();
			}
			if (strText != this.strButtonText) {
				this.attr("strButtonText", strText);

				// Webkit bug workaround
				this.domButtonText.style.opacity = "0.99";
				this.domButtonText.style.opacity = "";
			}
		},

		/**
		 * Replaces key icon
		 * @param {String} strIcon new icon for the key
		 */
		replaceIcon: function (strIcon) {
			this.attr("strButtonIcon", strIcon);
		},

		/**
		 * Remove key icon
		 */
		removeIcon: function() {
			this.attr("strButtonIcon", null);
		},

		/**
		 * Remove key icon
		 * @param {String} strIcon new icon for the key
		 */
		setIcon: function(strIcon) {
			this.attr("strButtonIcon", strIcon);
			this._updateClasses();
		},

		/**
		 * Set the long press callback
		 * @param {Function} longCallback function to call when the button is pressed for a while, null does disable the effect
		 */
		setLongCallback: function(longCallback) {
			this._clearTimers();
			this.funcCallbackLong = longCallback;
		},

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * Because marketing wants the whole zone to be active and not only the button itself, attach events to our parent
		 * @private
		 */
		_attachEventParent: function() {
			// Node where all the events will be attached
			var node = this.boolAttachParent ? this.domNode.parentNode : this.domNode;
			if (!this._eventsAttached && node) {
				this.connect(node, "mousedown", this._startHighlight);
				if( typeof this.funcCallbackClick === "function" ) {
					this.connect(node, "onclick", this._actionMouseUp);
				}
				dojo.toggleClass(node, "debug_zoneActive", this.strButtonType !== "empty");
				this._eventsAttached = true;
			}
		},

		_startHighlight: function(e, args) {
			var boolHighlight = this.inherited(arguments);
			if(boolHighlight && typeof this.funcCallbackClick !== "function") {
				this.actionNotification(e);
			}
		},

		 _stopHighlight : function() {
			this._clearTimers();
			var clicked = this.inherited(arguments);
			if(clicked) {
				this.releaseNotification();
			}
		},

		_actionMouseUp:function(clickEvent) {
			var boolHighlight = this.inherited(arguments);
 	                this._feedBack(clickEvent); 
			// Behavior for Cancel, favorites, contact...
			if( typeof this.funcCallbackClick == "function" ) {
				this.funcCallbackClick(this.strButtonText);
			}
			dojo.stopEvent(clickEvent);
		},


		/**
		 * Launch this.funcCallback on mouse down event
		 *  @param {Event} clickEvent mouse down event
		 */
		actionNotification:function(clickEvent){
			this._feedBack(clickEvent);
			// Use a different behavior is a longpress callback is present
			if (typeof this.funcCallbackLong == "function") {
				this._handlerLongPress = setTimeout(dojo.hitch(this, this.actionLongpress), 1000);
			}
			// Normal behavior, callback on mousedown
			if( typeof this.funcCallback == "function" ) {
				// If auto-repat feature is wanted
				if (this.boolAutorepeat) {
					this._autorepeatTimeout = setTimeout(dojo.hitch(this, this._repeatNotification, true), 1000);
				}
				this.funcCallback(this.strButtonText);
			}
			dojo.stopEvent(clickEvent);
		},

		actionLongpress: function() {
			// Longpress has been triggered don't callback on mouseup
			this._handlerLongPress = null;
			ICTouchAPI.keyboardServices.delChars();
			this.funcCallbackLong(this.strButtonText);
		},

		/**
		 * Launch this.funcCallbackRelease on mouse up event
		 */
		releaseNotification: function() {
			this._clearTimers();

			if (typeof this.funcCallbackRelease == "function") {
				this.funcCallbackRelease(this.strButtonText);
			}
		},

		/**
		 * @private
		 */
		_repeatNotification: function(first) {
			// Because we check if the function exist before creating the timeout we should always have a callback
			this.funcCallback(this.strButtonText);
			this._autorepeatTimeout = null;
			if (first) {
				this._autorepeatInterval = setInterval(dojo.hitch(this, this._repeatNotification, false), 100);
			}
		},

		/**
		 * @private
		 */
		_clearTimers: function() {
			if (this._handlerLongPress) {
				clearTimeout(this._handlerLongPress);
			}
			if (this._autorepeatInterval) {
				clearInterval(this._autorepeatInterval);
			}
			if (this._autorepeatTimeout) {
				clearTimeout(this._autorepeatTimeout);
			}
			this._autorepeatInterval = null;
			this._autorepeatTimeout = null;
			this._handlerLongPress = null;
		},

		/**
		 * @private
		 */
		_feedBack:function(clickEvent){
			// Don't feedback on empty buttons
			if( this.strButtonType !== "empty" ) {
				this.myFeedback.feedback(clickEvent);
			}
		},

		/**
		 * @private
		 */
		_getIconSize: function() {
			// Special case for dialpads ok button ( rotated button )
			if( this.strCustomClass === "ok" && this.intWidth > 1 ) {
				return "48";
			}
			return ( this.intWidth > 1 && this.intHeight > 1 ) ? "64" : "32";
		},

		/**
		 * @private
		 */
		_setStrNotificationAttr: function(notif) {
			if( this.strNotification != null ) {
				dojo.removeClass(this.domNotification, this.strNotification);
			}
			if( notif != null ) {
				dojo.addClass(this.domNotification, notif);
			}
			this.strNotification = notif;
		},

		/**
		 * @private
		 */
		_setStrButtonIconAttr: function (strIcon) {
			if (this.strButtonIcon != strIcon || this._initializing ) {
				if( typeof this.strButtonIcon == "string" ) {
					var oldIcon = this.strButtonIcon + "-" + this._getIconSize();
					dojo.removeClass(this.domButtonIcon, oldIcon);
				}
				this.strButtonIcon = strIcon;
				if( typeof this.strButtonIcon == "string" ) {
					strIcon = strIcon + "-" + this._getIconSize();
					dojo.addClass(this.domButtonIcon, strIcon);
				}
				this._updateClasses();
			}
		},

		/**
		 * @private
		 */
		_setStrCustomClass: function(value) {
			if (this.strCustomClass != value || this._initializing) {
				this.strCustomClass = value;
				this._updateClasses();
			}
		},

		/**
		 * @private
		 */
		_setIntWidthAttr: function(value) {
			if (this.intWidth != value || this._initializing) {
				this.intWidth = value;
				this._updateClasses();
			}
		},

		/**
		 * @private
		 */
		_setIntHeightAttr: function(value) {
			if (this.intHeight != value || this._initializing) {
				this.intHeight = value;
				this._updateClasses();
			}
		},

		/**
		 * @private
		 */
		_updateClasses: function() {
			var name = this._strBaseClass;
			if( this.strButtonType ) {
				name += " " + this.strButtonType;
			}
			if( this.strCustomClass ) {
				name += " " + this.strCustomClass;
			}
			if( this.strButtonIcon ) {
				name += " icon";
			}
			if( this.strButtonText ) {
				name += " text";
			}
			name = name + " width-" + this.intWidth + " height-" + this.intHeight;
			this.domNode.className = name;
		}
	}
);
/**
 * @class UIElements.Keyboard.Key
 * @ignore
 * @extends UIElements.Keyboard.KeyBase
 */
dojo.require("UIElements.Keyboard.KeyBase");
dojo.provide("UIElements.Keyboard.Key");
dojo.declare("UIElements.Keyboard.Key",
	UIElements.Keyboard.KeyBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @private
		 */
		templatePath				: dojo.moduleUrl("UIElements.Keyboard","templates/Key8082.html"),
		/**
		 * @private
		 */
		cssFile                     : ["8082.css","icon.css"]

		/* ------------------------------------ Constructor --------------------------------------- */

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */

	});
/**
* @class UIElements.PresentationList.PresentationListControlBase
* @extends UIElements._base
* Abstract Class - Do not use it directly
*/
dojo.provide("UIElements.PresentationList.PresentationListControlBase");
dojo.declare("UIElements.PresentationList.PresentationListControlBase",
	[UIElements._base, dojox.dtl._Templated], {

		/* --------------------------------- Public attributes ------------------------------------ */
		/**
		 * Items to display in the list
		 * @property
		 * @type {Array}
		 */
		arrItems		: [ ],
		/**
		 * Show Icons
		 * @property
		 * @type {Boolean}
		 */
		boolShowIcon	: false,
		/**
		 * Show Highlight when user presses on PresentationList item?
		 * @property
		 * @type {Boolean}
		 */
		boolShowHighlight	: false,
		/**
		 * Show Labels
		 * @property
		 * @type {Boolean}
		 */
		boolShowLabel	: false,
		/**
		 * Scroll
		 * @property
		 * @type {Boolean}
		 */
		boolScroll		: true,
        /**
         * Display style of the list, possible values : default, contactcard
         * @property
         * @type {String}
         */
        strDisplay     : "default",

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		domObjContent : 0,
		/**
		 * @ignore
		 */
		domItems		: [ ],
		/**
		 * @ignore
		 */
		domArrIcons		: [ ],
		/**
		 * @ignore
		 */
		domArrIconsLabel : [ ],
		/**
		 * @ignore
		 */
		domList			: null,
		/**
		 * @ignore
		 */
		domTextContent	: [ ],
		/**
		 * @ignore
		 */
		domContent	: [ ],
		/**
		 * @ignore
		 */
		domLabel	: [ ],
		/**
		 * @ignore
		 * Max items on screen
		 */
		_intMaxItems	: 8,
		/**
		 * @ignore
		 * Index of first element shown
		 */
		_intStart		: 0,
		/**
		 * @ignore
		 * Index of last element shown
		 */
		_intEnd			: 0,
		/**
		 * true if we must wait for highlighting, false otherwise
		 * @ignore
		 * @type {Boolean}
		 */
		_enableWaitingHighlight : true,

		/* ------------------------------------ Constructor --------------------------------------- */

		constructor     : function(){
			this.arrItems = [ ];
			this.domArrIcons = [ ];
			this.domArrIconsLabel = [ ];
			this.domItems = [ ];
			this.domContent = [ ];
			this.domLabel = [ ];
		},

		postMixInProperties : function() {
			for(var i=0; i<this.arrItems.length; ++i) {
				var obj = this.arrItems[i];

				var strLabel = obj.strLabel?obj.strLabel:"";
				var strContent = obj.strContent?obj.strContent:"";

				var handlers = ICTouchAPI.i18nServices.i18nList([strLabel], this._swapLang(i, 0) );
				ICTouchAPI.i18nServices.addI18nHandlers(this, handlers);
				var handlers2 = ICTouchAPI.i18nServices.i18nList([strContent], this._swapLang(i, 1) );
				ICTouchAPI.i18nServices.addI18nHandlers(this, handlers2);

				/*if (strLabel){
					obj.strLabel = strLabel;
				}
				if (strContent){
					obj.strContent = strContent;
				}*/

				if (!obj.strLabel){
					obj.strLabel = "";
				}

				if (obj.objContent){
					obj.boolObjContent = true;
				}
			}
		},

		postCreate : function () {
			var len = this.arrItems.length;
			for(var i=0; i<len; ++i) {
				if(this.arrItems[i].callback){
					this.arrItems[i].callback = ICTouchAPI.tools.callbackToHitch(this.arrItems[i].callback);

					// Attach to the dom element the function to attach and remove an event from _base. (as on function "newElement" from _base)
					this.domItems[i].attachEvent = this._attachEvent();
					this.domItems[i]._removeEvent = this._removeEvent(this.domItems[i]);

					// Call the attachEvent function that has just been created. (will automatically fire _removeEvent when the list is destroyed)
					this.domItems[i].attachEvent({event: "mousedown", func: "_startHighlight", capture: true});
				}

				if (this.arrItems[i].objIconLabel) {
				    dojo.addClass(this.domArrIconsLabel[i], this.arrItems[i].objIconLabel.strIcon);
				    if(this.arrItems[i].objIconLabel.width) {
					dojo.style(this.domArrIconsLabel[i], {
					    "min-width": this.arrItems[i].objIconLabel.width + "px",
					    "max-width": this.arrItems[i].objIconLabel.width + "px"
					})
				    }
				}

				if (this.arrItems[i].strLabel.isI18Ned){
					this.domLabel[i].innerHTML = this.arrItems[i].strLabel.getTranslation();
					if (this.arrItems[i].objIconLabel && this.arrItems[i].objIconLabel.width) {
					    dojo.style(this.domLabel[i], {
						"min-width": (110 - this.arrItems[i].objIconLabel.width) + "px",
						"max-width": (110 - this.arrItems[i].objIconLabel.width) + "px"
					    })
					}
				//this.arrItems[i].strLabel.fillInnerHTML(this.domLabel[i])
				}
				else{
					this.domLabel[i].innerHTML = this.arrItems[i].strLabel;
				}

				if (this.arrItems[i].strIcon && this.boolShowIcon) {
					dojo.addClass(this.domArrIcons[i], this.arrItems[i].strIcon+"-"+this._iconSize);
				}
				if (this.arrItems[i].strHighlight && this.boolShowHighlight) {
					dojo.addClass(this.domItems[i], this.arrItems[i].strHighlight);
				}
				if (this.arrItems[i].additionalStyle) {
					dojo.addClass(this.domContent[i], this.arrItems[i].additionalStyle);
				}
				if (this.arrItems[i].boolEnabled === false){
					dojo.addClass(this.domItems[i], "disabledMode");
					this.domItems[i].isDisabled = true;
				}
				this.domItems[i].callback = this.arrItems[i].callback;
				if (this.arrItems[i].boolObjContent === true) {
					this.arrItems[i].objContent.placeAt(this.domContent[i], "replace");
				}
				else {
					if (this.arrItems[i].strContent.isI18Ned){
						this.domContent[i].innerText = "";
						this.domContent[i].innerHTML = this.arrItems[i].strContent.getTranslation();
					//this.arrItems[i].strContent.fillInnerHTML(this.domContent[i])
					}
					else{
						this.domContent[i].innerHTML = "";
						this.domContent[i].innerText = this.arrItems[i].strContent;
					}
					if(typeof this.arrItems[i].callback == "function") {
						dojo.addClass(this.domItems[i], "hasAction");
					}
				}

				var positionalClass = "middle";
				positionalClass = i === 0 ? "first" : positionalClass;
				positionalClass = i === len-1 ? "last" : positionalClass;
				dojo.addClass(this.domItems[i], positionalClass);

				dojo.toggleClass(this.domItems[i], "emphasis", (this.arrItems[i].boolEmphasis == true));

				// Only keep the first _intMaxItems into the dom
				if( i >= this._intMaxItems && this.boolScroll) {
					this.domList.removeChild(this.domItems[i]);
				}
			}

            // Change the list display
            if(this.strDisplay != "default") {
                dojo.addClass(this.domNode, this.strDisplay);
            }

			this._intEnd = Math.min(len, this._intMaxItems)-1;

			if(this.boolScroll) {
				this._initScroll();
			}
		},

		// return true if the current list is not equal to the arrItems argument
		// This function may be called to avoid a refresh of the list when it is not requiered.
		isListModified : function (arrItems) {
			var modified = false;

			//No items: list deleted. modified is true
			if (!arrItems) {
				modified = true;
			}

			// check if the number of fields is the same
			if (!modified){
				if(arrItems.length != this.arrItems.length){
					modified = true;
				}
			}

			// Then, check if every field of the new array is the same in the old one
			if(!modified){
				for(var i in arrItems){
					if(arrItems[i].strId != this.arrItems[i].strId || arrItems[i].strContent != this.arrItems[i].strContent || arrItems[i].strIcon != this.arrItems[i].strIcon){
						modified = true;
						break;
					}
				}
			}
			// Finally, check if every field of the old array is the same in the new one
			if(!modified){
				for(i in this.arrItems){
					if(arrItems[i].strId != this.arrItems[i].strId || arrItems[i].strContent != this.arrItems[i].strContent){
						modified = true;
						break;
					}
				}
			}

			return modified;
		},

		destroy: function() {

			for(var i=0; i<this.arrItems.length; ++i) {
				if (this.arrItems[i].boolObjContent) {
					this.arrItems[i].objContent.destroy();
				}
			}

			this.inherited(arguments);
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		 * @ignore
		 */
		_swapLang : function (intItem, intType)  {
			var that = this;
			if(intType === 0) {
				return function (name, value) {
					if (that.domLabel && that.domLabel[intItem]) {
						return that.domLabel[intItem].innerHTML = value;
					} else {
						return false;
					}
				}
			} else if(intType === 1) {
				return function (name, value) {
					if (that.domContent && that.domContent[intItem] && !that.arrItems[intItem].boolObjContent) {
						return that.domContent[intItem].innerHTML = value;
					} else {
						return false;
					}
				}
			} else {
				return function () {
					return;
				}
			}
		},

		/**
		 * @ignore
		 */
		_initScroll: function() {
			var objScrollParams = {
				"handler" : this.domNode,
				"easing" : true,
				"scrollType" : "vertical"
			};
			// If we have more items than _intMaxItems then start the pagination mechanism
			if( this.domItems.length > this._intMaxItems ) {
				objScrollParams.pagination = {
					"callback" : this.pagination,
					"context" : this
				}
			}
			// Add the scrolling functionality
			this.scroll = FXScroll.init(objScrollParams);
		},

		/**
		 * @ignore
		 * @param {HTMLElement} element to be scrolled
		 * @return {Function} return a function ...
		 */
		pagination : function (element) {

			// If the list is scrolled more than 50 pixels
			if (this.domNode.scrollTop >= 50) {
				// Push an item at the end of the list and remove the first one.
				// Return the amount of pixels to "unscroll"
				return -50 * this.push(1);
			// If the list is scrolled to the top most
			} else if (!this.domNode.scrollTop) {
				// Add an item at the top of the list and remove the last one
				return 50 *this.unshift(1);
			} else {
				// If nothing changed, return 0.
				return 0;
			}

		},

		/**
		 * @ignore
		 * If the number of elements gets to big, removes the top nodes
		 * @param {Number} nb number of elements to be added
		 * @return {Boolean} result
		*/
		push : function(nb) {
			var node;
			// Do nothing if we are already at the end
			if( this._intEnd >= this.domItems.length-1 ) {
				return 0;
			}
			node = this.domItems[this._intStart];
			this.domList.removeChild(node);
			this._intStart++;
			this._intEnd++;
			node = this.domItems[this._intEnd];
			dojo.place(node, this.domList, "last");
			var item = this.arrItems[this._intEnd];
			if(item.objContent && item.objContent.refreshDisplay){
				item.objContent.refreshDisplay();
			}
			return 1;
		},

		/**
		 * @ignore
		 * @param {Number} nb number of elements to be unshifted
		 * @return {Boolean} result
		 */
		unshift : function(nb) {
			var node;
			// Do nothing if we are already at the end
			if( this._intStart <= 0 ) {
				return 0;
			}
			node = this.domItems[this._intEnd];
			this.domList.removeChild(node);
			this._intStart--;
			this._intEnd--;
			node = this.domItems[this._intStart];
			dojo.place(node, this.domList, "first");
			return 1;
		},

		/**
		 * @ignore
		 * run the event callback
		 * @param {Event} event number of elements to be unshifted
		 *
		 */
		itemAction : function(event) {
			if (event.currentTarget.isDisabled == true){
				return;
			}
			if (event.currentTarget.callback)
				event.currentTarget.callback();
		},

		/**
		 * @ignore
		 * move to the Item
		 * @param {Number} index index of the targeted Item
		 *
		 */
		moveToItem : function(intIndex) {
			this.scroll.getScroll().moveToItem(intIndex)
		},

		/**
		 * @ignore
		 * move to the Item
		 * @param {Number} index index of the targeted Item
		 * @param {Boolean} emphasis is Emphasis enabled ?
		 *
		 */
		toggleEmphasis: function (intIndex, boolEmphasis) {
			if(intIndex >= 0 && intIndex < this.arrSecondary.length)
			{
				dojo.toggleClass(this.arrSecondary[intIndex], "emphasis", (boolEmphasis == true));
			}
		},

		/**
		 * @ignore
		 */
		toggleDisabledMode: function(intIndex, disabledMode) {
			if(intIndex >= 0 && intIndex < this.domItems.length)
			{
				dojo.toggleClass(this.domItems[intIndex], "disabledMode", (disabledMode == true));
				this.domItems[intIndex].isDisabled = disabledMode;
			}
		},

		/**
		 * @ignore
		 * setItemTextContent change text of an item according to this position
		 * @param {Number} intPos Position
		 * @param {String} strText Text
		 */
		setItemTextContent: function(intPos, strText) {
			if(this.arrItems[intPos].boolObjContent!==true) {
				if (strText.isI18Ned) {
					strText = strText.getTranslation();
				};
				this.domContent[intPos].innerHTML = strText;
			}
		},

		/**
		 * @public
		 * Change the label of an item according to this position
		 * @param {Number} intPos Position
		 * @param {String} strText Text
		 */
		setItemLabel : function(intPos, strText) {
			if (strText.isI18Ned) {
				strText = strText.getTranslation();
			}
			this.domLabel[intPos].innerHTML = strText;
		}

	});
/**
* @class UIElements.PresentationList.PresentationListControl
* @extends UIElements.PresentationList.PresentationListControlBase
* UIElements used to display a list of properties of a specific elemement
*/
dojo.require("UIElements.PresentationList.PresentationListControlBase");
dojo.provide("UIElements.PresentationList.PresentationListControl");
dojo.declare("UIElements.PresentationList.PresentationListControl",
	UIElements.PresentationList.PresentationListControlBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		templatePath	: dojo.moduleUrl("UIElements.PresentationList","templates/PresentationList8082.html"),
		/**
		 * @ignore
		 */
		cssFile			: ["base.css", "8082.css"],
		/**
		 * @ignore
		 */
		_objFeedback	: null,

        /**
         * @ignore
         */
        _iconSize : 32,
		/* ------------------------------------ Constructor --------------------------------------- */

		constructor : function() {
		},

		destroy : function() {
			this.inherited(arguments);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 */
		_feedback : function (event) {
		},
	}
	);
/**
* @class UIElements.NotificationButton.NotificationButtonControlBase
* @extends UIElements._base
* Abstract Class - Do not use it directly
*/
dojo.provide("UIElements.NotificationButton.NotificationButtonControlBase");
dojo.declare("UIElements.NotificationButton.NotificationButtonControlBase",
	[UIElements._base, dojox.dtl._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Notification value
		 * @property
		 * @type Number
		 */
		intNotificationValue		: 0,
		/**
		 * Notification name
		 * @property
		 * @type String
		 */
		strNotificationName			: "",
		/**
		 * Notification type
		 * @property
		 * @type String
		 */
		strNotificationType			: "",
		/**
		 * Notification icone name
		 * @property
		 * @type String
		 */
		strNotificationIcon			: "",
		/**
		 * Notification label
		 * @property
		 * @type String
		 */
		strNotificationLabelIcon			: "",
		/**
		 * Notification bottom label
		 * @property
		 * @type String
		 */
		strNotificationLabelBottomIcon			: "",

		/**
		 * Notification icons displayed in the second line
		 * @property
		 * @type Array
		 */
		arrNotificationIconBottom : [],

		/**
		 * Notification label
		 * @property
		 * @type String
		 */
		strNotificationLabel		: "",

		/**
		 * Notification bottom label
		 * @property
		 * @type String
		 */
		strNotificationLabelBottom: "",

		/**
		 * callback function
		 * @property
		 * @type Function
		 */
		callback					: null,
		/**
		 * my feedback
		 * @property
		 * @type Function
		 */
		myFeedback					: null,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		_callbackFunc				: null,
		/**
		 * @ignore
		 */
		_intPosition				: -1,
		/**
		 * @ignore
		 */
		intNotifPosition			: -1,
		/**
		 * @ignore
		 */
		arrPositionClass			: ["firstNotif","secondNotif","thirdNotif","largeNotif","hiddenNotif"],
		/**
		 * @ignore
		 */
		domNotificationLabel : null,
		/**
		 * @ignore
		 */
		domNotificationLabelIcon : null,

		/**
		 * @ignore
		 */
		domNotificationLabelBottom : null,

		/**
		 * @ignore
		 */
		domNotificationLabelBottomIcon : null,

		/**
		 * @ignore
		 */
		domNotificationIconBottom: [],

		/**
		 * @ignore
		 */
		domNotificationType			: null,
		/**
		 * @ignore
		 */
		domNotificationValue		: null,

		/**
		 * @ignore
		 */
		_state:                         "IDLE",

		/**
		 * @ignore
		 */
		attributeMap: {
			strNotificationLabel: {
				node: "domNotificationLabel",
				type: "innerHTML"
			},

			strNotificationLabelBottom : {
				node: "domNotificationLabelBottom",
				type: "innerHTML"
			}
		},

		/* ------------------------------------ Constructor --------------------------------------- */

		constructor : function() {
		//this.myFeedback = ICTouchAPI.feedbackServices.initFeedback("click");
		    this.domNotificationIconBottom = [];
		},


		postCreate : function () {
			if(!generalConfig.simulation) {
				if (this.strNotificationType){
					dojo.addClass(this.domNotificationType, this.strNotificationType);
				}
				if (this.strNotificationIcon){
					dojo.addClass(this.domNotificationValue, this.strNotificationIcon);
				}
				if (this.strNotificationLabelIcon){
					dojo.addClass(this.domNotificationLabelIcon, this.strNotificationLabelIcon);
				}
				if (this.strNotificationLabelBottomIcon){
					dojo.addClass(this.domNotificationLabelBottomIcon, this.strNotificationLabelBottomIcon);
				}
				if (!this.strNotificationLabelBottom){
					dojo.style(this.domNotificationLabelBottom, "display", "none");
				}

				if (this.arrNotificationIconBottom) {
					// Define the length in pixels of all bottom icons
					var widthIcons = 140-9;
					var maxHeight = 15;
					for(var counterIcon = 0; counterIcon < this.arrNotificationIconBottom.length; counterIcon++) {
						// Calculate the total length and the highest icon of the bottom icons
						if (this.arrNotificationIconBottom[counterIcon].width) {
							widthIcons -= this.arrNotificationIconBottom[counterIcon].width;
						} else {
							// Default value, see largeNotification CSS
							widthIcons -= 15;
						}
						// Determines the highest icon
						if (this.arrNotificationIconBottom[counterIcon].width) {
							if (this.arrNotificationIconBottom[counterIcon].width > maxHeight) {
								maxHeight = this.arrNotificationIconBottom[counterIcon].width;
							}
						}
					}

					for(var counterIcon = 0; counterIcon < this.arrNotificationIconBottom.length; counterIcon++) {
						dojo.addClass(this.domNotificationIconBottom[counterIcon], this.arrNotificationIconBottom[counterIcon].icon);
						if (!counterIcon) {
							// Define the default left margin for the first icon
							dojo.style(this.domNotificationIconBottom[counterIcon],"margin-left","9px");
						}
						dojo.style(this.domNotificationIconBottom[counterIcon],"width",maxHeight+"px");
						dojo.style(this.domNotificationIconBottom[counterIcon],"height",maxHeight+"px");
					}

					/*	If bottom icons are defined, the label has no margin-left.
					The available width is calculated depending of the number of defined bottom icons.
					The font style is set to italic to fit the large notification.
					*/
					if (this.strNotificationLabelBottom) {
						dojo.style(this.domNotificationLabelBottom,"margin-left","0px");
						dojo.style(this.domNotificationLabelBottom,"width", widthIcons+"px");
						dojo.style(this.domNotificationLabelBottom,"font-style","italic");
					}
				}
			}
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * Set the position
		 * @param {Number} intPosition the new position
		 */
		setPosition : function(intPosition) {
			var strClass = null;
			if(this.arrPositionClass[intPosition]) {
				strClass = this.arrPositionClass[intPosition];
			}
			if(intPosition != this.intNotifPosition) {
				if(this.arrPositionClass[this.intNotifPosition]) {
					dojo.removeClass(this.domNotification,this.arrPositionClass[this.intNotifPosition]);
				} else {
					dojo.removeClass(this.domNotification,"hiddenNotif");
				}
			}
			if(strClass) {
				dojo.addClass(this.domNotification,strClass);
			} else {
				dojo.addClass(this.domNotification,"hiddenNotif");
			}
			this.intNotifPosition = intPosition;
		},

		/**
		 * Set the value
		 * @param {Number} value new value
		 */
		setValue : function(value){
			this.intNotificationValue=value;
			this.domNotificationValue.innerHTML = value;
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */

		_startHighlight : function() {
			// Launch default highlight and get its status
			var boolHighlight = this.inherited(arguments);
			if(boolHighlight) {
                    this._state = "PENDING";
			}
                },

		_stopHighlight : function() {
			var clicked = this.inherited(arguments);
			if(clicked) {
                    if(this._state == "PENDING") {
                        if( this._callbackFunc ) {
                                this._callbackFunc();
                        }
                        this._state = "IDLE";
                    }
			}
                },

		/**
		 * @ignore
		 */
		_setCallbackAttr : function(callback) {
			this._callbackFunc = ICTouchAPI.tools.callbackToHitch(callback);
		}
	}
	);
/**
 * @class UIElements.NotificationButton.NotificationButtonControl
 * @extends UIElements.NotificationButton.NotificationButtonControlBase
 * These buttons are used to warn the user of webapp events .
 * These events might require a specific action from the user but not necessarily.
 */
dojo.require("UIElements.NotificationButton.NotificationButtonControlBase");
dojo.provide("UIElements.NotificationButton.NotificationButtonControl");
dojo.declare("UIElements.NotificationButton.NotificationButtonControl",
	UIElements.NotificationButton.NotificationButtonControlBase,
	{

		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 * The path to the template is defined here
		 */
		templatePath			   : dojo.moduleUrl("UIElements.NotificationButton","templates/NotificationButton8082.html"),
		/**
		 * @ignore
		 */
		cssFile                    : ["8082.css", "icon.css"]

		/* ------------------------------------ Constructor --------------------------------------- */

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */

	}
	);
/**
 * @class UIElements.ProgressBar.ProgressBarControlBase
 * @extends UIElements._base
 * Abstract Class - Do not use it directly
 */
dojo.provide("UIElements.ProgressBar.ProgressBarControlBase");
dojo.declare("UIElements.ProgressBar.ProgressBarControlBase",
	[UIElements._base, dijit._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * current value
		 * @property
		 * @type Number
		 */
		intValue			: 0,
		/**
		 * minimum value
		 * @property
		 * @type Number
		 */
		intMinimum			: 0,
		/**
		 * maximum value
		 * @property
		 * @type Number
		 */
		intMaximum			: 100,		
		/**
		 * Do we have to show the time ?
		 * @property
		 * @type String
		 */
		boolShowTime		: false,		

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		domBar				: null,		
		/**
		 * @ignore
		 * Length of the bar
		 */
		_intRealWidth		: 0,
		/**
		 * @ignore
		 * marginBox of the bar
		 */
		_mbBar				: null,
		/**
		 * @ignore
		 * Pixel where the button begins
		 */
		_intStart			: 0,
		/**
		 * @ignore
		 * Pixel where the button ends
		 */
		_intEnd				: 0,		
		/**
		 * @ignore
		 * FrameId where this UI is contained
		 */
		_strFrameId			: null,

		/* ------------------------------------ Constructor --------------------------------------- */


		constructor : function() {
		},

		postMixInProperties : function() {
		},

		postCreate: function() {
			ICTouchAPI.tools.removeWhitespaceNodes(this.domBar);

			if(this.boolShowTime )
			{
				this.domPlayedTime.innerHTML = this._floatToTime(this.intValue);
				this.domTotalTime.innerHTML = (this.intMaximum)? this._floatToTime(this.intMaximum) : '';
			}


			this.setBarWidth(this.intBarWidth);
			this.setValue(this.intValue);
			this.subscribe("iframe.show", this._display);
		},

		destroy: function() {			
			this.inherited(arguments);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * Set the progressBar width
		 * @param {Number} width new width
		 */
		setBarWidth: function(width) {
			this.intBarWidth = width;
			if( this.boolShowTime )
			{
				var mbTotal = dojo.marginBox(this.domTotalTime);
				width -= mbTotal.w*2;
			}
			
			this._intRealWidth = width;			
			dojo.style(this.domBar, "width", width + "px");

			// Need to workaround a webkit bug where the width: inherit isn't applied when the div is outside of display
			var list = dojo.query(".applyWidth", this.domNode);
			list.style("width", width + "px");

			// These 2 values tweaks where the button start and end ( magic values )
			this._intEnd = width;
			this._intStart = 0;

			// set value and update display
			this._updateDisplay();
		},

		/**
		 * Set the maximum value
		 * @param {Number} max new max
		 */
		setMax : function(intMax) {
			this.intMaximum = intMax;
			if(this.boolShowTime ) {
				this.domTotalTime.innerHTML = (this.intMaximum)? this._floatToTime(this.intMaximum) : '';
			}
		},
		/**
		 * Set the current value
		 * @param {Number} value new value
		 */
		setValue: function(value) {
			this.intValue = Math.max(this.intMinimum, Math.min(this.intMaximum, value)); // Clamp between min & max
			//Apply godd color on bumper
			if(this.intValue===this.intMinimum) {
				dojo.addClass(this.domLeftBump,"minValue");
			}
			else {
				dojo.removeClass(this.domLeftBump,"minValue");
			}
			if(this.intValue===this.intMaximum) {
				dojo.addClass(this.domRightBump,"maxValue");
			}
			else {
				dojo.removeClass(this.domRightBump,"maxValue");
			}
			this._updateDisplay();
			this.valuechange();
		},
		/**
		 * get the current value
		 * @return {Number} return the value
		 */
		getValue: function() {
			return this.intValue;
		},

		/* ----------------------------------- Public methods ------------------------------------- */
		

		/**
		 * Event triggered when value changed
		 */
		valuechange : function() {

		},

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 * When an iframe is shown reset the size so everything can be computed correctly
		 * This function is tricky because sometimes the styles aren't loaded when the UI is displayed, so we wait some more
		 */
		_display: function(strCurrentFrameId) {
			// Retrieve our frame's frameId
			if( this._strFrameId == null ) {
				this._strFrameId = ICTouchAPI.tools.getFrameId(this.domNode);
			}

			if( this._strFrameId == strCurrentFrameId ) {
				// It's our frame
				this.setBarWidth(this.intBarWidth);

				if( this.domBar.scrollWidth == 0 || this.domBar.scrollLeft == 0 ) {
					// Style didn't load yet when scrollWidth or scrollLeft equals 0, retry after 100ms
					var context = this;
					setTimeout(function(){
						context._display(strCurrentFrameId);
					}, 100);
				}
			}
		},
		
		/**
		 * @ignore
		 */
		_setValueAbsolute : function(pixelValue) {
			// Convert value from pixel to meaningful intValue
			this.intValue = ((pixelValue-this._intStart)*(this.intMaximum-this.intMinimum)/(this._intEnd-this._intStart))+this.intMinimum;
			this.setValue(this.intValue);
		},
		/**
		 * @ignore
		 */
		placeAt : function() {
			this.inherited(arguments);
			this.setBarWidth(this.intBarWidth);
		},
		/**
		 * @ignore
		 */
		_updateDisplay : function() {
			// Compute the left position of the button and progressBar
			var left = this._intStart+(this.intValue-this.intMinimum) * (this._intEnd-this._intStart)/(this.intMaximum-this.intMinimum);			
			this.domBar.scrollLeft = this._intRealWidth-left;

			if( this.boolShowTime )
				this.domPlayedTime.innerHTML = this._floatToTime(this.intValue);
		},
		/**
		 * @ignore
		 */
		_floatToTime : function(intValue) {
			var val = Math.round(intValue);
			var secs = val%60;
			var pad = secs < 10 ? '0' : '';
			return Math.floor(val/60) + ":" + pad + secs;
		},
		
		/**
		 * @ignore
		 */
		refreshDisplay : function (){
			this._updateDisplay();
		}
	});
/**
 * @class UIElements.ProgressBar.ProgressBarControl
 * @extends UIElements.ProgressBar.ProgressBarControlBase
 * Displays a progressBar to show a value which can vary from a minimum value to a maximum value, linearly.
 */
dojo.provide("UIElements.ProgressBar.ProgressBarControl");
dojo.require("UIElements.ProgressBar.ProgressBarControlBase");
dojo.declare("UIElements.ProgressBar.ProgressBarControl",
	UIElements.ProgressBar.ProgressBarControlBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
	 * Bar width
	 * @property
	 * @type Number
	 */
		intBarWidth		: 400,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		templatePath		: dojo.moduleUrl("UIElements.ProgressBar", "templates/ProgressBarControl8082.html"),
		/**
		 * @ignore
		 */
		cssFile				: ["base.css", "8082.css"],	

		/* ------------------------------------ Constructor --------------------------------------- */
		
		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */
		

	});

/**
 * @class UIElements.SelectList.SelectListControlBase
 * @extends UIElements._base
 * Abstract Class - Do not use it directly
 */
dojo.provide("UIElements.SelectList.SelectListControlBase");
dojo.declare("UIElements.SelectList.SelectListControlBase",
	[UIElements._base, dijit._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Items
		 * @property
		 * @type Array
		 */
		arrItems                    : [ ],
		/**
		 * enable multi-selection
		 * @property
		 * @type Boolean
		 */
		boolMultiselect				: false,
		/**
		 * select limit
		 * @property
		 * @type Number
		 */
		intSelectLimit				: -1,
		/**
		 * enable scroll
		 * @property
		 * @type Boolean
		 */
		boolScroll					: true,
		/**
		 * Display an empty state when this list is empty
		 * @property
		 * @type {Boolean}
		 */
		showEmptyState: false,

		/**
		 * Empty state text, null show the default text from EmptyState
		 * @property
		 * @type {String}
		 */
		emptyText: null,



		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		scroll						: null,
		/**
		 * @ignore
		 * UL that will contain every item that is displayed
		 */
		domList						: null,
		/**
		 * @ignore
		 */
		_itemCallbackFunc			: null,
		/**
		 * @ignore
		 */
		_objFeedback				: null,
		/**
		 * @ignore
		 * Contain the instantiated SelectListItem
		 */
		_arrListItem				: null,
		/**
		 * @ignore
		 * Max items on screen
		 */
		_intMaxItems				: 8,
		/**
		 * @ignore
		 * Contain index of items currently selected
		 */
		_arrSelected				: null,
		/**
		 * @ignore
		 * Index of first element shown
		 */
		_intStart					: 0,
		/**
		 * @ignore
		 * Index of last element shown
		 */
		_intEnd						: 0,
		/**
		 * @ignore
		 */
		itemCallback				: null,
		/**
		 * @ignore
		 */
		buttonCallback				: null,
		/**
		 * @ignore
		 */
		buttonCallbackParams		: {},
		/**
		 * EmptyState object
		 * @ignore
		 */
		_emptyState : null,
		/**
		 * true if we must wait for highlighting, false otherwise
		 * @ignore
		 * @type {Boolean}
		 */
		_enableWaitingHighlight : true,


		/* ------------------------------------ Constructor --------------------------------------- */

		constructor	: function(){
			this.arrItems		= [ ];
			this._arrListItem	= [ ];
			this._arrSelected	= [ ];
		},

		postCreate : function () {
			// Precreate every item
			for(var i=0; i<this.arrItems.length; ++i)
			{
				var item = new UIElements.SelectList.SelectListItem(this.arrItems[i]);
				if(item.isSelected()) {
					this._arrSelected.push(i);
				}
				if(item.isEnabled()){
					item.setClickCallback(this, this._clicked, i);
					item.setActionCallback(this, this._applyAction, i);
				}
				this._arrListItem.push(item);
				// Place the first _intMaxItems into the dom
				if( i < this._intMaxItems ) {
					item.placeAt(this.domList);
				}
			}
			this._intEnd = Math.min(this.arrItems.length, this._intMaxItems)-1;
			if (!this.boolMultiselect){
				dojo.addClass(this.domNode, "SimpleSelect");
			}
			if (this.boolScroll && this.arrItems.length){
				this._initScroll();
			}

			if (this.arrItems.length <= 0 && this.showEmptyState) {
				var params = {
					boolVisible: true
				};
				if (this.emptyText) {
					params.strEmptyMessage = this.emptyText;
				}
				this._emptyState = new UIElements.EmptyState.EmptyStateControl(params);
				this._emptyState.placeAt(this.domNode, "first");
			}
		},

		/**
		 * Destroy
		 */
		destroy : function() {
			var objItem;
			while( (objItem = this._arrListItem.pop()) != null ) {
				objItem.destroy();
			}
			delete this._arrSelected;

			this.inherited(arguments);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * return a table containing the index of every selected item
		 * @return {Array} Index of every selected item
		 */
		getSelected: function() {
			// Return a clone so the application can modify our version
			return dojo.clone(this._arrSelected);
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		 * paginate a given UI
		 * @param {HTMLElement} element UI element to be paginated
                 * @ignore
		 * @return {Function} function
		 */
		pagination : function (element) {
			// If the list is scrolled more than 50 pixels
			if (this.domNode.scrollTop >= 50) {
				// Push an item at the end of the list and remove the first one.
				// Return the amount of pixels to "unscroll"
				return -50 * this.push(1);
			// If the list is scrolled to the top most
			} else if (!this.domNode.scrollTop) {
				// Add an item at the top of the list and remove the last one
				return 50 *this.unshift(1);
			} else {
				// If nothing changed, return 0.
				return 0;
			}
		},

		/**
		 * Remove the first selectable dom element and add a new one at the end of the list.
		 * Elements come from the private liste of items
		 * @return {Boolean} true if an element has been pushed, else return false
		 */
		push : function() {
			var node;
			// Do nothing if we are already at the end
			if( this._intEnd >= this._arrListItem.length-1 ) {
				return 0;
			}
			node = this._arrListItem[this._intStart].domNode;
			this.domList.removeChild(node);
			this._intStart++;
			this._intEnd++;
			node = this._arrListItem[this._intEnd].domNode;
			dojo.place(node, this.domList, "last");
			return 1;
		},

		/**
		 * Remove the last selectable dom element and add a new one at the beginning of the list.
		 * Elements come from the private liste of items
		 * @return {Boolean} true if an element has been unshifted, else return false
		 */
		unshift : function() {
			var node;
			// Do nothing if we are already at the end
			if( this._intStart <= 0 ) {
				return 0;
			}
			node = this._arrListItem[this._intEnd].domNode;
			this.domList.removeChild(node);
			this._intStart--;
			this._intEnd--;
			node = this._arrListItem[this._intStart].domNode;
			dojo.place(node, this.domList, "first");
			return 1;
		},



		/**
		 * Select every item up to the limit defined by intSelectLimit
		 */
		selectAll: function() {
			var len = this._arrListItem.length;
			if( this.intSelectLimit > -1 ) {
				len = Math.min(len, this.intSelectLimit);
			}
			for (var i = 0; i<len; ++i) {
				this._arrListItem[i].select();
				this._arrSelected.push(i);
			}
		},

		/**
		 * If possible select the item
		 * @param  {Number} intIndex Index of the item in the arrItems list
		 * @return {Boolean} did the item get selected
		 */
		selectAtIndex: function(intIndex) {
			// Did we reach any limit
			var canCheck = this.intSelectLimit < 0 || this._arrSelected.length < this.intSelectLimit;
			if( intIndex >= 0 && intIndex < this._arrListItem.length && canCheck ) {
				this._arrListItem[intIndex].select();
				this._arrSelected.push(intIndex);
				return true;
			}
			return false;
		},

		/**
		 * If the item is selected, unselect it
		 * @param {Number} intIndex Index of the item in the arrItems list
		 * @return {Boolean} was the item selected
		 */
		unselectAtIndex: function(intIndex) {
			var selectedIndex = -1;
			for(var i in this._arrSelected){
				if(this._arrSelected[i]==intIndex){
					selectedIndex = i;
				}
			}
			//var selectedIndex = this._arrSelected.indexOf(intIndex);
			if( selectedIndex > -1 ) {
				this._arrListItem[intIndex].unselect();
				this._arrSelected.splice(selectedIndex, 1);
				return true;
			}
			return false;
		},

		/**
		* Unselect every item unconditionally
		*/
		unselectAll: function() {
			var intIndex;
			while( (intIndex = this._arrSelected.pop()) != null ) {
				this._arrListItem[intIndex].unselect();
			}
		},

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 */
		_initScroll: function() {
			var objScrollParams = {
				"handler" : this.domNode,
				"easing" : true,
				"scrollType" : "vertical"
			};
			// If we have more items than _intMaxItems then start the pagination mechanism
			if( this._arrListItem.length > this._intMaxItems ) {
				objScrollParams.pagination = {
					"callback" : this.pagination,
					"context" : this
				}
			}

                        /*
                        if(this._arrListItem.length < 6) {
                            objScrollParams.bounce = false;
                        }
                        */

			// Add the scrolling functionality
			this.scroll = FXScroll.init(objScrollParams);
		},

		/**
		 * @ignore
		 */
		_setItemCallbackAttr : function(callback) {
			this._itemCallbackFunc = ICTouchAPI.tools.callbackToHitch(callback);
		},

		/**
		 * @ignore
		 */
		_clicked : function(intItem) {
			// Get the clicked item
			var objItem = this._arrListItem[intItem];
			var intIndex = this.arrItems[intItem].intIndex;

			if (!this.boolMultiselect)  {
				this.unselectAll();
			}
			if( objItem.isSelected() ) {
				// Try to unselect it and if successful call the callback
				if( this.unselectAtIndex(intItem) && this._itemCallbackFunc ) {
					this._itemCallbackFunc(intIndex, false);
				}
			}
			else {
				// Try to select it and if successful call the callback
				if( this.selectAtIndex(intItem) && this._itemCallbackFunc ) {
					this._itemCallbackFunc(intIndex, true);
				}
			}
		},

		/**
		 * @ignore
		 */
		_applyAction : function(intItem) {
			var intIndex = this.arrItems[intItem].intIndex;
			if (this.buttonCallback){
				this.buttonCallback(intIndex, this.buttonCallbackParams);
			}
		}
	}
);
/**
 * @class UIElements.SelectList.SelectListControl
 * @extends UIElements.SelectList.SelectListControlBase
 * This UIElement is used to gather a choice or list a choices made by a user. The label of a list element can be rather long and the element can contain a preview icon (in form of an image (<img> tag) or in form of a CSS class. An action icon and a callback on it can be associated to the list element (i.e. preview of a wallpaper).
 */
dojo.require("UIElements.SelectList.SelectListControlBase");
dojo.provide("UIElements.SelectList.SelectListControl");
dojo.declare("UIElements.SelectList.SelectListControl",
	UIElements.SelectList.SelectListControlBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		templatePath			: dojo.moduleUrl("UIElements.SelectList","templates/SelectList8082.html"),
		/**
		 * @ignore
		 */
		cssFile                 : ["base.css", "8082.css"]

		/* ------------------------------------ Constructor --------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */
	}
	);
/**
 * @class UIElements.SelectList.SelectListItemBase
 * @extends UIElements._base
 * Abstract Class - Do not use it directly
 */
dojo.provide("UIElements.SelectList.SelectListItemBase");
dojo.declare("UIElements.SelectList.SelectListItemBase",
	[UIElements._base, dojox.dtl._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Secondary content icon
		 * @property
		 * @type String
		 */
		strSecondaryContentIcon		: "",
		/**
		 * Secondary design icon
		 * @property
		 * @type String
		 */
		strSecondaryDesignIcon		: "",
		/**
		 * Checked icon
		 * @property
		 * @type String
		 */
		strCheckedIcon				: null,
		/**
		 * Unchecked icon
		 * @property
		 * @type String
		 */
		strUncheckedIcon         	: null,
		/**
		 * label
		 * @property
		 * @type String
		 */
		strLabel					: "",
		/**
		 * Action icon
		 * @property
		 * @type String
		 */
		strActionIcon				: "",
		/**
		 * Additionnal style: alternative css class (optional, provided in the item)
		 * @property
		 * @type String
		 */
		additionalStyle		: "",
		/**
		 * Button callback function
		 * @property
		 * @type Function
		 */
		buttonCallback				: null,
		/**
		 * Items callback function
		 * @property
		 * @type Function
		 */
		itemCallback				: null,
		/**
		 * Selected
		 * @property
		 * @type Boolean
		 */
		boolSelected				: false,
		/**
		 * Enabled (if item is not enabled, it appears grey tint and not selectable)
		 * @property
		 * @type Boolean
		 */
		boolEnabled				: true,
		/**
		 * Index
		 * @property
		 * @type Number
		 */
		intIndex					: -1,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		_itemCallbackFunc			: null,
		/**
		 * @ignore
		 */
		_buttonCallbackFunc			: null,
		/**
		 * @ignore
		 */
		_boolSecondaryIcon			: false,
		/**
		 * @ignore
		 */
		_funcCallback				: null,
		/**
		 * @ignore
		 */
		_objContext					: null,
		/**
		 * @ignore
		 */
		_callbackValue				: null,
		/**
		 * @ignore
		 */
		_objLastEvent				: null,
		/**
		 * @ignore
		 */
		_checked					: false,
		/**
		 * @ignore
		 */
		_funActioncCallback			: null,
		/**
		 * @ignore
		 */
		_objActionContext			: null,
		/**
		 * @ignore
		 */
		_callbackActionValue		: null,
		/**
		 * @ignore
		 */
		domCheckIcon				: null,
		/**
		 * @ignore
		 */
		domUncheckIcon				: null,
		/**
		 * @ignore
		 */
		domSecondaryIcon			: null,
		/**
		 * @ignore
		 */
		domActionButton				: null,
		/**
		 * @ignore
		 */
		domActionIcon				: null,
		/**
		 * @ignore
		 */
		attributeMap: {
			"strLabel": {
				node: "domLabel",
				type: "innerHTML"
			}
		},
		/**
		 * true if we must wait for highlighting, false otherwise
		 * @ignore
		 * @type {Boolean}
		 */
		_enableWaitingHighlight : true,

		/* ------------------------------------ Constructor --------------------------------------- */

		postMixInProperties : function() {
			this.strActionIcon = this.strActionIcon || "";
			if (this.strSecondaryDesignIcon || this.strSecondaryContentIcon){
				this._boolSecondaryIcon = true;
			}
		},

		postCreate : function () {
			var icon;
			if (this.strSecondaryDesignIcon){
				icon = this.strSecondaryDesignIcon + "-" + this._intIconSize;
				dojo.addClass(this.domSecondaryIcon, icon);
			}
			if (this.additionalStyle){
				dojo.addClass(this.domNode, this.additionalStyle);
			}
			if (this.strActionIcon){
				icon = this.strActionIcon + "-" + this._intIconSize;
				dojo.addClass(this.domActionIcon, icon);
				this.domActionButton.intIndex = this.intIndex;
			}
			if (this.boolSelected) {
				this.select();
			}
			else {
				this.unselect();
			}
			if (this.boolEnabled == false){
				dojo.toggleClass(this.domNode, "disabledModeSelectList", true);
				dojo.toggleClass(this.domNode, "selectableLine", false);
			}
			this.domNode.intIndex = this.intIndex;
		},

		destroy: function() {
			delete this._funcCallback;
			delete this._objContext;
			delete this._objLastEvent;
			this.inherited(arguments);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * set the click event's callback function
		 * @param {Object} context the event context
		 * @param {Function} funcCallback callback function to be runned
		 * @param {Number} value ???
		 */
		setClickCallback: function(context, funcCallback, value) {
			this._funcCallback = funcCallback;
			this._objContext = context;
			this._callbackValue = value;
		},

		/**
		 * return true if the dom icon has the "checked" class
		 * @return {Boolean} domIcon has checked class ?
		 */
		isSelected : function() {
			return this._checked;
		},

		/**
		 * return the Enabled state of the item
		 * @return {Boolean} domNode has not disabledMode class ?
		 */
		isEnabled : function() {
			return this.boolEnabled;
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		 * Select
		 */
		select : function () {
			this._checked = true;
			this._updateIcon(true);
		},

		/**
		 * Unselect
		 */
		unselect : function() {
			this._checked = false;
			this._updateIcon(false);
		},

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 */
		setActionCallback: function(context, funcCallback, value) {
			this._funcActionCallback = funcCallback;
			this._objActionContext = context;
			this._callbackActionValue = value;
		},

		/**
		 * @ignore
		 */
		_clicked: function(event) {
			//There is a very odd behavior here, every event comes two times so for now this is a workaround
			if( event === this._objLastEvent ) {
				return;
			}
			this._objLastEvent = event;
			if( typeof this._funcCallback == "function" ) {
				this._funcCallback.call(this._objContext, this._callbackValue);
			}
		},

		/**
		 * @ignore
		 */
		_actionClicked: function(event) {
			//There is a very odd behavior here, every event comes two times so for now this is a workaround
			if( event === this._objLastEvent ) {
				return;
			}
			this._objLastEvent = event;
			if( typeof this._funcActionCallback == "function" ) {
				this._funcActionCallback.call(this._objActionContext, this._callbackActionValue);
			}
		},

		/**
		 * Every remove and addClass are done here according to the boolChecked flag
		 * @ignore
		 */
		_updateIcon: function(boolChecked) {
			// Check that strCheckedIcon and strUncheckedIcon isn't garbage
			if( typeof this.strCheckedIcon == "string" && this.strCheckedIcon.length ) {
				dojo.toggleClass(this.domIcon, this.strCheckedIcon, boolChecked);
			}
			else {
				dojo.toggleClass(this.domIcon, "checked", boolChecked);
			}
			if( typeof this.strUncheckedIcon == "string" && this.strUncheckedIcon.length ) {
				dojo.toggleClass(this.domIcon, this.strUncheckedIcon, !boolChecked);
			}
			else {
				dojo.toggleClass(this.domIcon, "unchecked", !boolChecked);
			}
		},

		/**
		 * @ignore
		 */
		_setItemCallbackAttr : function(callback) {
			this._itemCallbackFunc = ICTouchAPI.tools.callbackToHitch(callback);
		},

		/**
		 * @ignore
		 */
		_setButtonCallbackAttr : function(callback) {
			this._buttonCallbackFunc = ICTouchAPI.tools.callbackToHitch(callback);
		}

	}
	);
/**
 * @class UIElements.SelectList.SelectListItem
 * @extends UIElements.SelectList.SelectListItemBase
 * Represents an item of the SelectList
 */
dojo.require("UIElements.SelectList.SelectListItemBase");
dojo.provide("UIElements.SelectList.SelectListItem");
dojo.declare("UIElements.SelectList.SelectListItem",
	UIElements.SelectList.SelectListItemBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		strActionIcon			: "",

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		templatePath			: dojo.moduleUrl("UIElements.SelectList","templates/SelectListItem8082.html"),
		/**
		 * @ignore
		 */
		domActionIcon			: null,
		/**
		 * @ignore
		 */
		_intIconSize			: 32,

		/* ------------------------------------ Constructor --------------------------------------- */

		postMixInProperties : function() {
			this.strActionIcon = this.strActionIcon || "";
			this.inherited(arguments);
		},

		postCreate : function () {
			if (this.strActionIcon){
				dojo.addClass(this.domActionIcon, this.strActionIcon);
				this.domActionButton.intIndex = this.intIndex;
			}
			this.inherited(arguments);
		}
	/* ----------------------------------- Getter / Setter------------------------------------- */

	/* ----------------------------------- Public methods ------------------------------------- */

	/* --------------------------------- Private Methods -------------------------------------- */
	}
	);
dojo.provide("UIElements.SimpleButton.SimpleButtonControlBase");

dojo.declare("UIElements.SimpleButton.SimpleButtonControlBase",
	[UIElements._base, dijit._Templated],
	{
		// All attributes of this class are available in the template
		strButtonName			: null,
		strButtonType			: null,
		strOtherClass			: null,
		strButtonText			: null,
		strButtonIcon			: null,
		strSubText				: null,

		intWidth				: 1,
		intHeight				: 1,

		domButtonIcon			: null,
		domButtonText			: null,
		domSubText				: null,
		funcCallback			: null,
		funcCallbackRelease		: null,
		myFeedback				: null,

		_strBaseClass			: "simpleButton", // Base class of this UI

		attributeMap: {
			strButtonText: { node: "domButtonText", type: "innerHTML" },
			strSubText: { node: "domSubText", type: "innerHTML" }
		},

		constructor : function(args) {
			this.myFeedback = ICTouchAPI.feedbackServices.initFeedback("click");
		},

		// Function called before displaying the output (allows you to check variables)
		postMixInProperties : function() {
			this.strButtonText = this.strButtonText || "";
			this.strSubText = this.strSubText || "";
		},

		postCreate : function () {
			if (this.strButtonIcon)
				dojo.addClass(this.domButtonIcon, this.strButtonIcon);
			if (this.strButtonName) {
				this.domNode.setAttribute("buttonname",this.strButtonName);
			}
			this._updateClasses();
		},

		actionNotification:function(clickEvent){
			this.myFeedback.feedback(clickEvent);
			this.funcCallback(this.strButtonText);
		},

		// WRK : crms00204576
		releaseNotification:function(){
			if (typeof this.funcCallbackRelease === "function") {
				this.funcCallbackRelease();
			}
		},

		_feedBack:function(clickEvent){
			this.myFeedback.feedback(clickEvent);
		},

		replaceText: function (strText) {
			if (strText.isI18Ned) {
				strText = strText.getTranslation();
			};
			this.attr("strButtonText", strText);
			// Workaround webkit bug ?
			this.domButtonText.style.opacity = "0.99";
			this.domButtonText.style.opacity = "";
		},

		replaceIcon: function (strIcon) {
			this.attr("strButtonIcon", strIcon);
		},

		_getIconSize: function(){
			var size = 0;
			switch(this.strButtonType){
				case "TextButton":
					size = "10";
					break;
				case "ActionButton":
					// Smaller icon if we have text
					var smallSize = ( this.strButtonText.length > 0 ) ? "24" : "32";
					size = ( this.intWidth > 1 ) ? "64" : smallSize;
					break;
			}
			return size;
		},

		_setStrButtonIconAttr: function (strIcon) {
			strIcon = strIcon + "-" + this._getIconSize();
			/*if( this.init )
			{
				var func = dojo.hitch(this, function(item) {
					this.domButtonIcon = item;
				});
				var transition = new UIEffects.Transition.CssClass(this.domButtonIcon, {
					strNewClass: strIcon,
					strOldClass: this.strButtonIcon,
					funcTransitionEnd: func
				});
				transition.launch();
			}
			else
			{*/
			dojo.removeClass(this.domButtonIcon, this.strButtonIcon);
			dojo.addClass(this.domButtonIcon, strIcon);
			/*	this.init = true;
			}*/
			this.strButtonIcon = strIcon;
		},

		_setStrButtonType: function(value) {
			this.strButtonType = value;
			this._updateClasses();
		},

		_setStrOtherClass: function(value) {
			this.strOtherClass = value;
			this._updateClasses();
		},

		_setIntWidthAttr: function(value) {
			this.intWidth = value;
			this._updateClasses();
		},

		_setIntHeightAttr: function(value) {
			this.intHeight = value;
			this._updateClasses();
		},

		_updateClasses: function() {
			var name = this._strBaseClass;
			if( this.strButtonType ) {
				name = name + " " + this.strButtonType;
			}
			if( this.strOtherClass ) {
				name = name + " " + this.strOtherClass;
			}
			name = name + " width-" + this.intWidth + " height-" + this.intHeight;
			this.domNode.className = name;
		}
	}
);
dojo.require("UIElements.SimpleButton.SimpleButtonControlBase");
dojo.provide("UIElements.SimpleButton.SimpleButtonControl");
// This class is related to:
//- dijit._Widget: the basic framework for each Widget
//- dojox.dtl._Templated: enable to use the django syntax in the template layout
dojo.declare("UIElements.SimpleButton.SimpleButtonControl", UIElements.SimpleButton.SimpleButtonControlBase, {
	// The path to the template is defined here
	templatePath				: dojo.moduleUrl("UIElements.SimpleButton","templates/SimpleButton8082.html"),
	cssFile                     : "base.css"
});
/**
 * @class UIElements.Toaster.ToasterControlBase
 * @extends UIElements._base
 * Abstract Class - Do not use it directly
 */
dojo.provide("UIElements.Toaster.ToasterControlBase");
dojo.declare("UIElements.Toaster.ToasterControlBase",
	[UIElements._base, dijit._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		domToaster     : null,

		/* ------------------------------------ Constructor --------------------------------------- */

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		 * Hide a given node
		 * @param {HTMLElement} node node to be hidden
		 */
		hide : function (node) {
			node.style.bottom = '-1000px';
		},
		/**
		 * Show a given node
		 * @param {HTMLElement} node shown to be hidden
		 */
		show : function(node) {
			node.style.bottom = '0';
		}

	/* --------------------------------- Private Methods -------------------------------------- */
	}
	);
/**
 * @class UIElements.Toaster.ToasterControl
 * @extends UIElements.Toaster.ToasterControlBase
 * Displays a retractable panel allowing to display content
 */
dojo.require("UIElements.Toaster.ToasterControlBase");
dojo.provide("UIElements.Toaster.ToasterControl");
dojo.declare("UIElements.Toaster.ToasterControl",
	UIElements.Toaster.ToasterControlBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		templatePath	: dojo.moduleUrl("UIElements.Toaster","templates/Toaster8082.html"),
		/**
		 * @ignore
		 */
		cssFile			: ["base.css", "8082.css"]

		/* ------------------------------------ Constructor --------------------------------------- */

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */
	}
);
dojo.provide("UIElements.InfoBar.InfoBarControlBase");

dojo.declare("UIElements.InfoBar.InfoBarControlBase",
	[UIElements._base, dojox.dtl._Templated],
	{

/* --------------------------------- Public attributes ------------------------------------ */

		/*
		 * Array of items
		 * @property
		 * @type Array
		 */
		arrItems					: [ ],
		/*
		 * Array of info bar elements
		 * @property
		 * @type Array
		 */
		arrInfoBarElements			: [ ],

/* --------------------------------- Private attributes ----------------------------------- */

/* ------------------------------------ Constructor --------------------------------------- */

		constructor				: function(){
			this.arrItems			= [ ];
			this.arrInfoBarElements	= [ ];
		},

		postMixInProperties : function() {
			for(var i=0; i<this.arrItems.length; ++i)
			{
				if( !this.arrItems[i] )
					this.arrItems[i] = "";
			}
		},

		postCreate : function () {
			for(var i=0; i<this.arrItems.length; ++i)
			{
				var tempObj = ICTouchAPI.tools.getObjectReference(this.arrItems[i]);
				new tempObj({}, this.arrInfoBarElements[i]);
			}
		}

/* ------------------------------------- Destructor --------------------------------------- */

/* ----------------------------------- Getter / Setter------------------------------------- */

/* ----------------------------------- Public methods ------------------------------------- */

/* --------------------------------- Private Methods -------------------------------------- */

});
dojo.require("UIElements.InfoBar.InfoBarControlBase");
dojo.provide("UIElements.InfoBar.InfoBarControl");

dojo.declare("UIElements.InfoBar.InfoBarControl",
	UIElements.InfoBar.InfoBarControlBase,
	{
		templatePath				: dojo.moduleUrl("UIElements.InfoBar","templates/InfoBar8082.html"),
		cssFile                     : ["base.css", "8082.css"]

/* --------------------------------- Public attributes ------------------------------------ */

/* --------------------------------- Private attributes ----------------------------------- */

/* ------------------------------------ Constructor --------------------------------------- */

/* ------------------------------------- Destructor --------------------------------------- */

/* ----------------------------------- Getter / Setter------------------------------------- */

/* ----------------------------------- Public methods ------------------------------------- */

/* --------------------------------- Private Methods -------------------------------------- */

});
/**
 * @class UIElements.ValueSelect.ScrollSelectBase
 * @extends UIElements._base
 * Abstract Class - Do not use it directly
 * 
 */
dojo.provide("UIElements.ValueSelect.ScrollSelectBase");
dojo.declare("UIElements.ValueSelect.ScrollSelectBase",
	[UIElements._base, dijit._Templated],
	{


		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * value
		 * @property
		 * @type Number
		 */
		intValue			: 0,
		/**
		 * Minimal range
		 * @property
		 * @type Number
		 */
		intRangeMin			: null,
		/**
		 * Maximal range
		 * @property
		 * @type Number
		 */
		intRangeMax			: null,
		/**
		 * step size
		 * @property
		 * @type Number
		 */
		intStep				: 1,
		/**
		 * endless loop
		 * @property
		 * @type Boolean
		 */
		boolLoop			: true,
		/**
		 * prefix string
		 * @property
		 * @type String
		 */
		strPrefix			: "",
		/**
		 * suffix string
		 * @property
		 * @type String
		 */
		strSuffix			: "",
		/**
		 * format
		 * @property
		 * @type String
		 */
		strFormat			: "",
		/**
		 * size string
		 * @property
		 * @type String
		 */
		strSize				: "",

		/**
		 * Callback called value change
		 * This property is set when instanciating the UIElement
		 * @property
		 * @type {Function}
		 */
		callback : {},

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		_objFeedback		: null,
		/**
		 * @ignore
		 */
		domInputText		: null,
		/**
		 * @ignore
		 */
		domUpButton			: null,
		/**
		 * @ignore
		 */
		domDownButton		: null,
		/**
		 * @ignore
		 */
		attributeMap: {
			strPrefix: {
				node: "domPrefixText",
				type: "innerHTML"
			},
			strSuffix: {
				node: "domSuffixText",
				type: "innerHTML"
			}
		},

		/* ------------------------------------ Constructor --------------------------------------- */

		constructor : function() {
			this.callback = {};
		},

		postMixInProperties : function() {
			if (this.intRangeMin === null || this.intRangeMax === null)
				this.boolLoop = false;   // can't loop when no range is defined
		},

		postCreate : function () {
			this.domInputText.value = dojox.string.sprintf(this.strFormat, this.intValue);
			dijit.typematic.addMouseListener(this.domUpButton, this, this._increaseValue, 50, 500);
			dijit.typematic.addMouseListener(this.domDownButton, this, this._decreaseValue, 50, 500);
			this.connect(this.domUpButton, "onmousedown", this._feedback);
			this.connect(this.domDownButton, "onmousedown", this._feedback);
			if (this.strSize){
				dojo.addClass(this.domNode, this.strSize);
			}
			this.subscribe("iframe.hide", this._stop);
			this.subscribe("toaster.show", this._stop);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * return the value
		 * @return {Number} return the value
		 */

		getValue : function() {
			return this.attr("intValue");
		},

		/**
		 * Set the value up to the given one
		 * @param {Number} value new value
		 */
		setValue : function(value) {
			this.attr("intValue", value);
		},

		/**
		 * Set Minimal range
		 * @param {Number} value new value
		 */
		setMaxRangeValue : function(value) {
			this.intRangeMax = value;
		},

		/**
		 * Set Maximal range
		 * @param {Number} value new value
		 */
		setMinRangeValue : function(value) {
			this.intRangeMin = value;
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 */
		_increaseValue : function(count) {
			// When count < 0 it's to signal that the iteration has stopped
			if( count < 0 )
				return;
			var value = this.intValue + this.intStep;
			if (this.intRangeMax !== null && value > this.intRangeMax) {
				if (this.boolLoop)
					value = this.intRangeMin;
				else
					value = this.intRangeMax;
			}
			this.attr('intValue', value);
			this._executeCallback("callback");
		},

		/**
		 * @ignore
		 */
		_decreaseValue : function(count) {
			if( count < 0 )
				return;
			var value = this.intValue - this.intStep;
			if (this.intRangeMin !== null && value < this.intRangeMin) {
				if (this.boolLoop)
					value = this.intRangeMax;
				else
					value = this.intRangeMin;
			}
			this.attr('intValue', value);
			this._executeCallback("callback");
		},

		/**
		 * @ignore
		 */
		_setIntValueAttr : function(newValue) {
			// Round to 0.1 ( avoid rounding errors )
			newValue = Math.round(newValue*10)/10;
			if( this.strFormat !== "" )
				this.domInputText.innerHTML = dojox.string.sprintf(this.strFormat, newValue);
			else
				this.domInputText.innerHTML = newValue;
			this.intValue = newValue;
		},

		/**
		 * @ignore
		 */
		_feedback : function (event) {
		},


		/**
		 * Stop automatic increasing/decreasing values
		 * @ignore
		 */
        _stop : function () {
            dijit.typematic.stop();
        },

		/**
		 * A generic function for executing the UI element callbacks
		 * @ignore
		 * @param {String} name the name of the callback to execute
		 * @param {Object} params the callback's parameters
		 * @return {Boolean} the result of the callback execution or false
		 */
		_executeCallback : function(name, params) {
			// If the callback is a function
			var cb = this[name];
			if (typeof cb === "function") {
				// Execute it
				return cb(params);
			// If it's an object
			} else if (cb && typeof cb.func === "function") {
				// Execute it in the given context, or in the current one.
				cb.func.call(cb.context || window, params || null);
			} else {
				return false;
			}
		}
	});
/**
 * @class UIElements.ValueSelect.ScrollSelect
 * @extends UIElements.ValueSelect.ScrollSelectBase
 * Displays a selectbox that allows the user to select a value (integer or decimal) which can range from a minimum value to a maximum value.
 * A suffix and/or a prefix can be applied to this value (for exemple a unit).
 */
dojo.require("UIElements.ValueSelect.ScrollSelectBase");
dojo.provide("UIElements.ValueSelect.ScrollSelect");
dojo.declare("UIElements.ValueSelect.ScrollSelect",
	UIElements.ValueSelect.ScrollSelectBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		templatePath : dojo.moduleUrl("UIElements.ValueSelect","templates/ScrollSelect8082.html"),
		/**
		 * @ignore
		 */
		cssFile : ["base.css", "8082.css"]

	/* ------------------------------------ Constructor --------------------------------------- */

	/* ----------------------------------- Getter / Setter------------------------------------- */

	/* ----------------------------------- Public methods ------------------------------------- */

	/* --------------------------------- Private Methods -------------------------------------- */
	}
	);
/**
 * @class UIElements.ValueSelect.NumeralSelectBase
 * @extends UIElements.ValueSelect.ScrollSelect
 * Abstract Class - Do not use it directly
 */
dojo.require("UIElements.ValueSelect.ScrollSelect");
dojo.provide("UIElements.ValueSelect.NumeralSelectBase");
dojo.declare("UIElements.ValueSelect.NumeralSelectBase",
	UIElements.ValueSelect.ScrollSelect,
	{
		/* --------------------------------- Public attributes ------------------------------------ */
		/**
		 * loop
		 * @property
		 * @type Boolean
		 */
		booLoop        : false

	/* --------------------------------- Private attributes ----------------------------------- */

	/* ------------------------------------ Constructor --------------------------------------- */

	/* ----------------------------------- Getter / Setter------------------------------------- */

	/* ----------------------------------- Public methods ------------------------------------- */

	/* --------------------------------- Private Methods -------------------------------------- */
	}
	);
/**
 * @class UIElements.ValueSelect.NumeralSelect
 * @extends UIElements.ValueSelect.NumeralSelectBase
 * Displays a selectbox that allows the user to select a value (integer or decimal) which can range from a minimum value to a maximum value.
 * A suffix and/or a prefix can be applied to this value (for exemple a unit).
 */
dojo.require("UIElements.ValueSelect.NumeralSelectBase");
dojo.provide("UIElements.ValueSelect.NumeralSelect");
dojo.declare("UIElements.ValueSelect.NumeralSelect",
	UIElements.ValueSelect.NumeralSelectBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */



		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		templatePath	: dojo.moduleUrl("UIElements.ValueSelect","templates/NumeralSelect8082.html")

	/* ------------------------------------ Constructor --------------------------------------- */

	/* ----------------------------------- Getter / Setter------------------------------------- */

	/* ----------------------------------- Public methods ------------------------------------- */

	/* --------------------------------- Private Methods -------------------------------------- */
	}
	);
/**
 * @class UIElements.ValueSelect.TimeSelectBase
 * @extends UIElements._base
 * Abstract Class - Do not use it directly
 */
dojo.provide("UIElements.ValueSelect.TimeSelectBase");
dojo.declare("UIElements.ValueSelect.TimeSelectBase",
	[UIElements._base, dijit._Templated],
	{

		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * hour value
		 * @property
		 * @type Number
		 */
		intHourValue			: 0,

		/**
		 * hour label
		 * @property
		 * @type String
		 */
		strHour					: null,

		/**
		 * minute value
		 * @property
		 * @type Number
		 */
		intMinuteValue			: 0,

		/**
		 * minute label
		 * @property
		 * @type String
		 */
		strMinute				: null,

		/**
		 * change hour
		 * @property
		 * @type Boolean
		 */
		boolChangeHour			: false, // Should the hour counter changer when we over/underflow minute
		/**
		 * time format
		 * @property
		 * @type Boolean
		 */
		is24hourFormat			: true,

		/**
		 * Callback called value change
		 * This property is set when instanciating the UIElement
		 * @property
		 * @type {Function}
		 */
		callback				: null,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		uiHour					: null,
		/**
		 * @ignore
		 */
		uiMinute				: null,
		/**
		 * @ignore
		 */
		uiAMPM				: null,
		/**
		 * @ignore
		 */
		domHour					: null,
		/**
		 * @ignore
		 */
		domMinute				: null,
		/**
		 * @ignore
		 */
		i18n					: "UIElements.ValueSelect",
		/**
		 * @ignore
		 */
		_boolMorning			: true,

		/* ------------------------------------ Constructor --------------------------------------- */

		constructor : function(args){
			if(args.strHour){
				this.strHour=args.strHour;
			}
			if(args.strMinute){
				this.strMinute=args.strMinute;
			}
		},

		postMixInProperties : function(){
			var min=1;
			var max=12;
			var value=this.intHourValue%12
			if(this.is24hourFormat){
				min=0;
				max=23;
				value=this.intHourValue;
			}
			else{
				value = ICTouchAPI.tools.get12HourFrom24(this.intHourValue).intHours;
			}
			this.uiHourParams = {
				intRangeMin	: min,
				intRangeMax	: max,
				intValue	: value,
				strFormat	: "%02d",
				intStep		: 1,
				callback	: this.callback
			};

			this.uiMinuteParams = {
				intRangeMin	: 0,
				intRangeMax	: 59,
				intValue	: this.intMinuteValue,
				strFormat	: "%02d",
				intStep		: 1,
				callback	: this.callback
			}
		},

		destroy: function() {
			if( this.uiHour ) {
				this.uiHour.destroy();
			}
			if( this.uiMinute ) {
				this.uiMinute.destroy();
			}
			this.inherited(arguments);
		},

		postCreate : function () {
			// Prepare parameters sent to hour rotate slider

			this.uiHour = new UIElements.ValueSelect.ScrollSelect(this.uiHourParams, this.domHour);
			this.uiMinute = new UIElements.ValueSelect.ScrollSelect(this.uiMinuteParams, this.domMinute);
			// need to know if we are in am or pm
			this._boolMorning = true;

			if(this.is24hourFormat && this.intHourValue  >= 12){
				this._boolMorning = false;
			}
			if(!this.is24hourFormat)
			{
				this._boolMorning = ICTouchAPI.tools.get12HourFrom24(this.intHourValue).boolIsMorning ? true : false;
				this._createAMPMUI();
			}
			delete this.uiHourParams;
			delete this.uiMinuteParams;

			this.setValue(this.intHourValue, this.intMinuteValue);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * Return the time value
		 * @return {Array} hour and minute
		 */

		getValue : function() {
			var hour = this.uiHour.intValue;
			var minute = this.uiMinute.intValue;
			if( !this.is24hourFormat) {
				hour = ICTouchAPI.tools.get24HourFrom12(hour, this._boolMorning);
			}
			return [hour, minute];
		},

		/**
		 * set value
		 * @param {Number} hour new hour
		 * @param {Number} minute new minute
		 */
		setValue : function(hour, minute) {
			this.uiMinute.attr("intValue", minute);
			if(this.is24hourFormat){
				this.uiHour.attr("intValue", hour);
			}
			else{
				var value = ICTouchAPI.tools.get12HourFrom24(hour);
				this.uiHour.attr("intValue", value.intHours);
				this._boolMorning = value.boolIsMorning;
				if(this.uiAMPM){
					this.uiAMPM.attr("intIndex", value.boolIsMorning ? 0 : 1);
				}
				else{
					this._createAMPMUI();
				}

			}
		},

		set24hourFormat : function(is24hourFormat){
			if(this.domAmPmSlider){
				if(is24hourFormat){
					dojo.addClass(this.domHalfDaySelect, "AMPMFormat");
					this.uiHour.setMinRangeValue(0);
					this.uiHour.setMaxRangeValue(23);
				}
				else{
					if(!this.uiAMPM){
						this._createAMPMUI();
					}
					dojo.removeClass(this.domHalfDaySelect, "AMPMFormat");
					this.uiHour.setMinRangeValue(1);
					this.uiHour.setMaxRangeValue(12);
				}
			}
			this.is24hourFormat = is24hourFormat;
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 */
		_createAMPMUI : function(){
			var _arrBool = [_("am", "ICTouchAPI"), _("pm", "ICTouchAPI")];
			this.uiAMPM = new UIElements.OptionChooser.OptionChooserControl({
				arrItems: _arrBool,
                intOptionWidth: 95,
				intIndex: this._boolMorning ? 0 : 1,
				funcCallback:dojo.hitch(this,this._halfDayChanged)
			}, this.domAmPmSlider);
		},

		/**
		 * @ignore
		 * Update the hour UI according to the format and the current moment in day
		 */
		_halfDayChanged: function() {
			this._boolMorning = !this._boolMorning;
			if(this.callback){
				this.callback();
			}
			}

	});
/**
 * @class UIElements.ValueSelect.TimeSelect
 * @extends UIElements.ValueSelect.TimeSelectBase
 * Association of two rotate sliders to define hours and minutes
 */
dojo.require("UIElements.ValueSelect.TimeSelectBase");
dojo.provide("UIElements.ValueSelect.TimeSelect");
dojo.declare("UIElements.ValueSelect.TimeSelect",
	UIElements.ValueSelect.TimeSelectBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * @ignore
		 */
		templatePath	: dojo.moduleUrl("UIElements.ValueSelect","templates/TimeSelect8082.html"),
		/**
		 * @ignore
		 */
		cssFile : ["base.css", "8082.css"]
	/* --------------------------------- Private attributes ----------------------------------- */

	/* ------------------------------------ Constructor --------------------------------------- */

	/* ----------------------------------- Getter / Setter------------------------------------- */

	/* ----------------------------------- Public methods ------------------------------------- */

	/* --------------------------------- Private Methods -------------------------------------- */
	}
	);
/**
* @class UIElements.Media.MediaBase
* @extends UIElements._base
* Abstract Class - Do not use it directly.
*/
dojo.provide("UIElements.Media.MediaBase");
dojo.declare("UIElements.Media.MediaBase",
	[UIElements._base, dojox.dtl._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Interval in ms between two calls of function update
		 * @property
		 * @type Number
		 */
		intUpdateTimeStep			: 250,
		/**
		 * Media duration in ms
		 * @property
		 * @type Number
		 */
		intMediaDuration			: 10000,

		/**
		 * Show or not Slider button
		 * @property
		 * @type Boolean
		 */
		boolShowSliderButton		: true,
		/**
		 * Show or not icon in Slider button
		 * @property
		 * @type Boolean
		 */
		boolIconOnSliderButton		: true,
		/**
		 * Enable or disable auto play
		 * @property
		 * @type Boolean
		 */
		boolAutoplay				: false,
		/**
		 * Enable or disable possiblity to pause
		 * @property
		 * @type Boolean
		 */
		boolCanPause				: true,
		/**
		 * Show or hide time slider
		 * @property
		 * @type Boolean
		*/
		boolShowSliderTime			: true,
		/**
		 * Enable or disable possiblity to seek file
		 * @property
		 * @type Boolean
		*/
		boolSeekable				: true,
		/**
		 * play callback
		 * @property
		 * @type Function
		*/
		clbkPlay					: null,
		/**
		 * pause callback
		 * @property
		 * @type Function
		 */
		clbkPause					: null,
		/**
		 * stop callback
		 * @property
		 * @type Function
		 */
		clbkStop					: null,
		/**
		 * Action to do after move on time slider
		 * @property
		 * @type String
		*/
		actionOnMoveEnd				: null,
		/**
		 * Auto pause or not after move on time slider
		 * @property
		 * @type Boolean
		*/
		autopauseOnMove				: true,
		/**
		 * Current position on time slider
		 * @property
		 * @type Number
		*/
		currentSliderPosition		: 0,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		_arrEvents					: [],
		/**
		 * @ignore
		 */
		_step 						: null,
		/**
		 * @ignore
		 */
		_boolPlaying				: true,
		/**
		 * @ignore
		 */
		_sliderMoved				: false,
		/**
		 * @ignore
		 * Start date in Milliseconds
		 */
		_startDateMs				: 0,

		/* ------------------------------------ Constructor --------------------------------------- */

		postCreate : function () {
			this._mediaInitialized();
			if(this.boolShowSliderButton) {
				// Add the media Class to the Slider to display the button
				this.slider.addClassOnButton("media");
				if(this.boolIconOnSliderButton) {
					dojo.addClass(this.slider.domButtonIcon, "onPause");
					dojo.removeClass(this.slider.domButtonIcon, "onPlay");
				}
			}
			// Place the slider
			dojo.style(this.slider.domNode, "left",  this.intSliderX  + "px");
			dojo.style(this.slider.domNode, "top",  this.intSliderY  + "px");			
		},

		destroy: function () {
			// Disconnect all events in the arrEvents array
			var h;
			while( h = this._arrEvents.pop() )
				dojo.disconnect(h);
			// Destroy slider
			if( this.slider )
				this.slider.destroy();
			this.inherited(arguments);

			// Clear timer
			if( this.intervalHandler )
				clearInterval(this.intervalHandler);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * Set the offset in s
		 * @param {Number} offset
		 */
		setToTime : function(offset) {
			this.currentSliderPosition = offset*1000;
			if (this.currentSliderPosition < this.intMediaDuration){
				this.slider.setValue(offset);
			}
			else {
				this.slider.setValue(this.intMediaDuration);
			}
		},

		/**
		 * Set a duration in ms
		 * @param {Number} duration
		 */
		setMediaDuration : function(intDuration) {
			this.intMediaDuration = intDuration;
			this._setStep();
			this.slider.setMax(intDuration/1000);
		},

		/**
		 * Return the sliding state
		 * @return {Boolean} return the sliding state (moving or not)
		 */
		isSliderMoved : function() {
			return this._sliderMoved;
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
         * Play or hold a media
         */
		mediaPlayPause : function() {
			if (!this._boolPlaying){
				if(this.clbkPlay) {
					this.clbkPlay();
				} else {
					this.mediaPlay();
				}
			}
			else{
				if (this.boolCanPause) {
					if(this.clbkPause) {
						this.clbkPause();
					} else {
						this.mediaPause();
					}
				}
			}
		},

		/**
         * Play
         */
		mediaPlay : function() {
			// Store a date to compute the duration on each iteration of setInterval
			this._startDateMs = (new Date()).getTime() - (this.currentSliderPosition || 0);
			//  Change the button to "Pause" status and start/continue playing the media
			if(this.boolShowSliderButton && this.boolIconOnSliderButton) {
				dojo.addClass(this.slider.domButtonIcon, "onPlay");
				dojo.removeClass(this.slider.domButtonIcon, "onPause");
			}
			//setInterval is a javascript function that calls the update function every X ms (defined by the intUpdateTimeStep param
			if(!this.intervalHandler) {
				var that = this;
				var func = function () {
					that._updateTime();
				}
				this.intervalHandler = setInterval( func, this.intUpdateTimeStep);
			}
			this._boolPlaying = true;
			this._sliderMoved = false;
		},

		/**
         * Hold
         */
		mediaPause : function() {
			// Change the button to "Play" status and  stop playing the media
			if(this.boolShowSliderButton && this.boolIconOnSliderButton) {
				dojo.addClass(this.slider.domButtonIcon, "onPause");
				dojo.removeClass(this.slider.domButtonIcon, "onPlay");
			}
			if( this.intervalHandler ){
				clearInterval(this.intervalHandler);
				this.intervalHandler = null;
			}
			this._boolPlaying = false;
			this._sliderMoved = false;
		},

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
         * @ignore
         */
		_mediaInitialized : function() {
			dojo.disconnect(this._arrEvents.pop());

			var min = 0;
			var max = this.intMediaDuration;
			this._setStep();
			this._sliderMoved = false;

			if(this.boolShowSliderButton) {
				// Media not seekable if streaming
				var params = {
					intMinimum: min,
					intMaximum: max/1000,
					intValue: this.currentSliderPosition/1000,
					intSliderWidth: this.intSliderWidth,
					boolSeekable : this.boolSeekable,
					boolShowTime: this.boolShowSliderTime,
				};
				var seekable = this.boolSeekable && ( this.intMediaDuration < Infinity );
				params.boolSeekable = seekable;
				this.slider = new UIElements.Slider.Linear(params, this.domSlider);
			}
			else {
				var params = {
					intMinimum: min,
					intMaximum: max/1000,
					intValue: this.currentSliderPosition/1000,
					intBarWidth: this.intSliderWidth,
					boolSeekable : this.boolSeekable,
					boolShowTime: this.boolShowSliderTime,
				};
				this.slider = new UIElements.ProgressBar.ProgressBarControl(params, this.domSlider);
			}

			
			// Enable to connect to the "buttonpressed" event of the ValueSlider to launch the  mediaPlayPause function
			var h;			
			if(seekable) {
				h = dojo.connect(this.slider, "buttonpressed", this, this.mediaPlayPause);
				this._arrEvents.push(h);
			}
			else {
				h = dojo.connect(this.slider, "slidermousedown", this, this._sliderMouseDown);
				this._arrEvents.push(h);
			}

			// Enable to connect to the "seekingstart" event of the ValueSlider to launch the   _mediaStartSeek function
			var h = dojo.connect(this.slider, "seekingstart", this, this._mediaStartSeek);
			this._arrEvents.push(h);

			// Enable to connect to the "seekingend" event of the ValueSlider to launch the   _mediaEndSeek function
			var h = dojo.connect(this.slider, "seekingend", this, this._mediaEndSeek);
			this._arrEvents.push(h);
		},

		_sliderMouseDown : function() {
			this.mediaPlayPause();
		},

		/**
         * @ignore
         */
		_updateTime : function() {
			if (this.currentSliderPosition < this.intMediaDuration){
				// Set the position according to the date and not to a step to avoid desync.
				this.currentSliderPosition = (new Date()).getTime() - this._startDateMs;
				this.slider.setValue(this.currentSliderPosition/1000);
			}			
			else if(this._boolPlaying) {
				this.mediaPause();
				if(this.clbkStop){
					this.clbkStop();
				}
			}
		},
		/**
         * @ignore
         */
		_mediaStartSeek : function() {
			if(this.autopauseOnMove){
				this.mediaPause();
			}
		},
		/**
         * @ignore
         */
		_mediaEndSeek : function() {
			this.currentSliderPosition = this.slider.getValue()*1000;
			if(!this._sliderMoved){
				this._sliderMoved = true;
				if(this.actionOnMoveEnd == "play") {
					if(this.clbkPlay && !this._boolPlaying) {
						this.clbkPlay();
					}
				} else if(this.actionOnMoveEnd == "pause") {
					if(this.clbkPause && this._boolPlaying) {
						this.clbkPause();
					}
				} else if(this.actionOnMoveEnd == "stop") {
					if(this.clbkStop) {
						this.clbkStop();
					}
				}
			}
		},
		/**
         * @ignore
         */
		_setStep : function() {
			this._step = this.intUpdateTimeStep;
		}

	}
	);
/**
* @class UIElements.Media.Media
* @extends UIElements.Media.MediaBase
* Media
*/
dojo.require("UIElements.Media.MediaBase");
dojo.provide("UIElements.Media.Media");
dojo.declare("UIElements.Media.Media",
	UIElements.Media.MediaBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * slider distance from the left
         * @property
         * @type Number
        */
		intSliderX		: 0,
		/**
		 * slider distance from the top
         * @property
         * @type Number
        */
		intSliderY		: 225,
        /**
		 * width of the slider
         * @property
         * @type Number
         */
		intSliderWidth	: 490,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		cssFile			: ["base.css", "8082.css"],

		/* ------------------------------------ Constructor --------------------------------------- */

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */

        /**
         * @ignore
         */
		_mediaInitialized : function() {
			this.inherited(arguments);
		}
	}
);
/**
* @class UIElements.Media.VideoBase
* @extends UIElements.Media.Media
* Abstract Class - Do not use it directly.
*/
dojo.require("UIElements.Media.Media");
dojo.provide("UIElements.Media.VideoBase");

dojo.declare("UIElements.Media.VideoBase",
	UIElements.Media.Media,
	{

		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/* ------------------------------------ Constructor --------------------------------------- */

		postCreate : function () {
			// Place the video
			dojo.style(this.domLoadVideo, "width", this.intVideoWidth + "px");
			dojo.style(this.domLoadVideo, "height", this.intVideoHeight + "px");
			dojo.style(this.domLoadVideo, "top", this.intVideoY + "px");
			dojo.style(this.domLoadVideo, "left", this.intVideoX + "px");

			this.inherited(arguments);
		}

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */

	}
);
/**
* @class UIElements.Media.Video
* @extends UIElements.Media.VideoBase
* Video
*/
dojo.require("UIElements.Media.VideoBase");
dojo.provide("UIElements.Media.Video");
dojo.declare("UIElements.Media.Video",
	UIElements.Media.VideoBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
        * intVideoWidth
        * Width of the video
        * @property
        * @type Number
        */
		intVideoWidth				: 450,
		/**
        * intVideoHeight
        * Height of the video
        * @property
        * @type Number
        */
		intVideoHeight				: 200,
		/**
        * intVideoX
        * X position
        * @property
        * @type Number
        */
		intVideoX					: 20,
		/**
        * intVideoY
        * Y position
        * @property
        * @type Number
        */
		intVideoY					: 5,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		templatePath				: dojo.moduleUrl("UIElements.Media","templates/Video8082.html"),

	/* ------------------------------------ Constructor --------------------------------------- */

	/* ----------------------------------- Getter / Setter------------------------------------- */

	/* ----------------------------------- Public methods ------------------------------------- */

	/* --------------------------------- Private Methods -------------------------------------- */

	}
	);
/**
 * @class UIElements.Media.AudioBase
 * @extends UIElements.Media.Media
 * Abstract Class - Do not use it directly.
 */
dojo.require("UIElements.Media.Media");
dojo.provide("UIElements.Media.AudioBase");
dojo.declare("UIElements.Media.AudioBase",
	UIElements.Media.Media,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/* ------------------------------------ Constructor --------------------------------------- */

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */
	}
);
/**
* @class UIElements.Media.Audio
* @extends UIElements.Media.AudioBase
* Audio media : mp3, ringing ...
*/


dojo.require("UIElements.Media.AudioBase");
dojo.provide("UIElements.Media.Audio");


dojo.declare("UIElements.Media.Audio",
	UIElements.Media.AudioBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		templatePath	: dojo.moduleUrl("UIElements.Media","templates/Audio8082.html")

	/* ------------------------------------ Constructor --------------------------------------- */

	/* ----------------------------------- Getter / Setter------------------------------------- */

	/* ----------------------------------- Public methods ------------------------------------- */

	/* --------------------------------- Private Methods -------------------------------------- */
	}
	);
/**
* @class UIElements.Media.PictureBase
* @extends UIElements._base
* Abstract Class - Do not use it directly.
*/
dojo.provide("UIElements.Media.PictureBase");
dojo.declare("UIElements.Media.PictureBase",
	[UIElements._base, dijit._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
        * Path of the media
        * Width of the video
        * @property
        * @type Number
        */
		strMediaPath		: ""

	/* --------------------------------- Private attributes ----------------------------------- */

	/* ------------------------------------ Constructor --------------------------------------- */

	/* ----------------------------------- Getter / Setter------------------------------------- */

	/* ----------------------------------- Public methods ------------------------------------- */

	/* --------------------------------- Private Methods -------------------------------------- */
	}
	);
/**
* @class UIElements.Media.Picture
* @extends UIElements.Media.PictureBase
* Pictures
*/
dojo.require("UIElements.Media.PictureBase");
dojo.provide("UIElements.Media.Picture");
dojo.declare("UIElements.Media.Picture",
	UIElements.Media.PictureBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		templatePath				: dojo.moduleUrl("UIElements.Media","templates/Picture8082.html")

	/* ------------------------------------ Constructor --------------------------------------- */

	/* ----------------------------------- Getter / Setter------------------------------------- */

	/* ----------------------------------- Public methods ------------------------------------- */

	/* --------------------------------- Private Methods -------------------------------------- */
	}
	);
/**
* @class UIElements.DialogBox.DialogBoxControlBase
* @extends UIElements._base
* @ignore
* Abstract Class<br>
* Element that can be displayed as an Popup or a Toaster. Unless there is a good reason don't use it directly.
*/
dojo.provide("UIElements.DialogBox.DialogBoxControlBase");

dojo.declare("UIElements.DialogBox.DialogBoxControlBase",
	[UIElements._base, dojox.dtl._Templated],
	{

		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Title of the popup.
		 * @property
		 * @type String
		 */
		strTitle					: "",
		/**
		 * Type of the popup. Will provide this element with a default title and icon.
		 * Recognized values are: info, warning, error and fatal_error.
		 * @property
		 * @type String
		 */
		strType						: "",
		/**
		 * Add an icon to the popup
		 * @property
		 * @type String
		 */
		strIcon						: "",
		/**
		 * Text that will be shown unless there is a widgetName.
		 * @property
		 * @type String
		 */
		strContent					: "",
		/**
		 * Instead of showing text, display an UI element.
		 * @property
		 * @type String
		 */
		strWidgetName				: "",
		/**
		 * Display the content verticaly or horizontaly,
		 * @property
		 * @type Bool
		 */
		boolHorizontalMode				: false,
		/**
		 * Parameters used to create the UI element.
		 * @property
		 * @type Object
		 */
		params						: {},

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @private
		 * Contain the reference to the inner widget
		 */
		_objWidget					: null,

		/* ------------------------------------ Constructor --------------------------------------- */

		postMixInProperties : function (){
			if (!this.strTitle) {
				switch(this.strType){
					case "info":
						this.strTitle = _("Information","UIElements.DialogBox");
						break;
					case "warning":
						this.strTitle = _("Warning","UIElements.DialogBox");
						break;
					case "error":
						this.strTitle = _("Error","UIElements.DialogBox");
						break;
					case "fatal_error":
						this.strTitle = _("Fatal error","UIElements.DialogBox");
						break;
					default:
						break;
				}
			}
		},

		postCreate : function () {			
			if (this.strIcon){
				var icon = this.strIcon + "-" + this._intIconSize;
				dojo.addClass(this.domPopupIcon, icon);
				dojo.addClass(this.domPopup, "withIcon");
			}
			else if (this.strType){
				dojo.addClass(this.domPopup, this.strType);
				dojo.addClass(this.domPopup, "withIcon");
				dojo.addClass(this.domPopup, "defaultIcon");
			}
			
			if (typeof this.strContent == "string" && this.strContent.length) {				
				dojo.create("p", {
					innerHTML: this.strContent,
					className: "textContent"
				}, this.domPopupContent, "last");
			}
			else if(typeof this.strContent == "object" && this.strContent.isI18Ned) {
				dojo.create("p", {
					innerHTML: this.strContent.getTranslation(),
					className: "textContent"
				}, this.domPopupContent, "last");
			}
			
			if (this.strWidgetName){
				var tempObj = ICTouchAPI.tools.getObjectReference(this.strWidgetName);
				var objParams = {
					params: this.params
				};
				for (var i in objParams.params) {
					objParams[i] = objParams.params[i];
				}
				this._objWidget = new tempObj(objParams, this.domPopupContent);
			//eval("new "+this.strWidgetName+"({},this.domPopupContent)");
			}
		},

		destroy : function () {
			if( this._objWidget ) {
				this._objWidget.destroy();
				delete this._objWidget;
			}
			this.inherited(arguments);
		},

		/* ------------------------------------ Getter/Setter ------------------------------------- */

		/**
		 * Get then instantiated widget
		 * @return {Widget} instantiated widget
		 */
		getContent : function () {
			return this._objWidget;
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		 * Place a new widget inside this element
		 * @param {Widget} objWidget that will be placed inside
		 */
		placeContent : function (objWidget) {
			this._objWidget = objWidget;
			this._objWidget.placeAt(this.domPopupContent);
		}

	/* --------------------------------- Private Methods -------------------------------------- */

	});
/**
* @class UIElements.DialogBox.DialogBoxControl
* @namespace UIElements.DialogBox
* @extends UIElements.DialogBox.DialogBoxControlBase
* Element that can be displayed as an Popup or a Toaster. Unless there is a good reason don't use it directly.
* @ignore
*/
dojo.require("UIElements.DialogBox.DialogBoxControlBase");
dojo.provide("UIElements.DialogBox.DialogBoxControl");

dojo.declare("UIElements.DialogBox.DialogBoxControl",
	UIElements.DialogBox.DialogBoxControlBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * For backward compatibility
		 * @property
		 * @type Array
		 */
		arrPopupButtons				: null,
		/**
		 * Array of params that will be passed to UIElements.AppButton.AppButton
		 * @property
		 * @type Array
		 */
		arrButtons					: null,
		/**
		 * Array
		 * @property
		 * @type Array
		 */
		domArrButtons				: [ ],
		/**
		 * Callback function when force Dialog box closing
		 * @property
		 * @type Function
		 */
		selfDestroyCallback : null,
		/**
		 * Boolean to know if a selfDestroyCallback can be used
		 * @property
		 * @type Boolean
		 */
		boolSelfDestroyCallback : false,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @private
		 */
		templatePath				: dojo.moduleUrl("UIElements.DialogBox","templates/DialogBox8082.html"),
		/**
		 * @private
		 */
		cssFile         		    : ["base.css", "8082.css", "icons.css"],
		/**
		 * @private
		 */
		_intIconSize				: 64,

		/* ------------------------------------ Constructor --------------------------------------- */

		constructor : function() {
			this.domArrButtons	= [ ];
			this.arrButtons = [ ];
		},

		postMixInProperties : function () {
			if( this.arrPopupButtons != null ) {
				this.arrButtons = this.arrPopupButtons;
			}
			if(this.selfDestroyCallback && this.arrButtons.length <=1){
				this.boolSelfDestroyCallback = true;
			}
			this.inherited(arguments);
		},

		postCreate : function () {
			var i;
			var nbButtons = this.arrButtons.length;
			for(i=0; i<nbButtons; ++i)
			{
				this.arrButtons[i].boolPopup = true;
				// Joel: add "parent" information in buttons, in order to facilitate "auto-close" popup functionnality
				this.arrButtons[i].objParent = this;
				this.arrButtons[i] = new UIElements.AppButton.AppButtonControl(this.arrButtons[i], this.domArrButtons[i]);
			}
			this.inherited(arguments);
			if (nbButtons > 0){
				dojo.addClass(this.domPopup, "withButtons");
			}
		},

		forceClose : function(){
			if (this.arrButtons.length == 1 && this.arrButtons[0].callback){
				this.arrButtons[0].callback();
			}
			if (this.selfDestroyCallback){
				this.selfDestroyCallback(this);
			}
			this.destroy();
		},

		destroy: function() {
			var nbButtons = this.arrButtons.length;
			for(var i=0; i<nbButtons; ++i)
			{
				if(this.arrButtons[i]){
				// Break the circular reference
					if(this.arrButtons[i].objParent){
						delete this.arrButtons[i].objParent;
					}
					this.arrButtons[i].destroy();
				}
				delete this.arrButtons[i];
			}
			this.inherited(arguments);
			
			dojo.publish( "dialogbox.hide", [] );
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		 * In case of a popup widget resize this element according to its content.
		 * And align it at the center of the screen
		 */
		show : function() {			
			var mb = dojo.marginBox(this.popupContainer);
			this.intPopupWidth = mb.w;
			this.intPopupHeight = mb.h;
			if (this.intPopupWidth){
				dojo.style(this.domPopup, "width", this.intPopupWidth + "px");
				dojo.style(this.domPopup, "marginLeft", "-" +(this.intPopupWidth/2) + "px");
			}
			if (this.intPopupHeight){
				dojo.style(this.domPopup, "height", this.intPopupHeight + "px");
				dojo.style(this.domPopup, "marginTop", "-" +(this.intPopupHeight/2) + "px");
			}
			
			dojo.publish( "dialogbox.show", [] );
		}

	/* --------------------------------- Private Methods -------------------------------------- */

	});/*
 * @class UIElements.ScrollPanel.ScrollPanelControlBase
 * @namespace UIElements.ScrollPanel
 * @ignore
 */
dojo.provide("UIElements.ScrollPanel.ScrollPanelControlBase");
dojo.declare("UIElements.ScrollPanel.ScrollPanelControlBase",
	[UIElements._base, dijit._Templated],
	{

		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		_objScroll: null,

		/* ------------------------------------ Constructor --------------------------------------- */

		postCreate : function () {
			// Parameter for scrollList not used in ScrollPanel
			this.boolHasScrollPanel = true;
			this.refresh(this);
		},

		destroy : function () {
			if( this._objScroll )
				this._objScroll.destroy();
			this.inherited(arguments);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/*
		 * is moving
		 */
		isMoving : function () {
			//return false;
			return this._objScroll.isMoving();
		},

		/*
		 * return the scroll object
		 */
		getScroll : function () {
			return this._objScroll;
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/*
		 * Refresh the scroll
		 * @param {Object} params : parameter for the scroll object refresh function
		 */
		refresh : function(params) {
			if( this._objScroll )
			{
				this._objScroll.refresh(params);
			}
			else
			{
				var listRoot = dojo.query(">", this.containerNode);
				listRoot = (listRoot.length > 1) ? this.containerNode : listRoot[0];
				this._objScroll = new UIEffects.Scroll.VerticalList(listRoot, params);
			}
		},

		/* --------------------------------- Private Methods -------------------------------------- */
	}
	);
/**
 * @class UIElements.ScrollPanel.ScrollPanelControlBase
 * @extends UIElements._base
 * @ignore
 * Abstract Class - Do not use it directly
 */
dojo.require("UIElements.ScrollPanel.ScrollPanelControlBase");
dojo.provide("UIElements.ScrollPanel.ScrollPanelControl");
dojo.declare("UIElements.ScrollPanel.ScrollPanelControl",
	UIElements.ScrollPanel.ScrollPanelControlBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/*
		 *@private
		 */
		templatePath				: dojo.moduleUrl("UIElements.ScrollPanel","templates/ScrollPanel8082.html"),

		/* ------------------------------------ Constructor --------------------------------------- */

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */

	}
);
/**
 * @class UIElements.Slider.LinearBase
 * @extends UIElements._base
 * Abstract Class - Do not use it directly
 */
dojo.provide("UIElements.Slider.LinearBase");
dojo.declare("UIElements.Slider.LinearBase",
	[UIElements._base, dijit._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * value
		 * @property
		 * @type Number
		 */
		intValue			: 0,
		/**
		 * minimum
		 * @property
		 * @type Number
		 */
		intMinimum			: 0,
		/**
		 * maximum
		 * @property
		 * @type Number
		 */
		intMaximum			: 100,
		/**
		 * Do we have to show the button ?
		 * @property
		 * @type Boolean
		 */
		boolShowButton		: true,
		/**
		 * Do we have to show the time ?
		 * @property
		 * @type String
		 */
		boolShowTime		: false,
		/**
		 * suffix
		 * @property
		 * @type String
		 */
		boolSeekable		: true,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		domBar				: null,
		/**
		 * @ignore
		 */
		domButtonContent	: null,
		/**
		 * @ignore
		 * Length of the slider
		 */
		_intRealWidth		: 0,
		/**
		 * @ignore
		 * marginBox of the slider
		 */
		_mbBar				: null,
		/**
		 * @ignore
		 * Pixel where the button begins
		 */
		_intStart			: 0,
		/**
		 * @ignore
		 * Pixel where the button ends
		 */
		_intEnd				: 0,
		/**
		 * @ignore
		 * Absolute value where the bar begins in x
		 */
		_intPageX			: 0,
		/**
		 * @ignore
		 */
		_handlerUp			: null,
		/**
		 * @ignore
		 */
		_handlerDown		: null,
		/**
		 * @ignore
		 */
		_handlerLeave		: null,
		/**
		 * @ignore
		 */
		_handlerMove		: null,
		/**
		 * @ignore
		 * Was already a mousemove event fired ?
		 */
		_boolFirstMove		: false,
		/**
		 * @ignore
		 * FrameId where this UI is contained
		 */
		_strFrameId			: null,

		/* ------------------------------------ Constructor --------------------------------------- */

		postMixInProperties : function() {
			this.intMinimum = Math.floor(this.intMinimum);
			this.intMaximum = Math.round(this.intMaximum);
		},

		postCreate: function() {
			ICTouchAPI.tools.removeWhitespaceNodes(this.domSlider);

			if(this.boolShowTime )
			{
				this.domPlayedTime.innerHTML = this._floatToTime(this.intValue);
				this.domTotalTime.innerHTML = (this.intMaximum)? this._floatToTime(this.intMaximum) : '';
			}
			if(!this.boolShowButton)
				dojo.style(this.domButton, "display", "none");

			this.setSliderWidth(this.intSliderWidth);
			this.setValue(this.intValue);
			this.subscribe("iframe.show", this._display);
		},

		destroy: function() {
			if( this._handlerMove != null )
				dojo.disconnect(this._handlerMove);
			this.inherited(arguments);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * Set the slider width
		 * @param {Number} width new width
		 */
		setSliderWidth: function(width) {
			this.intSliderWidth = width;
			if( this.boolShowTime )
			{
				var mbTotal = dojo.marginBox(this.domTotalTime);
				width -= mbTotal.w*2;
			}
			this._intRealWidth = width;
			dojo.style(this.domSlider, "width", width + "px");

			// Need to workaround a webkit bug where the width: inherit isn't applied when the div is outside of display
			var list = dojo.query(".applyWidth", this.domNode);
			list.style("width", width + "px");

			// These 2 values tweaks where the button start and end ( magic values )
			this._intEnd = width - 10;
			this._intStart = 17;

			// set value and update display
			this._updateDisplay();
		},

		/**
		 * Set the max
		 * @param {Number} max new max
		 */
		setMax : function(intMax) {
			this.intMaximum = intMax;
			if(this.boolShowTime ) {
				this.domTotalTime.innerHTML = (this.intMaximum)? this._floatToTime(this.intMaximum) : '';
			}
		},
		/**
		 * Set the value
		 * @param {Number} value new value
		 */
		setValue: function(value) {
			this.intValue = Math.max(this.intMinimum, Math.min(this.intMaximum, value)); // Clamp between min & max
			this._updateDisplay();
			this.valuechange();
		},
		/**
		 * get the value
		 * @return {Number} return the value
		 */
		getValue: function() {
			return this.intValue;
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		 * Add a class on button
		 * @param {String} buttonClass the path to the css class to be added to the button
		 */
		addClassOnButton: function(buttonClass) {
			dojo.addClass(this.domButtonContent, buttonClass);
		},

		/**
		 * Enable seek on the slider
		 * @param {Boolean} boolSeekable if true we can manually change the position on the slider
		 */
		setSeekable: function(boolSeekable) {
			this.boolSeekable = boolSeekable;
		},

		/**
		 * Event triggered when start seeking
		 */
		seekingstart : function() {
			dojo.addClass(this.domButtonContent, "sliding");
		},
		/**
		 * Event triggered when end seeking
		 */
		seekingend : function() {
			dojo.removeClass(this.domButtonContent, "sliding");
		},
		/**
		 * Event triggered when button pressed
		 */
		buttonpressed : function() {

		},
		/**
		 * Event triggered when slider pressed
		 */
		slidermousedown : function() {

		},
		/**
		 * Event triggered when slider released
		 */
		slidermouseup : function() {

		},
		/**
		 * Event triggered when value changed
		 */
		valuechange : function() {

		},

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 * When an iframe is shown reset the size so everything can be computed correctly
		 * This function is tricky because sometimes the styles aren't loaded when the UI is displayed, so we wait some more
		 */
		_display: function(strCurrentFrameId) {
			// Retrieve our frame's frameId
			if( this._strFrameId == null ) {
				this._strFrameId = ICTouchAPI.tools.getFrameId(this.domNode);
			}

			if( this._strFrameId == strCurrentFrameId ) {
				// It's our frame
				this.setSliderWidth(this.intSliderWidth);

				if( this.domSlider.scrollWidth == 0 || this.domSlider.scrollLeft == 0 ) {
					// Style didn't load yet when scrollWidth or scrollLeft equals 0, retry after 100ms
					var context = this;
					setTimeout(function(){
						context._display(strCurrentFrameId);
					}, 100);
				}
			}
		},

		/**
		 * @ignore
		 */
		_mouseDown: function(e) {
			this.slidermousedown();
			if (this.boolSeekable){
				dojo.stopEvent(e);
				this._intPageX = dojo.coords(this.domBar, true).x;
				// Disconnect any previous handler ( might happen when the cursor moves out of windows )
				if( this._handlerMove != null ) {
					dojo.disconnect(this._handlerMove);
				}
				this._handlerMove = dojo.connect(this.domNode, "mousemove", this, this._mouseMove);
				this._boolFirstMove = true;
			}
		},
		/**
		 * @ignore
		 */
		_mouseMove: function(e) {
			dojo.stopEvent(e);

			this._setValueAbsolute(e.pageX - this._intPageX);
			if( this._boolFirstMove ) {
				this.seekingstart();
				this._boolFirstMove = false;
			}
		},
		/**
		 * @ignore
		 */
		_mouseUp: function(e) {
			this.slidermouseup();
			if (this.boolSeekable){
				dojo.stopEvent(e);
				// If we didn't move while pushing the button then fire the "buttonpressed" event
				if (this._boolFirstMove){
					this.buttonpressed();
				}
				else{
					this.seekingend();
				}
				if( this._handlerMove != null ){
					dojo.disconnect(this._handlerMove);
					this._handlerMove = null;
				}
			}
		},
		/**
		 * @ignore
		 */
		_setValueAbsolute : function(pixelValue) {
			// Convert value from pixel to meaningful intValue
			this.intValue = ((pixelValue-this._intStart)*(this.intMaximum-this.intMinimum)/(this._intEnd-this._intStart))+this.intMinimum;
			this.setValue(this.intValue);
		},
		/**
		 * @ignore
		 */
		placeAt : function() {
			this.inherited(arguments);
			this.setSliderWidth(this.intSliderWidth);
		},
		/**
		 * @ignore
		 */
		_updateDisplay : function() {
			// Compute the left position of the button and progressBar
			var left = this._intStart+(this.intValue-this.intMinimum) * (this._intEnd-this._intStart)/(this.intMaximum-this.intMinimum);
			this.domSlider.scrollLeft = this._intRealWidth-left;

			if( this.boolShowTime )
				this.domPlayedTime.innerHTML = this._floatToTime(this.intValue);
		},
		/**
		 * @ignore
		 */
		_floatToTime : function(intValue) {
			var val = Math.round(intValue);
			var secs = val%60;
			var pad = secs < 10 ? '0' : '';
			return Math.floor(val/60) + ":" + pad + secs;
		},

		/**
		 * @ignore
		 */
		_feedback : function () {
		},
		/**
		 * @ignore
		 */
		refreshDisplay : function (){
			this._updateDisplay();
		}
	});
/**
 * @class UIElements.Slider.Linear
 * @extends UIElements.Slider.LinearBase
 * Displays a slider bar to change a value which can vary from a minimum value to a maximum value, linearly.
 */
dojo.provide("UIElements.Slider.Linear");
dojo.require("UIElements.Slider.LinearBase");
dojo.declare("UIElements.Slider.Linear",
	UIElements.Slider.LinearBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
	 * Slider width
	 * @property
	 * @type Number
	 */
		intSliderWidth		: 400,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
	 * @ignore
	 */
		templatePath		: dojo.moduleUrl("UIElements.Slider", "templates/Linear8082.html"),
		/**
	 * @ignore
	 */
		cssFile				: ["base.css", "8082.css"],
		/**
	 * @ignore
	 */
		_objFeedback		: null,

		/* ------------------------------------ Constructor --------------------------------------- */

		constructor : function() {
		//this._objFeedback = ICTouchAPI.feedbackServices.initFeedback("click");
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
	 * @ignore
	 */
		_feedback : function (event) {
		//this._objFeedback.feedback(event);
		}

	});

/**
 * @class UIElements.Slider.GraduateBase
 * @extends UIElements.Slider.Linear
 * Abstract Class - Do not use it directly
 */
dojo.require("UIElements.Slider.Linear");
dojo.provide("UIElements.Slider.GraduateBase");
dojo.declare("UIElements.Slider.GraduateBase",
	UIElements.Slider.Linear,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * show touch
		 * @property
		 * @type Boolean
		 */
		boolShowTouch		: false,
		/**
		 * show Label on the button
		 * @property
		 * @type Boolean
		 */
		boolShowButtonLabel	: true,
		/**
		 * maximum
		 * @property
		 * @type Number
		 */
		intMaximum			: 10,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		domGraduations		: null,

		/* ------------------------------------ Constructor --------------------------------------- */

		postCreate: function() {
			this._showGraduations();

			//add a class if the tooltip feature is enabled
			if (this.boolShowTouch)
				dojo.addClass(this.domNode, "showTooltip");

			this.inherited(arguments);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * Set the value
		 * @param {Number} value : new value
		 */
		setValue: function(value) {
			value = Math.round(value);
			this.inherited(arguments, [value]);
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 */
		_showGraduations: function() {
			var range = this.intMaximum - this.intMinimum + 1;
			for(var i=0; i<range; ++i)
			{
				dojo.create("td", {
					"class": "border"
				}, this.domGraduations, "last");
				dojo.create("td", { }, this.domGraduations, "last");
			}
		},
		/**
		 * @ignore
		 */
		_updateDisplay : function() {
			this.inherited(arguments);
			var toShow = this.getValue();
			if (this.boolShowButtonLabel){
			if( this.boolShowTouch )
				this.domButtonContent.innerHTML = "<em>" + toShow + "</em>" + "<span>" + toShow + "</span>";
			else
				this.domButtonContent.innerHTML = "<em>" + toShow + "</em>";
		}
		}
	});
/**
 * @class UIElements.Slider.Graduate
 * @extends UIElements.Slider.GraduateBase
 * Displays a slider bar to change a value which can vary from a minimum value to a maximum value, gradually.
 */
dojo.require("UIElements.Slider.GraduateBase");
dojo.provide("UIElements.Slider.Graduate");
dojo.declare("UIElements.Slider.Graduate",
	UIElements.Slider.GraduateBase,
	{

		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		templatePath		: dojo.moduleUrl("UIElements.Slider","templates/Graduate8082.html")

		/* ------------------------------------ Constructor --------------------------------------- */

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */
	}
);
/**
 * @class UIElements.Slider.RotateBase
 * @extends UIElements._base
 * Abstract Class - Do not use it directly
 */
dojo.provide("UIElements.Slider.RotateBase");
dojo.declare("UIElements.Slider.RotateBase",
	[UIElements._base, dijit._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * suffix
		 * @property
		 * @type String
		 */
		strSuffix	: "",
		/**
		 * prefix
		 * @property
		 * @type String
		 */
		strPrefix	: "",
		/**
		 * title
		 * @property
		 * @type String
		 */
		strTitle	: null,
		/**
		 * format
		 * @property
		 * @type String
		 */
		strFormat	: null,
		/**
		 * minimum
		 * @property
		 * @type Number
		 */
		intMinimum	: 0,
		/**
		 * Maximum
		 * @property
		 * @type Number
		 */
		intMaximum	: 23,
		/**
		 * value
		 * @property
		 * @type Number
		 */
		intValue	: 0,
		/**
		 * Is only used for display. Setting this attribute will force minimum to 0 and maximum to arrItems.length-1
		 * @property
		 * @type Array
		 */
		arrItems	: null,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		_floatValue		: 0,
		/**
		 * @ignore
		 */
		domTitle		: null,
		/**
		 * @ignore
		 */
		domValue		: null,
		/**
		 * @ignore
		 */
		domPrefix		: null,
		/**
		 * @ignore
		 */
		domSuffix		: null,
		/**
		 * @ignore
		 */
		domSprite		: null,
		/**
		 * @ignore
		 */
		domIncrement	: null,
		/**
		 * @ignore
		 */
		domDecrement	: null,
		/**
		 * @ignore
		 */
		attributeMap: {
			strTitle: {
				node: "domTitle",
				type: "innerHTML"
			},
			strSuffix: {
				node: "domSuffix",
				type: "innerHTML"
			},
			strPrefix: {
				node: "domPrefix",
				type: "innerHTML"
			}
		},
		/**
		 * @ignore
		 */
		_strBaseClass	: "",
		/**
		 * @ignore
		 */
		_objCoords		: null,
		/**
		 * @ignore
		 */
		_handlerMove	: null,
		/**
		 * @ignore
		 */
		_intRange		: 0,

		/**
		 * @ignore
		 * At which point are we in the outter ( scrollable ) zone ?
		 */
		_outerRadius	: 55,

		/**
		 * @ignore
		 * How many steps are in the sprite
		 */
		_intSprites			: 24,
		/**
		 * @ignore
		 * Half length of the spinner
		 */
		_intSpinnerLength	: 83,
		/**
		 * @ignore
		 * Distance from the middle where we doesn't start any action
		 */
		_intButtonDeadZone	: 15,

		/* ------------------------------------ Constructor --------------------------------------- */

		constructor : function(args){
			if(args.strTitle){
				this.strTitle=args.strTitle;
			}
		},

		postMixInProperties: function() {
			this._intRange = this.intMaximum - this.intMinimum + 1;
		},

		postCreate: function() {
			// Force first refresh
			var realValue = this.intValue;
			this.intValue = -1;
			this.attr('intValue', realValue);
			this.connect(this, "onMouseLeave", this._spinnerLeave);			
		},

		destroy: function() {
			if( this._handlerMove )
				dojo.disconnect(this._handlerMove);
			this.inherited(arguments);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		 * Increment the current value
		 * @param {Number} count Only used by typematic, shouldn't be used by anything else
		 */
		increment: function(count) {
			// When count < 0 it's to signal that the iteration has stopped
			if( count < 0 )
				return;

			// Bypass dijits attr function to enhance speed
			if( this.intValue == this.intMaximum ) {
				this._setIntValueAttr(this.intMinimum);
				this.overrun();
			}
			else {
				this._setIntValueAttr(this.intValue+1);
			}
		},

		/**
		 * Decrement the current value
		 * @param {Number} count Only used by typematic, shouldn't be used by anything else
		 */
		decrement: function(count) {
			// When count < 0 it's to signal that the iteration has stopped
			if( count < 0 )
				return;

			// Bypass dijits attr function to enhance speed
			if( this.intValue == this.intMinimum ) {
				this._setIntValueAttr(this.intMaximum);
				this.underrun();
			}
			else {
				this._setIntValueAttr(this.intValue-1);
			}
		},

		/**
		 * Event triggered when we cycle from the upper bound to the lower bound
		 */
		overrun: function() {  },

		/**
		 * Event triggered when we cycle from the lower bound to upper bound
		 */
		underrun: function() {  },

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 * intValue setter
		 */
		_setIntValueAttr: function(value) {
			if( value == this.intValue ) {
				return;
			}
			// Clamp value
			this.intValue = Math.min(this.intMaximum, Math.max(this.intMinimum, value));
			// Display the content of arrItems if available
			if( this.arrItems != null ) {
				this.domValue.innerHTML = this.arrItems[this.intValue];
			}
			// Only use sprintf when necessary
			else if( typeof this.strFormat == "string" ) {
				this.domValue.innerHTML = dojox.string.sprintf(this.strFormat, this.intValue);
			}
			// Default display method
			else {
				this.domValue.innerHTML = this.intValue;
			}

			// Change the value to the internal representation
			this._floatValue = (this.intValue - this.intMinimum) / this._intRange;
			this._refresh();
		},

		/**
		 * @ignore
		 * You must set intValue for this to take effect
		 */
		_setArrItemsAttr: function(items) {
			if( items != null ) {
				this.intMinimum = 0;
				this.intMaximum = items.length-1;
				this._intRange = this.intMaximum - this.intMinimum + 1;
			}

			this.arrItems = items;
		},

		/**
		 * @ignore
		 * You must set intValue for this to take effect
		 */
		_setIntMinimumAttr: function(minimum) {
			if( this.arrItems == null ) {
				this.intMinimum = minimum;
				this._intRange = this.intMaximum - this.intMinimum + 1;
			}
		},

		/**
		 * @ignore
		 * You must set intValue for this to take effect
		 */
		_setIntMaximumAttr: function(maximum) {
			if( this.arrItems == null ) {
				this.intMaximum = maximum;
				this._intRange = this.intMaximum - this.intMinimum + 1;
			}
		},

		/**
		 * @ignore
		 * Update the position of the button and change sprite ( normalize should only be used when the range is changed )
		 */
		_refresh: function(normalizeFloatValue) {
			if( normalizeFloatValue ) {
				var value = this._floatValue*this._intRange;
				this._floatValue = Math.round(value)/this._intRange;
			}

			if( this._strBaseClass == "" ) {
				this._strBaseClass = this.domSprite.className;
			}
			this.domSprite.className = this._strBaseClass + " sprite-"+Math.round(this._floatValue*this._intSprites);
		},

		/**
		 * @ignore
		 * This will call the function dijit.typematic.trigger when we aren't in the dead zone
		 * If y > 0 then it's increment, else it's decrement
		 */
		_triggerTypematic: function(event) {
			var y = event.pageY - this._objCoords.y - this._intSpinnerLength;
			if( Math.abs(y) < this._intButtonDeadZone )
				return;
			if( y < 0 )
				dijit.typematic.trigger(event, this, this.domIncrement, this.increment, this.domIncrement, 0.85);
			else
				dijit.typematic.trigger(event, this, this.domDecrement, this.decrement, this.domDecrement, 0.85);
		},

		/**
		 * @ignore
		 * Event handler when a mouse button is pushed
		 */
		_spinnerDown: function(event) {
			dojo.stopEvent(event);
			this._objCoords = dojo.coords(this.domSprite.parentNode, false);
			// If we are in the outer radius start scrolling else the click will be passed down to the buttons
			if( this._spinnerMove(event, true) )
				this._handlerMove = dojo.connect(this.domNode, "mousemove", this, this._spinnerMove);
			else
				this._triggerTypematic(event);
		},

		/**
		 * @ignore
		 * Event handler when mouse is moved
		 */
		_spinnerMove: function(event, first) {
			var x = event.pageX - this._objCoords.x - this._intSpinnerLength;
			var y = event.pageY - this._objCoords.y - this._intSpinnerLength;

			var rad = Math.sqrt(x*x + y*y);

			/*
			 * If this is called the first time during a scroll ( on mouseDown ) and we are in the inner radius of the scroll,
			 * do nothing and return false to show it to _spinnerDown
			 */
			if( first && rad < this._outerRadius )
				return false;

			x /= rad;

			var angle = y < 0 ? -Math.acos(x) : Math.acos(x);

			// Save the last value for the overrun / underrun test
			var lastValue = this._floatValue;

			this._floatValue = angle / ( 2 * Math.PI ) + 0.25;
			if( this._floatValue < 0 )
				this._floatValue += 1.0;

			// Bypass dijit's attr function and directly call the setter
			this._setIntValueAttr(Math.round(this._floatValue*this._intRange)+this.intMinimum);

			if( lastValue >= 0.75 && this._floatValue < 0.25 ) {
				this.overrun();
			}

			if( lastValue < 0.25 && this._floatValue >= 0.75 ) {
				this.underrun();
			}

			return true;
		},

		/**
		 * @ignore
		 * Event handler when mouse leaves the UIElement
		 */
		_spinnerLeave: function(event) {
			this._spinnerUp(event);
		},

		/**
		 * @ignore
		 * Event handler when mouse button is released or the mouse leaves the UIElement
		 */
		_spinnerUp: function(event) {
			dojo.stopEvent(event);
			if( this._handlerMove ) {
				dojo.disconnect(this._handlerMove);
				this._handlerMove = null;
				this._objCoords = null;
			}
			else {
				dijit.typematic.stop();
			}
		}		
	});

/**
 * @class UIElements.Slider.Rotate
 * @extends UIElements.Slider.RotateBase
 * Displays a rotate slider bar to change a value which can vary from a minimum value to a maximum value.
 */
dojo.provide("UIElements.Slider.Rotate");
dojo.require("UIElements.Slider.RotateBase");
dojo.declare("UIElements.Slider.Rotate",
	UIElements.Slider.RotateBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		templatePath		: dojo.moduleUrl("UIElements.Slider", "templates/Rotate8082.html"),
		/**
		 * @ignore
		 */
		cssFile				: "rotate8082.css"

	/* ------------------------------------ Constructor --------------------------------------- */

	/* ----------------------------------- Getter / Setter------------------------------------- */

	/* ----------------------------------- Public methods ------------------------------------- */

	/* --------------------------------- Private Methods -------------------------------------- */

	});
/**
* @class UIElements.OptionChooser.OptionChooserControlBase
* @extends UIElements._base
* Abstract Class - Do not use it directly
*/
dojo.provide("UIElements.OptionChooser.OptionChooserControlBase");

dojo.declare("UIElements.OptionChooser.OptionChooserControlBase",
	[UIElements._base, dojox.dtl._Templated],
	{

		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Array of string that form the available options
		 * @property
		 * @type Array
		 */
		arrItems			: null,

		/**
		 * Index of selected option
		 * @property
		 * @type Number
		 */
		intIndex			: 0, // Can be changed with ui.attr('intIndex', newValue);

		/**
		 * Width of an individual option in px
		 * @property
		 * @type Number
		 */
		intOptionWidth		: 76, // Can be changed with ui.attr('intOptionWidth', newValue); (80px if no padding)

		/**
		 * Function called when the value change
		 * @property
		 * @type Function
		 */
		funcCallback		: null,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		_funcCallbackFunc	: null,
		/**
		 * @ignore
		 */
		_boolActive			: false,
		/**
		 * @ignore
		 */
		_arrRow				: null,
		/**
		 * @ignore
		 */
		_intStart			: -1,
		/**
		 * @ignore
		 */
		domLabel				: null,

		/* ------------------------------------ Constructor --------------------------------------- */

		constructor: function() {
			this._arrRow = [];
			this.domLabel = [];
		},

		postMixInProperties : function() {
			if(this.arrItems) {
				for(var i=0; i<this.arrItems.length; ++i) {
					var obj = this.arrItems[i];
					var handler = ICTouchAPI.i18nServices.i18nList([obj], this._swapLang(i));
					ICTouchAPI.i18nServices.addI18nHandlers(this, handler);
				}
			}
		},

		postCreate: function() {
			if(!generalConfig.simulation) {
				if(this.arrItems) {
					for(var i=0; i<this.arrItems.length; ++i) {
						dojo.toggleClass(this._arrRow[i], "active", ( i === this.intIndex ));
						this._arrRow[i].intIndex = i;
						if (this.arrItems[i].isI18Ned) {
							this.domLabel[i].innerHTML = this.arrItems[i].getTranslation();
						}
						else {
							this.domLabel[i].innerHTML = this.arrItems[i];
						}
					}
				}
				this._setIntOptionWidthAttr(this.intOptionWidth);
				// This cannot be done with a dojoAttachEvent
				this.connect(this, "onMouseLeave", this._finish);
			}
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * method for changing the UIElement's value
		 * @param {Number} newValue This will change the main selected item
		 */
		setValue: function (newValue) {
			this._setIntIndexAttr(newValue);
		},

		/**
		 * get the current selected index
		 * @return {Number} the current selected item
		 */
		getValue: function() {
			return this.intIndex;
		},

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 */
		_mousedown: function(event) {
			this._intStart = this.intIndex;
			var newIndex = event.currentTarget.intIndex;
			this._setIntIndexAttr(newIndex);
			// Mark this as active
			this._boolActive = true;
		},

		/*
		 * This function must be removed when the new scroll ( with axis constrait ) will be implemented
		 * Do not forget to remove all reference to this
		 * @ignore
		 */
		_fixScroll: function() {
			if(this._containingScroll == undefined){
				var currentNode = this.domNode;
				while(!this._containingScroll && currentNode){
					if(currentNode.IctScroll){
						this._containingScroll = currentNode.IctScroll;
					}
					currentNode = currentNode.parentNode;
				}
			}
			if(this._containingScroll){
				this._containingScroll.cancelEvent = false;
			}else{
				this._containingScroll = false;
			}
		},

		/**
		 * @ignore
		 */
		_mousemove: function(event) {
			// Ignore the event if it isn't active
			if( this._boolActive ) {
				var newIndex = event.currentTarget.intIndex;
				this._setIntIndexAttr(newIndex);
				// This remove the capture feature of the scroll
				this._fixScroll();
			}
		},

		/**
		 * @ignore
		 * Callback when the user stop using the option chooser
		 */
		_finish: function(event) {
			if( this._boolActive && this._intStart != this.intIndex ) {
				// When we stop with the OptionChooser launch callback

				if( this._funcCallbackFunc )
					this._funcCallbackFunc(this.intIndex);
				this._feedback(event);
			}
			//crms00247940 : mousemove function was called before mouse down,
			//because of the boolActive value kept at "true" the intIndex was always changed.
			this._boolActive = false;

		},

		/**
		 * @ignore
		 */
		_setIntIndexAttr: function(newIndex) {
			if( newIndex != this.intIndex ) {
				dojo.removeClass(this._arrRow[this.intIndex], "active");
				dojo.addClass(this._arrRow[newIndex], "active");
				this.intIndex = newIndex;
			}
		},

		/**
		 * @ignore
		 */
		_setIntOptionWidthAttr: function(newWidth) {
			var i, len, realWidth, totalWidth = 0;
			len = this._arrRow.length;
			for(i=0; i<len; ++i) {
				// -3px ( border: 2px, margin: 1px ) but last item that doesn't have any margin
				realWidth = i === len-1 ? newWidth - 3 : newWidth - 2;
				dojo.style(this._arrRow[i], "width", realWidth + "px");
				totalWidth += realWidth+3;
			}
			dojo.style(this.domNode, "width", totalWidth+"px");
			this.intOptionWidth = newWidth;
		},

		/**
		 * @ignore
		 */
		_feedback: function() {

		},

		/**
		 * @ignore
		 */
		_setFuncCallbackAttr : function(callback) {
			this._funcCallbackFunc = ICTouchAPI.tools.callbackToHitch(callback);
		},
		
		_swapLang : function (intItem) {
			var context = this;
			return function (name, value) {
				if (context.domLabel && context.domLabel[intItem]) {
					return context.domLabel[intItem].innerHTML = value;
				} else {
					return false;
				}
			}
		}

	});
/**
* @class UIElements.OptionChooser.OptionChooserControl
* @extends UIElements.OptionChooser.OptionChooserControlBase
* This UIelement displays a slider with a dragable button allowing the user to modify the switch value (chosen between several predefined values)
*/
dojo.require("UIElements.OptionChooser.OptionChooserControlBase");
dojo.provide("UIElements.OptionChooser.OptionChooserControl");

dojo.declare("UIElements.OptionChooser.OptionChooserControl",
	UIElements.OptionChooser.OptionChooserControlBase,
	{
		/* --------------------------------- Public attributes ----------------------------------- */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		templatePath		: dojo.moduleUrl("UIElements.OptionChooser", "templates/OptionChooser8082.html"),
		/**
		 * @ignore
		 */
		cssFile				: ["base.css", "8082.css"],
		/**
		 * @ignore
		 */
		_objFeedback		: null,

		/* ------------------------------------ Constructor --------------------------------------- */
		constructor : function() {
		//this._objFeedback = ICTouchAPI.feedbackServices.initFeedback("click");
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* --------------------------------- Public Methods -------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 */
		_feedback: function(event) {
		//this._objFeedback.feedback(event);
		}
	}
	);
/**
 * @class UIElements.Spinner.SpinnerControlBase
 * @extends UIElements._base
 * Abstract Class - Do not use it directly
 */
dojo.provide("UIElements.Spinner.SpinnerControlBase");
dojo.declare("UIElements.Spinner.SpinnerControlBase",
	[UIElements._base, dijit._Templated],
	{

		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Message to be displayed, override default spinner message
		 * @property
		 * @type String
		 */
		strMessage			: "Loading...", // Override default spinner message

		/**
		 * Should the spinner be over the content
		 * @property
		 * @type Boolean
		 */
		boolForeground		: false,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		domRotating			: null,
		/**
		 * @ignore
		 */
		_i18n				: "UIElements.Spinner",
		/**
		 * @ignore
		 */
		_boolRotating       : false,
		/**
		 * @ignore
		 */
		_intRotatingSpeed   : 0.25,
		/**
		 * @ignore
		 */
		_intInterval        : 100,
		/**
		 * @ignore
		 */
		_handlerInterval    : null,
		/**
		 * @ignore
		 */
		_intAngle       : 0,
		/**
		 * @ignore
		 */
		attributeMap: {
			strMessage: {
				node: "domMessage",
				type: "innerHTML"
			}
		},

		/* ------------------------------------ Constructor --------------------------------------- */

		constructor: function() {
                    if(!generalConfig.simulation) {
                         // Translate the default message
                        this.strMessage = _(this.strMessage, this._i18n);
                    }
		},

		postMixInProperties: function() {
			this.srcNodeRef = dojo.create("div", { }, this.srcNodeRef);
		},

		postCreate: function() {
			if( this._boolRotating ) {
				var that = this;
				var func = function() {
					that._rotateSpinner();
				}
				this._handlerInterval = setInterval(func, this._intInterval);
				this._rotateSpinner();
			}
			dojo.toggleClass(this.domNode, "foreground", this.boolForeground);
		},

		destroy: function() {
			if( this._handlerInterval ) {
				clearInterval(this._handlerInterval);
			}
			if (this.domRotating) {
				this.domRotating.parentNode.removeChild(this.domRotating);
			}
			this.inherited(arguments);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 */
		_rotateSpinner: function() {
			var angle = this._intAngle;
			this.domRotating.style['-webkit-transform'] = "rotate(" + angle + "deg)";
			this._intAngle = angle+20;
		},

	});
/**
 * @class UIElements.Spinner.SpinnerControl
 * @extends UIElements.Spinner.SpinnerControlBase
 * Loading widget
 */
dojo.require("UIElements.Spinner.SpinnerControlBase");
dojo.provide("UIElements.Spinner.SpinnerControl");
dojo.declare("UIElements.Spinner.SpinnerControl",
	UIElements.Spinner.SpinnerControlBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * @ignore
		 */
		templatePath	: dojo.moduleUrl("UIElements.Spinner", "templates/SpinnerControl8082.html"),
		/**
		 * @ignore
		 */
		cssFile         : ["base.css", "8082.css"]

	/* --------------------------------- Private attributes ----------------------------------- */

	/* ------------------------------------ Constructor --------------------------------------- */

	/* ----------------------------------- Getter / Setter------------------------------------- */

	/* ----------------------------------- Public methods ------------------------------------- */

	/* --------------------------------- Private Methods -------------------------------------- */

	});/**
* @class UIElements.EmptyState.EmptyStateControlBase
* @extends UIElements._base
* @extends  dijit._Templated
* Abstract class - Do not use it directly
*/
dojo.provide("UIElements.EmptyState.EmptyStateControlBase");
dojo.declare("UIElements.EmptyState.EmptyStateControlBase",
    [UIElements._base, dijit._Templated],
    {

	/* --------------------------------- Public attributes ------------------------------------ */
	
		/**
		 * Display or not the EmptyItem
		 * @cfg {Boolean} boolVisible
		 */
		boolVisible: false,
		
		/**
		 * Message displayed on screen
		 * @cfg {String} strEmptyMessage
		 */
		strEmptyMessage: _("Empty", "ICTouchAPI"),
		
		/**
		 * Show or not the EmptyItem border
		 * @cfg {Boolean} boolBackground
		 */
		boolBackground: true,
		
		/**
		 * Show or not the icon
		 * @cfg {Boolean} boolIcon
		 */
		boolIcon: true,

	/* --------------------------------- Private attributes ----------------------------------- */
	
		domIcon: null,
		domText: null,
		attributeMap: {
			strEmptyMessage: {
				node: "domText",
				type: "innerHTML"
			}
		},

	/* ------------------------------------ Constructor --------------------------------------- */
		
		postCreate: function() {
			this.setVisibility(this.boolVisible);
			dojo.toggleClass(this.domNode, "setBackground", this.boolBackground);
			dojo.toggleClass(this.domNode, "setIcon", this.boolIcon);
		},
	
	/* ----------------------------------- Getter / Setter------------------------------------- */
	
		/**
                 * Change the visibility of the EmptyItem
                 * @param {Boolean} visible True to display the EmptyItem
                 */
                setVisibility: function(visible) {
			this.boolVisible = visible;
			dojo.toggleClass(this.domNode, "visible", this.boolVisible);
		},

                /**
                 * Change the text displayed on the EmptyItem
                 * @param {String} text The text to display
                 */
		setEmptyText: function(text) {
			this.attr("strEmptyMessage", text);
		}

	/* ----------------------------------- Public methods ------------------------------------- */

	/* --------------------------------- Private Methods -------------------------------------- */
    }
);
/**
* @class UIElements.EmptyState.EmptyStateControl
* @extends UIElements.EmptyState.EmptyStateControlBase
*
* This class is the graphical representation of an empty item.
*
* An EmptyItem is an item that can be displayed when there is nothing to display (no item in a list...).
*
* An EmptyItem displays a text (that can be modified), an icon (not customizable) and a border. The icon and the border can be set or not.
*
* Here is an example of creating an EmptyItem in a container.
*
*     postCreate : function() {
*
*          // Create the container
*          var container = new UIElements.Container.ContainerControl({
*               title: {
*                   strLabel: "Empty Item"
*               }
*               content: {
*                   name:"UIElements.EmptyItem.EmptyItemControl",
*                   params:{
*                       boolVisible: true,
*                       strEmptyMessage: "No result",
*                       boolBackground: true,
*                       boolIcon: true
*                   }
*               }
*          }, this.emptyItemDomNode);
*     },
*
* Don't forget to add the attachPoint in the template associated to the view
*     <div dojoAttachPoint="emptyItemDomNode" />
*
*/
dojo.require("UIElements.EmptyState.EmptyStateControlBase");
dojo.provide("UIElements.EmptyState.EmptyStateControl");
dojo.declare("UIElements.EmptyState.EmptyStateControl",
    UIElements.EmptyState.EmptyStateControlBase,
    {
/* --------------------------------- Public attributes ------------------------------------ */

        templatePath			: dojo.moduleUrl("UIElements.EmptyState","templates/EmptyState8082.html")

/* --------------------------------- Private attributes ----------------------------------- */

/* ------------------------------------ Constructor --------------------------------------- */

/* ----------------------------------- Getter / Setter------------------------------------- */

/* ----------------------------------- Public methods ------------------------------------- */

/* --------------------------------- Private Methods -------------------------------------- */
    }
);
/**
 * @class UIElements.NavigationBar.NavigationBarBase
 * @namespace UIElements.NavigationBar
 * @extends UIElements._base
 * Abstract Class<br>
 */
dojo.provide("UIElements.NavigationBar.NavigationBarBase");
dojo.declare("UIElements.NavigationBar.NavigationBarBase",
	[UIElements._base,
	dijit._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */


		

		/* --------------------------------- Private attributes ----------------------------------- */

		domArrListPage : null,
		intMaxPage : 1,
		intCurrentPage : 1,
		displayCallBack : null,
		domList : null,
		domArrowLeft : null,
		domArrowRight : null,

		/* ------------------------------------ Constructor --------------------------------------- */

		constructor : function(){
			this.domArrListPage = [];
		},

		postMixInProperties : function(){

		},

		postCreate : function(){
			if(!generalConfig.simulation) {
				this._updatePageList();
				this._updateArrows();
			}
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */




		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		 * Go to specific page and update navigation bar items
		 * @param {number} intPageNumber the new current page number
		 */
		goToPage : function (intPageNumber){
			var page = null;

			// Update current page (and correct if not existant)
			if (intPageNumber.currentTarget && intPageNumber.currentTarget.attributes.getNamedItem('position').value) {
				// Automatic case if we want to authorize a click on each icon, not used for the moment
				page = parseInt(intPageNumber.currentTarget.attributes.getNamedItem('position').value);
			} else if (typeof(intPageNumber)=='number' && parseInt(intPageNumber)== intPageNumber) {
				page = intPageNumber;
			} else {
				page = parseInt(this.intCurrentPage);
			}

			if(typeof page != "number" || page == NaN){
				ICTouchAPI.debugServices.error("UIElements.NavigationBar.NavigationBarBase - goToPage / The page number entered is not a number : "+intPageNumber);
				return;
			}

			if(page != this.intCurrentPage){

				// Remove current class to last page and add normal class instead
				dojo.removeClass(this.domArrListPage[this.intCurrentPage-1], "current");
				dojo.addClass(this.domArrListPage[this.intCurrentPage-1], "normal");

				if (page>this.intMaxPage) {
					this.intCurrentPage = parseInt(this.intMaxPage);
				} else if (page<1) {
					this.intCurrentPage = 1;
				} else {
					this.intCurrentPage = parseInt(page);
				}

				// Remove normal class to new current page and add current class instead
				dojo.removeClass(this.domArrListPage[this.intCurrentPage-1], "normal");
				dojo.addClass(this.domArrListPage[this.intCurrentPage-1], "current");

				// Show selected page
				this.displayCallBack(this.intCurrentPage);

				this._updateArrows();
			} else {
				ICTouchAPI.debugServices.warning("UIElements.NavigationBar.NavigationBarBase - goToPage / The page number was already the same : "+page);
			}
		},

		/**
		 * Change the number of pages and update the icons display.
		 * @param {Number} intMaxPage the new number of pages
		 */
		setMaxPage : function(intMaxPage) {
			if(this.intMaxPage != intMaxPage){
				var newMaxPage = parseInt(intMaxPage);
				if(newMaxPage != NaN){
					this.intMaxPage = parseInt(intMaxPage);

					// Check that the selected page is not higher than the new maxPage
					if (this.intCurrentPage>this.intMaxPage) {
						this.intCurrentPage = parseInt(this.intMaxPage);
					}

					this._updatePageList();

					this._updateArrows();
				} else {
					ICTouchAPI.debugServices.error("UIElements.NavigationBar.NavigationBarBase - setMaxPage / The number of pages entered is not a number : "+intMaxPage);
				}
			} else {
				ICTouchAPI.debugServices.warning("UIElements.NavigationBar.NavigationBarBase - setMaxPage / The number of pages was already the same : "+intMaxPage);
			}
		},

		/**
		 * Go to previous page
		 */
		previous : function() {
			this.goToPage(parseInt(this.intCurrentPage)-1);
		},

		/**
		 * Go to next page
		 */
		next : function() {
			this.goToPage(parseInt(this.intCurrentPage)+1);
		},

		/* --------------------------------- Private Methods -------------------------------------- */

		_updatePageList : function() {
			// Update navigation bar : list of pages
			dojo.empty(this.domList);
			this.domArrListPage = [];

			for (var i=1;i<=this.intMaxPage;i++) {
				var li = dojo.create("li", {"class":"item", dojoAttachPoint:"domPageItem"+i, position:i}, this.domList);
				if (i == 1) {
						dojo.addClass(li, "home");
				}
				if (i == this.intCurrentPage) {
					dojo.addClass(li, "current");
				} else {
					dojo.addClass(li, "normal");
				}
				this.domArrListPage.push(li);
			}
		},

		_updateArrows : function() {
			// Update navigation bar : hide or show arrows
			if (this.intCurrentPage <= 1) {
				dojo.addClass(this.domArrowLeft, "hide");
			} else {
				dojo.removeClass(this.domArrowLeft, "hide");
			}
			if (this.intCurrentPage >= this.intMaxPage) {
				dojo.addClass(this.domArrowRight, "hide");
			} else {
				dojo.removeClass(this.domArrowRight, "hide");
			}
		}

	});
/**
 * @class UIElements.NavigationBar.NavigationBar
 * @namespace UIElements.NavigationBar
 * @extends UIElements.NavigationBar.NavigationBarBase
 * @link UIElements.NavigationBar.NavigationBarBase
 */
dojo.require("UIElements.NavigationBar.NavigationBarBase");
dojo.provide("UIElements.NavigationBar.NavigationBar");
dojo.declare("UIElements.NavigationBar.NavigationBar",
	UIElements.NavigationBar.NavigationBarBase,
	{

		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 * The path to the template is defined here
		 */
		templatePath: dojo.moduleUrl("UIElements.NavigationBar","templates/NavigationBar8082.html"),
		/**
		 * @ignore
		 */
		cssFile    : ["base.css","8082.css"]

		/* ------------------------------------ Constructor --------------------------------------- */



		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */



		/* --------------------------------- Private Methods -------------------------------------- */




	});
/**
 * @class UIElements.Video.VideoControlBase
 * @namespace UIElements.Video
 * @extends UIElements._base
 * @ignore
 * Abstract Class<br>
 * Video is a UIElements placeholder for live video display. Actual video pixels
 * are overlaid on the screen at lower level (QT)
 */
dojo.provide("UIElements.Video.VideoControlBase");

dojo.declare("UIElements.Video.VideoControlBase",
	[UIElements._base, dijit._Templated],
	{
		/* --------------------------------- Private attributes ----------------------------------- */

		/**
         * @private
         */
		_strBaseClass			: "Video", // Base class of this UI
		_strCallerDeclaredClass	: null,
		_handlerUIDisplayed		: null,
		sipCallId: -1,

		/* --------------------------------- Public attributes ------------------------------------ */

		/**
         * @public
         */

		callId: -1,

		pip: 1,


		/**
         * @property
         * @type Array
         */
		attributeMap: {
		},

		/* ------------------------------------ Constructor --------------------------------------- */

		constructor : function(args) {
		},

		// Function called before displaying the output (allows you to check variables)
		postMixInProperties : function() {
		},


		/* ----------------------------------- Public methods ------------------------------------- */

		showVideoContent : function () {
			ICTouchAPI.debugServices.info('UIElements.Video.VideoControlBase - showVideoContent');
			var context = this;
			setTimeout(function() {
				/* Read current CSS properties */
				context.setStyle();

				if( context.callId >= 0 ) {
					ICTouchAPI.debugServices.debug('UIElements.Video.VideoControlBase - showVideoContent / call VideoServices.show for callId: ' + context.callId);
					ICTouchAPI.VideoServices.show( {
						callId:context.callId,
						sipCallId : context.sipCallId,
						callerUI: context.id,
						callerScreen: context._strCallerDeclaredClass,
						pip: context.pip
					} );
				} else {
					ICTouchAPI.debugServices.debug('UIElements.Video.VideoControlBase - showVideoContent / no callId provided, call VideoServices.showLocal');
					ICTouchAPI.VideoServices.showLocal( {
						callerUI: context.id,
						callerScreen: context._strCallerDeclaredClass
					} );
				}
			}, 200);
		},

		hideVideoContent : function () {
			ICTouchAPI.debugServices.info('UIElements.Video.VideoControlBase - hideVideoContent');
			ICTouchAPI.VideoServices.hide();
		},


		getCoordinate : function() {
			var objCoordinate = {};
			// Get the VideoContent div domNode
			var contentDomNode = this.domNode; //dojo.query('div',this.domNode)[0];
			// Get the relative position and size of the VideoContent div
			var clientRect = contentDomNode.getBoundingClientRect();
			// width rounded to a multiple of 8
			objCoordinate.w = clientRect.width & 0xFFF8;
			// center the window horizontally
			objCoordinate.x = clientRect.left + ((clientRect.width - objCoordinate.w) >> 1);
			// height rounded to a multiple of 2
			objCoordinate.h = clientRect.height & 0xFFFE;
			objCoordinate.y = clientRect.top;
			return objCoordinate;
		},

		displayError: function() {
			dojo.addClass( this.domNode, "Error");
		},

		displayVideo: function() {
			dojo.removeClass( this.domNode, "Error");
		},

		/* --------------------------------- Private Methods -------------------------------------- */
		/**
         * @private
         */
		postCreate : function () {
			ICTouchAPI.debugServices.info('UIElements.Video.VideoControlBase - postCreate');
			this._strCallerDeclaredClass = this._getCallerDeclaredClass(this);
		},

		startup : function () {
			ICTouchAPI.debugServices.info('UIElements.Video.VideoControlBase - startup');
			if (ICTouchAPI.transitionServices.getCurrentScreenName() === this._strCallerDeclaredClass) {
				ICTouchAPI.debugServices.debug('UIElements.Video.VideoControlBase - startup / view ' + this._strCallerDeclaredClass + ' is displayed, call showVideoContent');
				this.showVideoContent();
			}
			else {
				ICTouchAPI.debugServices.debug('UIElements.Video.VideoControlBase - startup / view ' + this._strCallerDeclaredClass + ' is not displayed, subscribe to WEBAPP_SHOW and call showVideoContent when the view will be displayed');
				this._handlerUIDisplayed = dojo.subscribe( "ICTouchAPI.transitionServices.WEBAPP_SHOW", this, function (evt) {
					if (evt.id === this._strCallerDeclaredClass) {
						ICTouchAPI.debugServices.debug('UIElements.Video.VideoControlBase - startup / view ' + this._strCallerDeclaredClass + ' is now displayed, call showVideoContent');
						dojo.unsubscribe(this._handlerUIDisplayed);
						var context = this;
						setTimeout(function(){
							context.showVideoContent();
						}, 0);
					}
				});
			}
		},

		destroy : function (destroyVideo) {
			ICTouchAPI.debugServices.info('UIElements.Video.VideoControlBase - destroy');
			if (this._handlerUIDisplayed) {
				dojo.unsubscribe(this._handlerUIDisplayed);
			}
			if(undefined === destroyVideo){
				destroyVideo = false;
			}
			ICTouchAPI.debugServices.debug('UIElements.Video.VideoControlBase / destroy - callId: '+this.callId);
			if (this.callId >= 0) {
				if(destroyVideo){
					ICTouchAPI.debugServices.debug('UIElements.Video.VideoControlBase - destroy / call VideoServices.destroy');
					ICTouchAPI.VideoServices.destroy( this.callId );
					this.callId = -1;
				} else {
					ICTouchAPI.debugServices.debug('UIElements.Video.VideoControlBase - destroy / call VideoServices.hide');
					// Video is hidden but not stopped (ex: held, hold, transfer, etc...)
					ICTouchAPI.VideoServices.hide({
						callId: this.callId,
						videoStableState: 2
					}); // stable state = 2 for _VIDEO_HIDDEN
				}
			} else {
				if(destroyVideo){
					ICTouchAPI.VideoServices.destroy( undefined, this._strCallerDeclaredClass);
				}
				else {
					ICTouchAPI.debugServices.debug('UIElements.Video.VideoControlBase - destroy / hide video default');
					ICTouchAPI.VideoServices.hide();
				}
			}
			this.inherited(arguments);
		},

		getStyle : function() {
			var bgRGBString = dojo.getComputedStyle(this.domNode).backgroundColor,
			textRGBString = dojo.getComputedStyle(this.domNode).color;

			var bgRGBColor = /^rgb\((\d{1,3}),\s*(\d{1,3}),\s*(\d{1,3})\)$/.exec( bgRGBString ),
			textRGBColor = /^rgb\((\d{1,3}),\s*(\d{1,3}),\s*(\d{1,3})\)$/.exec( textRGBString );

			if( bgRGBColor && undefined !== bgRGBColor[1] && textRGBColor && undefined !== textRGBColor[1] )
			{
				var color = [ parseInt(bgRGBColor[1]), parseInt(bgRGBColor[2]), parseInt(bgRGBColor[3]), 0 ];
				var tColor = [ parseInt(textRGBColor[1]), parseInt(textRGBColor[2]), parseInt(textRGBColor[3]), 0 ];

				var bgColor = color[2]|color[1]<<8|color[0]<<16|color[3]<<24,
				textColor = tColor[2]|tColor[1]<<8|tColor[0]<<16|tColor[3]<<24,
				mainMessage = ( this.callId >= 0 ) ? _("waitingVideo","UIElements.Video") : "",//"Waiting for the video of the other party..." : "", // _('ERROR_DIR_MAX_REACHED', "webapp.communication"), // "Waiting for the video of the other party..."
				pipMessage = ( this.callId >= 0 ) ? _("pipMessage","UIElements.Video") : ""; // _('ERROR_DIR_MAX_REACHED', "webapp.communication");  // "Tap here to broadcast or mute your video"

				/*  */
				ICTouchAPI.VideoServices.setParams( bgColor, mainMessage, textColor, pipMessage );
			}

		},
        setStyle : function() {
			ICTouchAPI.debugServices.info('UIElements.Video.VideoControlBase - setStyle');
			var params = this.getStyle();
			if (params) {
				ICTouchAPI.VideoServices.setParams( params.bgColor, params.mainMessage, params.textColor, params.pipMessage );
            }
		},

		_getCallerDeclaredClass : function(context) {
			var object = dijit.getEnclosingWidget(context.domNode.parentNode);
			if (object !== null) {
				return this._getCallerDeclaredClass(object);
			}
			else {
				return context.declaredClass;
			}
		}

	}
	);
dojo.require("UIElements.Video.VideoControlBase");
dojo.provide("UIElements.Video.VideoControl");
// This class is related to:
//- dijit._Widget: the basic framework for each Widget
//- dojox.dtl._Templated: enable to use the django syntax in the template layout
dojo.declare("UIElements.Video.VideoControl", UIElements.Video.VideoControlBase, {
	// The path to the template is defined here
	templatePath				: dojo.moduleUrl("UIElements.Video","templates/Video8082.html"),
	cssFile                     : ["base.css", "8082.css"]
});
/**
 * @class UIElements.SideTray.SideTrayControlBase
 * @extends UIElements._base
 * @ignore
 * Abstract Class - Do not use it directly
 */
dojo.provide("UIElements.SideTray.SideTrayControlBase");
dojo.declare("UIElements.SideTray.SideTrayControlBase",
	[UIElements._base, dijit._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		domArrow     : null,
		domContent	 : null,
		domSTContent : null,
		isTrayOpen	 : false,
		domAppBar	 : null,
		appBarUi	 : null,
		objVideoButton : null,
		videoButtonUi : null,
		
		
		/* ------------------------------------ Constructor --------------------------------------- */

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */


		constructor : function(){
	
		},
		
		postCreate : function(){
			this.buildAppBar();
			//this.domSTContent.style.display = 'none';

			this.openTray();
		},
		
		/**
		 * Hide a given node
		 * @param {HTMLElement} node node to be hidden
		 */
		hide : function () {
			dojo.toggleClass(this.domArrow,"generic-arrow-right-32", false);
			dojo.toggleClass(this.domArrow,"generic-arrow-left-32", true);
		},
		/**
		 * Show a given node
		 * @param {HTMLElement} node shown to be hidden
		 */
		switchState : function() {
			var that = this;
			setTimeout(function(){that.updateState();},15);
		},
		
		updateState : function() {
			if(this.isTrayOpen) {
				this.closeTray();
			} else {
				this.openTray();
			}
		},
		
		openTray : function() {
			this.domSTContent.style.display = 'block';
			this.isTrayOpen = true;
			this.domArrow.className = "Arrow generic-arrow-right-32";
		},
		
		closeTray : function() {
			this.domSTContent.style.display = 'none';
			this.isTrayOpen = false;
			this.domArrow.className = "Arrow generic-arrow-left-32";
		},
		
		buildAppBar: function() {
			//Create the appbar
			this.appBarUi = new UIElements.AppBar.AppBarWidget({
				webAppName:"communication",
				widgetName:"videoSideTray"
			});
			//Create the video button
			this.videoButtonUi = new UIElements.AppButton.AppButtonControl(this.objVideoButton);
			//Add it to the appbar
			this.appBarUi.addActionButton(this.videoButtonUi);
			
			//Place the appBar in the dom
			this.appBarUi.placeAt(this.domAppBar);
		},
		
		destroy: function() {

			this.inherited(arguments);
		}

	/* --------------------------------- Private Methods -------------------------------------- */
	}
	);
/**
 * @class UIElements.SideTray.SideTrayControl
 * @extends UIElements.SideTray.SideTrayControlBase
 * @ignore
 * Displays a retractable panel allowing to display content
 */
dojo.require("UIElements.SideTray.SideTrayControlBase");
dojo.provide("UIElements.SideTray.SideTrayControl");
dojo.declare("UIElements.SideTray.SideTrayControl",
	UIElements.SideTray.SideTrayControlBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @private
		 */
		templatePath	: dojo.moduleUrl("UIElements.SideTray","templates/SideTray8082.html"),
		/**
		 * @private
		 */
		cssFile			: ["base.css", "8082.css"]

		/* ------------------------------------ Constructor --------------------------------------- */

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */
	}
);
