/**
 * @class UIElements.Slider.LinearBase
 * @extends UIElements._base
 * Abstract Class - Do not use it directly
 */
dojo.provide("UIElements.Slider.LinearBase");
dojo.declare("UIElements.Slider.LinearBase",
	[UIElements._base, dijit._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * value
		 * @property
		 * @type Number
		 */
		intValue			: 0,
		/**
		 * minimum
		 * @property
		 * @type Number
		 */
		intMinimum			: 0,
		/**
		 * maximum
		 * @property
		 * @type Number
		 */
		intMaximum			: 100,
		/**
		 * Do we have to show the button ?
		 * @property
		 * @type Boolean
		 */
		boolShowButton		: true,
		/**
		 * Do we have to show the time ?
		 * @property
		 * @type String
		 */
		boolShowTime		: false,
		/**
		 * suffix
		 * @property
		 * @type String
		 */
		boolSeekable		: true,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		domBar				: null,
		/**
		 * @ignore
		 */
		domButtonContent	: null,
		/**
		 * @ignore
		 * Length of the slider
		 */
		_intRealWidth		: 0,
		/**
		 * @ignore
		 * marginBox of the slider
		 */
		_mbBar				: null,
		/**
		 * @ignore
		 * Pixel where the button begins
		 */
		_intStart			: 0,
		/**
		 * @ignore
		 * Pixel where the button ends
		 */
		_intEnd				: 0,
		/**
		 * @ignore
		 * Absolute value where the bar begins in x
		 */
		_intPageX			: 0,
		/**
		 * @ignore
		 */
		_handlerUp			: null,
		/**
		 * @ignore
		 */
		_handlerDown		: null,
		/**
		 * @ignore
		 */
		_handlerLeave		: null,
		/**
		 * @ignore
		 */
		_handlerMove		: null,
		/**
		 * @ignore
		 * Was already a mousemove event fired ?
		 */
		_boolFirstMove		: false,
		/**
		 * @ignore
		 * FrameId where this UI is contained
		 */
		_strFrameId			: null,

		/* ------------------------------------ Constructor --------------------------------------- */

		postMixInProperties : function() {
			this.intMinimum = Math.floor(this.intMinimum);
			this.intMaximum = Math.round(this.intMaximum);
		},

		postCreate: function() {
			ICTouchAPI.tools.removeWhitespaceNodes(this.domSlider);

			if(this.boolShowTime )
			{
				this.domPlayedTime.innerHTML = this._floatToTime(this.intValue);
				this.domTotalTime.innerHTML = (this.intMaximum)? this._floatToTime(this.intMaximum) : '';
			}
			if(!this.boolShowButton)
				dojo.style(this.domButton, "display", "none");

			this.setSliderWidth(this.intSliderWidth);
			this.setValue(this.intValue);
			this.subscribe("iframe.show", this._display);
		},

		destroy: function() {
			if( this._handlerMove != null )
				dojo.disconnect(this._handlerMove);
			this.inherited(arguments);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * Set the slider width
		 * @param {Number} width new width
		 */
		setSliderWidth: function(width) {
			this.intSliderWidth = width;
			if( this.boolShowTime )
			{
				var mbTotal = dojo.marginBox(this.domTotalTime);
				width -= mbTotal.w*2;
			}
			this._intRealWidth = width;
			dojo.style(this.domSlider, "width", width + "px");

			// Need to workaround a webkit bug where the width: inherit isn't applied when the div is outside of display
			var list = dojo.query(".applyWidth", this.domNode);
			list.style("width", width + "px");

			// These 2 values tweaks where the button start and end ( magic values )
			this._intEnd = width - 10;
			this._intStart = 17;

			// set value and update display
			this._updateDisplay();
		},

		/**
		 * Set the max
		 * @param {Number} max new max
		 */
		setMax : function(intMax) {
			this.intMaximum = intMax;
			if(this.boolShowTime ) {
				this.domTotalTime.innerHTML = (this.intMaximum)? this._floatToTime(this.intMaximum) : '';
			}
		},
		/**
		 * Set the value
		 * @param {Number} value new value
		 */
		setValue: function(value) {
			this.intValue = Math.max(this.intMinimum, Math.min(this.intMaximum, value)); // Clamp between min & max
			this._updateDisplay();
			this.valuechange();
		},
		/**
		 * get the value
		 * @return {Number} return the value
		 */
		getValue: function() {
			return this.intValue;
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		 * Add a class on button
		 * @param {String} buttonClass the path to the css class to be added to the button
		 */
		addClassOnButton: function(buttonClass) {
			dojo.addClass(this.domButtonContent, buttonClass);
		},

		/**
		 * Enable seek on the slider
		 * @param {Boolean} boolSeekable if true we can manually change the position on the slider
		 */
		setSeekable: function(boolSeekable) {
			this.boolSeekable = boolSeekable;
		},

		/**
		 * Event triggered when start seeking
		 */
		seekingstart : function() {
			dojo.addClass(this.domButtonContent, "sliding");
		},
		/**
		 * Event triggered when end seeking
		 */
		seekingend : function() {
			dojo.removeClass(this.domButtonContent, "sliding");
		},
		/**
		 * Event triggered when button pressed
		 */
		buttonpressed : function() {

		},
		/**
		 * Event triggered when slider pressed
		 */
		slidermousedown : function() {

		},
		/**
		 * Event triggered when slider released
		 */
		slidermouseup : function() {

		},
		/**
		 * Event triggered when value changed
		 */
		valuechange : function() {

		},

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 * When an iframe is shown reset the size so everything can be computed correctly
		 * This function is tricky because sometimes the styles aren't loaded when the UI is displayed, so we wait some more
		 */
		_display: function(strCurrentFrameId) {
			// Retrieve our frame's frameId
			if( this._strFrameId == null ) {
				this._strFrameId = ICTouchAPI.tools.getFrameId(this.domNode);
			}

			if( this._strFrameId == strCurrentFrameId ) {
				// It's our frame
				this.setSliderWidth(this.intSliderWidth);

				if( this.domSlider.scrollWidth == 0 || this.domSlider.scrollLeft == 0 ) {
					// Style didn't load yet when scrollWidth or scrollLeft equals 0, retry after 100ms
					var context = this;
					setTimeout(function(){
						context._display(strCurrentFrameId);
					}, 100);
				}
			}
		},

		/**
		 * @ignore
		 */
		_mouseDown: function(e) {
			this.slidermousedown();
			if (this.boolSeekable){
				dojo.stopEvent(e);
				this._intPageX = dojo.coords(this.domBar, true).x;
				// Disconnect any previous handler ( might happen when the cursor moves out of windows )
				if( this._handlerMove != null ) {
					dojo.disconnect(this._handlerMove);
				}
				this._handlerMove = dojo.connect(this.domNode, "mousemove", this, this._mouseMove);
				this._boolFirstMove = true;
			}
		},
		/**
		 * @ignore
		 */
		_mouseMove: function(e) {
			dojo.stopEvent(e);

			this._setValueAbsolute(e.pageX - this._intPageX);
			if( this._boolFirstMove ) {
				this.seekingstart();
				this._boolFirstMove = false;
			}
		},
		/**
		 * @ignore
		 */
		_mouseUp: function(e) {
			this.slidermouseup();
			if (this.boolSeekable){
				dojo.stopEvent(e);
				// If we didn't move while pushing the button then fire the "buttonpressed" event
				if (this._boolFirstMove){
					this.buttonpressed();
				}
				else{
					this.seekingend();
				}
				if( this._handlerMove != null ){
					dojo.disconnect(this._handlerMove);
					this._handlerMove = null;
				}
			}
		},
		/**
		 * @ignore
		 */
		_setValueAbsolute : function(pixelValue) {
			// Convert value from pixel to meaningful intValue
			this.intValue = ((pixelValue-this._intStart)*(this.intMaximum-this.intMinimum)/(this._intEnd-this._intStart))+this.intMinimum;
			this.setValue(this.intValue);
		},
		/**
		 * @ignore
		 */
		placeAt : function() {
			this.inherited(arguments);
			this.setSliderWidth(this.intSliderWidth);
		},
		/**
		 * @ignore
		 */
		_updateDisplay : function() {
			// Compute the left position of the button and progressBar
			var left = this._intStart+(this.intValue-this.intMinimum) * (this._intEnd-this._intStart)/(this.intMaximum-this.intMinimum);
			this.domSlider.scrollLeft = this._intRealWidth-left;

			if( this.boolShowTime )
				this.domPlayedTime.innerHTML = this._floatToTime(this.intValue);
		},
		/**
		 * @ignore
		 */
		_floatToTime : function(intValue) {
			var val = Math.round(intValue);
			var secs = val%60;
			var pad = secs < 10 ? '0' : '';
			return Math.floor(val/60) + ":" + pad + secs;
		},

		/**
		 * @ignore
		 */
		_feedback : function () {
		},
		/**
		 * @ignore
		 */
		refreshDisplay : function (){
			this._updateDisplay();
		}
	});
