/**
 * @class UIElements.SelectList.SelectListControlBase
 * @extends UIElements._base
 * Abstract Class - Do not use it directly
 */
dojo.provide("UIElements.SelectList.SelectListControlBase");
dojo.declare("UIElements.SelectList.SelectListControlBase",
	[UIElements._base, dijit._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Items
		 * @property
		 * @type Array
		 */
		arrItems                    : [ ],
		/**
		 * enable multi-selection
		 * @property
		 * @type Boolean
		 */
		boolMultiselect				: false,
		/**
		 * select limit
		 * @property
		 * @type Number
		 */
		intSelectLimit				: -1,
		/**
		 * enable scroll
		 * @property
		 * @type Boolean
		 */
		boolScroll					: true,
		/**
		 * Display an empty state when this list is empty
		 * @property
		 * @type {Boolean}
		 */
		showEmptyState: false,

		/**
		 * Empty state text, null show the default text from EmptyState
		 * @property
		 * @type {String}
		 */
		emptyText: null,



		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		scroll						: null,
		/**
		 * @ignore
		 * UL that will contain every item that is displayed
		 */
		domList						: null,
		/**
		 * @ignore
		 */
		_itemCallbackFunc			: null,
		/**
		 * @ignore
		 */
		_objFeedback				: null,
		/**
		 * @ignore
		 * Contain the instantiated SelectListItem
		 */
		_arrListItem				: null,
		/**
		 * @ignore
		 * Max items on screen
		 */
		_intMaxItems				: 8,
		/**
		 * @ignore
		 * Contain index of items currently selected
		 */
		_arrSelected				: null,
		/**
		 * @ignore
		 * Index of first element shown
		 */
		_intStart					: 0,
		/**
		 * @ignore
		 * Index of last element shown
		 */
		_intEnd						: 0,
		/**
		 * @ignore
		 */
		itemCallback				: null,
		/**
		 * @ignore
		 */
		buttonCallback				: null,
		/**
		 * @ignore
		 */
		buttonCallbackParams		: {},
		/**
		 * EmptyState object
		 * @ignore
		 */
		_emptyState : null,
		/**
		 * true if we must wait for highlighting, false otherwise
		 * @ignore
		 * @type {Boolean}
		 */
		_enableWaitingHighlight : true,


		/* ------------------------------------ Constructor --------------------------------------- */

		constructor	: function(){
			this.arrItems		= [ ];
			this._arrListItem	= [ ];
			this._arrSelected	= [ ];
		},

		postCreate : function () {
			// Precreate every item
			for(var i=0; i<this.arrItems.length; ++i)
			{
				var item = new UIElements.SelectList.SelectListItem(this.arrItems[i]);
				if(item.isSelected()) {
					this._arrSelected.push(i);
				}
				if(item.isEnabled()){
					item.setClickCallback(this, this._clicked, i);
					item.setActionCallback(this, this._applyAction, i);
				}
				this._arrListItem.push(item);
				// Place the first _intMaxItems into the dom
				if( i < this._intMaxItems ) {
					item.placeAt(this.domList);
				}
			}
			this._intEnd = Math.min(this.arrItems.length, this._intMaxItems)-1;
			if (!this.boolMultiselect){
				dojo.addClass(this.domNode, "SimpleSelect");
			}
			if (this.boolScroll && this.arrItems.length){
				this._initScroll();
			}

			if (this.arrItems.length <= 0 && this.showEmptyState) {
				var params = {
					boolVisible: true
				};
				if (this.emptyText) {
					params.strEmptyMessage = this.emptyText;
				}
				this._emptyState = new UIElements.EmptyState.EmptyStateControl(params);
				this._emptyState.placeAt(this.domNode, "first");
			}
		},

		/**
		 * Destroy
		 */
		destroy : function() {
			var objItem;
			while( (objItem = this._arrListItem.pop()) != null ) {
				objItem.destroy();
			}
			delete this._arrSelected;

			this.inherited(arguments);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * return a table containing the index of every selected item
		 * @return {Array} Index of every selected item
		 */
		getSelected: function() {
			// Return a clone so the application can modify our version
			return dojo.clone(this._arrSelected);
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		 * paginate a given UI
		 * @param {HTMLElement} element UI element to be paginated
                 * @ignore
		 * @return {Function} function
		 */
		pagination : function (element) {
			// If the list is scrolled more than 50 pixels
			if (this.domNode.scrollTop >= 50) {
				// Push an item at the end of the list and remove the first one.
				// Return the amount of pixels to "unscroll"
				return -50 * this.push(1);
			// If the list is scrolled to the top most
			} else if (!this.domNode.scrollTop) {
				// Add an item at the top of the list and remove the last one
				return 50 *this.unshift(1);
			} else {
				// If nothing changed, return 0.
				return 0;
			}
		},

		/**
		 * Remove the first selectable dom element and add a new one at the end of the list.
		 * Elements come from the private liste of items
		 * @return {Boolean} true if an element has been pushed, else return false
		 */
		push : function() {
			var node;
			// Do nothing if we are already at the end
			if( this._intEnd >= this._arrListItem.length-1 ) {
				return 0;
			}
			node = this._arrListItem[this._intStart].domNode;
			this.domList.removeChild(node);
			this._intStart++;
			this._intEnd++;
			node = this._arrListItem[this._intEnd].domNode;
			dojo.place(node, this.domList, "last");
			return 1;
		},

		/**
		 * Remove the last selectable dom element and add a new one at the beginning of the list.
		 * Elements come from the private liste of items
		 * @return {Boolean} true if an element has been unshifted, else return false
		 */
		unshift : function() {
			var node;
			// Do nothing if we are already at the end
			if( this._intStart <= 0 ) {
				return 0;
			}
			node = this._arrListItem[this._intEnd].domNode;
			this.domList.removeChild(node);
			this._intStart--;
			this._intEnd--;
			node = this._arrListItem[this._intStart].domNode;
			dojo.place(node, this.domList, "first");
			return 1;
		},



		/**
		 * Select every item up to the limit defined by intSelectLimit
		 */
		selectAll: function() {
			var len = this._arrListItem.length;
			if( this.intSelectLimit > -1 ) {
				len = Math.min(len, this.intSelectLimit);
			}
			for (var i = 0; i<len; ++i) {
				this._arrListItem[i].select();
				this._arrSelected.push(i);
			}
		},

		/**
		 * If possible select the item
		 * @param  {Number} intIndex Index of the item in the arrItems list
		 * @return {Boolean} did the item get selected
		 */
		selectAtIndex: function(intIndex) {
			// Did we reach any limit
			var canCheck = this.intSelectLimit < 0 || this._arrSelected.length < this.intSelectLimit;
			if( intIndex >= 0 && intIndex < this._arrListItem.length && canCheck ) {
				this._arrListItem[intIndex].select();
				this._arrSelected.push(intIndex);
				return true;
			}
			return false;
		},

		/**
		 * If the item is selected, unselect it
		 * @param {Number} intIndex Index of the item in the arrItems list
		 * @return {Boolean} was the item selected
		 */
		unselectAtIndex: function(intIndex) {
			var selectedIndex = -1;
			for(var i in this._arrSelected){
				if(this._arrSelected[i]==intIndex){
					selectedIndex = i;
				}
			}
			//var selectedIndex = this._arrSelected.indexOf(intIndex);
			if( selectedIndex > -1 ) {
				this._arrListItem[intIndex].unselect();
				this._arrSelected.splice(selectedIndex, 1);
				return true;
			}
			return false;
		},

		/**
		* Unselect every item unconditionally
		*/
		unselectAll: function() {
			var intIndex;
			while( (intIndex = this._arrSelected.pop()) != null ) {
				this._arrListItem[intIndex].unselect();
			}
		},

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 */
		_initScroll: function() {
			var objScrollParams = {
				"handler" : this.domNode,
				"easing" : true,
				"scrollType" : "vertical"
			};
			// If we have more items than _intMaxItems then start the pagination mechanism
			if( this._arrListItem.length > this._intMaxItems ) {
				objScrollParams.pagination = {
					"callback" : this.pagination,
					"context" : this
				}
			}

                        /*
                        if(this._arrListItem.length < 6) {
                            objScrollParams.bounce = false;
                        }
                        */

			// Add the scrolling functionality
			this.scroll = FXScroll.init(objScrollParams);
		},

		/**
		 * @ignore
		 */
		_setItemCallbackAttr : function(callback) {
			this._itemCallbackFunc = ICTouchAPI.tools.callbackToHitch(callback);
		},

		/**
		 * @ignore
		 */
		_clicked : function(intItem) {
			// Get the clicked item
			var objItem = this._arrListItem[intItem];
			var intIndex = this.arrItems[intItem].intIndex;

			if (!this.boolMultiselect)  {
				this.unselectAll();
			}
			if( objItem.isSelected() ) {
				// Try to unselect it and if successful call the callback
				if( this.unselectAtIndex(intItem) && this._itemCallbackFunc ) {
					this._itemCallbackFunc(intIndex, false);
				}
			}
			else {
				// Try to select it and if successful call the callback
				if( this.selectAtIndex(intItem) && this._itemCallbackFunc ) {
					this._itemCallbackFunc(intIndex, true);
				}
			}
		},

		/**
		 * @ignore
		 */
		_applyAction : function(intItem) {
			var intIndex = this.arrItems[intItem].intIndex;
			if (this.buttonCallback){
				this.buttonCallback(intIndex, this.buttonCallbackParams);
			}
		}
	}
);
