/**
 * @class UIElements.MenuList.MenuListControlBase
 * @extends UIElements.List.ListControlBase
 * Abstract Class - Do not use it directly
 * - {Object} objIconLeft The object defining the icon displayed between the avatar and the primary and secondary content
 * - {Object} arrSecondaryIcon The array containing icons displayed left of the second text content
 */
dojo.provide("UIElements.MenuList.MenuListControlBase");
dojo.declare("UIElements.MenuList.MenuListControlBase",
	UIElements.List.ListControl,
	{
		/**
		 * CSS highlight class
		 * @ignore
		 */
		_highlightClass : "highlighted",

		/**
		 * Height (in pixel) of an element of the list (li) (Default height is 50px)
		 * @cfg {Number}
		 */
		intLiHeight : 50,

		/**
		 * Set boolPreview to true when the MenuList UIElement is instantiated in a container in a preview of Homepage preview.
		 *
		 * This property can be set when instantiating the UIElement.
		 * @cfg {Boolean}
		 */
		boolPreview : false,

		/**
		 * @ignore
		 */
		constructor : function() {
			this.types.normal.callbackLeftIcon = {
				event : "click",
				func : "_clickedIconLeft",
				capture : true
			}
		},

		/**
		 * Update an item its primary content
		 * @param {Number} itemNb number of element
		 * @param {String} text new value
		 * @return {Boolean} result of the operation
		 */
		updatePrimaryContent : function (itemNb, text) {
			return this._updateItemText(itemNb, "strPrimaryContent", text);
		},

		/**
		 * Update an item its secondary content
		 * @param {Number} itemNb number of element
		 * @param {String} text new value
		 * @return {Boolean} result of the operation
		 */
		updateSecondaryContent : function (itemNb, text) {
			return this._updateItemText(itemNb, "strSecondaryContent", text);
		},

		/**
		 * Backward compatibility
		 * @deprecated
		 * @return {Boolean} the value returned by the right function call
		 */
		updateTimeForItem : function () {
			return this._deprecated("updateTimeForItem", "updateSecondaryContent", arguments);
		},

		/**
		 * Returns an li dom node for a given item.
		 * @ignore
		 * @param {Object} item the item from which the template is built
		 * @return {HTMLElement} an li dom node
		 */
		_getLiTemplate: function (item) {

			// Compute the content of the item:
			// - if the item is a principal item, unfolded, and it has an alternate unfoldedContent object defined, the alternate content
			//   provided into this unfoldedContent object will overwrite the content provided basically to the item
			//   (if the item is folded, the content provided at the root of the item will be applied)
			// - otherwise, the content provided at the root of item will be applied
			var objContent = this._computeContent(item);
			
			// li is a new Element
			var li = this.newElement("li")
			// In which I mix in a new className
			.mixin({
				"className" :  this.types[item.strType || "normal"]["className"]
					+ (this.isPrincipalItem(item) ?	" " + "Principal" : "")
					+ (this.isSecondaryItem(item) ?	" " + "Secondary" : "")
					+ (this._isAppearSelected(item) ?	" " + "appearSelected" : "")
					+ ( objContent.additionalStyle ?	" " + objContent.additionalStyle : "" ),
				"style" : "height:" + (this.intLiHeight + 1) + "px"
			})
			// I can also attach an event to it
			.attachEvent(this.types[item.strType || "normal"]["callback"], {
				// the Element can even receive custom params, this is an example :
				"myArgs" : "example"
			})
			.attachEvent({event: "mousedown", func: "_startHighlight", capture: false});

			// compute the width of the item label according the elements which are displayed or not in the item
			// (the base width is not the same if the menulist is in preview, as containers in preview are narrower
			// than containers in thirdControl)
			var menuListLabelWidth = this.boolPreview ? 238 : 268;

			// If the subtype is set to "PRINCIPAL_ENTRY", add the possibility on the left to collapse/uncollapse.
			if(this.isPrincipalItem(item)){
				// A div element
				var divIconPrincipalLeft = this.newElement("div")
				// With this className
				.mixin({
					"className" : "MenuListIconLeft " + "MenuListLeftPrincipal " + (this._isItemFolded(item) ? "folded " : "unfolded "),
					"style" : "height:" + this.intLiHeight + "px"
				})
				.attachEvent({event: "click", func: "_clickedIconLeft", capture:false}, {item: item});
				li.append(divIconPrincipalLeft);

				var leftSeparator = this._createSeparator();
				li.append(leftSeparator);
				menuListLabelWidth -= 45 ; //(size of separator : 1 + size of the clickable div : 44)

			}

			// If the item is a secondary item, add the left 'L' symbol
			if(this.isSecondaryItem(item)){
				// A div element
				var divSecondaryLeft = this.newElement("div")
					// With this className
					.mixin({
						"className" : "MenuListIconLeft " + "MenuListLeftSecondary " + (this._isLastSecondaryItem(item) ? "LastOne" : "")
					});
				li.append(divSecondaryLeft);
				menuListLabelWidth -= 45 ; //(size of the crochet div : 45)
			}
			
			// A div element
			var divEntry = this.newElement("div")
			// With this className
			.mixin({
				"className" : "MenuListItem",
				"style" : "height:" + this.intLiHeight + "px"
			})

			if (typeof objContent.strPicture != "undefined" && objContent.strPicture != "" && objContent.strPicture != null) {
				// Include img in a div set to the size of the MenuList item
				var divPicture = this.newElement("div")
					.mixin({
						"style" : "height:" + this.intLiHeight + "px" + ";" +
								"width:" + this.intLiHeight + "px"
					});
				divPicture.append(this.newElement("img")
					.mixin({
						// Max height/witdh of the picture is the size of the MenuList item
						"className" : "MenuListPicture",
						"src" : objContent.strPicture,
						"style" : "max-height:" + this.intLiHeight + "px" + ";" +
							"max-width:" + this.intLiHeight + "px"
					}));
				divEntry.append(divPicture);
				menuListLabelWidth -= this.intLiHeight;
			}

			// Default margin-left between items
			var marginLeft = 12;
			var divIconsLeft;
			if (objContent.objIconLeft) {
			    // Calculate the general margin-left space between items
			    // By default margin-left is set to 12px; If a width has been defined for the icon left; then the margin-left (spacing)
			    // between items is calculated in a proportion of (8/3)
			    marginLeft = objContent.objIconLeft.width ? Math.floor(objContent.objIconLeft.width/(8/3)) : 12;

				// A div element
				divIconsLeft = this.newElement("div")
					// With this className
					.mixin({
						"className" : "MenuListIconLeft " + (objContent.objIconLeft.icon ? objContent.objIconLeft.icon : ""),
						"style" :   "width:" + (objContent.objIconLeft.width ? objContent.objIconLeft.width : 32) + "px" + ";" +
							    "margin-left:" + marginLeft + "px"
					    });
				divEntry.append(divIconsLeft);
			    menuListLabelWidth -= marginLeft + (objContent.objIconLeft.width ? objContent.objIconLeft.width : 32);
			}

			// Then add a div for the texts
			var MenuListLabel = this.newElement("div")
			.mixin({
				"className" : "MenuListLabel" + (objContent.alternateStyle ? " alternate" :""),
				"style" : "margin-left:" + marginLeft + "px"
			});
			menuListLabelWidth -= marginLeft;

			divEntry.append(MenuListLabel);

			// Set the primary icon if any
			if (item.strType == "title" || (typeof objContent.strPrimaryIcon !== "undefined" && objContent.strPrimaryIcon !== "")) {

				var divPrimaryIcon = this.newElement("span")
				.mixin({
					"className" : (objContent.strPrimaryIcon || "") + " MenuListIconRight",
					"style" : "width: " + this.intLiHeight + "px;" +
						  "height: " + this.intLiHeight + "px;"
				});

				// If a callback is defined for right icon, add a separator and attack the callback to the dom.
				if(this.callbackPrimaryIcon){
					var rightSeparator = this._createSeparator();
					divEntry.append(rightSeparator);
					menuListLabelWidth -= 1 ; //(size of separator)

					divPrimaryIcon.attachEvent({event: "click", func: "_clickedPrimaryIcon", capture:false}, {item: item});

					divPrimaryIcon.className += " MenuListIconRightClickable";
				}
				
				divEntry.append(divPrimaryIcon);
				menuListLabelWidth -= this.intLiHeight;
			}

			if (typeof objContent.strPrimaryContent != "undefined") {
				MenuListLabel.append(
					this.newElement("span")
					// Every text content is automatically l10n if i18ned
					.mixin({
						"className" : "strPrimaryContent",
						"innerHTML" :  objContent.strPrimaryContent,
						"style"		: "width : "+menuListLabelWidth+"px;"
					})
				);
			}

			if (typeof objContent.strPrimaryContent != "undefined") {
				MenuListLabel.append(
					this.newElement("br")
				);
			}

			// Add icons in the secondary line content
			if (typeof objContent.arrSecondaryIcon != "undefined") {
			    for(var counterIcon = 0; counterIcon < objContent.arrSecondaryIcon.length; counterIcon++) {
				MenuListLabel.append (
				    this.newElement("span").mixin({
					"className" : objContent.arrSecondaryIcon[counterIcon].icon + " MenuListSecondaryIcon",
					"style" : "width:" + objContent.arrSecondaryIcon[counterIcon].width + "px;"
				    })
				);
				if (typeof objContent.arrSecondaryIcon[counterIcon].width != "undefined") {
				    menuListLabelWidth -= objContent.arrSecondaryIcon[counterIcon].width;
				}
			    }
			}

			// Add label in the secondary line content
			if (objContent.strSecondaryContent) {
			    if ((objContent.strSecondaryContent != null) && (objContent.strSecondaryContent != "")) {
				MenuListLabel.append(
					this.newElement("span")
					.mixin({
						"className" : "strSecondaryContent",
						"innerHTML" : objContent.strSecondaryContent,
						"style" : "width: "+menuListLabelWidth+"px;" +
							  "vertical-align: top;" +
							  (objContent.strSecondaryContentStyle ? "font-style: " + objContent.strSecondaryContentStyle + ";" : "") +
							  (objContent.strSecondaryContentWeight ? "font-weight: " + objContent.strSecondaryContentWeight + ";" : "")
					})
				);
			    }
			}
			
			li.append(divEntry);

			// Returns the li.
			return li;
		},

		/**
		 * Create an HTMLElement separator of 1 pixel
		 * @return {HTMLElement} separator
		 * @ignore
		 */
		_createSeparator : function() {
			var objSeparator = this.newElement("div").mixin({
				className: "MenuListSeparatorExtDiv",
				style: "height: " + this.intLiHeight + "px;"
			});
			objSeparator.append(
				this.newElement("div").mixin({
					className: "MenuListSeparator",
					// 3/5 of the hight for the separator, and 1/5 free on the top and on the bottom
					style: "margin-top: " + Math.ceil((this.intLiHeight)*0.2) + "px; height: "+ Math.floor((this.intLiHeight)*0.6) + "px;"
				})
			);

			return objSeparator;
		},


		/**
		 * Compare oldItem and newItem and determine if the item needs to be updated
		 * (i.e. fields of the newItem have been modified and have an impact graphically)
		 * @param {Object} oldItem the current state of the item
		 * @param {Object} newItem the new item
		 * @return {Boolean} true if an update is needed, false otherwise
		 * @ignore
		 */
		_isUpdateNeeded: function (oldItem, newItem) {
			if (oldItem.strPrimaryIcon === newItem.strPrimaryIcon &&
				oldItem.strPrimaryContent === newItem.strPrimaryContent &&
				ICTouchAPI.tools.compareObjects(oldItem.arrSecondaryIcon, newItem.arrSecondaryIcon) &&
				oldItem.strSecondaryContent === newItem.strSecondaryContent &&
				oldItem.strSecondaryContentStyle === newItem.strSecondaryContentStyle &&
				oldItem.strSecondaryContentWeight === newItem.strSecondaryContentWeight &&
				oldItem.strPicture === newItem.strPicture &&
				ICTouchAPI.tools.compareObjects(oldItem.objIconLeft, newItem.objIconLeft) &&
				oldItem.alternateStyle === newItem.alternateStyle &&
				oldItem.additionalStyle === newItem.additionalStyle &&
				ICTouchAPI.tools.compareObjects(oldItem.unfoldedContent, newItem.unfoldedContent)) {
				
				return false;
			}
			return true;
		},


		/**
		 * Update the class "LastOne" of a given secondary item.
		 * @ignore
		 * @param {Object} item the secondary item
		 * @param {Boolean} bLastOne if true, add the "LastOne" class, remove it otherwise
		 */
		_updateSecondaryItemLastOneClass : function (item, bLastOne) {
			var domItem = this._getItemInDom(item);
			var elem = domItem && domItem.getElementsByClassName("MenuListLeftSecondary")[0];
			if (elem) { // If elem is not defined, that's because item is not currently displayed in dom (because of pagination or folded)
				if (bLastOne) {
					dojo.addClass(elem,"LastOne");
				}
				else {
					dojo.removeClass(elem,"LastOne");
				}
			}
		},

		/**
		 * Update the arraw displayed in front of principal items, according to the state folded/unfolded of the item.
		 * @ignore
		 * @abstract
		 * @param {Object} item the principal item
		 */
		_updateArrawPrincipalItem : function (item) {
			var domItem = this._getItemInDom(item);
			var elem = domItem && domItem.getElementsByClassName("MenuListLeftPrincipal")[0];
			if (elem) { // If elem is not defined, that's because item is not currently displayed in dom (because of pagination)
				if (this._isItemFolded(item)) {
					dojo.removeClass(elem,"unfolded");
					dojo.addClass(elem,"folded");
				}
				else {
					dojo.removeClass(elem,"folded");
					dojo.addClass(elem,"unfolded");
				}
			}
		},

		/**
		 * Compute the content of the item:
		 * - if the item is a principal item, unfolded, and it has an alternate unfoldedContent object defined, the alternate content
		 *   provided into this unfoldedContent object will overwrite the content provided basically to the item
		 *   (if the item is folded, the content provided at the root of the item will be applied)
		 *  - otherwise, the content provided at the root of item will be applied
		 * @return {Object} an object containing all the computed styles
		 * @ignore
		 */
		_computeContent : function(item) {
			// Compute the content of the item:
			// - if the item is a principal item, unfolded, and it has an alternate unfoldedContent object defined, the alternate content
			//   provided into this unfoldedContent object will overwrite the content provided basically to the item
			//   (if the item is folded, the content provided at the root of the item will be applied)
			// - otherwise, the content provided at the root of item will be applied
			var objContent;
			if(this.isPrincipalItem(item) && item.unfoldedContent && !this._isItemFolded(item)){
				objContent = {
					strPrimaryIcon : item.unfoldedContent.strPrimaryIcon !== undefined ? item.unfoldedContent.strPrimaryIcon : item.strPrimaryIcon,
					strPrimaryContent : item.unfoldedContent.strPrimaryContent !== undefined ? item.unfoldedContent.strPrimaryContent : item.strPrimaryContent,
					arrSecondaryIcon : item.unfoldedContent.arrSecondaryIcon !== undefined ? item.unfoldedContent.arrSecondaryIcon : item.arrSecondaryIcon,
					strSecondaryContent : item.unfoldedContent.strSecondaryContent !== undefined ? item.unfoldedContent.strSecondaryContent : item.strSecondaryContent,
					strSecondaryContentStyle : item.unfoldedContent.strSecondaryContentStyle !== undefined ? item.unfoldedContent.strSecondaryContentStyle : item.strSecondaryContentStyle,
					strSecondaryContentWeight : item.unfoldedContent.strSecondaryContentWeight !== undefined ? item.unfoldedContent.strSecondaryContentWeight : item.strSecondaryContentWeight,
					strPicture : item.unfoldedContent.strPicture !== undefined ? item.unfoldedContent.strPicture : item.strPicture,
					objIconLeft : item.unfoldedContent.objIconLeft !== undefined ? item.unfoldedContent.objIconLeft : item.objIconLeft,
					alternateStyle : item.unfoldedContent.alternateStyle !== undefined ? item.unfoldedContent.alternateStyle : item.alternateStyle,
					additionalStyle : item.unfoldedContent.additionalStyle !== undefined ? item.unfoldedContent.additionalStyle : item.additionalStyle
				}
			}
			else {
				objContent = {
					strPrimaryIcon : item.strPrimaryIcon,
					strPrimaryContent : item.strPrimaryContent,
					arrSecondaryIcon : item.arrSecondaryIcon,
					strSecondaryContent : item.strSecondaryContent,
					strSecondaryContentStyle : item.strSecondaryContentStyle,
					strSecondaryContentWeight : item.strSecondaryContentWeight,
					strPicture : item.strPicture,
					objIconLeft : item.objIconLeft,
					alternateStyle : item.alternateStyle,
					additionalStyle : item.additionalStyle
				}
			}
			return objContent;
		},

		/**
		 * @param {HTMLElement} element UI element to be paginated
		 * @ignore
		 */
		_clickedIconLeft : function(event, args) {
			if (args.PARENT_WIDGET.foldUnfoldPrincipalItem) {
				event.cancelBubble = true;
				args.PARENT_WIDGET.foldUnfoldPrincipalItem.call(args.PARENT_WIDGET, args.item);
			}
		},

		/**
		 * @param {HTMLElement} element UI element to be paginated
		 * @ignore
		 */
		_clickedPrimaryIcon : function(event, args) {

			if (args.PARENT_WIDGET.callbackPrimaryIcon) {
				event.cancelBubble = true;
				args.PARENT_WIDGET._executeCallback("callbackPrimaryIcon", args.item);
			}
		}

	});
