/**
* @class UIElements.Media.MediaBase
* @extends UIElements._base
* Abstract Class - Do not use it directly.
*/
dojo.provide("UIElements.Media.MediaBase");
dojo.declare("UIElements.Media.MediaBase",
	[UIElements._base, dojox.dtl._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Interval in ms between two calls of function update
		 * @property
		 * @type Number
		 */
		intUpdateTimeStep			: 250,
		/**
		 * Media duration in ms
		 * @property
		 * @type Number
		 */
		intMediaDuration			: 10000,

		/**
		 * Show or not Slider button
		 * @property
		 * @type Boolean
		 */
		boolShowSliderButton		: true,
		/**
		 * Show or not icon in Slider button
		 * @property
		 * @type Boolean
		 */
		boolIconOnSliderButton		: true,
		/**
		 * Enable or disable auto play
		 * @property
		 * @type Boolean
		 */
		boolAutoplay				: false,
		/**
		 * Enable or disable possiblity to pause
		 * @property
		 * @type Boolean
		 */
		boolCanPause				: true,
		/**
		 * Show or hide time slider
		 * @property
		 * @type Boolean
		*/
		boolShowSliderTime			: true,
		/**
		 * Enable or disable possiblity to seek file
		 * @property
		 * @type Boolean
		*/
		boolSeekable				: true,
		/**
		 * play callback
		 * @property
		 * @type Function
		*/
		clbkPlay					: null,
		/**
		 * pause callback
		 * @property
		 * @type Function
		 */
		clbkPause					: null,
		/**
		 * stop callback
		 * @property
		 * @type Function
		 */
		clbkStop					: null,
		/**
		 * Action to do after move on time slider
		 * @property
		 * @type String
		*/
		actionOnMoveEnd				: null,
		/**
		 * Auto pause or not after move on time slider
		 * @property
		 * @type Boolean
		*/
		autopauseOnMove				: true,
		/**
		 * Current position on time slider
		 * @property
		 * @type Number
		*/
		currentSliderPosition		: 0,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		_arrEvents					: [],
		/**
		 * @ignore
		 */
		_step 						: null,
		/**
		 * @ignore
		 */
		_boolPlaying				: true,
		/**
		 * @ignore
		 */
		_sliderMoved				: false,
		/**
		 * @ignore
		 * Start date in Milliseconds
		 */
		_startDateMs				: 0,

		/* ------------------------------------ Constructor --------------------------------------- */

		postCreate : function () {
			this._mediaInitialized();
			if(this.boolShowSliderButton) {
				// Add the media Class to the Slider to display the button
				this.slider.addClassOnButton("media");
				if(this.boolIconOnSliderButton) {
					dojo.addClass(this.slider.domButtonIcon, "onPause");
					dojo.removeClass(this.slider.domButtonIcon, "onPlay");
				}
			}
			// Place the slider
			dojo.style(this.slider.domNode, "left",  this.intSliderX  + "px");
			dojo.style(this.slider.domNode, "top",  this.intSliderY  + "px");			
		},

		destroy: function () {
			// Disconnect all events in the arrEvents array
			var h;
			while( h = this._arrEvents.pop() )
				dojo.disconnect(h);
			// Destroy slider
			if( this.slider )
				this.slider.destroy();
			this.inherited(arguments);

			// Clear timer
			if( this.intervalHandler )
				clearInterval(this.intervalHandler);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * Set the offset in s
		 * @param {Number} offset
		 */
		setToTime : function(offset) {
			this.currentSliderPosition = offset*1000;
			if (this.currentSliderPosition < this.intMediaDuration){
				this.slider.setValue(offset);
			}
			else {
				this.slider.setValue(this.intMediaDuration);
			}
		},

		/**
		 * Set a duration in ms
		 * @param {Number} duration
		 */
		setMediaDuration : function(intDuration) {
			this.intMediaDuration = intDuration;
			this._setStep();
			this.slider.setMax(intDuration/1000);
		},

		/**
		 * Return the sliding state
		 * @return {Boolean} return the sliding state (moving or not)
		 */
		isSliderMoved : function() {
			return this._sliderMoved;
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
         * Play or hold a media
         */
		mediaPlayPause : function() {
			if (!this._boolPlaying){
				if(this.clbkPlay) {
					this.clbkPlay();
				} else {
					this.mediaPlay();
				}
			}
			else{
				if (this.boolCanPause) {
					if(this.clbkPause) {
						this.clbkPause();
					} else {
						this.mediaPause();
					}
				}
			}
		},

		/**
         * Play
         */
		mediaPlay : function() {
			// Store a date to compute the duration on each iteration of setInterval
			this._startDateMs = (new Date()).getTime() - (this.currentSliderPosition || 0);
			//  Change the button to "Pause" status and start/continue playing the media
			if(this.boolShowSliderButton && this.boolIconOnSliderButton) {
				dojo.addClass(this.slider.domButtonIcon, "onPlay");
				dojo.removeClass(this.slider.domButtonIcon, "onPause");
			}
			//setInterval is a javascript function that calls the update function every X ms (defined by the intUpdateTimeStep param
			if(!this.intervalHandler) {
				var that = this;
				var func = function () {
					that._updateTime();
				}
				this.intervalHandler = setInterval( func, this.intUpdateTimeStep);
			}
			this._boolPlaying = true;
			this._sliderMoved = false;
		},

		/**
         * Hold
         */
		mediaPause : function() {
			// Change the button to "Play" status and  stop playing the media
			if(this.boolShowSliderButton && this.boolIconOnSliderButton) {
				dojo.addClass(this.slider.domButtonIcon, "onPause");
				dojo.removeClass(this.slider.domButtonIcon, "onPlay");
			}
			if( this.intervalHandler ){
				clearInterval(this.intervalHandler);
				this.intervalHandler = null;
			}
			this._boolPlaying = false;
			this._sliderMoved = false;
		},

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
         * @ignore
         */
		_mediaInitialized : function() {
			dojo.disconnect(this._arrEvents.pop());

			var min = 0;
			var max = this.intMediaDuration;
			this._setStep();
			this._sliderMoved = false;

			if(this.boolShowSliderButton) {
				// Media not seekable if streaming
				var params = {
					intMinimum: min,
					intMaximum: max/1000,
					intValue: this.currentSliderPosition/1000,
					intSliderWidth: this.intSliderWidth,
					boolSeekable : this.boolSeekable,
					boolShowTime: this.boolShowSliderTime,
				};
				var seekable = this.boolSeekable && ( this.intMediaDuration < Infinity );
				params.boolSeekable = seekable;
				this.slider = new UIElements.Slider.Linear(params, this.domSlider);
			}
			else {
				var params = {
					intMinimum: min,
					intMaximum: max/1000,
					intValue: this.currentSliderPosition/1000,
					intBarWidth: this.intSliderWidth,
					boolSeekable : this.boolSeekable,
					boolShowTime: this.boolShowSliderTime,
				};
				this.slider = new UIElements.ProgressBar.ProgressBarControl(params, this.domSlider);
			}

			
			// Enable to connect to the "buttonpressed" event of the ValueSlider to launch the  mediaPlayPause function
			var h;			
			if(seekable) {
				h = dojo.connect(this.slider, "buttonpressed", this, this.mediaPlayPause);
				this._arrEvents.push(h);
			}
			else {
				h = dojo.connect(this.slider, "slidermousedown", this, this._sliderMouseDown);
				this._arrEvents.push(h);
			}

			// Enable to connect to the "seekingstart" event of the ValueSlider to launch the   _mediaStartSeek function
			var h = dojo.connect(this.slider, "seekingstart", this, this._mediaStartSeek);
			this._arrEvents.push(h);

			// Enable to connect to the "seekingend" event of the ValueSlider to launch the   _mediaEndSeek function
			var h = dojo.connect(this.slider, "seekingend", this, this._mediaEndSeek);
			this._arrEvents.push(h);
		},

		_sliderMouseDown : function() {
			this.mediaPlayPause();
		},

		/**
         * @ignore
         */
		_updateTime : function() {
			if (this.currentSliderPosition < this.intMediaDuration){
				// Set the position according to the date and not to a step to avoid desync.
				this.currentSliderPosition = (new Date()).getTime() - this._startDateMs;
				this.slider.setValue(this.currentSliderPosition/1000);
			}			
			else if(this._boolPlaying) {
				this.mediaPause();
				if(this.clbkStop){
					this.clbkStop();
				}
			}
		},
		/**
         * @ignore
         */
		_mediaStartSeek : function() {
			if(this.autopauseOnMove){
				this.mediaPause();
			}
		},
		/**
         * @ignore
         */
		_mediaEndSeek : function() {
			this.currentSliderPosition = this.slider.getValue()*1000;
			if(!this._sliderMoved){
				this._sliderMoved = true;
				if(this.actionOnMoveEnd == "play") {
					if(this.clbkPlay && !this._boolPlaying) {
						this.clbkPlay();
					}
				} else if(this.actionOnMoveEnd == "pause") {
					if(this.clbkPause && this._boolPlaying) {
						this.clbkPause();
					}
				} else if(this.actionOnMoveEnd == "stop") {
					if(this.clbkStop) {
						this.clbkStop();
					}
				}
			}
		},
		/**
         * @ignore
         */
		_setStep : function() {
			this._step = this.intUpdateTimeStep;
		}

	}
	);
