/**
 * @class UIElements.List.ListControlBase
 * @namespace UIElements.List
 * @extends UIElements._base
 * Abstract Class - Do not use it directly
 */
dojo.provide("UIElements.List.ListControlBase");
dojo.declare("UIElements.List.ListControlBase",
	[UIElements._base, dojox.dtl._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Items to display in the list.
		 * 
	 * This property is public for backward compatibility reasons.
		 * <b>It is recommended not to modify it but instead to use the available functions</b>
		 * (i.e. pushItems, popItems, updateItem....).
	 * @property
	 * @type {Array}
	 */
		arrItems : [],

		/**
		 * Nb of elements Max to be displayed in the DOM for pagination.
		 *
		 * This property can be set when instantiating the UIElement.
		 * @cfg {Number}
	 */
		nbElementsMax : 0,

		/**
		 * Describes the differences between item types (normal, title...).
		 *
		 * One of these types can be set for each item. Default type of the item is "normal".
		 * "Title" corresponds to title of collapsable items.
		 * @property {Object}
	 */
		types : {},

		/**
	 * Do categories have to automatically collapse?
		 *
		 * This property can be set when instantiating the UIElement.
		 * Then it can be changed using the appropriated getter/setter.
		 * @cfg {Boolean}
	 */
		autoCollapse : true,

		/**
		 * Do principal entries have to automatically fold when another one is unfolding?
		 *
		 * This property can be set when instantiating the UIElement.
		 * Then it can be changed using the appropriated getter/setter.
		 * @cfg {Boolean}
		 */
		autoFold : true,

		/**
		 * Do principal entries have to automatically unfold when clicking on a principal item?
		 *
		 * This property can be set when instantiating the UIElement.
		 * Then it can be changed using the appropriated getter/setter.
		 * @cfg {Boolean}
		 */
		autoUnfoldOnClick : true,

		/**
		 * Do principal entries have to be automatically folded when they are added to the list?
		 *
		 * This property can be set when instantiating the UIElement.
		 * Then it can be changed using the appropriated getter/setter.
		 * @cfg {Boolean}
		 */
		insertAsFolded : true,

		/**
	 * Does the list have to autoSelect the first item?
		 *
		 * This property can be set when instantiating the UIElement.
		 * @cfg {Boolean}
	 */
		autoSelectFirst : false,

		/**
		 * Does the list have to scroll ?
		 *
		 * This property can be set when instantiating the UIElement.
		 * @cfg {Boolean}
		 */
		boolScroll : true,

		/**
		 * Callback called when an element is selected.
		 *
		 * This property is set when instantiating the UIElement.
		 * @cfg {Function}
	 */
		callback : {},

		/**
		 * Callback called when a category is collapsed.
		 *
		 * This property is set when instantiating the UIElement.
		 * @cfg {Function}
	 */
		funcCollapsed : null,

		/**
		 * Display an empty state when this list is empty.
		 *
		 * This property can be set when instantiating the UIElement.
		 * @cfg {Boolean}
		 */
		showEmptyState: false,

		/**
		 * Empty state text, null show the default text from EmptyState.
		 *
		 * This property can be set when instantiating the UIElement.
		 * @cfg {String}
		 */
		emptyText: "",

		/**
		 * Do we select graphically an item we selected (by click or by js) ?
		 * 
		 * This property can be set when instantiating the UIElement.
		 * @cfg {Boolean}
		 */
		boolNotSelectable : false,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
	 * the list's root dom node
	 * @ignore
	 */
		_list: null,
		/**
	 * The current selected item
	 * @ignore
	 */
		_currentSelected : null,
		/**
	 * The current uncollapsed category
	 * @ignore
	 */
		_currentUnCollapsed : null,
		/**
	 * An array listing every LIs in the dom
	 * @ignore
	 */
		_LisInDom : [],

		/**
		 * EmptyState object
		 * @ignore
		 */
		_emptyState : null,

		/**
		 * Static value for _subType
		 * PRINCIPAL_ENTRY is the _subType for principal items
		 * @static
		 * @type {String}
		 * @ignore
		 */
		PRINCIPAL_ENTRY : "principalEntry",

		/**
		 * Static value for _subType
		 * SUB_ENTRY is the _subType for secondary items
		 * @static
		 * @type {String}
		 * @ignore
		 */
		SUB_ENTRY : "subEntry",

		/* ------------------------------------ Constructor --------------------------------------- */


		/**
	 * @ignore
	 */
		constructor : function () {
			// Describes every type used in these lists.
			// Can't see more than two, this is linked with the style sheet
			this.types = {
				title : {
					className : "Title",
					callback : {
						event : "click",
						func : "toggleCollapse",
						capture : true
					}
				},
				normal : {
					className : "Content",
					callback : {
						event : "click",
						func : "selectedLi",
						capture : false
					}
				}
			};

			// Take care to correctly init every attribute
			// so they are not shared between UIElements of same type.
			this.arrItems = [];


                        //crms00331797: Due to scroll enhancement and back color change, sometimes it seems that the end of the list is reached
			this.nbElementsMax = 10;
			this.autoCollapse = true;
			this.autoSelectFirst = true;

			this._list = null;
			this._currentSelected = null;
			this._currentUnCollapsed = null;
			this._LisInDom = [];

			this.callback = {};
			this.funcCollapsed = null;
		},

		/**
	 * @ignore
	 */
		postMixInProperties : function() {
			if(!generalConfig.simulation) {
				// Categories can only auto collapse if asked and if there are categories!
				this.setAutoCollapse(this._isListWithCategories() && !!this.getAutoCollapse());

				// Clean types in arrItems.
				this._cleanTypes(this.arrItems);

				// Add bCollapse to all items
				// The List will be (un)collapsed at start according to autoCollapse.
				this._setItemsCollapse(this.arrItems, this.getAutoCollapse());

				// Initializes the UL dom node
				this._initList()

				// And adds elements to it.
				.push(this.nbElementsMax);

				// Auto select first item
				this.autoSelectFirst &&  this.selectFirst();
		    }
		},

		/**
	 * @ignore
	 */
		postCreate : function () {                    
			if(!generalConfig.simulation) {

				// Appends the list to the visible dom
				this.domNode.appendChild(this._list);
				if(this.boolScroll){
					// Adds the scroll functionality
					FXScroll.init({
						handler : this.domNode,
						// Scroll of vertical type.
						scrollType : "vertical",
						// The pagination object
						// When scrolling up/down, the callback is called and does
						// what is necessary to add and remove dom nodes.
						pagination : {
							"callback" : this.pagination,
							"context" : this
						}
					});
				}
	
				if (this.showEmptyState) {
					var params = {
						boolBackground: true,
						boolIcon: false
					};
					if (this.emptyText) {
						params.strEmptyMessage = this.emptyText;
					}
					this._emptyState = new UIElements.EmptyState.EmptyStateControl(params);
					this._emptyState.placeAt(this.domNode, "first");
				}
			}
		},

		/**
	 * @ignore
	 */
		destroy : function () {
			// Go over each li and "remove" it to unsubscribe all i18n object
			var len = this._LisInDom.length;
			for (var i=0; i<len; ++i) {
				this._LisInDom[i].remove(false);
			}

			if (this._emptyState) {
				this._emptyState.destroy();
				delete this._emptyState;
			}
			
			this.inherited(arguments);
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
	 * Returns the selected item
	 * @return {Object} the current selected item
	 */
		getCurrentSelectedItem : function () {
			var currentSelected = this.getCurrentSelected();
			return (currentSelected && currentSelected._itemSource);
		},

		/**
		 * Returns the number of items in the list (ignore the number of secondary items)
		 * @return {Number} the number of items (without taking into account potential secondary items)
		 */
		getNbItems : function () {
			return this._getArrItemsWithoutSecondaryItems().length;
		},

		/**
	 * Scrolls the list to an item
	 * @param {Number} itemNb index of the item to scroll to
	 * @return {Object} Returns the li of the pointer that we've scrolled to.
	 */
		scrollToItemByPosition : function (itemNb) {
			// Before uncollapsing a new category,
			// collapse the previous one, if autoCollapse is enabled.
			this.getAutoCollapse() && this.collapseAll();
			// Get previous item's category
			var previousCat = this._getPreviousCatItem(itemNb);
			// Unollapse it.
			this._setCategoryCollapse(previousCat, false);
			// Empty the list of domNodes
			this._emptyList();
			// Reset the scroll
			this._resetDomScroll();
			// Insert the new nodes
			var firstDomNodeDisplayed = this.insert(0, itemNb, this.nbElementsMax, false);
			// Fill the list from the top until max nb of elements to display is reached.
			this._fillList();
			// return the first dom node
			return firstDomNodeDisplayed;
		},

		/**
	 * Select an item according to its position.
		 * 
		 * Note that:
		 * <ul>
		 * <li> it will scroll to the item's position (except if "doNotScroll" is set to true) and select the item </li>
		 * <li> it will not execute the clickCallback function of the list, except if "click" is set to true </li>
		 * <li> if the item to select is a principal item and the list's flag "autoUnfoldOnClick" is true (that's the case by default),
		 * the item will be automatically unfolded (except if "doNotUnfold" is set to true) </li>
		 * <li> if the item to select is a secondary item and the principal item is currently folded, the principal item will be
		 * automatically unfolded and the secondary item will be selected (except if "doNotUnfold" is set to true) </li>
		 * </ul>
		 * @param {Object} itemPos position of the item to select
		 * @param {Boolean} click handle click ? If true, the clickCallback function of the list will be executed
		 * @param {Boolean} doNotScroll if true, select the item without scrolling to its position
		 * @param {Boolean} doNotUnfold if true, do not unfold the primary entry when selecting it
		 */
		selectItemByPosition : function (itemPos, click, doNotScroll, doNotUnfold) {
		    var item = this.arrItems[itemPos];
			if (item) {
				this.selectItemByIndex(item.intIndex, click, doNotScroll, doNotUnfold);
			}
			else {
				ICTouchAPI.debugServices.warning("UIElements.List.ListControlBase - selectItemByPosition / Couldn't select item as item at position " + itemPos + " was not defined!");
			}
		},

		/**
		 * Select an item according to its intIndex property.
		 * 
		 * Note that:
		 * <ul>
		 * <li> it will scroll to the item's position (except if "doNotScroll" is set to true) and select the item </li>
		 * <li> it will not execute the clickCallback function of the list, except if "click" is set to true </li>
		 * <li> if the item to select is a principal item and the list's flag "autoUnfoldOnClick" is true (that's the case by default),
		 * the item will be automatically unfolded (except if "doNotUnfold" is set to true) </li>
		 * <li> if the item to select is a secondary item and the principal item is currently folded, the principal item will be
		 * automatically unfolded and the secondary item will be selected (except if "doNotUnfold" is set to true) </li>
		 * </ul>
		 * @param {String} index intIndex of the itemm to select
		 * @param {Boolean} click handle click ? If true, the clickCallback function of the list will be executed
		 * @param {Boolean} doNotScroll if true, select the item without scrolling to its position
		 * @param {Boolean} doNotUnfold if true, do not unfold the primary entry when selecting it
		 */
		selectItemByIndex : function (index, click, doNotScroll, doNotUnfold) {
		    var item = this.findItem("intIndex", index);
			if (item) {
				// if the selected entry is a secondary item and the principal item is folded, unfold it
				if (!doNotUnfold && this.isSecondaryItem(item) && this._isItemFolded(this.getPrincipalItem(item))) {
					this.unfoldPrincipalItem(this.getPrincipalItem(item));
				}
				if(doNotScroll) {
					// Select the item
					this._selectLi(this._getItemInDom(item), !click, doNotUnfold);
				} else {
					// Select the item we're scrolling to
					this._selectLi(this.scrollToItemByPosition(this._getItemPosition(item)), !click, doNotUnfold);
				}
			}
			else {
				ICTouchAPI.debugServices.warning("UIElements.List.ListControlBase - selectItemByIndex / Couldn't select item, item with intIndex " + index + " was not found!");
		    }
		},


		/**
	 * Select first list item
	 * Only works if there's no categories.
	 */
		selectFirst : function () {
			// If not a list with categories and there are items
			if (!this._isListWithCategories()) {
				// Select first item
				this._selectLi(this._LisInDom[0]);
			}
		},

		/**
	 * Uncollapse a category
	 * By default, the first item will be selected.
	 * @param {Object} item elem to uncollapse
	 * @param {Boolean} dontSelectFirstItem don't select the first item
	 * @return {Object} this
	 */
		uncollapseCategory : function (item, dontSelectFirstItem) {
			// Before uncollapsing a new one,
			// collapse the previous one, if autoCollapse is enabled.
			this.getAutoCollapse() && this.collapseAll();
			// Set the whole category uncollapsed
			this._setCategoryCollapse(item, false);
			// Insert its domNodes
			// The first inserted domNode will be returned.
			this._emptyList();
			this._resetDomScroll();
			var firstNode = this.insert(0, this._getItemPosition(item),  this.nbElementsMax, false);
			// fill the list at the top if needed.
			this._fillList();
			// Select first item if required and it exists.
			if (!dontSelectFirstItem) {
				// Select the domNode
				this._selectLi(this._getNextLiInDom(firstNode));
			}
			return this;

		},

		/**
	 * Collapse a category
	 * @param {Object} item elem to collapse
	 * @return {Object} this
	 */
		collapseCategory : function (item) {
			// Set all the category collapsed
			this._setCategoryCollapse(item, true);
			// Remove its domNodes if any
			var elem = this._getItemInDom(item);

			if  (elem) {
				this.slice(this.getLiPosition(elem)+1, this._getNbLisDisplayedInCat(elem), true);
			}
			// fill the list at the top if needed.
			this._fillList();
			// Call the associated callback.
			this._executeCallback("funcCollapsed");

			return this;

		},

		/**
	 * Collapse all categories
	 * @return {Object} this
	 */
		collapseAll : function () {
			// Collapse current if any
			var currentUnCollapsed = this._getCurrentUnCollapsed();
			if (currentUnCollapsed) {
				this.collapseCategory(currentUnCollapsed);
			}
			return this;
		},

		/**
	 * Returns current selected dom node
	 * @return {Object} currently selected object
	 */
		getCurrentSelected : function () {
			return this._currentSelected;
		},

		/**
	 * Deselect the current item.
	 */
		deselect : function () {
			this._deSelectCurrent();
		},

		/**
	 * Get autoCollapse value
	 * @return {Boolean} autoCollapse value
	 */
		getAutoCollapse : function () {
			return this.autoCollapse;
		},

		/**
	 * Set autoCollapse value;
	 * @param {Boolean} bool new value for autoCollapse
	 * @return {Boolean} results of the affectation
	 */
		setAutoCollapse : function (bool) {
			return this.autoCollapse = bool;
		},

		/**
		 * Get autoFold value
		 * @return {Boolean} autoFold value
		 */
		getAutoFold : function () {
			return this.autoFold;
		},

		/**
		 * Set autoFold value;
		 * @param {Boolean} bool new value for autoFold
		 * @return {Boolean} results of the affectation
		 */
		setAutoFold : function (bool) {
			return this.autoFold = bool;
		},

		/**
		 * Get autoUnfoldOnClick value
		 * @return {Boolean} autoUnfoldOnClick value
		 */
		getAutoUnfoldOnClick : function () {
			return this.autoUnfoldOnClick;
		},

		/**
		 * Set autoUnfoldOnClick value;
		 * @param {Boolean} bool new value for autoUnfoldOnClick
		 * @return {Boolean} results of the affectation
		 */
		setAutoUnfoldOnClick : function (bool) {
			return this.autoUnfoldOnClick = bool;
		},

		/**
		 * Get insertAsFolded value
		 * @return {Boolean} insertAsFolded value
		 */
		getInsertAsFolded : function () {
			return this.insertAsFolded;
		},

		/**
		 * Set insertAsFolded value;
		 * @param {Boolean} bool new value for insertAsFolded
		 * @return {Boolean} results of the affectation
		 */
		setInsertAsFolded : function (bool) {
			return this.insertAsFolded = bool;
		},

		/**
	 * Just give an item the select's style
	 * @param {Number} intIndex index of the element
	 * @param {Boolean} boolSelect
	 * @return {Object}
	 */
		toggleEmphasis : function (intIndex, boolSelect) {
			return this._toggleSelectEntryByItem(this.findItem("intIndex", intIndex), boolSelect);
		},

		/**
	 * Hides an item
		 * @param {Number} intPosition position of item to hide
	 */
		hideItem : function (intPosition) {
			this._setItemInvisibility(this.arrItems[intPosition], true);
			this._removeItemInDom(this.arrItems[intPosition]);
		},

		/**
	 * Hides an item
	 * @param {Number} intIndex index of the item to hide
	 */
		hideItemByIntIndex : function (intIndex) {
			this.hideItem(this._getItemPositionByIntIndex(intIndex));
		},

		/**
	 * Shows an item
		 * @param {Object} intPosition position of item to show
	 */
		showItem : function (intPosition) {
			this._setItemInvisibility(this.arrItems[intPosition], false);
			// if the item to show is in first position of arrItems, unshift it to the DOM
			if (intPosition === 0) {
				this.unshift(1);
			}
			// if the item to show is in last position of arrItems, push it to the DOM
			else if (intPosition === this.arrItems.length-1) {
				this.push(1);
			}
			// else insert it after the previous displayable item of arrItems
			else {
				// get the previous item which is displayable (not callapsed and not invisible)
				var previousDisplayableItem = this._getPreviousDisplayableItem(intPosition);
				if (previousDisplayableItem) {
					var insertPointDom = this.getLiPosition(this._getItemInDom(previousDisplayableItem));
					// if we found an item displayable, get its position in DOM (if it is not in DOM, it is not visible because of the pagination,
					// so do not insert the shownItem in the DOM - it will be inserted when its position will be in the visible area)
					if (insertPointDom !== undefined) {
						this.insert(insertPointDom, intPosition, 1, true);
					}
				}
				// if there is no item displayable before the one to show, unshift if to the DOM
				else {
					this.unshift(1);
				}
			}
		},

		/**
		 * Shows an item
		 * @param {String/Number} intIndex index of the item to show
	 */
		showItemByIntIndex : function (intIndex) {
			this.showItem(this._getItemPositionByIntIndex(intIndex));
		},

		/**
		 * Find item according to one of their attributes value.
	 * @param {Object} what the property to test
	 * @param {Object} equals the value of the property that is looked for
		 * @return {Object} The first element that match the search
	 */
		findItem : function (what, equals) {
			return this.findItems(what, equals)[0];
		},

		/**
		 * Find items according to one of their attributes value.
		 * @param {Object} what the property to test
		 * @param {Object} equals the value of the property that is looked for
		 * @return {Array} The elements that match the search
		 */
		findItems : function (what, equals) {
			var func = function (item) {
				// If this item's attribute equals..
				return (item && item[what] === equals);
			}
			return this.arrItems.filter(func);
		},

		/**
		 * Get an item from its intIndex.
		 * @param {String/Number} intIndex the item's index
		 * @return {Object} the first item matching the given intIndex
		 */
		getItemByIntIndex : function(intIndex) {
			return this.findItem("intIndex", intIndex);
		},

		/**
		 * Get an item from its position (in arrItems).
		 * @param {Number} intPosition the item's position
		 * @return {Object} the item at the given position (in arrItems)
		 */
		getItemByPosition : function (intPosition) {
			return this.arrItems[intPosition];
		},


		/**
		 * Check if an item is a principal item.
		 * @param {Object} item the item to check
		 * @return {Boolean} true if the item is a principal item, false otherwise
		 */
		isPrincipalItem: function (item) {
			// Check if an item is a principal item (i.e. it has the _subType equal to this.PRINCIPAL_ENTRY)
			return (item && this._getItemSubType(item) === this.PRINCIPAL_ENTRY);
		},
		

		/**
		 * Check if an item is a secondary item.
		 * @param {Object} item the item to check
		 * @return {Boolean} true if the item is a secondary item, false otherwise
		 */
		isSecondaryItem: function (item) {
			// Check if the item is a secondary item (i.e. it has the _subType equal to this.SUB_ENTRY and its field _principalEntryIndex is defined)
			return (item && (this._getItemSubType(item) === this.SUB_ENTRY && item._principalEntryIndex !== undefined));
		},


		/**
		 * Get the secondary items associated to a principal item.
		 * @param {String/Number} principalEntryIndex the principal item's intIndex
		 * @return {Array} an array containing the secondary items of this principal item (empty array if there is no secondary items)
		 */
		getSecondaryItems : function (principalEntryIndex) {
			return this.findItems("_principalEntryIndex", principalEntryIndex);
		},


		/**
		 * Get the number of secondary items associated to a principal item.
		 * @param {String/Number} principalEntryIndex the principal item's intIndex
		 * @return {Number} the number of secondary items for this principal item (0 if there is no secondary items for this principal item)
		 */
		getNbSecondaryItems : function (principalEntryIndex) {
			var secondaryItems = this.getSecondaryItems(principalEntryIndex);
			return (secondaryItems && secondaryItems.length) || 0;
		},


		/**
		 * Get the principal item's intIndex from a secondary item.
		 * @param {Object} secondaryItem the secondary item
		 * @return {String/Number} the principal item's intIndex
		 */
		getPrincipalItemIndex : function (secondaryItem) {
			return (secondaryItem && secondaryItem._principalEntryIndex);
		},


		/**
		 * Get the principal item's position (in arrItems) from a secondary item.
		 * @param {Object} secondaryItem the secondary item
		 * @return {Number} the principal item's position (in arrItems)
		 */
		getPrincipalItemPosition : function (secondaryItem) {
			return this._getItemPositionByIntIndex(this.getPrincipalItemIndex(secondaryItem));
		},


		/**
		 * Get the principal item from a secondary item.
		 * @param {Object} secondaryItem the secondary item
		 * @return {Object} the principal item
		 */
		getPrincipalItem : function (secondaryItem) {
			return this.getItemByIntIndex(this.getPrincipalItemIndex(secondaryItem));
		},

		/**
	 * Push a new item at the end.
	 * @param {Object} item item to be pushed
	 * @param {Boolean} addNow if the element is to be displayed right away
	 * @return {Boolean} result of the push operation
	 */
		pushItem : function (item,  addNow) {
			// Split collapsable items if arrSecondaryItems is filled
			var items = this._splitSecondaryItems(item);
			var nbItems = items.length;
			this.arrItems = this.arrItems.concat(items);
			this._checkForEmpty();
			return addNow && this.push(nbItems);
		},

		/**
	 * Push an array of new items at the end.
	 * @param {Array} arrItems array of the items to be pushed
	 * @param {Boolean} addNow if the element is to be displayed right away
	 * @return {Boolean} result of the push operation
	 */
		pushItems : function (arrItems, addNow) {
			var func = function (item) {
				this.pushItem(item, addNow);
			}
			var returnValue = arrItems.forEach(func, this);
			this._checkForEmpty();
			return returnValue;
		},

		/**
		 * Push secondary item(s) to a principal item.
		 * Note: If the item was not a principal item (i.e. it had 0 secondary items), it will be displayed as a principal item after the push operation.
		 * ex: [..., item_x, sub_item_x1, sub_item_x2, item_x+1,...] -> [..., item_x, sub_item_x1, sub_item_x2, <b>sub_item_x3</b>, item_x+1,...]
		 * @param {String/Number} principalEntryIndex the index of the principal entry to which the secondary items will be added
		 * @param {Array} secondaryItems the secondary items to be added
		 * @return {Array} the result of the insert
		 */
		pushSecondaryItems : function (principalEntryIndex, secondaryItems) {
			return this.insertSecondaryItems(principalEntryIndex, this.getNbSecondaryItems(principalEntryIndex)+1, secondaryItems);
		},

		/**
		 * Add a new item at the begining.
	 * @param {Object} item item to be unshifted
	 * @param {Boolean} addNow to be added now ?
	 * @return {Boolean} result of the operation
	 */
		unshiftItem : function (item, addNow) {
			// Split collapsable items if arrSecondaryItems is filled
			var items = this._splitSecondaryItems(item);
			var nbItems = items.length;
			this.arrItems = items.concat(this.arrItems);
			this._checkForEmpty();
			return addNow && this.unshift(nbItems);
		},

		/**
		 * Add multiple items from the begining.
	 * @param {Array} arrItems array of the items to be unshifted
	 * @param {Boolean} addNow to be added now ?
	 * @return {Boolean} result of the operation
	 */
		unshiftItems : function (arrItems, addNow) {
			var func = function (item) {
				this.unshiftItem(item, addNow);
			}
			var returnValue = arrItems.forEach(func, this);
			this._checkForEmpty();
			return returnValue;
		},

		/**
		 * Unshift secondary item(s) to a principal item.
		 * Note: If the item was not a principal item (i.e. it had 0 secondary items), it will be displayed as a principal item after the unshift operation.
		 * ex: [..., item_x, sub_item_x1, sub_item_x2, item_x+1,...] -> [..., item_x, <b>sub_item_x0</b>, sub_item_x1, sub_item_x2, item_x+1,...]
		 * @param {String/Number} principalEntryIndex the index of the principal entry to which the secondary items will be added
		 * @param {Array} secondaryItems the secondary items to be added
		 * @return {Array} the result of the insert
		 */
		unshiftSecondaryItems : function (principalEntryIndex, secondaryItems) {
			return this.insertSecondaryItems(principalEntryIndex, 0, secondaryItems);
		},

		/**
		 * Update an item and refresh its corresponding li element.
		 * 
		 * If the item provided is a secondary item, it will be updated.
		 *
		 * If the item provided is a principal item, it will be updated and its secondary items will be updated too:
		 * <ul>
		 * <li> modification/addition/suppression of secondary items is managed,
		 * <li> if the item was not a principal item but the item provided in updateItem is, its secondary items will be automatically added
		 *		and the item will appear as a principal item (according to the graphical implementation of the UIElements inheriting List),
		 * <li> if the item was a principal item but the item provided in updateItem doesn't have secondary items, its secondary items will
		 *		be removed and it will not appear as a principal item anymore (according to the graphical implementation of the UIElements inheriting List).
		 * </ul>
		 *
		 * For performance reason, <b>updateItem</b> checks if the item needs to be updated (i.e. fully destroyed and rebuilt, really CPU coslty) using the
		 * abstract function "\_isUpdateNeeded" which has to be implemented by UIElements inheriting MenuList. If the function "\_isUpdateNeeded"
		 * is not overwritten, by default the item will be updated.
		 *
		 * It is possible to bypass the test done in "\_isUpdateNeeded" by setting the argument bForceUpdate to true <u>(not recommended)</u>.
		 * @param {Object} item item to be updtated (can be standard item, principal item or secondary item)
		 * @param {Boolean} bForceUpdate force the update (don't use the function "\_isUpdateNeeded" to check if an update is needed) <u>(not recommended)</u>
		 */
		updateItem : function (item, bForceUpdate) {
			// Get the old item.
			var oldItem = this.findItem("intIndex", item.intIndex);
			if (oldItem) {
				var selectedItem;
				// if item has secondary items provided, check if the item was already a principal item (look for secondary items to be added/removed/updated)
				// or not (add the secondary items and the item will become a principal item)
				if (item.arrSecondaryItems) {
					// if the old item was already a principal item, check if some of its secondary items have to be added/removed/updated
					if (this.isPrincipalItem(oldItem)) {
						// if the selected item was a secondary item of this item, save its index to re-select it after, as it may have been moved
						selectedItem = (this.isSecondaryItem(this.getCurrentSelectedItem()) && this.getPrincipalItem(this.getCurrentSelectedItem()) == oldItem) ? this.getCurrentSelectedItem() : undefined;
						var i;
						// browse through the new item's secondary items and look for secondary items to update/add in oldItem
						for (i = 0 ; i < item.arrSecondaryItems.length ; i++) {
							// if the secondary item at position 'i' was already at position 'i' in the oldItem, update this secondary item (if needed)
							if (oldItem.arrSecondaryItems[i] && item.arrSecondaryItems[i].intIndex === oldItem.arrSecondaryItems[i].intIndex) {
								this.updateItem(item.arrSecondaryItems[i]);
							}
							// else the secondary item at position 'i' was not at the same position in the oldItem, insert this secondary item at position 'i'
							else {
								this.insertSecondaryItems(item.intIndex, i, item.arrSecondaryItems[i]);
							}
						}
						// browse through oldItem's secondary items and look for secondary items to remove
						for (i = 0 ; i < oldItem.arrSecondaryItems.length ; i++) {
							// if the secondary item at position 'i' in the oldItem is not at the same position in the new item, remove this secondary item and
							// all the secondary items following this one; then exit the loop
							if (!item.arrSecondaryItems[i]) {
								this.sliceSecondaryItems(item.intIndex, i, oldItem.arrSecondaryItems.length-i);
								break;
							}
						}

					}
					// else the old item was not a principal item, so insert all the secondary items (the item will automatically become a principal item)
					else {
						this.insertSecondaryItems(item.intIndex, 0, item.arrSecondaryItems);
					}
				}
				// item doesn't have secondary items provided, check if the item was already a principal item. If it was, remove all its secondary items
				// (the item will automatically become a standard item)
				else if (this.isPrincipalItem(oldItem)) {
					// if the selected item was a secondary item of this item, save its index to select the principal item, as the secondaries will be removed
					selectedItem = (this.isSecondaryItem(this.getCurrentSelectedItem()) && this.getPrincipalItem(this.getCurrentSelectedItem()) == oldItem) ? this.getCurrentSelectedItem() : undefined;
					this.emptySecondaryItems(oldItem.intIndex);
				}
				// determine if the item need to be updated graphycally:
				// if bForceUpdate is not set to true, check if the item really need to be updated, using function _isUpdateNeeded (to be
				// overwritten by UIElements inheriting List, otherwise the update will always be done even if it is not necessary)
				var boolUpdate = bForceUpdate || this._isUpdateNeeded(oldItem, item );
			// Get its index
			var idx = this._getItemPosition(oldItem);
			// Mix the new item in the old one.
			this.arrItems[idx] = dojo.mixin(oldItem, item);
				// update graphycally or not according to the computed boolUpdate
				if (boolUpdate) {
			// Get its position in the dom.
			var lipos = this.getLiPosition(this._getItemInDom(oldItem));
			// Refresh the li
			this._updateLi(lipos, idx);
				}
				// if selectedItem is defined, the selected item before the update was a secondary item of the principal item updating.
				if (selectedItem) {
					// if the item which was selected is still in the list, select it again (it may have been removed/re-added during the update)
					if (this.getItemByIntIndex(selectedItem.intIndex)) {
						this.selectItemByIndex(selectedItem.intIndex, false, true, true);
					}
					// if the item which was selected is no longer in the list, select the item updating (i.e. the parent of seletedItem, which has been deleted during the update)
					else {
						this.selectItemByIndex(this.getPrincipalItemIndex(selectedItem), true);
					}
				}
			}
			else {
				ICTouchAPI.debugServices.warning("UIElements.List.ListControlBase - updateItem / Couldn't update item as the item provided don't exists in the list.");
			}
		},

		/**
		 * Pop an item from the list.
		 * Note: If the item to pop is a secondary item, the whole corresponding principal item is popped (including its secondary items)
		 * ex: [..., item_x, sub_item_x1, sub_item_x2, item_x+1,...] -> [..., item_x, sub_item_x1, item_x+1,...]
		 * @param {Array} completeList if the list should'nt be completed
		 * @return {Object} popped Item(s)
	 */
		popItem : function (completeList) {
			// Check if the item to pop is a secondary item. If that's the case, pop the whole last item (including its secondary items)
			if (this._getItemSubType(this.arrItems[this.arrItems.length-1]) === this.SUB_ENTRY) {
				return this.sliceItems(this.getPrincipalItemPosition(this.arrItems[this.arrItems.length-1]), this.arrItems.length, !completeList);
			}
			else {
			var item = this.arrItems.pop();
			this._checkForEmpty();
			return this._removeItemInDom(item, !completeList);
			}
		},

		/**
		 * Pop a secondary item from a principal item.
		 * Note: If the principal item don't have secondary items anymore after the pop operation, it will not be displayed as a principal item anymore.
		 * @param {String/Number} principalEntryIndex the index of the principal item from which the secondary item will be popped
		 * @param {Array} completeList if the list should'nt be completed
		 * @return {Object} popped item
		 */
		popSecondaryItem : function (principalEntryIndex, completeList) {
			return this.sliceItem(this._getLastSecondaryItemPosition(principalEntryIndex), !completeList);
		},

		/**
	 * Pop multiple items from the list
	 * @param {Number} nb number of element to pop out
		 * @return {Array} popped elements
	 */
		popItems : function (nb) {
			for (var i=0; i<nb; i++) {
				this.popItem();
			}
			this._checkForEmpty();
			return this.arrItems;
		},

		/**
		 * Remove one item from the top of the list.
		 * Note: If the item to shift is a principal item, the whole principal item is shifted, including its secondary items
	 * @param {Array} completeList List from which to shift an item
	 * @return {Object} li or this
	 */
		shiftItem : function (completeList) {
			// Check if the item to shift is a principal item. If that's the case, shift the whole first item (including its secondary items)
			if (this.arrItems[0]._subType === this.PRINCIPAL_ENTRY) {
				return this.sliceItems(0, this._getLastSecondaryItemPosition(this.arrItems[0].intIndex)+1, !completeList);
			}
			else {
			var item = this.arrItems.shift();
			this._checkForEmpty();
			return this._removeItemInDom(item, !completeList);
			}
		},

		/**
		 * Shift a secondary item from a principal item.
		 * Note: If the principal item don't have secondary items anymore after the shift operation, it will not be displayed as a principal item anymore.
		 * ex: [..., item_x, sub_item_x1, sub_item_x2, item_x+1,...] -> [..., item_x, sub_item_x2, item_x+1,...]
		 * @param {String/Number} principalEntryIndex the index of the principal item from which the secondary item will be shifted
		 * @param {Array} completeList if the list should'nt be completed
		 * @return {Object} shifted item
		 */
		shiftSecondaryItem : function (principalEntryIndex, completeList) {
			return this.sliceItem(this._getFirstSecondaryItemPosition(principalEntryIndex), !completeList);
		},

		/**
	 * Remove multiple items from the top of the list.
	 * @param {Number} nb number of elements to shifted
	 * @return {Array} local array of items
	 */
		shiftItems : function (nb) {
			for (var i=0; i<nb; i++) {
				this.shiftItem();
			}
			this._checkForEmpty();
			return this.arrItems;
		},

		/**
		 * Remove an item and its corresponding li.
		 * Note: If the item to slice is a principal item, the whole principal item is sliced, including its secondary items.
		 * @param {Number} itemPos position of element to slice
		 * @param {Boolean} dontCompleteList don't complete the list
		 * @return {Object} sliced li or false
		 */
		sliceItem : function (itemPos, dontCompleteList) {
			if (this.arrItems[itemPos]) {
				if (this.arrItems[itemPos]._subType === this.PRINCIPAL_ENTRY) {
					// Splice arrItems
					return this.sliceItems(itemPos, this._getLastSecondaryItemPosition(this.arrItems[itemPos].intIndex) + 1 - itemPos, dontCompleteList);
				} else {
					var isSecondaryItem = this.isSecondaryItem(this.arrItems[itemPos]);
					var secondaryItemPosition = isSecondaryItem && this._getSecondaryItemPosition(this.arrItems[itemPos]);
					var wasLastSecondaryItem = isSecondaryItem && this._isLastSecondaryItem(this.arrItems[itemPos]);
					var item = this.arrItems.splice(itemPos, 1)[0];
					// manage the case a secondary item is spliced (refresh the principal item and the last secondary item if needed)
					if (isSecondaryItem) {
						var principalItem = this.getPrincipalItem(item);
						// update the array of secondary items of the principal item
						principalItem.arrSecondaryItems.splice(secondaryItemPosition, 1);
						// Update graphically the principal item if needed
						this._checkUpdatePrincipalItem(principalItem.intIndex);
						// if we removed the last secondary item and the principal item is still a principal item after the slice,
						// add the class LastOne to its last secondary item
						if (wasLastSecondaryItem && this.isPrincipalItem(principalItem)) {
							this._updateSecondaryItemLastOneClass(this._getLastSecondaryItem(principalItem.intIndex), true);
						}
					}
				this._checkForEmpty();
				return this._removeItemInDom(item, dontCompleteList);
				}
			} else {
				return false;
			}
		},

		/**
		 * Remove secondary item(s) from a principal item (according to its position in arrSecondaryItems) and its corresponding li.
		 * If the secondaryItemStart is lower than 0, secondaryItemStart is set to 0 (the slice will start from the first secondary item).
		 * If secondaryItemStart + range is greater than the number of secondary items currently present, the slice will stop at the last secondary item
		 * Note: If the principal item don't have secondary items anymore after the slice operation, it will not be displayed as a principal item anymore
		 * ex: [..., item_x, sub_item_x1, sub_item_x2, sub_item_x3, sub_item_x4, item_x+1,...] -> [..., item_x, sub_item_x1, sub_item_x4, item_x+1,...]
		 * @param {String/Number} principalEntryIndex index of the principal item
		 * @param {Number} secondaryItemStart position of first secondary item to slice
		 * @param {Number} range range of secondary items to be sliced
		 * @param {Boolean} dontCompleteList don't complete the list
		 * @return {Object} sliced li or false
		 */
		sliceSecondaryItems : function (principalEntryIndex, secondaryItemStart, range, dontCompleteList) {
			// check that the principal item well exists... if not, just return empty array
			if (!this.getItemByIntIndex(principalEntryIndex)) {
				ICTouchAPI.debugServices.warning("UIElements.List.ListControlBase - sliceSecondaryItems / Couldn't remove secondary items as there is no item having the given principalEntryIndex: " + principalEntryIndex);
				return [];
			}
			else {
				// check that secondaryItemPos is well in the range of secondaryItems, else if out of range slice the last secondary item
				var nbSecondaryItems = this.getNbSecondaryItems(principalEntryIndex);
				secondaryItemStart = (0 < secondaryItemStart) ? secondaryItemStart : 0;
				range = (secondaryItemStart + range <= nbSecondaryItems) ? range : nbSecondaryItems;
				// compute the position of secondaryItemStart in arrItems
				var startSlice = this._getItemPosition(this.getSecondaryItems(principalEntryIndex)[secondaryItemStart]);
				// remove the secondary items
				var slicedItems = this.sliceItems(startSlice, range, dontCompleteList);
				// update the array of secondary items of the principal item
				this.getItemByIntIndex(principalEntryIndex).arrSecondaryItems.splice(secondaryItemStart, range);
				// if there is no longuer secondary items for the principal item, update it
				this._checkUpdatePrincipalItem(principalEntryIndex);
				// if we removed the last secondary item and the principal item is still a principal item after the slice, add the class LastOne to its last secondary item
				if (secondaryItemStart+range === nbSecondaryItems && this.isPrincipalItem(this.getItemByIntIndex(principalEntryIndex))) {
					this._updateSecondaryItemLastOneClass(this._getLastSecondaryItem(principalEntryIndex), true);
				}
				return slicedItems;
			}
		},
		
		/**
		 * Remove an item and its corresponding li given its intIndex.
		 * The item to remove can also be a secondary item.
		 * If the item to remove is a principal item, the whole principal item is removed, including its secondary items.
	 * If the deleted item was selected, it selects the new one and executes the callback.
	 * dontAutoSelectNext true if you don't want to have the next item to be selected
		 * @param {String/Number} intIndex the intIndex property of the item to delete
	 * @param {Boolean} dontAutoSelectNext to prevent the next item to be selected
		 * @return {Object} the removed item (an array is a principal item is removed with its secondary items)
	 */
		removeItemByIntIndex : function (intIndex, dontAutoSelectNext) {
			var sliceItem = null;
			// Get the item in the list according to the intIndex
			var item = this.getItemByIntIndex(intIndex);
			if(item) {
				// If the item is in the dom
				var isInDom = this._getItemInDom(item);
				// Get the position of the item in the dom
				var itemPos = this._LisInDom.indexOf(isInDom);
				// Is the current select item.
				var currentSelected = !!(isInDom === this.getCurrentSelected() || (this.getCurrentSelectedItem() && this.isSecondaryItem(this.getCurrentSelectedItem()) && this.getPrincipalItem(this.getCurrentSelectedItem()) == item));
				// Remove the item from the list and from the dom
				sliceItem =  this.sliceItem(this._getItemPosition(item));
				// And select the new one which took the place.
				if (currentSelected) {
					// the currentSelected item (or the principal item of the currentSelected item) has been removed, so set CurrentSelected to null
					this._setCurrentSelected(null);
					if (!dontAutoSelectNext) {
					// Select the new one which took the place (the "next one")
					// Or select the previous one if we were at the end of the list.
					this._selectLi(this._LisInDom[itemPos]) || this._selectLi(this._LisInDom[itemPos-1]);
				}
				}

			}
			return sliceItem;
		},

		/**
		 * Remove item(s) and their corresponding li
		 * Do not use if the list contains secondary items (do not managed by sliceItems)
		 * @ignore
		 * @param {Number} start starting position
	 * @param {Number} range range od items to be sliced
	 * @param {Boolean} dontCompleteList don't complete the list
	 * @return {Array} sliced Items
	 */
		sliceItems : function (start, range, dontCompleteList) {
			
			// Splice arrItems
			var items = this.arrItems.splice(start, range);
			// function remove lis
			var func = function (item) {
				this._removeItemInDom(item, dontCompleteList);
			}
			// Execution function for each item
			items.forEach(func, this);
			this._checkForEmpty();
			// Return an array of removed elements
			return items;
		},

		/**
		 * Insert item(s) at a given insert point.
		 * Note that the provided insert point don't take into account the secondary items that some items can have.
		 * Ex: if insertItems is called with insertPoint = 2, the items will be inserted before the 2nd item
		 * (whatever the item at position 0 and 1 have secondary items or not).
		 * @param {Number} insertPoint insert point
		 * @param {Array} items elements to be added
	 * @return {Array} the result of the insert
	 */
		insertItems : function (insertPoint, items) {
			// Check if items is a list, else fake one
			items = dojo.isArray(items) ? items : [items];
			// Split collapsable items if arrSecondaryItems is filled
			items = this._splitSecondaryItemsFromArray(items);
			// Insert the items in the items list (take into account the potential secondary items that some items can have by using _computeInsertPoint)
			this.arrItems = this._insertInArray(this.arrItems, this._computeInsertPoint(insertPoint), items);
			this._checkForEmpty();
			// Get the next index of the first inserted item.
			var newIndex = this._getItemPosition(items[0]);
			// Get the previous item in the dom, if any
			var  previousItemIndex = this.getLiPosition(this._getItemInDom(this.arrItems[newIndex-1]));
			// Insert after the previous item or at the end.
			var elt = this.insert(previousItemIndex, newIndex, items.length, true);
			items = null;
			return elt;
		},

		/**
		 * insert secondary item(s) to a principal item at a given position
		 * If the insertPoint is lower than 0, the secondary items are inserted at the position 0 (in first position of the secondary items).
		 * If the insertPoint is greater than the number of secondary items currently present, the secondary items are inserted at the end of the secondary items array.
		 * Note: If the item was not a principal item (i.e. it had 0 secondary items), it will be displayed as a principal item
		 * ex: [..., item_x, sub_item_x1, sub_item_x2, item_x+1,...] -> [..., item_x, sub_item_x1, <b>sub_item_x3</b>, sub_item_x2, item_x+1,...]
		 * @param {String/Number} principalEntryIndex the index of the principal entry to which the secondary items will be added
		 * @param {Number} insertPoint insert point
		 * @param {Array} secondaryItems the secondary elements to be added
		 * @return {Array} the result of the insert
		 */
		insertSecondaryItems : function (principalEntryIndex, insertPoint, secondaryItems) {
			// check that the principal item well exists... if not, just return empty array
			if (!this.getItemByIntIndex(principalEntryIndex)) {
				ICTouchAPI.debugServices.warning("UIElements.List.ListControlBase - insertSecondaryItems / Couldn't insert secondary items as there is no item having the given argument principalEntryIndex (" + principalEntryIndex + ')!');
				return [];
			}
			if (!secondaryItems) {
				ICTouchAPI.debugServices.warning("UIElements.List.ListControlBase - insertSecondaryItems / Couldn't insert secondary items as argument secondaryItems is not defined!");
				return [];
			}
			else {
				// Check if items is a list, else fake one
				secondaryItems = dojo.isArray(secondaryItems) ? secondaryItems : [secondaryItems];
				// format secondary items
				var principalEntry = this.getItemByIntIndex(principalEntryIndex);
				this._formatSecondaryItems(secondaryItems, principalEntry);
				// compute the position of insertPoint in arrItems
				var insertPosition;
				// check that the principal item we wants to insert items is already a principal item ; if that's not the case, we insert the secondary items directly after the principal item
				if (this.isPrincipalItem(principalEntry)) {
					// check that insertPoint is well in the range of secondaryItems, else if out of range insert them after the last secondary item
					var nbSecondaryItems = this.getNbSecondaryItems(principalEntryIndex);
					insertPoint = (0 < insertPoint) ? insertPoint : 0;
					insertPoint = (insertPoint <= nbSecondaryItems) ? insertPoint : nbSecondaryItems;
					// If we insert the secondaryItem in last position, remove the class "LastOne" from the current last item 
					// (the class will be automatically added to the new one at its creation by _getLiTemplate)
					if (insertPoint === nbSecondaryItems) {
						this._updateSecondaryItemLastOneClass(this._getLastSecondaryItem(principalEntryIndex), false);
					}
					insertPosition = this._getFirstSecondaryItemPosition(principalEntryIndex) + insertPoint;
					// update the array of secondary items of the principal item
					principalEntry.arrSecondaryItems = this._insertInArray(principalEntry.arrSecondaryItems, insertPoint, secondaryItems);
				}
				else {
					// insert the secondary items directly after the principal item
					insertPosition = this._getItemPosition(principalEntry) + 1;
					// create the array of secondary items of the principal item
					principalEntry.arrSecondaryItems = secondaryItems;
				}
				// Insert the items in the items list
				this.arrItems = this._insertInArray(this.arrItems, insertPosition, secondaryItems);
				// Update the principal item if needed
				this._checkUpdatePrincipalItem(principalEntryIndex);
				this._checkForEmpty();
				// if the principal item is unfolded, add the secondary items in the DOM
				if (!this._isItemFolded(principalEntry)) {
					// Get the next index of the first inserted item.
					var newIndex = this._getItemPosition(secondaryItems[0]);
					// Get the previous item in the dom, if any
					var  previousItemIndex = this.getLiPosition(this._getItemInDom(this._getPreviousDisplayableItem(newIndex)));
					// Insert after the previous item or at the end.
					var elt = this.insert(previousItemIndex, newIndex, secondaryItems.length, true);
				}
				secondaryItems = null;
				return elt;
			}
		},

		/**
	 * Remove all items from the list
	 */
		emptyItems : function() {
			return this.sliceItems(0, this.arrItems.length);
		},


		/**
		 * Remove all secondary items of a principal item and their corresponding li
		 * Note: If the principal item will not be displayed as a principal item anymore
		 * ex: [..., item_x, sub_item_x1, sub_item_x2, sub_item_x3, sub_item_x4, item_x+1,...] -> [..., item_x, item_x+1,...]
		 * @param {String/Number} principalEntryIndex index of the principal item
		 * @return {Object} sliced li or false
		 */
		emptySecondaryItems : function(principalEntryIndex) {
			return this.sliceSecondaryItems(principalEntryIndex, 0, this.getNbSecondaryItems(principalEntryIndex));
		},

		/**
		 * Unfold a principal item
		 * @param {Object} principalEntry principal item to unfold
		 */
		unfoldPrincipalItem : function (principalEntry) {
			// unfold the item only if it is curretnly folded
			if (this._isItemFolded(principalEntry)) {
				// Before unfolding a new one,
				// fold the other ones, if autoFold is enabled.
				this.getAutoFold() && this.foldAllPrincipalItems();
				// set principal item's folded flag as unfolded
				this._setItemFolded(principalEntry, false);
				// refresh principal item to appear unfolded
				this._updatePrincipalItem(principalEntry);
				// show all the secondary items associated to this principal item
				var func = function (item) {
					this.showItemByIntIndex(item.intIndex);
				}
				this.getSecondaryItems(principalEntry.intIndex).forEach(func, this);
			}
		},

		/**
		 * Unfold a principal item from its intIndex
		 * @param {String/Number} principalEntryIndex principal item's intIndex to unfold
		 */
		unfoldPrincipalItemByIndex : function (principalEntryIndex) {
			this.unfoldPrincipalItem(this.getItemByIntIndex(principalEntryIndex));
		},

		/**
		 * Fold a principal item
		 * @param {Object} principalEntry principal item to fold
		 */
		foldPrincipalItem : function (principalEntry) {
			// fold the item only if it is currently unfolded
			if (!this._isItemFolded(principalEntry)) {
				// set principal item's folded flag as folded
				this._setItemFolded(principalEntry, true);
				// refresh principal item to appear folded
				this._updatePrincipalItem(principalEntry);
				// hide all the secondary items associated to this principal item
				// and look if one of the secondary items was selected before the folding
				var currentSelectedItem = this.getCurrentSelectedItem();
				var selectedSecondaryItem = false;
				var func = function (item) {
					if (item === currentSelectedItem) {
						selectedSecondaryItem = true;
					}
					this.hideItemByIntIndex(item.intIndex);
				}
				this.getSecondaryItems(principalEntry.intIndex).forEach(func, this);
				// if a secondary item was selected when folding, select the principal item after all the secondary items has been hidden
				if (selectedSecondaryItem) {
					this.selectItemByIndex(principalEntry.intIndex, true, true, true);
				}
			}
		},

		/**
		 * Fold a principal item from its intIndex
		 * @param {String/Number} principalEntryIndex principal item's intIndex to fold
		 */
		foldPrincipalItemByIndex : function (principalEntryIndex) {
			this.foldPrincipalItem(this.getItemByIntIndex(principalEntryIndex));
		},

		/**
		 * Fold all the unfolded principal items
		 */
		foldAllPrincipalItems : function () {
			var arrUnfoldedItems = this.findItems("_isFolded", false);

			var func = function (item) {
				this.foldPrincipalItem(item);
			}
			arrUnfoldedItems.forEach(func, this);
		},

		/**
		 * Fold a primary item if it is unfolded, unfold it if it is folded
		 * @param {Object} principalEntry principal item to fold/unfold
		 */
		foldUnfoldPrincipalItem : function (principalEntry) {
			if (this._isItemFolded(principalEntry)) {
				this.unfoldPrincipalItem(principalEntry);
			}
			else {
				this.foldPrincipalItem(principalEntry);
			}
		},

		/**
		 * Fold a primary item if it is unfolded, unfold it if it is folded
		 * @param {String/Number} principalEntryIndex principal item's intIndex to fold/unfold
		 */
		foldUnfoldPrincipalItemByIndex : function (principalEntryIndex) {
			this.foldUnfoldPrincipalItem(this.getItemByIntIndex(principalEntryIndex));
		},

		/**
	 *   If long press callback is defined, execute callback.
	 *   The selected entry item index is given in parameter
	 *   @param {Object} liNode the selected item entry
	 */
		eventLongPress: function(liNode) {
			if (this.callbackLongPress && liNode && liNode._itemSource) {
				this._executeCallback("callbackLongPress", liNode._itemSource.intIndex);
			}
		},

		/* ----------------------------------- Private methods ------------------------------------- */


		/**
	 * @ignore
	 * launch pagination if the list has been scrolled
	 * @return {number} the amount of pixel scrolled
	 */
		pagination : function () {
                        var scrollTop = this.domNode.scrollTop;

                        // If the list is scrolled more than 50 pixels
			if (scrollTop >= 50) {
				// Push an item at the end of the list and remove the first one.
				// Return the amount of pixels to "unscroll"
				return -50 * this.push(1, true);
			// If the list is scrolled to the top most
			} else if (!scrollTop) {
				// Add an item at the top of the list and remove the last one
				return 50 *this.unshift(1, true);
			} else {
				// If nothing changed, return 0.
				return 0;
			}

		},

		/**
	 * A method to add (nb) elements at the end of the dom node.
	 * If the number of elements gets to big, removes the top nodes
	 * @ignore
	 * @param {Number} nb  number of elements to be added
	 * @param {Boolean} remove  true if we have to remove when nb > max
	 * @return {Number} the final array length
	 */
		push : function (nb, remove) {

			// We want to add stuff after the last displayed item :
			var start = this._getLastDisplayedItem()+1;
			// Can't add more than nbElementsMax item
			var range = remove && nb > this.nbElementsMax ? this.ElementsMax : nb;
			// Get a list of Lis to display, from a list of items
			var newElements = this._buildList(this._getNodesToDisplay(start, range));
			// Insert the domNodes in the main list and store the references as well.
			this._insertNodes(newElements[0], newElements[1], 0);
			// If asked, remove any exceeding node.
			remove && this.shift(this._getNbOfExceedingNodes());
			// Return the number of elements really pushed
			return newElements[1].length;
		},

		/**
	 * A method to remove (nb) elements at the end of the dom node.
	 * @ignore
	 * @param {Number} nb number of elements to pop out
	 * @return {Array} poped elements
	 */
		pop : function (nb) {
			// Calling the slice method with the starting point and the range
			// calculated from the current list length
			return this.slice(this._LisInDom.length-nb, nb);
		},

		/**
	 * Removes (nb) elements at the begining of the dom node
	 * @ignore
	 * @param {Number} nb number of element to pop out
	 * @return {Number} number of sliced arrays
	 */
		shift : function (nb) {
			// Calling the slice method with the starting point and the range
			return this.slice(0, nb);
		},

		/**
	 * Add (nb) elements at the begining of the dom node
	 * @ignore
	 * @param {Number} nb number of elements to be added
	 * @param {Boolean} remove true if we have to remove when nb > max
	 * @return {Number} the final array length
	 */
		unshift : function (nb, remove) {
			// As we want to insert nodes before the first displayed
			var start = this._getFirstDisplayedItem();
			// Be sure not to add more elements than possible
			var range = remove && nb > this.nbElementsMax ? this.ElementsMax : nb;
			// Get a list of Lis to display, from a list of items
			var newElements = this._buildList(this._getNodesToDisplay(start, range, true));
			// Insert the domNodes in the main list and store the references as well.
			this._insertNodes(newElements[0], newElements[1], this._LisInDom[0]);
			// If asked, remove any exceeding node.
			remove && this.pop(this._getNbOfExceedingNodes());
			// Return the number of elements really unshifted
			return newElements[1].length;
		},

		/**
	 * Removes a number of dom nodes from a specific point
	 * @ignore
	 * @param {Number} start starting point
	 * @param {Number} range range of item to be sliced
	 * @param {Boolean} add notice the query for addign element at the end
	 * @return {Array} number of sliced elements
	 */
		slice : function (start, range, add) {
			// Remove n domNodes
			for (var i=0, nb=0; i<range; i++, nb++) {
				this._removeOneLi(start);
			}
			// If asked, add n elements at the end.
			add && this.push(-this._getNbOfExceedingNodes());
			//add && this.unshift(-this._getNbOfExceedingNodes());
			// Returns the number of domNodes sliced.
			return nb;
		},

		/**
	 * Insert items from an Li
	 * @ignore
	 * @param {Object} fromLi  Elements to be added
	 * @param {Object} fromItem  starting Element
	 * @param {Number} nb  starting Element
	 * @param {Boolean} remove do we have to remove some elements ?
	 * @return {Array} poped elements
	 */
		insert : function (fromLi, fromItem, nb, remove) {

			// In this case, range is of course limited by the available nodes to display.
			var nextLi = this._LisInDom[fromLi+1];
			// Can't be more than the nbElementsMax
			// (check that the number of displayable elements between fromItem and the item after the insert point (fromLi) is not greater than nbElementsMax)
			var maxRange = nextLi ? (this._getNbDisplayableItems(fromItem, this._getItemPosition(nextLi._itemSource))) : this.nbElementsMax;
			// As long as nb is defined and less than maxRange
			var range = (nb > maxRange) || !nb ? maxRange : nb;
			// Get the elements
			var newElements = this._buildList(this._getNodesToDisplay(fromItem, range));
			// Insert them
			this._insertNodes(newElements[0], newElements[1], nextLi);

			// If asked, remove any exceeding node starting from the end
			remove && (this._getNbOfExceedingNodes() > 0) && this.pop(this._getNbOfExceedingNodes());
			// Then from the start
			remove && (this._getNbOfExceedingNodes() > 0) && this.shift(this._getNbOfExceedingNodes());
			// Or if not enough domNodes were added
			remove && (this._getNbOfExceedingNodes() < 0) && this.unshift(-this._getNbOfExceedingNodes());
			// Actually its more like a verify nb of nodes than a remove boolean
			remove && (this._getNbOfExceedingNodes() < 0) && this.push(-this._getNbOfExceedingNodes());

			// Returns a reference to the first domNode displayed.
			return newElements[1][0];
		},

		/**
		 * Find the previous displayable item in arrItems from a given pisition.
		 * @ignore
		 * @param {Number} intPosition position from which search the previous displayable item
		 * @return {Object} The previous displayable item (undefined if not found)
		 */
		_getPreviousDisplayableItem : function (intPosition) {
			var previousDisplayableItem;
			for (var i = intPosition-1; i >= 0 ; i--) {
				if (this._itemDisplayable(this.arrItems[i])) {
					previousDisplayableItem = this.arrItems[i];
					break;
				}
			}
			return previousDisplayableItem;
		},

		/**
		 * Get the number of displayable items in arrItems from a item to another one.
		 * @ignore
		 * @param {Number} fromItem position from which get the number of displayable items
		 * @param {Number} toItem position to which get the number of displayable items
		 * @return {Number} The number of displayable items
		 */
		_getNbDisplayableItems : function (fromItem, toItem) {
			var nbDisplayable = 0;
			for (var i = fromItem ; i < toItem ; i++) {
				if (this._itemDisplayable(this.arrItems[i])) {
					nbDisplayable++;
				}
			}
			return nbDisplayable;
		},


		/**
	 * @ignore
	 * Returns the index of a given li in the list.
	 * @param {Object} li  targeted element
	 * @return {Number} the position
	 */
		getLiPosition : function (li) {
			return +(this._LisInDom.indexOf(li));
		},

		/**
	 * un/collapse a category
	 * This method is called when a click occurs on a category.
	 * @ignore
	 * @param {Event} event the click event, unused
	 * @param {Array} args objects to collapse or not
	 */
		toggleCollapse : function (event, args) {
			// Is it collapsed?
			if (this._itemSource.bCollapse) {
				// Uncollapse the new one
				args.PARENT_WIDGET.uncollapseCategory(this._itemSource);
			} else {
				// Else, collapse the current.
				args.PARENT_WIDGET.collapseCategory(this._itemSource);
			}

		},

		/**
	 * Set a collapse value + style to a dom node
	 * @ignore
	 * @param {Object} item element to be setted up
	 * @param {Boolean} bCollapse collapse state
	 * @return {Boolean} new collapse state
	 */
		setCollapse : function (item, bCollapse) {
			if (item) {
				var elem = this._getItemInDom(item);
				if (elem) {
					dojo.toggleClass(elem, "collapsed", bCollapse);
				}
				this._setItemCollapse(item, bCollapse);
				return bCollapse;
			}
		},

		/**
	 * Called when a click occured on an item
	 * @ignore
	 * @param {Event} event the click event
	 * @param {Object} args args to be browsed
	 */
		selectedLi : function (event, args) {
			args.PARENT_WIDGET._selectLi(this, false, false, true);
		},

		/**
	 * Abstract Class - To be overloaded by child class
	 * Returns an li dom node for a given item.
	 * @ignore
		 * @abstract
	 * @param {Object} item the item from which the template is built
	 * @return {HTMLElement} an li dom node
	 */
		_getLiTemplate: function (item) {
			
		},

		/**
		 * Abstract function called when a request to update an item is done.
		 * By default, if this function is not overwritten, the function returns true so that the whole item will always be destroyed and rebuilt graphically.
		 * As the graphical operations are costly for CPU, overwrite this function in UIElements inheriting "List" would allow to gain performances by not
		 * refreshing the whole item when it's not needed.
		 * Ex:
		 * <ul>
		 * <li> if nothing has changed in the fields of the object concerning graphical element's part (label/icons/style...), the UIElement can return false as
		 *		an update is not needed (will avoid a useless costly refresh). </li>
		 * <li> if only a label changed in the newItem compared to what it is in the item displayed, the UIElement can change it direcly using _updateItemText
		 *		(will change only the text direcly in the dom) and return false (will be better for performances than rebuilding all the entry). </li>
		 *	</ul>
		 * @ignore
		 * @abstract
		 * @param {Object} oldItem the current state of the item
		 * @param {Object} newItem the new item
		 * @return {Boolean} true if an update is needed, false otherwise
		 */
		_isUpdateNeeded: function (oldItem, newItem) {			
			return true;
		},


		/**
		 * Abstract function called when the state folded/unfolded of a principal item changes.
		 * Function to be overwitten by UIElements inheriting "List" to manage graphically this changement.
		 * @ignore
		 * @abstract
		 * @param {Object} item the principal item
		 */
		_updateArrawPrincipalItem: function (item) {

		},

		/**
		 * Abstract function called when a secondary item [become/is not anymore] the last secondary item of its principal item.
		 * Function to be overwitten by UIElements inheriting "List" to manage graphically this changement.
		 * @ignore
		 * @abstract
		 * @param {Object} item the secondary item
		 * @param {Boolean} bLastOne true if the item is the last secondary item of its principal item, false otherwise
		 */
		_updateSecondaryItemLastOneClass: function (item, bLastOne) {

		},

		/**
	 *    Update an li given its index and the item's index
	 *    @ignore
	 *    @param {Number} liIndex the index of the li to update
	 *    @param {Number} itemIndex the index of the corresponding item
	 */
		_updateLi : function (liIndex, itemIndex) {
			// If the li element exists.
			if (this._LisInDom[liIndex]) {
				// Remove the old one.
				this._removeOneLi(liIndex);
				// Build the new one
				var newElement = this._buildList(this._getNodesToDisplay(itemIndex, 1));
				// Insert it.
				this._insertNodes(newElement[0], newElement[1], this._LisInDom[liIndex]);
			}
		},

		/**
	 * Reset the scroll for this UIElement
	 * @ignore
	 */
		_resetDomScroll : function () {
			if(this.boolScroll){
				FXScroll.reset(this.domNode);
			}
		},

		/**
	 * A generic function for executing the UI element callbacks
	 * @ignore
	 * @param {String} name the name of the callback to execute
	 * @param {Object} params the callback's parameters
		 * @param {Boolean} boolByClick true if the callback is executed following a user click (not a selection called by the webapp)
	 * @return {Boolean} the result of the callback execution or false
	 */
		_executeCallback : function(name, params, boolByClick) {
			// If the callback is a function
			var cb = this[name];
			if (typeof cb === "function") {
				// Execute it
				return cb(params, boolByClick || false);
			// If it's an object
			} else if (cb && typeof cb.func === "function") {
				// Execute it in the given context, or in the current one.
				cb.func.call(cb.context || window, params || null, boolByClick || false);
			} else {
				return false;
			}
		},

		/**
	 *   Get nb of domNodes currently displayed
	 *   @ignore
	 *   @return {Number} the number of displayed domNodes
	 */
		_getNbOfDomNodes : function () {
			return this._LisInDom.length;
		},

		/**
	 * Get the difference between the number of existing domNodes and the nbElementsMax
	 * @ignore
	 * @return {Number} the difference
	 */
		_getNbOfExceedingNodes : function () {
			var maxRange = this.arrItems.length > this.nbElementsMax ? this.nbElementsMax : this.arrItems.length;
			return (this._getNbOfDomNodes() - maxRange);
		},

		/**
	 * Empty the list of domNodes
	 * @ignore
	 * @return {Number} the result of the pop function
	 */
		_emptyList : function () {
			return this.pop(this._getNbOfDomNodes());
		},

		/**
	 * Removes one dom node
	 * @ignore
	 * @param {Number} liNb the index of the dom node
	 * @return the result of the remove operation or this
	 */
		_removeOneLi : function (liNb) {
			var li = this._removeLiNb(liNb);
			return li ? li.remove() : this;
		},

		/**
	 * Collapse one li
	 * @ignore
	 * @param {Number} liNb the index of the li
	 */
		_collapseOneLi : function (liNb) {
			this._setItemCollapse(this._LisInDom[liNb]._itemSource, true);
		},

		/**
	 * Uncollapse one li
	 * @ignore
	 * @param {Number} liNb the index of the li
	 */
		_uncollapseOneLi : function (liNb) {
			this._setItemCollapse(this._LisInDom[liNb]._itemSource, false);
		},

		/**
	 * Removes the index from the list of references.
	 * @ignore
	 * @param {Number} liNb the index of the li
	 */
		_removeLiNb : function (liNb) {
			return this._LisInDom.splice(liNb, 1)[0] || false;
		},


		/**
	 * Add nodes at a specific point
	 * @ignore
	 * @param {HTMLElement} domNodes a dom fragment with the li domNodes
	 * @param {HTMLElement} nodesRef the references of the domNodes to insert
	 * @param {HTMLElement} position a domNode from which to insert
	 */
		_insertNodes : function (domNodes, nodesRef, position) {
			// Insert nodes before an other one, or append at the end.
			this._list.insertBefore(domNodes, position);
			// Also insert the references in the _LisInDom array
			this._LisInDom = this._insertInArray(this._LisInDom, this.getLiPosition(position), nodesRef);

		},

		/**
	 * A generic function for adding items to an array from an insertion point
	 * @ignore
	 * @param {Array} list the list to which to add items
	 * @param {Number} insertPoint the index from which to add the items
	 * @param {Array} items the items to add
	 */
		_insertInArray : function (list, insertPoint, items) {
			items = dojo.isArray(items) ? items : [items];
			var head = insertPoint >= 0 ? list.splice(insertPoint) : [];
			return list.concat(items).concat(head);
		},

		/**
	 * Returns the last displayed element in the list.
	 * @ignore
	 * @return {Number} the position or -1 if none
	 */
		_getLastDisplayedItem : function () {
			var length = this._LisInDom.length;
			if (length > 0) {
				return this._getItemPosition(this._LisInDom[this._LisInDom.length-1]._itemSource);
			} else {
				return -1;
			}
		},

		/**
	 * Returns the first displayed element in the list
	 * @ignore
	 * @return {Number} the position
	 */
		_getFirstDisplayedItem : function () {
			if (typeof this._LisInDom[0] === "object") {
				return this._getItemPosition(this._LisInDom[0]._itemSource);
			}
		},

		/**
	 *  Return the type of a domNode
	 *  @ignore
	 *  @param {HTMLElement} li the domNode
	 *  @return {String} title || normal
	 */
		_getLiType : function (li) {
			if (typeof li === "object" && li.getAttribute) {
				if (li.className.split(" ").indexOf(this.types["title"].className) >= 0) {
					return "title";
				} else {
					return "normal";
				}
			} else {
				return false;
			}
		},

		/**
	 * Checks if the given list contains categories
	 * @ignore
	 * @return {Boolean} true if it's a list with categories
	 */
		_isListWithCategories : function () {
			var func = function (item) {
				if (item.strType === "title") {
					return true;
				}
			};
			return this.arrItems.some(func);
		},

		/**
	 * Clearly identify title type from normal type
	 * @ignore
	 * @param {Array} array the list of items from which to clean types
	 */
		_cleanTypes : function (array) {
			// Do the cleaning on arrItems.
			if (array && array.forEach) {
				array.forEach(this._cleanItemType, this);
			}
		},

		/**
	 * Set the strType property to "normal" if it's not defined
	 * @ignore
	 * @param {Object} item the item to clean
	 * @return {Boolean} if the item was cleaned
	 */
		_cleanItemType : function (item) {
			// If no type is defined
			if (!item.strType) {
				// set it to normal.
				item.strType = "normal";
				return true;
			} else {
				return false;
			}
		},


		/**
		 * @ignore
		 * Return the list of items without the potential secondary items
		 * @return {Array} the array of list's items without the potential secondary items
		 */
		_getArrItemsWithoutSecondaryItems : function () {
			var func = function (item) {
				// return this item if it is not a secondary item
				return (item && !this.isSecondaryItem(item));
			}
			return this.arrItems.filter(func,this);
		},


		/**
		 * @ignore
		 * Convert an insert point given which don't take into account the secondary items that some items can have to the real position in the whole arrItems array
		 * (which contains the secondary items).
		 * ex: [item_a, item_b, sub_item_b1, sub_item_b2, item_c,...] -> we want insert before item_c, so to position 2, but real position is arrItems is in fact 4
		 * (because of the secondary items of item_b)
		 * @param {Number} insertPoint insert point wanted without taking into account the secondary items that some items can have
		 * @return {Number} the insert point in arrItems
		 */
		_computeInsertPoint : function (insertPoint) {
			var arrayWithoutSecondariesItems = this._getArrItemsWithoutSecondaryItems();
			return this._getItemPosition(arrayWithoutSecondariesItems[insertPoint]);
		},


		/**
		 * @ignore
		 * Get the position (relative to arrSecondaryItems of a principal item, not relative to arrItems) of a secondary item.
		 * @param {Object} secondaryItem the secondary item to get the position
		 * @return {Number} the position (relative to arrSecondaryItems) of the secondary item
		 */
		_getSecondaryItemPosition : function (secondaryItem) {
			return this.getSecondaryItems(this.getPrincipalItem(secondaryItem).intIndex).indexOf(secondaryItem);
		},


		/**
		 * @ignore
		 * Get the first secondary item associated to a given principal item (from the principal item's intIndex).
		 * @param {String/Number} principalEntryIndex the intIndex of the principal item
		 * @return {Object} the first secondary item associated to this principal item
		 */
		_getFirstSecondaryItem : function (principalEntryIndex) {
			var secondaryItems = this.getSecondaryItems(principalEntryIndex);
			return (secondaryItems && secondaryItems[0]);
		},

		/**
		 * @ignore
		 * Get the intIndex of the first secondary item associated to a given principal item (from the principal item's intIndex).
		 * @param {String/Number} principalEntryIndex the intIndex of the principal item
		 * @return {Number} the intIndex of the first secondary item associated to this principal item
		 */
		_getFirstSecondaryItemIndex : function (principalEntryIndex) {
			var firstSecondaryItem = this._getFirstSecondaryItem(principalEntryIndex);
			return (firstSecondaryItem && firstSecondaryItem.intIndex);
		},

		/**
		 * @ignore
		 * Get the position (in arrItems) of the first secondary item associated to a given principal item (from the principal item's intIndex).
		 * ex: [item_a, item_b, sub_item_b1, sub_item_b2, item_c,...] -> position of sub_item_b1 is 2
		 * @param {String/Number} principalEntryIndex the intIndex of the principal item
		 * @return {Number} the position (in arrItems) of the first secondary item associated to this principal item
		 */
		_getFirstSecondaryItemPosition : function (principalEntryIndex) {
			return this._getItemPosition(this._getFirstSecondaryItem(principalEntryIndex));
		},


		/**
		 * @ignore
		 * Get the last secondary item associated to a given principal item (from the principal item's intIndex).
		 * @param {String/Number} principalEntryIndex the intIndex of the principal item
		 * @return {Object} the last secondary item associated to this principal item
		 */
		_getLastSecondaryItem : function (principalEntryIndex) {
			var secondaryItems = this.getSecondaryItems(principalEntryIndex);
			return (secondaryItems && secondaryItems[secondaryItems.length-1]);
		},

		/**
		 * @ignore
		 * Get the intIndex of the last secondary item associated to a given principal item (from the principal item's intIndex).
		 * @param {String/Number} principalEntryIndex the intIndex of the principal item
		 * @return {Number} the intIndex of the last secondary item associated to this principal item
		 */
		_getLastSecondaryItemIndex : function (principalEntryIndex) {
			var lastSecondaryItem = this._getLastSecondaryItem(principalEntryIndex);
			return (lastSecondaryItem && lastSecondaryItem.intIndex);
		},

		/**
		 * @ignore
		 * Get the position (in arrItems) of the last secondary item associated to a given principal item (from the principal item's intIndex).
		 * ex: [item_a, item_b, sub_item_b1, sub_item_b2, item_c,...] -> position of sub_item_b2 is 3
		 * @param {String/Number} principalEntryIndex the intIndex of the principal item
		 * @return {Number} the position (in arrItems) of the last secondary item associated to this principal item
		 */
		_getLastSecondaryItemPosition : function (principalEntryIndex) {
			return this._getItemPosition(this._getLastSecondaryItem(principalEntryIndex));
		},

		/**
		 * @ignore
		 * Check if an item is the last secondary item of its principal item.
		 * @param {Object} secondaryItem the secondary item to check
		 * @return {Boolean} true if the item is the secondary item, false otherwise
		 */
		_isLastSecondaryItem: function (secondaryItem) {
			var principalItem = this.getPrincipalItem(secondaryItem);
			return (principalItem && this._getLastSecondaryItem(principalItem.intIndex) === secondaryItem);
		},


		/**
		 * @ignore
		 * Format an array of items provided with secondary items (item[i].arrSecondaryItems) to an array of item with secondary items split
		 * after their respective principal item. Specific fields of principal items and secondary items are filled.
		 * The array returned is directly insertable into arrItems.
		 * ex: [item_a.arrSecondaryItems=[sub_item_a1, sub_item_a2], item_b.arrSecondaryItems=[sub_item_b1, sub_item_b2]]
		 * -> [item_a, sub_item_a1, sub_item_a2, item_b, sub_item_b1, sub_item_b2]
		 * @param {Array} array the initial array of items to format
		 * @return {Array} the formatted array with secondary items split and entries's fields filled
		 */
		_splitSecondaryItemsFromArray: function (array) {
			var itemsList = [];
			if (array && dojo.isArray(array)) {
				for (var i = 0 ; i < array.length ; i++) {
					itemsList = itemsList.concat(this._splitSecondaryItems(array[i]));
				}
			}
			return itemsList;
		},

		/**
		 * @ignore
		 * Format an item provided with secondary items (item[i].arrSecondaryItems) to an array of item with secondary items split
		 * after their principal item. Specific fields of principal item and secondary items are filled.
		 * ex: item_a.arrSecondaryItems=[sub_item_a1, sub_item_a2]
		 * -> [item_a, sub_item_a1, sub_item_a2]
		 * Note:
		 *  - if the item provided in input don't have secondary items in its field arrSecondaryItems, the function returns an array with just this item: [item]
		 *  - if the item provided in input is not defined, the function returns an empty array: []
		 * @param {Object} item the initial item format
		 * @return {Array} the array with secondary items split and entries's fields filled
		 */
		_splitSecondaryItems: function (item) {
			if (item) {
				// Add missing normal type to item
				this._cleanItemType(item);
				if (item.arrSecondaryItems && item.arrSecondaryItems.length > 0) {
					// set the principal item 's flag _subType to PRINCIPAL_ENTRY
					this._setItemSubType(item, this.PRINCIPAL_ENTRY);
					// set the principal item's flag _isFolded according to the list's flag insertAsFolded
					this._setItemFolded(item, this.getInsertAsFolded());
					// add the principal item to the first position of the array
					var itemsList = [item];
					// format the secondary items
					this._formatSecondaryItems(item.arrSecondaryItems, item);
					// push the formatted secondary items after the principal item
					itemsList = itemsList.concat(item.arrSecondaryItems);
					return itemsList;
				} else {
					return [item];
				}
			} else {
				return [];
			}
		},

		/**
		 * @ignore
		 * Format an array of secondary items (the array stored in principalItem.arrSecondaryItems) by filling the specific fields of each secondary items.
		 * Specific fields are:
		 *  - _subType (set to this.SUB_ENTRY)
		 *  - _principalEntryIndex (set to the intIndex of the principal item provided in input)
		 * @param {Array} arrSecondaryItems the array of secondary items to format
		 * @param {Object} principalEntry the principal item associated to the secondary items
		 */
		_formatSecondaryItems: function (arrSecondaryItems, principalEntry) {
			for (var i = 0 ; i < arrSecondaryItems.length ; i++) {
				this._formatSecondaryItem(arrSecondaryItems[i], principalEntry);
			}
		},

		/**
		 * @ignore
		 * Format a secondary item by filling its specific fields.
		 * The secondary item is automatically formatted as invisible if the principal item is folded,
		 * visible is the principal item is unfolded.
		 * Specific fields are:
		 *  - _principalEntryIndex (set to the intIndex of the principal item provided in input)
		 *  - _subType (set to this.SUB_ENTRY)
		 * @param {Object} secondaryItem the secondary item to format
		 * @param {Object} principalEntry the principal item associated to this secondary item
		 * @return {Boolean} return true if the item is successfully formatted, false if the item was not defined or not an object
		 */
		_formatSecondaryItem: function (secondaryItem, principalEntry) {
			if (secondaryItem && typeof secondaryItem === "object") {
				// Add missing normal type to item
				this._cleanItemType(secondaryItem);
				// Add specific fields for secondary entry item
				secondaryItem._principalEntryIndex = principalEntry.intIndex;
				this._setItemSubType(secondaryItem, this.SUB_ENTRY);
				// set the invisibility property of the secondary item according to the state fold/unfold of the principal item
				// if the state fold/unfold of the principal item is undefined, it means that we are adding a secondary item to an item
				// which was not principal before, so set invisibility to the value of autoFold
				var boolVisibility = (this._isItemFolded(principalEntry) === undefined) ? this.getAutoFold() : this._isItemFolded(principalEntry);
				this._setItemInvisibility(secondaryItem, boolVisibility);
				return true;
			} else {
				return false;
			}
		},

		/**
		 * @ignore
		 * Update, if needed, a principal item:
		 *  - set the sepcific field (_subType = this.PRINCIPAL_ENTRY) and update graphically the item if it had no secondary items but it has now,
		 *  - delete the specific field (_subType) and update graphically the item if it had secondary items but it has no one anymore.
		 * Usefull to check if the principal entry has to be updated after an addition or deletion of secondary entries.
		 * @param {String/Number} principalEntryIndex the intIndex of the principal item to check for update
		 * @return {Boolean} return true if the item is a principal item (it has secondary item(s)), false if the item is not a principal item (it has no secondary items)
		 */
		_checkUpdatePrincipalItem: function (principalEntryIndex) {
			var principalItem = this.getItemByIntIndex(principalEntryIndex);
			if (this.getNbSecondaryItems(principalEntryIndex) > 0) {
				// If the principal item was not a principal item before, it is now, so set its internal variable and update it graphically
				if (!this.isPrincipalItem(principalItem)) {
					// set the principal item subtype to PRINCIPAL_ENTRY
					this._setItemSubType(principalItem, this.PRINCIPAL_ENTRY);
					// if the _isFolded property is not defined, set it to the value of autoFold
					if (this._isItemFolded(principalItem) === undefined) {
						this._setItemFolded(principalItem, this.getAutoFold())
					}
					// Update graphically the principal item so that it appears as a principal item
					this.updateItem(principalItem, true);
				}
				return true;
			} else {
				// If the principal item was a principal item before, it isn't anymore, so remove its internal variable and update it graphically
				if (this.isPrincipalItem(principalItem)) {
					// Remove subType PRINCIPAL_ENTRY
					delete principalItem._subType;
					// Remove isFolded flag
					delete principalItem._isFolded;
					// Remove the array of secondary items
					delete principalItem.arrSecondaryItems;
					// Update graphically the principal item so that it doesn't appear as a principal item anymore
					this.updateItem(principalItem, true);
				}
				return false;
			}
		},

		/**
		 * @ignore
		 * Refresh principal item to appear folded or unfolded, according to its state
		 *  - if the principal item has a unfoldedContent object defined (different data to display when the item is unfolded), refresh the whole item
		 *  - otherwise, just refresh the arraw (better performances).
		 * @param {Object} principalItem the principal item to update
		 */
		_updatePrincipalItem: function (principalItem) {
			if (principalItem.unfoldedContent !== undefined) {
				this.updateItem(principalItem, true);
			}
			else {
				this._updateArrawPrincipalItem(principalItem);
			}
		},


		/**
	 * Creates the root of the list
	 * @ignore
	 * @return this
	 */
		_initList : function () {
			this._list = this.newElement("ul")
			.mixin( {
				"className" : this.namespace.split(".")[1]
			})
			return this;
		},

		/**
	 * Converts a list of items to a list of LIs
	 * @ignore
	 * @param {Array} list a list of items to convert to LIs
	 * @return {Array} the list of dom elements and their references
	 */
		_buildList : function (list) {
			var nb ,  i, newLi, lisReference = [], domFragment = document.createDocumentFragment();
			nb = list.length;
			// As long as there'll be item to transform to domNodes
			for (i=0; i<nb; i++) {
				// Get the li's template
				newLi = this._getLiTemplate(list[i]);
				// And save its former item
				newLi._itemSource =  list[i];
				// Add its reference to the list
				lisReference.push(newLi);

				// Add specific information
				if (list[i].strType === "title")  {
					dojo.toggleClass(newLi, "collapsed", list[i].bCollapse);
				} else {
					!!list[i].bSelect && this._selectCurrent(newLi);
				}
				// Append the domNode to the list
				domFragment.appendChild(newLi);
			}
			// Returns both.
			return [domFragment, lisReference];
		},


		/**
	 * Returns the list of items in a category
	 * @ignore
	 * @param {Number} start the index of the item from which to search items
	 * @return {Array} the list of items
	 */
		_getItemsInCategory : function (start) {
			return this._getItemsList(start+1, 0, "normal", true);
		},


		/**
	 * From start, find the next category domNode
	 * @ignore
	 * @param {Number} start the index from which to search for the next category
	 * @return {HTMLElement} the LI dom node
	 */
		_getNextCatLi : function (start) {
			// Searching from next Li, looking for only 1 result of type "title"
			// Returns the first element in the list.
			return this._getLisList(start+1, 1, "title")[0];
		},

		/**
	 * From start, find the previous category domNode
	 * @ignore
	 * @param {Number} start the index from which to search for the previous category
	 * @return {HTMLElement} the LI dom node
	 */
		_getPreviousCatLi : function (start) {
			// Searching from this li, looking for only 1 li of type title,
			// don't stop until found, and reverse the search array
			return this._getLisList(start, 1, "title", false, true)[0];
		},


		/**
	 * From itemNb, find the previous category item
	 * @ignore
	 * @param {Number} itemNb the index from which to search for the previous category item
	 * @return {Object} the item object
	 */
		_getPreviousCatItem : function (itemNb) {
			// Searching from this item, looking for only 1 item of type title,
			// don't stop until found, and reverse the search array
			return this._getItemsList(itemNb, 1, "title", false, true)[0];
		},

		/**
	 * Return a list of items, searching with the parameters :
	 * @ignore
	 * @param {Number} start From start
	 * @param {Number} range A maximum of range elements
	 * @param {String} type With specific type or all
	 * @param {Boolean} stop Must stop right after the next one if it doesn't match required type.
	 * @param {Boolean} reverse Reverses the list
	 * @return {Array} the list of items found
	 */
		_getItemsList : function (start, range, type, stop, reverse) {
			// Of type... or all
			type = type || (this.getAutoCollapse() ? "title" : "*");
			var results = [];
			range = range || undefined;

			var arrCopy;
			if (reverse) {
				// If reverse, use the list from 0 to start point and reverse it.
				arrCopy = this.arrItems.slice(0,start).reverse();
			} else {
				// Else, use from start to end.
				arrCopy = this.arrItems.slice(start);
			}
			// If some items correspond to criterium, push them into results[]
			var func = function (item) {
				if (item.strType === type || type === "*") {
					results.push(item);
					return false;
				} else {
					// Stop if needed, else continue
					return stop || false;
				}
			};
			arrCopy.some(func);
			// Keep range
			results = results.slice(0, range);
			// Returns the results
			if (reverse) {
				return results.reverse();
			} else {
				return results;
			}

		},

		/**
	 * Return a list of Lis, searching with the parameters :
	 * @ignore
	 * @param {Number} start From start
	 * @param {Number} range A maximum of range elements
	 * @param {String} type With specific type or all
	 * @param {Boolean} stop Must stop right after the next one if it doesn't match required type.
	 * @param {Boolean} reverse Reverses the list
	 * @return {Array} the list of items found
	 */
		_getLisList : function (start, range, type, stop, reverse) {
			var results = [];
			var arrCopy;
			if (reverse) {
				arrCopy = this._LisInDom.slice(0,start).reverse();
			} else {
				arrCopy = this._LisInDom.slice(start);
			}
			var func = function (item) {
				if (this._getLiType(item) === type) {
					results.push(item);
					return false;
				} else {
					return stop || false;
				}
			};
			arrCopy.some(func, this);
			results = results.slice(0, range);
			if (reverse) {
				return results.reverse();
			} else {
				return results;
			}
		},

		/**
	 * Get an item's corresponding item in dom
	 * @ignore
	 * @param {Object} item the item to be searched for in the dom
	 * @return {HTMLElement} the item in the DOM
	 *
	 */
		_getItemInDom : function (item) {
			var itemInDom;
			var func = function (li) {
				if (li._itemSource == item)  {
					itemInDom = li;
					return li;
				}
			}
			this._LisInDom.some(func, this);
			return itemInDom;
		},

		/**
	 * From the items list, return the nodes that can be displayed
	 * @ignore
	 * @param {Number} start get the nodes from this item's index
	 * @param {Number} range the number of items to get
	 * @param {Boolean} reverse reverse the array?
	 * @return {Array} the list of items.
	 */
		_getNodesToDisplay : function (start, range, reverse) {
			range = range || undefined;
			var arrCopy;
			if (reverse) {
				arrCopy = this.arrItems.slice(0,start).reverse();
			} else {
				arrCopy = this.arrItems.slice(start);
			}
			var func = function (item) {
				// These items are all of type "title" and all the others that are not collapsed
				if (this._itemDisplayable(item)) {
					return item;
				}
			};
			arrCopy = arrCopy.filter(func, this).slice(0, range);
			if (reverse) {
				return arrCopy.reverse();
			} else {
				return arrCopy;
			}
		},

		/**
	 * Tests if an item is displayable or not
	 * @ignore
	 * @param {Object} item the item to test
	 * @return {Boolean} if the item is displayable
	 */
		_itemDisplayable : function (item) {
			if (item.strType === "title" ||
				!item.bCollapse) {
				if (!this._getItemInvisibility(item)) {
					return true;
				} else {
					return false;
				}
			} else {
				return false;
			}
		},

		/**
	 * Get the index of an item in the items list
	 * @ignore
	 * @param {Object} item the item to get the index from
	 * @return {Number} the index
	 */
		_getItemPosition : function (item) {
			return this.arrItems.indexOf(item);
		},

		/**
	 * Get the index of an item in the items list
	 * @ignore
	 * @param {String} intIndex the item's index
	 * @return {Number} the index of the item in the list
	 */
		_getItemPositionByIntIndex : function(intIndex) {
			return this._getItemPosition(this.findItem("intIndex", intIndex));
		},

		/**
	 * Get the next dom li
	 * @ignore
	 * @param {HTMLElement} li the dom node to search from
	 *
	 */
		_getNextLiInDom : function (li) {
			// Get the position of the given li
			var liPos = this._LisInDom.indexOf(li);
			// If it's in the array and is not the last one
			if (liPos >= 0 && liPos < this._getNbOfDomNodes()) {
				// Return the next element
				return this._LisInDom[liPos+1];
			} else {
				// Return false if there's not next element
				return false;
			}
		},

		/**
	 * Fills in the list from its top without moving the domNodes
	 * @ignore
	 */
		_fillList : function () {
			// Number of pixels to scroll after the dom nodes are added
			var nbOfPixelsToScroll = 0;
			// As long as there are nodes to be added
			while ((this._getNbOfExceedingNodes() < 0) && (this.unshift(1, false)) ) {
				// Increase the nb pixels to scroll by the height of an element
				// TODO : what if 50 changes?
				nbOfPixelsToScroll +=50;
			}
			// Scroll it
			if (nbOfPixelsToScroll > 0)
				this.domNode.scrollTop += nbOfPixelsToScroll;
		},

		/**
	 * Un/collapse a category
	 * @ignore
	 * @param {Object} item the category item
	 * @param {Boolean} bCollapse true if the category is to collapse
	 * @return {Array} the list of items that've been collapsed
	 */
		_setCategoryCollapse : function (item, bCollapse) {
			if (item) {
				// Its domNode
				this.setCollapse(item, bCollapse);
				!bCollapse && this._setCurrentUnCollapsed(item);
				// What's inside
				var itemsList = this._getItemsInCategory(this._getItemPosition(item));
				this._setItemsCollapse(itemsList, bCollapse);
				// Returns the list of items collapsed
				return itemsList;
			} else {
				return false;
			}
		},

		/**
	 * Returns the number of lis displayed in a category
	 * @ignore
	 * @param {Object} elem the li of the category
	 * @return {Number} the number of lis displayed
	 *
	 */
		_getNbLisDisplayedInCat : function (elem) {
			// Get the position of this li
			var start = this.getLiPosition(elem);
			// find the next category
			var nextCat = this.getLiPosition(this._getNextCatLi(start));
			nextCat = nextCat >= 0 ? nextCat : this._LisInDom.length;
			return nextCat-start-1;
		},
		
				/**
	 * Select an li
	 * @ignore
	 * @param {HTMLElement} liNode the li domNode to select
	 * @param {Boolean} dontExecCallback if the callback must be executed or not
		 * @param {Boolean} dontUnfold if the entry must be unfold or not
		 * @param {Boolean} boolByClick if _selectLi was called following to a user click (not a click called by the webapp)
		 */
		_selectLi : function (liNode, dontExecCallback, dontUnfold, boolByClick) {
			if (liNode && this._selectCurrent(liNode,liNode._itemSource.boolNotSelectable)) {
			    if (!dontExecCallback && this.callback && liNode._itemSource) {
					this._executeCallback("callback", liNode._itemSource.intIndex, boolByClick);
				}
				// if auto unfold on click is enabled and the selected entry is a principal item, unfold it
				if (!dontUnfold && this.getAutoUnfoldOnClick() && this.isPrincipalItem(liNode._itemSource)) {
					this.unfoldPrincipalItem(liNode._itemSource);
				}
				// if the selected entry is a secondary item and the principal item is folded, unfold it
				if (!dontUnfold && this.isSecondaryItem(liNode._itemSource) && this._isItemFolded(this.getPrincipalItem(liNode._itemSource))) {
					this.unfoldPrincipalItem(this.getPrincipalItem(liNode._itemSource));
				}
				// if auto fold is enabled and the selected entry is not a principal or a secondary item, fold all the principal items
				if (!dontUnfold && this.getAutoFold() && !this.isPrincipalItem(liNode._itemSource) && !this.isSecondaryItem(liNode._itemSource)) {
					this.foldAllPrincipalItems();
			    }
			    return liNode;		
			} else {
				return false;
			}
		},

		/**
	 * Set current selected dom node info
	 * @ignore
	 * @param {HTMLElement} elem the element that is selected
	 */
		_setCurrentSelected : function (elem) {
			this._currentSelected = elem;
		},

		/**
	 * Select a dom node and unselect the old one.
		 * Note that if the dom node to select corresponds to a secondary item, the associated principal item's dom node will also appear selected.
	 * @ignore
	 * @param {HTMLElement} elem the dom node to select
	 * @param {boolean} boolNotSelectable specify if the element appears or not graphically selected
	 * @return {HTMLElement} the selected element or false if it can't be selected
	 */
		_selectCurrent : function (elem, boolNotSelectable) {
			if (this._getLiType(elem) === "normal") {
				this._deSelectCurrent(elem);
				if (!this.boolNotSelectable && !boolNotSelectable) {
				    this._toggleSelectEntryByLi(elem, true);
					// if the item to select is a secondary item, select its principal item
					if (elem._itemSource && this.isSecondaryItem(elem._itemSource)) {
						var liPrincipalItem = this._getItemInDom(this.getPrincipalItem(elem._itemSource));
						// the principal item can not beeing in the dom (pagination), so check if it exists before trying to unselect it
						// if the principal item is not displayed currently, set it flag bSelect so that it will be displayed as unselected if we scroll to its position
						if (liPrincipalItem) {
							this._toggleAppearSelectedEntryByLi(liPrincipalItem, true);
						}
						else {
							this._toggleAppearSelectedItem(this.getPrincipalItem(elem._itemSource), true);
						}
					}
				}
				this._setCurrentSelected(elem);
				return elem;
			} else {
				return false;
			}
		},

		/**
	 * Toggle select an entry given its li
	 * @ignore
	 * @param {HTMLElement} li the li domNode to de/select
	 * @param {Boolean} bSelect select or deselect
	 * @return {HTMLElement} the selected li
	 */
		_toggleSelectEntryByLi : function (li, bSelect) {
			this._toggleSelectLi(li, bSelect);
			this._toggleSelectItem(li._itemSource, bSelect);
			return li;
		},


		/**
	 * Toggle select an entry given its item
	 * @ignore
	 * @param {Object} item the item to de/select
	 * @param {Boolean} bSelect select or deselect
	 * @return {Object} the selected item
	 */
		_toggleSelectEntryByItem : function (item, bSelect) {
			this._toggleSelectItem(item, bSelect);
			var func = function (li) {
				if (li._itemSource == item) {
					return this._toggleSelectLi(li, bSelect);
				}
			};
			return this._LisInDom.filter(func, this);
		},

		/**
	 * Add/Remove the selected class of a domNode
	 * @ignore
	 * @param {HTMLElement} li the li node to modify
	 * @param {Boolean} boolSelect true : add, false : remove
	 * @return dojo.toggleClass return
	 */
		_toggleSelectLi : function (li, boolSelect) {
			return dojo.toggleClass(li, "selected", boolSelect);
		},

		/**
	 * Set the selected property of an item
	 * @ignore
	 * @param {Object} item the item to modify
	 * @param {Boolean} boolSelect true : selected, false : not selected
	 * @return {Object} the item
	 */
		_toggleSelectItem : function (item, boolSelect) {
			return item.bSelect = boolSelect;
		},


		/**
		 * Toggle appear selected or not appear selected an entry given its li
		 * @ignore
		 * @param {HTMLElement} li the li domNode to toggle appear selected or not appear selected
		 * @param {Boolean} boolAppearSelected appear selected or not appear selected
		 * @return {HTMLElement} the appear selected li
		 */
		_toggleAppearSelectedEntryByLi : function (li, boolAppearSelected) {
			this._toggleAppearSelectedLi(li, boolAppearSelected);
			this._toggleAppearSelectedItem(li._itemSource, boolAppearSelected);
			return li;
		},

		/**
		 * Toggle appear selected or not appear selected an entry given its item
		 * @ignore
		 * @param {Object} item the item to toggle appear selected or not appear selected
		 * @param {Boolean} boolAppearSelected appear selected or not appear selected
		 * @return {Object} the appear selected item
		 */
		_toggleAppearSelectedEntryByItem : function (item, boolAppearSelected) {
			this._toggleAppearSelectedItem(item, boolAppearSelected);
			var func = function (li) {
				if (li._itemSource == item) {
					return this._toggleAppearSelectedLi(li, boolAppearSelected);
				}
			};
			return this._LisInDom.filter(func, this);
		},

		/**
		 * Add/Remove the appearSelected class of a domNode
		 * @ignore
		 * @param {HTMLElement} li the li node to modify
		 * @param {Boolean} boolAppearSelected true : add, false : remove
		 * @return dojo.toggleClass return
		 */
		_toggleAppearSelectedLi : function (li, boolAppearSelected) {
			return dojo.toggleClass(li, "appearSelected", boolAppearSelected);
		},

		/**
		 * Set the appear selected property of an item
		 * @ignore
		 * @param {Object} item the item to modify
		 * @param {Boolean} boolAppearSelected true : appear selected, false : not appear selected
		 * @return {Object} the item
		 */
		_toggleAppearSelectedItem : function (item, boolAppearSelected) {
			return item.bAppearSelected = !!boolAppearSelected;
		},

		/**
		 * Get the appear selected property of an item
		 * @ignore
		 * @param {Object} item the item to test
		 * @return {Boolean} true if the item has the property appear selected enabled, false otherwise
		 */
		_isAppearSelected : function (item) {
			return !!item.bAppearSelected;
		},

		/**
	 * Deselect current item/domNode
	 * @ignore
		 * @param {HTMLElement} elem (optional parameter) the dom node currently beeing selected
		 */
		_deSelectCurrent : function (elem) {
			if (this.getCurrentSelected() != null) {
				this._toggleSelectEntryByLi(this.getCurrentSelected(), false);
				// if the current item to deselect is a secondary item, also deselect its principal item
				// (only if the item currently beeing selected is not a secondary item of the same principal item than the one to deselect)
				if (this.isSecondaryItem(this.getCurrentSelectedItem()) && this.getPrincipalItem(this.getCurrentSelectedItem()) && (!elem || this.getPrincipalItemIndex(elem) !== this.getPrincipalItemIndex(this.getCurrentSelectedItem()))) {
					var liPrincipalItem = this._getItemInDom(this.getPrincipalItem(this.getCurrentSelectedItem()));
					// the principal item can not beeing in the dom (pagination), so check if it exists before trying to unselect it
					// if the principal item is not displayed currently, set it flag bSelect so that it will be displayed as unselected if we scroll to its position
					if (liPrincipalItem) {
						this._toggleAppearSelectedEntryByLi(liPrincipalItem, false);
					}
					else {
						this._toggleAppearSelectedItem(this.getPrincipalItem(this.getCurrentSelectedItem()), false);
					}
				}
			}
		},

		/**
	 * Set a list of items to un/collapse
	 * @ignore
	 * @param {Array} list the list of items to set to un/collapse
	 * @param {Boolean} bCollapse the value of collapse
	 * @return {Array} the list of modified items
	 */
		_setItemsCollapse : function (list, bCollapse) {
			var func = function (item) {
				item.bCollapse = bCollapse;
			}
			list.map(func);

			return list;
		},

		/**
	 * Set an item to un/collapse
	 * @ignore
	 * @param {Object} item the item to set to un/collapse
	 * @param {Boolean} bCollapse the value of collapse
	 * @return {Boolean} the value of bCollapse
	 */
		_setItemCollapse : function (item, bCollapse) {
			if (item && (typeof item == "object")) {
				item.bCollapse = bCollapse;
				return bCollapse;
			}
		},

		/**
	 * Set the current uncollapsed property
	 * @ignore
	 * @param {Object} item the category's item
	 * @return {Object} the category's item
	 */
		_setCurrentUnCollapsed : function (item) {
			return this._currentUnCollapsed = item;
		},

		/**
	 * Get the current uncollapsed category
	 * @ignore
	 * @return {Object} the category's item
	 */
		_getCurrentUnCollapsed : function () {
			return this._currentUnCollapsed;
		},

		/**
	 * Update an item's text (and its Li)
	 * @ignore
	 * @param {Number} itemNb the index of the item
	 * @param {String} which the name of the property to modify
	 * @param {String} newText the new text to assign
	 */
		_updateItemText : function (itemNb,  which, newText) {
			// If its attribute has a changeText function
			if (this.arrItems[itemNb][which] && (typeof this.arrItems[itemNb][which].changeText == "function")) {
				// Then call it to change the text.
				this.arrItems[itemNb][which].changeText(newText);
				return true;
			} else {
				// Else find the domNode to refresh in the list of lis.
				var func = function (item) {
					// Is it this one?
					if (item._itemSource === this.arrItems[itemNb]) {
						// If the new text is a fresh i18nObject
						if (typeof newText === "object" && newText.isI18Ned) {
							// Do i18n stuff
							ICTouchAPI.i18nServices.addI18nHandler(this, newText.fillInnerHTML(item));
						} else {
							// Or just innerHTML the new text.
							item.getElementsByClassName(which)[0].innerHTML = newText;
						}
					}
				};
				return this._LisInDom.forEach(func, this);
			}
		},

		/**
	 * Get an item invisibility property
	 * @ignore
	 * @param {Object} item the item from which to get the property
	 * @return {Boolean} the value
	 */
		_getItemInvisibility : function (item) {
			return item._invisible;
		},

		/**
	 * Set an item invisibility property
	 * @ignore
	 * @param {Object} item the item to set the property to
	 * @param {Boolean} invisibility the value to set
	 * @return {Boolean} the value
	 */
		_setItemInvisibility : function (item, invisibility) {
			return item._invisible = !!invisibility;
		},
		
		/**
		 * Get the item's subType property
		 * @ignore
		 * @param {Object} item the item from which to get the property
		 * @return {String} the subType value
		 */
		_getItemSubType : function (item) {
			return item._subType;
		},

		/**
		 * Set the item's subType property
		 * @ignore
		 * @param {Object} item the item to set the property to
		 * @param {Boolean} subType the value to set
		 * @return {String} the subType value
		 */
		_setItemSubType : function (item, subType) {
			return item._subType = subType;
		},

		/**
		 * Get the item's folded property
		 * @ignore
		 * @param {Object} item the item from which to get the property
		 * @return {Boolean} the value
		 */
		_isItemFolded : function (item) {
			return item._isFolded;
		},

		/**
		 * Set the item's folded property
		 * @ignore
		 * @param {Object} item the item to set the property to
		 * @param {Boolean} folded the value to set
		 * @return {Boolean} the value
		 */
		_setItemFolded : function (item, folded) {
			return item._isFolded = !!folded;
		},

		/**
	 * METHODE 1.18
	 * NE PAS UTILISER EN DEHORS DE CONTACTS
	 * DOIT DISPARAITRE AU PROFIT DE FILTRES/TRIS DES DATASTORE
	 * @ignore
	 */
		_DONT_USE_locateItem : function (what, superiorTo) {
			var func = function (item) {
				var superior = false;
				for(var i in what) {
					if (item.objContact[what[i]].toLowerCase() > superiorTo[i].toLowerCase()) {
						superior = true;
						break;
					}
					else if (item.objContact[what[i]].toLowerCase() < superiorTo[i].toLowerCase()) {
						break;
					}
				}
				if(superior === true) {
					return item;
				}
				else {
					return null;
				}
			}
			return this._getItemPosition(this.arrItems.filter(func)[0]);
		},

		/**
	 * Remove a dom element given its item.
	 * By default, add new elements to the visible list
	 * @ignore
	 * @param {Object} item the item to remove
	 * @param {Boolean} dontAdd to set to true if the list must not autocomplete itself after removing
	 * @return {HTMLElement} the li that is removed
	 */
		_removeItemInDom : function (item, dontAdd) {
			var removed = this._removeOneLi(this._LisInDom.indexOf(this._getItemInDom(item)));
			!dontAdd && (this._getNbOfExceedingNodes() < 0) && this.push(-this._getNbOfExceedingNodes(), true);
			!dontAdd && (this._getNbOfExceedingNodes() < 0) && this.unshift(-this._getNbOfExceedingNodes(), true);
			return removed;
		},

		/**
		 * Check if the list is empty and display the empty state if needed
		 * @ignore
		 */
		_checkForEmpty : function() {
			if (this._emptyState) {
				var isEmpty = this.arrItems.length <= 0;
				this._emptyState.setVisibility(isEmpty);
				if (typeof this.domNode.deactivate === "function" && isEmpty) {
					this.domNode.deactivate();
				}
				else if (typeof this.domNode.activate === "function" && !isEmpty) {
					this.domNode.activate();
				}
			}
		}

	});
