/**
 * @class UIElements.Keyboard.KeyBase
 * @ignore
 * @extends UIElements._base
 * Abstract Class<br>
 * Represents a key of the keyboard
 */
dojo.provide("UIElements.Keyboard.KeyBase");
dojo.declare("UIElements.Keyboard.KeyBase",
	[UIElements._base, dijit._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * name of button
		 * @property
		 * @type String
		 */
		strButtonName			: null,
		/**
		 * button type
		 * @property
		 * @type String
		 */
		strButtonType			: "button",
		/**
		 * Css class for keys with notification
		 * @property
		 * @type String
		 */
		strNotification			: null,
		/**
		 * CssClass of the key
		 * @property
		 * @type String
		 */
		strCustomClass			: null,
		/**
		 * Text of the key
		 * @property
		 * @type String
		 */
		strButtonText			: null,
		/**
		 * icon of the key
		 * @property
		 * @type String
		 */
		strButtonIcon			: null,
		/**
		 * sub text of the key
		 * @property
		 * @type String
		 */
		strSubText				: null,

		/**
		 * Width of the key
		 * @property
		 * @type Number
		 */
		intWidth				: 1,
		/**
		 * Height of the key
		 * @property
		 * @type Number
		 */
		intHeight				: 1,
		/**
		 * Function called on mouse down event
		 * @property
		 * @type Function
		 */
		funcCallback			: null,
		/**
		 * Function called on long press on button. Mutually exclusive with autorepeat
		 * @property
		 * @type Function
		 */
		funcCallbackLong		: null,
		/**
		 * Function called on mouse up event
		 * @property
		 * @type Function
		 */
		funcCallbackRelease		: null,
		/**
		 * Function called on click event
		 * @property
		 * @type Function
		 */
		funcCallbackClick		: null,
		/**
		 * Should the callback function be repeated when the user keeps pressing. Mutually exclusive with long press
		 * @property
		 * @type Boolean
		 */
		boolAutorepeat			: false,
		/**
		 * Attach on parent node, attach mousedown and mouseup on parent instead of domNode.
		 * Useful when the whole td should be active instead of the button
		 * @property
		 * @type Boolean
		 */
		boolAttachParent		: true,


		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @private
		 */
		myFeedback				: null,
		/**
		 * @private
		 */
		domButtonIcon			: null,
		/**
		 * @private
		 */
		domButtonText			: null,
		/**
		 * @private
		 */
		domSubText				: null,
		/**
		 * @private
		 * Base class of this UI
		 */
		_strBaseClass			: "Key keyMainDiv",
		/**
		 * setTimeout handler for the long press callback feature
		 * @private
		 */
		_handlerLongPress		: null,
		/**
		 * @private
		 */
		_autorepeatTimeout		: null,
		/**
		 * @private
		 */
		_autorepeatInterval		: null,

		/**
		 * @private
		 * Are mouse events connected
		 */
		_eventsAttached			: false,

		/**
		 * @private
		 */
		attributeMap: {
			strButtonText: {
				node: "domButtonText",
				type: "innerHTML"
			},
			strSubText: {
				node: "domSubText",
				type: "innerHTML"
			}
		},

		/**
		 * @private
		 * True untill the postCreate is finished
		 */
		_initializing : true,

		/* ------------------------------------ Constructor --------------------------------------- */

		constructor : function(args) {
			this.myFeedback = ICTouchAPI.feedbackServices.initFeedback("click");
		},

		postMixInProperties : function() {
			this.strButtonText = this.strButtonText || "";
			this.strSubText = this.strSubText || "";

			//
		},

		postCreate : function () {
			if (this.strButtonName) {
				this.domNode.setAttribute("buttonname", this.strButtonName);
			}
			this._updateClasses();
			this._attachEventParent();

			this._initializing = false;
		},

		placeAt: function() {
			/*
			 Keyboard does create this UIElements a domNode so in postCreate this.domNode.parentNode is null
			 But after placeAt has been called there should be a parentNode
			*/
			this.inherited(arguments);
			this._attachEventParent();
		},

		destroy: function() {
			this._clearTimers();
			this.inherited(arguments);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * Replaces key text
		 * @param {String} strText new text for the key
		 */
		replaceText: function (strText) {
			if(strText==null || strText=="") {
				strText = " ";
			}
			if (strText.isI18Ned) {
				strText = strText.getTranslation();
			}
			if (strText != this.strButtonText) {
				this.attr("strButtonText", strText);

				// Webkit bug workaround
				this.domButtonText.style.opacity = "0.99";
				this.domButtonText.style.opacity = "";
			}
		},

		/**
		 * Replaces key icon
		 * @param {String} strIcon new icon for the key
		 */
		replaceIcon: function (strIcon) {
			this.attr("strButtonIcon", strIcon);
		},

		/**
		 * Remove key icon
		 */
		removeIcon: function() {
			this.attr("strButtonIcon", null);
		},

		/**
		 * Remove key icon
		 * @param {String} strIcon new icon for the key
		 */
		setIcon: function(strIcon) {
			this.attr("strButtonIcon", strIcon);
			this._updateClasses();
		},

		/**
		 * Set the long press callback
		 * @param {Function} longCallback function to call when the button is pressed for a while, null does disable the effect
		 */
		setLongCallback: function(longCallback) {
			this._clearTimers();
			this.funcCallbackLong = longCallback;
		},

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * Because marketing wants the whole zone to be active and not only the button itself, attach events to our parent
		 * @private
		 */
		_attachEventParent: function() {
			// Node where all the events will be attached
			var node = this.boolAttachParent ? this.domNode.parentNode : this.domNode;
			if (!this._eventsAttached && node) {
				this.connect(node, "mousedown", this._startHighlight);
				if( typeof this.funcCallbackClick === "function" ) {
					this.connect(node, "onclick", this._actionMouseUp);
				}
				dojo.toggleClass(node, "debug_zoneActive", this.strButtonType !== "empty");
				this._eventsAttached = true;
			}
		},

		_startHighlight: function(e, args) {
			var boolHighlight = this.inherited(arguments);
			if(boolHighlight && typeof this.funcCallbackClick !== "function") {
				this.actionNotification(e);
			}
		},

		 _stopHighlight : function() {
			this._clearTimers();
			var clicked = this.inherited(arguments);
			if(clicked) {
				this.releaseNotification();
			}
		},

		_actionMouseUp:function(clickEvent) {
			var boolHighlight = this.inherited(arguments);
 	                this._feedBack(clickEvent); 
			// Behavior for Cancel, favorites, contact...
			if( typeof this.funcCallbackClick == "function" ) {
				this.funcCallbackClick(this.strButtonText);
			}
			dojo.stopEvent(clickEvent);
		},


		/**
		 * Launch this.funcCallback on mouse down event
		 *  @param {Event} clickEvent mouse down event
		 */
		actionNotification:function(clickEvent){
			this._feedBack(clickEvent);
			// Use a different behavior is a longpress callback is present
			if (typeof this.funcCallbackLong == "function") {
				this._handlerLongPress = setTimeout(dojo.hitch(this, this.actionLongpress), 1000);
			}
			// Normal behavior, callback on mousedown
			if( typeof this.funcCallback == "function" ) {
				// If auto-repat feature is wanted
				if (this.boolAutorepeat) {
					this._autorepeatTimeout = setTimeout(dojo.hitch(this, this._repeatNotification, true), 1000);
				}
				this.funcCallback(this.strButtonText);
			}
			dojo.stopEvent(clickEvent);
		},

		actionLongpress: function() {
			// Longpress has been triggered don't callback on mouseup
			this._handlerLongPress = null;
			ICTouchAPI.keyboardServices.delChars();
			this.funcCallbackLong(this.strButtonText);
		},

		/**
		 * Launch this.funcCallbackRelease on mouse up event
		 */
		releaseNotification: function() {
			this._clearTimers();

			if (typeof this.funcCallbackRelease == "function") {
				this.funcCallbackRelease(this.strButtonText);
			}
		},

		/**
		 * @private
		 */
		_repeatNotification: function(first) {
			// Because we check if the function exist before creating the timeout we should always have a callback
			this.funcCallback(this.strButtonText);
			this._autorepeatTimeout = null;
			if (first) {
				this._autorepeatInterval = setInterval(dojo.hitch(this, this._repeatNotification, false), 100);
			}
		},

		/**
		 * @private
		 */
		_clearTimers: function() {
			if (this._handlerLongPress) {
				clearTimeout(this._handlerLongPress);
			}
			if (this._autorepeatInterval) {
				clearInterval(this._autorepeatInterval);
			}
			if (this._autorepeatTimeout) {
				clearTimeout(this._autorepeatTimeout);
			}
			this._autorepeatInterval = null;
			this._autorepeatTimeout = null;
			this._handlerLongPress = null;
		},

		/**
		 * @private
		 */
		_feedBack:function(clickEvent){
			// Don't feedback on empty buttons
			if( this.strButtonType !== "empty" ) {
				this.myFeedback.feedback(clickEvent);
			}
		},

		/**
		 * @private
		 */
		_getIconSize: function() {
			// Special case for dialpads ok button ( rotated button )
			if( this.strCustomClass === "ok" && this.intWidth > 1 ) {
				return "48";
			}
			return ( this.intWidth > 1 && this.intHeight > 1 ) ? "64" : "32";
		},

		/**
		 * @private
		 */
		_setStrNotificationAttr: function(notif) {
			if( this.strNotification != null ) {
				dojo.removeClass(this.domNotification, this.strNotification);
			}
			if( notif != null ) {
				dojo.addClass(this.domNotification, notif);
			}
			this.strNotification = notif;
		},

		/**
		 * @private
		 */
		_setStrButtonIconAttr: function (strIcon) {
			if (this.strButtonIcon != strIcon || this._initializing ) {
				if( typeof this.strButtonIcon == "string" ) {
					var oldIcon = this.strButtonIcon + "-" + this._getIconSize();
					dojo.removeClass(this.domButtonIcon, oldIcon);
				}
				this.strButtonIcon = strIcon;
				if( typeof this.strButtonIcon == "string" ) {
					strIcon = strIcon + "-" + this._getIconSize();
					dojo.addClass(this.domButtonIcon, strIcon);
				}
				this._updateClasses();
			}
		},

		/**
		 * @private
		 */
		_setStrCustomClass: function(value) {
			if (this.strCustomClass != value || this._initializing) {
				this.strCustomClass = value;
				this._updateClasses();
			}
		},

		/**
		 * @private
		 */
		_setIntWidthAttr: function(value) {
			if (this.intWidth != value || this._initializing) {
				this.intWidth = value;
				this._updateClasses();
			}
		},

		/**
		 * @private
		 */
		_setIntHeightAttr: function(value) {
			if (this.intHeight != value || this._initializing) {
				this.intHeight = value;
				this._updateClasses();
			}
		},

		/**
		 * @private
		 */
		_updateClasses: function() {
			var name = this._strBaseClass;
			if( this.strButtonType ) {
				name += " " + this.strButtonType;
			}
			if( this.strCustomClass ) {
				name += " " + this.strCustomClass;
			}
			if( this.strButtonIcon ) {
				name += " icon";
			}
			if( this.strButtonText ) {
				name += " text";
			}
			name = name + " width-" + this.intWidth + " height-" + this.intHeight;
			this.domNode.className = name;
		}
	}
);
