/**
* @class UIElements.InputField.InputFieldControlBase
* @extends UIElements._base
* Abstract Class - Do not use it directly
*/
dojo.provide("UIElements.InputField.InputFieldControlBase");
dojo.declare("UIElements.InputField.InputFieldControlBase",
	[UIElements._base, dojox.dtl._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Text displayed into the InputField element
		 * @cfg {String} strInputText
		 */
		strInputText	: "",

		/**
		 * Text displayed into the InputField (used when boolPassword = true)
		 * @cfg {String} strInputContent
		 */
		strInputContent		: "",

		/**
		 * Keyboard to display. See {@link ICTouchAPI.KeyboardTypes} for more information about the available keyboards
		 * @cfg {String} keyboardType
		 */
		keyboardType	: "",

		/**
		 * Initial mode for alphanumeric keyboard (ICTouchAPI.KeyboardAlphanumMode enumeration) (default: NORMAL)
		 * @property
		 * @type String
		 */
		strDefaultMode	: null,

		/**
		 * Default text to be displayed in the keyboard when activated
		 * @cfg {String} strKeyboardDefaultText
		 */
		strKeyboardDefaultText	: null,

		/**
		 * Callback fired when the OK button is pressed
		 * @cfg {Function} funcCallback
		 */
		funcCallback	: null,

		/**
		 * Callback fired when the OK button of the keyboard is pressed
		 * @cfg {Function} funcCallbackOk
                 * @ignore
		 */
		funcCallbackOk	: null,

		/**
		 * Callback fired when the EXIT button of the keyboard is pressed
		 * @cfg {Function} funcCallbackExit
		 */
		funcCallbackExit: null,

		/**
		 * Name of the group of fields this input belongs to.
		 * @property
		 * @type Function
                 * @ignore
		 */
		linkedFields	: "",

		/**
		 * Displayed an InputField with several lines. Not compatible with a PresentationList (Should be set to false when used with a PresentationList)
		 * @cfg {Boolean} isMultiline
		 */
		isMultiline	: false,

		/**
		 * Maximum number of lines allowed. Not compatible with a PresentationList (Should not be used with a PresentationList)
		 * @cfg {Number} intMaxLine
		 */
		intMaxLine		: 3,

/**
		 * Should the inputfield directly use the max line number, even if the content is empty ? (note, isMultiline should be set to true)
		 * @cfg {Boolean} boolForceMaxLineSize
		 */
		boolForceMaxLineSize : false,

		/**
		 * if true, the value of the input filed has chanegd
		 * @property
		 * @type Boolean
                 * @ignore
		 */
		valueChanged	: false,

		/**
		 * Label of the keyboard OK button
		 * @cfg {String} strOK
		 */
		strOk			: null,

		/**
		 * Label of the Exit button
		 * @cfg {String} strExit
		 */
		strExit			: null,

		/**
		 * Callback called when clicking outside the keyboard
		 * @property
		 * @type Function
                 * @ignore
		 */
		actionOutsideClick : null,

		/**
		 * Display InputField as a password field. Show stars instead of actual characters.
		 * @cfg {Boolean} boolPassword
		 */
		boolPassword : false,

		/**
		 * Enable of disable the inputField. If the InputField is disabled, the keyboard will not be displayed.
		 * @cfg {Boolean} boolEnabled
		 */
		boolEnabled : true,

		/**
		 * Hides or not stars if password is empty
		 * @cfg {Boolean} hideStarsIfPasswordEmpty
		 */
		hideStarsIfPasswordEmpty : false,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		numLockKeyboard	: 0,
		/**
		 * @ignore
		 */
		strEventId		: null,
		/**
		 * @ignore
		 */
		domInput		: null,
		/**
		 * @ignore
		 */
		domTextarea		: null,
		/**
		 * @ignore
		 */
		oldScrollHeight : 0,
		/**
		 * @ignore
		 */
		domExtraLabel	: null,
		/**
		 * @ignore
		 */
		domExtra		: null,
		/**
		 * @ignore
		 */
		domExtraIcon	: null,
		/**
		 * @ignore
		 */
		_extraCallback	: null,
		/**
		 * @ignore
		 */
		_extraAutorepeat: false,
		/**
		 * @ignore
		 */
		_autorepeatInterval: null,
		/*
		 * @ignore
		 */
		_autorepeatTimeout: null,
		/**
		 * @ignore
		 */
		_extraLabelSubscribe : null,
		/**
		 * @ignore
		 */
		myFeedback				: null,

		/* ------------------------------------ Constructor --------------------------------------- */

		constructor: function () {
			this._arrInputThreshold = [];
			this.myFeedback = ICTouchAPI.feedbackServices.initFeedback("click");
		},

		postMixInProperties: function() {
			// Backward compatibility
			if (typeof this.isMultiLines === "boolean")
			{
				this.isMultiline = this.isMultiLines;
				delete this.isMultiLines;
			}
		},

		postCreate : function () {
			if (this.strInputIcon){
				dojo.addClass(this.domWithIcon, "opIcon");
				dojo.addClass(this.domIcon, this.strInputIcon);
			}
			if(this.strInputText && !this.strInputContent) {
				this.strInputContent = this.strInputText;
			}
			if(!this.strInputText && this.strInputContent) {
				this.strInputText = this.strInputContent;
			}
			if (!this.boolEnabled){
				this.getInputFieldObject().setAttribute("disabled","disabled");
			}
			this.setMultiline(this.isMultiline);
		},

		destroy: function() {
			if (this._extraLabelSubscribe) {
				dojo.unsubscribe(this._extraLabelSubscribe);
				this._extraLabelSubscribe = null;
			}
			this._clearExtraTimers();
			this.inherited(arguments);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * Get the text of this element
		 * @return {String} text of element
		 */
		getInputFieldContent: function () {
			return dojox.dtl.filter.htmlstrings.striptags(this.strInputContent);
		},

		/**
		 * Set the text of this element
		 * @param {String} strContent Text to display in this element
		 * @param {Boolean} boolCallback Execute callback or not
		 * @param {Boolean} hideStarsIfPasswordEmpty In password mode, the empty string will be replace by "*****", if parameter is set to true, it will not replace the content in case of an empty string
		 * @return {String} text of element
                 * @ignore
		 */
		setInputFieldContent: function (strContent, boolCallback, hideStarsIfPasswordEmpty) {
			var field = this.getInputFieldObject();
			this.strInputText = strContent;
			this.strInputContent = strContent;
			if(this.boolPassword && (this.strInputText || (!hideStarsIfPasswordEmpty && !this.hideStarsIfPasswordEmpty))){
				this._convertInputToPassword(this.strInputText.length);
			}
			if (boolCallback == undefined || boolCallback == null || boolCallback == true){
				if(this.funcCallback){
					this.funcCallback(strContent);
				}
			}
			field.value = this.strInputText;
			field.selectionStart = this.strInputText.length;
			field.selectionEnd = this.strInputText.length;
			this.domTextarea.rows = 1;
			this._updateInputFieldRows();

			return strContent;
		},

		/**
		 * Set a new text int the InputField
		 * @param {String} newValue New text to display
		 */
		setValue: function (newValue) {
			this.setInputFieldContent(newValue, false);
		},

		/**
		 * Should this field be a password field. This will erase the content
		 * @param {Boolean} password is this text field a password field
		 */
		setPassword: function (password) {
			this.boolPassword = password;
			this.setInputFieldContent("", false);
		},

		/**
		 * Retrieve maximum number of lines allowed
		 * @return {Number}  maximum number of lines
		 */
		getMaxLine: function() {
			return this.intMaxLine;
		},

		/**
		 * Set maximum number of lines allowed
		 * @param {Number} nbLine new maximum number of lines
		 */
		setMaxLine: function(nbLine) {
			this.intMaxLine = nbLine;
		},

		/**
		 * Retrieve html dom element of the input field
		 * @return {HTMLElement}  Html dom element
                 * @ignore
		 */
		getInputFieldObject: function () {
			return this.isMultiline ? this.domTextarea : this.domInput;
		},

		/**
		 * Retrieve html dom element's length
		 * @return {Number}  Html dom element's length
                 * @ignore
		 */
		getInputFieldLength: function () {
			return this.getInputFieldObject().length;
		},

		/**
		 * Retrieve input field selection start index and selection end index
		 * @return {Array}  Input field selection start and end
		 */
		getSelection: function () {
			var field = this.getInputFieldObject();
			return [
			field.selectionStart,
			field.selectionEnd
			];
		},

		/**
		 * Retrieve cursor position in the input field
		 * @return {Number}  cursor position
		 */
		getCaretPos: function () {
			return this.getInputFieldObject().selectionStart;
		},

		/**
		 * Retrieve Multiline status
		 * @return {Boolean}  true if multiline is allowed
		 */
		getMultiline: function () {
			return this.isMultiline;
		},

		/**
                 * Switch to a multi lines or single line InputField
                 * @param {Boolean} boolMultiline True to switch to multi lines InputField
                 */
                setMultiline: function(boolMultiline) {
			var multiline = false;
			if(boolMultiline===true) {
				multiline = true;
			}
			this.isMultiline = multiline;
			dojo.toggleClass(this.domInput, "hidden", multiline);
			dojo.toggleClass(this.domTextarea, "hidden", !multiline);
			this.setInputFieldContent(this.strInputContent, false, this.hideStarsIfPasswordEmpty);
		},

		/**
		 * Retrieve ForceMaxLineSize status
		 * @return {Boolean}  true if ForceMaxLineSize is set
		 */
		getForceMaxLineSize: function () {
			return this.boolForceMaxLineSize;
		},

		/**
		* Set ForceMaxLineSize
		* @param {Boolean} boolForceMaxLineSize True to activate boolForceMaxLineSize
		*/
		setForceMaxLineSize: function(boolForceMaxLineSize) {
			this.boolForceMaxLineSize = boolForceMaxLineSize;
			// update the textarea
			this._updateInputFieldRows();
		},

		/**
		 * Add or remove the extra button
		 * @param {Function} callback function to call when the user click or null to remove the button
		 * @param {Boolean} should the button autorepeat when the button stay pressed
		 * @param {String} label to show or null to remove the label
		 * @param {String} icon for the button
                 * @ignore
		 */
		setExtraButton: function(callback, autorepeat, label, icon) {
			// Because this label can't be handled by this widget lifecycle
			if (this._extraLabelSubscribe) {
				dojo.unsubscribe(this._extraLabelSubscribe);
				this._extraLabelSubscribe = null;
			}

			this.domExtraLabel.innerHTML = "";
			if (typeof callback == "function") {
				this._extraCallback = callback;
				if (label && label.fillInnerHTML) {
					this._extraLabelSubscribe = label.fillInnerHTML(this.domExtraLabel);
				}
				if (icon && typeof icon == "string") {
					this.domExtraIcon.className = "icon "+icon+"-32";
				}
				else {
					this.domExtraIcon.className = "";
				}
				dojo.addClass(this.domNode, "withExtra");
				this._extraAutorepeat = autorepeat;
			}
			else {
				this._extraCallback = null;
				dojo.removeClass(this.domNode, "withExtra");
			}
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		 * Del characters from selection's start to end.
		 * If selection's start and end are equals, dels the previous char.
                 * @ignore
		 */
		delChars: function() {
			// Dels chars from selectionStart to selectionEnd.
			// If selectionStart equals selectionEnd, then dels the previous char.
			var field = this.getInputFieldObject();
			var intSelectionStart = field.selectionStart;
			var intSelectionEnd = field.selectionEnd;

			// If no selection is made
			if (intSelectionStart === intSelectionEnd) {
				// If caret is at position 0, no need to do anything
				if (intSelectionStart > 0) {
					// Removes the character before caret position
					if (intSelectionStart === this.strInputText.length) {
						this.strInputText = this.strInputText.substring(0, this.strInputText.length - 1);
						this.strInputContent = this.strInputContent.substring(0, this.strInputContent.length - 1);
					}
					else {
						this.strInputText = this.strInputText.substring(0, intSelectionStart - 1) + this.strInputText.substring(intSelectionStart);
						this.strInputContent = this.strInputContent.substring(0, intSelectionStart - 1) + this.strInputContent.substring(intSelectionStart);
					}
					field.value = this.strInputText;
					// Sets back the caret position.
					field.selectionStart = intSelectionStart-1;
					field.selectionEnd = intSelectionStart-1;
				}
			}
			// If several characters are selected
			else {
				// Removes selected chars
				this.strInputText = this.strInputText.substring(0, intSelectionStart) + this.strInputText.substring(intSelectionEnd);
				this.strInputContent = this.strInputContent.substring(0, intSelectionStart) + this.strInputContent.substring(intSelectionEnd);
				field.value = this.strInputText;
				// Sets back the caret position.
				field.selectionStart = intSelectionStart;
				field.selectionEnd = intSelectionStart;
			}

			this._updateInputFieldRows();
		},

		/**
		 * This method is called when the virtual keyboard is used
		 * this.strInputContent is manually updated with given UTF8char
		 * @param {String} UTF8char
                 * @ignore
		 */
		addNewChar : function(UTF8char) {
			var field = this.getInputFieldObject();
			var intSelectionStart = field.selectionStart;
			var intSelectionEnd = field.selectionEnd;

			if (UTF8char != '') {
				this.strInputContent = this.strInputContent.substring(0, intSelectionStart) + UTF8char + this.strInputContent.substring(intSelectionEnd);
				this.strInputText = this.strInputText.substring(0, intSelectionStart) + UTF8char + this.strInputText.substring(intSelectionEnd);
				field.value = this.strInputText;
				if (this.boolPassword) {
					this._convertInputButLastCharToPassword();
				}
				field.selectionEnd = field.selectionStart = intSelectionStart + 1;
			}

			this.domInput.scrollLeft = this.domInput.scrollWidth;
			this._updateInputFieldRows();
		},

		/**
		 * Launch the Keyboard
		 * @param {Event} event
                 * @ignore
		 */
		deployKeyboard: function (event) {
			if (this.boolEnabled){
				this.valueChanged = true;
				if(this.strEventId!=null)
					dojo.publish(this.strEventId, [this]);
				if( this.keyboardType !== "" )
				{
					ICTouchAPI.keyboardServices.deployKeyboard(this.keyboardType, this, {
						strOkLabel : this.strOk,
						strExitLabel : this.strExit,
						funcCallbackOk : this.funcCallbackOk,
						funcCallbackExit : this.funcCallbackExit,
						actionOutsideClick : this.actionOutsideClick,
						strDefaultText : this.strKeyboardDefaultText,
						strInputTitle : this.strInputTitle,
						hideStarsIfPasswordEmpty : this.hideStarsIfPasswordEmpty,
						strDefaultMode: this.strDefaultMode
					});
					if(event)
						dojo.stopEvent(event);
				}
			}
		},

		/**
		 * Enable or disable the inputField element
		 * @param {Boolean} disable True to disable the InputField
		 */
		toggleDisabledMode : function(disable){
			this.boolEnabled = !disable;
			if (disable){
				this.getInputFieldObject().setAttribute("disabled","disabled");
			}
			else{
				this.getInputFieldObject().removeAttribute("disabled");
			}
		},

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 */
		_extraButtonDown: function(clickEvent) {
			this._feedBack(clickEvent);
			if (typeof this._extraCallback == "function") {
				this._extraCallback();
				if (this._extraAutorepeat) {
					this._autorepeatTimeout = setTimeout(dojo.hitch(this, this._extraRepeat), 500);
				}
			}
		},

                /**
                 * @ignore
                 */
		_clearExtraTimers: function() {
			if (this._autorepeatTimeout) {
				clearTimeout(this._autorepeatTimeout);
				this._autorepeatTimeout = null;
			}
			if (this._autorepeatInterval) {
				clearInterval(this._autorepeatInterval);
				this._autorepeatInterval = null;
			}
		},


                /**
                 * @ignore
                 */
		_extraRepeat: function() {
			this._extraCallback();
			this._autorepeatTimeout = null;
			this._autorepeatInterval = setInterval(this._extraCallback, 100);
		},

		/**
		 * @ignore
		 */
		_updateInputFieldRows: function () {
			if (this.isMultiline) {
				// If we force the line size
				if (this.boolForceMaxLineSize) {
					this.domTextarea.rows = this.intMaxLine;
				}else {
					var rows = Math.floor(this.domTextarea.scrollHeight / 30);
					// Clamp value between 1 and intMaxLine
					rows = Math.max(Math.min(rows, this.intMaxLine), 1);
					this.domTextarea.rows = rows;
					if(this.oldScrollHeight !== this.domTextarea.scrollHeight) {
						var diffScrollHeight = this.domTextarea.scrollHeight - this.oldScrollHeight;
						this.oldScrollHeight = this.domTextarea.scrollHeight;
						this.domTextarea.scrollTop += diffScrollHeight;

					}
				}
			}
		},

		/**
		 * @ignore
		 * This method is triggered when a physical keyboard is used
		 * The value in inputField is not refreshed at this time
		 * Both this.strInputText and this.strInputContent are manually updated with new event.keychar
		 */
		_onKeyPress : function(event) {
			var field = this.domInput;
			var intSelectionStart = field.selectionStart;
			var intSelectionEnd = field.selectionEnd;

			// A new letter / number was typed
			if (event.keyChar !== '') {
				this.addNewChar(event.keyChar);
				dojo.stopEvent(event);
			}
			// Backspace key was hit
			else if (event.keyCode === 8) {
				if (intSelectionStart === intSelectionEnd) {
					if (intSelectionStart > 0) {
						if (intSelectionStart === this.strInputText.length) {
							this.strInputText = this.strInputText.substring(0, this.strInputText.length - 1);
							this.strInputContent = this.strInputContent.substring(0, this.strInputContent.length - 1);
						}
						else {
							this.strInputText = this.strInputText.substring(0, intSelectionStart - 1) + this.strInputText.substring(intSelectionStart);
							this.strInputContent = this.strInputContent.substring(0, intSelectionStart - 1) + this.strInputContent.substring(intSelectionStart);
						}
						// Replace the carrot at the right place ( else it will show up a the end )
						field.selectionStart = intSelectionStart-1;
					}
				}
				else {
					this.strInputText = this.strInputText.substring(0, intSelectionStart) + this.strInputText.substring(intSelectionEnd);
					this.strInputContent = this.strInputContent.substring(0, intSelectionStart) + this.strInputContent.substring(intSelectionEnd);
				}
			}
			// Delete key was hit
			else if (event.keyCode === 46) {
				if (intSelectionStart === intSelectionEnd) {
					if (intSelectionStart < this.strInputText.length) {
						if (intSelectionStart === 0) {
							this.strInputText = this.strInputText.substring(1);
							this.strInputContent = this.strInputContent.substring(1);
						}
						else {
							this.strInputText = this.strInputText.substring(0, intSelectionStart) + this.strInputText.substring(intSelectionStart+1);
							this.strInputContent = this.strInputContent.substring(0, intSelectionStart) + this.strInputContent.substring(intSelectionStart+1);
						}
					}
				}
				else {
					this.strInputText = this.strInputText.substring(0, intSelectionStart) + this.strInputText.substring(intSelectionEnd);
					this.strInputContent = this.strInputContent.substring(0, intSelectionStart) + this.strInputContent.substring(intSelectionEnd);
				}
			}
			this.domInput.value = this.strInputText;
			dojo.stopEvent(event);
		},

		/**
		 * @ignore
		 * This method converts field.value but last char in stars
		 */
		_convertInputButLastCharToPassword : function() {
			// The whole string but last char is replaced by ? chars
			var field = this.getInputFieldObject();
			if(this.strInputContent.length == 1){
				this.strInputText = this.strInputContent;
			}
			var strLength = this.strInputText.length;
			var str = ""
			for (var i=0; i<strLength; i++) {
				if (i == strLength - 1) {
					str += this.strInputText.charAt(i);
				}
				else {
					str += '*';
				}
			}
			this.strInputText = str;
			field.value = this.strInputText;
			// 1 second later, the whole string including last char needs to be replaced by * chars
			var func = dojo.hitch(this, this._convertInputToPassword);
			setTimeout (function(){
				func(strLength);
			}, 1000);
		},

		/**
		 * @ignore
		 * This method converts field.value in stars
		 */
		_convertInputToPassword : function (currentStrLength) {
			// This test is made so no replacement is made if another char has been typed during the 3 seconds delay
			if(!currentStrLength || currentStrLength <= 0){
				var field = this.getInputFieldObject();
				this.strInputText = "*****";
				field.value = this.strInputText;
			}else{
				if (this.strInputText.length == currentStrLength) {
					var field = this.getInputFieldObject();
					var str = "";
					for (var i=0; i<currentStrLength; i++) {
						str += '*';
					}
					this.strInputText = str;
					var intSelectionStart = field.selectionStart;
					var intSelectionEnd = field.selectionEnd;
					// After setting the value of a field we must restore selectionStart and selectionEnd
					field.value = this.strInputText;
					field.selectionStart = intSelectionStart;
					field.selectionEnd = intSelectionEnd;
				}
			}
		},


                /**
                 * @ignore
                 */
		_feedBack:function(clickEvent){
				this.myFeedback.feedback(clickEvent);
		}
	}
	);
