/**
* @class UIElements.ApplicationMode.ThirdControl
* @extends UIElements.ApplicationMode.ThirdControlBase
* This class is the graphical representation of a webapp screen with a 1/3 - 2/3 layout.
*
* The 1/3 - 2/3 layout displays two containers that take all the available space in the screen while keeping the top bar and an action bar.
*
* - The 1/3 container that is used to display a {@link UIElements.MenuList.MenuListControl}
*
* - The 2/3 container that is used to display a list like a {@link UIElements.DataList.DataListControl} or a {@link UIElements.PresentationList.PresentationListControl} or any other graphical elements.
*
* If you want to display a webapp view using a full layout, see the {@link UIElements.ApplicationMode.FullControl} class.
*
* Here is an example of creating a 1/3 - 2/3 layout in the main view:
*     // Declare the layout
*     layout:null,
*
*     postCreate : function() {
*           // Define the callback function that is called when an item is selected in the 1/3 container list.
*           var itemMenuPressedCallback = dojo.hitch(this, this.onMenuItemPressed);
*
*           // Create the 1/3 - 2/3 layout
*           this.layout = new UIElements.ApplicationMode.ThirdControl({
*               menu : {
*                   name: "webapp.myWebapp.getMenuContent",
*                   params: {callBackFunc:itemMenuPressedCallback}
*               },
*               container : {
*                   name: "webapp.myWebapp.getContainerContent",
*                   params: {}
*               }
*           }, this.thirdDomNode);
*     },
*
*     // Callback when an item in the view getMenuContent is selected.
*     onMenuItemPressed : function() {
*           // Do something, ie: reload an other container in the 2/3 part
*           this.layout.setContainer("webapp.myWebapp.getAnOtherContainer", {});
*     }
*
* In the template associated to this view, don't forget to add the thirdDomNode attach point
*     <div dojoAttachPoint="thirdDomNode" />
*
* In the view getMenuContent (1/3 container), define the MenuList
*
*     // Items in the list see the {@link UIElements.MenuList.MenuListControl} class to see an example of how to populate a list.
*     arrItems: []
*
*     // The callBack function (value is given by the main view using the 'params' property
*     callBackFunc: null,
*
*     postCreate : function() {
*            var container = new UIElements.Container.ContainerControl({
*               title: {
*                   strLabel: "1/3 Layout title"
*               }
*               content: {
*                   name:"UIElements.MenuList.MenuListControl",
*                   params:{
*                       arrItems:[],
*                       callback: this.callBackFunc
*                   }
*               }
*           }, this.menuDomNode);
*     }
*
* In the template associated to the getMenuContent, don't forget to add the associated DOM attach point
*     <div dojoAttachPoint="menuDomNode" />
*
* In the view getContainerContent (2/3 container), the minimum that must be defined is a container
*     postCreate : function() {
*           // Create the container
*           var container = new UIElements.Container.ContainerControl({
*               title: {
*                   strLabel: "2/3 Layout title"
*               }
*               content: {
*                   name:"UIElements.DataList.DataListControl",
*                   params:{arrItems:[]}
*               }
*           }, this.containerDomNode);
*      }
*
* Instead of adding a DataList, other UIElement like a {@link UIElements.PresentationList.PresentationListControl} or an other view can be added using the objContent parameter
*      objContent: {
*          name:"webapp.myWebapp.getOtherView",
*          params:{}
*      }
*
* In the template associated to the getContainerContent, don't forget to add the DOM attach points of all UIElements added
*     <div dojoAttachPoint="containerDomNode" />
*
*/
dojo.require("UIElements.ApplicationMode.ThirdControlBase");
dojo.provide("UIElements.ApplicationMode.ThirdControl");
dojo.declare("UIElements.ApplicationMode.ThirdControl",
	UIElements.ApplicationMode.ThirdControlBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Reference on menu object
		 * @property
		 * @type HTMLElement
		 * @ignore
		 */
		objMenu				: null,
		/**
		 * Reference on the container object
		 * @property
		 * @type HTMLElement
		 * @ignore
		 */
		objContainer		: null,
		/**
		 * Reference on the menu title object
		 * @property
		 * @type HTMLElement
		 * @ignore
		 */
		objMenuTitle		: null,
		/**
		 * Reference on the contaienr title object
		 * @property
		 * @type HTMLElement
		 * @ignore
		 */
		objContainerTitle	: null,

		/**
		 * JSON object that defines the view displayed in the 1/3 part of the layout
		 * This field must contain the name of the view and the parameters to send to this view
		 * @cfg {Object} menu (required)
		 * @cfg {String} menu.name
		 * @cfg {Object} menu.params
		 */
		menu: null,

		/**
		 * JSON object that defines the view displayed in the 2/3 part of the layout
		 * This field must contain the name of the view and the parameters to send to this view
		 * @cfg {Object} container (required)
		 * @cfg {String} container.name
		 * @cfg {Object} container.params
		 */
		container: null,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		domMenu				: null,
		/**
		 * @ignore
		 */
		domContainer		: null,
		/**
		 * @ignore
		 * The path to the template is defined here
		 */
		templatePath	: dojo.moduleUrl("UIElements.ApplicationMode","templates/Third8082.html"),
		/**
		 * @ignore
		 */
		cssFile         : ["base.css","8082.css"],
		/**
		 * @ignore
		 */
		_uiMenu				: null,
		/**
		 * @ignore
		 */
		_uiContainer		: null,

		/* ------------------------------------ Constructor --------------------------------------- */

		postMixInProperties : function() {
			if (this.menu)
				this.objMenu = this.menu;
			if (this.container)
				this.objContainer = this.container;
		},

		postCreate: function() {
			if(!generalConfig.simulation) {
				// If strMenuTitle was defined, load an intermediate "ContainerControl" instead of directly loading the widget.
				if (this.objMenuTitle != null) {
					this._uiMenu = new UIElements.Container.ContainerControl({
						objTitle: this.objMenuTitle,
						objContent: {
							name: this.objMenu.name,
							params: this.objMenu.params
						}
					}, this.domMenu);
				}
				else {
					eval("this._uiMenu = new "+this.objMenu.name+"(this.objMenu.params,this.domMenu)");
				}
				// If strContainerTitle was defined, load an intermediate "ContainerControl" instead of directly loading the widget.
				var div = dojo.create("div",{},this.domContainer);
				if (this.objContainerTitle != null) {
					this._uiContainer = new UIElements.Container.ContainerControl({
						objTitle: this.objContainerTitle,
						objContent: {
							name: this.objContainer.name,
							params: this.objContainer.params
						}
					}, div);
				}
				else{
					eval("this._uiContainer = new "+this.objContainer.name+"(this.objContainer.params,div)");
				}
			}
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * Get the container associated to view contained into 1/3 part of the layout
		 * @return {UIElements.Container.ContainerControl} the object representing the 1/3 part of the layout
         */
		getMenu: function() {
			return this._uiMenu;
		},

		/**
		 * Get the container associated to view contained into 2/3 part of the layout
		 * @return {UIElements.Container.ContainerControl} the object representing the 2/3 part of the layout
         */
		getContainer: function() {
			return this._uiContainer;
		},

		/**
		 * Set or update the view contained into the 2/3 par of the layout
		 * @param {String} strContainerName the name of the view
		 * @param {Object} objContainerParams the params (JSON Object with keys / values)
         */
		setContainer: function(strContainerName,objContainerParams) {
			this.objContainer.name   = strContainerName;
			this.objContainer.params = objContainerParams;
			if( this._uiContainer != null )
				this._uiContainer.destroy();
			this.domContainer.id = ""; // resets

			// new objContainer
			var div = dojo.create("div",{},this.domContainer);
			var tempObj = ICTouchAPI.tools.getObjectReference(this.container.name);
			this._uiContainer = new tempObj(this.container.params, div);
		}

	/* ----------------------------------- Public methods ------------------------------------- */

	/* --------------------------------- Private Methods -------------------------------------- */

	}
	);
