/**
 * @class UIElements.AppButton.ButtonControlBase
 * @extends UIElements._base
 * Abstract Class - Do not use it directly.
 */
dojo.provide("UIElements.AppButton.ButtonControlBase");
dojo.declare("UIElements.AppButton.ButtonControlBase",
	[UIElements._base, dijit._Templated],
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Status of button (either onDisplay,offDisplay or unavailable)
		 * @cfg {String} strButtonStatus
		 */
		strButtonStatus : null,
		/**
		 * Name of the button
		 * @cfg {String} strButtonName
		 */
		strButtonName   : "",

		/**
		 * Status text
		 * @property
		 * @type String
		 * @ignore
		 */
		strStatusText   : "",

		/**
		 * Status icon (either 'notif-on', 'notif-off', or '')
		 * @cfg {String} strStatusIcon
		 */
		strStatusIcon   : "",

		/**
		 * Button label
		 * @cfg {String} strButtonLabel
		 */
		strButtonLabel	: "",

		/**
		 * Button icon. CSS style that embeds your image
		 * @cfg {String} strButtonIcon
		 */
		strButtonIcon   : "",

		/**
		 * Is the button embedded inside a popup
		 * @cfg {String} boolPopup
		 */
		boolPopup		: false,

		/**
		 * Call back function
		 * @cfg {Function} callback
		 */
		callback		: null,

		/**
		 * call back function for long press
		 * @property
		 * @type Function
		 */
		callbackLong	: null,
		
		/**
		 * feedback
		 * @ignore
		 * @property
		 * @type Function
		 */
		myFeedback		: null,
		/**
		 * scroll object
		 * @ignore
		 * @property
		 * @type UIEffects.Scroll
		 */
		objScroll		: null,
		/**
		 * Array of showing events
		 * @ignore
		 * @property
		 * @type Array
		 */
		eventsShowing	: [],
		/**
		 * Array of hiding events
		 * @ignore
		 * @property
		 * @type Array
		 */
		eventsHiding	: [],
		/**
		 * Array of Tinting events
		 * @ignore
		 * @property
		 * @type Array
		 */
		eventsTinting	: [],
		/**
		 * Array of capabilities which impact this button
		 * @ignore
		 * @property
		 * @type Array
		 */
		arrCapabilities : [],
		/**
		 * Array of capabilities handler related to this button
		 * @ignore
		 * @property
		 * @type Array
		 */
		arrCapabilitiesHandler : [],
		/**
		 *  Array of instances handler related to this button
		 *  @ignore
		 * @property
		 * @type Array
		 */
		arrCapabilitiesInstancesHandler : [],
		/**
		 * mode icon
		 * @ignore
		 * @property
		 * @type String
		 */
		strModeIcon   : null,

		/* --------------------------------- Private attributes ----------------------------------- */

		/*
		 * Hide / show / grey status depending of the capabilities the button subscribed to.
		 * A webapp can not show a button that is hidden by capability.
		 * @ignore
		 */
		_capabilityStatus : null,

		/*
		 * Hide / show / grey status depending of custom management of the button (by webapps).
		 * A webapp can hide / grey a button that is shown by the capabalities.
		 * @ignore
		 */
		_customStatus : null,

		/*
		 * List of available status
		 * @ignore
		 */
		_lstStatus : {
			ON : "onDisplay",
			OFF : "offDisplay",
			GREY : "unavailable"
		},

		/**
		 * @ignore
		 */
		domStatus		: null,
		/**
		 * @ignore
		 */
		domButton		: null,
		/**
		 * @ignore
		 * true if there is already an onmousedown on element
		 * useful in order to interpret an onclick after an onmouseup
		 */
		_eventOnMouseDownDone : false,
		/**
		 * @ignore
		 */
		_callbackFunc	: null,
		/**
		 * @ignore
		 */
		_handlerLongPress : null,
		/**
		 * @ignore
		 */
		_arrLastCapabilityCounter : null,
		/**
		 * @ignore
		 */
		attributeMap: {
			strButtonLabel: {
				node: "domLabel",
				type: "innerHTML"
			},
			strStatusText: {
				node: "domStatus",
				type: "innerHTML"
			}
		},

		_highlightClass: "active",
		/**
		 * @ignore
		 */
		_boolMoveable :true,//boolean in order to know it the progkey webapp can move the button or not

		/* ------------------------------------ Constructor --------------------------------------- */

		constructor : function() {
			//this.myFeedback = ICTouchAPI.feedbackServices.initFeedback("click");
			this.eventsHiding = [];
			this.eventsShowing = [];
			this.eventsTinting = [];

			// Initialize capabilities counters.
			this._arrLastCapabilityCounter = [];
			for(var i = 0 ; i > this.arrCapabilities.length ; i++) {
				this._arrLastCapabilityCounter[this.arrCapabilities[i].capability] = 0;
			}

			// Initialize capability and custom status.
			this._capabilityStatus = this._lstStatus.ON;
			this._customStatus = this._lstStatus.ON;
			this.strButtonStatus = this._lstStatus.ON;
		},


		postCreate : function() {
			if(this.strButtonStatus && this.domNode)
				dojo.addClass(this.domNode, this.strButtonStatus);
			if(this.strButtonIcon && this.domIcon)
				dojo.addClass(this.domIcon, this.strButtonIcon);
			if (this.strButtonName || this.strButtonName == 0){
				this.domNode.setAttribute("buttonname",this.strButtonName);
			}
			else {
				console.warn("Button '" + this.strButtonLabel + "' doesn't have a strButtonName!");
			}
			if (this.strStatusIcon){
				dojo.addClass(this.domStatus, "withStatusIcon");
				dojo.addClass(this.domStatus, this.strStatusIcon);
			}

			this.subscribeCapabilities();
		},

		destroy : function() {
			this.removeAllEvents();

			this.unSubscribeCapabilities();

			//delete this.myFeedback;
			delete this.callback;

			this.inherited(arguments);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/**
		 * Returns the name of the button
		 * @return {String} name of button
		 */
		getButtonName : function(){
			return this.strButtonName;
		},

		/**
		 * Get visibility status of this button
		 * @return {Boolean} the visibility of this button
		 */
		getVisibility: function() {
			return this.strButtonStatus == "onDisplay" || this.strButtonStatus == "unavailable";
		},

		/**
		* Updates the label
        * @param {String} strButtonLabel new button label
        */
		setButtonLabel : function (strButtonLabel) {
			this.attr("strButtonLabel", strButtonLabel);
		},

		/**
		 * Updates the text status by the given argument
		 * @param {String} strStatusText status of button in text (on or off)
		 * @ignore
		 */
		setStatusText: function(strStatusText){
			dojo.removeClass(this.domStatus, this.strStatusText);
			this.strStatusText = strStatusText;
			dojo.addClass(this.domStatus, this.strStatusText);
		},

		/**
		 * Updates the icon status by the given argument
		 * @param {String} strStatusIcon status of button led (either notif-on or notif-off)
		 */
		setStatusIcon    : function(strStatusIcon){
			this.attr("strStatusIcon", strStatusIcon);
		},

		/**
		 * setModeIcon replaces the icon mode by the given argument
		 * @ignore
		 * @param {String} strModeIcon mode of button in icon
		 */
		setModeIcon : function(strModeIcon){
			this.attr("strModeIcon", strModeIcon);
		},

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		 * removes mode icon
		 * @ignore
		 */
		removeModeIcon : function(){
			this.attr("strModeIcon", "");
		},

		/**
		 * clone method clones this in order to make a button with the same
		 * behavior. The clone is displayed in action bar of widgetName of
		 * webAppName.
		 * This method should be only called by the framework, especially appBarServicesis.
		 * It is useful when the webapp requires to display the same button in several action bars.
		 * @ignore
		 * @param {String} strWebAppName name of webapp
		 * @param {String} strWidgetName name of widget
		 */
		subscribeCapabilities:function() {
			var length = this.arrCapabilities.length;

			var subscribeHandler = null;
			this.arrCapabilitiesHandler = [];
			this.arrCapabilitiesInstancesHandler = [];
			for (var i = 0; i < length; i++) {

				var objCap = this.arrCapabilities[i];
				if (objCap.instance == undefined) {
					subscribeHandler = ICTouchAPI.CapabilityServices.subscribeToCapability(objCap.module, objCap.capability, this, this._capabilityChanged);
					this.arrCapabilitiesHandler[objCap.module + objCap.capability] = subscribeHandler;
				}
				else if (objCap.instance >= 0) {
					subscribeHandler = ICTouchAPI.CapabilityServices.subscribeToInstance(objCap.module, objCap.instance, objCap.capability, this, this._capabilityChanged);
					this.arrCapabilitiesInstancesHandler[objCap.module + objCap.capability + objCap.instance] = subscribeHandler;
				}
			}
		},

		/**
		 *  Disconnect an capability handler and reconnect an instance handler
		 *  @ignore
		 *  @param {String} strModule the module whch has to reconnect its capability
		 *  @param {Number} intInstance the instance to be econnected
		 */
		reconnectCapabilitiesToInstance:function(strModule, intInstance) {
			ICTouchAPI.debugServices.info("UIElements.AppButton.ButtonControlBase - reconnectCapabilitiesToInstance $ input function params | (button " + this.strButtonName + ") | strModule: " + strModule + " | intInstance: " + intInstance);
			var length = this.arrCapabilities.length;
			var subscribeHandler = null;
			var noResubscribeToInstance = false; // bool used to avoid unecessary treatment (do not unsubscribe/resubscribe the instance if it is already subscribed)
			for (var i = 0; i < length; i++) {
				var objCap = this.arrCapabilities[i];
				if (objCap.module == strModule && objCap.instance != undefined) {
					// remove all subsciption to the given capability whatever the instance is
					for (var j in this.arrCapabilitiesInstancesHandler) {
						if (j.indexOf(objCap.module + objCap.capability) !== -1) {
							if (j == objCap.module + objCap.capability + intInstance) {
								ICTouchAPI.debugServices.debug("UIElements.AppButton.ButtonControlBase - reconnectCapabilitiesToInstance / j: " + j + ", button " + this.strButtonName + " do not need to unsubscribe to intInstance " + intInstance + " because it is already susbribed");
								// instance to reconnect is already subscribed, do not unsubscribe/resubscribe it
								noResubscribeToInstance = true;
							}
							else {
								ICTouchAPI.debugServices.debug("UIElements.AppButton.ButtonControlBase - reconnectCapabilitiesToInstance / call disconnectHandler for " + this.strButtonName + " | objCap.instance: " + objCap.instance + " (disconnect j: " + j + ")");
								subscribeHandler = this.arrCapabilitiesInstancesHandler[j];
								ICTouchAPI.CapabilityServices.disconnectHandler(subscribeHandler);
								delete (this.arrCapabilitiesInstancesHandler[j]);
							}
						}
					}
					objCap.instance = intInstance;
					if (objCap.instance >= 0 && !noResubscribeToInstance) {
						ICTouchAPI.debugServices.debug("UIElements.AppButton.ButtonControlBase - reconnectCapabilitiesToInstance / call subscribeToInstance for " + this.strButtonName + " | objCap.instance: " + objCap.instance);
						subscribeHandler = ICTouchAPI.CapabilityServices.subscribeToInstance(objCap.module, objCap.instance, objCap.capability, this, this._capabilityChanged);
						this.arrCapabilitiesInstancesHandler[objCap.module + objCap.capability + objCap.instance] = subscribeHandler;
					}
					else {
						ICTouchAPI.debugServices.debug("UIElements.AppButton.ButtonControlBase - reconnectCapabilitiesToInstance / noResubscribeToInstance: " + noResubscribeToInstance + ", button " + this.strButtonName + " do not need to resubscribe to objCap.instance " + objCap.instance + " because it is already susbribed");
					}
				}
			}
		},

		/**
		 * unsubscribe all capabilities
		 * @ignore
		 */
		unSubscribeCapabilities: function() {
			var subscribeHandler;

			for(var i in this.arrCapabilitiesHandler ) {
				subscribeHandler = this.arrCapabilitiesHandler[i];
				ICTouchAPI.CapabilityServices.disconnectHandler(subscribeHandler);
				delete this.arrCapabilitiesHandler[i];
			}

			for(var j in this.arrCapabilitiesInstancesHandler ) {
				subscribeHandler = this.arrCapabilitiesInstancesHandler[j];
				ICTouchAPI.CapabilityServices.disconnectHandler(subscribeHandler);
				delete this.arrCapabilitiesInstancesHandler[j];
			}
		},

		/**
		 * Set capabilityStatus to "OFF"
		 */
		hide: function(){
			this._customStatus = this._lstStatus.OFF;
			this._applyStatus();
		},

		/**
		 * Set capabilityStatus to "ON"
		 */
		show: function(){
			this._customStatus = this._lstStatus.ON;
			this._applyStatus();
		},

		/**
		 * Set capabilityStatus to "GREY"
		 */
		greyTint: function(){
			this._customStatus = this._lstStatus.GREY;
			this._applyStatus();
		},

		/**
		 * addEventsShowing method adds the events given in arguments
		 * which shows the button
		 * @ignore
		 * @param {String[]}    list of event names
		 */
		addEventsShowing: function(){
			for(var i=0;i<arguments.length;i++){
				var handle = dojo.subscribe(arguments[i], this, this.show);
				// note: with this.show(), show() is running
				// when subscribe occurs write only
				// the name of method (without ())

				// save the handle in the eventsShowing array
				this.eventsShowing[arguments[i]]=handle;
			}
		},

		/**
		 * removeEventsShowing method removes the events given in arguments
		 * which shows the button
		 * @ignore
		 * @param {String[]} list of event names
		 */
		removeEventsShowing: function(){
			for(var i=0;i<arguments.length;i++){
				// get the handle from the eventsShowing array
				// before unsubscribing
				dojo.unsubscribe(this.eventsShowing[arguments[i]]);
				// delete the reference of handle
				delete (this.eventsShowing[arguments[i]]);
			}
		},

		/**
		 * addEventsHiding method adds the events given in arguments
		 * which hides the button
		 * @ignore
		 * @param {String[]} list of event names
		 */
		addEventsHiding: function(){
			for(var i=0;i<arguments.length;i++){
				var handle = dojo.subscribe(arguments[i], this, this.hide);
				// save the handle in the eventsHiding array
				this.eventsHiding[arguments[i]]=handle;
			}
		},

		/**
		 * removeEventsHiding method removes the events given in arguments
		 * which hides the button
		 * @ignore
		 * @param {String[]} list of event names
		 */
		removeEventsHiding: function(){
			for(var i=0;i<arguments.length;i++){
				dojo.unsubscribe(this.eventsHiding[arguments[i]]);
				delete (this.eventsHiding[arguments[i]]);
			}
		},

		/**
		 * addEventsGreyTinting method adds the events given in arguments
		 * which greytints the button
		 * @ignore
		 * @param {String[]} list of event names
		 */
		addEventsGreyTinting: function(){
			for(var i=0;i<arguments.length;i++){
				var handle = dojo.subscribe(arguments[i], this, this.greyTint);
				// save the handle in the eventsTinting array
				this.eventsTinting[arguments[i]]=handle;
			}
		},

		/**
		 * removeEventsGreyTinting method removes the events given in arguments
		 * which greytints the button
		 * @ignore
		 * @param {String[]}    list of event names
		 */
		removeEventsGreyTinting: function(){
			for(var i=0;i<arguments.length;i++){
				dojo.unsubscribe(this.eventsTinting[arguments[i]]);
				delete (this.eventsTinting[arguments[i]]);
			}
		},

		/**
		 * removeAllEvents method removes all events currently subscribed
		 * @ignore
		 */
		removeAllEvents: function(){
			var i;
			for (i in this.eventsTinting) {
				dojo.unsubscribe(this.eventsTinting[i]);
				delete (this.eventsTinting[i]);
			}
			for (i in this.eventsHiding) {
				dojo.unsubscribe(this.eventsHiding[i]);
				delete (this.eventsHiding[i]);
			}
			for (i in this.eventsShowing) {
				dojo.unsubscribe(this.eventsShowing[i]);
				delete (this.eventsShowing[i]);
			}
		},



		/* --------------------------------- Private Methods -------------------------------------- */

		_startHighlight: function() {
			this.boolLongPress = false;
			// Launch default highlight and get its status
			var activated = this.inherited(arguments);
			// Add the active class on the icon if necessary
			dojo.toggleClass(this.domIcon, "active", activated);
			if (typeof this.callbackLong == "function") {
				this._handlerLongPress = setTimeout(dojo.hitch(this, this._actionLongpress), 500);
			}
		},
		
		_actionLongpress : function() {
			// Longpress has been triggered don't callback on mouseup
			this.boolLongPress = true;
			this._handlerLongPress = null;
			this.callbackLong(this.strButtonName);
		},
		
		_stopHighlight: function() {
			// Launch default stopHighlight and check if a click was issued
			var clicked = this.inherited(arguments);
			dojo.removeClass(this.domIcon, "active");
			if (clicked && this.strButtonStatus === "onDisplay" && !this.boolLongPress) {
				this._callbackFunc(this.strButtonName);
			}
			
			if (this._handlerLongPress) {
				clearTimeout(this._handlerLongPress);
				this._handlerLongPress = null;
			}
		},
		
		/**
		 * @ignore
		 */
		_setStrStatusIconAttr : function(value) {
			if(value && value.length){
				value = value + "-" + this.intStatusIconSize;
				if(this.strStatusIcon != value){
					dojo.toggleClass(this.domStatus, "withStatusIcon", value != "");
					dojo.removeClass(this.domStatus, this.strStatusIcon);
					dojo.addClass(this.domStatus, value);
					this.strStatusIcon = value;
				}
			}
		},

		/**
		 * @ignore
		 */
		_setStrModeIconAttr : function(value) {
			if(this.strModeIcon && this.strModeIcon != ""){
				dojo.removeClass(this.domMode, this.strModeIcon+"-32");
				dojo.removeClass(this.domMode, "modeOn");
			}
			if(value && value.length){
				dojo.addClass(this.domMode, value + "-32");
				dojo.addClass(this.domMode, "modeOn");
				this.strModeIcon = value;
			}
		},

		/**
		 * @ignore
		 */
		_setCallbackAttr : function(callback) {
			this._callbackFunc = ICTouchAPI.tools.callbackToHitch(callback);
		},


		/**
		 * @ignore
		 * Function is called when a capability related to this button has changed
		 * The button is set visible and active only when all capabilities associated with it are AVAILABLE
		 * If only one capability is UNAVAILABLE, button is hidden
		 * If at least one is TEMP_UNAVAILABLE or UNKNOWN and all other are AVAILABLE, button is grey tinted
		 */
		_capabilityChanged : function(capability, status, instance, intCapabilityCounter) {
			ICTouchAPI.debugServices.info("UIElements.AppButton.ButtonControlBase - _capabilityChanged $ input function params | capability: " + capability + " | status: " + status + " | instance: " + instance + " | intCapabilityCounter: " + intCapabilityCounter);
			if(intCapabilityCounter){
				if(!this._arrLastCapabilityCounter[capability] || intCapabilityCounter > this._arrLastCapabilityCounter[capability]){
					this._arrLastCapabilityCounter[capability] = intCapabilityCounter;
				} else {
					ICTouchAPI.debugServices.warning("UIElements.AppButton.ButtonControlBase - _capabilityChanged / Do not treat capability " + capability + " with instance " + instance + " because a more recent capability value was already treated. Old number : " + this._arrLastCapabilityCounter[capability] + " , new number : " +intCapabilityCounter);
					return;
				}
			}
			var newState = 1;	// show
			for (var i in this.arrCapabilities) {
				var objCap = this.arrCapabilities[i];
				var capstate;
				if (instance != undefined)
					capstate = ICTouchAPI.CapabilityServices.getCapability(objCap.module, instance, objCap.capability);
				else
					capstate = ICTouchAPI.CapabilityServices.getCapability(objCap.module, objCap.capability);
				if (capstate == ICTouchAPI.CapabilityServices.UNAVAILABLE) {
					newState = 0;	// hide
					break;	// If unavailable we can break the loop
				}
				else if (capstate != ICTouchAPI.CapabilityServices.AVAILABLE)
					newState = 2;	// grey tinted
			}
			switch (newState) {
				case 1:
					this._capabilityShow();
					break;
				case 0:
					this._capabilityHide();
					break;
				default:
					this._capabilityHide();
					break;
			}
		},

		/**
		 * Shows/hides/tints in grey the button according to capabilityStatus and customStatus
		 * @ignore
		 */
		_applyStatus: function(){
			if(this._capabilityStatus == this._lstStatus.OFF || this._customStatus == this._lstStatus.OFF){
				this._hide();
			} else if(this._capabilityStatus == this._lstStatus.GREY || this._customStatus == this._lstStatus.GREY) {
				this._greyTint();
			} else {
				this._show();
			}
		},

		/**
		 * Set capabilityStatus to "OFF"
		 *  @ignore
		 */
		_capabilityHide: function(){
			this._capabilityStatus = this._lstStatus.OFF;
			this._applyStatus();
		},

		/**
		 * Set capabilityStatus to "ON"
		 *  @ignore
		 */
		_capabilityShow: function(){
			this._capabilityStatus = this._lstStatus.ON;
			this._applyStatus();
		},

		/**
		 * Set capabilityStatus to "GREY"
		 * @ignore
		 */
		_capabilityGreyTint: function(){
			this._capabilityStatus = this._lstStatus.GREY;
			this._applyStatus();
		},

		/**
		 * Hides the button
		 *  @ignore
		 */
		_hide: function(){
			if(this.strButtonStatus != this._lstStatus.OFF){
				dojo.removeClass(this.domNode, this.strButtonStatus);
				this.strButtonStatus = this._lstStatus.OFF;
				dojo.addClass(this.domNode, this.strButtonStatus);
			}
		},

		/**
		 * Unhides / shows the button
		 *  @ignore
		 */
		_show: function(){
			if(this.strButtonStatus != this._lstStatus.ON){
				dojo.removeClass(this.domNode, this.strButtonStatus);
				this.strButtonStatus = this._lstStatus.ON;
				dojo.addClass(this.domNode, this.strButtonStatus);
			}
		},

		/**
		 * Deactivates and tints the button in grey.
		 * @ignore
		 */
		_greyTint: function(){
			if(this.strButtonStatus != this._lstStatus.GREY){
				dojo.removeClass(this.domNode, this.strButtonStatus);
				this.strButtonStatus = this._lstStatus.GREY;
				dojo.addClass(this.domNode, this.strButtonStatus);
			}
		},

	}
	);
