/**
 * @class UIElements.AppButton.ButtonControl
 * @extends UIElements.AppButton.ButtonControlBase
 *
 * This class is the graphical representation of a button.
 *
 * A button can be used in a container with a 2/3 or a full layout to propose interaction with the user.
 *
 * A button contains a label, an icon an optionally a led that can be switched on or not.
 *
 * You can detect if the user has pressed on a button by given a callback function that will be called each time the user presses the button.
 *
 * Here is an example of creating a button:
 *
 * The first step is to create the DOM attach point in the HTML template file
 *     // Creating the HTML template associated to the button
 *     // Use a CSS style to place your button at the right place and to define the correct size
 *     <div dojoAttachPoint="myButton" class="buttonCls" />
 *
 * Next, you can create the button in your JavaScript view
 *     postCreate : function() {
 *          // Initialize the callback function
 *          var func = dojo.hitch(this, this.onButtonPress);
 *
 *          // Create the Button
 *          var btn = new UIElements.AppButton.ButtonControl({
 *              strButtonName:  "aButton",          // Button identifier
 *              strButtonLabel: "A Button",         // Label displayed
 *              strButtonIcon:  "buttonImgCss",     // CSS class of the image button
 *              strStatusIcon:  "notif-on",        // Led is switched off
 *              callback:       func                // Callback function called
 *          });
 *      },
 *
 *      onButtonPress : function() {
 *          // Do something
 *          // ...
 *      }
 *      
 * Finally, define the buttonCls CSS style
 *     .buttonCls {
 *         // Mandatory, define button size
 *         width:120px,
 *         height:124px,
 *         // Optional, define button position
 *         // ...
 *     }
 *
 */
dojo.require("UIElements.AppButton.ButtonControlBase");
dojo.provide("UIElements.AppButton.ButtonControl");
dojo.declare("UIElements.AppButton.ButtonControl",
	UIElements.AppButton.ButtonControlBase,
	{
		/* --------------------------------- Public attributes ------------------------------------ */

		/**
		 * Status text
		 * @property
		 * @type String
                 * @ignore
		 */
		strStatusText		: "",

                /**
		 * Button icon. Use a CSS style that embeds your image
		 * @cfg {String} strButtonIcon
		 */

		strButtonIcon		: "",

                /**
		 * status icon size
                 * @ignore
		 * @property
		 * @type Number
		 */
		intStatusIconSize	: 16,

                /**
		 * icon size
                 * @ignore
		 * @property
		 * @type Number
		 */
		intIconSize			: 64,

                /**
		 * popup icon size
                 * @ignore
		 * @property
		 * @type Number
		 */
		intPopupIconSize	: 48,

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 * The path to the template is defined here
		 */
		templatePath	: dojo.moduleUrl("UIElements.AppButton", "templates/Button8082.html"),
		/**
		 * @ignore
		 */
		cssFile         : ["base.css", "8082.css"],

		/* ------------------------------------ Constructor --------------------------------------- */

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 */
		_setStrButtonIconAttr : function(value) {
			if( this.boolPopup ) {
				value = value + "-" + this.intPopupIconSize;
			}
			else {
				value = value + "-" + this.intIconSize;
			}

			// Toggle status icon
			dojo.removeClass(this.domIcon, this.strButtonIcon);
			dojo.addClass(this.domIcon, value);
			this.strButtonIcon = value;
		}
	}
	);
