/**
 * @class UIElements.AppBar.AppBarWidget
 * @extends UIElements.AppBar.AppBarWidgetBase
 * This class represents the graphical user interface of an AppBar widget.
 *
 * Each view contains an AppBar widget that can be used to propose action buttons for the user so than he can interact with your web application.
 *
 * The web application requests the {ICTouchAPI.AppBarServices} to provide the action bar corresponding to its widget. In other words,no webapp has to instanciate an AppBar.
 * As soon as the webapp gets the AppBar, the webapp can add/remove some action buttons.
 *
 * When the AppBar of a view is building or updating, the AppBar requests automatically the static buttons and the others buttons attached to this view.
 *
 * Here is an example of adding a new action button in the AppBar widget:
 *
 *
 *     // Add an action button to an AppBar widget
 *     postCreate : function() {
 *
 *          // Initialize the callback function
 *          var func = dojo.hitch(this, this.onButtonPress);
 *
 *          // Create the Button
 *          var saveBtn = new UIElements.AppButton.AppButtonControl({
 *              strButtonName:  "saveButton",       // Button identifier
 *              strButtonLabel: "Save",             // Label displayed
 *              strButtonIcon:  "saveImgCss",       // CSS class of the image button
 *              callback:       func                // Callback function called
 *          });
 *
 *          // Get the AppBar Widget
 *          var appBar = ICTouchAPI.AppBarServices.getAppBar("webappName", "viewName");
 *
 *          // Add the button
 *          appBar.addActionButton(saveBtn);
 *     },
 *
 *     // Define the callback function when the user presses on the save button
 *     onButtonPress : function() {
 *
 *          // Do something
 *          ...
 *
 *     },
 *
 */
dojo.require("UIElements.AppBar.AppBarWidgetBase");
dojo.provide("UIElements.AppBar.AppBarWidget");
dojo.declare("UIElements.AppBar.AppBarWidget",
	UIElements.AppBar.AppBarWidgetBase,
	{

		/* --------------------------------- Public attributes ------------------------------------ */

		/* --------------------------------- Private attributes ----------------------------------- */

		/**
		 * @ignore
		 */
		domStaticButtons: null,
		/**
		 * @ignore
		 */
		domAppBarLeft	: null,
		/**
		 * @ignore
		 */
		domArrowLeft	: null,
		/**
		 * @ignore
		 */
		domArrowRight	: null,
		/**
		 * @ignore
		 * 0: nowhere, 1: utmost left, 2: somewhere in the middle, 3: utmost right
		 */
		_intLastArrow	: 0,
		/**
		 * @ignore
		 */
		_boolArrow		: false,
		/**
		 * @ignore
		 * The path to the template is defined here
		 */
		templatePath: dojo.moduleUrl("UIElements.AppBar","templates/AppBar8082.html"),
		/**
		 * @ignore
		 */
		cssFile    : ["base.css","8082.css"],

		/* ------------------------------------ Constructor --------------------------------------- */

		postCreate : function(){
			var i;

			// place the buttons into this bar
			for( i=0; i<this.arrStaticButtons.length; i++){
				this.addStaticButton(this.arrStaticButtons[i], true);
			}

			for( i=0; i<this.arrActionButtons.length; i++) {
				this.addActionButton(this.arrActionButtons[i], true);
			}

			for( i=0; i<this.arrOtherWebAppButtons.length; i++) {
				this.addOtherWebAppButton(this.arrOtherWebAppButtons[i], true,true);
			}

			if( this.domButtons != null && !this.boolDisableScroll) {
				this._checkArrows();
				FXScroll.init({
					"handler"	: this.domButtons,
					"easing"	: true,
					"scrollType" : "horizontal",
					// Not used for pagination but more as an "onscroll" event
					"pagination": {
						"callback"	: this._updateArrows,
						"context"	: this
					}
				});
			}

			this.inherited(arguments);
		},

		/* ----------------------------------- Getter / Setter------------------------------------- */

		/* ----------------------------------- Public methods ------------------------------------- */

		/**
		 * Add a static button into this bar. Only the 'homepage' button should be added as static.
		 * @param {UIElements.AppButton.AppButtonControl} objButton UI Button to add
                 */
		addStaticButton : function(objButton, boolAvoidPush) {
			// Adopt it so chrome doesn't throw an exception
			this.domNode.ownerDocument.adoptNode(objButton.domNode)

			objButton.placeAt(this.domStaticButtons, "last");
			dojo.addClass(this.domNode, "hasStatic");

			if( boolAvoidPush == false || typeof boolAvoidPush == "undefined" ) {
				this.arrStaticButtons.push(objButton);
			}

			objButton.setAppBar(this);
			this._checkArrows();
		},

		/**
		 * Remove a static button from this bar. Only the 'homepage' button should be removed
		 * @param {UIElements.AppButton.AppButtonControl} objButton UI Button to remove
		 */
		removeStaticButton : function(objButton) {
			for(var i in this.arrStaticButtons) {
				if(this.arrStaticButtons[i]==objButton) {
					this.arrStaticButtons.splice(i,1);
				}
			}

			this._checkArrows();
		},

		/**
		 * Add an action ( from an other webapp ) button into this bar
		 * @param {UIElements.AppButton.AppButtonControl} objButton UI Button to add
                 */
		addOtherWebAppButton : function(objButton, boolAvoidPush, boolRebuild) {
			// Adopt it so chrome doesn't throw an exception
			this.domNode.ownerDocument.adoptNode(objButton.domNode)

			objButton.placeAt(this.domButtons, "last");

			if( boolAvoidPush == false || typeof boolAvoidPush == "undefined" ) {
				this.arrOtherWebAppButtons.push(objButton);
			}
			if(boolRebuild){
				this.rebuildDom();
			}
			objButton.setAppBar(this);
			this._checkArrows();
		},

		/**
		 * Remove an action ( from an other webapp ) button from this bar
		 * @param {UIElements.AppButton.AppButtonControl} objButton UI Button to remove
		 */
		removeOtherWebAppButton : function(objButton) {
			for(var i in this.arrOtherWebAppButtons) {
				if(this.arrOtherWebAppButtons[i]==objButton) {
					this.arrOtherWebAppButtons.splice(i,1);
				}
			}
			this._checkArrows();
		},

		/**
		 * change the scroll capture
                 * @ignore
		 * @param {Object} capture
		 */
		changeScrollCapture : function(capture) {
			this.scroll.changeCapture(capture);
		},

		/* --------------------------------- Private Methods -------------------------------------- */

		/**
		 * @ignore
		 */
		_updateArrows: function() {
			if( !this._boolArrow ) {
				return 0;
			}
			var status = 2;

			var s = this.domButtons;
			// If the buttons are scrolled to the left most.
			//   This condition should be first so when the appBar is offscreen both condition can be true
			//   but the scroll is on the left
			if (s.scrollLeft < 5) {
				status = 1;
			// If the buttons are scrolled to the right most
			} else if (s.scrollLeft + 5 >= s.scrollWidth - s.clientWidth) {
				status = 3
			}

			if( this._intLastArrow !== status ) {
				dojo.toggleClass(this.domArrowLeft, "show", status > 1);
				dojo.toggleClass(this.domArrowRight, "show", status < 3);
				this._intLastArrow = status;
			}
			return 0;
		},

		/**
		 * @ignore
		 */
		_checkArrows: function() {
			//var i;
			// Number buttons before we need to scroll ( with and without static buttons )
			var max = this.arrStaticButtons.length > 0 ? 5 : 8;
			// Number of visible buttons
			var nbr = 0;
			for (var i in this.arrActionButtons){
				if(typeof this.arrActionButtons[i] != "string"){
					if(this.arrActionButtons[i].getVisibility()){
						nbr+=1;
					}
				}
			}
			for (var j in this.arrOtherWebAppButtons){
				if(typeof this.arrOtherWebAppButtons[j] != "string"){
					if(this.arrOtherWebAppButtons[j].getVisibility()){
						nbr+=1;
					}
				}
			}
			if( nbr > max ) {
				if(!this._boolArrow){
					this._boolArrow = true;
				}
				// 2 means somewhere in the middle. Otherwise, we do not have to test
				if(this._intLastArrow !== 1){
					// Call this to show the arrows
					this._updateArrows();
				}
			}
			else {
				this._intLastArrow = 0;
				this._boolArrow = false;
				dojo.removeClass(this.domArrowLeft, "show");
				dojo.removeClass(this.domArrowRight, "show");
			}
		}

	});
