/**
 * @ignore
 */
dojo.provide("UIEffects.Scroll._base");

dojo.declare("UIEffects.Scroll._base", dojo.dnd.Moveable, {
	_liList: null,
	_liDescendant: null,
	_liLength: 1,
	_boolInitialized: false,
	_boolLargeEnough: true,
	_boolMoving: false,
	boolSnap: true, // The list snaps on the edge
	boolSmooth: false, // Does the list come back smoothly
	boolLoop: false, // Can we loop endlessly
	boolHasScrollPanel: false, // Do we have a ScrollPanel around ?
	containerQuantity: 0,
	_totalLength: 0,
	_minOffset: 0,
	scrollFeedback : null,
	_boolFeedbackDone : false,

	constructor: function(node, params){
		console.assert(node);
		if(!params){ params = {}; }

		// Safeguard
		if( params.arrItemsList && params.boolLoop )
			console.error("You cannot loop a custom list !");

		if(params.boolSnap != undefined){ this.boolSnap = params.boolSnap; }
		if(params.boolSmooth != undefined){ this.boolSmooth = params.boolSmooth; }
		if(params.boolLoop != undefined){ this.boolLoop = params.boolLoop; }
		if(params.arrItemsList != undefined){ this._liList = params.arrItemsList; }
		if(params.intItemSize != undefined){ this._liLength = params.intItemSize; }
		if(params.boolHasScrollPanel != undefined){ this.boolHasScrollPanel = params.boolHasScrollPanel; }

		this.containerQuantity = params.containerQuantity ? params.containerQuantity : 0;

		// Changing default delay to 5 ( how many pixels before it starts scrolling )
		// Note from Ludovic Cadart: 5 pixels is too few,
		//this.delay = params.delay > 0 ? params.delay : 0;
		this.delay = 20;

		// Scroll feedback object
		this.scrollFeedback = ICTouchAPI.feedbackServices.initFeedback("endScroll");

		// Overload dojo standard function to disable autoscrolling
		dojo.dnd.autoScroll = function(){};

		// Apply some basic styles
		dojo.style(node, {
			"position": "relative",
			// Needed or dojo.dnd.Mover won't return a correct leftTop object
			"left": "0px",
			"top": "0px"
		});
		if (this.node && this.node.parentNode)
			dojo.style(this.node.parentNode, "overflow", "hidden");
	},

	refresh: function(params)
	{
		this._boolInitialized = false;
		if(!params){ params = {}; }
		if(params.arrItemsList != undefined){ this._liList = params.arrItemsList; }
		else{ this._liList = null; }
	},

	_findContainerQuantity: function(){
		// summary: This will try to find the smallest box that contains the list
		//   then how many items will fit in that box

		// Get the parent containerSize
		var minSize = this.getItemInnerLength(this.node.parentNode);
		for(var node = this.node.parentNode; node !== null; node = node.parentNode)
		{
			var size =  this.getItemInnerLength(node);
			if( minSize > size )
				minSize = size;
		}
//		return Math.round(minSize / this._liLength);
		// Joel Takvorian, correction: decrease size by one to allow access to last item
		var result = Math.round(minSize / this._liLength);
		return (result > 1) ? result - 1 : result;
	},

	_initList: function()
	{
		if( this._liList === null )
			this._liList = dojo.query("> *", this.node);

		if( this.boolSnap )
		{
			// Compute length of an item
			if( this._liLength == 1 )
				this._liLength = this.getItemLength(this._liList[0]);

			// Compute how many items can fit into this container
			if( this.containerQuantity < 1 )
				this.containerQuantity = this._findContainerQuantity();

			if( this.boolLoop && this._liList.length < this.containerQuantity + 2 )
			{
				console.warn("Not enough items ("+this._liList.length+") "+
					"to make this list scrollable endlessly, needed: "+Math.floor(this.containerQuantity) + 2);
				this.boolLoop = false;
			}

			if( this._liList.length < this.containerQuantity )
			{
				this._boolLargeEnough = false;
			}

			this._totalLength = this._liList.length * this._liLength;
			// How many pixels are "too much" for the container
			this._minOffset = ( this.containerQuantity - this._liList.length )* this._liLength;
		}

		this.initItem(this.node);
		this._boolInitialized = true;
	},

	isMoving: function(){
		return this._boolMoving;
	},

	onMove: function(/* dojo.dnd.Mover */ mover, /* Object */ leftTop, e){
		// summary: called during every move notification,
		//	should actually move the node, can be overwritten.

		this._boolMoving = true;
		this.onMoving(mover, leftTop, e);
		this.moveItem(this.node, leftTop);
		this.onMoved(mover, leftTop);
	},

	onDragDetected: function(/* Event */ e){
		// summary: called when the drag is detected,
		// responsible for creation of the mover

		if( ! this._boolInitialized )
		{
			this._initList();
		}

		// Only start scrolling when the list is large enough
		if( ! this._boolLargeEnough )
			return;

		this.movedOffset = 0;
		this._boolFeedbackDone = false;

		new this.mover(this.node, e, this);
	},

	onMoveStop: function(/* dojo.dnd.Mover */ mover){
		var first = this._liList[0];
		var context = this;

		if( this.boolSnap )
		{
			// Get the actual value
			var offset = this.getItemOffset(this.node);

			if( offset < this._minOffset )
				offset = this._minOffset;
			else
			{
				// Do the math to make it snap on a border
				offset = Math.round(offset / this._liLength);

				// Clamp between - (size - 1) and 0
				offset = Math.min(Math.max(offset, 1-this._liList.length), 0);
				offset = offset * this._liLength;
			}

			setTimeout(function(){
				context.setItemOffset(context.node, offset);
			}, 10);
		}

		this.scrollFeedback.stop();

		// Little trick to be able to catch onclick event with .isMoving()
		setTimeout(function(){
			context._boolMoving = false;
			dojo.publish("/dnd/move/stop", [mover]);
			dojo.removeClass(dojo.body(), "dojoMove");
			dojo.removeClass(context.node, "dojoMoveItem");
		}, 100);
	},

	onMoving: function(/* dojo.dnd.Mover */ mover, /* Object */ leftTop, e){
		if( this.boolLoop )
		{
			var move = this.checkLooping(leftTop);
			if( move == -1 )
			{
				// Moving LI from start to end
				var li = this._liList.shift();
				dojo.place(li, this.node, "last");
				this._liList.push(li);
			}

			if( move == 1 )
			{
				// Moving LI from end to start
				var li = this._liList.pop();
				dojo.place(li, this.node, "first");
				this._liList.unshift(li);
			}
		}
		if (this.boolSnap && !this._boolFeedbackDone) {
			// Check if out of list bounds
			// Get the current offset
			var offset = this.getItemOffset(this.node);
			if (offset < this._minOffset || offset > 0) {
				// do feedback
				this.scrollFeedback.feedback(e);
				this._boolFeedbackDone = true;
			}
		}
	},

	// move to item N° intIndex
	moveToItem : function(intIndex){
		if( ! this._boolInitialized )
			this._initList();

		// wrong value
		if(intIndex < 0 && intIndex > this._liList.length-1 )
			return;

		// init if not
		if( ! this._boolInitialized )
		{
			this._initList();
		}


		// want to go to the item Index, so offset =  - size of an item * N°
		var offset = - (this._liLength * intIndex); // (negative Offset) cause 0 is top


		// move
		var context = this;
		setTimeout(function(){
			if( context.boolHasScrollPanel )
			{
				context.setItemOffset(context.node, offset);
			}
			else
			{
				for(var i=0; i<context._liList.length; ++i)
				{
					context.setItemOffset(context._liList[i], offset);
				}
			}
		}, 10);
	},

	moveToDom : function(domTarget) {
		// init if not
		if( ! this._boolInitialized )
			this._initList();
		this.setItemOffset(this.node, -domTarget.offsetTop);
	}
});
/**
 * @ignore
 */
dojo.provide("UIEffects.Scroll.HorizontalList");

dojo.declare("UIEffects.Scroll.HorizontalList", UIEffects.Scroll._base, {
	movedOffset: 0,
	startOffset: 0,

	initItem : function(domNode) {
		var minWidth = dojo.contentBox(domNode.parentNode).w;
		for(var node = domNode; node !== null; node = node.parentNode)
		{
			var width = dojo.marginBox(node).w;
			if( minWidth > width )
				minWidth = width;
		}

		dojo.style(domNode.parentNode, "width", minWidth+"px");
	},

	onDragDetected : function() {
		this.inherited(arguments);
		this.startOffset = this.node.parentNode.scrollLeft;
	},

	moveItem : function(domNode, leftTop) {
		domNode.parentNode.scrollLeft = - (leftTop.l + this.movedOffset * this._liLength) + this.startOffset;
	},

	getItemLength : function(domNode) {
		return dojo.marginBox(domNode).w;
	},

	getItemInnerLength : function(domNode) {
		return dojo.contentBox(domNode).w;
	},

	getItemOffset : function(domNode) {
		return Math.round(-domNode.parentNode.scrollLeft);
	},

	setItemOffset : function(domNode, offset) {
		domNode.parentNode.scrollLeft = -offset;
	},

	/*
	 * Return -1, 0 or 1
	 * -1 : An item needs to be added at the end of the list
	 * 0  : Do nothing
	 * 1  : An item needs to be added at the start
	 */
	checkLooping : function(leftTop) {
		var left = leftTop.l + this.movedOffset * this._liLength;
		var result = 0;
		if( left <= -this._liLength )
		{
			result = -1;
			this.movedOffset++;
		}

		if( left >= 0 )
		{
			result = 1;
			this.movedOffset--;
		}

		return result;
	}
});
/**
 * @ignore
 */
dojo.provide("UIEffects.Scroll.VerticalList");

dojo.declare("UIEffects.Scroll.VerticalList", UIEffects.Scroll._base, {
	movedOffset: 0,
	startOffset: 0,

	initItem : function(domNode) {
		var minHeight = dojo.contentBox(domNode.parentNode).h;
		for(var node = domNode; node !== null; node = node.parentNode)
		{
			var height = dojo.marginBox(node).h;
			if( minHeight > height )
				minHeight = height;
		}

		dojo.style(domNode.parentNode, "height", minHeight+"px");
	},

	onDragDetected : function() {
		this.inherited(arguments);
		this.startOffset = this.node.parentNode.scrollTop;
	},

	moveItem : function(domNode, leftTop) {
		domNode.parentNode.scrollTop = - (leftTop.t + this.movedOffset * this._liLength) + this.startOffset;
	},
	getItemLength : function(domNode) {
		return dojo.marginBox(domNode).h;
	},

	getItemInnerLength : function(domNode) {
		return dojo.contentBox(domNode).h;
	},

	getItemOffset : function(domNode) {
		return Math.round(-domNode.parentNode.scrollTop);
	},

	setItemOffset : function(domNode, offset) {
		domNode.parentNode.scrollTop = -offset;
	},

	/*
	 * Return -1, 0 or 1
	 * -1 : An item needs to be added at the end of the list
	 * 0  : Do nothing
	 * 1  : An item needs to be added at the start
	 */
	checkLooping : function(leftTop) {
		if( this.movedOffset == undefined )
			this.movedOffset = 0;

		var top = leftTop.t + this.movedOffset * this._liLength;
		var result = 0;
		if( top <= -this._liLength )
		{
			result = -1;
			++this.movedOffset;
		}

		if( top >= this._liLength )
		{
			result = 1;
			--this.movedOffset;
		}

		return result;
	}
});
/**
 *
 *	ICTouchAPI's Scrolling Effect
 *
 *	Changelog
 *	---------
 *
 *	* 3.6 - 2001/09/07
 *              - crms00331791: Elastic Bounce effect
 *
 *	* 3.5 - 2011/09/06
 *
 *              - crms00333894: Slow easing
 *              - crms00334334: Scroll detection versus tap
 *              - crms00334319: Magnify
 *
 *	* 3.4 - 2011/08/23
 *
 *              - crms00331797: scrolling doesn't always start + easing stops brutally + enhance easing stop
 *
 *	* 3.3 - 2011/05/12
 *
 *		- Added a test in _decorate function : some webapp crashed because the handler.firstElementChild / handler.lastElementChild was null
 *
 *	* 3.2 - 2011/05/03
 *
 *		- Added Elastic effects
 *
 *	* 3.1 - 2011/05/02
 *
 *		- Added methods to completely remove a DOM element's scroll
 *
 *
 *	* 3.0 - 2011/24/03
 *
 *		- Simplified scrolling calculations by basing them on relative coordinates instead of absolute
 *		- Added back easing
 *		- Improved clicking/scrolling decision to make it faster, which gives better user experience (based on speed)
 *		- Reinforced defense mechanisms to avoid scroll locking in random state
 *
 *	* 2.2 - 2010/11/30
 *
 *		- Improved pagination.
 *
 *	* 2.1 - 2010/08/27
 *
 *		- Added methods for pagination
 *
 *	* 2.0 - 2010/07/20
 *
 *		- Complete redesign.
 *		- Scroll starts after a minimum offset (5px by default).
 *		- No other scroll can happen when there's one running.
 *		- Decorator-like design pattern kept.
 *		- Split into 3 states :
 *		- Waiting state > Observing State > Scrolling State > Waiting State.
 *			See "_initListeners" method for more information.
 *
 *
 *	* 1.4 - 2010/04/13
 *
 *		- Make it compatible with Firefox
 *
 *	* 1.3 - 2010/03/23
 *
 *		- No more click should be lost while scrolling
 *
 *	* 1.2 - 2010/03/08
 *
 *		- Make it compatible with Firefox
 *
 *	* 1.1 - 2010/02/24
 *
 *		- Check if valid dom node at initialization
 *
 *	* 1.0 - 2010/02/23
 *
 *		- Regrouped scrolling variables/methods in a unique root to prevent from erasing
 *		- Switched to decorator-like design pattern
 *
 *	* 0.2 - 2010/02/23
 *
 *		- Better events handling that allows to ignore what the dom tree looks like.
 *		- Greatly reduced init time.
 *
 *	* 0.1 - 2010/02/19
 *
 *		- Handles both vertical and horizontal scroll
 *		- Can scroll whatever content in a container, as long as it overflows.
 *		- Ease effect is de/activable.
 * @ignore
 *
 */
//var ICTouchAPI = ICTouchAPI || {};
//ICTouchAPI.UIEffects = ICTouchAPI.UIEffects || {};
FXScroll = {

	// Global variable that indicates if an element is currently scrolling
	// Allows for locking any other scroll.
	whoslocked : "",

	// Method to call to lock the scroll
	scrollLock : function scrollLock(who) {
		this.whoslocked = who;
	},

	// Method to call to unlock the scroll
	scrollUnLock : function scrollUnLock(who) {
		if (this.whoslocked === who) {
			this.whoslocked = null;
		}

	},


	observeState : function observeState(state, scope, func) {
		if (["start", "stop"].indexOf(state) >= 0 ) {
			return dojo.subscribe("_Scroll." + state, scope, func);
		} else {
			return false;
		}
	},

	unobserveState: function unobserveState(handler) {
		return dojo.unsubscribe(handler);
	},

	// Get scroll lock state
	lockState : function lockState() {
		return !!(this.whoslocked);
	},

	// Send an event to its target's parent document.
	_dispatchToUpperDocument : function _dispatchToUpperDocument (event) {
		// Get the target's window
		var defaultView = event.target.ownerDocument.defaultView;

		// If it's in an iframe, there's a parent document
		if (typeof defaultView.frameElement.ownerDocument.defaultView.frameElement != "undefined") {
			// The event is dispatched to the frame element (in the parent document then).
			defaultView.frameElement.dispatchEvent(event);
		} else {
			// Else return false.
			return false;
		}
	},

	// Dispatch the event to the current scrolling element.
	_dispatchToScrollingElement : function _dispatchToScrollingElement(event) {
		// If it's a new event.

		if (this.whoslocked && this.whoslocked._lastEvent != event ) {
			this.whoslocked._lastEvent = event ;
			this._delta = new Date().getTime() - event.timeStamp ;
			// Dispatch it to the current scrolling element.
			this.whoslocked.dispatchEvent(event);
		}
	},

	// Treat an iframe event
	treatEvent : function treatEvent(event) {

		// If something is scrolling
		if (this.lockState()) {
			// Dispatch the new event to the scrolling element.
			this._dispatchToScrollingElement(event);
			// If nothing scrolls
		} else {
			// Send the event to its parent document.
			this._dispatchToUpperDocument(event);
		}

	},

	// To initialize a scroll on an element.
	init : function init(args) {
		// If the decorated container is scrollable
		if (typeof args.handler.scrollTop != "undefined") {
			// Set scroll's attributes to handler
			this._decorate(args.handler);
			// Mixin with params
			this._mixinProperties(args.handler.IctScroll, args);
			// Set event listeners on handler
			this._initListeners(args.handler, this);
		}
		return args.handler;
	},

	// To initialize a scroll on an element.
	unInit : function unInit(handler) {
		// If the decorated container is scrollable
		if (handler.IctScroll) {
			// Reset coords
			this.reset(handler);
			// Make sure to remove all listeners
			handler.IctScroll.unInit();
			// Remove IctScroll object
			this._removeDecoration(handler);
			return true;
		} else {
			return false;
		}
	},

	// Reset node like it never scrolled
	reset : function reset(domNode) {
		// Is it an IctScroll scroll node
		if (domNode && domNode.IctScroll) {
			domNode.scrollTop = 0;
			domNode.scrollLeft = 0;
		}
	},

	// Initialize listeners :
	// The heart of this script
	_initListeners : function _initListeners(handler, FXScroll) {
		// How it works :

		// - Waiting State
		//		One listener on mousedown, nothing happens before mousedown
		//			When a mousedown occurs, the element enters Observing state

		// - Observing State
		//		The element listens to mousemove events.
		//		If scroll requirements are met (>2px speed, right scrolling direction) the element enters
		//			the scrolling state.
		//		If mouseup or mouseleave occurs before requirements are met, the elements goes back to waiting State.

		// - Scrolling State
		//		The element scrolls and no other scroll can happen.
		//		No click or mouseup events propagates nor bubbles.
		//		When scroll ends (mouseup or mouseleave), the element goes back to easing state.

		// - Easing state
		//		Starts when a scrolling state end.
		//		On touch it goes back to scrolling state
		//		On easing end it goes back to waiting state.

		// Method used to check if the element needs to start scrolling
		var doIScroll = function doIScroll(event) {
			// If the element is free to scroll.
			if (FXScroll.lockState()) {
				IDontScroll();
				return false;
			}

			// Update coords
			handler.IctScroll.setLastCoords(event);

			// Check the scroll direction
			var horizontal = Math.abs(handler.IctScroll.speedX)  >= handler.IctScroll.minimumSpeedX;
			var vertical = Math.abs(handler.IctScroll.speedY)  >= handler.IctScroll.minimumSpeedY;

			var bScroll = false;

			if(handler.IctScroll.scrollType == "both") {
				bScroll = (horizontal || vertical);
			}
			else {
				bScroll = ((horizontal && handler.IctScroll.scrollType == "horizontal") || (vertical && handler.IctScroll.scrollType == "vertical"));
			}

			// If a scroll is detected, switch to scrolling state
			if(bScroll) {
				// Remove "observing state" event listeners
				observingState("leave");
				// Add "scrolling state event listeners"
				scrollingState("enter");
				// Initial speed is used to begin the scroll
				onscroll(event);
			}

			return bScroll;

		};

		// Method called to quit everything and get back to waitingState
		var IDontScroll = function IDontScroll() {
			// This is a defense mechanism
			observingState("leave", true);
			scrollingState("leave", true);
			easingState("leave", true);
			// This is the desired state.
			waitingState("enter", true);
		};


		// On mousedown on a scrolling area, switch from "waiting state" to "observing state"
		var onmousedown = function onmousedown(event) {
			if (handler.IctScroll && handler.IctScroll.getState() != "disabled") {
				// Defense mechanisms
				if  (handler.IctScroll.bumperInterval) {
					clearInterval(handler.IctScroll.bumperInterval);
					handler.IctScroll.bumperInterval = null;
				}
				scrollingState("leave");
				easingState("leave");
				observingState("leave");
				// Set current position to check offset
				handler.IctScroll.setInitCoords(event);
				handler.IctScroll.setLastCoords(event);
				// Add "observing state" event listeners
				waitingState("leave");
				observingState("enter");
			}
		};

		// Method called when on "scrolling state"
		var onscroll = function onscroll(event) {
			// Save current click coords as last ones
			handler.IctScroll.setLastCoords(event);
			// Fix container size to avoid bad scroll/bounce effect
			if (!handler.IctScroll.dimensionsFixed) {
				var dimensions = window.getComputedStyle(handler);
				handler.style.height = dimensions['height'];
				handler.style.width = dimensions['width'];
				handler.IctScroll.dimensionsFixed = true;
			}
			// Scrolls according to current state
			scroll();
			// No need for the event to go any further
			event.stopPropagation();
			event.cancelBubble = true;
		};

		// The scrolling function.
		var scroll = function scroll() {

			// If moved ends up being false, the element hasn't moved.
			var val, ictscroll = handler.IctScroll, moved = false, distance = 0;

			// If scrollType is vertical, or "both"
			if (ictscroll.scrollType != "horizontal") {
				// Paginates (remove and add dom nodes if needed)
				val  = paginate() || 0;
				// Scrolls the element and compare the previous position to the new one.
				var beforeTop = handler.scrollTop;

				// Calculates the distance to scroll :
				// It must scroll by the speed, plus the height of an eventual pagination, plus an offset that could remain from a previous scroll
				distance = ictscroll.speedY - val - ictscroll.remainingY;

				// Scroll
				handler.scrollTop -= distance;

				// And erase the remaining offset
				ictscroll.remainingY = 0;

				// If the distance is greater than the handler.scrollTop value before it scrolled,
				// and only if it has paginated
				if ((distance > beforeTop) ) {
					// scrollTop value would be negative, but as it can't be negative, it will stop at 0.
					// The difference between the two gives an amount of pixels that are missing to the last scroll.
					// We save this offset and will add it to the next scroll.
					ictscroll.remainingY = beforeTop - distance;
				}



				// Set the flag
				////////////////////////////////////////////////////////////////////// START PATCH

				// crms00331797_Begin
				// If scrollTop == beforeTop, test if a pagination has been done in order to continue the scroll
				// Fix the problem of scroll blocks (Easing stops brutally)
				if (handler.scrollTop != beforeTop || val != 0) {
					// crms00331797_End
					moved = true;
					if(ictscroll.speedY<0 && ictscroll.currentTop) {
						handler.firstElementChild.style['padding-top']=ictscroll.previousTop;
						ictscroll.remainingY = 0;
						ictscroll.currentTop = 0 ;
					}
					if(ictscroll.speedY>0 && ictscroll.currentBottom) {
						handler.lastElementChild.style['padding-bottom']=ictscroll.previousBottom;
						ictscroll.remainingY = 0;
						ictscroll.currentBottom = 0 ;
					}
				} else {
					if (handler.IctScroll.getState() != "easing") {
						
					 	if(ictscroll.speedY<0) {
					 		ictscroll.currentBottom += -ictscroll.speedY ;
					 		handler.lastElementChild.style['padding-bottom']=''+ictscroll.currentBottom+'px';
							ictscroll.remainingY = 0;
					 	}
					 	if(ictscroll.speedY>0) {
					 		ictscroll.currentTop += (ictscroll.speedY/2) ;
					 		handler.firstElementChild.style['padding-top']=''+ictscroll.currentTop+'px'; 
							ictscroll.remainingY = 0;
					 	}
					}
				}
				////////////////////////////////////////////////////////////////////// END PATCH

			}

			// Same thing horizontally.
			if (ictscroll.scrollType != "vertical") {
				val  = paginate() || 0;
				var beforeLeft = handler.scrollLeft;
				handler.scrollLeft -= ictscroll.speedX - val;
				// crms00331797_Begin
				// If scrollLeft == beforeLeft, test if a pagination has been done in order to continue the scroll
				// Fix the problem of scroll blocks (Easing stops brutally)
				if (handler.scrollLeft != beforeLeft || val != 0) {
					moved = true;
				}
				// crms00331797_End
			}

			// Return true if the handler moved
			return moved;

		};

		// The easing function
		var startEasing = function startEasing() {
			// Only start easing if the last event was received before the easingTimeout
			// This prevents the handler from easing if the user slowly released its finger
			//if ((new Date().getTime() - (handler.IctScroll._lastEvent.timeStamp + handler.IctScroll._delta) ) <= handler.IctScroll.easingTimeout) {
			if(handler.IctScroll._delta <= handler.IctScroll.easingTimeout) {

				//update the initial easing speed
				var speed = Math.sqrt(Math.pow(handler.IctScroll.speedY, 2) + Math.pow(handler.IctScroll.speedX, 2));
				handler.IctScroll.speedX = handler.IctScroll.speedX < 0 ? -1 * speed : speed ;
				handler.IctScroll.speedY = handler.IctScroll.speedY < 0 ? -1 * speed : speed ;

				// The function that will tick at the desired fps rate.
				handler.IctScroll._easingHandler = setInterval(function() {
					// The speed has to be superior to 1px, no other scroll must be running, and easing must be activated
					var absSpeedX = Math.abs(handler.IctScroll.speedX);
					var absSpeedY = Math.abs(handler.IctScroll.speedY);

					if (Math.max(absSpeedX, absSpeedY) >=1 && !FXScroll.lockState() && handler.IctScroll.ease) {
						// crms00331797_Begin
						// Enhance the end of the easing step (linear end)
						if(absSpeedY < handler.IctScroll.linearEasingStep) {
							handler.IctScroll.speedY = handler.IctScroll.speedY < 0 ? handler.IctScroll.speedY + 1 : handler.IctScroll.speedY - 1;
						}
						else {
							handler.IctScroll.speedY *= handler.IctScroll.easingSpeed;
						}

						if(absSpeedX < handler.IctScroll.linearEasingStep) {
							handler.IctScroll.speedX = handler.IctScroll.speedX < 0 ? handler.IctScroll.speedX + 1 : handler.IctScroll.speedX - 1;
						}
						else {
							handler.IctScroll.speedX *= handler.IctScroll.easingSpeed;
						}
						// crms00331797_End

						// If despite everything the handler hasn't moved
						if (!scroll()) {
							// Stop scrolling (back to waiting state)
							IDontScroll();
						}
					} else {
						// Stop scrolling (back to waiting state)
						IDontScroll();
					}
				}, Math.round(1000/handler.IctScroll.fps));
			} else {
				// Stop scrolling (back to waiting state)
				IDontScroll();
			}
		};

		var stopEasing = function stopEasing() {
			// Clears the interval so the easing function stops ticking
			clearInterval(handler.IctScroll._easingHandler);
		};

		// Calls the pagination method
		var paginate = function paginate() {
			// If there's a need for pagination
			if (typeof handler.IctScroll.pagination == "object"){
				return handler.IctScroll.pagination.callback.call(handler.IctScroll.pagination.context || this);
			} else {
				return false;
			}
		};

		// Method called when leaving the area (mouse out, mouse up)
		var onmouseup = function onmouseup(event) {
			// Stop propagation and bubbling
			// We have been scrolling, no mouseup must be received by any other element.
			setTimeout(function () {
				scrollingState("leave");
				easingState("enter");
			}, 0);
			// Stops the event propagation as a mouseup here should only end the scrolling state
			event.stopPropagation();
			event.cancelBubble = true;
		};


		// Can call this function to stop a specific event to propagate.
		var stopEventPropagation = function stopEventPropagation(event) {
			event.stopPropagation();
		};

		// Can call this function to stop a specific event to bubble.
		var stopEventBubbling = function stopEventBubbling(event) {
			event.cancelBubble = true;
		};

		var disableClick = function disableClick(event)  {
			if (FXScroll.lockState()) {
				//console.log("cancelled", event.type, event.timeStamp);
				event.stopPropagation();
				event.cancelBubble = true;
			}
		}

		var waitingState  = function waitingState(way, force) {
			if (way == "enter") {
				if (handler.IctScroll.getState() != "waiting" || force) {
					//console.log(handler.className, "enter waiting");
					handler.IctScroll.setState("waiting");
					handler.addEventListener("mousedown", onmousedown, true);
					//handler.addEventListener("mouseup", disableClick, true);
					handler.addEventListener("click", disableClick, true);
				}
			} else {
				if (handler.IctScroll.getState() == "waiting" || force) {
					//console.log(handler.className, "enter waiting");
					handler.removeEventListener("mousedown", onmousedown, true);
					//handler.removeEventListener("mouseup", disableClick, true);
					handler.removeEventListener("click", disableClick, true);
				}
			}
		};

		var observingState = function observingState(way, force) {
			if (way == "enter") {
				if (handler.IctScroll.getState() != "observing") {
					handler.IctScroll.setState("observing");
					//console.log(handler.className, "enter observing");
					handler.addEventListener("mousemove", doIScroll, true);
					handler.addEventListener("mouseup", IDontScroll, true);
				}
			} else {
				if (handler.IctScroll.getState() == "observing" || force) {
					//console.log(handler.className, "leave observing");
					handler.removeEventListener("mousemove", doIScroll, true);
					handler.removeEventListener("mouseup", IDontScroll, true);
				}
			}
		};

		var scrollingState = function scrollingState(way, force) {
			if (way == "enter") {
				if (handler.IctScroll.getState() != "scrolling") {
					FXScroll.scrollLock(handler);
					dojo.publish("_Scroll.start");
					handler.IctScroll.setState("scrolling");
					//console.log(handler.className, "enter scrolling");
					handler.addEventListener("mousemove", onscroll, true);
					handler.addEventListener("mouseup", onmouseup, true);

					handler.addEventListener("click", stopEventBubbling, true);
					handler.addEventListener("click", stopEventPropagation, true);
				}
			} else {
				if (handler.IctScroll.getState() == "scrolling" || force) {
					FXScroll.scrollUnLock(handler);
						!force && dojo.publish("_Scroll.stop");
					//console.log(handler.className, "leave scrolling");
					handler.removeEventListener("mousemove", onscroll, true);
					handler.removeEventListener("mouseup", onmouseup, true);

					handler.removeEventListener("click", stopEventBubbling, true);
					handler.removeEventListener("click", stopEventPropagation, true);

					////////////////////////////////////////////////////////////////////// START PATCH
					if (handler.IctScroll.currentTop > 0 && !handler.IctScroll.bumperInterval) {

						handler.IctScroll.bumperInterval = setInterval(function () {
							handler.IctScroll.currentTop *= handler.IctScroll.bumpSpeed;
							handler.firstElementChild.style['padding-top'] = handler.IctScroll.currentTop + handler.IctScroll.previousTop + "px";

							if (handler.IctScroll.currentTop  < 1 && handler.IctScroll.bumperInterval) {
								clearInterval(handler.IctScroll.bumperInterval);
								handler.IctScroll.currentTop = 0;
								handler.IctScroll.bumperInterval  = null;
								handler.firstElementChild.style['padding-top']= handler.IctSroll.previousTop;
							}
						},  Math.round(1000/handler.IctScroll.fps));

					}

					if (handler.IctScroll.currentBottom > 0 && !handler.IctScroll.bumperInterval) {

						handler.IctScroll.bumperInterval = setInterval(function () {
							handler.IctScroll.currentBottom *= handler.IctScroll.bumpSpeed;
							handler.lastElementChild.style['padding-bottom'] = handler.IctScroll.currentBottom + handler.IctScroll.previousBottom + "px";

							if (handler.IctScroll.currentBottom  < 1 && handler.IctScroll.bumperInterval) {
								clearInterval(handler.IctScroll.bumperInterval);
								handler.IctScroll.currentBottom = 0;
								handler.IctScroll.bumperInterval  = null;
								handler.lastElementChild.style['padding-bottom']= handler.IctSroll.previousBottom;
							}
						},  Math.round(1000/handler.IctScroll.fps));

					}
				////////////////////////////////////////////////////////////////////// END PATCH

			}
		}
	};

	var backToScroll = function backToScroll(event) {
		easingState("leave");
		handler.IctScroll.setInitCoords(event);
		handler.IctScroll.setLastCoords(event);
		scrollingState("enter");
		//console.log("trying to block mousedown", event.timeStamp);
		event.stopPropagation();
		event.cancelBubble = true;
	};

	var easingState = function easingState(way, force) {
		if (way == "enter") {
			if (handler.IctScroll.getState() != "easing") {
				handler.IctScroll.setState("easing");
				//console.log(handler.className, "enter easing");
				handler.addEventListener("mousedown", backToScroll, true);
				handler.addEventListener("click", stopEventBubbling, true);
				handler.addEventListener("click", stopEventPropagation, true);
				startEasing();
			}
		} else {
			if (handler.IctScroll.getState() == "easing" || force) {
				stopEasing();
				//console.log(handler.className, "leave easing");
				handler.removeEventListener("mousedown", backToScroll, true);
				handler.removeEventListener("click", stopEventBubbling, true);
				handler.removeEventListener("click", stopEventPropagation, true);
			}
		}
	};

	waitingState("enter");

	handler.IctScroll.unInit= function unInit() {
		waitingState("leave", true);
		// defense
		observingState("leave", true);
		easingState("leave", true);
		scrollingState("leave", true);
	};

},

// Returns an object that is the handler decorator.
_decorate : function _decorate(handler) {

	////////////////////////////////////////////////////////////////////// START PATCH
	var initSpringVars = function initSpringVars(handler) {
		if (handler.firstElementChild) {
			this.previousTop = handler.firstElementChild.style['padding-top'];
			this.currentTop = parseInt(document.defaultView.getComputedStyle(handler.firstElementChild,null)['padding-top']);
		}
		if (handler.lastElementChild) {
			this.previousBottom = handler.lastElementChild.style['padding-bottom'];
			this.currentBottom = parseInt(document.defaultView.getComputedStyle(handler.lastElementChild,null)['padding-bottom']);
		}
	};
	initSpringVars(handler);
	////////////////////////////////////////////////////////////////////// END PATCH


	// Simply save the events coordinates
	var setInitCoords = function setInitCoords(event)  {
		this.lastCoordX = event.screenX;
		this.lastCoordY = event.screenY;
	};

	// Save the init coordinates but calculate the speed.
	// Makes sure the event is treated only once

	var setLastCoords = function setLastCoords(event) {
		// Defense mechanism
		if (this._lastEvent != event) {
			this.speedX =  event.screenX - this.lastCoordX;
			this.speedY = event.screenY - this.lastCoordY;
			this.lastCoordX = event.screenX;
			this.lastCoordY = event.screenY;
			if(this._lastEvent) {
				this._delta = event.timeStamp - this._lastEvent.timeStamp;
			}
			else {
				this._delta = event.timeStamp;
			}
			this._lastEvent = event;
			//this._delta = new Date().getTime() - event.timeStamp ;
		}


	};

	var setState = function setState(state) {
		return this._scrollStatus = state;
	};

	var getState = function getState() {
		return this._scrollStatus;
	};

	var setDisableScroll = function setDisableScroll (bDisable) {
		return this._scrollStatus = bDisable ? "disabled" : "enabled";
	};

	//
	//		var addOffset = function addOffset(axis, amount) {
	//			return this["initCoord" + (axis.toUpperCase())] += amount;
	//		};

	// A set of attributes and methods to set them.
	handler.IctScroll =  {
		// Coords
		lastCoordX : 0,
		lastCoordY : 0,
		// Scroll detection
		minimumSpeedX : 12,
		minimumSpeedY : 6,
		// Effects
		bounce: true,
		ease : true,
		//Scroll parameters
		easingSpeed : .9,
		bumpSpeed: .4,
		fps : 20,
		easingTimeout : 200,
		_delta : 0,
		remainingY: 0,
		linearEasingStep : 8,
		scrollType : "both",
		_scrollStatus : "freshInit",
		// The first scroll fixes the container's dimensions
		dimensionsFixed: false,
		// Functions
		setLastCoords : setLastCoords,
		setInitCoords : setInitCoords,
		setState : setState,
		getState : getState,
		setDisableScroll: setDisableScroll,
		////////////////////////////////////////////////////////////////////// START PATCH
		previousTop: '',
		previousBottom: '',
		currentTop: 0,
		currentBottom: 0,
		initSpringVars: initSpringVars,
		bumperInterval: null
		////////////////////////////////////////////////////////////////////// END PATCH
	}

	handler.deactivate = function deactivate () {
		this.IctScroll.setDisableScroll(true);
	};

	handler.activate = function activate () {
		this.IctScroll.setDisableScroll(false);
	};

},

_removeDecoration : function _removeDecoration(handler) {
	return delete handler.IctScroll;
},

// Mixes two objects
_mixinProperties : function _mixinProperties(list, args) {
	for (var i in args) {
		if (args.hasOwnProperty(i)) {
			list[i] = args[i];
		}
	}
	return list;
}


}



/**
 * @ignore
 */
dojo.provide("UIEffects.Resize");

dojo.declare("UIEffects.Resize", [ ], {
	domContainer: null,
	
	constructor: function(container)
	{
		this.domContainer = container;
	},
	
	resizeWidth: function (){
		var arrElements = dojo.query("> *", this.domContainer);
		var width = 0;
		arrElements.forEach(function(elt){
			var cs = dojo.getComputedStyle(elt);
			width += parseInt(cs.width) + parseInt(cs.marginLeft) + parseInt(cs.marginRight) + parseInt(cs.borderLeftWidth) + parseInt(cs.borderRightWidth);
		});
		
		this.domContainer.style.width = width + "px";		
	},
	
	resizeHeight: function (){
		var arrElements = dojo.query("> *", this.domContainer);
		var height = 0;
		arrElements.forEach(function(elt){
			var cs = dojo.getComputedStyle(elt);
			height += parseInt(cs.height) + parseInt(cs.marginTop) + parseInt(cs.marginBottom) + parseInt(cs.borderTopWidth) + parseInt(cs.borderBottomWidth);
		});
		
		this.domContainer.style.height = height + "px";		
	},
});
/**
 * @ignore
 */
dojo.provide("UIEffects.Transition._base");

dojo.declare("UIEffects.Transition._base", null,
	{
		floatDuration: 0.5,
		_hasTransition: false,
		funcTransitionEnd: null,
		_launched: false,
		
		constructor: function(srcNode, params)
		{
			this.domNode = srcNode;
			
			if( !params ) params = { };
			if(params.floatDuration){ this.floatDuration = params.floatDuration; }
			if(params.funcTransitionEnd && dojo.isFunction(params.funcTransitionEnd)){ this.funcTransitionEnd = params.funcTransitionEnd; }
			
			this.floatDuration = parseFloat(this.floatDuration);
			this._hasTransition = dojo.isWebkit || dojo.isSafari;
			if( this._hasTransition )
			{
				this._setTransition();
				if( this.funcTransitionEnd )
					this._handle = dojo.connect(this.domNode, "webkitTransitionEnd", this.funcTransitionEnd);
			}
		},
		
		destroy: function()
		{
			this._removeTransition();
			this._applyStart();
			this.funcTransitionEnd = null;
			if( this._handle )
				dojo.disconnect(this._handle);
		},
		
		_setTransition: function()
		{
			this.domNode.style.setProperty("-webkit-transition-duration", this.floatDuration+"s");
		},	
		
		_removeTransition: function()
		{
			this.domNode.style.removeProperty("-webkit-transition-duration");
		},
		
		launch: function()
		{
			this._applyEnd();
			this._launched = true;
			if( this.funcTransitionEnd && !this._hasTransition )
				this.funcTransitionEnd();
		},
		
		warpLaunch: function(funcCallback)
		{
			if( !this._hasTransition )
			{
				this._applyEnd();
				if( funcCallback )
					funcCallback();
				return;
			}
			
			this._removeTransition();
			this._applyEnd();
			var step = dojo.hitch(this, function() {
				this._setTransition();
				if( funcCallback )
					funcCallback();
			});
			setTimeout(step, 0);
		},
		
		revert: function()
		{
			this._applyStart();
			this._launched = false;
			if( this.funcTransitionEnd && !this._hasTransition )
				this.funcTransitionEnd();
		},
		
		warpRevert: function(funcCallback)
		{
			if( !this._hasTransition )
			{
				this._applyStart();
				if( funcCallback )
					funcCallback();
				return;
			}
			
			this._removeTransition();
			this._applyStart();
			var step = dojo.hitch(this, function() {
				this._setTransition();
				if( funcCallback )
					funcCallback();
			});
			setTimeout(step, 0);
		},
		
		toggle: function()
		{
			if( this._launched )
				this.revert();
			else
				this.launch();
		},
	}
);
/**
 * @ignore
 */
dojo.provide("UIEffects.Transition.Slide");

dojo.declare("UIEffects.Transition.Slide", UIEffects.Transition._base,
	{
		strDirection: "left",
		
		_initLeft: 0,
		_initTop: 0,
		_finalLeft: 0,
		_finalTop: 0,
		
		constructor: function(srcNode, params)
		{
			if( !params ) params = { };
			if(params.strDirection){ this.strDirection = params.strDirection; }
			ICTouchAPI.addOnInit(this, this._loaded);
		},
		
		_loaded: function()
		{
			var mb = dojo.marginBox(this.domNode);
			var s = this.domNode.style;

			if( !s.position )
				s.position = "relative";
			if( s.left )
				this._finalLeft = this._initLeft = parseInt(s.left);
			if( s.top )
				this._finalTop = this._initTop = parseInt(s.top);
				
			switch(this.strDirection)
			{
				case "left":
				default:
					this._finalLeft -= mb.w;
					break;
				case "right":
					this._finalLeft += mb.w;
					break;
				case "top":
					this._finalTop -= mb.h;
					break;
				case "bottom":
					this._finalTop += mb.h;
					break;
			}
			
			this._applyStart();
		},
		
		_applyStart: function()
		{
			this._setStyle(this._initLeft, this._initTop);
		},
		
		_applyEnd: function()
		{
			this._setStyle(this._finalLeft, this._finalTop);
		},
		
		_setStyle: function(left, top)
		{
			dojo.style(this.domNode, "left", left+"px");
			dojo.style(this.domNode, "top", top+"px");
		},
		
		_setTransition: function()
		{
			this.inherited(arguments);
			this.domNode.style.setProperty("-webkit-transition-property", "top, left");
		},
		
		_removeTransition: function()
		{
			this.inherited(arguments);
			this.domNode.style.removeProperty("-webkit-transition-property");
		}
	}
);
/**
 * @ignore
 */
dojo.provide("UIEffects.Transition.Fade");

dojo.declare("UIEffects.Transition.Fade", UIEffects.Transition._base,
	{
		constructor: function(srcNode, params)
		{
			this._applyStart();
		},
		
		_applyStart: function()
		{
			dojo.style(this.domNode, "opacity", 0);
		},
		
		_applyEnd: function()
		{
			dojo.style(this.domNode, "opacity", 1);
		},
		
		_setTransition: function()
		{
			this.inherited(arguments);
			this.domNode.style.setProperty("-webkit-transition-property", "opacity");
		},
		
		_removeTransition: function()
		{
			this.inherited(arguments);
			this.domNode.style.removeProperty("-webkit-transition-property");
		}
	}
);
/**
 * @ignore
 */
dojo.provide("UIEffects.Transition.CssClass");

dojo.declare("UIEffects.Transition.CssClass", null, {
	floatDuration: -1,
	funcTransitionEnd: null,
	strOldClass: null,
	strNewClass: "",
	objTransition: null,
	domNode: null,
	domNew: null,
	
	constructor: function(srcNode, params)
	{
		this.domNode = srcNode;
		
		if( !params ) params = { };
		if( params.floatDuration ) this.floatDuration = params.floatDuration;
		if( params.funcTransitionEnd ) this.funcTransitionEnd = params.funcTransitionEnd;
		if( params.strOldClass ) this.strOldClass = params.strOldClass;
		if( params.strNewClass ) this.strNewClass = params.strNewClass;
	},
	
	launch: function()
	{
		this.domNew = dojo.clone(this.domNode);
		dojo.addClass(this.domNew, this.strNewClass);
		if( this.strOldClass !== null )
			dojo.removeClass(this.domNew, this.strOldClass);
		dojo.place(this.domNew, this.domNode, "after");
		
		var func = dojo.hitch(this, this.ended);
		var params = { "funcTransitionEnd": func };
		
		if( this.floatDuration > 0 )
			params.floatDuration = this.floatDuration;
		var transition = this.objTransition = new UIEffects.Transition.Fade(this.domNew, params);
		setTimeout(function(){
			transition.launch();
		}, 10);
	},
	
	ended: function()
	{
		dojo.destroy(this.domNode);
		this.domNode = this.domNew;
		
		if( this.funcTransitionEnd )
			this.funcTransitionEnd(this.domNew);
	}
});
